package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.40.2 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.40.2) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@("`0`````````````#`+<````!````````(%``````````0```````
M`*2H``````!``#@`"`!``!D`&`````8````$`````````$``````````0```
M``````!``````````<`````````!P``````````(`````P````0````````"
M``````````(``````````@``````````%P`````````7``````````$````!
M````!0```````````````````````````````````````)R,````````G(P`
M``````$```````$````&````````G4````````&=0````````9U`````````
M!3`````````%R````````0```````@````8```````"=6````````9U8````
M```!G5@````````!X`````````'@``````````@````$````!`````````(8
M`````````A@````````"&``````````L`````````"P`````````!&1TY5``
M```$````````E0````````"5`````````)4``````````/P`````````_```
M```````$9'3E4@````0```````"=0````````9U````````!G4`````````"
MP`````````+```````````$O=7-R+VQI8F5X96,O;&0N96QF7W-O```````'
M````!`````%.971"4T0``#N:R@`````$````!`````-085@`````````````
M``!A````90``````````````,`````T``````````````#<````.````/P``
M```````O``````````,```!+````7@`````````?````2@````8````=````
M%`````````!!`````````%H`````````%P```#L```!5``````````````!A
M````8P```"X````2````5````$,````5````60```$0`````````1P```"(`
M```T````6````"8````J``````````<```!-``````````L````*````9```
M`%<```!)````(````$(```!B`````````%P`````````&````"T```!,````
M````````````````$0```#(``````````````%T````K``````````D`````
M````)````#@`````````3@````4```!%````&P```#H`````````````````
M````````2``````````Y````40```%\`````````8```````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````!,`````````````````````````````````
M```````````````````C````(0`````````I`````````"<````H````+```
M````````````#``````````<````&@```!D``````````````!``````````
M,P```#$````V````'@`````````````````````````$``````````````!`
M````)0```````````````````!8``````````````#X````(````````````
M````````/0```%``````````4P``````````````/```````````````3P``
M``````!&````#P````````!2````5@```#4```!;````````````````````
M```````````````````````#```*````````(%`````````````````#```0
M```````!G5```````````````)(2``````````````````````````````P2
M`````````````````````````````!02````````````````````````````
M`U$2```*````````*5``````````"````!PB````````````````````````
M`````"L2`````````````````````````````O@2````````````````````
M`````````RH0```5```````!HG```````````````#$2````````````````
M`````````````V(2```*````````,V``````````(````#@2````````````
M`````````````````P42`````````````````````````````$`2````````
M`````````````````````WT2```*````````/,``````````9````(,2````
M`````````````````````````SD0```5```````!HP@``````````````$@2
M`````````````````````````````XD0```4```````!HG``````````````
M`%@2`````````````````````````````&<2````````````````````````
M`````&T2`````````````````````````````'02````````````````````
M`````````'X2`````````````````````````````(@2````````````````
M`````````````S@0```5```````!HP@``````````````)$2````````````
M`````````````````@,2`````````````````````````````)<2````````
M`````````````````````)X2`````````````````````````````*P2````
M`````````````````````````+T1`````````````````````````````HDB
M`````````````````````````````,H2````````````````````````````
M`Y`2```*````````*X0`````````+````YT2```*````````*[``````````
M;````ZD2```*````````-(`````````%O````-@6````````````````````
M`````````.L2`````````````````````````````Q<2````````````````
M`````````````/82`````````````````````````````/\1```5```````!
MHP``````````"````0<2`````````````````````````````NX2````````
M`````````````````````1,2`````````````````````````````1H2````
M`````````````````````````[82```*````````3>0````````!)````2T1
M`````````````````````````````3\2````````````````````````````
M`4\2`````````````````````````````5T2````````````````````````
M`````6(2`````````````````````````````\$1```4```````!HF@`````
M````"````7$2`````````````````````````````7T2````````````````
M`````````````9(2`````````````````````````````9T2````````````
M`````````````````TP0```5```````!HP@``````````````J\2````````
M`````````````````````;`2`````````````````````````````_P2```*
M````````(%``````````"````;X2`````````````````````````````<82
M`````````````````````````````=(2````````````````````````````
M`T00```5```````!HP@``````````````\L2```*````````+"`````````$
MQ````=@2`````````````````````````````>,2````````````````````
M`````````]@2```*````````3Q`````````!N````?02````````````````
M`````````````^(0```5```````!HG```````````````^X2```*````````
M*+``````````0````_82```*````````49`````````"&````?L2````````
M`````````````````````FHB`````````````````````````````_L2```*
M````````(%``````````"```!`,2```*````````*6`````````")```!!`2
M```*````````,X``````````^````@(2````````````````````````````
M!"(2```*````````,.0`````````<````@L2````````````````````````
M`````H(2````````````````````````````!#`2```*````````,50`````
M````G````AP2`````````````````````````````B42````````````````
M`````````````BL2`````````````````````````````N$2````````````
M`````````````````C`2`````````````````````````````M01```5````
M```!HH@`````````"````C@2`````````````````````````````PP2````
M`````````````````````````C\2````````````````````````````!#@2
M```*````````39``````````5````D81````````````````````````````
M`LD1```4```````!H>@`````````"```!$$2```*````````.D`````````"
M>````282`````````````````````````````DL2````````````````````
M`````````QX2``````````````````````````!L:6)P97)L+G-O`')E86QL
M;V,`;65M;6]V90!?7V-X85]F:6YA;&EZ90!R;61I<@!S=')C:'(`<W1R;F-M
M<`!S=')R8VAR`&)O;W1?1'EN84QO861E<@!P97)L7V-O;G-T<G5C=`!A8F]R
M=`!U;FQI;FL`7U]L<W1A=#4P`'!E<FQ?9G)E90!?7W-T870U,`!?97AI=`!M
M96UC<'D`<&5R;%]D97-T<G5C=`!?7W-T86-K7V-H:U]F86EL`%!,7V1O7W5N
M9'5M<`!097)L7W-V7W-E='!V`%!,7V-U<G)E;G1?8V]N=&5X=`!P97)L7V%L
M;&]C`&-L;W-E9&ER`&5N=FER;VX`7U]O<&5N9&ER,S``;65M<V5T`%!E<FQ?
M<V%F97-Y<VUA;&QO8P!?7W-T86-K7V-H:U]G=6%R9`!097)L7V=V7V9E=&-H
M<'8`7U]S<')I;G1F7V-H:P!R96%D`%!E<FQ?<WES7VEN:70S`%!E<FQ?9V5T
M7W-V`%!E<FQ?8W)O86M?;F]C;VYT97AT`%!E<FQ?;F5W6%,`4&5R;%]A=&9O
M<FM?=6YL;V-K`%!E<FQ?<WES7W1E<FT`7U]E<G)N;P!097)L7VUG7W-E=`!M
M:V1I<@!P97)L7W!A<G-E`%!E<FQ?;F5W6%-?9FQA9W,`86-C97-S`&=E='!I
M9`!V9G!R:6YT9@!097)L7V%T9F]R:U]L;V-K`'!E<FQ?<G5N`&QS965K`&]P
M96X`<W!R:6YT9@!G971U:60`<W1R;&5N`%]?<T8`<W1R8VUP`&QI8FTN<V\N
M,`!L:6)C<GEP="YS;RXQ`%]?9&5R96=I<W1E<E]F<F%M95]I;F9O`'-T<F1U
M<`!?7W)E9VES=&5R7V9R86UE7VEN9F\`;&EB<'1H<F5A9"YS;RXQ`'!T:')E
M861?871F;W)K`&QI8F,N<V\N,3(`7U]P<F]G;F%M90!?7W!S7W-T<FEN9W,`
M7U]G971P=W5I9#4P`%]?<WES8V%L;`!?7W-T<&-P>5]C:&L`871E>&ET`%]L
M:6)C7VEN:70`<W1R=&]K`%]?<F5A9&1I<C,P`%]?8G-S7W-T87)T7U\`7U]B
M<W-?96YD7U\`7U]E;F1?7P!?96YD`'!A<E]C=7)R96YT7V5X96,`9V5T7W5S
M97)N86UE7V9R;VU?9V5T<'=U:60`<&%R7V-L96%N=7``7V5D871A`'!A<E]B
M87-E;F%M90!P87)?9&ER;F%M90!P87)?;6MT;7!D:7(`<VAA7W5P9&%T90!M
M>5]P87)?<&P`<&%R7VEN:71?96YV`'-H85]F:6YA;`!?7V)S<U]S=&%R=`!X
M<U]I;FET`&UA:6X`7U]S=&%R=`!P87)?9FEN9'!R;V<`<&%R7W-E='5P7VQI
M8G!A=&@`<&%R7V5N=E]C;&5A;@!P87)?9&EE`'-H85]I;FET`%A37TEN=&5R
M;F%L<U]005)?0D]/5``O=7-R+W!K9R]L:6(O<&5R;#4O-2XT,"XP+V%A<F-H
M-C1E8BUN971B<V0M=&AR96%D+6UU;'1I+T-/4D4Z+W5S<B]P:V<O;&EB````
M``````````````&=0`````````0#````````45````````&=2`````````0#
M````````4-````````&?0`````````0#```````!G5````````&?6```````
M``0#```````!G4````````&?>`````````0#```````!HP````````&?B```
M``````0#```````!G4@```````&?D`````````0#```````!HF@```````&?
ML`````````0#```````!G5@```````&?R`````````0#```````!HH@`````
M``&?V`````````0#```````!H>@```````&?X`````````0#````````.D``
M``````&AZ`````````0#```````!HH````````&A\`````````0#```````!
MH?````````&B"`````````0#````````4\````````&B$`````````0#````
M````4\@```````&B(`````````0#````````4]````````&B*`````````0#
M````````4^````````&B,`````````0#````````4^@```````&B.```````
M``0#````````4_````````&B0`````````0#````````4_@```````&B4```
M``````0#````````5`````````&B6`````````0#````````5`@```````&B
M:`````````0#````````5W@```````&?2`````<```0!````````````````
M``&?4````!,```0!``````````````````&?8````"$```0!````````````
M``````&?:````"(```0!``````````````````&?<````"<```0&````````
M``````````&?@````#$```0!``````````````````&?F````#H```0!````
M``````````````&?J````$P```0!``````````````````&?P````%(```0!
M``````````````````&?T````%\```0!``````````````````&@``````,`
M``0"``````````````````&@"`````0```0"``````````````````&@$```
M``4```0"``````````````````&@&`````<```0"``````````````````&@
M(`````@```0"``````````````````&@*`````D```0"````````````````
M``&@,`````L```0"``````````````````&@.`````T```0"````````````
M``````&@0`````X```0"``````````````````&@2`````\```0"````````
M``````````&@4````!$```0"``````````````````&@6````!4```0"````
M``````````````&@8````!8```0"``````````````````&@:````!<```0"
M``````````````````&@<````!@```0"``````````````````&@>````!D`
M``0"``````````````````&@@````!H```0"``````````````````&@B```
M`!P```0"``````````````````&@D````!T```0"``````````````````&@
MF````!X```0"``````````````````&@H````!\```0"````````````````
M``&@J````"````0"``````````````````&@L````"(```0"````````````
M``````&@N````",```0"``````````````````&@P````"@```0"````````
M``````````&@R````"D```0"``````````````````&@T````"H```0"````
M``````````````&@V````"P```0"``````````````````&@X````"T```0"
M``````````````````&@Z````"X```0"``````````````````&@\````"\`
M``0"``````````````````&@^````#(```0"``````````````````&A````
M`#,```0"``````````````````&A"````#0```0"``````````````````&A
M$````#4```0"``````````````````&A&````#<```0"````````````````
M``&A(````#@```0"``````````````````&A*````#D```0"````````````
M``````&A,````#P```0"``````````````````&A.````#T```0"````````
M``````````&A0````#\```0"``````````````````&A2````$````0"````
M``````````````&A4````$$```0"``````````````````&A6````$0```0"
M``````````````````&A8````$4```0"``````````````````&A:````$<`
M``0"``````````````````&A<````$L```0"``````````````````&A>```
M`$P```0"``````````````````&A@````%````0"``````````````````&A
MB````%,```0"``````````````````&AD````%4```0"````````````````
M``&AF````%8```0"``````````````````&AH````%<```0"````````````
M``````&AJ````%@```0"``````````````````&AL````%D```0"````````
M``````````&AN````%L```0"``````````````````&AP````%P```0"````
M``````````````&AR````%T```0"``````````````````&AT````&(```0"
M``````````````````&AV````&,```0"``````````````````&AX````&0`
M``0"`````````````````````/![OZG0``"0$?Y'^1#B/Y$@`A_6'R`#U1\@
M`]4?(`/5T```L!$"0/D0`@"1(`(?UM```+`1!D#Y$"(`D2`"'];0``"P$0I`
M^1!"`)$@`A_6T```L!$.0/D08@"1(`(?UM```+`1$D#Y$((`D2`"'];0``"P
M$19`^1"B`)$@`A_6T```L!$:0/D0P@"1(`(?UM```+`1'D#Y$.(`D2`"'];0
M``"P$2)`^1`"`9$@`A_6T```L!$F0/D0(@&1(`(?UM```+`1*D#Y$$(!D2`"
M'];0``"P$2Y`^1!B`9$@`A_6T```L!$R0/D0@@&1(`(?UM```+`1-D#Y$*(!
MD2`"'];0``"P$3I`^1#"`9$@`A_6T```L!$^0/D0X@&1(`(?UM```+`10D#Y
M$`("D2`"'];0``"P$49`^1`B`I$@`A_6T```L!%*0/D00@*1(`(?UM```+`1
M3D#Y$&("D2`"'];0``"P$5)`^1""`I$@`A_6T```L!%60/D0H@*1(`(?UM``
M`+`16D#Y$,("D2`"'];0``"P$5Y`^1#B`I$@`A_6T```L!%B0/D0`@.1(`(?
MUM```+`19D#Y$"(#D2`"'];0``"P$6I`^1!"`Y$@`A_6T```L!%N0/D08@.1
M(`(?UM```+`1<D#Y$((#D2`"'];0``"P$79`^1"B`Y$@`A_6T```L!%Z0/D0
MP@.1(`(?UM```+`1?D#Y$.(#D2`"'];0``"P$8)`^1`"!)$@`A_6T```L!&&
M0/D0(@21(`(?UM```+`1BD#Y$$($D2`"'];0``"P$8Y`^1!B!)$@`A_6T```
ML!&20/D0@@21(`(?UM```+`1ED#Y$*($D2`"'];0``"P$9I`^1#"!)$@`A_6
MT```L!&>0/D0X@21(`(?UM```+`1HD#Y$`(%D2`"'];0``"P$:9`^1`B!9$@
M`A_6T```L!&J0/D00@61(`(?UM```+`1KD#Y$&(%D2`"'];0``"P$;)`^1""
M!9$@`A_6T```L!&V0/D0H@61(`(?UM```+`1ND#Y$,(%D2`"'];0``"P$;Y`
M^1#B!9$@`A_6T```L!'"0/D0`@:1(`(?UM```+`1QD#Y$"(&D2`"'];0``"P
M$<I`^1!"!I$@`A_6T```L!'.0/D08@:1(`(?UM```+`1TD#Y$((&D2`"'];0
M``"P$=9`^1"B!I$@`A_6T```L!':0/D0P@:1(`(?UM```+`1WD#Y$.(&D2`"
M'];0``"0$>)`^1`"!Y$@`A_6T```D!'F0/D0(@>1(`(?UM```)`1ZD#Y$$('
MD2`"'];0``"0$>Y`^1!B!Y$@`A_6T```D!'R0/D0@@>1(`(?UN$#`JHT```4
M``````````#]>[VIH```\/T#`)$`H$?Y]1,`^;4``/"U(C61OP(`ZX(!`%3S
M4P&I%`0`T90"%<L3`(#2E/Y#TZ!Z<_@``#_6?P(4ZW,&`)&!__]4\U-!J?43
M0/G]>\.HP`-?UA\@`]7]>[VI_0,`D?-3`:FS``#P]1,`^;4``/!STD?YM=Y'
M^7\"%>MB`0!48`Y`N1\@$/&!`0!4=`)`^7-B`)%@@E_X```_UH`"`/E_`A7K
MX_[_5/-30:GU$T#Y_7O#J,`#7]8(__^7_7N\J?T#`)'S4P&I]5L"J?<;`/EA
M"0"T]`,!JJ$``/"U``#P(>1'^8("0/FDOD?Y0P!`^30``/F!"D#Y@0``^:$`
M`/`C!P"T(>Q'^2,``/E"`$#Y0P!`.>,``#1"!`"1?[P`<8'__U0B``#Y0P!`
M.6/__S5```"TV?[_EYC__Y>@``#P`-A'^<`&`+2@``#PMP``\/<"-9$`K$?Y
M_P(`ZT(!`%06!`#1$P"`TM8"%\O6_D/3X'IS^```/]9_`A;K<P8`D8'__U0`
M``"0`(`!D</^_Y>W``#PH```\/<"-9$`Q$?Y_P(`ZT(!`%06!`#1$P"`TM8"
M%\O6_D/3X'IS^```/];?`A/K<P8`D8'__U2UOD?Y@`I`N8$"0/FB`D#YTPL`
ME([^_Y<A[$?YP@``D$(`"I$B``#Y`/K_M=#__Q<"``#P0J`.D6,"@-)!`(!2
M@`"`TO+^_Y<@`(!2Q/[_EX___Y?*__\7`````/U[O*G]`P"1\U,!J:`&`+3W
M8P.I]P,`JJ```/``O$?Y&`!`^3@&`+3U6P*I]@,!JA4`@%+@`D`Y'_0`<008
M0'K!`@!4$P-`^=,!`+2U?D"3]`,8JA\@`]7B`Q6JX0,7JN`#$ZI__O^7@```
M-6!J=3@?]`!Q``(`5)..0/CS_O^U]5M"J?=C0ZG@`Q.J\U-!J?U[Q*C``U_6
M]0,7JJ`>0#@?]`!Q!!A`>J'__U2U`A=+Y?__%Y0"&,NU!@"1<P(5BY3^0Y/4
M`@"YX`,3JO-30:GU6T*I]V-#J?U[Q*C``U_6$P"`TNG__Q<3`(#2]V-#J>;_
M_Q?]>[*IH0``\/T#`)$AP$?Y(@!`^>)O`/D"`(#2P```M/-3`:GS`P"J`0!`
M.6$!`#7S4T&IH```\`#`1_GA;T#Y`@!`^2$``NL"`(#2X0@`5/U[SJC``U_6
ME?[_E_0#`*J`_O^T]V,#J1<``/#X`P&1]V(0D?5;`JG@`Q2J$/__ER`%`+0!
M-$`Y%30`D3^X`'%A``!4H09`.0'__S0`-$`Y'[@`<0`%`%3@`Q.J^/[_E_8#
M`*K@`Q6J]?[_E\`"``L`"``1`'Q`D_7^_Y?C`Q6JX@,3JO4#`*KA`Q>JX/[_
ME^$#&*K@`Q6J1?[_EQ\$`#&@``!4X$M`N0`,%!(?$$!QX`(`5.`#%:HQ_O^7
MX`,5JB/^_Y?@`Q2JZ?[_ET#[_[4?(`/5X`,4JEW^_Y?@`Q.J`_[_E_-30:GU
M6T*I]V-#J;W__Q>@!D`Y'[@`<>'Z_U2@"D`Y(/G_--3__Q?@`Q6JJ/__E^K_
M_Q?S4P&I]5L"J?=C`ZDV_O^7'R`#U?U[M*FB``#P_0,`D4+`1_GS"P#YX8,`
MD?,#`*I#`$#YXU\`^0,`@-(._O^7(`(`->$K0+D"`)12(0P4$C\00'$D$$)Z
MH`$`5*$``/`AP$?YXE]`^2,`0/E"``/K`P"`TH$!`%3S"T#Y_7O,J,`#7]8`
M`(!2]?__%^`#$ZI!`(!2=/[_EQ\``''@%Y\:[___%Q#^_Y?_PQ#1HP``\.$#
M`*H"@(#2_7L`J?T#`)%CP$?Y\PL`^?.C`)%D`$#YY!<"^00`@-+@`Q.JPOW_
MEP`$`-$?`!/KJ```5`<``!0?]!\X'P`3ZX```%0!`$`Y/[P`<6#__U3@`Q.J
MX06`4L7]_Y=``@"T'P`3Z\@!`%3@`Q.J8/[_EZ$``/`AP$?YXA="^2,`0/E"
M``/K`P"`TH$"`%3]>T"I\PM`^?_#$)'``U_6'P``.?+__Q>@``#P`,!'^>$7
M0OD"`$#Y(0`"ZP(`@-+A``!4_7M`J0```/#S"T#Y`"`/D?_#$)%&_O\7U?W_
ME_U[NJFB``#P_0,`D?-3`:GS`P"J]`,!JO5;`JGW8P.I0L!'^2``0#E!`$#Y
MX2\`^0$`@-(?]`!QE!:4FN`#%*I4_O^7]@,`JN%3`9'@`Q.J\/[_ET`(`+3U
M`P"J3?[_E]\"`.OI!@!4]U>`N;@``/#W\GW38`)`.1_T`'$$&$!Z0`P`5.,#
M$ZIB'$`X7_0`<4080'JA__]48P`3RV-\0),8OT?YP`H`D0```XL6`T#Y//[_
ME\!J-_C``@"TXU>`N>`#$ZH"`T#Y`11`.$-X8_@_]`!Q)!A`>F$``#G```!4
M`A1`.&(<`#A?]`!Q1!A`>H'__U2A!X!2``"`TF$4`#B":F`X8F@@.``$`)&B
M__\UH```\`#`1_GA+T#Y`@!`^2$``NL"`(#2H0<`5/-30:GU6T*I]V-#J?U[
MQJC``U_6`0"`TA\@`]6":F$XHFHA."$$`)&B__\U[O__%[@``/#Y:P2I`+]'
M^1H`0/E``T#Y``4`M.`#&JH5`(!2'R`#U0&,0/CB`Q4JM08`$:'__[5`#``1
M`'Q]D\$``)`BX$*Y(@(`-2(`@%(BX`*Y_?W_E_D#`*J``P"TMWY]D^$#&JKB
M`Q>J%_W_EP"_1_D9``#YH`8`$3_;(/CU5P"Y^6M$J:7__Q?A`P"JX`,:J@G]
M_Y?Y`P"J`;]'^2```/E``0"TMWY]D_/__Q<#`(#2I?__%P`"@-(5`(!2X/__
M%_EK!*E,_?^7^6M$J;K__Q<?(`/5_7N_J4'0.]6B``#P_0,`D:```/``N$?Y
M'R`#U1\@`]7]>\&H(P``\$*H1_ECH!.1(&A@^`$``/`A@!"1>?W_%_U[OJE!
MT#O5_0,`D?,+`/F@``#P`+A'^1\@`]4?(`/5,P``BR!H8/CF__^7H@``\`0`
M`/!@`D#YA(`=D?,+0/GC`P2J_7O"J`$``/!"\$?Y(0`1D04`@%)^_?\7'R`#
MU0``@-+``U_6'R`#U1\@`]7_`Q31H@``\/U[`*G]`P"10L!'^?-3`:GS`P&J
M]5L"J>%S`9'U`P"J0P!`^>-_`OD#`(#2````\`!@$9$]_O^7]`,`JN$%@%+@
M`Q6JS?S_ET`"`+3A`Q6J````\`"@$9$P__^7X`,5JJ$``/`AP$?YXG]"^2,`
M0/E"``/K`P"`TL$,`%3]>T"I\U-!J?5;0JG_`Q21P`-?UN`#$ZIE_?^7%@``
M\,'B$9$"_?^7\P,`JB#]_[3W8P.I&```\!<``/`8(P^1]P(2D?DC`/D?(`/5
M8`)`.>`&`#2T``"TX0,3JN`#%*I[_?^70`4`-.`#$ZIP_?^7`@0`T?D#`*IB
M`@*+?P("ZZ,``%0*```47_0?.'\"`NN```!400!`.3^\`'%@__]4X`,3JF+]
M_Y?Y`P"JX`,5JE_]_Y<9`!F+.0<`D3__#_$(`P!4Y`,3JN8#%:KE`Q>J`P``
M\&,@$I'SXP.1`H"`T@$`@%+@`Q.J[?S_E^`#$ZKA@P&1JOS_EZ```#7@:T"Y
M``P4$A\@0'%@`@!4P>(1D0``@-+&_/^7\P,`JH#Y_[7W8T.I^2-`^:K__Q?T
M`@"T@`)`.1^X`'&```!4\P,8JCD`@-+:__\7@`9`.0#^_S3[__\7X`,3JB$`
M@%($_?^78/W_->$#$ZH```#P`*`1D<O^_Y?@`Q.J#?W_E_=C0ZGY(T#YF/__
M%_,#&*HY`(#2Q___%_=C`ZGY(P#YE/S_E_U[OJGA!8!2_0,`D?,+`/GS`P"J
M7OS_EQ\``/%@!H":\PM`^?U[PJCY_/\7``,`M/U[OJG]`P"1\PL`^?,#`*H!
M`$`Y(0$`-!']_Y<`!`"1'P`0\4@!`%3@`Q.J\PM`^?U[PJAK_O\7\PM`^0``
M`/#]>\*H`"`/D>7\_Q<``(#2\PM`^?U[PJC``U_6````\``@#Y'>_/\7'R`#
MU?U[NJF@``#P_0,`D0#`1_GU6P*IM0``\/-3`:D4``#P\U,!D?=C`ZF40A*1
M^2,`^;6^1_D!`$#YX2\`^0$`@-+A`Q.JX`,4JHK]_Y?@`0"TH@)`^>-7@+E`
M#`.+`01`^4%X(_CA_O^T`0A`^0$$`/D`(`"1H?__M>$#$ZK@`Q2J?/W_EV#^
M_[6U``#P%```\)2"$I&UOD?YX0,3JN`#%*IT_?^7X`$`M*("0/GC5X"Y0`P#
MBP$$0/E!>"/XX?[_M`$(0/D!!`#Y`"``D:'__[7A`Q.JX`,4JF;]_Y=@_O^U
MM0``\!0``/"4HA*1M;Y'^>$#$ZK@`Q2J7OW_E^`!`+2B`D#YXU>`N4`,`XL!
M!$#Y07@C^.'^_[0!"$#Y`00`^0`@`)&A__^UX0,3JN`#%*I0_?^78/[_M;4`
M`/`4``#PE.(2D;6^1_GA`Q.JX`,4JDC]_Y?@`0"TH@)`^>-7@+E`#`.+`01`
M^4%X(_CA_O^T`0A`^0$$`/D`(`"1H?__M>$#$ZK@`Q2J.OW_EV#^_[6U``#P
M%```\)8"$Y&UOD?YX0,3JN`#%JHR_?^7X`$`M*("0/GC5X"Y0`P#BP$$0/E!
M>"/XX?[_M`$(0/D!!`#Y`"``D:'__[7A`Q.JX`,6JB3]_Y=@_O^UM@``\!4`
M`/"U0A.1UKY'^>$#$ZK@`Q6J'/W_E^`!`+3"`D#YXU>`N4`,`XL!!$#Y07@C
M^.'^_[0!"$#Y`00`^0`@`)&A__^UX0,3JN`#%:H._?^78/[_M;8``/`5``#P
MMV(1D=:^1_GA`Q.JX`,7J@;]_Y?@`0"TP@)`^>-7@+E`#`.+`01`^4%X(_CA
M_O^T`0A`^0$$`/D`(`"1H?__M>$#$ZK@`Q>J^/S_EV#^_[6W``#P%@``\-B"
M$Y'WOD?YX0,3JN`#&*KP_/^7X`$`M.("0/GC5X"Y0`P#BP$$0/E!>"/XX?[_
MM`$(0/D!!`#Y`"``D:'__[7A`Q.JX`,8JN+\_Y=@_O^UN```\!<``/#YPA.1
M&+]'^>$#$ZK@`QFJVOS_E^`!`+0"`T#YXU>`N4`,`XL!!$#Y07@C^.'^_[0!
M"$#Y`00`^0`@`)&A__^UX0,3JN`#&:K,_/^78/[_M;D``/`8``#P&`,4D3F_
M1_GA`Q.JX`,8JL3\_Y?@`0"T(@-`^>-7@+E`#`.+`01`^4%X(_CA_O^T`0A`
M^0$$`/D`(`"1H?__M>$#$ZK@`QBJMOS_EV#^_[6Y``#P&```\!BC$9$YOT?Y
MX0,3JN`#&*JN_/^7X`$`M"(#0/GC5X"Y0`P#BP$$0/E!>"/XX?[_M`$(0/D!
M!`#Y`"``D:'__[7A`Q.JX`,8JJ#\_Y=@_O^UX0,3J@```-``0!21F_S_E^$#
M`*I@``"TX,(3D9/]_Y?A`Q.J````T`"@%)&3_/^7X0,`JH```+0```#0`$`/
MD8K]_Y?A`Q.J````T```%9&*_/^7X0,`JF`"`+2@8A&1@OW_EZ```-``P$?Y
MX2]`^0(`0/DA``+K`@"`TF$"`%2``A.1`0``T/-30:DAH!61]5M"J?=C0ZGY
M(T#Y_7O&J'+]_Q?A`Q.J````T`!`%9%R_/^7X0,`JF#]_[3`@A.1:OW_E^C_
M_Q<\^_^7_7N^J:(``-````#0_0,`D4+`1_GA4P"1`(`3D4,`0/GC#P#Y`P"`
MTF'\_Y?``0"T``!`.1_``'$$&$!ZX`>?&J$``-`AP$?YX@]`^2,`0/E"``/K
M`P"`TJ$``%3]>\*HP`-?U@``@%+V__\7(/O_E_U[KJFH``#0[`:`$OT#`)$*
MP4?YX!N`/>V#`Y'A'X`]ZP^`$NA#`)'B(X`]J0``T.,G@#WD*X`]Y2^`/>8S
M@#WG-X`]X8L.J>$#`*KB`PBJXY,/J>6;$*GGCP#Y(.E'^4,!0/GC+P#Y`P"`
MTN.#!)'CCP.I`,`$D>TG`/GL+PHIY)=#J>07`:GDET2I!!4!J6?[_Y?@'X!2
MI?K_E_]#`=&C``#0_T-`T0$`@-+D0T"10@"`4OU[`*G]`P"18\!'^?-3`:GT
M`P`J90!`^84D`/D%`(#28OO_EP`$`-'S`P#K`3Q`DG,^0)(S1)/:$P`3Z\0'
M`%3U6P*I%@``T!4!@-+6`A:1]V,#J1<``-#X`P&1]\(5D34`H/(?(`/5X0,3
MJN`#%"H"`(!23?O_EQ\$`+$`!@!4X0,8JN(#%:K@`Q0J__K_EP%\0),?!``Q
M``0`5#\<`/$)!`!4(2``T0$#`:NB`P!4H```\`0``!0A!`#1/P`8ZP,#`%0#
M_$#Y(@!`^7\``NM!__]4(``8R_5;0JD``!.+]V-#J:$``-#B0T"1(<!'^4,D
M0/DD`$#Y8P`$ZP0`@-)!`@!4_7M`J?-30:G_0P&1_T-`D<`#7];@`Q:JC/__
MEW-"0-%_0D"Q`?K_5/5;0JGW8T.I``"`DNO__Q?@`Q>J@___E\___Q?U6P*I
M]V,#J9[Z_Y<?(`/5_7N_J?T#`)$B^_^7&?O_ET```+0``$#Y_7O!J,`#7];]
M>[RIH@``T/T#`)%"P$?Y\U,!J>'3`)'U6P*I%0``T+9"%I%#`$#YXQ\`^0,`
M@-+T`P"JX`,6JK;[_Y>```"T\P,`J@``0#G``0`UH```T`#`1_GA'T#Y`@!`
M^2$``NL"`(#2@00`5.$#%*J@0A:1\U-!J?5;0JG]>\2HH?S_%^`#%*H#^_^7
M]0,`JN`#$ZH`^_^7H`(`BP`(`)$!^_^7Y`,3JN(#%*KS`P"J`P``T`$``-!C
MX!&1(2`2D>GZ_Y>@``#0`,!'^>$?0/D"`$#Y(0`"ZP(`@-+A``!4X0,3JN`#
M%JKS4T&I]5M"J?U[Q*B$_/\75_K_EQ\@`]4?(`/5_7NGJ:(``/"A``#00N`'
MD?T#`)$AP$?Y\U,!J>,C`I'U6P*I\\,!D?5C`9'W8P.I]P,`JO13`9%&'$&I
M(`!`^>#'`/D``(#22JQ"J>:?!:D8``#02*1#J>$#%*I$E$2I`&,1D4RT1:EH
M)`&I3A!`^:X*`/E&-$#Y9@H`^604`JGL-P>IZJ\(J6+[_Y>```"T`0!`.?8#
M`*K!'@`U^6L$J9#__Y?Y`P"JH!L`M.`#&:JX^O^7`/A_TP`$`)&Y^O^7(@-`
M.?8#`*I"`0`T&@``T/,#%JI:HQ:1X`,3JN$#&JJ@^O^7<PH`D2(?0#AB__\U
M`0``T/E#`I$@0`^1`P``%""'0/A`"`"TX0,4JD'[_Y?S`P"J8/__M`$`0#DA
M__\TV/O_E^#^_S3@`Q.J>?K_E_,#`*K`!@"TX`,3JI7Z_Y?U`P"JX`,6JI+Z
M_Y>Y`@`+.1,0$1H``-`Y?T"3X`,9JI#Z_Y?E`Q:JX@,3JD,#$I$$``#0A,`6
MD?4#`*H!``#0(>`6D7?Z_Y?@`Q6J`3B`4D3Z_Y<?!``Q`!(`5.'C`I'@`Q6J
MS_G_E_,#`"H@!0`TH```T`$``-#B`D#YXP,5J@#H1_DA(!B1%`"`T@#`!)'4
M^?^7H```T`#`1_GAQT#Y`@!`^2$``NL"`(#2(1X`5.`#%*KS4T&I]5M"J?=C
M0ZGY:T2I_7O9J,`#7]89`(!2L]IY^+,!`+1@`D`Y8`$`-#D'`!'@`Q.JEOO_
MER#__S3@`Q.J-_K_E_,#`*K@]_^UL]IY^+/^_[43`(#2N___%^##0+D`#!02
M'Q!`<:'Z_U3VUT"Y0_K_E]\"`&LA^O]4X,-`N0`@`!(?``=QH?G_5.`#&:I'
M^O^7X0,4JO0#`*H```#0`(`9D=[Z_Y?A`P"J0!``M.`"0/F._/^7^0,`JL`/
M`+3@`QFJ-OK_E_8#`*H?#`!Q[0$`5`!\0),!``#0`!``T2&@&9$@`P"+-?K_
MEP`!`#7?$@!Q@!0`5-86`%$@RW8X'[P`<0`4`%0?(`/56?[_E\```#7@`QFJ
M`0"`4@SZ_Y?V`P`JP`P`-0$``-````#0(:`5D0"`$Y&U^_^76@,2D1<``-#W
M@AV1Z?G_EQD``-#D`P`JY0,7JN,#&JKB`Q6JX`,4J@$``-`YHQJ1(6`:D?_Y
M_Y</```4Q?G_EP``0+D?1`!Q`0(`5-GY_Y=S!@`1Y`,`*N4#$RKF`Q>JXP,:
MJN(#%:KA`QFJX`,4JO#Y_Y?@`Q2J`3B`4KWY_Y<?!``QP/W_5.`#%:HY^?^7
MX0,4J@!C$9&.^_^7X`,4JLS^_Y>`__\7JOG_EP``0+D?1`!QP.W_5/,"0/FE
M^?^7I0``T`0`0+GB`Q.JXP,5JJ7H1_D4`(#2`0``T"$@%Y&@P`210_G_EV__
M_Q<6`(!28-IV^(`!`+3A`Q2JU@8`$7CZ_Y=@__^T`0!`.2'__S2T^?^7^0,`
MJB#C_[5@VG;XP/[_M:`!@-+2^?^7&0``T/8#`*HY@Q:18@J`4AC__Q>D_O^7
MH```T`#`1_GAQT#Y`@!`^2$``NL"`(#200D`5.`#%JKS4T&I]5M"J?=C0ZG]
M>]FHF_G_%_D"0/F"__\7,/[_E_<#`*H@\_^W`2@`T0(`@%+@`Q8J^4,%D9GY
M_Y?A`QFJX`,6*L(`@-)-^?^7'Q@`\8```%0```#0`.`9D?G]_Y?@4T&Y`-!0
M40`,!7'A\/]4(0M`>:`(B5(_``!K8?#_5.'*`-$"`(!2X`,6*O-C!9&#^?^7
MX`,6*N$#$ZH"!8#2-_G_EQ^@`/&```!4````T`#@&9'C_?^7Y`,3JD,#$I'B
M`Q6JX`,4J@4``-`!``#0I8`=D2$@&I'_`P8Y?/G_EY'__Q?V!D#Y-NS_M.`#
M%JJ#^?^7'PP`\:GK_U0`$`#1`0``T,`"`(LAP!F1A/G_EQ\``'$Y$Y::5?__
M%^CX_Y?Y:P2IYOC_EQ\@`]7]>[NIH@``T`0``-#]`P"1\U,!J5/0.]6@``#0
M`+A'^1\@`]4?(`/5]5L"J?43`9'A`Q6J]V,#J6-J8/B``!.10L!'^63`0CE%
M`$#YY2<`^04`@-)DQ`(Y_OG_EX```+0!`$`Y/\0`<8`+`%1=_/^7H```T`"X
M1_D?(`/5'R`#U78"`(M@:F#X`0``T&,`@%(A`!N1(@"`4N;X_Y?T`P"J8`$`
MM`$(0/D"``#0P`)`^4(@&Y$A`$#YOOC_EX`*0/D!`$#Y(`Q`N2`+L#>@``#0
M`+A'^1\@`]4?(`/5=P(`BV!J8/@!``#08P"`4B%`&Y$B`(!2M@``\,[X_Y?T
M`P"J8`$`M.$#%:H```#0`*`1D=#Y_Y?B`P"J0`<`M($*0/G@`D#Y(0!`^:+X
M_Y>@``#0`+A'^1\@`]4?(`/5=`(`BW-J8/@B`(!2`0``T"%@&Y'@`Q.JQ_C_
MEP(``-#A`P"J0L`;D>`#$ZJ2^/^7TX(+D8("0/GA`Q6J````T`!@$9%?Q`(Y
ML_G_EV`*`/G``@"T`3B`4M/X_Y<!!``1/P0`<02(47K!!0!4H```T`#`1_GA
M)T#Y`@!`^2$``NL"`(#2P00`5/-30:GU6T*I]V-#J?U[Q:C``U_6``1`.:#T
M_S2C__\78`9`^1;^_Y=@"@#Y`/W_M0```-``X!N1H?C_ET3[_Y?X`P"J@`$`
MM($*0/GB`P"JX`)`^2$`0/EE^/^7X`,8JB_X_Y?!__\7P`)`^:CX_Y>F__\7
MP((+D8$*0/D"!$#Y(0!`^4(`0/G@`D#Y6/C_E[;__Q=.^/^7UH(+D>,#`"H`
M``#0`(`<D<&*0*DA`$#Y@_C_EQ\@`]4?(`/5_7N^J?T#`)'S"P#Y\P,`J@7]
M_Y=_`@#Q!!A`>H$``%3S"T#Y_7O"J,`#7]9@`D`Y@/__-.`#$ZJN^_^7HOO_
MEP$``-""`(#2(<`6D?GW_Y>`_O\UX`,3JO,+0/G]>\*HF?G_%P``````````
M`````/^#"M&A``#0XF,`D03@`)$&X`&1XP,"JOU[`*G]`P"1(<!'^24`0/GE
M3P'Y!0"`TH&$0/@E?$"2(?Q@TV$4`*EC0`"1WP`$ZT'__U1%``B100!`^4(@
M`)%#!$#Y1!Q`^6,``<I!,$#Y(0`$RB$``\HC_%_3800!JB%\0))!/`#Y7P`%
MZV'^_U0'*$"I)#./T@@D0:E$4*ORY9M!J2-TG=+B_%O3[/Q"TT(4!ZJ,>0>J
M(0$(RB/;K?(A``J*(0`)RJ4`!(LB0"*+0?U"TR%X"JI"``6+"Q!`^2U\0)*E
M`0C*Q@`$BT(`"PNE``>*I0`(RB$`#,I._%O33_Q"TZ4`"8O.%0*JI0`&BR$`
M`HKB>0*JI0`."^:_0JDA``W*KOQ;TR$`"(O.%06JL/Q"T\8`!(OO`02+(0`&
MBX8!`LHA``X+C'U`DL8`!8K&``S*+OQ;T\8`#8O.%0&J#7H%JL8`#XO&``X+
M10`-RNX?0/E"?$"2I0`!BC#\0M.E``+*S@$$BZ4`#(O/_%O3#'H!JN\5!JJA
M`0S*I0`.B^XC0/FE``\+K7U`DB$`!HHA``W*S_Q"T^9Y!JHA``*+P@$$BZ[\
M6],A``*+K_Q"TX(!!LK.%06J+@`."T(`!8KE>06JC'U`DN&_1*E"``S*0@`-
MB\W]6].M%0ZJT'Q`DB$`!(M"``&+P0`%RDT`#0LA``Z*(0`0RL;]0M/B`02+
MSG@.JB$`#(NO_5O3IGQ`DB$``HOO%0VJH@`.RJ7]0M,O``\+I7@-JD(`#8KA
MMT6I0@`&RD(`$(OL_5O3C!4/JO#]0M,A``2+0@`!B\$!!<HA``^*SGU`DB$`
M#LI,``P+#WH/JJ(!!(LA``:+C?U;TR$``HNB``_*0@`,BJ5\0))"``7*K14,
MJD(`#HLM``T+\+M&J8;]0M/&>`RJH?U;TRP4#:KA`0;*(0`-BN]]0)(A``_*
M$`($BR$`!8M"`!"+L/U"TTP`#`L->@VJP@$$BR$``HO"``W*0@`,BL9\0))"
M``;*COU;TT(`#XO.%0RJ\+]'J2X`#@N%_4+3I7@,JL']6],L%`ZJH0$%RB$`
M#HJM?4"2$`($BR$`#<I"`!"+T/U"TTP`#`L.>@ZJX@$$BR$`!HLA``*+H@`.
MRD(`#(JE?$"20@`%RH_]6]-"``V+[Q4,JO"W2*DO``\+AOU"T\9X#*KA_5O3
M+!0/JL$!!LHA``^*SGU`DA`"!(LA``[*0@`0B_#]0M-,``P+#WH/JJ(!!(LA
M``6+(0`"B\(`#\I"``R*QGQ`DD(`!LJ-_5O30@`.BZT5#*KPNTFI+0`-"X7]
M0M.E>`RJH?U;TRP4#:KA`07*(0`-BN]]0)(0`@2+(0`/RD(`$(NP_4+33``,
M"PUZ#:K"`02+(0`&BR$``HNB``W*COU;TZ5\0))"``R*SA4,JD(`!<HN``X+
M0@`/BX;]0M/AOTJIS'@,JL;]6]/&%`ZJ(0`$B^0!!(M"``&+H0$,RB$`#HJM
M?4"2(0`-RL_]0M/2``(+[GD.JB$`!8M"0":+[U]`^2$`!(N$`0[*1?Y;TX0`
M`DJE%!*JX@$#BX0`#8M-_D+3L``!"[)Y$JJ$``*+XF-`^2%`)8O%`1+*!OY;
MTZ4``4K&%!"J00`#BPW^0M.B0"R+SP`$"ZUY$*I"``&+X6=`^81`)HM&`@W*
MY?U;T\0`!$JE%`^J)@`#BX%`+HNQ``(+(0`&B^S]0M-"0"6+YFM`^8]Y#ZJE
M`0_*)/Y;TX04$:JE``)*P@`#BR;^0M.0``$+T7@1JN9O0/DA0"2+Y`$1RJQ`
M,HN$``%*!?Y;TZ44$*J"`0*+P0`#BPS^0M.&0"V+K@`""Y!Y$*K!``&+YG-`
M^4)`)8LE`A#*Q/U;TZ(``DJ$%`ZJQ@`#BT)`+XO,_4+3C0`!"XYY#JI"``:+
MYG=`^2%`)(L$`@[*I?U;TX$``4JE%`VJQ@`#BR%`,8NO``(+(0`&BZS]0M-"
M0"6+YGM`^8UY#:K%`0W*Y/U;TX04#ZJE``)*P@`#B^;]0M.1``$+SW@/JN9_
M0/DA0"2+I`$/RJQ`,(N$``%*)?Y;TZ44$:J"`0*+P0`#BRS^0M.&0"Z+L``"
M"XQY$:K!``&+YH-`^4)`)8OE`0S*!/Y;TZ(``DJ$%!"JQ@`#BT)`+8L-_D+3
MC@`!"T(`!HNF>1"J\(=`^2%`)(N$`0;*Q?U;TX$``4JE%`ZJ$`(#BR%`+XNM
M``(+(0`0B\_]0M-"0"6+\(M`^>YY#JK%``[*I/U;TX04#:JE``)*`@(#B[#]
M0M./``$+#7H-JO"/0/DA0"2+Q`$-RJQ`+(N$``%*Y?U;TZ44#ZJ&0":+`0(#
MBX(!`HOP_4+3K``""\$``8L&>@^J\)-`^4)`)8NE`0;*A/U;TZ(``DJ$%`RJ
M$`(#BT)`+HN._4+3CP`!"\QY#*I"`!"+\)=`^2%`)(O$``S*Y?U;TX$``4JE
M%`^J$`(#BR%`+8OM_4+3K@`""Z]Y#ZHA`!"+\)M`^4)`)8N%`0_*Q/U;TZ(`
M`DJ$%`ZJ0D`FBP4"`XO0_4+3C0`!"T4`!8L">@ZJ\)]`^2%`)(OD`0+*IOU;
MTX0``4K&%`VJ`0(#BX1`+(NP_4+3A``!BP%Z#:K.``4+I4`FBT8``<K%``5*
MS/U;TZ5`+XO/_4+3C!4.JNYY#JKP/U2IC0$$"X1`+(NF_5O3QA0-JA`"`XNE
M`!"+,``.RLP`!0L0`@1*`D(BB^0!`XNP_4+3C_U;TPUZ#:KO%0RJ0@`$BZ5`
M)HODJT#Y0@`/"\8!#<K%``5*@P`#BX;]0M.A0"&+1/Q;T\QX#*J$%`*J(0`#
MBR$`!`NC`0RJY*]`^84!#0IC``**3_Q"TV,`!:J&FY?29N.Q\N)Y`JJ$``:+
M8T`NBR7\6],N_$+3I10!JF,`!(N$`0*J8P`%"X0``8K!>0&JY3M6J4\`#`J$
M``^J;_Q"TX1`+8MM_%O3K14#JJ4`!HO.`0:+A``%BT4``:J$``T+I0`#BBT`
M`@JE``VJC?Q;TZ5`+(OL>0.JI0`.BZ,5!*HM``RJHP`#"^Z[0/FE`02*C0$!
M"H_\0M.E``VJS@$&BZ5`(HMM_%O3XGD$JJT5`ZJ$`0*JI0`.B^Z_0/FE``T+
MA``#BDT`#`J$``VJ;_Q"T\X!!HN$0"&+K?Q;T^%Y`ZJM%06JA``.BX0`#0M#
M``&J[<-`^2X``@IC``6*K_Q"TV,`#JKE>06JK0$&BV-`+(N._%O3+``%JLX5
M!*IC``V+[<=`^6,`#@N,`02*K@`!"HP!#JJ._$+3@D$BBZP!!HMM_%O3Q'D$
MJJT5`ZI"``R+0@`-"ZP`!*KMRT#YC@`%"HP!`XIO_$+3C`$.JN-Y`ZJM`0:+
M@4$ABT[\6].,``.JSA4"JB$`#8LA``X+C`$"BFX`!`J,`0ZJ3OQ"T\)Y`JJ%
M026+[;M9J2_\0M.L`0:++?Q;TZT5`:JE``R+;``"JJ4`#0N,`0&*30`#"HP!
M#:KA>0&JS@$&BX1!)(NM_%O33``!JJT5!:J$``Z+[M=`^80`#0N,`06*+0`"
M"HP!#:JO_$+3S@$&BX-!(XN-_%O3['D%JJT5!*HE``RJ8P`.BZ4`!(KNVT#Y
M8P`-"XT!`0J/_$+3I0`-JN1Y!*K.`0:+HD`BBVW\6].%`02JK14#JD(`#HON
MWT#Y0@`-"Z4``XJ-``P*;_Q"TZ4`#:K.`0:+H4`ABTW\6]/E>0.JK14"JH,`
M!:HA``Z+3OQ"TR$`#0MC``**PGD"JJ\`!`KM.URI8P`/JFQ`+(LC_%O38Q0!
MJB_\0M.M`0:+C`$-BZT``JJ#`0,+K`$!BDT`!0KA>0&JC`$-JFW\6].$022+
MS`$&BX0`#(M,``&J+@`""HP!`XJ,`0ZJK14#JH5!)8N$``T+[S-=J6W\0M.M
M>0.JCOQ;T\X5!*KO`0:+HP`/BR4`#:IC``X+I0`$BJX!`0JE``ZJCOQ"T\1Y
M!*JB0"*+A0$&BVS\6].,%0.J0@`%BV[\0M.E`02J0@`,"Z4``XK#>0.JCP`-
M"NP[7JFE``^JI4`ABT'\6],A%`*J3_Q"TXP!!HNE``R+C``#JJ4``0N,`0**
M80`$"N)Y`JJ,`0&J4``#"H%!+8O-`0:+K/Q;TR$`#8N,%06J;0`"JN_[0/F.
M`0$+K0$%BBQ`+(NM`1"JT/U"TZ1!)(OM`0:+K_Q"T\;]6]/&%`ZJA``-B^UY
M!:K2``0+[_]`^4$`#<HA``Q*Q3J8TD5,N?(A0".+[`$%BT/^6],.>@ZJ8Q02
MJB$`#(N$0":+[`-!^7$``0NF`0[*(4`CB\0`!$J&`06+@D`BBTS^0M,D_EO3
MDGD2JH04$:I"``:+[`=!^9```@O#`1+*)OY"TV,``4J!`06+;$`MBP/^6]/1
M>!&JC`$!BV,4$*I"0"2+Y@M!^6\`#`M$`A'*@T$CBX0``DKA_5O3P@`%BX9`
M+HL$_D+3Q@`"BY!X$*HA%`^JY`]!^2X`!@LB`A#*[/U"TT(``TJ#``6+1$`R
MB\+]6]./>0^JA``#BT(4#JK!0"&+XQ-!^4T`!`L&`@_*S/U"T\$``4IF``6+
M(T`QBZ']6]..>0ZJ8P`&BR$4#:KL`0[*YA=!^3(``PN"0"*+84`ABX(!`DK$
M``6+3$`PBZ;]0M-"_EO3S7@-JH0!!(M"%!*JYAM!^5$`!`O#`0W*3/Y"TV$`
M`4K#``6+)D`OBR'^6]/#``.+C'D2JB$4$:J"0"*+YA]!^3```PND`0S*84`A
MBX(``DK&``6+1$`NBR[^0M,"_EO3A``&BT(4$*K&>1&J[B-!^4\`!`N#`0;*
M@D`BBV$``4K.`06+(T`MBPW^0M/A_5O3L'D0JF,`#HLA%`^J[B=!^2T``PO$
M`!#*84`ABX(``DK$`06+3$`LB^[]0M.B_5O3SWD/JH0!!(M"%`VJ["M!^4X`
M!`L#`@_*@D`BBV$``4J#`06+)D`FBZS]0M/!_5O3PP`#BR$4#JJ&>0VJ["]!
M^2T``POD`0;*84`ABX(``DJ,`06+1$`PB]#]0M.B_5O3#GH.JH0`#(M"%`VJ
M[#-!^5$`!`O#``[*@D`BBV$``4J,`06+(T`OBZ_]0M,A_EO3[7D-JF,`#(LA
M%!&J[#=!^3```PO$`0W*+_Y"TX(``DJ$`06+3$`FBP+^6].&`02+0A00JNQY
M$:IA0"&+Y#M!^4\`!@NC`0S*PD`BBV$``4J#``6+)$`NBP[^0M/A_5O3A``#
MBR$4#ZK#>1"J\#]!^2X`!`N&`0/*@4`AB\(``DH0`@6+1D`MB^W]0M/"_5O3
MKWD/JD(4#JK&`!"+[4-!^5``!@MD``_*PD`BBX0``4JA`06+C$`LB\W]0M,$
M_EO3C`$!BZYY#JJ$%!"JX4=!^9$`#`OF`0[*YT`OB\8``DHB``6+PT`CBR'^
M6]-C``*+(101JF,``0L!_D+3[4M!^2%X$*J&022+P@$!RF3\6].L``"P1@`&
M2H04`ZJE`06+(OY"TX0`!HNE``>+AL%'^4)X$:HA``D+RP$+"V,`"@N$``4+
M2``("P0,`*D(!`&I"Q``^>!/0?G!`$#Y```!ZP$`@-*!``!4_7M`J?^#"I'`
M`U_6$O3_EQ\@`]7]>[^I`!"`TOT#`)&A]/^7'Q@`^29@A-(E<972Q)^;TL..
MBM("/IS2INBL\J7YO?)$%[/R0P:B\D)ZN/(&%`"I!`P!J0)\`JD?>`"Y_7O!
MJ,`#7];]>[RI0GQ`D_T#`)'S4P&I%.``D?,#`:KU6P*I]0,`JO8#`JKW8P.I
M!8Q"J0!X0+FD#`(+OP`$ZV.4@YIB=$*+I(H"J>`"`#07"(!2]P(`2]\"%VN`
MPB"+U]*7&OA^0)/B`QBJVO/_EZ!Z0+G@`@`+H'H`N1\``7'```!4\U-!J?5;
M0JGW8T.I_7O$J,`#7];6`A=+<P(8B^`#%:JL^_^7W_X`<<T#`%36`@%1V'X&
M4Q<'`!'W:GK3=P(7BV`&0*F`!@"I<P(!D6(.?:F"#@&IX`,5JF(.?JF"#@*I
M8@Y_J8(.`ZF:^_^7?P(7ZX'^_U36&AA+X0,7JL)^0)/@`Q2JL_/_E[9Z`+GS
M4T&I]5M"J?=C0ZG]>\2HP`-?UO<#$ZKU__\7'R`#U1\@`]7]>[RI_0,`D?-3
M`:GS`P&J]`,`JO5;`JGA#X`2]QL`^7?B`)%UVD*IH"(#4P($`!'ARB`XX,(B
MBU_@`'$M"P!4`0B`4B(``DL!`(!20GQ`D[SS_Y?@`Q.J<OO_EW_^`ZE__@2I
M?_X%J7\V`/G%_E#3Q/Y(TZ/^6-.B_E#3H?Y(T\#^6--@P@$YX`,3JF7&`3ED
MR@$Y=LX!.6/2`3EBU@$Y8=H!.77>`3E>^_^780)`^>`#$ZHA_%C3@0(`.6$"
M0/DA_%#3@08`.6$"0/DA_$C3@0H`.6$"0/F!#@`Y809`^2'\6-.!$@`Y809`
M^2'\4-.!%@`Y809`^2'\2-.!&@`Y809`^8$>`#EA"D#Y(?Q8TX$B`#EA"D#Y
M(?Q0TX$F`#EA"D#Y(?Q(TX$J`#EA"D#Y@2X`.6$.0/DA_%C3@3(`.6$.0/DA
M_%#3@38`.6$.0/DA_$C3@3H`.6$.0/F!/@`Y81)`^2'\6-.!0@`Y81)`^2'\
M4-.!1@`Y81)`^2'\2-.!2@`Y81)`^8%.`#GS4T&I]5M"J?<;0/G]>\2H'?/_
M%P$'@%(B``)+`0"`4D)\0)-D\_^7KO__%P``````````H```L`!`"I$!($$Y
M00,`-:$``)`B`(!2`B`!.2"D1_F@`0"T_7N_J:```+#]`P"1`/A`^>OR_Y>@
M``"0`-1'^6`!`+3]>\&H(```D`!@&I&4\_\7H```D`#41_G```"T(```D`!@
M&I&.\_\7_7O!J,`#7];``U_6``````````"@``"P`4`*D0)`2CE"`0`UH@``
MD",`@%(#0`HY0+1'^:```+0A(`"1(```D`!@&I$8\_\7P`-?U@``````````
M_7NZJ:,``)#]`P"18\!'^?-3`:GU6P*IXH<#J>+C`)'A`P&1X$\`N>`S`9%D
M`$#YY"\`^00`@-(V\_^7H```D*(``)``X$?Y0LQ'^>$#`JI`\_^7H```D`"P
M1_D``$`Y@`D`-$'0.]6S``"PH```D`"X1_D?(`/5'R`#U2-H8/CB3T"Y80A2
M.4`0`!%4#``1(00`,F$($CD`?'V3#?/_E^,#`*JA``"08H(+D>8C0/D```"0
M),A'^0#`'9'E3T"Y`0``D,<`0/DAH!V1A`!`^6<$`*E#!`#Y9``!J;\$`'$M
M"0!4(0"`TB`,`!'$>&'X(00`D638(/B_``%K;/__5*4,`!%U@@N1?]@E^.'_
M_[#B`Q0J(<`CD00`@-*@#D#Y(//_ER`%`#1U@@N1]E,!D:`.0/F_\O^7]`,`
M*N$#%JH```"0`$`3D>[S_Y=`!`"T<X(+D6`.0/FB\O^7`?/_EZ```)``P$?Y
MX2]`^0(`0/DA``+K`@"`TN$$`%3@`Q0J\U-!J?5;0JG]>\:HP`-?UKCR_Y>S
M``"P8H(+D4`,`/E@`P"T?_+_ET'0.]6@``"0`+A'^1\@`]4?(`/5(&A@^!\8
M)#FJ__\7H`Y`^1'S_Y?6__\7H`I`^6```+15^O^7W/__%^$#%JH```"0`&`1
MD<3S_Y>@"@#YP/K_M$WZ_Y?4__\7A0"`4K___Q<@`(!2./+_EXOR_Y=P<U]S
M=')I;F=S(&UI<W-I;F<*```````O=&UP+P```"X`````````4$%27U1-4$1)
M4@```````%1-4$1)4@``5$5-4$1)4@!414U0`````%1-4```````55-%4@``
M``!54T523D%-10``````````)7,O)7,```!$>6YA3&]A9&5R.CIB;V]T7T1Y
M;F%,;V%D97(``````$EN=&5R;F%L<SHZ4$%2.CI"3T]4`````%!!4E]414U0
M``````````!005)?4%)/1TY!344`````.@`````````O`````````"5S)7,E
M<P``4$523#5,24(``````````%!%4DQ,24(`4$523#5/4%0``````````%!%
M4DQ)3P``4$%27TE.251)04Q)6D5$`%!!4E]34$%73D5$``````!005)?0TQ%
M04X`````````4$%27T1%0E5'`````````%!!4E]#04-(10````````!005)?
M1TQ/0D%,7T1%0E5'``````````!005)?1TQ/0D%,7U1-4$1)4@````````!0
M05)?1TQ/0D%,7U1%35``4$%27T=,3T)!3%]#3$5!3@``````````,0``````
M``!L<V5E:R!F86EL960`````<F5A9"!F86EL960``````$Q$7TQ)0E)!4EE?
M4$%42`!365-414T``"4P,G@`````<&%R+0`````E<R5S)7,E<P``````````
M)7,Z(&-R96%T:6]N(&]F('!R:79A=&4@<W5B9&ER96-T;W)Y("5S(&9A:6QE
M9"`H97)R;F\])6DI"@```````"5S.B!P<FEV871E('-U8F1I<F5C=&]R>2`E
M<R!I<R!U;G-A9F4@*'!L96%S92!R96UO=F4@:70@86YD(')E=')Y('EO=7(@
M;W!E<F%T:6]N*0H```````!0051(`````'!A<FP`````+G!A<@````!S:&]R
M="!R96%D````````)7,E<V-A8VAE+25S)7,``"5S)7-T96UP+25U)7,````E
M<R5S=&5M<"TE=2TE=25S```````````8`````````'!E<FP`````,```````
M``!005(Z.E!A8VME<CHZ5D524TE/3@`````Q+C`V,P```%5N86)L92!T;R!C
M<F5A=&4@8V%C:&4@9&ER96-T;W)Y```````````E<SH@8W)E871I;VX@;V8@
M<')I=F%T92!C86-H92!S=6)D:7)E8W1O<GD@)7,@9F%I;&5D("AE<G)N;STE
M:2D*```````````M90```````"TM````````979A;"`G97AE8R`O=7-R+V)I
M;B]P97)L("`M4R`D,"`D>S$K(B1`(GTG"FEF(#`[(",@;F]T(')U;FYI;F<@
M=6YD97(@<V]M92!S:&5L;`IP86-K86=E(%]?<&%R7W!L.PIM>2`H)%!!4E]-
M04=)0RP@)$9)3$5?;V9F<V5T7W-I>F4L("1C86-H95]N86UE7W-I>F4I.PI"
M14=)3B!["B1005)?34%'24,@/2`B7&Y005(N<&U<;B(["B1&24Q%7V]F9G-E
M=%]S:7IE(#T@-#L@("`C('!A8VLH(DXB*0HD8V%C:&5?;F%M95]S:7IE(#T@
M-#`["B102UI)4%]-04=)0R`](")02UPP,#-<,#`T(CL*?0IS=6(@9FEN9%]P
M87)?;6%G:6,*>PIM>2`H)&9H*2`]($!?.PIM>2`D8VAU;FM?<VEZ92`](#8T
M("H@,3`R-#L*;7D@)&)U9CL*;7D@)'-I>F4@/2`M<R`D9F@["FUY("1P;W,@
M/2`H)'-I>F4M,2D@+2`H)'-I>F4M,2D@)2`D8VAU;FM?<VEZ93L@("`@("`C
M($Y/5$4Z("1P;W,@:7,@82!M=6QT:7!L92!O9B`D8VAU;FM?<VEZ90IW:&EL
M92`H)'!O<R`^/2`P*2!["G-E96L@)&9H+"`D<&]S+"`P.PIR96%D("1F:"P@
M)&)U9BP@)&-H=6YK7W-I>F4@*R!L96YG=&@H)%!!4E]-04=)0RD["FEF("@H
M;7D@)&D@/2!R:6YD97@H)&)U9BP@)%!!4E]-04=)0RDI(#X](#`I('L*<F5T
M=7)N("1P;W,@*R`D:3L*?0HD<&]S("T]("1C:'5N:U]S:7IE.PI]"G)E='5R
M;B`M,3L*?0IM>2`H)'!A<E]T96UP+"`D<')O9VYA;64L($!T;7!F:6QE+"`E
M36]D=6QE0V%C:&4I.PI%3D0@>R!I9B`H)$5.5GM005)?0TQ%04Y]*2!["G)E
M<75I<F4@1FEL93HZ5&5M<#L*<F5Q=6ER92!&:6QE.CI"87-E;F%M93L*<F5Q
M=6ER92!&:6QE.CI3<&5C.PIM>2`D=&]P9&ER(#T@1FEL93HZ0F%S96YA;64Z
M.F1I<FYA;64H)'!A<E]T96UP*3L*;W5T<RAQ<5M296UO=FEN9R!F:6QE<R!I
M;B`B)'!A<E]T96UP(ETI.PI&:6QE.CI&:6YD.CIF:6YD9&5P=&@H<W5B('L@
M*"`M9"`I(#\@<FUD:7(@.B!U;FQI;FL@?2P@)'!A<E]T96UP*3L*<FUD:7(@
M)'!A<E]T96UP.PII9B`H+60@)'!A<E]T96UP("8F("1>3R!N92`G35-7:6XS
M,B<I('L*;7D@)'1M<"`](&YE=R!&:6QE.CI496UP*`I414U03$%412`]/B`G
M=&UP6%A86%@G+`I$25(@/3X@1FEL93HZ0F%S96YA;64Z.F1I<FYA;64H)'1O
M<&1I<BDL"E-51D9)6"`]/B`G+F-M9"<L"E5.3$E.2R`]/B`P+`HI.PIM>2`D
M9FEL96YA;64@/2`D=&UP+3YF:6QE;F%M93L*<')I;G0@)'1M<"`\/"(N+BXB
M.PIX/3$[('=H:6QE(%L@7"1X("UL="`Q,"!=.R!D;PIR;2`M<F8@)R1P87)?
M=&5M<"<*:68@6R!<(2`M9"`G)'!A<E]T96UP)R!=.R!T:&5N"F)R96%K"F9I
M"G-L965P(#$*>#U@97AP<B!<)'@@*R`Q8`ID;VYE"G)M("<D9FEL96YA;64G
M"BXN+@IC;&]S92`D=&UP.PIC:&UO9"`P-S`P+"`D9FEL96YA;64["FUY("1C
M;60@/2`B)&9I;&5N86UE(#XO9&5V+VYU;&P@,CXF,2`F(CL*<WES=&5M*"1C
M;60I.PIO=71S*'%Q6U-P87=N960@8F%C:V=R;W5N9"!P<F]C97-S('1O('!E
M<F9O<FT@8VQE86YU<#H@)&9I;&5N86UE72D["GT*?2!]"D)%1TE.('L*26YT
M97)N86QS.CI005(Z.D)/3U0H*2!I9B!D969I;F5D("9);G1E<FYA;',Z.E!!
M4CHZ0D]/5#L*979A;"!["E]P87)?:6YI=%]E;G8H*3L*;7D@)'%U:65T(#T@
M(21%3E9[4$%27T1%0E5'?3L*;7D@)7-Y<R`]("@*<&%T:%]S97`@("`@/3X@
M*"1>3R`]?B`O7DU35VEN+R`_("<[)R`Z("<Z)RDL"E]E>&4@("`@("`@(#T^
M("@D7D\@/7X@+UXH/SI-4U=I;GQ/4S)\8WEG=VEN*2\@/R`G+F5X92<@.B`G
M)RDL"E]D96QI;2`@("`@(#T^("@D7D\@/7X@+UY-4U=I;GQ/4S(O(#\@)UQ<
M)R`Z("<O)RDL"BD["E]S971?<')O9VYA;64H*3L*;W5T<RAQ<5M<)'!R;V=N
M86UE(#T@(B1P<F]G;F%M92)=*3L*7W-E=%]P87)?=&5M<"@I.PIO=71S*'%Q
M6UPD14Y6>U!!4E]414U0?2`]("(D14Y6>U!!4E]414U0?2)=*3L*;7D@*"1S
M=&%R=%]P;W,L("1S=&%R=%]O9E]&24Q%7W-E8W1I;VXI.PI-04=)0SH@>PIL
M;V-A;"`D4TE'>U]?5T%23E]??2`]('-U8B![?3L*=6YL97-S("AO<&5N(%]&
M2"P@)SPZ<F%W)RP@)'!R;V=N86UE*2!["F]U=',H<7%;0V%N)W0@<F5A9"!F
M<F]M(&9I;&4@(B1P<F]G;F%M92)=*3L@(",@9&]N)W0@=7-E("0A(&AE<F4@
M87,@:70@<F5Q=6ER97,@17)R;F\N<&T*;&%S="!-04=)0SL*?0IM>2`D;6%G
M:6-?<&]S(#T@9FEN9%]P87)?;6%G:6,H*E]&2"D["FEF("@D;6%G:6-?<&]S
M(#P@,"D@>PIO=71S*'%Q6T-A;B=T(&9I;F0@;6%G:6,@<W1R:6YG("(D4$%2
M7TU!1TE#(B!I;B!F:6QE("(D<')O9VYA;64B72D["FQA<W0@34%'24,["GT*
M;W5T<R@B1F]U;F0@4$%2(&UA9VEC(&%T('!O<VET:6]N("1M86=I8U]P;W,B
M*3L*<V5E:R!?1D@L("1M86=I8U]P;W,@+2`D1DE,15]O9F9S971?<VEZ92`M
M(&QE;F=T:"@B7#!#04-(12(I+"`P.PIR96%D(%]&2"P@)&)U9BP@;&5N9W1H
M*")<,$-!0TA%(BD["FEF("@D8G5F(&YE(")<,$-!0TA%(B`F)B`D8G5F(&YE
M(")<,$-,14%.(BD@>PIO=71S*").;R!C86-H92!M87)K97(@9F]U;F0B*3L*
M;&%S="!-04=)0SL*?0IO=71S*'%Q6T-A8VAE(&UA<FME<B`B)&)U9B(@9F]U
M;F1=*3L*<V5E:R!?1D@L("1M86=I8U]P;W,@+2`D1DE,15]O9F9S971?<VEZ
M92P@,#L*<F5A9"!?1D@L("1B=68L("1&24Q%7V]F9G-E=%]S:7IE.PIM>2`D
M;V9F<V5T(#T@=6YP86-K*").(BP@)&)U9BD["F]U=',H(D]F9G-E="!F<F]M
M('-T87)T(&]F($9)3$5S(&ES("1O9F9S970B*3L*<V5E:R!?1D@L("1M86=I
M8U]P;W,@+2`D1DE,15]O9F9S971?<VEZ92`M("1O9F9S970L(#`["B1S=&%R
M=%]O9E]&24Q%7W-E8W1I;VX@/2!T96QL(%]&2#L*;7D@)7)E<75I<F5?;&ES
M=#L*<F5A9"!?1D@L("1B=68L(#0[("`@("`@("`@("`@("`@("`@("`@("`@
M("`@(R!R96%D('1H92!F:7)S="`B1DE,12(*=VAI;&4@*"1B=68@97$@(D9)
M3$4B*2!["G)E860@7T9(+"`D8G5F+"`T.PIR96%D(%]&2"P@)&)U9BP@=6YP
M86-K*").(BP@)&)U9BD["FUY("1F=6QL;F%M92`]("1B=68["F]U=',H<7%;
M56YP86-K:6YG($9)3$4@(B1F=6QL;F%M92(N+BY=*3L*;7D@)&-R8R`]("@@
M)&9U;&QN86UE(#U^('-\7BA;82UF7&1=>SA]*2]\?"`I(#\@)#$@.B!U;F1E
M9CL*;7D@*"1B87-E;F%M92P@)&5X="D@/2`H)&)U9B`]?B!M?"@_.BXJ+RD_
M*"XJ*2A<+BXJ*7PI.PIR96%D(%]&2"P@)&)U9BP@-#L*<F5A9"!?1D@L("1B
M=68L('5N<&%C:R@B3B(L("1B=68I.PII9B`H9&5F:6YE9"@D97AT*2!A;F0@
M)&5X="`A?B`O7"XH/SIP;7QP;'QI>'QA;"DD+VDI('L*;7D@)&9I;&5N86UE
M(#T@7W-A=F5?87,H(B1C<F,D97AT(BP@)&)U9BP@,#<U-2D["B1005(Z.DAE
M879Y.CI&=6QL0V%C:&5[)&9U;&QN86UE?2`]("1F:6QE;F%M93L*)%!!4CHZ
M2&5A=GDZ.D9U;&Q#86-H97LD9FEL96YA;65](#T@)&9U;&QN86UE.PI]"F5L
M<V4@>PHD36]D=6QE0V%C:&5[)&9U;&QN86UE?2`]('L*8G5F(#T^("1B=68L
M"F-R8R`]/B`D8W)C+`IN86UE(#T^("1F=6QL;F%M92P*?3L*)')E<75I<F5?
M;&ES='LD9G5L;&YA;65]*RL["GT*<F5A9"!?1D@L("1B=68L(#0["GT*;&]C
M86P@0$E.0R`]("AS=6(@>PIM>2`H)'-E;&8L("1M;V1U;&4I(#T@0%\["G)E
M='5R;B!I9B!R968@)&UO9'5L92!O<B`A)&UO9'5L93L*;7D@)&EN9F\@/2`D
M36]D=6QE0V%C:&5[)&UO9'5L97T["G)E='5R;B!U;FQE<W,@)&EN9F\["F1E
M;&5T92`D<F5Q=6ER95]L:7-T>R1M;V1U;&5].PHD24Y#>R1M;V1U;&5](#T@
M(B]L;V%D97(O)&EN9F\O)&UO9'5L92(["FEF("@D14Y6>U!!4E]#3$5!3GT@
M86YD(&1E9FEN960H)DE/.CI&:6QE.CIN97<I*2!["FUY("1F:"`]($E/.CI&
M:6QE+3YN97=?=&UP9FEL92!O<B!D:64@(D-A;B=T(&-R96%T92!T96UP(&9I
M;&4Z("0A(CL*)&9H+3YB:6YM;V1E*"D["B1F:"T^<')I;G0H)&EN9F\M/GMB
M=69]*3L*)&9H+3YS965K*#`L(#`I.PIR971U<FX@)&9H.PI]"F5L<V4@>PIM
M>2`D9FEL96YA;64@/2!?<V%V95]A<R@B)&EN9F\M/GMC<F-]+G!M(BP@)&EN
M9F\M/GMB=69]*3L*)&EN9F\M/GMF:6QE?2`]("1F:6QE;F%M93L*;W!E;B!M
M>2`D9F@L("<\.G)A=R<L("1F:6QE;F%M92!O<B!D:64@<7%;0V%N)W0@<F5A
M9"`B)&9I;&5N86UE(CH@)"%=.PIR971U<FX@)&9H.PI]"F1I92`B0F]O='-T
M<F%P<&EN9R!F86EL960Z(&-A;B=T(&9I;F0@;6]D=6QE("1M;V1U;&4A(CL*
M?2P@0$E.0RD["G)E<75I<F4@6%-,;V%D97(["G)E<75I<F4@4$%2.CI(96%V
M>3L*<F5Q=6ER92!#87)P.CI(96%V>3L*<F5Q=6ER92!%>'!O<G1E<CHZ2&5A
M=GD["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*<F5Q=6ER92!)
M3SHZ1FEL93L*=VAI;&4@*&UY("1F:6QE;F%M92`]("AS;W)T(&ME>7,@)7)E
M<75I<F5?;&ES="E;,%TI('L*=6YL97-S("@D24Y#>R1F:6QE;F%M97T@;W(@
M)&9I;&5N86UE(#U^("]"4T1004XO*2!["FEF("@D9FEL96YA;64@/7X@+UPN
M<&UC/R0O:2D@>PIR97%U:7)E("1F:6QE;F%M93L*?0IE;'-E('L*9&\@)&9I
M;&5N86UE('5N;&5S<R`D9FEL96YA;64@/7X@+W-I=&5C=7-T;VUI>F5<+G!L
M)"\["GT*?0ID96QE=&4@)')E<75I<F5?;&ES='LD9FEL96YA;65].PI]"FEF
M("@D8G5F(&YE("102UI)4%]-04=)0RD@>PIO=71S*'%Q6TYO('II<"!F;W5N
M9"!A9G1E<B!&24Q%('-E8W1I;VX@:6X@9FEL92`B)'!R;V=N86UE(ETI.PIL
M87-T($U!1TE#(#L*?0HD<W1A<G1?<&]S(#T@*'1E;&P@7T9(*2`M(#0[("`@
M("`@("`@("`@("`@(",@<W1A<G0@;V8@>FEP"GT*;7D@0'!A<E]A<F=S.PIM
M>2`H)&]U="P@)&)U;F1L92P@)&QO9V9H+"`D8V%C:&5?;F%M92D["F1E;&5T
M92`D14Y6>U!!4E]!4%!?4D554T5].R`C('-A;FET:7IE("A2155312!M87D@
M8F4@82!S96-U<FET>2!P<F]B;&5M*0II9B`H(21S=&%R=%]P;W,@;W(@*"1!
M4D=66S!=(&5Q("<M+7!A<BUO<'1I;VYS)R`F)B!S:&EF="DI('L*;7D@)61I
M<W1?8VUD(#T@<7<H"G`@("!B;&EB7W1O7W!A<@II("`@:6YS=&%L;%]P87(*
M=2`@('5N:6YS=&%L;%]P87(*<R`@('-I9VY?<&%R"G8@("!V97)I9GE?<&%R
M"BD["FEF("A`05)'5B!A;F0@)$%21U9;,%T@97$@)RTM<F5U<V4G*2!["G-H
M:69T($!!4D=6.PHD14Y6>U!!4E]!4%!?4D554T5](#T@<VAI9G0@0$%21U8[
M"GT*96QS92![(",@;F]R;6%L('!A<FP@8F5H879I;W5R"FUY($!A9&1?=&]?
M:6YC.PIW:&EL92`H0$%21U8I('L*)$%21U9;,%T@/7X@+UXM*%M!24U/0DQB
M<7!I=7-4=ETI*"XJ*2\@;W(@;&%S=#L*:68@*"0Q(&5Q("=))RD@>PIP=7-H
M($!A9&1?=&]?:6YC+"`D,CL*?0IE;'-I9B`H)#$@97$@)TTG*2!["F5V86P@
M(G5S92`D,B(["GT*96QS:68@*"0Q(&5Q("=!)RD@>PIU;G-H:69T($!P87)?
M87)G<RP@)#(["GT*96QS:68@*"0Q(&5Q("=/)RD@>PHD;W5T(#T@)#(["GT*
M96QS:68@*"0Q(&5Q("=B)RD@>PHD8G5N9&QE(#T@)W-I=&4G.PI]"F5L<VEF
M("@D,2!E<2`G0B<I('L*)&)U;F1L92`]("=A;&PG.PI]"F5L<VEF("@D,2!E
M<2`G<2<I('L*)'%U:65T(#T@,3L*?0IE;'-I9B`H)#$@97$@)TPG*2!["F]P
M96X@)&QO9V9H+"`B/CXB+"`D,B!O<B!D:64@<7%;0V%N)W0@;W!E;B!L;V<@
M9FEL92`B)#(B.B`D(5T["GT*96QS:68@*"0Q(&5Q("=4)RD@>PHD8V%C:&5?
M;F%M92`]("0R.PI]"G-H:69T*$!!4D=6*3L*:68@*&UY("1C;60@/2`D9&ES
M=%]C;61[)#%]*2!["F1E;&5T92`D14Y6>R=005)?5$5-4"=].PII;FET7VEN
M8R@I.PIR97%U:7)E(%!!4CHZ1&ES=#L*)GLB4$%2.CI$:7-T.CHD8VUD(GTH
M*2!U;FQE<W,@0$%21U8["B9[(E!!4CHZ1&ES=#HZ)&-M9")]*"1?*2!F;W(@
M0$%21U8["F5X:70["GT*?0IU;G-H:69T($!)3D,L($!A9&1?=&]?:6YC.PI]
M"GT*:68@*"1O=70I('L*>PIR97%U:7)E($E/.CI&:6QE.PIR97%U:7)E($%R
M8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E<W0Z.E-(03L*?0IM>2`D<&%R(#T@
M<VAI9G0H0$%21U8I.PIM>2`D>FEP.PII9B`H9&5F:6YE9"`D<&%R*2!["F]P
M96X@;7D@)&9H+"`G/#IR87<G+"`D<&%R(&]R(&1I92!Q<5M#86XG="!F:6YD
M('!A<B!F:6QE("(D<&%R(CH@)"%=.PIB;&5S<R@D9F@L("=)3SHZ1FEL92<I
M.PHD>FEP(#T@07)C:&EV93HZ6FEP+3YN97<["B@@)'II<"T^<F5A9$9R;VU&
M:6QE2&%N9&QE*"1F:"P@)'!A<BD@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I
M("D*;W(@9&EE('%Q6T5R<F]R(')E861I;F<@>FEP(&%R8VAI=F4@(B1P87(B
M73L*?0IM>2`E;65T85]P87(@/2!D;R!["FEF("@D>FEP(&%N9"!M>2`D;65T
M82`]("1Z:7`M/F-O;G1E;G1S*"=-151!+GEM;"<I*2!["B1M971A(#U^(',O
M+BI><&%R.B0O+VUS.PHD;65T82`]?B!S+UY<4RXJ+R]M<SL*)&UE=&$@/7X@
M+UX@("A;7CI=*RDZ("@N*RDD+VUG.PI]"GT["FEF("AD969I;F5D("1P87(I
M('L*;W!E;B!M>2`D<&@L("<\.G)A=R<L("1P87(@;W(@9&EE('%Q6T-A;B=T
M(')E860@<&%R(&9I;&4@(B1P87(B.B`D(5T["FUY("1B=68["G)E860@)'!H
M+"`D8G5F+"`T.PID:64@<7%;(B1P87(B(&ES(&YO="!A('!A<B!F:6QE72!U
M;FQE<W,@)&)U9B!E<2`D4$M:25!?34%'24,["F-L;W-E("1P:#L*?0I#<F5A
M=&50871H*"1O=70I(#L*;7D@)&9H(#T@24\Z.D9I;&4M/FYE=R@*)&]U="P*
M24\Z.D9I;&4Z.D]?0U)%050H*2!\($E/.CI&:6QE.CI/7U)$5U(H*2!\($E/
M.CI&:6QE.CI/7U1254Y#*"DL"C`W-S<L"BD@;W(@9&EE('%Q6T-A;B=T(&-R
M96%T92!F:6QE("(D;W5T(CH@)"%=.PHD9F@M/F)I;FUO9&4H*3L*<V5E:R!?
M1D@L(#`L(#`["FUY("1L;V%D97(["FEF("AD969I;F5D("1S=&%R=%]O9E]&
M24Q%7W-E8W1I;VXI('L*<F5A9"!?1D@L("1L;V%D97(L("1S=&%R=%]O9E]&
M24Q%7W-E8W1I;VX["GT@96QS92!["FQO8V%L("0O(#T@=6YD968["B1L;V%D
M97(@/2`\7T9(/CL*?0II9B`H(21%3E9[4$%27U9%4D)!5$E-?2!A;F0@)&QO
M861E<B`]?B`O7B@_.B,A?%Q`<F5M*2\I('L*<F5Q=6ER92!005(Z.D9I;'1E
M<CHZ4&]D4W1R:7`["E!!4CHZ1FEL=&5R.CI0;V13=')I<"T^87!P;'DH7"1L
M;V%D97(L("0P*3L*?0HD9F@M/G!R:6YT*"1L;V%D97(I"F]R(&1I92!Q<5M%
M<G)O<B!W<FET:6YG(&QO861E<B!T;R`B)&]U="(Z("0A73L*:68@*"1B=6YD
M;&4I('L*<F5Q=6ER92!005(Z.DAE879Y.PI005(Z.DAE879Y.CI?:6YI=%]D
M>6YA;&]A9&5R*"D["FEN:71?:6YC*"D["G)E<75I<F5?;6]D=6QE<R@I.PIM
M>2!`:6YC(#T@9W)E<"![("$O0E-$4$%.+R!]"F=R97`@>PHH)&)U;F1L92!N
M92`G<VET92<I(&]R"B@D7R!N92`D0V]N9FEG.CI#;VYF:6=[87)C:&QI8F5X
M<'T@86YD"B1?(&YE("1#;VYF:6<Z.D-O;F9I9WMP<FEV;&EB97AP?2D["GT@
M0$E.0SL*<WPO*WPO?&<L('-\+R1\?"!F;W)E86-H($!I;F,["FUY("5F:6QE
M<SL*)&9I;&5S>R1??2LK(&9O<B!`1'EN84QO861E<CHZ9&Q?<VAA<F5D7V]B
M:F5C=',L('9A;'5E<R`E24Y#.PIM>2`D;&EB7V5X="`]("1#;VYF:6<Z.D-O
M;F9I9WML:6)?97AT?3L@("`@("`@("`C(%A86"!L:6)?97AT('9S(&1L97AT
M(#\*;7D@)7=R:71T96X["F9O<F5A8V@@;7D@)&ME>2`H<V]R="!K97ES("5F
M:6QE<RD@>PIM>2`H)&9I;&4L("1N86UE*3L*:68@*&1E9FEN960H;7D@)&9C
M(#T@)%!!4CHZ2&5A=GDZ.D9U;&Q#86-H97LD:V5Y?2DI('L**"1F:6QE+"`D
M;F%M92D@/2`H)&ME>2P@)&9C*3L*?0IE;'-E('L*9F]R96%C:"!M>2`D9&ER
M("A`:6YC*2!["FEF("@D:V5Y(#U^(&U\7EQ1)&1I<EQ%+R@N*BDD?&DI('L*
M*"1F:6QE+"`D;F%M92D@/2`H)&ME>2P@)#$I.PIL87-T.PI]"FEF("@D:V5Y
M(#U^(&U\7B]L;V%D97(O6UXO72LO*"XJ*21\*2!["FEF("AM>2`D<F5F(#T@
M)$UO9'5L94-A8VAE>R0Q?2D@>PHH)&9I;&4L("1N86UE*2`]("@D<F5F+"`D
M,2D["FQA<W0["GT*:68@*"UF("(D9&ER+R0Q(BD@>PHH)&9I;&4L("1N86UE
M*2`]("@B)&1I<B\D,2(L("0Q*3L*;&%S=#L*?0I]"GT*?0IN97AT('5N;&5S
M<R!D969I;F5D("1N86UE.PIN97AT(&EF("1W<FET=&5N>R1N86UE?2LK.PIN
M97AT(&EF("%R968H)&9I;&4I(&%N9"`D9FEL92`]?B`O7"Y<421L:6)?97AT
M7$4D+VD["F]U=',H<W!R:6YT9BAQ<5M086-K:6YG($9)3$4@(B5S(BXN+ETL
M(')E9B`D9FEL92`_("1F:6QE+3Y[;F%M97T@.B`D9FEL92DI.PIM>2`D8V]N
M=&5N=#L*:68@*')E9B@D9FEL92DI('L*)&-O;G1E;G0@/2`D9FEL92T^>V)U
M9GT["GT*96QS92!["FQO8V%L("0O(#T@=6YD968["F]P96X@;7D@)'1H+"`G
M/#IR87<G+"`D9FEL92!O<B!D:64@<7%;0V%N)W0@<F5A9"`B)&9I;&4B.B`D
M(5T["B1C;VYT96YT(#T@/"1T:#X["F-L;W-E("1T:#L*4$%2.CI&:6QT97(Z
M.E!O9%-T<FEP+3YA<'!L>2A<)&-O;G1E;G0L("(\96UB961D960^+R1N86UE
M(BD*:68@(21%3E9[4$%27U9%4D)!5$E-?2!A;F0@)&YA;64@/7X@+UPN*#\Z
M<&U\:7A\86PI)"]I.PI005(Z.D9I;'1E<CHZ4&%T8VA#;VYT96YT+3YN97<M
M/F%P<&QY*%PD8V]N=&5N="P@)&9I;&4L("1N86UE*3L*?0HD9F@M/G!R:6YT
M*")&24Q%(BP*<&%C:R@G3B<L(&QE;F=T:"@D;F%M92D@*R`Y*2P*<W!R:6YT
M9B@B)3`X>"\E<R(L($%R8VAI=F4Z.EII<#HZ8V]M<'5T94-20S,R*"1C;VYT
M96YT*2P@)&YA;64I+`IP86-K*"=.)RP@;&5N9W1H*"1C;VYT96YT*2DL"B1C
M;VYT96YT*0IO<B!D:64@<7%;17)R;W(@=W)I=&EN9R!E;6)E9&1E9"!&24Q%
M('1O("(D;W5T(CH@)"%=.PIO=71S*'%Q6U=R:71T96X@87,@(B1N86UE(ETI
M.PI]"GT*:68@*"1Z:7`I('L*)'II<"T^=W)I=&54;T9I;&5(86YD;&4H)&9H
M*2`]/2!!<F-H:79E.CI::7`Z.D%:7T]+"F]R(&1I92!Q<5M%<G)O<B!W<FET
M:6YG('II<"!P87)T(&]F("(D;W5T(ET["GT*:68@*"1M971A7W!A<GMC;&5A
M;GTI('L*)&9H+3YP<FEN="@B7#!#3$5!3B(I.PI]"F5L<V4@>PII9B`H(61E
M9FEN960@)&-A8VAE7VYA;64I('L*;7D@)&-T>"`]($1I9V5S=#HZ4TA!+3YN
M97<H,2D["G-E96L@)&9H+"`P+"`P.PHD8W1X+3YA9&1F:6QE*"1F:"D["G-E
M96L@)&9H+"`P+"`R.PHD8V%C:&5?;F%M92`]("1C='@M/FAE>&1I9V5S=#L*
M?0HD8V%C:&5?;F%M92`]('-U8G-T<B`D8V%C:&5?;F%M92P@,"P@)&-A8VAE
M7VYA;65?<VEZ93L*)&-A8VAE7VYA;64@+CT@(EPP(B!X("@D8V%C:&5?;F%M
M95]S:7IE("T@;&5N9W1H("1C86-H95]N86UE*3L*)&9H+3YP<FEN="@D8V%C
M:&5?;F%M92P*(EPP0T%#2$4B*3L*?0IM>2`D;V9F<V5T(#T@)&9H+3YT96QL
M("T@;&5N9W1H*"1L;V%D97(I.PHD9F@M/G!R:6YT*'!A8VLH)TXG+"`D;V9F
M<V5T*2P*)%!!4E]-04=)0RD["B1F:"T^8VQO<V4@;W(@9&EE('%Q6T5R<F]R
M('=R:71I;F<@=')A:6QE<B!O9B`B)&]U="(Z("0A73L*8VAM;V0@,#<U-2P@
M)&]U=#L*97AI=#L*?0I["FQA<W0@=6YL97-S(&1E9FEN960@)'-T87)T7W!O
M<SL*7V9I>%]P<F]G;F%M92@I.PIR97%U:7)E(%!!4CL*4$%2.CI(96%V>3HZ
M7VEN:71?9'EN86QO861E<B@I.PI["G)E<75I<F4@1FEL93HZ1FEN9#L*<F5Q
M=6ER92!!<F-H:79E.CI::7`["GT*;7D@)&9H(#T@24\Z.D9I;&4M/FYE=SL@
M("`@("`@("`@("`@("`@("`@("`@("`@("`@(",@07)C:&EV93HZ6FEP(&]P
M97)A=&5S(&]N(&%N($E/.CI(86YD;&4*)&9H+3YF9&]P96XH9FEL96YO*%]&
M2"DL("=R)RD@;W(@9&EE('%Q6V9D;W!E;B@I(&9A:6QE9#H@)"%=.PI!<F-H
M:79E.CI::7`Z.G-E=$-H=6YK4VEZ92@M<R!?1D@I.PIM>2`D>FEP(#T@07)C
M:&EV93HZ6FEP+3YN97<["B@D>FEP+3YR96%D1G)O;49I;&5(86YD;&4H)&9H
M+"`D<')O9VYA;64I(#T]($%R8VAI=F4Z.EII<#HZ05I?3TLH*2D*;W(@9&EE
M('%Q6T5R<F]R(')E861I;F<@>FEP(&%R8VAI=F4@(B1P<F]G;F%M92)=.PI!
M<F-H:79E.CI::7`Z.G-E=$-H=6YK4VEZ92@V-"`J(#$P,C0I.PIP=7-H($!0
M05(Z.DQI8D-A8VAE+"`D>FEP.PHD4$%2.CI,:6)#86-H97LD<')O9VYA;65]
M(#T@)'II<#L*;W5T<R@B17AT<F%C=&EN9R!Z:7`N+BXB*3L*:68@*&1E9FEN
M960@)$5.5GM005)?5$5-4'TI('L@(R!S:&]U;&0@8F4@<V5T(&%T('1H:7,@
M<&]I;G0A"F9O<F5A8V@@;7D@)&UE;6)E<B`H("1Z:7`M/FUE;6)E<G,@*2![
M"FYE>'0@:68@)&UE;6)E<BT^:7-$:7)E8W1O<GD["FUY("1M96UB97)?;F%M
M92`]("1M96UB97(M/F9I;&5.86UE.PIN97AT('5N;&5S<R`D;65M8F5R7VYA
M;64@/7X@;7L*7@HO/W-H;&EB+PHH/SHD0V]N9FEG.CI#;VYF:6=[87)C:&YA
M;65]+RD_"BA;7B]=*RD*)`I]>#L*;7D@)&5X=')A8W1?;F%M92`]("0Q.PIM
M>2`D9&5S=%]N86UE(#T@1FEL93HZ4W!E8RT^8V%T9FEL92@D14Y6>U!!4E]4
M14U0?2P@)&5X=')A8W1?;F%M92D["FEF("@M9B`D9&5S=%]N86UE("8F("US
M(%\@/3T@)&UE;6)E<BT^=6YC;VUP<F5S<V5D4VEZ92@I*2!["F]U=',H<7%;
M4VMI<'!I;F<@(B1M96UB97)?;F%M92(@<VEN8V4@:70@86QR96%D>2!E>&ES
M=',@870@(B1D97-T7VYA;64B72D["GT@96QS92!["F]U=',H<7%;17AT<F%C
M=&EN9R`B)&UE;6)E<E]N86UE(B!T;R`B)&1E<W1?;F%M92(N+BY=*3L**"1M
M96UB97(M/F5X=')A8W14;T9I;&5.86UE9"@D9&5S=%]N86UE*2`]/2!!<F-H
M:79E.CI::7`Z.D%:7T]+*"DI"F]R(&1I92!Q<5M%<G)O<B!E>'1R86-T:6YG
M('II<"!M96UB97(@=&\@(B1D97-T7VYA;64B73L*8VAM;V0H,#4U-2P@)&1E
M<W1?;F%M92D@:68@)%Y/(&5Q(")H<'5X(CL*?0I]"GT*?0IU;FQE<W,@*"10
M05(Z.DQI8D-A8VAE>R1P<F]G;F%M97TI('L*9&EE(#P\("(N(B!U;FQE<W,@
M0$%21U8["E5S86=E.B`D,"!;("U!;&EB+G!A<B!=(%L@+4ED:7(@72!;("U-
M;6]D=6QE(%T@6R!S<F,N<&%R(%T@6R!P<F]G<F%M+G!L(%T*)#`@6R`M0GPM
M8B!=(%LM3V]U=&9I;&5=('-R8RYP87(*+@HD14Y6>U!!4E]04D]'3D%-17T@
M/2`D<')O9VYA;64@/2`D,"`]('-H:69T*$!!4D=6*3L*?0IS=6(@0W)E871E
M4&%T:"!["FUY("@D;F%M92D@/2!`7SL*<F5Q=6ER92!&:6QE.CI"87-E;F%M
M93L*;7D@*"1B87-E;F%M92P@)'!A=&@L("1E>'0I(#T@1FEL93HZ0F%S96YA
M;64Z.F9I;&5P87)S92@D;F%M92P@*"=<+BXJ)RDI.PIR97%U:7)E($9I;&4Z
M.E!A=&@["D9I;&4Z.E!A=&@Z.FUK<&%T:"@D<&%T:"D@=6YL97-S*"UE("1P
M871H*3L@(R!M:W!A=&@@9&EE<R!W:71H(&5R<F]R"GT*<W5B(')E<75I<F5?
M;6]D=6QE<R!["G)E<75I<F4@;&EB.PIR97%U:7)E($1Y;F%,;V%D97(["G)E
M<75I<F4@:6YT96=E<CL*<F5Q=6ER92!S=')I8W0["G)E<75I<F4@=V%R;FEN
M9W,["G)E<75I<F4@=F%R<SL*<F5Q=6ER92!#87)P.PIR97%U:7)E($-A<G`Z
M.DAE879Y.PIR97%U:7)E($5R<FYO.PIR97%U:7)E($5X<&]R=&5R.CI(96%V
M>3L*<F5Q=6ER92!%>'!O<G1E<CL*<F5Q=6ER92!&8VYT;#L*<F5Q=6ER92!&
M:6QE.CI496UP.PIR97%U:7)E($9I;&4Z.E-P96,["G)E<75I<F4@6%-,;V%D
M97(["G)E<75I<F4@0V]N9FEG.PIR97%U:7)E($E/.CI(86YD;&4["G)E<75I
M<F4@24\Z.D9I;&4["G)E<75I<F4@0V]M<')E<W,Z.EIL:6(["G)E<75I<F4@
M07)C:&EV93HZ6FEP.PIR97%U:7)E($1I9V5S=#HZ4TA!.PIR97%U:7)E(%!!
M4CL*<F5Q=6ER92!005(Z.DAE879Y.PIR97%U:7)E(%!!4CHZ1&ES=#L*<F5Q
M=6ER92!005(Z.D9I;'1E<CHZ4&]D4W1R:7`["G)E<75I<F4@4$%2.CI&:6QT
M97(Z.E!A=&-H0V]N=&5N=#L*<F5Q=6ER92!A='1R:6)U=&5S.PIE=F%L('L@
M<F5Q=6ER92!#=V0@?3L*979A;"![(')E<75I<F4@5VEN,S(@?3L*979A;"![
M(')E<75I<F4@4V-A;&%R.CI5=&EL('T["F5V86P@>R!R97%U:7)E($%R8VAI
M=F4Z.E5N>FEP.CI"=7)S="!].PIE=F%L('L@<F5Q=6ER92!4:64Z.DAA<V@Z
M.DYA;65D0V%P='5R92!].PIE=F%L('L@<F5Q=6ER92!097)L24\[(')E<75I
M<F4@4&5R;$E/.CIS8V%L87(@?3L*979A;"![(')E<75I<F4@=71F."!].PI]
M"G-U8B!?<V5T7W!A<E]T96UP('L*:68@*&1E9FEN960@)$5.5GM005)?5$5-
M4'T@86YD("1%3E9[4$%27U1%35!](#U^("\H+BLI+RD@>PHD<&%R7W1E;7`@
M/2`D,3L*<F5T=7)N.PI]"F9O<F5A8V@@;7D@)'!A=&@@*`HH;6%P("1%3E9[
M)%]]+"!Q=R@@4$%27U1-4$1)4B!435!$25(@5$5-4$1)4B!414U0(%1-4"`I
M*2P*<7<H($,Z7%Q414U0("]T;7`@+B`I"BD@>PIN97AT('5N;&5S<R!D969I
M;F5D("1P871H(&%N9"`M9"`D<&%T:"!A;F0@+7<@)'!A=&@["FUY("1U<V5R
M;F%M93L*;7D@)'!W=6ED.PIE=F%L('LH)'!W=6ED*2`](&=E='!W=6ED*"0^
M*2!I9B!D969I;F5D("0^.WT["FEF("@@9&5F:6YE9"@F5VEN,S(Z.DQO9VEN
M3F%M92D@*2!["B1U<V5R;F%M92`]("97:6XS,CHZ3&]G:6Y.86UE.PI]"F5L
M<VEF("AD969I;F5D("1P=W5I9"D@>PHD=7-E<FYA;64@/2`D<'=U:60["GT*
M96QS92!["B1U<V5R;F%M92`]("1%3E9[55-%4DY!345]('Q\("1%3E9[55-%
M4GT@?'P@)U-94U1%32<["GT*;7D@)'-T;7!D:7(@/2`B)'!A=&@D<WES>U]D
M96QI;7UP87(M(BYU;G!A8VLH(D@J(BP@)'5S97)N86UE*3L*;6MD:7(@)'-T
M;7!D:7(L(#`W-34["FUY("1C86-H95]D:7(["FEF("@D14Y6>U!!4E]#3$5!
M3GTI('L*)&-A8VAE7V1I<B`](")T96UP+20D(CL*?0IE;'-E('L*;W!E;B!M
M>2`D9F@L("(\.G)A=R(L("1P<F]G;F%M92!O<B!D:64@<7%;0V%N)W0@<F5A
M9"`B)'!R;V=N86UE(CH@)"%=.PII9B`H*&UY("1M86=I8U]P;W,@/2!F:6YD
M7W!A<E]M86=I8R@D9F@I*2`^/2`P*2!["G-E96L@)&9H+"`D;6%G:6-?<&]S
M"BT@)$9)3$5?;V9F<V5T7W-I>F4*+2!L96YG=&@H(EPP0T%#2$4B*2P@,#L*
M;7D@)&)U9CL*<F5A9"`D9F@L("1B=68L(&QE;F=T:"@B7#!#04-(12(I.PII
M9B`H)&)U9B!E<2`B7#!#04-(12(I('L*<V5E:R`D9F@L("1M86=I8U]P;W,*
M+2`D1DE,15]O9F9S971?<VEZ90HM(&QE;F=T:"@B7#!#04-(12(I"BT@)&-A
M8VAE7VYA;65?<VEZ92P@,#L*<F5A9"`D9F@L("1B=68L("1C86-H95]N86UE
M7W-I>F4["B1B=68@/7X@<R]<,"\O9SL*)&-A8VAE7V1I<B`](")C86-H92TD
M8G5F(CL*?0I]"F-L;W-E("1F:#L*?0II9B`H(21C86-H95]D:7(I('L*)&-A
M8VAE7V1I<B`](")T96UP+20D(CL*)$5.5GM005)?0TQ%04Y](#T@,3L*?0HD
M<W1M<&1I<B`N/2`B)'-Y<WM?9&5L:6U])&-A8VAE7V1I<B(["FUK9&ER("1S
M=&UP9&ER+"`P-S4U.PHD14Y6>U!!4E]414U0?2`]("1S=&UP9&ER.PIL87-T
M.PI]"B1P87)?=&5M<"`]("0Q(&EF("1%3E9[4$%27U1%35!](&%N9"`D14Y6
M>U!!4E]414U0?2`]?B`O*"XK*2\["GT*<W5B(%]S879E7V%S('L*;7D@*"1N
M86UE+"`D8V]N=&5N=',L("1M;V1E*2`]($!?.PIM>2`D9G5L;&YA;64@/2`B
M)'!A<E]T96UP+R1N86UE(CL*=6YL97-S("@M92`D9G5L;&YA;64I('L*;7D@
M)'1E;7!N86UE(#T@(B1F=6QL;F%M92XD)"(["FUY("1F:#L**&]P96X@)&9H
M+"`G/CIR87<G+"`D=&5M<&YA;64I"B8F("AP<FEN="`D9F@@)&-O;G1E;G1S
M*0HF)B`H8VQO<V4@)&9H*0IO<B!D:64@<7%;17)R;W(@=W)I=&EN9R`B)'1E
M;7!N86UE(CH@)"%=.PIC:&UO9"`D;6]D92P@)'1E;7!N86UE(&EF(&1E9FEN
M960@)&UO9&4["G)E;F%M92@D=&5M<&YA;64L("1F=6QL;F%M92D@;W(@=6YL
M:6YK*"1T96UP;F%M92D["GT*<F5T=7)N("1F=6QL;F%M93L*?0IS=6(@7W-E
M=%]P<F]G;F%M92!["FEF("AD969I;F5D("1%3E9[4$%27U!23T=.04U%?2!A
M;F0@)$5.5GM005)?4%)/1TY!345](#U^("\H+BLI+RD@>PHD<')O9VYA;64@
M/2`D,3L*?0HD<')O9VYA;64@?'P]("0P.PII9B`H)$5.5GM005)?5$5-4'T@
M86YD(&EN9&5X*"1P<F]G;F%M92P@)$5.5GM005)?5$5-4'TI(#X](#`I('L*
M)'!R;V=N86UE(#T@<W5B<W1R*"1P<F]G;F%M92P@<FEN9&5X*"1P<F]G;F%M
M92P@)'-Y<WM?9&5L:6U]*2`K(#$I.PI]"FEF("@A)$5.5GM005)?4%)/1TY!
M345](&]R(&EN9&5X*"1P<F]G;F%M92P@)'-Y<WM?9&5L:6U]*2`^/2`P*2![
M"FEF("AO<&5N(&UY("1F:"P@)SPG+"`D<')O9VYA;64I('L*<F5T=7)N(&EF
M("US("1F:#L*?0II9B`H+7,@(B1P<F]G;F%M921S>7-[7V5X97TB*2!["B1P
M<F]G;F%M92`N/2`D<WES>U]E>&5].PIR971U<FX["GT*?0IF;W)E86-H(&UY
M("1D:7(@*'-P;&ET("]<421S>7-[<&%T:%]S97!]7$4O+"`D14Y6>U!!5$A]
M*2!["FYE>'0@:68@97AI<W1S("1%3E9[4$%27U1%35!](&%N9"`D9&ER(&5Q
M("1%3E9[4$%27U1%35!].PHD9&ER(#U^(',O7%$D<WES>U]D96QI;7U<120O
M+SL**"@D<')O9VYA;64@/2`B)&1I<B1S>7-[7V1E;&EM?21P<F]G;F%M921S
M>7-[7V5X97TB*2P@;&%S="D*:68@+7,@(B1D:7(D<WES>U]D96QI;7TD<')O
M9VYA;64D<WES>U]E>&5](CL**"@D<')O9VYA;64@/2`B)&1I<B1S>7-[7V1E
M;&EM?21P<F]G;F%M92(I+"!L87-T*0II9B`M<R`B)&1I<B1S>7-[7V1E;&EM
M?21P<F]G;F%M92(["GT*?0IS=6(@7V9I>%]P<F]G;F%M92!["B0P(#T@)'!R
M;V=N86UE('Q\/2`D14Y6>U!!4E]04D]'3D%-17T["FEF("AI;F1E>"@D<')O
M9VYA;64L("1S>7-[7V1E;&EM?2D@/"`P*2!["B1P<F]G;F%M92`]("(N)'-Y
M<WM?9&5L:6U])'!R;V=N86UE(CL*?0IM>2`D<'=D(#T@*&1E9FEN960@)D-W
M9#HZ9V5T8W=D*2`_($-W9#HZ9V5T8W=D*"D*.B`H*&1E9FEN960@)E=I;C,R
M.CI'971#=V0I(#\@5VEN,S(Z.D=E=$-W9"@I(#H@8'!W9&`I.PIC:&]M<"@D
M<'=D*3L*)'!R;V=N86UE(#U^(',O7B@_/5PN7"X_7%$D<WES>U]D96QI;7U<
M12DO)'!W9"1S>7-[7V1E;&EM?2\["B1%3E9[4$%27U!23T=.04U%?2`]("1P
M<F]G;F%M93L*?0IS=6(@7W!A<E]I;FET7V5N=B!["FEF("@@)$5.5GM005)?
M24Y)5$E!3$E:141]*RL@/3T@,2`I('L*<F5T=7)N.PI](&5L<V4@>PHD14Y6
M>U!!4E])3DE424%,25I%1'T@/2`R.PI]"F9O<B`H<7<H(%-005=.140@5$5-
M4"!#3$5!3B!$14)51R!#04-(12!04D]'3D%-12`I("D@>PID96QE=&4@)$5.
M5GLB4$%27R1?(GT["GT*9F]R("AQ=R\@5$U01$E2(%1%35`@0TQ%04X@1$5"
M54<@+RD@>PHD14Y6>R)005)?)%\B?2`]("1%3E9[(E!!4E]'3$]"04Q?)%\B
M?2!I9B!E>&ES=',@)$5.5GLB4$%27T=,3T)!3%\D7R)].PI]"FEF("@D14Y6
M>U!!4E]414U0?2D@>PID96QE=&4@)$5.5GM005)?0TQ%04Y].PI]"GT*<W5B
M(&]U=',@>PIR971U<FX@:68@)'%U:65T.PII9B`H)&QO9V9H*2!["G!R:6YT
M("1L;V=F:"`B0%]<;B(["GT*96QS92!["G!R:6YT(")`7UQN(CL*?0I]"G-U
M8B!I;FET7VEN8R!["G)E<75I<F4@0V]N9FEG.PIP=7-H($!)3D,L(&=R97`@
M9&5F:6YE9"P@;6%P("1#;VYF:6<Z.D-O;F9I9WLD7WTL('%W*`IA<F-H;&EB
M97AP('!R:79L:6)E>'`@<VET96%R8VAE>'`@<VET96QI8F5X<`IV96YD;W)A
M<F-H97AP('9E;F1O<FQI8F5X<`HI.PI]"G!A8VMA9V4@;6%I;CL*<F5Q=6ER
M92!005(["G5N<VAI9G0@0$E.0RP@7"9005(Z.F9I;F1?<&%R.PI005(M/FEM
M<&]R="A`<&%R7V%R9W,I.PID:64@<7%;<&%R+G!L.B!#86XG="!O<&5N('!E
M<FP@<V-R:7!T("(D<')O9VYA;64B.B!.;R!S=6-H(&9I;&4@;W(@9&ER96-T
M;W)Y7&Y="G5N;&5S<R`M92`D<')O9VYA;64["F1O("1P<F]G;F%M93L*0T]2
M13HZ97AI="@D,2D@:68@*"1`(#U^+UY?5$M?15A)5%]<*"A<9"LI7"DO*3L*
M9&EE("1`(&EF("1`.PI].PHD.CI?7T524D]2(#T@)$`@:68@)$`["GT*0T]2
M13HZ97AI="@D,2D@:68@*"0Z.E]?15)23U(@/7XO7E]42U]%6$E47UPH*%QD
M*RE<*2\I.PID:64@)#HZ7U]%4E)/4B!I9B`D.CI?7T524D]2.PHQ.PI?7T5.
M1%]?"@!P97)L>'-I+F,``````````$-!0TA%```!&P,[````_````![__XM@
M```!%/__B\````%`__^,)````6S__XV0```!V/__CH0```)`__^0`````J#_
M_Y"4```"R/__D8````+\__^3L````U#__Y/P```#</__E%````.4__^48```
M`ZC__Y:$```$"/__EK````0L__^7(```!&#__YOD```$G/__G%0```2\__^<
M\```!-C__YY@```%9/__GH````6$__^?@```!<#__Z5````&,/__I\````9H
M__^H,```!LC__[B0```&[/__N.0```<,__^Z$```!U#__[O0```!K/__O%``
M``&8__^\D```!I0`````````$``````!>E(`!'@>`1L,'P`````H````&/__
MBD0```!<`$$.,)T&G@5$E0)%E`.3!$K4TT+>W=4.`````````"@```!$__^*
M>````&0`00XPG0:>!4*3!)0#0I4"4@K>W=73U`X`00L`````*````'#__XJP
M```!:`!!#D"=")X'0I0%DP9!E@.5!$&7`@`````````````0````G/__NK``
M```X`````````"@```"P__^Z'````'@`2@X0G0*>`4C>W0X`20X0G0*>`4'>
MW0X`````````9````-S__XNP````]`!!#D"=")X'0I,&E`5"F`&7`D:6`Y4$
M5M;50=C70][=T]0.`$$.0),&E`65!)8#EP*8`9T(G@=/UM5!V-=!WMW3U`X`
M00Y`DP:4!9T(G@="EP*8`4+8UP````!<```!1/__C#P```%X`$$.X`&=')X;
M2)09DQI$U--(WMT.`$$.X`&3&I09G1R>&T28%9<61)87E1AP"M330=;50=C7
M00M)T]35UM?80909DQI!EA>5&$&8%9<6```````D```!I/__C5@```"4`$$.
MP`&=&)X71),65@K>W=,.`$$+````````,````<S__XW$````[`!!#K`(1)V&
M`9Z%`4.3A`%B"MW>TPX`00M."MW>TPX`00L``````%````(`__^.?````BP`
M00Y@G0R>"T.3"I0)3)4(E@>7!I@%?PK>W=?8U=;3U`X`00M)F@.9!%\*VME!
M"TG9VD*9!)H#0]G:09H#F01"VMD``````!P```)4__^06````$``00X0G0*>
M`4C>W0X`````````(````G3__Y!X````7`!!#B"=!)X#0Y,"3M[=TPX`````
M````$````IC__Y"T````"`````````!<```"K/__D+````(D`$$.@`I"G:`!
MGI\!0Y.>`92=`4.5G`&6FP%<"MW>U=;3U`X`00M(F)D!EYH!19F8`7<*V-=!
MV4$+50K8UT'900M#U]C909B9`9>:`4&9F`$````@```##/__DG0````L`$$.
M()T$G@-#DP)&WMW3#@`````````P```#,/__DGP```!L`$(.()T$G@-"DP)*
M"M[=TPX`00M#"M[=TPX`0@M#WMW3#@``````.````V3__Y*X```$Q`!!#F"=
M#)X+1)4(E@="DPJ4"467!I@%F00#`1H*WMW9U]C5UM/4#@!!"P``````'```
M`Z#__Y=`````<`!!#B"=!)X#5PK>W0X`00L````8```#P/__EY````"<`$$.
MH`*=))XC````````B````]S__Y@0```!;`!!#E!"#M"`!$2=BD">B4!#DXA`
ME(=`39:%0)6&0$28@T"7A$!BUM5"V-=*W=[3U$$.@(`$00X`00[0@`23B$"4
MAT"5AD"6A4"7A$"8@T"=BD">B4!&UM5!V-="E89`EH5`EX1`F(-`0]76U]A!
MEH5`E89`09B#0)>$0``````<```$:/__F/0````@`$$.$)T"G@%&WMT.````
M`````#@```2(__^8]````/@`00Y`G0B>!T:3!I0%E026`U@*WMW5UM/4#@!!
M"UP*WMW5UM/4#@!!"P```````&P```3$__^9N```!;P`00Z0`YTRGC%(DS"4
M+Y4NEBU#ERR8*UR:*9DJ`ED*VME!WMW7V-76T]0.`$$+`IG9VDW>W=?8U=;3
MU`X`00Z0`Y,PE"^5+I8MERR8*YDJFBF=,IXQ`D/9VD&:*9DJ```````T```%
M-/__GP@```)X`$$.4)T*G@E$DPB4!T:5!I8%0Y<$F`,"9PK>W=?8U=;3U`X`
M00L``````"@```5L__^A4````&0`00X@G02>`T*3`D<*WMW3#@!!"TW>W=,.
M````````,```!9C__[7T```"&`!!#F"=#)X+19,*E`F5")8'`ET*WMW5UM/4
M#@!!"P```````````"````7,__^A8```$%P`00Z@!4:=5)Y3`P0."MW>#@!!
M"P```!P```7P__^QG````%0`00X0G0*>`5/>W0X`````````0```!A#__['0
M```!)`!!#D"=")X'0Y,&E`5#E026`T.7`I@!60K>W=?8U=;3U`X`00MB"M[=
MU]C5UM/4#@!!"P`````P```&5/__LK@```&X`$$.0)T(G@="DP:4!465!)8#
MEP("7PK>W=?5UM/4#@!!"P``````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````````````!14````````%#0````
M`````````````````0`````````!``````````$````````"4@`````````!
M`````````EP``````````0````````*?``````````$````````"O@``````
M```/````````!%<`````````&0```````9U``````````!L`````````"```
M```````:```````!G4@`````````'``````````(``````````0````````"
M2``````````%````````#N``````````!@````````5H``````````H`````
M```$H0`````````+`````````!@`````````%0````````````````````,`
M``````&?Z``````````"````````!;@`````````%``````````'````````
M`!<````````6H``````````'````````$X@`````````"`````````,8````
M``````D`````````&`````!O___[``````@`````````;___^0`````````7
M````````````````````````````````````````````````````````````
M``````````````````````````````````````````````````````&=6```
M`````9U0`````````````````````````````9U`````````````````````
M```````````````````!HP```````````````````9U(```````!HF@`````
M``````````````````````````````````&=6```````````````````````
M``````&BB``````````````````!H>@````````Z0```````````````````
M`````````````````````!Q@````````'&`````````<8````````!Q@````
M````'&`````````<8````````!Q@````````'&`````````<8````````!Q@
M````````'&`````````<8````````!Q@````````'&`````````<8```````
M`!Q@````````'&`````````<8````````!Q@````````'&`````````<8```
M`````!Q@````````'&`````````<8````````!Q@````````'&`````````<
M8````````!Q@````````'&`````````<8````````!Q@````````'&``````
M```<8````````!Q@````````'&`````````<8````````!Q@````````'&``
M```````<8````````!Q@````````'&`````````<8````````!Q@````````
M'&`````````<8````````!Q@````````'&`````````<8````````!Q@````
M````'&`````````<8````````!Q@````````'&`````````<8````````!Q@
M````````'&`````````<8````````!Q@````````'&`````````<8```````
M`!Q@```````!HH````````&A\`I005(N<&T*``````````````````!3P```
M`````%/(``````````````````!3T````````%/@````````4^@```````!3
M\````````%/X``````````````````!4`````````%0(````````````````
M``!7>$=#0SH@*$YE=$)31"!N8C(@,C`R,S`W,3`I(#$P+C4N,`!'0T,Z("A.
M971"4T0@;F(S(#(P,C,Q,#`X*2`Q,"XU+C``)$YE=$)31#H@8W)T,"Y3+'8@
M,2XR(#(P,3@O,3$O,C8@,3<Z,S<Z-#0@:F]E<F<@17AP("0`)$YE=$)31#H@
M8W)T,"UC;VUM;VXN8RQV(#$N,C<@,C`R,B\P-B\R,2`P-CHU,CHQ-R!S:W)L
M;"!%>'`@)``D3F5T0E-$.B!C<G1I+E,L=B`Q+C$@,C`Q-"\P."\Q,"`P-3HT
M-SHS-B!M871T($5X<"`D`"1.971"4T0Z(&-R=&)E9VEN+F,L=B`Q+C$W(#(P
M,3@O,3(O,C@@,3@Z,3<Z,3$@8VAR:7-T;W,@17AP("0`)$YE=$)31#H@8W)T
M96YD+E,L=B`Q+C$@,C`Q-"\P."\Q,"`P-3HT-SHS-B!M871T($5X<"`D```N
M<VAS=')T86(`+FEN=&5R<``N;F]T92YN971B<V0N:61E;G0`+FYO=&4N;F5T
M8G-D+G!A>``N:&%S:``N9'EN<WEM`"YD>6YS='(`+G)E;&$N9'EN`"YR96QA
M+G!L=``N=&5X=``N<F]D871A`"YE:%]F<F%M95]H9'(`+F5H7V9R86UE`"YI
M;FET7V%R<F%Y`"YF:6YI7V%R<F%Y`"YJ8W(`+F1Y;F%M:6,`+F=O=``N9V]T
M+G!L=``N9&%T80`N8G-S`"YC;VUM96YT````````````````````````````
M````````````````````````````````````````````````````````````
M````````````"P````$``````````@````````(``````````@``````````
M%P````````````````````$``````````````!,````'``````````(`````
M```"&`````````(8`````````!@````````````````````$````````````
M```F````!P`````````"`````````C`````````",``````````4````````
M````````````!```````````````-P````4``````````@````````)(````
M`````D@````````#(`````4```````````````@`````````!````#T````+
M``````````(````````%:`````````5H````````"7@````&`````P``````
M```(`````````!@```!%`````P`````````"````````#N`````````.X```
M``````2A`````````````````````0``````````````30````0`````````
M`@```````!.(````````$X@````````#&`````4```````````````@`````
M````&````%<````$`````````$(````````6H````````!:@````````!;@`
M```%````$P`````````(`````````!@```!<`````0`````````&````````
M'&`````````<8`````````/P````````````````````$``````````0````
M80````$`````````!@```````"!0````````(%`````````S6```````````
M`````````!```````````````&<````!``````````(```````!3J```````
M`%.H````````05<````````````````````(``````````````!O`````0``
M```````"````````E0````````"5``````````#\````````````````````
M!```````````````?0````$``````````@```````)8`````````E@``````
M```&C`````````````````````@``````````````(<````.``````````,`
M``````&=0````````)U```````````@````````````````````(````````
M``@```"3````#P`````````#```````!G4@```````"=2``````````(````
M````````````````"``````````(````GP````$``````````P```````9U0
M````````G5``````````"`````````````````````@``````````````*0`
M```&``````````,```````&=6````````)U8`````````>`````&````````
M```````(`````````!````"M`````0`````````#```````!GS@```````"?
M.`````````"P````````````````````"``````````(````L@````$`````
M`````P```````9_H````````G^@````````"``````````````````````@`
M````````"````+L````!``````````,```````&AZ````````*'H````````
M`(@````````````````````(``````````````#!````"``````````#````
M```!HH````````"B<`````````"(````````````````````0```````````
M````Q@````$`````````,```````````````````HG``````````1```````
M``````````````$``````````0```!\````!`````````#``````````````
M`````**T`````````1X````````````````````!``````````$````!````
M`P````````````````````````````"CT@````````#/````````````````
,`````0``````````
