// Copyright 2000-2018 JetBrains s.r.o.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.jetbrains.php;

import com.intellij.codeInsight.completion.PrefixMatcher;
import com.intellij.openapi.project.Project;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.util.Processor;
import com.intellij.util.indexing.ID;
import com.jetbrains.php.lang.psi.elements.*;
import com.jetbrains.php.lang.psi.resolve.types.PhpType;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Set;

public abstract class PhpIndex {

  @NotNull
  public static PhpIndex getInstance(@NotNull Project project) {
    return project.getComponent(PhpIndex.class);
  }

  /**
   * @param name "\the\namespace"
   */
  public abstract Collection<PhpNamespace> getNamespacesByName(String name);

  @NotNull
  public abstract Collection<String> getAllConstantNames(@Nullable PrefixMatcher prefixMatcher);

  @NotNull
  public abstract Collection<String> getAllVariableNames(@Nullable PrefixMatcher prefixMatcher);

  @NotNull
  public abstract Collection<String> getAllFunctionNames(@Nullable PrefixMatcher prefixMatcher);

  /**
   * @see #getAllClassFqns(PrefixMatcher)
   */
  @NotNull
  public abstract Collection<String> getAllClassNames(@Nullable PrefixMatcher prefixMatcher);

  /**
   * @param prefixMatcher if null, all fqns will be returned
   */
  @NotNull
  public abstract Collection<String> getAllClassFqns(@Nullable final PrefixMatcher prefixMatcher);

  /**
   * @param prefixMatcher if null, all fqns will be returned
   */
  @NotNull
  public abstract Collection<String> getAllInterfacesFqns(@Nullable PrefixMatcher prefixMatcher);

  /**
   * @return all interface names in lowercase
   */
  @NotNull
  public abstract Collection<String> getAllInterfaceNames();

  /**
   * @return all trait names in lowercase
   */
  @NotNull
  public abstract Collection<String> getAllTraitNames();

  @NotNull
  public abstract Collection<String> getChildNamespacesByParentName(@Nullable String name);

  @NotNull
  public abstract Collection<String> getAllChildNamespacesFqns(@NotNull String parentNamespaceFqn);

  @NotNull
  public abstract Collection<String> getTraitUsagesByFQN(@Nullable String name);

  @SuppressWarnings({"UnusedDeclaration"})
  public abstract Collection<PhpUse> getUseAliasesByName(@Nullable String name);

  @NotNull
  public abstract Collection<PhpUse> getUseAliasesByReferenceName(@Nullable String name);

  @NotNull
  public abstract Collection<Constant> getConstantsByFQN(@Nullable String fqn);

  @NotNull
  public abstract Collection<Constant> getConstantsByName(@Nullable String name);

  @NotNull
  public abstract Collection<Variable> getVariablesByName(@Nullable String name);

  @NotNull
  public abstract Collection<Function> getFunctionsByName(@Nullable String name);

  @NotNull
  public abstract Collection<Function> getFunctionsByFQN(@Nullable String fqn);

  @NotNull
  public abstract Collection<PhpClass> getInterfacesByName(@Nullable String name);

  public abstract Collection<PhpClass> getTraitsByName(String name);

  @NotNull
  public abstract Collection<PhpClass> getClassesByName(@Nullable String name);

  @NotNull
  public abstract Collection<PhpClass> getClassesByNameInScope(@Nullable String name, GlobalSearchScope scope);

  @Nullable
  public abstract PhpClass getClassByName(@Nullable String name);

  @NotNull
  public abstract Collection<PhpClass> getDirectSubclasses(@Nullable String fqn);

  @NotNull
  public abstract Collection<PhpClass> getAllSubclasses(@Nullable String fqn);

  /**
   * This scope should be used if results from php lib stubs are needed.
   */
  public abstract GlobalSearchScope getSearchScope();

  @NotNull
  public abstract Collection<? extends PhpNamedElement> getBySignature(@NotNull String s);

  @NotNull
  public abstract Collection<? extends PhpNamedElement> getBySignature(@NotNull String s, @Nullable Set<String> visited, int depth);

  @NotNull
  protected abstract Collection<? extends PhpNamedElement> getBySignatureInternal(@NotNull String s,
                                                                                  @Nullable Set<String> visited,
                                                                                  int depth);

  public abstract Collection<PhpClass> getClasses(@Nullable Set<String> visited, @NotNull String classRef);

  @NotNull
  public abstract Collection<PhpClass> getClassesByFQN(String fqn);

  @NotNull
  public abstract Collection<PhpClass> getInterfacesByFQN(String fqn);

  @NotNull
  public abstract Collection<PhpClass> getTraitsByFQN(String fqn);

  @NotNull
  public abstract Collection<PhpClass> getAnyByFQN(String fqn);

  @NotNull
  public abstract Collection<PhpClass> getCoveringTestClasses(@NotNull Project project, @NotNull String targetClassFqn);

  @NotNull
  public abstract Collection<Method> getCoveringTestMethods(@NotNull Project project, @NotNull Method method);

  public abstract Collection<PhpClass> getTraitUsages(PhpClass me);

  /**
   * @param me trait to look for
   * @param visited cyclic reference prevention
   * @return all traits using this, classes directly using it and their superclasses AND THEIR TRAITS ETC.
   * @deprecated SLOW!
   * @see processNestedTraitUsages
   */
  public abstract Collection<PhpClass> getNestedTraitUsages(PhpClass me, @Nullable Collection<String> visited);

  /**
   * processes all traits using this, classes directly using it and their superclasses AND THEIR TRAITS ETC..
   * @param me trait to look for
   * @param visited cyclic reference prevention
   */
  public abstract boolean processNestedTraitUsages(PhpClass me, @Nullable Collection<String> visited, Processor<? super PhpClass> processor);

  protected abstract Collection<String> filterKeys(Collection<String> keys, ID id);

  @NotNull
  public abstract <T extends PhpNamedElement> Collection<T> filterByNamespace(@NotNull Collection<T> elements,
                                                                              @Nullable String namespaceName,
                                                                              boolean allowGlobal);

  /**
   * @param elements collection to filter
   * @param namespaceName trailing '\' included!
   * @return filtered collection
   */
  @NotNull
  public abstract <T extends PhpNamedElement> Collection<T> filterByNamespace(@NotNull Collection<T> elements, @Nullable String namespaceName);

  public abstract PhpType completeType(@NotNull Project p, @NotNull PhpType type, @NotNull Set<String> visited);

  public abstract PhpType completeThis(@NotNull PhpType type,
                                       @Nullable String thisClass, final @NotNull Set<String> visited);

  @NotNull
  public abstract Collection<String> getClassAliasesNames();
}
