/* Copyright (C) 1998 Ulrich Drepper, <drepper@cygnus.com>.
   The GPL applies to this file.
   As a special restriction the file must not be used in this or a modified
   form on Microsoft and Be systems.  */

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/extensions/XShm.h>
#include "PlotP.h"

#define offset(field) XtOffsetOf(PlotRec, field)
static XtResource resources[] =
{
  { XtNforeground, XtCForeground, XtRPixel, sizeof (Pixel),
    offset (plot.foreground), XtRString, XtDefaultForeground },
  { XtNlow, XtClow, XtRFloat, sizeof (float),
    offset (plot.low), XtRFloat, "-1.0" },
  { XtNhigh, XtChigh, XtRFloat, sizeof (float),
    offset (plot.high), XtRFloat, "1.0" },
  { XtNmaxptr, XtCmaxptr, XtRPointer, sizeof (double *),
    offset (plot.maxerror), XtRPointer, NULL },
};


static void Initialize (Widget request, Widget new, ArgList args,
			Cardinal *num_args);
static void Redisplay (Widget gw, XEvent *event, Region region);

PlotClassRec plotClassRec =
{
  {
    /* core_class fields.  */
    /* superclass		*/	(WidgetClass) &simpleClassRec,
    /* class_name		*/	"Plot",
    /* widget_size		*/	sizeof (PlotRec),
    /* class_initialize		*/	NULL,
    /* class_part_initialize	*/	NULL,
    /* class_inited		*/	FALSE,
    /* initialize		*/	Initialize,
    /* initialize_hook		*/	NULL,
    /* realize			*/	XtInheritRealize,
    /* actions			*/	NULL,
    /* num_actions		*/	0,
    /* resources		*/	resources,
    /* num_resources		*/	XtNumber (resources),
    /* xrm_class		*/	NULLQUARK,
    /* compress_motion		*/	TRUE,
    /* compress_exposure	*/	TRUE,
    /* compress_enterleave	*/	TRUE,
    /* visible_interest		*/	FALSE,
    /* destroy			*/	NULL,
    /* resize			*/	NULL,
    /* expose			*/	Redisplay,
    /* set_values		*/	NULL,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	XtInheritSetValuesAlmost,
    /* get_values_hook		*/	NULL,
    /* accept_focus		*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private		*/	NULL,
    /* tm_table			*/	NULL,
    /* query_geometry		*/	NULL,
    /* display_accelerator	*/	XtInheritDisplayAccelerator,
    /* extension		*/	NULL
  },
  /* Simple class fields initialization.  */
  {
    /* change_sensitive		*/	XtInheritChangeSensitive
  },
  /* Content class fields initialization.  */
  {
    /* ignore			*/      0
  }
};
WidgetClass plotWidgetClass = (WidgetClass) &plotClassRec;


static int
InitializeShm (Widget request, Widget new, ArgList args, Cardinal *num_args)
{
  PlotWidget pw = (PlotWidget) new;

  pw->plot.image = XShmCreateImage
    (XtDisplay (pw),
     DefaultVisual (XtDisplay (pw), DefaultScreen (XtDisplay (pw))),
     1,
     XYBitmap,
     NULL,
     &pw->plot.shminfo,
     pw->core.width - 40, pw->core.height - 30);

  if (pw->plot.image == NULL)
    {
      pw->plot.shm = False;
      return 0;
    }

  pw->plot.shminfo.shmid = shmget (IPC_PRIVATE,
				   pw->plot.image->bytes_per_line
				   * pw->plot.image->height,
				   IPC_CREAT | 0777);
  if (pw->plot.shminfo.shmid == -1)
    {
      XDestroyImage (pw->plot.image);
      pw->plot.image = False;
      // fprintf (stderr, "cannot allocate shared memory\n");
      return 0;
    }

  pw->plot.shminfo.shmaddr
    = pw->plot.image->data = shmat (pw->plot.shminfo.shmid, 0, 0);
  pw->plot.shminfo.readOnly = True;

  if (XShmAttach (XtDisplay (pw), &pw->plot.shminfo) == 0)
    {
      XDestroyImage (pw->plot.image);
      shmdt (pw->plot.shminfo.shmaddr);
      shmctl (pw->plot.shminfo.shmid, IPC_RMID, 0);
      pw->plot.image = False;
      return 0;
    }

  return 1;
}

static void
Initialize (Widget request, Widget new, ArgList args, Cardinal *num_args)
{
  PlotWidget pw = (PlotWidget) new;
  XGCValues values;
  Font fn;

  /* See whether we can use shared memory for the pixmap.  */
  pw->plot.shm = 0; // XShmQueryExtension (XtDisplay (new));
  if (!pw->plot.shm
      || InitializeShm (request, new, args, num_args))
    {
      fprintf (stderr, "not using shm\n");
      pw->plot.image = XCreateImage
	(XtDisplay (pw),
	 DefaultVisual (XtDisplay (pw), DefaultScreen (XtDisplay (pw))),
	 1,
	 XYBitmap,
	 0,
	 NULL,
	 pw->core.width - 40, pw->core.height - 30,
	 8,
	 (pw->core.width + 7) / 8);
      if (pw->plot.image == NULL)
	{
	  fprintf (stderr, "cannot create image\n");
	  abort ();
	}
      pw->plot.image->data = calloc (pw->plot.image->bytes_per_line,
				     pw->plot.image->height);
      if (pw->plot.image->data == NULL)
	abort ();
    }

  /* Get the font.  */
  fn = XLoadFont (XtDisplay (new), "6x13");

  /* Create the GCs.  */
  values.function = GXcopy;
  values.foreground = BlackPixel (XtDisplay (new), DefaultScreen (XtDisplay (pw)));
  values.background = WhitePixel (XtDisplay (new), DefaultScreen (XtDisplay (pw)));
  values.font = fn;
  pw->plot.pixgc = XtGetGC (new, GCForeground | GCBackground | GCFunction | GCFont,
			    &values);

  pw->plot.drawgc = XtGetGC (new, 0, NULL);

  pw->plot.lastmax = *pw->plot.maxerror;
}


void
plot_update (PlotWidget pw, double *r, unsigned int n)
{
  double step;
  double count;
  double maxerror = *pw->plot.maxerror;
  char buf[10];

  if (pw->plot.lastmax != maxerror && r != NULL && n > 0)
    {
      long int cnt = n;

      /* Clear the bitmap and redraw it.  */
      pw->plot.lastmax = maxerror;

      XClearArea (XtDisplay (pw),
		  XtWindow (pw),
		  0, 0, pw->core.width, pw->core.height, True);

      memset (pw->plot.image->data, '\0',
	      pw->plot.image->bytes_per_line * pw->plot.image->height);

      while (--cnt >= 0)
        {
          double x = r[2 * cnt];
          double e = r[2 * cnt + 1];
          int xp;
          int yp;

          xp = ((x - pw->plot.low) * (pw->core.width - 40)
	        / (pw->plot.high - pw->plot.low));
          yp = (pw->core.height - 30) / 2 * (1.0 - e / maxerror);

          XPutPixel (pw->plot.image, xp, yp, 1);
        }
    }

  if (pw->plot.shm)
    XShmPutImage (XtDisplay (pw),
		  XtWindow (pw),
		  pw->plot.pixgc,
		  pw->plot.image,
		  0, 0,
		  35, 15,
		  pw->core.width - 40, pw->core.height - 30,
		  0);
  else
    XPutImage (XtDisplay (pw),
	       XtWindow (pw),
	       pw->plot.pixgc,
	       pw->plot.image,
	       0, 0,
	       35, 15,
	       pw->core.width - 40, pw->core.height - 30);

  XDrawRectangle (XtDisplay (pw),
		  XtWindow (pw),
		  pw->plot.drawgc,
		  34,
		  14,
		  pw->core.width - 38,
		  pw->core.height - 28);
  XDrawLine (XtDisplay (pw),
	     XtWindow (pw),
	     pw->plot.drawgc,
	     34, pw->core.height / 2,
	     pw->core.width - 19, pw->core.height / 2);
  XDrawString (XtDisplay (pw),
	       XtWindow (pw),
	       pw->plot.pixgc,
	       5, pw->core.height / 2 + 6,
	       " 0.0", 4);

  if (pw->plot.low < 0.0 && pw->plot.high > 0.0)
    {
      XDrawLine (XtDisplay (pw),
		 XtWindow (pw),
		 pw->plot.drawgc,
		 35 + ((pw->core.width - 40) * -pw->plot.low
		       / (pw->plot.high - pw->plot.low)),
		 15,
		 35 + ((pw->core.width - 40) * -pw->plot.low
		       / (pw->plot.high - pw->plot.low)),
		 pw->core.height - 15);
    }

  step = pw->plot.lastmax > 5.0 ? 1.0 : 0.5;
  for (count = step; count <= pw->plot.lastmax; count += step)
    {
      int n = snprintf (buf, sizeof (buf), "% .1f", count);
      XDrawString (XtDisplay (pw),
		   XtWindow (pw),
		   pw->plot.pixgc,
		   5,
		   21 + (pw->core.height - 30) / 2 * ((pw->plot.lastmax - count) / pw->plot.lastmax),
		   buf, n);

      XDrawLine (XtDisplay (pw),
		 XtWindow (pw),
		 pw->plot.drawgc,
		 34,
		 15 + (pw->core.height - 30) / 2 * (1.0 - (pw->plot.lastmax - count) / pw->plot.lastmax),
		 pw->core.width - 4,
		 15 + (pw->core.height - 30) / 2 * (1.0 - (pw->plot.lastmax - count) / pw->plot.lastmax));

      n = snprintf (buf, sizeof (buf), "% .1f", -count);
      XDrawString (XtDisplay (pw),
		   XtWindow (pw),
		   pw->plot.pixgc,
		   5,
		   21 + (pw->core.height - 30) / 2 * (2.0 - (pw->plot.lastmax - count) / pw->plot.lastmax),
		   buf, n);

      XDrawLine (XtDisplay (pw),
		 XtWindow (pw),
		 pw->plot.drawgc,
		 34,
		 15 + (pw->core.height - 30) / 2 * (1.0 + (pw->plot.lastmax - count) / pw->plot.lastmax),
		 pw->core.width - 4,
		 15 + (pw->core.height - 30) / 2 * (1.0 + (pw->plot.lastmax - count) / pw->plot.lastmax));
    }
}


static void
Redisplay (Widget gw, XEvent *event, Region region)
{
  PlotWidget pw = (PlotWidget) gw;

  plot_update (pw, NULL, 0);
}


void
plot (PlotWidget pw, double *r, unsigned long int from, unsigned long int to)
{
  long int cnt = to;
  while (--cnt >= 0)
    {
      double x = r[2 * cnt];
      double e = r[2 * cnt + 1];
      int xp;
      int yp;

      while (fabs (e) > *pw->plot.maxerror)
	*pw->plot.maxerror += 0.5;

      xp = ((x - pw->plot.low) * (pw->core.width - 40)
	    / (pw->plot.high - pw->plot.low));
      yp = (pw->core.height - 30) / 2 * (1.0 - e / *pw->plot.maxerror);

      XPutPixel (pw->plot.image, xp, yp, 1);
    }

  plot_update (pw, r, to);
}
