/*
// YAP2LC - an LDAP migration tool
//
// yap2lc.c
// License: GPL
// Author: Radu Negut <radun@romsys.ro>
// Version: 0.5.9 [23.12.2003]
// Section: main code, option processing,
//	    cli processing, workhorse section
*/

#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <pwd.h>
#include <sys/param.h>
#include <regex.h>
#if HAVE_SHADOW_H == 1
#include <shadow.h>
#endif
#include <sys/time.h>
#include <ctype.h>
#include "yap2lc.h"

extern int optind;
global_t global;

int
main(int argc, char *argv[])
{
    struct passwd *pwd;
#if HAVE_SHADOW_H == 1
    struct spwd *shadow;
#endif
#if defined(BSD)
    FILE *ldifp, *ltpl;
#else
    FILE *ldifp, *lpwd, *ltpl;
#endif
    struct timeval tval_start;
    struct timeval tval_end;
#if HAVE_SHADOW_H == 1
    FILE *lshd;
#endif
    int x = 0;
    int c;
    global.impdup = 0;
    global.pipelined = 0;
    global.malswitch = 0;
    global.dbsupport = 0;
#ifdef NDBM
    global.dbsupport++;
#endif
#ifdef BDB
    global.dbsupport += 2;
#endif
    if(argc > 1)
	{
	 while((c = getopt(argc, argv, "pclhd")) != -1)
	     {
	      switch(c)
	          {
	           case '-':
		       break;
		   case 'p':
		       global.pipelined++;
		       break;
		   case 'c':
		       global.pipelined += 2;
		       break;
		   case 'l':
		       global.pipelined += 4;
		       break;
		   case 'd':
		       global.pipelined += 8;
		       break;
		   case 'h':
		       usage();
		       exit(0);       
		   default:
		       printf("Wrong command line switches.\nTry 'yap2lc -h' for answers.\n\n");
		       usage();
		       exit(-1);    
		  }
	       }
	      if((global.pipelined % 2) != 1 || global.pipelined == 5)
	          {
		   printf("Something's missing from the command line switches.\nTry 'yap2lc -h' for answers.\n\n");
		   exit(-1);
		  }
	     }
    if(global.pipelined == 0)
	printf("\nCopyright 2003 by Radu Negut. Licensed under the terms of the GPL\n\n");
    if(fill_global() == -1)
	{
	 printf("Unable to read configuration file.\n");
	 return -1;
	}
    if(global.pipelined == 0 && (ltpl = fopen(global.ldftpl, "r")) == NULL)
	{
	 printf("Unable to open LDIF template. Check your conf.\n");
	 return -1;
	}
    if(global.pipelined == 0 && (ldifp = fopen(global.outfile, "w")) == NULL)
	{
	 printf("Unable to open ldif export outfile from yap2lc.conf.\n");
	 return -1;
	}
#if defined(BSD)
#else
    if((lpwd = fopen(global.locpasswd, "r")) == NULL)
	{
	 printf("Unable to open local passwd file. Check your conf.\n");
	 return -1;
	}
#endif
#if HAVE_SHADOW_H == 1
    if((lshd = fopen(global.locshadow, "r")) == NULL)
	{
	 printf("Unable to open local shadow file. Check your conf.\n");
	 return -1;
	}
#endif
    if(global.malswitch != 0)
	 yap2lc_dbopen();
    if((global.min_uid >= global.max_uid) && (global.max_uid != 0))
	{
	 printf("The min and max UIDs specified do not make sense. Check your conf.\n");
	 return -1;
	}
    if(global.pipelined == 0 || (global.pipelined > 4 && global.pipelined != 11 && global.pipelined != 9))
	{
	 if((global.logfp = fopen(global.logfile, "w+")) == NULL)
	     {
	      if((global.lognocn == 1 || global.lognopwd == 1 || global.logugids == 1 || global.logduplicates == 1) && \
	          (strlen(global.logfile) == 0))
	          {
	           if(global.pipelined == 0)
	               printf("Logging was requested, but no log file has been specified; using default, 'yap2lclog'.\n");
	           if((global.logfp = fopen("./yap2lclog", "w+")) == NULL)
	    		{
		         printf("Unable to create logfile. Exiting.\n");
		         return -1;
		        }
	          }
	      else 
	          if(global.pipelined == 0)
	              printf("No log file specified; none required.\n");
	     }
	}
    if(global.pipelined != 0)
	{
	 for(x = 0; x < (MAX_LDIF_SIZ - MAX_VAR_ALLOC); x++)
	     {
	      if((c = getc(stdin)) != EOF)
	           global.ldifbuf[x] = (char)c;
	      else
	          break;
	     }
	 global.ldifbuf[MAX_LDIF_SIZ - MAX_VAR_ALLOC] = '\0';
	}
    else
	{
	 fread(global.ldifbuf, sizeof(char), MAX_LDIF_SIZ - MAX_VAR_ALLOC, ltpl);
	 fclose(ltpl);
	}
    global.accstat = 1;
    gettimeofday(&tval_start, 0);
#if HAVE_SHADOW_H == 1
    while((pwd = fgetpwent(lpwd)) != NULL && (shadow = fgetspent(lshd)) != NULL)
	{
	 if(global.pipelined == 0)
	     printf(".");
	 if(global.sanitize == 1)
	    if(sanity_check(pwd, shadow) > 0)
	    	{
	         ++global.baddies;
	         continue;
		}
	 ldif_assembly(pwd, shadow, ldifp);
	 ++global.counter;
	}
#else
#if defined(BSD)
    if((int)geteuid() != 0)
	{
	 printf("On BSD systems we need to be root to get the 'shadowed' passwords. Exiting.\n");
	 return -1;
	}
    while((pwd = getpwent()) != NULL)
#else
    while((pwd = fgetpwent(lpwd)) != NULL)
#endif
	{
	 if(global.pipelined == 0)
	     printf(".");
	 if(global.sanitize == 1)
	    if(sanity_check(pwd) > 0)
		{
		 ++global.baddies;
		 continue;
		}
	 ldif_assembly(pwd, ldifp);
	 ++global.counter;
	}
#endif
    gettimeofday(&tval_end, 0);
    if(global.pipelined == 0)
	fclose(ldifp);
#if defined(BSD)
#else
    fclose(lpwd);
#endif
#if HAVE_SHADOW_H == 1
    fclose(lshd);
#endif
    if(global.pipelined == 0 || (global.pipelined > 4 && global.pipelined != 9 \
	&& global.pipelined != 11))
	fclose(global.logfp);
    if(global.regexpfilter == 1)
	free((void *)global.reg_struct);
    if(global.remduplicates == 1)
	unlink("./cnlist");
    if(global.malswitch != 0)
	yap2lc_dbclose();
    if(global.pipelined != 0)
	 return 0;
    if((u_long)((tval_end.tv_usec - tval_start.tv_usec) / 1000) < 1000)
	vanity_table((tval_end.tv_usec - tval_start.tv_usec) / 1000, MSECS);
    else
	vanity_table((tval_end.tv_sec - tval_start.tv_sec), SECS);
    return 0;
}

int
fill_global()
{
    FILE *fp;
    char *cp;
    char confline[MAX_CONFLINE_LEN], buf[10];
    int opt;
    int safe, x = 11, y = 0, z;
    global.sanitize = 1;
    if(global.pipelined == 1 || global.pipelined == 9 || global.pipelined == 11)
	return(pipe_defs());
    if((fp = fopen("./yap2lc.conf", "r")) == NULL)
	return -1;
    while((cp = fgets(confline, MAX_CONFLINE_LEN, fp)) != NULL)
	{
	 opt = ident_opt(confline);
	 switch(opt)
	    {
	     case 1:
	        strncpy(global.outfile, &confline[8], MAX_NAME_LEN);
		global.outfile[strlen(global.outfile) - 1] = '\0';
	        break;
	     case 2:
	        strncpy(global.ldftpl, &confline[13], MAX_NAME_LEN);
		global.ldftpl[strlen(global.ldftpl) - 1] = '\0';
		break;
	     case 3:
	        if((strncmp(&confline[15], "yes", 3)) == 0)
		    global.regexpfilter = 1;
		else
		    global.regexpfilter = 0;
	        break;
	     case 4:
	        global.reg_struct = regex_compiler(&confline[13]);
	        break;
	     case 5:
	        if((strncmp(&confline[17], "yes", 3)) == 0)
		    global.logregexp = 1;
		else
		    global.logregexp = 0;
	        break;
	     case 6:
	        if((strncmp(&confline[14], "uname", 5)) == 0)
		    global.regextarget = RGXUNAME;
		if((strncmp(&confline[14], "rname", 5)) == 0)
		    global.regextarget = RGXRNAME;
	        break;
	     case 7:
	        if((strncmp(&confline[18], "yes", 3)) == 0)
		    global.igngrperr =1;
		else
		    global.igngrperr = 0;
		break;
	     case 8:
	         strncpy(global.cryptscheme, &confline[12], MAX_NAME_LEN);
		 global.cryptscheme[strlen(global.cryptscheme) - 1] = '\0';
		 break;
	     case 9:
	         for(z = 0; z < 10; z++)
		     global.exgids[z] = -1;
	         while(confline[x] != '\n' || confline[x] == '\0')
		     {
		      if(y == 10)
		          break;
		      bzero(buf, 10);
		      safe = x + 10;
		      z = 0;
		      while(confline[x] != ',' && x < safe)
		          {
			   strncpy(&buf[z], &confline[x], 1);
			   ++z;
			   ++x;
			  }
		      sscanf(buf, "%d", &global.exgids[y]);
		      ++x;
		      ++y;
		     }
	         break;
	     case 10:
	         if((strncmp(&confline[8], "yes", 3)) == 0)
		     global.lognocn = 1;
		 else
		     global.lognocn = 0;
	         break; 
	     case 11:
	         if((strncmp(&confline[9], "yes", 3)) == 0)
		     global.lognopwd = 1;
		 else
		     global.lognopwd = 0;
	         break;
	     case 12:
	         if((strncmp(&confline[15], "yes", 3)) == 0)
		     global.logugids = 1;
		 else
		     global.logugids = 0;
	         break;
	     case 13:
	         if((strncmp(&confline[14], "yes", 3)) == 0)
		     global.logduplicates = 1;
		 else
		     global.logduplicates = 0;
	         break;
	     case 14:
	         if((strncmp(&confline[17], "yes", 3)) == 0)
		     global.remduplicates = 1;
		 else
		     global.remduplicates = 0;
		 break;
	     case 15:
	         strncpy(global.logfile, &confline[8], MAX_NAME_LEN);
		 global.logfile[strlen(global.logfile) - 1] = '\0';
		 break;
	     case 16:
	         if((strncmp(&confline[11], "yes", 3)) == 0)
		     global.allownopwd = 1;
		 else
		     global.allownopwd = 0;
		 break;
	     case 17:
	         if((strncmp(&confline[15], "yes", 3)) == 0)
		     global.allowlockedout = 1;
		 else
		     global.allowlockedout = 0;
		 break;
	     case 18:
	         if((strncmp(&confline[16], "yes", 3)) == 0)
		     global.idacstat = 1;
		 else
		     global.idacstat = 0;
	        break;
	     case 19:
	        break;
	     case 20:
	        if(global.pipelined >= 8)
		    break;
	        strncpy(global.locpasswd, &confline[12], MAX_NAME_LEN);
		global.locpasswd[strlen(global.locpasswd) - 1] = '\0';
	        break;
#if HAVE_SHADOW_H == 1
	     case 21:
	        if(global.pipelined >= 8)
		    break;
	        strncpy(global.locshadow, &confline[12], MAX_NAME_LEN);
		global.locshadow[strlen(global.locshadow) - 1] = '\0';
		break;
#endif
	     case 22:
	        if((strncmp(&confline[9], "no", 2)) == 0)
		    global.sanitize = 0;
	        break;
	     case 23:
	         sscanf(&confline[7], "%lu", &global.min_uid);
	         break;
	     case 24:
	         sscanf(&confline[7], "%lu", &global.max_uid);
	         break;
	     case 25:
		 strncpy(global.localiases, &confline[12], MAX_NAME_LEN);
		 global.localiases[strlen(global.localiases) - 1] = '\0';
	         break;
	     case 26:
	         if((strncmp(&confline[15], "yes", 3)) == 0)
		     global.malswitch = 1;
	         break;
	     case 27:
	         switch((char)confline[14])
		     {
		      case 'b':
		          global.malsdbtype = YAP2LC_DB_BDB;
		          break;
		      case 'n':
		          global.malsdbtype = YAP2LC_DB_NDBM;
		          break;
		      default:
		          global.malsdbtype = YAP2LC_DB_UNDEF;
			  break;
		     }
	         break;
	     case 28:
	         strncpy(global.malstring, &confline[17], MAX_NAME_LEN);
		 global.malstring[strlen(global.malstring) - 1] = '\0';
		 strcat(global.malstring, ": ");
	         break;
	     default:
	        break;
	    }
	}
    if(global.pipelined >= 13)
	pipe_defs();
    return 0;
}

int
pipe_defs(void)
{
    if(global.pipelined == 9 || global.pipelined >= 13)
	{
	 strncpy(global.locpasswd, "/etc/passwd", 11);
	 global.locpasswd[11] = '\0';
	 strncpy(global.locshadow, "/etc/shadow", 11);
	 global.locshadow[11] = '\0';
	 strncpy(global.localiases, "./aliases.db", 12);
	 global.localiases[12] = '\0';
	}
    else
	{
	 strncpy(global.locpasswd, "./passwd", 8);
	 global.locpasswd[8] = '\0';
	 strncpy(global.locshadow, "./shadow", 8);
	 global.locshadow[8] = '\0';
	 strncpy(global.localiases, "./aliases.db", 9);
	 global.localiases[9] = '\0';
	}
    strncpy(global.logfile, "./yap2lclog", 11);
    global.logfile[11] = '\0';
    if(global.pipelined < 13)
	global.sanitize = 0;
    return 0;
}

int
ident_opt(const char *line)
{
    if(line[0] == '#' || line[0] == ' ')
	return 19;
    if((strncmp(line, "outfile", 7)) == 0)
	return 1;
    if((strncmp(line, "localPasswd", 11)) == 0)
	return 20;
    if((strncmp(line, "ldifTemplate", 12)) == 0)
	return 2;
    if((strncmp(line, "removeByRegexp", 14)) == 0)
	return 3;
    if((strncmp(line, "filterRegexp", 12)) == 0)
	return 4;
    if((strncmp(line, "logRegexpRemoved", 16)) == 0)
	return 5;
    if((strncmp(line, "applyRegexpTo", 13)) == 0)
	return 6;
    if((strncmp(line, "ignoreGroupErrors", 17)) == 0)
	return 7;
    if((strncmp(line, "cryptScheme", 11)) == 0)
	return 8;
    if((strncmp(line, "excludeGID", 10)) == 0)
	return 9;
    if((strncmp(line, "logNoCN", 7)) == 0)
	return 10;
    if((strncmp(line, "logNoPwd", 8)) == 0)
	return 11;
    if((strncmp(line, "logExcludedGID", 14)) == 0)
	return 12;
    if((strncmp(line, "logDuplicates", 13)) == 0)
	return 13;
    if((strncmp(line, "removeDuplicates", 16)) == 0)
	return 14;
    if((strncmp(line, "logFile", 7)) == 0)
	return 15;
    if((strncmp(line, "allowNoPwd", 10)) == 0)
	return 16;
    if((strncmp(line, "allowLockedOut", 14)) == 0)
	return 17;
    if((strncmp(line, "idAccountStatus", 15)) == 0)
	return 18;
#if HAVE_SHADOW_H == 1
    if((strncmp(line, "localShadow", 11)) == 0)
	return 21;
#endif
    if((strncmp(line, "sanitize", 8)) == 0)
	return 22;
    if((strncmp(line, "minUID", 6)) == 0)
	return 23;
    if((strncmp(line, "maxUID", 6)) == 0)
	return 24;
    if((strncmp(line, "aliasesFile", 11)) == 0)
	return 25;
    if((strncmp(line, "processAliases", 14)) == 0)
	return 26;
    if((strncmp(line, "aliasesDBType", 13)) == 0)
	return 27;
    if((strncmp(line, "aliasesLdifEntry", 16)) == 0)
	return 28;
    return 200;
}

#if HAVE_SHADOW_H == 1
void
ldif_assembly(struct passwd *pwd, struct spwd *shadow, FILE *ldifp)
#else
void
ldif_assembly(struct passwd *pwd, FILE *ldifp)
#endif
{
    u_long pos, full;
    memset((void *)global.assembly, '\0', MAX_LDIF_SIZ);
    pos = strcspn(global.ldifbuf, "$");
    strncat(global.assembly, global.ldifbuf, pos);
#if HAVE_SHADOW_H == 1
    fillvar(pos, pwd, shadow);
#else
    fillvar(pos, pwd);
#endif
    full = pos;
    while((pos = strcspn(&global.ldifbuf[pos + 4], "$")) != 0)
	{
	 full += 4;
	 if(global.havemals != 0)
	    { /* this takes care of accounts without mail aliases */
	     full += global.havemals;
	     pos -= global.havemals;
	     global.havemals = 0;
	    }
	 strncat(global.assembly, &global.ldifbuf[full], pos);
	 full += pos;
	 pos = full;
#if HAVE_SHADOW_H == 1
	 fillvar(full, pwd, shadow);
#else
	 fillvar(full, pwd);
#endif
	}
    if(global.pipelined != 0)
	printf("%s\n", global.assembly);
    else
	fprintf(ldifp, "%s\n", global.assembly);
}

#if HAVE_SHADOW_H == 1
inline void
fillvar(u_long pos, struct passwd *pwd, struct spwd *shadow)
#else
inline void
fillvar(u_long pos, struct passwd *pwd)
#endif
{
    char buf[MAX_LDIF_NAME];
    int x;
    bzero(buf, MAX_LDIF_NAME);
    if(strncmp(&global.ldifbuf[pos], "$PWD", 4) == 0)
	{
	 strcat(global.assembly, global.cryptscheme);
#if HAVE_SHADOW_H == 1
	 strcat(global.assembly, shadow->sp_pwdp);
#else
	 strcat(global.assembly, pwd->pw_passwd);
#endif
	}
    if(strncmp(&global.ldifbuf[pos], "$RUN", 4) == 0)
	strcat(global.assembly, pwd->pw_gecos);
    if(strncmp(&global.ldifbuf[pos], "$UNM", 4) == 0)
	strcat(global.assembly, pwd->pw_name);
    if(strncmp(&global.ldifbuf[pos], "$UID", 4) == 0)
	{
	 sprintf(buf, "%d", (int)pwd->pw_uid);
	 strcat(global.assembly, buf);
	}
    if(strncmp(&global.ldifbuf[pos], "$GID", 4) == 0)
	{
	 sprintf(buf, "%d", (int)pwd->pw_gid);
	 strcat(global.assembly, buf);
	}
    if(strncmp(&global.ldifbuf[pos], "$HDR", 4) == 0)
	strcat(global.assembly, pwd->pw_dir);
    if(strncmp(&global.ldifbuf[pos], "$LSH", 4) == 0)
	strcat(global.assembly, pwd->pw_shell);
    if(strncmp(&global.ldifbuf[pos], "$YAS", 4) == 0 && global.idacstat == 1)
	{
	 if(global.accstat == 1)
	     strcat(global.assembly, "active");
	 else
	    {
	     strcat(global.assembly, "inactive");
	     global.accstat = 1;
	    }
	}
#if HAVE_SHADOW_H == 1
    if(strncmp(&global.ldifbuf[pos], "$SLC", 4) == 0)
	sprintf(&global.assembly[strlen(global.assembly) - 1], "%ld", shadow->sp_lstchg);
    if(strncmp(&global.ldifbuf[pos], "$SMN", 4) == 0)
	sprintf(&global.assembly[strlen(global.assembly) - 1], "%ld", shadow->sp_min);
    if(strncmp(&global.ldifbuf[pos], "$SMX", 4) == 0)
	sprintf(&global.assembly[strlen(global.assembly) - 1], "%ld", shadow->sp_max);
    if(strncmp(&global.ldifbuf[pos], "$SWR", 4) == 0)
	sprintf(&global.assembly[strlen(global.assembly) - 1], "%ld", shadow->sp_warn);
    if(strncmp(&global.ldifbuf[pos], "$SIN", 4) == 0)
	sprintf(&global.assembly[strlen(global.assembly) - 1], "%ld", shadow->sp_inact);
    if(strncmp(&global.ldifbuf[pos], "$SXP", 4) == 0)
	sprintf(&global.assembly[strlen(global.assembly) - 1], "%ld", shadow->sp_expire);
    if(strncmp(&global.ldifbuf[pos], "$SFG", 4) == 0)
	sprintf(&global.assembly[strlen(global.assembly) - 1], "%ld", shadow->sp_flag);
#endif
    if(strncmp(&global.ldifbuf[pos], "$MLS", 4) == 0 && global.malswitch == 1)
	 mals_assembly(pwd->pw_name, &global.ldifbuf[pos + 4], global.assembly);
}


