#include <unistd.h>
#include <fcntl.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <sys/select.h>
#include <sys/time.h>
#include <netdb.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "xtux.h"
#include "server.h"
#include "entity.h"
#include "clients.h"
#include "misc.h"
#include "sv_net.h"
#include "sv_netmsg_recv.h"
#include "sv_netmsg_send.h"

#define BACKLOG 2  /* Amount of pending connections we allow on sock */
/* Maximum times we will have read errors from a client before dropping them */
#define MAX_BAD 64

extern server_t server;
extern entity *root;
extern float sin_lookup[DEGREES];
extern float cos_lookup[DEGREES];

client_t *cl_root = NULL;
client_t *cl_tail = NULL;

/* Sock is used for listening for incoming connections */
static int sock;

static void sv_net_update(void);
static void sv_net_update_statusbar(client_t *cl);
static void sv_net_handle_client(client_t *cl);
static void sv_net_add_client(int fd);
static int sv_net_handle_message(client_t *cl, netmsg msg);
static void sv_net_add_client(int fd);
static int sv_net_handle_message(client_t *cl, netmsg msg);
static void sv_net_handle_client(client_t *cl);
static void sv_net_update_clients_entity(client_t *cl);

short sv_net_init(short port)
{
    struct sockaddr_in addr;
    int bound = 0;
    short pt;
    
    sock = net_init();
    memset((char*)&addr, 0, sizeof(addr));
    addr.sin_family = AF_INET;
    addr.sin_addr.s_addr = INADDR_ANY;

    /* Try to bind to "port", but if that fails, bind to the first free port */
    for( pt = port ; port ; pt++ ) {
	addr.sin_port = htons(pt);
	if (bind(sock, (struct sockaddr *) &addr, sizeof(addr)) < 0) {
	    perror("bind");
	    bound = 0;
	} else {
	    bound = 1;
	    break;
	}
    }
    
    if( !bound ) {
	fprintf(stderr, "Error, could not bind a free port!\n");
	return -1;
    }

    /* listen for new connections, allow BACKLOG pending */
    if (listen(sock, BACKLOG) < 0) {
	perror("listen");
	return -1;
    }
    
    /* Set server socket to nonblocking */
    fcntl(sock, F_SETFL, O_NONBLOCK);
    return pt; /* Port we are bound to */

}


void sv_net_get_client_input(void)
{
    client_t *cl, *next;
    int rfd;

    sv_net_update();
    for( cl = cl_root ; cl != NULL ; cl = next ) {
	next = cl->next;
	if( cl->incoming ) {
	    if( (rfd = net_buffered_read( cl->fd )) > 0 ) {
		sv_net_handle_client( cl );
		cl->incoming = 0;
	    } else if( rfd == 0 ) {
		if( ++cl->bad >= MAX_BAD )
		    sv_net_remove_client(cl);
		else
		    printf("Read error %d for %s\n", cl->bad, cl->name);
	    } else {
		/* printf("buffered read fd < 0\n"); */
		sv_net_remove_client(cl);
	    }
	}
	if( cl->status == ACTIVE )
	    sv_net_update_clients_entity( cl );
    }

}


void sv_net_send_start_frame(void)
{
    client_t *cl;
    netmsg msg;

    for( cl = cl_root ; cl != NULL ; cl = cl->next ) {
	if( cl->fd == -1 || cl->status != ACTIVE )
	    continue; /* Skip inactive clients */
	cl->screenpos = calculate_screenpos(cl);
	msg.type = NETMSG_START_FRAME;
	msg.start_frame.screenpos = cl->screenpos;
	net_send_message(cl->fd, msg);
    }

}


void sv_net_update_clients(void)
{
    netmsg msg;
    entity *ent;
    animation_t *ani;
    client_t *cl;

    for( cl = cl_root ; cl != NULL ; cl = cl->next ) {
	/* Skip inactive clients */
	if( cl->fd == -1 || cl->status != ACTIVE )
	    continue;

	sv_net_update_statusbar(cl);

	for( ent = root ; ent != NULL ; ent = ent->next ) {
	    if( ent->mode == LIMBO )
		continue;
	    /* Client can see their own invisible entity */
	    if( ent->visible == 0 && cl->ent != ent )
		continue;

	    ani = entity_animation(ent);

	    /* Skip if X is out of range */
	    if( ent->x + ani->img_w < cl->screenpos.x )
		continue;
	    if( ent->x > cl->screenpos.x + cl->view_w )
		continue;
	    /* Skip if Y is out of range */
	    if( ent->y + ani->img_h < cl->screenpos.y )
		continue;
	    if( ent->y > cl->screenpos.y + cl->view_h )
		continue;

	    msg = entity_to_netmsg(ent);
	    if( cl->ent == ent )
		msg.type = NETMSG_MYENTITY;
	    else
		msg.type = NETMSG_ENTITY;
	    net_send_message(cl->fd, msg);
	}
	/* Done with this client */
	msg.type = NETMSG_END_FRAME;
	net_send_message(cl->fd, msg);
    }

}


void sv_net_send_to_all(netmsg msg)
{
    client_t *cl;

    for( cl = cl_root ; cl != NULL ; cl = cl->next )
	if( cl->status == ACTIVE && cl->fd > 0 ) {
	    net_send_message( cl->fd, msg);
	}

}


void sv_net_send_text_to_all(char *message)
{
    netmsg msg;

    printf("%s\n", message);

    msg.type = NETMSG_TEXTMESSAGE;
    strcpy( msg.textmessage.sender, "SERVER");
    strncpy( msg.textmessage.string, message, TEXTMESSAGE_STRLEN);

    sv_net_send_to_all(msg);

}


void sv_net_tell_clients_to_changelevel(void)
{
    client_t *cl;
    netmsg msg;

    for( cl = cl_root ; cl != NULL ; cl = cl->next ) {
	sv_netmsg_send_changelevel(cl);

	msg.type = NETMSG_END_FRAME;
	net_send_message(cl->fd, msg);
	cl->status = JOINING; /* Clients will send ready notices when done */
    }

}


void sv_net_remove_client(client_t *cl)
{
    char buf[TEXTMESSAGE_STRLEN];

    server.clients--;
    snprintf(buf, TEXTMESSAGE_STRLEN, "%s disconnected (%d clients)",
	     cl->name, server.clients);
    close( cl->fd );

    if( cl->ent ) {
	entity_delete( cl->ent );
    }

    /* Make sure that root & Tail will still point at the
       right spot after cl is deleted */
    if( cl_root == cl )
	cl_root = cl->next;
    if( cl_tail == cl )
	cl_tail = cl->prev;

    if( cl->prev )
	cl->prev->next = cl->next;

    if( cl->next )
	cl->next->prev = cl->prev;

    free(cl);
    sv_net_send_text_to_all(buf);

}


static void sv_net_update(void)
{
    char buf[32];
    client_t *cl;
    fd_set read_fds;
    struct timeval tv;
    struct sockaddr_in naddr;
    int newsock;
    int addrlen; /* addrlen is passed to accept() and is changed to be the
		    length of the new address */

    FD_ZERO(&read_fds);

    /* If status of sock changes, then we have a new client wishing to join
       the server */
    FD_SET(sock, &read_fds);
    /* The fd's change status when there is new client data to be read */
    for( cl = cl_root ; cl != NULL ; cl = cl->next )
	FD_SET(cl->fd, &read_fds);
    
    tv.tv_sec = 0;
    tv.tv_usec = M_SEC / 20; /* 1/20 of a second */
    select(42, &read_fds, 0, 0, &tv); /* FIXME: 42 might not always work! */

    if( FD_ISSET(sock, &read_fds) ) {
	/* printf("FD activity on sock!\n"); */
	addrlen = sizeof(naddr);
	newsock = accept(sock,(struct sockaddr *)&naddr,(socklen_t *)&addrlen);
	if( newsock < 0 ) {
	    perror("accept");
	} else {
	    if( server.clients < server.max_clients )
		sv_net_add_client(newsock);
	    else {
		snprintf(buf, 32, "Server full (max=%d)", server.max_clients);
		sv_netmsg_send_rejection(newsock, buf);
		close(newsock);
	    }
	}
    }

    /* Read data from each ACTIVE client if it's fd status has changed */
    for( cl = cl_root ; cl != NULL ; cl = cl->next )
	if( FD_ISSET(cl->fd, &read_fds) )
	    cl->incoming = 1;

}


#define CHECK_CHANGE(a)       \
          if( cl-> a != cl->ent-> a ) { cl-> a = cl->ent-> a ; change = 1; } 

/* Checks to see if client cl needs it's statusbar updated */
static void sv_net_update_statusbar(client_t *cl)
{
    weap_type_t *wt;
    int change = 0;

    CHECK_CHANGE(health);
    CHECK_CHANGE(weapon);
    CHECK_CHANGE(frags);

    wt = weapon_type( cl->ent->weapon );
    if( cl->ammo != cl->ent->ammo[ wt->ammo_type ] ) {
	cl->ammo = cl->ent->ammo[ wt->ammo_type ];
	change = 1;
    }

    if( change )
	sv_netmsg_send_update_statusbar(cl);

}


static void sv_net_handle_client(client_t *cl)
{
    netmsg msg;

    while( (msg = net_get_message()).type != NETMSG_NONE )
	if( sv_net_handle_message(cl, msg) == NETMSG_QUIT )
	    break; /* If we get a quit, the rest are irrelevant */
}


/* Change entities velocity in game according to clients keypresses */
static void sv_net_update_clients_entity(client_t *cl)
{
    entity *ent;
    int accel;
    byte dir;

    ent = cl->ent; /* Saves typing */
    if( ent->mode < ALIVE )
	return; /* ent is not allowed to move */

    accel = entity_type(ent->type)->accel;

    if( cl->keypress & FORWARD_BIT ) {
	ent->x_v += sin_lookup[ ent->dir ] * accel;
	ent->y_v -= cos_lookup[ ent->dir ] * accel;
    }
	
    if( cl->keypress & BACK_BIT ) {
	ent->x_v -= sin_lookup[ ent->dir ] * accel;
	ent->y_v += cos_lookup[ ent->dir ] * accel;
    }
	
    if( cl->keypress & S_LEFT_BIT ) {
	dir = ent->dir - DEGREES / 4; /* 1/4 rotation LEFT */
	ent->x_v += sin_lookup[ dir ] * accel;
	ent->y_v -= cos_lookup[ dir ] * accel;
    }
    
    if( cl->keypress & S_RIGHT_BIT ) {
	dir = ent->dir + DEGREES / 4; /* 1/4 rotation RIGHT */
	ent->x_v += sin_lookup[ dir ] * accel;
	ent->y_v -= cos_lookup[ dir ] * accel;
    }

    /* Have entity (attempt to) fire it's weapon */
    if( cl->keypress & B1_BIT ) {
	ent->trigger = 1;
    } else
	ent->trigger = 0;
    
    if( cl->keypress & B3_BIT )
	ent->weapon_switch = 1; /* UP */
    else if( cl->keypress & B4_BIT )
	ent->weapon_switch = -1; /* DOWN */

}


/* Add the client that's been accept()-ed onto fd */
static void sv_net_add_client(int fd)
{
    client_t *cl;

    if( (cl = (client_t *)malloc(sizeof(client_t))) == NULL ) {
	perror("Malloc");
	fprintf(stderr, "Couldn't allocate new client!\n");
	sv_netmsg_send_rejection(fd, "memory error");
	close(fd);
	return;
    }

    /* Add client to the list */
    if( cl_root == NULL )
	cl_root = cl;

    if( cl_tail != NULL )
	cl_tail->next = cl;
    cl->prev = cl_tail;
    cl->next = NULL;
    cl_tail = cl;

    fcntl( fd, F_SETFL, O_NONBLOCK);
    cl->fd = fd;
    cl->status = JOINING;
    cl->incoming = 0;
    cl->bad = 0;
    cl->ent = NULL;

    /* Display info about the new client */
    printf("Added client from \"%s\" (%d clients)\n",
	   net_get_address(fd), server.clients);

}


/*
  Function pointer table for handling incoming messages.
  All of the below functions are in "sv_netmsg_recv.c" and have
  the form "int function(client_t *cl, netmsg msg)" (arg1 = pointer to
  client_t, arg2 = netmsg, returning void).
  We shouldn't get the messages that are NOTHANDLED, because
  well behaved clients shouldn't send them.
*/


static int sv_netmsg_not_handled(client_t *cl, netmsg msg)
{
    printf("NOT HANDLING MESSAGE \"%s\" from \"%s\" (%s)\n",
	   net_message_name(msg.type), cl->name, net_get_address(cl->fd));

    return 0;

}

#define NOTHANDLED sv_netmsg_not_handled
static int (*recv_message[NUM_NETMESSAGES])(client_t *,  netmsg) = {
    NOTHANDLED, /* NETMSG_NONE */
    sv_netmsg_recv_noop,
    sv_netmsg_recv_query_version,
    sv_netmsg_recv_version,
    sv_netmsg_recv_textmessage,
    sv_netmsg_recv_quit,
    NOTHANDLED, /* NETMSG_REJECTION */
    NOTHANDLED, /* NETMSG_SV_INFO */
    NOTHANDLED, /* NETMSG_CHANGELEVEL */
    NOTHANDLED, /* NETMSG_START_FRAME */
    NOTHANDLED, /* NETMSG_END_FRAMS */
    NOTHANDLED, /* NETMSG_ENTITY */
    NOTHANDLED, /* NETMSG_MYENTITY */
    NOTHANDLED, /* NETMSG_RAILSLUG */
    NOTHANDLED, /* NETMSG_UPDATE_STATUSBAR */
    sv_netmsg_recv_join,
    sv_netmsg_recv_ready,
    sv_netmsg_recv_query_sv_info,
    sv_netmsg_recv_cl_update,
    NOTHANDLED, /* NETMSG_GAMEMESSAGE */
};


/* Handle message and return the message type (-1 on error) */
static int sv_net_handle_message(client_t *cl, netmsg msg)
{

    if( msg.type < NUM_NETMESSAGES ) {
	if( recv_message[msg.type] == NULL )
	    sv_netmsg_not_handled(cl, msg);
	else if( recv_message[msg.type](cl, msg) < 0 )
	    return NETMSG_QUIT; /* Something went wrong */

	return msg.type;
    } else {
	printf("sv_netmsg_recv: Unknown message type %d from %s!\n",
	       msg.type, cl->name);
	return -1;
    }

}


