#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "xtux.h"
#include "server.h"
#include "entity.h"
#include "world.h"
#include "sv_map.h"
#include "clients.h"
#include "sv_netmsg_send.h"
#include "event.h"
#include "game.h"

extern map_t *map;
extern game_t game;
extern float sin_lookup[DEGREES];
extern float cos_lookup[DEGREES];

/* Each event read in from the map (section EVENT) is identified by a symbol
   (printable character). Multiple map tiles (EVENTDATA) can have the same
   event value. event_symbol_table keeps track of these, which are used in
   read_event_data() to fill in the event map layer with the number of the
   event, that is looked up in the event_tab[] array. */
static char event_symbol_table[256];
static int events = 0;
static event_t *event_tab; /* malloc'd to size events */

/* Delete all current events */
void event_reset(void)
{

    if( events ) {
	free(event_tab);
	events = 0;
    }
    memset(event_symbol_table, 0, 256);

}


static char *event_name[NUM_EVENTS] = {
    "NONE",
    "MESSAGE",
    "TELEPORT",
    "ENDLEVEL",
    "DAMAGE"
};


static int event_trigger_none(entity *ent, event_t *event)
{
    return 0;
}

static int event_trigger_message(entity *ent, event_t *event)
{

    if( ent->cl )
	sv_netmsg_send_gamemessage(ent->cl->fd, event->data.message.text, 0);

    return 0;
}

static int event_trigger_teleport(entity *ent, event_t *event)
{

    printf("Teleporting to %d,%d\n",
	   event->data.teleport.x, event->data.teleport.y);
    ent->x = event->data.teleport.x * TILE_W;
    ent->y = event->data.teleport.y * TILE_H;
    ent->dir = event->data.teleport.dir;
    ent->speed = event->data.teleport.speed;
    ent->x_v = sin_lookup[ent->dir] * ent->speed;
    ent->y_v = -cos_lookup[ent->dir] * ent->speed;
    entity_spawn_effect(ent);
    return 1;

}

static int event_trigger_damage(entity *ent, event_t *event)
{
    ent->health -= event->data.damage.damage;
    return 0;
}

static int event_trigger_endlevel(entity *ent, event_t *event)
{

    strcpy(game.next_map_name, event->data.endlevel.map);
    game.changelevel = 1;
    return 0;
}


static int (*event_trigger_handle[NUM_EVENTS])(entity *, event_t *) = {
    event_trigger_none,
    event_trigger_message,
    event_trigger_teleport,
    event_trigger_endlevel,
    event_trigger_damage
};



int event_trigger(entity *ent, byte id)
{
    event_t *event;

    if( id < events ) {
	event = &event_tab[id];

	if( ent->class != GOODIE || event->active == 0 ) {
	    return 0; /* Doesn't trigger this event */
	}

	/* Trigger once only if no repeat? */
	event->active = event->repeat;

	printf("EVENT %d, type %s triggered\n",
	       id, event_name[ event->type ]);

	/* event_trigger_handle's return 1 if entity has moved */
	return event_trigger_handle[event->type](ent, event);
    }

    return 0;

}


typedef struct tmp_node_struct {
    struct tmp_node_struct *next;
    event_t event;
} tmp_node;

tmp_node *event_root, *event_tail;

static event_t *event_new(void)
{
    tmp_node *ptr;

    if( (ptr = (tmp_node *)malloc( sizeof(tmp_node) )) == NULL ) {
	perror("malloc");
	ERR_QUIT("Allocating new tmp node for an event", -1);
    }

    /* Add it to the list */
    if( event_root == NULL )
	event_root = ptr; /* The new event_root entity */

    if( event_tail != NULL )
	event_tail->next = ptr;

    event_tail = ptr;
    event_tail->next = NULL; /* end of list */

    events++;

    return &ptr->event;

}


/* Setup the event layer to point to the right events */
int event_setup_layer(void)
{
    tmp_node *ptr, *next;
    int i;

    if( events <= 0 )
	return 0; /* No events */

    events++; /* We want an EVENT_NONE at the start */

    if( (event_tab = malloc( sizeof(event_t) * events )) == NULL )
	ERR_QUIT("Error allocating event_tab[] array", -1);

    event_tab[0].type = EVENT_NONE;
    event_tab[0].active = 0;
    event_tab[0].repeat = 0;

    ptr = event_root;
    for( i=1 ; ptr != NULL ; i++ ) {
	next = ptr->next;
	event_tab[i] = ptr->event;
	free(ptr);
	ptr = next;
    }

    event_root = NULL;
    event_tail = NULL;

    printf(" id  |     name     | A | R | V |\n");
    for(i=0 ; i<events ;i++ ) {
	printf(" %3d | %12s | %c | %c | %c |\n",
	       i, event_name[ event_tab[i].type ],
	       event_tab[i].active? 'y' : 'n',
	       event_tab[i].repeat? 'y' : 'n',
	       event_tab[i].visible? 'y' : 'n');
    }

    for(i=0 ; i<map->width * map->height ; i++ )
	map->event[i] = event_symbol_table[ (int)map->event[i] ];

    return 1;

}


void event_send_initial_effects(client_t *cl)
{
    netmsg msg;
    int x, y;
    byte e;

    if( map->event == NULL )
	return;

    for( y=0 ; y<map->height ; y++ ) {
	for( x=0 ; x<map->width ; x++ ) {
	    if( (e = map->event[ y*map->width + x ]) ) {
		if( !event_tab[e].active || !event_tab[e].visible )
		    continue;
		if( event_tab[e].type == EVENT_TELEPORT ) {
		    msg.type = NETMSG_PARTICLES;
		    msg.particles.effect = P_TELEPORT;
		    msg.particles.color1 = COL_GRAY;
		    msg.particles.color2 = COL_YELLOW;
		    msg.particles.length = TILE_W/2;
		    msg.particles.x = x * TILE_W + TILE_W/2;
		    msg.particles.y = y * TILE_H + TILE_H/2;
		    net_send_message(cl->fd, msg);
		} else if( event_tab[e].type == EVENT_ENDLEVEL ) {
		    msg.type = NETMSG_PARTICLES;
		    msg.particles.effect = P_TELEPORT;
		    msg.particles.color1 = COL_PINK;
		    msg.particles.color2 = COL_PURPLE;
		    msg.particles.length = TILE_W/2;
		    msg.particles.x = x * TILE_W + TILE_W/2;
		    msg.particles.y = y * TILE_H + TILE_H/2;
		    net_send_message(cl->fd, msg);
		}

	    }
	}
    }

}





static void event_new_none(event_t *event, const char *args)
{

}


static void event_new_message(event_t *event, const char *args)
{

    strncpy(event->data.message.text, args, TEXTMESSAGE_STRLEN);

}

static void event_new_teleport(event_t *event, const char *args)
{
    int x, y, dir, speed;

    if( sscanf(args, "%d %d %d %d", &x, &y, &dir, &speed) == 4 ) {
	event->data.teleport.x = x;
	event->data.teleport.y = y;
	event->data.teleport.dir = dir;
	event->data.teleport.speed = speed;
    } else {
	printf("Error parsing teleport event\n");
    }

}

static void event_new_endlevel(event_t *event, const char *args)
{

    strncpy(event->data.endlevel.map, args, NETMSG_STRLEN);

}

static void event_new_damage(event_t *event, const char *args)
{
    int damage;

    if( sscanf(args, "%d", &damage) == 1 ) {
	event->data.damage.damage = damage;
    } else {
	printf("Error parsing damage object\n");
    }

}


static void (*event_handle[NUM_EVENTS])(event_t *event, const char *args) = {
    event_new_none,
    event_new_message,
    event_new_teleport,
    event_new_endlevel,
    event_new_damage
};


void read_event(const char *line)
{
    event_t *event;
    int i, len, event_type;
    const char *args;
    char c, a, r, v; /* id character, active, repeat, visible */

    event_type = -1;
    for(i=0 ; i<NUM_EVENTS; i++ ) {
	len = strlen(event_name[i]);
	if( !strncasecmp(line, event_name[i], len) ) {
	    args = line + len + 1; /* Args are 1 space after the event name */
	    /* Get symbol representation for event */
	    if( sscanf(args, "%c", &c) != 1 ) {
		break;
	    }
	    args += 2; /* Move past the character */

	    /* Event initially active setting (y = true) */
	    if( sscanf(args, "%c", &a) != 1 ) {
		break;
	    }
	    args += 2; /* Move past the character */

	    /* Event trigger repeat setting (y = true) */
	    if( sscanf(args, "%c", &r) != 1 ) {
		break;
	    }
	    args += 2; /* Move past the character */

	    /* Event trigger visible? (y = true) */
	    if( sscanf(args, "%c", &v) != 1 ) {
		break;
	    }
	    args += 2; /* Move past the character */

	    /* Make a new event, add it's symbol to the symbol table and
	       pass the rest of the line off to the individual routines to
	       fill in the rest of the event */
	    event = event_new();
	    event->type = i;
	    event->active = (a == 'y');
	    event->repeat = (r == 'y');
	    event->visible = (v == 'y');
	    event_symbol_table[(int)c] = events;
	    event_handle[i](event, args);
	    break;
	}
    }

}
