#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

#include "xtux.h"
#include "server.h"
#include "entity.h"
#include "clients.h"
#include "sv_map.h"
#include "weapon.h"
#include "world.h"
#include "sv_net.h"
#include "ai.h"
#include "item.h"
#include "game.h"

extern server_t server;
extern game_t game;
extern byte num_weapon_types;

/* Start and end of entity list, initially NULL */
entity *root;
entity *tail;

int num_entities;
byte num_entity_types;

/* Keeps track of how entities of each class there are */
int class_population[NUM_ENT_CLASSES];

static int id_num = 1;

static void entity_update_movement(float secs);
static void entity_check_collisions(void);
static void entity_animate(void);
static void entity_die(entity *ent);
static entity *entity_alloc(void);


/* Called once at program initialisation */
void entity_init(void)
{
    num_entity_types = entity_type_init();
}


void entity_drip(entity *ent, byte color1, byte color2)
{
    ent_type_t *et;
    netmsg msg;
    
    et = entity_type(ent->type);

    msg.type = NETMSG_PARTICLES;
    msg.particles.effect = P_DRIP;
    msg.particles.dir = ent->dir;
    msg.particles.length = 1;
    msg.particles.color1 = color1;
    msg.particles.color2 = color2;
    /* Draw with offsets to make it look good on the client */
    msg.particles.x = ent->x + ent->width / 2;
    msg.particles.y = ent->y + ent->height / 2;
    sv_net_send_to_all(msg);

}


void entity_spawn_effect(entity *ent)
{
    netmsg msg;
    ent_type_t *et;

    et = entity_type(ent->type);

    /* Send respawning effect to clients */
    msg.type = NETMSG_PARTICLES;
    msg.particles.effect = P_SPAWN;
    msg.particles.color1 = COL_WHITE;
    msg.particles.color2 = COL_GRAY;
    msg.particles.length = 2 * et->width / 3;
    msg.particles.x = ent->x + et->width/2;
    msg.particles.y = ent->y + et->height/2;
    sv_net_send_to_all(msg);

}

/* How often the entity drips while invisible */
#define INVISIBLE_DRIP_TIME 500

void entity_update(float secs)
{
    ent_type_t *et;
    entity *ent, *next;
    msec_t now, time;
    int i;

    now = gettime();
    for( ent = root ; ent != NULL ; ent = ent->next ) {
	if( ent->dripping ) {
	    if( (et = entity_type(ent->type)) == NULL )
		continue;
	    time = (ent->dripping==1)? et->drip_time : INVISIBLE_DRIP_TIME;
	    if( now - ent->last_drip >= time ) {
		entity_drip(ent, ent->drip1, ent->drip2);
		ent->last_drip = now;
	    }
	}

	/* Respawn items */
	if( ent->mode == LIMBO )
	    if( now >= ent->respawn_time ) {
		ent->mode = ALIVE;
		ent->respawn_time = 0;
		entity_spawn_effect(ent);
	    }

	if( ent->powerup ) {
	    /* Entity is wounded */
	    if( ent->powerup & (1<<PU_WOUND) )
		if( now - ent->last_wound >= M_SEC ) {
		    entity_drip(ent, COL_RED, COL_RED);	/* Drip blood */
		    ent->health -= 2;
		    ent->last_wound = now;
		}

	    if( ent->powerup & (1<<PU_INVISIBILITY) ) {
		ent->visible = 0;
		if( ent->x_v || ent->y_v ) { /* Drip if moving */
		    ent->dripping = 2;
		    ent->drip1 = ent->color1;
		    ent->drip2 = ent->color2;
		} else {
		    if( (et = entity_type(ent->type)) == NULL )
			continue;
		    ent->dripping = et->dripping;
		    ent->drip1 = et->drip1;
		    ent->drip2 = et->drip2;
		}
	    }

	    for( i=0 ; i<NUM_POWERUPS ; i++ ) {
		if( ent->powerup & (1<<i) && now > ent->powerup_expire[i] ) {
		    if( i == PU_INVISIBILITY ) {
			if( (et = entity_type(ent->type)) == NULL )
			    continue;
			ent->dripping = et->dripping;
			ent->drip1 = et->drip1;
			ent->drip2 = et->drip2;
			ent->visible = 1;
		    } else if( i == PU_FROZEN ) {
			if( ent->mode == FROZEN )
			    ent->mode = ALIVE;
		    }
		    printf("powerup %d expired.\n", i);
		    ent->powerup &= ~(1<<i);
		    ent->powerup_expire[i] = 0;
		}
	    }
	}
    }

    entity_update_movement(secs);
    entity_check_collisions();
    entity_animate();
 
    for( ent = root ; ent != NULL ; ent = next ) {
	next = ent->next;
	if( ent->mode >= FROZEN && ent->health <= 0 )
	    entity_die(ent); /* Set to either dead or dying */

	if( ent->mode == DEAD ) {
	    if( ent->controller == CTRL_CLIENT )
		spawn(ent, 1); /* Respawn */
	    else
		entity_delete(ent);
	}
    }

}


static void entity_update_movement(float secs)
{
    ent_type_t *et;
    entity  *ent;

    for( ent = root ;  ent != NULL ; ent = ent->next ) {
	if( ent->mode >= ALIVE && ent->ai && ent->controller == CTRL_AI ) {
	    switch( ent->ai ) {
	    case AI_NONE:
		break;
	    case AI_FIGHT:
		if( ent->weapon ) {
		    /*
		    printf("Ent %s has weapon type %s\n", entity_type(ent->type)->name,
			   weapon_type(ent->weapon)->name);
		    */
		    ai_shooter_think(ent);
		} else
		    ai_fight_think(ent);
		break;
	    case AI_FLEE:
		ai_flee_think(ent);
		break;
	    case AI_SEEK:
		ai_seek_think(ent);
		break;
	    case AI_ROTATE:
		ai_rotate_think(ent);
		break;
	    default:
		break;
	    }
	    ai_move(ent);
	}

	if( ent->class != PROJECTILE )
	    world_friction(ent, secs);

	/* Keep speed in check */
	et = entity_type(ent->type);
	if( ent->x_v > 0 ) {
	    if( ent->x_v > et->speed )
		ent->x_v = et->speed;
	} else if( ent->x_v < 0 ) {
	    if( ent->x_v < -et->speed )
		ent->x_v = -et->speed;
	}
	
	if( ent->y_v > 0 ) {
	    if( ent->y_v > et->speed )
		ent->y_v = et->speed;
	} else if( ent->y_v < 0 ) {
	    if( ent->y_v < -et->speed )
		ent->y_v = -et->speed;
	}

	world_check_entity_move(ent, secs);

    }

}




/*

    Here's the order (from ../common/entity_type.h). You only need
    to handle collision between first and the second->class's that
    come after first's->class.

    GOODIE,
    BADDIE,
    NEUTRAL,
    ITEM,
    PROJECTILE,
    NUM_ENT_CLASSES
*/


static void col_goodie(entity *goodie, entity *second)
{
    ent_type_t *et;
    entity *shooter;
    point pt;

    switch( second->class ) {
    case GOODIE:
	/* Nothing.... */
	break;
    case BADDIE:
	et = entity_type( second->type );
	/* Touch damage is delt over a second */
	goodie->health -= et->touchdamage / server.fps;
	break;
    case NEUTRAL:
	/* Nothing... */
	break;
    case ITEM:
	item_collision(second, goodie);
	break;
    case PROJECTILE:
	shooter = findent(second->pid);
	pt.x = second->x;
	pt.y = second->y;
	weapon_hit( shooter, goodie, pt, second->weapon );
	second->health = 0; /* Remove projectile */
	break;
    default:
	printf("NOT HANDLED!\n");
    }

}


static void col_baddie(entity *baddie, entity *second)
{
    entity *shooter;
    point pt;

    switch( second->class ) {
    case BADDIE:
	/* Nothing.... */
	break;
    case NEUTRAL:
	/* Nothing.... */
	break;
    case ITEM:
	/* item_collision(second, baddie); */
	break;
    case PROJECTILE:
	shooter = findent(second->pid);
	pt.x = second->x;
	pt.y = second->y;
	weapon_hit( shooter, baddie, pt, second->weapon);
	second->health = 0; /* Remove projectile */
	break;
    default:
	printf("NOT HANDLED!\n");
    }

}


static void col_neutral(entity *neutral, entity *second)
{
    entity *shooter;
    point pt;

    switch( second->class ) {
    case NEUTRAL:
	break;
    case ITEM:
	/* Nothing.... */
	break;
    case PROJECTILE:
	shooter = findent(second->pid);
	pt.x = second->x;
	pt.y = second->y;
	weapon_hit( shooter, neutral, pt, second->weapon);
	second->health = 0; /* Remove projectile */
	break;
    default:
	printf("NOT HANDLED!\n");
    }

}


static void col_item(entity *item, entity *second)
{

    switch( second->class ) {
    case ITEM:
	/* Nothing.... */
	break;
    case PROJECTILE:
	/* Nothing.... */
	break;
    default:
	printf("NOT HANDLED!\n");
    }

}


/* In theory this should only be called with both first and second
   being projectiles */
static void col_projectile(entity *proj, entity *second)
{
    entity *shooter;
    point pt;

    switch( second->class ) {
    case PROJECTILE:
	/* Projectile hits second */
	if( (shooter = findent(proj->pid)) != NULL ) {
	    pt.x = proj->x;
	    pt.y = proj->y;
	    weapon_hit( shooter, second, pt, proj->weapon);
	}

	/* Second hits projectile */
	if( (shooter = findent(second->pid)) != NULL ) {
	    pt.x = second->x;
	    pt.y = second->y;
	    weapon_hit( shooter, proj, pt, second->weapon);
	}

	break;
    default:
	printf("NOT HANDLED!\n");
    }

}


static void (*handle_collision[NUM_ENT_CLASSES])(entity *,  entity *) = {
    col_goodie,
    col_baddie,
    col_neutral,
    col_item,
    col_projectile
};


/* FIXME: Optimise and make correct (use intersection tests). */
void entity_check_collisions(void)
{
    entity *ent1, *ent2;
    entity *first, *second;
    int d1, d2;

    for( ent1 = root ; ent1 != NULL ; ent1 = ent1->next ) {
	d1 = MAX( ent1->width, ent1->height );
	if( ent1->mode < FROZEN || ent1->health <= 0 )
	    continue; /* Skip LIMBO, DEAD & DYING entities */
	for( ent2 = ent1->next ; ent2 != NULL ; ent2 = ent2->next ) {
	    if( ent1->class == NEUTRAL && ent2->class == NEUTRAL )
		continue;
	    if( ent2->mode < FROZEN || ent1->health <= 0 )
		continue; /* Skip LIMBO, DEAD & DYING entities */
	    if( ent1->pid == ent2->id || ent2->pid == ent1->id )
		continue; /* Don't clip on parent */
	    d2 = MAX( ent2->width, ent2->height );
	    d1 = (d1 + d2) / 2;
	    if( entity_dist(ent1, ent2) < d1 ) {
		/* Call the function for the "lowest" class (in the
		   entity-class enumeration) with args (lowest, highest) */
		if( ent1->class < ent2->class ) {
		    first = ent1;
		    second = ent2;
		} else {
		    first = ent2;
		    second = ent1;
		}
		handle_collision[first->class]( first, second );
	    }
	}
    }

}


/* Creates a new entity on the end of the list of type "type" and returns
   a pointer to the newly created entity */
entity *entity_new(byte type, float x, float y, float x_v, float y_v)
{
    ent_type_t *et;
    entity *ent;

    /* Create entity */
    if( (ent = entity_alloc()) == NULL)
	ERR_QUIT("Error creating new entity", EXIT_FAILURE);

    memset( ent, 0, sizeof(entity) );

    /* Add it to the list */
    if( root == NULL )
	root = ent; /* The new root entity */

    if( tail != NULL )
	tail->next = ent;

    ent->prev = tail;
    tail = ent;
    tail->next = NULL;

    num_entities++;
    id_num++;

    ent->id = id_num;
    ent->x = x;
    ent->y = y;
    ent->x_v = x_v;
    ent->y_v = y_v;

    /* Set values from default type for this particular entity */
    et = entity_type(type);
    ent->type = type;
    ent->class = et->class;
    ent->mode = ALIVE;
    ent->speed = et->speed;
    ent->width = et->width;
    ent->height = et->height;
    ent->health = et->health;
    ent->dripping = et->dripping;
    ent->drip1 = et->drip1;
    ent->drip2 = et->drip2;
    ent->color1 = COL_WHITE;
    ent->color2 = COL_BLUE;
    ent->controller = CTRL_AI;
    ent->visible = 1;

    if( num_weapon_types > 0 ) {
	if( (ent->has_weapon = (char *)malloc(num_weapon_types)) == NULL ) {
	    perror("Malloc");
	    ERR_QUIT("Error malloc'ing has_weapon array", -1);
	}
	weapon_reset(ent);
    } else {
	ERR_QUIT("num_weapon_types out of range!\n", num_weapon_types);
    }

    memset(ent->powerup_expire, 0, sizeof(ent->powerup_expire));

    if( et->ai )
	ent->ai = et->ai;
    else { /* Default AI's for each class */
	switch( et->class ) {
	case GOODIE:
	case BADDIE:
	    ent->ai = AI_FIGHT;
	    break;
	case NEUTRAL:
	    ent->ai = AI_FLEE;
	    break;
	case PROJECTILE:
	    ent->ai = AI_NONE;
	    break;
	case ITEM:
	    if( et->item_action == GIVEWEAPON )
		ent->ai = AI_ROTATE;
	    break;
	}
    }

    class_population[ ent->class ]++;

    /*
      printf("created new %s (type %d, class %d) %d entities\n",
      et->name, type, ent->class, num_entities);
    */

    return ent;

}


void entity_delete(entity *ent)
{
    weap_type_t *wt;

    /* Does entity explode when it dies? */
    if( ent->class == PROJECTILE ) {
	wt = weapon_type(ent->weapon);
	if( wt->explosion )
	    weapon_explode(ent, wt);
    }

    if( num_weapon_types > 0 ) {
	free( ent->has_weapon );
    }

    num_entities--;
    class_population[ ent->class ]--;

    /*
      printf("Deleted a %s (%d entities left)\n", entity_type(ent->type)->name,
      num_entities);
    */

    /* Make sure that root & Tail will still point at the
       right spot after the entity is deleted */
    if( root == ent )
	root = ent->next;
    if( tail == ent )
	tail = ent->prev;

    if( ent->prev )
	ent->prev->next = ent->next;

    if( ent->next )
	ent->next->prev = ent->prev;

    free(ent);

}


/* Deletes all entities, returns the number of entities deleted */
int entity_remove_all_non_clients(void)
{
    entity *ent, *next;
    int num = 0;

    printf("Removing all non client entities\n");

    ent = root;
    while( ent != NULL ) {
	num++;
	next = ent->next;
	if( ent->cl ) { /* Don't delete clients... */
	    printf("Saving %s...\n", ent->name);
	    ent->mode = LIMBO;
	} else
	    entity_delete(ent);
	ent = next;
    }

    return num;

}


netmsg entity_to_netmsg(entity *ent)
{
    netmsg msg;

    msg.entity.entity_type = ent->type;
    msg.entity.dir = ent->dir;
    msg.entity.mode = ent->mode;
    msg.entity.x = ent->x + 0.5; /* Round floats to integers */
    msg.entity.y = ent->y + 0.5;
    msg.entity.img = ent->img;
    msg.entity.weapon = ent->weapon;

    return msg;

}


void entity_killed(entity *shooter, entity *victim, point P, byte weaptype)
{
    char buf[TEXTMESSAGE_STRLEN];
    int change = 0;
    weap_type_t *wt;

    /* No frags for killing items or projectiles */
    if( victim->class == PROJECTILE || victim->class == ITEM ) {
	return;
    }

    if( victim == shooter ) {    /* Suicide */
	victim->frags--;
	if( victim->name ) {
	    snprintf(buf, TEXTMESSAGE_STRLEN, "%s couldn't cope",victim->name);
	    sv_net_send_text_to_all(buf);
	}
	return;
    } else {
	/* Who was the victim? */
	switch( game.mode ) {
	case SAVETHEWORLD:
	    if( victim->class != GOODIE )
		change++;
	    break;
	case HOLYWAR:
	    if( victim->class == NEUTRAL )
		change--;
	    else {
		change++;
		/* OBITUARY IF CLIENT?? */
	    }
	    break;
	case BUNNYBASH:
	    change++;
	    break;
	default:
	    ERR_QUIT("Game mode out of range!", game.mode );
	}
    }

    if( shooter )	/* Update frag count */
	shooter->frags += change;

    if( victim->name && shooter->name ) {
	if( (wt = weapon_type(weaptype)) && wt->obituary )
	    snprintf(buf, TEXTMESSAGE_STRLEN, wt->obituary,
		     victim->name, shooter->name);
	else
	    snprintf(buf, TEXTMESSAGE_STRLEN,
		     "A %s was killed by a %s.", victim->name, shooter->name);
	sv_net_send_text_to_all(buf);
    }


}


void entity_animate(void)
{
    animation_t *ani;
    ent_type_t *et;
    entity *ent;
    msec_t now;
    int animate;

    now = gettime();

    for( ent = root ;  ent != NULL ; ent = ent->next ) {
	ani = entity_animation(ent);

	animate = 0; /* Don't animate by default */
	if( ani->images > 1 ) {
	    if( ani->stationary_ani ) /* Animates when stationary */
		animate = 1;
	    else if( ent->x_v || ent->y_v ) /* Moving */
		animate = 1;
	}

	if( animate ) {
	    /* If it's time do next frame in animation loop */
	    if( now - ent->last_frame >= ani->framelen ) {
		ent->last_frame = now;
		if( ++ent->frame >= ani->frames ) {
		    /* End of this loop of animation. If the entity is
		       in it's dying animation, make it dead. */
		    if( ent->mode == DYING )
			ent->mode = DEAD;
		    else
			ent->frame = 0;
		}
	    }

	    ent->img = ani->order[ent->frame];

	} else {
	    et = entity_type(ent->type);
	    if( ent->mode == DYING ) {
		printf("Killed off %s\n", et->name);
		ent->mode = DEAD;
	    } else
		ent->img = 0;
	}
    }

}



animation_t *entity_animation(entity *ent)
{
    animation_t *ani;
    ent_type_t *et;

    et = entity_type(ent->type);
    ani = et->animation[ ent->mode ];

    if( ani == NULL ) {
	/* Plan B, use Alive animation */
	if( (ani = et->animation[ ALIVE ]) == NULL ) {
	    printf("%s: Animation modes %d & ALIVE undefined for ent %s\n",
		   __FILE__, ent->mode, et->name);
	    ERR_QUIT("No ALIVE animation. Fix your \"entity\" file!", -1);
	}
    }

    return ani;

}

/* returns pixels between the CENTER of ent0 & ent1 */
float entity_dist(entity *ent0, entity *ent1)
{
    ent_type_t *et_0, *et_1;
    float x_dist, y_dist;

    et_0 = entity_type(ent0->type);
    et_1 = entity_type(ent1->type);

    x_dist = (ent0->x + et_0->width/2.0) - (ent1->x + et_1->width/2.0);
    y_dist = (ent0->y + et_0->height/2.0) - (ent1->y + et_1->height/2.0);

    return sqrt( x_dist * x_dist + y_dist * y_dist );

}


/* Find entity that has the entity-id of id */ 
entity *findent(int id)
{
    entity *ent;

    for( ent=root ; ent != NULL ; ent = ent->next ) {
	if( ent->id == id )
	    return ent;
    }

    return NULL;

}


/* Called when victim's health <= 0 */
static void entity_die(entity *ent)
{
    ent_type_t *et;
	entity *seeker;

    et = entity_type(ent->type);
    
    if( et->animation[ DYING ] ) {
		ent->mode = DYING;
		ent->last_frame = gettime();
		ent->frame = 0;
    } else
		ent->mode = DEAD;

	/* If something has current entity as target, remove it */
	for(  seeker =  root ; seeker != NULL ; seeker = seeker->next ) {
			if( seeker->target == ent )
					seeker->target = NULL;
	}
	
}


/* Returns a pointer to newly created entity, returns NULL on error */
static entity *entity_alloc(void)
{
    entity *ent;
    
    if( (ent = (entity *)malloc( sizeof(entity) )) == NULL)
	perror("Malloc");
    else
	ent->next = NULL;

    return ent;

}
