/* XTux Arena Client. David Lawrence - philaw@camtech.net.au
 * Client network code.
 * Started Jan 18 2000
 */

#include <unistd.h>
#include <sys/signal.h>
#include <fcntl.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <sys/select.h>
#include <sys/time.h>
#include <netdb.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>


#include "xtux.h"
#include "client.h"
#include "win.h"
#include "entity.h"
#include "cl_net.h"
#include "cl_netmsg_send.h"
#include "cl_netmsg_recv.h"

extern win_t win;
extern client_t client;
extern netmsg my_entity;

int sock = -1;

void cl_network_init(void)
{

    client.server_address = "localhost";
    client.port = DEFAULT_PORT;

}


void sigcatcher(int signal)
{

    if( signal == SIGALRM )
	printf("Timed out!\n");

    return;

}


int cl_network_connect(char *server, int port)
{
    struct hostent *hp;
    struct sockaddr_in addr;
    netmsg msg;

    if( sock <= 0 )
	sock = net_init();
    else {
	printf("Already connected!\n");
	return 0;
    }

    memset(&addr, 0, sizeof(addr));
    addr.sin_family = AF_INET;
    addr.sin_port = htons(port);

    if( (addr.sin_addr.s_addr = inet_addr(server)) == INADDR_NONE ) {
	if((hp = gethostbyname(server)) == NULL) {
	    close(sock);
	    sock = -1;
	    return 0;
	} else {
	    addr.sin_addr.s_addr = *(long *) hp->h_addr;
	}
    }

    if( connect(sock, (struct sockaddr *) &addr, sizeof(addr)) < 0 ) {
	perror("connect");
	close(sock);
	sock = -1;
	return 0;
    }

    client.connected = 1;
    client.sv_status = JOINING;

    printf("Joining Game....\n");
    cl_netmsg_send_join();
    printf("Querying Server info......\n");
    cl_netmsg_send_query_sv_info();

    alarm(10); /* Timeout after 10 secs */
    signal(SIGALRM, sigcatcher);
    
    printf("buffered read....\n");
    while( net_buffered_read(sock) > 0 ) {
	printf("Got something\n");
	while( (msg = net_get_message()).type != NETMSG_NONE ) {
	    if( msg.type == NETMSG_SV_INFO ) {
		/* Got server info ok, we can join the game */
		alarm(0);
		cl_net_handle_message(msg);
		return 1;
	    } else if(msg.type == NETMSG_REJECTION || msg.type == NETMSG_QUIT){
		/* We were rejected or another error occured trying to join */
		alarm(0);
		cl_net_handle_message(msg);
		return 0;
	    } else {
	      printf("IGNORING message type %d!\n", msg.type);
	    }
	}
    }

    alarm(0);
    printf("TIMED OUT waiting for server info!\n");
    cl_network_disconnect();
    return 0;

}


void cl_network_disconnect(void)
{

    if( client.connected ) {
	printf("Disconnecting\n");
	close(sock);
	sock = -1;
	client.connected = 0;
	client.health = 0;
    } else /* Not connected (shouldn't be allowed to happen) */
	printf("Not Connected!\n");

}


void cl_net_connection_lost(char *str)
{
    if( str ) {
	clear_area(win.buf, 0, client.view_h/2, client.view_w, 32, "black");
	win_center_print(win.buf, str, client.view_h/2, 3, "white");
    } else
	win_center_print(win.buf, "Connection to server lost", 
			 client.view_h/2, 3, "white");
    win_update();
    delay(2*M_SEC);
    cl_network_disconnect();
    client.state = MENU;
}


/*
  Function pointer table for handling incoming messages.
  All of the below functions are in "cl_netmsg_recv.c"
  We shouldn't get the messages that are NOTHANDLED, because
  the server shouldn't send them. In theory anyway.
*/
static void cl_netmsg_not_handled(netmsg msg)
{
    printf("NOT HANDLING MESSAGE \"%s\"\n", net_message_name(msg.type));
}


#define NOTHANDLED cl_netmsg_not_handled
void (*recv_message[NUM_NETMESSAGES])(netmsg) = {
    NOTHANDLED, /* NETMSG_NONE */
    NOTHANDLED, /* NETMSG_NOOP */
    cl_netmsg_recv_query_version,
    cl_netmsg_recv_version,
    cl_netmsg_recv_textmessage,
    cl_netmsg_recv_quit,
    cl_netmsg_recv_rejection,
    cl_netmsg_recv_sv_info,
    cl_netmsg_recv_changelevel,
    cl_netmsg_recv_start_frame,
    cl_netmsg_recv_end_frame,
    cl_netmsg_recv_entity,
    cl_netmsg_recv_myentity,
    cl_netmsg_recv_particles,
    cl_netmsg_recv_update_statusbar,
    NOTHANDLED,  /* NETMSG_JOIN */
    NOTHANDLED,  /* NETMSG_READY */
    NOTHANDLED,  /* NETMSG_QUERY_SV_INFO */
    NOTHANDLED,  /* NETMSG_CL_UPDATE */
    cl_netmsg_recv_gamemessage
};
    

/* Handle message and return the message type */
int cl_net_handle_message(netmsg msg)
{

    if( client.debug )
	printf("cl_net_handle_message: type = %s\n",
	       net_message_name(msg.type));

    if( msg.type < NUM_NETMESSAGES ) {
	if( recv_message[msg.type] == NULL )
	    cl_netmsg_not_handled(msg);
	else
	    recv_message[msg.type](msg);
	return msg.type;
    } else {
	printf("cl_netmsg_recv: Unknown message type %d!\n", msg.type);
	return -1;
    }

}


void cl_net_update(void)
{
    netmsg msg;

    if( client.debug ) {
	printf("**************************\n");
	printf("About to get read() from server.....\n");
    }

    switch( net_buffered_read( sock ) ) {
    case 1:
	break;
    case -1:
	printf("net_read_frame() returned -1\n");
	cl_net_connection_lost(NULL);
	return;
    case 0:
	printf("Skipping this loop, but trying again\n");
	return;
    }


    while( (msg = net_get_message()).type != NETMSG_NONE )
	cl_net_handle_message(msg);


}



