/* XNibbles - A simple X11 snake game
   Copyright (C) 1998 Sean MacIsaac and Ian Peters
  
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY of FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
  
   Specific questions about this program can be addressed to the authors,
   Sean Macisaac (sjm@acm.org), and Ian Peters (ipeters@acm.org). */

#include "common.h"
#include <malloc.h>
#include <limits.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <linux/soundcard.h>
#include <sys/ioctl.h>
#include <sys/wait.h>

struct Mix {
  unsigned char *ClippedBuffer;
  int *UnclippedBuffer;
  int size;
};

typedef struct Mix Mix;

struct Channel {
  unsigned char *start, *current;
  int length;
  int left;
};

typedef struct Channel Channel;

static int MixerStatus = 0;
static const Sound *sounds = NULL;
static int NumberSounds = 0;
static int SoundDev = -1;
static int Pipe[2] = {-1, -1};
static int PID = -1;
static const char *device = NULL;
static int NumberChannels = 6;
static int mixrate = 0;

int SoundInitDevice();
int SoundRestoreDevice();

void ChannelReset(Channel *channel);
void ChannelAssign(Channel *channel, const Sound *sound);
int ChannelMixAll(Mix *mix, Channel *channel);
int ChannelCopyIn(Channel *channel, Mix *mix);
int ChannelMixIn(Channel *channel, Mix *mix);
int ChannelFinalMix(Channel *channel, Mix *mix);
void MixAlloc(Mix *mix, int size);
void MixDealloc(Mix *mix);

int SoundInit(int number, const Sound *SoundArray, int MixRate, int channels, const char *Device) {
  int result;

  NumberSounds = number;
  sounds = SoundArray;
  mixrate = MixRate;
  NumberChannels = channels;
  device = Device;

  if (sounds == NULL) return EXIT_FAILURE;
  result = SoundInitDevice();
  if (result == EXIT_SUCCESS) {
    MixerStatus = 1;
  } else {
    MixerStatus = 0;
  }
  return result;
}

int SoundRestore() {
  int result;

  if (!MixerStatus) return EXIT_FAILURE;
  result = SoundRestoreDevice();
  MixerStatus = 0;
  return result;
}

int SoundPlay(int sound, int channel) {

  if (!MixerStatus) return EXIT_FAILURE;

  if (sounds[sound].data != NULL) {
    write(Pipe[1], &sound, sizeof(sound));
    write(Pipe[1], &channel, sizeof(channel));
  }

  return EXIT_SUCCESS;
}

int SoundInitDevice() {
  int whoami;

  SoundDev = -1;
  PID = 0;
  
  if (access(device, W_OK) != 0) {
    perror("Could not access sound device");
    return EXIT_FAILURE;
  }

  SoundDev = open(device, O_WRONLY);
 
  if (SoundDev < 0) {
    fprintf(stderr,"Cannot open sound device\n");
    return EXIT_FAILURE;
  }

  close(SoundDev);

  if (pipe(Pipe) < 0) {
    fprintf(stderr, "Cannot create pipe to sound controller\n");
    return EXIT_FAILURE;
  }

  if ((whoami = fork()) < 0) {
    fprintf(stderr, "Cannot fork sound driver\n");
    return EXIT_FAILURE;
  }

  if (whoami != 0) {
    close(Pipe[0]);
    PID = whoami;
    return EXIT_SUCCESS;
  }

  {
    int sound_num, ch, i;
    struct timeval tval = {0L, 0L};
    fd_set readfds, dsp;
    Mix mix;
    int frag, fragsize;
    Channel *chan = (Channel*)malloc(sizeof(Channel)*NumberChannels);

    for (i = 0; i < NumberChannels; i++) ChannelReset(chan + i);
    SoundDev= open(device, O_WRONLY);
    if (SoundDev < 0) {
      perror("Cannot open sound device: ");
      exit(1);
    }

    frag = FRAG_SPEC;

    ioctl(SoundDev, SNDCTL_DSP_SETFRAGMENT, &frag);

    if (ioctl(SoundDev, SNDCTL_DSP_SPEED, &mixrate) == -1)
      perror("Sound driver ioctl ");

    fragsize = 0;
    if (ioctl(SoundDev, SNDCTL_DSP_GETBLKSIZE, &fragsize) == -1)
      perror("Sound driver ioctl ");

    MixAlloc(&mix,fragsize);

    close(Pipe[1]);

    FD_ZERO(&dsp);
    FD_SET(SoundDev, &dsp);
    FD_ZERO(&readfds);
    FD_SET(Pipe[0], &readfds);

    for ( ; ; ) {
      FD_SET(Pipe[0], &readfds);
      tval.tv_sec = 0L;
      tval.tv_usec = 0L;
      select(Pipe[0]+1, &readfds, NULL, NULL, &tval);
      if (FD_ISSET(Pipe[0], &readfds)) {
        if (read(Pipe[0], &sound_num, sizeof(int)) == 0)
          break;
        read(Pipe[0], &ch, sizeof(int));
        ChannelAssign(&(chan[ch]), &(sounds[sound_num]));
      }
      ChannelMixAll(&mix, chan);
      write(SoundDev, mix.ClippedBuffer, fragsize);
    }
    MixDealloc(&mix);
    close(Pipe[0]);
    close(Pipe[1]);
    exit(0);
  }
}

int SoundRestoreDevice() {
  close(Pipe[0]);
  close(Pipe[1]);
  wait(NULL);
  return EXIT_SUCCESS;
}

void ChannelReset(Channel *chan) {
  chan->start = NULL;
  chan->current = NULL;
  chan->length = 0;
  chan->left = 0;
}

void ChannelAssign(Channel *chan, const Sound *snd) {
  chan->start = snd->data;
  chan->current = chan->start;
  chan->length = snd->length;
  chan->left = snd->length;
}

int ChannelCopyIn(Channel *chan, Mix *mix) {

  int i, *p = mix->UnclippedBuffer, result, min;

  result = (chan->left > 0 ) ? 1 : 0;
  min= ( chan->left < mix->size ) ? chan->left : mix->size;
  for (i = 0; i < min; i++) {
    *p++ = (int) *chan->current++;
  }
  chan->left -= i;
  while(i < mix->size) {
    *p++ = 128;
    i++;
  }
  return result;
}

int ChannelMixIn(Channel *chan, Mix *mix) {
  int i, *p = mix->UnclippedBuffer, result, min;
  result = (chan->left > 0) ? 1 : 0;
  min = (chan->left < mix->size) ? chan->left : mix->size;

  for ( i = 0; i < min; i++) {
    *p++ += (int) (*chan->current++) - 128;
  }
  
  chan->left -= i;
  return result;
}

static inline unsigned char clip(int i) {
  return (i < 0) ? 0 : ((i > 255) ? 255 : i);
}

int ChannelFinalMix(Channel *chan, Mix *mix) {
  register int i;
  int *p = mix->UnclippedBuffer, result, min;
  unsigned char *final = mix->ClippedBuffer;

  result = (chan->left > 0) ? 1 : 0;
  min = (chan->left < mix->size) ? chan->left : mix->size;

  for (i = 0; i < min; i++) {
    *p += (int) (*chan->current++) - 128;
    *final++ = clip(*p++);
  }
  chan->left -= 1;
  while (i < mix->size) {
    *final++ = clip(*p++);
    i++;
  }
  return result;
}

void MixAlloc(Mix *mix, int size) {
  mix->ClippedBuffer = (unsigned char *)calloc(sizeof(char), size);
  mix->UnclippedBuffer = (int *)calloc(sizeof(int), size);
  mix->size = size;

  if ((mix->ClippedBuffer == NULL) || (mix->UnclippedBuffer == NULL)) {
    fprintf(stderr, "Unable to allocate memory for mixer buffer\n");
    exit(-1);
  }
}

void MixDealloc(Mix *mix) {
  if (mix->ClippedBuffer) free(mix->ClippedBuffer);
  if (mix->UnclippedBuffer) free(mix->UnclippedBuffer);
}

int ChannelMixAll(Mix *mix, Channel *chan) {
  int result = 0, i = 0;

  result = ChannelCopyIn(chan, mix);
  for (i = 2; i < NumberChannels; i++)
    result += ChannelMixIn(++chan, mix);

  result += ChannelFinalMix(++chan, mix);

  return result;
}

int SoundLoad(const char *file, Sound *sound) {
  FILE *fp;

  sound->data = NULL;
  sound->length = 0;
  fp = fopen(file, "r");
  if (fp == NULL) return -1;
  sound->length = lseek(fileno(fp), 0, SEEK_END);
  lseek(fileno(fp), 0, SEEK_SET);
  sound->data = (unsigned char *)malloc(sound->length);

  if (sound->data == NULL) {
    fclose(fp);
    return -2;
  }
  
  fread(sound->data, 1, sound->length, fp);
  fclose(fp);
  return 0;
}
