/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   zb.c								*
*									*
*	Client zero buffer monitor routines.  This module provides	*
*	an abstraction of implicit outgoing buffers used for all	*
*	protocol generated by xmx itself (aka client zero).		*
*									*
*	Establishes the notion of a current destination, and prevents	*
*	the buffer from becoming too large - frees internal routines	*
*	from tedious buffer management.					*
*									*
************************************************************************/
#define NEED_EVENTS
#include <X11/Xproto.h>
#include <xmc.h>
#include "xmx.h"
#include "zb.h"
#include "df.h"
#include "cx.h"
#include "rx.h"
#include "incl/zb.pvt.h"

zb_t zb;		/* client zero buffer monitor */

/************************************************************************
*									*
*   destination info							*
*									*
************************************************************************/
static int fdest;		/* forward destination */
static int rdest;		/* reverse destination */
static server_t *dsp;		/* dest server */

static int event;		/* boolean - event? */
static client_t *dcp;		/* dest client */
static window_t *dwp;		/* dest window */
static mask_t dmask;		/* dest event mask */

/************************************************************************
*									*
*   zb_init								*
*									*
************************************************************************/
void
zb_init
   VOID
{
   fdest = ZD_NONE;
   rdest = ZD_NONE;

   zb.zseqno = 0;	/* first seqno is 1 */
   zb.bp = buf_new(B_STATIC);
   zb.rbp = buf_new(B_STATIC);
}

/************************************************************************
*									*
*   zb_dest								*
*									*
*	Establishes the current server destination for the buffer	*
*	monitor.  Subsequent calls to zb_queue will queue all active	*
*	data to servers identified with the destination.		*
*									*
************************************************************************/
void
zb_dest
   AL((dest, sp))
   DB int dest
   DD server_t *sp
   DE
{
   if (dest != fdest || sp != dsp)
      if (buf_active(zb.bp))
         zb_queue(0);
   fdest = dest;
   dsp = sp;
}

int
zb_get_dest
   VOID
{
   return fdest;
}

void
zb_server_death
   AL((sp))
   DB server_t *sp
   DE
{
   register chunk_t *chp;

   switch (fdest) {
      case ZD_SERV:
      case ZD_SERV_ASYNC:
         if (sp == dsp) {
            fdest = ZD_NONE;
            dsp = 0;
            /*
            **  throw it all out
            */
            chp = buf_split(zb.bp, 0);
            buf_clear(chp);
         }
         break;
      case ZD_ALLBUTONE:
         if (sp == dsp)
            dsp = 0;
         break;
      default:
         break;
   }
}

server_t *
zb_get_server
   VOID
{
   return dsp;
}

/************************************************************************
*									*
*   zb_client_dest							*
*									*
*	Establishes the current client destination for the buffer	*
*	monitor.  Subsequent calls to zb_client_queue will queue	*
*	all active data to servers identified with the destination.	*
*									*
************************************************************************/
void
zb_client_dest
   AL((cp))
   DB client_t *cp	/* if zero, send to all clients */
   DE
{
   if (event || cp != dcp) {
      if (buf_active(zb.rbp))
         zb_client_queue();
      dcp = cp;
      event = 0;
   }
}

void
zb_client_death
   AL((cp))
   DB client_t *cp
   DE
{
   register chunk_t *chp;

   if (cp == dcp && event == 0) {
      chp = buf_split(zb.rbp, 0);
      buf_clear(chp);

      dcp = 0;
   }
}

/************************************************************************
*									*
*   zb_event_dest							*
*									*
************************************************************************/
void
zb_event_dest
   AL((wp, mask))
   DB window_t *wp
   DD mask_t mask
   DE
{
   if (buf_active(zb.rbp))
      zb_client_queue();
   dwp = wp;
   dmask = mask;
   event = 1;
}

/************************************************************************
*									*
*   zb_queue								*
*									*
*	Queue the active contents of the forward buffer to its current	*
*	destination.  Assumes the buffer is not empty.			*
*									*
************************************************************************/
void
zb_queue
   AL((pp))
   DB pp_t *pp	/* if trailing partial packet info for this */
   DE
{
   register int i, n;
   register chunk_t *chp;

   chp = buf_split(zb.bp, 0);
   pp_assign(chp->tpp, pp);

   n = zb_queue_chunk(chp);

   if (rx_repeats(zrxqp)) {
      rx_incr(zrxqp, n);
      if (fdest != ZD_SERV_ASYNC)
         rx_clear_repeat(zrxqp);
   }
   buf_clear(chp);
}

/************************************************************************
*									*
*   zb_queue_chunk							*
*									*
*	Queue the given chunk to the current forward destination.	*
*									*
************************************************************************/
int
zb_queue_chunk
   AL((chp))
   DB chunk_t *chp
   DE
{
   register int i, n;

   n = 0;
   switch (fdest) {
      case ZD_NONE:
         break;
      case ZD_SERV:
      case ZD_SERV_ASYNC:
         queue_add(dsp->qp, chp);
         n++;
         break;
      case ZD_ALLBUTONE:
         for (i=0; i<num_serv; i++)
            if (servers[i]->state == S_READY && servers[i] != dsp) {
               queue_add(servers[i]->qp, chp);
               n++;
            }
         break;
      case ZD_KETCHUP:
         for (i=0; i<num_serv; i++)
            if (servers[i]->state == S_KETCHUP) {
               queue_add(servers[i]->qp, chp);
               n++;
            }
         break;
      case ZD_QSERV:
         queue_add(qservp->qp, chp);
         n++;
         break;
      case ZD_ALLSERV:
         for (i=0; i<num_serv; i++)
            if (servers[i]->state == S_READY) {
               queue_add(servers[i]->qp, chp);
               n++;
            }
         break;
      case ZD_FLOOR:
         for (i=0; i<num_serv; i++)
            if (servers[i]->state == S_READY && servers[i]->mode == Floor) {
               queue_add(servers[i]->qp, chp);
               n++;
            }
         break;
      case ZD_SEAT:
         for (i=0; i<num_serv; i++)
            if (servers[i]->state == S_READY && servers[i]->mode == Seat) {
               queue_add(servers[i]->qp, chp);
               n++;
            }
         break;
      case ZD_VIEW:
         for (i=0; i<num_serv; i++)
            if (servers[i]->state == S_READY && servers[i]->mode == View) {
               queue_add(servers[i]->qp, chp);
               n++;
            }
         break;
      default:
         warn("zb_queue: destination is not valid [%d]\n", fdest);
         break;
   }
   return n;
}

/************************************************************************
*									*
*   zb_client_queue							*
*									*
*	Queue the active contents of the return buffer to its current	*
*	destination.  Assumes the buffer is not empty.			*
*									*
*	Note that event_send (as invoked below) may cause this routine	*
*	to be called recursively, but never to an event destination,	*
*	which probably would be bad if it did.  The exact progression	*
*	is event_send -> proto_KeymapNotify -> zb_client_queue.		*
*	Careful what you change here.					*
*									*
************************************************************************/
void
zb_client_queue
   VOID
{
   register int i, j;
   chunk_t *chp;

   chp = buf_split(zb.rbp, 0);
   if (event)
      event_send(chp, 1, dwp, dmask, 0);	/* not perfect */
   else if (dcp)
      queue_add(dcp->qp, chp);
   else				/* if client is zero, send to all */
      for (i=j=1; j<num_clients; i++)
         if (clients[i]) {
            j++;
            queue_add(clients[i]->qp, chp);
         }

   buf_clear(chp);
}
