/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   vconf.c								*
*									*
*	Create and maintain the virtual server configuration.		*
*	The virtual configuration is private to this module.		*
*									*
*	The virtual configuration is created in several passes:		*
*		1) It is partially initialized with default values.	*
*		2) The remaining values are set to those of the first	*
*		   server encountered.					*
*		3) These are then pared down and adjusted as other	*
*		   servers are merged with it.				*
*		4) Finally, any user or default constraints are		*
*		   applied.						*
*									*
*	The virtual configuration may only be modified, either by	*
*	changing constraints or merging a new server, before it has	*
*	been fixed (by vconf_fix()).  Once it has been fixed, new	*
*	constraints are not accepted and new servers must be mapped	*
*	without merging.						*
*									*
************************************************************************/
#include <X11/X.h>
#define NEED_REPLIES
#define NEED_EVENTS
#include <X11/Xproto.h>

#include <xmc.h>
#include <xmcp.h>

#include "xmx.h"
#include "df.h"
#include "res.h"
#include "ptc.h"
#include "version.h"
#include "patchlevel.h"
#include "vconf.h"

#include "incl/vconf.pvt.h"

static buffer_t *bp;
static vconf_t vconf;
static chunk_t *vc_chp;
static chunk_t *vc_swap_chp;
static vcons_t vcons;		/* virtual configuration constraints */

/************************************************************************
*									*
*  vconf_init								*
*									*
*	Initialize the virtual configuration.  After it is		*
*	initialized, it exists, has some default values in it, but is	*
*	otherwise empty and undefined.					*
*									*
************************************************************************/
void
vconf_init
   VOID
{
   register format_t *fp, *lfp;
   register depth_t *dp, *ldp;
   register vtype_t *vp, *lvp;
   register extname_t *ep, *lep;

   if (bp == 0)
      bp = buf_new(B_STATIC);
   /*
   **  free chunks
   */
   if (vc_chp) {
      buf_clear(vc_chp);
      vc_chp = 0;
   }
   if (vc_swap_chp) {
      buf_clear(vc_swap_chp);
      vc_swap_chp = 0;
   }
   /*
   **  free virtual configuration
   */
   free_formats();
   free_depths();
   free_extnames();
   /*
   **  Initialize values
   */
   vconf.prefix.success = 1;
   vconf.prefix.majorVersion = X_MAJOR_VERSION;
   vconf.prefix.minorVersion = X_MINOR_VERSION;
   vconf.prefix.length = 0;

   vconf.setup.release = RELEASE;
   vconf.setup.ridBase = 0;
   vconf.setup.ridMask = RID_MASK;
   vconf.setup.motionBufferSize = 0;
   vconf.setup.nbytesVendor = strlen(VENDOR);
   vconf.setup.maxRequestSize = MAXREQSIZE;
   vconf.setup.numRoots = 1;	/* there's always only one root window */
   vconf.setup.numFormats = 0;
   vconf.setup.imageByteOrder = 0;
   vconf.setup.bitmapBitOrder = 0;
   vconf.setup.bitmapScanlineUnit = 0;
   vconf.setup.bitmapScanlinePad = 0;
   vconf.setup.minKeyCode = MIN_KEYCODE;
   vconf.setup.maxKeyCode = MAX_KEYCODE;

   vconf.root.whitePixel = 0;	/* these get set in color_root_cmap */
   vconf.root.blackPixel = 0;
   vconf.root.currentInputMask = 0;
   vconf.root.pixWidth = 0;
   vconf.root.pixHeight = 0;
   vconf.root.mmWidth = 0;
   vconf.root.mmHeight = 0;
   vconf.root.minInstalledMaps = 0;
   vconf.root.maxInstalledMaps = 0;
   vconf.root.rootVisualID = 0;
   vconf.root.backingStore = 0;
   vconf.root.saveUnders = 0;
   vconf.root.rootDepth = 0;
   vconf.root.nDepths = 0;

   vconf.nexts = 0;
   /*
   **  zero all pointers
   */
   vconf.prefixp = 0;
   vconf.setupp = 0;
   vconf.fp = 0;
   vconf.rootp = 0;
   vconf.dp = 0;
   vconf.rootvp = 0;
   vconf.ep = 0;

   vcstate = VC_MUSH;
}

/************************************************************************
*									*
*  vconf_fix								*
*									*
*									*
************************************************************************/
int
vconf_fix
   VOID
{
   register int i;
   register depth_t *dp;
   register vtype_t *vp;
   register format_t *fp;
   s16_t x, y;

   if (config_mode == ConfigDelay)		/* delay configuration */
      return -1;
   /*
   **  merge all servers
   */
   for (i=0; i<num_serv; i++)
      if (servers[i]->state == S_KETCHUP)
         if (merge(servers[i])) {
            warn("can't merge server %s:%d.%d into virtual config\n",
		servers[i]->tag, servers[i]->display, servers[i]->screen);
            xmc_drop(servers[i--]);	/* slides server array left */
         }

   if (i == 0)
      return -1;	/* no configuration possible */

   apply_constraints();

   /*
   ** create server resource model
   */
   viordv[1] = image_xy_iord(	vconf.setup.bitmapBitOrder,
				vconf.setup.imageByteOrder,
				vconf.setup.bitmapScanlineUnit,
				vconf.setup.bitmapScanlinePad);

   for (fp=vconf.fp; fp; fp=fp->next)
      if (fp->format.depth > 1)
         viordv[fp->format.depth] = image_z_iord(fp->format.bitsPerPixel,
						fp->format.scanLinePad);

   sres_init_depths(vconf.root.nDepths);
   for (dp=vconf.dp; dp; dp=dp->next) {
      sres_new_depth(dp->depth.depth, vconf.root.rootDepth);
      for (vp=dp->vp; vp; vp=vp->next)
         if (sres_new_visual(dp->depth.depth, &vp->visual))
            quit(1, "vconf_fix: sres_new_visual failed\n");
   }
   if (sres_init_screen(&vconf.root, vconf.rootvp->visual.visualID))
      quit(1, "vconf_reset: sres_new_screen [%d] failed\n", i);

   if (opt.geomp) {
      /*
      ** do we really want to set the virtual size this way?
      */
      util_adjsize(	opt.geomp,
			vconf.root.pixWidth,
			vconf.root.pixHeight,
			vconf.root.mmWidth,
			vconf.root.mmHeight,
			&vconf.root.pixWidth,
			&vconf.root.pixHeight,
			&vconf.root.mmWidth,
			&vconf.root.mmHeight);
      vscreen.wp->dwb.width = vconf.root.pixWidth;
      vscreen.wp->dwb.height = vconf.root.pixHeight;
   }
   tptr_init();	/* initialize default telepointer */

   D_CALL0(D_PAINFUL, vconf_print);

   vcstate = VC_FIXED;
   return 0;
}

/************************************************************************
*									*
*  vconf_map								*
*									*
*									*
************************************************************************/
int
vconf_map
   AL((sp))
   DB server_t *sp
   DE
{
   register mask_t m;
   register format_t *fp;
   register depth_t *dp;
   register vtype_t *vp;
   register extname_t *ep;
   register geom_t *geomp;

   if (vcstate != VC_FIXED)
      return err(-1, "vconf_map: virtual configuration is not fixed!\n");

   if (compare(sp))
      return -1;

   /*
   **  minorVersion
   */
   if (	vconf.prefix.minorVersion > vconf.prefixp->minorVersion) {
      DEBUG3(D_VCONF, "vconf_map: server [%d/%s] %s\n",
				sp->fd, sp->tag, "minor version mismatch");
      return -1;
   }
   /*
   **  maxRequestSize
   */
   if (vconf.setup.maxRequestSize > vconf.setupp->maxRequestSize) {
      DEBUG3(D_VCONF, "vconf_map: server [%d/%s] %s\n",
				sp->fd, sp->tag, "maxRequestSize mismatch");
      return -1;
   }
   /*
   **  formats
   */
   for (fp=vconf.fp; fp; fp=fp->next)
      if (!fp->match) {
         DEBUG3(D_VCONF, "vconf_map: server [%d/%s] %s\n",
				sp->fd, sp->tag, "format mismatch");
         D_CALL1(D_VCONF, dprx_PixmapFormat, &fp->format);
         return -1;
      }
   /*
   **  backingStore (nonfatal)
   */
   if (vconf.root.backingStore > vconf.rootp->backingStore) {
      DEBUG3(D_VCONF, "vconf_map: server [%d/%s] %s\n",
			sp->fd, sp->tag, "backingStore (nonfatal) mismatch");
      D_CALL1(D_VCONF, dprx_WindowRoot, &vconf.root);
   }
   /*
   **  saveUnders (nonfatal)
   */
   if (vconf.root.saveUnders && !vconf.rootp->saveUnders) {
      DEBUG3(D_VCONF, "vconf_map: server [%d/%s] %s\n",
			sp->fd, sp->tag, "saveUnders (nonfatal) mismatch");
      D_CALL1(D_VCONF, dprx_WindowRoot, &vconf.root);
   }
   /*
   **  depths
   */
   for (dp=vconf.dp; dp; dp=dp->next) {
      if (!dp->match) {
         DEBUG3(D_VCONF, "vconf_map: server [%d/%s] depth [%d] mismatch\n",
				sp->fd, sp->tag, dp->depth.depth);
         return -1;
      }
      /*
      **  visuals
      */
      for (vp=dp->vp; vp; vp=vp->next)
         if (!vp->match) {
            DEBUG3(D_VCONF, "vconf_map: server [%d/%s] %s\n",
					sp->fd, sp->tag, "visual mismatch");
            D_CALL1(D_VCONF, dprx_VisualType, &vp->visual);
            return -1;
         }
   }
   /*
   **  extensions
   */
   for (ep=vconf.ep; ep; ep=ep->next)
      if (!ep->match) {
         DEBUG3(D_VCONF, "vconf_map: server [%d/%s] extension [%s] mismatch\n",
				sp->fd, sp->tag, ep->name);
         return -1;
      }

   /*
   **  SUCCESS - a mapping is possible
   */
   sp->smap.base = vconf.setupp->ridBase;
   sp->smap.mask = vconf.setupp->ridMask;
   sp->smap.minkey = vconf.setupp->minKeyCode;
   sp->smap.keycnt = vconf.setupp->maxKeyCode - vconf.setupp->minKeyCode + 1;
   sp->smap.shift = 0;
   for (m=sp->smap.mask; (m&1)==0; m>>=1)
      sp->smap.shift++;
   sp->smap.iordv[1] = image_xy_iord(	vconf.setupp->bitmapBitOrder,
					vconf.setupp->imageByteOrder,
					vconf.setupp->bitmapScanlineUnit,
					vconf.setupp->bitmapScanlinePad);
   for (fp=vconf.fp; fp; fp=fp->next)
      if (fp->formatp->depth > 1)
         sp->smap.iordv[fp->formatp->depth] =
				image_z_iord(	fp->formatp->bitsPerPixel,
				      		fp->formatp->scanLinePad);

   if (sp->smap.mp == 0)
      sp->smap.mp = hash_new(VCONF_HASHSIZE);
   else
      hash_reset(sp->smap.mp);

   sp->smap.root.whitePixel = vconf.rootp->whitePixel;
   sp->smap.root.blackPixel = vconf.rootp->blackPixel;
   sp->smap.root.pixWidth = vconf.rootp->pixWidth;
   sp->smap.root.pixHeight = vconf.rootp->pixHeight;
   sp->smap.root.width = vconf.root.pixWidth;
   sp->smap.root.height = vconf.root.pixHeight;
   sp->smap.root.realcmap = vconf.rootp->defaultColormap;

   sp->smap.root.currentInputMask = vconf.rootp->currentInputMask;

   hash_add(sp->smap.mp, vscreen.vservroot, vconf.rootp->windowId, 0);
   hash_add(sp->smap.mp, vconf.root.windowId, mkrid(sp,vscreen.rootvid), 0);

   for (dp=vconf.dp; dp; dp=dp->next)
      for (vp=dp->vp; vp; vp=vp->next) {
         hash_add(sp->smap.mp, vp->visual.visualID, vp->visualp->visualID, 0);
         /*
         **	if the real and virtual root visuals differ, it's not
         **	possible to share the root colormap.
         */
         if (	vp == vconf.rootvp &&
		vp->visualp->visualID != vconf.rootp->rootVisualID)
            sp->smap.root.flags |= RF_OWNCMAP;
      }

   if (sp->smap.root.flags & RF_OWNROOT)
      sp->smap.root.realshell = mkrid(sp, vscreen.shellroot);
   /*
   **  if pixels are mapped for any server, they are mapped for all
   **  servers, irrespective of RF_OWNCMAP
   */
   if (vscreen.wp->mp->mappixels)
      pmap_new(sp);

   hash_add(		sp->smap.mp,
			vscreen.shellroot,
			sp->smap.root.realshell,
			(char *)vscreen.shdp);

   if (sp->smap.root.flags & RF_OWNCMAP) {
      hash_add(	sp->smap.mp,
			vconf.root.defaultColormap,
			mkrid(sp, vscreen.cmapvid),
			0);
      hash_add(	sp->smap.mp,
			vscreen.vservcmap,
			vconf.rootp->defaultColormap,
			0);
   }
   else {
      hash_add(	sp->smap.mp,
			vconf.root.defaultColormap, 
			vconf.rootp->defaultColormap,
			0);
   }
   return 0;
}

/************************************************************************
*									*
*   vconf_write								*
*									*
************************************************************************/
void
vconf_write
   AL((cp))
   DB client_t *cp
   DE
{
   register chunk_t *chp;
   register xConnSetupPrefix *prefix;
   register format_t *fp;
   register depth_t *dp;
   register vtype_t *vp;
   static char padsrc[4] = {0,0,0,0};

   if (vc_chp == 0) {
      vconf.setup.nbytesVendor = strlen(VENDOR);

      buf_put(bp, (char *)&vconf.prefix, sz_xConnSetupPrefix);
      buf_put(bp, (char *)&vconf.setup, sz_xConnSetup);
      buf_put(bp, VENDOR, vconf.setup.nbytesVendor);
      buf_put(bp, padsrc, PAD(vconf.setup.nbytesVendor));

      for (fp=vconf.fp; fp; fp=fp->next)
         buf_put(bp, (char *)&fp->format, sz_xPixmapFormat);

      buf_put(bp, (char *)&vconf.root, sz_xWindowRoot);

      for (dp=vconf.dp; dp; dp=dp->next) {
         buf_put(bp, (char *)&dp->depth, sz_xDepth);

         for (vp=dp->vp; vp; vp=vp->next)
            buf_put(bp, (char *)&vp->visual, sz_xVisualType);
      }
      vc_chp = buf_split(bp, 0);
      vc_chp->type = P_CONNECT;
      prefix = (xConnSetupPrefix *)buf_data(vc_chp);
      prefix->length = (buf_chunksize(vc_chp) - sz_xConnSetupPrefix) / 4;
   }
   if (cp->swap) {
      if (vc_swap_chp == 0) {
         buf_put(bp, buf_data(vc_chp), buf_chunksize(vc_chp));
         vc_swap_chp = buf_split(bp, 0);
         vc_swap_chp->type = P_CONNECT_SWAPPED;
         swap_connsetup(buf_data(vc_swap_chp));
      }
      chp = vc_swap_chp;
   }
   else
      chp = vc_chp;

   queue_add(cp->qp, chp);
   /* no buf_clear, since we reuse these */
}

/************************************************************************
*									*
*  vconf_use_size							*
*									*
*	Set size constraint on the virtual configuration.		*
*									*
************************************************************************/
int
vconf_use_size
   AL((width, height))
   DB u16_t width
   DD u16_t height
   DE
{
   if (vcstate != VC_MUSH)
      return -1;

   if (width > 0 && height > 0) {
      vcons.mask |= VCSET_SIZE;
      vcons.width = width;
      vcons.height = height;
   }
}

/************************************************************************
*									*
*  vconf_use_depthtype							*
*									*
*	Add a depth/classtype pair to the virtual config		*
*	constraints.  The first time this is called, the values		*
*	passed become the default root depth and class.			*
*									*
************************************************************************/
int
vconf_use_depthtype
   AL((depth, class))
   DB u8_t depth
   DD u8_t class
   DE
{
   register dcons_t *dcp, *ldcp;

   if (vcstate != VC_MUSH)
      return -1;

   if ((vcons.mask & VCSET_DEPTHS) == 0) {	/* first one is default */
      vcons.mask |= VCSET_DEPTHS;
      vcons.rootdepth = depth;
      vcons.rootclass = class;
   }
   for (ldcp=0,dcp=vcons.dcp; dcp; ldcp=dcp,dcp=dcp->next)
      if (dcp->depth == depth)
         break;
      else if (dcp->depth > depth) {
         dcp = 0;
         break;
      }

   if (dcp == 0) {
      if (MALLOC(dcp, dcons_t *, sizeof(dcons_t)))
         quit(-1, "vconf_use_depthtype: malloc error unrecoverable\n");
      if (ldcp == 0) {
         dcp->next = 0;
         vcons.dcp = dcp;
      }
      else {
         dcp->next = ldcp->next;
         ldcp->next = dcp;
      }
      dcp->depth = depth;
      dcp->classmask = 0;
   }
   dcp->classmask |= 1<<class;

   return 0;
}

/************************************************************************
*									*
*  vconf_use_extension							*
*									*
*	Add a protocol extension name to the list of names in the	*
*	virtual configuration constraints.  The virtual configuration	*
*	will support no extensions other than those in this list.  If	*
*	the given name is a null pointer, and this routine is not	*
*	otherwise called, no extensions will be supported.  If this	*
*	routine is not called, the intersection of the extensions	*
*	supported by the servers merged will be supported.		*
*									*
************************************************************************/
int
vconf_use_extension
   AL((name))
   DB char *name	/* if zero, use no extensions */
   DE
{
   register int i, n, cmp;
   register char *extname;
   register char *t1, *t2;

   if (vcstate != VC_MUSH)
      return -1;

   n = ext_supported(0);

   if ((vcons.mask & VCSET_EXTS) == 0) {
      vcons.mask |= VCSET_EXTS;
      if (n)
         if (CALLOC(vcons.extnamep, char **, n, sizeof(char *)))
            return -1;	/* not very graceful... */
      vcons.nextnames = 0;
   }
   if (name == 0 || ext_supported(name) == 0)
      return -1;

   if (MALLOC(extname, char *, strlen(name) + 1))
      return -1;
   (void) strcpy(extname, name);

   for (i=0; i<n; i++)
      if (vcons.extnamep[i]) {
         cmp = strcmp(extname, &extname[i]);
         if (cmp == 0) {		/* already there */
            free(extname);
            break;
         }
         else if (cmp < 0) {		/* insert it */
            for (t1=extname; t1 && i<n; i++) {	/* slide array right */
               t2 = vcons.extnamep[i];
               vcons.extnamep[i] = t1;
               t1 = t2;
            }
            if (t1)
               warn("vconf_use_extension: sanity check - can't happen\n");
               /*
               **  Here's why: the length of the array is equal to the
               **  number of extensions supported, and only supported
               **  extension names are inserted into it.  Therefore t1
               **  can never be non-zero if i==n.
               */
            vcons.nextnames++;
            break;
         }
      }
      else {				/* insert last */
         vcons.extnamep[i] = extname;
         vcons.nextnames++;
         break;
      }
   return 0;
}

/************************************************************************
*									*
*  vconf_clear_constraints						*
*									*
*	Remove all constraints on the virtual configuration.		*
*									*
************************************************************************/
void
vconf_clear_constraints
   VOID
{
   vcons.mask = VCSET_NONE;

   free_dcons();
   free_econs();
}

/************************************************************************
*									*
*   vconf_print								*
*									*
************************************************************************/
void
vconf_print
   VOID
{
   register format_t *fp;
   register depth_t *dp;
   register vtype_t *vp;

   warn("VIRTUAL CONFIGURATION:\n");
   dprx_ConnSetupPrefix(&vconf.prefix);
   dprx_ConnSetup(&vconf.setup);
   dprx_vender(VENDOR);

   for (fp=vconf.fp; fp; fp=fp->next)
      dprx_PixmapFormat(&fp->format);

   dprx_WindowRoot(&vconf.root);

   for (dp=vconf.dp; dp; dp=dp->next) {
      dprx_Depth(&dp->depth);

   for (vp=dp->vp; vp; vp=vp->next)
      dprx_VisualType(&vp->visual);
   }
}

/*
**  make_like
**
**	Copy the configuration details of a real live server
**	connection block into the virtual configuration.  This ensures
**	a close match with at least one participating X server, and
**	gives a starting point for later comparisons.
**
**	If successful, the virtual configuration is left ready to
**	merge or map with the given server.  If unsuccessful, the
**	virtual configuration is rewound to its initialized state.
*/
int
make_like
   AL((sp))
   DB server_t *sp
   DE
{
   register int i;
   register xConnSetupPrefix *prefix;
   register xConnSetup *setup;
   register xWindowRoot *root;
   register depth_t *dp;
   register xDepth *depth;
   register format_t *fp, *fn;
   register xPixmapFormat *format;
   register vtype_t *vp, *vn;
   register xVisualType *visual;
   register extname_t *ep, *en;

   prefix = (xConnSetupPrefix *)cblock_prefix(sp->block);
   vconf.prefixp = prefix;

   setup = cblock_setup(sp->block);

   vconf.setup.motionBufferSize = setup->motionBufferSize;
   vconf.setup.imageByteOrder = setup->imageByteOrder;
   vconf.setup.bitmapBitOrder = setup->bitmapBitOrder;
   vconf.setup.bitmapScanlineUnit = setup->bitmapScanlineUnit;
   vconf.setup.bitmapScanlinePad = setup->bitmapScanlinePad;
#ifdef awaiting_key_mapping_module_TODO
vconf.setup.minKeyCode = setup->minKeyCode;
vconf.setup.maxKeyCode = setup->maxKeyCode;
#endif

   vconf.setupp = setup;

   for (i=0; i<=sp->screen; i++)
      if ((root = cblock_root(0)) == 0) {  /* that screen doesn't exist */
         vconf_init();
         return -1;
      }
   vconf.root.pixWidth = root->pixWidth;
   vconf.root.pixHeight = root->pixHeight;
   vconf.root.mmWidth = root->mmWidth;
   vconf.root.mmHeight = root->mmHeight;
   vconf.root.minInstalledMaps = root->minInstalledMaps;
   vconf.root.maxInstalledMaps = root->maxInstalledMaps;
   vconf.root.backingStore = root->backingStore;
   vconf.root.saveUnders = root->saveUnders;
   vconf.root.rootDepth = root->rootDepth;
   vconf.root.nDepths = 1;

   vconf.rootp = root;
   /*
   **  Depth 1 is always listed and supported.
   */
   if (MALLOC(dp, depth_t *, sizeof(depth_t))) {
      vconf_init();
      return -1;
   }
   dp->depth.depth = 1;
   dp->depth.nVisuals = 0;
   dp->vp = 0;
   dp->match = 1;
   dp->next = 0;

   vconf.dp = dp;

   while (depth=(xDepth *)cblock_depth(0)) {
      if (depth->depth > 1) {
         if (MALLOC(dp->next, depth_t *, sizeof(depth_t))) {
            vconf_init();
            return -1;
         }
         dp = dp->next;
         dp->next = 0;

         dp->depth.depth = depth->depth;
         dp->depth.nVisuals = 0;
         dp->match = 1;

         vconf.root.nDepths++;
      }
      else
         dp = vconf.dp;

      for (vp=0; visual=(xVisualType *)cblock_visual(0);) {
         if (MALLOC(vn, vtype_t *, sizeof(vtype_t))) {
            vconf_init();
            return -1;
         }
         if (vp)
            vp->next = vn;
         else
            dp->vp = vn;
         (vp = vn)->next = 0;

         vp->visual.visualID = 0;	/* set later in those that survive */
         vp->visual.class = visual->class;
         vp->visual.bitsPerRGB = visual->bitsPerRGB;
         vp->visual.colormapEntries = visual->colormapEntries;
         vp->visual.redMask = visual->redMask;
         vp->visual.greenMask = visual->greenMask;
         vp->visual.blueMask = visual->blueMask;

         if (visual->visualID == root->rootVisualID)
            vconf.rootvp = vp;

         vp->visualp = visual;
         vp->match = 1;

         dp->depth.nVisuals++;
      }

      for (; dp->next; dp=dp->next);	/* paranoid */
   }
   /*
   **  Do formats next, to cull any that don't match a depth
   **  supported by our chosen root window.
   */
   vconf.setup.numFormats = 0;

   for (fp=0; format=(xPixmapFormat *)cblock_format(0);) {
      for (dp=vconf.dp; dp && dp->depth.depth != format->depth; dp=dp->next);
      if (dp == 0)
         continue;

      if (MALLOC(fn, format_t *, sizeof(format_t))) {
         vconf_init();
         return -1;
      }
      if (fp)
         fp->next = fn;
      else
         vconf.fp = fn;
      (fp = fn)->next = 0;

      fp->format.depth = format->depth;
      fp->format.bitsPerPixel = format->bitsPerPixel;
      fp->format.scanLinePad = format->scanLinePad;

      fp->formatp = format;
      fp->match = 1;

      vconf.setup.numFormats++;
   }
   /*
   **  Copy the list of extensions supported.
   */
   for (ep=0, i=0; i<sp->nexts; i++)
      if (ext_supported(sp->exts[i].name)) {
         if (MALLOC(en, extname_t *, sizeof(extname_t))) {
            vconf_init();
            return -1;
         }
         if (MALLOC(en->name, char *, strlen(sp->exts[i].name) + 1)) {
            vconf_init();
            return -1;
         }
         if (ep)
            ep->next = en;
         else
            vconf.ep = en;
         (ep = en)->next = 0;
   
         strcpy(ep->name, sp->exts[i].name);
         ep->match = 1;

	 vconf.nexts++;
      }

   if (check_constraints()) {
      vconf_init();
      return -1;
   }
   vcstate = VC_INFLUX;

   return 0;
}

/*
**  compare
**
**	Compare a server setup block to the virtual configuration.
**
**	Returns zero if the configuration given overlaps the virtual
**	configuration sufficiently to support a virtual X session.
**	Marks configuration component matches for subsequent calls
**	to merge, map or prune.
**
**	If this routine succeeds, then a mapping exists from all or
**	a subset of the virtual configuration to the server.  The
**	virtual configuration is left undisturbed in all cases.
*/
static int
compare
   AL((sp))
   DB server_t *sp
   DE
{
   register int i, lasti;
   register int nvisuals;
   register xConnSetupPrefix *prefix;
   register xConnSetup *setup;
   register xWindowRoot *root;
   register depth_t *dp, *lastdp;
   register xDepth *depth, *lastdepth;
   register format_t *fp;
   register xPixmapFormat *format, *lastformat;
   register vtype_t *vp;
   register extname_t *ep;
 
   if (vcstate == VC_MUSH)
      return make_like(sp);

   prefix = (xConnSetupPrefix *)cblock_prefix(sp->block);
   /*
   **  well, it's gotta be X11!
   */
   if (prefix->majorVersion != vconf.prefix.majorVersion)
      return -1;
   vconf.prefixp = prefix;
 
   setup = (xConnSetup *)cblock_setup(sp->block);
   vconf.setupp = setup;
   /*
   **  find designated screen
   */
   cblock_root(1);
   for (i=0; i<=sp->screen; i++)
      if ((root = cblock_root(0)) == 0) {  /* that screen doesn't exist */
         return -1;
      }
   vconf.rootp = root;
   /*
   **  compare depth list
   */
   nvisuals = 0;
   lastdepth = 0;
   (void) cblock_depth(1);

   for (dp=vconf.dp; dp; dp=dp->next) {
      dp->match = dp->depth.depth == 1 ? 1 : 0;	/* depth 1 always supported */
      do {
         if (depth = cblock_depth(0))
            if (dp->depth.depth == depth->depth) {
               dp->match = 1;
               /*
               **  compare visual list
               */
               closest_visual(0);	/* reset */
               for (vp=dp->vp; vp; vp=vp->next)
                  if (vp->visualp = closest_visual(&vp->visual)) {
                     vp->match = 1;
                     nvisuals++;
                  }
                  else
                     vp->match = 0;
               break;			/* exit loop */
            }
      } while (depth != lastdepth);

      lastdepth = depth;
   }
   if (nvisuals == 0)		/* no visuals, no match */
      return -1;

   /*
   **  match up formats
   */
   dp = lastdp = 0;
   lastformat = 0;
   cblock_format(1);
   /*
   **  match depths from three lists: the virtual config list of
   **  depths, the virtual config list of formats and the prospective
   **  joiner's list of formats.  if a particular depth is listed in
   **  all three, it is a match in the virtual config.
   **
   **  this is somewhat unnecessarily complicated.  i just hate
   **  excessive looping, so each inner loop picks up where it
   **  left off last time, automatically restarting and quitting
   **  if all are seen.
   */
   for (fp=vconf.fp; fp; fp=fp->next) {
      fp->match = 0;
      do {
         if (dp = dp ? dp->next : vconf.dp)
            if (fp->format.depth == dp->depth.depth) {
               if (dp->match) {	/* no depth, no format */
                  do {
                     if (format = cblock_format(0))
                        if (fp->format.depth == format->depth) {
                           fp->match = 1;
                           fp->formatp = format;
                           break;
                        }
                  } while (format != lastformat);
   
                  lastformat = format;
               }
               break;
            }
      } while (dp != lastdp);

      lastdp = dp;
   }
   /*
   **  mark the extensions that match
   */
   if (sp->nexts) {
      i = lasti = sp->nexts - 1;
      for (ep=vconf.ep; ep; ep=ep->next) {
         ep->match = 0;
         do {
            i = (i + 1) % sp->nexts;
            if (strcmp(ep->name, sp->exts[i].name) == 0) {
               ep->match = 1;
               lasti = i;
            }
         } while (i != lasti);
      }
   }
   return check_constraints();
}

/*
**  closest_visual
**
**	Compare the given xVisualType with those in the current cblock
**	and return the best match.
**
**	See the comments in the code below for an explanation of what
**	is the "best match."
*/
static xVisualType *
closest_visual
   AL((cmp))
   DB xVisualType *cmp
   DE
{
   xVisualType *tmp, *best;
   static int i, nseen, sz;
   static xVisualType **seen, **tseen;

   if (cmp == 0) {	/* reset */
      nseen = 0;
      return 0;
   }
   if (nseen >= sz) {
      if (MALLOC(tseen, xVisualType **, sizeof(xVisualType **) * (sz + 16)))
         return 0;
      if (seen) {
         bcopy((char *)seen, (char *)tseen, sizeof(xVisualType **) * sz);
         free(seen);
      }
      seen = tseen;
      sz += 16;
   }
   best = 0;

   (void) cblock_visual(1);	/* rewind list */

   while (tmp = cblock_visual(0))
      /*
      **  classes must match
      */
      if (tmp->class == cmp->class) {
         /*
         **  masks must match for visuals that have them
         */
         switch (tmp->class) {
            case DirectColor:
            case TrueColor:
               if (	tmp->redMask != cmp->redMask ||
			tmp->greenMask != cmp->greenMask ||
			tmp->blueMask != cmp->blueMask)
                  continue;
         }
         /*
         **  pick a visual with at least as many colormap entries, but
         **  if more, choose the one with the smallest number
         */
         if (	best == 0 ||
		(best->colormapEntries < cmp->colormapEntries ?
			tmp->colormapEntries > best->colormapEntries :
			tmp->colormapEntries < best->colormapEntries)) {
            /*
            **  never select one we've seen
            */
            for (i=0; i<nseen; i++)
               if (seen[i] == tmp)
                  break;
            if (i == nseen) {
               best = tmp;
               seen[nseen++] = tmp;
               /*
               **  short circuit search if a match is found
               */
               if (best->colormapEntries == cmp->colormapEntries)
                  break;		/* all done */
            }
         }
      }

   return best;
}

/*   prune
**
**	Unlink and free any nodes in the virtual config that have a
**	zero match field.
**
**	Note that this routine does not update the length field of
**	xConnSetupPrefix.  It should be calculated when needed.
**	
*/
static void
prune
   VOID
{
   register format_t *fp, **fpp;
   register depth_t *dp, **dpp;
   register vtype_t *vp, **vpp;
   register extname_t *ep, **epp;

   fpp = &vconf.fp;
   for (fp=vconf.fp; fp; fp = *fpp)
      if (fp->match)
         fpp = &fp->next;
      else {
         *fpp = fp->next;
         free(fp);
         vconf.setup.numFormats--;
      }
   dpp = &vconf.dp;
   for (dp=vconf.dp; dp; dp = *dpp) {
      vpp = &dp->vp;
      for (vp=dp->vp; vp; vp = *vpp)
         if (vp->match && dp->match)
            vpp = &vp->next;
         else {
            *vpp = vp->next;
            free(vp);
            dp->depth.nVisuals--;
         }
      if (dp->match)
         dpp = &dp->next;
      else {
         *dpp = dp->next;
         free(dp);
         vconf.root.nDepths--;
      }
   }
   epp = &vconf.ep;
   for (ep=vconf.ep; ep; ep = *epp)
      if (ep->match)
         epp = &ep->next;
      else {
         *epp = ep->next;
         free(ep);
         vconf.nexts--;
      }
}


/*
**  merge
**
**	Modify the virtual configuration to permit a mapping to the
**	given server.
*/
static int
merge
   AL((sp))
   DB server_t *sp
   DE
{
   register int i, j;
   register depth_t *dp;
   register vtype_t *vp, *rootvp;
   register uint_t rootdepth;

   if (compare(sp))
      return -1;

   prune();

   /*
   **  use smaller minor version number
   */
   if (vconf.prefix.minorVersion > vconf.prefixp->minorVersion)
      vconf.prefix.minorVersion = vconf.prefixp->minorVersion;
   /*
   **  use smaller motion buffer size
   */
   if (vconf.setup.motionBufferSize > vconf.setupp->motionBufferSize)
      vconf.setup.motionBufferSize = vconf.setupp->motionBufferSize;
   /*
   **  use smaller max request size
   */
   if (vconf.setup.maxRequestSize > vconf.setupp->maxRequestSize)
      vconf.setup.maxRequestSize = vconf.setupp->maxRequestSize;
   /*
   **  use largest screen dimensions
   */
   if (vconf.root.pixWidth < vconf.rootp->pixWidth)
      vconf.root.pixWidth = vconf.rootp->pixWidth;
   if (vconf.root.pixHeight < vconf.rootp->pixHeight)
      vconf.root.pixHeight = vconf.rootp->pixHeight;
   /*
   **  min min installed maps
   */
   if (vconf.root.minInstalledMaps > vconf.rootp->minInstalledMaps)
      vconf.root.minInstalledMaps = vconf.rootp->minInstalledMaps;
   /*
   **  max max installed maps
   */
   if (vconf.root.maxInstalledMaps < vconf.rootp->maxInstalledMaps)
      vconf.root.maxInstalledMaps = vconf.rootp->maxInstalledMaps;
   /*
   **  use least backing store
   */
   if (vconf.root.backingStore > vconf.rootp->backingStore)
      vconf.root.backingStore = vconf.rootp->backingStore;
   /*
   **  only offer saveunders if everyone can
   */
   if (vconf.root.saveUnders && !vconf.rootp->saveUnders)
      vconf.root.saveUnders = vconf.rootp->saveUnders;
   /*
   **        the following loop does not explicitly compare the vconf
   **        to the block, but rather ensures the self-consistency of
   **        the vconf following a prune.  compare ensures that a
   **        rootdepth and rootvp will be found here.
   */
   rootdepth = 0;		/* make sure that rootDepth and... */
   for (dp=vconf.dp; dp; dp=dp->next) {
      if (vconf.root.rootDepth == dp->depth.depth || rootdepth == 0) {
         rootdepth = dp->depth.depth;
         rootvp = 0;		/* ...rootVisualID exist */
         for (vp=dp->vp; vp; vp=vp->next)
            if (vconf.rootvp == vp || rootvp == 0)
               rootvp = vp;
      }
      if (rootvp == 0)
         rootdepth = 0;
   }
   vconf.root.rootDepth = rootdepth;	/* if they don't exist, */
   vconf.rootvp = rootvp;		/* assign ones that do */

   return 0;
}

/*
**  check_constraints
**
**	Does the current virtual config conform to the current
**	set of constraints?  Honors match information in the virtual
**	config.  This should be run immediately after a compare().
*/
static int
check_constraints
   VOID
{
   register int i;
   register int failed;
   register u8_t mask;
   register dcons_t *dcp;
   register depth_t *dp, *lastdp;
   register vtype_t *vp;
   register extname_t *ep, *lastep;

   /*
   **  are all the depths required present?
   */
   if (vcons.mask & VCSET_DEPTHS) {
      dp = lastdp = 0;
      for (dcp=vcons.dcp; dcp; dcp=dcp->next) {
         failed = 1;
         do {
            if (dp = dp ? dp->next : vconf.dp)
               if (dcp->depth == dp->depth.depth)
                  if (dp->match) {
                     for (mask=0, vp=dp->vp; vp; vp=vp->next)
                        if (vp->match)
                           mask |= 1 << vp->visual.class;
                     if ((dcp->classmask & mask) == dcp->classmask) {
                        failed = 0;
                        break;
                     }
                  }
         } while (dp != lastdp);
         lastdp = dp;

         if (failed)
            return -1;
      }
   }
   /*
   **  are all the extensions required present?
   */
   if (vcons.mask & VCSET_EXTS) {
      ep = lastep = 0;
      for (i=0; i<vcons.nextnames; i++) {
         failed = 1;
         do {
            if (ep = ep ? ep->next : vconf.ep)
               if (ep->match)
                  if (strcmp(vcons.extnamep[i], ep->name) == 0) {
                     failed = 0;
                     break;
                  }
         } while (ep != lastep);
         lastep = ep;

         if (failed)
            return -1;
      }
   }
   return 0;
}

/*
**  apply_constraints
**
**	Set the screen size and root depth and visual, if needed.
**	If check_constraints was called diligently, prior to this,
**	then this routine cannot fail.
*/
static int
apply_constraints
   VOID
{
   register u32_t old, new;
   register depth_t *dp;
   register vtype_t *vp, *tvp;

   if (vcons.mask & VCSET_SIZE) {
      old = (u32_t)vconf.root.pixWidth;
      new = (u32_t)vcons.width;
      vconf.root.mmWidth = (((u32_t)vconf.root.mmWidth * new) / old);

      old = (u32_t)vconf.root.pixHeight;
      new = (u32_t)vcons.height;
      vconf.root.mmHeight = (((u32_t)vconf.root.mmHeight * new) / old);

      vconf.root.pixWidth = vcons.width;
      vconf.root.pixHeight = vcons.height;
   }
   if (vcons.mask & VCSET_DEPTHS) {
      vconf.root.rootDepth = vcons.rootdepth;

      for (dp=vconf.dp; dp; dp=dp->next)
         if (dp->depth.depth == vcons.rootdepth)
            break;
      if (dp == 0)
         return -1;	/* desired root depth not found */

      for (tvp=0,vp=dp->vp; vp; vp=vp->next)
         if (vp->visual.class == vcons.rootclass)
            if (vp == vconf.rootvp)
               break;	/* all done */
            else
               tvp = vp;	/* candidate for root visual */
      if (vp == 0)
         if (tvp)
            vconf.rootvp = tvp;
         else
            return -1;	/* desired root visual not found */
   }
   return 0;
}

/*
**  free depth constraint list
*/
void
free_dcons
   VOID
{
   register dcons_t *dcp, *ldcp;

   for (dcp=vcons.dcp; ldcp=dcp;) {
      dcp = dcp->next;
      free(ldcp);
   }
   vcons.dcp = 0;
}

/*
**  free extension constraint list
*/
void
free_econs
   VOID
{
   register int i;

   for (i=0; vcons.extnamep[i]; i++)
      free(vcons.extnamep[i]);
   free(vcons.extnamep);

   vcons.extnamep = 0;
}

/*
**  free virtual config format list
*/
static void
free_formats
   VOID
{
   register format_t *fp, *lfp;

   for (fp=vconf.fp; lfp=fp;) {
      fp = fp->next;
      free(lfp);
   }
}

/*
**  free virtual config depth list
*/
static void
free_depths
   VOID
{
   register depth_t *dp, *ldp;
   register vtype_t *vp, *lvp;

   for (dp=vconf.dp; ldp=dp;) {
      dp = dp->next;
      for (vp=dp->vp; lvp=vp;) {
         vp = vp->next;
         free(lvp);
      }
      free(ldp);
   }
}

/*
**  free virtual config protocol extension list
*/
static void
free_extnames
   VOID
{
   register extname_t *ep, *lep;

   for (ep=vconf.ep; lep=ep;) {
      ep = ep->next;
      free(lep->name);
      free(lep);
   }
}
