/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   sres.c								*
*									*
*	Server resources.						*
*									*
************************************************************************/
#include <X11/Xproto.h>
#include <X11/X.h>
#include "xmx.h"
#include "df.h"
#include "incl/sres.pvt.h"

static visual_t *visuals;

/************************************************************************
*									*
*   sres_init_depths							*
*   sres_new_depth							*
*									*
*	Initialize the depth vector component of the virtual screen.	*
*	This is done in two parts because the source of the data is	*
*	a linked list in the virtual configuration and that list is	*
*	walked in vconf_fix, not here.					*
*									*
************************************************************************/
void
sres_init_depths
   AL((ndepths))
   DB int ndepths
   DE
{
   register int i;

   if (MALLOC(vscreen.dv, screendepth_t *, ndepths * sizeof(screendepth_t)))
      quit(-1, "sres_init_depths: malloc error fatal\n");

   vscreen.ndepths = ndepths;

   for (i=0; i<ndepths; i++) {
      vscreen.dv[i].pixmap = util_new_client_id();	/* pre-allocate rids */
      vscreen.dv[i].gc = util_new_client_id();
      hash_add(vmap, vscreen.dv[i].pixmap, vscreen.dv[i].pixmap, 0);
      hash_add(vmap, vscreen.dv[i].gc, vscreen.dv[i].gc, 0);
      vscreen.dv[i].depth = 0;
   }
   vscreen.dv[0].depth = 1;		/* first is always depth one */
}

void
sres_new_depth
   AL((depth, rootdepth))
   DB u8_t depth
   DD u8_t rootdepth
   DE
{
   register int i;

   if (depth > 1) {
      for (i=1; i<vscreen.ndepths; i++)
         if (vscreen.dv[i].depth == 0) {
            vscreen.dv[i].depth = depth;
            break;
         }
      if (i == vscreen.ndepths) {
         warn("sres_new_depth: sanity check - can't happen\n");
         return;
      }
   }
   else
      i = 0;

   if (depth == rootdepth)
      vscreen.dp = &vscreen.dv[i];
}

/************************************************************************
*									*
*   sres_new_visual							*
*									*
************************************************************************/
int
sres_new_visual
   AL((depth, visual))
   DB u8_t depth
   DD xVisualType *visual
   DE
{
   visual_t *vp;

   if (MALLOC(vp, visual_t *, sizeof(visual_t)))
      return -1;

   visual->visualID = util_new_server_id();

   vp->cid = visual->visualID;
   vp->class = visual->class;
   vp->depth = depth;
   vp->bprgb = visual->bitsPerRGB;
   vp->nentries = visual->colormapEntries;
   vp->rmask = visual->redMask;
   vp->gmask = visual->greenMask;
   vp->bmask = visual->blueMask;

   vp->next = visuals;
   visuals = vp;

   hash_add(vmap, vp->cid, vp->cid, (char *)vp);

   return 0;
}

/************************************************************************
*									*
*   sres_init_screen							*
*									*
*	Note: this routine needs to clean up better on failure!		*
*									*
************************************************************************/
int
sres_init_screen
   AL((root, rootVisualID))
   DB xWindowRoot *root
   DD rid_t rootVisualID
   DE
{
   register int i, maxcols;
   register rid_t shellroot;
   register window_t *wp;
   register drawable_t *dp;	/* shell window */
   register colormap_t *mp;
   register visual_t *vp;

   if ((vp = (visual_t *)hash_data(vmap, rootVisualID)) == 0)
      return -1;
   /*
   **  Assign new resource ids
   */
   root->windowId = util_new_server_id();
   root->defaultColormap = util_new_server_id();
   root->rootVisualID = rootVisualID;
   /*
   **	Establish upper limit on number of cells allocable in server
   **	default colormaps.
   */
   maxcols = opt.maxcolsp->abs ?
			opt.maxcolsp->n : vp->nentries * opt.maxcolsp->r;

   if (maxcols < 2)
      maxcols = 2;
   else if (maxcols > (int)vp->nentries)
      maxcols = vp->nentries;

   shellroot = util_new_server_id();

   mp = color_root_cmap(root->defaultColormap,
			vp,
			&root->whitePixel,
			&root->blackPixel,
			vp->nentries - maxcols,
			shellroot,
			root->windowId,
			root->rootDepth);
   if (mp == 0)
      return -1;

   wp = window_new(	0,				/* client */
			root->rootDepth,		/* depth */
			root->windowId,			/* client id */
			1,				/* mapped? */
			0, 0,				/* x, y */
			root->pixWidth, root->pixHeight,/* width, height */
			0,				/* borderwidth */
			InputOutput,			/* window class */
			vp);				/* visual */

   if (wp == 0)
      return -1;

   wp->vid = root->windowId;       /* vid not used?  TODO */

   if (MALLOC(dp, drawable_t *, sizeof(drawable_t))) {
      free(wp);
      free(vscreen.dv);
      return -1;
   }
   dp->res.client = 0;
   dp->type = W_SHELL;
   dp->depth = root->rootDepth;
   dp->width = root->pixWidth;
   dp->height = root->pixHeight;

   vscreen.whitePixel = root->whitePixel;
   vscreen.blackPixel = root->blackPixel;
   vscreen.pixWidth = root->pixWidth;
   vscreen.pixHeight = root->pixHeight;
   vscreen.mmWidth = root->mmWidth;
   vscreen.mmHeight = root->mmHeight;
   vscreen.minInstalledMaps = root->minInstalledMaps;
   vscreen.maxInstalledMaps = root->maxInstalledMaps;
   vscreen.maxcols = maxcols;
   vscreen.backingStore = root->backingStore;
   vscreen.saveUnders = root->saveUnders;
   vscreen.colormap = mp;
   vscreen.vservroot = util_new_server_id();	/* maps to real root */
   vscreen.vservcmap = util_new_server_id();	/* maps to real default cmap */
   vscreen.rootpixmap = util_new_client_id();
   vscreen.borderpixmap = util_new_client_id();
   vscreen.rootcmap = root->defaultColormap; /* virtual default cmap */
   vscreen.iconpixmap = util_new_client_id();
   vscreen.shellroot = shellroot;	/* maps to shell window */
   vscreen.rootvid = util_new_client_id();
   vscreen.cmapvid = util_new_client_id();
   vscreen.cursor = util_new_client_id();
   vscreen.shdp = dp;		/* shell window info */
   vscreen.wp = wp;
   vscreen.ndepths = root->nDepths;

   for (i=0; i<IWCount; i++)    /* initialize attributes */
      switch (i) {
         case IWBackPixmap:	/* root window default - cannot be queried */
            wp->atts[i] = vscreen.rootpixmap;
            break;
         case IWBorderPixmap:	/* root window default - cannot be queried */
            wp->atts[i] = vscreen.borderpixmap;
            break;
         case IWColormap:	/* root window default */
            wp->atts[i] = (u32_t)vscreen.colormap->cid;
            break;
         case IWCursor:		/* root window default */
            wp->atts[i] = (u32_t)vscreen.cursor;
            break;
         default:
            wp->atts[i] = win_defs[i];
      }
   wp->mp = mp;
   color_winlist_add(mp, wp);

   wp->parent = 0;
   wp->prevsib = 0;
   wp->nextsib = 0;
   /*
   **  root window event mask is special
   */
   imask_put(wp->xmask, (char *)0, IE_ROOTSET);
   MkCompMask(wp);

#ifdef DEBUG
   if (sizeof(screen_t) != 92)
      warn("sres_new_screen: not updated since screen_t last changed\n");
#endif

			/* these are for hash_map */
   hash_add(vmap, vscreen.rootpixmap, vscreen.rootpixmap, 0);
   hash_add(vmap, vscreen.iconpixmap, vscreen.iconpixmap, 0);
   hash_add(vmap, vscreen.cmapvid, vscreen.cmapvid, 0);
   hash_add(vmap, vscreen.cursor, vscreen.cursor, 0);

			/* this is for hash_data, only */
   hash_add(vmap, wp->cid, vscreen.rootvid, (char *)wp);
   hash_add(vmap, vscreen.shellroot, vscreen.shellroot, (char *)dp);

   /* clients never see rootcmap, they see      */
   /* (vconf->) rp->root.defaultColormap        */
   /* so no mapping is created here             */

   /* TODO - these should be elsewhere */
   focus_assign(wp, 0);
   inp_motion_init(wp);

   if (opt.geomp) {
   }
   return 0;
}

/************************************************************************
*									*
*   sres_reset								*
*									*
*	Free virtual server resources so that they may be recreated	*
*	for a server reset.						*
*									*
*	We assume that hash table entries for these resources are	*
*	reset, both in the vmap and each server map (smap).		*
*									*
*	Be careful to reset focuswp TODO				*
*									*
************************************************************************/
void
sres_reset
   VOID
{
   visual_t *vp, *lvp;

   for (vp=visuals; lvp=vp;) {
      vp = vp->next;
      free(lvp);
   }

   if (vscreen.colormap)
         /* how free this? TODO */;
   free(vscreen.wp);
}

/************************************************************************
*									*
*   sres_match								*
*									*
*	Verify that a depth and visual are supported by the screen.	*
*									*
************************************************************************/
int
sres_match
   AL((depth, vp))
   DB u8_t depth
   DD visual_t *vp
   DE
{
   register int i;

   if (vp->depth == depth)
      for (i=0; i<vscreen.ndepths; i++)
         if (vscreen.dv[i].depth == depth)
            return 0;

   return 1;
}
