/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   ptc.h								*
*									*
*	Protocol translation code macros.				*
*									*
************************************************************************/

#define PTC_BLOCKSZ	256
#define PTC_LASTPC	(PTC_BLOCKSZ - sizeof(op_jump_t))

/************************************************************************
*									*
*   ugly, nasty address alignment cruft					*
*									*
************************************************************************/
typedef struct {char a; u32_t b;}s_word_t;
typedef struct {char a; void *b;}s_addr_t;

/*
** Address multiple required for ints and pointers, respectively, in memory.
**
** These are compile-time constants, nice nice.
*/
#define WORDALIGN (sizeof(s_word_t) - sizeof(u32_t))
#define ADDRALIGN (sizeof(s_addr_t) - sizeof(void *))

/************************************************************************
*									*
*   opcodes								*
*									*
************************************************************************/
/*
** ptc flow of control
*/
#define PTC_DONE_OP		0	/* all done */
#define PTC_SKIP_OP		1	/* do nothing, pc+=2+*(pc+1) */
#define PTC_JUMP_OP		2	/* increment block, pc=0 */
/*
** protocol primitive operations
*/
/*
** request primitives
*/
#define PTC_ATOM_OP		3
#define PTC_CLIENT_RID_OP	4
#define PTC_SERVER_RID_OP	5
#define PTC_EVENT_MASK_OP	6
#define PTC_ROOT_PIXEL_OP	7
#define PTC_IMAGE_LEN_OP	8
#define PTC_G_EXPOSE_OP		9
/*
** for protocol extensions
*/
#define PTC_EXTENSION_OP	10

/************************************************************************
*									*
*   types								*
*									*
************************************************************************/
typedef struct _ptc_t {
   uint_t		pc;	/* program counter */
   u16_t		count;	/* use depends on context */
   u8_t			clinum;	/* client that originated protocol */
   u16_t		seqno;	/* base client seq number for this block */
   char *		head;
   char *		tail;
   struct _ptc_t *	next;
}ptc_t;

#define ptc_clinum(chp)		((ptc_t *)((chp)->dptr))->clinum
#define ptc_seqno(chp)		((ptc_t *)((chp)->dptr))->seqno
#define ptc_count(chp)		((ptc_t *)((chp)->dptr))->count

/************************************************************************
*									*
*   instructions							*
*									*
************************************************************************/
typedef struct {
	u8_t		opcode;		/* PTC_DONE_OP */
	u8_t		pad;
}op_done_t;

typedef struct {
	u8_t		opcode;		/* PTC_SKIP_OP */
	u8_t		count;			/* must be >1 */
}op_skip_t;

typedef struct {
	u8_t		opcode;		/* PTC_JUMP_OP */
	char *		blkp;
}op_jump_t;

typedef struct {
	u8_t		opcode;		/* PTC_ATOM_OP */
	u16_t		offset;
	atom_t		atom;
}op_atom_t;

typedef struct {
	u8_t		opcode;		/* PTC_CLIENT_RID_OP or PTC_SERVER_RID_OP */
	u16_t		offset;
	rid_t		rid;
}op_rid_t;

typedef struct {
	u8_t		opcode;		/* PTC_EVENT_MASK_OP */
	u16_t		offset;
	mask_t		mask;
}op_mask_t;

typedef struct {
	u8_t		opcode;		/* PTC_ROOT_PIXEL_OP */
	u16_t		offset;
	pixel_t		pixel;
}op_pixel_t;

typedef struct {
	u8_t		opcode;		/* PTC_IMAGE_LEN_OP */
	u8_t		format;
	u16_t		offset;
	u16_t		length;
	u16_t		width;
	u16_t		height;
	u8_t		leftPad;
	u8_t		depth;
}op_imagelen_t;

typedef struct {
	u8_t		opcode;		/* PTC_G_EXPOSE_OP */
	u16_t		offset;
}op_gexpose_t;

typedef struct {
	u8_t		opcode;		/* PTC_EXTENSION_OP */
	u8_t		code;		/* extension code */
	u16_t		length;		/* length of op */
}op_extension_t;


/************************************************************************
*									*
*   Ops - inlined							*
*									*
************************************************************************/
#define ptc_empty(ptcp)	((ptcp)->pc == 0 && (ptcp)->head == (ptcp)->tail)

#define PTC_ALIGN(ptcp, _m) {\
	register int n;\
\
	n = (ptcp)->pc % (_m);\
	if (n)\
	   PTC_SKIP(ptcp, n);\
}

#define PTC_SKIP(ptcp, _n) {\
	register op_skip_t *op;\
	op = (op_skip_t *)&ptcp->tail[(ptcp)->pc];\
	op->opcode = PTC_SKIP_OP;\
	op->count = (_n);\
	(ptcp)->pc += (_n);\
}

#define PTC_JUMP(ptcp, _blkp) {\
	register op_jump_t *op;\
\
	if ((ptcp)->pc < PTC_LASTPC) {\
		PTC_SKIP((ptcp), PTC_LASTPC - (ptcp)->pc);\
	}\
	op = (op_jump_t *)&ptcp->tail[(ptcp)->pc];\
	op->opcode = PTC_JUMP_OP;\
	op->blkp = (_blkp);\
}

#define PTC_NEXT(ptcp, _type, _ptr)\
	if ((ptcp)->pc > PTC_LASTPC - sizeof(_type)) {\
		register char *_cp;\
\
		_cp = (char *)ptc_new_block();\
		PTC_JUMP((ptcp), _cp);\
		(ptcp)->tail = _cp;\
		(ptcp)->pc = 0;\
	}\
	(_ptr) = (_type *)&ptcp->tail[(ptcp)->pc];\
	(ptcp)->pc += sizeof(_type)

/*
**   Primitives
*/
#define PTC_DONE(ptcp) {\
	register op_done_t *op;\
\
	PTC_NEXT(ptcp, op_done_t, op);\
	op->opcode = PTC_DONE_OP;\
}

#define PTC_GET(bp, clinum, seqno) ((buf_chunk(bp))->type == P_REQUEST ?\
	(ptc_t *)((buf_chunk(bp))->dptr) : (ptc_t *)ptc_set(bp, clinum, seqno))

#define PTC_INCR_COUNT(ptcp)	(ptcp)->count++
#define PTC_CLEAR_COUNT(ptcp)	(ptcp)->count = 0
#define PTC_COUNT(ptcp)		((ptcp)->count)

#define PTC_ATOM(ptcp, basep, _atom) {\
	register op_atom_t *op;\
\
	PTC_ALIGN(ptcp, WORDALIGN);\
	PTC_NEXT(ptcp, op_atom_t, op);\
	op->opcode = PTC_ATOM_OP;\
	op->offset = (u16_t)((char *)&(_atom) - (basep));\
	op->atom = (_atom);\
}

#define PTC_RID(ptcp, basep, _id) {\
	register op_rid_t *op;\
\
	PTC_ALIGN(ptcp, WORDALIGN);\
	PTC_NEXT(ptcp, op_rid_t, op);\
	if ((_id) & RID_SERVER_MASK) {\
		op->opcode = PTC_SERVER_RID_OP;\
		op->rid = (_id);\
	}\
	else {\
		op->opcode = PTC_CLIENT_RID_OP;\
		op->rid = hash_map(vmap, (_id));\
	}\
	op->offset = (u16_t)((char *)&(_id) - (basep));\
}

#define PTC_EVENTMASK(ptcp, basep, _mask) {\
	register op_mask_t *op;\
\
	PTC_ALIGN(ptcp, WORDALIGN);\
	PTC_NEXT(ptcp, op_mask_t, op);\
	op->opcode = PTC_EVENT_MASK_OP;\
	op->mask = _mask;\
	op->offset = (u16_t)((char *)&(_mask) - (basep));\
}

#define PTC_PIXEL(ptcp, basep, _pxl) {\
	register op_pixel_t *op;\
\
	PTC_ALIGN(ptcp, WORDALIGN);\
	PTC_NEXT(ptcp, op_pixel_t, op);\
	op->opcode = PTC_ROOT_PIXEL_OP;\
	op->pixel = (_pxl);\
	op->offset = (u16_t)((char *)&(_pxl) - (basep));\
}

#define PTC_IMAGELEN(ptcp, basep, len, _format, _width, _height, _lpad, _depth) {\
	register op_imagelen_t *op;\
\
	PTC_NEXT(ptcp, op_imagelen_t, op);\
	op->opcode = PTC_IMAGE_LEN_OP;\
	op->format = (_format);\
	op->width = (_width);\
	op->height = (_height);\
	op->leftPad = (_lpad);\
	op->depth = (_depth);\
	op->offset = (u16_t)((char *)&(len) - (basep));\
}

#define PTC_GEXPOSE(ptcp, basep, _offset) {\
	register op_gexpose_t *op;\
\
	PTC_NEXT(ptcp, op_gexpose_t, op);\
        op->opcode = PTC_G_EXPOSE_OP;\
        op->offset = (u16_t)((char *)&(_offset) - (basep));\
}

#define PTC_EXTENSION(ptcp, basep, _code) {\
	register op_extension_t *op;\
\
	PTC_NEXT(ptcp, op_extension_t, op);\
	op->opcode = PTC_EXTENSION_OP;\
	op->code = (_code);\
}
