/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   proto.c								*
*									*
*	Like an Xlib that writes to memory (ie buffers), rather than	*
*	i/o streams.  Note that this is deliberately incomplete.  Only	*
*	the protocol requests actually needed by xmx are defined here.	*
*	They're easy enough to add.					*
*									*
*	Some quirky things:  replies and errors take a client pointer	*
*	as an argument and always queue the result.  That's because	*
*	they are one per chunk and always have a single destination.	*
*	Requests and events may go to multiple destinations, so they	*
*	must be queued outside of these routines.			*
*									*
*	This construction is primarily to reduce typing.		*
*									*
************************************************************************/
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/Xutil.h>
#define NEED_REPLIES
#define NEED_EVENTS
#include <X11/Xproto.h>
/* doh! */
#ifndef sz_xGetModifierMappingReply
#define sz_xGetModifierMappingReply sz_xReply
#endif

#include "xmx.h"
#include "am.h"
#include "zb.h"
#include "ptc.h"
#include "ptc_x.h"
#include "rs.h"
#include "incl/proto.pvt.h"

static char padsrc[4];		/* null bytes for padding */

/*
**       proto_ConnClientPrefix
*/
void
proto_ConnClientPrefix
   AL((nProto, authProto, nString, authString))
   DB u16_t nProto
   DD char *authProto
   DD u16_t nString
   DD char *authString
   DE
{
   register buffer_t *bp = zb_fbuf();
   xConnClientPrefix *p;

   /*
   **  need a clean buffer, since this is not a P_REQUEST, but goes
   **  out as a P_NONE, since no translation is necessary.
   */
   if (buf_active(bp))
      zb_queue(0);

   if (buf_avail(bp) < sz_xConnClientPrefix)
      buf_adjust(bp, sz_xConnClientPrefix);

   p = (xConnClientPrefix *)buf_next(bp);

   p->byteOrder = endian;
   p->majorVersion = 11;
   p->minorVersion = 0;
   p->nbytesAuthProto = nProto;
   p->nbytesAuthString = nString;

   buf_grow(bp, sz_xConnClientPrefix);

   if (nProto) {
      buf_put(bp, authProto, nProto);
      if (PAD(nProto))
         buf_put(bp, padsrc, PAD(nProto));
   }
   if (nString) {
      buf_put(bp, authString, nString);
      if (PAD(nString))
         buf_put(bp, padsrc, PAD(nString));
   }
}

/*
**       proto_ConnSetup_deny
*/
void
proto_ConnSetup_deny
   AL((cp, len, reason))
   DB client_t *cp
   DD u8_t len
   DD char *reason
   DE
{
   register buffer_t *bp = zb_rbuf();
   register uint_t xtra;
   xConnSetupPrefix *p;

   xtra = len + PAD(len);
   if (buf_avail(bp) < sz_xConnSetupPrefix + xtra)
      buf_adjust(bp, sz_xConnSetupPrefix + xtra);

   zb_client_dest(cp);

   p = (xConnSetupPrefix *)buf_next(bp);

   p->success = 0;
   p->lengthReason = len;
   p->majorVersion = 11;
   p->minorVersion = 0;
   p->length = xtra / 4;

   buf_grow(bp, sz_xConnSetupPrefix);
   if (len) {
      buf_put(bp, reason, len);
      if (PAD(len))
         buf_put(bp, padsrc, PAD(len));
   }
   (buf_chunk(bp))->type = P_CONNECT;
   zb_client_queue();
}

/************************************************************************
*	REQUESTS							*
************************************************************************/

/*
** [001] proto_CreateWindow
*/
void
proto_CreateWindow
   AL((depth, rid, parent, x, y, w, h, bw, class, vid, mask, values, mappixels))
   DB u8_t depth
   DD rid_t rid
   DD rid_t parent
   DD s16_t x
   DD s16_t y
   DD u16_t w
   DD u16_t h
   DD u16_t bw
   DD u16_t class
   DD rid_t vid
   DD mask_t mask
   DD u32_t *values
   DD int mappixels
   DE
{
   register buffer_t *bp = zb_fbuf();
   register int n;
   register uint_t xtra, len;
   xCreateWindowReq *p;

   NUMONES(mask, n);
   xtra = n * 4;
   len = sz_xCreateWindowReq + xtra;
   zb_mon(len);

   if (buf_avail(bp) < len)
      buf_adjust(bp, len);

   p = (xCreateWindowReq *)buf_next(bp);

   p->reqType = X_CreateWindow;
   p->depth = depth;
   p->length = sz_xCreateWindowReq / 4 + n;
   p->wid = rid;
   p->parent = parent;
   p->x = x;
   p->y = y;
   p->width = w;
   p->height = h;
   p->borderWidth = bw;
   p->class = class;
   p->visual = vid;
   p->mask = mask;

   buf_grow(bp, sz_xCreateWindowReq);
   if (n)
      buf_put(bp, (char *)values, xtra);

   zb_incr_seqno();
   ptc_CreateWindow(bp, 0, zb_seqno(), p, mappixels);
}

/*
** [002] proto_ChangeWindowAttributes
*/
void
proto_ChangeWindowAttributes
   AL((window, mask, values, mappixels))
   DB rid_t window
   DD mask_t mask
   DD u32_t *values
   DD int mappixels
   DE
{
   register buffer_t *bp = zb_fbuf();
   register int n;
   register uint_t len, xtra;
   xChangeWindowAttributesReq *p;

   NUMONES(mask, n);
   xtra = n * 4;
   len = sz_xChangeWindowAttributesReq + xtra;
   zb_mon(len);

   if (buf_avail(bp) < len)
      buf_adjust(bp, len);

   p = (xChangeWindowAttributesReq *)buf_next(bp);

   p->reqType = X_ChangeWindowAttributes;
   p->length = sz_xChangeWindowAttributesReq / 4 + n;
   p->window = window;
   p->valueMask = mask;

   buf_grow(bp, sz_xChangeWindowAttributesReq);
   if (n)
      buf_put(bp, (char *)values, xtra);

   zb_incr_seqno();
   ptc_ChangeWindowAttributes(bp, 0, zb_seqno(), p, mappixels);
}

/*
** [004] proto_DestroyWindow
*/
void
proto_DestroyWindow
   AL((w))
   DB rid_t w
   DE
{
   register buffer_t *bp = zb_fbuf();
   xResourceReq *p;

   zb_mon(sz_xResourceReq);

   if (buf_avail(bp) < sz_xResourceReq)
      buf_adjust(bp, sz_xResourceReq);

   p = (xResourceReq *)buf_next(bp);

   p->reqType = X_DestroyWindow;
   p->length = sz_xResourceReq / 4;
   p->id = w;

   buf_grow(bp, sz_xResourceReq);

   zb_incr_seqno();
   ptc_DestroyWindow(bp, 0, zb_seqno(), p);
}
   

/*
** [007] proto_ReparentWindow
*/
void
proto_ReparentWindow
   AL((window, parent, x, y))
   DB rid_t window
   DD rid_t parent
   DD int x
   DD int y
   DE
{
   register buffer_t *bp = zb_fbuf();
   xReparentWindowReq *p;

   zb_mon(sz_xReparentWindowReq);

   if (buf_avail(bp) < sz_xReparentWindowReq)
      buf_adjust(bp, sz_xReparentWindowReq);

   p = (xReparentWindowReq *)buf_next(bp);

   p->reqType = X_ReparentWindow;
   p->length = sz_xReparentWindowReq / 4;
   p->window = window;
   p->parent = parent;
   p->x = x;
   p->y = y;

   buf_grow(bp, sz_xReparentWindowReq);

   zb_incr_seqno();
   ptc_ReparentWindow(bp, 0, zb_seqno(), p);
}

/*
** [008] proto_MapWindow
*/
void
proto_MapWindow
   AL((w))
   DB rid_t w
   DE
{
   register buffer_t *bp = zb_fbuf();
   xResourceReq *p;

   zb_mon(sz_xResourceReq);

   if (buf_avail(bp) < sz_xResourceReq)
      buf_adjust(bp, sz_xResourceReq);

   p = (xResourceReq *)buf_next(bp);

   p->reqType = X_MapWindow;
   p->length = sz_xResourceReq / 4;
   p->id = w;

   buf_grow(bp, sz_xResourceReq);

   zb_incr_seqno();
   ptc_MapWindow(bp, 0, zb_seqno(), p);
}

/*
** [010] proto_UnmapWindow
*/
void
proto_UnmapWindow
   AL((w))
   DB rid_t w
   DE
{
   register buffer_t *bp = zb_fbuf();
   xResourceReq *p;

   zb_mon(sz_xResourceReq);

   if (buf_avail(bp) < sz_xResourceReq)
      buf_adjust(bp, sz_xResourceReq);

   p = (xResourceReq *)buf_next(bp);

   p->reqType = X_UnmapWindow;
   p->length = sz_xResourceReq / 4;
   p->id = w;

   buf_grow(bp, sz_xResourceReq);

   zb_incr_seqno();
   ptc_UnmapWindow(bp, 0, zb_seqno(), p);
}

/*
** [012] proto_ConfigureWindow
*/
void
proto_ConfigureWindow
   AL((w, mask, values))
   DB rid_t w
   DD mask_t mask
   DD u32_t *values
   DE
{
   register buffer_t *bp = zb_fbuf();
   register int n;
   register uint_t len, xtra;
   xConfigureWindowReq *p;

   NUMONES(mask, n);
   xtra = n * 4;
   len = sz_xConfigureWindowReq + xtra;
   zb_mon(len);

   if (buf_avail(bp) < len)
      buf_adjust(bp, len);

   p = (xConfigureWindowReq *)buf_next(bp);

   p->reqType = X_ConfigureWindow;
   p->length = sz_xConfigureWindowReq / 4 + n;
   p->window = w;
   p->mask = (u16_t)mask;

   buf_grow(bp, sz_xConfigureWindowReq);
   if (n)
      buf_put(bp, (char *)values, xtra);

   zb_incr_seqno();
   ptc_ConfigureWindow(bp, 0, zb_seqno(), p);
}

/*
** [014] proto_GetGeometry
*/
void
proto_GetGeometry
   AL((w))
   DB rid_t w
   DE
{
   register buffer_t *bp = zb_fbuf();
   xResourceReq *p;

   zb_mon(sz_xResourceReq);

   if (buf_avail(bp) < sz_xResourceReq)
      buf_adjust(bp, sz_xResourceReq);

   p = (xResourceReq *)buf_next(bp);

   p->reqType = X_GetGeometry;
   p->length = sz_xResourceReq / 4;
   p->id = w;

   buf_grow(bp, sz_xResourceReq);

   zb_incr_seqno();
   ptc_GetGeometry(bp, 0, zb_seqno(), p);
}

/*
** [016] proto_InternAtom
*/
void
proto_InternAtom
   AL((ifexists, name, len))
   DB u8_t ifexists
   DD char *name
   DD int len
   DE
{
   register buffer_t *bp = zb_fbuf();
   register uint_t totlen, xtra;
   xInternAtomReq *p;

   xtra = len + PAD(len);
   totlen = sz_xInternAtomReq + xtra;
   zb_mon(totlen);

   if (buf_avail(bp) < totlen)
      buf_adjust(bp, totlen);

   p = (xInternAtomReq *)buf_next(bp);

   p->reqType = X_InternAtom;
   p->length = (sz_xInternAtomReq + xtra) / 4;
   p->onlyIfExists = ifexists;
   p->nbytes = len;

   buf_grow(bp, sz_xInternAtomReq);
   if (len) {
      buf_put(bp, name, len);
      if (PAD(len))
         buf_put(bp, padsrc, PAD(len));
   }

   zb_incr_seqno();
   ptc_InternAtom(bp, 0, zb_seqno(), p);
}

/*
** [017] proto_GetAtomName
*/
void
proto_GetAtomName
   AL((atom))
   DB atom_t atom
   DE
{
   register buffer_t *bp = zb_fbuf();
   xResourceReq *p;

   zb_mon(sz_xResourceReq);

   if (buf_avail(bp) < sz_xResourceReq)
      buf_adjust(bp, sz_xResourceReq);

   p = (xResourceReq *)buf_next(bp);

   p->reqType = X_GetAtomName;
   p->length = sz_xResourceReq / 4;
   p->id = atom;

   buf_grow(bp, sz_xResourceReq);

   zb_incr_seqno();
   ptc_GetAtomName(bp, 0, zb_seqno(), p);
}

/*
** [018] proto_ChangeProperty
*/
void
proto_ChangeProperty
   AL((wid, prop, type, mode, format, nunits, data))
   DB rid_t wid
   DD atom_t prop
   DD atom_t type
   DD u8_t mode
   DD u8_t format
   DD u32_t nunits
   DD char *data
   DE
{
   register buffer_t *bp = zb_fbuf();
   register int nbytes;
   register uint_t len, xtra;
   xChangePropertyReq *p;

   nbytes = (format >> 3) * nunits;
   xtra = nbytes + PAD(nbytes);
   len = sz_xChangePropertyReq + xtra;
   zb_mon(len);

   if (buf_avail(bp) < len)
      buf_adjust(bp, len);

   p = (xChangePropertyReq *)buf_next(bp);

   p->reqType = X_ChangeProperty;
   p->length = (sz_xChangePropertyReq + xtra) / 4;
   p->mode = mode;
   p->window = wid;
   p->property = prop;
   p->type = type;
   p->format = format;
   p->nUnits = nunits;

   buf_grow(bp, sz_xChangePropertyReq);
   if (nbytes) {
      buf_put(bp, data, nbytes);
      if (PAD(nbytes))
         buf_put(bp, padsrc, PAD(nbytes));
   }

   zb_incr_seqno();
   ptc_ChangeProperty(bp, 0, zb_seqno(), p);
}

/*
** [026] proto_GrabPointer
*/
void
proto_GrabPointer
   AL((win, oe, mask, pmode, kmode, cwin, curs, time))
   DB rid_t win
   DD u8_t oe
   DD u16_t mask
   DD u8_t pmode
   DD u8_t kmode
   DD rid_t cwin
   DD rid_t curs
   DD u32_t time
   DE
{
   register buffer_t *bp = zb_fbuf();
   xGrabPointerReq *p;

   zb_mon(sz_xGrabPointerReq);

   if (buf_avail(bp) < sz_xGrabPointerReq)
      buf_adjust(bp, sz_xGrabPointerReq);

   p = (xGrabPointerReq *)buf_next(bp);

   p->reqType = X_GrabPointer;
   p->ownerEvents = oe;
   p->length = sz_xGrabPointerReq / 4;
   p->grabWindow = win;
   p->eventMask = mask;
   p->pointerMode = pmode;
   p->keyboardMode = kmode;
   p->confineTo = cwin;
   p->cursor = curs;
   p->time = time;

   buf_grow(bp, sz_xGrabPointerReq);

   zb_incr_seqno();
   ptc_GrabPointer(bp, 0, zb_seqno(), p);
}

/*
** [027] proto_UngrabPointer
*/
void
proto_UngrabPointer
   AL((time))
   DB u32_t time
   DE
{
   register buffer_t *bp = zb_fbuf();
   xResourceReq *p;

   zb_mon(sz_xResourceReq);

   if (buf_avail(bp) < sz_xResourceReq)
      buf_adjust(bp, sz_xResourceReq);

   p = (xResourceReq *)buf_next(bp);

   p->reqType = X_UngrabPointer;
   p->length = sz_xResourceReq / 4;
   p->id = time;

   buf_grow(bp, sz_xResourceReq);

   zb_incr_seqno();
   ptc_UngrabPointer(bp, 0, zb_seqno(), p);
}

/*
** [031] proto_GrabKeyboard
*/
void
proto_GrabKeyboard
   AL((win, oe, pmode, kmode, time))
   DB rid_t win
   DD u8_t oe
   DD u8_t pmode
   DD u8_t kmode
   DD u32_t time
   DE
{
   register buffer_t *bp = zb_fbuf();
   xGrabKeyboardReq *p;

   zb_mon(sz_xGrabKeyboardReq);

   if (buf_avail(bp) < sz_xGrabKeyboardReq)
      buf_adjust(bp, sz_xGrabKeyboardReq);

   p = (xGrabKeyboardReq *)buf_next(bp);

   p->reqType = X_GrabKeyboard;
   p->ownerEvents = oe;
   p->length = sz_xGrabKeyboardReq / 4;
   p->grabWindow = win;
   p->pointerMode = pmode;
   p->keyboardMode = kmode;
   p->time = time;

   buf_grow(bp, sz_xGrabKeyboardReq);

   zb_incr_seqno();
   ptc_GrabKeyboard(bp, 0, zb_seqno(), p);
}

/*
** [032] proto_UngrabKeyboard
*/
void
proto_UngrabKeyboard
   AL((time))
   DB u32_t time
   DE
{
   register buffer_t *bp = zb_fbuf();
   xResourceReq *p;

   zb_mon(sz_xResourceReq);

   if (buf_avail(bp) < sz_xResourceReq)
      buf_adjust(bp, sz_xResourceReq);

   p = (xResourceReq *)buf_next(bp);

   p->reqType = X_UngrabKeyboard;
   p->length = sz_xResourceReq / 4;
   p->id = time;

   buf_grow(bp, sz_xResourceReq);

   zb_incr_seqno();
   ptc_UngrabKeyboard(bp, 0, zb_seqno(), p);
}

/*
** [041] proto_WarpPointer
*/
void
proto_WarpPointer
   AL((swin, dwin, sx, sy, sw, sh, dx, dy))
   DB rid_t swin
   DD rid_t dwin
   DD s16_t sx
   DD s16_t sy
   DD u16_t sw
   DD u16_t sh
   DD s16_t dx
   DD s16_t dy
   DE
{
   register buffer_t *bp = zb_fbuf();
   xWarpPointerReq *p;

   zb_mon(sz_xWarpPointerReq);

   if (buf_avail(bp) < sz_xWarpPointerReq)
      buf_adjust(bp, sz_xWarpPointerReq);

   p = (xWarpPointerReq *)buf_next(bp);

   p->reqType = X_WarpPointer;
   p->length = sz_xWarpPointerReq / 4;
   p->srcWid = swin;
   p->dstWid = dwin;
   p->srcX = sx;
   p->srcY = sy;
   p->srcWidth = sw;
   p->srcHeight = sh;
   p->dstX = dx;
   p->dstY = dy;

   buf_grow(bp, sz_xWarpPointerReq);

   zb_incr_seqno();
   ptc_WarpPointer(bp, 0, zb_seqno(), p);
}

/*
** [043] proto_GetInputFocus
*/
void
proto_GetInputFocus
   VOID
{
   register buffer_t *bp = zb_fbuf();
   xReq *p;

   zb_mon(sz_xReq);

   if (buf_avail(bp) < sz_xReq)
      buf_adjust(bp, sz_xReq);

   p = (xReq *)buf_next(bp);

   p->reqType = X_GetInputFocus;
   p->length = sz_xReq / 4;

   buf_grow(bp, sz_xReq);

   zb_incr_seqno();
   ptc_GetInputFocus(bp, 0, zb_seqno(), p);
}

/*
** [045] proto_OpenFont
*/
void
proto_OpenFont
   AL((fid, name, len))
   DB rid_t fid
   DD char *name
   DD int len
   DE
{
   register buffer_t *bp = zb_fbuf();
   register uint_t totlen, xtra;
   xOpenFontReq *p;

   xtra = len + PAD(len);
   totlen = sz_xOpenFontReq + xtra;
   zb_mon(totlen);

   if (buf_avail(bp) < totlen)
      buf_adjust(bp, totlen);

   p = (xOpenFontReq *)buf_next(bp);

   p->reqType = X_OpenFont;
   p->length = (sz_xOpenFontReq + xtra) / 4;
   p->fid = fid;
   p->nbytes = len;

   buf_grow(bp, sz_xOpenFontReq);
   if (len) {
      buf_put(bp, name, len);
      if (PAD(len))
         buf_put(bp, padsrc, PAD(len));
   }
   zb_incr_seqno();
   ptc_OpenFont(bp, 0, zb_seqno(), p);
}

/*
** [046] proto_CloseFont
*/
void
proto_CloseFont
   AL((font))
   DB rid_t font
   DE
{
   register buffer_t *bp = zb_fbuf();
   xResourceReq *p;

   zb_mon(sz_xResourceReq);

   if (buf_avail(bp) < sz_xResourceReq)
      buf_adjust(bp, sz_xResourceReq);

   p = (xResourceReq *)buf_next(bp);

   p->reqType = X_CloseFont;
   p->length = sz_xResourceReq / 4;
   p->id = font;

   buf_grow(bp, sz_xResourceReq);

   zb_incr_seqno();
   ptc_CloseFont(bp, 0, zb_seqno(), p);
}

/*
** [048] proto_QueryTextExtents
*/
void
proto_QueryTextExtents
   AL((font, n, sp))
   DB rid_t font
   DD u8_t n
   DD u16_t *sp
   DE
{
   register buffer_t *bp = zb_fbuf();
   register int nbytes, xtra, pad;
   xQueryTextExtentsReq *p;

   xtra = n * 2;
   pad = PAD(xtra);
   nbytes = sz_xQueryTextExtentsReq + xtra + pad;
   zb_mon(nbytes);

   if (buf_avail(bp) < nbytes)
      buf_adjust(bp, nbytes);

   p = (xQueryTextExtentsReq *)buf_next(bp);

   p->reqType = X_QueryTextExtents;
   p->oddLength = pad == 2 ? xTrue : xFalse;
   p->length = nbytes / 4;
   p->fid = font;

   buf_grow(bp, sz_xQueryTextExtentsReq);
   if (n) {
      buf_put(bp, (char *)sp, xtra);
      if (pad)
         buf_put(bp, padsrc, pad);
   }
   zb_incr_seqno();
   ptc_QueryTextExtents(bp, 0, zb_seqno(), p);
}

/*
** [051] proto_SetFontPath
*/
void
proto_SetFontPath
   AL((npaths, paths))
   DB int npaths
   DD char **paths	/* the first byte of each is the length */
   DE
{
   register int i, len, totlen, pad;
   register buffer_t *bp = zb_fbuf();
   xSetFontPathReq *p;

   totlen = sz_xSetFontPathReq;
   for (i=0; i<npaths; i++)
      totlen += (int)*paths[i] + 1;
   pad = PAD(totlen);
   totlen += pad;

   zb_mon(totlen);

   if (buf_avail(bp) < totlen)
      buf_adjust(bp, totlen);

   p = (xSetFontPathReq *)buf_next(bp);

   p->reqType = X_SetFontPath;
   p->length = totlen / 4;
   p->nFonts = npaths;

   buf_grow(bp, sz_xSetFontPathReq);

   if (npaths) {
      for (i=0; i<npaths; i++) {
         len = (int)*paths[i];
         buf_put(bp, paths[i], len + 1);
      }
      if (pad)
         buf_put(bp, padsrc, pad);
   }
   zb_incr_seqno();
   ptc_SetFontPath(bp, 0, zb_seqno(), p);
}

/*
** [053] proto_CreatePixmap
*/
void
proto_CreatePixmap
   AL((pid, drawable, w, h, depth))
   DB rid_t pid
   DD rid_t drawable
   DD u16_t w
   DD u16_t h
   DD u16_t depth
   DE
{
   register buffer_t *bp = zb_fbuf();
   xCreatePixmapReq *p;

   zb_mon(sz_xCreatePixmapReq);

   if (buf_avail(bp) < sz_xCreatePixmapReq)
      buf_adjust(bp, sz_xCreatePixmapReq);

   p = (xCreatePixmapReq *)buf_next(bp);

   p->reqType = X_CreatePixmap;
   p->depth = depth;
   p->length = sz_xCreatePixmapReq / 4;
   p->pid = pid;
   p->drawable = drawable;
   p->width = w;
   p->height = h;

   buf_grow(bp, sz_xCreatePixmapReq);

   zb_incr_seqno();
   ptc_CreatePixmap(bp, 0, zb_seqno(), p);
}

/*
** [054] proto_FreePixmap
*/
void
proto_FreePixmap
   AL((pid))
   DB rid_t pid
   DE
{
   register buffer_t *bp = zb_fbuf();
   xResourceReq *p;

   zb_mon(sz_xResourceReq);

   if (buf_avail(bp) < sz_xResourceReq)
      buf_adjust(bp, sz_xResourceReq);

   p = (xResourceReq *)buf_next(bp);

   p->reqType = X_FreePixmap;
   p->length = sz_xResourceReq / 4;
   p->id = pid;

   buf_grow(bp, sz_xResourceReq);

   zb_incr_seqno();
   ptc_FreePixmap(bp, 0, zb_seqno(), p);
}

/*
** [055] proto_CreateGC
*/
void
proto_CreateGC
   AL((gc, drawable, mask, values, mappixels))
   DB rid_t gc
   DD rid_t drawable
   DD mask_t mask
   DD u32_t *values
   DD int mappixels
   DE
{
   register buffer_t *bp = zb_fbuf();
   register int n;
   register uint_t len, xtra;
   xCreateGCReq *p;

   NUMONES(mask, n);
   xtra = n * 4;
   len = sz_xCreateGCReq + xtra;
   zb_mon(len);

   if (buf_avail(bp) < len)
      buf_adjust(bp, len);

   p = (xCreateGCReq *)buf_next(bp);

   p->reqType = X_CreateGC;
   p->length = sz_xCreateGCReq / 4 + n;
   p->gc = gc;
   p->drawable = drawable;
   p->mask = mask;

   buf_grow(bp, sz_xCreateGCReq);
   if (n)
      buf_put(bp, (char *)values, xtra);

   zb_incr_seqno();
   ptc_CreateGC(bp, 0, zb_seqno(), p, mappixels);
}

/*
** [056] proto_ChangeGC
*/
void
proto_ChangeGC
   AL((gc, mask, values, mappixels))
   DB rid_t gc
   DD mask_t mask
   DD u32_t *values
   DD int mappixels
   DE
{
   register buffer_t *bp = zb_fbuf();
   register int n;
   register uint_t len, xtra;
   xChangeGCReq *p;

   NUMONES(mask, n);
   xtra = n * 4;
   len = sz_xChangeGCReq + xtra;
   zb_mon(len);

   if (buf_avail(bp) < len)
      buf_adjust(bp, len);

   p = (xChangeGCReq *)buf_next(bp);

   p->reqType = X_ChangeGC;
   p->length = sz_xChangeGCReq / 4 + n;
   p->gc = gc;
   p->mask = mask;

   buf_grow(bp, sz_xChangeGCReq);
   if (n)
      buf_put(bp, (char *)values, xtra);

   zb_incr_seqno();
   ptc_ChangeGC(bp, 0, zb_seqno(), p, mappixels);
}

/*
** [060] proto_FreeGC
*/
void
proto_FreeGC
   AL((gid))
   DB rid_t gid
   DE
{
   register buffer_t *bp = zb_fbuf();
   xResourceReq *p;

   zb_mon(sz_xResourceReq);

   if (buf_avail(bp) < sz_xResourceReq)
      buf_adjust(bp, sz_xResourceReq);

   p = (xResourceReq *)buf_next(bp);

   p->reqType = X_FreeGC;
   p->length = sz_xResourceReq / 4;
   p->id = gid;

   buf_grow(bp, sz_xResourceReq);

   zb_incr_seqno();
   ptc_FreeGC(bp, 0, zb_seqno(), p);
}

/*
** [061] proto_ClearArea
*/
void
proto_ClearArea
   AL((window, x, y, w, h, exposures))
   DB rid_t window
   DD s16_t x
   DD s16_t y
   DD u16_t w
   DD u16_t h
   DD u8_t exposures
   DE
{
   register buffer_t *bp = zb_fbuf();
   xClearAreaReq *p;

   zb_mon(sz_xClearAreaReq);

   if (buf_avail(bp) < sz_xClearAreaReq)
      buf_adjust(bp, sz_xClearAreaReq);

   p = (xClearAreaReq *)buf_next(bp);

   p->reqType = X_ClearArea;
   p->length = sz_xClearAreaReq / 4;
   p->exposures = exposures;
   p->window = window;
   p->x = x;
   p->y = y;
   p->width = w;
   p->height = h;

   buf_grow(bp, sz_xClearAreaReq);

   zb_incr_seqno();
   ptc_ClearArea(bp, 0, zb_seqno(), p);
}

/*
** [062] proto_CopyArea
*/
void
proto_CopyArea
   AL((srcId, dstId, gid, sx, sy, dx, dy, w, h))
   DB rid_t srcId
   DD rid_t dstId
   DD rid_t gid
   DD s16_t sx
   DD s16_t sy
   DD s16_t dx
   DD s16_t dy
   DD u16_t w
   DD u16_t h
   DE
{
   register buffer_t *bp = zb_fbuf();
   xCopyAreaReq *p;

   zb_mon(sz_xCopyAreaReq);

   if (buf_avail(bp) < sz_xCopyAreaReq)
      buf_adjust(bp, sz_xCopyAreaReq);

   p = (xCopyAreaReq *)buf_next(bp);

   p->reqType = X_CopyArea;
   p->length = sz_xCopyAreaReq / 4;
   p->srcDrawable = srcId;
   p->dstDrawable = dstId;
   p->gc = gid;
   p->srcX = sx;
   p->srcY = sy;
   p->dstX = dx;
   p->dstY = dy;
   p->width = w;
   p->height = h;

   buf_grow(bp, sz_xCopyAreaReq);

   zb_incr_seqno();
   ptc_CopyArea(bp, 0, zb_seqno(), p);
}

/*
** [063] proto_CopyPlane
*/
void
proto_CopyPlane
   AL((srcId, dstId, gid, sx, sy, dx, dy, w, h, plane))
   DB rid_t srcId
   DD rid_t dstId
   DD rid_t gid
   DD s16_t sx
   DD s16_t sy
   DD s16_t dx
   DD s16_t dy
   DD u16_t w
   DD u16_t h
   DD u32_t plane
   DE
{
   register buffer_t *bp = zb_fbuf();
   xCopyPlaneReq *p;

   zb_mon(sz_xCopyPlaneReq);

   if (buf_avail(bp) < sz_xCopyPlaneReq)
      buf_adjust(bp, sz_xCopyPlaneReq);

   p = (xCopyPlaneReq *)buf_next(bp);

   p->reqType = X_CopyPlane;
   p->length = sz_xCopyPlaneReq / 4;
   p->srcDrawable = srcId;
   p->dstDrawable = dstId;
   p->gc = gid;
   p->srcX = sx;
   p->srcY = sy;
   p->dstX = dx;
   p->dstY = dy;
   p->width = w;
   p->height = h;
   p->bitPlane = plane;

   buf_grow(bp, sz_xCopyPlaneReq);

   zb_incr_seqno();
   ptc_CopyPlane(bp, 0, zb_seqno(), p);
}

/*
** [064] proto_PolyPoint
*/
void
proto_PolyPoint
   AL((mode, drawable, gc, n, points))
   DB u8_t mode
   DD rid_t drawable
   DD rid_t gc
   DD int n
   DD u16_t *points
   DE
{
   register buffer_t *bp = zb_fbuf();
   register uint_t len, xtra;
   xPolyPointReq *p;

   xtra = n * 4;
   len = sz_xPolyPointReq + xtra;
   zb_mon(len);

   if (buf_avail(bp) < len)
      buf_adjust(bp, len);

   p = (xPolyPointReq *)buf_next(bp);

   p->reqType = X_PolyPoint;
   p->coordMode = mode;
   p->length = sz_xPolyPointReq / 4 + n;
   p->drawable = drawable;
   p->gc = gc;

   buf_grow(bp, sz_xPolyPointReq);
   if (n)
      buf_put(bp, (char *)points, xtra);

   zb_incr_seqno();
   ptc_PolyPoint(bp, 0, zb_seqno(), p);
}

/*
** [070] proto_PolyFillRectangle
*/
void
proto_PolyFillRectangle
   AL((drawable, gc, x, y, w, h))
   DB rid_t drawable
   DD rid_t gc
   DD s16_t x
   DD s16_t y
   DD u16_t w
   DD u16_t h
   DE
{
   register buffer_t *bp = zb_fbuf();
   register uint_t len, xtra;
   xPolySegmentReq *p;
   struct {
      s16_t	x, y;
      u16_t	w, h;
   }rect;

   xtra = sizeof(rect);
   len = sz_xPolySegmentReq + xtra;
   zb_mon(len);

   if (buf_avail(bp) < len)
      buf_adjust(bp, len);

   p = (xPolySegmentReq *)buf_next(bp);

   p->reqType = X_PolyFillRectangle;
   p->length = (sz_xPolySegmentReq + xtra) / 4;
   p->drawable = drawable;
   p->gc = gc;

   buf_grow(bp, sz_xPolySegmentReq);
   rect.x = x;
   rect.y = y;
   rect.w = w;
   rect.h = h;
   buf_put(bp, (char *)&rect, sizeof(rect));

   zb_incr_seqno();
   ptc_PolyFillRectangle(bp, 0, zb_seqno(), p);
}

/*
** [072] proto_PutImage
*/
void
proto_PutImage
   AL((format, drawable, gc, x, y, w, h, left_pad, depth))
   DB u8_t format
   DD rid_t drawable
   DD rid_t gc
   DD s16_t x
   DD s16_t y
   DD u16_t w
   DD u16_t h
   DD u8_t left_pad
   DD u8_t depth
   DE
{
   register buffer_t *bp = zb_fbuf();
   xPutImageReq *p;

   zb_mon(sz_xPutImageReq);

   if (buf_avail(bp) < sz_xPutImageReq)
      buf_adjust(bp, sz_xPutImageReq);

   p = (xPutImageReq *)buf_next(bp);

   p->reqType = X_PutImage;
   p->format = format;
   p->length = sz_xPutImageReq / 4;
   p->drawable = (Drawable)drawable;
   p->gc = (GContext)gc;
   p->width = w;
   p->height = h;
   p->dstX = x;
   p->dstY = y;
   p->leftPad = left_pad;
   p->depth = depth;

   buf_grow(bp, sz_xPutImageReq);

   zb_incr_seqno();
   ptc_PutImage(bp, 0, zb_seqno(), p);
}

/*
** [073] proto_GetImage
*/
void
proto_GetImage
   AL((format, drawable, x, y, w, h, planemask))
   DB u8_t format
   DD rid_t drawable
   DD s16_t x
   DD s16_t y
   DD u16_t w
   DD u16_t h
   DD mask_t planemask
   DE
{
   register buffer_t *bp = zb_fbuf();
   xGetImageReq *p;

   zb_mon(sz_xGetImageReq);

   if (buf_avail(bp) < sz_xGetImageReq)
      buf_adjust(bp, sz_xGetImageReq);

   p = (xGetImageReq *)buf_next(bp);

   p->reqType = X_GetImage;
   p->format = format;
   p->length = sz_xGetImageReq / 4;
   p->drawable = (Drawable)drawable;
   p->x = x;
   p->y = y;
   p->width = w;
   p->height = h;
   p->planeMask = planemask;

   buf_grow(bp, sz_xGetImageReq);

   zb_incr_seqno();
   ptc_GetImage(bp, 0, zb_seqno(), p);
}

/*
** [077] proto_ImageText16
*/
void
proto_ImageText16
   AL((drawable, gc, x, y, n, sp))
   DB rid_t drawable
   DD rid_t gc
   DD s16_t x
   DD s16_t y
   DD u8_t n
   DD u16_t *sp
   DE
{
   register buffer_t *bp = zb_fbuf();
   register int nbytes, xtra, pad;
   xImageText16Req *p;

   xtra = n * 2;
   pad = PAD(xtra);
   nbytes = sz_xImageText16Req + xtra + pad;
   zb_mon(nbytes);

   if (buf_avail(bp) < nbytes)
      buf_adjust(bp, nbytes);

   p = (xImageText16Req *)buf_next(bp);

   p->reqType = X_ImageText16;
   p->nChars = n;
   p->length = nbytes / 4;
   p->drawable = (Drawable)drawable;
   p->gc = (GContext)gc;
   p->x = x;
   p->y = y;

   buf_grow(bp, sz_xImageText16Req);
   if (n) {
      buf_put(bp, (char *)sp, xtra);
      if (pad)
         buf_put(bp, padsrc, pad);
   }
   zb_incr_seqno();
   ptc_ImageText16(bp, 0, zb_seqno(), p);
}

/*
** [078] proto_CreateColormap
*/
void
proto_CreateColormap
   AL((cmap, visual, window, alloc))
   DB rid_t cmap
   DD rid_t visual
   DD rid_t window
   DD u8_t alloc
   DE
{
   register buffer_t *bp = zb_fbuf();
   xCreateColormapReq *p;

   zb_mon(sz_xCreateColormapReq);

   if (buf_avail(bp) < sz_xCreateColormapReq)
      buf_adjust(bp, sz_xCreateColormapReq);

   p = (xCreateColormapReq *)buf_next(bp);

   p->reqType = X_CreateColormap;
   p->length = sz_xCreateColormapReq / 4;
   p->alloc = alloc;
   p->mid = (Colormap)cmap;
   p->window = (Window)window;
   p->visual = (VisualID)visual;

   buf_grow(bp, sz_xCreateColormapReq);

   zb_incr_seqno();
   ptc_CreateColormap(bp, 0, zb_seqno(), p);
}

/*
** [079] proto_FreeColormap
*/
void
proto_FreeColormap
   AL((cmap))
   DB rid_t cmap
   DE
{
   register buffer_t *bp = zb_fbuf();
   xResourceReq *p;

   zb_mon(sz_xResourceReq);

   if (buf_avail(bp) < sz_xResourceReq)
      buf_adjust(bp, sz_xResourceReq);

   p = (xResourceReq *)buf_next(bp);

   p->reqType = X_FreeColormap;
   p->length = sz_xResourceReq / 4;
   p->id = cmap;

   buf_grow(bp, sz_xResourceReq);

   zb_incr_seqno();
   ptc_FreeColormap(bp, 0, zb_seqno(), p);
}

/*
** [080] proto_CopyColormapAndFree
*/
void
proto_CopyColormapAndFree
   AL((newid, oldid))
   DB rid_t newid
   DD rid_t oldid
   DE
{
   register buffer_t *bp = zb_fbuf();
   xCopyColormapAndFreeReq *p;

   zb_mon(sz_xCopyColormapAndFreeReq);

   if (buf_avail(bp) < sz_xCopyColormapAndFreeReq)
      buf_adjust(bp, sz_xCopyColormapAndFreeReq);

   p = (xCopyColormapAndFreeReq *)buf_next(bp);

   p->reqType = X_CopyColormapAndFree;
   p->length = sz_xCopyColormapAndFreeReq / 4;
   p->mid = newid;
   p->srcCmap = oldid;

   buf_grow(bp, sz_xCopyColormapAndFreeReq);

   zb_incr_seqno();
   ptc_CopyColormapAndFree(bp, 0, zb_seqno(), p);
}

/*
** [081] proto_InstallColormap
*/
void
proto_InstallColormap
   AL((cmap))
   DB rid_t cmap
   DE
{
   register buffer_t *bp = zb_fbuf();
   xResourceReq *p;

   zb_mon(sz_xResourceReq);

   if (buf_avail(bp) < sz_xResourceReq)
      buf_adjust(bp, sz_xResourceReq);

   p = (xResourceReq *)buf_next(bp);

   p->reqType = X_InstallColormap;
   p->length = sz_xResourceReq / 4;
   p->id = cmap;

   buf_grow(bp, sz_xResourceReq);

   zb_incr_seqno();
   ptc_InstallColormap(bp, 0, zb_seqno(), p);
}

/*
** [082] proto_UninstallColormap
*/
void
proto_UninstallColormap
   AL((cmap))
   DB rid_t cmap
   DE
{
   register buffer_t *bp = zb_fbuf();
   xResourceReq *p;

   zb_mon(sz_xResourceReq);

   if (buf_avail(bp) < sz_xResourceReq)
      buf_adjust(bp, sz_xResourceReq);

   p = (xResourceReq *)buf_next(bp);

   p->reqType = X_UninstallColormap;
   p->length = sz_xResourceReq / 4;
   p->id = cmap;

   buf_grow(bp, sz_xResourceReq);

   zb_incr_seqno();
   ptc_UninstallColormap(bp, 0, zb_seqno(), p);
}

/*
** [084] proto_AllocColor
*/
void
proto_AllocColor
   AL((cmap, red, green, blue))
   DB rid_t cmap
   DD u16_t red
   DD u16_t green
   DD u16_t blue
   DE
{
   register buffer_t *bp = zb_fbuf();
   xAllocColorReq *p;

   zb_mon(sz_xAllocColorReq);

   if (buf_avail(bp) < sz_xAllocColorReq)
      buf_adjust(bp, sz_xAllocColorReq);

   p = (xAllocColorReq *)buf_next(bp);

   p->reqType = X_AllocColor;
   p->length = sz_xAllocColorReq / 4;
   p->cmap = (Colormap)cmap;
   p->red = red;
   p->green = green;
   p->blue = blue;

   buf_grow(bp, sz_xAllocColorReq);

   zb_incr_seqno();
   ptc_AllocColor(bp, 0, zb_seqno(), p);
}

/*
** [086] proto_AllocColorCells
*/
void
proto_AllocColorCells
   AL((cmap, ncol, nplanes, contig))
   DB rid_t cmap
   DD u16_t ncol
   DD u16_t nplanes
   DD u8_t contig
   DE
{
   register buffer_t *bp = zb_fbuf();
   xAllocColorCellsReq *p;

   zb_mon(sz_xAllocColorCellsReq);

   if (buf_avail(bp) < sz_xAllocColorCellsReq)
      buf_adjust(bp, sz_xAllocColorCellsReq);

   p = (xAllocColorCellsReq *)buf_next(bp);

   p->reqType = X_AllocColorCells;
   p->length = sz_xAllocColorCellsReq / 4;
   p->planes = nplanes;
   p->contiguous = contig;
   p->cmap = (Colormap)cmap;
   p->colors = ncol;

   buf_grow(bp, sz_xAllocColorCellsReq);

   zb_incr_seqno();
   ptc_AllocColorCells(bp, 0, zb_seqno(), p);
}

/*
** [087] proto_AllocColorPlanes
*/
void
proto_AllocColorPlanes
   AL((cmap, ncol, reds, greens, blues, contig))
   DB rid_t cmap
   DD u16_t ncol
   DD u16_t reds
   DD u16_t greens
   DD u16_t blues
   DD u8_t contig
   DE
{
   register buffer_t *bp = zb_fbuf();
   xAllocColorPlanesReq *p;

   zb_mon(sz_xAllocColorPlanesReq);

   if (buf_avail(bp) < sz_xAllocColorPlanesReq)
      buf_adjust(bp, sz_xAllocColorPlanesReq);

   p = (xAllocColorPlanesReq *)buf_next(bp);

   p->reqType = X_AllocColorPlanes;
   p->length = sz_xAllocColorPlanesReq / 4;
   p->cmap = (Colormap)cmap;
   p->colors = ncol;
   p->red = reds;
   p->green = greens;
   p->blue = blues;
   p->contiguous = contig;

   buf_grow(bp, sz_xAllocColorPlanesReq);

   zb_incr_seqno();
   ptc_AllocColorPlanes(bp, 0, zb_seqno(), p);
}

/*
** [088] proto_FreeColors
*/
void
proto_FreeColors
   AL((cmap, mask, n, pp, mappixels))
   DB rid_t cmap
   DD mask_t mask
   DD int n
   DD pixel_t *pp
   DD int mappixels
   DE
{
   register buffer_t *bp = zb_fbuf();
   register uint_t totlen, xtra;
   xFreeColorsReq *p;

   xtra = sizeof(pixel_t) * n;
   totlen = sz_xFreeColorsReq + xtra;

   zb_mon(totlen);

   if (buf_avail(bp) < totlen)
      buf_adjust(bp, totlen);

   p = (xFreeColorsReq *)buf_next(bp);

   p->reqType = X_FreeColors;
   p->length = totlen / 4;
   p->cmap = (Colormap)cmap;
   p->planeMask = mask;

   buf_grow(bp, sz_xFreeColorsReq);
   if (n)
      buf_put(bp, (char *)pp, xtra);

   zb_incr_seqno();
   ptc_FreeColors(bp, 0, zb_seqno(), p, mappixels);
}

/*
** [089] proto_StoreColors
*/
void
proto_StoreColors
   AL((cmap, ncol, c, mappixels))
   DB rid_t cmap
   DD int ncol
   DD xColorItem *c
   DD int mappixels
   DE
{
   register buffer_t *bp = zb_fbuf();
   register uint_t len, xtra;
   xStoreColorsReq *p;

   xtra = ncol * sz_xColorItem;
   len = sz_xStoreColorsReq + xtra;
   zb_mon(len);

   if (buf_avail(bp) < len)
      buf_adjust(bp, len);

   p = (xStoreColorsReq *)buf_next(bp);

   p->reqType = X_StoreColors;
   p->length = (sz_xStoreColorsReq + xtra) / 4;
   p->cmap = (Colormap)cmap;

   buf_grow(bp, sz_xStoreColorsReq);
   if (ncol)
      buf_put(bp, (char *)c, xtra);

   zb_incr_seqno();
   ptc_StoreColors(bp, 0, zb_seqno(), p, mappixels);
}

/*
** [091] proto_QueryColors
*/
void
proto_QueryColors
   AL((cmap, lstp))
   DB rid_t cmap
   DD list_t *lstp
   DE
{
   register buffer_t *bp = zb_fbuf();
   register uint_t n, len, xtra;
   register xQueryColorsReq *p;

   n = list_size(lstp);
   xtra = n * 4;
   len = sz_xQueryColorsReq + xtra;
   zb_mon(len);

   if (buf_avail(bp) < len)
      buf_adjust(bp, len);

   p = (xQueryColorsReq *)buf_next(bp);

   p->reqType = X_QueryColors;
   p->length = (sz_xQueryColorsReq + xtra) / 4;
   p->cmap = (Colormap)cmap;

   buf_grow(bp, sz_xQueryColorsReq);
   if (n)
      list_move(lstp, bp);

   zb_incr_seqno();
   ptc_QueryColors(bp, 0, zb_seqno(), p);
}

/*
** [093] proto_CreateCursor
*/
void
proto_CreateCursor
   AL((cid, pid, mid, fr, fg, fb, br, bg, bb, x, y))
   DB rid_t cid
   DD rid_t pid
   DD rid_t mid
   DD u16_t fr
   DD u16_t fg
   DD u16_t fb
   DD u16_t br
   DD u16_t bg
   DD u16_t bb
   DD u16_t x
   DD u16_t y
   DE
{
   register buffer_t *bp = zb_fbuf();
   xCreateCursorReq *p;

   zb_mon(sz_xCreateCursorReq);

   if (buf_avail(bp) < sz_xCreateCursorReq)
      buf_adjust(bp, sz_xCreateCursorReq);

   p = (xCreateCursorReq *)buf_next(bp);

   p->reqType = X_CreateCursor;
   p->length = sz_xCreateCursorReq / 4;
   p->cid = (Cursor)cid;
   p->source = (Pixmap)pid;
   p->mask = (Pixmap)mid;
   p->foreRed = fr;
   p->foreGreen = fg;
   p->foreBlue = fb;
   p->backRed = br;
   p->backGreen = bg;
   p->backBlue = bb;
   p->x = x;
   p->y = y;

   buf_grow(bp, sz_xCreateCursorReq);

   zb_incr_seqno();
   ptc_CreateCursor(bp, 0, zb_seqno(), p);
}

/*
** [094] proto_CreateGlyphCursor
*/
void
proto_CreateGlyphCursor
   AL((cid, fid, mid, csrc, cmask, fr, fg, fb, br, bg, bb))
   DB rid_t cid
   DD rid_t fid
   DD rid_t mid
   DD u16_t csrc
   DD u16_t cmask
   DD u16_t fr
   DD u16_t fg
   DD u16_t fb
   DD u16_t br
   DD u16_t bg
   DD u16_t bb
   DE
{
   register buffer_t *bp = zb_fbuf();
   xCreateGlyphCursorReq *p;

   zb_mon(sz_xCreateGlyphCursorReq);

   if (buf_avail(bp) < sz_xCreateGlyphCursorReq)
      buf_adjust(bp, sz_xCreateGlyphCursorReq);

   p = (xCreateGlyphCursorReq *)buf_next(bp);

   p->reqType = X_CreateGlyphCursor;
   p->length = sz_xCreateGlyphCursorReq / 4;
   p->cid = (Cursor)cid;
   p->source = (Font)fid;
   p->mask = (Font)mid;
   p->sourceChar = csrc;
   p->maskChar = cmask;
   p->foreRed = fr;
   p->foreGreen = fg;
   p->foreBlue = fb;
   p->backRed = br;
   p->backGreen = bg;
   p->backBlue = bb;

   buf_grow(bp, sz_xCreateGlyphCursorReq);

   zb_incr_seqno();
   ptc_CreateGlyphCursor(bp, 0, zb_seqno(), p);
}

/*
** [095] proto_FreeCursor
*/
void
proto_FreeCursor
   AL((cursor))
   DB rid_t cursor
   DE
{
   register buffer_t *bp = zb_fbuf();
   xResourceReq *p;

   zb_mon(sz_xResourceReq);

   if (buf_avail(bp) < sz_xResourceReq)
      buf_adjust(bp, sz_xResourceReq);

   p = (xResourceReq *)buf_next(bp);

   p->reqType = X_FreeCursor;
   p->length = sz_xResourceReq / 4;
   p->id = cursor;

   buf_grow(bp, sz_xResourceReq);

   zb_incr_seqno();
   ptc_FreeCursor(bp, 0, zb_seqno(), p);
}

/*
** [099] proto_ListExtensions
*/
void
proto_ListExtensions
   VOID
{
   register buffer_t *bp = zb_fbuf();
   xReq *p;

   zb_mon(sz_xReq);

   if (buf_avail(bp) < sz_xReq)
      buf_adjust(bp, sz_xReq);

   p = (xReq *)buf_next(bp);

   p->reqType = X_ListExtensions;
   p->length = sz_xReq / 4;

   buf_grow(bp, sz_xReq);

   zb_incr_seqno();
   ptc_ListExtensions(bp, 0, zb_seqno(), p);
}

/*
** [101] proto_GetKeyboardMapping
*/
void
proto_GetKeyboardMapping
   AL((first, count))
   DB u8_t first
   DD u8_t count
   DE
{
   register buffer_t *bp = zb_fbuf();
   xGetKeyboardMappingReq *p;

   zb_mon(sz_xGetKeyboardMappingReq);

   if (buf_avail(bp) < sz_xGetKeyboardMappingReq)
      buf_adjust(bp, sz_xGetKeyboardMappingReq);

   p = (xGetKeyboardMappingReq *)buf_next(bp);

   p->reqType = X_GetKeyboardMapping;
   p->length = sz_xGetKeyboardMappingReq / 4;
   p->firstKeyCode = first;
   p->count = count;

   buf_grow(bp, sz_xGetKeyboardMappingReq);

   zb_incr_seqno();
   ptc_GetKeyboardMapping(bp, 0, zb_seqno(), p);
}

/*
** [117] proto_GetPointerMapping
*/
void
proto_GetPointerMapping
   VOID
{
   register buffer_t *bp = zb_fbuf();
   xReq *p;

   zb_mon(sz_xReq);

   if (buf_avail(bp) < sz_xReq)
      buf_adjust(bp, sz_xReq);

   p = (xReq *)buf_next(bp);

   p->reqType = X_GetPointerMapping;
   p->length = sz_xReq / 4;

   buf_grow(bp, sz_xReq);

   zb_incr_seqno();
   ptc_GetPointerMapping(bp, 0, zb_seqno(), p);
}

/*
** [119] proto_GetModifierMapping
*/
void
proto_GetModifierMapping
   VOID
{
   register buffer_t *bp = zb_fbuf();
   xReq *p;

   zb_mon(sz_xReq);

   if (buf_avail(bp) < sz_xReq)
      buf_adjust(bp, sz_xReq);

   p = (xReq *)buf_next(bp);

   p->reqType = X_GetModifierMapping;
   p->length = sz_xReq / 4;

   buf_grow(bp, sz_xReq);

   zb_incr_seqno();
   ptc_GetModifierMapping(bp, 0, zb_seqno(), p);
}

/************************************************************************
*	REPLIES								*
************************************************************************/

/*
** [003] proto_GetWindowAttributesReply
*/
void
proto_GetWindowAttributesReply
   AL((cp, vid, class, installed, state, ymask, atts))
   DB client_t *cp
   DD rid_t vid
   DD u16_t class
   DD u8_t installed
   DD u8_t state
   DD mask_t ymask
   DD u32_t *atts
   DE
{
   register buffer_t *bp = zb_rbuf();
   xGetWindowAttributesReply *p;

   if (buf_avail(bp) < sz_xGetWindowAttributesReply)
      buf_adjust(bp, sz_xGetWindowAttributesReply);

   zb_client_dest(cp);

   p = (xGetWindowAttributesReply *)buf_next(bp);

   p->type = X_Reply;
   p->sequenceNumber = cp->seqno;
   p->length = 3;

   p->visualID = vid;
   p->class = class;
   p->mapInstalled = installed;
   p->mapState = state;
   p->yourEventMask = ymask;

   p->backingStore = atts[IWBackingStore];
   p->bitGravity = atts[IWBitGravity];
   p->winGravity = atts[IWWinGravity];
   p->backingBitPlanes = atts[IWBackingPlanes];
   p->backingPixel = atts[IWBackingPixel];
   p->saveUnder = atts[IWSaveUnder];
   p->override = atts[IWOverrideRedirect];
   p->colormap = atts[IWColormap];
   p->allEventMasks = atts[IWEventMask];
   p->doNotPropagateMask = atts[IWDontPropagate];

   buf_grow(bp, sz_xGetWindowAttributesReply);

   rs_mark(buf_chunk(bp), X_GetWindowAttributes, 0);
   zb_client_queue();
}

/*
** [014] proto_GetGeometryReply
*/
void
proto_GetGeometryReply
   AL((cp, depth, root, x, y, width,height, borderwidth))
   DB client_t *cp
   DD u8_t depth
   DD u32_t root
   DD s16_t x
   DD s16_t y
   DD u16_t width
   DD u16_t height
   DD u16_t borderwidth
   DE
{
   register buffer_t *bp = zb_rbuf();
   xGetGeometryReply *p;

   if (buf_avail(bp) < sz_xGetGeometryReply)
      buf_adjust(bp, sz_xGetGeometryReply);

   zb_client_dest(cp);

   p = (xGetGeometryReply *)buf_next(bp);

   p->type = X_Reply;
   p->sequenceNumber = cp->seqno;
   p->length = 0;

   p->depth = depth;
   p->root = root;
   p->x = x;
   p->y = y;
   p->width = width;
   p->height = height;
   p->borderWidth = borderwidth;

   buf_grow(bp, sz_xGetGeometryReply);

   rs_mark(buf_chunk(bp), X_GetGeometry, 0);
   zb_client_queue();
}

/*
** [015] proto_QueryTreeReply
*/
void
proto_QueryTreeReply
   AL((cp, root, parent, lstp))
   DB client_t *cp
   DD u32_t root
   DD u32_t parent
   DD list_t *lstp
   DE
{
   register buffer_t *bp = zb_rbuf();
   register uint_t n, xtra;
   xQueryTreeReply *p;

   n = list_size(lstp);
   xtra = n * 4;
   if (buf_avail(bp) < sz_xQueryTreeReply + xtra)
      buf_adjust(bp, sz_xQueryTreeReply + xtra);

   zb_client_dest(cp);

   p = (xQueryTreeReply *)buf_next(bp);

   p->type = X_Reply;
   p->sequenceNumber = cp->seqno;
   p->length = p->nChildren = n;
   p->root = root;
   p->parent = parent;

   buf_grow(bp, sz_xQueryTreeReply);
   if (n)
      list_move(lstp, bp);

   rs_mark(buf_chunk(bp), X_QueryTree, 0);
   zb_client_queue();
}

/*
** [016] proto_InternAtomReply
*/
void
proto_InternAtomReply
   AL((cp, atom))
   DB client_t *cp
   DD atom_t atom
   DE
{
   register buffer_t *bp = zb_rbuf();
   xInternAtomReply *p;

   if (buf_avail(bp) < sz_xInternAtomReply)
      buf_adjust(bp, sz_xInternAtomReply);

   zb_client_dest(cp);

   p = (xInternAtomReply *)buf_next(bp);

   p->type = X_Reply;
   p->sequenceNumber = cp->seqno;
   p->length = 0;
   p->atom = atom;

   buf_grow(bp, sz_xInternAtomReply);

   rs_mark(buf_chunk(bp), X_InternAtom, 0);
   zb_client_queue();
}

/*
** [017] proto_GetAtomNameReply
*/
void
proto_GetAtomNameReply
   AL((cp, name, nbytes))
   DB client_t *cp
   DD char *name
   DD u32_t nbytes
   DE
{
   register buffer_t *bp = zb_rbuf();
   register uint_t xtra;
   xGetAtomNameReply *p;

   xtra = nbytes + PAD(nbytes);
   if (buf_avail(bp) < sz_xGetAtomNameReply + xtra)
      buf_adjust(bp, sz_xGetAtomNameReply + xtra);

   zb_client_dest(cp);

   p = (xGetAtomNameReply *)buf_next(bp);

   p->type = X_Reply;
   p->sequenceNumber = cp->seqno;
   p->length = xtra / 4;
   p->nameLength = (u16_t)nbytes;

   buf_grow(bp, sz_xGetAtomNameReply);

   if (nbytes) {
      buf_put(bp, name, nbytes);
      if (PAD(nbytes))
         buf_put(bp, padsrc, PAD(nbytes));
   }

   rs_mark(buf_chunk(bp), X_GetAtomName, 0);
   zb_client_queue();
}

/*
** [020] proto_GetPropertyReply
*/
void
proto_GetPropertyReply
   AL((cp, type, format, after, nbytes, data))
   DB client_t *cp
   DD atom_t type
   DD u8_t format
   DD u32_t after
   DD u32_t nbytes
   DD char *data
   DE
{
   register buffer_t *bp = zb_rbuf();
   register uint_t xtra;
   register xGetPropertyReply *p;

   xtra = nbytes + PAD(nbytes);
   if (buf_avail(bp) < sz_xGetPropertyReply + xtra)
      buf_adjust(bp, sz_xGetPropertyReply + xtra);

   zb_client_dest(cp);

   p = (xGetPropertyReply *)buf_next(bp);

   p->type = X_Reply;
   p->format = format;
   p->sequenceNumber = cp->seqno;
   p->length = xtra / 4;
   p->propertyType = type;
   p->bytesAfter = after;
   p->nItems = format ? nbytes / (format>>3) : 0;

   buf_grow(bp, sz_xGetPropertyReply);
   if (nbytes) {
      buf_put(bp, data, nbytes);
      if (PAD(nbytes))
         buf_put(bp, padsrc, PAD(nbytes));
   }

   rs_mark(buf_chunk(bp), X_GetProperty, 0);
   zb_client_queue();
}

/*
** [021] proto_ListPropertiesReply
*/
void
proto_ListPropertiesReply
   AL((cp, lstp))
   DB client_t *cp
   DD list_t *lstp
   DE
{
   register buffer_t *bp = zb_rbuf();
   register uint_t n, xtra;
   register xListPropertiesReply *p;

   n = list_size(lstp);
   xtra = n * 4;
   if (buf_avail(bp) < sz_xListPropertiesReply + xtra)
      buf_adjust(bp, sz_xListPropertiesReply + xtra);

   zb_client_dest(cp);

   p = (xListPropertiesReply *)buf_next(bp);

   p->type = X_Reply;
   p->sequenceNumber = cp->seqno;
   p->nProperties = p->length = n;

   buf_grow(bp, sz_xListPropertiesReply);
   if (n)
      list_move(lstp, bp);

   rs_mark(buf_chunk(bp), X_ListProperties, 0);
   zb_client_queue();
}

/*
** [023] proto_GetSelectionOwnerReply
*/
void
proto_GetSelectionOwnerReply
   AL((cp, window))
   DB client_t *cp
   DD rid_t window
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xGetSelectionOwnerReply *p;

   if (buf_avail(bp) < sz_xGetSelectionOwnerReply)
      buf_adjust(bp, sz_xGetSelectionOwnerReply);

   zb_client_dest(cp);

   p = (xGetSelectionOwnerReply *)buf_next(bp);

   p->type = X_Reply;
   p->sequenceNumber = cp->seqno;
   p->length = 0;
   p->owner = window;

   buf_grow(bp, sz_xGetSelectionOwnerReply);

   rs_mark(buf_chunk(bp), X_GetSelectionOwner, 0);
   zb_client_queue();
}

/*
** [026] proto_GrabPointerReply
*/
void
proto_GrabPointerReply
   AL((cp, status))
   DB client_t *cp
   DD u8_t status
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xGrabPointerReply *p;

   if (buf_avail(bp) < sz_xGrabPointerReply)
      buf_adjust(bp, sz_xGrabPointerReply);

   zb_client_dest(cp);

   p = (xGrabPointerReply *)buf_next(bp);

   p->type = X_Reply;
   p->sequenceNumber = cp->seqno;
   p->length = 0;
   p->status = status;

   buf_grow(bp, sz_xGrabPointerReply);

   rs_mark(buf_chunk(bp), X_GrabPointer, 0);
   zb_client_queue();
}

/*
** [031] proto_GrabKeyboardReply
*/
void
proto_GrabKeyboardReply
   AL((cp, status))
   DB client_t *cp
   DD u8_t status
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xGrabKeyboardReply *p;

   if (buf_avail(bp) < sz_xGrabKeyboardReply)
      buf_adjust(bp, sz_xGrabKeyboardReply);

   zb_client_dest(cp);

   p = (xGrabKeyboardReply *)buf_next(bp);

   p->type = X_Reply;
   p->sequenceNumber = cp->seqno;
   p->length = 0;
   p->status = status;

   buf_grow(bp, sz_xGrabKeyboardReply);

   rs_mark(buf_chunk(bp), X_GrabKeyboard, 0);
   zb_client_queue();
}

/*
** [038] proto_QueryPointerReply
*/
void
proto_QueryPointerReply
   AL((cp, root,child, rootx,rooty, winx,winy, mask))
   DB client_t *cp
   DD rid_t root
   DD rid_t child
   DD s16_t rootx
   DD s16_t rooty
   DD s16_t winx
   DD s16_t winy
   DD u16_t mask
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xQueryPointerReply *p;

   if (buf_avail(bp) < sz_xQueryPointerReply)
      buf_adjust(bp, sz_xQueryPointerReply);

   zb_client_dest(cp);

   p = (xQueryPointerReply *)buf_next(bp);

   p->type = X_Reply;
   p->sequenceNumber = cp->seqno;
   p->length = 0;
   p->sameScreen = xTrue;
   p->root = root;
   p->child = child;
   p->rootX = rootx;
   p->rootY = rooty;
   p->winX = winx;
   p->winY = winy;
   p->mask = mask;

   buf_grow(bp, sz_xQueryPointerReply);

   rs_mark(buf_chunk(bp), X_QueryPointer, 0);
   zb_client_queue();
}

/*
** [039] proto_GetMotionEvents
*/
void
proto_GetMotionEventsReply
   AL((cp, nevents, events))
   DB client_t *cp
   DD u8_t nevents
   DD xTimecoord *events
   DE
{
   register buffer_t *bp = zb_rbuf();
   register uint_t xtra;
   register xGetMotionEventsReply *p;

   xtra = nevents * sz_xTimecoord;
   if (buf_avail(bp) < sz_xGetMotionEventsReply + xtra)
      buf_adjust(bp, sz_xGetMotionEventsReply + xtra);

   zb_client_dest(cp);

   p = (xGetMotionEventsReply *)buf_next(bp);

   p->type = X_Reply;
   p->sequenceNumber = cp->seqno;
   p->length = xtra / 4;
   p->nEvents = nevents;

   buf_grow(bp, sz_xGetMotionEventsReply);
   if (nevents)
      buf_put(bp, (char *)events, xtra);

   rs_mark(buf_chunk(bp), X_GetMotionEvents, 0);
   zb_client_queue();
}

/*
** [040] proto_TranslateCoordsReply
*/
void
proto_TranslateCoordsReply
   AL((cp, child, x, y))
   DB client_t *cp
   DD rid_t child
   DD s16_t x
   DD s16_t y
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xTranslateCoordsReply *p;

   if (buf_avail(bp) < sz_xTranslateCoordsReply)
      buf_adjust(bp, sz_xTranslateCoordsReply);

   zb_client_dest(cp);

   p = (xTranslateCoordsReply *)buf_next(bp);

   p->type = X_Reply;
   p->sequenceNumber = cp->seqno;
   p->length = 0;
   p->sameScreen = xTrue;
   p->child = child;
   p->dstX = x;
   p->dstY = y;

   buf_grow(bp, sz_xTranslateCoordsReply);

   rs_mark(buf_chunk(bp), X_TranslateCoords, 0);
   zb_client_queue();
}

/*
** [043] proto_GetInputFocusReply
*/
void
proto_GetInputFocusReply
   AL((cp, reverto, focus))
   DB client_t *cp
   DD u8_t reverto
   DD rid_t focus
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xGetInputFocusReply *p;

   if (buf_avail(bp) < sz_xGetInputFocusReply)
      buf_adjust(bp, sz_xGetInputFocusReply);

   zb_client_dest(cp);

   p = (xGetInputFocusReply *)buf_next(bp);

   p->type = X_Reply;
   p->sequenceNumber = cp->seqno;
   p->length = 0;
   p->revertTo = reverto;
   p->focus = focus;

   buf_grow(bp, sz_xGetInputFocusReply);

   rs_mark(buf_chunk(bp), X_GetInputFocus, 0);
   zb_client_queue();
}

/*
** [044] proto_QueryKeymapReply
*/
void
proto_QueryKeymapReply
   AL((cp, keymap))
   DB client_t *cp
   DD u8_t *keymap
   DE
{
   register buffer_t *bp = zb_rbuf();
   register int i;
   register xQueryKeymapReply *p;

   if (buf_avail(bp) < sz_xQueryKeymapReply)
      buf_adjust(bp, sz_xQueryKeymapReply);

   zb_client_dest(cp);

   p = (xQueryKeymapReply *)buf_next(bp);

   p->type = X_Reply;
   p->sequenceNumber = cp->seqno;
   p->length = 2;
   for (i=0; i<32; i++)
      p->map[i] = keymap[i];

   buf_grow(bp, sz_xQueryKeymapReply);

   rs_mark(buf_chunk(bp), X_QueryKeymap, 0);
   zb_client_queue();
}

/*
** [083] proto_ListInstalledColormapsReply
*/
void
proto_ListInstalledColormapsReply
   AL((cp, lstp))
   DB client_t *cp
   DD list_t *lstp
   DE
{
   register buffer_t *bp = zb_rbuf();
   register uint_t n, xtra;
   register xListInstalledColormapsReply *p;

   n = list_size(lstp);
   xtra = n * 4;
   if (buf_avail(bp) < sz_xListInstalledColormapsReply + xtra)
      buf_adjust(bp, sz_xListInstalledColormapsReply + xtra);

   zb_client_dest(cp);

   p = (xListInstalledColormapsReply *)buf_next(bp);

   p->type = X_Reply;
   p->sequenceNumber = cp->seqno;
   p->length = n;
   p->nColormaps = (u16_t)p->length;

   buf_grow(bp, sz_xListInstalledColormapsReply);
   if (n)
      list_move(lstp, bp);

   rs_mark(buf_chunk(bp), X_ListInstalledColormaps, 0);
   zb_client_queue();
}

/*
** [084] proto_AllocColorReply
*/
void
proto_AllocColorReply
   AL((cp, pixel, r, g, b))
   DB client_t *cp
   DD pixel_t pixel
   DD u16_t r
   DD u16_t g
   DD u16_t b
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xAllocColorReply *p;

   if (buf_avail(bp) < sz_xAllocColorReply)
      buf_adjust(bp, sz_xAllocColorReply);

   zb_client_dest(cp);

   p = (xAllocColorReply *)buf_next(bp);

   p->type = X_Reply;
   p->sequenceNumber = cp->seqno;
   p->length = 0;
   p->red = r;
   p->green = g;
   p->blue = b;
   p->pixel = pixel;

   buf_grow(bp, sz_xAllocColorReply);

   rs_mark(buf_chunk(bp), X_AllocColor, 0);
   zb_client_queue();
}

/*
** [085] proto_AllocNamedColorReply
*/
void
proto_AllocNamedColorReply
   AL((cp, pixel, exr, exg, exb, r, g, b))
   DB client_t *cp
   DD pixel_t pixel
   DD u16_t exr
   DD u16_t exg
   DD u16_t exb
   DD u16_t r
   DD u16_t g
   DD u16_t b
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xAllocNamedColorReply *p;

   if (buf_avail(bp) < sz_xAllocNamedColorReply)
      buf_adjust(bp, sz_xAllocNamedColorReply);

   zb_client_dest(cp);

   p = (xAllocNamedColorReply *)buf_next(bp);

   p->type = X_Reply;
   p->sequenceNumber = cp->seqno;
   p->length = 0;
   p->pixel = pixel;
   p->exactRed = exr;
   p->exactGreen = exg;
   p->exactBlue = exb;
   p->screenRed = r;
   p->screenGreen = g;
   p->screenBlue = b;

   buf_grow(bp, sz_xAllocNamedColorReply);

   rs_mark(buf_chunk(bp), X_AllocNamedColor, 0);
   zb_client_queue();
}

/*
** [086] proto_AllocColorCellsReply
*/
void
proto_AllocColorCellsReply
   AL((cp, ncols, nmsks, pixels, rm, gm, bm))
   DB client_t *cp
   DD u16_t ncols
   DD u16_t nmsks
   DD pixel_t *pixels
   DD mask_t rm
   DD mask_t gm
   DD mask_t bm
   DE
{
   register buffer_t *bp = zb_rbuf();
   register int i;
   register uint_t xtra;
   register mask_t tm, *mp;
   register xAllocColorCellsReply *p;

   xtra = (nmsks + ncols) * 4;
   if (buf_avail(bp) < sz_xAllocColorCellsReply + xtra)
      buf_adjust(bp, sz_xAllocColorCellsReply + xtra);

   zb_client_dest(cp);

   p = (xAllocColorCellsReply *)buf_next(bp);

   p->type = X_Reply;
   p->sequenceNumber = cp->seqno;
   p->length = ncols + nmsks;
   p->nPixels = ncols;
   p->nMasks = nmsks;

   buf_grow(bp, sz_xAllocColorCellsReply);
   if (ncols)
      buf_put(bp, (char *)pixels, ncols*4);

   mp = (mask_t *)buf_next(bp);
   for (i=0,tm=1; tm<=rm; tm<<=1)
      if (tm & rm)
         mp[i++] = tm;
   for (i=0,tm=1; tm<=gm; tm<<=1)
      if (tm & gm)
         mp[i++] |= tm;
   for (i=0,tm=1; tm<=bm; tm<<=1)
      if (tm & bm)
         mp[i++] |= tm;

   buf_grow(bp, nmsks*4);

   rs_mark(buf_chunk(bp), X_AllocColorCells, 0);
   zb_client_queue();
}

/*
** [087] proto_AllocColorPlanesReply
*/
void
proto_AllocColorPlanesReply
   AL((cp, ncols, rm, gm, bm, pixels))
   DB client_t *cp
   DD u16_t ncols
   DD mask_t rm
   DD mask_t gm
   DD mask_t bm
   DD pixel_t *pixels
   DE
{
   register buffer_t *bp = zb_rbuf();
   register uint_t xtra;
   register xAllocColorPlanesReply *p;

   xtra = ncols * 4;
   if (buf_avail(bp) < sz_xAllocColorPlanesReply + xtra)
      buf_adjust(bp, sz_xAllocColorPlanesReply + xtra);

   zb_client_dest(cp);

   p = (xAllocColorPlanesReply *)buf_next(bp);

   p->type = X_Reply;
   p->sequenceNumber = cp->seqno;
   p->length = ncols;
   p->nPixels = ncols;
   p->redMask = rm;
   p->greenMask = gm;
   p->blueMask = bm;

   buf_grow(bp, sz_xAllocColorPlanesReply);
   if (ncols)
      buf_put(bp, (char *)pixels, xtra);

   rs_mark(buf_chunk(bp), X_AllocColorPlanes, 0);
   zb_client_queue();
}

/*
** [091] proto_QueryColorsReply
*/
void
proto_QueryColorsReply
   AL((cp, lstp))
   DB client_t *cp
   DD list_t *lstp
   DE
{
   register buffer_t *bp = zb_rbuf();
   register uint_t n, xtra;
   register xQueryColorsReply *p;

   n = list_size(lstp);
   xtra = n * 4;
   if (buf_avail(bp) < sz_xQueryColorsReply + xtra)
      buf_adjust(bp, sz_xQueryColorsReply + xtra);

   zb_client_dest(cp);

   p = (xQueryColorsReply *)buf_next(bp);

   p->type = X_Reply;
   p->sequenceNumber = cp->seqno;
   p->length = n;
   p->nColors = p->length / 2;

   buf_grow(bp, sz_xQueryColorsReply);
   if (n)
      list_move(lstp, bp);

   rs_mark(buf_chunk(bp), X_QueryColors, 0);
   zb_client_queue();
}

/*
** [092] proto_LookupColorReply
*/
void
proto_LookupColorReply
   AL((cp, exr, exg, exb, r, g, b))
   DB client_t *cp
   DD u16_t exr
   DD u16_t exg
   DD u16_t exb
   DD u16_t r
   DD u16_t g
   DD u16_t b
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xLookupColorReply *p;

   if (buf_avail(bp) < sz_xLookupColorReply)
      buf_adjust(bp, sz_xLookupColorReply);

   zb_client_dest(cp);

   p = (xLookupColorReply *)buf_next(bp);

   p->type = X_Reply;
   p->sequenceNumber = cp->seqno;
   p->length = 0;
   p->exactRed = exr;
   p->exactGreen = exg;
   p->exactBlue = exb;
   p->screenRed = r;
   p->screenGreen = g;
   p->screenBlue = b;

   buf_grow(bp, sz_xLookupColorReply);

   rs_mark(buf_chunk(bp), X_LookupColor, 0);
   zb_client_queue();
}

/*
** [098] proto_QueryExtensionReply
*/
void
proto_QueryExtensionReply
   AL((cp, present, majorop, firstevent, firsterror))
   DB client_t *cp
   DD u8_t present
   DD u8_t majorop
   DD u8_t firstevent
   DD u8_t firsterror
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xQueryExtensionReply *p;

   if (buf_avail(bp) < sz_xQueryExtensionReply)
      buf_adjust(bp, sz_xQueryExtensionReply);

   zb_client_dest(cp);

   p = (xQueryExtensionReply *)buf_next(bp);

   p->type = X_Reply;
   p->sequenceNumber = cp->seqno;
   p->length = 0;
   p->present = present;
   p->major_opcode = majorop;
   p->first_event = firstevent;
   p->first_error = firsterror;

   buf_grow(bp, sz_xQueryExtensionReply);

   rs_mark(buf_chunk(bp), X_QueryExtension, 0);
   zb_client_queue();
}

/*
** [099] proto_ListExtensionsReply
*/
void
proto_ListExtensionsReply
   AL((cp, nexts, exts))
   DB client_t *cp
   DD int nexts
   DD char **exts
   DE
{
   register buffer_t *bp = zb_rbuf();
   register int i, len, totlen = 0;
   register uint_t xtra;
   register xListExtensionsReply *p;

   xtra = 0;
   for (i=0; i<nexts; i++)
      xtra += strlen(exts[i]) + 1;
   xtra += PAD(xtra);
   if (buf_avail(bp) < sz_xListExtensionsReply + xtra)
      buf_adjust(bp, sz_xListExtensionsReply + xtra);

   zb_client_dest(cp);

   p = (xListExtensionsReply *)buf_next(bp);

   p->type = X_Reply;
   p->nExtensions = nexts;
   p->sequenceNumber = cp->seqno;
   p->length = xtra / 4;

   buf_grow(bp, sz_xListExtensionsReply);

   for (i=0; i<nexts; i++) {
      len = strlen(exts[i]);
      *(buf_next(bp)) = (char)len;
      buf_grow(bp, 1);
      if (len)
         buf_put(bp, exts[i], len);
      totlen += len + 1;
   }
   if (totlen)
      if (PAD(totlen))
         buf_put(bp, padsrc, PAD(totlen));

   rs_mark(buf_chunk(bp), X_ListExtensions, 0);
   zb_client_queue();
}

/*
** [101] proto_GetKeyboardMappingReply
*/
void
proto_GetKeyboardMappingReply
   AL((cp, nspc, n, data))
   DB client_t *cp
   DD u8_t nspc		/* number of keysyms per keycode */
   DD int n		/* number of keycodes */
   DD u32_t *data
   DE
{
   register int totlen, xtra, pad;
   register buffer_t *bp = zb_rbuf();
   register xGetKeyboardMappingReply *p;

   xtra = n * nspc * sizeof(u32_t);
   pad = PAD(xtra);
   totlen = sz_xGetKeyboardMappingReply + xtra + pad;
   if (buf_avail(bp) < totlen)
      buf_adjust(bp, totlen);

   zb_client_dest(cp);

   p = (xGetKeyboardMappingReply *)buf_next(bp);

   p->type = X_Reply;
   p->sequenceNumber = cp->seqno;
   p->length = (xtra + pad) / 4;
   p->keySymsPerKeyCode = nspc;

   buf_grow(bp, sz_xGetKeyboardMappingReply);

   buf_put(bp, (char *)data, xtra);
   if (pad)
      buf_put(bp, padsrc, pad);

   rs_mark(buf_chunk(bp), X_GetKeyboardMapping, 0);
   zb_client_queue();
}

/*
** [110] proto_ListHostsReply
*/
void
proto_ListHostsReply
   AL((cp, mode, addrlen, nhosts))
   DB client_t *cp
   DD u8_t mode
   DD u32_t addrlen
   DD u16_t nhosts
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xListHostsReply *p;

   if (buf_avail(bp) < sz_xListHostsReply)
      buf_adjust(bp, sz_xListHostsReply);

   zb_client_dest(cp);

   p = (xListHostsReply *)buf_next(bp);

   p->type = X_Reply;
   p->sequenceNumber = cp->seqno;
   p->enabled = mode;
   p->length = (addrlen + nhosts * sz_xHostEntry)/4;
   p->nHosts = nhosts;

   buf_grow(bp, sz_xListHostsReply);

   rs_mark(buf_chunk(bp), X_ListHosts, 0);

   /* proto_HostEntry calls should be made following this */
}

/*
** [116] proto_SetPointerMappingReply
*/
void
proto_SetPointerMappingReply
   AL((cp, success))
   DB client_t *cp
   DD u8_t success
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xSetPointerMappingReply *p;

   if (buf_avail(bp) < sz_xSetPointerMappingReply)
      buf_adjust(bp, sz_xSetPointerMappingReply);

   zb_client_dest(cp);

   p = (xSetPointerMappingReply *)buf_next(bp);

   p->type = X_Reply;
   p->sequenceNumber = cp->seqno;
   p->length = 0;
   p->success = success;

   buf_grow(bp, sz_xSetPointerMappingReply);

   rs_mark(buf_chunk(bp), X_SetPointerMapping, 0);
   zb_client_queue();
}

/*
** [117] proto_GetPointerMappingReply
*/
void
proto_GetPointerMappingReply
   AL((cp, nElts, map))
   DB client_t *cp
   DD u8_t nElts
   DD u8_t *map
   DE
{
   register int totlen, xtra, pad;
   register buffer_t *bp = zb_rbuf();
   register xGetPointerMappingReply *p;

   xtra = nElts;
   pad = PAD(xtra);
   totlen = sz_xGetPointerMappingReply + xtra + pad;
   if (buf_avail(bp) < totlen)
      buf_adjust(bp, totlen);

   zb_client_dest(cp);

   p = (xGetPointerMappingReply *)buf_next(bp);

   p->type = X_Reply;
   p->sequenceNumber = cp->seqno;
   p->length = (xtra + pad) / 4;
   p->nElts = nElts;

   buf_grow(bp, sz_xGetPointerMappingReply);
   if (xtra) {
      buf_put(bp, (char *)map, xtra);
      if (pad)
         buf_put(bp, padsrc, pad);
   }
   rs_mark(buf_chunk(bp), X_GetPointerMapping, 0);
   zb_client_queue();
}

/*
** [118] proto_SetModifierMappingReply
*/
void
proto_SetModifierMappingReply
   AL((cp, success))
   DB client_t *cp
   DD u8_t success
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xSetModifierMappingReply *p;

   if (buf_avail(bp) < sz_xSetModifierMappingReply)
      buf_adjust(bp, sz_xSetModifierMappingReply);

   zb_client_dest(cp);

   p = (xSetModifierMappingReply *)buf_next(bp);

   p->type = X_Reply;
   p->sequenceNumber = cp->seqno;
   p->length = 0;
   p->success = success;

   buf_grow(bp, sz_xSetModifierMappingReply);

   rs_mark(buf_chunk(bp), X_SetModifierMapping, 0);
   zb_client_queue();
}

/*
** [119] proto_GetModifierMappingReply
*/
void
proto_GetModifierMappingReply
   AL((cp, ncpm, map))
   DB client_t *cp
   DD u8_t ncpm
   DD u8_t *map
   DE
{
   register int totlen, xtra, pad;
   register buffer_t *bp = zb_rbuf();
   register xGetModifierMappingReply *p;

   xtra = ncpm * 8;
   pad = PAD(xtra);
   totlen = sz_xGetModifierMappingReply + xtra + pad;
   if (buf_avail(bp) < totlen)
      buf_adjust(bp, totlen);

   zb_client_dest(cp);

   p = (xGetModifierMappingReply *)buf_next(bp);

   p->type = X_Reply;
   p->sequenceNumber = cp->seqno;
   p->length = (xtra + pad) / 4;
   p->numKeyPerModifier = ncpm;

   buf_grow(bp, sz_xGetModifierMappingReply);
   if (xtra) {
      buf_put(bp, (char *)map, xtra);
      if (pad)
         buf_put(bp, padsrc, pad);
   }
   rs_mark(buf_chunk(bp), X_GetModifierMapping, 0);
   zb_client_queue();
}

/************************************************************************
*	EVENTS								*
************************************************************************/
/*
** [02]	KeyPress (detail)
*/
/*
** [03]	KeyRelease (detail)
*/
/*
** [04]	ButtonPress (detail)
*/
/*
** [05]	ButtonRelease (detail)
*/
/*
** [06]	MotionNotify (detail)
*/
/*
** [07]	EnterNotify (detail)
*/
void
proto_EnterNotify
   AL((time, root, event, child, rx,ry, ex,ey, mask, mode, focus, detail))
   DB timestamp_t time
   DD rid_t root
   DD rid_t event
   DD rid_t child
   DD s16_t rx
   DD s16_t ry
   DD s16_t ex
   DD s16_t ey
   DD u16_t mask
   DD u8_t mode
   DD u8_t focus
   DD u8_t detail
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xEvent *p;

   if (buf_avail(bp) < sz_xEvent)
      buf_adjust(bp, sz_xEvent);

   p = (xEvent *)buf_next(bp);

   p->u.u.type = EnterNotify;
   p->u.u.detail = detail;
   p->u.enterLeave.time = time;
   p->u.enterLeave.root = root;
   p->u.enterLeave.event = event;
   p->u.enterLeave.child = child;
   p->u.enterLeave.rootX = rx;
   p->u.enterLeave.rootY = ry;
   p->u.enterLeave.eventX = ex;
   p->u.enterLeave.eventY = ey;
   p->u.enterLeave.state = mask;
   p->u.enterLeave.mode = mode;
   p->u.enterLeave.flags = (focus ? 1 : 0) | 2;

   buf_grow(bp, sz_xEvent);

   es_mark(bp);
   zb_client_queue();
}

/*
** [08]	LeaveNotify (detail)
*/
void
proto_LeaveNotify
   AL((time, root, event, child, rx,ry, ex,ey, mask, mode, focus, detail))
   DB timestamp_t time
   DD rid_t root
   DD rid_t event
   DD rid_t child
   DD s16_t rx
   DD s16_t ry
   DD s16_t ex
   DD s16_t ey
   DD u16_t mask
   DD u8_t mode
   DD u8_t focus
   DD u8_t detail
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xEvent *p;

   if (buf_avail(bp) < sz_xEvent)
      buf_adjust(bp, sz_xEvent);

   p = (xEvent *)buf_next(bp);

   p->u.u.type = LeaveNotify;
   p->u.u.detail = detail;
   p->u.enterLeave.time = time;
   p->u.enterLeave.root = root;
   p->u.enterLeave.event = event;
   p->u.enterLeave.child = child;
   p->u.enterLeave.rootX = rx;
   p->u.enterLeave.rootY = ry;
   p->u.enterLeave.eventX = ex;
   p->u.enterLeave.eventY = ey;
   p->u.enterLeave.state = mask;
   p->u.enterLeave.mode = mode;
   p->u.enterLeave.flags = (focus ? 1 : 0) | 2;

   buf_grow(bp, sz_xEvent);

   es_mark(bp);
   zb_client_queue();
}

/*
** [09]	FocusIn (detail)
*/
void
proto_FocusIn
   AL((window, mode, detail))
   DB rid_t window
   DD u8_t mode
   DD u8_t detail
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xEvent *p;

   if (buf_avail(bp) < sz_xEvent)
      buf_adjust(bp, sz_xEvent);

   p = (xEvent *)buf_next(bp);

   p->u.u.type = FocusIn;
   p->u.u.detail = detail;
   p->u.focus.window = window;
   p->u.focus.mode = mode;

   buf_grow(bp, sz_xEvent);

   es_mark(bp);
   zb_client_queue();
}

/*
** [10]	FocusOut (detail)
*/
void
proto_FocusOut
   AL((window, mode, detail))
   DB rid_t window
   DD u8_t mode
   DD u8_t detail
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xEvent *p;

   if (buf_avail(bp) < sz_xEvent)
      buf_adjust(bp, sz_xEvent);

   p = (xEvent *)buf_next(bp);

   p->u.u.type = FocusOut;
   p->u.u.detail = detail;
   p->u.focus.window = window;
   p->u.focus.mode = mode;

   buf_grow(bp, sz_xEvent);

   es_mark(bp);
   zb_client_queue();
}

/*
** [11]	KeymapNotify
*/
void
proto_KeymapNotify
   AL((keymap))
   DB u8_t *keymap	/* a 32-byte vector, byte 0 is unused */
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xKeymapEvent *p;

   if (buf_avail(bp) < sz_xKeymapEvent)
      buf_adjust(bp, sz_xKeymapEvent);

   p = (xKeymapEvent *)buf_next(bp);

   p->type = KeymapNotify;
   bcopy((char *)&keymap[1], (char *)&p->map[0], 31);

   buf_grow(bp, sz_xKeymapEvent);

   es_mark(bp);
   zb_client_queue();
}

/*
** [12]	Expose
*/
void
proto_Expose
   AL((window, x, y, w, h, cnt))
   DB rid_t window
   DD u16_t x
   DD u16_t y
   DD u16_t w
   DD u16_t h
   DD u16_t cnt
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xEvent *p;

   if (buf_avail(bp) < sz_xEvent)
      buf_adjust(bp, sz_xEvent);

   p = (xEvent *)buf_next(bp);

   p->u.u.type = Expose;
   p->u.expose.window = window;
   p->u.expose.x = x;
   p->u.expose.y = y;
   p->u.expose.width = w;
   p->u.expose.height = h;
   p->u.expose.count = cnt;

   buf_grow(bp, sz_xEvent);

   es_mark(bp);
   zb_client_queue();
}

/*
** [13]	GraphicsExposure
*/
/*
** [14]	NoExposure
*/
/*
** [15]	VisibilityNotify
*/

/*
** [16]	CreateNotify
*/
void
proto_CreateNotify
   AL((parent, window, x, y, w, h, bw, override))
   DB rid_t parent
   DD rid_t window
   DD s16_t x
   DD s16_t y
   DD u16_t w
   DD u16_t h
   DD u16_t bw
   DD u8_t override
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xEvent *p;

   if (buf_avail(bp) < sz_xEvent)
      buf_adjust(bp, sz_xEvent);

   p = (xEvent *)buf_next(bp);

   p->u.u.type = CreateNotify;
   p->u.createNotify.parent = parent;
   p->u.createNotify.window = window;
   p->u.createNotify.x = x;
   p->u.createNotify.y = y;
   p->u.createNotify.width = w;
   p->u.createNotify.height = h;
   p->u.createNotify.borderWidth = bw;
   p->u.createNotify.override = override;

   buf_grow(bp, sz_xEvent);

   es_mark(bp);
   zb_client_queue();
}


/*
** [17]	DestroyNotify
*/
void
proto_DestroyNotify
   AL((event, window))
   DB rid_t event
   DD rid_t window
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xEvent *p;

   if (buf_avail(bp) < sz_xEvent)
      buf_adjust(bp, sz_xEvent);

   p = (xEvent *)buf_next(bp);

   p->u.u.type = DestroyNotify;
   p->u.destroyNotify.event = event;
   p->u.destroyNotify.window = window;

   buf_grow(bp, sz_xEvent);

   es_mark(bp);
   zb_client_queue();
}

/*
** [18]	UnmapNotify
*/
void
proto_UnmapNotify
   AL((event, window, fromconf))
   DB rid_t event
   DD rid_t window
   DD u8_t fromconf
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xEvent *p;

   if (buf_avail(bp) < sz_xEvent)
      buf_adjust(bp, sz_xEvent);

   p = (xEvent *)buf_next(bp);

   p->u.u.type = UnmapNotify;
   p->u.unmapNotify.event = event;
   p->u.unmapNotify.window = window;
   p->u.unmapNotify.fromConfigure = fromconf;

   buf_grow(bp, sz_xEvent);

   es_mark(bp);
   zb_client_queue();
}

/*
** [19]	MapNotify
*/
void
proto_MapNotify
   AL((event, window, override))
   DB rid_t event
   DD rid_t window
   DD u8_t override
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xEvent *p;

   if (buf_avail(bp) < sz_xEvent)
      buf_adjust(bp, sz_xEvent);

   p = (xEvent *)buf_next(bp);

   p->u.u.type = MapNotify;
   p->u.mapNotify.event = event;
   p->u.mapNotify.window = window;
   p->u.mapNotify.override = override;

   buf_grow(bp, sz_xEvent);

   es_mark(bp);
   zb_client_queue();
}

/*
** [20]	MapRequest
*/
void
proto_MapRequest
   AL((parent, window))
   DB rid_t parent
   DD rid_t window
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xEvent *p;

   if (buf_avail(bp) < sz_xEvent)
      buf_adjust(bp, sz_xEvent);

   p = (xEvent *)buf_next(bp);

   p->u.u.type = MapRequest;
   p->u.mapRequest.parent = parent;
   p->u.mapRequest.window = window;

   buf_grow(bp, sz_xEvent);

   es_mark(bp);
   zb_client_queue();
}

/*
** [21]	ReparentNotify
*/
void
proto_ReparentNotify
   AL((event, window, parent, x, y, override))
   DB rid_t event
   DD rid_t window
   DD rid_t parent
   DD s16_t x
   DD s16_t y
   DD u8_t override
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xEvent *p;

   if (buf_avail(bp) < sz_xEvent)
      buf_adjust(bp, sz_xEvent);

   p = (xEvent *)buf_next(bp);

   p->u.u.type = ReparentNotify;
   p->u.reparent.event = event;
   p->u.reparent.window = window;
   p->u.reparent.parent = parent;
   p->u.reparent.x = x;
   p->u.reparent.y = y;
   p->u.reparent.override = override;

   buf_grow(bp, sz_xEvent);

   es_mark(bp);
   zb_client_queue();
}

/*
** [22]
*/
void
proto_ConfigureNotify
   AL((event, window, sibling, x, y, w, h, bw, override))
   DB rid_t event
   DD rid_t window
   DD rid_t sibling
   DD s16_t x
   DD s16_t y
   DD u16_t w
   DD u16_t h
   DD u16_t bw
   DD u8_t override
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xEvent *p;

   if (buf_avail(bp) < sz_xEvent)
      buf_adjust(bp, sz_xEvent);

   p = (xEvent *)buf_next(bp);

   p->u.u.type = ConfigureNotify;
   p->u.configureNotify.event = event;
   p->u.configureNotify.window = window;
   p->u.configureNotify.aboveSibling = sibling;
   p->u.configureNotify.x = x;
   p->u.configureNotify.y = y;
   p->u.configureNotify.width = w;
   p->u.configureNotify.height = h;
   p->u.configureNotify.borderWidth = bw;
   p->u.configureNotify.override = override;

   buf_grow(bp, sz_xEvent);

   es_mark(bp);
   zb_client_queue();
}

/*
** [23]
*/
void
proto_ConfigureRequest
   AL((parent,window,sibling, x,y, w,h,bw, stackmode, mask))
   DB rid_t parent
   DD rid_t window
   DD rid_t sibling
   DD s16_t x
   DD s16_t y
   DD u16_t w
   DD u16_t h
   DD u16_t bw
   DD u8_t stackmode
   DD mask_t mask
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xEvent *p;

   if (buf_avail(bp) < sz_xEvent)
      buf_adjust(bp, sz_xEvent);

   p = (xEvent *)buf_next(bp);

   p->u.u.type = ConfigureRequest;
   p->u.u.detail = stackmode;
   p->u.configureRequest.parent = parent;
   p->u.configureRequest.window = window;
   p->u.configureRequest.sibling = sibling;
   p->u.configureRequest.x = x;
   p->u.configureRequest.y = y;
   p->u.configureRequest.width = w;
   p->u.configureRequest.height = h;
   p->u.configureRequest.borderWidth = bw;
   p->u.configureRequest.valueMask = mask;

   buf_grow(bp, sz_xEvent);

   es_mark(bp);
   zb_client_queue();
}

/*
** [24]	GravityNotify
*/
void
proto_GravityNotify
   AL((event, window, x, y))
   DB rid_t event
   DD rid_t window
   DD s16_t x
   DD s16_t y
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xEvent *p;

   if (buf_avail(bp) < sz_xEvent)
      buf_adjust(bp, sz_xEvent);

   p = (xEvent *)buf_next(bp);

   p->u.u.type = GravityNotify;
   p->u.gravity.event = event;
   p->u.gravity.window = window;
   p->u.gravity.x = x;
   p->u.gravity.y = y;

   buf_grow(bp, sz_xEvent);

   es_mark(bp);
   zb_client_queue();
}

/*
** [25]
*/
void
proto_ResizeRequest
   AL((window, w, h))
   DB rid_t window
   DD u16_t w
   DD u16_t h
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xEvent *p;

   if (buf_avail(bp) < sz_xEvent)
      buf_adjust(bp, sz_xEvent);

   p = (xEvent *)buf_next(bp);

   p->u.u.type = ResizeRequest;
   p->u.resizeRequest.window = window;
   p->u.resizeRequest.width = w;
   p->u.resizeRequest.height = h;

   buf_grow(bp, sz_xEvent);

   es_mark(bp);
   zb_client_queue();
}

/*
** [26]	CirculateNotify
*/
void
proto_CirculateNotify
   AL((event, window, place))
   DB rid_t event
   DD rid_t window
   DD u8_t place
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xEvent *p;

   if (buf_avail(bp) < sz_xEvent)
      buf_adjust(bp, sz_xEvent);

   p = (xEvent *)buf_next(bp);

   p->u.u.type = CirculateRequest;
   p->u.circulate.event = event;
   p->u.circulate.window = window;
   p->u.circulate.place = place;

   buf_grow(bp, sz_xEvent);

   es_mark(bp);
   zb_client_queue();
}

/*
** [27]	CirculateRequest
*/
void
proto_CirculateRequest
   AL((parent, window, place))
   DB rid_t parent
   DD rid_t window
   DD u8_t place
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xEvent *p;

   if (buf_avail(bp) < sz_xEvent)
      buf_adjust(bp, sz_xEvent);

   p = (xEvent *)buf_next(bp);

   p->u.u.type = CirculateRequest;
   p->u.circulate.event = parent;
   p->u.circulate.window = window;
   p->u.circulate.place = place;

   buf_grow(bp, sz_xEvent);

   es_mark(bp);
   zb_client_queue();
}

/*
** [28]	PropertyNotify
*/
void
proto_PropertyNotify
   AL((window, atom, time, state))
   DB rid_t window
   DD atom_t atom
   DD timestamp_t time
   DD u8_t state
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xEvent *p;

   if (buf_avail(bp) < sz_xEvent)
      buf_adjust(bp, sz_xEvent);

   p = (xEvent *)buf_next(bp);

   p->u.u.type = PropertyNotify;
   p->u.property.window = window;
   p->u.property.atom = atom;
   p->u.property.time = time;
   p->u.property.state = state;

   buf_grow(bp, sz_xEvent);

   es_mark(bp);
   zb_client_queue();
}

/*
** [29]	SelectionClear
*/
void
proto_SelectionClear
   AL((time, window, atom))
   DB timestamp_t time
   DD rid_t window
   DD atom_t atom
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xEvent *p;

   if (buf_avail(bp) < sz_xEvent)
      buf_adjust(bp, sz_xEvent);

   p = (xEvent *)buf_next(bp);

   p->u.u.type = SelectionClear;
   p->u.selectionClear.time = time;
   p->u.selectionClear.window = window;
   p->u.selectionClear.atom = atom;

   buf_grow(bp, sz_xEvent);

   es_mark(bp);
   zb_client_queue();
}

/*
** [30]	SelectionRequest
*/
void
proto_SelectionRequest
   AL((time, owner, requestor, selection, target, property))
   DB timestamp_t time
   DD rid_t owner
   DD rid_t requestor
   DD atom_t selection
   DD atom_t target
   DD atom_t property
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xEvent *p;

   if (buf_avail(bp) < sz_xEvent)
      buf_adjust(bp, sz_xEvent);

   p = (xEvent *)buf_next(bp);

   p->u.u.type = SelectionRequest;
   p->u.selectionRequest.time = time;
   p->u.selectionRequest.owner = owner;
   p->u.selectionRequest.requestor = requestor;
   p->u.selectionRequest.selection = selection;
   p->u.selectionRequest.target = target;
   p->u.selectionRequest.property = property;

   buf_grow(bp, sz_xEvent);

   es_mark(bp);
   zb_client_queue();
}

/*
** [31]	SelectionNotify
*/
void
proto_SelectionNotify
   AL((time, requestor, selection, target, property))
   DB timestamp_t time
   DD rid_t requestor
   DD atom_t selection
   DD atom_t target
   DD atom_t property
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xEvent *p;

   if (buf_avail(bp) < sz_xEvent)
      buf_adjust(bp, sz_xEvent);

   p = (xEvent *)buf_next(bp);

   p->u.u.type = SelectionNotify;
   p->u.selectionNotify.time = time;
   p->u.selectionNotify.requestor = requestor;
   p->u.selectionNotify.selection = selection;
   p->u.selectionNotify.target = target;
   p->u.selectionNotify.property = property;

   buf_grow(bp, sz_xEvent);

   es_mark(bp);
   zb_client_queue();
}

/*
** [32]	ColormapNotify
*/
void
proto_ColormapNotify
   AL((window, colormap, new, state))
   DB rid_t window
   DD rid_t colormap
   DD u8_t new
   DD u8_t state
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xEvent *p;

   if (buf_avail(bp) < sz_xEvent)
      buf_adjust(bp, sz_xEvent);

   p = (xEvent *)buf_next(bp);

   p->u.u.type = ColormapNotify;
   p->u.colormap.window = window;
   p->u.colormap.colormap = colormap;
   p->u.colormap.new = new;
   p->u.colormap.state = state;

   buf_grow(bp, sz_xEvent);

   es_mark(bp);
   zb_client_queue();
}

/*
** [33]	ClientMessage (detail)
*/
/*
** [34]	MappingNotify
*/
void
proto_MappingNotify
   AL((request, first, count))
   DB u8_t request
   DD u8_t first
   DD u8_t count
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xEvent *p;

   if (buf_avail(bp) < sz_xEvent)
      buf_adjust(bp, sz_xEvent);

   p = (xEvent *)buf_next(bp);

   p->u.u.type = MappingNotify;
   p->u.mappingNotify.request = request;
   p->u.mappingNotify.firstKeyCode = first;
   p->u.mappingNotify.count = count;

   buf_grow(bp, sz_xEvent);

   es_mark(bp);
   zb_client_queue();
}

/************************************************************************
*	ERRORS								*
************************************************************************/
void
proto_Error
   AL((cp, code, value, minorop, majorop))
   DB client_t *cp
   DD u8_t code
   DD u32_t value
   DD u16_t minorop
   DD u8_t majorop
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xError *p;

   if (buf_avail(bp) < sz_xError)
      buf_adjust(bp, sz_xError);

   zb_client_dest(cp);

   p = (xError *)buf_next(bp);

   p->type = X_Error;
   p->errorCode = code;
   p->sequenceNumber = cp->seqno;
   p->resourceID = value;
   p->minorCode = minorop;
   p->majorCode = majorop;

   buf_grow(bp, sz_xError);

   (buf_chunk(bp))->type = P_ERROR;
   zb_client_queue();
}

/************************************************************************
*	MISCELLANEOUS							*
************************************************************************/

/*
**       proto_ColorItem
*/
void
proto_ColorItem
   AL((pixel, red, green, blue, flags))
   DB pixel_t pixel
   DD u16_t red
   DD u16_t green
   DD u16_t blue
   DD u8_t flags
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xColorItem *p;

   if (buf_avail(bp) < sz_xColorItem)
      buf_adjust(bp, sz_xColorItem);

   p = (xColorItem *)buf_next(bp);

   p->pixel = pixel;
   p->red = red;
   p->green = green;
   p->blue = blue;
   p->flags = flags;
   buf_grow(bp, sz_xColorItem);
}
/*
**       proto_HostEntry
*/
void
proto_HostEntry
   AL((family, length, address))
   DB u8_t family
   DD u16_t length
   DD char *address
   DE
{
   register buffer_t *bp = zb_rbuf();
   register xHostEntry *p;

   if (buf_avail(bp) < sz_xHostEntry)
      buf_adjust(bp, sz_xHostEntry);

   p = (xHostEntry *)buf_next(bp);

   p->family = family;
   p->length = length;
   buf_grow(bp, sz_xHostEntry);

   if (length) {
      buf_put(bp, address, length);
      if (PAD(length))
         buf_put(bp, padsrc, PAD(length));
   }
}
