/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   pp.c								*
*									*
*	Partial packet pool routines.					*
*									*
*	They don't belong anywhere else, so they're here.		*
*									*
************************************************************************/
#define NEED_REPLIES
#define NEED_EVENTS
#include <X11/Xproto.h>

#include "xmx.h"
#include "df.h"
#include "incl/pp.pvt.h"

/*
**  partial data types
*/
#define PP_NONE		0
#define PP_IMAGE	1
#define PP_SHORTS	2
#define PP_BYTES	3

static pp_t *ppfree;

/************************************************************************
*									*
*   pp_new								*
*									*
************************************************************************/
pp_t *
pp_new
   VOID
{
   register pp_t *pp;

   if (ppfree) {
      pp = ppfree;
      ppfree = ppfree->next;
   }
   else if (MALLOC(pp, pp_t *, sizeof(pp_t)))
      return (pp_t *)err(0, "malloc returned zero");

   return pp;
}

/************************************************************************
*									*
*   pp_free								*
*									*
************************************************************************/
void
pp_free
   AL((pp))
   DB pp_t *pp
   DE
{
   if (--pp->refs <= 0) {
      pp->next = ppfree;
      ppfree = pp;
   }
}

/************************************************************************
*									*
*   pp_free_freelist							*
*									*
************************************************************************/
void
pp_free_freelist
   VOID
{
   register pp_t *pp, *last;

   for (pp=ppfree; last=pp;) {
      pp = pp->next;
      free(last);
   }
   ppfree = 0;
}

/************************************************************************
*									*
*   pp_clone								*
*									*
************************************************************************/
pp_t *
pp_clone
   AL((pp, minus))
   DB pp_t *pp
   DD int minus
   DE
{
   register pp_t *npp;

   npp = pp_new();
   bcopy(pp, npp, sizeof(pp_t));
   npp->nmore -= minus;
   npp->refs = 0;

   return npp;
}

/************************************************************************
*									*
*   pp_image								*
*									*
*	Create a partial packet record for an image request or reply.	*
*									*
************************************************************************/
pp_t *
pp_image
   AL((nmore, iord, width, depth))
   DB uint_t nmore	/* total bytes of image data, including padding */
   DD iord_t iord	/* image order of image */
   DD u16_t width	/* width in pixels */
   DD u8_t depth	/* no. of planes */
   DE
{
   register pp_t *pp;

   pp = pp_new();
   pp->type = PP_IMAGE;
   pp->nmore = nmore;
   pp->depth = depth;
   pp->width = width;
   pp->iord = iord;
   pp->refs = 0;

   return pp;
}

/************************************************************************
*									*
*   pp_process								*
*									*
*	The given buffer contains the continuation of a partially	*
*	received packet.  The given partial packet record describes	*
*	what remains.  This routine splits off a chunk with the		*
*	largest piece that can be processed and returns it.  It also	*
*	takes care of clearing and setting the proper pp_t's for this	*
*	and any data which will follow.					*
*									*
************************************************************************/
chunk_t *
pp_process
   AL((bp, ppp))
   DB buffer_t *bp	/* this can contain zero bytes */
   DD pp_t **ppp	/* if last partial, this will be cleared */
   DE
{
   register uint_t n;
   register chunk_t *chp = 0;
   register pp_t *npp;
   register pp_t *pp = *ppp;

   if (n = MIN(pp->nmore, buf_active(bp)))
      switch (pp->type) {
         case PP_IMAGE:
            chp = image_source(pp->iord, pp->width, pp->depth, bp, n);
            break;
         case PP_SHORTS:
            n &= ~1;	/* even number */
            if (n)
               chp = buf_split(bp, n);
            break;
         case PP_BYTES:
            if (n)
               chp = buf_split(bp, n);
            break;
         default:
            warn("pp_process: bad pp data type [%d]\n", pp->type);
            if (n)
               chp = buf_split(bp, n);
            break;
      }
   if (chp) {
      chp->lpp = pp;	/* instead of pp_assign, because ref stays the same */

      if (buf_chunksize(chp) < pp->nmore) {
         npp = pp_clone(pp, buf_chunksize(chp));
         pp_assign(*ppp, npp);
         pp_assign(chp->tpp, npp);
      }
      else
         *ppp = 0;
   }
   return chp;
}

/************************************************************************
*									*
*   pp_print								*
*									*
************************************************************************/
void
pp_print
   AL((pp))
   DB pp_t *pp
   DE
{
   warn("pp->nmore [%d]\n", pp->nmore);
   warn("pp->depth [%d]\n", pp->depth);
   warn("pp->width [%d]\n", pp->width);
   warn("pp->refs [%d]\n", pp->refs);
   warn("pp->iord [%d]\n", pp->iord);
   warn("pp->next [0x%x]\n", pp->next);
}
