/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   main.c								*
*									*
*	Xmx is an X protocol multiplexor.				*
*									*
************************************************************************/
#include <stdio.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/resource.h>
#ifdef _AIX
#include <sys/select.h>
#endif
#include <unistd.h>
#include <errno.h>

#define NEED_REPLIES
#define NEED_EVENTS
#include <X11/Xproto.h>
#include <xmc.h>
#include <xmcp.h>

#define DEFINE_GLOBALS
#include "xmx.h"
#include "df.h"
#include "cx.h"
#include "rx.h"
#include "zb.h"
#include "res.h"
#include "ptc.h"
#include "incl/main.pvt.h"
#include "fd.h"
#include "version.h"
#include "patchlevel.h"

#ifdef NEVER
/* hack to force list.c dependency */
list_new();
#endif

/*
**   main
*/
int
main
   AL((argc, argv))
   DB int argc
   DD char **argv
   DE
{
   register int i, j, s;
   register int nfds;
   register int rv;
   register rx_t *rxp;
   chunk_t *vchp;
   u16_t Bl = 0x426c;
   int *fdv;
   int fdts;			/* file descriptor table size */
   fd_set rfds, wfds;
   hostaddr_t addr;
   struct timeval tv;
   void *dp;

   xmx_version_major = VERSION_MAJOR;
   xmx_version_minor = VERSION_MINOR;
   xmx_release_status = RELEASE_STATUS;
   xmx_patchlevel = PATCHLEVEL;

   config_mode = ConfigAllow;

   util_setsig();

   endian = *((char *)&Bl);	/* establish endianness */
   fdts = get_fdts();
   fd_init();
   if (atom_reset())
      exit(-1);
   vconf_init();
   vmap = hash_new(VMAPSIZE);
   host_me();

   bmiord = image_xy_iord(0, 0, 8, 8);	/* bitmap image order */

   util_init();
   zb_init();
   inp_init();
   args_get(argc, argv);	/* also fills "servers" array from argv */
   auth_reset();
   rgb_init();
   cliz_init();

   ext_init();
   xmc_init();
   event_init();

   if (CALLOC(cv, conn_t *, (unsigned)fdts, sizeof(conn_t)))
      exit(-1);

   xmc_reset(0, 1);		/* kicks things off */
   zb_queue_all();

   if (socket_as_Xserver(opt.dpy))	/* x server sockets */
      quit(-1, "unable to establish X server ports\n");
   if (opt.xmc)				/* xmc server sockets */
      if (socket_as_xmc(opt.mdpynum))
         warn("unable to establish multiplexor control ports\n");
   if (opt.xdm)				/* xdm server sockets */
      for (i=socket_as_xdm(&fdv); --i>=0;)
         main_open_conn(fdv[i], XDMCP, 0);

   for (;;) {			/* main loop */
      fd_fds(&rfds, &wfds);

      DEBUG2(D_SELECT, "main: selecting... rfds 0x%04x wfds 0x%04x\n",
					rfds.fds_bits[0], wfds.fds_bits[0]);

      if (exposures) {		/* exposure processing delay - brief! */
         tv.tv_sec = 0;
         tv.tv_usec = 1000;		/* 1 millisecond */
      }
      else if (ntimeouts) {	/* polling for server timeout */
         tv.tv_sec = 1;			/* 1 sec */
         tv.tv_usec = 0;
      }
      else {			/* poll for hash table garbage collection */
         tv.tv_sec = 10;
         tv.tv_usec = 0;
      }
      /*
      ** kneel before the monolithic select!
      */
      if ((nfds = select(MAXFD+1, fdcast(&rfds), fdcast(&wfds), 0, &tv)) < 0) {
         /*
         ** if the signal handler didn't exit, don't exit
         */
         if (errno == EINTR)
            continue;
         pquit(1, "select");
      }
      else if (nfds == 0) {		/* select timed out */
         if (exposures)
            expos_timeout();
         else if (ntimeouts) {
            while ((i = time_out()) >= 0) {
               switch (cv[i].type) {
                  case CLIENT:
                     /*
                     **  Timeout occurs before the client has had a chance
                     **  to actually do anything, so there is nothing to
                     **  clean up, just free the storage and close the socket.
                     */
                     client_free(cv[i].u.cp);
                     main_close_conn(i);
                     break;
                  case SERVER:
                     warn("server '%s' timed out\n", cv[i].u.sp->tag);
                     xmc_drop(cv[i].u.sp);
                     break;
                  default:
                     warn("main: connection type %s timed out!\n",
					debug_conn_type_str(cv[i].type));
                     break;
               }
            }
         }
         else if (hash_dirty(vmap))
            if (! fd_inreset()) {	/* this test is awkward... TODO */
               zb_dest(ZD_ALLSERV, 0);
               cliz_sync(hash_cleanup_cont);
               zb_queue(0);
            }
         continue;
      }
      DEBUG3(D_SELECT, "main: selected...  rfds 0x%04x wfds 0x%04x nfds %d\n",
				rfds.fds_bits[0], wfds.fds_bits[0], nfds);
      /*
      **   Is nfds the sum of the number of bits set in the fd_set's
      **   set by select(2), or is it the number of fd's that
      **   have some i/o condition pending?  Guess what?  The answer
      **   is implementation dependent (SunOS is bits, IRIX is fds).
      **
      **   That's the reason we spin on MAXFD instead of nfds.  Pfft.
      */
      /*
      **   First, flush anything waiting to go out.
      */
      for (i=0; i<=MAXFD; i++) {
         if (FD_ISSET(i, &wfds)) {
            switch (cv[i].type) {
               case CLIENT:
                  queue_flush(cv[i].u.cp->qp);
                  break;
               case SERVER:
                  queue_flush(cv[i].u.sp->qp);
                  break;
               case XMC:
                  queue_flush(cv[i].u.xp->qp);
                  break;
            }
         }
      }
      /*
      **   Those flushes may have cleared the (incoming) buffers
      **   of some clients that are blocked.  If so, unblock them.
      */
      if (fd_blocked())		/* blocked clients? */
         for (i=j=1; j<num_clients; i++)
            if (clients[i]) {
               j++;
               if (	FD_ISSET(clients[i]->fd, &blockfds) &&
			buf_used(cv[clients[i]->fd].bp) < BIGBUFFER) {
                  fd_unblock(clients[i]->fd);
               }
            }
      /*
      **   Reconsider previously ignored inputs
      */
      if (fd_free_saves())
         fd_restore(&rfds);
      /*
      **   Proceed to incoming data.
      */
      for (i=0; i<=MAXFD; i++) {
         if (FD_ISSET(i, &rfds)) {
            switch (cv[i].type) {
               case XPORT:
                  DEBUG1(D_NETWORK, "XPORT socket [%d] in\n", i);
                  if ((s = socket_accept(i)) < 0)
                     break;
                  if (	socket_addr(s, cv[i].u.pp->family, &addr) ||
			(dp = (void *)client_alloc(s, &addr)) == 0) {
                     close(s);
                     break;
                  }
                  time_start(s, opt.to);
                  main_open_conn(s, CLIENT, dp);
                  /*
                  **  This little hack vastly simplifies things.
                  */
                  cv[s].bp->clinum = ((client_t *)dp)->clinum;
                  break;
               case XMCPORT:
                  DEBUG1(D_NETWORK, "XMCPORT socket [%d] in\n", i);
                  if ((s = socket_accept(i)) < 0)
                     break;
                  if (	socket_addr(s, cv[i].u.pp->family, &addr) ||
			(dp = (void *)xmc_alloc(s, &addr)) == 0) {
                     close(s);
                     break;
                  }
                  main_open_conn(s, XMC, dp);
                  break;
               case CLIENT:
                  DEBUG1(D_NETWORK, "CLIENT socket [%d] in\n", i);
                  /*
                  **  spin here instead of blocking in select because
                  **  we can't rely on wfds to trigger the select
                  */
                  if (FD_ISSET(i, &blockfds))
                     break;

                  rv = buf_read(i, cv[i].bp);
                  /*
                  **   always process data if there's any
                  */
                  if (buf_active(cv[i].bp)) {
                     DEBUG1(D_NETWORK, "\tchunksize [%d]\n",
					buf_chunksize(buf_chunk(cv[i].bp)));
                     if (cv[i].u.cp->state == K_NEW) { /* connection block */
                        if (connect_client_in(cv[i].u.cp, cv[i].bp)) {
                           queue_flush(cv[i].u.cp->qp); /* send denial */
                           client_free(cv[i].u.cp);
                           main_close_conn(i);
                        }
                     }
                     else {
                        request_process(cv[i].u.cp, cv[i].bp);
                        /*
                        **   If that read pushed the buffer over an
                        **   arbitrary limit, block further reads until
                        **   it drains alittle.  Note that this has the
                        **   effect of slowing the entire session to
                        **   let slower machines keep up (and xmx from
                        **   running out of memory).
                        */
                        if (buf_used(cv[i].bp) > BIGBUFFER) {
                           fd_block(i);
                        }
                     }
                  }
                  /*
                  **   now deal with results of read
                  */
                  switch (rv) {
                     case -1:
                        pwarn("read");
                        /* fall through */
                     case 0:
                        /*
                        **   make sure client didn't die since read
                        */
                        if (cv[i].type == CLIENT)
                           cliz_kill_client(cv[i].u.cp);
                        break;
                  }
                  break;
               case SERVER:
                  DEBUG2(D_NETWORK, "SERVER socket [%d] [%s] in\n",
							i, cv[i].u.sp->tag);
                  rv = buf_read(i, cv[i].bp);

                  if (buf_active(cv[i].bp)) {
                     DEBUG1(D_NETWORK, "\tchunksize [%d]\n",
					buf_chunksize(buf_chunk(cv[i].bp)));
                     reply_process(cv[i].u.sp, cv[i].bp);
                  }
                  switch (rv) {
                     case -1:
                        pwarn("read");
                        /* fall through */
                     case 0:
                        /*
                        **   make sure server didn't die since read
                        */
                        if (cv[i].type == SERVER)
                           xmc_drop(cv[i].u.sp);
                        break;
                  }
                  break;
               case XMC:
                  DEBUG1(D_NETWORK, "XMC socket [%d] in\n", i);

                  rv = buf_read(i, cv[i].bp);
                  if (buf_active(cv[i].bp)) {
                        if (cv[i].u.xp->accepted == 0)
                           xmc_connect(cv[i].u.xp, cv[i].bp);
                        else
                           xmc_process(cv[i].u.xp, cv[i].bp);
                  }
                  switch (rv) {
                     case -1:
                        pwarn("read");
                        /* fall through */
                     case 0:
                        /*
                        **   make sure client didn't die since read
                        */
                        if (cv[i].type == XMC)
                           xmc_close(cv[i].u.xp);
                        break;
                  }
                  break;
               case XDMCP:
                  DEBUG1(D_NETWORK, "XDMCP socket [%d] in\n", i);
                  break;
            }
            if (fd_chill()) {	/* a grab of some sort has occurred */
               /*
               **  we now skip all unprocessed pending data, so we
               **  must remember the connections that have data pending,
               **  so we look at them when the grab is over
               */
               for (i++; i<=MAXFD; i++)
                  if (FD_ISSET(i, &rfds))
                     fd_save(i);

               break;	/* loop exit */
            }
         }
      }
      zb_queue_all();
   }
   /*NOTREACHED*/
}

/************************************************************************
*									*
*   main_open_conn							*
*									*
************************************************************************/
void
main_open_conn
   AL((fd, type, datap))
   DB int fd
   DD u16_t type
   DD void *datap
   DE
{
   switch (type) {
      case XPORT:
         fd_client(fd);
         cv[fd].bp = 0;
         cv[fd].u.pp = (port_t *)datap;
         break;
      case CLIENT:
         fd_client(fd);
         cv[fd].bp = buf_new(B_STATIC);
         cv[fd].u.cp = (client_t *)datap;
         break;
      case SERVER:
         fd_server(fd);
         cv[fd].bp = buf_new(B_STATIC);
         cv[fd].u.sp = (server_t *)datap;
         break;
      case XMCPORT:
         fd_client(fd);
         cv[fd].bp = 0;
         cv[fd].u.pp = (port_t *)datap;
         break;
      case XMC:
         fd_control(fd);
         cv[fd].bp = buf_new(B_STATIC);
         cv[fd].u.xp = (xmc_t *)datap;
         break;
      case XDMCP:
         fd_client(fd);
         cv[fd].bp = buf_new(B_STATIC);
         break;
      case CLOSED:
         warn("main_open_conn: can't open connection of type CLOSED\n");
         return;
      default:
         warn("main_open_conn: bad connection type [%d]\n", type);
         return;
   }
   cv[fd].type = type;
}

/************************************************************************
*									*
*   main_close_conn							*
*									*
************************************************************************/
void
main_close_conn
   AL((fd))
   DB int fd
   DE
{
   fd_clear(fd);
   cv[fd].type = CLOSED;
   if (cv[fd].bp) {
      buf_free(cv[fd].bp);
      cv[fd].bp = 0;
   }
   if (close(fd))
      pwarn("close");

   time_end(fd);
}

/*
**	get_fdts
**
**	Increase the file descriptor table size, if possible, and return it.
*/
static int
get_fdts
   VOID
{
#ifdef RLIMIT_NOFILE
   struct rlimit rlim;

   if (getrlimit(RLIMIT_NOFILE, &rlim) == 0) {
      rlim.rlim_cur = rlim.rlim_max;
      if (setrlimit(RLIMIT_NOFILE, &rlim) == 0)
         return rlim.rlim_cur;
   }
#endif
#ifdef _SC_OPEN_MAX
   return sysconf(_SC_OPEN_MAX);
#else
#ifdef NOFILE
   return NOFILE;
#else
   return getdtablesize();
#endif
#endif
}
