/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   fd.h								*
*									*
*	Macros to manage file descriptor bit vectors.			*
*									*
*	This file precedes xmx.h (due to AIX name conflicts).		*
*									*
************************************************************************/
#ifdef DEFINE_GLOBALS
#define extern
#endif

extern fd_set allfds;		/* all active fd's */
extern fd_set readfds;		/* all read fd's */
extern fd_set serverfds;	/* all server fd's */
extern fd_set resetfds;		/* server fd's during reset */
extern fd_set writefds;		/* write fd's */
extern fd_set blockfds;		/* blocked fd's */
extern fd_set holdfds;		/* hold (blocked) fd's */
extern fd_set heldfds;		/* previously held fd's (ready to go) */
extern fd_set partialfds;	/* client fd's with partial requests */
extern fd_set savefds;		/* client fd's saved */
extern int fdstate;		/* switch */
extern int nblocked;		/* how many clients blocked */
extern int grab;		/* fd of grabbing client */
extern int pending;		/* fd of source of partial request or reply */
extern int maxfd;		/* max fd */
extern int npending;		/* count of pushed pending fd's */
extern int nheld;		/* count of fd's held */
extern int npartial;		/* count of fd's with partial requests */
extern int nsaved;		/* count of fd's saved */
extern int fdchill;		/* stop processing */
extern list_t *plstp;		/* pushed pending fd's */

#ifdef extern
#undef extern
#endif

/*
**	fd states
*/
#define FS_ALL		0x0
#define FS_PENDING	0x1
#define FS_GRAB		0x2
#define FS_RESET	0x4

#define MAXFD (maxfd)		/* largest fd encountered */

#ifdef hpux
#define fdcast(a) (int *)a
#else
#define fdcast(a) a
#endif

/************************************************************************
*									*
*   fd_init								*
*									*
************************************************************************/
#define fd_init() {\
   grab = pending = maxfd = -1;\
   FD_ZERO(&allfds);\
   FD_ZERO(&readfds);\
   FD_ZERO(&serverfds);\
   FD_ZERO(&resetfds);\
   FD_ZERO(&writefds);\
   FD_ZERO(&blockfds);\
   FD_ZERO(&holdfds);\
   FD_ZERO(&heldfds);\
   FD_ZERO(&partialfds);\
   FD_ZERO(&savefds);\
   fdstate = FS_ALL;\
   plstp = list_new();\
   fdchill = 0;\
}

/************************************************************************
*									*
*   fd_server, fd_client, fd_control					*
*									*
************************************************************************/
#define fd_server(fd) {\
   FD_SET(fd, &allfds);\
   FD_SET(fd, &readfds);\
   FD_SET(fd, &serverfds);\
   if (fd > maxfd)\
      maxfd = fd;\
}

#define fd_client(fd) {\
   FD_SET(fd, &allfds);\
   FD_SET(fd, &readfds);\
   if (fd > maxfd)\
      maxfd = fd;\
}

#define fd_control(fd) {\
   FD_SET(fd, &allfds);\
   FD_SET(fd, &readfds);\
   FD_SET(fd, &serverfds);\
   if (fd > maxfd)\
      maxfd = fd;\
}

#define fd_write(fd) {\
   FD_SET(fd, &allfds);\
   FD_SET(fd, &writefds);\
   if (fd > maxfd)\
      maxfd = fd;\
}

/************************************************************************
*									*
*   fd_block, fd_unblock						*
*									*
*	Temporarily remove a fd from those selected for input.		*
*									*
*	This is conceptually the same as fd_hold/fd_unhold, but		*
*	fd_block causes xmx to spin on these fd's.  If select(2)	*
*	could be relied upon to report when a fd is truly ready		*
*	to receive data, fd_hold/fd_unhold could replace these.		*
*									*
************************************************************************/
#define fd_block(fd) {\
   DEBUG1(D_SELECT, "fd_block: %d\n", (fd));\
   if (! FD_ISSET((fd), &readfds))\
      warn("fd_block: bad file descriptor [%d]\n", fd);\
/* FD_CLR((fd), &readfds); */\
   FD_SET((fd), &blockfds);\
   nblocked++;\
}

#define fd_unblock(fd) {\
   DEBUG0(D_SELECT, "fd_unblock\n");\
/* FD_SET((fd), &readfds); */\
   FD_CLR((fd), &blockfds);\
   nblocked--;\
}

#define fd_blocked()	(nblocked)

/************************************************************************
*									*
*   fd_clear								*
*   fd_wclear								*
*									*
************************************************************************/
#define fd_clear(fd) {\
   FD_CLR((fd), &allfds);\
   FD_CLR((fd), &readfds);\
   FD_CLR((fd), &serverfds);\
   FD_CLR((fd), &resetfds);\
   FD_CLR((fd), &writefds);\
   FD_CLR((fd), &blockfds);\
   FD_CLR((fd), &holdfds);\
   FD_CLR((fd), &heldfds);\
   FD_CLR((fd), &partialfds);\
   FD_CLR((fd), &savefds);\
   if (fdstate & FS_GRAB) {\
      fd_ungrab();\
   }\
   if (fdstate & FS_PENDING && pending == (fd)) {\
      /* this is a serious problem TODO */\
      fd_unpending();\
   }\
   if ((fd) == maxfd)\
      while (! FD_ISSET(maxfd, &allfds))\
         if (maxfd-- == 0)\
            break;\
}

#define fd_wclear(fd) {\
   FD_CLR((fd), &writefds);\
   if ((fd) == maxfd)\
      while (! FD_ISSET(maxfd, &allfds))\
         if (maxfd-- == 0)\
            break;\
}

/************************************************************************
*									*
*   fd_fds								*
*									*
*	Create bit vectors for select(2) based on the current state	*
*	of the client/server dialogs.					*
*									*
*	Note that in PENDING or GRAB state, partial client requests	*
*	cause xmx to ignore all other i/o until they are completed.	*
*	This is probably not ideal, but is more efficient and speeds	*
*	the server grab along.						*
*									*
************************************************************************/
#define fd_fds(rfdsp, wfdsp)\
   if (fdstate == FS_ALL)\
      bcopy((char *)&readfds, (char *)rfdsp, sizeof(fd_set));\
   else if (fdstate & (FS_PENDING | FS_GRAB)) {\
      if (npartial) {\
         bcopy((char *)&partialfds, (char *)rfdsp, sizeof(fd_set));\
      }\
      else if (fdstate & FS_PENDING) {\
         FD_ZERO(rfdsp);\
         FD_SET(pending, rfdsp);\
      }\
      else if (fdstate & FS_GRAB) {\
         bcopy((char *)&serverfds, (char *)rfdsp, sizeof(fd_set));\
         FD_SET(grab, rfdsp);\
      }\
   }\
   else		/* server reset */\
      bcopy((char *)&resetfds, (char *)rfdsp, sizeof(fd_set));\
   \
   bcopy((char *)&writefds, (char *)wfdsp, sizeof(fd_set))

/************************************************************************
*									*
*   fd_save								*
*   fd_restore								*
*   fd_free_saves							*
*									*
*	Save and restore fds whose input we skip.			*
*									*
************************************************************************/
#define fd_save(fd) {\
   DEBUG1(D_SELECT, "fd_save: %d\n", (fd));\
   FD_SET((fd), &savefds);\
   nsaved++;\
}

#define fd_restore(rfdsp) {\
   register int _i;\
   DEBUG1(D_SELECT, "fd_restore: nsaved %d\n", nsaved);\
   for (_i=0; nsaved && _i<maxfd; _i++)\
      if (FD_ISSET(_i, &savefds)) {\
         FD_CLR(_i, &savefds);\
         FD_SET(_i, (rfdsp));\
         nsaved--;\
      }\
}

#define fd_free_saves() (nsaved && fdstate == FS_ALL)

/************************************************************************
*									*
*   fd_hold								*
*   fd_unhold								*
*   fd_setheld								*
*   fd_held								*
*									*
*	Note: server fds may bypass a hold during a grab or pending	*
*	request.  These macros only modify readfds.			*
*									*
************************************************************************/
#define fd_hold(fd) {\
   DEBUG1(D_SELECT, "fd_hold: %d\n", (fd));\
   if (! FD_ISSET((fd), &readfds))\
      warn("fd_hold: bad file descriptor [%d]\n", fd);\
   FD_CLR((fd), &readfds);\
   FD_SET((fd), &holdfds);\
   if (FD_ISSET((fd), &heldfds)) {\
      nheld--;\
      FD_CLR((fd), &heldfds);\
   }\
   nhold++;\
}

#define fd_unhold(fd) {\
   DEBUG1(D_SELECT, "fd_unhold: %d\n", (fd));\
   if (FD_ISSET((fd), &holdfds)) {\
      nhold--;\
      FD_CLR((fd), &holdfds);\
      nheld++;\
      FD_SET((fd), &heldfds);\
   }\
}

#define fd_setheld(rfdsp) {\
   register int _i;\
   for (_i=0; _i<maxfd; _i++)\
      if (FD_ISSET(_i, &heldfds)) {\
         FD_SET(_i, (rfdsp));\
         nheld--;\
         FD_CLR(_i, &heldfds);\
         /* (nfds)++; */ /* not used - see comment in main.c */\
      }\
}

#define fd_held()	(nheld)

/************************************************************************
*									*
*   fd_grab								*
*   fd_ungrab								*
*									*
*	Set or clear main loop monopolization for a client-requested	*
*	server grab.							*
*									*
************************************************************************/
#define fd_grab(fd)\
   if ((fdstate & FS_GRAB) == 0) {\
      DEBUG1(D_SELECT, "fd_grab: %d\n", (fd));\
      fdstate |= FS_GRAB;\
      fdchill = 1;\
      grab = fd;\
   }

#define fd_ungrab()\
   if (fdstate & FS_GRAB) {\
      DEBUG0(D_SELECT, "fd_ungrab\n");\
      fdstate &= ~FS_GRAB;\
   }

/************************************************************************
*									*
*   fd_partial, fd_unpartial						*
*									*
*	Note in-progress partial requests.  Fd's are those of the	*
*	clients which originated them.					*
*									*
*	This is only of consequence if a server grab or "pending"	*
*	state is attempted.  Both must wait or risk deadlock.		*
*									*
************************************************************************/
#define fd_partial(fd)\
   if (! FD_ISSET((fd), &partialfds)) {\
      DEBUG1(D_SELECT, "fd_partial %d\n", (fd));\
      FD_SET(fd, &partialfds);\
      npartial++;\
   }

#define fd_unpartial(fd)\
   if (FD_ISSET((fd), &partialfds)) {\
      DEBUG1(D_SELECT, "fd_unpartial %d\n", (fd));\
      FD_CLR(fd, &partialfds);\
      npartial--;\
   }

/************************************************************************
*									*
*   fd_pending, fd_unpending						*
*									*
*	Statements.  Set or clear main loop monopolization for an	*
*	incomplete request.						*
*									*
************************************************************************/
#define fd_pending(fd)\
   if ((fdstate & FS_PENDING) == 0) {\
      fdstate |= FS_PENDING;\
      pending = fd;\
      fdchill = 1;\
   }\
   else if (fd != pending)\
      warn("fd_pending: bad nested pending!\n");\
   npending++

#define fd_unpending()\
   if (fdstate & FS_PENDING)\
      if (--npending == 0)\
         fdstate &= ~FS_PENDING

/************************************************************************
*									*
*   fd_reset, fd_unreset						*
*									*
*	Statements.  Set or clear main loop monopolization for a	*
*	"client zero" dialog.						*
*									*
************************************************************************/
#define fd_reset(fd)\
   fdstate |= FS_RESET;\
   fdchill = 1;\
   if (fd < 0)\
      bcopy((char *)&serverfds, (char *)&resetfds, sizeof(fd_set));\
   else {\
      FD_SET(fd, &resetfds);\
   }

#define fd_unreset()\
   if (fdstate & FS_RESET) {\
      fdstate &= ~FS_RESET;\
      FD_ZERO(&resetfds);\
   }\
   else {\
      warn("fd_unreset: not in reset!\n");\
      abort();\
   }

#define fd_inreset()	(fdstate & FS_RESET)

/************************************************************************
*									*
*   fd_chill								*
*									*
*	Expression.  Should main loop chill out?  True if a connection	*
*	needs to monopolize input for awhile.				*
*									*
*	Resets each time it is tested (that is, it breaks out of the	*
*	main loop exactly once).					*
*									*
************************************************************************/
#define fd_chill()	(fdchill ? (fdchill=0, 1) : 0)
