/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   event.c								*
*									*
*	Send events to clients.						*
*									*
************************************************************************/
#define NEED_EVENTS
#include <X11/Xproto.h>
#include <X11/X.h>

#include "xmx.h"
#include "df.h"
#include "incl/event.pvt.h"

static buffer_t *bp;

/************************************************************************
*									*
*   event_init								*
*									*
************************************************************************/
void
event_init
   VOID
{
   bp = buf_new(B_STATIC);
}

/************************************************************************
*									*
*   event_send_event							*
*									*
************************************************************************/
int
event_send_event
   AL((cp, p))
   DB client_t *cp
   DD xSendEventReq *p
   DE
{
   register window_t *wp;
   register chunk_t *chp;
   register xEvent *ep;

   switch (p->destination) {
      case PointerWindow:
         wp = ptrwp;
         break;
      case InputFocus:
         if (focuswp) {
            for (wp=ptrwp; wp->level > focuswp->level; wp=wp->parent);
            wp = wp == focuswp ? ptrwp : focuswp;
         }
         else
            return 0;
         break;
      default:
         if ((wp = (window_t *)hash_data(vmap, p->destination)) == 0) {
            proto_Error(cp, BadWindow, p->destination, 0, X_SendEvent);
            return -1;
         }
         break;
   }
   buf_put(bp, (char *)&p->event, sz_xEvent);
   es_mark(bp);
   chp = buf_split(bp, 0);

   ep = (xEvent *)buf_data(chp);
   ep->u.u.type |= 0x80;	/* flip send event bit */
   if (p->eventMask == 0 && wp->dwb.res.client)
      queue_add(wp->dwb.res.client->qp, chp);
   else
      event_send(chp, 1, wp, p->eventMask, p->propagate);
   buf_clear(chp);

   return 0;
}

/************************************************************************
*									*
*   event_send								*
*									*
*	Send the event(s) to the clients who have selected for them	*
*	on the given window.						*
*									*
*	If more than one event is in the buffer, all must be of the	*
*	same type (eg Expose events).					*
*									*
************************************************************************/
int
event_send
   AL((chp, nevents, wp, mask, propagate))
   DB chunk_t *chp
   DD int nevents
   DD window_t *wp
   DD mask_t mask
   DD int propagate	/* caution: always honored */
   DE
{
   register mask_t j;
   register int i, n;
   client_t *cp;
   client_t *grabcp;
   register xEvent *p;
   register u8_t etype;

   p = (xEvent *)buf_data(chp);
   switch (etype = p->u.u.type & 0x7f) {
      case KeyPress:
      case KeyRelease:
      case ButtonPress:
      case ButtonRelease:
      case MotionNotify:
      case EnterNotify:
      case LeaveNotify:
         grabcp = inp_ptr_grab_client();
         break;
      default:
         grabcp = 0;
         break;
   }
   if ((wp->xmask->mask & mask) == 0) {

      if (propagate == 0)
         return 0;
      /*
      **  climb the window heirarchy in search of an event window
      */
      for (;;)
         if ((wp->atts[IWDontPropagate] & mask) == 0 && wp->parent) {
            p->u.keyButtonPointer.event = wp->parent->cid;
            p->u.keyButtonPointer.eventX += XOFFSET(wp);
            p->u.keyButtonPointer.eventY += YOFFSET(wp);
            if (wp == ptrwp)
               p->u.keyButtonPointer.child = wp->cid;
            else
               p->u.keyButtonPointer.child = 0;

            wp = wp->parent;

            if (wp->xmask->mask & mask) {
               if (grabcp) {
                  if ((j = imask_get(wp->xmask, (char *)grabcp)) && j & mask)
                     goto grab_mask_found;	/* sue me */

                  /* else keep searching */
               }
               else
                  break;	/* found it - exit loop */
            }
         }
         else
            return 0;
   }
   else if (grabcp) {
      if ((j = imask_get(wp->xmask, (char *)grabcp)) && j & mask) {
         grab_mask_found:	/* jump in from the loop above */

         queue_add(grabcp->qp, chp);
         /*
         **  See comment below.  Note that FocusIn events are not
         **  modified by grabs, and so never show up here.
         */
         if (etype == EnterNotify && j & KeymapStateMask) {
            zb_client_dest(grabcp);
            proto_KeymapNotify(inp_keymap());
         }
         return 1;
      }
      return 0;
   }
   /*
   **  There's no grab affecting this event and wp gets it.  Send it
   **  to all clients that have selected for it.
   */
   n = 0;
   for (	j=imask_first(wp->xmask, mask, (char **)&cp);
		j;
		j=imask_next((char **)&cp)) {
      /*
      **  Off it goes...ta ta!
      */
      queue_add(cp->qp, chp);
      n++;
      switch (etype) {
         case ButtonPress:
            /*
            **  Only one client could have selected for this, so only one
            **  autograb can result.
            */
            if (j & (ButtonPressMask | ButtonReleaseMask))
               inp_autograb(wp, cp, j);
            break;
         case EnterNotify:
         case FocusIn:
            /*
            **  EnterNotify and FocusIn events are always followed by
            **  KeymapNotify events to clients that select for both.  Don't
            **  set the zb dest to an event destination or this will do ugly
            **  things - see the comment for zb_client_queue().
            **
            **  No, there's no need to count this (n++).
            */
            if (j & KeymapStateMask) {
               zb_client_dest(cp);
               proto_KeymapNotify(inp_keymap());
            }
            break;
      }
   }
   return n;
}

/************************************************************************
*									*
*   event_sendto							*
*									*
************************************************************************/
void
event_sendto
   AL((chp, cp))
   DB chunk_t *chp
   DD client_t *cp
   DE
{
   queue_add(cp->qp, chp);
}

/*
**	[07]
*/
void
event_EnterNotify
   AL((wp, childwp, time, rx,ry, x,y, keybutmsk, mode, focus, detail))
   DB window_t *wp
   DD window_t *childwp
   DD timestamp_t time
   DD s16_t rx
   DD s16_t ry
   DD s16_t x
   DD s16_t y
   DD u16_t keybutmsk
   DD u8_t mode
   DD u8_t focus
   DD u8_t detail
   DE
{
   zb_event_dest(wp, EnterWindowMask);
   proto_EnterNotify(	time,
			vscreen.wp->cid,
			wp->cid,
			childwp ? childwp->cid : None,
			rx, ry,
			x, y,
			keybutmsk,
			mode,
			focus,
			detail);
}

/*
**	[08]
*/
void
event_LeaveNotify
   AL((wp, childwp, time, rx,ry, x,y, keybutmsk, mode, focus, detail))
   DB window_t *wp
   DD window_t *childwp
   DD timestamp_t time
   DD s16_t rx
   DD s16_t ry
   DD s16_t x
   DD s16_t y
   DD u16_t keybutmsk
   DD u8_t mode
   DD u8_t focus
   DD u8_t detail
   DE
{
   zb_event_dest(wp, LeaveWindowMask);
   proto_LeaveNotify(	time,
			vscreen.wp->cid,
			wp->cid,
			childwp ? childwp->cid : None,
			rx, ry,
			x, y,
			keybutmsk,
			mode,
			focus,
			detail);
}

/*
**	[09]
*/
void
event_FocusIn
   AL((wp, mode, detail))
   DB window_t *wp
   DD u8_t mode
   DD u8_t detail
   DE
{
   zb_event_dest(wp, FocusChangeMask);
   proto_FocusIn(wp->cid, mode, detail);
}

/*
**	[10]
*/
void
event_FocusOut
   AL((wp, mode, detail))
   DB window_t *wp
   DD u8_t mode
   DD u8_t detail
   DE
{
   zb_event_dest(wp, FocusChangeMask);
   proto_FocusOut(wp->cid, mode, detail);
}

/*
**	[12]
*/
void
event_Expose		/* should support rectangles here... */
   AL((wp))
   DB window_t *wp
   DE
{
   zb_event_dest(wp, ExposureMask);
   proto_Expose(wp->cid, 0, 0, wp->dwb.width, wp->dwb.height, 0);
}

/*
**	[16]
*/
void
event_CreateNotify
   AL((wp))
   DB window_t *wp
   DE
{
   zb_event_dest(wp->parent, SubstructureNotifyMask);
   proto_CreateNotify(	wp->parent->cid,
			wp->cid,
			wp->x,
			wp->y,
			wp->dwb.width,
			wp->dwb.height,
			wp->borderwidth,
			wp->atts[IWOverrideRedirect]);
}

/*
**	[17]
*/
void
event_DestroyNotify
   AL((wp))
   DB window_t *wp
   DE
{
   zb_event_dest(wp, StructureNotifyMask);
   proto_DestroyNotify(wp->cid, wp->cid);

   if (wp->parent) {
      zb_event_dest(wp->parent, SubstructureNotifyMask);
      proto_DestroyNotify(wp->parent->cid, wp->cid);
   }
}

/*
**	[18]
*/
void
event_UnmapNotify
   AL((wp, fromconf))
   DB window_t *wp
   DD u8_t fromconf
   DE
{
   zb_event_dest(wp, StructureNotifyMask);
   proto_UnmapNotify(wp->cid, wp->cid, fromconf);

   if (wp->parent) {
      zb_event_dest(wp->parent, SubstructureNotifyMask);
      proto_UnmapNotify(wp->parent->cid, wp->cid, fromconf);
   }
}

/*
**	[19]
*/
void
event_MapNotify
   AL((wp))
   DB window_t *wp
   DE
{
   zb_event_dest(wp, StructureNotifyMask);
   proto_MapNotify(wp->cid, wp->cid, wp->atts[IWOverrideRedirect]);

   if (wp->parent) {
      zb_event_dest(wp->parent, SubstructureNotifyMask);
      proto_MapNotify(wp->parent->cid, wp->cid, wp->atts[IWOverrideRedirect]);
   }
}

/*
**	[20]
*/
void
event_MapRequest
   AL((cp, wp))
   DB client_t *cp
   DD window_t *wp
   DE
{
   zb_client_dest(cp);
   proto_MapRequest(wp->parent->cid, wp->cid);
}

/*
**	[21]
*/
void
event_ReparentNotify
   AL((oparent, wp))
   DB window_t *oparent
   DD window_t *wp
   DE
{
   zb_event_dest(wp, StructureNotifyMask);
   proto_ReparentNotify(	wp->cid,
				wp->cid,
				wp->parent->cid,
				wp->x,
				wp->y,
				wp->atts[IWOverrideRedirect]);

   zb_event_dest(oparent, SubstructureNotifyMask);
   proto_ReparentNotify(	oparent->cid,
				wp->cid,
				wp->parent->cid,
				wp->x,
				wp->y,
				wp->atts[IWOverrideRedirect]);

   zb_event_dest(wp->parent, SubstructureNotifyMask);
   proto_ReparentNotify(	wp->parent->cid,
				wp->cid,
				wp->parent->cid,
				wp->x,
				wp->y,
				wp->atts[IWOverrideRedirect]);
}

/*
**	[22]
*/
void
event_ConfigureNotify
   AL((wp))
   DB window_t *wp
   DE
{
   zb_event_dest(wp, StructureNotifyMask);
   proto_ConfigureNotify(	wp->cid,
				wp->cid,
				wp->nextsib ? wp->nextsib->cid : 0,
				wp->x, wp->y,
				wp->dwb.width, wp->dwb.height, wp->borderwidth,
				(u8_t)wp->atts[IWOverrideRedirect]);

   if (wp->parent) {
      zb_event_dest(wp->parent, SubstructureNotifyMask);
      proto_ConfigureNotify(	wp->parent->cid,
				wp->cid,
				wp->nextsib ? wp->nextsib->cid : 0,
				wp->x, wp->y,
				wp->dwb.width, wp->dwb.height, wp->borderwidth,
				(u8_t)wp->atts[IWOverrideRedirect]);
   }
}

/*
**	[23]
*/
void
event_ConfigureRequest
   AL((cp, parent, window, sibling, x, y, w, h, bw, stackmode, mask))
   DB client_t *cp
   DD rid_t parent
   DD rid_t window
   DD rid_t sibling
   DD s16_t x
   DD s16_t y
   DD u16_t w
   DD u16_t h
   DD u16_t bw
   DD u8_t stackmode
   DD mask_t mask
   DE
{
   zb_client_dest(cp);
   proto_ConfigureRequest(	parent, window, sibling,
				x, y, w, h, bw,
				stackmode,
				mask);
}

/*
**	[24]
*/
void
event_GravityNotify
   AL((wp))
   DB window_t *wp
   DE
{
   zb_event_dest(wp, StructureNotifyMask);
   proto_GravityNotify(wp->cid, wp->cid, wp->x, wp->y);

   if (wp->parent) {
      zb_event_dest(wp->parent, SubstructureNotifyMask);
      proto_GravityNotify(wp->parent->cid, wp->cid, wp->x, wp->y);
   }
}

/*
**	[25]
*/
void
event_ResizeRequest
   AL((cp, window, w, h))
   DB client_t *cp
   DD rid_t window
   DD u16_t w
   DD u16_t h
   DE
{
   zb_client_dest(cp);
   proto_ResizeRequest(window, w, h);
}

/*
**	[26]
*/
void
event_CirculateNotify
   AL((wp, window, place))
   DB window_t *wp
   DD rid_t window
   DD u8_t place
   DE
{
   register chunk_t *chp;

   zb_event_dest(wp, StructureNotifyMask);
   proto_CirculateNotify(wp->cid, window, place);

   if (wp->parent) {
      zb_event_dest(wp->parent, SubstructureNotifyMask);
      proto_CirculateNotify(wp->parent->cid, window, place);
   }
}

/*
**	[27]
*/
void
event_CirculateRequest
   AL((cp, parent, window, place))
   DB client_t *cp
   DD rid_t parent
   DD rid_t window
   DD u8_t place
   DE
{
   zb_client_dest(cp);
   proto_CirculateRequest(parent, window, place);
}

/*
**	[28]
*/
void
event_PropertyNotify
   AL((wp, atom, time, state))
   DB window_t *wp
   DD atom_t atom
   DD timestamp_t time
   DD u8_t state
   DE
{
   zb_event_dest(wp, PropertyChangeMask);
   proto_PropertyNotify(wp->cid, atom, time, state);
}

/*
**	[29]
*/
void
event_SelectionClear
   AL((cp, time, window, selection))
   DB client_t *cp
   DD timestamp_t time
   DD rid_t window
   DD atom_t selection
   DE
{
   zb_client_dest(cp);
   proto_SelectionClear(time, window, selection);
}

/*
**	[30]
*/
void
event_SelectionRequest
   AL((cp, time, owner, requestor, sel, targ, prop))
   DB client_t *cp
   DD timestamp_t time
   DD rid_t owner
   DD rid_t requestor
   DD atom_t sel
   DD atom_t targ
   DD atom_t prop
   DE
{
   zb_client_dest(cp);
   proto_SelectionRequest(time, owner, requestor, sel, targ, prop);
}

/*
**	[31]
*/
void
event_SelectionNotify
   AL((cp, time, requestor, sel, targ, prop))
   DB client_t *cp
   DD timestamp_t time
   DD rid_t requestor
   DD atom_t sel
   DD atom_t targ
   DD atom_t prop
   DE
{
   zb_client_dest(cp);
   proto_SelectionNotify(time, requestor, sel, targ, prop);
}

/*
**	[32]
*/
void
event_ColormapNotify
   AL((wp, new, state))
   DB window_t *wp
   DD u8_t new
   DD u8_t state
   DE
{
   zb_event_dest(wp, ColormapChangeMask);
   proto_ColormapNotify(wp->cid, wp->mp->cid, new, state);
}

/*
**	[34]
*/
void
event_MappingNotify
   AL((request, first, count))
   DB u8_t request
   DD u8_t first
   DD u8_t count
   DE
{
   zb_client_dest(0);		/* MappingNotify goes to everyone */
   proto_MappingNotify(request, first, count);
}
