/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   cmap.c								*
*									*
*	Model colormaps.						*
*									*
************************************************************************/
#include <stdlib.h>
#include <X11/X.h>
#include <X11/Xproto.h>

#include "xmx.h"
typedef struct _cell_t cell_t;
#include "incl/cmap.pvt.h"

typedef struct _cell_t {
	u16_t	red;
	u16_t	green;
	u16_t	blue;
	u8_t	status;
	u8_t	refs;
	u8_t	clirefs[NOCLIENTS];
}cell_xxx;	/* cell_t defined above */

typedef struct _cmap_t {
	u8_t		class;
	u8_t		alloc;
	union {
		struct {
			u16_t		n;
			cell_t *	rgb;
		}pseudo;
		struct {
			cell_t		*red, *green, *blue;
			mask_t		rmask, gmask, bmask;
			u16_t		rn, gn, bn;
			u8_t		rshift, gshift, bshift;
		}direct;
		struct {
			u16_t		rdiv, gdiv, bdiv;
			mask_t		rmask, gmask, bmask;
			u8_t		rshift, gshift, bshift;
		}true;
	}u;
}cmap_xxx;	/* cmap_t defined in xmx.h */

/************************************************************************
*									*
*   cmap_create								*
*									*
************************************************************************/
cmap_t *
cmap_create
   AL((class, ncols, rmask, gmask, bmask, alloc, nhide))
   DB u8_t class
   DD u16_t ncols
   DD mask_t rmask
   DD mask_t gmask
   DD mask_t bmask
   DD u8_t alloc
   DD int nhide
   DE
{
   register int i;
   register mask_t m;
   cmap_t *cmp;

   if (MALLOC(cmp, cmap_t *, sizeof(cmap_t)))
      return 0;

   cmp->class = class;
   cmp->alloc = alloc;

   switch (cmp->class) {
      case GrayScale:
      case PseudoColor:
      case StaticGray:
      case StaticColor:
         cmp->u.pseudo.n = ncols;
         if (CALLOC(cmp->u.pseudo.rgb,cell_t *,sizeof(cell_t),cmp->u.pseudo.n)){
            cmap_free(cmp);
            return 0;
         }
         break;
      case DirectColor:
         cmp->u.direct.rmask = rmask;
         cmp->u.direct.gmask = gmask;
         cmp->u.direct.bmask = bmask;
         NUMONES(rmask, i);
         cmp->u.direct.rn = 1 << i;
         NUMONES(gmask, i);
         cmp->u.direct.gn = 1 << i;
         NUMONES(bmask, i);
         cmp->u.direct.bn = 1 << i;
         if (	CALLOC(cmp->u.direct.red, cell_t *, cmp->u.direct.rn,
							sizeof(cell_t)) ||
         	CALLOC(cmp->u.direct.green, cell_t *, cmp->u.direct.gn,
							sizeof(cell_t)) ||
         	CALLOC(cmp->u.direct.blue, cell_t *, cmp->u.direct.bn,
							sizeof(cell_t))) {
            cmap_free(cmp);
            return 0;
         }
         for (cmp->u.direct.rshift=0, m=rmask; (m & 1) == 0; m>>=1)
            cmp->u.direct.rshift++;

         for (cmp->u.direct.gshift=0, m=gmask; (m & 1) == 0; m>>=1)
            cmp->u.direct.gshift++;

         for (cmp->u.direct.bshift=0, m=bmask; (m & 1) == 0; m>>=1)
            cmp->u.direct.bshift++;
         break;
      case TrueColor:
         cmp->u.true.rmask = rmask;
         cmp->u.true.gmask = gmask;
         cmp->u.true.bmask = bmask;

         cmp->u.true.rshift = 0;
         for (m=rmask; (m & 1) == 0; m>>=1)
            cmp->u.true.rshift++;
         cmp->u.true.rdiv = MAXCOLVAL / m;

         cmp->u.true.gshift = 0;
         for (m=gmask; (m & 1) == 0; m>>=1)
            cmp->u.true.gshift++;
         cmp->u.true.gdiv = MAXCOLVAL / m;

         cmp->u.true.bshift = 0;
         for (m=bmask; (m & 1) == 0; m>>=1)
            cmp->u.true.bshift++;
         cmp->u.true.bdiv = MAXCOLVAL / m;
         break;
   }
   hide_cells(cmp, nhide);

   return cmp;
}

/************************************************************************
*									*
*   cmap_unhide_cells							*
*									*
************************************************************************/
int
cmap_unhide_cells
   AL((cmp, n))
   DB cmap_t *cmp
   DD int n
   DE
{
   register int t;

   t = unhide_cells(cmp, n);

   return (t < n);
}

/************************************************************************
*									*
*   cmap_copy_and_free							*
*									*
************************************************************************/
cmap_t *
cmap_copy_and_free
   AL((ocmp, clinum))
   DB cmap_t *ocmp
   DD u8_t clinum
   DE
{
   cmap_t *ncmp;

   switch (ocmp->class) {
      case GrayScale:
      case PseudoColor:
      case StaticGray:
      case StaticColor:
         if (ncmp = cmap_create(ocmp->class,
				ocmp->u.pseudo.n,
				0, 0, 0,
				ocmp->alloc, 0))
            copy_free(	ocmp->u.pseudo.rgb,
			ncmp->u.pseudo.rgb,
			ocmp->u.pseudo.n,
			clinum);
         break;
      case DirectColor:
         if (ncmp = cmap_create(ocmp->class, 0,
				ocmp->u.direct.rmask,
				ocmp->u.direct.gmask,
				ocmp->u.direct.bmask,
				ocmp->alloc, 0)) {
            copy_free(	ocmp->u.direct.red,
			ncmp->u.direct.red,
			ocmp->u.direct.rn,
			clinum);
            copy_free(	ocmp->u.direct.green,
			ncmp->u.direct.green,
			ocmp->u.direct.gn,
			clinum);
            copy_free(	ocmp->u.direct.blue,
			ncmp->u.direct.blue,
			ocmp->u.direct.bn,
			clinum);
         }
         break;
      case TrueColor:
         ncmp = cmap_create(	ocmp->class, 0,
				ocmp->u.true.rmask,
				ocmp->u.true.gmask,
				ocmp->u.true.bmask,
				ocmp->alloc, 0);
         break;
      default:
         warn("cmap_copy_and_free: bad class (this can't happen)\n");
         return 0;
   }
   return ncmp;
}

/************************************************************************
*									*
*   cmap_alloc_white_black						*
*									*
*	Sets and allocates the white and black pixels in a new		*
*	colormap.							*
*									*
*	Note: this routine does not check if a particular cell was	*
*	previously hidden or allocated.					*
*									*
*	That leads to the following very subtle behavior: if the user	*
*	requests n cells to be "hidden" - reserved for other clients	*
*	to use - that number may or may not include the default		*
*	white and black pixels.  If it does not, the user may end up	*
*	with fewer allocable cells than expected.  C'est la vie.	*
*									*
************************************************************************/
void
cmap_alloc_white_black
   AL((cmp, whitep, blackp))
   DB cmap_t *cmp
   DD pixel_t *whitep
   DD pixel_t *blackp
   DE
{
   register pixel_t rpxl, gpxl, bpxl;

   switch (cmp->class) {
      case GrayScale:
         *blackp = 0;
         *whitep = 1;
         cmp->u.pseudo.rgb[*whitep].status = CS_READ;
         cmp->u.pseudo.rgb[*whitep].red = MAXCOLVAL;
         cmp->u.pseudo.rgb[*whitep].refs++;
         cmp->u.pseudo.rgb[*whitep].clirefs[0]++;

         cmp->u.pseudo.rgb[*blackp].status = CS_READ;
         cmp->u.pseudo.rgb[*blackp].red = 0;
         cmp->u.pseudo.rgb[*blackp].refs++;
         cmp->u.pseudo.rgb[*blackp].clirefs[0]++;
         break;
      case PseudoColor:
         *blackp = 0;
         *whitep = 1;
         cmp->u.pseudo.rgb[*whitep].status = CS_READ;
         cmp->u.pseudo.rgb[*whitep].red = MAXCOLVAL;
         cmp->u.pseudo.rgb[*whitep].green = MAXCOLVAL;
         cmp->u.pseudo.rgb[*whitep].blue = MAXCOLVAL;
         cmp->u.pseudo.rgb[*whitep].refs++;
         cmp->u.pseudo.rgb[*whitep].clirefs[0]++;

         cmp->u.pseudo.rgb[*blackp].status = CS_READ;
         cmp->u.pseudo.rgb[*blackp].red = 0;
         cmp->u.pseudo.rgb[*blackp].green = 0;
         cmp->u.pseudo.rgb[*blackp].blue = 0;
         cmp->u.pseudo.rgb[*blackp].refs++;
         cmp->u.pseudo.rgb[*blackp].clirefs[0]++;
         break;
      case DirectColor:
         *whitep =	(1 << cmp->u.direct.rshift) |
			(1 << cmp->u.direct.gshift) |
			(1 << cmp->u.direct.bshift);
         rpxl = (*whitep & cmp->u.direct.rmask) >> cmp->u.direct.rshift;
         gpxl = (*whitep & cmp->u.direct.gmask) >> cmp->u.direct.gshift;
         bpxl = (*whitep & cmp->u.direct.bmask) >> cmp->u.direct.bshift;

         cmp->u.direct.red[rpxl].status = CS_READ;
         cmp->u.direct.red[rpxl].red = MAXCOLVAL;
         cmp->u.direct.red[rpxl].refs++;
         cmp->u.direct.red[rpxl].clirefs[0]++;

         cmp->u.direct.green[gpxl].status = CS_READ;
         cmp->u.direct.green[gpxl].red = MAXCOLVAL;
         cmp->u.direct.green[gpxl].refs++;
         cmp->u.direct.green[gpxl].clirefs[0]++;

         cmp->u.direct.blue[bpxl].status = CS_READ;
         cmp->u.direct.blue[bpxl].red = MAXCOLVAL;
         cmp->u.direct.blue[bpxl].refs++;
         cmp->u.direct.blue[bpxl].clirefs[0]++;

         *blackp = 0;
         rpxl = (*blackp & cmp->u.direct.rmask) >> cmp->u.direct.rshift;
         gpxl = (*blackp & cmp->u.direct.gmask) >> cmp->u.direct.gshift;
         bpxl = (*blackp & cmp->u.direct.bmask) >> cmp->u.direct.bshift;

         cmp->u.direct.red[rpxl].status = CS_READ;
         cmp->u.direct.red[rpxl].red = 0;
         cmp->u.direct.red[rpxl].refs++;
         cmp->u.direct.red[rpxl].clirefs[0]++;

         cmp->u.direct.green[gpxl].status = CS_READ;
         cmp->u.direct.green[gpxl].red = 0;
         cmp->u.direct.green[gpxl].refs++;
         cmp->u.direct.green[gpxl].clirefs[0]++;

         cmp->u.direct.blue[bpxl].status = CS_READ;
         cmp->u.direct.blue[bpxl].red = 0;
         cmp->u.direct.blue[bpxl].refs++;
         cmp->u.direct.blue[bpxl].clirefs[0]++;
         break;
      case StaticGray:
      case StaticColor:
         *blackp = 0;
         *whitep = 1;
         cmp->u.pseudo.rgb[*whitep].status = CS_READ;
         cmp->u.pseudo.rgb[*whitep].refs++;
         cmp->u.pseudo.rgb[*whitep].clirefs[0]++;
         cmp->u.pseudo.rgb[*whitep].red = MAXCOLVAL;
         cmp->u.pseudo.rgb[*whitep].green = MAXCOLVAL;
         cmp->u.pseudo.rgb[*whitep].blue = MAXCOLVAL;

         cmp->u.pseudo.rgb[*blackp].status = CS_READ;
         cmp->u.pseudo.rgb[*blackp].refs++;
         cmp->u.pseudo.rgb[*blackp].clirefs[0]++;
         cmp->u.pseudo.rgb[*blackp].red = 0;
         cmp->u.pseudo.rgb[*blackp].green = 0;
         cmp->u.pseudo.rgb[*blackp].blue = 0;
         break;
      case TrueColor:
         *blackp = 0;
         *whitep = cmp->u.true.rmask | cmp->u.true.gmask | cmp->u.true.bmask;
         break;
   }
}

/************************************************************************
*									*
*   cmap_alloc_static							*
*									*
*	Initialize a pixel in a static colormap.			*
*									*
************************************************************************/
int
cmap_alloc_static
   AL((cmp, red, green, blue, pixel))
   DB cmap_t *cmp
   DD u16_t red
   DD u16_t green
   DD u16_t blue
   DD pixel_t pixel
   DE
{
   switch (cmp->class) {
      case GrayScale:
      case PseudoColor:
      case DirectColor:
         warn("cmap_alloc_static: wrong class - this can't happen!\n");
         break;
      case StaticGray:
      case StaticColor:
         if (cmp->u.pseudo.rgb[pixel].status != CS_FREE) {
            warn("cmap_alloc_static: pixel %d already allocated\n", pixel);
            return -1;
         }
         cmp->u.pseudo.rgb[pixel].status = CS_READ;
         cmp->u.pseudo.rgb[pixel].refs++;
         cmp->u.pseudo.rgb[pixel].clirefs[0]++;
         cmp->u.pseudo.rgb[pixel].red = red;
         cmp->u.pseudo.rgb[pixel].green = green;
         cmp->u.pseudo.rgb[pixel].blue = blue;
         break;
      case TrueColor:
         break;
   }
   return 0;
}

/************************************************************************
*									*
*   cmap_alloc_read							*
*									*
************************************************************************/
int
cmap_alloc_read
   AL((cmp, clinum, r, g, b, pp, rp, gp, bp, new))
   DB cmap_t *cmp
   DD u8_t clinum		/* client number */
   DD u16_t r
   DD u16_t g
   DD u16_t b
   DD pixel_t *pp
   DD u16_t *rp
   DD u16_t *gp
   DD u16_t *bp
   DD int *new
   DE
{
   pixel_t i, j, k;
   register u8_t flag;

   *new = 0;
   switch (cmp->class) {
      case GrayScale:
         if (find_color1(cmp->u.pseudo.rgb, cmp->u.pseudo.n, r, &i, 0))
            return -1;
         if (cmp->u.pseudo.rgb[i].status == CS_FREE) {
            cmp->u.pseudo.rgb[i].status = CS_READ;
            cmp->u.pseudo.rgb[i].red = r;
            *new = 1;
         }
         cmp->u.pseudo.rgb[i].refs++;
         cmp->u.pseudo.rgb[i].clirefs[clinum]++;
         *pp = i;
         *rp = *gp = *bp = cmp->u.pseudo.rgb[i].red;
         break;
      case PseudoColor:
         if (find_color3(cmp->u.pseudo.rgb,cmp->u.pseudo.n, r,g,b, &i, 0))
            return -1;
         if (cmp->u.pseudo.rgb[i].status == CS_FREE) {
            cmp->u.pseudo.rgb[i].status = CS_READ;
            cmp->u.pseudo.rgb[i].red = r;
            cmp->u.pseudo.rgb[i].green = g;
            cmp->u.pseudo.rgb[i].blue = b;
            *new = 1;
         }
         cmp->u.pseudo.rgb[i].refs++;
         cmp->u.pseudo.rgb[i].clirefs[clinum]++;
         *pp = i;
         *rp = cmp->u.pseudo.rgb[i].red;
         *gp = cmp->u.pseudo.rgb[i].green;
         *bp = cmp->u.pseudo.rgb[i].blue;
         break;
      case DirectColor:
         if (find_color1(cmp->u.direct.red, cmp->u.direct.rn, r, &i, 0))
            return -1;
         if (find_color1(cmp->u.direct.green, cmp->u.direct.gn, g, &j, 0))
            return -1;
         if (find_color1(cmp->u.direct.blue, cmp->u.direct.bn, b, &k, 0))
            return -1;

         flag = 0;
         if (cmp->u.direct.red[i].status == CS_FREE) {		/* RED */
            cmp->u.direct.red[i].status = CS_READ;
            cmp->u.direct.red[i].red = r;
            flag |= DoRed;
         }
         cmp->u.direct.red[i].refs++;
         cmp->u.direct.red[i].clirefs[clinum]++;

         if (cmp->u.direct.green[j].status == CS_FREE) {	/* GREEN */
            cmp->u.direct.green[j].status = CS_READ;
            cmp->u.direct.green[j].red = g;	/* only red channel is used */
            flag |= DoGreen;
         }
         cmp->u.direct.green[j].refs++;
         cmp->u.direct.green[j].clirefs[clinum]++;

         if (cmp->u.direct.blue[k].status == CS_FREE) {		/* BLUE */
            cmp->u.direct.blue[k].status = CS_READ;
            cmp->u.direct.blue[k].red = b;	/* only red channel is used */
            flag |= DoBlue;
         }
         cmp->u.direct.blue[k].refs++;
         cmp->u.direct.blue[k].clirefs[clinum]++;
         *pp =	i << cmp->u.direct.rshift |
		j << cmp->u.direct.gshift |
		k << cmp->u.direct.bshift;

         if (flag)
            *new = 1;

         *rp = cmp->u.direct.red[i].red;
         *gp = cmp->u.direct.green[j].red;
         *bp = cmp->u.direct.blue[k].red;
         break;
      case StaticGray:
      case StaticColor:
         *pp = (r | g | b) ? 1 : 0;	/* HACK-O-RAMA (TODO) */
         *rp = *gp = *bp = *pp ? MAXCOLVAL : 0;
         break;
      case TrueColor:
         i = r / cmp->u.true.rdiv;	/* we assume a simple linear    */
         j = g / cmp->u.true.gdiv;	/* relationship between channel */
         k = b / cmp->u.true.bdiv;	/* values and pixel sub-masks.  */

         *pp =	i << cmp->u.true.rshift |
		j << cmp->u.true.gshift |
		k << cmp->u.true.bshift;
         *rp = i * cmp->u.true.rdiv;
         *gp = j * cmp->u.true.gdiv;
         *bp = k * cmp->u.true.bdiv;
         break;
   }
   return 0;
}

/************************************************************************
*									*
*   cmap_lookup								*
*									*
*	This isn't entirely correct.  We don't really know what the	*
*	hardware supports, but we can make a pretty good guess.		*
*	Basically, if the given colormap is static, calculate the	*
*	closest match, otherwise return the exact values.		*
*									*
************************************************************************/
void
cmap_lookup
   AL((cmp, r, g, b, rp, gp, bp))
   DB cmap_t *cmp
   DD u16_t r
   DD u16_t g
   DD u16_t b
   DD u16_t *rp
   DD u16_t *gp
   DD u16_t *bp
   DE
{
   pixel_t i, j, k;

   switch (cmp->class) {
      case GrayScale:
      case PseudoColor:
      case DirectColor:
         *rp = r;
         *gp = g;
         *bp = b;
         break;
      case StaticGray:
      case StaticColor:
         *rp = *gp = *bp = (r | g | b) ? MAXCOLVAL : 0;		/* not */
         break;
      case TrueColor:
         i = r / cmp->u.true.rdiv;	/* we assume a simple linear	*/
         j = g / cmp->u.true.gdiv;	/* relationship between channel	*/
         k = b / cmp->u.true.bdiv;	/* values and pixel sub-masks.	*/

         *rp = i * cmp->u.true.rdiv;
         *gp = j * cmp->u.true.gdiv;
         *bp = k * cmp->u.true.bdiv;
         break;
   }
}

/************************************************************************
*									*
*   cmap_free								*
*									*
************************************************************************/
void
cmap_free
   AL((cmp))
   DB cmap_t *cmp
   DE
{
   switch (cmp->class) {
      case GrayScale:
      case PseudoColor:
         if (cmp->u.pseudo.rgb)
            free(cmp->u.pseudo.rgb);
         break;
      case DirectColor:
         if (cmp->u.direct.red)
            free(cmp->u.direct.red);
         if (cmp->u.direct.green)
            free(cmp->u.direct.green);
         if (cmp->u.direct.blue)
            free(cmp->u.direct.blue);
         break;
   }
   free(cmp);
}

/************************************************************************
*									*
*   cmap_alloc_cells							*
*									*
************************************************************************/
int
cmap_alloc_cells
   AL((cmp, clinum, ncols, nplanes, contig, ppp, rmask, gmask, bmask))
   DB cmap_t *cmp
   DD u8_t clinum
   DD int ncols
   DD int nplanes
   DD int contig
   DD pixel_t **ppp
   DD mask_t *rmask
   DD mask_t *gmask
   DD mask_t *bmask
   DE
{
   register int i;
   register int pixallocd = 0;
   static int npix, ntpix;
   static pixel_t *pp, *tpp;

   if (cmp->alloc)
      return -1;

   if (ncols > npix) {
      if (pp)
         free(pp);
      npix = RUP(ncols, 256);
      if (MALLOC(pp, pixel_t *, npix * sizeof(pixel_t))) {
         npix = 0;
         return -1;
      }
   }
   switch (cmp->class) {
      case GrayScale:
      case PseudoColor:
         *gmask = *bmask = 0;
         if (alloc_cells(cmp->u.pseudo.rgb, cmp->u.pseudo.n, clinum,
					contig, ncols, nplanes, pp, rmask))
            break;
         *ppp = pp;
         return 0;
      case DirectColor:
         *rmask = *gmask = *bmask = 0;
         if (ncols > ntpix) {
            if (tpp)
               free(tpp);
            ntpix = RUP(ncols, 256);
            if (MALLOC(tpp, pixel_t *, ntpix * sizeof(pixel_t))) {
               ntpix = 0;
               return -1;
            }
         }
         if (alloc_cells(cmp->u.direct.red, cmp->u.direct.rn,
				clinum, contig, ncols, nplanes, tpp, rmask))
            break;
         for (i=0; i<ncols; i++)
            pp[i] = tpp[i] << cmp->u.direct.rshift;
         *rmask <<= cmp->u.direct.rshift;
         pixallocd = 1;

         if (alloc_cells(cmp->u.direct.green, cmp->u.direct.gn,
				clinum, contig, ncols, nplanes, tpp, gmask))
            break;
         for (i=0; i<ncols; i++)
            pp[i] |= tpp[i] << cmp->u.direct.gshift;
         *gmask <<= cmp->u.direct.gshift;

         if (alloc_cells(cmp->u.direct.blue, cmp->u.direct.bn,
				clinum, contig, ncols, nplanes, tpp, bmask))
            break;
         for (i=0; i<ncols; i++)
            pp[i] |= tpp[i] << cmp->u.direct.bshift;
         *bmask <<= cmp->u.direct.bshift;

         *ppp = pp;
         return 0;
   }
   if (pixallocd)		/* this could be an error TODO */
      cmap_free_cells(cmp, clinum, ncols, pp, *rmask | *gmask | *bmask);

   return -1;
}

/************************************************************************
*									*
*   cmap_alloc_planes							*
*									*
************************************************************************/
int
cmap_alloc_planes
   AL((cmp, clinum, contig, ncols, nreds, ngreens, nblues,
						ppp, rmask, gmask, bmask))
   DB cmap_t *cmp
   DD u8_t clinum
   DD int contig
   DD int ncols
   DD int nreds
   DD int ngreens
   DD int nblues
   DD pixel_t **ppp
   DD mask_t *rmask
   DD mask_t *gmask
   DD mask_t *bmask
   DE
{
   register int i;
   register mask_t tm;
   int nmasks;
   static int npix, ntpix;
   static pixel_t *pp, *tpp;
   static mask_t mask;

   if (cmp->alloc)
      return -1;

   if (ncols > npix) {
      if (pp)
         free(pp);
      npix = RUP(ncols, 256);
      if (MALLOC(pp, pixel_t *, npix * sizeof(pixel_t))) {
         npix = 0;
         return -1;
      }
   }
   switch (cmp->class) {
      case GrayScale:
      case PseudoColor:
         nmasks = nreds + ngreens + nblues;
         alloc_cells(cmp->u.pseudo.rgb, cmp->u.pseudo.n,
				clinum, contig, ncols, nmasks, pp, &mask);
         for (tm=1; (tm & mask) == 0; tm<<=1);
         for (; nreds--; tm<<=1)
            *rmask |= tm;
         for (; ngreens--; tm<<=1)
            *gmask |= tm;
         for (; nblues--; tm<<=1)
            *bmask |= tm;
         return 0;
      case DirectColor:
         *rmask = *gmask = *bmask = 0;	/* for error processing... */
         nmasks = nreds + ngreens + nblues;
         if (ncols > ntpix) {
            if (tpp)
               free(tpp);
            ntpix = RUP(ncols, 256);
            if (MALLOC(tpp, pixel_t *, ntpix * sizeof(pixel_t))) {
               ntpix = 0;
               return -1;
            }
         }
         if (alloc_cells(cmp->u.direct.red, cmp->u.direct.rn,
				clinum, contig, ncols, nreds, tpp, rmask))
            break;
         for (i=0; i<ncols; i++)
            pp[i] = tpp[i] << cmp->u.direct.rshift;
         *rmask <<= cmp->u.direct.rshift;

         if (alloc_cells(cmp->u.direct.green, cmp->u.direct.gn,
				clinum, contig, ncols, ngreens, tpp, gmask))
            break;
         for (i=0; i<ncols; i++)
            pp[i] |= tpp[i] << cmp->u.direct.gshift;
         *gmask <<= cmp->u.direct.gshift;

         if (alloc_cells(cmp->u.direct.blue, cmp->u.direct.bn,
				clinum, contig, ncols, nblues, tpp, bmask))
            break;
         for (i=0; i<ncols; i++)
            pp[i] |= tpp[i] << cmp->u.direct.bshift;
         *bmask <<= cmp->u.direct.bshift;

         *ppp = pp;
         return 0;
   }
				/* this could be an error TODO */
   cmap_free_cells(cmp, clinum, ncols, pp, *rmask | *gmask | *bmask);
   return -1;
}

/************************************************************************
*									*
*   cmap_store								*
*									*
************************************************************************/
int
cmap_store
   AL((cmp, clinum, pixel, red, green, blue, rflag, gflag, bflag))
   DB cmap_t *cmp
   DD u8_t clinum
   DD pixel_t pixel
   DD u16_t red
   DD u16_t green
   DD u16_t blue
   DD u8_t rflag
   DD u8_t gflag
   DD u8_t bflag
   DE
{
   register pixel_t rpxl, gpxl, bpxl;

   switch (cmp->class) {
      case GrayScale:
         if (	cmp->alloc == 0 &&
		(cmp->u.pseudo.rgb[pixel].status != CS_WRITE ||
		cmp->u.pseudo.rgb[pixel].clirefs[clinum] == 0))
            return -1;
         cmp->u.pseudo.rgb[pixel].red = red;
         cmp->u.pseudo.rgb[pixel].green = red;
         cmp->u.pseudo.rgb[pixel].blue = red;
         break;
      case PseudoColor:
         if (	cmp->alloc == 0 &&
		(cmp->u.pseudo.rgb[pixel].status != CS_WRITE ||
		cmp->u.pseudo.rgb[pixel].clirefs[clinum] == 0))
            return -1;
         if (rflag) cmp->u.pseudo.rgb[pixel].red = red;
         if (gflag) cmp->u.pseudo.rgb[pixel].green = green;
         if (bflag) cmp->u.pseudo.rgb[pixel].blue = blue;
         break;
      case DirectColor:
         rpxl = (pixel & cmp->u.direct.rmask) >> cmp->u.direct.rshift;
         gpxl = (pixel & cmp->u.direct.gmask) >> cmp->u.direct.gshift;
         bpxl = (pixel & cmp->u.direct.bmask) >> cmp->u.direct.bshift;
         if (cmp->alloc == 0 &&
		(rflag == 0 ||
			cmp->u.direct.red[rpxl].status != CS_WRITE ||
                        cmp->u.direct.red[rpxl].clirefs[clinum] == 0) &&
		(gflag == 0 ||
			cmp->u.direct.green[gpxl].status != CS_WRITE ||
                        cmp->u.direct.green[gpxl].clirefs[clinum] == 0) &&
		(bflag == 0 ||
			cmp->u.direct.blue[bpxl].status != CS_WRITE ||
                        cmp->u.direct.blue[bpxl].clirefs[clinum] == 0))
            return -1;

         if (rflag) cmp->u.direct.red[rpxl].red = red;
         if (gflag) cmp->u.direct.green[gpxl].red = green;
         if (bflag) cmp->u.direct.blue[bpxl].red = blue;
         break;
   }
   return 0;
}

/************************************************************************
*									*
*   cmap_free_cells							*
*									*
************************************************************************/
int
cmap_free_cells
   AL((cmp, clinum, ncols, pixels, mask))
   DB cmap_t *cmp
   DD u8_t clinum
   DD int ncols
   DD pixel_t *pixels
   DD mask_t mask
   DE
{
   register int i;
   register pixel_t pixel;
   register mask_t tm, rmsk, gmsk, bmsk;

   switch (cmp->class) {
      case GrayScale:
      case PseudoColor:
         for (i=0; i<ncols; i++)	/* check'em all first */
            for (tm=0; tm <= mask; tm = tm ? tm<<1 : 1)
               if (tm == (tm & mask))
                  if (cmp->u.pseudo.rgb[pixels[i]|tm].clirefs[clinum] == 0)
                     return -1;
         for (i=0; i<ncols; i++)
            for (tm=0; tm <= mask; tm = tm ? tm<<1 : 1)
               if (tm == (tm & mask)) {
                  cmp->u.pseudo.rgb[pixels[i]].clirefs[clinum]--;
                  if (--cmp->u.pseudo.rgb[pixels[i]].refs == 0)
                     cmp->u.pseudo.rgb[pixels[i]].status = CS_FREE;
               }
         break;
      case DirectColor:
         rmsk = mask & cmp->u.direct.rmask;
         gmsk = mask & cmp->u.direct.gmask;
         bmsk = mask & cmp->u.direct.bmask;
         for (i=0; i<ncols; i++)
            for (tm=0; tm <= mask; tm = tm ? tm<<1 : 1) {
               if (tm == (tm & rmsk)) {
                  pixel = ((pixels[i]|tm) & cmp->u.direct.rmask) >>
							cmp->u.direct.rshift;
                  if (cmp->u.direct.red[pixel].clirefs[clinum] == 0)
                     return -1;
               }
               if (tm == (tm & gmsk)) {
                  pixel = ((pixels[i]|tm) & cmp->u.direct.gmask) >>
							cmp->u.direct.gshift;
                  if (cmp->u.direct.green[pixel].clirefs[clinum] == 0)
                     return -1;
               }
               if (tm == (tm & bmsk)) {
                  pixel = ((pixels[i]|tm) & cmp->u.direct.bmask) >>
							cmp->u.direct.bshift;
                  if (cmp->u.direct.blue[pixel].clirefs[clinum] == 0)
                     return -1;
               }
            }
         for (i=0; i<ncols; i++)
            for (tm=0; tm <= mask; tm = tm ? tm<<1 : 1) {
               if (tm == (tm & rmsk)) {
                  pixel = ((pixels[i]|tm) & cmp->u.direct.rmask) >>
							cmp->u.direct.rshift;
                  cmp->u.direct.red[pixel].clirefs[clinum]--;
                  if (--cmp->u.direct.red[pixel].refs == 0)
                     cmp->u.direct.red[pixel].status = CS_FREE;
               }
               if (tm == (tm & gmsk)) {
                  pixel = ((pixels[i]|tm) & cmp->u.direct.gmask) >>
							cmp->u.direct.gshift;
                  cmp->u.direct.green[pixel].clirefs[clinum]--;
                  if (--cmp->u.direct.green[pixel].refs == 0)
                     cmp->u.direct.green[pixel].status = CS_FREE;
               }
               if (tm == (tm & bmsk)) {
                  pixel = ((pixels[i]|tm) & cmp->u.direct.bmask) >>
							cmp->u.direct.bshift;
                  cmp->u.direct.blue[pixel].clirefs[clinum]--;
                  if (--cmp->u.direct.blue[pixel].refs == 0)
                     cmp->u.direct.blue[pixel].status = CS_FREE;
               }
            }
         break;
   }
   return 0;
}

/************************************************************************
*									*
*   cmap_query								*
*									*
*	sanity check pixels - return bad vals in lstp[0].		*
*									*
************************************************************************/
int
cmap_query
   AL((cmp, ncols, pixels, lstp))
   DB cmap_t *cmp
   DD int ncols
   DD pixel_t *pixels
   DD list_t *lstp
   DE
{
   register pixel_t i, pixel;
   register u16_t val;
   union {
      struct {
         u16_t	s[4];
      }shorts;
      struct {
         u32_t	l[2];
      }longs;
   }u;

   switch (cmp->class) {
      case GrayScale:
         for (i=0; i<ncols; i++) {
            u.shorts.s[0] = cmp->u.pseudo.rgb[pixels[i]].red;
            u.shorts.s[1] = cmp->u.pseudo.rgb[pixels[i]].red;
            u.shorts.s[2] = cmp->u.pseudo.rgb[pixels[i]].red;
            list_put(lstp, u.longs.l[0]);
            list_put(lstp, u.longs.l[1]);
         }
         break;
      case PseudoColor:
         for (i=0; i<ncols; i++) {
            u.shorts.s[0] = cmp->u.pseudo.rgb[pixels[i]].red;
            u.shorts.s[1] = cmp->u.pseudo.rgb[pixels[i]].green;
            u.shorts.s[2] = cmp->u.pseudo.rgb[pixels[i]].blue;
            list_put(lstp, u.longs.l[0]);
            list_put(lstp, u.longs.l[1]);
         }
         break;
      case DirectColor:
         for (i=0; i<ncols; i++) {
            pixel = (pixels[i] & cmp->u.direct.rmask) >> cmp->u.direct.rshift;
            u.shorts.s[0] = cmp->u.direct.red[pixel].red;
            pixel = (pixels[i] & cmp->u.direct.gmask) >> cmp->u.direct.gshift;
            u.shorts.s[1] = cmp->u.direct.green[pixel].red;
            pixel = (pixels[i] & cmp->u.direct.bmask) >> cmp->u.direct.bshift;
            u.shorts.s[2] = cmp->u.direct.blue[pixel].red;
            list_put(lstp, u.longs.l[0]);
            list_put(lstp, u.longs.l[1]);
         }
         break;
      case StaticGray:
      case StaticColor:
         for (i=0; i<ncols; i++) {
            if (pixels[i] == 1)
               val = MAXCOLVAL;
            else
               val = 0;
            u.shorts.s[0] = val;
            u.shorts.s[1] = val;
            u.shorts.s[2] = val;
            list_put(lstp, u.longs.l[0]);
            list_put(lstp, u.longs.l[1]);
         }
         break;
      case TrueColor:
         for (i=0; i<ncols; i++) {
            pixel = (pixels[i] & cmp->u.true.rmask) >> cmp->u.true.rshift;
            u.shorts.s[0] = pixel * cmp->u.true.rdiv;
            pixel = (pixels[i] & cmp->u.true.gmask) >> cmp->u.true.gshift;
            u.shorts.s[1] = pixel * cmp->u.true.rdiv;
            pixel = (pixels[i] & cmp->u.true.bmask) >> cmp->u.true.bshift;
            u.shorts.s[2] = pixel * cmp->u.true.rdiv;
            list_put(lstp, u.longs.l[0]);
            list_put(lstp, u.longs.l[1]);
         }
         break;
   }
   return 0;
}

/************************************************************************
*									*
*   cmap_all_pixels							*
*									*
*	Set the pixel values corresponding to all cell entries in the	*
*	the given colormap.						*
*									*
************************************************************************/
void
cmap_all_pixels
   AL((cmp, lstp))
   DB cmap_t *cmp
   DD list_t *lstp
   DE
{
   register int i, ncol;
   register int ri, gi, bi;
   register int rmax, gmax, bmax;
   register pixel_t pixel;

   switch (cmp->class) {
      case GrayScale:
      case PseudoColor:
         ncol = cmp->u.pseudo.n;
         for (i=0; i<ncol; i++) {
            list_put(lstp, (u32_t)i);
         }
         break;
      case DirectColor:
         ri = gi = bi = 0;
         rmax = cmp->u.direct.rn - 1;
         gmax = cmp->u.direct.gn - 1;
         bmax = cmp->u.direct.bn - 1;
         ncol = MAX(ri,gi);
         ncol = MAX(ncol,bi);
         for (i=0; i<ncol; i++) {
            pixel =	((ri << cmp->u.direct.rshift) & cmp->u.direct.rmask) |
			((gi << cmp->u.direct.gshift) & cmp->u.direct.gmask) |
			((bi << cmp->u.direct.bshift) & cmp->u.direct.bmask);
            list_put(lstp, (u32_t)pixel);
            if (ri < rmax) ri++;
            if (gi < gmax) gi++;
            if (bi < bmax) bi++;
         }
         break;
      case StaticGray:	/* these shouldn't be needed */
      case StaticColor:
      case TrueColor:
         warn("cmap_all_pixels: called for static cmap - can't happen\n");
         break;
   }
}

/************************************************************************
*									*
*   cmap_query_items							*
*									*
*	Retrieve cell values as X ColorItem's.  Only sets items that	*
*	correspond to allocated cells.					*
*									*
************************************************************************/
void
cmap_query_items
   AL((cmp, ncols, cip))
   DB cmap_t *cmp
   DD int ncols
   DD xColorItem *cip
   DE
{
   register pixel_t i, pixel;

   switch (cmp->class) {
      case GrayScale:
         for (i=0; i<ncols; i++) {
            pixel = cip[i].pixel;
            if (cmp->u.pseudo.rgb[pixel].status == CS_READ ||
		cmp->u.pseudo.rgb[pixel].status == CS_WRITE) {
               cip[i].red = cmp->u.pseudo.rgb[pixel].red;
               cip[i].green = cmp->u.pseudo.rgb[pixel].red;
               cip[i].blue = cmp->u.pseudo.rgb[pixel].red;
            }
         }
         break;
      case PseudoColor:
         for (i=0; i<ncols; i++) {
            pixel = cip[i].pixel;
            if (cmp->u.pseudo.rgb[pixel].status == CS_READ ||
		cmp->u.pseudo.rgb[pixel].status == CS_WRITE) {
               cip[i].red = cmp->u.pseudo.rgb[pixel].red;
               cip[i].green = cmp->u.pseudo.rgb[pixel].green;
               cip[i].blue = cmp->u.pseudo.rgb[pixel].blue;
            }
         }
         break;
      case DirectColor:
         for (i=0; i<ncols; i++) {
            pixel = (cip[i].pixel&cmp->u.direct.rmask) >> cmp->u.direct.rshift;
            cip[i].red = cmp->u.direct.red[pixel].red;
            pixel = (cip[i].pixel&cmp->u.direct.gmask) >> cmp->u.direct.gshift;
            cip[i].green = cmp->u.direct.green[pixel].red;
            pixel = (cip[i].pixel&cmp->u.direct.bmask) >> cmp->u.direct.bshift;
            cip[i].blue = cmp->u.direct.blue[pixel].red;
         }
         break;
      case StaticGray:	/* these shouldn't be needed */
      case StaticColor:
      case TrueColor:
         break;
   }
}

/************************************************************************
*									*
*   cmap_free_client							*
*									*
************************************************************************/
void
cmap_free_client
   AL((cmp, cp))
   DB cmap_t *cmp
   DD client_t *cp
   DE
{
   switch (cmp->class) {
      case GrayScale:
      case PseudoColor:
         free_client_cells(cmp->u.pseudo.rgb, cmp->u.pseudo.n, cp->clinum);
         break;
      case DirectColor:
         free_client_cells(cmp->u.direct.red, cmp->u.direct.rn, cp->clinum);
         free_client_cells(cmp->u.direct.green, cmp->u.direct.gn, cp->clinum);
         free_client_cells(cmp->u.direct.blue, cmp->u.direct.bn, cp->clinum);
         break;
   }
}

/************************************************************************
*									*
*   cmap_get_cells							*
*									*
*	Stores xColorItem's for up to the n cells in the given array.	*
*	Returns the number of cells retrieved.  For direct cmaps,	*
*	returns pixel values constructed arbitrarily, based on a	*
*	linear search of each of the r, g and b cell vectors.		*
*	Assuming there is no r-g-b coherency in the direct colormaps	*
*	of X servers, this should do the right thing.			*
*									*
*	This routine maintains internal state on the current place in	*
*	the search and should be called repeatedly until it returns	*
*	zero.  It then resets itself for the next search.		*
*									*
************************************************************************/
int
cmap_get_cells
   AL((cmp, status, n, cip))
   DB cmap_t *cmp
   DD u8_t status
   DD int n
   DD xColorItem *cip
   DE
{
   register pixel_t i, rpix, gpix, bpix;
   register int ncol = 0;
   static pixel_t rstart, gstart, bstart;	/* current starting point */
   
   switch (cmp->class) {
      case GrayScale:
         for (i=rstart; i<cmp->u.pseudo.n && ncol<n; i++)
            if (status == 0 || cmp->u.pseudo.rgb[i].status & status) {
               cip[ncol].pixel = i;
               cip[ncol].red = cmp->u.pseudo.rgb[i].red;
               cip[ncol].green = cmp->u.pseudo.rgb[i].red;
               cip[ncol].blue = cmp->u.pseudo.rgb[i].red;
               cip[ncol].flags = DoRed | DoGreen | DoBlue;
               ncol++;
            }
         rstart = ncol ? i : 0;
         break;
      case PseudoColor:
         for (i=rstart; i<cmp->u.pseudo.n && ncol<n; i++)
            if (status == 0 || cmp->u.pseudo.rgb[i].status & status) {
               cip[ncol].pixel = i;
               cip[ncol].red = cmp->u.pseudo.rgb[i].red;
               cip[ncol].green = cmp->u.pseudo.rgb[i].green;
               cip[ncol].blue = cmp->u.pseudo.rgb[i].blue;
               cip[ncol].flags = DoRed | DoGreen | DoBlue;
               ncol++;
            }
         rstart = ncol ? i : 0;
         break;
      case DirectColor:
         rpix = rstart;
         gpix = gstart;
         bpix = bstart;

         for (; ncol < n; ncol++) {
            cip[ncol].flags = 0;
            cip[ncol].pixel = 0;

            for (; rpix < cmp->u.direct.rn; rpix++)
               if (status == 0 || cmp->u.direct.red[rpix].status & status) {
                  cip[ncol].red = cmp->u.direct.red[rpix].red;
                  cip[ncol].flags |= DoRed;
                  cip[ncol].pixel |= rpix++ << cmp->u.direct.rshift;
                  break;
               }
            for (; gpix < cmp->u.direct.gn; gpix++)
               if (status == 0 || cmp->u.direct.green[gpix].status & status) {
                  cip[ncol].green = cmp->u.direct.green[gpix].red;
                  cip[ncol].flags |= DoGreen;
                  cip[ncol].pixel |= gpix++ << cmp->u.direct.gshift;
                  break;
               }
            for (; bpix < cmp->u.direct.bn; bpix++)
               if (status == 0 || cmp->u.direct.blue[bpix].status & status) {
                  cip[ncol].blue = cmp->u.direct.blue[bpix].red;
                  cip[ncol].flags |= DoBlue;
                  cip[ncol].pixel |= bpix++ << cmp->u.direct.bshift;
                  break;
               }
            if (cip[ncol].flags == 0)
               break;			/* ran out of colors first... */
         }
         if (ncol) {
            rstart = rpix;
            gstart = gpix;
            bstart = bpix;
         }
         else
            rstart = gstart = bstart = 0;
         break;
   }
   return ncol;
}

/*
**   hide_cells
**
**	Hides the first ncells cells in a colormap, making them
**	inaccessible to xmx or any shared clients.
**
**	This should only be run on a newly allocated colormap.
*/
static void
hide_cells
   AL((cmp, ncells))
   DB cmap_t *cmp
   DD int ncells
   DE
{
   register int i, n;

   switch (cmp->class) {
      case GrayScale:
      case PseudoColor:
         for (i=0; i<ncells; i++) {
            cmp->u.pseudo.rgb[i].status = CS_HIDDEN;
            cmp->u.pseudo.rgb[i].refs++;
            cmp->u.pseudo.rgb[i].clirefs[0]++;
         }
         break;
      case DirectColor:
         /*
         **  this is pretty badly hacked TODO
         */
         n = ncells <= (int)cmp->u.direct.rn ? ncells : (int)cmp->u.direct.rn;
         for (i=0; i<n; i++) {
            cmp->u.direct.red[i].status = CS_HIDDEN;
            cmp->u.direct.red[i].refs++;
            cmp->u.direct.red[i].clirefs[0]++;
         }
         n = ncells <= (int)cmp->u.direct.gn ? ncells : (int)cmp->u.direct.gn;
         for (i=0; i<n; i++) {
            cmp->u.direct.green[i].status = CS_HIDDEN;
            cmp->u.direct.green[i].refs++;
            cmp->u.direct.green[i].clirefs[0]++;
         }
         n = ncells <= (int)cmp->u.direct.bn ? ncells : (int)cmp->u.direct.bn;
         for (i=0; i<n; i++) {
            cmp->u.direct.blue[i].status = CS_HIDDEN;
            cmp->u.direct.blue[i].refs++;
            cmp->u.direct.blue[i].clirefs[0]++;
         }
         break;
   }
}

/*
**   unhide_cells
**
**	This routine is abit sloppy (wrt DirectColor) - it can unhide
**	an excess of cells, which will probably never be used.
*/
static int
unhide_cells
   AL((cmp, n))
   DB cmap_t *cmp
   DD int n
   DE
{
   register int i, j;

   switch (cmp->class) {
      case GrayScale:
      case PseudoColor:
         for (i=j=0; i<(int)cmp->u.pseudo.n && j<n; i++)
            if (cmp->u.pseudo.rgb[i].status == CS_HIDDEN) {
               cmp->u.pseudo.rgb[i].status = CS_FREE;
               cmp->u.pseudo.rgb[i].refs = 0;
               cmp->u.pseudo.rgb[i].clirefs[0] = 0;
               j++;
            }
         break;
      case DirectColor:
         for (i=j=0; i<(int)cmp->u.direct.rn && j<n; i++)
            if (cmp->u.direct.red[i].status == CS_HIDDEN) {
               cmp->u.direct.red[i].status = CS_FREE;
               cmp->u.direct.red[i].refs = 0;
               cmp->u.direct.red[i].clirefs[0] = 0;
               j++;
            }
         if (j < n)
            n = j;
         for (i=j=0; i<(int)cmp->u.direct.gn && j<n; i++)
            if (cmp->u.direct.green[i].status == CS_HIDDEN) {
               cmp->u.direct.green[i].status = CS_FREE;
               cmp->u.direct.green[i].refs = 0;
               cmp->u.direct.green[i].clirefs[0] = 0;
               j++;
            }
         if (j < n)
            n = j;
         for (i=j=0; i<(int)cmp->u.direct.bn && j<n; i++)
            if (cmp->u.direct.blue[i].status == CS_HIDDEN) {
               cmp->u.direct.blue[i].status = CS_FREE;
               cmp->u.direct.blue[i].refs = 0;
               cmp->u.direct.blue[i].clirefs[0] = 0;
               j++;
            }
         break;
   }
   return j;
}

/*
**   alloc_cells
*/
static int
alloc_cells
   AL((cp, n, clinum, contig, ncols, nplanes, pp, mask))
   DB cell_t *cp
   DD int n
   DD u8_t clinum
   DD int contig
   DD int ncols
   DD int nplanes
   DD pixel_t *pp
   DD mask_t *mask
   DE
{
   register int i;
   register int pixcnt;
   pixel_t pixval, pixel;
   mask_t planeval, lomask;
   int npixvals, nplanevals;

   if (nplanes == 0) {	/* just allocate pixels  (not continguous! TODO ) */
      pixcnt = 0;
      for (i=0; i<n; i++)
         if (cp[i].status == CS_FREE) {
            pp[pixcnt++] = i;
            if (pixcnt == ncols) {
               mark_cells(cp, clinum, ncols, pp, 0, *mask=0);
               return 0;
            }
         }
      return -1;
   }
   else if (nplanes == 32)	/* this probably won't ever happen anyway */
      return -1;
   else {			/* contiguous planes */
      pixcnt = 0;
      lomask = 0x0;
      nplanevals = 1 << nplanes;
      npixvals = n / nplanevals;
      for (i=0; (1<<i)<=npixvals; i++, lomask+=lomask+1) {
         for (pixval=0; pixval<npixvals; pixval++) {
            pixel = (pixval & lomask) | (pixval & ~lomask) << nplanes;
            for (planeval=0; planeval<nplanevals; planeval++)
               if (cp[pixel | planeval << i].status != CS_FREE)
                  break;
            if (planeval == nplanevals) {	/* found one */
               pp[pixcnt++] = pixel;
               if (pixcnt == ncols) {	/* done */
                  *mask = (nplanevals - 1) << i;
                  mark_cells(cp, clinum, ncols, pp, nplanes, *mask);
                  return 0;
               }
            }
         }
         pixcnt = 0;
      }
   }
   if (contig == 0) {		/* non-contiguous is okay */
      return -1;	/* not yet implemented */
   }
   return -1;
}

/*
**   mark_cells
**
**	Mark cells as allocated writable by the given client.
*/
static void
mark_cells
   AL((cp, clinum, ncols, pp, nbits, mask))
   DB cell_t *cp
   DD u8_t clinum
   DD int ncols
   DD pixel_t *pp
   DD int nbits			/* no. of bits set in mask */
   DD mask_t mask
   DE
{
   register int i, j, t, nmaskvals;
   register mask_t tm, invm;

   nmaskvals = 1 << nbits;
   invm = ~mask;

   for (i=0; i<ncols; i++) {
      tm = invm;
      for (j=0; j<nmaskvals; j++) {
         t = pp[i] | (tm & mask);

         cp[t].status = CS_WRITE;
         cp[t].refs++;
         cp[t].clirefs[clinum]++;

         tm = (tm + 1) | invm;
      }
   }
}

/*
**   free_client_cells
**
**	Remove client's references to cells in list
*/
static void
free_client_cells
   AL((cp, n, clinum))
   DB cell_t *cp
   DD u16_t n
   DD u8_t clinum
   DE
{
   register pixel_t i;

   for (i=0; i<n; i++)
      if (cp[i].clirefs[clinum]) {
         do {
            cp[i].clirefs[clinum]--;
            cp[i].refs--;
         } while (cp[i].clirefs[clinum]);

         if (cp[i].refs == 0)
            cp[i].status = CS_FREE;
      }
}

/*
**   find_color1
**
**	Look for, in order of preference, a matching channel value, a free
**	cell, or the closest channel value (stored as red).
*/
static int
find_color1
   AL((cp, n, val, pixel, lookup))
   DB cell_t *cp
   DD u16_t n
   DD u16_t val
   DD pixel_t *pixel
   DD int lookup
   DE
{
   register pixel_t i;
   register s32_t psave = -1;
#ifdef NO_CLOSEST
   register u16_t diff, mindiff;
#endif

   for (i=0; i<n; i++)
      switch (cp[i].status) {
         case CS_FREE:
            if (lookup == 0 && psave < 0)
               psave = (s32_t)i;
            break;
         case CS_READ:
            if (cp[i].red == val) {
               *pixel = i;
               return 0;
            }
            break;
      }

#ifdef NO_CLOSEST
   if (psave < 0) {
      mindiff = ~((u16_t)0);
      for (i=0; i<n; i++)
         if (cp[i].status == CS_READ) {
            diff = val > cp[i].red ? val - cp[i].red : cp[i].red - val;
            if (diff < mindiff) {
               mindiff = diff;
               psave = (s32_t)i;
            }
         }
   }
#endif
   if (psave < 0)
      return -1;
   else {
      *pixel = (pixel_t)psave;
      return 0;
   }
}

/*
**   find_color3
**
**	Look for, in order of preference, a matching color, a free
**	cell, or the closest color.
*/
static int
find_color3
   AL((cp, n, r, g, b, pixel, lookup))
   DB cell_t *cp
   DD u16_t n
   DD u16_t r
   DD u16_t g
   DD u16_t b
   DD pixel_t *pixel
   DD int lookup
   DE
{
   register int i;
   register s32_t psave = -1;
   register u32_t diff, mindiff;
/*
   register int rdiff, gdiff, bdiff;
*/

   for (i=0; i<n; i++)
      switch (cp[i].status) {
         case CS_FREE:
            if (lookup == 0 && psave < 0)
               psave = (s32_t)i;
            break;
         case CS_READ:
            if (cp[i].red == r && cp[i].green == g && cp[i].blue == b) {
               *pixel = i;
               return 0;
            }
            break;
      }

#ifdef NO_CLOSEST
   if (psave < 0) {
      mindiff = ~((u32_t)0);
      for (i=0; i<n; i++)
         if (cp[i].status == CS_READ) {
            diff = r > cp[i].red ? r - cp[i].red : cp[i].red - r;
            diff += g > cp[i].green ? g - cp[i].green : cp[i].green - g;
            diff += b > cp[i].blue ? b - cp[i].blue : cp[i].blue - b;
/*	better hues, worse values
            rdiff = r - cp[i].red;
            gdiff = g - cp[i].green;
            bdiff = b - cp[i].blue;
            diff = rdiff > gdiff ? rdiff - gdiff : gdiff - rdiff;
            diff += rdiff > bdiff ? rdiff - bdiff : rdiff - bdiff;
            diff += gdiff > bdiff ? gdiff - bdiff : gdiff - bdiff;
*/
            if (diff < mindiff) {
               mindiff = diff;
               psave = (s32_t)i;
            }
         }
   }
#endif
   if (psave < 0)
      return -1;
   else {
      *pixel = (pixel_t)psave;
      return 0;
   }
}

/*
**   copy_free
**
**	Scan old cell list for client-owned cells - allocate them in new
**	cell list and free them in the old one.
*/
static void
copy_free
   AL((ocp, ncp, n, clinum))
   DB cell_t *ocp
   DD cell_t *ncp
   DD u16_t n
   DD u8_t clinum
   DE
{
   register pixel_t i;

   for (i=0; i<n; i++)
      if (	ocp[i].status != CS_FREE &&
		ocp[i].clirefs[clinum]) {
         ncp[i].status = ocp[i].status;
         ncp[i].refs++;
         ncp[i].clirefs[clinum] = ocp[i].clirefs[clinum];
         ncp[i].red = ocp[i].red;
         ncp[i].green = ocp[i].green;
         ncp[i].blue = ocp[i].blue;

         ocp[i].clirefs[clinum] = 0;
         if (--ocp[i].refs == 0)
            ocp[i].status = CS_FREE;
      }
}
