/*
 * Copyright 1991-1997, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   ig.c								*
*									*
*	Icon grid.  This is a kind of pseudo-widget.  It does alot of	*
*	widget-like things in the window of a widget, but it lacks	*
*	the usual widget structure and semantics.  Maybe someday it	*
*	it will be a real widget.					*
*									*
*	Basically, it manages a collection of labelled icons, arranged	*
*	in a grid on the window of a widget supplied by the caller.	*
*	Operations include selection and arbitrary movement within	*
*	the grid.  It's kind of mac-like.				*
*									*
*	This code is widget-independent, and can be wrapped in any	*
*	instrinsic-based widget set.					*
*									*
************************************************************************/
/*** VERSION 0.09 ***/

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include "ig.h"
#include "ig_priv.h"

#ifdef SYSV
#define bcopy(a,b,c)	memcpy(b,a,c)
#define bzero(a,b)	memset(a,0,b)
#endif

void ig_down();
void ig_move();
void draw_box();
void ig_up();
void ig_expose();
void ig_configure();

XtActionsRec actions[] = {
	{"do-ig_down",		ig_down},
	{"do-ig_move",		ig_move},
	{"do-ig_up",		ig_up},
	{"do-ig_expose",	ig_expose},
	{"do-ig_configure",	ig_configure},
};

static char *ttstr = "\
<BtnDown>		: do-ig_down()		\n\
<BtnMotion>		: do-ig_move()		\n\
<BtnUp>			: do-ig_up()		\n\
<Expose>		: do-ig_expose()	\n\
<Configure>		: do-ig_configure()	\n\
";

static acn_t *contexts;
static IconGrid hackgp;

static void realize();
static void realize_view();
static void free_view();
static void realize_image();
static void free_image();
static IGIcon *find_icon_tag();
static void draw_icon();
static void realize_icon();
static int visible();
static void grid2coord();
static IGIcon *coord2grid();
static int read_bitmap_file();

/************************************************************************
*									*
*	ig_init								*
*									*
*	Establish a new icon grid on the given widget for the given	*
*	application context.						*
*									*
************************************************************************/
IconGrid
ig_init(context, widget, nstates)
   XtAppContext context;
   Widget widget;
   int nstates;
{
   register int sz;
   IconGrid gp;
   acn_t *acp;
   XGCValues vals;
   Pixmap pm;

   if ((gp = (IconGrid)malloc(sizeof(IconGrid_t))) == 0) {
      return 0;
   }
   gp->realized = 0;
   gp->context = context;
   gp->widget = widget;
   gp->viewp = 0;
   gp->curviewp = 0;
   gp->nstates = nstates;
   gp->nicons = 0;
   gp->icons = 0;

   gp->gridsz = DEFAULT_GRID_WIDTH * DEFAULT_GRID_HEIGHT;
   gp->gridw = DEFAULT_GRID_WIDTH;
   gp->gridh = DEFAULT_GRID_HEIGHT;
   sz = sizeof(IGIcon *) * gp->gridsz;
   if ((gp->grid = (IGIcon **)malloc(sz)) == 0) {
      free(gp);
      return 0;
   }
   bzero((char *)gp->grid, sz);

   gp->gridlw = gp->gridlh = 0;
   gp->sqwidth = gp->sqheight = 0;	/* fix me TODO */
   gp->padx = gp->pady = 2;

   /*
   ** have we seen this context before?
   */
   for (acp=contexts; acp; acp=acp->next)
      if (acp->context == context)
         break;
   if (acp == 0) {
      if ((acp = (acn_t *)malloc(sizeof(acn_t))) == 0) {
         free(gp->grid);
         free(gp);
         return 0;
      }
      acp->context = context;
      acp->next = contexts;
      contexts = acp;
      XtAppAddActions(context, actions, XtNumber(actions));
/*
      XtRegisterGrabAction(	ig_down,
				FALSE,
				ButtonMotionMask | ButtonReleaseMask,
				GrabModeAsync,
				GrabModeSync);
*/
   }
   /*
   ** add actions to widget's translation table
   */
   XtOverrideTranslations(widget, XtParseTranslationTable(ttstr));

hackgp = gp;
   return gp;
}

void
ig_realize(gp)
   IconGrid gp;
{
   register Widget widget;
   register IGView vp;
   register IGIcon *ip;
   XGCValues vals;
   Pixmap pm;
   
   widget = gp->widget;
   pm = XCreatePixmap(		XtDisplay(widget),
				XtWindow(widget),
				1, 1, 1);
   vals.function = GXxor;
   gp->gc_xor = XCreateGC(	XtDisplay(widget),
				pm,
				GCFunction,
				&vals);
   XFreePixmap(XtDisplay(widget), pm);

   vals.function = GXcopy;
   vals.foreground = BlackPixel(XtDisplay(widget), 0);	/* wrong TODO */
   vals.background = WhitePixel(XtDisplay(widget), 0);
   gp->gc_copy = XCreateGC(	XtDisplay(widget),
				XtWindow(widget),
				GCFunction | GCForeground | GCBackground,
				&vals);
   vals.foreground = WhitePixel(XtDisplay(widget), 0);	/* wrong TODO */
   vals.background = BlackPixel(XtDisplay(widget), 0);
   gp->gc_invert = XCreateGC(	XtDisplay(widget),
				XtWindow(widget),
				GCFunction | GCForeground | GCBackground,
				&vals);
   vals.function = GXinvert;
   gp->gc_flip = XCreateGC(	XtDisplay(widget),
				XtWindow(widget),
				GCFunction,
				&vals);

   XtVaGetValues(widget, XtNwidth, &gp->width, XtNheight, &gp->height, NULL);
   gp->vx = gp->vy = 0;
   gp->vwidth = gp->width;
   gp->vheight = gp->height;

   for (vp=gp->viewp; vp; vp=vp->next)
      realize_view(gp, vp);

   for (ip=gp->icons; ip; ip=ip->next)
      realize_icon(gp, ip);

   gp->realized = 1;

   ig_set_view(gp, gp->curviewp);
}

/************************************************************************
*									*
*	ig_register_view						*
*									*
*	Establish an alternative set of icons to represent the states	*
*	that icons can attain.  The images are all initially blank.	*
*									*
************************************************************************/
IGView
ig_register_view(gp, fontname)
   IconGrid gp;
   char *fontname;
{
   register int i, len;
   IGView vp;

   if ((vp = (IGView)malloc(sizeof(IGView_t))) == 0)
      return 0;

   len = sizeof(image_t *) * gp->nstates;
   if ((vp->imagev = (image_t **)malloc(len)) == 0) {
      free(vp);
      return 0;
   }
   bzero((char *)vp->imagev, len);

   vp->width = vp->height = 0;
   if (vp->fontname = (char *)malloc(strlen(fontname)+1))
      strcpy(vp->fontname, fontname);
   else {
      free(vp->imagev);
      free(vp);
      return 0;
   }
   vp->fsp = 0;
   vp->next = gp->viewp;
   gp->viewp = vp;

   if (gp->realized)
      realize_view(gp, vp);

   return vp;
}

static void
realize_view(gp, vp)
   IconGrid gp;
   IGView vp;
{
   register int i;
   int dr;
   XCharStruct fmet;

   vp->fsp = XLoadQueryFont(XtDisplay(gp->widget), vp->fontname);
   if (vp->fsp == 0)
      return;  /* need better error recovery here TODO */

   if (gp->curviewp == vp) {	/* view already set */
      XSetFont(XtDisplay(gp->widget), gp->gc_copy, vp->fsp->fid);
      XSetFont(XtDisplay(gp->widget), gp->gc_invert, vp->fsp->fid);
   }
   XTextExtents(vp->fsp, "Ij", 2, &dr, &vp->ascent, &vp->descent, &fmet);

   for (i=0; i<(int)gp->nstates; i++)
      if (vp->imagev[i])
         realize_image(gp, vp->imagev[i]);
}

static void
free_view(gp, vp)
   IconGrid gp;
   IGView vp;
{
   register int i;

   if (vp->imagev) {
      for (i=0; i<gp->nstates; i++)
         if (vp->imagev[i])
            free_image(vp->imagev[i]);
      free(vp->imagev);
   }
   if (vp->fsp)
      XFreeFont(XtDisplay(gp->widget), vp->fsp);
   if (vp->fontname)
      free(vp->fontname);
   free(vp);
}

/************************************************************************
*									*
*	ig_register_image_file						*
*	ig_register_image_data						*
*									*
*	Establish an icon image for a state in a view.  Replaces any	*
*	existing image.							*
*									*
************************************************************************/
int
ig_register_image_file(gp, vp, state, imagefile, maskfile)
   IconGrid gp;
   IGView vp;
   int state;
   char *imagefile;
   char *maskfile;
{
   uint_t iwidth, iheight, width, height;
   uchar_t *image, *mask;

   if (read_bitmap_file(imagefile, &iwidth, &iheight, &image)) {
      return -1;
   }
   if (read_bitmap_file(maskfile, &width, &height, &mask)) {
      return -1;
   }

   if (iwidth != width || iheight != height)
      return -1;

   return ig_register_image_data(gp, vp, state, image, mask, width, height);
}

int
ig_register_image_data(gp, vp, state, imagebits, maskbits, width, height)
   IconGrid gp;
   IGView vp;
   int state;
   uchar_t *imagebits;
   uchar_t *maskbits;
   uint_t width, height;
{
   register int len;
   register image_t *imp;

   if (state < 0 || state >= gp->nstates)
      return -1;

   if ((imp = (image_t *)malloc(sizeof(image_t))) == 0)
      return -1;

   len = (int)(((Dimension)(width + 7) / 8) * height);
   if ((imp->image = (uchar_t *)malloc(len)) == 0) {
      free(imp);
      return -1;
   }
   if ((imp->mask = (uchar_t *)malloc(len)) == 0) {
      free_image(imp);
      return -1;
   }
   bcopy((char *)imagebits, (char *)imp->image, len);
   bcopy((char *)maskbits, (char *)imp->mask, len);

   imp->width = width;
   imp->height = height;

   if (width > vp->width)
      vp->width = width;
   if (height > vp->height)
      vp->height = height;

   if (vp->imagev[state])
      free_image(vp->imagev[state]);

   vp->imagev[state] = imp;

   if (gp->realized)
      realize_image(gp, imp);

   return 0;
}

static void
realize_image(gp, imp)
   IconGrid gp;
   image_t *imp;
{
   imp->normal = XCreateBitmapFromData(	XtDisplay(gp->widget),
					XtWindow(gp->widget),
					(char *)imp->image,
					(uint_t)imp->width,
					(uint_t)imp->height);
   imp->invert = XCreateBitmapFromData(	XtDisplay(gp->widget),
					XtWindow(gp->widget),
					(char *)imp->mask,
					(uint_t)imp->width,
					(uint_t)imp->height);
   XCopyArea(	XtDisplay(gp->widget),
		imp->normal,
		imp->invert,
		gp->gc_xor,
		0, 0,
		(uint_t)imp->width,
		(uint_t)imp->height,
		0, 0);
}

static void
free_image(imp)
   image_t *imp;
{
   if (imp->image)
      free(imp->image);
   if (imp->mask)
      free(imp->mask);
   free(imp);
}

/************************************************************************
*									*
*	ig_set_view							*
*									*
*	Cause a repaint of the window after setting the view.		*
*									*
*	TODO - need to change tag clipping for each icon when view	*
*	changes. (or do it a completely different way).			*
*									*
************************************************************************/
void
ig_set_view(gp, vp)
   IconGrid gp;
   IGView vp;
{
   register Dimension width, height;

   width = vp->width + gp->padx * 2;
   height = vp->height + vp->ascent + vp->descent + 1 + gp->pady * 2;

   if (gp->realized)
      if (gp->curviewp == 0 || gp->curviewp->fsp != vp->fsp) {
         XSetFont(XtDisplay(gp->widget), gp->gc_copy, vp->fsp->fid);
         XSetFont(XtDisplay(gp->widget), gp->gc_invert, vp->fsp->fid);
         /* need to adjust tagwidth for all icons here */
      }
   gp->curviewp = vp;
   gp->sqwidth = width;
   gp->sqheight = height;
   if (gp->vwidth > gp->gridlw * width)
      gp->gridlw = gp->vwidth / width;
   if (gp->vheight > gp->gridlh * height)
      gp->gridlh = gp->vheight / height;

   if (gp->realized)
      ig_refresh(gp);
}

/************************************************************************
*									*
*	ig_add_icon							*
*									*
*	Add an icon.							*
*									*
************************************************************************/
IGIcon *
ig_add_icon(gp, tag, state, selected, row, col, data)
   IconGrid gp;
   char *tag;
   int state;
   int selected;
   int row, col;	/* if -1, next open grid space */
   void *data;
{
   register int i, j, len, rv, resize;
   IGIcon *ip, *next, *last;

   for (last=0,next=gp->icons; next; last=next,next=next->next) {
      rv = strcmp(tag, next->tag);
      if (rv == 0)
         return 0;		/* error - tag not unique */
      else if (rv < 0)
         break;
   }
   if ((ip = (IGIcon *)malloc(sizeof(IGIcon))) == 0)
      return 0;

   ip->taglen = strlen(tag);
   len = ip->taglen + 1;
   if ((ip->tag = (char *)malloc(len)) == 0) {
      free(ip);
      return 0;
   }
   bcopy(tag, ip->tag, len);

   ip->cliptaglen = ip->taglen;
   if (gp->realized)
      realize_icon(gp, ip);

   ip->state = state;
   ip->selected = selected;
   resize = 0;
   if (row < 0 || col < 0 || gridval(gp, row, col)) {
      for (i=0; i<(int)gp->gridlh; i++) {
         for (j=0; j<(int)gp->gridlw; j++)
            if (gridval(gp, i, j) == 0)
               break;
         if (j < (int)gp->gridlw)
            break;
      }
      if (gridval(gp, i, j)) {
         i = 0;
         j = gp->gridlh++;
         resize++;
      }
      row = i;
      col = j;
   }
   else {
      if (row > gp->gridlw)
         gp->gridlw = resize = row;
      if (col > gp->gridlh)
         gp->gridlh = resize = col;
   }
   if (!gp->realized && resize)
      ig_resize(gp);
   ip->row = row;
   ip->col = col;
   gridval(gp, row, col) = ip;

   ip->data = data;

   ip->prev = 0;

   /*
   **	insert icon into list in alphabetical order by tag
   */
   if (last) {
      if (ip->next = last->next)
         ip->next->prev = ip;
      ip->prev = last;
      last->next = ip;
   }
   else {
      if (gp->icons)
         gp->icons->prev = ip;
      ip->next = gp->icons;
      gp->icons = ip;
   }
   if (gp->realized)
      draw_icon(gp, ip, 0);

   return ip;
}

static void
realize_icon(gp, ip)
   IconGrid gp;
   IGIcon *ip;
{
   ip->tagwidth = XTextWidth(gp->curviewp->fsp, ip->tag, ip->taglen);
   while (ip->tagwidth > gp->sqwidth)
      ip->tagwidth = XTextWidth(gp->curviewp->fsp, ip->tag, --ip->cliptaglen);
}

void *
ig_set_data(ip, data)
   IGIcon *ip;
   void *data; 
{
   void *old;

   old = ip->data;
   ip->data = data;
   return old;
}

/*
**	ig_find_icon
**
**	Search ordered icon list for tag, return icon if found
*/
IGIcon *
ig_find_icon(gp, tag)
   IconGrid gp;
   char *tag;
{
   register int rv;
   IGIcon *ip;

   for (ip=gp->icons; ip; ip=ip->next) {
      rv = strcmp(tag, ip->tag);
      if (rv == 0)
         return ip;
      else if (rv < 0)
         break;
   }
   return 0;
}

void
ig_change_state(gp, ip, state)
   IconGrid gp;
   IGIcon *ip;
   int state;
{
   if (state < 0 || state >= gp->nstates)
      return;	/* error! */

   if (ip->state != state) {
      ip->state = state;
      if (gp->realized)
         draw_icon(gp, ip, 1);
   }
}

void
ig_select(gp, ip)
   IconGrid gp;
   IGIcon *ip;	/* if null, all */
{
   if (ip) {
      ip->selected = 1;
      if (gp->realized)
         draw_icon(gp, ip, 1);
   }
   else
      for (ip=gp->icons; ip; ip=ip->next) {
         ip->selected = 1;
         if (gp->realized)
            draw_icon(gp, ip, 1);
      }
}

void
ig_unselect(gp, ip)
   IconGrid gp;
   IGIcon *ip;	/* if null, all */
{
   if (ip) {
      ip->selected = 0;
      if (gp->realized)
         draw_icon(gp, ip, 1);
   }
   else
      for (ip=gp->icons; ip; ip=ip->next) {
         ip->selected = 0;
         if (gp->realized)
            draw_icon(gp, ip, 1);
      }
}

void
ig_remove_icon(gp, ip)
   IconGrid gp;
   IGIcon *ip;
{
   if (gp->curicon == ip)
      gp->curicon = 0;

   if (ip->next)
      ip->next->prev = ip->prev;

   if (ip->prev)
      ip->prev->next = ip->next;
   else
      gp->icons = ip->next;

   gridval(gp, ip->row, ip->col) = 0;
   ig_clear(gp, ip->row, ip->col);
   if (ip->tag)
      free(ip->tag);
   free(ip);
}

/************************************************************************
*									*
*   ig_call_callback							*
*									*
*	Call the given callback routine once for each selected icon.	*
*	Always calls the given routine once with a null pointer to	*
*	indicate completion.						*
*									*
************************************************************************/
void
ig_call_callback(gp, rtn)
   IconGrid gp;
   IGproc rtn;
{
   IGIcon *ip;

   for (ip=gp->icons; ip; ip=ip->next) {
      if (ip->selected)
         (*rtn)(gp, ip);
   }
   (*rtn)(gp, 0);
}

void
ig_refresh(gp)
   IconGrid gp;
{
   register IGIcon *nip;

   XClearWindow(XtDisplay(gp->widget), XtWindow(gp->widget));

   for (nip=gp->icons; nip; nip=nip->next)
      draw_icon(gp, nip, 0);
}

static void
draw_icon(gp, ip, clear)
   IconGrid gp;
   IGIcon *ip;
   int clear;
{
   int lx, ty, rx, by;

   grid2coord(gp, ip->row, ip->col, &lx, &ty);

   if (!visible(gp, lx, ty))
      return;

   if (gp->curviewp) {
      if (clear)
         XClearArea(	XtDisplay(gp->widget),
			XtWindow(gp->widget),
			lx, ty,
			gp->sqwidth, gp->sqheight,
			False);

      if (gp->curviewp->imagev[ip->state]) {
         XCopyPlane(	XtDisplay(gp->widget),
			ip->selected ?
				gp->curviewp->imagev[ip->state]->invert:
				gp->curviewp->imagev[ip->state]->normal,
			XtWindow(gp->widget),
			gp->gc_copy,
			0, 0,
			gp->curviewp->imagev[ip->state]->width,
			gp->curviewp->imagev[ip->state]->height,
			lx + gp->padx,
			ty + gp->pady,
			0x1);
         XDrawImageString(XtDisplay(gp->widget),
			XtWindow(gp->widget),
			ip->selected ? gp->gc_invert : gp->gc_copy,
			lx + gp->sqwidth/2 - ip->tagwidth/2,
			ty + gp->sqheight - gp->pady - gp->curviewp->descent,
			ip->tag,
			ip->cliptaglen);
      }
   }
   else {
   }
}

/*
**	visible
**
**	is grid square visible?
*/
static int
visible(gp, lx, ty)
   IconGrid gp;
   int lx, ty;
{
   register int rx, by;

   rx = lx + gp->sqwidth - 1;
   by = ty + gp->sqheight - 1;

   if (	rx < gp->vx ||
	lx >= (int)(gp->vx + gp->vwidth) ||
	by < gp->vy ||
	ty >= (int)(gp->vy + gp->vheight))
      return 0;
   else
      return 1;
}

void
ig_clear(gp, row, col)
   IconGrid gp;
   int row, col;
{
   int x, y;

   if (!gp->realized)
      return;

   grid2coord(gp, row, col, &x, &y);
   if (!visible(gp, x, y))
      return;

   XClearArea(	XtDisplay(gp->widget),
		XtWindow(gp->widget),
		x, y,
		gp->sqwidth, gp->sqheight,
		False);
}

static void
grid2coord(gp, row, col, xp, yp)
   IconGrid gp;
   int row, col;
   int *xp, *yp;
{
   *xp = col * gp->sqwidth;
   *yp = row * gp->sqheight;
}

static IGIcon *
coord2grid(gp, x, y, rowp, colp, hotxp, hotyp)
   IconGrid gp;
   int x, y;
   int *rowp, *colp;
   int *hotxp, *hotyp;
{
   int row, col;
   int width, height;
   int hotx, hoty;
   int ix, bit;
   IGIcon *ip;

   row = y / (int)gp->sqheight;
   col = x / (int)gp->sqwidth;
   ip = gridval(gp, row, col);

   hotx = x % (int)gp->sqwidth;
   hoty = y % (int)gp->sqheight;

   if (hotxp)
      *hotxp = hotx;
   if (hotyp)
      *hotyp = hoty;

   if (ip) {
      hotx -= gp->padx;
      hoty -= gp->pady;

      /*
      **   in the pad area
      */
      if (	hotx < 0 ||
		hoty < 0 ||
		hotx >= (int)gp->sqwidth - 2 * gp->padx ||
		hoty >= (int)gp->sqheight - 2 * gp->pady)
         ip = 0;
      /*
      **   in the tag area
      */
      else if (hoty >= (int)gp->sqheight -
			(gp->curviewp->ascent + gp->curviewp->descent)) {
         if (	hotx < ((int)gp->sqwidth/2 - (int)ip->tagwidth/2) ||
		hotx >= ((int)gp->sqwidth/2 + (int)ip->tagwidth/2))
            ip = 0;
      }
      /*
      **   in the icon image area
      */
      else if (gp->curviewp->imagev[ip->state]) {
         width = (int)gp->curviewp->imagev[ip->state]->width;
         height = (int)gp->curviewp->imagev[ip->state]->height;
         hotx -= (int)gp->curviewp->width/2 - width/2;
         hoty -= (int)gp->curviewp->height/2 - height/2;
         if (	hotx < 0 || hotx >= width ||
		hoty < 0 || hoty >= height)
            ip = 0;
         else {				/* correlate with mask */
            ix = hoty * ((width + 7) / 8) + hotx / 8;
            bit = hotx % 8;
            if ((gp->curviewp->imagev[ip->state]->mask[ix] & (1<<bit)) == 0)
               ip = 0;
         }
      }
      /*
      **   (no icon, no hit)
      */
      else
         ip = 0;
   }
   if (rowp)
      *rowp = row;
   if (colp)
      *colp = col;

   return ip;
}

void
ig_down(widget, event, params, nparams)
   Widget widget;
   XEvent *event;
   String *params;
   Cardinal *nparams;
{
   int row, col;
   int hotx, hoty;
   IGIcon *ip;
   IconGrid gp;

   gp = hackgp;

   ip = coord2grid(gp, event->xbutton.x, event->xbutton.y,
						&row, &col, &hotx, &hoty);
   if (ip) {
      gp->curicon = ip;
      gp->origx = hotx;
      gp->origy = hoty;
      gp->lastx = event->xbutton.x;
      gp->lasty = event->xbutton.y;
   }
   else {
      gp->origx = gp->lastx = event->xbutton.x;
      gp->origy = gp->lasty = event->xbutton.y;
   }
}

void
ig_move(widget, event, params, nparams)
   Widget widget;
   XEvent *event;
   String *params;
   Cardinal *nparams;
{
   int dx, dy;
   IconGrid gp;

   gp = hackgp;

   if (gp->motion) {
      draw_box(gp, gp->lastx, gp->lasty);
      draw_box(gp, event->xbutton.x, event->xbutton.y);
   }
   else {
      dx = gp->lastx > event->xbutton.x ?
				gp->lastx - event->xbutton.x :
				event->xbutton.x - gp->lastx;
      dy = gp->lasty > event->xbutton.y ?
				gp->lasty - event->xbutton.y :
				event->xbutton.y - gp->lasty;
      if (dx > 3 || dy > 3) {
         draw_box(gp, event->xbutton.x, event->xbutton.y);
         gp->motion = 1;
      }
   }
   gp->lastx = event->xbutton.x;
   gp->lasty = event->xbutton.y;
}

void
draw_box(gp, x, y)
   IconGrid gp;
   int x, y;
{
   register int lx, ty;
   register int w, h;

   if (!gp->realized)
      return;

   if (gp->curicon) {
      XDrawRectangle(	XtDisplay(gp->widget),
			XtWindow(gp->widget),
			gp->gc_flip,
			x - gp->origx,
			y - gp->origy,
			gp->sqwidth,
			gp->sqheight);
   }
   else {
      if (x < gp->origx) {
         lx = x;
         w = gp->origx - x;
      }
      else {
         lx = gp->origx;
         w = x - gp->origx;
      }
      if (y < gp->origy) {
         ty = y;
         h = gp->origy - y;
      }
      else {
         ty = gp->origy;
         h = y - gp->origy;
      }
      XDrawRectangle(	XtDisplay(gp->widget),
			XtWindow(gp->widget),
			gp->gc_flip,
			lx, ty,
			w, h);
   }
}

void
ig_up(widget, event, params, nparams)
   Widget widget;
   XEvent *event;
   String *params;
   Cardinal *nparams;
{
   int row, col, row1, col1, row2, col2, t;
   IGIcon *ip;
   IconGrid gp;

   gp = hackgp;

   ip = coord2grid(gp, event->xbutton.x, event->xbutton.y, &row1, &col1, 0, 0);

   if (gp->motion) {
      draw_box(gp, event->xbutton.x, event->xbutton.y);
      gp->motion = 0;

      if (gp->curicon) {
         if (gridval(gp, row1, col1) == 0) {	/* empty grid square */
            gridval(gp, gp->curicon->row, gp->curicon->col) = 0;
            gridval(gp, row1, col1) = gp->curicon;
            ig_clear(gp, gp->curicon->row, gp->curicon->col);
            gp->curicon->row = row1;
            gp->curicon->col = col1;
            draw_icon(gp, gp->curicon, 0);
         }
         gp->curicon = 0;
      }
      else {
         coord2grid(gp, gp->origx, gp->origy, &row2, &col2, 0, 0);
         if (row1 > row2) {
            t = row1;
            row1 = row2;
            row2 = t;
         }
         if (col1 > col2) {
            t = col1;
            col1 = col2;
            col2 = t;
         }
         for (row=row1; row<=row2; row++)
            for (col=col1; col<=col2; col++)
               if ((ip = gridval(gp, row, col)) && ip->selected == 0) {
                  ip->selected = 1;
                  draw_icon(gp, ip, 1);
               }
      }
   }
   else if (gp->curicon == ip)
      if (ip) {
         ip->selected = !ip->selected;
         draw_icon(gp, ip, 1);
         gp->curicon = 0;
      }
      else
         for (ip=gp->icons; ip; ip=ip->next)
            if (ip->selected) {
               ip->selected = 0;
               draw_icon(gp, ip, 1);
            }
/*
   XtUngrabPointer(widget, CurrentTime);
*/
}

/*
** ig_expose
**
**	Collect contiguous exposures - redraw only the smallest
**	enclosing rectangle.
*/
void
ig_expose(widget, event, params, nparams)
   Widget widget;
   XEvent *event;
   String *params;
   Cardinal *nparams;
{
   register IconGrid gp;
   register IGIcon *nip;
   int lx, ty;
   static pending;
   static int x, y, width, height;

   if (event->type == Expose) {
      if (pending == 0) {
         x = event->xexpose.x;
         y = event->xexpose.y;
         width = event->xexpose.width;
         height = event->xexpose.height;
      }
      else {
         if (event->xexpose.x < x) {
            x = event->xexpose.x;
            width += x - event->xexpose.x;
         }
         if (event->xexpose.y < y) {
            y = event->xexpose.y;
            height += y - event->xexpose.y;
         }
         if (event->xexpose.x + event->xexpose.width > x + width)
            width = event->xexpose.x + event->xexpose.width - x;
         if (event->xexpose.y + event->xexpose.height > y + height)
            height = event->xexpose.y + event->xexpose.height - y;
      }
      if (event->xexpose.count == 0) {
         gp = hackgp;
         if (!gp->realized)
            ig_realize(gp);

         XClearArea(	XtDisplay(gp->widget),
			XtWindow(gp->widget),
			x, y,
			width, height,
			False);

         for (nip=gp->icons; nip; nip=nip->next) {
            grid2coord(gp, nip->row, nip->col, &lx, &ty);
            if (	lx <= x + width &&
			ty <= y + height &&
			lx + (int)gp->sqwidth > x &&
			ty + (int)gp->sqheight > y)
               draw_icon(gp, nip, 0);
         }
         pending = 0;
      }
      else
         pending = 1;
   }
}

void
ig_configure(widget, event, params, nparams)
   Widget widget;
   XEvent *event;
   String *params;
   Cardinal *nparams;
{
   register IconGrid gp;

   gp = hackgp;

   XtVaGetValues(	gp->widget,
			XtNwidth, &gp->vwidth,
			XtNheight, &gp->vheight,
			NULL);
}

void
ig_resize(gp)
   IconGrid gp;
{
   gp->vwidth = gp->gridlw * gp->sqwidth;
   gp->vheight = gp->gridlh * gp->sqheight;
   XtVaSetValues(	gp->widget,
			XtNwidth, gp->vwidth,
			XtNheight, gp->vheight,
			NULL);
}

static int
read_bitmap_file(filename, wp, hp, dp)
   char *filename;
   uint_t *wp;
   uint_t *hp;
   uchar_t *dp;
{
   printf("read_bitmap_file: not implemented!\n");

   return -1;
}
