/*----------------------------------------------------------------------------
--
--  Module:           XmUbNoteBook
--
--  Project:          XmUb - Ulle's Motif widgets
--  System:           
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    This is the implementation of the widget.
--
--  Filename:         XmUbNoteBk.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1993-03-06
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: XmUbNoteBk.c, Version: 1.1, Date: 95/02/18 15:10:09";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <Xm/XmP.h>

#include "System.h"

/* Private widget header file. */
#include "XmUbNoteBkP.h"

/* Bitmaps for predefined bindings. */
#include "XmUbNoteBkSpiralTop.h"
#include "XmUbNoteBkSpiralBottom.h"
#include "XmUbNoteBkSpiralLeft.h"
#include "XmUbNoteBkSpiralRight.h"

#include "XmUbNoteBkSolidTop.h"
#include "XmUbNoteBkSolidBottom.h"
#include "XmUbNoteBkSolidLeft.h"
#include "XmUbNoteBkSolidRight.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

#define DEFAULT_MARGIN        5
#define INVALID_SIDE         -1
#define INVALID_TAB_NUMBER   -1
#define BOOK_PAGE_WIDTH       4


#define START_PIXMAP_INDEX   0
#define CENTER_PIXMAP_INDEX  1
#define END_PIXMAP_INDEX     2

/* Resource converters. */
#define XmRbindingType      "BindingType"
#define XmRchildType        "ChildType"
#define XmRcoveredPagesPos  "CoveredPagesPos"
#define XmRhorizDir         "HorizDir"
#define XmRscrollDirection  "ScrollDirection"
#define XmRsideAttachment   "SideAttachment"
#define XmRtabPacking       "TabPacking"
#define XmRvertDir          "VertDir"

/* Internal resources. */
#define XmUbNinternalKidRepos  "internal.kid.repos"

#define InitArray( array, size, value )  \
  { \
    int xxx_array_ix; \
    for( xxx_array_ix = 0; xxx_array_ix < size; xxx_array_ix ++ ) \
      array[ xxx_array_ix ] = value; \
  }

#ifdef MAX
#undef MAX
#endif

#ifdef MIN
#undef MIN
#endif

#define MAX( x, y )  ( ( x ) > ( y ) ? ( x ) : ( y ) )
#define MIN( x, y )  ( ( x ) < ( y ) ? ( x ) : ( y ) )

/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

typedef struct {
  int  width;
  int  height;
} DimensionDescr, *DimensionDescrRef;


typedef struct {
  int        num_managed;
  Dimension  max_width;
  Dimension  max_height;
  Dimension  total_width;
  Dimension  total_height;
} ChildSizes, *ChildSizesRef;


typedef struct {
  Dimension   work_area_width;
  Dimension   work_area_height;

  Dimension   binding_width;
  Dimension   binding_height;

  ChildSizes  tab_sizes[ 4 ];

  ChildSizes  scroll_sizes[ 4 ];
  Dimension   back_scroll_dim[ 4 ];    /* width for horiz, height for vert */
  Dimension   forward_scroll_dim[ 4 ];
} NoteBookSizes, *NoteBookSizesRef;

typedef struct {
  int            width;
  int            height;
  unsigned char  *bits;
} BitmapDescriptor, *BitmapDescriptorRef;

typedef struct {
  Boolean  binding;
  Boolean  book_side[ 4 ];
  Boolean  front_area;
} RedrawDescr, *RedrawRef;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Internal Motif functions (how do we know about this....?) */
extern void
  _XmBackgroundColorDefault();

/* Quarks. */
static XrmQuark   quark_bottom;
static XrmQuark   quark_left;
static XrmQuark   quark_right;
static XrmQuark   quark_top;

/* Predefined bindings. */
static BitmapDescriptor  predef_solid_bitmaps[ 4 ];
static BitmapDescriptor  predef_spiral_bitmaps[ 4 ];

/* The notebook resource list. */
static XtResource resources[] = {
  {
    XmUbNbindingPixmapCenter,
    XmUbCBindingPixmapCenter,
    XtRPixmap, sizeof( Pixmap ),
    XtOffset( XmUbNoteBookWidget, notebook.binding_pixmap_center ),
    XtRImmediate,
    (XtPointer) NULL
  },
  {
    XmUbNbindingPixmapEnd,
    XmUbCBindingPixmapEnd,
    XtRPixmap, sizeof( Pixmap ),
    XtOffset( XmUbNoteBookWidget, notebook.binding_pixmap_end ),
    XtRImmediate,
    (XtPointer) NULL
  },
  {
    XmUbNbindingPixmapStart,
    XmUbCBindingPixmapStart,
    XtRPixmap, sizeof( Pixmap ),
    XtOffset( XmUbNoteBookWidget, notebook.binding_pixmap_start ),
    XtRImmediate,
    (XtPointer) NULL
  },
  {
    XmUbNbindingSide,
    XmUbCBindingSide,
    XmRsideAttachment, sizeof( sideAttachment ),
    XtOffset( XmUbNoteBookWidget, notebook.binding_side ),
    XtRImmediate,
    (XtPointer) XmUbTOP
  },
  {
    XmUbNbindingType,
    XmUbCBindingType,
    XmRbindingType, sizeof( bindingType ),
    XtOffset( XmUbNoteBookWidget, notebook.binding_type ),
    XtRImmediate,
    (XtPointer) XmUbBINDING_SPIRAL
  },
  {
    XmUbNbookSideExtent,
    XmUbCBookSideExtent,
    XtRDimension, sizeof( Dimension ),
    XtOffset( XmUbNoteBookWidget, notebook.book_side_extent ),
    XtRImmediate,
    (XtPointer) 20
  },
  {
    XmUbNcoveredPageBackground,
    XmCBackground,
    XtRPixel, sizeof( Pixel ),
    XtOffset( XmUbNoteBookWidget, notebook.covered_page_background ),
    XmRCallProc,
    (XtPointer) _XmBackgroundColorDefault
  },
  {
    XmUbNcoveredPagesPos,
    XmUbCCoveredPagesPos,
    XmRcoveredPagesPos, sizeof( coveredPagesPos ),
    XtOffset( XmUbNoteBookWidget, notebook.book_pages_sides ),
    XtRImmediate,
    (XtPointer) XmBOTTOM_RIGHT
  },
  {
    XmUbNequalizeTabsBottom,
    XmUbCEqualizeTabs,
    XtRBoolean, sizeof( Boolean ),
    XtOffset( XmUbNoteBookWidget, notebook.equalize_tabs_bottom ),
    XtRImmediate,
    (XtPointer) True
  },
  {
    XmUbNequalizeTabsLeft,
    XmUbCEqualizeTabs,
    XtRBoolean, sizeof( Boolean ),
    XtOffset( XmUbNoteBookWidget, notebook.equalize_tabs_left ),
    XtRImmediate,
    (XtPointer) True
  },
  {
    XmUbNequalizeTabsRight,
    XmUbCEqualizeTabs,
    XtRBoolean, sizeof( Boolean ),
    XtOffset( XmUbNoteBookWidget, notebook.equalize_tabs_right ),
    XtRImmediate,
    (XtPointer) True
  },
  {
    XmUbNequalizeTabsTop,
    XmUbCEqualizeTabs,
    XtRBoolean, sizeof( Boolean ),
    XtOffset( XmUbNoteBookWidget, notebook.equalize_tabs_top ),
    XtRImmediate,
    (XtPointer) True
  },
  {
    XmUbNfrontPageBackground,
    XmCBackground,
    XtRPixel, sizeof( Pixel ),
    XtOffset( XmUbNoteBookWidget, notebook.front_page_background ),
    XmRCallProc,
    (XtPointer) _XmBackgroundColorDefault
  },
  {
    XmUbNfrontPageFixed,
    XmUbCFrontPageFixed,
    XtRBoolean, sizeof( Boolean ),
    XtOffset( XmUbNoteBookWidget, notebook.front_page_fixed ),
    XtRImmediate,
    (XtPointer) False
  },
  {
    XmUbNtabPacking,
    XmUbCTabPacking,
    XmRtabPacking, sizeof( tabPacking ),
    XtOffset( XmUbNoteBookWidget, notebook.tab_packing ),
    XtRImmediate,
    (XtPointer) XmUbTAB_PACK_TIGHT
  },
  {
    XmNrecomputeSize,
    XmCRecomputeSize,
    XtRBoolean, sizeof( Boolean ),
    XtOffset( XmUbNoteBookWidget, notebook.resize_allowed ),
    XtRImmediate,
    (XtPointer) True
  },
  {
    XmUbNrepeatCenterPixmap,
    XmUbCRepeatCenterPixmap,
    XtRBoolean, sizeof( Boolean ),
    XtOffset( XmUbNoteBookWidget, notebook.repeat_center_binding ),
    XtRImmediate,
    (XtPointer) True
  },
  {
    XmUbNresizeWorkArea,
    XmUbCResizeWorkArea,
    XtRBoolean, sizeof( Boolean ),
    XtOffset( XmUbNoteBookWidget, notebook.resize_work_area ),
    XtRImmediate,
    (XtPointer) True
  },
  {
    XmUbNsizeIsWorkArea,
    XmUbCSizeIsWorkArea,
    XtRBoolean, sizeof( Boolean ),
    XtOffset( XmUbNoteBookWidget, notebook.work_area_decides_size ),
    XtRImmediate,
    (XtPointer) True
  },
  {
    XmUbNtabDirBottom,
    XmUbCTabHdirection,
    XmRhorizDir, sizeof( horizDir ),
    XtOffset( XmUbNoteBookWidget, notebook.tab_dir_bottom ),
    XtRImmediate,
    (XtPointer) XmUbLEFT_TO_RIGHT
  },
  {
    XmUbNtabDirLeft,
    XmUbCTabVdirection,
    XmRvertDir, sizeof( vertDir ),
    XtOffset( XmUbNoteBookWidget, notebook.tab_dir_left ),
    XtRImmediate,
    (XtPointer) XmUbTOP_TO_BOTTOM
  },
  {
    XmUbNtabDirRight,
    XmUbCTabVdirection,
    XmRvertDir, sizeof( vertDir ),
    XtOffset( XmUbNoteBookWidget, notebook.tab_dir_right ),
    XtRImmediate,
    (XtPointer) XmUbTOP_TO_BOTTOM
  },
  {
    XmUbNtabDirTop,
    XmUbCTabHdirection,
    XmRhorizDir, sizeof( horizDir ),
    XtOffset( XmUbNoteBookWidget, notebook.tab_dir_top ),
    XtRImmediate,
    (XtPointer) XmUbLEFT_TO_RIGHT
  },
  {
    XmUbNtabSpacingBottom,
    XmUbCTabSpacing,
    XtRInt, sizeof( int ),
    XtOffset( XmUbNoteBookWidget, notebook.tab_spacing_bottom ),
    XtRImmediate,
    (XtPointer) 2
  },
  {
    XmUbNtabSpacingLeft,
    XmUbCTabSpacing,
    XtRInt, sizeof( int ),
    XtOffset( XmUbNoteBookWidget, notebook.tab_spacing_left ),
    XtRImmediate,
    (XtPointer) 2
  },
  {
    XmUbNtabSpacingRight,
    XmUbCTabSpacing,
    XtRInt, sizeof( int ),
    XtOffset( XmUbNoteBookWidget, notebook.tab_spacing_right ),
    XtRImmediate,
    (XtPointer) 2
  },
  {
    XmUbNtabSpacingTop,
    XmUbCTabSpacing,
    XtRInt, sizeof( int ),
    XtOffset( XmUbNoteBookWidget, notebook.tab_spacing_top ),
    XtRImmediate,
    (XtPointer) 2
  },
  {
    XmUbNscrollOffsetBottom,
    XmUbCScrollOffset,
    XtRInt, sizeof( int ),
    XtOffset( XmUbNoteBookWidget, notebook.scroll_offset_bottom ),
    XtRImmediate,
    (XtPointer) 0
  },
  {
    XmUbNscrollOffsetLeft,
    XmUbCScrollOffset,
    XtRInt, sizeof( int ),
    XtOffset( XmUbNoteBookWidget, notebook.scroll_offset_left ),
    XtRImmediate,
    (XtPointer) 0
  },
  {
    XmUbNscrollOffsetRight,
    XmUbCScrollOffset,
    XtRInt, sizeof( int ),
    XtOffset( XmUbNoteBookWidget, notebook.scroll_offset_right ),
    XtRImmediate,
    (XtPointer) 0
  },
  {
    XmUbNscrollOffsetTop,
    XmUbCScrollOffset,
    XtRInt, sizeof( int ),
    XtOffset( XmUbNoteBookWidget, notebook.scroll_offset_top ),
    XtRImmediate,
    (XtPointer) 0
  },
  {
    XmUbNworkArea,
    XmUbCWorkArea,
    XtRWindow, sizeof( Widget ),
    XtOffset( XmUbNoteBookWidget, notebook.work_area ),
    XtRImmediate,
    (XtPointer) NULL
  },

  /* Internal resources. */

  {
    XmUbNinternalKidRepos,
    XmUbNinternalKidRepos,
    XtRBoolean, sizeof( Boolean ),
    XtOffset( XmUbNoteBookWidget, notebook._internal_repos_kids ),
    XtRImmediate,
    (XtPointer) False
  },
};  /* resources */


/* The constraint resource list. */
static XtResource notebookConstraintResources[] = {
  {
    XmUbNchildType,
    XmUbCChildType,
    XmRchildType, sizeof( childType ),
    XtOffset( XmUbNoteBookConstraints, notebook.child_type ),
    XtRImmediate,
    (XtPointer) XmUbTAB
  },
  {
    XmUbNposition,
    XmUbCPosition,
    XmRsideAttachment, sizeof( sideAttachment ),
    XtOffset( XmUbNoteBookConstraints, notebook.side ),
    XtRImmediate,
    (XtPointer) XmUbBOTTOM
  },
  {
    XmUbNscrollDirection,
    XmUbCScrollDirection,
    XmRscrollDirection, sizeof( scrollDirection ),
    XtOffset( XmUbNoteBookConstraints, notebook.scroll_direction ),
    XtRImmediate,
    (XtPointer) XmUbSCROLL_FORWARD
  },

};  /* notebookConstraintResources */



/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

/* Core methods. */
static void              
  ClassInitialize();

static void
  ChangeManaged( Widget  widget );

static void 
  ConstraintInitialize( Widget     req,
                        Widget     new,
                        ArgList    args,
                        Cardinal   *num_args );

static Boolean
  ConstraintSetValues( Widget     current,
                       Widget     request,
                       Widget     new,
                       ArgList    args,
                       Cardinal   *num_args );

static void 
  Destroy( Widget   widget );

static void
  ExposeIt( Widget  widget,
            XEvent  *ev,
            Region  region );

static XtGeometryResult
  GeometryManager( Widget            widget,
                   XtWidgetGeometry  *request,
                   XtWidgetGeometry  *reply );

static void
  Initialize( Widget     treq,
              Widget     tnew,
              ArgList    args,
              Cardinal   *num_args );

static XtGeometryResult
  QueryGeometry( Widget             widget,
                 XtWidgetGeometry  *proposed,
                 XtWidgetGeometry  *answer );

static void
  Resize( Widget    widget );

static Boolean 
  SetValues( Widget     current,
             Widget     request,
             Widget     new,
             ArgList    args,
             Cardinal   *num_args );


/* These core methods are not needed. 
*
*  static void 
*    Realize( Widget               widget,
*             XtValueMask          mask,
*             XSetWindowAttributes *attributes );
*
*/

/* Internal functions. */

/*
  BindingDimensions :
    Parameter pixmap_dimensions must be the address of an array with 3 elements
    or NULL. If not NULL, array is updated with the dimensions of the 
    different pixmaps.
*/
static void
  BindingDimensions( XmUbNoteBookWidget  w,
                     Dimension           *binding_width,
                     Dimension           *binding_height,
                     DimensionDescrRef   pixmap_dimensions );

static void
  CheckRedrawArea( XmUbNoteBookWidget  w,
                   XExposeEvent        *event,
                   RedrawRef           redraw );
                   

static void
  ConvertToBindingType( XrmValue   *args,
                        Cardinal   *num_args,
                        XrmValue   *from,
                        XrmValue   *to );

static void
  ConvertToChildType( XrmValue   *args,
                      Cardinal   *num_args,
                      XrmValue   *from,
                      XrmValue   *to );

static void
  ConvertToCoveredPagesPos( XrmValue   *args,
                            Cardinal   *num_args,
                            XrmValue   *from,
                            XrmValue   *to );

static void
  ConvertToHorizontalDirection( XrmValue   *args,
                                Cardinal   *num_args,
                                XrmValue   *from,
                                XrmValue   *to );

static void
  ConvertToSideAttachment( XrmValue   *args,
                           Cardinal   *num_args,
                           XrmValue   *from,
                           XrmValue   *to );

static void
  ConvertToScrollDirection( XrmValue   *args,
                            Cardinal   *num_args,
                            XrmValue   *from,
                            XrmValue   *to );

static void
  ConvertToTabPacking( XrmValue   *args,
                       Cardinal   *num_args,
                       XrmValue   *from,
                       XrmValue   *to );

static void
  ConvertToVerticalDirection( XrmValue   *args,
                              Cardinal   *num_args,
                              XrmValue   *from,
                              XrmValue   *to );

/* DoLayout bases its calculations on the width and height set in the core
   part of w. */
static void
  DoLayout( XmUbNoteBookWidget  w,
            NoteBookSizesRef    preferred_sizes );

static void
  DrawBinding( XmUbNoteBookWidget  w );

static void
  DrawBookPages( XmUbNoteBookWidget  w,
                 int                 side );

static void
  DrawFrontPageFrame( XmUbNoteBookWidget  w,
                      int                 side );

static void
  GetPreferredSize( XmUbNoteBookWidget  w,
                    Widget              initiator,
                    XtWidgetGeometry    *request,
                    NoteBookSizesRef    preferred_sizes );

static Widget
  GetScrollButton( XmUbNoteBookWidget  w,
                   sideAttachment      side,
                   scrollDirection     direction,
                   Boolean             must_be_managed );

static void 
  KidsPreferredGeometry( Widget            kid,
                         Widget            initiator,
                         XtWidgetGeometry  *request,
                         XtWidgetGeometry  *desired );

static void
  LayoutPositions( XmUbNoteBookWidget  w,
                   NoteBookSizesRef    preferred_sizes,
                   double              tab_x_pos[],
                   double              tab_y_pos[],
                   Position            tab_allowance[],
                   Position            back_scroll_x_pos[],
                   Position            back_scroll_y_pos[],
                   Position            forw_scroll_x_pos[],
                   Position            forw_scroll_y_pos[] );

static void
  PreferredOwnDimensions( XmUbNoteBookWidget  w,
                          NoteBookSizesRef    preferred_sizes,
                          Dimension           *pref_width,
                          Dimension           *pref_height );

/* ResizeIfNeeded returns True if the layout has been calculated and the 
   positions and sizes of the children has been set. */
static Boolean
  ResizeIfNeeded( XmUbNoteBookWidget  w,
                  NoteBookSizesRef    preferred_sizes );

static void
  SetBindingPixmap( XmUbNoteBookWidget  w,
                    bindingType         old_binding_type );

/* The GC routines GC free old if non-NULL. */
static void
  SetLineDrawGC( XmUbNoteBookWidget  w );

static void
  SetFrontPageGC( XmUbNoteBookWidget  w );

static void
  SetCoveredPageGC( XmUbNoteBookWidget  w );

static int
  TabNumber( XmUbNoteBookWidget  w,
             Widget              tab,
             sideAttachment      side );



/*----------------------------------------------------------------------------
--  Initialization of the class record.
----------------------------------------------------------------------------*/

/* This initialization has to be done after the methods have been declared. */
XmUbNoteBookClassRec xmUbNoteBookClassRec = {

  { /* Core class fields. */
    /* superclass */                    (WidgetClass) &xmManagerClassRec,
    /* class_name */                    "XmUbNoteBook",
    /* widget_size */                   sizeof( XmUbNoteBookRec ),
    /* class_initialize */              ClassInitialize,
    /* class_part_initialize */         NULL,
    /* class_inited */                  False,
    /* initialize */                    Initialize,
    /* initialize_hook */               NULL,
    /* realize */                       XtInheritRealize, 
    /* actions */                       NULL,
    /* num_actions */                   0,
    /* resources */                     resources,
    /* num_resources */                 XtNumber( resources ),
    /* xrm_class */                     NULLQUARK,
    /* compress_motion */               True,
    /* compress_exposure */             True,
    /* compress_enterleave */           True,
    /* visible_interest */              False,
    /* destroy */                       Destroy,
    /* resize */                        Resize,
    /* expose */                        ExposeIt,
    /* set_values */                    SetValues,
    /* set_values_hook */               NULL,
    /* set_values_almost */             XtInheritSetValuesAlmost,
    /* get_values_hook */               NULL,
    /* accept_focus */                  NULL,
    /* version */                       XtVersion,
    /* callback_private */              NULL,
    /* tm_table */                      NULL,
    /* query_geometry */                QueryGeometry,
    /* display_accelerator */           XtInheritDisplayAccelerator,
    /* extension */                     NULL
  },
  { /* Composite class part. */
    /* geometry_manager */              GeometryManager,
    /* change_managed */                ChangeManaged,
    /* insert_child */                  XtInheritInsertChild,
    /* delete_child */                  XtInheritDeleteChild,
    /* extension */                     NULL
  },
  { /* Constraint class fields. */
    /* subresources */                  notebookConstraintResources,
    /* subresource_count */             XtNumber( 
                                          notebookConstraintResources ),
    /* constraint_size */               sizeof( XmUbNoteBookConstraintsRec ),
    /* initialize */                    ConstraintInitialize,
    /* destroy */                       NULL,
    /* set_values */                    ConstraintSetValues,
    /* extension */                     NULL
  },
  { /* XmManager class part. */
    /* translations */                  NULL,
    /* get_resources */                 NULL,
    /* num_get_resources */             0,
    /* get_constraint_resources */      NULL,
    /* num_get_constraint_resources */  0,
    /* extension */                     NULL
  },
  { /* NoteBook class part. */
    /* extension */                     NULL
  },

}; /* xmUbMgrClassRec */



/* Class record pointer. */
WidgetClass 
  xmUbNoteBookWidgetClass = (WidgetClass) &xmUbNoteBookClassRec;



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/


static void
  BindingDimensions( XmUbNoteBookWidget  w,
                     Dimension           *binding_width,
                     Dimension           *binding_height,
                     DimensionDescrRef   pixmap_dimensions )
{

  /* Variables. */
  unsigned int  center_width;
  unsigned int  center_height;
  int           dummy_int;
  unsigned int  dummy_uint;
  Window        dummy_window;
  unsigned int  end_width;
  unsigned int  end_height;
  int           index;
  unsigned int  start_width;
  unsigned int  start_height;


  /* Code. */

  /* Initialize. */
  *binding_width  = 0;
  *binding_height = 0;

  if( pixmap_dimensions != NULL ){
    for( index = 0; index < 3; index ++ ){
      pixmap_dimensions[ index ].width  = 0;
      pixmap_dimensions[ index ].height = 0;
    }
  }

  /* Return the max of center/start/end for the dimension where the pixmap is
     drawn, and the sum in the other dimension. */

  switch( w -> notebook.binding_type ){

    case XmUbBINDING_NONE:
      return;

    case XmUbBINDING_PIXMAP:
      /* The pixmap must be set in the widget structure. */
      /* If not, it has not yet been created. */

      if( w -> notebook.binding_pixmap_center != None )
        XGetGeometry( XtDisplay( (Widget) w ), 
                      w -> notebook.binding_pixmap_center,
                      &dummy_window, &dummy_int, &dummy_int,
                      &center_width, &center_height, 
                      &dummy_uint, &dummy_uint );
      else {
        center_width  = 0;
        center_height = 0;
      }

      if( w -> notebook.binding_pixmap_start != None )
        XGetGeometry( XtDisplay( (Widget) w ), 
                      w -> notebook.binding_pixmap_start,
                      &dummy_window, &dummy_int, &dummy_int,
                      &start_width, &start_height, 
                      &dummy_uint, &dummy_uint );
      else {
        start_width  = 0;
        start_height = 0;
      }

      if( w -> notebook.binding_pixmap_end != None )
        XGetGeometry( XtDisplay( (Widget) w ), 
                      w -> notebook.binding_pixmap_end,
                      &dummy_window, &dummy_int, &dummy_int,
                      &end_width, &end_height, 
                      &dummy_uint, &dummy_uint );
      else {
        end_width  = 0;
        end_height = 0;
      }

      switch( w -> notebook.binding_side ){
        case XmUbTOP:
        case XmUbBOTTOM:

          *binding_width = 
            (Dimension) ( start_width + center_width + end_width );
          *binding_height =
            (Dimension) MAX( start_height, MAX( center_height, end_height ));

          break;

        case XmUbLEFT:
        case XmUbRIGHT:

          *binding_width = 
            (Dimension) MAX( start_width, MAX( center_width, end_width ));
          *binding_height =
            (Dimension) ( start_height + center_height + end_height );

          break;
      }

      if( pixmap_dimensions != NULL ){
        pixmap_dimensions[ START_PIXMAP_INDEX ].width  = (int) start_width;
        pixmap_dimensions[ START_PIXMAP_INDEX ].height = (int) start_height;

        pixmap_dimensions[ CENTER_PIXMAP_INDEX ].width  = (int) center_width;
        pixmap_dimensions[ CENTER_PIXMAP_INDEX ].height = (int) center_height;

        pixmap_dimensions[ END_PIXMAP_INDEX ].width  = (int) end_width;
        pixmap_dimensions[ END_PIXMAP_INDEX ].height = (int) end_height;
      }


      break;

    case XmUbBINDING_SOLID:
      *binding_width = 
        predef_solid_bitmaps[ w -> notebook.binding_side ].width;
      *binding_height = 
        predef_solid_bitmaps[ w -> notebook.binding_side ].height;


      if( pixmap_dimensions != NULL ){
        pixmap_dimensions[ CENTER_PIXMAP_INDEX ].width  = (int) *binding_width;
        pixmap_dimensions[ CENTER_PIXMAP_INDEX ].height = 
          (int) *binding_height;
      }

      break;

    case XmUbBINDING_SPIRAL:
      *binding_width = 
        predef_spiral_bitmaps[ w -> notebook.binding_side ].width;
      *binding_height = 
        predef_spiral_bitmaps[ w -> notebook.binding_side ].height;

      if( pixmap_dimensions != NULL ){
        pixmap_dimensions[ CENTER_PIXMAP_INDEX ].width  = (int) *binding_width;
        pixmap_dimensions[ CENTER_PIXMAP_INDEX ].height = 
          (int) *binding_height;
      }

      break;

  }

  
  return;

} /* BindingDimensions */


/*----------------------------------------------------------------------*/

static void
  CheckRedrawArea( XmUbNoteBookWidget  w,
                   XExposeEvent        *event,
                   RedrawRef           redraw )
{
  /* Variables. */
  int  index;

  /* Code. */

  /* Initialize. */
  redraw -> binding    = False;
  redraw -> front_area = False;
  for( index = 0; index < 4; index ++ )
    redraw -> book_side[ index ] = False;

  /* For each component, check if it must be redrawn. */

  if( w -> notebook.binding_type != XmUbBINDING_NONE ){

    switch( w -> notebook.binding_side ){
      case XmUbTOP:
        if( ( (Position) event -> y < w -> notebook.front_page_top_pos ) &&
            ( (Position) event -> x < w -> notebook.front_page_right_pos ) &&
            ( (Position) event -> x + (Position) event -> width > 
              w -> notebook.front_page_left_pos ) )
          redraw -> binding = True;

        break;

      case XmUbBOTTOM:
        if( ( (Position) event -> y + (Position) event -> height >
              w -> notebook.front_page_bottom_pos ) &&
            ( (Position) event -> x < w -> notebook.front_page_right_pos ) &&
            ( (Position) event -> x + (Position) event -> width > 
              w -> notebook.front_page_left_pos ) )
          redraw -> binding = True;

        break;

      case XmUbLEFT:
        if( ( (Position) event -> x < w -> notebook.front_page_left_pos ) &&
            ( (Position) event -> y < w -> notebook.front_page_bottom_pos ) &&
            ( (Position) event -> y + (Position) event -> height > 
              w -> notebook.front_page_top_pos ) )
          redraw -> binding = True;

        break;

      case XmUbRIGHT:
        if( ( (Position) event -> x + (Position) event -> width >
               w -> notebook.front_page_right_pos ) &&
            ( (Position) event -> y < w -> notebook.front_page_bottom_pos ) &&
            ( (Position) event -> y + (Position) event -> height > 
              w -> notebook.front_page_top_pos ) )
          redraw -> binding = True;

        break;
    }
  }

  /* Book sides. */

  switch( w -> notebook.book_pages_sides ){

    case XmTOP_LEFT:
      if( ( (Position) event -> y < w -> notebook.front_page_top_pos ) &&
          ( (Position) event -> y + (Position) event -> height > 
            w -> notebook.front_page_top_pos -
            (Position) w -> notebook.book_side_extent ) &&
          ( (Position) event -> x < w -> notebook.front_page_right_pos ) &&
          ( (Position) event -> x + (Position) event -> width >
            w -> notebook.front_page_left_pos - 
            (Position) w -> notebook.book_side_extent ) )
        redraw -> book_side[ XmUbTOP ] = True;

      if( ( (Position) event -> x < w -> notebook.front_page_left_pos ) &&
          ( (Position) event -> x + (Position) event -> width >
            w -> notebook.front_page_left_pos - 
            (Position) w -> notebook.book_side_extent ) &&
          ( (Position) event -> y < w -> notebook.front_page_bottom_pos ) &&
          ( (Position) event -> y + (Position) event -> height >
            w -> notebook.front_page_top_pos - 
            (Position) w -> notebook.book_side_extent ) )
        redraw -> book_side[ XmUbLEFT ] = True;

      break;

    case XmTOP_RIGHT:
      if( ( (Position) event -> y < w -> notebook.front_page_top_pos ) &&
          ( (Position) event -> y + (Position) event -> height > 
            w -> notebook.front_page_top_pos -
            (Position) w -> notebook.book_side_extent ) &&
          ( (Position) event -> x < w -> notebook.front_page_right_pos +
            (Position) w -> notebook.book_side_extent ) &&
          ( (Position) event -> x + (Position) event -> width >
            w -> notebook.front_page_left_pos ) )
        redraw -> book_side[ XmUbTOP ] = True;

      if( ( (Position) event -> x + (Position) event -> width >
             w -> notebook.front_page_right_pos ) &&
          ( (Position) event -> x < w -> notebook.front_page_right_pos +
            (Position) w -> notebook.book_side_extent ) &&
          ( (Position) event -> y < w -> notebook.front_page_bottom_pos ) &&
          ( (Position) event -> y + (Position) event -> height >
            w -> notebook.front_page_top_pos - 
            (Position) w -> notebook.book_side_extent ) )
        redraw -> book_side[ XmUbRIGHT ] = True;

      break;

    case XmBOTTOM_LEFT:
      if( ( (Position) event -> y + (Position) event -> height >
            w -> notebook.front_page_bottom_pos ) &&
          ( (Position) event -> y < w -> notebook.front_page_bottom_pos +
            (Position) w -> notebook.book_side_extent ) &&
          ( (Position) event -> x < w -> notebook.front_page_right_pos ) &&
          ( (Position) event -> x + (Position) event -> width >
            w -> notebook.front_page_left_pos - 
            (Position) w -> notebook.book_side_extent ) )
        redraw -> book_side[ XmUbBOTTOM ] = True;

      if( ( (Position) event -> x < w -> notebook.front_page_left_pos ) &&
          ( (Position) event -> x + (Position) event -> width >
            w -> notebook.front_page_left_pos - 
            (Position) w -> notebook.book_side_extent ) &&
          ( (Position) event -> y < w -> notebook.front_page_bottom_pos +
            (Position) w -> notebook.book_side_extent ) &&
          ( (Position) event -> y + (Position) event -> height >
            w -> notebook.front_page_top_pos - 
            (Position) w -> notebook.book_side_extent ) )
        redraw -> book_side[ XmUbLEFT ] = True;

      break;

    case XmBOTTOM_RIGHT:
      if( ( (Position) event -> y + (Position) event -> height >
            w -> notebook.front_page_bottom_pos ) &&
          ( (Position) event -> y < w -> notebook.front_page_bottom_pos +
            (Position) w -> notebook.book_side_extent ) &&
          ( (Position) event -> x < w -> notebook.front_page_right_pos +
            (Position) w -> notebook.book_side_extent ) &&
          ( (Position) event -> x + (Position) event -> width >
            w -> notebook.front_page_left_pos ) )
        redraw -> book_side[ XmUbBOTTOM ] = True;


      if( ( (Position) event -> x + (Position) event -> width >
             w -> notebook.front_page_right_pos ) &&
          ( (Position) event -> x < w -> notebook.front_page_right_pos +
            (Position) w -> notebook.book_side_extent ) &&
          ( (Position) event -> y < w -> notebook.front_page_bottom_pos +
            (Position) w -> notebook.book_side_extent ) &&
          ( (Position) event -> y + (Position) event -> height >
            w -> notebook.front_page_top_pos ) )
        redraw -> book_side[ XmUbRIGHT ] = True;

      break;
  }

  /* Front area. */

  if( ( (Position) event -> x < w -> notebook.front_page_right_pos ) &&
      ( (Position) event -> x + (Position) event -> width <
        w -> notebook.front_page_left_pos ) &&
      ( (Position) event -> y < w -> notebook.front_page_bottom_pos ) &&
      ( (Position) event -> y + (Position) event -> height >
        w -> notebook.front_page_top_pos ) )
    redraw -> front_area = True;


  return;

} /* CheckRedrawArea */


/*----------------------------------------------------------------------*/

static void              
  ClassInitialize()
{

  /* Code. */

  /* Set bitmap addresses and dimensions in predefined arrays. */
  predef_spiral_bitmaps[ XmUbTOP ].width  = XmUbNoteBkSpiralTop_width;
  predef_spiral_bitmaps[ XmUbTOP ].height = XmUbNoteBkSpiralTop_height;
  predef_spiral_bitmaps[ XmUbTOP ].bits   = XmUbNoteBkSpiralTop_bits;

  predef_spiral_bitmaps[ XmUbBOTTOM ].width  = XmUbNoteBkSpiralBottom_width;
  predef_spiral_bitmaps[ XmUbBOTTOM ].height = XmUbNoteBkSpiralBottom_height;
  predef_spiral_bitmaps[ XmUbBOTTOM ].bits   = XmUbNoteBkSpiralBottom_bits;
 
  predef_spiral_bitmaps[ XmUbLEFT ].width  = XmUbNoteBkSpiralLeft_width;
  predef_spiral_bitmaps[ XmUbLEFT ].height = XmUbNoteBkSpiralLeft_height;
  predef_spiral_bitmaps[ XmUbLEFT ].bits   = XmUbNoteBkSpiralLeft_bits;

  predef_spiral_bitmaps[ XmUbRIGHT ].width  = XmUbNoteBkSpiralRight_width;
  predef_spiral_bitmaps[ XmUbRIGHT ].height = XmUbNoteBkSpiralRight_height;
  predef_spiral_bitmaps[ XmUbRIGHT ].bits   = XmUbNoteBkSpiralRight_bits;


  predef_solid_bitmaps[ XmUbTOP ].width  = XmUbNoteBkSolidTop_width;
  predef_solid_bitmaps[ XmUbTOP ].height = XmUbNoteBkSolidTop_height;
  predef_solid_bitmaps[ XmUbTOP ].bits   = XmUbNoteBkSolidTop_bits;

  predef_solid_bitmaps[ XmUbBOTTOM ].width  = XmUbNoteBkSolidBottom_width;
  predef_solid_bitmaps[ XmUbBOTTOM ].height = XmUbNoteBkSolidBottom_height;
  predef_solid_bitmaps[ XmUbBOTTOM ].bits   = XmUbNoteBkSolidBottom_bits;
 
  predef_solid_bitmaps[ XmUbLEFT ].width  = XmUbNoteBkSolidLeft_width;
  predef_solid_bitmaps[ XmUbLEFT ].height = XmUbNoteBkSolidLeft_height;
  predef_solid_bitmaps[ XmUbLEFT ].bits   = XmUbNoteBkSolidLeft_bits;

  predef_solid_bitmaps[ XmUbRIGHT ].width  = XmUbNoteBkSolidRight_width;
  predef_solid_bitmaps[ XmUbRIGHT ].height = XmUbNoteBkSolidRight_height;
  predef_solid_bitmaps[ XmUbRIGHT ].bits   = XmUbNoteBkSolidRight_bits;



  /* Convert quarks for comparison. */
  quark_bottom = XrmStringToQuark( "BOTTOM" );
  quark_left   = XrmStringToQuark( "LEFT" );
  quark_right  = XrmStringToQuark( "RIGHT" );
  quark_top    = XrmStringToQuark( "TOP" );


  /* Register type converters. */
  XtAddConverter( XmRString, XmRbindingType, ConvertToBindingType, NULL, 0 );

  XtAddConverter( XmRString, XmRchildType, ConvertToChildType, NULL, 0 );

  XtAddConverter( XmRString, XmRcoveredPagesPos, ConvertToCoveredPagesPos,
                  NULL, 0 );

  XtAddConverter( XmRString, XmRhorizDir, ConvertToHorizontalDirection,
                  NULL, 0 );

  XtAddConverter( XmRString, XmRscrollDirection, ConvertToScrollDirection,
                  NULL, 0 );

  XtAddConverter( XmRString, XmRsideAttachment, ConvertToSideAttachment,
                  NULL, 0 );

  XtAddConverter( XmRString, XmRtabPacking, ConvertToTabPacking, NULL, 0 );

  XtAddConverter( XmRString, XmRvertDir, ConvertToVerticalDirection,
                  NULL, 0 );

  return;

}


/*----------------------------------------------------------------------*/

static void
  ChangeManaged( Widget  widget )
{
  /* Variables. */
  int                 index;
  Boolean             layout_done = False;
  NoteBookSizes       preferred_sizes;
  XmUbNoteBookWidget  w;


  /* Code. */

  w = (XmUbNoteBookWidget) widget;

  w -> notebook.internal_resize = True;

  /* We may have to restack the tabs. */
  for( index = 0; index < 4; index ++ )
    w -> notebook.unstacked_tabs[ index ] = True;

  /* Find out the preferred sizes of the kids. */
  GetPreferredSize( w, NULL, NULL, &preferred_sizes );

  /* Check if we need to resize. */
  if( w -> notebook.resize_allowed )
    layout_done = ResizeIfNeeded( w, &preferred_sizes );

  /* Do the layout. */
  if( !layout_done )
    DoLayout( w, &preferred_sizes );

  w -> notebook.internal_resize = False;


  return;

} /* ChangeManaged */


/*----------------------------------------------------------------------*/

static void 
  ConstraintInitialize( Widget     req,
                        Widget     new,
                        ArgList    args,
                        Cardinal   *num_args )
{

  /* Variables. */
  XmUbNoteBookConstraints  constraints;
  XmUbNoteBookWidget       w;


  /* Code. */

  /* Make sure the work area resources are correct. */
  constraints = (XmUbNoteBookConstraints) new -> core.constraints;
  w = (XmUbNoteBookWidget) XtParent( new );

  if( ( w -> notebook.work_area == NULL ) &&
      ( constraints -> notebook.child_type == XmUbWORK_AREA ) )
    w -> notebook.work_area = new;

  else if( w -> notebook.work_area == new )
    constraints -> notebook.child_type = XmUbWORK_AREA;


  return;

} /* ConstraintInitialize */


/*----------------------------------------------------------------------*/

static Boolean
  ConstraintSetValues( Widget     current,
                       Widget     request,
                       Widget     new,
                       ArgList    args,
                       Cardinal   *num_args )
{
  /* Variables. */
  XmUbNoteBookConstraints  new_ctr;
  XmUbNoteBookConstraints  old_ctr;
  XmUbNoteBookWidget       w;

  /* Code. */

  w = (XmUbNoteBookWidget) XtParent( current );

  new_ctr = (XmUbNoteBookConstraints) new -> core.constraints;
  old_ctr = (XmUbNoteBookConstraints) current -> core.constraints;

  /* If the side was changed, we will have to restack the tabs for both the
     old and the new side. */
  if( new_ctr -> notebook.side != old_ctr -> notebook.side ){
    w -> notebook.unstacked_tabs[ new_ctr -> notebook.side ] = True;
    w -> notebook.unstacked_tabs[ old_ctr -> notebook.side ] = True;
  }


  /* DoLayout ? */


  /* Everything we set here must lead to a redisplay. */

  return True;

}  /* ConstraintSetValues */


/*----------------------------------------------------------------------*/

static void
  ConvertToBindingType( XrmValue   *args,
                        Cardinal   *num_args,
                        XrmValue   *from,
                        XrmValue   *to )
{

  /* Variables. */
  static bindingType  conv_binding_type;


  /* Code. */

  if( *num_args != 0 )
    XtWarningMsg( "wrongParameters", "ConvertToBindingType", 
      "XtToolkitError", "Conversion needs no extra arguments", 
      (String *) NULL, (Cardinal *) NULL );
 
  if( strcmp( (char *) from -> addr, "BINDING_NONE" ) == 0 )
    conv_binding_type = XmUbBINDING_NONE;

  else if( strcmp( (char *) from -> addr, "BINDING_PIXMAP" ) == 0 )
    conv_binding_type = XmUbBINDING_PIXMAP;

  else if( strcmp( (char *) from -> addr, "BINDING_SOLID" ) == 0 )
    conv_binding_type = XmUbBINDING_SOLID;

  else if( strcmp( (char *) from -> addr, "BINDING_SPIRAL" ) == 0 )
    conv_binding_type = XmUbBINDING_SPIRAL;

  else {
    XtWarningMsg( "wrongParameters", "ConvertToBindingType", 
      "XtToolkitError", "Cannot convert to type bindingType",
      (String *) NULL, (Cardinal *) 0 );
    conv_binding_type = XmUbBINDING_SPIRAL;
  }

  ( *to ).size = sizeof( bindingType );
  ( *to ).addr = (XtPointer) &conv_binding_type;


  return;

} /* ConvertToBindingType */


/*----------------------------------------------------------------------*/

static void
  ConvertToChildType( XrmValue   *args,
                      Cardinal   *num_args,
                      XrmValue   *from,
                      XrmValue   *to )
{

  /* Variables. */
  static childType  conv_child_type;


  /* Code. */

  if( *num_args != 0 )
    XtWarningMsg( "wrongParameters", "ConvertToChildType", 
      "XtToolkitError", "Conversion needs no extra arguments", 
      (String *) NULL, (Cardinal *) NULL );
 
  if( strcmp( (char *) from -> addr, "TAB" ) == 0 )
    conv_child_type = XmUbTAB;

  else if( strcmp( (char *) from -> addr, "SCROLL_BUTTON" ) == 0 )
    conv_child_type = XmUbSCROLL_BUTTON;

  else if( strcmp( (char *) from -> addr, "WORK_AREA" ) == 0 )
    conv_child_type = XmUbWORK_AREA;

  else {
    XtWarningMsg( "wrongParameters", "ConvertToChildType", 
      "XtToolkitError", "Cannot convert to type childType",
      (String *) NULL, (Cardinal *) 0 );
    conv_child_type = XmUbTAB;
  }

  ( *to ).size = sizeof( childType );
  ( *to ).addr = (XtPointer) &conv_child_type;


  return;

} /* ConvertToChildType */


/*----------------------------------------------------------------------*/

static void
  ConvertToCoveredPagesPos( XrmValue   *args,
                            Cardinal   *num_args,
                            XrmValue   *from,
                            XrmValue   *to )
{

  /* Variables. */
  static coveredPagesPos  conv_covered_pages_pos;


  /* Code. */

  if( *num_args != 0 )
    XtWarningMsg( "wrongParameters", "ConvertToCoveredPagesPos", 
      "XtToolkitError", "Conversion needs no extra arguments", 
      (String *) NULL, (Cardinal *) NULL );
 
  if( strcmp( (char *) from -> addr, "BOTTOM_RIGHT" ) == 0 )
    conv_covered_pages_pos = XmBOTTOM_RIGHT;

  else if( strcmp( (char *) from -> addr, "TOP_RIGHT" ) == 0 )
    conv_covered_pages_pos = XmTOP_RIGHT;

  else if( strcmp( (char *) from -> addr, "TOP_LEFT" ) == 0 )
    conv_covered_pages_pos = XmTOP_LEFT;

  else if( strcmp( (char *) from -> addr, "BOTTOM_LEFT" ) == 0 )
    conv_covered_pages_pos = XmBOTTOM_LEFT;


  else {
    XtWarningMsg( "wrongParameters", "ConvertToCoveredPagesPos", 
      "XtToolkitError", "Cannot convert to type coveredPagesPos",
      (String *) NULL, (Cardinal *) 0 );
    conv_covered_pages_pos = XmBOTTOM_RIGHT;
  }

  ( *to ).size = sizeof( coveredPagesPos );
  ( *to ).addr = (XtPointer) &conv_covered_pages_pos;


  return;

} /* ConvertToCoveredPagesPos */


/*----------------------------------------------------------------------*/

static void
  ConvertToHorizontalDirection( XrmValue   *args,
				Cardinal   *num_args,
				XrmValue   *from,
				XrmValue   *to )
{

  /* Variables. */
  static horizDir  conv_horizontal_direction;


  /* Code. */

  if( *num_args != 0 )
    XtWarningMsg( "wrongParameters", "ConvertToHorizontalDirection", 
      "XtToolkitError", "Conversion needs no extra arguments", 
      (String *) NULL, (Cardinal *) NULL );
 
  if( strcmp( (char *) from -> addr, "LEFT_TO_RIGHT" ) == 0 )
    conv_horizontal_direction = XmUbLEFT_TO_RIGHT;

  else if( strcmp( (char *) from -> addr, "RIGHT_TO_LEFT" ) == 0 )
    conv_horizontal_direction = XmUbRIGHT_TO_LEFT;

  else {
    XtWarningMsg( "wrongParameters", "ConvertToHorizontalDirection", 
      "XtToolkitError", "Cannot convert to type horizDir",
      (String *) NULL, (Cardinal *) 0 );
    conv_horizontal_direction = XmUbLEFT_TO_RIGHT;
  }

  ( *to ).size = sizeof( horizDir );
  ( *to ).addr = (XtPointer) &conv_horizontal_direction;


  return;

} /* ConvertToHorizontalDirection */


/*----------------------------------------------------------------------*/

static void
  ConvertToScrollDirection( XrmValue   *args,
                            Cardinal   *num_args,
                            XrmValue   *from,
                            XrmValue   *to )
{

  /* Variables. */
  static scrollDirection  conv_scroll_direction;


  /* Code. */

  if( *num_args != 0 )
    XtWarningMsg( "wrongParameters", "ConvertToScrollDirection", 
      "XtToolkitError", "Conversion needs no extra arguments", 
      (String *) NULL, (Cardinal *) NULL );
 
  if( strcmp( (char *) from -> addr, "SCROLL_FORWARD" ) == 0 )
    conv_scroll_direction = XmUbSCROLL_FORWARD;

  else if( strcmp( (char *) from -> addr, "SCROLL_BACK" ) == 0 )
    conv_scroll_direction = XmUbSCROLL_BACK;

  else {
    XtWarningMsg( "wrongParameters", "ConvertToScrollDirection", 
      "XtToolkitError", "Cannot convert to type scrollDirection",
      (String *) NULL, (Cardinal *) 0 );
    conv_scroll_direction = XmUbSCROLL_FORWARD;
  }

  ( *to ).size = sizeof( scrollDirection );
  ( *to ).addr = (XtPointer) &conv_scroll_direction;


  return;

} /* ConvertToScrollDirection */


/*----------------------------------------------------------------------*/

static void
  ConvertToSideAttachment( XrmValue   *args,
                           Cardinal   *num_args,
                           XrmValue   *from,
                           XrmValue   *to )
{

  /* Variables. */
  XrmQuark               quark_resource;
  static sideAttachment  conv_side_attachment;


  /* Code. */

  if( *num_args != 0 )
    XtWarningMsg( "wrongParameters", "ConvertToSideAttachment", 
      "XtToolkitError", "Conversion needs no extra arguments", 
      (String *) NULL, (Cardinal *) NULL );
 
  /* Convert argument. */
  quark_resource = XrmStringToQuark( from -> addr );

  if( quark_resource == quark_bottom )
    conv_side_attachment = XmUbBOTTOM;

  else if( quark_resource == quark_left )
    conv_side_attachment = XmUbLEFT;

  else if( quark_resource == quark_right )
    conv_side_attachment = XmUbRIGHT;

  else if( quark_resource == quark_top )
    conv_side_attachment = XmUbTOP;

  else {
    XtWarningMsg( "wrongParameters", "ConvertToSideAttachment", 
      "XtToolkitError", "Cannot convert to type sideAttachment",
      (String *) NULL, (Cardinal *) 0 );
    conv_side_attachment = XmUbBOTTOM;
  }

  ( *to ).size = sizeof( sideAttachment );
  ( *to ).addr = (XtPointer) &conv_side_attachment;


  return;

} /* ConvertToSideAttachment */


/*----------------------------------------------------------------------*/

static void
  ConvertToTabPacking( XrmValue   *args,
                       Cardinal   *num_args,
                       XrmValue   *from,
                       XrmValue   *to )
{

  /* Variables. */
  static tabPacking  conv_tab_packing;


  /* Code. */

  if( *num_args != 0 )
    XtWarningMsg( "wrongParameters", "ConvertToTabPacking", 
      "XtToolkitError", "Conversion needs no extra arguments", 
      (String *) NULL, (Cardinal *) NULL );
 
  if( strcmp( (char *) from -> addr, "TAB_PACK_TIGHT" ) == 0 )
    conv_tab_packing = XmUbTAB_PACK_TIGHT;

  else if( strcmp( (char *) from -> addr, "TAB_PACK_COLUMN" ) == 0 )
    conv_tab_packing = XmUbTAB_PACK_COLUMN;

  else if( strcmp( (char *) from -> addr, "TAB_PACK_EQUAL" ) == 0 )
    conv_tab_packing = XmUbTAB_PACK_EQUAL;

  else {
    XtWarningMsg( "wrongParameters", "ConvertToTabPacking", 
      "XtToolkitError", "Cannot convert to type tabPacking",
      (String *) NULL, (Cardinal *) 0 );
    conv_tab_packing = XmUbTAB_PACK_TIGHT;
  }

  ( *to ).size = sizeof( tabPacking );
  ( *to ).addr = (XtPointer) &conv_tab_packing;


  return;

} /* ConvertToTabPacking */


/*----------------------------------------------------------------------*/

static void
  ConvertToVerticalDirection( XrmValue   *args,
			      Cardinal   *num_args,
			      XrmValue   *from,
			      XrmValue   *to )
{

  /* Variables. */
  static vertDir  conv_vertical_direction;


  /* Code. */

  if( *num_args != 0 )
    XtWarningMsg( "wrongParameters", "ConvertToVerticalDirection", 
      "XtToolkitError", "Conversion needs no extra arguments", 
      (String *) NULL, (Cardinal *) NULL );
 
  if( strcmp( (char *) from -> addr, "TOP_TO_BOTTOM" ) == 0 )
    conv_vertical_direction = XmUbTOP_TO_BOTTOM;

  else if( strcmp( (char *) from -> addr, "BOTTOM_TO_TOP" ) == 0 )
    conv_vertical_direction = XmUbBOTTOM_TO_TOP;

  else {
    XtWarningMsg( "wrongParameters", "ConvertToVerticalDirection", 
      "XtToolkitError", "Cannot convert to type vertDir",
      (String *) NULL, (Cardinal *) 0 );
    conv_vertical_direction = XmUbTOP_TO_BOTTOM;
  }

  ( *to ).size = sizeof( vertDir );
  ( *to ).addr = (XtPointer) &conv_vertical_direction;


  return;

} /* ConvertToVerticalDirection */


/*----------------------------------------------------------------------*/

static void 
  Destroy( Widget   widget )
{

  /* Variables. */
  XmUbNoteBookWidget  w;


  /* Code. */

  w = (XmUbNoteBookWidget) widget;

  /* Free GC's. */
  XtReleaseGC( widget, w -> notebook.covered_page_gc );
  XtReleaseGC( widget, w -> notebook.front_page_gc );
  XtReleaseGC( widget, w -> notebook.line_draw_gc );

  /* Since we allocate our own pixmaps for solid & spiral we should free it. */
  switch( w -> notebook.binding_type ){
    case XmUbBINDING_SOLID:
    case XmUbBINDING_SPIRAL:
      if( w -> notebook.binding_pixmap_center != None )
        XFreePixmap( XtDisplay( (Widget) w ), 
                     w -> notebook.binding_pixmap_center );

      break;

    default:
      break;
  }


  return;

} /* Destroy */


/*----------------------------------------------------------------------*/

static void
  DoLayout( XmUbNoteBookWidget  w,
            NoteBookSizesRef    preferred_sizes )
{

  /* NOTE:
     The variable positions that are set in the arrays are for the top and
     left corner if "western" layout, and for the right and bottom corner
     if the other layout. */

  /* Variables. */
  Position                 back_scroll_x_pos[ 4 ];
  Position                 back_scroll_y_pos[ 4 ];
  XmUbNoteBookConstraints  constraints;
  Dimension                decoration_offset[ 4 ];
  XtWidgetGeometry         desired;
  Position                 forw_scroll_x_pos[ 4 ];
  Position                 forw_scroll_y_pos[ 4 ];
  Dimension                height = 0;
  int                      index;
  Widget                   kid;
  Window                   last_sibling[ 4 ];
  unsigned int             mask;
  Position                 offset;
  int                      selected_tab_num;
  sideAttachment           side;
  Position                 tab_allowance[ 4 ];
  Dimension                tab_height[ 4 ];
  int                      tab_num_offset[ 4 ];
  double                   tab_pos_incr[ 4 ];
  double                   tab_spacing[ 4 ];
  Dimension                tab_width[ 4 ];
  double                   tab_x_pos[ 4 ];
  double                   tab_y_pos[ 4 ];
  Dimension                tmp_dimension;
  Dimension                wa_height;
  Dimension                wa_width;
  Dimension                width = 0;
  Position                 x = 0;
  XWindowChanges           xwc;
  Position                 y = 0;

  /* Code. */

  /* Initialize. */
  /* Macro initializations for brain-dead Ultrix compiler. */
  InitArray( decoration_offset, 4, 0 );
  InitArray( tab_allowance, 4, 0 );
  InitArray( tab_height, 4, 0 );
  InitArray( tab_num_offset, 4, 0 );
  InitArray( tab_pos_incr, 4, 0.0 );
  InitArray( tab_spacing, 4, 0.0 );
  InitArray( tab_width, 4, 0 );
  InitArray( last_sibling, 4, None );

  switch( w -> notebook.binding_side ){
    case XmUbTOP:
      decoration_offset[ XmUbTOP ] = preferred_sizes -> binding_height;
      break;

    case XmUbBOTTOM:
      decoration_offset[ XmUbBOTTOM ] = preferred_sizes -> binding_height;
      break;

    case XmUbLEFT:
      decoration_offset[ XmUbLEFT ] = preferred_sizes -> binding_width;
      break;

    case XmUbRIGHT:
      decoration_offset[ XmUbRIGHT ] = preferred_sizes -> binding_width;
      break;
  }

  /* Front area. */
  /* If the front page should be fixed, don't touch those positions, unless
     the resize comes from outside. */
  if( ! ( w -> notebook.front_page_fixed && w -> notebook.internal_resize ) ){

    w -> notebook.front_page_top_pos =  (Position) (
      decoration_offset[ XmUbTOP ] + 
      MAX( preferred_sizes -> tab_sizes[ XmUbTOP ].max_height,
           preferred_sizes -> scroll_sizes[ XmUbTOP ].max_height ) +
      ( ( ( w -> notebook.book_pages_sides == XmTOP_LEFT ) ||
          ( w -> notebook.book_pages_sides == XmTOP_RIGHT ) ) ?
        w -> notebook.book_side_extent : 0 ) );

    w -> notebook.front_page_left_pos = (Position) (
      decoration_offset[ XmUbLEFT ] + 
      MAX( preferred_sizes -> tab_sizes[ XmUbLEFT ].max_width,
           preferred_sizes -> scroll_sizes[ XmUbLEFT ].max_width ) +
      ( ( ( w -> notebook.book_pages_sides == XmTOP_LEFT ) ||
          ( w -> notebook.book_pages_sides == XmBOTTOM_LEFT ) ) ?
        w -> notebook.book_side_extent : 0 ) );
                                        
    tmp_dimension =
      decoration_offset[ XmUbRIGHT ] + 
      MAX( preferred_sizes -> tab_sizes[ XmUbRIGHT ].max_width,
           preferred_sizes -> scroll_sizes[ XmUbRIGHT ].max_width  ) +
      ( ( ( w -> notebook.book_pages_sides == XmTOP_RIGHT ) ||
          ( w -> notebook.book_pages_sides == XmBOTTOM_RIGHT ) ) ?
        w -> notebook.book_side_extent : 0 );

    if( tmp_dimension < w -> core.width - 1 )
      w -> notebook.front_page_right_pos = 
        (Position) ( w -> core.width - tmp_dimension - 1 );
    else
      w -> notebook.front_page_right_pos = 0;


    tmp_dimension =
      decoration_offset[ XmUbBOTTOM ] + 
      MAX( preferred_sizes -> tab_sizes[ XmUbBOTTOM ].max_height,
           preferred_sizes -> scroll_sizes[ XmUbBOTTOM ].max_height ) +
      ( ( ( w -> notebook.book_pages_sides == XmBOTTOM_LEFT ) ||
          ( w -> notebook.book_pages_sides == XmBOTTOM_RIGHT ) ) ?
        w -> notebook.book_side_extent : 0 );

    if( tmp_dimension < w -> core.height - 1 )
      w -> notebook.front_page_bottom_pos = 
        (Position) ( w -> core.height - tmp_dimension - 1 );
    else
      w -> notebook.front_page_bottom_pos = 0;

  }

  /* TABS. */

  /* Dimensions. */

  /* A value in the height or width field means that the dimension is set. */
  /* 0 means variable in that dimension. */

  switch( w -> notebook.tab_packing ){

    case XmUbTAB_PACK_TIGHT:
      /* Variable in both dimensions. */
      break;

    case XmUbTAB_PACK_COLUMN:
      /* Fixed in the dimension attached to the side. */
      tab_width[ XmUbTOP ] = 
        preferred_sizes -> tab_sizes[ XmUbTOP ].max_width;
      tab_width[ XmUbBOTTOM ] = 
        preferred_sizes -> tab_sizes[ XmUbBOTTOM ].max_width;

      tab_height[ XmUbLEFT ] = 
        preferred_sizes -> tab_sizes[ XmUbLEFT ].max_height;
      tab_height[ XmUbRIGHT ] = 
        preferred_sizes -> tab_sizes[ XmUbRIGHT ].max_height;

      break;

    case XmUbTAB_PACK_EQUAL:
      /* Fixed to max in both width and height. */
      tab_width[ XmUbTOP ] = 
        preferred_sizes -> tab_sizes[ XmUbTOP ].max_width;
      tab_width[ XmUbLEFT ] = 
        preferred_sizes -> tab_sizes[ XmUbLEFT ].max_width;
      tab_width[ XmUbRIGHT ] = 
        preferred_sizes -> tab_sizes[ XmUbRIGHT ].max_width;
      tab_width[ XmUbBOTTOM ] = 
        preferred_sizes -> tab_sizes[ XmUbBOTTOM ].max_width;

      tab_height[ XmUbTOP ] = 
        preferred_sizes -> tab_sizes[ XmUbTOP ].max_height;
      tab_height[ XmUbLEFT ] = 
        preferred_sizes -> tab_sizes[ XmUbLEFT ].max_height;
      tab_height[ XmUbRIGHT ] = 
        preferred_sizes -> tab_sizes[ XmUbRIGHT ].max_height;
      tab_height[ XmUbBOTTOM ] = 
        preferred_sizes -> tab_sizes[ XmUbBOTTOM ].max_height;

      break;
  }


  /* Tab positions. */
  LayoutPositions( w, preferred_sizes,
                      tab_x_pos, tab_y_pos, tab_allowance,
                      back_scroll_x_pos, back_scroll_y_pos,
                      forw_scroll_x_pos, forw_scroll_y_pos );

  /* Get the position increment for the tabs. */

  if( ( w -> notebook.book_pages_sides == XmTOP_LEFT ) ||
      ( w -> notebook.book_pages_sides == XmTOP_RIGHT ) ){

    if( preferred_sizes -> tab_sizes[ XmUbTOP ].num_managed > 1 )
      tab_pos_incr[ XmUbTOP ] = (double) w -> notebook.book_side_extent /
        (double) ( preferred_sizes -> tab_sizes[ XmUbTOP ].num_managed - 1 );
    else
      tab_pos_incr[ XmUbTOP ] = 0.0;
  }

  if( ( w -> notebook.book_pages_sides == XmTOP_LEFT ) ||
      ( w -> notebook.book_pages_sides == XmBOTTOM_LEFT ) ){

    if( preferred_sizes -> tab_sizes[ XmUbLEFT ].num_managed > 1 )
      tab_pos_incr[ XmUbLEFT ] = (double) w -> notebook.book_side_extent /
	(double) ( preferred_sizes -> tab_sizes[ XmUbLEFT ].num_managed - 1 );
    else
      tab_pos_incr[ XmUbLEFT ] = 0.0;
  }

  if( ( w -> notebook.book_pages_sides == XmTOP_RIGHT ) ||
      ( w -> notebook.book_pages_sides == XmBOTTOM_RIGHT ) ){

    if( preferred_sizes -> tab_sizes[ XmUbRIGHT ].num_managed > 1 )
      tab_pos_incr[ XmUbRIGHT ] = (double) w -> notebook.book_side_extent /
        (double) ( preferred_sizes -> tab_sizes[ XmUbRIGHT ].num_managed - 1 );
    else
      tab_pos_incr[ XmUbRIGHT ] = 0.0;
  }

  if( ( w -> notebook.book_pages_sides == XmBOTTOM_LEFT ) ||
      ( w -> notebook.book_pages_sides == XmBOTTOM_RIGHT ) ){

    if( preferred_sizes -> tab_sizes[ XmUbBOTTOM ].num_managed > 1 )
      tab_pos_incr[ XmUbBOTTOM ] = (double) w -> notebook.book_side_extent /
       (double) ( preferred_sizes -> tab_sizes[ XmUbBOTTOM ].num_managed - 1 );
    else
      tab_pos_incr[ XmUbBOTTOM ] = 0.0;
  }



  /* The selected tab should be on top. Adjust start offset. */
  for( index = 0; index < 4; index ++ ){

    if( w -> notebook.selected_tabs[ index ] != NULL ){
      selected_tab_num = 
	TabNumber( w, w -> notebook.selected_tabs[ index ], index );

      if( selected_tab_num == INVALID_TAB_NUMBER )
	selected_tab_num = 0;

    } else
      selected_tab_num = 0;

    if( selected_tab_num == 0 )
      tab_num_offset[ index ] = 0;
    else
      tab_num_offset[ index ] = 
	preferred_sizes -> tab_sizes[ index ].num_managed - selected_tab_num;
  }

  /* Set the widget values. */

  /* Backwards scroll buttons. */

  for( index = 0; index < 4; index ++ ){

    kid = GetScrollButton( w, index, XmUbSCROLL_BACK, True );

    if( kid != NULL ){

      /* Find out child's preferred geometry without constraints. */
      (void) XtQueryGeometry( kid, NULL, &desired );

      side = ( (XmUbNoteBookConstraints) ( kid -> core.constraints ) ) ->
        notebook.side;

      switch( side ){

        case XmUbTOP:
          x = back_scroll_x_pos[ XmUbTOP ];
          if( w -> notebook.tab_dir_top == XmUbRIGHT_TO_LEFT )
            x -= (Position) desired.width;

          y = back_scroll_y_pos[ XmUbTOP ] - (Position) desired.height;

          /* Set the start position of the first tab. */
          if( w -> notebook.tab_dir_top == XmUbLEFT_TO_RIGHT )
            tab_x_pos[ XmUbTOP ] = (double) ( x + (Position) desired.width +
              (Position) w -> notebook.scroll_offset_top );
          else
            tab_x_pos[ XmUbTOP ] = (double ) ( x - (Position) 
                                   w -> notebook.scroll_offset_top );

          break;


        case XmUbBOTTOM:
          x = back_scroll_x_pos[ XmUbBOTTOM ];
          if( w -> notebook.tab_dir_bottom == XmUbRIGHT_TO_LEFT )
            x -= (Position) desired.width;

          y = back_scroll_y_pos[ XmUbBOTTOM ];

          /* Set the start position of the first tab. */
          if( w -> notebook.tab_dir_bottom == XmUbLEFT_TO_RIGHT )
            tab_x_pos[ XmUbBOTTOM ] = (double) ( x + (Position) desired.width +
              (Position) w -> notebook.scroll_offset_bottom );
          else
            tab_x_pos[ XmUbBOTTOM ] = (double) ( x - (Position) 
                                      w -> notebook.scroll_offset_bottom );

          break;


        case XmUbLEFT:
          x = back_scroll_x_pos[ XmUbLEFT ] - desired.width;

          y = back_scroll_y_pos[ XmUbLEFT ];
          if( w -> notebook.tab_dir_left == XmUbBOTTOM_TO_TOP )
            y -= (Position) desired.height;

          /* Set the start position of the first tab. */
          if( w -> notebook.tab_dir_left == XmUbTOP_TO_BOTTOM )
            tab_y_pos[ XmUbLEFT ] = (double) ( y + (Position) desired.height +
              (Position) w -> notebook.scroll_offset_left );
          else
            tab_y_pos[ XmUbLEFT ] = (double) ( y - (Position)
                                        w -> notebook.scroll_offset_left );

          break;


        case XmUbRIGHT:
          x = back_scroll_x_pos[ XmUbRIGHT ];

          y = back_scroll_y_pos[ XmUbRIGHT ];
          if( w -> notebook.tab_dir_right == XmUbBOTTOM_TO_TOP )
            y -= (Position) desired.height;

          /* Set the start position of the first tab. */
          if( w -> notebook.tab_dir_right == XmUbTOP_TO_BOTTOM )
            tab_y_pos[ XmUbRIGHT ] = (double) ( y + (Position) desired.height +
              (Position) w -> notebook.scroll_offset_right );
          else
            tab_y_pos[ XmUbRIGHT ] = (double) ( y - (Position) 
                                      w -> notebook.scroll_offset_right );

          break;

      }
      
      /* Set the values. */
      XtConfigureWidget( kid, x, y, desired.width, desired.height, 
                         kid -> core.border_width );

    } /* if */
  } /* for */
   

  /* Set tab spacings. */
  if( ( w -> notebook.equalize_tabs_top ) &&
      ( preferred_sizes -> tab_sizes[ XmUbTOP ].num_managed > 1 ) ){
    tab_spacing[ XmUbTOP ] = ( (double) tab_allowance[ XmUbTOP ] -
      (double) ( tab_width[ XmUbTOP ] == 0 ? 
                 preferred_sizes -> tab_sizes[ XmUbTOP ].total_width :
                 tab_width[ XmUbTOP ] * (Position) preferred_sizes -> 
                   tab_sizes[ XmUbTOP ].num_managed ) ) /
      (double) ( preferred_sizes -> tab_sizes[ XmUbTOP ].num_managed - 1 );
  } else
    tab_spacing[ XmUbTOP ] = (double) w -> notebook.tab_spacing_top;

  if( ( w -> notebook.equalize_tabs_bottom ) &&
      ( preferred_sizes -> tab_sizes[ XmUbBOTTOM ].num_managed > 1 ) ){
    tab_spacing[ XmUbBOTTOM ] = ( (double) tab_allowance[ XmUbBOTTOM ] -
      (double) ( tab_width[ XmUbBOTTOM ] == 0 ? 
                 preferred_sizes -> tab_sizes[ XmUbBOTTOM ].total_width :
                 tab_width[ XmUbBOTTOM ] * (Position) preferred_sizes -> 
                   tab_sizes[ XmUbBOTTOM ].num_managed ) ) /
      (double) ( preferred_sizes -> tab_sizes[ XmUbBOTTOM ].num_managed - 1 );
  } else
    tab_spacing[ XmUbBOTTOM ] = (double) w -> notebook.tab_spacing_bottom;

  if( ( w -> notebook.equalize_tabs_left ) &&
      ( preferred_sizes -> tab_sizes[ XmUbLEFT ].num_managed > 1 ) ){
    tab_spacing[ XmUbLEFT ] = ( (double) tab_allowance[ XmUbLEFT ] -
      (double) ( tab_height[ XmUbLEFT ] == 0 ? 
                 preferred_sizes -> tab_sizes[ XmUbLEFT ].total_height :
                 tab_height[ XmUbLEFT ] * (Position) preferred_sizes -> 
                   tab_sizes[ XmUbLEFT ].num_managed ) ) /
      (double) ( preferred_sizes -> tab_sizes[ XmUbLEFT ].num_managed - 1 );
  } else
    tab_spacing[ XmUbLEFT ] = (double) w -> notebook.tab_spacing_left;

  if( ( w -> notebook.equalize_tabs_right ) &&
      ( preferred_sizes -> tab_sizes[ XmUbRIGHT ].num_managed > 1 ) ){
    tab_spacing[ XmUbRIGHT ] = ( (double) tab_allowance[ XmUbRIGHT ] -
      (double) ( tab_height[ XmUbRIGHT ] == 0 ? 
                 preferred_sizes -> tab_sizes[ XmUbRIGHT ].total_height :
                 tab_height[ XmUbRIGHT ] * (Position) preferred_sizes -> 
                   tab_sizes[ XmUbRIGHT ].num_managed ) ) /
      (double) ( preferred_sizes -> tab_sizes[ XmUbRIGHT ].num_managed - 1 );
  } else
    tab_spacing[ XmUbRIGHT ] = (double) w -> notebook.tab_spacing_right;



  /* Tab buttons. */

  for( index = 0; index < w -> composite.num_children; index ++ ){
    kid = w -> composite.children[ index ];

    constraints = (XmUbNoteBookConstraints) kid -> core.constraints;

    if( ( constraints -> notebook.child_type == XmUbTAB ) && 
        XtIsManaged( kid ) ){

      /* Let it keep certain fields, depending on the side. */
      side = constraints -> notebook.side;

      /* Find out child's preferred geometry without constraints. */
      if(( tab_width[ side ] == 0 ) || ( tab_height[ side ] == 0 ))
        (void) XtQueryGeometry( kid, NULL, &desired );

      if( kid == w -> notebook.selected_tabs[ side ] )
        tab_num_offset[ side ] = 0;

      if( tab_height[ side ] == 0 )
        height = desired.height;
      else
        height = tab_height[ side ];

      if( tab_width[ side ] == 0 )
        width = desired.width;
      else
        width = tab_width[ side ];

      switch( side ){
        case XmUbTOP:
            x = (Position) tab_x_pos[ side ];
            if( w -> notebook.tab_dir_top == XmUbRIGHT_TO_LEFT )
              x -= (Position) width;

            offset = (Position) height + (Position) ( tab_pos_incr[ side ] * 
                       (double) tab_num_offset[ side ] );

            if( offset < tab_y_pos[ side ] )
              y = (Position) tab_y_pos[ side ] - offset + 1;
            else
              y = 0;

            if( w -> notebook.tab_dir_top == XmUbLEFT_TO_RIGHT )
              tab_x_pos[ side ] += ( (double) width + tab_spacing[ XmUbTOP ] );
            else
              tab_x_pos[ side ] = (double) x - tab_spacing[ XmUbTOP ];
            
            break;

        case XmUbBOTTOM:
            x = (Position) tab_x_pos[ side ];
            if( w -> notebook.tab_dir_bottom == XmUbRIGHT_TO_LEFT )
              x -= (Position) width;

            y = (Position) tab_y_pos[ side ] + (Position) ( 
                  tab_pos_incr[ side ] * (double) tab_num_offset[ side ] );

            if( w -> notebook.tab_dir_bottom == XmUbLEFT_TO_RIGHT )
              tab_x_pos[ side ] += ( (double) width + 
                                     tab_spacing[ XmUbBOTTOM ] );
            else
              tab_x_pos[ side ] = (double) x - tab_spacing[ XmUbBOTTOM ];

          break;

        case XmUbLEFT:
            offset = (Position) width + (Position) ( tab_pos_incr[ side ] * 
                     (double) tab_num_offset[ side ] );

            if( offset < tab_x_pos[ side ] )
              x = (Position) tab_x_pos[ side ] - offset + 1;
            else
              x = 0;

            y = (Position) tab_y_pos[ side ];
            if( w -> notebook.tab_dir_left == XmUbBOTTOM_TO_TOP )
              y -= (Position) height;

            if( w -> notebook.tab_dir_left == XmUbTOP_TO_BOTTOM )
              tab_y_pos[ side ] += ( (double) height + 
                                     tab_spacing[ XmUbLEFT ] );
            else
              tab_y_pos[ side ] = (double) y - tab_spacing[ XmUbLEFT ];

          break;

        case XmUbRIGHT:
            x = (Position) tab_x_pos[ side ] + (Position) ( 
                  tab_pos_incr[ side ] * (double) tab_num_offset[ side ] );

            y = (Position) tab_y_pos[ side ];
            if( w -> notebook.tab_dir_right == XmUbBOTTOM_TO_TOP )
              y -= (Position) height;

            if( w -> notebook.tab_dir_right == XmUbTOP_TO_BOTTOM )
              tab_y_pos[ side ] += ( (double) height +
                                     tab_spacing[ XmUbRIGHT ] );
            else
              tab_y_pos[ side ] = (double) y - tab_spacing[ XmUbRIGHT ];

          break;
      }

      XtConfigureWidget( kid, x, y, width, height, kid -> core.border_width );

      /* Set stacking order if applicable. */
      if( w -> notebook.unstacked_tabs[ side ] &&
          ( ( ( side == XmUbTOP ) && ( tab_spacing[ XmUbTOP ] < 0.0 ) ) ||
	    ( ( side == XmUbLEFT ) && ( tab_spacing[ XmUbLEFT ] < 0.0 ) ) ||
	    ( ( side == XmUbRIGHT ) && ( tab_spacing[ XmUbRIGHT ] < 0.0 ) ) ||
	    ( ( side == XmUbBOTTOM ) && 
              ( tab_spacing[ XmUbBOTTOM ] < 0.0 ) ) ) &&
        XtIsManaged( kid ) && XtWindowOfObject( kid ) != None ) {

	if( ( last_sibling[ side ] == None ) || 
	    ( tab_num_offset[ side ] == 0 ) ){
	  xwc.stack_mode = Above;
	  mask           = CWStackMode;
	} else {
	  xwc.stack_mode = Below;
	  xwc.sibling    = last_sibling[ side ];
	  mask           = CWStackMode | CWSibling;
	}

	XConfigureWindow( XtDisplay( kid ), XtWindow( kid ), mask, &xwc );

        last_sibling[ side ] = XtWindow( kid );
      }


      tab_num_offset[ side ] ++;

    }
  }

  /* Readjust to end of tab. */
  if( w -> notebook.tab_dir_top == XmUbLEFT_TO_RIGHT )
    tab_x_pos[ XmUbTOP ] -= tab_spacing[ XmUbTOP ];
  else
    tab_x_pos[ XmUbTOP ] += tab_spacing[ XmUbTOP ];

  if( w -> notebook.tab_dir_bottom == XmUbLEFT_TO_RIGHT )
    tab_x_pos[ XmUbBOTTOM ] -= tab_spacing[ XmUbBOTTOM ];
  else
    tab_x_pos[ XmUbBOTTOM ] += tab_spacing[ XmUbBOTTOM ];

  if( w -> notebook.tab_dir_left == XmUbTOP_TO_BOTTOM )
    tab_y_pos[ XmUbLEFT ] -= tab_spacing[ XmUbLEFT ];
  else
    tab_y_pos[ XmUbLEFT ] += tab_spacing[ XmUbLEFT ];

  if( w -> notebook.tab_dir_right == XmUbTOP_TO_BOTTOM )
    tab_y_pos[ XmUbRIGHT ] -= tab_spacing[ XmUbRIGHT ];
  else
    tab_y_pos[ XmUbRIGHT ] += tab_spacing[ XmUbRIGHT ];


  /* Forward scroll buttons. */

  for( index = 0; index < 4; index ++ ){
      
    kid = GetScrollButton( w, index, XmUbSCROLL_FORWARD, True );

    if( kid != NULL ){

      /* Find out child's preferred geometry without constraints. */
      (void) XtQueryGeometry( kid, NULL, &desired );

      side = ( (XmUbNoteBookConstraints) ( kid -> core.constraints ) ) ->
        notebook.side;

      switch( side ){

        case XmUbTOP:
          if( w -> notebook.tab_dir_top == XmUbLEFT_TO_RIGHT )
            x = MAX( ( forw_scroll_x_pos[ XmUbTOP ] - 
                       (Position) desired.width ), 
                     ( (Position) tab_x_pos[ XmUbTOP ] + 
                       (Position) w -> notebook.scroll_offset_top ) );
          else 
            x = MIN( forw_scroll_x_pos[ XmUbTOP ], 
                     ( (Position) tab_x_pos[ XmUbTOP ] - 
                       (Position) desired.width -
                       (Position) w -> notebook.scroll_offset_top ) );

          y = forw_scroll_y_pos[ XmUbTOP ] - (Position) desired.height;

          break;


        case XmUbBOTTOM:
          if( w -> notebook.tab_dir_bottom == XmUbLEFT_TO_RIGHT )
            x = MAX( ( forw_scroll_x_pos[ XmUbBOTTOM ] - 
                       (Position) desired.width ), 
                     ( (Position) tab_x_pos[ XmUbBOTTOM ] +
                       (Position) w -> notebook.scroll_offset_bottom ) );
          else 
            x = MIN( forw_scroll_x_pos[ XmUbBOTTOM ], 
                     ( (Position) tab_x_pos[ XmUbBOTTOM ] - 
                       (Position) desired.width -
                       (Position) w -> notebook.scroll_offset_bottom ) );

          y = forw_scroll_y_pos[ XmUbBOTTOM ];

          break;


        case XmUbLEFT:
          if( w -> notebook.tab_dir_left == XmUbTOP_TO_BOTTOM )
            y = MAX( ( forw_scroll_y_pos[ XmUbLEFT ] - 
                       (Position) desired.height ), 
                     ( (Position) tab_y_pos[ XmUbLEFT ] + 
                       (Position) w -> notebook.scroll_offset_left ) );
          else 
            y = MIN( forw_scroll_y_pos[ XmUbLEFT ], 
                     ( (Position) tab_y_pos[ XmUbLEFT ] - 
                       (Position) desired.height -
                       (Position) w -> notebook.scroll_offset_left ) );

          x = forw_scroll_x_pos[ XmUbLEFT ] - (Position) desired.width;

          break;


        case XmUbRIGHT:
          if( w -> notebook.tab_dir_right == XmUbTOP_TO_BOTTOM )
            y = MAX( ( forw_scroll_y_pos[ XmUbRIGHT ] - 
                       (Position) desired.height ), 
                     ( (Position) tab_y_pos[ XmUbRIGHT ] +
                       (Position) w -> notebook.scroll_offset_right ) );
          else 
            y = MIN( forw_scroll_y_pos[ XmUbRIGHT ], 
                     ( (Position) tab_y_pos[ XmUbRIGHT ] - 
                       (Position) desired.height -
                       (Position) w -> notebook.scroll_offset_right ) );

          x = forw_scroll_x_pos[ XmUbRIGHT ];

          break;

      }
      
      /* Set the values. */
      XtConfigureWidget( kid, x, y, desired.width, desired.height, 
                         kid -> core.border_width );

    } /* if */
  } /* for */


  /* WORK AREA. */

  if( ( w -> notebook.work_area != NULL ) && 
      XtIsManaged( w -> notebook.work_area )){

    /* Let the work area take what's left over. */
    if( w -> notebook.front_page_top_pos < 
        w -> notebook.front_page_bottom_pos )
      wa_height = (Dimension) ( w -> notebook.front_page_bottom_pos -
				w -> notebook.front_page_top_pos - 1 );
    else
      wa_height = 0;


    if( w -> notebook.front_page_right_pos > 
        w -> notebook.front_page_left_pos )
      wa_width = (Dimension) ( w -> notebook.front_page_right_pos -
			       w -> notebook.front_page_left_pos - 1 );
    else
      wa_width = 0;

    if( ! w -> notebook.resize_work_area ){
      /* Work area should only be resized if too big. */

      if( preferred_sizes -> work_area_width < wa_width )
	wa_width = preferred_sizes -> work_area_width;
      if( preferred_sizes -> work_area_height < wa_height )
	wa_height = preferred_sizes -> work_area_height;
    }

    XtConfigureWidget( w -> notebook.work_area, 
                       w -> notebook.front_page_left_pos + 1, 
                       w -> notebook.front_page_top_pos + 1,
                       wa_width, wa_height, 
                       w -> notebook.work_area -> core.border_width );
  }

  /* Reset stacking indication. */
  for( index = 0; index < 4; index ++ )
    w -> notebook.unstacked_tabs[ index ] = False;


  return;

} /* DoLayout */


/*----------------------------------------------------------------------*/

static void
  DrawBinding( XmUbNoteBookWidget  w )
{

  /* Type definitions. */
  typedef struct {

    int     x_pos;
    int     y_pos;

    int     x_stop;
    int     y_stop;

    int     x_increment;
    int     y_increment;

    Pixmap  pixmap;

  } PixmapDrawingPositions;


  /* Variables. */
  Dimension               binding_height;
  Dimension               binding_width;
  PixmapDrawingPositions  drawing_pos[ 3 ];
  DimensionDescr          dimension_list[ 3 ];
  int                     index;
  int                     middle;
  unsigned int            x_extent;
  unsigned int            y_extent;

  /* Code. */

  if( w -> notebook.binding_type == XmUbBINDING_NONE )
    return;

  BindingDimensions( w, &binding_width, &binding_height, dimension_list );

  drawing_pos[ START_PIXMAP_INDEX ].pixmap = 
    w -> notebook.binding_pixmap_start;
  drawing_pos[ CENTER_PIXMAP_INDEX ].pixmap = 
    w -> notebook.binding_pixmap_center;
  drawing_pos[ END_PIXMAP_INDEX ].pixmap = w -> notebook.binding_pixmap_end;


  switch( w -> notebook.binding_side ){

    case XmUbTOP:
    case XmUbBOTTOM:
      /* Pixmaps drawn from left to right. X positions the same. */

      drawing_pos[ START_PIXMAP_INDEX ].x_increment = 
        dimension_list[ START_PIXMAP_INDEX ].width;

      drawing_pos[ CENTER_PIXMAP_INDEX ].x_increment = 
        dimension_list[ CENTER_PIXMAP_INDEX ].width;

      drawing_pos[ END_PIXMAP_INDEX ].x_increment = 
        dimension_list[ END_PIXMAP_INDEX ].width;


      if( w -> notebook.repeat_center_binding ){
        /* Start and end are once-only, repeat the center. */

        drawing_pos[ START_PIXMAP_INDEX ].x_pos = 
          (int) w -> notebook.front_page_left_pos;
        drawing_pos[ START_PIXMAP_INDEX ].x_stop = 
          MIN( ( drawing_pos[ START_PIXMAP_INDEX ].x_pos +
                 drawing_pos[ START_PIXMAP_INDEX ].x_increment ),
               (int) w -> notebook.front_page_right_pos );

        drawing_pos[ END_PIXMAP_INDEX ].x_stop = 
          (int) w -> notebook.front_page_right_pos;

        if( drawing_pos[ END_PIXMAP_INDEX ].x_stop - 
            drawing_pos[ END_PIXMAP_INDEX ].x_increment <
            drawing_pos[ START_PIXMAP_INDEX ].x_pos )
          drawing_pos[ END_PIXMAP_INDEX ].x_pos = 
            drawing_pos[ START_PIXMAP_INDEX ].x_pos;

        else
          drawing_pos[ END_PIXMAP_INDEX ].x_pos = 
            drawing_pos[ END_PIXMAP_INDEX ].x_stop -   
            drawing_pos[ END_PIXMAP_INDEX ].x_increment;

        drawing_pos[ CENTER_PIXMAP_INDEX ].x_pos = 
          drawing_pos[ START_PIXMAP_INDEX ].x_stop;      
        drawing_pos[ CENTER_PIXMAP_INDEX ].x_stop = 
          drawing_pos[ END_PIXMAP_INDEX ].x_pos;

      } else {

        /* Repeat start and end, center only drawn once. */

        middle = (int) ( w -> notebook.front_page_left_pos +
                          ( w -> notebook.front_page_right_pos -
                            w -> notebook.front_page_left_pos ) / 2 );

        drawing_pos[ CENTER_PIXMAP_INDEX ].x_pos = middle - 
          drawing_pos[ CENTER_PIXMAP_INDEX ].x_increment / 2;
        drawing_pos[ CENTER_PIXMAP_INDEX ].x_stop = middle +
          drawing_pos[ CENTER_PIXMAP_INDEX ].x_increment / 2;

        /* Symmetric, we only need to test one side. */
        if( drawing_pos[ CENTER_PIXMAP_INDEX ].x_pos < 
            (int) w -> notebook.front_page_left_pos ){

          drawing_pos[ CENTER_PIXMAP_INDEX ].x_pos = 
            (int) w -> notebook.front_page_left_pos;
          drawing_pos[ CENTER_PIXMAP_INDEX ].x_stop = 
            (int) w -> notebook.front_page_right_pos;

          /* The center pixmap is so large it fills up the whole side. */
          drawing_pos[ START_PIXMAP_INDEX ].x_pos = 
            (int) w -> notebook.front_page_left_pos;
          drawing_pos[ START_PIXMAP_INDEX ].x_stop = 
            drawing_pos[ START_PIXMAP_INDEX ].x_pos - 1;

          drawing_pos[ END_PIXMAP_INDEX ].x_pos = 
            (int) w -> notebook.front_page_right_pos;
          drawing_pos[ END_PIXMAP_INDEX ].x_stop = 
            drawing_pos[ END_PIXMAP_INDEX ].x_pos - 1;

        } else {
          /* There is some room for the start and end pixmaps. */

          drawing_pos[ START_PIXMAP_INDEX ].x_pos = 
            (int) w -> notebook.front_page_left_pos;
          drawing_pos[ START_PIXMAP_INDEX ].x_stop = 
            drawing_pos[ CENTER_PIXMAP_INDEX ].x_pos;

          drawing_pos[ END_PIXMAP_INDEX ].x_pos = 
            drawing_pos[ CENTER_PIXMAP_INDEX ].x_stop;
          drawing_pos[ END_PIXMAP_INDEX ].x_stop =
            (int) w -> notebook.front_page_right_pos;
        }
      }

      /* Y positions. */

      if( w -> notebook.binding_side == XmUbTOP ){

        for( index = 0; index <= END_PIXMAP_INDEX; index ++ ){
          drawing_pos[ index ].y_pos  = 0;
          drawing_pos[ index ].y_stop = (int) w -> notebook.front_page_top_pos;

          drawing_pos[ index ].y_increment = 0;
        }

      } else {

        for( index = 0; index <= END_PIXMAP_INDEX; index ++ ){

          if( w -> core.height > binding_height )
            drawing_pos[ index ].y_pos =
              (int) ( w -> core.height - binding_height );
          else
            drawing_pos[ index ].y_pos = 0;

          drawing_pos[ index ].y_stop = (int) w -> core.height;

          drawing_pos[ index ].y_increment = 0;
        }

      }

      break;


    case XmUbLEFT:
    case XmUbRIGHT:
      /* Pixmaps drawn from top to bottom. Y positions the same. */

      drawing_pos[ START_PIXMAP_INDEX ].y_increment = 
        dimension_list[ START_PIXMAP_INDEX ].height;

      drawing_pos[ CENTER_PIXMAP_INDEX ].y_increment = 
        dimension_list[ CENTER_PIXMAP_INDEX ].height;

      drawing_pos[ END_PIXMAP_INDEX ].y_increment = 
        dimension_list[ END_PIXMAP_INDEX ].height;


      if( w -> notebook.repeat_center_binding ){
        /* Start and end are once-only, repeat the center. */

        drawing_pos[ START_PIXMAP_INDEX ].y_pos = 
          (int) w -> notebook.front_page_top_pos;
        drawing_pos[ START_PIXMAP_INDEX ].y_stop = 
          MIN( ( drawing_pos[ START_PIXMAP_INDEX ].y_pos +
                 drawing_pos[ START_PIXMAP_INDEX ].y_increment ),
               (int) w -> notebook.front_page_bottom_pos );

        drawing_pos[ END_PIXMAP_INDEX ].y_stop = 
          (int) w -> notebook.front_page_bottom_pos;

        if( drawing_pos[ END_PIXMAP_INDEX ].y_stop - 
            drawing_pos[ END_PIXMAP_INDEX ].y_increment <
            drawing_pos[ START_PIXMAP_INDEX ].y_pos )
          drawing_pos[ END_PIXMAP_INDEX ].y_pos = 
            drawing_pos[ START_PIXMAP_INDEX ].y_pos;

        else
          drawing_pos[ END_PIXMAP_INDEX ].y_pos = 
            drawing_pos[ END_PIXMAP_INDEX ].y_stop -   
            drawing_pos[ END_PIXMAP_INDEX ].y_increment;

        drawing_pos[ CENTER_PIXMAP_INDEX ].y_pos = 
          drawing_pos[ START_PIXMAP_INDEX ].y_stop;      
        drawing_pos[ CENTER_PIXMAP_INDEX ].y_stop = 
          drawing_pos[ END_PIXMAP_INDEX ].y_pos;

      } else {

        /* Repeat start and end, center only drawn once. */

        middle = (int) ( w -> notebook.front_page_top_pos +
                          ( w -> notebook.front_page_bottom_pos -
                            w -> notebook.front_page_top_pos ) / 2 );

        drawing_pos[ CENTER_PIXMAP_INDEX ].y_pos = middle - 
          drawing_pos[ CENTER_PIXMAP_INDEX ].y_increment / 2;
        drawing_pos[ CENTER_PIXMAP_INDEX ].y_stop = middle +
          drawing_pos[ CENTER_PIXMAP_INDEX ].y_increment / 2;

        /* Symmetric, we only need to test one side. */
        if( drawing_pos[ CENTER_PIXMAP_INDEX ].y_pos < 
            (int) w -> notebook.front_page_top_pos ){

          drawing_pos[ CENTER_PIXMAP_INDEX ].y_pos = 
            (int) w -> notebook.front_page_top_pos;
          drawing_pos[ CENTER_PIXMAP_INDEX ].y_stop = 
            (int) w -> notebook.front_page_bottom_pos;

          /* The center pixmap is so large it fills up the whole side. */
          drawing_pos[ START_PIXMAP_INDEX ].y_pos = 
            (int) w -> notebook.front_page_top_pos;
          drawing_pos[ START_PIXMAP_INDEX ].y_stop = 
            drawing_pos[ START_PIXMAP_INDEX ].y_pos - 1;

          drawing_pos[ END_PIXMAP_INDEX ].y_pos = 
            (int) w -> notebook.front_page_bottom_pos;
          drawing_pos[ END_PIXMAP_INDEX ].y_stop = 
            drawing_pos[ END_PIXMAP_INDEX ].y_pos - 1;

        } else {
          /* There is some room for the start and end pixmaps. */

          drawing_pos[ START_PIXMAP_INDEX ].y_pos = 
            (int) w -> notebook.front_page_top_pos;
          drawing_pos[ START_PIXMAP_INDEX ].y_stop = 
            drawing_pos[ CENTER_PIXMAP_INDEX ].y_pos;

          drawing_pos[ END_PIXMAP_INDEX ].y_pos = 
            drawing_pos[ CENTER_PIXMAP_INDEX ].y_stop;
          drawing_pos[ END_PIXMAP_INDEX ].y_stop =
            (int) w -> notebook.front_page_bottom_pos;
        }
      }

      /* X positions. */

      if( w -> notebook.binding_side == XmUbLEFT ){

        for( index = 0; index <= END_PIXMAP_INDEX; index ++ ){
          drawing_pos[ index ].x_pos  = 0;
          drawing_pos[ index ].x_stop = 
            (int) w -> notebook.front_page_left_pos - 1;

          drawing_pos[ index ].x_increment = 0;
        }

      } else {

        for( index = 0; index <= END_PIXMAP_INDEX; index ++ ){

          if( w -> core.width - 1 > binding_width )
            drawing_pos[ index ].x_pos =
              (int) ( w -> core.width - binding_width - 1 );
          else
            drawing_pos[ index ].x_pos = 0;

          drawing_pos[ index ].x_stop = (int) w -> core.width - 1;

          drawing_pos[ index ].x_increment = 0;
        }

      }

      break;

  }

  /* Draw the pixmap repeatedly. */
  for( index = 0; index <= END_PIXMAP_INDEX; index ++ ){

    if( drawing_pos[ index ].pixmap == None )
      continue;

    for( ; ( drawing_pos[ index ].x_pos <= drawing_pos[ index ].x_stop ) && 
           ( drawing_pos[ index ].y_pos <= drawing_pos[ index ].y_stop ); 
         drawing_pos[ index ].x_pos += drawing_pos[ index ].x_increment, 
         drawing_pos[ index ].y_pos += drawing_pos[ index ].y_increment ){

      binding_width  = dimension_list[ index ].width;
      binding_height = dimension_list[ index ].height;


      if( drawing_pos[ index ].y_pos + binding_height > 
          drawing_pos[ index ].y_stop )
        y_extent = (unsigned int) ( drawing_pos[ index ].y_stop - 
                                    drawing_pos[ index ].y_pos );
      else 
        y_extent = (unsigned int) binding_height;

      if( drawing_pos[ index ].x_pos + binding_width > 
          drawing_pos[ index ].x_stop )
        x_extent = (unsigned int) ( drawing_pos[ index ].x_stop - 
                                    drawing_pos[ index ].x_pos );
      else 
        x_extent = (unsigned int) binding_width;

      XCopyArea( XtDisplay( (Widget) w ), 
                 drawing_pos[ index ].pixmap, w -> core.window,
                 w -> manager.background_GC,
                 0, 0, x_extent, y_extent, 
                 drawing_pos[ index ].x_pos, drawing_pos[ index ].y_pos );
    }

  }

  return;

} /* DrawBinding */


/*----------------------------------------------------------------------*/

static void
  DrawBookPages( XmUbNoteBookWidget  w,
                 int                 side )
{

  int        end_point = 0;
  Dimension  front_page_height;
  Dimension  front_page_width;
  int        height_rect_x = 0;
  int        height_rect_y = 0;
  Boolean    horiz_bar = True;
  Dimension  offset;
  XPoint     points[ 5 ];
  int        start_point = 0;
  Boolean    vert_bar = True;
  int        width_rect_x = 0;
  int        width_rect_y = 0;

  /* Code. */

  if( w -> notebook.front_page_top_pos < w -> notebook.front_page_bottom_pos )
    front_page_height = w -> notebook.front_page_bottom_pos -
                        w -> notebook.front_page_top_pos + 1;
  else
    front_page_height = 1;

  if( w -> notebook.front_page_right_pos > w -> notebook.front_page_left_pos )
    front_page_width = w -> notebook.front_page_right_pos -
                       w -> notebook.front_page_left_pos + 1;
  else
    front_page_width = 1;

  for( offset = 0; offset < w -> notebook.book_side_extent;
       offset += BOOK_PAGE_WIDTH ){

    /* Set points. Depend on which side the pages are displayed on. */
    switch( w -> notebook.book_pages_sides ){

      case XmTOP_LEFT:

        if( ( side == XmUbBOTTOM ) || ( side == XmUbRIGHT ) )
          return;

        /* Point 0. */
        if( offset + BOOK_PAGE_WIDTH < front_page_width ) 
          points[ 0 ].x = (short) ( w -> notebook.front_page_right_pos -
                                    offset - BOOK_PAGE_WIDTH );
        else
          points[ 0 ].x = (short) w -> notebook.front_page_left_pos;

        if( (Dimension) w -> notebook.front_page_top_pos > offset )
          points[ 0 ].y = (short) ( w -> notebook.front_page_top_pos - 
                                    offset );
        else
          points[ 0 ].y = (short) 0;

        /* Point 1. */
        points[ 1 ].x = points[ 0 ].x;

        if( points[ 0 ].y > (short) BOOK_PAGE_WIDTH )
          points[ 1 ].y = points[ 0 ].y - (short) BOOK_PAGE_WIDTH;
        else
          points[ 1 ].y = 0;

        /* Point 2. */
        points[ 2 ].y = points[ 1 ].y;

        if( (Dimension) points[ 1 ].x > front_page_width )
          points[ 2 ].x = points[ 1 ].x - (short) front_page_width + 1;
        else
          points[ 2 ].x = (short) 0;

        /* Point 3. */
        points[ 3 ].x = points[ 2 ].x;

        if( (Dimension) points[ 2 ].y + front_page_height < w -> core.height )
          points[ 3 ].y = points[ 2 ].y + (short) front_page_height - 1;
        else
          points[ 3 ].y = (short) w -> core.height - 1;

        /* Point 4. */
        if( (Position) points[ 3 ].x + (Position) BOOK_PAGE_WIDTH <  
            w -> notebook.front_page_left_pos )
          points[ 4 ].x = points[ 3 ].x + (short) BOOK_PAGE_WIDTH;
        else
          points[ 4 ].x = w -> notebook.front_page_left_pos;

        points[ 4 ].y = points[ 3 ].y;

        height_rect_x = (int) points[ 2 ].x + 1;
        height_rect_y = (int) points[ 2 ].y + 1;

        width_rect_x  = (int) points[ 2 ].x + 1;
        width_rect_y  = (int) points[ 2 ].y + 1;

        break;


      case XmTOP_RIGHT:

        if( ( side == XmUbBOTTOM ) || ( side == XmUbLEFT ) )
          return;

        /* Point 0. */
        if( offset + BOOK_PAGE_WIDTH < front_page_width ) 
          points[ 0 ].x = (short) ( w -> notebook.front_page_left_pos + 
                                    offset + BOOK_PAGE_WIDTH );
        else
          points[ 0 ].x = (short) w -> notebook.front_page_right_pos;

        if( (Dimension) w -> notebook.front_page_top_pos > offset )
          points[ 0 ].y = (short) ( w -> notebook.front_page_top_pos - 
                                    offset );
        else
          points[ 0 ].y = (short) 0;

        /* Point 1. */
        points[ 1 ].x = points[ 0 ].x;

        if( points[ 0 ].y > (short) BOOK_PAGE_WIDTH )
          points[ 1 ].y = points[ 0 ].y - (short) BOOK_PAGE_WIDTH;
        else
          points[ 1 ].y = 0;

        /* Point 2. */
        points[ 2 ].y = points[ 1 ].y;

        if( (Dimension) points[ 1 ].x + front_page_width + 1 < 
            w -> core.width )
          points[ 2 ].x = points[ 1 ].x + (short) front_page_width - 1;
        else
          points[ 2 ].x = (short) w -> core.width - 1;

        /* Point 3. */
        points[ 3 ].x = points[ 2 ].x;

        if( (Dimension) points[ 2 ].y + front_page_height < w -> core.height )
          points[ 3 ].y = points[ 2 ].y + (short) front_page_height - 1;
        else
          points[ 3 ].y = (short) w -> core.height - 1;

        /* Point 4. */
        if( (Position) points[ 3 ].x > (short) BOOK_PAGE_WIDTH )
          points[ 4 ].x = points[ 3 ].x - (short) BOOK_PAGE_WIDTH;
        else
          points[ 4 ].x = w -> notebook.front_page_right_pos;

        points[ 4 ].y = points[ 3 ].y;


        if( points[ 2 ].x > (short) BOOK_PAGE_WIDTH )
          height_rect_x = (int) points[ 2 ].x - BOOK_PAGE_WIDTH + 1;
        else
          height_rect_x = 1;

        height_rect_y = (int) points[ 2 ].y + 1;

        width_rect_x  = (int) points[ 1 ].x + 1;
        width_rect_y  = (int) points[ 1 ].y + 1;

        break;


      case XmBOTTOM_LEFT:

        if( ( side == XmUbTOP ) || ( side == XmUbRIGHT ) )
          return;

        /* Point 0. */
        if( offset + BOOK_PAGE_WIDTH < front_page_width ) 
          points[ 0 ].x = (short) ( w -> notebook.front_page_right_pos -
                                    offset - BOOK_PAGE_WIDTH );
        else
          points[ 0 ].x = (short) w -> notebook.front_page_left_pos;

        if( w -> notebook.front_page_bottom_pos + offset <= w -> core.height )
          points[ 0 ].y = (short) w -> notebook.front_page_bottom_pos + offset;
        else
          points[ 0 ].y = (short) w -> core.height - 1;

        /* Point 1. */
        points[ 1 ].x = points[ 0 ].x;

        if( points[ 0 ].y + (short) offset <= w -> core.height )
          points[ 1 ].y = points[ 0 ].y + (short) BOOK_PAGE_WIDTH;
        else
          points[ 1 ].y = (short) w -> core.height - 1;

        /* Point 2. */
        points[ 2 ].y = points[ 1 ].y;

        if( (Dimension) points[ 1 ].x > front_page_width )
          points[ 2 ].x = points[ 1 ].x - (short) front_page_width + 1;
        else
          points[ 2 ].x = (short) 0;

        /* Point 3. */
        points[ 3 ].x = points[ 2 ].x;

        if( (Dimension) points[ 2 ].y > front_page_height )
          points[ 3 ].y = points[ 2 ].y - (short) front_page_height + 1;
        else
          points[ 3 ].y = (short) 0;

        /* Point 4. */
        if( (Position) points[ 3 ].x + (Position) BOOK_PAGE_WIDTH <  
            w -> notebook.front_page_left_pos )
          points[ 4 ].x = points[ 3 ].x + (short) BOOK_PAGE_WIDTH;
        else
          points[ 4 ].x = w -> notebook.front_page_left_pos;

        points[ 4 ].y = points[ 3 ].y;


        height_rect_x = (int) points[ 3 ].x + 1;
        height_rect_y = (int) points[ 3 ].y + 1;

        width_rect_x  = (int) points[ 2 ].x + 1;
        width_rect_y  = (int) points[ 0 ].y + 1;

        break;


      case XmBOTTOM_RIGHT:

        if( ( side == XmUbTOP ) || ( side == XmUbLEFT ) )
          return;

        /* Point 0. */
        if( offset + BOOK_PAGE_WIDTH < front_page_width ) 
          points[ 0 ].x = (short) ( w -> notebook.front_page_left_pos + 
                                    offset + BOOK_PAGE_WIDTH );
        else
          points[ 0 ].x = (short) w -> notebook.front_page_right_pos;

        if( w -> notebook.front_page_bottom_pos + offset <= w -> core.height )
          points[ 0 ].y = (short) w -> notebook.front_page_bottom_pos + offset;
        else
          points[ 0 ].y = (short) w -> core.height - 1;

        /* Point 1. */
        points[ 1 ].x = points[ 0 ].x;

        if( points[ 0 ].y + (short) offset <= w -> core.height )
          points[ 1 ].y = points[ 0 ].y + (short) BOOK_PAGE_WIDTH;
        else
          points[ 1 ].y = (short) w -> core.height - 1;

        /* Point 2. */
        points[ 2 ].y = points[ 1 ].y;

        if( (Dimension) points[ 1 ].x + front_page_width <= w -> core.width )
          points[ 2 ].x = points[ 1 ].x + (short) front_page_width - 1;
        else
          points[ 2 ].x = (short) w -> core.width - 1;

        /* Point 3. */
        points[ 3 ].x = points[ 2 ].x;

        if( (Dimension) points[ 2 ].y > front_page_height )
          points[ 3 ].y = points[ 2 ].y - (short) front_page_height - 1;
        else
          points[ 3 ].y = (short) 0;

        /* Point 4. */
        if( (Position) points[ 3 ].x >=
            w -> notebook.front_page_right_pos + BOOK_PAGE_WIDTH )
          points[ 4 ].x = points[ 3 ].x - (short) BOOK_PAGE_WIDTH;
        else
          points[ 4 ].x = w -> notebook.front_page_right_pos;

        points[ 4 ].y = points[ 3 ].y;


        height_rect_x = (int) points[ 4 ].x + 1;
        height_rect_y = (int) points[ 4 ].y + 1;

        width_rect_x  = (int) points[ 0 ].x + 1;
        width_rect_y  = (int) points[ 0 ].y + 1;

        break;

    }    

    if( ( side == XmUbTOP ) || ( side == XmUbBOTTOM ) ){
      start_point = 0;
      end_point   = 2;
      vert_bar    = False;

    } else if( ( side == XmUbRIGHT ) || ( side == XmUbLEFT ) ){
      start_point = 2;
      end_point   = 4;
      horiz_bar   = False;

    } else {
      start_point = 0;
      end_point   = 4;
    }

    if( vert_bar )
      XFillRectangle( XtDisplay( (Widget) w ), w -> core.window, 
                      w -> notebook.covered_page_gc,
                      height_rect_x, height_rect_y,
                      (unsigned int) BOOK_PAGE_WIDTH - 1,
                      (unsigned int) front_page_height - 2 );

    if( horiz_bar )
      XFillRectangle( XtDisplay( (Widget) w ), w -> core.window, 
                      w -> notebook.covered_page_gc,
                      width_rect_x, width_rect_y,
                      (unsigned int) front_page_width - 2,
                      (unsigned int) BOOK_PAGE_WIDTH - 1 );

    XDrawLines( XtDisplay( (Widget) w ), w -> core.window, 
                w -> notebook.line_draw_gc, &( points[ start_point ] ), 
                end_point - start_point + 1, CoordModeOrigin );

  }


  return;

} /* DrawBookPages */


/*----------------------------------------------------------------------*/

static void
  DrawFrontPageFrame( XmUbNoteBookWidget  w,
                      int                 side )
{
  /* Code. */

  if( ( ( side == INVALID_SIDE ) || ( side == XmUbTOP ) ) &&
      ( ( w -> notebook.binding_side != XmUbTOP ) ||
        ( w -> notebook.binding_type == XmUbBINDING_NONE ) ) )
    XDrawLine( XtDisplay( (Widget) w ), w -> core.window, 
               w -> notebook.line_draw_gc, 
               (int) w -> notebook.front_page_left_pos, 
               (int) w -> notebook.front_page_top_pos,
               (int) w -> notebook.front_page_right_pos,
               (int) w -> notebook.front_page_top_pos );

  if( ( ( side == INVALID_SIDE ) || ( side == XmUbRIGHT ) ) &&
      ( ( w -> notebook.binding_side != XmUbRIGHT ) ||
        ( w -> notebook.binding_type == XmUbBINDING_NONE ) ) )
    XDrawLine( XtDisplay( (Widget) w ), w -> core.window, 
               w -> notebook.line_draw_gc, 
               (int) w -> notebook.front_page_right_pos,
               (int) w -> notebook.front_page_top_pos,
               (int) w -> notebook.front_page_right_pos,
               (int) w -> notebook.front_page_bottom_pos );

  if( ( ( side == INVALID_SIDE ) || ( side == XmUbBOTTOM ) ) &&
      ( ( w -> notebook.binding_side != XmUbBOTTOM ) ||
        ( w -> notebook.binding_type == XmUbBINDING_NONE ) ) )
    XDrawLine( XtDisplay( (Widget) w ), w -> core.window, 
               w -> notebook.line_draw_gc, 
               (int) w -> notebook.front_page_left_pos, 
               (int) w -> notebook.front_page_bottom_pos, 
               (int) w -> notebook.front_page_right_pos, 
               (int) w -> notebook.front_page_bottom_pos );

  if( ( ( side == INVALID_SIDE ) || ( side == XmUbLEFT ) ) &&
      ( ( w -> notebook.binding_side != XmUbLEFT ) ||
        ( w -> notebook.binding_type == XmUbBINDING_NONE ) ) )
    XDrawLine( XtDisplay( (Widget) w ), w -> core.window, 
               w -> notebook.line_draw_gc, 
               (int) w -> notebook.front_page_left_pos, 
               (int) w -> notebook.front_page_top_pos,
               (int) w -> notebook.front_page_left_pos, 
               (int) w -> notebook.front_page_bottom_pos );


  return;

} /* DrawFrontPageFrame */


/*----------------------------------------------------------------------*/

static void
  ExposeIt( Widget  widget,
            XEvent  *ev,
            Region  region )
{

  XExposeEvent        *event;
  RedrawDescr         redraw;
  XmUbNoteBookWidget  w;
  int                 height, width, x, y, index;

  /* Code. */

  w     = (XmUbNoteBookWidget) widget;
  event = (XExposeEvent *) ev;

  /* The front area and binding are only redrawn if we are not dealing with
     just one side. */

  /* Check the area that was exposed. */
  /* If resized, everything should be redrawn. */
  if( w -> notebook.nonexposed_resize ){
    redraw.binding    = True;
    redraw.front_area = True;
    for( index = 0; index < 4; index ++ )
      redraw.book_side[ index ] = True;

    /* We must clear the whole window. */
    XClearArea( XtDisplay( widget ), w -> core.window, 0, 0,
                (unsigned int) w -> core.width,
                (unsigned int) w -> core.height,
                False );
  } else {
    CheckRedrawArea( w, event, &redraw );

    /* We must clear the area that was exposed. */
    XClearArea( XtDisplay( widget ), w -> core.window,
                event -> x, event -> y,
                (unsigned int) event -> width,
                (unsigned int) event -> height,
                False );
  }

  /* Front page. Fill exposed area with background color.  */
  if( redraw.front_area ){  

    x = (int) w -> notebook.front_page_left_pos + 1;
    y = (int) w -> notebook.front_page_top_pos + 1;

    width  = (int) ( w -> notebook.front_page_right_pos -
                     w -> notebook.front_page_left_pos - 1 );
    height = (int) ( w -> notebook.front_page_bottom_pos -
                     w -> notebook.front_page_top_pos - 1 );
    if( width < 1 )
      width = 1;

    if( height < 1 )
      height = 1;

    XFillRectangle( XtDisplay( widget ), w -> core.window, 
                    w -> notebook.front_page_gc, x, y, width, height );
  } 

  if( redraw.binding )
    DrawBinding( w );

  for( index = 0; index < 4; index ++ ){
    if( redraw.book_side[ index ] )
      DrawBookPages( w, index );
  }

  /* We can afford to always draw the front frame. */
  DrawFrontPageFrame( w, INVALID_SIDE ); 

  w -> notebook.nonexposed_resize = False;


  return;

} /* ExposeIt */


/*----------------------------------------------------------------------*/

static XtGeometryResult
  GeometryManager( Widget            widget,
                   XtWidgetGeometry  *request,
                   XtWidgetGeometry  *reply )
{

  Dimension           old_height;
  Dimension           old_width;
  XtWidgetGeometry    own_request;
  NoteBookSizes       preferred_sizes;
  Dimension           pref_height;
  Dimension           pref_width;
  XtGeometryResult    result;
  XmUbNoteBookWidget  w;

  /* Code. */

  /* Find notebookwidget. */
  w = (XmUbNoteBookWidget) XtParent( widget );

  /* Work area can only be resized if we allow notebook resize. */
  if( widget == w -> notebook.work_area ){
    if( !w -> notebook.resize_allowed || w -> notebook.front_page_fixed )
      return XtGeometryNo;
  }

  w -> notebook.internal_resize = True;

  /* Find out how big the widget would be if the resize were allowed. */
  GetPreferredSize( w, widget, request, &preferred_sizes );
  PreferredOwnDimensions( w, &preferred_sizes, &pref_width, &pref_height );

  /* If no change in dimensions, allow the request. */
  if(( pref_width == w -> core.width ) && ( pref_height == w -> core.height )){
    DoLayout( w, &preferred_sizes );
    w -> notebook.internal_resize = False;

    return XtGeometryYes;
  }

  /* If no resize is set, we are dealing with a tab widget. Allow the child */
  /* resize but do not request a new size from our parent. */
  if( !w -> notebook.resize_allowed ){
    DoLayout( w, &preferred_sizes );
    w -> notebook.internal_resize = False;

    return XtGeometryYes;
  }

  /* We must ask our parent to resize us. */
  own_request.request_mode = CWWidth | CWHeight;
  own_request.width  = pref_width;
  own_request.height = pref_height;

  /* Save dimensions. */
  old_width  = w -> core.width;
  old_height = w -> core.height;

  w -> notebook.resize_called = False;

  /* We are not interested in any compromise geometry. */
  result = XtMakeGeometryRequest( (Widget) w, &own_request, NULL );

  /* Deny work area requests if new size not totally granted. */
  if( ( result != XtGeometryYes ) && ( widget == w -> notebook.work_area ) ){
    w -> core.width  = old_width;
    w -> core.height = old_height;

    w -> notebook.internal_resize = False;

    return XtGeometryNo;
  }

  if( !w -> notebook.resize_called )
    Resize( (Widget) w );

  w -> notebook.internal_resize = False;


  return XtGeometryYes;

} /* GeometryManager */


/*----------------------------------------------------------------------*/

static void
  GetPreferredSize( XmUbNoteBookWidget  w,
                    Widget              initiator,
                    XtWidgetGeometry    *request,
                    NoteBookSizesRef    preferred_sizes )
{

  Boolean           button_valid;
  XtWidgetGeometry  desired;
  int               index;
  Widget            kid;
  scrollDirection   scroll_direction;
  sideAttachment    side;

  /* Code. */
 
  /* Initialize. */
  preferred_sizes -> work_area_width  = 0;
  preferred_sizes -> work_area_height = 0;

  for( index = 0; index < 4; index ++ ){

    preferred_sizes -> tab_sizes[ index ].num_managed = 0;
    preferred_sizes -> tab_sizes[ index ].max_width    = 0;
    preferred_sizes -> tab_sizes[ index ].max_height   = 0;
    preferred_sizes -> tab_sizes[ index ].total_width  = 0;
    preferred_sizes -> tab_sizes[ index ].total_height = 0;

    preferred_sizes -> scroll_sizes[ index ].num_managed = 0;
    preferred_sizes -> scroll_sizes[ index ].max_width    = 0;
    preferred_sizes -> scroll_sizes[ index ].max_height   = 0;
    preferred_sizes -> scroll_sizes[ index ].total_width  = 0;
    preferred_sizes -> scroll_sizes[ index ].total_height = 0;

    preferred_sizes -> back_scroll_dim[ index ]    = 0;
    preferred_sizes -> forward_scroll_dim[ index ] = 0;

  }

  /* Get binding dimensions. */
  BindingDimensions( w, &( preferred_sizes -> binding_width ), 
                        &( preferred_sizes -> binding_height ), NULL );


  /* Get the children's preferred geometries. */

  for( index = 0; index < w -> composite.num_children; index ++ ){

    kid = w -> composite.children[ index ];

    if( ( XtIsManaged( kid ))){

      KidsPreferredGeometry( kid, initiator, request, &desired );

      if( kid == w -> notebook.work_area ){
        /* Work area child. */
        preferred_sizes -> work_area_width  = desired.width;
        preferred_sizes -> work_area_height = desired.height;

      } else {
        /* Tab or scroll button child. */
        side = ( (XmUbNoteBookConstraints) ( kid -> core.constraints ) ) ->
          notebook.side;
        
        switch( ( (XmUbNoteBookConstraints) 
                    ( kid -> core.constraints ) ) -> notebook.child_type ){

          case XmUbWORK_AREA:
            break;


          case XmUbTAB:

            preferred_sizes -> tab_sizes[ side ].num_managed ++;

            if( preferred_sizes -> tab_sizes[ side ].max_width < 
                desired.width )
              preferred_sizes -> tab_sizes[ side ].max_width = 
                desired.width;

            if( preferred_sizes -> tab_sizes[ side ].max_height < 
                desired.height )
              preferred_sizes -> tab_sizes[ side ].max_height = 
                desired.height;

            preferred_sizes -> tab_sizes[ side ].total_width  += 
              desired.width;
            preferred_sizes -> tab_sizes[ side ].total_height += 
              desired.height;

            break;


          case XmUbSCROLL_BUTTON:

            scroll_direction = ( (XmUbNoteBookConstraints) 
              ( kid -> core.constraints ) ) -> notebook.scroll_direction;

            button_valid = False;

            switch( scroll_direction ) {

              case XmUbSCROLL_BACK:
                if( preferred_sizes -> back_scroll_dim[ side ] == 0 ){

                  if( ( side == XmUbTOP ) || ( side == XmUbBOTTOM ) )
                    preferred_sizes -> back_scroll_dim[ side ] = desired.width;
                  else
                    preferred_sizes -> back_scroll_dim[ side ] = 
                      desired.height;

                  button_valid = True;

                }

                break;

              case XmUbSCROLL_FORWARD:
                if( preferred_sizes -> forward_scroll_dim[ side ] == 0 ){

                  if( ( side == XmUbTOP ) || ( side == XmUbBOTTOM ) )
                    preferred_sizes -> forward_scroll_dim[ side ] = 
                      desired.width;
                  else
                    preferred_sizes -> forward_scroll_dim[ side ] = 
                      desired.height;

                  button_valid = True;

                }

                break;
            }

            if( button_valid ){

              preferred_sizes -> scroll_sizes[ side ].num_managed ++;

              if( preferred_sizes -> scroll_sizes[ side ].max_width < 
                  desired.width )
                preferred_sizes -> scroll_sizes[ side ].max_width = 
                  desired.width;

              if( preferred_sizes -> scroll_sizes[ side ].max_height < 
                  desired.height )
                preferred_sizes -> scroll_sizes[ side ].max_height = 
                  desired.height;

              preferred_sizes -> scroll_sizes[ side ].total_width  += 
                desired.width;
              preferred_sizes -> scroll_sizes[ side ].total_height += 
                desired.height;
            }

            break;

        }
      }
    }
  }


  return;

} /* GetPreferredSize */


/*----------------------------------------------------------------------*/

static Widget
  GetScrollButton( XmUbNoteBookWidget  w,
                   sideAttachment      side,
                   scrollDirection     direction,
                   Boolean             must_be_managed )
{

  XmUbNoteBookConstraints  constraints;
  int                      index;
  Widget                   kid;

  /* Code. */

  for( index = 0; index < w -> composite.num_children; index ++ ){
    kid = w -> composite.children[ index ];

    constraints = (XmUbNoteBookConstraints) kid -> core.constraints;

    if( ( constraints -> notebook.child_type       == XmUbSCROLL_BUTTON ) &&
        ( constraints -> notebook.side             == side ) &&
        ( constraints -> notebook.scroll_direction == direction ) ){

      if( must_be_managed ){

        if( XtIsManaged( kid ) )
          return kid;

      } else 
        return kid;

    }

  } /* for */


  return( NULL );

} /* GetScrollButton */


/*----------------------------------------------------------------------*/

static void
  Initialize( Widget     treq,
              Widget     tnew,
              ArgList    args,
              Cardinal*  num_args )
{

  int                 index;
  XmUbNoteBookWidget  new;

  /* Code. */

  new = (XmUbNoteBookWidget) tnew;

  /* Resource checks. */
  if( (int) new -> notebook.book_side_extent % BOOK_PAGE_WIDTH != 0 ){
    new -> notebook.book_side_extent = (Dimension) (
      ( (int) new -> notebook.book_side_extent / BOOK_PAGE_WIDTH ) * 
      BOOK_PAGE_WIDTH );
    printf( "XmUbNbookSideExtent must be multiple of 4. Forced to %d.\n",
	    (int) new -> notebook.book_side_extent );
  }

  /* Initialize non-resource fields. */
  new -> notebook.resize_called         = False;
  new -> notebook.internal_resize       = False;
  new -> notebook.nonexposed_resize     = True;

  new -> notebook.front_page_top_pos    = 0;
  new -> notebook.front_page_left_pos   = 0;
  new -> notebook.front_page_right_pos  = 0;
  new -> notebook.front_page_bottom_pos = 0;


  new -> notebook.line_draw_gc          = NULL;
  new -> notebook.front_page_gc         = NULL;
  new -> notebook.covered_page_gc       = NULL;

  for( index = 0; index < 4; index ++ ){
    new -> notebook.selected_tabs[ index ]  = NULL;
    new -> notebook.unstacked_tabs[ index ] = True;
  }

  /* Set an initial size. We have no children yet. */
/*  if( new -> core.width == 0 ) */
    new -> core.width  = 100;

/*  if( new -> core.height == 0 ) */
    new -> core.height = 100;

  SetLineDrawGC( new );
  SetFrontPageGC( new );
  SetCoveredPageGC( new );

  SetBindingPixmap( new, XmUbBINDING_NONE );


  return;

} /* Initialize */


/*----------------------------------------------------------------------*/

static void 
  KidsPreferredGeometry( Widget            kid,
                         Widget            initiator,
                         XtWidgetGeometry  *request,
                         XtWidgetGeometry  *desired )
{

  /* Code. */

  if( kid == initiator ){
    /* The initiator should not be queried. */
    if( request -> request_mode & CWWidth )
      desired -> width = request -> width;
    else
      desired -> width = (Dimension) kid -> core.width;

    if( request -> request_mode & CWHeight )
      desired -> height = request -> height;
    else
      desired -> height = (Dimension) kid -> core.height;

  } else
    (void) XtQueryGeometry( kid, NULL, desired );


  return;

} /* KidsPreferredGeometry */


/*----------------------------------------------------------------------*/

static void
  LayoutPositions( XmUbNoteBookWidget  w,
                   NoteBookSizesRef    preferred_sizes,
                   double              double_tab_x_pos[],
                   double              double_tab_y_pos[],
                   Position            tab_allowance[],
                   Position            back_scroll_x_pos[],
                   Position            back_scroll_y_pos[],
                   Position            forw_scroll_x_pos[],
                   Position            forw_scroll_y_pos[] )
{
  /* Variables. */
  int       index;
  Position  left_pos, right_pos;
  Position  tab_x_pos[ 4 ];
  Position  tab_y_pos[ 4 ];


  /* Code. */

  /* Start with default values. */

  tab_y_pos[ XmUbTOP ]         = w -> notebook.front_page_top_pos;
  back_scroll_y_pos[ XmUbTOP ] = w -> notebook.front_page_top_pos;
  forw_scroll_y_pos[ XmUbTOP ] = w -> notebook.front_page_top_pos;

  if( w -> notebook.tab_dir_top == XmUbLEFT_TO_RIGHT ){
    tab_x_pos[ XmUbTOP ]         = w -> notebook.front_page_left_pos;
    back_scroll_x_pos[ XmUbTOP ] = w -> notebook.front_page_left_pos;
    forw_scroll_x_pos[ XmUbTOP ] = w -> notebook.front_page_right_pos;
  } else {
    tab_x_pos[ XmUbTOP ]         = w -> notebook.front_page_right_pos;
    back_scroll_x_pos[ XmUbTOP ] = w -> notebook.front_page_right_pos;
    forw_scroll_x_pos[ XmUbTOP ] = w -> notebook.front_page_left_pos;
  }

  tab_y_pos[ XmUbBOTTOM ]         = w -> notebook.front_page_bottom_pos;
  back_scroll_y_pos[ XmUbBOTTOM ] = w -> notebook.front_page_bottom_pos + 1;
  forw_scroll_y_pos[ XmUbBOTTOM ] = w -> notebook.front_page_bottom_pos + 1;

  if( w -> notebook.tab_dir_bottom == XmUbLEFT_TO_RIGHT ){
    tab_x_pos[ XmUbBOTTOM ]         = w -> notebook.front_page_left_pos;
    back_scroll_x_pos[ XmUbBOTTOM ] = w -> notebook.front_page_left_pos;
    forw_scroll_x_pos[ XmUbBOTTOM ] = w -> notebook.front_page_right_pos;
  } else {
    tab_x_pos[ XmUbBOTTOM ]         = w -> notebook.front_page_right_pos;
    back_scroll_x_pos[ XmUbBOTTOM ] = w -> notebook.front_page_right_pos;
    forw_scroll_x_pos[ XmUbBOTTOM ] = w -> notebook.front_page_left_pos;
  }

  tab_x_pos[ XmUbLEFT ]         = w -> notebook.front_page_left_pos;
  back_scroll_x_pos[ XmUbLEFT ] = w -> notebook.front_page_left_pos;
  forw_scroll_x_pos[ XmUbLEFT ] = w -> notebook.front_page_left_pos;

  if( w -> notebook.tab_dir_left == XmUbTOP_TO_BOTTOM ){
    tab_y_pos[ XmUbLEFT ]         = w -> notebook.front_page_top_pos;
    back_scroll_y_pos[ XmUbLEFT ] = w -> notebook.front_page_top_pos;
    forw_scroll_y_pos[ XmUbLEFT ] = w -> notebook.front_page_bottom_pos;
  } else {
    tab_y_pos[ XmUbLEFT ]         = w -> notebook.front_page_bottom_pos;
    back_scroll_y_pos[ XmUbLEFT ] = w -> notebook.front_page_bottom_pos;
    forw_scroll_y_pos[ XmUbLEFT ] = w -> notebook.front_page_top_pos;
  }

  tab_x_pos[ XmUbRIGHT ]         = w -> notebook.front_page_right_pos;
  back_scroll_x_pos[ XmUbRIGHT ] = w -> notebook.front_page_right_pos + 1;
  forw_scroll_x_pos[ XmUbRIGHT ] = w -> notebook.front_page_right_pos + 1;

  if( w -> notebook.tab_dir_right == XmUbTOP_TO_BOTTOM ){
    tab_y_pos[ XmUbRIGHT ]         = w -> notebook.front_page_top_pos;
    back_scroll_y_pos[ XmUbRIGHT ] = w -> notebook.front_page_top_pos;
    forw_scroll_y_pos[ XmUbRIGHT ] = w -> notebook.front_page_bottom_pos;
  } else {
    tab_y_pos[ XmUbRIGHT ]         = w -> notebook.front_page_bottom_pos;
    back_scroll_y_pos[ XmUbRIGHT ] = w -> notebook.front_page_bottom_pos;
    forw_scroll_y_pos[ XmUbRIGHT ] = w -> notebook.front_page_top_pos;
  }

  /* Adjustments depending on the display side of the covered pages. */
  switch( w -> notebook.book_pages_sides ){

    case XmTOP_LEFT:

      if( w -> notebook.tab_dir_top == XmUbRIGHT_TO_LEFT )
        tab_x_pos[ XmUbTOP ] -= (Position) w -> notebook.book_side_extent;

      back_scroll_x_pos[ XmUbTOP ] -= 
        (Position) w -> notebook.book_side_extent;
      forw_scroll_x_pos[ XmUbTOP ] -= 
        (Position) w -> notebook.book_side_extent;
      back_scroll_y_pos[ XmUbTOP ] -= 
        (Position) w -> notebook.book_side_extent;
      forw_scroll_y_pos[ XmUbTOP ] -= 
        (Position) w -> notebook.book_side_extent;

      if( w -> notebook.tab_dir_left == XmUbBOTTOM_TO_TOP )
        tab_y_pos[ XmUbLEFT ] -= (Position) w -> notebook.book_side_extent;

      back_scroll_x_pos[ XmUbLEFT ] -= 
        (Position) w -> notebook.book_side_extent;
      forw_scroll_x_pos[ XmUbLEFT ] -= 
        (Position) w -> notebook.book_side_extent;
      back_scroll_y_pos[ XmUbLEFT ] -= 
        (Position) w -> notebook.book_side_extent;
      forw_scroll_y_pos[ XmUbLEFT ] -= 
        (Position) w -> notebook.book_side_extent;

      break;


    case XmTOP_RIGHT:

      if( w -> notebook.tab_dir_top == XmUbLEFT_TO_RIGHT )
        tab_x_pos[ XmUbTOP ] += (Position) w -> notebook.book_side_extent;

      back_scroll_x_pos[ XmUbTOP ] += 
        (Position) w -> notebook.book_side_extent;
      forw_scroll_x_pos[ XmUbTOP ] += 
        (Position) w -> notebook.book_side_extent;
      back_scroll_y_pos[ XmUbTOP ] -= 
        (Position) w -> notebook.book_side_extent;
      forw_scroll_y_pos[ XmUbTOP ] -= 
        (Position) w -> notebook.book_side_extent;

      if( w -> notebook.tab_dir_right == XmUbBOTTOM_TO_TOP )
        tab_y_pos[ XmUbRIGHT ] -= (Position) w -> notebook.book_side_extent;

      back_scroll_x_pos[ XmUbRIGHT ] += 
        (Position) w -> notebook.book_side_extent;
      forw_scroll_x_pos[ XmUbRIGHT ] += 
        (Position) w -> notebook.book_side_extent;
      back_scroll_y_pos[ XmUbRIGHT ] -= 
        (Position) w -> notebook.book_side_extent;
      forw_scroll_y_pos[ XmUbRIGHT ] -= 
        (Position) w -> notebook.book_side_extent;

      break;


    case XmBOTTOM_LEFT:

      if( w -> notebook.tab_dir_bottom == XmUbRIGHT_TO_LEFT )
        tab_x_pos[ XmUbBOTTOM ] -= (Position) w -> notebook.book_side_extent;

      back_scroll_x_pos[ XmUbBOTTOM ] -= 
        (Position) w -> notebook.book_side_extent;
      forw_scroll_x_pos[ XmUbBOTTOM ] -= 
        (Position) w -> notebook.book_side_extent;
      back_scroll_y_pos[ XmUbBOTTOM ] += 
        (Position) w -> notebook.book_side_extent;
      forw_scroll_y_pos[ XmUbBOTTOM ] += 
        (Position) w -> notebook.book_side_extent;

      if( w -> notebook.tab_dir_left == XmUbTOP_TO_BOTTOM )
        tab_y_pos[ XmUbLEFT ] += (Position) w -> notebook.book_side_extent;

      back_scroll_x_pos[ XmUbLEFT ] -= 
        (Position) w -> notebook.book_side_extent;
      forw_scroll_x_pos[ XmUbLEFT ] -= 
        (Position) w -> notebook.book_side_extent;
      back_scroll_y_pos[ XmUbLEFT ] += 
        (Position) w -> notebook.book_side_extent;
      forw_scroll_y_pos[ XmUbLEFT ] += 
        (Position) w -> notebook.book_side_extent;

      break;


    case XmBOTTOM_RIGHT:

      if( w -> notebook.tab_dir_bottom == XmUbLEFT_TO_RIGHT )
        tab_x_pos[ XmUbBOTTOM ] += (Position) w -> notebook.book_side_extent;

      back_scroll_x_pos[ XmUbBOTTOM ] += 
        (Position) w -> notebook.book_side_extent;
      forw_scroll_x_pos[ XmUbBOTTOM ] += 
        (Position) w -> notebook.book_side_extent;
      back_scroll_y_pos[ XmUbBOTTOM ] += 
        (Position) w -> notebook.book_side_extent;
      forw_scroll_y_pos[ XmUbBOTTOM ] += 
        (Position) w -> notebook.book_side_extent;

      if( w -> notebook.tab_dir_right == XmUbTOP_TO_BOTTOM )
        tab_y_pos[ XmUbRIGHT ] += (Position) w -> notebook.book_side_extent;

      back_scroll_x_pos[ XmUbRIGHT ] += 
        (Position) w -> notebook.book_side_extent;
      forw_scroll_x_pos[ XmUbRIGHT ] += 
        (Position) w -> notebook.book_side_extent;
      back_scroll_y_pos[ XmUbRIGHT ] += 
        (Position) w -> notebook.book_side_extent;
      forw_scroll_y_pos[ XmUbRIGHT ] += 
        (Position) w -> notebook.book_side_extent;

      break;

  }

  /* Allowances for tabs in the different dimensions. */

  /* Top. */

  if( w -> notebook.tab_dir_top == XmUbLEFT_TO_RIGHT ){

    if( preferred_sizes -> back_scroll_dim[ XmUbTOP ] != 0 )
      left_pos = back_scroll_x_pos[ XmUbTOP ] +
                 (Position) preferred_sizes -> back_scroll_dim[ XmUbTOP ] +
                 (Position) w -> notebook.scroll_offset_top;
    else 
      left_pos = w -> notebook.front_page_left_pos +
                 (Position) w -> notebook.scroll_offset_top +
                 ( w -> notebook.book_pages_sides == XmTOP_RIGHT ?
                   (Position) w -> notebook.book_side_extent : 0 );

    if( preferred_sizes -> forward_scroll_dim[ XmUbTOP ] != 0 )
      right_pos = forw_scroll_x_pos[ XmUbTOP ] -
                  (Position) preferred_sizes -> forward_scroll_dim[ XmUbTOP ] -
                  (Position) w -> notebook.scroll_offset_top;
    else
      right_pos = w -> notebook.front_page_right_pos -
                  (Position) w -> notebook.scroll_offset_top -
                  ( w -> notebook.book_pages_sides == XmTOP_LEFT ?
                    (Position) w -> notebook.book_side_extent : 0 );

  } else {

    if( preferred_sizes -> forward_scroll_dim[ XmUbTOP ] != 0 )
      left_pos = forw_scroll_x_pos[ XmUbTOP ] +
                 (Position) preferred_sizes -> forward_scroll_dim[ XmUbTOP ] +
                 (Position) w -> notebook.scroll_offset_top;
    else 
      left_pos = w -> notebook.front_page_left_pos +
                 (Position) w -> notebook.scroll_offset_top +
                 ( w -> notebook.book_pages_sides == XmTOP_RIGHT ?
                   (Position) w -> notebook.book_side_extent : 0 );

    if( preferred_sizes -> back_scroll_dim[ XmUbTOP ] != 0 )
      right_pos = back_scroll_x_pos[ XmUbTOP ] -
                  (Position) preferred_sizes -> back_scroll_dim[ XmUbTOP ] -
                  (Position) w -> notebook.scroll_offset_top;
    else
      right_pos = w -> notebook.front_page_right_pos -
                  (Position) w -> notebook.scroll_offset_top -
                  ( w -> notebook.book_pages_sides == XmTOP_LEFT ?
                    (Position) w -> notebook.book_side_extent : 0 );
  }

  tab_allowance[ XmUbTOP ] = right_pos - left_pos + 1;


  /* Bottom. */

  if( w -> notebook.tab_dir_bottom == XmUbLEFT_TO_RIGHT ){

    if( preferred_sizes -> back_scroll_dim[ XmUbBOTTOM ] != 0 )
      left_pos = back_scroll_x_pos[ XmUbBOTTOM ] +
                 (Position) preferred_sizes -> back_scroll_dim[ XmUbBOTTOM ] +
                 (Position) w -> notebook.scroll_offset_bottom;
    else 
      left_pos = w -> notebook.front_page_left_pos +
                 (Position) w -> notebook.scroll_offset_bottom +
                 ( w -> notebook.book_pages_sides == XmBOTTOM_RIGHT ?
                   (Position) w -> notebook.book_side_extent : 0 );

    if( preferred_sizes -> forward_scroll_dim[ XmUbBOTTOM ] != 0 )
      right_pos = forw_scroll_x_pos[ XmUbBOTTOM ] -
                  (Position) preferred_sizes -> 
                               forward_scroll_dim[ XmUbBOTTOM ] -
                  (Position) w -> notebook.scroll_offset_bottom;
    else
      right_pos = w -> notebook.front_page_right_pos -
                  (Position) w -> notebook.scroll_offset_bottom -
                  ( w -> notebook.book_pages_sides == XmBOTTOM_LEFT ?
                    (Position) w -> notebook.book_side_extent : 0 );

  } else {

    if( preferred_sizes -> forward_scroll_dim[ XmUbBOTTOM ] != 0 )
      left_pos = forw_scroll_x_pos[ XmUbBOTTOM ] +
                 (Position) preferred_sizes -> 
                              forward_scroll_dim[ XmUbBOTTOM ] +
                 (Position) w -> notebook.scroll_offset_bottom;
    else 
      left_pos = w -> notebook.front_page_left_pos +
                 (Position) w -> notebook.scroll_offset_bottom +
                 ( w -> notebook.book_pages_sides == XmBOTTOM_RIGHT ?
                   (Position) w -> notebook.book_side_extent : 0 );

    if( preferred_sizes -> back_scroll_dim[ XmUbBOTTOM ] != 0 )
      right_pos = back_scroll_x_pos[ XmUbBOTTOM ] -
                  (Position) preferred_sizes -> back_scroll_dim[ XmUbBOTTOM ] -
                  (Position) w -> notebook.scroll_offset_bottom;
    else
      right_pos = w -> notebook.front_page_right_pos -
                  (Position) w -> notebook.scroll_offset_bottom -
                  ( w -> notebook.book_pages_sides == XmBOTTOM_LEFT ?
                    (Position) w -> notebook.book_side_extent : 0 );
  }

  tab_allowance[ XmUbBOTTOM ] = right_pos - left_pos + 1;


  /* Left. */

  if( w -> notebook.tab_dir_left == XmUbTOP_TO_BOTTOM ){

    if( preferred_sizes -> back_scroll_dim[ XmUbLEFT ] != 0 )
      left_pos = back_scroll_y_pos[ XmUbLEFT ] +
                 (Position) preferred_sizes -> back_scroll_dim[ XmUbLEFT ] +
                 (Position) w -> notebook.scroll_offset_left;
    else 
      left_pos = w -> notebook.front_page_top_pos +
                 (Position) w -> notebook.scroll_offset_left +
                 ( w -> notebook.book_pages_sides == XmBOTTOM_LEFT ?
                   (Position) w -> notebook.book_side_extent : 0 );

    if( preferred_sizes -> forward_scroll_dim[ XmUbLEFT ] != 0 )
      right_pos = forw_scroll_y_pos[ XmUbLEFT ] -
                  (Position) preferred_sizes -> 
                               forward_scroll_dim[ XmUbLEFT ] -
                  (Position) w -> notebook.scroll_offset_left;
    else
      right_pos = w -> notebook.front_page_bottom_pos -
                  (Position) w -> notebook.scroll_offset_left -
                  ( w -> notebook.book_pages_sides == XmTOP_LEFT ?
                    (Position) w -> notebook.book_side_extent : 0 );

  } else {

    if( preferred_sizes -> forward_scroll_dim[ XmUbLEFT ] != 0 )
      left_pos = forw_scroll_y_pos[ XmUbLEFT ] +
                 (Position) preferred_sizes -> forward_scroll_dim[ XmUbLEFT ] +
                 (Position) w -> notebook.scroll_offset_left;
    else 
      left_pos = w -> notebook.front_page_top_pos +
                 (Position) w -> notebook.scroll_offset_left +
                 ( w -> notebook.book_pages_sides == XmBOTTOM_LEFT ?
                   (Position) w -> notebook.book_side_extent : 0 );

    if( preferred_sizes -> back_scroll_dim[ XmUbLEFT ] != 0 )
      right_pos = back_scroll_y_pos[ XmUbLEFT ] -
                  (Position) preferred_sizes -> back_scroll_dim[ XmUbLEFT ] -
                  (Position) w -> notebook.scroll_offset_left;
    else
      right_pos = w -> notebook.front_page_bottom_pos -
                  (Position) w -> notebook.scroll_offset_left -
                  ( w -> notebook.book_pages_sides == XmTOP_LEFT ?
                    (Position) w -> notebook.book_side_extent : 0 );
  }

  tab_allowance[ XmUbLEFT ] = right_pos - left_pos + 1;


  /* Right. */

  if( w -> notebook.tab_dir_right == XmUbTOP_TO_BOTTOM ){

    if( preferred_sizes -> back_scroll_dim[ XmUbRIGHT ] != 0 )
      left_pos = back_scroll_y_pos[ XmUbRIGHT ] +
                 (Position) preferred_sizes -> back_scroll_dim[ XmUbRIGHT ] +
                 (Position) w -> notebook.scroll_offset_right;
    else 
      left_pos = w -> notebook.front_page_top_pos +
                 (Position) w -> notebook.scroll_offset_right +
                 ( w -> notebook.book_pages_sides == XmBOTTOM_RIGHT ?
                   (Position) w -> notebook.book_side_extent : 0 );

    if( preferred_sizes -> forward_scroll_dim[ XmUbRIGHT ] != 0 )
      right_pos = forw_scroll_y_pos[ XmUbRIGHT ] -
                  (Position) preferred_sizes -> 
                               forward_scroll_dim[ XmUbRIGHT ] -
                  (Position) w -> notebook.scroll_offset_right;
    else
      right_pos = w -> notebook.front_page_bottom_pos -
                  (Position) w -> notebook.scroll_offset_right -
                  ( w -> notebook.book_pages_sides == XmTOP_RIGHT ?
                    (Position) w -> notebook.book_side_extent : 0 );

  } else {

    if( preferred_sizes -> forward_scroll_dim[ XmUbRIGHT ] != 0 )
      left_pos = forw_scroll_y_pos[ XmUbRIGHT ] +
                 (Position) preferred_sizes -> 
                              forward_scroll_dim[ XmUbRIGHT ] +
                 (Position) w -> notebook.scroll_offset_right;
    else 
      left_pos = w -> notebook.front_page_top_pos +
                 (Position) w -> notebook.scroll_offset_right +
                 ( w -> notebook.book_pages_sides == XmBOTTOM_RIGHT ?
                   (Position) w -> notebook.book_side_extent : 0 );

    if( preferred_sizes -> back_scroll_dim[ XmUbRIGHT ] != 0 )
      right_pos = back_scroll_y_pos[ XmUbRIGHT ] -
                  (Position) preferred_sizes -> back_scroll_dim[ XmUbRIGHT ] -
                  (Position) w -> notebook.scroll_offset_right;
    else
      right_pos = w -> notebook.front_page_bottom_pos -
                  (Position) w -> notebook.scroll_offset_right -
                  ( w -> notebook.book_pages_sides == XmTOP_RIGHT ?
                    (Position) w -> notebook.book_side_extent : 0 );
  }

  tab_allowance[ XmUbRIGHT ] = right_pos - left_pos + 1;

  /* Convert to doubles for return arrays. */
  for( index = 0; index < 4; index ++ ){
    double_tab_x_pos[ index ] = (double) tab_x_pos[ index ];
    double_tab_y_pos[ index ] = (double) tab_y_pos[ index ];
  }


  return;

} /* LayoutPositions */


/*----------------------------------------------------------------------*/

static void
  PreferredOwnDimensions( XmUbNoteBookWidget  w,
                          NoteBookSizesRef    preferred_sizes,
                          Dimension           *pref_width,
                          Dimension           *pref_height )
{

  Dimension  bottom_height;
  Dimension  bottom_width = 0;
  Dimension  left_height = 0;
  Dimension  left_width;
  Dimension  middle_height;
  Dimension  middle_width;
  Dimension  right_height = 0;
  Dimension  right_width;
  Dimension  top_height;
  Dimension  top_width = 0;
  int        total_num;

  /* Code. */

  /* Initialize. */
  *pref_width  = 0;
  *pref_height = 0;


  if( w -> notebook.work_area_decides_size ){

    /* The size should be based on how big the work area wants to be. */

    left_width  = MAX( preferred_sizes -> tab_sizes[ XmUbLEFT ].max_width,
                       preferred_sizes -> scroll_sizes[ XmUbLEFT ].max_width );

    right_width = MAX( preferred_sizes -> tab_sizes[ XmUbRIGHT ].max_width,
                    preferred_sizes -> scroll_sizes[ XmUbRIGHT ].max_width );

    *pref_width = preferred_sizes -> work_area_width + 2 + /* frame */
                  w -> notebook.book_side_extent +
                  ( w -> notebook.binding_side == XmUbRIGHT || 
                    w -> notebook.binding_side == XmUbLEFT ?
                    preferred_sizes -> binding_width : 0 ) +
                  left_width + right_width;


    top_height    = MAX( preferred_sizes -> tab_sizes[ XmUbTOP ].max_height,
                      preferred_sizes -> scroll_sizes[ XmUbTOP ].max_height );

    bottom_height = MAX( preferred_sizes -> tab_sizes[ XmUbBOTTOM ].max_height,
                    preferred_sizes -> scroll_sizes[ XmUbBOTTOM ].max_height );

    *pref_height = preferred_sizes -> work_area_height + 2 + /* frame */
                   w -> notebook.book_side_extent +
		   ( w -> notebook.binding_side == XmUbTOP || 
		     w -> notebook.binding_side == XmUbBOTTOM ?
		     preferred_sizes -> binding_height : 0 ) +
                   top_height + bottom_height;

    /* Guard against zero requests. */
    if( *pref_width == 0 )
      *pref_width = 1;
    if( *pref_height == 0 )
      *pref_height = 1;


    return;

  }


  /* The dimensions have to be calculated based on all components. */

  /* Heights. */
  /* Only middle portion counted -- not tabs on bottom or top. */

  switch( w -> notebook.tab_packing ){

    case XmUbTAB_PACK_TIGHT:
      total_num = preferred_sizes -> tab_sizes[ XmUbLEFT ].num_managed +
                  preferred_sizes -> scroll_sizes[ XmUbLEFT ].num_managed;

      left_height = (Dimension) (
        (int) ( preferred_sizes -> tab_sizes[ XmUbLEFT ].total_height +
                preferred_sizes -> scroll_sizes[ XmUbLEFT ].total_height ) +
        ( total_num > 1 ? 
          ( total_num - 1 ) * w -> notebook.tab_spacing_left : 0 ));


      total_num = preferred_sizes -> tab_sizes[ XmUbRIGHT ].num_managed +
                  preferred_sizes -> scroll_sizes[ XmUbRIGHT ].num_managed;

      right_height = (Dimension) (
        (int) ( preferred_sizes -> tab_sizes[ XmUbRIGHT ].total_height +
                preferred_sizes -> scroll_sizes[ XmUbRIGHT ].total_height ) +
        ( total_num > 1 ? 
          ( total_num - 1 ) * w -> notebook.tab_spacing_right : 0 ));

      break;

    case XmUbTAB_PACK_COLUMN:
    case XmUbTAB_PACK_EQUAL:
      total_num = preferred_sizes -> tab_sizes[ XmUbLEFT ].num_managed +
                  preferred_sizes -> scroll_sizes[ XmUbLEFT ].num_managed;

      left_height = (Dimension) (
        (int) preferred_sizes -> tab_sizes[ XmUbLEFT ].max_height *
        preferred_sizes -> tab_sizes[ XmUbLEFT ].num_managed +
        (int) preferred_sizes -> scroll_sizes[ XmUbLEFT ].max_height *
        preferred_sizes -> scroll_sizes[ XmUbLEFT ].num_managed +
        ( total_num > 1 ? 
          ( total_num - 1 ) * w -> notebook.tab_spacing_left : 0 ));

      total_num = preferred_sizes -> tab_sizes[ XmUbRIGHT ].num_managed +
                  preferred_sizes -> scroll_sizes[ XmUbRIGHT ].num_managed;

      right_height = (Dimension) (
        (int) preferred_sizes -> tab_sizes[ XmUbRIGHT ].max_height *
        preferred_sizes -> tab_sizes[ XmUbRIGHT ].num_managed +
        (int) preferred_sizes -> scroll_sizes[ XmUbRIGHT ].max_height *
        preferred_sizes -> scroll_sizes[ XmUbRIGHT ].num_managed +
        ( total_num > 1 ?
          ( total_num - 1 ) * w -> notebook.tab_spacing_right : 0 ));

      break;
  }

  middle_height = preferred_sizes -> work_area_height + 2; /* frame */

  /* The book side extent is added for the side where it is displayed. */
  if( ( w -> notebook.book_pages_sides == XmTOP_LEFT ) ||
      ( w -> notebook.book_pages_sides == XmBOTTOM_LEFT ) )
    left_height += w -> notebook.book_side_extent;

  else if( ( w -> notebook.book_pages_sides == XmTOP_RIGHT ) || 
           ( w -> notebook.book_pages_sides == XmBOTTOM_RIGHT ) )
    right_height += w -> notebook.book_side_extent;


  /* Widths. */
  /* Only middle portion counted -- not tabs on left or right. */


  switch( w -> notebook.tab_packing ){

    case XmUbTAB_PACK_TIGHT:
      total_num = preferred_sizes -> tab_sizes[ XmUbTOP ].num_managed +
                  preferred_sizes -> scroll_sizes[ XmUbTOP ].num_managed;

      top_width = (Dimension) (
        (int) ( preferred_sizes -> tab_sizes[ XmUbTOP ].total_width +
                preferred_sizes -> scroll_sizes[ XmUbTOP ].total_width ) +
        ( total_num > 1 ? 
          ( total_num - 1 ) * w -> notebook.tab_spacing_top : 0 ));

      total_num = preferred_sizes -> tab_sizes[ XmUbBOTTOM ].num_managed +
                  preferred_sizes -> scroll_sizes[ XmUbBOTTOM ].num_managed;

      bottom_width = (Dimension) (
        (int) ( preferred_sizes -> tab_sizes[ XmUbBOTTOM ].total_width +
                preferred_sizes -> scroll_sizes[ XmUbBOTTOM ].total_width ) +
        ( total_num > 1 ?
          ( total_num - 1 ) * w -> notebook.tab_spacing_bottom : 0 ));

      break;

    case XmUbTAB_PACK_COLUMN:
    case XmUbTAB_PACK_EQUAL:
      total_num = preferred_sizes -> tab_sizes[ XmUbTOP ].num_managed +
                  preferred_sizes -> scroll_sizes[ XmUbTOP ].num_managed;

      top_width = (Dimension) (
        (int) preferred_sizes -> tab_sizes[ XmUbTOP ].max_width *
        preferred_sizes -> tab_sizes[ XmUbTOP ].num_managed +
        (int) preferred_sizes -> scroll_sizes[ XmUbTOP ].max_width *
        preferred_sizes -> scroll_sizes[ XmUbTOP ].num_managed +
        ( total_num > 1 ? 
          ( total_num - 1 ) * w -> notebook.tab_spacing_top : 0 ));

      total_num = preferred_sizes -> tab_sizes[ XmUbBOTTOM ].num_managed +
                  preferred_sizes -> scroll_sizes[ XmUbBOTTOM ].num_managed;

      bottom_width = (Dimension) (
        (int) preferred_sizes -> tab_sizes[ XmUbBOTTOM ].max_width *
        preferred_sizes -> tab_sizes[ XmUbBOTTOM ].num_managed +
        (int) preferred_sizes -> scroll_sizes[ XmUbBOTTOM ].max_width *
        preferred_sizes -> scroll_sizes[ XmUbBOTTOM ].num_managed +
        ( total_num > 1 ?
          ( total_num - 1 ) * w -> notebook.tab_spacing_bottom : 0 ));

      break;
  }

  middle_width = preferred_sizes -> work_area_width + 2; /* frame */

  /* The book side extent is added for the side where it is displayed. */
  if( ( w -> notebook.book_pages_sides == XmTOP_LEFT ) ||
      ( w -> notebook.book_pages_sides == XmTOP_RIGHT ) )
    top_width += w -> notebook.book_side_extent;

  else if( ( w -> notebook.book_pages_sides == XmBOTTOM_LEFT ) || 
           ( w -> notebook.book_pages_sides == XmBOTTOM_RIGHT ) )
    bottom_width += w -> notebook.book_side_extent;

  /* The preferred dimensions. */
  if( w -> notebook.front_page_fixed && w -> notebook.internal_resize ){

    middle_width = w -> notebook.front_page_right_pos + 
                   preferred_sizes -> tab_sizes[ XmUbRIGHT ].max_width;

    if( ( w -> notebook.book_pages_sides == XmTOP_RIGHT ) ||
        ( w -> notebook.book_pages_sides == XmBOTTOM_RIGHT ) )
      middle_width += w -> notebook.book_side_extent;

    middle_height = w -> notebook.front_page_bottom_pos + 
                    preferred_sizes -> tab_sizes[ XmUbBOTTOM ].max_height;

    if( ( w -> notebook.book_pages_sides == XmBOTTOM_LEFT ) ||
        ( w -> notebook.book_pages_sides == XmBOTTOM_RIGHT ) )
      middle_height += w -> notebook.book_side_extent;

    *pref_width  = MAX( middle_width, MAX( top_width, bottom_width ));
    *pref_height = MAX( middle_height, MAX( left_height, right_height ));

    /* Add binding only if on right or bottom. */
    if( w -> notebook.binding_side == XmUbRIGHT )
      *pref_width += preferred_sizes -> binding_width;

    else if( w -> notebook.binding_side == XmUbBOTTOM )
      *pref_height += preferred_sizes -> binding_height;

  } else {

    *pref_width  = MAX( middle_width, MAX( top_width, bottom_width )) +
      ( preferred_sizes -> tab_sizes[ XmUbLEFT ].max_width +
        preferred_sizes -> tab_sizes[ XmUbRIGHT ].max_width +
        w -> notebook.book_side_extent );

    *pref_height = MAX( middle_height, MAX( left_height, right_height )) +
      ( preferred_sizes -> tab_sizes[ XmUbBOTTOM ].max_height +
        preferred_sizes -> tab_sizes[ XmUbTOP ].max_height +
        w -> notebook.book_side_extent );

    /* We may have to add a binding. */
    switch( w -> notebook.binding_side ){

      case XmUbTOP:
      case XmUbBOTTOM:

        *pref_height += preferred_sizes -> binding_height;
        break;

      case XmUbLEFT:
      case XmUbRIGHT:

        *pref_width += preferred_sizes -> binding_width;
        break;
    }

  }

  /* Guard against zero requests. */
  if( *pref_width == 0 )
    *pref_width = 1;
  if( *pref_height == 0 )
    *pref_height = 1;


  return;

} /* PreferredOwnDimensions */


/*----------------------------------------------------------------------*/

static XtGeometryResult
  QueryGeometry( Widget             widget,
                 XtWidgetGeometry  *proposed,
                 XtWidgetGeometry  *answer )
{

  NoteBookSizes       preferred_sizes;
  Dimension           pref_height;
  Dimension           pref_width;
  XmUbNoteBookWidget  w;

  /* Code. */


  w = (XmUbNoteBookWidget) widget;

  /* Check what dimensions we want. */
  if( !w -> notebook.resize_allowed ){
    /* No resize allowed. */
    pref_width  = w -> core.width;
    pref_height = w -> core.height;

  } else {

    /* Get dimensions that we *really* want. */
    GetPreferredSize( w, NULL, NULL, &preferred_sizes );
    PreferredOwnDimensions( w, &preferred_sizes, &pref_width, &pref_height );
  }

  answer -> request_mode = CWWidth | CWHeight;
  answer -> width  = pref_width;
  answer -> height = pref_height;


  if( proposed == NULL ){
    /* This is a query for the requested geometry. */

    if(( answer -> height == (int) w -> core.height ) &&
       ( answer -> width  == (int) w -> core.width ))
      return XtGeometryNo;
    else
      return XtGeometryAlmost;
  }


  /* The parent supplied a geometry suggestion. */
  if(( answer -> height == proposed -> height ) &&
     ( answer -> width  == proposed -> width ))
    return XtGeometryYes;

  if( ( proposed -> height <= 1 ) ||
      ( proposed -> width  <= 1 ) )
    /* That's too small ! */
    return XtGeometryNo;


  /* Only a compromise left. */
  return XtGeometryAlmost;

} /* QueryGeometry */


/*----------------------------------------------------------------------*/

static void
  Resize( Widget    widget )
{

  NoteBookSizes       preferred_sizes;
  XmUbNoteBookWidget  w;

  /* Code. */

  w = (XmUbNoteBookWidget) widget;

  w -> notebook.resize_called = True;

  /* Check the preferred sizes of the children. */
  GetPreferredSize( w, NULL, NULL, &preferred_sizes );


  /* Do the layout with the new dimensions. */
  DoLayout( w, &preferred_sizes );

  /* We must redraw ourselves with the new size. */
  
  w -> notebook.nonexposed_resize = True;


  return;

} /* Resize */


/*----------------------------------------------------------------------*/

static Boolean
  ResizeIfNeeded( XmUbNoteBookWidget  w,
                  NoteBookSizesRef    preferred_sizes )
{

  Boolean           layout_done;
  Dimension         pref_height;
  Dimension         pref_width;
  XtWidgetGeometry  request;
  XtGeometryResult  result;

  /* Code. */

  /* Initialize. */
  layout_done = False;

  /* We cannot resize if not allowed. */
  if( XtIsManaged( (Widget) w ) && ( ! w -> notebook.resize_allowed ) )
    return False;

  /* Get the preferred dimensions of the notebook widget. */
  PreferredOwnDimensions( w, preferred_sizes, &pref_width, &pref_height );

  /* If the widget has not yet been managed, we should allow every resize. */
  if( ! XtIsManaged( (Widget) w ) ){
    w -> core.width  = pref_width;
    w -> core.height = pref_height;

    /* Returning False will trigger the DoLayout function in the caller. */
    return False;
  }

  /* If we want the same dimensions, no resizing is needed. */
  if(( pref_width  == w -> core.width ) &&
     ( pref_height == w -> core.height ))
    return False;

  /* Dimensions are different. Try to resize. */
  request.request_mode = CWWidth | CWHeight;

  request.width  = pref_width;
  request.height = pref_height;

  w -> notebook.resize_called = False;

  do {

    result = XtMakeGeometryRequest( (Widget) w, &request, &request );

  } while( result == XtGeometryAlmost );

  if( result == XtGeometryNo )
    return False;

  /* Resize done. Core fields have already been updated. */
  return( w -> notebook.resize_called );

} /* ResizeIfNeeded */


/*----------------------------------------------------------------------*/

static void
  SetBindingPixmap( XmUbNoteBookWidget  w,
                    bindingType         old_binding_type )
{

  unsigned char  *bm_bits = NULL;
  int            bm_height = 0;
  int            bm_width = 0;

  /* Code. */

  /* Since we allocate our own pixmaps for solid & spiral we should free
     the old one if that was the old type. */
  switch( old_binding_type ){
    case XmUbBINDING_SOLID:
    case XmUbBINDING_SPIRAL:
      if( w -> notebook.binding_pixmap_center != None )
        XFreePixmap( XtDisplay( (Widget) w ), 
                     w -> notebook.binding_pixmap_center );
      break;

    default:
      break;
  }

  switch( w -> notebook.binding_type ){
    case XmUbBINDING_NONE:
      w -> notebook.binding_pixmap_center = None;
      w -> notebook.binding_pixmap_start  = None;
      w -> notebook.binding_pixmap_end    = None;
      return;

    case XmUbBINDING_PIXMAP:
      /* New pixmap already set. */
      return;

    case XmUbBINDING_SPIRAL:
      bm_width  = predef_spiral_bitmaps[ w -> notebook.binding_side ].width;
      bm_height = predef_spiral_bitmaps[ w -> notebook.binding_side ].height;
      bm_bits   = predef_spiral_bitmaps[ w -> notebook.binding_side ].bits;

      w -> notebook.binding_pixmap_start  = None;
      w -> notebook.binding_pixmap_end    = None;
      w -> notebook.repeat_center_binding = True;

      break;
      
    case XmUbBINDING_SOLID:
      bm_width  = predef_solid_bitmaps[ w -> notebook.binding_side ].width;
      bm_height = predef_solid_bitmaps[ w -> notebook.binding_side ].height;
      bm_bits   = predef_solid_bitmaps[ w -> notebook.binding_side ].bits;

      w -> notebook.binding_pixmap_start  = None;
      w -> notebook.binding_pixmap_end    = None;
      w -> notebook.repeat_center_binding = True;

      break;
  }

  /* Create the new center pixmap from the predefined bitmaps. */
  w -> notebook.binding_pixmap_center = 
    XCreatePixmapFromBitmapData( 
      XtDisplay( (Widget) w ), 
      RootWindowOfScreen( XtScreen( (Widget) w )),
      (char *) bm_bits, (unsigned int) bm_width, (unsigned int) bm_height,
      (unsigned long) w -> manager.foreground,
      (unsigned long) w -> core.background_pixel,
      DefaultDepthOfScreen( XtScreen( (Widget) w )));


  return;

} /* SetBindingPixmap */


/*----------------------------------------------------------------------*/

static void
  SetCoveredPageGC( XmUbNoteBookWidget  w )
{

  XGCValues  gc_values;

  /* Code. */

  if( w -> notebook.covered_page_gc != NULL )
    XtReleaseGC( (Widget) w, w -> notebook.covered_page_gc );

  gc_values.foreground = w -> notebook.covered_page_background;

  w -> notebook.covered_page_gc = XtGetGC( (Widget) w, GCForeground,
                                           &gc_values );


  return;

} /* SetCoveredPageGC */


/*----------------------------------------------------------------------*/

static void
  SetFrontPageGC( XmUbNoteBookWidget  w )
{

  XGCValues  gc_values;

  /* Code. */

  if( w -> notebook.front_page_gc != NULL )
    XtReleaseGC( (Widget) w, w -> notebook.front_page_gc );

  gc_values.foreground = w -> notebook.front_page_background;

  w -> notebook.front_page_gc = XtGetGC( (Widget) w, GCForeground,
                                         &gc_values );


  return;

} /* SetFrontPageGC */


/*----------------------------------------------------------------------*/

static void
  SetLineDrawGC( XmUbNoteBookWidget  w )
{

  XGCValues  gc_values;

  /* Code. */

  if( w -> notebook.line_draw_gc != NULL )
    XtReleaseGC( (Widget) w, w -> notebook.line_draw_gc );

  gc_values.foreground = w -> manager.foreground;
  gc_values.line_width = 0;

  w -> notebook.line_draw_gc = XtGetGC( (Widget) w, 
                                        ( GCForeground | GCLineWidth ),
                                        &gc_values );


  return;

} /* SetLineDrawGC */


/*----------------------------------------------------------------------*/

static Boolean 
  SetValues( Widget     current,
             Widget     request,
             Widget     new,
             ArgList    args,
             Cardinal   *num_args )
{

#define Differs( field )  ( curW -> field != newW -> field )

  XmUbNoteBookWidget  curW;
  Boolean             layout_done = False;
  Boolean             new_kid_pos = False;
  XmUbNoteBookWidget  newW;
  NoteBookSizes       preferred_sizes;
  Boolean             redisplay = False;
  Boolean             visual_changed = False;
  Boolean             work_area_changed = False;

  /* Code. */

  curW = (XmUbNoteBookWidget) current;
  newW = (XmUbNoteBookWidget) new;

  /* Internal marker for kid repositioning. Resize not needed. */
  if( newW -> notebook._internal_repos_kids ){
    new_kid_pos = True;
    newW -> notebook._internal_repos_kids = False;
  }

  /* Resources with restrictions. */
  if( Differs( notebook.book_side_extent ) ){
    if( (int) newW -> notebook.book_side_extent % BOOK_PAGE_WIDTH != 0 ){
      newW -> notebook.book_side_extent = (Dimension) (
        ( (int) newW -> notebook.book_side_extent / BOOK_PAGE_WIDTH ) *
        BOOK_PAGE_WIDTH );
      printf( "XmUbNbookSideExtent must be multiple of 4. Forced to %d.\n",
              (int) newW -> notebook.book_side_extent );
    }

    if( Differs( notebook.book_side_extent ) )
      visual_changed = True;
  }


  /* Color changes. */

  if( Differs( manager.foreground )){
    SetLineDrawGC( newW );
    redisplay = True;
  }
  if( Differs( notebook.front_page_background )){
    SetFrontPageGC( newW );
    redisplay = True;
  }
  if( Differs( notebook.covered_page_background )){
    SetCoveredPageGC( newW );
    redisplay = True;
  }

  /* Resources that force a recalculation of the tab positions. */
  if( Differs( notebook.tab_dir_left )         ||
      Differs( notebook.tab_dir_right )        ||
      Differs( notebook.tab_dir_top )          ||
      Differs( notebook.tab_dir_bottom )       ||
      Differs( notebook.tab_packing )          ||
      Differs( notebook.scroll_offset_top )    ||
      Differs( notebook.scroll_offset_left )   ||
      Differs( notebook.scroll_offset_right )  ||
      Differs( notebook.scroll_offset_bottom ) ||
      Differs( notebook.equalize_tabs_top )    ||
      Differs( notebook.equalize_tabs_left )   ||
      Differs( notebook.equalize_tabs_right )  ||
      Differs( notebook.equalize_tabs_bottom ) )
    visual_changed = True;

  /* If resize setting changed, trigger recalculation. */
  if( Differs( notebook.resize_allowed )         ||
      Differs( notebook.work_area_decides_size ) ||
      Differs( notebook.resize_work_area )       ||
      Differs( notebook.front_page_fixed ))
    visual_changed = True;

  if( Differs( notebook.tab_spacing_top )   ||
      Differs( notebook.tab_spacing_left )  ||
      Differs( notebook.tab_spacing_right ) ||
      Differs( notebook.tab_spacing_bottom ))
    visual_changed = True;

  if( Differs( notebook.work_area )){
    work_area_changed = True;

    /* Make sure that the right child type is set for this widget. */
    /* Needed for correct calculation in DoLayout. */
    {
      XmUbNoteBookConstraints  constraints;
      Widget                   wa;

      wa = (Widget) newW -> notebook.work_area;
      constraints = (XmUbNoteBookConstraints) ( wa -> core.constraints );
      constraints -> notebook.child_type = XmUbWORK_AREA;
    }
  }

  if( Differs( notebook.repeat_center_binding )){

    /* Decline changes that do not comply with predefined pixmaps. */
    switch( newW -> notebook.binding_type ){
      case XmUbBINDING_SOLID:
      case XmUbBINDING_SPIRAL:

        if( ! newW -> notebook.repeat_center_binding ){
          newW -> notebook.repeat_center_binding = True;
        }

        break;

      default:
        break;
    }

    visual_changed = True;
  }

  if( Differs( notebook.binding_type )          ||
      Differs( notebook.binding_side )          ||
      Differs( notebook.binding_pixmap_center ) ||
      Differs( notebook.binding_pixmap_start )  ||
      Differs( notebook.binding_pixmap_end )){

    /* Decline changes that do not comply with predefined pixmaps. */
    switch( newW -> notebook.binding_type ){
      case XmUbBINDING_SOLID:
      case XmUbBINDING_SPIRAL:

        if( newW -> notebook.binding_pixmap_start != None ) {
          printf( "Start pixmap disallowed when binding type predefined.\n" );
          newW -> notebook.binding_pixmap_start = None;
        }

        if( newW -> notebook.binding_pixmap_end != None ) {
          printf( "End pixmap disallowed when binding type predefined.\n" );
          newW -> notebook.binding_pixmap_end = None;
        }

        break;

      default:
        break;
    }


    SetBindingPixmap( newW, curW -> notebook.binding_pixmap_center );

    visual_changed = True;
  }

  /***********************************************************************/

  newW -> notebook.internal_resize = True;

  /* Find out the preferred geometries. */
  GetPreferredSize( newW, NULL, NULL, &preferred_sizes );

  if( visual_changed ){

    /* May need to try a resize. */
    if( newW -> notebook.resize_allowed )
      layout_done = ResizeIfNeeded( newW, &preferred_sizes );
  }

  if( !layout_done && ( new_kid_pos || visual_changed || work_area_changed ))
    DoLayout( newW, &preferred_sizes );

  newW -> notebook.internal_resize = False;


  return( new_kid_pos || redisplay || visual_changed || work_area_changed );


#undef Differs

} /* SetValues */


/*----------------------------------------------------------------------*/

static int
  TabNumber( XmUbNoteBookWidget  w,
             Widget              tab,
             sideAttachment      side )
{

  /* Variables. */
  XmUbNoteBookConstraints  constraints;
  int                      index;
  Widget                   kid;
  int                      kid_index = 0;


  /* Code. */

  for( index = 0; index < w -> composite.num_children; index ++ ){
    kid = w -> composite.children[ index ];

    constraints = (XmUbNoteBookConstraints) kid -> core.constraints;

    if( XtIsManaged( kid ) &&
        ( constraints -> notebook.child_type == XmUbTAB ) &&
        ( constraints -> notebook.side       == side ) ){

      if( kid == tab )
        return kid_index;

      kid_index ++;

    }
  }


  return ( INVALID_TAB_NUMBER );

} /* TabNumber */


/*----------------------------------------------------------------------*/

Widget  
  XmUbCreateNoteBook( Widget    parent,
                      String    name,
                      ArgList   arglist,
                      Cardinal  argcount )
{
  /* Code. */

  return XtCreateWidget( name, xmUbNoteBookWidgetClass, 
                         parent, arglist, argcount );

} /* XmUbCreateNoteBook */


/*----------------------------------------------------------------------*/

void
  XmUbNoteBookDeselectTab( Widget   notebook,
                           int      side,
                           Boolean  redisplay )
{

  /* Variables. */
  NoteBookSizes       preferred_sizes;
  XmUbNoteBookWidget  w;


  /* Code. */

  w = (XmUbNoteBookWidget) notebook;

  w -> notebook.selected_tabs[ side ]  = NULL;
  w -> notebook.unstacked_tabs[ side ] = True;

  /* Trigger repositioning. */
  if( redisplay ){
    GetPreferredSize( w, NULL, NULL, &preferred_sizes );
    DoLayout( w, &preferred_sizes );
  }

  return;

} /* XmUbNoteBookDeselectTab */


/*----------------------------------------------------------------------*/


void
  XmUbNoteBkRestackTabs( Widget  notebook,
                         int     side )
{

  /* Variables. */
  int                      active_num;
  XmUbNoteBookConstraints  constraints;
  int                      index;
  Widget                   kid;
  int                      num_tabs;
  XmUbNoteBookWidget       w;
  int                      wi;
  Window                   *windows;


  /* Code. */

  w = (XmUbNoteBookWidget) notebook;

  /* Tabs are stacked with the first one in the array on top, so
     find out which one that is active and how many there are. */

  active_num = w -> composite.num_children;
  num_tabs   = 0;

  for( index = 0; index < w -> composite.num_children; index ++ ){
    kid = w -> composite.children[ index ];
  
    constraints = (XmUbNoteBookConstraints) kid -> core.constraints;

    if( ( constraints -> notebook.side == side ) &&
        ( constraints -> notebook.child_type == XmUbTAB ) &&
        XtIsManaged( kid ) ){

      num_tabs ++;

      if( w -> notebook.selected_tabs[ side ] == kid )
        active_num = index;

    }
  }

  if( num_tabs == 0 )
    return;

  /* We know how many tabs we should have, allocate array. */
  windows = (Window *) SysMalloc( sizeof( Window ) * num_tabs );

  /* Fill the array starting with the active tab. */

  wi = 0;
  for( index = active_num; index < w -> composite.num_children; index ++ ){
    kid = w -> composite.children[ index ];
    constraints = (XmUbNoteBookConstraints) kid -> core.constraints;

    if( ( constraints -> notebook.side == side ) &&
        ( constraints -> notebook.child_type == XmUbTAB ) &&
        XtIsManaged( kid ) && XtWindowOfObject( kid ) != None ) {
      windows[ wi ] = XtWindow( kid );
      wi++;
    }
  }

  for( index = 0; index < active_num; index ++ ){
    kid = w -> composite.children[ index ];
    constraints = (XmUbNoteBookConstraints) kid -> core.constraints;

    if( ( constraints -> notebook.side == side ) &&
        ( constraints -> notebook.child_type == XmUbTAB ) &&
        XtIsManaged( kid ) && XtWindowOfObject( kid ) != None ) {
      windows[ wi ] = XtWindow( kid );
      wi++;
    }
  }

  if( wi == 0 )
    return;

  /* Stack the kids. */

  XRestackWindows( XtDisplay( w ), windows, wi );


  SysFree( windows );

  return;

} /* XmUbNoteBkRestackTabs */


/*----------------------------------------------------------------------*/

void
  XmUbNoteBookSelectedTab( Widget  notebook,
                           Widget  selected,
                           int     side )
{

  /* Variables. */
  XmUbNoteBookWidget  w;
  NoteBookSizes       preferred_sizes;


  /* Code. */

  w = (XmUbNoteBookWidget) notebook;

  /* If the same tab is selected again, no reconfig is needed. */
  if( w -> notebook.selected_tabs[ side ] == selected )
    return;

  w -> notebook.selected_tabs[ side ]  = selected;
  w -> notebook.unstacked_tabs[ side ] = True;

  /* Trigger repositioning. */
  GetPreferredSize( w, NULL, NULL, &preferred_sizes );
  DoLayout( w, &preferred_sizes );


  return;

} /* XmUbNoteBookSelectedTab */
