/*----------------------------------------------------------------------------
--
--  Module:           XmUbArrowLabel
--
--  Project:          XmUb - Ulle's Motif widgets
--  System:           
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    This is the implementation of the widget.
--
--  Filename:         XmUbArrLab.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-04-28
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: XmUbArrLab.c, Version: 1.1, Date: 95/02/18 15:10:06";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <Xm/XmP.h>

#include <Xm/Label.h>
#include <Xm/ArrowB.h>

/* Private widget header file. */
#include "XmUbArrLabP.h"

/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

#ifdef MAX
#undef MAX
#endif

#ifdef MIN
#undef MIN
#endif

#define MAX( x, y )  ( ( x ) > ( y ) ? ( x ) : ( y ) )
#define MIN( x, y )  ( ( x ) < ( y ) ? ( x ) : ( y ) )

#define XmUbAL_CHILD_ERROR  -1

/* Resource converters. */
#define XmRarrowPlacement  "ArrowPlacement"


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

typedef struct {

  Dimension  width;
  Dimension  height;
  Position   x;
  Position   y;

} KidDimensionRec;

/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Internal Motif functions (how do we know about this....?) */
extern void
  _XmBackgroundColorDefault();


/* The arrow label resource list. */
static XtResource resources[] = {
  {
    XmNactivateCallback,
    XmCCallback,
    XmRCallback, sizeof( XtCallbackList ),
    XtOffset( XmUbArrowLabelWidget, al.activate_callback ),
    XmRCallback,
    NULL
  },
  {
    XmNlabelString,
    XmCXmString,
    XmRXmString, sizeof( XmString ),
    XtOffset( XmUbArrowLabelWidget, al.label_string ),
    XtRImmediate,
    (XtPointer) NULL
  },
  {
    XmNmarginHeight,
    XmCMarginHeight,
    XtRDimension, sizeof( Dimension ),
    XtOffset( XmUbArrowLabelWidget, al.margin_height ),
    XtRImmediate,
    (XtPointer) 0
  },
  {
    XmNmarginWidth,
    XmCMarginWidth,
    XtRDimension, sizeof( Dimension ),
    XtOffset( XmUbArrowLabelWidget, al.margin_width ),
    XtRImmediate,
    (XtPointer) 0
  },
  {
    XmUbNalArrowOrientation,
    XmCOrientation,
    XmROrientation, sizeof( unsigned char ),
    XtOffset( XmUbArrowLabelWidget, al.arrow_orientation ),
    XmRImmediate,
    (XtPointer) XmHORIZONTAL
  },
  {
    XmUbNalArrowPlacement,
    XmUbCAlArrowPlacement,
    XmRarrowPlacement, sizeof( arrowPlacement ),
    XtOffset( XmUbArrowLabelWidget, al.arrow_placement ),
    XmRImmediate,
    (XtPointer) XmUbARROWS_SPREAD_OUT
  },
  {
    XmNspacing,
    XmCSpacing,
    XtRDimension, sizeof( Dimension ),
    XtOffset( XmUbArrowLabelWidget, al.spacing ),
    XtRImmediate,
    (XtPointer) 2
  },
  {
    XmUbNalRecomputeHeight,
    XmUbCAlRecomputeHeight,
    XtRBoolean, sizeof( Boolean ),
    XtOffset( XmUbArrowLabelWidget, al.recompute_height ),
    XtRImmediate,
    (XtPointer) True
  },
  {
    XmUbNalRecomputeWidth,
    XmUbCAlRecomputeWidth,
    XtRBoolean, sizeof( Boolean ),
    XtOffset( XmUbArrowLabelWidget, al.recompute_width ),
    XtRImmediate,
    (XtPointer) True
  },

};  /* resources */


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

/* Class methods. */
static void              
  ClassInitialize();

static void
  ChangeManaged( Widget  widget );

static void
  DeleteChild( Widget  widget );

static void 
  Destroy( Widget   widget );

static XtGeometryResult
  GeometryManager( Widget            widget,
                   XtWidgetGeometry  *request,
                   XtWidgetGeometry  *reply );

static void
  Initialize( Widget     treq,
              Widget     tnew,
              ArgList    args,
              Cardinal   *num_args );

static void
  InsertChild( Widget  widget );

static XtGeometryResult
  QueryGeometry( Widget             widget,
                 XtWidgetGeometry  *proposed,
                 XtWidgetGeometry  *answer );

static void
  Resize( Widget    widget );

static Boolean 
  SetValues( Widget     current,
             Widget     request,
             Widget     new,
             ArgList    args,
             Cardinal   *num_args );


/* Internal functions. */


static void
  ArrowButtonActivatedCB( Widget                       arrow_button,
                          XmUbArrowLabelWidget         al,
                          XmArrowButtonCallbackStruct  *call_data );

/* If not in the child list, returns XmUbMD_CHILD_ERROR. */
static int
  ChildIndex( XmUbArrowLabelWidget  al,
              Widget                child );

static void
  ConvertToArrowPlacement( XrmValue   *args,
                           Cardinal   *num_args,
                           XrmValue   *from,
                           XrmValue   *to );


static void
  CreateInternalWidgets( XmUbArrowLabelWidget  al );

/* The sizes array  must have been processed by GetChildPrefSizes 
   before DoLayout is called. */
static void
  DoLayout( XmUbArrowLabelWidget  al,
            KidDimensionRec       sizes[] );

/* Fills in the width and height fields in the kids dimension recs. */
static void
  GetChildPrefSizes( XmUbArrowLabelWidget  al,
                     Widget                initiator,
                     XtWidgetGeometry      *request,
                     KidDimensionRec       sizes[] );


/* The sizes array must have been prepared by GetChildPrefSizes before this
   function is called. */
static void
  GetOwnPreferredSize( XmUbArrowLabelWidget  al,
                       KidDimensionRec       sizes[],
                       Dimension             *pref_width,
                       Dimension             *pref_height );

static void
  GetValuesHook( Widget    w,
                 ArgList   args,
                 Cardinal  *num_args );

static void 
  KidsPreferredGeometry( Widget            kid,
                         Widget            initiator,
                         XtWidgetGeometry  *request,
                         XtWidgetGeometry  *desired );

/* ResizeIfNeeded fills in the sizes array. Does not have to be initialized. */
static Boolean
  ResizeIfNeeded( XmUbArrowLabelWidget  al,
                  KidDimensionRec       sizes[] );

static void
  WarningNoResourceChange( XmUbArrowLabelWidget  al,
                           String                resource );

/*----------------------------------------------------------------------------
--  Initialization of the class record.
----------------------------------------------------------------------------*/

/* This initialization has to be done after the methods have been declared. */
XmUbArrowLabelClassRec xmUbArrowLabelClassRec = {

  { /* Core class fields. */
    /* superclass */                    (WidgetClass) &xmManagerClassRec,
    /* class_name */                    "XmUbArrowLabel",
    /* widget_size */                   sizeof( XmUbArrowLabelRec ),
    /* class_initialize */              ClassInitialize,
    /* class_part_initialize */         NULL,
    /* class_inited */                  False,
    /* initialize */                    Initialize,
    /* initialize_hook */               NULL,
    /* realize */                       XtInheritRealize, 
    /* actions */                       NULL,
    /* num_actions */                   0,
    /* resources */                     resources,
    /* num_resources */                 XtNumber( resources ),
    /* xrm_class */                     NULLQUARK,
    /* compress_motion */               True,
    /* compress_exposure */             True,
    /* compress_enterleave */           True,
    /* visible_interest */              False,
    /* destroy */                       Destroy,
    /* resize */                        Resize,
    /* expose */                        NULL,
    /* set_values */                    SetValues,
    /* set_values_hook */               NULL,
    /* set_values_almost */             XtInheritSetValuesAlmost,
    /* get_values_hook */               GetValuesHook,
    /* accept_focus */                  NULL,
    /* version */                       XtVersion,
    /* callback_private */              NULL,
    /* tm_table */                      NULL,
    /* query_geometry */                QueryGeometry,
    /* display_accelerator */           XtInheritDisplayAccelerator,
    /* extension */                     NULL
  },
  { /* Composite class part. */
    /* geometry_manager */              GeometryManager,
    /* change_managed */                ChangeManaged,
    /* insert_child */                  InsertChild,
    /* delete_child */                  DeleteChild,
    /* extension */                     NULL
  },
  { /* Constraint class fields. */
    /* subresources */                  NULL,
    /* subresource_count */             0,
    /* constraint_size */               sizeof( 
                                          XmUbArrowLabelConstraintsRec ),
    /* initialize */                    NULL,
    /* destroy */                       NULL,
    /* set_values */                    NULL,
    /* extension */                     NULL
  },
  { /* XmManager class part. */
    /* translations */                  NULL,
    /* get_resources */                 NULL,
    /* num_get_resources */             0,
    /* get_constraint_resources */      NULL,
    /* num_get_constraint_resources */  0,
    /* extension */                     NULL
  },
  { /* Arrow label class part. */
    /* extension */                     NULL
  },

}; 


/* Class record pointer. */
WidgetClass 
  xmUbArrowLabelWidgetClass = (WidgetClass) &xmUbArrowLabelClassRec;



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/


static void
  ArrowButtonActivatedCB( Widget                       arrow_button,
                          XmUbArrowLabelWidget         al,
                          XmArrowButtonCallbackStruct  *call_data )
{
  /* Variables. */
  XmUbArrowLabelCallbackStruct  cb;
  int                           child_index;

  /* Code. */

  /* Find out if it was the forwards or backwards button that was pressed. */
  child_index = ChildIndex( al, arrow_button );

  switch( child_index ){

    case XmUbAL_CHILD_BACK_ARROW:
      cb.reason = XmUbCR_BACK_ARROW_ACTIVATED;
      break;

    case XmUbAL_CHILD_FORWARD_ARROW:
      cb.reason = XmUbCR_FORWARD_ARROW_ACTIVATED;
      break;

    default:
      /* Something is wrong. */
      return;
  } 

  cb.event = call_data -> event;
  cb.child = arrow_button;

  /* Invoke the callbacks. */
  XtCallCallbackList( (Widget) al, al -> al.activate_callback,
                      (XtPointer) &cb );


  return;

} /* ArrowButtonActivatedCB */


/*----------------------------------------------------------------------*/

static void              
  ClassInitialize()
{
  /* Code. */

  /* Register type converters. */
  XtAddConverter( XmRString, XmRarrowPlacement,
                  ConvertToArrowPlacement, NULL, 0 );

  return;

} /* ClassInitialize */


/*----------------------------------------------------------------------*/

static void
  ChangeManaged( Widget  widget )
{
  /* Variables. */
  Boolean               layout_done;
  KidDimensionRec       kids_sizes[ NO_INTERNAL_CHILDREN ];
  XmUbArrowLabelWidget  al;


  /* Code. */

  al = (XmUbArrowLabelWidget) widget;

  /* ResizeIfNeeded checks the size of all children and resizes the widget. */
  layout_done = ResizeIfNeeded( al, kids_sizes );

  /* Do the layout. */
  if( !layout_done )
    DoLayout( al, kids_sizes );


  return;

} /* ChangeManaged */


/*----------------------------------------------------------------------*/

static int
  ChildIndex( XmUbArrowLabelWidget  al,
              Widget                child )
{
  /* Variables. */
  int  index;

  /* Code. */

  for( index = 0; index < NO_INTERNAL_CHILDREN; index ++ ){

    if( al -> al.internal_children[ index ] == child )
      return index;

  } /* for */

 
  /* Specified child not found. */
  return XmUbAL_CHILD_ERROR;

} /* ChildIndex */


/*----------------------------------------------------------------------*/

static void
  ConvertToArrowPlacement( XrmValue   *args,
                           Cardinal   *num_args,
                           XrmValue   *from,
                           XrmValue   *to )
{

  /* Variables. */
  static arrowPlacement  conv_arrow_placement;


  /* Code. */

  if( *num_args != 0 )
    XtWarningMsg( "wrongParameters", "ConvertToArrowPlacement", 
      "XtToolkitError", "Conversion needs no extra arguments", 
      (String *) NULL, (Cardinal *) NULL );

  if( strcmp( (char *) from -> addr, "ARROWS_LEFT" ) == 0 )
    conv_arrow_placement = XmUbARROWS_LEFT;

  else if( strcmp( (char *) from -> addr, "ARROWS_SPREAD_OUT" ) == 0 )
    conv_arrow_placement = XmUbARROWS_SPREAD_OUT;

  else if( strcmp( (char *) from -> addr, "ARROWS_RIGHT" ) == 0 )
    conv_arrow_placement = XmUbARROWS_RIGHT;

  else {
    XtWarningMsg( "wrongParameters", "ConvertToArrowPlacement", 
      "XtToolkitError", "Cannot convert to type arrowPlacement",
      (String *) NULL, (Cardinal *) 0 );
    conv_arrow_placement = XmUbARROWS_SPREAD_OUT;
  }

  ( *to ).size = sizeof( arrowPlacement );
  ( *to ).addr = (XtPointer) &conv_arrow_placement;


  return;

} /* ConvertToArrowPlacement */


/*----------------------------------------------------------------------*/

static void
  CreateInternalWidgets( XmUbArrowLabelWidget  al )
{
  /* Variables. */
  Arg           args[ 5 ];
  Cardinal      n;
  char          *name_buffer;
  String        wname;

  /* Code. */

  /* Get the name of the "parent" widget. */
  wname = XtName( (Widget)al );

  name_buffer = XtMalloc( strlen( wname ) + 4 );

  /* The two arrow buttons. */
  /* Backwards button. */
  sprintf( name_buffer, "%sBa", wname );
  n = 0;
  if( al -> al.arrow_orientation == XmHORIZONTAL ){
    XtSetArg( args[ n ], XmNarrowDirection, XmARROW_LEFT ); n++;
  } else {
    XtSetArg( args[ n ], XmNarrowDirection, XmARROW_UP ); n++;
  }

  al -> al.internal_children[ XmUbAL_CHILD_BACK_ARROW ] = 
    XmCreateArrowButton( (Widget) al, name_buffer, args, n );

  XtAddCallback( al -> al.internal_children[ XmUbAL_CHILD_BACK_ARROW ],
                 XmNactivateCallback, 
                 (XtCallbackProc) ArrowButtonActivatedCB, (XtPointer) al );

  /* Forward button. */
  sprintf( name_buffer, "%sFa", wname );
  n = 0;
  if( al -> al.arrow_orientation == XmHORIZONTAL ){
    XtSetArg( args[ n ], XmNarrowDirection, XmARROW_RIGHT ); n++;
  } else {
    XtSetArg( args[ n ], XmNarrowDirection, XmARROW_DOWN ); n++;
  }

  al -> al.internal_children[ XmUbAL_CHILD_FORWARD_ARROW ] = 
    XmCreateArrowButton( (Widget) al, name_buffer, args, n );

  XtAddCallback( al -> al.internal_children[ XmUbAL_CHILD_FORWARD_ARROW ],
                 XmNactivateCallback, 
                 (XtCallbackProc) ArrowButtonActivatedCB, (XtPointer) al );

  /* The label. */
  sprintf( name_buffer, "%sLb", wname );

  n = 0;
  if( al -> al.label_string != NULL ){
    XtSetArg( args[ n ], XmNlabelString, al -> al.label_string ); n++;
  }

  al -> al.internal_children[ XmUbAL_CHILD_LABEL ] = 
    XmCreateLabel( (Widget) al, name_buffer, args, n );


  XtManageChildren( al -> al.internal_children, NO_INTERNAL_CHILDREN );

  XtFree( name_buffer );


  return;

} /* CreateInternalWidgets */


/*----------------------------------------------------------------------*/

static void
  DeleteChild( Widget  widget )
{

  /* Variables. */
  int                   index;
  XmUbArrowLabelWidget  al;

  /* Code. */

  al = (XmUbArrowLabelWidget) XtParent( widget );

  /* Clear the internal reference. */
  for( index = 0; index < NO_INTERNAL_CHILDREN; index ++ ){
    if( al -> al.internal_children[ index ] == widget ){
      al -> al.internal_children[ index ] = NULL;
      break;
    }
  }

  /* Perform the actual operation */
  (* ( (CompositeWidgetClass) (xmUbArrowLabelWidgetClass ->
     core_class.superclass) ) -> composite_class.delete_child ) ( widget );


  return;

} /* DeleteChild */


/*----------------------------------------------------------------------*/

static void 
  Destroy( Widget   widget )
{
  /* Variables. */
  XmUbArrowLabelWidget  al;

  /* Code. */

  al = (XmUbArrowLabelWidget) widget;

  /* Free copied XmStrings. */
  if( al -> al.label_string != NULL )
    XmStringFree( al -> al.label_string );

  /* Remove callbacks. */
  XtRemoveAllCallbacks( widget, XmNactivateCallback );


  return;

} /* Destroy */


/*----------------------------------------------------------------------*/

static void
  DoLayout( XmUbArrowLabelWidget  al,
            KidDimensionRec       sizes[] )
{
  /* Variables. */
  int        index;
  Widget     kid;
  Dimension  max_height;
  Dimension  offset;

  /* Code. */

  /* Let the arrows keep their sizes, and let the label size shrink and 
     grow. */

  /* Start by calculating the label's width. */
  offset = sizes[ XmUbAL_CHILD_BACK_ARROW ].width +
           sizes[ XmUbAL_CHILD_FORWARD_ARROW ].width +
           2 * al -> al.margin_width + 2 * al -> al.spacing;

  if( offset > al -> core.width )
    sizes[ XmUbAL_CHILD_LABEL ].width = 1;
  else
    sizes[ XmUbAL_CHILD_LABEL ].width = al -> core.width - offset;

  /* The widgets may keep their heights. */

  /* The placement resource decides the x position. */
  switch( al -> al.arrow_placement ){

    case XmUbARROWS_LEFT:
      sizes[ XmUbAL_CHILD_BACK_ARROW ].x = (Position) al -> al.margin_width;

      sizes[ XmUbAL_CHILD_FORWARD_ARROW ].x = 
        sizes[ XmUbAL_CHILD_BACK_ARROW ].x + 
        (Position) sizes[ XmUbAL_CHILD_BACK_ARROW ].width + 
        (Position) al -> al.spacing;

      sizes[ XmUbAL_CHILD_LABEL ].x = sizes[ XmUbAL_CHILD_FORWARD_ARROW ].x +
        (Position) sizes[ XmUbAL_CHILD_FORWARD_ARROW ].width +
        (Position) al -> al.spacing;

      break;

    case XmUbARROWS_RIGHT:
      sizes[ XmUbAL_CHILD_LABEL ].x = (Position) al -> al.margin_width;

      sizes[ XmUbAL_CHILD_BACK_ARROW ].x = sizes[ XmUbAL_CHILD_LABEL ].x +
        (Position) sizes[ XmUbAL_CHILD_LABEL ].width +
        (Position) al -> al.spacing;

      sizes[ XmUbAL_CHILD_FORWARD_ARROW ].x = 
        sizes[ XmUbAL_CHILD_BACK_ARROW ].x + 
        (Position) sizes[ XmUbAL_CHILD_BACK_ARROW ].width + 
        (Position) al -> al.spacing;

      break;

    case XmUbARROWS_SPREAD_OUT:
      sizes[ XmUbAL_CHILD_BACK_ARROW ].x = (Position) al -> al.margin_width;

      sizes[ XmUbAL_CHILD_LABEL ].x = sizes[ XmUbAL_CHILD_BACK_ARROW ].x +
        (Position) sizes[ XmUbAL_CHILD_BACK_ARROW ].width +
        (Position) al -> al.spacing;

      sizes[ XmUbAL_CHILD_FORWARD_ARROW ].x = sizes[ XmUbAL_CHILD_LABEL ].x +
        (Position) sizes[ XmUbAL_CHILD_LABEL ].width + 
        (Position) al -> al.spacing;

      break;

  } /* switch */


  /* Vertical positions. */
  max_height = 0;
  for( index = XmUbAL_FIRST_CHILD; index < NO_INTERNAL_CHILDREN; index ++ ){
    if( sizes[ index ].height > max_height )
      max_height = sizes[ index ].height;
  }

  for( index = XmUbAL_FIRST_CHILD; index < NO_INTERNAL_CHILDREN; index ++ )
    sizes[ index ].y = (Position) ( max_height - sizes[ index ].height ) / 2;

  /* Configure the children. */
  /* All positions and dimensions are now in the sizes array. */
  for( index = XmUbAL_FIRST_CHILD; index < NO_INTERNAL_CHILDREN; index ++ ){

    kid = al -> al.internal_children[ index ];

    if( ( kid != NULL ) && XtIsManaged( kid ) )
      XtConfigureWidget( kid, sizes[ index ].x, sizes[ index ].y,
        sizes[ index ].width, sizes[ index ].height, 
        kid -> core.border_width );
  }


  return;

} /* DoLayout */


/*----------------------------------------------------------------------*/

static XtGeometryResult
  GeometryManager( Widget            widget,
                   XtWidgetGeometry  *request,
                   XtWidgetGeometry  *reply )
{

  XmUbArrowLabelWidget  al;
  XtWidgetGeometry      own_request;
  Dimension             old_width, old_height;
  Dimension             pref_height;
  Dimension             pref_width;
  KidDimensionRec       kids_sizes[ NO_INTERNAL_CHILDREN ];
  XtGeometryResult      result;

  /* Code. */

  al = (XmUbArrowLabelWidget) XtParent( widget );

  /* Find out how big the widget would be if the resize were allowed. */
  GetChildPrefSizes( al, NULL, request, kids_sizes );
  GetOwnPreferredSize( al, kids_sizes, &pref_width, &pref_height );

  /* If no change in dimensions, allow the request. */
  if( ( pref_width == al -> core.width ) && 
      ( pref_height == al -> core.height )){
    DoLayout( al, kids_sizes );
    return XtGeometryYes;
  }

  /* We must ask our parent to resize us. */
  own_request.request_mode = CWWidth | CWHeight;
  own_request.width  = pref_width;
  own_request.height = pref_height;

  /* Save dimensions. */
  old_width  = al -> core.width;
  old_height = al -> core.height;

  al -> al.resize_called = False;

  /* We are not interested in any compromise geometry. */
  result = XtMakeGeometryRequest( (Widget) al, &own_request, NULL );

  /* Reset to old dimensions if request not granted. */
  if( result != XtGeometryYes ){
    al -> core.width  = old_width;
    al -> core.height = old_height;

  } else {
    if( !al -> al.resize_called )
      Resize( (Widget) al );
  }

/* !!!!!!!!!!!!!!!!!!!!!! */
  /* Always grant child's request. */
  return XtGeometryYes;

} /* GeometryManager */


/*----------------------------------------------------------------------*/

static void
  GetChildPrefSizes( XmUbArrowLabelWidget  al,
                     Widget                initiator,
                     XtWidgetGeometry      *request,
                     KidDimensionRec       sizes[] )
{
  /* Variables. */
  XtWidgetGeometry  desired;
  int               index;
  Widget            kid;

  /* Code. */

  /* Initialize. */
  for( index = 0; index < NO_INTERNAL_CHILDREN; index ++ ){
    sizes[ index ].width  = 0;
    sizes[ index ].height = 0;
    sizes[ index ].x      = 0;
    sizes[ index ].y      = 0;
  }

  /* Get the preferred sizes for the children. */
  for( index = XmUbAL_FIRST_CHILD; index < NO_INTERNAL_CHILDREN; index ++ ){

    kid = al -> al.internal_children[ index ];

    if( ( kid != NULL ) && XtIsManaged( kid ) ){

      KidsPreferredGeometry( kid, initiator, request, &desired );

      sizes[ index ].width  = desired.width;
      sizes[ index ].height = desired.height;

    }

  } /* for */

  
  return;

} /* GetChildPrefSizes */


/*----------------------------------------------------------------------*/

static void
  GetOwnPreferredSize( XmUbArrowLabelWidget  al,
                       KidDimensionRec       sizes[],
                       Dimension             *pref_width,
                       Dimension             *pref_height )
{
  /* Code. */

  /* All layouts arrange the widgets horizontally, so calculating the 
     preferred size is simple. */

  *pref_width = sizes[ XmUbAL_CHILD_BACK_ARROW ].width +
                sizes[ XmUbAL_CHILD_FORWARD_ARROW ].width +
                sizes[ XmUbAL_CHILD_LABEL ].width +
                2 * al -> al.margin_width +
                2 * al -> al.spacing;

  *pref_height = MAX( sizes[ XmUbAL_CHILD_LABEL ].height,
                      MAX( sizes[ XmUbAL_CHILD_BACK_ARROW ].height,
                           sizes[ XmUbAL_CHILD_FORWARD_ARROW ].height ) ) +
                 2 * al -> al.margin_height;

  return;

} /* GetOwnPreferredSize */


/*----------------------------------------------------------------------*/

static void
  GetValuesHook( Widget    w,
                 ArgList   args,
                 Cardinal  *num_args )
{
  /* Variables. */
  int                   index;
  XmUbArrowLabelWidget  al;

  /* Code. */

  al = (XmUbArrowLabelWidget) w;

  /* Copy the XmStrings. */
  for( index = 0; index < *num_args; index ++ ){

    if( strcmp( args[ index ].name, XmNlabelString ) == 0 ){
      * ( XmString *) ( args[ index ].value ) = 
        XmStringCopy( al -> al.label_string );

    }
  }


  return;

} /* GetValuesHook */


/*----------------------------------------------------------------------*/

static void
  Initialize( Widget     treq,
              Widget     tnew,
              ArgList    args,
              Cardinal   *num_args )
{
  /* Variables. */
  int                   index;
  KidDimensionRec       kids_sizes[ NO_INTERNAL_CHILDREN ];
  XmUbArrowLabelWidget  new;
  Dimension             pref_height;
  Dimension             pref_width;
  XmString              xm;

  /* Code. */

  new = (XmUbArrowLabelWidget) tnew;

  /* Initialize private fields. */
  /* Nullify all widget ids. */
  for( index = XmUbAL_FIRST_CHILD; index < NO_INTERNAL_CHILDREN; index ++ )
    new -> al.internal_children[ index ] = NULL;

  /* Start size if none supplied. */
  if( new -> core.width == 0 )
    new -> core.width = 300;
  else
    new -> al.recompute_width = False;

  if( new -> core.height == 0 )
    new -> core.height = 100;
  else
    new -> al.recompute_height = False;

  /* Copy XmStrings. */
  if( new -> al.label_string != NULL ){
    xm = XmStringCopy( new -> al.label_string );
    new -> al.label_string = xm;
  }


  /* Create the internal widgets. */
  new -> al.internal_widgets_created = False;

  CreateInternalWidgets( new );

  new -> al.internal_widgets_created = True;

  /* Set the desired size of the widget. */
  GetChildPrefSizes( new, NULL, NULL, kids_sizes );
  GetOwnPreferredSize( new, kids_sizes, &pref_width, &pref_height );

  new -> core.width  = pref_width;
  new -> core.height = pref_height;

  DoLayout( new, kids_sizes );


  return;

} /* Initialize */


/*----------------------------------------------------------------------*/

static void
  InsertChild( Widget  widget )
{

  /* Variables. */
  Cardinal              num_params;
  String                params[ 1 ];
  XmUbArrowLabelWidget  al;

  /* Code. */

  al = (XmUbArrowLabelWidget) XtParent( widget );

  /* We do not allow the application to create children. */
  if( al -> al.internal_widgets_created ){

    params[ 0 ] = XtClass( (Widget) al ) -> core_class.class_name;
    num_params  = 1;
    XtAppErrorMsg( XtWidgetToApplicationContext( widget ),
                   "childError", "number", "WidgetError",
                   "Applications cannot add children to %s widgets.",
                   params, &num_params );
  }

  /* This is an internal child. Adding it is OK. */
  (* ( (CompositeWidgetClass) (xmUbArrowLabelWidgetClass ->
     core_class.superclass) ) -> composite_class.insert_child ) ( widget );


  return;

} /* InsertChild */


/*----------------------------------------------------------------------*/

static void 
  KidsPreferredGeometry( Widget            kid,
                         Widget            initiator,
                         XtWidgetGeometry  *request,
                         XtWidgetGeometry  *desired )
{

  /* Code. */

  if( ( kid == initiator ) && ( request != NULL ) ){
    /* The initiator should not be queried. */
    if( request -> request_mode & CWWidth )
      desired -> width = request -> width;
    else
      desired -> width = (Dimension) kid -> core.width;

    if( request -> request_mode & CWHeight )
      desired -> height = request -> height;
    else
      desired -> height = (Dimension) kid -> core.height;

  } else
    (void) XtQueryGeometry( kid, NULL, desired );


  return;

} /* KidsPreferredGeometry */


/*----------------------------------------------------------------------*/

static void
  Resize( Widget    widget )
{
  /* Variables. */
  KidDimensionRec       kids_sizes[ NO_INTERNAL_CHILDREN ];
  XmUbArrowLabelWidget  al;

  /* Code. */

  al = (XmUbArrowLabelWidget) widget;

  al -> al.resize_called = True;

  /* We have to get the preferred size of the children before we organize
     the layout. */
  GetChildPrefSizes( al, NULL, NULL, kids_sizes );
  DoLayout( al, kids_sizes );


  return;

} /* Resize */


/*----------------------------------------------------------------------*/

static Boolean
  ResizeIfNeeded( XmUbArrowLabelWidget  al,
                  KidDimensionRec       sizes[] )
{

  /* Variables. */
  Boolean           layout_done;
  Dimension         pref_height;
  Dimension         pref_width;
  XtWidgetGeometry  request;
  XtGeometryResult  result;

  /* Code. */

  /* Initialize. */
  layout_done = False;

  /* Get the preferred dimensions of the widget. */
  GetChildPrefSizes( al, NULL, NULL, sizes );
  GetOwnPreferredSize( al, sizes, &pref_width, &pref_height );

  /* If we want the same dimensions, no resizing is needed. */
  if(( pref_width  == al -> core.width ) &&
     ( pref_height == al -> core.height ))
    return False;

  /* Dimensions are different. Try to resize. */
  request.request_mode = CWWidth | CWHeight;

  request.width  = pref_width;
  request.height = pref_height;

  al -> al.resize_called = False;

  do {

    result = XtMakeGeometryRequest( (Widget) al, &request, &request );

  } while( result == XtGeometryAlmost );

  if( result == XtGeometryNo )
    return False;

  /* Resize done. Core fields have already been updated. */
  return( al -> al.resize_called );

} /* ResizeIfNeeded */


/*----------------------------------------------------------------------*/

static XtGeometryResult
  QueryGeometry( Widget             widget,
                 XtWidgetGeometry  *proposed,
                 XtWidgetGeometry  *answer )
{

  KidDimensionRec       kids_sizes[ NO_INTERNAL_CHILDREN ];
  XmUbArrowLabelWidget  al;
  Dimension             pref_height;
  Dimension             pref_width;

  /* Code. */

  al = (XmUbArrowLabelWidget) widget;

  /* Get dimensions that we *really* want. */
  GetChildPrefSizes( al, NULL, NULL, kids_sizes );
  GetOwnPreferredSize( al, kids_sizes, &pref_width, &pref_height );

  answer -> request_mode = CWWidth | CWHeight;
  answer -> width  = pref_width;
  answer -> height = pref_height;


  if( proposed == NULL ){
    /* This is a query for the requested geometry. */

    if(( answer -> height == (int) al -> core.height ) &&
       ( answer -> width  == (int) al -> core.width ))
      return XtGeometryNo;
    else
      return XtGeometryAlmost;
  }

  /* The parent supplied a geometry suggestion. */
  if(( answer -> height == proposed -> height ) &&
     ( answer -> width  == proposed -> width ))
    return XtGeometryYes;

  if( ( proposed -> height <= 1 ) ||
      ( proposed -> width  <= 1 ) )
    /* That's too small ! */
    return XtGeometryNo;


  /* Only a compromise left. */
  return XtGeometryAlmost;

} /* QueryGeometry */


/*----------------------------------------------------------------------*/

static Boolean 
  SetValues( Widget     current,
             Widget     request,
             Widget     new,
             ArgList    args,
             Cardinal   *num_args )
{
#ifdef Differs
#undef Differs
#endif

#define Differs( field )  ( curW -> field != newW -> field )

  /* Variables. */
  XmUbArrowLabelWidget  curW;
  Arg                   local_args[ 4 ];
  Cardinal              n;
  XmUbArrowLabelWidget  newW;
  Boolean               visual_changed = False;
  XmString              xm;

  /* Code. */

  curW = (XmUbArrowLabelWidget) current;
  newW = (XmUbArrowLabelWidget) new;

  /* Width and height. */
  /* Resizing is handled higher up. */

  if( Differs( al.recompute_width ) && newW -> al.recompute_width )
    visual_changed = True;
  else if( Differs( core.width ) )
    newW -> al.recompute_width = False;

  if( Differs( al.recompute_height ) && newW -> al.recompute_height )
    visual_changed = True;
  else if( Differs( core.height ) )
    newW -> al.recompute_height = False;

  /* Margins and spacings. */
  if( Differs( al.margin_width )  ||
      Differs( al.margin_height ) ||
      Differs( al.spacing ) )
    visual_changed = True;

  /* These resources may only be set at creation. */
  if( Differs( al.arrow_orientation ) ){
    WarningNoResourceChange( newW, "XmUbNalArrowOrientation" );
    newW -> al.arrow_orientation = curW -> al.arrow_orientation;
  }
  if( Differs( al.arrow_placement ) ){
    WarningNoResourceChange( newW, "XmUbNalArrowPlacement" );
    newW -> al.arrow_placement = curW -> al.arrow_placement;
  }

  /* These resources affect child widgets. */
  if( Differs( al.label_string ) ||
      Differs( manager.foreground ) ||
      Differs( core.background_pixel ) ){

    n = 0;
    if( Differs( manager.foreground ) ){
      XtSetArg( local_args[ n ], XmNforeground, 
                                 newW -> manager.foreground ); n++;
    }
    if( Differs( core.background_pixel ) ){
      XtSetArg( local_args[ n ], XmNbackground, 
                                 newW -> core.background_pixel ); n++;
    }

    if( n != 0 ){
      XtSetValues( newW -> al.internal_children[ XmUbAL_CHILD_BACK_ARROW ],
                   local_args, n );
      XtSetValues( newW -> al.internal_children[ XmUbAL_CHILD_FORWARD_ARROW ],
                   local_args, n );
    }

    /* The label is also affected by a label string change. */
    if( Differs( al.label_string ) ){
      if( curW -> al.label_string != NULL )
        XmStringFree( curW -> al.label_string );

      if( newW -> al.label_string != NULL ){
        xm = XmStringCopy( newW -> al.label_string );
        newW -> al.label_string = xm;
      }

      XtSetArg( local_args[ n ], XmNlabelString, 
                                 newW -> al.label_string ); n++;
    }

    if( n != 0 )
      XtSetValues( newW -> al.internal_children[ XmUbAL_CHILD_LABEL ], 
                   local_args, n );
  }


  if( visual_changed ){

    KidDimensionRec  kids_sizes[ NO_INTERNAL_CHILDREN ];
    Boolean          resized;

    /* Code. */

    resized = ResizeIfNeeded( newW, kids_sizes );

    /* If the widget was resized, the layout has already been done. */
    if( !resized )
      DoLayout( newW, kids_sizes );

  }

  return( visual_changed );

#undef Differs

} /* SetValues */


/*----------------------------------------------------------------------*/

static void
  WarningNoResourceChange( XmUbArrowLabelWidget  al,
                           String                resource )
{
  /* Variables. */
  Cardinal  num_params;
  String    params[ 2 ];

  /* Code. */

  params[ 0 ] = resource;
  params[ 1 ] = XtClass( al ) -> core_class.class_name;
  num_params  = 2;
  XtAppWarningMsg( XtWidgetToApplicationContext( (Widget) al ),
		   "resourceError", "setValues", "WidgetError",
		   "Resource %s may not be changed in %s widgets.",
		   params, &num_params );

  return;

} /* WarningNoResourceChange */


/*----------------------------------------------------------------------*/

Widget  
  XmUbCreateArrowLabel( Widget    parent,
                        String    name,
                        ArgList   arglist,
                        Cardinal  argcount )
{

  /* Code. */

  return XtCreateWidget( name, xmUbArrowLabelWidgetClass, 
                         parent, arglist, argcount );

} /* XmUbCreateArrowLabel */


/*----------------------------------------------------------------------*/

Widget 
  XmUbArrowLabelGetChild( Widget  widget,
                          int     child )
{
  /* Variables. */
  XmUbArrowLabelWidget  al;

  /* Code. */

  al = (XmUbArrowLabelWidget) widget;


  return( al -> al.internal_children[ child ] );

} /* XmUbArrowLabelGetChild */
