/*----------------------------------------------------------------------------
--
--  Module:           xtmFormatPrintout
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Prepares a printout. The input is a file created by the printer 
--    routine in XDiary (or by another program).
--
--  Filename:         xtmPrFormat.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1993-10-05
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmPrFormat.c, Version: 1.1, Date: 95/02/18 16:08:48";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <time.h>

#include <X11/Intrinsic.h>

#include "System.h"
#include "Message.h"
#include "TimDate.h"

#include "msgXdiary.h"
#include "xtmDbTools.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Name of program. */
#define PROGRAM_NAME   "xdprformat"

/* Start of backup entry. */
#define ENTRY_START  "@!XDiaryDump"

/* Print file identifier. */
#define PRINTER_FILE_ID  "#%XDiaryPrint"

/* Parameter keys. */
#define KEY_CALENDAR          "#%Calendar"
#define KEY_DATE_RANGE        "#%DateRange"
#define KEY_TIME_MIN          "#%TimeMin"
#define KEY_TIME_MAX          "#%TimeMax"
#define KEY_DEFAULT_TIME_MIN  "#%DefaultTimeMin"
#define KEY_DEFAULT_TIME_MAX  "#%DefaultTimeMax"
#define KEY_DATE_FORMAT       "#%DateFormat"
#define KEY_TIME_FORMAT       "#%TimeFormat"
#define KEY_ONLY_WORKDAYS     "#%PrOnlyWorkdays"
#define KEY_3D_LOOK           "#%PrIn3d"

#define max( a, b )  (a < b ? b : a )


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Printout parameters. */
typedef struct {
  Boolean       only_workdays;
  Boolean       print_3d_look;
  char          calendar_name[ 50 ];
  char          date_format[ 50 ];
  char          time_format[ 50 ];
  TIM_TIME_REF  start_date;
  TIM_TIME_REF  end_date;
  TIM_TIME_REF  start_time;
  TIM_TIME_REF  end_time;
} PRINT_PARAM, *PRINT_PARAM_REF;

/* One single entry. */
typedef struct {
  int                    duration;
  UINT32                 flags;
  char                   tag[ XTM_DB_TAG_LEN + 1 ];
  char                   *text;
  TIM_TIME_REF           date_stamp;
  TIM_TIME_REF           time_stamp;
  XTM_DB_ENTRY_TYPE      entry_type;
  XTM_DB_ENTRY_CATEGORY  entry_category;
} ENTRY, *ENTRY_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of program. */
static char  *program_name;

/* Name of module. */
static char  *module_name = "xtmFormatPrintout";

/* Name of text domain. */
static char  *text_domain = "XDiary";

/* Global settings. */
static char  *msg_language;
static char  *output_file;
static char  *paper_size;
static char  *prologue_dir;
static char  *sheets_per_page;
static char  *style;

static FILE  *input_file_ref;
static FILE  *output_file_ref;

static PRINT_PARAM  print_param;

/* Font-encoding for ISO-8859-1 character set. */
static int  ps_iso_encoding[] = {
  241, /* exclamdown */
  242, /* cent */
  243, /* sterling */
  250, /* currency */
  245, /* yen */
  246, /* brokenbar */
  247, /* section */
  310, /* dieresis */
  200, /* copyright */
  343, /* ordfeminine */
  253, /* guillemotleft */
  201, /* logicalnot */
  255, /* hyphen */
  202, /* registered */
  305, /* macron */
  312, /* degree */
  203, /* plusminus */
  262, /* twosuperior */
  263, /* threesuperior */
  302, /* acute */
  204, /* mu */
  266, /* paragraph */
  264, /* periodcentered */
  313, /* cedilla */
  271, /* onesuperior */
  353, /* ordmasculine */
  273, /* guillemotright */
  205, /* onequarter */
  206, /* onehalf */
  207, /* threequarters */
  277, /* questiondown */
  210, /* Agrave */
  211, /* Aacute */
  212, /* Acircumflex */
  213, /* Atilde */
  214, /* Adieresis */
  215, /* Aring */
  341, /* AE */
  216, /* Ccedilla */
  217, /* Egrave */
  220, /* Eacute */
  221, /* Ecircumflex */
  222, /* Edieresis */
  223, /* Igrave */
  224, /* Iacute */
  225, /* Icircumflex */
  226, /* Idieresis */
  227, /* Eth */
  230, /* Ntilde */
  231, /* Ograve */
  232, /* Oacute */
  233, /* Ocircumflex */
  234, /* Otilde */
  235, /* Odieresis */
  236, /* multiply */
  351, /* Oslash */
  237, /* Ugrave */
  240, /* Uacute */
  321, /* Ucircumflex */
  322, /* Udieresis */
  323, /* Yacute */
  324, /* Thorn */
  373, /* germandbls */
  325, /* agrave */
  326, /* aacute */
  327, /* acircumflex */
  330, /* atilde */
  331, /* adieresis */
  332, /* aring */
  361, /* ae */
  333, /* ccedilla */
  334, /* egrave */
  335, /* eacute */
  336, /* ecircumflex */
  337, /* edieresis */
  340, /* igrave */
  354, /* iacute */
  355, /* icircumflex */
  356, /* idieresis */
  357, /* eth */
  360, /* ntilde */
  265, /* ograve */
  276, /* oacute */
  300, /* ocircumflex */
  311, /* otilde */
  314, /* odieresis */
  342, /* divide */
  371, /* oslash */
  362, /* ugrave */
  363, /* uacute */
  364, /* ucircumflex */
  374, /* udieresis */
  375, /* yacute */
  376, /* thorn */
  377  /* ydieresis */
};


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  commonInit( FILE             *file_ref,
              PRINT_PARAM_REF  print_param );

static void
  displayUsage();

static Boolean
  fetchEntries( FILE           *file_ref,
                LST_DESC_TYPE  *entries_ref );

static Boolean
  fetchEntry( FILE       *file_ref,           
              ENTRY_REF  entry_ref,
              char       **entry_text );

static Boolean
  fetchParam( FILE             *file_ref,
              PRINT_PARAM_REF  print_param );

static void
  findTimeRange( LST_DESC_TYPE  entries_ref );

static Boolean
  formatAscii( char             *style,
               PRINT_PARAM_REF  print_param );

static Boolean
  formatDayList( char             *style,
                 PRINT_PARAM_REF  print_param );

static Boolean
  formatEntryList( char             *style,
                   PRINT_PARAM_REF  print_param );

static Boolean
  formatMonthList( char             *style,
                   PRINT_PARAM_REF  print_param );

static Boolean
  formatWeekList( char             *style,
                  PRINT_PARAM_REF  print_param );

static char
  *nlsText( char  *base_text );

static int
  noCaseStrcmp( char  *buffer1,
                char  *buffer2 );

static Boolean
  outputPrologue( char  *prologue,
                  FILE  *output_file,
                  int   level );

static void
  writeEntryText( FILE   *file_ref,
                  char   *entry_text,
                  ENTRY  *entry_ref );

static void
  writeMonthLayout( TIM_TIME_REF  month_date,
                    FILE          *file_ref );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

int
main( int argc, char *argv[] )
{

  /* Variables. */
  Boolean  ok;
  int      arg_ref;


  /* Code. */

  /* Fetch the name of the program. */
  program_name = PROGRAM_NAME;

  SysInitializeEnvironment();

  
  /* Reset global variables. */
  msg_language = NULL;
  output_file  = NULL;
  paper_size   = NULL;
  prologue_dir = NULL;
  style        = NULL;


  /* Fetch qualifiers. */
  arg_ref = 1;
  while( arg_ref < argc && argv[ arg_ref ][ 0 ] == '-' ) {

    if( noCaseStrcmp( argv[ arg_ref ], "-help" ) == 0 ||
        noCaseStrcmp( argv[ arg_ref ], "-h" ) == 0 ||
        noCaseStrcmp( argv[ arg_ref ], "-usage" ) == 0 ) {
      displayUsage();
      exit( 0 );

    } else if( noCaseStrcmp( argv[ arg_ref ], "-language" ) == 0 ||
               noCaseStrcmp( argv[ arg_ref ], "-lan" ) == 0 ) {
      arg_ref++;
      if( arg_ref < argc )
        msg_language = SysNewString( argv[ arg_ref ] );
      else {
        fprintf( stderr, "%s: -language requires parameter.\n", program_name );
        displayUsage();
        exit( 1 );
      }

    } else if( noCaseStrcmp( argv[ arg_ref ], "-paper" ) == 0 ||
               noCaseStrcmp( argv[ arg_ref ], "-pap" ) == 0 ) {
      arg_ref++;
      if( arg_ref < argc )
        paper_size = SysNewString( argv[ arg_ref ] );
      else {
        fprintf( stderr, "%s: -paper requires parameter.\n", program_name );
        displayUsage();
        exit( 1 );
      }

    } else if( noCaseStrcmp( argv[ arg_ref ], "-sheets" ) == 0 ||
               noCaseStrcmp( argv[ arg_ref ], "-she" ) == 0 ) {
      arg_ref++;
      if( arg_ref < argc )
        sheets_per_page = SysNewString( argv[ arg_ref ] );
      else {
        fprintf( stderr, "%s: -sheet requires parameter.\n", program_name );
        displayUsage();
        exit( 1 );
      }

    } else if( noCaseStrcmp( argv[ arg_ref ], "-output" ) == 0 ||
               noCaseStrcmp( argv[ arg_ref ], "-out" ) == 0 ) {
      arg_ref++;
      if( arg_ref < argc )
        output_file = SysNewString( argv[ arg_ref ] );
      else {
        fprintf( stderr, "%s: -output requires parameter.\n", program_name );
        displayUsage();
        exit( 1 );
      }

    } else if( noCaseStrcmp( argv[ arg_ref ], "-prologue" ) == 0 ||
               noCaseStrcmp( argv[ arg_ref ], "-pro" ) == 0 ) {
      arg_ref++;
      if( arg_ref < argc )
        prologue_dir = SysNewString( argv[ arg_ref ] );
      else {
        fprintf( stderr, "%s: -prologue requires parameter.\n", program_name );
        displayUsage();
        exit( 1 );
      }

    } else if( noCaseStrcmp( argv[ arg_ref ], "-style" ) == 0 ||
               noCaseStrcmp( argv[ arg_ref ], "-sty" ) == 0 ) {
      arg_ref++;
      if( arg_ref < argc )
        style = SysNewString( argv[ arg_ref ] );
      else {
        fprintf( stderr, "%s: -style requires parameter.\n", program_name );
        displayUsage();
        exit( 1 );
      }

    } else if( noCaseStrcmp( argv[ arg_ref ], "-version" ) == 0 ) {
      printf( "%s: Version: %s\n", program_name, VERSION_ID );
      exit( 0 );

    } /* if */

    arg_ref++;

  } /* while */


  /* Any language? */
  if( msg_language == NULL )
    msg_language = SysNewString( "En_US" );


  /* Initialize necessary text domains. */
  msgInitialize();
  msgInitCatalogue( text_domain, NULL, msg_language, msgXdiaryTexts );
  msgTextDomain( text_domain );


  /* Any prologue directory? */
  if( prologue_dir == NULL )
    prologue_dir = SysNewString( "." );

  /* Correct style? */
  if( style == NULL )
    style = SysNewString( "WeekList" );

  if( strcmp( style, "EntryList" )    != 0 &&
      strcmp( style, "WeekList" )     != 0 &&
      strcmp( style, "CompWeekList" ) != 0 &&
      strcmp( style, "DayList" )      != 0 &&
      strcmp( style, "CompDayList" )  != 0 &&
      strcmp( style, "MonthList" )    != 0 &&
      strcmp( style, "Ascii" )        != 0 ) {
    fprintf( stderr, "%s: Unknown style (%s).\n", program_name, style );

    displayUsage();
    exit( 1 );
  }

  /* Correct paper size? */
  if( paper_size == NULL )
    paper_size = SysNewString( "A4" );

  if( strcmp( paper_size, "A3" )     != 0 &&
      strcmp( paper_size, "A4" )     != 0 &&
      strcmp( paper_size, "A5" )     != 0 &&
      strcmp( paper_size, "Letter" ) != 0 &&
      strcmp( paper_size, "Legal" )  != 0 ) {
    fprintf( stderr, "%s: Unknown paper size (%s).\n",
             program_name, paper_size );

    displayUsage();
    exit( 1 );
  }


  /* Correct sheets_per_page? */
  if( sheets_per_page == NULL )
    sheets_per_page = SysNewString( "1" );

  if( strcmp( sheets_per_page, "1" ) != 0 &&
      strcmp( sheets_per_page, "2" ) != 0 ) {
    fprintf( stderr, "%s: Unknown sheets/page (%s).\n",
             program_name, sheets_per_page );

    displayUsage();
    exit( 1 );
  }


  /* Input file. */
  if( (argc - arg_ref) > 0 ) {
    input_file_ref = fopen( argv[ arg_ref ], "r" );
  } else {
    input_file_ref = stdin;
  }

  if( input_file_ref == NULL ) {
    fprintf( stderr, "%s: Cannot open input file.\n", program_name );

    displayUsage();
    exit( 1 );
  }

  /* Output file. */
  if( output_file == NULL )
    output_file_ref = stdout;
  else
    output_file_ref = fopen( output_file, "w" );

  if( output_file_ref == NULL ) {
    fprintf( stderr, "%s: Cannot open output file.\n", program_name );

    displayUsage();
    exit( 1 );
  }


  /* Fetch print parameters. */
  ok = fetchParam( input_file_ref, &print_param );
  if( ! ok )
    exit( 1 );


  /* Depending on style, format the input file. */
  ok = False;

  if( strcmp( style, "EntryList" ) == 0 )
    ok = formatEntryList( style, &print_param );

  if( strcmp( style, "DayList" ) == 0 ||
      strcmp( style, "CompDayList" ) == 0 )
    ok = formatDayList( style, &print_param );

  if( strcmp( style, "WeekList" ) == 0 ||
      strcmp( style, "CompWeekList" ) == 0 )
    ok = formatWeekList( style, &print_param );

  if( strcmp( style, "MonthList" ) == 0 )
    ok = formatMonthList( style, &print_param );

  if( strcmp( style, "Ascii" ) == 0 )
    ok = formatAscii( style, &print_param );

  if( ! ok )
    exit( 1 );


  exit( 0 );

} /* main */


/*----------------------------------------------------------------------*/

static void
  displayUsage()
{

  printf( 
    "\n"
    "%s (%s): Formats an XDiary printout.\n"
    "\n"
    "Usage:\n"
    "  %s [flags] [file]\n"
    "\n"
    "where:\n"
    "  file  : File created by the XDiary print routine. If no file is given\n"
    "          standard input is used.\n"
    "\n"
    "Flags:\n"
    "  -h[elp]         : Display this help.\n"
    "  -pap[er] paper  : Format for one of the paper sizes:\n"
    "                      A3, A4, A5, Letter, Legal\n"
    "  -out[put] file  : Save output in file. Default is to send the output\n"
    "                    to standard output.\n"
    "  -pro[logue] dir : Search prologue files in this directory.\n"
    "  -she[ets] no    : Number of sheets (1,2) per page.\n"
    "  -sty[le] style  : Format for one of the styles:\n"
    "                      EntryList, WeekList, CompWeekList, DayList,\n"
    "                      CompDayList, MonthList, Ascii\n"
    "  -usage          : Display this help.\n"
    "  -version        : Display the current version.\n"
    "\n",
    program_name, VERSION_ID, program_name );

  return;

} /* displayUsage */


/*----------------------------------------------------------------------*/

static Boolean
  fetchEntries( FILE           *file_ref,
                LST_DESC_TYPE  *entries_ref )
{

  /* Variables. */
  Boolean  ok;
  char     *entry_text;
  ENTRY    entry_record;


  /* Code. */

  *entries_ref = LstLinkNew( sizeof( ENTRY ), NULL );

  /* Fetch all entries. */
  do {

    ok = fetchEntry( file_ref, &entry_record, &entry_text );
    if( ! ok )
      break;

    (void) LstLinkInsertLast( *entries_ref, &entry_record );

  } while( True );


  /* Time range for entries. */
  findTimeRange( *entries_ref );


  return( True );

} /* fetchEntries */


/*----------------------------------------------------------------------*/

static Boolean
  fetchEntry( FILE       *file_ref,           
              ENTRY_REF  entry_ref,
              char       **entry_text )
{

  /* Variables. */
  int              integer1;
  int              items;
  int              segment_size = 500;
  int              text_buffer_size = 0;
  UINT32           flags;
  char             buffer[ 1024 ];
  char             buffer1[ 256 ];
  char             buffer2[ 256 ];
  char             *char_ref;
  TIM_STATUS_TYPE  time_status;
  TIM_TIME_REF     time_data;


  /* Code. */

  *entry_text = NULL;

  /* Skip all lines until an entry start tag. */
  do {
    char_ref = fgets( buffer, sizeof( buffer ), file_ref );
    if( char_ref == NULL )
      return( False );

    if( strncmp( buffer, ENTRY_START, strlen( ENTRY_START ) ) == 0 )
      break;
  } while( 1 );


  /* Allocate space for the appointment text. */
  *entry_text      = (char *) SysMalloc( segment_size );
  **entry_text     = '\0';
  text_buffer_size = segment_size;

  /* Record initialization. */
  entry_ref -> time_stamp     = 0;
  entry_ref -> date_stamp     = 0;
  entry_ref -> duration       = 0;

  entry_ref -> flags          = 0;
  entry_ref -> tag[ 0 ]       = '\0';
  entry_ref -> text           = NULL;
  entry_ref -> entry_type     = XTM_DB_DAY_ENTRY;
  entry_ref -> entry_category = XTM_DB_ENTRY_LIST;

  /* Start reading the entry data, stop at the end marker ($$). */
  do {
    char_ref = fgets( buffer, sizeof( buffer ), file_ref );
    if( char_ref == NULL )
      return( False );


    /* End of record? */
    if( strncmp( buffer, "$$", 2 ) == 0 ) {
      entry_ref -> text = *entry_text;

      return( True );
    }


    /* Entry date, time and duration. */
    if( strncmp( buffer, "Ti", 2 ) == 0 ) {

      items = sscanf( &buffer[ 2 ], "%s %s %d", buffer1, buffer2, &integer1 );
      if( items != 3 ) {
        fprintf( stderr, "%s: Don't know line: %s.\n", program_name, buffer );
        return( False );
      }

      /* Date. */
      time_status = TimMakeDateFromIsoString( &time_data, buffer1 );
      if( time_status != TIM_OK ) {
        fprintf( stderr, "%s: Date error: %s.\n", program_name, buffer1 );
        return( False );
      }

      entry_ref -> date_stamp = time_data;

      /* Time. */
      time_status = TimMakeTimeFromIsoString( &time_data, buffer2 );
      if( time_status != TIM_OK ) {
        fprintf( stderr, "%s: Time error: %s.\n", program_name, buffer1 );
        return( False );
      }

      entry_ref -> time_stamp = time_data;

      /* Duration. */
      if( integer1 < 0 || integer1 > 24 * 60 ) {
        fprintf( stderr, "%s: Duration error: %d.\n", program_name, integer1 );
        return( False );
      }

      entry_ref -> duration = integer1;

    } /* if */


    /* Flags. */
    if( strncmp( buffer, "Fl", 2 ) == 0 ) {

      items = sscanf( &buffer[ 2 ], "%s", buffer1 );
      if( items != 1 ) {
        fprintf( stderr, "%s: Don't know line: %s.\n", program_name, buffer );
        return( False );
      }

      flags = 0;

      if( strchr( buffer1, 'A' ) != NULL )
        flagSet( flags, XTM_DB_FLAG_ALARM );

      if( strchr( buffer1, 'D' ) != NULL )
        flagSet( flags, XTM_DB_FLAG_NOTE_DONE );

      if( strchr( buffer1, 'I' ) != NULL )
        flagSet( flags, XTM_DB_FLAG_IMPORTANT );
 
      if( strchr( buffer1, 'P' ) != NULL )
        flagSet( flags, XTM_DB_FLAG_PRIVATE );

      flagSet( entry_ref -> flags, flags );

      if( strchr( buffer1, 'S' ) != NULL )
        entry_ref -> entry_category = XTM_DB_REP_ENTRY_LIST;

      if( strchr( buffer1, 'T' ) != NULL )
        entry_ref -> entry_category = XTM_DB_STICKY_LIST;

      if( strchr( buffer1, 'N' ) != NULL )
        entry_ref -> entry_type = XTM_DB_DAY_NOTE;

    } /* if */


    /* Entry tag. */
    if( strncmp( buffer, "Tg", 2 ) == 0 ) {

      items = sscanf( &buffer[ 2 ], "%s", buffer1 );
      if( items == 1 ) {
        strncpy( entry_ref -> tag, buffer1, XTM_DB_TAG_LEN );
        entry_ref -> tag[ XTM_DB_TAG_LEN ] = '\0';
      }

    } /* if */


    /* Text line? */
    if( strncmp( buffer, "<<", 2 ) == 0 ) {

      /* Do we need to expand the buffer? */
      if( strlen( *entry_text ) + strlen( buffer ) + 2 > text_buffer_size ) {

        int  extend_size;

        extend_size = max( strlen( buffer ), segment_size ) + 10;
        text_buffer_size = text_buffer_size + extend_size;

        *entry_text = SysRealloc( *entry_text, text_buffer_size );

      } /* if */

      strcat( *entry_text, &buffer[ 2 ] );

    } /* if */

  } while( True );


} /* fetchEntry */


/*----------------------------------------------------------------------*/

static Boolean
  fetchParam( FILE             *file_ref,
              PRINT_PARAM_REF  print_param )
{

  /* Variables. */
  Boolean          valid_file = False;
  char             buffer[ 1024 ];
  char             buffer1[ 1024 ];
  char             buffer2[ 1024 ];
  char             key[ 1024 ];
  char             *char_ref;
  TIM_STATUS_TYPE  tstatus1;
  TIM_STATUS_TYPE  tstatus2;
  TIM_TIME_REF     time_value;


  /* Code. */

  /* Default values. */
  print_param -> only_workdays = False;
  print_param -> print_3d_look = True;
  print_param -> start_date    = (TIM_TIME_REF) 0;
  print_param -> end_date      = (TIM_TIME_REF) 0;
  print_param -> start_time    = TimMakeTime( 1970, 1, 1, 8,  0, 0 );
  print_param -> end_time      = TimMakeTime( 1970, 1, 1, 18, 0, 0 );

  strcpy( print_param -> calendar_name, "Default" );
  strcpy( print_param -> date_format, "YMD-1111" );
  strcpy( print_param -> time_format, "24    :0" );


  /* Read the parameters. */
  do {

    /* Fetch a line. */
    char_ref = fgets( buffer, sizeof( buffer ), file_ref );
    if( char_ref == NULL )
      return( True );

    while( isspace( *char_ref ) )
      char_ref++;

    /* The key (if there is one). */
    key[ 0 ] ='\0';
    sscanf( char_ref, "%[^:]", key );

    /* Read parameters. */
    if( *char_ref == '\0' ) {
      break;

    } else if( strcmp( key, PRINTER_FILE_ID ) == 0 ) {
      valid_file = True;

    } else if( strcmp( key, KEY_CALENDAR ) == 0 ) {
      sscanf( buffer, "%[^:]: %s", key, print_param -> calendar_name );

    } else if( strcmp( key, KEY_DATE_RANGE ) == 0 ) {
      sscanf( buffer, "%[^:]: %s %s", key, buffer1, buffer2 );

      tstatus1 = TimMakeDateFromIsoString( &print_param -> start_date,
                                           buffer1 );
      tstatus2 = TimMakeDateFromIsoString( &print_param -> end_date,
                                           buffer2 );

      if( tstatus1 != TIM_OK || tstatus2 != TIM_OK ) {
        fprintf( stderr, "%s: Invalid date range.\n", program_name );
        return( False );
      }

    } else if( strcmp( key, KEY_DEFAULT_TIME_MIN ) == 0 ) {
      sscanf( buffer, "%[^:]: %s", key, buffer1 );

      tstatus1 = TimMakeTimeFromIsoString( &time_value, buffer1 );

      if( tstatus1 != TIM_OK ) {
        fprintf( stderr, "%s: Invalid entry start time.\n", program_name );
        return( False );
      }

      if( time_value < print_param -> start_time )
        print_param -> start_time = time_value;

    } else if( strcmp( key, KEY_DEFAULT_TIME_MAX ) == 0 ) {
      sscanf( buffer, "%[^:]: %s", key, buffer1 );

      tstatus1 = TimMakeTimeFromIsoString( &time_value, buffer1 );

      if( tstatus1 != TIM_OK ) {
        fprintf( stderr, "%s: Invalid entry end time.\n", program_name );
        return( False );
      }

      if( time_value > print_param -> end_time )
        print_param -> end_time = time_value;

    } else if( strcmp( key, KEY_DATE_FORMAT ) == 0 ) {
      sscanf( buffer, "%[^:]: %8c", key, print_param -> date_format );

    } else if( strcmp( key, KEY_TIME_FORMAT ) == 0 ) {
      sscanf( buffer, "%[^:]: %8c", key, print_param -> time_format );

    } else if( strcmp( key, KEY_ONLY_WORKDAYS ) == 0 ) {
      sscanf( buffer, "%[^:]: %s", key, buffer1 );

      if( noCaseStrcmp( buffer1, "True" ) == 0 )
        print_param -> only_workdays = True;

    } else if( strcmp( key, KEY_3D_LOOK ) == 0 ) {
      sscanf( buffer, "%[^:]: %s", key, buffer1 );

      if( noCaseStrcmp( buffer1, "False" ) == 0 )
        print_param -> print_3d_look = False;

    } /* if */

  } while( True );

  if( ! valid_file ) {
    fprintf( stderr, "%s: The input file is not a printer file!\n",
             program_name );

    return( False );
  }

  /* Time format. */
  tstatus1 = TimInitializeFormat( print_param -> date_format,
                                  print_param -> time_format );
  if( tstatus1 != TIM_OK ) {
    fprintf( stderr, "%s: The date and/or time format is wrong.\n",
             program_name );

    return( False );
  }


  return( True );

} /* fetchParam */


/*----------------------------------------------------------------------*/

static void
  findTimeRange( LST_DESC_TYPE  entries_ref )
{

  /* Variables. */
  int           first_hour;
  int           last_hour;
  ENTRY         entry_record;
  LST_STATUS    lst_status;
  TIM_TIME_REF  max_time;
  TIM_TIME_REF  min_time;
  TIM_TIME_REF  tmp_time;


  /* Code. */

  min_time = print_param.start_time;
  max_time = print_param.end_time;

  lst_status = LstLinkCurrentFirst( entries_ref );

  while( lst_status == LST_OK ) {

    (void) LstLinkGetCurrent( entries_ref, &entry_record );

    if( entry_record.entry_type == XTM_DB_DAY_ENTRY ) {
      tmp_time = entry_record.time_stamp;
      if( tmp_time < min_time )
        min_time = tmp_time;

      tmp_time = entry_record.time_stamp;
      TimAddMinutes( &tmp_time, entry_record.duration );
      if( tmp_time > max_time )
        max_time = tmp_time;
    }

    lst_status = LstLinkCurrentNext( entries_ref );

  } /* while */


  /* Adjust up or down to the closest hour. */
  TimAddMinutes( &max_time, -1 );

  first_hour = TimHour( min_time );
  last_hour  = TimHour( max_time ) + 1;

  if( first_hour < 0 )
    first_hour = 0;

  if( last_hour > 23 )
    last_hour = 23;

  min_time = TimMakeTime( 1970, 1, 1, first_hour, 0, 0 );
  max_time = TimMakeTime( 1970, 1, 1, last_hour, 0, 0 );

  if( min_time < print_param.start_time )
    print_param.start_time = min_time;

  if( max_time > print_param.end_time )
    print_param.end_time = max_time;


  return;

} /* findTimeRange */


/*----------------------------------------------------------------------*/

static Boolean
  formatAscii( char             *style,
               PRINT_PARAM_REF  print_param )
{

  /* Variables. */
  Boolean        day_written = False;
  int            char_read;
  int            items;
  char           buffer[ 1024 ];
  char           *char_ref;
  ENTRY          entry_record;
  LST_DESC_TYPE  entries_ref;
  LST_STATUS     lst_status;
  TIM_TIME_REF   curr_date;
  TIM_TIME_REF   tmp_time;


  /* Code. */

  /* Fetch the entries. */
  (void) fetchEntries( input_file_ref, &entries_ref );


  /* Process all the entries. */
  curr_date  = (TIM_TIME_REF) 0;
  lst_status = LstLinkCurrentFirst( entries_ref );

  while( lst_status == LST_OK ) {

    (void) LstLinkGetCurrent( entries_ref, &entry_record );

    if( curr_date != entry_record.date_stamp ) {

      curr_date = entry_record.date_stamp;

      fprintf( output_file_ref, "%s\n",
               "===========================================================" );

      TimFormatStrTime( curr_date, "%A, %b ", buffer, sizeof( buffer ) );
      fprintf( output_file_ref, "%s", buffer );

      fprintf( output_file_ref, "%d %d\n\n",
               TimIndexOfDay( curr_date ), TimIndexOfYear( curr_date ) );


      day_written = True;

    } /* if */

    /* Write the note? */
    if( entry_record.entry_type == XTM_DB_DAY_NOTE ) {
      fprintf( output_file_ref, "%s\n",
               nlsText( msgGetText( MXDI_FOPR_NOTE ) ) );

    /* Write the entry? */
    } else {
      tmp_time = entry_record.time_stamp;
      TimAddMinutes( &tmp_time, entry_record.duration );

      TimFormatTime( entry_record.time_stamp, buffer, sizeof( buffer ) );
      fprintf( output_file_ref, "%s", buffer );

      if( entry_record.duration != 0 ) {
        TimFormatTime( tmp_time, buffer, sizeof( buffer ) );
        fprintf( output_file_ref, " - %s", buffer );
      }

      fprintf( output_file_ref, "\n" );
    }

    /* Entry lines. */
    char_ref = entry_record.text;

    while( *char_ref != '\0' ) {
      char_read = strlen( char_ref );

      items = sscanf( char_ref, "%[^\n]\n%n", buffer, &char_read );
      if( items != 1 )
        break;

      fprintf( output_file_ref, "    %s\n", buffer );

      char_ref = char_ref + char_read;
    }

    /* The end of the entry. */
    fprintf( output_file_ref, "  ---\n\n" );

    /* Next entry. */
    lst_status = LstLinkCurrentNext( entries_ref );

  } /* while */

  if( day_written )
    fprintf( output_file_ref, "%s\n",
             "===========================================================" );


  return( True );

} /* formatAscii */


/*----------------------------------------------------------------------*/

static Boolean
  formatDayList( char             *style,
                 PRINT_PARAM_REF  print_param )
{

  /* Variables. */
  Boolean        compact = False;
  Boolean        day_written = False;
  Boolean        ok;
  int            curr_month;
  char           buffer[ 1024 ];
  ENTRY          entry_record;
  LST_DESC_TYPE  entries_ref;
  LST_STATUS     lst_status;
  TIM_TIME_REF   curr_date;
  TIM_TIME_REF   month_date;


  /* Code. */

  /* Fetch the entries. */
  (void) fetchEntries( input_file_ref, &entries_ref );

  fprintf( output_file_ref, "/PageFormat (%s) def\n\n", paper_size );
  fprintf( output_file_ref, "/SheetsPerPage %s def\n", sheets_per_page );

  /* Specific initialization. */
  fprintf( output_file_ref, "/TickScaleStart %d.%d def\n", 
           TimHour( print_param -> start_time ),
           TimMinute( print_param -> start_time ) );

  fprintf( output_file_ref, "/TickScaleEnd %d.%d def\n", 
           TimHour( print_param -> end_time ),
           TimMinute( print_param -> end_time ) );

  ok = outputPrologue( "XDpsDay.pro", output_file_ref, 1 );
  if( ! ok )
    return( False );


  if( strcmp( style, "CompDayList" ) == 0 )
    compact = True;

  /* Specific initialization. */
  fprintf( output_file_ref, "\n" );

  fprintf( output_file_ref, "/DateLabel (%s) def\n",
           nlsText( msgGetText( MXDI_FOPR_DATE ) ) );
  fprintf( output_file_ref, "/DbLabel (%s) def\n",
           nlsText( msgGetText( MXDI_FOPR_CALENDAR ) ) );
  fprintf( output_file_ref, "/LabelApps (%s) def\n", 
           nlsText( msgGetText( MXDI_FOPR_APPS ) ) );
  fprintf( output_file_ref, "/LabelNotes (%s) def\n",
           nlsText( msgGetText( MXDI_FOPR_NOTES ) ) );

  /* Common nitialization. */
  commonInit( output_file_ref, print_param );

  /* Specific initialization. */
  fprintf( output_file_ref, "\n" );
  fprintf( output_file_ref, "%% Start a new page\n" );
  fprintf( output_file_ref, "bop\n\n" );


  /* Process all the entries. */
  lst_status = LstLinkCurrentFirst( entries_ref );
  if( lst_status == LST_OK )
    (void) LstLinkGetCurrent( entries_ref, &entry_record );

  curr_date  = print_param -> start_date;
  curr_month = 0;

  while( curr_date <= print_param -> end_date ) {

    /* If new month, write it. */
    if( curr_month != TimIndexOfMonth( curr_date ) ) {

      month_date = curr_date;
      TimAddMonths( &month_date, -1 );
      fprintf( output_file_ref, "/CalPrevMonth " );
      writeMonthLayout( month_date, output_file_ref );

      TimAddMonths( &month_date, 1 );
      fprintf( output_file_ref, "/CalThisMonth " );
      writeMonthLayout( month_date, output_file_ref );

      TimAddMonths( &month_date, 1 );
      fprintf( output_file_ref, "/CalNextMonth " );
      writeMonthLayout( month_date, output_file_ref );

      fprintf( output_file_ref, "\n" );

      curr_month = TimIndexOfMonth( curr_date );

    } /* if */

    if( print_param -> only_workdays && 
        TimIndexOfDayInIsoWeek( curr_date ) > 5 ) {
      TimAddDays( &curr_date, 1 );
      continue;
    }

    if( ! compact ||
        (lst_status == LST_OK && entry_record.date_stamp == curr_date ) ) {

      if( day_written ) {
        fprintf( output_file_ref, "%% End of the day.\n" );
        fprintf( output_file_ref, "endDay\n\n" );

        fprintf( output_file_ref, "%% Start a new page.\n" );
        fprintf( output_file_ref, "newPage\n\n" );
      }

      fprintf( output_file_ref, "%% Start a new day.\n\n" );

      TimFormatStrTime( curr_date, "%A", buffer, sizeof( buffer ) );
      fprintf( output_file_ref, "(%s) ", nlsText( buffer ) );

      TimFormatStrTime( curr_date, "%B", buffer, sizeof( buffer ) );
      fprintf( output_file_ref, "(%s %d %d) startDay\n",
               nlsText( buffer ), TimIndexOfDay( curr_date ),
               TimIndexOfYear( curr_date ) );

      day_written = True;

    } /* if */

    /* Do we have a valid entry? */
    while( lst_status == LST_OK && entry_record.date_stamp == curr_date ) {

      /* Write the note? */
      if( entry_record.entry_type == XTM_DB_DAY_NOTE ) {
        fprintf( output_file_ref, "%% Write a note.\n" );
        fprintf( output_file_ref, "startNote\n" );

        if( flagIsSet( entry_record.flags, XTM_DB_FLAG_IMPORTANT ) )
          fprintf( output_file_ref, "importantNote\n" );

      /* Write the entry? */
      } else {
        fprintf( output_file_ref, "%% Write an entry.\n" );
        fprintf( output_file_ref, "%d.%d hour %d min startEntry\n",
                 TimHour( entry_record.time_stamp ),
                 TimMinute( entry_record.time_stamp ),
                 entry_record.duration );

        if( flagIsSet( entry_record.flags, XTM_DB_FLAG_IMPORTANT ) )
          fprintf( output_file_ref, "importantEntry\n" );
      }

      /* Entry lines. */
      writeEntryText( output_file_ref, entry_record.text, &entry_record );

      /* The end of the entry. */
      if( entry_record.entry_type == XTM_DB_DAY_NOTE )
        fprintf( output_file_ref, "endNote\n\n" );
      else
        fprintf( output_file_ref, "endEntry\n\n" );

      /* Next entry. */
      lst_status = LstLinkCurrentNext( entries_ref );

      if( lst_status == LST_OK )
        (void) LstLinkGetCurrent( entries_ref, &entry_record );

    } /* while */


    TimAddDays( &curr_date, 1 );

  } /* while */

  if( day_written ) {
    fprintf( output_file_ref, "%% End of the day.\n" );
    fprintf( output_file_ref, "endDay\n\n" );
  }

  /* OK, we're done. */
  fprintf( output_file_ref, "eop\n" );
  fprintf( output_file_ref, "lastPage\n\n" );
  fprintf( output_file_ref, "%%%%Trailer\n\n" );


  return( True );

} /* formatDayList */


/*----------------------------------------------------------------------*/

static Boolean
  formatEntryList( char             *style,
                   PRINT_PARAM_REF  print_param )
{

  /* Variables. */
  Boolean        day_written = False;
  Boolean        ok;
  char           buffer[ 1024 ];
  ENTRY          entry_record;
  LST_DESC_TYPE  entries_ref;
  LST_STATUS     lst_status;
  TIM_TIME_REF   curr_date;
  TIM_TIME_REF   tmp_time;


  /* Code. */

  /* Fetch the entries. */
  (void) fetchEntries( input_file_ref, &entries_ref );

  fprintf( output_file_ref, "/PageFormat (%s) def\n\n", paper_size );
  fprintf( output_file_ref, "/SheetsPerPage %s def\n", sheets_per_page );

  ok = outputPrologue( "XDpsList.pro", output_file_ref, 1 );
  if( ! ok )
    return( False );


  /* Specific initialization. */
  fprintf( output_file_ref, "\n" );

  fprintf( output_file_ref, "/DateLabel (%s) def\n",
           nlsText( msgGetText( MXDI_FOPR_DATE ) ) );
  fprintf( output_file_ref, "/DbLabel (%s) def\n",
           nlsText( msgGetText( MXDI_FOPR_CALENDAR ) ) );


  /* Common nitialization. */
  commonInit( output_file_ref, print_param );


  /* Specific initialization. */
  fprintf( output_file_ref, "\n" );
  fprintf( output_file_ref, "%% Start a new page\n" );
  fprintf( output_file_ref, "bop\n\n" );


  /* Process all the entries. */
  curr_date  = (TIM_TIME_REF) 0;
  lst_status = LstLinkCurrentFirst( entries_ref );

  while( lst_status == LST_OK ) {

    (void) LstLinkGetCurrent( entries_ref, &entry_record );

    if( entry_record.date_stamp != curr_date ) {

      curr_date = entry_record.date_stamp;

      if( day_written ) {
        fprintf( output_file_ref, "%% End of the day.\n" );
        fprintf( output_file_ref, "daySeparator\n\n" );
      }

      fprintf( output_file_ref, "%% Start a new day.\n" );

      TimFormatStrTime( curr_date, "%b", buffer, sizeof( buffer ) );
      fprintf( output_file_ref, "(%s ", nlsText( buffer ) );

      fprintf( output_file_ref, "%d ", TimIndexOfDay( curr_date ) );

      TimFormatStrTime( curr_date, "%A", buffer, sizeof( buffer ) );
      fprintf( output_file_ref, "%s) newDay\n\n", nlsText( buffer ) );

      day_written = True;

    } /* if */

    /* Write the note? */
    if( entry_record.entry_type == XTM_DB_DAY_NOTE ) {
      fprintf( output_file_ref, "%% Write a note.\n" );
      fprintf( output_file_ref, "(%s) entryNoteHeader\n",
               msgGetText( MXDI_FOPR_NOTE ) );

      fprintf( output_file_ref, "beginEntryText\n" );

      if( flagIsSet( entry_record.flags, XTM_DB_FLAG_IMPORTANT ) )
        fprintf( output_file_ref, "importantNote\n" );

    /* Write the entry? */
    } else {
      tmp_time = entry_record.time_stamp;
      TimAddMinutes( &tmp_time, entry_record.duration );

      fprintf( output_file_ref, "%% Write an entry.\n" );
      fprintf( output_file_ref, "%d.%d hour %d.%d hour entryAppHeader\n",
               TimHour( entry_record.time_stamp ),
               TimMinute( entry_record.time_stamp ),
               TimHour( tmp_time ), TimMinute( tmp_time ) );

      fprintf( output_file_ref, "beginEntryText\n" );

      if( flagIsSet( entry_record.flags, XTM_DB_FLAG_IMPORTANT ) )
        fprintf( output_file_ref, "importantEntry\n" );
    }

    /* Entry lines. */
    writeEntryText( output_file_ref, entry_record.text, &entry_record );

    /* The end of the entry. */
    if( entry_record.entry_type == XTM_DB_DAY_NOTE )
      fprintf( output_file_ref, "endNote\n\n" );
    else
      fprintf( output_file_ref, "endEntry\n\n" );

    /* Next entry. */
    lst_status = LstLinkCurrentNext( entries_ref );

  } /* while */

  if( day_written ) {
    fprintf( output_file_ref, "%% End of the day.\n" );
    fprintf( output_file_ref, "daySeparator\n\n" );
  }

  /* OK, we're done. */
  fprintf( output_file_ref, "eop\n" );
  fprintf( output_file_ref, "lastPage\n\n" );
  fprintf( output_file_ref, "%%%%Trailer\n\n" );


  return( True );

} /* formatEntryList */


/*----------------------------------------------------------------------*/

static Boolean
  formatMonthList( char             *style,
                   PRINT_PARAM_REF  print_param )
{

  /* Variables. */
  Boolean        day_written = False;
  Boolean        ok;
  int            curr_month;
  int            days_per_page;
  int            index;
  char           buffer[ 1024 ];
  ENTRY          entry_record;
  LST_DESC_TYPE  entries_ref;
  LST_STATUS     lst_status;
  TIM_TIME_REF   curr_date;
  TIM_TIME_REF   month_date;


  /* Code. */

  /* Fetch the entries. */
  (void) fetchEntries( input_file_ref, &entries_ref );

  fprintf( output_file_ref, "/PageFormat (%s) def\n\n", paper_size );
  fprintf( output_file_ref, "/SheetsPerPage %s def\n", sheets_per_page );

  ok = outputPrologue( "XDpsMonth.pro", output_file_ref, 1 );
  if( ! ok )
    return( False );


  if( print_param -> only_workdays )
    days_per_page = 5;
  else
    days_per_page = 7;


  /* Specific initialization. */
  fprintf( output_file_ref, "\n" );

  fprintf( output_file_ref, "/DateLabel (%s) def\n",
           nlsText( msgGetText( MXDI_FOPR_DATE ) ) );
  fprintf( output_file_ref, "/DbLabel (%s) def\n",
           nlsText( msgGetText( MXDI_FOPR_CALENDAR ) ) );


  /* Common nitialization. */
  commonInit( output_file_ref, print_param );


  /* Specific initialization. */
  fprintf( output_file_ref, "\n" );
  fprintf( output_file_ref, "/DaysPerPage %d def\n", days_per_page );
  fprintf( output_file_ref, "\n" );

  if( TimIndexOfFirstDayInWeek() == 0 && print_param -> only_workdays )
    fprintf( output_file_ref, "/startDayInWeek 2 def \n" );
  else
    fprintf( output_file_ref, "/startDayInWeek 1 def \n" );
  fprintf( output_file_ref, "%s\n", 
           "/stopDayInWeek startDayInWeek DaysPerPage add 1 sub def" );
  fprintf( output_file_ref, "\n" );


  /* Weekdays. */
  curr_date = TimMakeTimeNow();

  index = TimIndexOfFirstDayInWeek();
  if( index == 0 )
    index = 7;

  while( TimIndexOfDayInIsoWeek( curr_date ) != index )
    TimAddDays( &curr_date, -1 );

  fprintf( output_file_ref, "/Weekdays [ " );

  for( index = 0; index < 7; index++ ) {
    TimFormatStrTime( curr_date, "%A", buffer, sizeof( buffer ) );
    fprintf( output_file_ref, "(%s) ", nlsText( buffer ) );
    TimAddDays( &curr_date, 1 );
  }

  fprintf( output_file_ref, "] def\n" );
  fprintf( output_file_ref, "\n" );


  /* Process all the entries. */
  lst_status = LstLinkCurrentFirst( entries_ref );
  if( lst_status == LST_OK )
    (void) LstLinkGetCurrent( entries_ref, &entry_record );

  curr_date  = print_param -> start_date;
  curr_month = 0;

  while( curr_date <= print_param -> end_date ) {

    /* If new month, write it. */
    if( curr_month != TimIndexOfMonth( curr_date ) ) {

      month_date = curr_date;
      TimAddMonths( &month_date, -1 );
      fprintf( output_file_ref, "/CalPrevMonth " );
      writeMonthLayout( month_date, output_file_ref );

      TimAddMonths( &month_date, 1 );
      fprintf( output_file_ref, "/CalThisMonth " );
      writeMonthLayout( month_date, output_file_ref );

      TimAddMonths( &month_date, 1 );
      fprintf( output_file_ref, "/CalNextMonth " );
      writeMonthLayout( month_date, output_file_ref );

      fprintf( output_file_ref, "\n" );

      fprintf( output_file_ref, "%% Start new page.\n" );
      if( day_written )
        fprintf( output_file_ref, "newPage\n" );
      else
        fprintf( output_file_ref, "bop\n" );
      fprintf( output_file_ref, "\n" );

      curr_month = TimIndexOfMonth( curr_date );

    } /* if */

    day_written = True;

    if( print_param -> only_workdays && 
        TimIndexOfDayInIsoWeek( curr_date ) > 5 ) {
      TimAddDays( &curr_date, 1 );
      continue;
    }

    /* Start a new day. */
    fprintf( output_file_ref, "%% Start a new day.\n" );
    fprintf( output_file_ref, "%d startDay\n", TimIndexOfDay( curr_date ) );
    fprintf( output_file_ref, "\n" );


    /* Write entries within the day. */
    while( lst_status == LST_OK && entry_record.date_stamp == curr_date ) {

      /* Write the note? */
      if( entry_record.entry_type == XTM_DB_DAY_NOTE ) {
        fprintf( output_file_ref, "%% Write a note.\n" );
        fprintf( output_file_ref, "startNote\n" );

        if( flagIsSet( entry_record.flags, XTM_DB_FLAG_IMPORTANT ) )
          fprintf( output_file_ref, "importantNote\n" );

      /* Write the entry? */
      } else {
        fprintf( output_file_ref, "%% Write an entry.\n" );
        fprintf( output_file_ref, "%d.%d hour %d min startEntry\n",
                 TimHour( entry_record.time_stamp ),
                 TimMinute( entry_record.time_stamp ),
                 entry_record.duration );

        if( flagIsSet( entry_record.flags, XTM_DB_FLAG_IMPORTANT ) )
          fprintf( output_file_ref, "importantEntry\n" );
      }

      /* Entry lines. */
      writeEntryText( output_file_ref, entry_record.text, &entry_record );

      /* The end of the entry. */
      if( entry_record.entry_type == XTM_DB_DAY_NOTE )
        fprintf( output_file_ref, "endNote\n\n" );
      else
        fprintf( output_file_ref, "endEntry\n\n" );

      /* Next entry. */
      lst_status = LstLinkCurrentNext( entries_ref );

      if( lst_status == LST_OK )
        (void) LstLinkGetCurrent( entries_ref, &entry_record );

    } /* while */


    /* Emd of the day. */
    fprintf( output_file_ref, "%% End of the day.\n" );
    fprintf( output_file_ref, "endDay\n" );
    fprintf( output_file_ref, "\n" );

    TimAddDays( &curr_date, 1 );

  } /* while */


  /* OK, we're done. */
  fprintf( output_file_ref, "eop\n" );
  fprintf( output_file_ref, "lastPage\n\n" );
  fprintf( output_file_ref, "%%%%Trailer\n\n" );


  return( True );

} /* formatMonthList */


/*----------------------------------------------------------------------*/

static Boolean
  formatWeekList( char             *style,
                  PRINT_PARAM_REF  print_param )
{

  /* Variables. */
  Boolean        compact = False;
  Boolean        day_written = False;
  Boolean        ok;
  int            day_count = 0;
  int            days_per_page;
  char           buffer[ 1024 ];
  ENTRY          entry_record;
  LST_DESC_TYPE  entries_ref;
  LST_STATUS     lst_status;
  TIM_TIME_REF   curr_date;


  /* Code. */

  /* Fetch the entries. */
  (void) fetchEntries( input_file_ref, &entries_ref );

  if( print_param -> only_workdays )
    days_per_page = 5;
  else
    days_per_page = 7;

  fprintf( output_file_ref, "/PageFormat (%s) def\n\n", paper_size );
  fprintf( output_file_ref, "/SheetsPerPage %s def\n", sheets_per_page );


  /* Specific initialization. */
  fprintf( output_file_ref, "/TickScaleStart %d.%d def\n", 
           TimHour( print_param -> start_time ),
           TimMinute( print_param -> start_time ) );

  fprintf( output_file_ref, "/TickScaleEnd %d.%d def\n", 
           TimHour( print_param -> end_time ),
           TimMinute( print_param -> end_time ) );

  fprintf( output_file_ref, "\n" );
  fprintf( output_file_ref, "/DaysPerPage %d def\n", days_per_page );

  ok = outputPrologue( "XDpsWeek.pro", output_file_ref, 1 );
  if( ! ok )
    return( False );

  if( strcmp( style, "CompWeekList" ) == 0 )
    compact = True;

  /* Specific initialization. */
  fprintf( output_file_ref, "\n" );

  fprintf( output_file_ref, "/DateLabel (%s) def\n",
           nlsText( msgGetText( MXDI_FOPR_DATE ) ) );
  fprintf( output_file_ref, "/DbLabel (%s) def\n",
           nlsText( msgGetText( MXDI_FOPR_CALENDAR ) ) );


  /* Common nitialization. */
  commonInit( output_file_ref, print_param );

  fprintf( output_file_ref, "\n" );
  fprintf( output_file_ref, "%% Start a new page\n" );
  fprintf( output_file_ref, "bop\n\n" );


  /* Process all the entries. */
  lst_status = LstLinkCurrentFirst( entries_ref );
  if( lst_status == LST_OK )
    (void) LstLinkGetCurrent( entries_ref, &entry_record );

  curr_date = print_param -> start_date;

  while( curr_date <= print_param -> end_date ) {

    if( print_param -> only_workdays && 
        TimIndexOfDayInIsoWeek( curr_date ) > 5 ) {
      TimAddDays( &curr_date, 1 );
      continue;
    }

    if( ! compact ||
        (lst_status == LST_OK && entry_record.date_stamp == curr_date ) ) {

      if( day_written ) {
        fprintf( output_file_ref, "%% End of the day.\n" );
        fprintf( output_file_ref, "endDay\n\n" );
      }

      /* Time for a new page? */
      if( day_count == days_per_page ) {
        fprintf( output_file_ref, "%% Start a new page.\n" );
        fprintf( output_file_ref, "newPage\n" );
        fprintf( output_file_ref, "\n" );

        day_count = 0;
      }

      fprintf( output_file_ref, "%% Start a new day.\n" );

      TimFormatStrTime( curr_date, "%A", buffer, sizeof( buffer ) );
      fprintf( output_file_ref, "(%s) ", nlsText( buffer ) );
      TimFormatStrTime( curr_date, "%b", buffer, sizeof( buffer ) );
      fprintf( output_file_ref, "(%s %d) startDay\n\n", 
               nlsText( buffer ), TimIndexOfDay( curr_date ) );

      day_written = True;
      day_count++;
    }


    /* Do we have a valid entry? */
    while( lst_status == LST_OK && entry_record.date_stamp == curr_date ) {

      /* Write the note? */
      if( entry_record.entry_type == XTM_DB_DAY_NOTE ) {
        fprintf( output_file_ref, "%% Write a note.\n" );
        fprintf( output_file_ref, "startNote\n" );

        if( flagIsSet( entry_record.flags, XTM_DB_FLAG_IMPORTANT ) )
          fprintf( output_file_ref, "importantNote\n" );

      /* Write the entry? */
      } else {
        fprintf( output_file_ref, "%% Write an entry.\n" );
        fprintf( output_file_ref, "%d.%d hour %d min startEntry\n",
                 TimHour( entry_record.time_stamp ),
                 TimMinute( entry_record.time_stamp ),
                 entry_record.duration );

        if( flagIsSet( entry_record.flags, XTM_DB_FLAG_IMPORTANT ) )
          fprintf( output_file_ref, "importantEntry\n" );
      }

      /* Entry lines. */
      writeEntryText( output_file_ref, entry_record.text, &entry_record );

      /* The end of the entry. */
      if( entry_record.entry_type == XTM_DB_DAY_NOTE )
        fprintf( output_file_ref, "endNote\n\n" );
      else
        fprintf( output_file_ref, "endEntry\n\n" );

      /* Next entry. */
      lst_status = LstLinkCurrentNext( entries_ref );

      if( lst_status == LST_OK )
        (void) LstLinkGetCurrent( entries_ref, &entry_record );

    } /* while */


    TimAddDays( &curr_date, 1 );

  } /* while */

  if( day_written ) {
    fprintf( output_file_ref, "%% End of the day.\n" );
    fprintf( output_file_ref, "endDay\n\n" );
  }

  /* OK, we're done. */
  fprintf( output_file_ref, "eop\n" );
  fprintf( output_file_ref, "lastPage\n\n" );
  fprintf( output_file_ref, "%%%%Trailer\n\n" );


  return( True );

} /* formatWeekList */


/*----------------------------------------------------------------------*/

static void
  commonInit( FILE             *file_ref,
              PRINT_PARAM_REF  print_param )
{

  /* Variables. */
  char  buffer[ 1024 ];
  char  buffer1[ 1024 ];
  char  buffer2[ 1024 ];


  /* Code. */

  /* Initialization. */
  if( strncmp( &print_param -> time_format[ 0 ], "12", 2 ) == 0 )
    fprintf( file_ref, "/Tf12Hour true def\n" );
  else
    fprintf( file_ref, "/Tf12Hour false def\n" );

  if( strncmp( &print_param -> time_format[ 2 ], "  ", 2 ) == 0 )
    fprintf( file_ref, "/Tf12HourSuffix () def\n" );
  else
    fprintf( file_ref, "/Tf12HourSuffix (%c) def\n", 
             print_param -> time_format[ 2 ] );

  if( strncmp( &print_param -> time_format[ 4 ], "  ", 2 ) == 0 )
    fprintf( file_ref, "/Tf24HourSuffix () def\n" );
  else
    fprintf( file_ref, "/Tf24HourSuffix (%c) def\n", 
             print_param -> time_format[ 4 ] );

  if( strncmp( &print_param -> time_format[ 6 ], "0", 1 ) == 0 )
    fprintf( file_ref, "/TfSeparator () def\n" );
  else
    fprintf( file_ref, "/TfSeparator (%c) def\n", 
             print_param -> time_format[ 6 ] );


  fprintf( file_ref, "/DbNames (%s) def\n",
           nlsText( print_param -> calendar_name ) );

  TimFormatDate( print_param -> start_date, buffer1, sizeof( buffer1 ) );
  TimFormatDate( print_param -> end_date, buffer2, sizeof( buffer2 ) );

  sprintf( buffer, "%s - %s", buffer1, buffer2 );
  fprintf( file_ref, "/DateRange (%s) def\n", nlsText( buffer ) );

  TimFormatDate( TimLocalTime( TimMakeTimeNow() ),
                 buffer1, sizeof( buffer1 ) );
  TimFormatTime( TimLocalTime( TimMakeTimeNow() ),
                 buffer2, sizeof( buffer2 ) );

  sprintf( buffer, "%s %s", buffer1, buffer2 );
  fprintf( file_ref, "/FooterDate (%s) def\n", nlsText( buffer ) );

  if( print_param -> print_3d_look )
    fprintf( file_ref, "/PrIn3d true def\n" );
  else
    fprintf( file_ref, "/PrIn3d false def\n" );

  if( print_param -> only_workdays )
    fprintf( file_ref, "/PrOnlyWorkdays true def\n" );
  else
    fprintf( file_ref, "/PrOnlyWorkdays false def\n" );

  if( TimIndexOfFirstDayInWeek() > 0 )
    fprintf( file_ref, "/MonDayOne true def\n" );
  else
    fprintf( file_ref, "/MonDayOne false def\n" );


  return;

} /* commonInit */


/*----------------------------------------------------------------------*/

static char
  *nlsText( char  *base_text )
{

  /* Variables. */
  int   char_index;
  char  check_char;
  char  buffer[ 50 ];
  char  *base_ref;
  char  *nls_ref;

  static char  nls_buffer[ 10000 ];


  /* Code. */

  base_ref = base_text;
  nls_ref  = nls_buffer;

  while( *base_ref != '\0' ) {

    check_char = *base_ref;
    char_index = (int) check_char;

    /* Re-encode? */
    if( *base_ref == '(' ) {
      strncpy( nls_ref, "\\(", 2 );
      nls_ref = nls_ref + 2;
    } else if( *base_ref == ')' ) {
      strncpy( nls_ref, "\\)", 2 );
      nls_ref = nls_ref + 2;
    } else if( char_index >= 161 && char_index <= 255 ) {
      sprintf( buffer, "\\%3.3d", ps_iso_encoding[ char_index - 161 ] );
      strncpy( nls_ref, buffer, 4 );
      nls_ref = nls_ref + 4;
    } else {
      *nls_ref = *base_ref;
      nls_ref++;
    }

    base_ref++;

  } /* while */

  *nls_ref = '\0';

  return( nls_buffer );

} /* nlsText */


/*----------------------------------------------------------------------*/

static int
  noCaseStrcmp( char  *buffer1,
                char  *buffer2 )
{

  /* Variables. */
  char  *char_ref1;
  char  *char_ref2;


  /* Code. */

  if( strlen( buffer1 ) != strlen( buffer2 ) )
    return( strcmp( buffer1, buffer2 ) );

  char_ref1 = buffer1;
  char_ref2 = buffer2;

  while( *char_ref1 != '\0' ) {
    if( tolower( *char_ref1 ) < tolower( *char_ref2 ) )
      return( -1 );
    else if( tolower( *char_ref1 ) > tolower( *char_ref2 ) )
      return( 1 );

    char_ref1++;
    char_ref2++;
  }

  return( 0 );

} /* noCaseStrcmp */


/*----------------------------------------------------------------------*/

static Boolean
  outputPrologue( char  *prologue,
                  FILE  *output_file,
                  int   level )
{

  /* Variables. */
  Boolean  status = True;
  char     buffer[ 1024 ];
  char     key[ 1024 ];
  char     prologue_file[ PATH_MAX ];
  char     *char_ref;
  FILE     *file_ref;


  /* Code. */

  if( level > 9 )
    return( False );

  /* Build the prologue filename. */
  sprintf( prologue_file, "%s/%s", prologue_dir, prologue );

  file_ref = fopen( prologue_file, "r" );
  if( file_ref == NULL )
    return( False );


  /* Read the prologue file. */
  do {

    /* Fetch a line. */
    char_ref = fgets( buffer, sizeof( buffer ), file_ref );
    if( char_ref == NULL )
      break;

    /* Include file? */
    key[ 0 ] ='\0';
    sscanf( buffer, "%s", key );

    if( strcmp( key, "#include" ) == 0 ) {
      sscanf( buffer, "%s %s", key, prologue_file );

      status = outputPrologue( prologue_file, output_file, level + 1 );
    } else {
      fprintf( output_file, "%s", buffer );
    }

  } while( True );

  fclose( file_ref );


  return( status );

} /* outputPrologue */


/*----------------------------------------------------------------------*/

static void
  writeEntryText( FILE   *file_ref,
                  char   *entry_text,
                  ENTRY  *entry_ref )
{

  /* Variables. */
  Boolean  first_line;
  int      char_read;
  int      items;
  char     buffer[ 1024 ];
  char     flag_buffer[ 100 ];
  char     *char_ref;


  /* Code. */

  char_ref   = entry_text;
  first_line = True;

  while( *char_ref != '\0' ) {
    char_read = strlen( char_ref );

    items = sscanf( char_ref, "%[^\n]\n%n", buffer, &char_read );
    if( items != 1 )
      break;

    flag_buffer[ 0 ] = '\0';

    if( first_line ) {
      if( flagIsSet( entry_ref -> flags, XTM_DB_FLAG_NOTE_DONE ) )
        strcat( flag_buffer, "\\201" );
    }
    if( strlen( flag_buffer ) > 0 )
      strcat( flag_buffer, " " );

    if( entry_ref -> entry_type == XTM_DB_DAY_NOTE )
      fprintf( file_ref, "(%s%s) nnl\n", flag_buffer, nlsText( buffer ) );
    else
      fprintf( file_ref, "(%s%s) nel\n", flag_buffer, nlsText( buffer ) );

    first_line = False;
    char_ref   = char_ref + char_read;
  }


  return;

} /* writeEntryText */


/*----------------------------------------------------------------------*/

static void
  writeMonthLayout( TIM_TIME_REF  month_date,
                    FILE          *file_ref )
{

  /* Variables. */
  int           index;
  int           start_on;
  char          buffer[ 50 ];
  TIM_TIME_REF  curr_date;
  TIM_TIME_REF  start_date;


  /* Code. */

  /* Introduction. */
  TimFormatStrTime( month_date, "%B", buffer, sizeof( buffer ) );

  fprintf( file_ref, "[ (%s) (%d) ",
           nlsText( buffer) , TimIndexOfYear( month_date ) );


  /* Weekdays. */
  curr_date = TimMakeTimeNow();

  index = TimIndexOfFirstDayInWeek();
  if( index == 0 )
    index = 7;

  while( TimIndexOfDayInIsoWeek( curr_date ) != index )
    TimAddDays( &curr_date, -1 );

  fprintf( output_file_ref, "( ) " );

  for( index = 0; index < 7; index++ ) {
    TimFormatStrTime( curr_date, "%A", buffer, sizeof( buffer ) );
    fprintf( output_file_ref, "(%2.2s) ", nlsText( buffer ) );
    TimAddDays( &curr_date, 1 );
  }


  /* The first in the month. */
  start_date = TimMakeTime( TimIndexOfYear(  month_date ),
                            TimIndexOfMonth( month_date ),
                            1, 0, 0, 0 );


  /* Find the first day in the week. */
  if( TimIndexOfFirstDayInWeek() == 1 )
    start_on = 1;
  else
    start_on = 7;

  curr_date = start_date;
  while( TimIndexOfDayInIsoWeek( curr_date ) != start_on )
    TimAddDays( &curr_date, -1 );


  /* Write the dates and week numbers. */
  for( index = 0; index < 6 * 8; index++ ) {

    /* Week number? */
    if( index % 8 == 0 ) {
      if( TimIndexOfMonth( curr_date ) > TimIndexOfMonth( start_date ) )
        fprintf( file_ref, "(0) " );
      else
        fprintf( file_ref, "(%d) ", TimIndexOfWeek( curr_date ) );

    /* Day. */
    } else {
      if( curr_date < start_date ||
          TimIndexOfMonth( curr_date ) != TimIndexOfMonth( start_date ) )
        fprintf( file_ref, "(0) " );
      else
        fprintf( file_ref, "(%d) ", TimIndexOfDay( curr_date ) );

      TimAddDays( &curr_date, 1 );

    } /* if */

  } /* loop */

  /* We're done. */
  fprintf( file_ref, "] def\n" );


  return;

} /* writeMonthLayout */
