/*----------------------------------------------------------------------------
--
--  Module:           xtmCustom
--
--  Project:          XDiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Module to handle data in XDiary the user can customize. The
--    data can be saved in a file and can be loaded from the same file.
--
--  Filename:         xtmCustom.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-06-15
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmCustom.c, Version: 1.1, Date: 95/02/18 16:03:35";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/signal.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/StringDefs.h>

#include <Xm/Protocols.h>

#include <Xm/Xm.h>
#include <Xm/CascadeB.h>
#include <Xm/RowColumn.h>

#include "System.h"
#include "LstLinked.h"
#include "Message.h"

#include "msgTopic.h"
#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmCustArch.h"
#include "xtmCustBase.h"
#include "xtmCustCals.h"
#include "xtmCustGroup.h"
#include "xtmCustIncl.h"
#include "xtmDbTools.h"
#include "xtmGroupDb.h"
#include "xtmHelp.h"
#include "xtmIcons.h"
#include "xtmPickDiary.h"
#include "xtmResource.h"
#include "xitError.h"
#include "xitParSet.h"
#include "xitStickyMsg.h"
#include "xitTools.h"
#include "xtmTools.h"
#include "xtmCustom.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Name of program. */
#define PROGRAM_NAME   "xdcustom"

/* Program class (also the name of the application defaults file). */
#define PROGRAM_CLASS  "XDiary"


/* Local widgets in the custom window. */
#define custRc             dataLocalW[  0 ]
#define genDescLa          dataLocalW[  1 ]
#define menuBr             dataLocalW[  2 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/



/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of program. */
static char  *program_name;


/* Name of module. */
static char  *module_name = "xtmCustom";

/* IDs for the help windows. */
static char  *custom_window_id = "Custom";


/* Name of text domain. */
static char  *text_domain      = "XDiary";
static char  *info_text_domain = "Topic";

/* Keyboard translations for newline in multi-line editor. */
XtTranslations  xtm_cu_newline_trans;

/* Application data. */
static  XTM_CU_BASE_DATA  appl_data;


/* Key translations. */
static char newlineTrans[] = 
  "<Key>Return:    newline()";


/* Program options. */
static XrmOptionDescRec options[] = {
  { "-geometry",     "CustomTl.geometry", XrmoptionSepArg, NULL },
  { "-h",            "*.StartupHelp",     XrmoptionNoArg,  "True" },
  { "-help",         "*.StartupHelp",     XrmoptionNoArg,  "True" },
  { "-iconic",       "CustomTl.iconic",   XrmoptionNoArg,  "True" },
  { "-lan",          "*.msgLanguage",     XrmoptionSepArg, NULL },
  { "-language",     "*.msgLanguage",     XrmoptionSepArg, NULL },
  { "-noFileLock",   "*.useFileLock",     XrmoptionNoArg,  "False" },
  { "-nofilelock",   "*.useFileLock",     XrmoptionNoArg,  "False" },
  { "-usage",        "*.StartupHelp",     XrmoptionNoArg,  "True" },
  { "-version",      "*.VersionHelp",     XrmoptionNoArg,  "True" },
};

/* Application resources. */
static XtResource  base_resources[] = {

  { "startupHelp", "StartupHelp", XtRBoolean, sizeof( Boolean ),
    XtOffset( XTM_CU_BASE_DATA_REF, startup_help ), 
    XtRString, "False" },

  { "versionHelp", "VersionHelp", XtRBoolean, sizeof( Boolean ),
    XtOffset( XTM_CU_BASE_DATA_REF, version_help ), 
    XtRString, "False" },

};


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void 
  calCustApplyCB( XTM_CC_REASON  reason,
                  XTM_CD_HANDLE  cal_db_handle,
                  void           *user_data );

static void 
  cancelCB( Widget                widget,
            XTM_CU_BASE_DATA_REF  appl_data_ref,
            XtPointer             call_data );

static Widget
  createCustomWindow( Widget                parent,
                      XTM_CU_BASE_DATA_REF  appl_data_ref );

static void 
  customParam( XTM_CU_BASE_DATA_REF  appl_data_ref );

static void
  displayUsage();

static void 
  doCustCB( Widget                     widget,
            XTM_CU_BASE_DATA_REF       appl_data_ref,
            XmRowColumnCallbackStruct  *call_data );

static void 
  exitCB( Widget     widget,
          XtPointer  client_data,
          XtPointer  call_data );

static void 
  groupCustApplyCB( XTM_CG_REASON  reason,
                    XTM_GD_HANDLE  new_group_db_handle,
                    void           *user_data );

static void 
  infoCB( Widget                     widget,
          XTM_CU_BASE_DATA_REF       appl_data_ref,
          XmRowColumnCallbackStruct  *call_data );

static void 
  okCB( Widget                widget,
        XTM_CU_BASE_DATA_REF  appl_data_ref,
        XtPointer             call_data );

static void 
  saveCB( Widget                widget,
          XTM_CU_BASE_DATA_REF  appl_data_ref,
          XtPointer             call_data );

static void
  saveCustomData( XTM_CU_BASE_DATA_REF  appl_data_ref );

static int
  startProcess( Widget  parent,
                char    *process_args[] );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

void 
  main( unsigned int argc, char *argv[] )
{

  /* Variables. */
  int                 index;
  char                *char_ref;
  Arg                 args[ 10 ];
  Cardinal            n;
  Display             *display;
  TIM_STATUS_TYPE     time_status;
  XTM_CB_STATUS       custom_status;
  XTM_GL_CUSTOM_DATA  custom_data;


  /* Code. */

  /* Fetch the name of the program. */
  program_name = PROGRAM_NAME;
  xitErSetApplicationName( PROGRAM_CLASS );


  /* Save the original command parameters. */
  custom_data.orig_argc = argc;
  custom_data.orig_argv = (char**) XtMalloc( argc * sizeof( char* ) );

  for( index = 0; index < argc; index++ )
    custom_data.orig_argv[ index ] = XtNewString( argv[ index ] );
  
  /* NLS enabled. */
  xtmToSetLocale( program_name );


  /* Initialize. */
  SysInitializeEnvironment();
  xtmDbInitializeProcessId();

  
  /* Initialize toolkit and open display. */
  XtToolkitInitialize();

  appl_data.context = XtCreateApplicationContext();
  display = XtOpenDisplay( appl_data.context, NULL,
                           NULL, PROGRAM_CLASS,
                           options, XtNumber( options ),
#if XtSpecificationRelease < 5
                           (Cardinal *) &argc,
#else
                           (int *) &argc,
#endif
                           argv );

  if( display == NULL )
    xitErMessage( NULL, XIT_ER_FATAL, 
                  module_name, "main",
                  "Cannot open display, check your DISPLAY variable." );

  /* Resource mapping.*/
  xtmToInitializeResourceMapping( argc, argv, display );
  
  /* Default translations */
  xtm_cu_newline_trans   = XtParseTranslationTable( newlineTrans );

  /* Create application shell. */
  n = 0;
  appl_data.toplevel = XtAppCreateShell( NULL, PROGRAM_CLASS,
                                         applicationShellWidgetClass,
                                         display,
                                         args, n );

  /* Get base application resources. */
  XtGetApplicationResources( appl_data.toplevel, (XtPointer) &appl_data, 
                             base_resources, 
                             XtNumber( base_resources ), 
                             NULL, 0 );

  /* Get base application resources. */
  XtGetApplicationResources( appl_data.toplevel, (XtPointer) &appl_data, 
                             base_resources, 
                             XtNumber( base_resources ), 
                             NULL, 0 );


  /* Get customize resources. */
  xtmRsFetchCustomResources( &custom_data, appl_data.toplevel );

  /* Initialize application data. */
  custom_data.cal_db_handle   = NULL;
  custom_data.group_db_handle = NULL;
  custom_data.archive_files   = NULL;
  custom_data.include_files   = NULL;

  appl_data.cal_handle   = NULL;
  appl_data.group_handle = NULL;
  appl_data.info_handle  = NULL;
  appl_data.paramW       = NULL;
  appl_data.custom_data  = &custom_data;


  /* Display current version? */
  if( appl_data.version_help ) {
    printf( "%s: Version: %s\n", program_name, VERSION_ID );
    exit( 0 );
  }

  /* Help requested? */
  if( appl_data.startup_help ) {
    displayUsage();
    exit( 0 );
  }


  /* Get customized data from file. */
  custom_status = xtmCbGetDataFromFile( appl_data.custom_data );

  if( custom_status == XTM_CB_WRONG_VERSION ) {
    char_ref = (char *) 
      SysMalloc( strlen( msgGetText( MXDI_CUST_WRONG_VERSION ) ) + 50 );

    sprintf( char_ref, msgGetText( MXDI_CUST_WRONG_VERSION ),
             xtmCbWhatVersion() );

    xitStDisplaySticky( appl_data.toplevel,
                        char_ref, XmUNSPECIFIED_PIXMAP,
                        msgGetText( MXDI_OK_BUTTON ), NULL,
                        NULL, NULL, NULL,
                        NULL );
    SysFree( char_ref );
  }



  /* Initialize necessary text domains. */
  msgInitialize();
  msgInitCatalogue( text_domain, NULL, custom_data.msg_language,
                    msgXdiaryTexts );
  msgInitCatalogue( info_text_domain, NULL, custom_data.msg_language,
                    msgTopicTexts );

  /* Default catalogue Xdiary. */
  msgTextDomain( text_domain );


  /* Initialize the time formats. */
  time_status = TimInitializeFormat( custom_data.date_format,
                                     custom_data.time_format );
  if( time_status != TIM_OK )
    xitErMessage( appl_data.toplevel, XIT_ER_ERROR, 
                  module_name, "main",
                  msgGetText( MXDI_ERRMSG_DATE_OR_TIME_FORMAT ) );


  /* Set colors and fonts in the resource database. */
  xtmRsFetchColors( &custom_data, appl_data.toplevel );


  /* Initialize the help system. */
  appl_data.info_handle = xitInInitialize( appl_data.toplevel,
                                           NULL, NULL );


  /* Create the customize window. */
  appl_data.customW = createCustomWindow( appl_data.toplevel, &appl_data );


  /* Display the customize window. */
  XtPopup( appl_data.customW, XtGrabNone );


  /* Session management. */
  xtmToSessionMgmt( appl_data.customW,
                    custom_data.orig_argc, custom_data.orig_argv );


  /* Enter the event loop. */
  XtAppMainLoop( appl_data.context );

} /* main */


/*----------------------------------------------------------------------*/

static Widget
  createCustomWindow( Widget                parent,
                      XTM_CU_BASE_DATA_REF  appl_data_ref )
{

  /* Variables. */
  int       index;
  char      *char_ref;
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    customTl;
  Widget    dataLocalW[ 3 ];
  Widget    doCustBu[ 5 ];
  Widget    menuCasc[ 2 ];
  Widget    menuHelpBu[ 7 ];
  Widget    menuFileBu[ 2 ];
  Widget    pulldownMenu[ 2 ];
  Widget    toplevel;
  Widget    workFo;

  static char  *pull_downs[] = { "pdown1", "pdown2" };

  static XIT_PUSH_STRUCT db_action_def[] = {
    { "AddPb",     "", "", True, NULL },
    { "DeletePb",  "", "", True, NULL },
  };

  static XIT_PUSH_STRUCT db_more_def[] = {
    { "MorePb",     "", "", True, NULL },
    { "IncludePb",  "", "", True, NULL },
    { "DbSelectPb", "", "", True, NULL },
  };

  static XIT_PUSH_STRUCT group_action_def[] = {
    { "AddPb",     "", "", True, NULL },
    { "DeletePb",  "", "", True, NULL },
  };

  static XIT_TEXT_STRUCT text_buffer[] = {
    { "DbNameTx",    NULL, 1, True },
    { "DbDirTx",     NULL, 1, True },
    { "GroupNameTx", NULL, 1, True },
    { "GroupMemTx",  NULL, 1, True },
  };

  static XIT_PUSH_STRUCT do_cust_def[] = {
    { "CustCalPb",     "", "", True, NULL },
    { "CustGroupsPb",  "", "", True, NULL },
    { "CustParPb",     "", "", True, NULL },
    { "CustArchivePb", "", "", True, NULL },
    { "CustAccessPb",  "", "", True, NULL },
  };

  static XIT_CASCADE_STRUCT menupane[] = {
    { "", "", "FilePane" },
    { "", "", "HelpPane" },
  };

  static XIT_MENU_BUTTON_STRUCT file_casc[] = {
    { "", "",  NULL, "CloseCancelBu", True, False, False },
    { "", "",  NULL, "CloseSaveBu",   True, False, False },
  };

  static XIT_MENU_BUTTON_STRUCT help_casc[] = {
    { "", "", NULL, "ContextBu", True, False, False },
    { "", "", NULL, "WindowsBu", True, False, False },
    { "", "", NULL, "KeysBu",    True, False, False },
    { "", "", NULL, "IndexBu",   True, False, False },
    { "", "", NULL, "HelpBu",    True, False, False },
    { "", "", NULL, "VersionBu", True, False, False },
    { "", "", NULL, "AboutBu",   True, False, False },
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "",   okCB,     NULL },
    { NULL, NULL,     NULL },
    { "",   saveCB,   NULL },
    { NULL, NULL,     NULL },
    { "",   cancelCB, NULL },
  };


  /* Code. */

  menupane[ 0 ].title    = msgGetText( MXDI_FILE_MENU );
  menupane[ 0 ].mnemonic = msgGetText( MXDI_FILE_MENU_ACC );
  menupane[ 1 ].title    = msgGetText( MXDI_HELP_MENU );
  menupane[ 1 ].mnemonic = msgGetText( MXDI_HELP_MENU_ACC );

  file_casc[ 0 ].title    = msgGetText( MXDI_CLOSE_CANCEL_MENU );
  file_casc[ 0 ].mnemonic = msgGetText( MXDI_CLOSE_CANCEL_MENU_ACC );
  file_casc[ 1 ].title    = msgGetText( MXDI_CLOSE_SAVE_MENU );
  file_casc[ 1 ].mnemonic = msgGetText( MXDI_CLOSE_SAVE_MENU_ACC );

  help_casc[ 0 ].title    = msgGetText( MXDI_HELP_CONTEXT );
  help_casc[ 0 ].mnemonic = msgGetText( MXDI_HELP_CONTEXT_ACC );
  help_casc[ 1 ].title    = msgGetText( MXDI_HELP_WINDOWS );
  help_casc[ 1 ].mnemonic = msgGetText( MXDI_HELP_WINDOWS_ACC );
  help_casc[ 2 ].title    = msgGetText( MXDI_HELP_KEYS );
  help_casc[ 2 ].mnemonic = msgGetText( MXDI_HELP_KEYS_ACC );
  help_casc[ 3 ].title    = msgGetText( MXDI_HELP_INDEX );
  help_casc[ 3 ].mnemonic = msgGetText( MXDI_HELP_INDEX_ACC );
  help_casc[ 4 ].title    = msgGetText( MXDI_HELP_HELP );
  help_casc[ 4 ].mnemonic = msgGetText( MXDI_HELP_HELP_ACC );
  help_casc[ 5 ].title    = msgGetText( MXDI_HELP_VERSION );
  help_casc[ 5 ].mnemonic = msgGetText( MXDI_HELP_VERSION_ACC );
  help_casc[ 6 ].title    = msgGetText( MXDI_HELP_ABOUT );
  help_casc[ 6 ].mnemonic = msgGetText( MXDI_HELP_ABOUT_ACC );

  action_buttons[ 0 ].label = msgGetText( MXDI_OK_BUTTON );
  action_buttons[ 0 ].data  = appl_data_ref;
  action_buttons[ 2 ].label = msgGetText( MXDI_SAVE_BUTTON );
  action_buttons[ 2 ].data  = appl_data_ref;
  action_buttons[ 4 ].label = msgGetText( MXDI_CANCEL_BUTTON );
  action_buttons[ 4 ].data  = appl_data_ref;

  do_cust_def[ 0 ].title = msgGetText( MXDI_CUST_CAL_BUTTON );
  do_cust_def[ 1 ].title = msgGetText( MXDI_CUST_GROUP_BUTTON );
  do_cust_def[ 2 ].title = msgGetText( MXDI_CUST_PARAM_BUTTON );
  do_cust_def[ 3 ].title = msgGetText( MXDI_CUST_ARCH_BUTTON );
  do_cust_def[ 4 ].title = msgGetText( MXDI_CUST_ACCESS_BUTTON );


  /* Create a toplevel dialog with buttons. */
  toplevel = xitGetToplevelWidget( parent );

  customTl = xitCreateToplevelDialog( toplevel, "CustomTl",
                                      1, 0,
                                      action_buttons, 
                                      XtNumber( action_buttons ) );

  char_ref = msgGetText( MXDI_CUSTOM_TITLE );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, char_ref ); n++;
  XtSetValues( customTl, args, n );

  char_ref = msgGetText( MXDI_CUSTOM_IC_TITLE );

  n = 0;
  XtSetArg( args[ n ], XmNiconName, char_ref ); n++;
  XtSetValues( customTl, args, n );


  /* Exit the application if this window is deleted. */
  {
    Atom  wm_delete_window;

    wm_delete_window = XmInternAtom( XtDisplay( customTl ),
                                     "WM_DELETE_WINDOW", False );

    XmAddWMProtocols(        customTl, &wm_delete_window, 1 );
    XmAddWMProtocolCallback( customTl, wm_delete_window, 
                             (XtCallbackProc) cancelCB,
                             (XtPointer) &appl_data );
  } /* block */


  /* Container for the contents of the window. */
  workFo = XtNameToWidget( customTl, "CustomTlBase.CustomTlFo" );


  /* Create the menubar and menu cascades. */
  menuBr = XmCreateMenuBar( workFo, "MenuBr", args, 0 );

  n = 0;
  for( index = 0; index < XtNumber( pulldownMenu ); index++ )
    pulldownMenu[ index ] = XmCreatePulldownMenu( menuBr, 
                                                  pull_downs[ index ], 
                                                  NULL, n );

  for( index = 0; index < XtNumber( menuCasc ); index++ )
    menuCasc[ index ] = xitCreateCascadeButton( menuBr, 
                                                pulldownMenu[ index ], 
                                                &menupane[ index ] );

  /* The help button should be placed to the right. */
  index = XtNumber( menuCasc ) - 1;
  n     = 0;
  XtSetArg( args[ n ], XmNmenuHelpWidget, menuCasc[ index ] ); n++;
  XtSetValues( menuBr, args, n );


  /* Create the file menu. */
  for( index = 0; index < XtNumber( menuFileBu ); index++ )
    menuFileBu[ index ] = xitCreateMenuPushButton( pulldownMenu[ 0 ],
                                                   &file_casc[ index ] );
  XtAddCallback( menuFileBu[ 0 ], XmNactivateCallback, 
                 (XtCallbackProc) cancelCB, (XtPointer) appl_data_ref );
  XtAddCallback( menuFileBu[ 1 ], XmNactivateCallback, 
                 (XtCallbackProc) okCB, (XtPointer) appl_data_ref );


  /* Create the help menu. */
  XtAddCallback( pulldownMenu[ 1 ], XmNentryCallback, 
                 (XtCallbackProc) infoCB, (XtPointer) appl_data_ref );

  for( index = 0; index < XtNumber( menuHelpBu ); index++ ) {
    menuHelpBu[ index ] = xitCreateMenuPushButton( pulldownMenu[ 1 ], 
                                                   &help_casc[ index ] );

    XtAddCallback( menuHelpBu[ index ], XmNactivateCallback, 
                   (XtCallbackProc) infoCB, (XtPointer) index );
  }

  /* We can't do context sensitive help. */
  XtSetSensitive( menuHelpBu[ 0 ], False );


  /* General description. */
  genDescLa = xitCreateLabel( workFo, "GenDescLa", 
                              msgGetText( MXDI_CUST_GENERAL_LABEL ), -1 );

  /* What kind of customization can we do? */
  n = 0;
  XtSetArg( args[ n ], XmNspacing,      10 ); n++;
  XtSetArg( args[ n ], XmNmarginHeight, 10 ); n++;
  custRc = XmCreateRowColumn( workFo, "CustRc", args, n );

  XtAddCallback( custRc, XmNentryCallback, 
                 (XtCallbackProc) doCustCB, (XtPointer) appl_data_ref );

  for( index = 0; index < XtNumber( doCustBu ); index++ ) {

    doCustBu[ index ] = xitCreatePushButton( custRc, 
                                             &do_cust_def[ index ] );

    XtAddCallback( doCustBu[ index ], XmNactivateCallback, 
                   (XtCallbackProc) doCustCB, (XtPointer) index );

  } /* loop */


  /* Put the elements together. */
  xitAttachWidget( menuBr,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_FORM, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( genDescLa,
                   XmATTACH_WIDGET, menuBr, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,   XmATTACH_NONE, NULL );
  xitAttachWidget( custRc,
                   XmATTACH_WIDGET, genDescLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );

  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( genDescLa, args, n );
  XtSetValues( custRc,    args, n );

  /* Manage the widgets. */
  XtManageChildren( doCustBu,    XtNumber( doCustBu ) );
  XtManageChildren( menuCasc,    XtNumber( menuCasc ) );
  XtManageChildren( menuHelpBu,  XtNumber( menuHelpBu ) );
  XtManageChildren( menuFileBu,  XtNumber( menuFileBu ) );

  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );


  /* Set icon for this window. */
  xtmIcSetSimpleIcon( customTl, workFo, XTM_IC_ICON_CUSTOM );

  /* Set the size of the window. */
  xitSetSizeToplevelDialog( customTl, True );


  return( customTl );

} /* createCustomWindow */


/*----------------------------------------------------------------------*/

static void 
  customParam( XTM_CU_BASE_DATA_REF  appl_data_ref )
{

  /* Variables. */
  Cardinal                n;
  XTM_GL_CUSTOM_DATA_REF  custom_data;
  XIT_PS_LABELS           param_labels;

  static  Boolean  true_value              = True;
  static  Boolean  false_value             = False;
  static  Boolean  workdays[ 7 ];
  static  int      alarm_lines             = 2;
  static  int      app_pane_height         = 300;
  static  int      day_view_days           = 2;
  static  int      day_view_day_switch     = 0;
  static  int      day_width               = 0;
  static  int      default_paper_size      = 1;
  static  int      default_sheets_per_page = 0;
  static  int      entry_delta             = 30;
  static  int      entry_lines             = 2;
  static  int      entry_pane_width        = 300;
  static  int      note_pane_height        = 50;
  static  int      print_lines             = 2;
  static  int      refresh_rate            = 20;
  static  int      summary_lines           = 2;
  static  int      start_day_view_on       = 0;
  static  int      start_hour              = 8;
  static  int      stop_hour               = 18;
  static  int      tune                    = 1;
  static  char     *date_format            = "YMD-1111";
  static  char     *time_format            = "12    :0";

  static XIT_PAR_REC  par_rec[] = {
    { "", "", XIT_PS_TYPE_DATE_FORMAT_FIELD,   /* Date format. */
      0, 0, 0, 0, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_SELECT_INDEX_FIELD,  /* Paper size. */
      0, 0, 20, 20, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_TIME_FORMAT_FIELD,   /* Time format. */
      0, 0, 0, 0, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_WORKDAYS_FIELD,      /* Work days. */
      0, 0, 0, 0, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_SEPARATOR,
      0, 0, 0, 0, "", 0, NULL, NULL, 1 },
    { "", "", XIT_PS_TYPE_INTEGER,             /* Day view days. */
      1, 31, 3, 3, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_SELECT_INDEX_FIELD,  /* Day view day switch. */
      0, 0, 15, 15, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_YES_NO,              /* Day view list. */
      0, 0, 0, 0, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_INTEGER,             /* Day view starts. */
      0, 23, 3, 3, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_INTEGER,             /* Day view stops. */
      0, 23, 3, 3, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_YES_NO,              /* Display nav cal. */
      0, 0, 0, 0, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_SELECT_INDEX_FIELD,  /* Start day view on */
      0, 0, 15, 15, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_YES_NO,              /* True color included. */
      0, 0, 0, 0, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_SEPARATOR,
      0, 0, 0, 0, "", 0, NULL, NULL, 1 },
    { "", "", XIT_PS_TYPE_INTEGER_FIELD,       /* Entry delta. */
      5, 360, 4, 4, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_YES_NO,              /* Entry flags. */
      0, 0, 0, 0, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_YES_NO,              /* Reminder for new entries. */
      0, 0, 0, 0, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_INTEGER,             /* Tune. */
      1, 5, 3, 3, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_SEPARATOR,
      0, 0, 0, 0, "", 0, NULL, NULL, 1 },
    { "", "", XIT_PS_TYPE_YES_NO,              /* Print in 3d. */
      0, 0, 0, 0, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_INTEGER,             /* Print lines. */
      1, 50, 3, 3, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_YES_NO,              /* Print only workdays. */
      0, 0, 0, 0, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_SELECT_INDEX_FIELD,  /* Sheets per page. */
      0, 0, 5, 5, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_SEPARATOR,
      0, 0, 0, 0, "", 0, NULL, NULL, 1 },
    { "", "", XIT_PS_TYPE_INTEGER,             /* Appointment pane height. */
      1, 1500, 5, 5, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_INTEGER,             /* Day width. */
      0, 500, 4, 4, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_INTEGER,             /* Entry pane width. */
      1, 2000, 4, 4, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_INTEGER,             /* Note pane height. */
      1, 1500, 5, 5, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_SEPARATOR,
      0, 0, 0, 0, "", 0, NULL, NULL, 1 },
    { "", "", XIT_PS_TYPE_INTEGER,             /* Alarm lines. */
      1, 100, 4, 4, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_INTEGER,             /* Entry lines. */
      1, 50, 3, 3, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_INTEGER,             /* Summary lines. */
      1, 50, 3, 3, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_SEPARATOR,
      0, 0, 0, 0, "", 0, NULL, NULL, 1 },
    { "", "", XIT_PS_TYPE_YES_NO,              /* Confirm. */
      0, 0, 0, 0, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_YES_NO,              /* Do refresh. */
      0, 0, 0, 0, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_YES_NO,              /* Entries interactive. */
      0, 0, 0, 0, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_YES_NO,              /* Give alarms. */
      0, 0, 0, 0, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_YES_NO,              /* Prompt repeat window. */
      0, 0, 0, 0, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_SEPARATOR,
      0, 0, 0, 0, "", 0, NULL, NULL, 1 },
    { "", "", XIT_PS_TYPE_YES_NO,              /* Calendar month extended. */
      0, 0, 0, 0, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_INTEGER,             /* Refresh rate. */
      1, 120, 4, 4, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_YES_NO,              /* Standing in cal. */
      0, 0, 0, 0, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_YES_NO,              /* Summary startup. */
      0, 0, 0, 0, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_YES_NO,              /* Week numbers. */
      0, 0, 0, 0, "", 0, NULL, NULL, 0 },
  };


  /* Code. */

  custom_data = appl_data_ref -> custom_data;

  /* Defaults. */
  workdays[ 0 ] = True;
  workdays[ 1 ] = True;
  workdays[ 2 ] = True;
  workdays[ 3 ] = True;
  workdays[ 4 ] = True;
  workdays[ 5 ] = False;
  workdays[ 6 ] = False;


  /* Labels for the parameter window. */
  param_labels.window_title        = msgGetText( MXDI_CUSTOM_TITLE );
  param_labels.parameter_list      = msgGetText( MXDI_PARAMETERS_LABEL );
  param_labels.description         = msgGetText( MXDI_DESCRIPTION_LABEL );
  param_labels.parameter_value     = msgGetText( MXDI_VALUE_LABEL );
  param_labels.yes_label           = msgGetText( MXDI_YES_LABEL );
  param_labels.no_label            = msgGetText( MXDI_NO_LABEL );
  param_labels.on_label            = msgGetText( MXDI_OFF_LABEL );
  param_labels.off_label           = msgGetText( MXDI_ON_LABEL );
  param_labels.cancel_button       = msgGetText( MXDI_CANCEL_BUTTON );
  param_labels.defaults_button     = msgGetText( MXDI_DEFAULTS_LABEL );
  param_labels.ok_button           = msgGetText( MXDI_OK_BUTTON );
  param_labels.reset_button        = msgGetText( MXDI_RESET_BUTTON );
  param_labels.date_order_label    = msgGetText( MXDI_DATE_ORDER_LABEL );
  param_labels.mdy_label           = msgGetText( MXDI_MDY_LABEL );
  param_labels.dmy_label           = msgGetText( MXDI_DMY_LABEL );
  param_labels.ymd_label           = msgGetText( MXDI_YMD_LABEL );
  param_labels.date_sep_label      = msgGetText( MXDI_DATE_SEPARATOR_LABEL );
  param_labels.day_zero_label      = msgGetText( MXDI_DAY_LEAD_ZERO_LABEL );
  param_labels.month_zero_label    = msgGetText( MXDI_MONTH_LEAD_ZERO_LABEL );
  param_labels.century_label       = msgGetText( MXDI_CENTURY_LABEL );
  param_labels.hour12_label        = msgGetText( MXDI_DATE_HOUR12_LABEL );
  param_labels.hour24_label        = msgGetText( MXDI_DATE_HOUR24_LABEL );
  param_labels.time_sep_label      = msgGetText( MXDI_TIME_SEPARATOR_LABEL );
  param_labels.hour_zero_label     = msgGetText( MXDI_HOUR_LEAD_ZERO_LABEL );
  param_labels.week_1st_day_label  = msgGetText( MXDI_WEEK_1ST_DAY_LABEL );
  param_labels.week_1st_days       = msgGetText( MXDI_WEEK_1ST_DAYS );
  param_labels.change_button       = msgGetText( MXDI_CHANGE_BUTTON );
  param_labels.change_hidden_label = msgGetText( MXDI_ENTER_HIDDEN_LABEL );
  param_labels.weekdays_short      = msgGetText( MXDI_WEEKDAYS_MO_FIRST );

  param_labels.msg_confirm_default = msgGetText( MXDI_OVERWRITE_PARAM_CONF );
  param_labels.msg_confirm_reset   = msgGetText( MXDI_OVERWRITE_PARAM_CONF );
  param_labels.msg_invalid_value   = msgGetText( MXDI_NOT_VALID_PARAM );


  /* Parameter values. */
  n = 0;

  /* Date format. */
  par_rec[ n ].parameter_label = msgGetText( MXDI_DATE_FORMAT_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_DATE_FORMAT_HELP );
  par_rec[ n ].value           = custom_data -> date_format;
  par_rec[ n ].default_value   = date_format;
  par_rec[ n ].size            = sizeof( custom_data -> date_format );

  /* Paper size. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_PAPER_SIZE_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_PAPER_SIZE_HELP );
  par_rec[ n ].value           = &custom_data -> default_paper_size;
  par_rec[ n ].default_value   = &default_paper_size;
  par_rec[ n ].size            = sizeof( custom_data -> default_paper_size );
  par_rec[ n ].select_values   = msgGetText( MXDI_PRINT_SIZE_FORMAT );

  /* Time format. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_TIME_FORMAT_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_TIME_FORMAT_HELP );
  par_rec[ n ].value           = custom_data -> time_format;
  par_rec[ n ].default_value   = time_format;
  par_rec[ n ].size            = sizeof( custom_data -> time_format );

  /* Workdays. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_WORKDAYS_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_WORKDAYS_HELP );
  par_rec[ n ].value           = custom_data -> workdays;
  par_rec[ n ].default_value   = &workdays;
  par_rec[ n ].size            = sizeof( custom_data -> workdays );

  /* Separator. */
  n++;

  /* Day view days. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_DAY_VIEW_DAYS_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_DAY_VIEW_DAYS_HELP );
  par_rec[ n ].value           = &custom_data -> day_view_days;
  par_rec[ n ].default_value   = &day_view_days;
  par_rec[ n ].size            = sizeof( custom_data -> day_view_days );

  /* Day view day switch. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_DAY_VIEW_DAY_SWITCH_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_DAY_VIEW_DAY_SWITCH_HELP );
  par_rec[ n ].value           = &custom_data -> day_view_day_switch;
  par_rec[ n ].default_value   = &day_view_day_switch;
  par_rec[ n ].size            = sizeof( custom_data -> day_view_day_switch );
  par_rec[ n ].select_values   = msgGetText( MXDI_DAY_VIEW_SWITCH_DAYS );

  /* Day view list. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_DAY_VIEW_LIST_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_DAY_VIEW_LIST_HELP );
  par_rec[ n ].value           = &custom_data -> list_layout;
  par_rec[ n ].default_value   = &false_value;
  par_rec[ n ].size            = sizeof( custom_data -> list_layout );

  /* Day view starts. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_DAY_VIEW_STARTS_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_DAY_VIEW_STARTS_HELP );
  par_rec[ n ].value           = &custom_data -> start_hour;
  par_rec[ n ].default_value   = &start_hour;
  par_rec[ n ].size            = sizeof( custom_data -> start_hour );

  /* Day view stops. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_DAY_VIEW_STOPS_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_DAY_VIEW_STOPS_HELP );
  par_rec[ n ].value           = &custom_data -> stop_hour;
  par_rec[ n ].default_value   = &stop_hour;
  par_rec[ n ].size            = sizeof( custom_data -> stop_hour );

  /* Display nav cal. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_DISPLAY_NAV_CAL_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_DISPLAY_NAV_CAL_HELP );
  par_rec[ n ].value           = &custom_data -> display_nav_cal;
  par_rec[ n ].default_value   = &false_value;
  par_rec[ n ].size            = sizeof( custom_data -> display_nav_cal );

  /* Start day view on. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_START_DV_ON_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_START_DV_ON_HELP );
  par_rec[ n ].value           = &custom_data -> start_day_view_on;
  par_rec[ n ].default_value   = &start_day_view_on;
  par_rec[ n ].size            = sizeof( custom_data -> start_day_view_on );
  par_rec[ n ].select_values   = msgGetText( MXDI_START_DV_DAYS );

  /* True color included. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_TRUE_COLOR_INCL_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_TRUE_COLOR_INCL_HELP );
  par_rec[ n ].value           = &custom_data -> true_color_included;
  par_rec[ n ].default_value   = &true_value;
  par_rec[ n ].size            = sizeof( custom_data -> true_color_included );

  /* Separator. */
  n++;

  /* Entry delta. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_ENTRY_DELTA_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_ENTRY_DELTA_HELP );
  par_rec[ n ].value           = &custom_data -> default_entry_delta;
  par_rec[ n ].default_value   = &entry_delta;
  par_rec[ n ].size            = sizeof( custom_data -> default_entry_delta );

  /* Entry flags. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_ENTRY_FLAGS_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_ENTRY_FLAGS_HELP );
  par_rec[ n ].value           = &custom_data -> display_entry_flags;
  par_rec[ n ].default_value   = &true_value;
  par_rec[ n ].size            = sizeof( custom_data -> display_entry_flags );

  /* Reminder for new entries. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_ALARM_NEW_APP_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_ALARM_NEW_APP_HELP );
  par_rec[ n ].value           = &custom_data -> alarm_for_new_app;
  par_rec[ n ].default_value   = &true_value;
  par_rec[ n ].size            = sizeof( custom_data -> alarm_for_new_app );

  /* Tune. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_TUNE_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_TUNE_HELP );
  par_rec[ n ].value           = &custom_data -> default_tune;
  par_rec[ n ].default_value   = &tune;
  par_rec[ n ].size            = sizeof( custom_data -> default_tune );

  /* Separator. */
  n++;

  /* Print in 3d. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_PR_3D_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_PR_3D_HELP );
  par_rec[ n ].value           = &custom_data -> pr_in_3d;
  par_rec[ n ].default_value   = &true_value;
  par_rec[ n ].size            = sizeof( custom_data -> pr_in_3d );

  /* Print lines. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_PRINT_LINES_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_PRINT_LINES_HELP );
  par_rec[ n ].value           = &custom_data -> print_def_lines;
  par_rec[ n ].default_value   = &print_lines;
  par_rec[ n ].size            = sizeof( custom_data -> print_def_lines );

  /* Print only workdays. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_PR_WORKDAYS_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_PR_WORKDAYS_HELP );
  par_rec[ n ].value           = &custom_data -> pr_only_workdays;
  par_rec[ n ].default_value   = &false_value;
  par_rec[ n ].size            = sizeof( custom_data -> pr_only_workdays );

  /* Sheets per page. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_SHEETS_PER_PAGE_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_SHEETS_PER_PAGE_HELP );
  par_rec[ n ].value           = &custom_data -> print_def_sheets;
  par_rec[ n ].default_value   = &default_sheets_per_page;
  par_rec[ n ].size            = sizeof( custom_data -> print_def_sheets );
  par_rec[ n ].select_values   = msgGetText( MXDI_SHEETS_PER_PAGE );

  /* Separator. */
  n++;

  /* Appointment pane height. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_APP_PANE_HEIGHT_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_APP_PANE_HEIGHT_HELP );
  par_rec[ n ].value           = &custom_data -> app_pane_height;
  par_rec[ n ].default_value   = &app_pane_height;
  par_rec[ n ].size            = sizeof( custom_data -> app_pane_height );

  /* Day width. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_DAY_WIDTH_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_DAY_WIDTH_HELP );
  par_rec[ n ].value           = &custom_data -> def_day_width;
  par_rec[ n ].default_value   = &day_width;
  par_rec[ n ].size            = sizeof( custom_data -> def_day_width );

  /* Entry pane width. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_ENTRY_PANE_WIDTH_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_ENTRY_PANE_WIDTH_HELP );
  par_rec[ n ].value           = &custom_data -> entry_pane_width;
  par_rec[ n ].default_value   = &entry_pane_width;
  par_rec[ n ].size            = sizeof( custom_data -> entry_pane_width );

  /* Note pane height. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_NOTE_PANE_HEIGHT_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_NOTE_PANE_HEIGHT_HELP );
  par_rec[ n ].value           = &custom_data -> note_pane_height;
  par_rec[ n ].default_value   = &note_pane_height;
  par_rec[ n ].size            = sizeof( custom_data -> note_pane_height );

  /* Separator. */
  n++;

  /* Alarm lines. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_ALARM_LINES_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_ALARM_LINES_HELP );
  par_rec[ n ].value           = &custom_data -> alarm_show_lines;
  par_rec[ n ].default_value   = &alarm_lines;
  par_rec[ n ].size            = sizeof( custom_data -> alarm_show_lines );

  /* Entry lines. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_ENTRY_LINES_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_ENTRY_LINES_HELP );
  par_rec[ n ].value           = &custom_data -> entry_show_lines;
  par_rec[ n ].default_value   = &entry_lines;
  par_rec[ n ].size            = sizeof( custom_data -> entry_show_lines );

  /* Summary lines. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_SUMMARY_LINES_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_SUMMARY_LINES_HELP );
  par_rec[ n ].value           = &custom_data -> summary_def_lines;
  par_rec[ n ].default_value   = &summary_lines;
  par_rec[ n ].size            = sizeof( custom_data -> summary_def_lines );

  /* Separator. */
  n++;

  /* Confirm. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_CONFIRM_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_CONFIRM_HELP );
  par_rec[ n ].value           = &custom_data -> confirm_actions;
  par_rec[ n ].default_value   = &true_value;
  par_rec[ n ].size            = sizeof( custom_data -> confirm_actions );

  /* Do refresh. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_DO_REFRESH_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_DO_REFRESH_HELP );
  par_rec[ n ].value           = &custom_data -> do_refresh;
  par_rec[ n ].default_value   = &true_value;
  par_rec[ n ].size            = sizeof( custom_data -> do_refresh );

  /* Entries interactive. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_ENTRIES_INTER_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_ENTRIES_INTER_HELP );
  par_rec[ n ].value           = &custom_data -> entries_interactive;
  par_rec[ n ].default_value   = &true_value;
  par_rec[ n ].size            = sizeof( custom_data -> entries_interactive );

  /* Give alarms. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_GIVE_ALARM_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_GIVE_ALARM_HELP );
  par_rec[ n ].value           = &custom_data -> give_alarm;
  par_rec[ n ].default_value   = &false_value;
  par_rec[ n ].size            = sizeof( custom_data -> give_alarm );

  /* Pronpt repeat. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_PROMPT_REPEAT_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_PROMPT_REPEAT_HELP );
  par_rec[ n ].value           = &custom_data -> prompt_repeat_win;
  par_rec[ n ].default_value   = &true_value;
  par_rec[ n ].size            = sizeof( custom_data -> prompt_repeat_win );

  /* Separator. */
  n++;

  /* Calendar month extend. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_CAL_MONTH_EXTEND_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_CAL_MONTH_EXTEND_HELP );
  par_rec[ n ].value           = &custom_data -> cal_month_extend;
  par_rec[ n ].default_value   = &false_value;
  par_rec[ n ].size            = sizeof( custom_data -> cal_month_extend );

  /* Refresh rate. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_REFRESH_RATE_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_REFRESH_RATE_HELP );
  par_rec[ n ].value           = &custom_data -> refresh_rate;
  par_rec[ n ].default_value   = &refresh_rate;
  par_rec[ n ].size            = sizeof( custom_data -> refresh_rate );

  /* Standing in cal. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_STANDING_CAL_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_STANDING_CAL_HELP );
  par_rec[ n ].value           = &custom_data -> show_stand_in_cal;
  par_rec[ n ].default_value   = &false_value;
  par_rec[ n ].size            = sizeof( custom_data -> show_stand_in_cal );

  /* Summary startup. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_SUMMARY_START_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_SUMMARY_START_HELP );
  par_rec[ n ].value           = &custom_data -> show_day_to_do;
  par_rec[ n ].default_value   = &false_value;
  par_rec[ n ].size            = sizeof( custom_data -> show_day_to_do );

  /* Week numbers. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_WEEK_NUMBERS_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_WEEK_NUMBERS_HELP );
  par_rec[ n ].value           = &custom_data -> show_week_numbers;
  par_rec[ n ].default_value   = &true_value;
  par_rec[ n ].size            = sizeof( custom_data -> show_week_numbers );


  /* Create the parameter window? */
  if( appl_data_ref -> paramW == NULL )
    appl_data_ref -> paramW = xitCreateParameterSet( 
                                appl_data_ref -> customW,
                                0, &param_labels,
                                NULL, NULL );

  /* Display the window and set the parameters. */
  XtManageChild( appl_data_ref -> paramW );

  xitParameterSetDisplay( appl_data_ref -> paramW, 
                          &par_rec[ 0 ], XtNumber( par_rec ) );


  return;

} /* customParam */


/*----------------------------------------------------------------------*/

static void
  displayUsage()
{

  printf( 
    "\n"
    "%s (%s): XDiary user customization.\n" 
    "\n"
    "Allows you to customize your XDiary session. The customization data\n"
    "will be saved in the file XDiary.custom in your home directory.\n"
    "\n"
    "Usage:\n"
    "  %s [flags]\n"
    "\n"
    "Flags:\n"
    "  -fmap large      : Use a large font.\n"
    "  -fmap medium     : Use a medium font.\n"
    "  -fmap small      : Use a small font.\n"
    "  -help            : Display this help.\n"
    "  -h               : See above.\n"
    "  -language <lan>  : Use the language <lan>.\n"
    "  -lan <lan>       : See above.\n"
    "  -noFileLock      : Don't use any file locking.\n"
    "  -palette gray    : Use color palette Gray.\n"
    "  -palette lila    : Use color palette Lila.\n"
    "  -palette motif   : Use color palette Motif.\n"
    "  -palette neon    : Use color palette Neon.\n"
    "  -palette nordic  : Use color palette Nordic.\n"
    "  -palette red     : Use color palette Red.\n"
    "  -palette sea     : Use color palette Sea.\n"
    "  -palette sky     : Use color palette Sky.\n"
    "  -palette wood    : Use color palette Wood.\n"
    "  -usage           : Display this help.\n"
    "  -version         : Display the current version.\n"
    "\n",
    program_name, VERSION_ID, program_name );


  return;

} /* displayUsage */


/*----------------------------------------------------------------------*/

static void
  saveCustomData( XTM_CU_BASE_DATA_REF  appl_data_ref )
{

  /* Variables. */
  TIM_STATUS_TYPE     time_status;
  XTM_CB_STATUS       status;
  XTM_GL_CUSTOM_DATA  *custom_data_ref;


  /* Code. */

  custom_data_ref = appl_data_ref -> custom_data;

  /* Save the customized data. */
  status = xtmCbPutDataToFile( custom_data_ref );

  if( status != XTM_CB_OK ) {
    xitErMessage( appl_data_ref -> customW, XIT_ER_INFO,
                  module_name, "saveCustomData",
                  msgGetText( MXDI_CANNOT_SAVE_CUSTOM ) );
    return;
  }

  /* New date/time formats? */
  time_status = TimInitializeFormat( custom_data_ref -> date_format,
                                     custom_data_ref -> time_format );
  if( time_status != TIM_OK )
    xitErMessage( appl_data_ref -> customW, XIT_ER_ERROR, 
                  module_name, "saveCustomData",
                  msgGetText( MXDI_ERRMSG_DATE_OR_TIME_FORMAT ) );


  return;

} /* saveCustomData */


/*----------------------------------------------------------------------*/

static int
  startProcess( Widget  parent,
                char    *process_args[] )
{

  /* Variables. */
  int      pid;
  Display  *display = NULL;
  Widget   tempW;


  /* Code. */

  if( parent != NULL )
    display = XtDisplay( parent );

  /* Start the sub-process. */
  pid = fork();

  switch( pid ) {

    /* Child running. */
    case 0:
      if( display != NULL )
        close( ConnectionNumber( display ) );

      /* Start the process. */
      execvp( process_args[ 0 ], process_args );

      {
        char  buffer[ 200 ];

        sprintf( buffer, 
                 msgGetText( MXDI_ERRMSG_EXECUTE_PROCESS ),
                 process_args[ 0 ] );

        xitErMessage( NULL, XIT_ER_ERROR, 
                      module_name, "startProcess",
                      buffer );
      } /* block */

      exit( 1 );

    /* Error in fork. */
    case -1:
      xitErMessage( parent, XIT_ER_ERROR,
                    module_name, "startProcess",
                    msgGetText( MXDI_PROCESS_CANNOT_FORK ) );
      return( 0 );

    /* Parent */
    default:
      break;

  } /* switch */

  /* Tell the user what we are doing since this might take time. */
  if( parent != NULL ) {
    tempW = xitCreateInformationDialog(
              parent, "InformationDialog", 
              msgGetText( MXDI_INFORMATION_LABEL ),
              msgGetText( MXDI_STARTING_PROCESS_MSG ),
              NULL, NULL );

    XtAppAddTimeOut( XtWidgetToApplicationContext( parent ),
                     5 * 1000,
                     (XtTimerCallbackProc) XtDestroyWidget,
                     (XtPointer) tempW );

  } /* if */


  return( pid );

} /* startProcess */


/*----------------------------------------------------------------------*/

static void 
  calCustApplyCB( XTM_CC_REASON  reason,
                  XTM_CD_HANDLE  cal_db_handle,
                  void           *user_data )
{

  /* Variables. */
  XTM_CU_BASE_DATA_REF  appl_data_ref;


  /* Code. */

  appl_data_ref = (XTM_CU_BASE_DATA_REF) user_data;

  if( reason != XTM_CC_REASON_OK )
    return;


  /* Free the old calendar database. */
  if( appl_data_ref -> custom_data -> cal_db_handle != NULL )
    xtmCdFree( appl_data_ref -> custom_data -> cal_db_handle );

  /* Store the new calendar DB. */
  appl_data_ref -> custom_data -> cal_db_handle = xtmCdCopy( cal_db_handle );


  return;

} /* calCustApplyCB */


/*----------------------------------------------------------------------*/

static void 
  cancelCB( Widget                widget,
            XTM_CU_BASE_DATA_REF  appl_data_ref,
            XtPointer             call_data )
{

  /* Code. */

  /* Make sure this is wanted. */
  (void) xitCreateQuestionDialog( 
           appl_data_ref -> customW, "QuestionDialog", 
           msgGetText( MXDI_QUESTION_MESSAGE_LABEL ),
           msgGetText( MXDI_CUST_CONFIRM_CANCEL ),
           exitCB, NULL,
           NULL,   NULL );


  return;

} /* cancelCB */


/*----------------------------------------------------------------------*/

static void 
  doCustCB( Widget                     widget,
            XTM_CU_BASE_DATA_REF       appl_data_ref,
            XmRowColumnCallbackStruct  *call_data )
{

  /* Variables. */
  int                     index;
  int                     n;
  char                    **process_args;
  Display                 *display;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  display = XtDisplay( appl_data_ref -> customW );
  custom_data_ref = appl_data_ref -> custom_data;


  /* Select what to do. */
  switch( (int) call_data -> data ) {

    /* Calendar customization? */
    case 0:
      /* Initialize calendar window? */
      if( appl_data_ref -> cal_handle == NULL )
        appl_data_ref -> cal_handle = xtmCcInitialize(
                                        appl_data_ref,
                                        appl_data_ref -> customW,
                                        calCustApplyCB,
                                        (void *) appl_data_ref );

      xtmCcCustomCalendars( appl_data_ref -> cal_handle,
                            custom_data_ref -> cal_db_handle );
      break;


    /* Group customization? */
    case 1:
      /* Initialize group window? */
      if( appl_data_ref -> group_handle == NULL )
        appl_data_ref -> group_handle = xtmCgInitialize(
                                          appl_data_ref,
                                          appl_data_ref -> customW,
                                          groupCustApplyCB,
                                          (void *) appl_data_ref );

      xtmCgCustomGroups( appl_data_ref -> group_handle );
      break;


    /* Parameter customization? */
    case 2:
      customParam( appl_data_ref );
      break;


    /* Archive customization? */
    case 3:
      xtmCrEditArchiveData( appl_data_ref -> customW, appl_data_ref );
      break;


    /* Access customization? */
    case 4:
      process_args = (char**) XtMalloc( (custom_data_ref -> orig_argc + 10) *
                                        sizeof( char* ) );

      n = 0;  
      process_args[ n ] = custom_data_ref -> acl_script; n++;

      for( index = 1; index < custom_data_ref -> orig_argc; index++ ) {
        process_args[ n ] = custom_data_ref -> orig_argv[ index ]; n++;
      }
    
      process_args[ n ] = NULL;

      (void) startProcess( appl_data_ref -> customW, process_args );

      XtFree( (char*) process_args );
      break;

  } /* switch */


  return;

} /* doCustCB */


/*----------------------------------------------------------------------*/

static void 
  exitCB( Widget     widget,
          XtPointer  client_data,
          XtPointer  call_data )
{

  /* Code. */

  exit( 0 );

} /* exitCB */


/*----------------------------------------------------------------------*/

static void 
  groupCustApplyCB( XTM_CG_REASON  reason,
                    XTM_GD_HANDLE  new_group_db_handle,
                    void           *user_data )
{

  /* Variables. */
  XTM_CU_BASE_DATA_REF  appl_data_ref;


  /* Code. */

  appl_data_ref = (XTM_CU_BASE_DATA_REF) user_data;

  if( reason != XTM_CG_REASON_OK )
    return;

  /* Free the old group list and make a copy of the new. */
  xtmGdFree( appl_data_ref -> custom_data -> group_db_handle );

  appl_data_ref -> custom_data -> group_db_handle =
    xtmGdCopyGroups( new_group_db_handle );


  return;

} /* groupCustApplyCB */


/*----------------------------------------------------------------------*/

static void 
  infoCB( Widget                     widget,
          XTM_CU_BASE_DATA_REF       appl_data_ref,
          XmRowColumnCallbackStruct  *call_data )
{

  /* Code. */

  /* About window? */
  if( (int) call_data -> data == 6 ) {

    Widget  tempW;

    tempW = XtNameToWidget( appl_data_ref -> customW, "CustomTlBase" );
    xtmHlDisplayAboutWindow( tempW );

    return;
  }

  /* Use the standard help. */
  xtmHlDisplayHelp( appl_data_ref -> info_handle,
                    (int) call_data -> data,
                    custom_window_id, "" );

  return;

} /* infoCB */


/*----------------------------------------------------------------------*/

static void 
  okCB( Widget                widget,
        XTM_CU_BASE_DATA_REF  appl_data_ref,
        XtPointer             call_data )
{

  /* Code. */

  /* Save to disk. */
  saveCustomData( appl_data_ref );

  /* Tell the user we have saved the data. */
  (void) xitCreateInformationDialog( 
           appl_data_ref -> customW, "InformationDialog", 
           msgGetText( MXDI_INFORMATION_LABEL ),
           msgGetText( MXDI_CUST_HOW_TO_ACTIVATE ),
           exitCB, NULL );


  return;

} /* okCB */


/*----------------------------------------------------------------------*/

static void 
  saveCB( Widget                widget,
          XTM_CU_BASE_DATA_REF  appl_data_ref,
          XtPointer             call_data )
{

  /* Variables. */
  char  *char_ref;


  /* Code. */

  /* Save the custom data. */
  saveCustomData( appl_data_ref );


  /* Tell the user we have saved the data. */
  char_ref = (char *) SysMalloc( 
                        strlen( msgGetText( MXDI_CUST_DATA_SAVED ) ) +
                        strlen( appl_data_ref -> custom_data -> custom_file ) +
                        50 );

  sprintf( char_ref, msgGetText( MXDI_CUST_DATA_SAVED ),
           appl_data_ref -> custom_data -> custom_file );

  (void) xitCreateInformationDialog( 
           appl_data_ref -> customW, "InformationDialog", 
           msgGetText( MXDI_INFORMATION_LABEL ),
           char_ref,
           NULL, NULL );

  SysFree( char_ref );


  return;

} /* saveCB */
