/*----------------------------------------------------------------------------
--
--  Module:           xtmTools
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Miscellaneous tools for the Xdiary modules.
--
--  Filename:         xtmTools.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1990-12-16
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmTools.c, Version: 1.1, Date: 95/02/18 15:52:54";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/signal.h>

#include <X11/Intrinsic.h>

#include <Xm/Protocols.h>

#if ! defined( XD_HAS_NO_NLS ) && XtSpecificationRelease > 4
#  include <X11/Xlocale.h>
#endif

#include <Xm/Xm.h>

#include "System.h"
#include "Message.h"
#include "SigHandler.h"

#include "msgXdiary.h"
#include "xitError.h"
#include "xitTools.h"
#include "xmubCustUtil.h"
#include "xtmTools.h"
#include "xitTune.h"

/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

#define  WM_STATE_ELEMENTS   1


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmTools";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static unsigned long
checkWindowState( Display  *display,
                  Window   window );

static void
saveSessionCB( Widget     widget,
               Widget     toplevelW,
               XtPointer  call_data );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

void
xtmToInitializeResourceMapping( unsigned int  argc,
                                char**        argv,
                                Display*      display )
{

  String       app_class;
  String       app_name;
  XrmDatabase  display_db = XtDatabase( display );

  xmubResourceMappingInfoRef  res_info = NULL;

  XtGetApplicationNameAndClass( display, &app_name, &app_class );
  
  XrmParseCommand( &display_db,
                   xmub_options, xmub_num_options,
                   app_name, (int*) &argc, argv );
  
  xmubInitResourceMapping( display,
                           xmub_resource_sets, xmub_num_resource_sets,
                           &res_info );

  if( res_info == NULL )
    xitErMessage( NULL, XIT_ER_FATAL, 
                  "", "main",
                  "Cannot find resource mapping file.\n"
                  "Please check the XDiary installation." );

  xmubResFreeInfo( res_info );

  
  return;
  
} /* xtmToInitializeResourceMapping */


/*----------------------------------------------------------------------*/

void
xtmToPlayMelody( Widget  parentW,
                 char    *tune_player,
                 char    *tune_player_param,
                 char    *tune_file,
                 int     volume,
                 int     duration_delta )
{

  /* Code. */

  /* Anything to play? */
  if( tune_file == NULL || *tune_file == '\0' ) {
    XBell( XtDisplay( parentW ), 100 );

    return;
  }

  /* Do we have our own tune player? */
  if( tune_player != NULL && *tune_player != '\0' ) {

    int   index;
    char  *process_args[ 10 ];

    /* Process arguments. */
    index = 0;
    process_args[ index++ ] = tune_player;

    if( tune_player_param != NULL && *tune_player_param != '\0' )
      process_args[ index++ ] = tune_player_param;

    process_args[ index++ ] = tune_file;
    process_args[ index++ ] = NULL;

    (void) xtmToStartProcess( NULL, True, NULL, process_args );

    return;
  }

  /* Use the built-in speaker. */
  xitTuPlayTuneFromFile( XtDisplay( parentW ), 
                         XtWidgetToApplicationContext( parentW ),
                         tune_file, volume, duration_delta );

  return;

} /* xtmToPlayMelody */


/*----------------------------------------------------------------------*/

void
xtmToSessionMgmt( Widget        shellW,
                  unsigned int  argc,
                  char**        argv )
{

  /* Variables. */
  Atom      wm_save_yourself;
  Display   *display;


  /* Code. */

  display = XtDisplay( shellW );

  /* Set command parameters on the window. */
  if( argv != NULL )
    XSetCommand( XtDisplay( shellW ), XtWindow( shellW ), argv, (int) argc );

  
  /* Save session callback. */
  wm_save_yourself = XmInternAtom( display, "WM_SAVE_YOURSELF", False );

  XmAddWMProtocols( shellW, &wm_save_yourself, 1 );
  XmAddWMProtocolCallback( shellW, wm_save_yourself,
                           (XtCallbackProc) saveSessionCB,
                           (XtPointer) shellW );


  return;

} /* xtmToSessionMgmt */


/*----------------------------------------------------------------------*/

void
xtmToSetLocale( char  *program_name )
{

  /* Code. */

  /* Only for release >X11R5. */
#if ! defined( XD_HAS_NO_NLS ) && XtSpecificationRelease > 4

  if( setlocale( LC_ALL, "" ) == NULL ) {
    if( setlocale( LC_ALL, "C" ) == NULL ) {
      fprintf( stderr, "%s: Cannot set locale.\n", program_name );
      exit( 1 );
    }
  }

  if( ! XSupportsLocale() ) {
    fprintf( stderr, "%s: X does not support locale.\n", program_name );
    exit( 1 );
  }

  if( XSetLocaleModifiers( "" ) == NULL ) {
    fprintf( stderr, "%s: X cannot set locale modifiers.\n", program_name );
  }

#endif


  return;

} /* xtmToSetLocale */


/*----------------------------------------------------------------------*/

int
xtmToStartProcess( Widget   parent,
                   Boolean  remove_message,
                   char     *starting_message,
                   char     *process_args[] )
{

  /* Variables. */
  int      pid;
  Display  *display = NULL;
  Widget   tempW;


  /* Code. */

  if( parent != NULL )
    display = XtDisplay( parent );

  /* Start the process. */
  pid = fork();

  switch( pid ) {

    /* Child running. */
    case 0:
      if( display != NULL )
        close( ConnectionNumber( display ) );


      /* Start the process. */
      execvp( process_args[ 0 ], process_args );

      {
        char  buffer[ 200 ];

        sprintf( buffer, 
                 msgGetText( MXDI_ERRMSG_EXECUTE_PROCESS ),
                 process_args[ 0 ] );

        xitErMessage( NULL, XIT_ER_ERROR, 
                      module_name, "startProcess",
                      buffer );
      } /* block */

      exit( 1 );

    /* Error in fork. */
    case -1:
      xitErMessage( parent, XIT_ER_ERROR,
                    module_name, "xtmToStartProcess",
                    msgGetText( MXDI_PROCESS_CANNOT_FORK ) );
      return( 0 );

    /* Parent */
    default:
      break;

  } /* switch */


  /* Tell the user what we are doing since this might take time. */
  if( starting_message != NULL ) {
    tempW = xitCreateInformationDialog(
              parent, "InformationDialog", 
              msgGetText( MXDI_INFORMATION_LABEL ),
              starting_message,
              NULL, NULL );

    if( remove_message )
      xitInformationDialogTimeout( tempW, 4 );
  }

  /* Catch this process when we return. */
  (void) SigRegister( SIGCHLD, pid, NULL, NULL );


  return( pid );

} /* xtmToStartProcess */


/*----------------------------------------------------------------------*/

static unsigned long
checkWindowState( Display  *display,
                  Window   window )
{

  /* Variables. */
  int            actual_format;
  int            status;
  unsigned long  *property = NULL;
  unsigned long  nitems;
  unsigned long  leftover;
  Atom           actual_type;
  Atom           wm_state;


  /* Code. */

  wm_state = XInternAtom( display, "WM_STATE", False );

  status = XGetWindowProperty( display, window,
                               wm_state, 0L, WM_STATE_ELEMENTS, False,
                               wm_state,
                               &actual_type, &actual_format, &nitems,
                               &leftover, (unsigned char **) &property );

  if( ! ((status == Success) && 
         (actual_type = wm_state) && 
         (nitems == WM_STATE_ELEMENTS)) ) {
    if( property ) {
      XFree( (char *) property );
      return( 0L );
    }
  }

  if( property == NULL )
    return( 0 );
  else
    return( *property );

} /* checkWindowState */


/*----------------------------------------------------------------------*/

static void
saveSessionCB( Widget     widget,
               Widget     toplevelW,
               XtPointer  call_data )
{

  /* Variables. */
  int     argc;
  char**  argv;


  /* Code. */

  /* Just "touch" the WM_COMMAND property. */
  if( XGetCommand( XtDisplay( toplevelW ), XtWindow( toplevelW  ),
                   &argv, &argc ) != 0 ) {
    XSetCommand( XtDisplay( toplevelW ), XtWindow( toplevelW ), argv, argc );

    XFreeStringList( argv );
  }

  
  return;

} /* saveSessionCB */


