/*----------------------------------------------------------------------------
--
--  Module:           xtmShowToDo
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Show what we have to do (a summary of the day entries).
--
--  Filename:         xtmShowToDo.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-01-24
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmShowToDo.c, Version: 1.1, Date: 95/02/18 15:52:51";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>

#include <Xm/Protocols.h>

#include <Xm/Xm.h>
#include <Xm/CascadeB.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/PushB.h>
#include <Xm/RowColumn.h>
#include <Xm/ScrolledW.h>
#include <Xm/SeparatoG.h>
#include <Xm/Text.h>
#include <Xm/ToggleB.h>

#include "System.h"
#include "Message.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmCalDb.h"
#include "xtmDbFilter.h"
#include "xtmDbInclude.h"
#include "xtmDbMisc.h"
#include "xtmDbTools.h"
#include "xtmFields.h"
#include "xtmFormat.h"
#include "xtmHelp.h"
#include "xtmIcons.h"
#include "xtmPrEntries.h"
#include "xtmTools.h"
#include "xitError.h"
#include "xitFieldSel.h"
#include "xitTools.h"
#include "XmUbTimeB.h"
#include "xtmShowToDo.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

#define max( a, b )  (a < b ? b : a )

/* Internal summary flags. */
#define ENTRY_FLAGS     (1<<0)

/* Windows in the cache? */
#define  MAX_CACHE_ENTRIES   3


/* Local widgets in the show window. */
#define calFs              dataLocalW[  0 ]
#define calLa              dataLocalW[  1 ]
#define calRc              dataLocalW[  2 ]
#define menuBr             dataLocalW[  3 ]
#define rangeTb            dataLocalW[  4 ]
#define shDayRc            dataLocalW[  5 ]
#define shDoLa             dataLocalW[  6 ]
#define shLineLa           dataLocalW[  7 ]
#define shLineRc           dataLocalW[  8 ]
#define shLineTx           dataLocalW[  9 ]
#define shWhatLa           dataLocalW[ 10 ]
#define textLa             dataLocalW[ 11 ]
#define textSw    	   dataLocalW[ 12 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Record for summary. */
typedef struct {

  /* Flags to use. */
  UINT32  flags;

  /* The summary is for this database. */
  char  db_name[ XTM_GL_MAX_CAL_NAME + 1 ];

  /* Look for entries with these tags. */
  char  search_tags[ 200 ];

  /* The summary window. */
  Widget  showW;

  /* Include handle. */
  XTM_DI_HANDLE  include_handle;

  /* Filter record. */
  XTM_DM_FILTER_REC  filter_rec;

  /* Filter handle. */
  XTM_FI_HANDLE  filter_handle;

  /* Application wide data. */
  XTM_GL_BASE_DATA_REF  appl_data_ref;

  /* Reference to printer window. */
  XTM_PR_HANDLE  pr_handle;

  /* Callback to inform our creator of specific actions. */
  void              *user_data;
  XTM_SD_ACTION_CB  actionCB;

} SHOW_REC, *SHOW_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmShowToDo";

/* IDs for the help windows. */
static char  *sum_window_id = "Summary";

/* Cache entries. */
static Boolean       init_cache = True;
static SHOW_REC_REF  cache_entries[ MAX_CACHE_ENTRIES ];


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  applyCB( Widget        widget,
           SHOW_REC_REF  show_ref,
           XtPointer     call_data );

static void 
  calChangedCB( Widget        widget,
                SHOW_REC_REF  show_ref,
                XtPointer     call_data );

static void
  closeCB( Widget        widget,
           SHOW_REC_REF  show_ref,
           XtPointer     call_data );

static Widget
  createShowWindow( SHOW_REC_REF  show_ref,
                    Widget        parent );

static void 
  ctrlMenuCB( Widget                     widget,
              SHOW_REC_REF               show_ref,
              XmRowColumnCallbackStruct  *call_data );

static void 
  destroyCB( Widget        widget,
             SHOW_REC_REF  show_ref,
             XtPointer     call_data );

static void 
  filterApplyCB( XTM_FI_REASON      reason,
                 XTM_DM_FILTER_REC  *filter_ref,
                 void               *user_data );

static void 
  filterCB( Widget        widget,
            SHOW_REC_REF  show_ref,
            XtPointer     call_data );

static void
  includeDbApplyCB( XTM_DI_REASON         reason,
                    XTM_CD_INCL_CALS_REF  new_db_incl_ref,
                    void                  *user_data );

static void 
  infoCB( Widget                     widget,
          SHOW_REC_REF               show_ref,
          XmRowColumnCallbackStruct  *call_data );

void
  mapCtrlMenuCB( Widget        widget,
                 SHOW_REC_REF  show_ref,
                 XtPointer     call_data );

static void
  printCB( Widget        widget,
           SHOW_REC_REF  show_ref,
           XtPointer     call_data );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XTM_SD_HANDLE
  xtmSdInitialize( XTM_GL_BASE_DATA_REF  appl_data_ref,
                   Widget                toplevel,
                   char                  *db_name,
                   XTM_SD_ACTION_CB      actionCB,
                   void                  *user_data )
{

  /* Variables. */
  Boolean                 ok;
  int                     index;
  int                     length;
  SHOW_REC_REF            show_ref;
  XTM_CD_CAL_INFO         db_info;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = appl_data_ref -> custom_data;


  /* Fetch database information. */
  ok = xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle,
                          db_name,
                          &db_info, NULL );
  if( ! ok )
    return( False );


  /* Minium permissions are read. */
  if( ! flagIsSet( db_info.operations, XTM_DB_FLAG_MODE_READ ) )
    return( NULL );


  /* Initialize the cache? */
  if( init_cache ) {
    for( index = 0; index < MAX_CACHE_ENTRIES; index++ )
      cache_entries[ index ] = NULL;

    init_cache = False;
  }


  /* Anything in the cache? */
  show_ref = NULL;

  for( index = 0; index < MAX_CACHE_ENTRIES; index++ ) {
    if( cache_entries[ index ] != NULL ) {
      show_ref = cache_entries[ index ];
      cache_entries[ index ] = NULL;

      break;
    }
  }

  /* Create the showrecord? */
  if( show_ref == NULL ) {

    /* Create and initialize our private data. */
    show_ref = SysNew( SHOW_REC );
    if( show_ref == NULL )
      return( NULL );

    show_ref -> showW = NULL;

  } /* if */

  show_ref -> appl_data_ref    = appl_data_ref;
  show_ref -> search_tags[ 0 ] = '\0';
  show_ref -> include_handle   = NULL;
  show_ref -> filter_handle    = NULL;
  show_ref -> pr_handle        = NULL;
  show_ref -> actionCB         = actionCB;
  show_ref -> user_data        = user_data;

  show_ref -> filter_rec.flags            = XTM_DM_FLAG_ALL;
  show_ref -> filter_rec.text_string[ 0 ] = '\0';
  show_ref -> filter_rec.tag_string[  0 ] = '\0';

  length = strlen( custom_data_ref -> entry_tags_filter );

  if( length > 0 && length < XTM_DM_MAX_TAG_SEARCH ) {
    strcpy( show_ref -> filter_rec.tag_string,
            custom_data_ref -> entry_tags_filter );
    flagSet( show_ref -> filter_rec.flags, XTM_DM_FLAG_SEARCH_TAG );
  }

  if( custom_data_ref -> display_entry_flags )
    flagSet( show_ref -> flags, ENTRY_FLAGS );

  if( strlen( custom_data_ref -> entry_tags_menu ) < 200 )
    strcpy( show_ref -> search_tags, custom_data_ref -> entry_tags_menu );


  /* Create a shadow calendar database. */
  ok = xtmCdCreateShadowEntry( custom_data_ref -> cal_db_handle,
                               db_name, show_ref -> db_name );
  if( ! ok ) {
    SysFree( show_ref );

    return( NULL );
  }


  /* Create the summary window. */
  if( show_ref -> showW == NULL ) {
    show_ref -> showW = createShowWindow( show_ref, toplevel );

    if( show_ref -> showW == NULL ) {
      xtmCdFreeShadowEntry( custom_data_ref -> cal_db_handle,
                            show_ref -> db_name );
      SysFree( show_ref );

      return( NULL );
    }
  }


  return( (XTM_SD_HANDLE) show_ref );

} /* xtmSdInitialize */


/*----------------------------------------------------------------------*/

void
  xtmSdDestroy( XTM_SD_HANDLE  show_handle )
{

  /* Variables. */
  SHOW_REC_REF  show_ref;


  /* Code. */

  if( show_handle == NULL )
    return;

  /* Our private data. */
  show_ref = (SHOW_REC_REF) show_handle;

  /* Destroy the window. */
  closeCB( NULL, show_ref, NULL );


  return;

} /* xtmSdDestroy */


/*----------------------------------------------------------------------*/

void 
  xtmSdDisplaySummary( XTM_SD_HANDLE  show_handle,
                       TIM_TIME_REF   from_date,
                       TIM_TIME_REF   to_date,
                       char           *search_tags )
{

  /* Variables. */
  Widget                  mainW;
  Widget                  tempW;
  SHOW_REC_REF            show_ref;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         db_info;


  /* Code. */

  show_ref        = (SHOW_REC_REF) show_handle;
  appl_data_ref   = show_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;


  mainW = XtNameToWidget( show_ref -> showW, "ShowTlBase.ShowTlFo" );


  /* Calendar we are using. */
  (void) xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle,
                            show_ref -> db_name,
                            &db_info, NULL );


  /* The fields containing from and to date. */
  tempW = XtNameToWidget( mainW, "ShDayRc.RangeTb" );

  XmUbTimeBoxSetStartDate( tempW, from_date );
  XmUbTimeBoxSetEndDate( tempW, to_date );


  /* Calendar to use. */
  tempW = XtNameToWidget( mainW, "CalRc.CalFs" );
  xitFieldSelectSetCurrent( tempW, db_info.short_name, False );


  /* Do we search special tags? */
  if( strlen( search_tags ) > 0 )
    strcpy( show_ref -> filter_rec.tag_string, search_tags );


  /* Display the current summary. */
  tempW = XtNameToWidget( mainW, "ActionFo.ApplyPb" );
  applyCB( tempW, show_ref, NULL );


  /* Display the window. */
  XtPopup( show_ref -> showW, XtGrabNone );


  /* Make sure the editor window is visable. */
  XRaiseWindow( XtDisplay( show_ref -> showW ), 
                XtWindow(  show_ref -> showW ) );

  XtMapWidget( show_ref -> showW );


  return;

} /* xtmSdDisplaySummary */


/*----------------------------------------------------------------------*/

void
  xtmSdEmptyCache()
{

  /* Variables. */
  int  index;


  /* Code. */

  if( ! init_cache )
    return;

  for( index = 0; index < MAX_CACHE_ENTRIES; index++ ) {
    if( cache_entries[ index ] != NULL )
      XtDestroyWidget( cache_entries[ index ] -> showW );

    cache_entries[ index ] = NULL;
  }


  return;

} /* xtmSdEmptyCache */


/*----------------------------------------------------------------------*/

static Widget
  createShowWindow( SHOW_REC_REF  show_ref,
                    Widget        parent )
{

  /* Variables. */
  int                     index;
  char                    buffer[ 50 ];
  Arg                     args[ 10 ];
  Cardinal                n;
  Widget                  dataLocalW[ 13 ];
  Widget                  menuCasc[ 3 ];
  Widget                  menuCtrlBu[ 4 ];
  Widget                  menuFileBu[ 1 ];
  Widget                  menuHelpBu[ 6 ];
  Widget                  pulldownMenu[ 3 ];
  Widget                  showTl;
  Widget                  toplevel;
  Widget                  workFo;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;

  static char  *pull_downs[] = { "pdown1", "pdown2", "pdown3" };

  static XIT_TEXT_STRUCT text_field_def[] = {
    { "ShLineTx", NULL, 1, True },
  };

  static XIT_CASCADE_STRUCT menupane[] = {
    { "", "", "FilePane" },
    { "", "", "ControlPane" },
    { "", "", "HelpPane" },
  };

  static XIT_MENU_BUTTON_STRUCT file_casc[] = {
    { "", "", NULL, "CloseBu", True, False, False },
  };

  static XIT_MENU_BUTTON_STRUCT ctrl_casc[] = {
    { "",      "",  NULL, "EntryFlagsBu", True,  True,  False },
    { XIT_SEP, " ", NULL, "",             False, False, False },
    { "",      "",  NULL, "FilterBu",     True,  False, False },
    { "",      "",  NULL, "IncludeBu",    True,  False, False },
  };

  static XIT_MENU_BUTTON_STRUCT help_casc[] = {
    { "", "", NULL, "ContextBu", True, False, False },
    { "", "", NULL, "WindowsBu", True, False, False },
    { "", "", NULL, "KeysBu",    True, False, False },
    { "", "", NULL, "IndexBu",   True, False, False },
    { "", "", NULL, "HelpBu",    True, False, False },
    { "", "", NULL, "VersionBu", True, False, False },
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "", applyCB,  NULL },
    { "", printCB,  NULL },
    { "", filterCB, NULL },
    { "", closeCB,  NULL },
  };


  /* Code. */

  appl_data_ref   = show_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;


  /* Get text for menues and buttons. */
  action_buttons[ 0 ].label = msgGetText( MXDI_APPLY_BUTTON );
  action_buttons[ 0 ].data  = show_ref;
  action_buttons[ 1 ].label = msgGetText( MXDI_PRINT_MENU );
  action_buttons[ 1 ].data  = show_ref;
  action_buttons[ 2 ].label = msgGetText( MXDI_FILTER_MENU );
  action_buttons[ 2 ].data  = show_ref;
  action_buttons[ 3 ].label = msgGetText( MXDI_CLOSE_BUTTON );
  action_buttons[ 3 ].data  = show_ref;

  menupane[ 0 ].title    = msgGetText( MXDI_FILE_MENU );
  menupane[ 0 ].mnemonic = msgGetText( MXDI_FILE_MENU_ACC );
  menupane[ 1 ].title    = msgGetText( MXDI_CTRL_MENU );
  menupane[ 1 ].mnemonic = msgGetText( MXDI_CTRL_MENU_ACC );
  menupane[ 2 ].title    = msgGetText( MXDI_HELP_MENU );
  menupane[ 2 ].mnemonic = msgGetText( MXDI_HELP_MENU_ACC );

  file_casc[ 0 ].title    = msgGetText( MXDI_CLOSE_MENU );
  file_casc[ 0 ].mnemonic = msgGetText( MXDI_CLOSE_MENU_ACC );

  ctrl_casc[ 0 ].title    = msgGetText( MXDI_FLAGS_DISPLAY_MENU );
  ctrl_casc[ 0 ].mnemonic = msgGetText( MXDI_FLAGS_DISPLAY_MENU_ACC );
  ctrl_casc[ 2 ].title    = msgGetText( MXDI_FILTER_MENU );
  ctrl_casc[ 2 ].mnemonic = msgGetText( MXDI_FILTER_MENU_ACC );
  ctrl_casc[ 3 ].title    = msgGetText( MXDI_INCLUDE_MENU );
  ctrl_casc[ 3 ].mnemonic = msgGetText( MXDI_INCLUDE_MENU_ACC );

  help_casc[ 0 ].title    = msgGetText( MXDI_HELP_CONTEXT );
  help_casc[ 0 ].mnemonic = msgGetText( MXDI_HELP_CONTEXT_ACC );
  help_casc[ 1 ].title    = msgGetText( MXDI_HELP_WINDOWS );
  help_casc[ 1 ].mnemonic = msgGetText( MXDI_HELP_WINDOWS_ACC );
  help_casc[ 2 ].title    = msgGetText( MXDI_HELP_KEYS );
  help_casc[ 2 ].mnemonic = msgGetText( MXDI_HELP_KEYS_ACC );
  help_casc[ 3 ].title    = msgGetText( MXDI_HELP_INDEX );
  help_casc[ 3 ].mnemonic = msgGetText( MXDI_HELP_INDEX_ACC );
  help_casc[ 4 ].title    = msgGetText( MXDI_HELP_HELP );
  help_casc[ 4 ].mnemonic = msgGetText( MXDI_HELP_HELP_ACC );
  help_casc[ 5 ].title    = msgGetText( MXDI_HELP_VERSION );
  help_casc[ 5 ].mnemonic = msgGetText( MXDI_HELP_VERSION_ACC );


  /* Create a separate window. */
  toplevel = xitGetToplevelWidget( parent );

  showTl = xitCreateToplevelDialog( toplevel, "ShowTl",
                                    1, 0,
                                    action_buttons, 
                                    XtNumber( action_buttons ) );

  XtAddCallback( showTl, XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) show_ref );


  /* Close the window if this window is deleted. */
  {
    Atom  wm_delete_window;

    wm_delete_window = XmInternAtom( XtDisplay( showTl ),
                                     "WM_DELETE_WINDOW", False );

    XmAddWMProtocols( showTl,  &wm_delete_window, 1 );
    XmAddWMProtocolCallback( showTl,  wm_delete_window, 
                             (XtCallbackProc) closeCB, (XtPointer) show_ref );
  } /* block */


  /* Form to enclose the show window. */
  workFo = XtNameToWidget( showTl, "ShowTlBase.ShowTlFo" );


  /* Create the menubar and menu cascades. */
  menuBr = XmCreateMenuBar( workFo, "MenuBr", args, 0 );

  n = 0;
  for( index = 0; index < XtNumber( pulldownMenu ); index++ )
    pulldownMenu[ index ] = XmCreatePulldownMenu( menuBr, 
                                                  pull_downs[ index ], 
                                                  NULL, n );

  for( index = 0; index < XtNumber( menuCasc ); index++ )
    menuCasc[ index ] = xitCreateCascadeButton( menuBr, 
                                                pulldownMenu[ index ], 
                                                &menupane[ index ] );

  /* The help button should be placed to the right. */
  index = XtNumber( menuCasc ) - 1;
  n     = 0;
  XtSetArg( args[ n ], XmNmenuHelpWidget, menuCasc[ index ] ); n++;
  XtSetValues( menuBr, args, n );


  /* Create the file menu. */
  for( index = 0; index < XtNumber( menuFileBu ); index++ )
    menuFileBu[ index ] = xitCreateMenuPushButton( pulldownMenu[ 0 ], 
                                                   &file_casc[ index ] );

  XtAddCallback( menuFileBu[ 0 ], XmNactivateCallback, 
                 (XtCallbackProc) closeCB, (XtPointer) show_ref );


  /* Create the control menu. */
  XtAddCallback( pulldownMenu[ 1 ], XmNentryCallback, 
                 (XtCallbackProc) ctrlMenuCB, (XtPointer) show_ref );
  XtAddCallback( pulldownMenu[ 1 ], XmNmapCallback, 
                 (XtCallbackProc) mapCtrlMenuCB, (XtPointer) show_ref );

  for( index = 0; index < XtNumber( menuCtrlBu ); index++ ) {
    menuCtrlBu[ index ] = xitCreateMenuPushButton( pulldownMenu[ 1 ], 
                                                   &ctrl_casc[ index ] );

    if( XmIsPushButton( menuCtrlBu[ index ] ) )
      XtAddCallback( menuCtrlBu[ index ], XmNactivateCallback, 
                     (XtCallbackProc) ctrlMenuCB, (XtPointer) index );
    else if( XmIsToggleButton( menuCtrlBu[ index ] ) )
      XtAddCallback( menuCtrlBu[ index ], XmNvalueChangedCallback, 
                     (XtCallbackProc) ctrlMenuCB, (XtPointer) index );
  }


  /* Create the help menu. */
  XtAddCallback( pulldownMenu[ 2 ], XmNentryCallback, 
                 (XtCallbackProc) infoCB, (XtPointer) show_ref );

  for( index = 0; index < XtNumber( menuHelpBu ); index++ ) {
    menuHelpBu[ index ] = xitCreateMenuPushButton( pulldownMenu[ 2 ], 
                                                   &help_casc[ index ] );

    XtAddCallback( menuHelpBu[ index ], XmNactivateCallback, 
                   (XtCallbackProc) infoCB, (XtPointer) index );
  }

  /* We can't do context sensitive help. */
  XtSetSensitive( menuHelpBu[ 0 ], False );


  /* Label to do. */
  shDoLa = xitCreateLabel( workFo, "ShDoLa", 
                           msgGetText( MXDI_SUMMARY_LABEL ), 
                           XmALIGNMENT_BEGINNING );


  /* Create lablel to hold the text (in a scrolled win). */
  n = 0;
  XtSetArg( args[ n ], XmNscrollingPolicy,        XmAUTOMATIC ); n++;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmAS_NEEDED ); n++;
  textSw = XmCreateScrolledWindow( workFo, "TextSw", args, n );

  textLa = xitCreateLabelWidget( textSw, "TextLa", " ", -1 );

  n = 0;
  XtSetArg( args[ n ], XmNworkWindow, textLa ); n++;
  XtSetValues( textSw, args, n );


  /* Calendar selection. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  calRc = XmCreateRowColumn( workFo, "CalRc", args, n );

  calLa = xitCreateLabel( calRc, "CalLa", 
                          msgGetText( MXDI_DATABASE_LABEL ), -1 );

  calFs = xtmFlCreateDbField( calRc, "CalFs",
                              custom_data_ref -> cal_db_handle,
                              True,
                              (XtCallbackProc) calChangedCB,
                              (void *) show_ref );


  /* Create a show date form. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  XtSetArg( args[ n ], XmNpacking,     XmPACK_TIGHT ); n++;
  shDayRc = XmCreateRowColumn( workFo, "ShDayRc", args, n );

  shWhatLa = xitCreateLabel( shDayRc, "ShWhatLa", 
                             msgGetText( MXDI_BETWEEN_DATES ), -1 );

  /* Date range selector. */
  rangeTb = xtmFlCreateRangeField( shDayRc, "RangeTb" );


  /* Lines to display. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  shLineRc = XmCreateRowColumn( workFo, "ShLineRc", args, n );

  shLineLa = xitCreateLabel( shLineRc, "ShLineLa", 
                             msgGetText( MXDI_SUMMARY_LINES ), -1 );

  shLineTx = xitCreateTextCols( shLineRc, &text_field_def[ 0 ], 5 );

  n = 0;
  XtSetArg( args[ n ], XmNmaxLength, 5 );  n++;
  XtSetValues( shLineTx, args, n );  


  /* Put the Parts together. */
  xitAttachWidget( menuBr, 
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_FORM, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( shDoLa,
                   XmATTACH_WIDGET, menuBr, XmATTACH_FORM, NULL,
                   XmATTACH_FORM,   NULL,   XmATTACH_NONE, NULL );
  xitAttachWidget( textSw, 
                   XmATTACH_WIDGET, shDoLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,   XmATTACH_NONE, NULL );
  xitAttachWidget( calRc,
                   XmATTACH_WIDGET, textSw, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,   XmATTACH_NONE, NULL );
  xitAttachWidget( shDayRc,
                   XmATTACH_WIDGET, calRc, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,  XmATTACH_NONE, NULL );
  xitAttachWidget( shLineRc,
                   XmATTACH_WIDGET, shDayRc, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,    XmATTACH_NONE, NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( shDoLa,   args, n );
  XtSetValues( textSw,   args, n );
  XtSetValues( shDayRc,  args, n );
  XtSetValues( shLineRc, args, n );
  XtSetValues( calRc,    args, n );


  /* Manage all the children. */
  XtManageChildren( menuCasc,     XtNumber( menuCasc ) );
  XtManageChildren( menuCtrlBu,   XtNumber( menuCtrlBu ) );
  XtManageChildren( menuFileBu,   XtNumber( menuFileBu ) );
  XtManageChildren( menuHelpBu,   XtNumber( menuHelpBu ) );
  XtManageChildren( pulldownMenu, XtNumber( pulldownMenu ) );

  xitManageChildren( dataLocalW,  XtNumber( dataLocalW ) );

  XtManageChild( showTl );


  /* Set icon for this window. */
  xtmIcSetSimpleIcon( showTl, workFo, XTM_IC_ICON_SUMMARY );

  /* Set the initial sizes. */
  xitSetSizeToplevelDialog( showTl, True );


  /* Make the final attachments. */
  xitAttachWidget( shLineRc,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL );
  xitAttachWidget( shDayRc,
                   XmATTACH_NONE, NULL, XmATTACH_FORM,   NULL,
                   XmATTACH_NONE, NULL, XmATTACH_WIDGET, shLineRc );
  xitAttachWidget( calRc,
                   XmATTACH_NONE, NULL, XmATTACH_FORM,   NULL,
                   XmATTACH_NONE, NULL, XmATTACH_WIDGET, shDayRc );
  xitAttachWidget( textSw, 
                   XmATTACH_WIDGET, shDoLa, XmATTACH_FORM,   NULL,
                   XmATTACH_FORM,   NULL,   XmATTACH_WIDGET, calRc );

  /* Number of lines to display. */
  sprintf( buffer, "%d", custom_data_ref -> summary_def_lines );
  XmTextSetString( shLineTx, buffer );


  return( showTl );

} /* createShowWindow */


/*----------------------------------------------------------------------*/

static void
  applyCB( Widget        widget,
           SHOW_REC_REF  show_ref,
           XtPointer     call_data )
{

  /* Variables. */
  int                     SEGMENT_SIZE = 10000;
  UINT32                  flags;
  Boolean                 ok;
  int                     display_lines;
  int                     items;
  int                     text_size;
  char                    buffer[ 100 ];
  char                    *char_ref;
  char                    *text_ref;
  Arg                     args[ 10 ];
  Cardinal                n;
  Widget                  mainW;
  Widget                  tempW;
  LST_DESC_TYPE           entries;
  LST_STATUS              lst_status;
  TIM_DELTA_TYPE          delta;
  TIM_TIME_REF            from_date;
  TIM_TIME_REF            last_date = 0;
  TIM_TIME_REF            to_date;
  XTM_DB_STATUS           db_status;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         db_info;


  /* Code. */

  appl_data_ref   = show_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;

  mainW = XtNameToWidget( show_ref -> showW, "ShowTlBase.ShowTlFo" );


  /* Calendar we are using. */
  (void) xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle,
                            show_ref -> db_name,
                            &db_info, NULL );


  /* Fetch the date range. */
  tempW = XtNameToWidget( mainW, "ShDayRc.RangeTb" );

  ok = xtmFoFetchDate( mainW, tempW, XTM_FO_START_DATE, False, &from_date );
  if( ! ok )
    return;

  ok = xtmFoFetchDate( mainW, tempW, XTM_FO_END_DATE, False, &to_date );
  if( ! ok )
    return;

  if( from_date > to_date ) {
    xitErMessage( show_ref -> showW, XIT_ER_ERROR, 
                  module_name, "applyCB",
                  msgGetText( MXDI_INVALID_DATE_RANGE ) );
    return;
  }


  /* Number of lines to display? */
  tempW    = XtNameToWidget( mainW, "ShLineRc.ShLineTx" );
  char_ref = xitStringGetText( tempW );

  items = sscanf( char_ref, "%d", &display_lines );
  if( items != 1 || display_lines < 0 )
    display_lines = 10000;

  SysFree( char_ref );


  /* How many days can we display? */
  (void) TimDelta( from_date, to_date, &delta );

  if( delta.days > 366 ) {
    xitErMessage( show_ref -> showW, XIT_ER_ERROR, 
                  module_name, "applyCB",
                  msgGetText( MXDI_PRINT_SHOW_TO_MANY_DAYS ) );
    return;
  }


  /* The title contains the current calendar. */
  sprintf( buffer, "%s %s",
           db_info.short_name, msgGetText( MXDI_TODO_TITLE ) );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, buffer ); n++;
  XtSetValues( show_ref -> showW, args, n );

  sprintf( buffer, "%s", db_info.short_name );

  n = 0;
  XtSetArg( args[ n ], XmNiconName, buffer ); n++;
  XtSetValues( show_ref -> showW, args, n );


  /* Create the list of entries to display. */
  flags = XTM_DB_FETCH_ALL_TEXT;

  db_status = xtmDmFetchFilteredEntriesRange( 
                appl_data_ref,
                show_ref      -> db_name,
                appl_data_ref -> context,
                show_ref      -> showW,
                from_date,
                to_date,
                flags,
                &show_ref -> filter_rec,
                &entries );

  if( db_status != XTM_DB_OK ) {
    xtmDmDeleteEntriesList( entries );
    return;
  }


  /* Initialize data. */
  text_size = SEGMENT_SIZE;
  text_ref  = SysMalloc( SEGMENT_SIZE );
  *text_ref = '\0';

  lst_status = LstLinkCurrentFirst( entries );

  /* Create the summary text. */
  while( lst_status == LST_OK ) {

    int                   append_size;
    int                   extend_size;
    char                  buffer[ 100 ];
    char                  entry_buffer[ 1000 ];
    char                  *entry_text;
    char                  *tmp_ref;
    XTM_DB_ALL_ENTRY_DEF  entry_record;

    lst_status = LstLinkGetCurrent( entries, &entry_record );

    /* Skip this entry? */
    if( flagIsSet( entry_record.entry.flags, XTM_DB_FLAG_HIDE_IN_SUMMARY ) ) {
      lst_status = LstLinkCurrentNext( entries );
      continue;
    }

    /* Start to build the entry. */
    strcpy( entry_buffer, "\n" );

    /* New date? */
    if( entry_record.entry.date_stamp != last_date ) {
      TimFormatStrTime( entry_record.entry.date_stamp, "%d %B %Y, %A\n \n", 
                        buffer, sizeof( buffer ) );
      if( buffer[ 0 ] == '0' )
        buffer[ 0 ] = ' ';

      strcat( entry_buffer, "-------------------------------------------" );
      strcat( entry_buffer, "-------------------------------------------\n" );
      strcat( entry_buffer, buffer );

      last_date = entry_record.entry.date_stamp;
    }

    /* Time for the entry. */
    if( entry_record.entry.entry_type != XTM_DB_DAY_NOTE ) {
      xtmFoFormatEntryTimes( &entry_record, buffer, sizeof( buffer ) );

      strcat( entry_buffer, buffer );
      strcat( entry_buffer, " " );
    }

    /* Flags for the entry. */
    xtmFoFormatEntryFlags( appl_data_ref -> custom_data, &entry_record,
                           buffer, sizeof( buffer ) );

    if( strlen( buffer ) > 0 &&
        flagIsSet( show_ref -> flags, ENTRY_FLAGS ) ) {
      strcat( entry_buffer, buffer );
      strcat( entry_buffer, " " );
    }

    /* Included database? */
    if( flagIsSet( entry_record.entry.flags, XTM_DB_FLAG_INCLUDE ) ) {
      sprintf( buffer, "(%s) ", entry_record.db_name );
      strcat( entry_buffer, buffer );
    }

    if( entry_record.entry.entry_type != XTM_DB_DAY_NOTE )
      strcat( entry_buffer, "\n" );

    /* Format entry text. */
    if( entry_record.all_text != NULL )
      tmp_ref = entry_record.all_text;
    else
      tmp_ref = entry_record.entry.text;

    if( entry_record.entry.entry_type == XTM_DB_DAY_NOTE )
      entry_text = xtmFoFormatText( tmp_ref, 0, display_lines, 80 );
    else
      entry_text = xtmFoFormatText( tmp_ref, 2, display_lines, 80 );

    /* Append entry information and text. */
    append_size = strlen( entry_buffer );

    if( entry_text != NULL )
      append_size = append_size + strlen( entry_text );

    /* Allocate space. */
    append_size = append_size + 10;

    if( strlen( text_ref ) + append_size > text_size ) {
      extend_size = max( append_size, SEGMENT_SIZE );

      text_size = text_size + extend_size;
      text_ref  = SysRealloc( text_ref, text_size );
    }

    /* Append the entry information and entry text. */
    strcat( text_ref, entry_buffer );

    if( entry_text != NULL ) {
      strcat( text_ref, entry_text );

      SysFree( entry_text );
    }

    strcat( text_ref, "\n " );

    /* Fetch the next entry. */
    lst_status = LstLinkCurrentNext( entries );

  } /* while */


  /* Set the text. */
  tempW = XtNameToWidget( mainW, "TextSw.ClipWindow.TextLa" );

  xitStringSetLabel( tempW, text_ref );


  /* Clear allocated data. */
  xtmDmDeleteEntriesList( entries );

  SysFree( text_ref );


  return;

} /* applyCB */


/*----------------------------------------------------------------------*/

static void 
  calChangedCB( Widget        widget,
                SHOW_REC_REF  show_ref,
                XtPointer     call_data )
{

  /* Variables. */
  char                    *cal_name;
  Widget                  mainW;
  Widget                  tempW;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         db_info;


  /* Code. */

  custom_data_ref = show_ref -> appl_data_ref -> custom_data;
  mainW = XtNameToWidget( show_ref -> showW, "ShowTlBase.ShowTlFo" );


  /* Calendar we are using. */
  (void) xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle,
                            show_ref -> db_name,
                            &db_info, NULL );


  /* Fetch the new database. */
  tempW = XtNameToWidget( mainW, "CalRc.CalFs" );
  xitFieldSelectGetCurrent( tempW, &cal_name );


  /* If the calendar is the same as before, just return. */
  if( strcmp( cal_name, db_info.short_name ) == 0 ) {
    SysFree( cal_name );
    return;
  }


  /* Remove the shadow entry and make a new one. */
  xtmCdFreeShadowEntry( custom_data_ref -> cal_db_handle,
                        show_ref -> db_name );

  (void) xtmCdCreateShadowEntry( custom_data_ref -> cal_db_handle,
                                 cal_name, show_ref -> db_name );

  SysFree( cal_name );


  return;

} /* calChangedCB */


/*----------------------------------------------------------------------*/

static void
  closeCB( Widget        widget,
           SHOW_REC_REF  show_ref,
           XtPointer     call_data )
{

  /* Variables. */
  int  index;


  /* Code. */

  /* Keep the window in cache? */
  for( index = 0; index < MAX_CACHE_ENTRIES; index++ ) {
    if( cache_entries[ index ] == NULL ) {
      cache_entries[ index ] = show_ref;

      destroyCB( NULL, show_ref, NULL );
      XtPopdown( show_ref -> showW );

      return;
    }
  }

  /* Window not kept in cache, really remove it. */
  XtDestroyWidget( show_ref -> showW );


  return;

} /* closeCB */


/*----------------------------------------------------------------------*/

static void 
  ctrlMenuCB( Widget                     widget,
              SHOW_REC_REF               show_ref,
              XmRowColumnCallbackStruct  *call_data )
{

  /* Variables. */
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = show_ref -> appl_data_ref -> custom_data;

  /* Select what to do. */
  switch( (int) call_data -> data ) {

    /* Display entry flags? */
    case 0:
      if( XmToggleButtonGetState( call_data -> widget ) )
        flagSet( show_ref -> flags, ENTRY_FLAGS );
      else
        flagClear( show_ref -> flags, ENTRY_FLAGS );

      /* Display again... */
      applyCB( show_ref -> showW, show_ref, NULL );
      break;


    /* Filter window. */
    case 2:
      filterCB( widget, show_ref, NULL );
      break;


    /* Include calendars. */
    case 3:
      if( show_ref -> include_handle == NULL )
        show_ref -> include_handle = xtmDiInitialize(
                                       show_ref -> appl_data_ref,
                                       show_ref -> showW,
                                       False,
                                       includeDbApplyCB, (void *) show_ref );

      /* Display the include window. */
      xtmDiDisplayIncludeWindow( show_ref -> include_handle,
                                 show_ref -> db_name );
      break;

  } /* switch */


  return;

} /* ctrlMenuCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget        widget,
             SHOW_REC_REF  show_ref,
             XtPointer     call_data )
{

  /* Variables. */
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = show_ref -> appl_data_ref -> custom_data;


  /* Do we have a user action callback registered? */
  if( show_ref -> actionCB != NULL )
    (* show_ref -> actionCB)( XTM_SD_REASON_DESTROY, show_ref -> user_data );


  /* Destroy child windows. */
  if( show_ref -> filter_handle != NULL ) {
    xtmFiDestroy( show_ref -> filter_handle );
    show_ref -> filter_handle = NULL;
  }

  if( show_ref -> include_handle != NULL ) {
    xtmDiDestroy( show_ref -> include_handle );
    show_ref -> include_handle = NULL;
  }

  if( show_ref -> pr_handle != NULL ) {
    xtmPrDestroy( show_ref -> pr_handle );
    show_ref -> pr_handle = NULL;
  }


  /* Release the user data. */
  xtmCdFreeShadowEntry( custom_data_ref -> cal_db_handle,
                        show_ref -> db_name );


  /* Release the user data (only if not cached). */
  if( widget != NULL )
    SysFree( show_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void 
  filterApplyCB( XTM_FI_REASON      reason,
                 XTM_DM_FILTER_REC  *filter_ref,
                 void               *user_data )
{

  /* Variables. */
  SHOW_REC_REF  show_ref;


  /* Code. */

  if( reason != XTM_FI_REASON_APPLY && reason != XTM_FI_REASON_OK )
    return;

  show_ref = (SHOW_REC_REF) user_data;

  /* Save the filter. */
  memcpy( (void *) &show_ref -> filter_rec,
          (void *) filter_ref,
          sizeof( XTM_DM_FILTER_REC ) );


  /* Search again... */
  applyCB( show_ref -> showW, show_ref, NULL );


  return;

} /* filterApplyCB */


/*----------------------------------------------------------------------*/

static void 
  filterCB( Widget        widget,
            SHOW_REC_REF  show_ref,
            XtPointer     call_data )
{

  /* Code. */

  /* Initialize the filter? */
  if( show_ref -> filter_handle == NULL )
    show_ref -> filter_handle = xtmFiInitialize(
                                  show_ref -> appl_data_ref,
                                  show_ref -> showW,
                                  show_ref -> search_tags,
                                  False,
                                  filterApplyCB, (void *) show_ref );

  /* Display the filter window. */
  xtmFiDisplayFilterWindow( show_ref  -> filter_handle,
                            &show_ref -> filter_rec );


  return;

} /* filterCB */


/*----------------------------------------------------------------------*/

static void
  includeDbApplyCB( XTM_DI_REASON         reason,
                    XTM_CD_INCL_CALS_REF  new_db_incl_ref,
                    void                  *user_data )
{

  /* Variables. */
  SHOW_REC_REF            show_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         db_info;
  XTM_CD_INCL_CALS        db_incl;


  /* Code. */

  if( reason != XTM_DI_REASON_APPLY && reason != XTM_DI_REASON_OK )
    return;

  show_ref        = (SHOW_REC_REF) user_data;
  custom_data_ref = show_ref -> appl_data_ref -> custom_data;


  /* Fetch the current database. */
  (void) xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle,
                            show_ref -> db_name,
                            &db_info, &db_incl );

  /* Save the new include data. */
  (void) xtmCdChangeEntry( custom_data_ref -> cal_db_handle,
                           show_ref -> db_name,
                           &db_info, new_db_incl_ref );

  /* Search again... */
  applyCB( show_ref -> showW, show_ref, NULL );


  return;

} /* includeDbApplyCB */


/*----------------------------------------------------------------------*/

static void 
  infoCB( Widget                     widget,
          SHOW_REC_REF               show_ref,
          XmRowColumnCallbackStruct  *call_data )
{

  /* Code. */

  xtmHlDisplayHelp( show_ref -> appl_data_ref -> info_handle,
                    (int) call_data -> data,
                    sum_window_id, "" );

  return;

} /* infoCB */


/*----------------------------------------------------------------------*/

void
  mapCtrlMenuCB( Widget        widget,
                 SHOW_REC_REF  show_ref,
                 XtPointer     call_data )
{

  /* Variables. */
  Widget  tempW;


  /* Code. */

  /* Display entry flags? */
  tempW = XtNameToWidget( widget, "EntryFlagsBu" );
  if( flagIsSet( show_ref -> flags, ENTRY_FLAGS ) )
    XmToggleButtonSetState( tempW, True, False );
  else
    XmToggleButtonSetState( tempW, False, False );


  return;

} /* mapCtrlMenu */


/*----------------------------------------------------------------------*/

static void
  printCB( Widget        widget,
           SHOW_REC_REF  show_ref,
           XtPointer     call_data )
{

  /* Variables. */
  Boolean               ok;
  int                   display_lines;
  int                   items;
  char                  *char_ref;
  Widget                mainW;
  Widget                tempW;
  TIM_TIME_REF          from_date;
  TIM_TIME_REF          to_date;
  XTM_GL_BASE_DATA_REF  appl_data_ref;


  /* Code. */

  appl_data_ref = show_ref -> appl_data_ref;

  mainW = XtNameToWidget( show_ref -> showW, "ShowTlBase.ShowTlFo" );


  /* Fetch the date range. */
  tempW = XtNameToWidget( mainW, "ShDayRc.RangeTb" );

  ok = xtmFoFetchDate( mainW, tempW, XTM_FO_START_DATE, False, &from_date );
  if( ! ok )
    return;

  ok = xtmFoFetchDate( mainW, tempW, XTM_FO_END_DATE, False, &to_date );
  if( ! ok )
    return;

  if( from_date > to_date ) {
    xitErMessage( show_ref -> showW, XIT_ER_ERROR, 
                  module_name, "printCB",
                  msgGetText( MXDI_INVALID_DATE_RANGE ) );
    return;
  }


  /* Number of lines to display? */
  tempW    = XtNameToWidget( mainW, "ShLineRc.ShLineTx" );
  char_ref = xitStringGetText( tempW );

  items = sscanf( char_ref, "%d", &display_lines );
  if( items != 1 || display_lines < 0 )
    display_lines = 0;

  SysFree( char_ref );


  /* Create the printer window. */
  if( show_ref -> pr_handle == NULL ) {
    show_ref -> pr_handle = xtmPrInitialize( 
                              appl_data_ref,
                              appl_data_ref -> toplevel,
                              show_ref -> db_name,
                              NULL, NULL );

    if( show_ref -> pr_handle == NULL )
      return;
  }

  /* Display the printer window. */
  xtmPrDisplayPrinter( show_ref -> pr_handle, 
                       from_date, to_date, 
                       display_lines,
                       &show_ref -> filter_rec );

  /* Inherit the calendar settings. */
  xtmPrSetCalendar( show_ref -> pr_handle, show_ref -> db_name );


  return;

} /* printCB */
