/*----------------------------------------------------------------------------
--
--  Module:           xtmSchedMain
--
--  Project:          XDiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Schedule tool for an overview of diary appointments and notes.
--
--  Filename:         xtmSchedMain.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-04-01
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmSchedMain.c, Version: 1.1, Date: 95/02/18 15:52:44";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>

#include <Xm/Protocols.h>

#include <Xm/Xm.h>
#include <Xm/ArrowB.h>
#include <Xm/BulletinB.h>
#include <Xm/CascadeB.h>
#include <Xm/Form.h>
#include <Xm/LabelP.h>
#include <Xm/PanedW.h>
#include <Xm/RowColumn.h>
#include <Xm/Separator.h>
#include <Xm/ScrolledW.h>
#include <Xm/Text.h>
#include <Xm/ToggleB.h>

#include "xitRwPixB.h"

#include "LstLinked.h"
#include "Message.h"
#include "System.h"
#include "TimDate.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmAccBase.h"
#include "xtmArchive.h"
#include "xtmCpMvEntry.h"
#include "xtmCustBase.h"
#include "xtmDbFilter.h"
#include "xtmDbInclude.h"
#include "xtmDbTools.h"
#include "xtmDbMisc.h"
#include "xtmDelEntry.h"
#include "xtmDuplEntry.h"
#include "xtmEditEntry.h"
#include "xtmFields.h"
#include "xtmFormat.h"
#include "xtmFreeze.h"
#include "xtmHelp.h"
#include "xtmIcons.h"
#include "xtmMsgSend.h"
#include "xtmOpenView.h"
#include "xtmPlan.h"
#include "xtmRemote.h"
#include "xtmSchedAct.h"
#include "xtmSchedTool.h"
#include "xtmSchedWin.h"
#include "xtmSelHidden.h"
#include "xtmShowToDo.h"
#include "xtmTools.h"
#include "xtmUpdate.h"
#include "xitError.h"
#include "xitFieldSel.h"
#include "xitTools.h"
#include "XmUbArrLab.h"
#include "xtmSchedMain.h"

#include "xtmSchedPriv.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Day views in the cache? */
#define  MAX_CACHE_ENTRIES   3

/* Local widgets in the database select window. */
#define appLa             dataLocalW[  0 ]
#define dateInfoLa        dataLocalW[  1 ]
#define dateLa            dataLocalW[  2 ]
#define dateRc            dataLocalW[  3 ]
#define dayDispBb         dataLocalW[  4 ]
#define dayDispSw         dataLocalW[  5 ]
#define entryBb           dataLocalW[  6 ]
#define entryBottomSp     dataLocalW[  7 ]
#define entrySw           dataLocalW[  8 ]
#define fromTb            dataLocalW[  9 ]
#define makeAppPb         dataLocalW[ 10 ]
#define makeNotePb        dataLocalW[ 11 ]
#define menuBr            dataLocalW[ 12 ]
#define navDayAl          dataLocalW[ 13 ]
#define navMonthAl        dataLocalW[ 14 ]
#define navFo             dataLocalW[ 15 ]
#define navWeekAl         dataLocalW[ 16 ]
#define navYearAl         dataLocalW[ 17 ]
#define noteBb            dataLocalW[ 18 ]
#define noteBottomSp      dataLocalW[ 19 ]
#define noteSw            dataLocalW[ 20 ]
#define pane1Fo           dataLocalW[ 21 ]
#define pane2Fo           dataLocalW[ 22 ]
#define schedPa           dataLocalW[ 23 ]
#define timeDispBb        dataLocalW[ 24 ]
#define timeDispLa        dataLocalW[ 25 ]
#define timeDispSw        dataLocalW[ 26 ]
#define trackLa           dataLocalW[ 27 ]
#define trackPx           dataLocalW[ 28 ]
#define zoomFs            dataLocalW[ 29 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmSchedMain";

/* IDs for the help windows. */
static char  *day_view_window_id = "DayView";

/* Trace double clicks on entries. */
static Time  time_last = 0;

/* Cache entries. */
static Boolean        init_cache = True;
static SCHED_REC_REF  cache_entries[ MAX_CACHE_ENTRIES ];

/* Last active day view window. */
static SCHED_REC_REF  last_active_sched_ref = NULL;

/* Tick pixmaps to use. */
static Boolean  init_tick_pixmap = True;
static Pixmap   tick_pixmap[ 7 ];


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void 
  applyCB( Widget         widget,
           SCHED_REC_REF  sched_ref,
           XtPointer      call_data );

static void 
  closeCB( Widget         widget,
           SCHED_REC_REF  sched_ref,
           XtPointer      call_data );

static void 
  closeReallyCB( Widget         widget,
                 SCHED_REC_REF  sched_ref,
                 XtPointer      call_data );

static Widget
  createScheduleWindow( Widget         parent,
                        SCHED_REC_REF  sched_ref );

static void 
  ctrlMenuCB( Widget                     widget,
              SCHED_REC_REF              sched_ref,
              XmRowColumnCallbackStruct  *call_data );

static void 
  destroyCB( Widget         widget,
             SCHED_REC_REF  sched_ref,
             XtPointer      call_data );

static void
  displayEntryInfo( SCHED_REC_REF  sched_ref );

static void 
  editMenuCB( Widget                     widget,
              SCHED_REC_REF              sched_ref,
              XmRowColumnCallbackStruct  *call_data );

static void 
  fileMenuCB( Widget                     widget,
              SCHED_REC_REF              sched_ref,
              XmRowColumnCallbackStruct  *call_data );

static void 
  filterApplyCB( XTM_FI_REASON      reason,
                 XTM_DM_FILTER_REC  *filter_ref,
                 void               *user_data );

static void 
  gotFocusCB( Widget               widget,
              SCHED_REC_REF        sched_ref,
              XmAnyCallbackStruct  *call_data );

static void
  includeDbApplyCB( XTM_DI_REASON         reason,
                    XTM_CD_INCL_CALS_REF  new_db_incl_ref,
                    void                  *user_data );

static void 
  infoCB( Widget                     widget,
          SCHED_REC_REF              sched_ref,
          XmRowColumnCallbackStruct  *call_data );

static void
  leaveHookCB( Widget    widget,
               XEvent    *event,
               char      *leave_args[],
               Cardinal  *num_args );

static void 
  makeAppointmentCB( Widget               widget,
                     SCHED_REC_REF        sched_ref,
                     XmAnyCallbackStruct  *call_data );

static void 
  makeNoteCB( Widget               widget,
              SCHED_REC_REF        sched_ref,
              XmAnyCallbackStruct  *call_data );

static void 
  navigateCB( Widget                        widget,
              SCHED_REC_REF                 sched_ref,
              XmUbArrowLabelCallbackStruct  *call_data );

static void
  newViewApplyCB( XTM_OV_REASON    reason,
                  XTM_CD_CAL_INFO  *db_info_ref,
                  void             *user_data );

static void 
  optionsMenuCB( Widget                     widget,
                 SCHED_REC_REF              sched_ref,
                 XmRowColumnCallbackStruct  *call_data );

static void
  printerActionCB( XTM_PR_REASON  reason,
                   void           *user_data );

static void
  selHideApplyCB( XTM_SH_REASON  reason,
                  UINT32         entry_id,
                  char           *calendar,
                  TIM_TIME_REF   as_date,
                  TIM_TIME_REF   as_time,
                  void           *user_data );

static void
  summaryActionCB( XTM_SD_REASON  reason,
                   void           *user_data );

static void
  timeActionHookCB( Widget         widget,
                    SCHED_REC_REF  sched_ref,
                    XEvent         *event );

static void
  updateCB( UINT32  flags,
            void    *user_data,
            void    *update_user_data );

static void 
  zoomCB( Widget         widget,
          SCHED_REC_REF  sched_ref,
          XtPointer      call_data );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XTM_SM_HANDLE
  xtmSmInitialize( XTM_GL_BASE_DATA_REF  appl_data_ref,
                   Widget                toplevel,
                   char                  *db_name,
                   XTM_SM_INIT_REF       init_ref,
                   XTM_SM_ACTION_CB      actionCB,
                   void                  *user_data )
{

  /* Variables. */
  Boolean                 ok;
  int                     index;
  int                     length;
  SCHED_REC_REF           sched_ref;
  TIM_TIME_REF            now;
  XTM_CD_CAL_INFO         db_info;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;

  static XtActionsRec  leaveActions[] = {
    { "LeaveHook", leaveHookCB }
  };


  /* Code. */

  custom_data_ref = appl_data_ref -> custom_data;

  now = TimLocalTime( TimMakeTimeNow() );
  now = TimMakeTime(  TimIndexOfYear(  now ),
                      TimIndexOfMonth( now ),
                      TimIndexOfDay(   now ),
                      0, 0, 0 );


  /* Fetch database information. */
  ok = xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle,
                          db_name, &db_info, NULL );
  if( ! ok )
    return( False );


  /* Minium permissions are read. */
  if( flagIsClear( db_info.operations, XTM_DB_FLAG_MODE_READ ) )
    return( NULL );


  /* Initialize the cache? */
  if( init_cache ) {
    for( index = 0; index < MAX_CACHE_ENTRIES; index++ )
      cache_entries[ index ] = NULL;

    init_cache = False;
  }


  /* Anything in the cache? */
  sched_ref = NULL;

  for( index = 0; index < MAX_CACHE_ENTRIES; index++ ) {
    if( cache_entries[ index ] != NULL ) {
      sched_ref = cache_entries[ index ];
      cache_entries[ index ] = NULL;

      break;
    }
  }

  /* Create the schedule record? */
  if( sched_ref == NULL ) {

    sched_ref = SysNew( SCHED_REC );
    if( sched_ref == NULL )
      return( NULL );

    sched_ref -> scheduleW       = NULL;
    sched_ref -> update_id       = 0;
    sched_ref -> counter         = 1;
    sched_ref -> selected_widget = NULL;
    sched_ref -> animate_ref     = NULL;

  } /* if */

  sched_ref -> appl_data_ref   = appl_data_ref;
  sched_ref -> editor_handle   = NULL;
  sched_ref -> filter_handle   = NULL;
  sched_ref -> include_handle  = NULL;
  sched_ref -> nav_cal_handle  = NULL;
  sched_ref -> open_handle     = NULL;
  sched_ref -> planner_handle  = NULL;
  sched_ref -> pr_handle       = NULL;
  sched_ref -> remote_handle   = NULL;
  sched_ref -> sel_hide_handle = NULL;
  sched_ref -> show_handle     = NULL;
  sched_ref -> force_update    = False;
  sched_ref -> schedule_start  = now;
  sched_ref -> day_width       = (Dimension) custom_data_ref -> def_day_width;
  sched_ref -> actionCB        = actionCB;
  sched_ref -> user_data       = user_data;

  sched_ref -> old_entry_delta    = 0;
  sched_ref -> old_schedule_start = 0;
  sched_ref -> old_schedule_stop  = 0;
  sched_ref -> old_entry_start    = 0;
  sched_ref -> old_entry_stop     = 0;

  sched_ref -> filter_rec.flags            = XTM_DM_FLAG_ALL;
  sched_ref -> filter_rec.text_string[ 0 ] = '\0';
  sched_ref -> filter_rec.tag_string[  0 ] = '\0';

  length = strlen( custom_data_ref -> entry_tags_filter );

  if( length > 0 && length < XTM_DM_MAX_TAG_SEARCH ) {
    strcpy( sched_ref -> filter_rec.tag_string,
            custom_data_ref -> entry_tags_filter );
    flagSet( sched_ref -> filter_rec.flags, XTM_DM_FLAG_SEARCH_TAG );
  }

  sched_ref -> show_no_days = custom_data_ref -> day_view_days;
  if( sched_ref -> show_no_days < 1 || sched_ref -> show_no_days > 31 )
    sched_ref -> show_no_days = 2;

  /* We have no default day width :-(. */
  if( sched_ref -> day_width == 0 )
    sched_ref -> day_width = 130;


  /* Default values? */
  if( init_ref == NULL ) {
    sched_ref -> entry_start = TimMakeTime( 1970, 1, 1, 
                                            custom_data_ref -> start_hour,
                                            0, 0 );
    sched_ref -> entry_stop  = TimMakeTime( 1970, 1, 1, 
                                            custom_data_ref -> stop_hour,
                                            0, 0 );

    sched_ref -> entry_delta = custom_data_ref -> default_entry_delta;
    sched_ref -> flags       = (XTM_SM_USE_GRID | XTM_SM_ENTRY_HANDLES);

    if( custom_data_ref -> confirm_actions )
      flagSet( sched_ref -> flags, XTM_SM_CONFIRM );
    if( custom_data_ref -> list_layout )
      flagSet( sched_ref -> flags, XTM_SM_LIST_LAYOUT );
    if( custom_data_ref -> display_nav_cal )
      flagSet( sched_ref -> flags, XTM_SM_AUTO_NAV_CAL );
    if( custom_data_ref -> display_entry_flags )
      flagSet( sched_ref -> flags, XTM_SM_ENTRY_FLAGS );
    if( custom_data_ref -> read_only_mode )
      flagSet( sched_ref -> flags, XTM_SM_READ_ONLY );
    if( custom_data_ref -> true_color_included )
      flagSet( sched_ref -> flags, XTM_SM_TRUE_COLOR_INC );

  } else {
    sched_ref -> entry_start = init_ref -> entry_start;
    sched_ref -> entry_stop  = init_ref -> entry_stop;
    sched_ref -> entry_delta = init_ref -> entry_delta;
    sched_ref -> flags       = init_ref -> default_flags;

  } /* if */


  /* Create a shadow calendar database. */
  ok = xtmCdCreateShadowEntry( custom_data_ref -> cal_db_handle,
                               db_name, sched_ref -> db_name );
  if( ! ok ) {
    SysFree( sched_ref );

    return( NULL );
  }


  /* Data for our animation. */
  if( sched_ref -> animate_ref == NULL ) {
    sched_ref -> animate_ref = SysNew( ANIMATE_REC );

    if( sched_ref -> animate_ref == NULL ) {
      SysFree( sched_ref );
      return( NULL );
    }

    sched_ref -> animate_ref -> aniEntryW = NULL;
    sched_ref -> animate_ref -> aniNoteW  = NULL;
    sched_ref -> animate_ref -> aniTimeW  = NULL;
  }

  sched_ref -> animate_ref -> animate_on = False;


  /* Create the schedule window and entry cache? */
  if( sched_ref -> scheduleW == NULL ) {

    /* Entry cache. */
    sched_ref -> entry_cache_ref = SysNew( ENTRY_CACHE );

    sched_ref -> entry_cache_ref -> max_used = 500;
    sched_ref -> entry_cache_ref -> used = 
      (ENTRY_INFO_REF *) SysMalloc( sizeof( ENTRY_INFO_REF ) * 500 );

    for( index = 0; index < 500; index++ )
      sched_ref -> entry_cache_ref -> used[ index ] = NULL;

    /* Schedule window. */
    sched_ref -> scheduleW = createScheduleWindow( toplevel, sched_ref );

    if( sched_ref -> scheduleW == NULL ) {
      xtmCdFreeShadowEntry( custom_data_ref -> cal_db_handle,
                            sched_ref -> db_name );

      SysFree( sched_ref -> animate_ref );
      SysFree( sched_ref );

      return( NULL );
    }

    /* Initialize the tick pixmaps? */
    if( init_tick_pixmap ) {
      Widget  tempW;

      tempW = XtNameToWidget( sched_ref -> scheduleW, 
                              "SchedTlBase.SchedTlFo" );

      tick_pixmap[ 0 ] = xtmIcFetchSimplePixmap( tempW, XTM_IC_ICON_ANI_MOVE,
                                                 False );
      tick_pixmap[ 1 ] = xtmIcFetchSimplePixmap( tempW, XTM_IC_ICON_ANI_MOVE1,
                                                 False );
      tick_pixmap[ 2 ] = xtmIcFetchSimplePixmap( tempW, XTM_IC_ICON_ANI_MOVE2,
                                                 False );
      tick_pixmap[ 3 ] = xtmIcFetchSimplePixmap( tempW, XTM_IC_ICON_ANI_MOVE3,
                                                 False );
      tick_pixmap[ 4 ] = xtmIcFetchSimplePixmap( tempW, XTM_IC_ICON_ANI_MOVE4,
                                                 False );
      tick_pixmap[ 5 ] = xtmIcFetchSimplePixmap( tempW, XTM_IC_ICON_ANI_MOVE5,
                                                 False );
      tick_pixmap[ 6 ] = xtmIcFetchSimplePixmap( tempW, XTM_IC_ICON_ANI_MOVE6,
                                                 False );
      init_tick_pixmap = False;
    }

    /* Animation pixmaps. */
    for( index = 0; index < 7; index++ )
      sched_ref -> tick_pixmap[ index ] = tick_pixmap[ index ];

    sched_ref -> tick_pixmap_index = 0;
    xtmStClearEntryTime( sched_ref );

    /* We will need this for the animation stuff. */
    XtAppAddActions( XtWidgetToApplicationContext( toplevel ), 
                     leaveActions, XtNumber( leaveActions ) );

  } /* if */


  return( sched_ref );

} /* xtmSmInitialize */


/*----------------------------------------------------------------------*/

void
  xtmSmEmptyCache()
{

  /* Variables. */
  int  index;


  /* Code. */

  if( ! init_cache )
    return;

  for( index = 0; index < MAX_CACHE_ENTRIES; index++ ) {
    if( cache_entries[ index ] != NULL )
      XtDestroyWidget( cache_entries[ index ] -> scheduleW );

    cache_entries[ index ] = NULL;
  }


  return;

} /* xtmSmEmptyCache */


/*----------------------------------------------------------------------*/

void
  xtmSmDestroy( XTM_SM_HANDLE  sched_handle )
{

  /* Variables. */
  SCHED_REC_REF  sched_ref;


  /* Code. */

  if( sched_handle == NULL )
    return;


  /* Our private data. */
  sched_ref = (SCHED_REC_REF) sched_handle;

  /* Destroy the schedule widget. */
  if( sched_ref -> scheduleW != NULL )
    closeReallyCB( NULL, sched_ref, NULL );


  return;

} /* xtmSmDestroy */


/*----------------------------------------------------------------------*/

void
  xtmSmSetFlags( XTM_SM_HANDLE  sched_handle,
                 UINT32         flags )
{

  /* Variables. */
  SCHED_REC_REF  sched_ref;


  /* Code. */

  if( sched_handle == NULL )
    return;


  /* Our private data. */
  sched_ref = (SCHED_REC_REF) sched_handle;

  flagSet( sched_ref -> flags, flags );


  return;

} /* xtmSmSetFlags */


/*----------------------------------------------------------------------*/

void
  xtmSmViewSchedule( XTM_SM_HANDLE  sched_handle,
                     TIM_TIME_REF   start_date,
                     Boolean        iconic )
{

  /* Variables. */
  Arg            args[ 10 ];
  Cardinal       n;
  Widget         tempW;
  SCHED_REC_REF  sched_ref;


  /* Code. */

  if( sched_handle == NULL )
    return;


  /* Our private data. */
  sched_ref = (SCHED_REC_REF) sched_handle;

  if( start_date != 0 )
    sched_ref -> schedule_start = start_date;


  /* Start the day view on a specific day? */
  xtmStStartOnDay( sched_ref, &sched_ref -> schedule_start );


  /* Set the start and stop dates and display the schedule. */
  xtmSwSetScheduleDates( sched_ref, sched_ref -> scheduleW, False );

  xtmSwSetSchedule( sched_ref, sched_ref -> scheduleW );


  /* Start as icon? */
  n = 0;
  if( iconic ) {
    XtSetArg( args[ n ], XmNinitialState, IconicState ); n++;
  } else {
    XtSetArg( args[ n ], XmNinitialState, NormalState ); n++;
  }
  XtSetValues( sched_ref -> scheduleW, args, n );


  /* Make sure the schedule window is visable. */
  XtPopup( sched_ref -> scheduleW, XtGrabNone );


  if( ! iconic ) {
    XRaiseWindow( XtDisplay( sched_ref -> scheduleW ), 
                  XtWindow(  sched_ref -> scheduleW ) );

    XtMapWidget( sched_ref -> scheduleW );
  }

  /* Don't let our children spoil our size. */
  n = 0;
  XtSetArg( args[ n ], XmNallowShellResize, False ); n++;
  XtSetValues( sched_ref -> scheduleW, args, n );


  /* Register for updates? */
  if( sched_ref -> update_id == 0 )
    sched_ref -> update_id = xtmUpRegister( 
                               (XTM_UP_MINUTE_TICK | XTM_UP_SCHEDULE),
                               updateCB, (void *) sched_ref );


  /* Keep track of when we are active. */
#ifndef XD_HAS_FOCUS_BUG
  tempW = XtNameToWidget( sched_ref -> scheduleW, "SchedTlBase" );
  XtAddCallback( tempW, XmNfocusCallback,
                 (XtCallbackProc) gotFocusCB, (XtPointer) sched_ref );
#endif

  gotFocusCB( NULL, sched_ref, NULL );


  return;

} /* xtmSmViewSchedule */


/*----------------------------------------------------------------------*/

Boolean
  xtmSmViewLastUsed( TIM_TIME_REF   start_date )
{

  /* Variables. */
  SCHED_REC_REF  sched_ref;


  /* Code. */

  /* Any 'last used day' view? */
  if( last_active_sched_ref == NULL )
    return( False );

  sched_ref = last_active_sched_ref;

  if( start_date != 0 )
    sched_ref -> schedule_start = start_date;


  /* Start the day view on a specific day? */
  xtmStStartOnDay( sched_ref, &sched_ref -> schedule_start );

  /* Set the start and stop dates and display the schedule. */
  xtmSwSetScheduleDates( sched_ref, sched_ref -> scheduleW, False );

  xtmSwSetSchedule( sched_ref, sched_ref -> scheduleW );


  /* The window should be 'on top'. */
  XRaiseWindow( XtDisplay( sched_ref -> scheduleW ), 
                XtWindow(  sched_ref -> scheduleW ) );

  XtMapWidget( sched_ref -> scheduleW );


  return( True );

} /* xtmSmViewLastUsed */


/*----------------------------------------------------------------------*/

static Widget
  createScheduleWindow( Widget         parent,
                        SCHED_REC_REF  sched_ref )
{

  /* Variables. */
  int         index;
  char        buffer[ 100 ];
  Arg         args[ 10 ];
  Cardinal    n;
  Dimension   height;
  Dimension   width;
  Dimension   sb_height;
  Dimension   time_width;
  Widget      dataLocalW[ 30 ];
  Widget      menuCasc[ 5 ];
  Widget      menuCtrlBu[ 12 ];
  Widget      menuEditBu[ 12 ];
  Widget      menuFileBu[ 8 ];
  Widget      menuHelpBu[ 7 ];
  Widget      menuOptionsBu[ 6 ];
  Widget      pulldownMenu[ 5 ];
  Widget      schedTl;
  Widget      timeIndicatorAr;
  Widget      tempW;
  Widget      toplevel;
  Widget      workFo;
  XmString    xstr;

  static char  *zoom_times = "0:10 0:15 0:30 0:45 1:00 1:30 2:00 4:00 6:00";

  static char  *pull_downs[] = { 
    "pdown1", "pdown2", "pdown3", "pdown4", "pdown5" };

  static XIT_PUSH_STRUCT button_def[] = {
    { "MakeAppPb",   "", "", True, NULL },
    { "MakeNotePb",  "", "", True, NULL }
  };

  static XIT_CASCADE_STRUCT menupane[] = {
    { "", "", "FilePane" },
    { "", "", "EditPane" },
    { "", "", "ControlPane" },
    { "", "", "OptionsPane" },
    { "", "", "HelpPane" },
  };

  static XIT_MENU_BUTTON_STRUCT file_casc[] = {
    { "",      "",  NULL, "NewViewBu",  True,  False, False },
    { "",      "",  NULL, "OpenSameBu", True,  False, False },
    { "",      "",  NULL, "PrintBu",    True,  False, False },
    { XIT_SEP, " ", NULL, "",           False, False, False },
    { "",      "",  NULL, "UploadBu",   True,  False, False },
    { "",      "",  NULL, "DownloadBu", True,  False, False },
    { XIT_SEP, " ", NULL, "",           False, False, False },
    { "",      "",  NULL, "CloseBu",    True,  False, False },
  };

  static XIT_MENU_BUTTON_STRUCT edit_casc[] = {
    { "",      "",  NULL, "EditBu",     True,  False, False },
    { "",      "",  NULL, "DeleteBu",   True,  False, False },
    { "",      "",  NULL, "MoveBu",     True,  False, False },
    { "",      "",  NULL, "CopyBu",     True,  False, False },
    { "",      "",  NULL, "DuplBu",     True,  False, False },
    { "",      "",  NULL, "ArchBu",     True,  False, False },
    { "",      "",  NULL, "FreezeBu",   True,  False, False },
    { "",      "",  NULL, "InfoBu",     True,  False, False },
    { XIT_SEP, " ", NULL, "",           False, False, False },
    { "",      "",  NULL, "CopyBufBu",  True,  False, False },
    { "",      "",  NULL, "CutBufBu",   True,  False, False },
    { "",      "",  NULL, "PasteBufBu", True,  False, False },
  };

  static XIT_MENU_BUTTON_STRUCT ctrl_casc[] = {
    { "",      "",  NULL, "ConfirmBu",      True,  True,  False },
    { "",      "",  NULL, "HandleBu",       True,  True,  False },
    { "",      "",  NULL, "ListLayBu",      True,  True,  False },
    { "",      "",  NULL, "UseGridBu",      True,  True,  False },
    { "",      "",  NULL, "EntryFlagsBu",   True,  True,  False },
    { "",      "",  NULL, "TrueColorIncBu", True,  True,  False },
    { XIT_SEP, " ", NULL, "",               False, False, False },
    { "",      "",  NULL, "DispDaysBu",     True,  False, False },
    { "",      "",  NULL, "DayWidthBu",     True,  False, False },
    { "",      "",  NULL, "FilterBu",       True,  False, False },
    { "",      "",  NULL, "IncludeBu",      True,  False, False },
    { "",      "",  NULL, "SelHideBu",      True,  False, False },
  };

  static XIT_MENU_BUTTON_STRUCT opt_casc[] = {
    { "",      "",  NULL, "ApEditBu",    True,  False, False },
    { "",      "",  NULL, "SummaryBu",   True,  False, False },
    { "",      "",  NULL, "SendMsgBu",   True,  False, False },
    { "",      "",  NULL, "PlannerBu",   True,  False, False },
    { XIT_SEP, " ", NULL, "",            False, False, False },
    { "",      "",  NULL, "RedisplayBu", True,  False, False },
  };

  static XIT_MENU_BUTTON_STRUCT help_casc[] = {
    { "", "", NULL, "ContextBu", True, False, False },
    { "", "", NULL, "WindowsBu", True, False, False },
    { "", "", NULL, "KeysBu",    True, False, False },
    { "", "", NULL, "IndexBu",   True, False, False },
    { "", "", NULL, "HelpBu",    True, False, False },
    { "", "", NULL, "VersionBu", True, False, False },
    { "", "", NULL, "AboutBu",   True, False, False },
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "",   applyCB, NULL },
    { NULL, NULL,    NULL },
    { NULL, NULL,    NULL },
    { NULL, NULL,    NULL },
    { "",   closeCB, NULL },
  };

  static XIT_ARROW_STRUCT time_ind_def[] = {
    { "TimeIndicatorAr", False, XmARROW_RIGHT, NULL },
  };


  /* Code. */

  /* Initialize the menu and label items. */
  action_buttons[ 0 ].label = msgGetText( MXDI_APPLY_BUTTON );
  action_buttons[ 0 ].data  = sched_ref;
  action_buttons[ 4 ].label = msgGetText( MXDI_CLOSE_BUTTON );
  action_buttons[ 4 ].data  = sched_ref;

  menupane[ 0 ].title    = msgGetText( MXDI_FILE_MENU );
  menupane[ 0 ].mnemonic = msgGetText( MXDI_FILE_MENU_ACC );
  menupane[ 1 ].title    = msgGetText( MXDI_EDIT_MENU );
  menupane[ 1 ].mnemonic = msgGetText( MXDI_EDIT_MENU_ACC );
  menupane[ 2 ].title    = msgGetText( MXDI_CTRL_MENU );
  menupane[ 2 ].mnemonic = msgGetText( MXDI_CTRL_MENU_ACC );
  menupane[ 3 ].title    = msgGetText( MXDI_OPT_MENU );
  menupane[ 3 ].mnemonic = msgGetText( MXDI_OPT_MENU_ACC );
  menupane[ 4 ].title    = msgGetText( MXDI_HELP_MENU );
  menupane[ 4 ].mnemonic = msgGetText( MXDI_HELP_MENU_ACC );

  file_casc[ 0 ].title    = msgGetText( MXDI_OPEN_SAME_MENU );
  file_casc[ 0 ].mnemonic = msgGetText( MXDI_OPEN_SAME_MENU_ACC );
  file_casc[ 1 ].title    = msgGetText( MXDI_NEW_VIEW_MENU );
  file_casc[ 1 ].mnemonic = msgGetText( MXDI_NEW_VIEW_MENU_ACC );
  file_casc[ 2 ].title    = msgGetText( MXDI_PRINT_MENU );
  file_casc[ 2 ].mnemonic = msgGetText( MXDI_PRINT_MENU_ACC );
  file_casc[ 4 ].title    = msgGetText( MXDI_UPLOAD_MENU );
  file_casc[ 4 ].mnemonic = msgGetText( MXDI_UPLOAD_MENU_ACC );
  file_casc[ 5 ].title    = msgGetText( MXDI_DOWNLOAD_MENU );
  file_casc[ 5 ].mnemonic = msgGetText( MXDI_DOWNLOAD_MENU_ACC );
  file_casc[ 7 ].title    = msgGetText( MXDI_CLOSE_MENU );
  file_casc[ 7 ].mnemonic = msgGetText( MXDI_CLOSE_MENU_ACC );

  if( flagIsSet( sched_ref -> flags, XTM_SM_READ_ONLY ) ) {
    edit_casc[ 0 ].title    = msgGetText( MXDI_SHOW_MENU );
    edit_casc[ 0 ].mnemonic = msgGetText( MXDI_SHOW_MENU_ACC );
  } else {
    edit_casc[ 0 ].title    = msgGetText( MXDI_EDIT_MENU );
    edit_casc[ 0 ].mnemonic = msgGetText( MXDI_EDIT_MENU_ACC );
  }

  edit_casc[  1 ].title    = msgGetText( MXDI_DELETE_MENU );
  edit_casc[  1 ].mnemonic = msgGetText( MXDI_DELETE_MENU_ACC );
  edit_casc[  2 ].title    = msgGetText( MXDI_MOVE_MENU );
  edit_casc[  2 ].mnemonic = msgGetText( MXDI_MOVE_MENU_ACC );
  edit_casc[  3 ].title    = msgGetText( MXDI_COPY_MENU );
  edit_casc[  3 ].mnemonic = msgGetText( MXDI_COPY_MENU_ACC );
  edit_casc[  4 ].title    = msgGetText( MXDI_DUPL_MENU );
  edit_casc[  4 ].mnemonic = msgGetText( MXDI_DUPL_MENU_ACC );
  edit_casc[  5 ].title    = msgGetText( MXDI_ARCH_MENU );
  edit_casc[  5 ].mnemonic = msgGetText( MXDI_ARCH_MENU_ACC );
  edit_casc[  6 ].title    = msgGetText( MXDI_FREEZE_MENU );
  edit_casc[  6 ].mnemonic = msgGetText( MXDI_FREEZE_MENU_ACC );
  edit_casc[  7 ].title    = msgGetText( MXDI_ENTRY_INFO_MENU );
  edit_casc[  7 ].mnemonic = msgGetText( MXDI_ENTRY_INFO_MENU_ACC );
  edit_casc[  9 ].title    = msgGetText( MXDI_COPY_BUF_MENU );
  edit_casc[  9 ].mnemonic = msgGetText( MXDI_COPY_BUF_MENU_ACC );
  edit_casc[ 10 ].title    = msgGetText( MXDI_CUT_BUF_MENU );
  edit_casc[ 10 ].mnemonic = msgGetText( MXDI_CUT_BUF_MENU_ACC );
  edit_casc[ 11 ].title    = msgGetText( MXDI_PASTE_BUF_MENU );
  edit_casc[ 11 ].mnemonic = msgGetText( MXDI_PASTE_BUF_MENU_ACC );

  ctrl_casc[  0 ].title    = msgGetText( MXDI_CONF_MENU );
  ctrl_casc[  0 ].mnemonic = msgGetText( MXDI_CONF_MENU_ACC );
  ctrl_casc[  1 ].title    = msgGetText( MXDI_HANDLE_MENU );
  ctrl_casc[  1 ].mnemonic = msgGetText( MXDI_HANDLE_MENU_ACC );
  ctrl_casc[  2 ].title    = msgGetText( MXDI_TIME_LAY_MENU );
  ctrl_casc[  2 ].mnemonic = msgGetText( MXDI_TIME_LAY_MENU_ACC );
  ctrl_casc[  3 ].title    = msgGetText( MXDI_GRID_MENU );
  ctrl_casc[  3 ].mnemonic = msgGetText( MXDI_GRID_MENU_ACC );
  ctrl_casc[  4 ].title    = msgGetText( MXDI_FLAGS_DISPLAY_MENU );
  ctrl_casc[  4 ].mnemonic = msgGetText( MXDI_FLAGS_DISPLAY_MENU_ACC );
  ctrl_casc[  5 ].title    = msgGetText( MXDI_TRUE_COLOR_MENU );
  ctrl_casc[  5 ].mnemonic = msgGetText( MXDI_TRUE_COLOR_MENU_ACC );
  ctrl_casc[  7 ].title    = msgGetText( MXDI_DISP_DAYS_MENU );
  ctrl_casc[  7 ].mnemonic = msgGetText( MXDI_DISP_DAYS_MENU_ACC );
  ctrl_casc[  8 ].title    = msgGetText( MXDI_DAY_WIDTH_MENU );
  ctrl_casc[  8 ].mnemonic = msgGetText( MXDI_DAY_WIDTH_MENU_ACC );
  ctrl_casc[  9 ].title    = msgGetText( MXDI_FILTER_MENU );
  ctrl_casc[  9 ].mnemonic = msgGetText( MXDI_FILTER_MENU_ACC );
  ctrl_casc[ 10 ].title    = msgGetText( MXDI_INCLUDE_MENU );
  ctrl_casc[ 10 ].mnemonic = msgGetText( MXDI_INCLUDE_MENU_ACC );
  ctrl_casc[ 11 ].title    = msgGetText( MXDI_SEL_HIDDEN_MENU );
  ctrl_casc[ 11 ].mnemonic = msgGetText( MXDI_SEL_HIDDEN_MENU_ACC );

  if( flagIsSet( sched_ref -> flags, XTM_SM_READ_ONLY ) ) {
    opt_casc[ 0 ].title    = msgGetText( MXDI_APP_SHOW_MENU );
    opt_casc[ 0 ].mnemonic = msgGetText( MXDI_APP_SHOW_MENU_ACC );
  } else {
    opt_casc[ 0 ].title    = msgGetText( MXDI_APP_EDITOR_MENU );
    opt_casc[ 0 ].mnemonic = msgGetText( MXDI_APP_EDITOR_MENU_ACC );
  }

  opt_casc[ 1 ].title    = msgGetText( MXDI_SUM_MENU );
  opt_casc[ 1 ].mnemonic = msgGetText( MXDI_SUM_MENU_ACC );
  opt_casc[ 2 ].title    = msgGetText( MXDI_SEND_MSG_MENU );
  opt_casc[ 2 ].mnemonic = msgGetText( MXDI_SEND_MSG_MENU_ACC );
  opt_casc[ 3 ].title    = msgGetText( MXDI_PLAN_MENU );
  opt_casc[ 3 ].mnemonic = msgGetText( MXDI_PLAN_MENU_ACC );
  opt_casc[ 5 ].title    = msgGetText( MXDI_REDISPLAY_MENU );
  opt_casc[ 5 ].mnemonic = msgGetText( MXDI_REDISPLAY_MENU_ACC );

  help_casc[ 0 ].title    = msgGetText( MXDI_HELP_CONTEXT );
  help_casc[ 0 ].mnemonic = msgGetText( MXDI_HELP_CONTEXT_ACC );
  help_casc[ 1 ].title    = msgGetText( MXDI_HELP_WINDOWS );
  help_casc[ 1 ].mnemonic = msgGetText( MXDI_HELP_WINDOWS_ACC );
  help_casc[ 2 ].title    = msgGetText( MXDI_HELP_KEYS );
  help_casc[ 2 ].mnemonic = msgGetText( MXDI_HELP_KEYS_ACC );
  help_casc[ 3 ].title    = msgGetText( MXDI_HELP_INDEX );
  help_casc[ 3 ].mnemonic = msgGetText( MXDI_HELP_INDEX_ACC );
  help_casc[ 4 ].title    = msgGetText( MXDI_HELP_HELP );
  help_casc[ 4 ].mnemonic = msgGetText( MXDI_HELP_HELP_ACC );
  help_casc[ 5 ].title    = msgGetText( MXDI_HELP_VERSION );
  help_casc[ 5 ].mnemonic = msgGetText( MXDI_HELP_VERSION_ACC );
  help_casc[ 6 ].title    = msgGetText( MXDI_HELP_ABOUT );
  help_casc[ 6 ].mnemonic = msgGetText( MXDI_HELP_ABOUT_ACC );

  button_def[ 0 ].title   = msgGetText( MXDI_MAKE_APPOINTMENT_LABEL );
  button_def[ 1 ].title   = msgGetText( MXDI_MAKE_NOTE_LABEL );


  /* No popups are defined. */
  sched_ref -> entry_popupW      = NULL;
  sched_ref -> entry_text_popupW = NULL;


  /* Create a toplevel dialog with buttons. */
  toplevel = xitGetToplevelWidget( parent );

  schedTl = xitCreateToplevelDialog( toplevel, "SchedTl",
                                     1, 0,
                                     action_buttons,
                                     XtNumber( action_buttons ) );

  XtAddCallback( schedTl, XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) sched_ref );

  /* Exit the application if this window is deleted. */
  {
    Atom  wm_delete_window;

    wm_delete_window = XmInternAtom( XtDisplay( schedTl ),
                                     "WM_DELETE_WINDOW", False );

    XmAddWMProtocols( schedTl, &wm_delete_window, 1 );
    XmAddWMProtocolCallback( schedTl, wm_delete_window, 
                             (XtCallbackProc) closeCB, (XtPointer) sched_ref );
  } /* block */


  /* Container for the contents of the window. */
  workFo = XtNameToWidget( schedTl, "SchedTlBase.SchedTlFo" );

  XtAddCallback( schedTl,   XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) sched_ref );

  /* Create the menubar and menu cascades. */
  menuBr = XmCreateMenuBar( workFo, "MenuBr", args, 0 );

  n = 0;
  for( index = 0; index < XtNumber( pulldownMenu ); index++ )
    pulldownMenu[ index ] = XmCreatePulldownMenu( menuBr, 
                                                  pull_downs[ index ], 
                                                  args, n );

  for( index = 0; index < XtNumber( menuCasc ); index++ )
    menuCasc[ index ] = xitCreateCascadeButton( menuBr, 
                                                pulldownMenu[ index ], 
                                                &menupane[ index ] );

  /* The help button should be placed to the right. */
  index = XtNumber( menuCasc ) - 1;
  n     = 0;
  XtSetArg( args[ n ], XmNmenuHelpWidget, menuCasc[ index ] ); n++;
  XtSetValues( menuBr, args, n );


  /* Create the file menu. */
  XtAddCallback( pulldownMenu[ 0 ], XmNentryCallback, 
                 (XtCallbackProc) fileMenuCB, (XtPointer) sched_ref );
  XtAddCallback( pulldownMenu[ 0 ],  XmNmapCallback, 
                 (XtCallbackProc) xtmSaMapFileMenuCB, (XtPointer) sched_ref );

  for( index = 0; index < XtNumber( menuFileBu ); index++ ) {
    menuFileBu[ index ] = xitCreateMenuPushButton( pulldownMenu[ 0 ], 
                                                   &file_casc[ index ] );

    if( XmIsPushButton( menuFileBu[ index ] ) )
      XtAddCallback( menuFileBu[ index ], XmNactivateCallback, 
                     (XtCallbackProc) fileMenuCB, (XtPointer) index );
  }

  if( flagIsSet( sched_ref -> flags, XTM_SM_ONLY_ONE ) )
    XtSetSensitive( menuFileBu[ 0 ], False );


  /* Create the edit menu. */
  XtAddCallback( pulldownMenu[ 1 ],  XmNentryCallback, 
                 (XtCallbackProc) editMenuCB, (XtPointer) sched_ref );
  XtAddCallback( pulldownMenu[ 1 ],  XmNmapCallback, 
                 (XtCallbackProc) xtmSaMapEditMenuCB, (XtPointer) sched_ref );

  for( index = 0; index < XtNumber( menuEditBu ); index++ ) {
    menuEditBu[ index ] = xitCreateMenuPushButton( pulldownMenu[ 1 ], 
                                                   &edit_casc[ index ] );

    if( XmIsPushButton( menuEditBu[ index ] ) )
      XtAddCallback( menuEditBu[ index ], XmNactivateCallback, 
                     (XtCallbackProc) editMenuCB, (XtPointer) index );
  }


  /* Create the control menu. */
  XtAddCallback( pulldownMenu[ 2 ],  XmNentryCallback, 
                 (XtCallbackProc) ctrlMenuCB, (XtPointer) sched_ref );
  XtAddCallback( pulldownMenu[ 2 ],  XmNmapCallback, 
                 (XtCallbackProc) xtmSaMapCtrlMenuCB, (XtPointer) sched_ref );

  for( index = 0; index < XtNumber( menuCtrlBu ); index++ ) {
    menuCtrlBu[ index ] = xitCreateMenuPushButton( pulldownMenu[ 2 ], 
                                                   &ctrl_casc[ index ] );

    if( XmIsPushButton( menuCtrlBu[ index ] ) )
      XtAddCallback( menuCtrlBu[ index ], XmNactivateCallback, 
                     (XtCallbackProc) ctrlMenuCB, (XtPointer) index );
    else if( XmIsToggleButton( menuCtrlBu[ index ] ) )
      XtAddCallback( menuCtrlBu[ index ], XmNvalueChangedCallback, 
                     (XtCallbackProc) ctrlMenuCB, (XtPointer) index );
  }


  /* Create the options menu. */
  XtAddCallback( pulldownMenu[ 3 ], XmNentryCallback, 
                 (XtCallbackProc) optionsMenuCB, (XtPointer) sched_ref );

  for( index = 0; index < XtNumber( menuOptionsBu ); index++ ) {
    menuOptionsBu[ index ] = xitCreateMenuPushButton( pulldownMenu[ 3 ], 
                                                      &opt_casc[ index ] );

    if( XmIsPushButton( menuOptionsBu[ index ] ) )
      XtAddCallback( menuOptionsBu[ index ], XmNactivateCallback, 
                     (XtCallbackProc) optionsMenuCB, (XtPointer) index );
  }


  /* Create the help menu. */
  XtAddCallback( pulldownMenu[ 4 ], XmNentryCallback, 
                 (XtCallbackProc) infoCB, (XtPointer) sched_ref );

  for( index = 0; index < XtNumber( menuHelpBu ); index++ ) {
    menuHelpBu[ index ] = xitCreateMenuPushButton( pulldownMenu[ 4 ], 
                                                   &help_casc[ index ] );

    if( XmIsPushButton( menuHelpBu[ index ] ) )
      XtAddCallback( menuHelpBu[ index ], XmNactivateCallback, 
                     (XtCallbackProc) infoCB, (XtPointer) index );
  }

  /* We can't do context sensitive help. */
  XtSetSensitive( menuHelpBu[ 0 ], False );


  /* Create a select date form. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  dateRc = XmCreateRowColumn( workFo, "DateRc", args, n );

  dateLa = xitCreateLabel( dateRc, "DateLa",
                           msgGetText( MXDI_SCHED_BTW_DATES ), -1 );

  fromTb = xtmFlCreateDateField( dateRc, "FromTb" );
  XtAddCallback( fromTb,  XmNactivateCallback, 
                 (XtCallbackProc) applyCB, (XtPointer) sched_ref );

  dateInfoLa = xitCreateLabel( dateRc, "DateInfoLa", " ", -1 );


  /* Day display window. */
  n = 0;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmSTATIC ); n++;
  XtSetArg( args[ n ], XmNscrollingPolicy,        XmAUTOMATIC ); n++;
  XtSetArg( args[ n ], XmNscrollBarPlacement,     XmTOP_RIGHT ); n++;
  XtSetArg( args[ n ], XmNshadowThickness,        1 );  n++;
  XtSetArg( args[ n ], XmNspacing,                0 );  n++;
  dayDispSw = XmCreateScrolledWindow( workFo, "DayDispSw", args, n );

  n = 0;
  XtSetArg( args[ n ], XmNhorizontalScrollBar, &tempW ); n++;
  XtGetValues( dayDispSw, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNheight, 1 ); n++;
  XtSetArg( args[ n ], XmNmappedWhenManaged, False ); n++;
  XtSetValues( tempW, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNallowOverlap, True ); n++;
  XtSetArg( args[ n ], XmNmarginHeight, 0 ); n++;
  XtSetArg( args[ n ], XmNmarginWidth,  0 ); n++;
  XtSetArg( args[ n ], XmNresizePolicy, XmRESIZE_NONE ); n++;
  dayDispBb = XmCreateBulletinBoard( dayDispSw, "DayDispBb", args, n );

  tempW = xitCreateLabel( dayDispBb, "", " ", -1 );

  n = 0;
  XtSetArg( args[ n ], XmNheight, &height ); n++;
  XtGetValues( tempW, args, n );

  XtDestroyWidget( tempW );

  n = 0;
  XtSetArg( args[ n ], XmNheight, height + 10 ); n++;
  XtSetValues( dayDispSw, args, n );


  /* Paned window for notes and appointments. */
  n = 0;
  XtSetArg( args[ n ], XmNmarginWidth, 0 ); n++;
  schedPa = XmCreatePanedWindow( workFo, "SchedPa", args, n );


  /* Two form containers within the panes. */
  height = (Dimension) sched_ref -> appl_data_ref -> custom_data -> 
                         note_pane_height;
  width = (Dimension) sched_ref -> appl_data_ref -> custom_data -> 
                        entry_pane_width;

  n = 0;
  XtSetArg( args[ n ], XmNpaneMinimum, height ); n++;
  XtSetArg( args[ n ], XmNwidth,       width ); n++;
  pane1Fo = XmCreateForm( schedPa, "Pane1Fo", args, n );

  height = (Dimension) sched_ref -> appl_data_ref -> custom_data -> 
                         app_pane_height;
  n = 0;
  XtSetArg( args[ n ], XmNpaneMinimum, height ); n++;
  XtSetArg( args[ n ], XmNwidth,       width ); n++;
  pane2Fo = XmCreateForm( schedPa, "Pane2Fo", args, n );


  /* Start to build the note part of the paned window. */

  /* Create a note. */
  makeNotePb = xitCreatePushButton( pane1Fo, &button_def[ 1 ] );

  XtAddCallback( makeNotePb, XmNactivateCallback, 
                 (XtCallbackProc) makeNoteCB, (XtPointer) sched_ref );

  /* Scrolled window for the notes. */
  n = 0;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmSTATIC ); n++;
  XtSetArg( args[ n ], XmNscrollingPolicy,        XmAUTOMATIC ); n++;
  XtSetArg( args[ n ], XmNshadowThickness,        1 );  n++;
  noteSw = XmCreateScrolledWindow( pane1Fo, "NoteSw", args, n );

  n = 0;
  XtSetArg( args[ n ], XmNhorizontalScrollBar, &tempW ); n++;
  XtGetValues( noteSw, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNheight, 1 ); n++;
  XtSetArg( args[ n ], XmNmappedWhenManaged, False ); n++;
  XtSetValues( tempW, args, n );

  /* Bulletin board for row columns with notes. */
  n = 0;
  XtSetArg( args[ n ], XmNallowOverlap, True ); n++;
  XtSetArg( args[ n ], XmNmarginHeight, 0 ); n++;
  XtSetArg( args[ n ], XmNmarginWidth,  0 ); n++;
  XtSetArg( args[ n ], XmNresizePolicy, XmRESIZE_NONE ); n++;
  noteBb = XmCreateBulletinBoard( noteSw, "NoteBb", args, n );

  if( flagIsClear( sched_ref -> flags, XTM_SM_READ_ONLY ) ) {
    XtAddEventHandler( noteBb, ButtonPressMask, False,
                       (XtEventHandler) xtmSaActionNoteHook,
                       (XtPointer) sched_ref );
    XtAddEventHandler( noteBb, ButtonMotionMask, False,
                       (XtEventHandler) xtmSaTrackAnimation,
                       (XtPointer) sched_ref );
    XtAddEventHandler( noteBb, ButtonReleaseMask, False,
                       (XtEventHandler) xtmSaStopAnimation,
                       (XtPointer) sched_ref );
  }

  /* Bottom delimiter in the notes window. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL );  n++;
  noteBottomSp = XmCreateSeparator( noteBb, "NoteBottomSp", args, n );
  

  /* Start to build the appointments part of the paned window. */

  /* Create an appointment. */
  makeAppPb = xitCreatePushButton( pane2Fo, &button_def[ 0 ] );

  XtAddCallback( makeAppPb, XmNactivateCallback, 
                 (XtCallbackProc) makeAppointmentCB, (XtPointer) sched_ref );

  /* Create the 'zoom' field. */
  zoomFs = xtmFlCreateSelField( pane2Fo, "ZoomFs", 4, zoom_times,
                                ' ', True,
                                (XtCallbackProc) zoomCB, (void *) sched_ref );

  tempW = xitFieldSelectGetChild( zoomFs, xitFIELD_SELECT_TEXT_FIELD );

  n = 0;
  XtSetArg( args[ n ], XmNeditable, False ); n++;
  XtSetArg( args[ n ], XmNcursorPositionVisible, False ); n++;
  XtSetValues( tempW, args, n );

  sprintf( buffer, "%1d:%2.2d",
           sched_ref -> entry_delta / 60,
           sched_ref -> entry_delta % 60 ); 

  xitFieldSelectSetCurrent( zoomFs, buffer, False );

  /* Some text saying what this is all about. */
  appLa = xitCreateLabel( pane2Fo, "AppLa", 
                          msgGetText( MXDI_APPOINTMENTS_PANE ), -1 );

  /* Label to use during animation. */
  trackPx = xitCreateLabel( pane2Fo, "TrackPx", " " , -1 );

  n = 0;
  XtSetArg( args[ n ], XmNlabelType, XmPIXMAP ); n++;
  XtSetValues( trackPx, args, n );

  trackLa = xitCreateLabel( pane2Fo, "TrackLa", 
                            "                          ", -1 );

  /* Time display window. */
  n = 0;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmSTATIC ); n++;
  XtSetArg( args[ n ], XmNscrollingPolicy, XmAUTOMATIC ); n++;
  XtSetArg( args[ n ], XmNscrollBarPlacement, XmBOTTOM_LEFT ); n++;
  XtSetArg( args[ n ], XmNshadowThickness, 1 );  n++;
  timeDispSw = XmCreateScrolledWindow( pane2Fo, "TimeDispSw", args, n );

  n = 0;
  XtSetArg( args[ n ], XmNverticalScrollBar, &tempW ); n++;
  XtGetValues( timeDispSw, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNwidth, 1 ); n++;
  XtSetArg( args[ n ], XmNmappedWhenManaged, False ); n++;
  XtSetValues( tempW, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNallowOverlap, True ); n++;
  XtSetArg( args[ n ], XmNmarginHeight, 0 ); n++;
  XtSetArg( args[ n ], XmNmarginWidth,  0 ); n++;
  timeDispBb = XmCreateBulletinBoard( timeDispSw, "TimeDispBb", args, n );

  /* Label with the times. */
  timeDispLa = xitCreateLabelWidget( timeDispBb, "TimeDispLa", 
                                     "        ", -1 );

  /* Read only mode? */
  if( flagIsClear( sched_ref -> flags, XTM_SM_READ_ONLY ) ) {
    XtAddEventHandler( timeDispLa, ButtonPressMask,   False,
                       (XtEventHandler) timeActionHookCB,
                       (XtPointer) sched_ref );
    XtAddEventHandler( timeDispLa, ButtonMotionMask, False,
                       (XtEventHandler) xtmSaTrackAnimation,
                       (XtPointer) sched_ref );
    XtAddEventHandler( timeDispLa, ButtonReleaseMask, False,
                       (XtEventHandler) xtmSaStopAnimation,
                       (XtPointer) sched_ref );
  }

  n = 0;
  XtSetArg( args[ n ], XmNx, 10 ); n++;
  XtSetArg( args[ n ], XmNy, 0 ); n++;
  XtSetValues( timeDispLa, args, n );

  /* Current time indicator (do not manage yet). */
  timeIndicatorAr = xitCreateArrowPushButton( timeDispBb,
                                              &time_ind_def[ 0 ] );
  n = 0;
  XtSetArg( args[ n ], XmNshadowThickness, 0 ); n++;
  XtSetArg( args[ n ], XmNtraversalOn, False ); n++;
  XtSetArg( args[ n ], XmNhighlightThickness, 0 ); n++;
  XtSetValues( timeIndicatorAr, args, n );


  /* Size of the time window. */
  n = 0;
  XtSetArg( args[ n ], XmNwidth, &width ); n++;
  XtGetValues( timeDispLa, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNwidth, width + 20 ); n++;
  XtSetValues( timeDispSw, args, n );

  /* Window for the appointments. */
  n = 0;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmSTATIC ); n++;
  XtSetArg( args[ n ], XmNscrollingPolicy, XmAUTOMATIC ); n++;
  XtSetArg( args[ n ], XmNshadowThickness, 1 );  n++;
  entrySw = XmCreateScrolledWindow( pane2Fo, "EntrySw", args, n );

  n = 0;
  XtSetArg( args[ n ], XmNhorizontalScrollBar, &tempW ); n++;
  XtGetValues( entrySw, args, n );

  XtAddCallback( tempW, XmNincrementCallback, 
                 (XtCallbackProc) xtmSwEntryScrolledHorizCB,
                 (XtPointer) sched_ref );
  XtAddCallback( tempW, XmNdecrementCallback, 
                 (XtCallbackProc) xtmSwEntryScrolledHorizCB,
                 (XtPointer) sched_ref );
  XtAddCallback( tempW, XmNpageIncrementCallback, 
                 (XtCallbackProc) xtmSwEntryScrolledHorizCB,
                 (XtPointer) sched_ref );
  XtAddCallback( tempW, XmNpageDecrementCallback, 
                 (XtCallbackProc) xtmSwEntryScrolledHorizCB,
                 (XtPointer) sched_ref );
  XtAddCallback( tempW, XmNvalueChangedCallback, 
                 (XtCallbackProc) xtmSwEntryScrolledHorizCB,
                 (XtPointer) sched_ref );
  XtAddCallback( tempW, XmNdragCallback, 
                 (XtCallbackProc) xtmSwEntryScrolledHorizCB,
                 (XtPointer) sched_ref );

  n = 0;
  XtSetArg( args[ n ], XmNverticalScrollBar, &tempW ); n++;
  XtGetValues( entrySw, args, n );

  XtAddCallback( tempW, XmNincrementCallback, 
                 (XtCallbackProc) xtmSwEntryScrolledVertCB,
                 (XtPointer) sched_ref );
  XtAddCallback( tempW, XmNdecrementCallback, 
                 (XtCallbackProc) xtmSwEntryScrolledVertCB,
                 (XtPointer) sched_ref );
  XtAddCallback( tempW, XmNpageIncrementCallback, 
                 (XtCallbackProc) xtmSwEntryScrolledVertCB,
                 (XtPointer) sched_ref );
  XtAddCallback( tempW, XmNpageDecrementCallback, 
                 (XtCallbackProc) xtmSwEntryScrolledVertCB,
                 (XtPointer) sched_ref );
  XtAddCallback( tempW, XmNvalueChangedCallback, 
                 (XtCallbackProc) xtmSwEntryScrolledVertCB,
                 (XtPointer) sched_ref );
  XtAddCallback( tempW, XmNdragCallback, 
                 (XtCallbackProc) xtmSwEntryScrolledVertCB,
                 (XtPointer) sched_ref );

  n = 0;
  XtSetArg( args[ n ], XmNallowOverlap, True ); n++;
  XtSetArg( args[ n ], XmNmarginHeight, 0 ); n++;
  XtSetArg( args[ n ], XmNmarginWidth,  0 ); n++;
  XtSetArg( args[ n ], XmNresizePolicy, XmRESIZE_NONE ); n++;
  entryBb = XmCreateBulletinBoard( entrySw, "EntryBb", args, n );

  if( flagIsClear( sched_ref -> flags, XTM_SM_READ_ONLY ) ) {
    XtAddEventHandler( entryBb, ButtonPressMask, False,
                       (XtEventHandler) xtmSaActionAppHook,
                       (XtPointer) sched_ref );
    XtAddEventHandler( entryBb, ButtonMotionMask, False,
                       (XtEventHandler) xtmSaTrackAnimation,
                       (XtPointer) sched_ref );
    XtAddEventHandler( entryBb, ButtonReleaseMask, False,
                       (XtEventHandler) xtmSaStopAnimation, 
                       (XtPointer) sched_ref );
  }

  /* Bottom delimiter in the entry window. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL );  n++;
  entryBottomSp = XmCreateSeparator( entryBb, "EntryBottomSp", args, n );
  

  /* Create the navigate buttons. */
  n = 0;
  navFo = XmCreateForm( workFo, "NavFo", args, n );


  /* Create next day action. */
  xstr = XmStringCreateLtoR( msgGetText( MXDI_DAY_LABEL ), CS );

  n = 0;
  XtSetArg( args[ n ], XmNlabelString, xstr ); n++;
  navDayAl = XmUbCreateArrowLabel( navFo, "NavDayAl", args, n );

  XmStringFree( xstr );

  XtAddCallback( navDayAl, XmNactivateCallback,
                 (XtCallbackProc) navigateCB, (XtPointer) sched_ref );


  /* Create next week action. */
  xstr = XmStringCreateLtoR( msgGetText( MXDI_WEEK_LABEL ), CS );

  n = 0;
  XtSetArg( args[ n ], XmNlabelString, xstr ); n++;
  navWeekAl = XmUbCreateArrowLabel( navFo, "NavWeekAl", args, n );

  XmStringFree( xstr );

  XtAddCallback( navWeekAl, XmNactivateCallback,
                 (XtCallbackProc) navigateCB, (XtPointer) sched_ref );


  /* Create next month action. */
  xstr = XmStringCreateLtoR( msgGetText( MXDI_MONTH_LABEL ), CS );

  n = 0;
  XtSetArg( args[ n ], XmNlabelString, xstr ); n++;
  navMonthAl = XmUbCreateArrowLabel( navFo, "NavMonthAl", args, n );

  XmStringFree( xstr );

  XtAddCallback( navMonthAl, XmNactivateCallback,
                 (XtCallbackProc) navigateCB, (XtPointer) sched_ref );


  /* Create next year action. */
  xstr = XmStringCreateLtoR( msgGetText( MXDI_YEAR_LABEL ), CS );

  n = 0;
  XtSetArg( args[ n ], XmNlabelString, xstr ); n++;
  navYearAl = XmUbCreateArrowLabel( navFo, "NavYearAl", args, n );

  XmStringFree( xstr );

  XtAddCallback( navYearAl, XmNactivateCallback,
                 (XtCallbackProc) navigateCB, (XtPointer) sched_ref );


  /* Fetch dimensions to use for offset values. */
  n = 0;
  XtSetArg( args[ n ], XmNhorizontalScrollBar, &tempW ); n++;
  XtGetValues( entrySw, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNheight, &sb_height ); n++;
  XtGetValues( tempW, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNwidth, &time_width ); n++;
  XtGetValues( timeDispSw, args, n );


  /* Set offsets for the components. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    10 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( dateRc, args, n );

  /* Day names window. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,  time_width + 6 ); n++;
  XtSetArg( args[ n ], XmNrightOffset, sb_height + 18 ); n++;
  XtSetValues( dayDispSw, args, n );

  /* Pane window. */
  n = 0;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( schedPa, args, n );

  /* Navigate form. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetValues( navFo, args, n );

  /* Navigate buttons. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,     5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   10 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  10 ); n++;
  XtSetValues( navDayAl,   args, n );
  XtSetValues( navWeekAl,  args, n );
  XtSetValues( navMonthAl, args, n );
  XtSetValues( navYearAl,  args, n );

  /* Notes window. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    0 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   time_width ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  0 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 0 ); n++;
  XtSetValues( noteSw, args, n );

  /* Zoom button. */
  n = 0;
  XtSetArg( args[ n ], XmNleftOffset,   2 ); n++;
  XtSetValues( zoomFs, args, n );

  /* New entry button. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset, 2 ); n++;
  XtSetValues( makeAppPb, args, n );

  /* Time display window. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   0 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  0 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, sb_height ); n++;
  XtSetValues( timeDispSw, args, n );

  /* Entry display window. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   0 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  0 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 0 ); n++;
  XtSetValues( entrySw, args, n );


  /* Parts in the notes pane. */
  xitAttachWidget( noteSw,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL );

  /* Parts in the appointments pane. */
  xitAttachWidget( zoomFs,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( appLa,
                   XmATTACH_OPPOSITE_WIDGET, zoomFs,
                   XmATTACH_WIDGET, timeDispSw,
                   XmATTACH_NONE, NULL, XmATTACH_OPPOSITE_WIDGET, zoomFs );
  xitAttachWidget( trackPx,
                   XmATTACH_OPPOSITE_WIDGET, zoomFs,
                   XmATTACH_WIDGET, appLa,
                   XmATTACH_NONE, NULL, XmATTACH_OPPOSITE_WIDGET, zoomFs );
  xitAttachWidget( trackLa,
                   XmATTACH_OPPOSITE_WIDGET, zoomFs,
                   XmATTACH_WIDGET, trackPx,
                   XmATTACH_NONE, NULL, XmATTACH_OPPOSITE_WIDGET, zoomFs );
  xitAttachWidget( makeAppPb,
                   XmATTACH_WIDGET, zoomFs, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,   XmATTACH_NONE, NULL );
  xitAttachWidget( timeDispSw,
                   XmATTACH_WIDGET, zoomFs, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,   XmATTACH_FORM, NULL );
  xitAttachWidget( entrySw,
                   XmATTACH_WIDGET, zoomFs, XmATTACH_WIDGET, timeDispSw,
                   XmATTACH_FORM, NULL,     XmATTACH_FORM,   NULL );

  /* Parts in the navigate window. */
  xitAttachWidget( navDayAl,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( navWeekAl,
                   XmATTACH_FORM, NULL, XmATTACH_WIDGET, navDayAl,
                   XmATTACH_NONE, NULL, XmATTACH_NONE,   NULL );
  xitAttachWidget( navMonthAl,
                   XmATTACH_FORM, NULL, XmATTACH_WIDGET, navWeekAl,
                   XmATTACH_NONE, NULL, XmATTACH_NONE,   NULL );
  xitAttachWidget( navYearAl,
                   XmATTACH_FORM, NULL, XmATTACH_WIDGET, navMonthAl,
                   XmATTACH_NONE, NULL, XmATTACH_NONE,   NULL );

  /* Parts in the main window. */
  xitAttachWidget( menuBr,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_FORM, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( dateRc,
                   XmATTACH_WIDGET, menuBr, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,   XmATTACH_NONE, NULL );
  xitAttachWidget( dayDispSw,
                   XmATTACH_WIDGET, dateRc, XmATTACH_FORM, NULL,
                   XmATTACH_FORM,   NULL,   XmATTACH_NONE, NULL );
  xitAttachWidget( schedPa,
                   XmATTACH_WIDGET, dayDispSw, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );
  xitAttachWidget( navFo,
                   XmATTACH_WIDGET, schedPa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,    XmATTACH_NONE, NULL );


  /* Manage all the children. */
  XtManageChildren( menuCasc,      XtNumber( menuCasc ) );
  XtManageChildren( menuCtrlBu,    XtNumber( menuCtrlBu ) );
  XtManageChildren( menuEditBu,    XtNumber( menuEditBu ) );
  XtManageChildren( menuFileBu,    XtNumber( menuFileBu ) );
  XtManageChildren( menuHelpBu,    XtNumber( menuHelpBu ) );
  XtManageChildren( menuOptionsBu, XtNumber( menuOptionsBu ) );

  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );


  /* We don't display all scroll bars. */
  n = 0;
  XtSetArg( args[ n ], XmNhorizontalScrollBar, &tempW ); n++;
  XtGetValues( timeDispSw, args, n );
  XtUnmanageChild( tempW );

  n = 0;
  XtSetArg( args[ n ], XmNverticalScrollBar, &tempW ); n++;
  XtGetValues( dayDispSw, args, n );
  XtUnmanageChild( tempW );


  /* Set icon for this window. */
  xtmIcSetSimpleIcon( schedTl, workFo, XTM_IC_ICON_SCHEDULE );

  /* Set the size of the window. */
  xitSetSizeToplevelDialog( schedTl, True );


  /* Make the final attachments. */
  xitAttachWidget( navFo,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL );
  xitAttachWidget( schedPa,
                   XmATTACH_WIDGET, dayDispSw, XmATTACH_FORM,   NULL,
                   XmATTACH_FORM,   NULL,      XmATTACH_WIDGET, navFo );

  n = 0;
  XtSetArg( args[ n ], XmNpaneMinimum, 1 ); n++;
  XtSetValues( pane1Fo, args, n );
  XtSetValues( pane2Fo, args, n );


  return( schedTl );

} /* createScheduleWindow */


/*----------------------------------------------------------------------*/

static void
  displayEntryInfo( SCHED_REC_REF  sched_ref )
{

  /* Variables. */
  Boolean                 ok;
  char                    buffer[ 50 ];
  char                    entry_type[ 50 ];
  char                    last_changed[ 50 ];
  char                    last_changed_by[ 100 ];
  char                    message[ 500 ];
  ENTRY_INFO              *entry_info_ref;
  TIM_TIME_REF            last_update;
  XTM_DB_ALL_ENTRY_DEF    entry_record;
  XTM_DB_ENTRY_DATABASES  database;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  struct passwd           *password_ref;


  /* Code. */

  custom_data_ref = sched_ref -> appl_data_ref -> custom_data;

  /* Selected entry. */
  entry_info_ref = xtmStFetchSelectedInfo( sched_ref );
  if( entry_info_ref == NULL )
    return;

  /* Open the database and fetch the entry. */
  ok = xtmDmOpenDatabase( sched_ref -> appl_data_ref,
                          entry_info_ref -> db_name, XTM_DB_FLAG_MODE_READ,
                          &database );
  if( ! ok )
    return;

  ok = xtmDmFetchEntry( sched_ref -> scheduleW,
                        &database, entry_info_ref -> id,
                        &entry_record, NULL );

  xtmDbCloseEntryDb( &database );
  if( ! ok )
    return;


  /* Type of entry? */
  entry_type[ 0 ] = '\0';

  if( entry_record.entry.entry_type == XTM_DB_DAY_ENTRY ) {
    if( entry_record.entry.entry_category == XTM_DB_REP_ENTRY_LIST )
      strcpy( entry_type, msgGetText( MXDI_TYPE_STAND_APPOINT ) );
    else
      strcpy( entry_type, msgGetText( MXDI_TYPE_APPOINT ) );
  }

  if( entry_record.entry.entry_type == XTM_DB_DAY_NOTE ) {
    if( entry_record.entry.entry_category == XTM_DB_REP_ENTRY_LIST )
      strcpy( entry_type, msgGetText( MXDI_TYPE_STAND_NOTE ) );
    else if( entry_record.entry.entry_category == XTM_DB_STICKY_LIST )
      strcpy( entry_type, msgGetText( MXDI_TYPE_STICKY_NOTE ) );
    else
      strcpy( entry_type, msgGetText( MXDI_TYPE_NOTE ) );
  }


  /* Last changed. */
  last_update = TimLocalTime( (TIM_TIME_REF) entry_record.entry.last_update );

  xtmFoFormatDate( last_update, buffer, sizeof( buffer ) );
  sprintf( last_changed, "%s ", buffer );

  xtmFoFormatTime( last_update, buffer, sizeof( buffer ) );
  strcat( last_changed, buffer );


  /* Last changed by. */
  password_ref = getpwuid( entry_record.entry.owner );
  if( password_ref != NULL )
    strcpy( last_changed_by, password_ref -> pw_name );
  else
    strcpy( last_changed_by, "-" );


  /* Display the information. */
  sprintf( message, msgGetText( MXDI_ENTRY_INFO_TEMPLATE ),
           entry_type, entry_record.db_name,
           last_changed, last_changed_by );

  (void) xitCreateInformationDialog( 
           sched_ref -> scheduleW, "InformationDialog", 
	   msgGetText( MXDI_INFORMATION_LABEL ),
           message,
           NULL, NULL );


  return;

} /* displayEntryInfo */


/*----------------------------------------------------------------------*/

static void 
  applyCB( Widget         widget,
           SCHED_REC_REF  sched_ref,
           XtPointer      call_data )
{

  /* Code. */

  /* Update the start and stop dates. */
  xtmSwSetScheduleDates( sched_ref, widget, True );

  /* Display the new schedule. */
  xtmSwSetSchedule( sched_ref, widget );


  return;

} /* applyCB */


/*----------------------------------------------------------------------*/

static void 
  closeCB( Widget         widget,
           SCHED_REC_REF  sched_ref,
           XtPointer      call_data )
{

  /* Variables. */
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = sched_ref -> appl_data_ref -> custom_data;

  /* Do we really want this? */
  if( flagIsSet( sched_ref -> flags, XTM_SM_ONLY_ONE ) &&
      custom_data_ref -> confirm_actions ) {

    XRaiseWindow( XtDisplay( sched_ref -> scheduleW ), 
                  XtWindow(  sched_ref -> scheduleW ) );

    XtMapWidget( sched_ref -> scheduleW );

    (void) xitCreateQuestionDialog( 
             sched_ref -> scheduleW, "QuestionDialog", 
  	     msgGetText( MXDI_QUESTION_MESSAGE_LABEL ),
             msgGetText( MXDI_SCHED_CONFIRM_CLOSE ),
             closeReallyCB, sched_ref,
             NULL,          NULL );

  } else {
    closeReallyCB( widget, sched_ref, NULL );
  }


  return;

} /* closeCB */


/*----------------------------------------------------------------------*/

static void 
  closeReallyCB( Widget         widget,
                 SCHED_REC_REF  sched_ref,
                 XtPointer      call_data )
{

  /* Variables. */
  int     index;
  Widget  tempW;


  /* Code. */

  /* Do we have a user action callback registered? */
  if( sched_ref -> actionCB != NULL )
    (* sched_ref -> actionCB)( XTM_SM_REASON_EXIT,
                               sched_ref -> user_data );

  /* Not last active window. */
  if( last_active_sched_ref == sched_ref )
    last_active_sched_ref = NULL;

  /* No focus callback. */
#ifndef XD_HAS_FOCUS_BUG
  tempW = XtNameToWidget( sched_ref -> scheduleW, "SchedTlBase" );
  XtRemoveCallback( tempW, XmNfocusCallback,
                    (XtCallbackProc) gotFocusCB, (XtPointer) sched_ref );
#endif


  /* Keep the window in cache? */
  for( index = 0; index < MAX_CACHE_ENTRIES; index++ ) {
    if( cache_entries[ index ] == NULL ) {
      cache_entries[ index ] = sched_ref;

      destroyCB( NULL, sched_ref, NULL );
      XtPopdown( sched_ref -> scheduleW );

      return;
    }
  }


  /* Window not kept in cache, really remove it. */
  XtDestroyWidget( sched_ref -> scheduleW );


  return;

} /* closeReallyCB */


/*----------------------------------------------------------------------*/

static void 
  ctrlMenuCB( Widget                     widget,
              SCHED_REC_REF              sched_ref,
              XmRowColumnCallbackStruct  *call_data )
{

  /* Variables. */
  Widget                  mainW;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = sched_ref -> appl_data_ref -> custom_data;

  mainW = XtNameToWidget( sched_ref -> scheduleW, "SchedTlBase.SchedTlFo" );

  /* Select what to do. */
  switch( (int) call_data -> data ) {

    /* Confirm actions? */
    case 0:
      if( XmToggleButtonGetState( call_data -> widget ) ) {
        flagSet( sched_ref -> flags, XTM_SM_CONFIRM );
        custom_data_ref -> confirm_actions = True;
      } else {
        flagClear( sched_ref -> flags, XTM_SM_CONFIRM );
        custom_data_ref -> confirm_actions = False;
      }
      break;


    /* Entry handles? */
    case 1:
      if( XmToggleButtonGetState( call_data -> widget ) )
        flagSet( sched_ref -> flags, XTM_SM_ENTRY_HANDLES );
      else
        flagClear( sched_ref -> flags, XTM_SM_ENTRY_HANDLES );

      if( flagIsClear( sched_ref -> flags, XTM_SM_LIST_LAYOUT ) )
        xtmUpDoUpdate( XTM_UP_SCHEDULE, NULL );
      break;


    /* List layout? */
    case 2:
      if( XmToggleButtonGetState( call_data -> widget ) )
        flagSet( sched_ref -> flags, XTM_SM_LIST_LAYOUT );
      else
        flagClear( sched_ref -> flags, XTM_SM_LIST_LAYOUT );

      sched_ref -> force_update = True;
      xtmUpDoUpdate( XTM_UP_SCHEDULE, NULL );
      break;


    /* Use grid? */
    case 3:
      if( XmToggleButtonGetState( call_data -> widget ) )
        flagSet( sched_ref -> flags, XTM_SM_USE_GRID );
      else
        flagClear( sched_ref -> flags, XTM_SM_USE_GRID );
      break;


    /* Display entry flags? */
    case 4:
      if( XmToggleButtonGetState( call_data -> widget ) )
        flagSet( sched_ref -> flags, XTM_SM_ENTRY_FLAGS );
      else
        flagClear( sched_ref -> flags, XTM_SM_ENTRY_FLAGS );

      sched_ref -> force_update = True;
      xtmUpDoUpdate( XTM_UP_SCHEDULE, NULL );
      break;


    /* True color for included entries? */
    case 5:
      if( XmToggleButtonGetState( call_data -> widget ) )
        flagSet( sched_ref -> flags, XTM_SM_TRUE_COLOR_INC );
      else
        flagClear( sched_ref -> flags, XTM_SM_TRUE_COLOR_INC );

      xtmUpDoUpdate( XTM_UP_SCHEDULE, NULL );
      break;


    /* Days to display? */
    case 7:
      xtmSwSelectScaleValue( sched_ref, SCHED_DISP_DAYS );
      break;


    /* Width of displayed days? */
    case 8:
      xtmSwSelectScaleValue( sched_ref, SCHED_DAY_WIDTH );
      break;


    /* Filter database. */
    case 9:
      /* Initialize the filter? */
      if( sched_ref -> filter_handle == NULL )
        sched_ref -> filter_handle = xtmFiInitialize(
                                       sched_ref -> appl_data_ref,
                                       sched_ref -> scheduleW,
                                       custom_data_ref -> entry_tags_menu,
                                       False,
                                       filterApplyCB, (void *) sched_ref );

      /* Display the filter window. */
      xtmFiDisplayFilterWindow( sched_ref  -> filter_handle,
                                &sched_ref -> filter_rec );
      break;


    /* Include database. */
    case 10:
      /* Initialize include. */
      if( sched_ref -> include_handle == NULL )
        sched_ref -> include_handle = xtmDiInitialize(
                                        sched_ref -> appl_data_ref,
                                        sched_ref -> scheduleW,
                                        False,
                                        includeDbApplyCB,
                                        (void *) sched_ref );

      /* Display the include window. */
      xtmDiDisplayIncludeWindow( sched_ref -> include_handle,
                                 sched_ref -> db_name );
      break;


    /* All standing entries? */
    case 11:
      /* Initialize the window? */
      if( sched_ref -> sel_hide_handle == NULL )
        sched_ref -> sel_hide_handle = xtmShInitialize(
                                         sched_ref -> appl_data_ref,
                                         sched_ref -> scheduleW,
                                         False, 
                                         selHideApplyCB,
                                         (void *) sched_ref );

      /* Display the window. */
      xtmShDisplaySelectWindow( sched_ref -> sel_hide_handle,
                                sched_ref -> db_name,
                                sched_ref -> schedule_start );
      break;

  } /* switch */


  return;

} /* ctrlMenuCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget         widget,
             SCHED_REC_REF  sched_ref,
             XtPointer      call_data )
{

  /* Variables. */
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = sched_ref -> appl_data_ref -> custom_data;


  /* Do we have a user action callback registered? */
  if( sched_ref -> actionCB != NULL )
    (* sched_ref -> actionCB)( XTM_SM_REASON_DESTROY, 
                               sched_ref -> user_data );


  /* De-register updates. */
  if( sched_ref -> update_id != 0 )
    xtmUpRemove( sched_ref -> update_id );

  sched_ref -> update_id = 0;


  /* Destroy child windows. */
  if( sched_ref -> editor_handle != NULL ) {
    xtmEdDestroy( sched_ref -> editor_handle );
    sched_ref -> editor_handle = NULL;
  }

  if( sched_ref -> filter_handle != NULL ) {
    xtmFiDestroy( sched_ref -> filter_handle );
    sched_ref -> filter_handle = NULL;
  }

  if( sched_ref -> include_handle != NULL ) {
    xtmDiDestroy( sched_ref -> include_handle );
    sched_ref -> include_handle = NULL;
  }

  if( sched_ref -> pr_handle != NULL ) {
    xtmPrDestroy( sched_ref -> pr_handle );
    sched_ref -> pr_handle = NULL;
  }

  if( sched_ref -> nav_cal_handle != NULL ) {
    xtmNcDestroy( sched_ref -> nav_cal_handle );
    sched_ref -> nav_cal_handle = NULL;
  }

  if( sched_ref -> open_handle != NULL ) {
    xtmOvDestroy( sched_ref -> open_handle );
    sched_ref -> open_handle = NULL;
  }

  if( sched_ref -> planner_handle != NULL ) {
    xtmPlDestroy( sched_ref -> planner_handle );
    sched_ref -> planner_handle = NULL;
  }

  if( sched_ref -> show_handle != NULL ) {
    xtmSdDestroy( sched_ref -> show_handle );
    sched_ref -> show_handle = NULL;
  }

  if( sched_ref -> sel_hide_handle != NULL ) {
    xtmShDestroy( sched_ref -> sel_hide_handle );
    sched_ref -> sel_hide_handle = NULL;
  }


  /* Free the shadow calendar. */
  xtmCdFreeShadowEntry( custom_data_ref -> cal_db_handle,
                        sched_ref -> db_name );


  /* Release the user data (only if not cached). */
  if( widget != NULL ) {

    int              index;
    ENTRY_CACHE_REF  cache_ref;

    /* Entry cache. */
    cache_ref = sched_ref -> entry_cache_ref;

    for( index = 0; index < cache_ref -> max_used; index++ ) {
      if( cache_ref -> used[ index ] == NULL )
        continue;

      if( cache_ref -> used[ index ] -> entry_text != NULL )
        SysFree( cache_ref -> used[ index ] -> entry_text );

      SysFree( cache_ref -> used[ index ] );
    }

    SysFree( sched_ref -> entry_cache_ref -> used );
    SysFree( sched_ref -> entry_cache_ref );


    /* Animation record. */
    if( sched_ref -> animate_ref != NULL )
      SysFree( sched_ref -> animate_ref );


    /* Main record. */
    SysFree( sched_ref );

  } /* if */


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void 
  gotFocusCB( Widget               widget,
              SCHED_REC_REF        sched_ref,
              XmAnyCallbackStruct  *call_data )
{

  /* Code. */

  last_active_sched_ref = sched_ref;


  return;

} /* gotFocusCB */


/*----------------------------------------------------------------------*/

static void 
  editMenuCB( Widget                     widget,
              SCHED_REC_REF              sched_ref,
              XmRowColumnCallbackStruct  *call_data )
{

  /* Variables. */
  Widget                  mainW;
  ENTRY_INFO              *entry_info_ref;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  appl_data_ref   = sched_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;

  mainW = XtNameToWidget( sched_ref -> scheduleW, "SchedTlBase.SchedTlFo" );

  entry_info_ref = xtmStFetchSelectedInfo( sched_ref );


  /* Select what to do. */
  switch( (int) call_data -> data ) {

    /* Edit the entry. */
    case 0:
      xtmSaAppointmentEditorShow( sched_ref );
      break;


    /* Delete the entry. */
    case 1:
      xtmDlDeleteEntry( appl_data_ref, sched_ref -> scheduleW,
                        entry_info_ref -> db_name, 
                        entry_info_ref -> id );
      break;


    /* Move the entry. */
    case 2:
      xtmCmCopyMoveEntry( appl_data_ref, sched_ref -> scheduleW,
                          XTM_CM_MOVE,
                          entry_info_ref -> db_name, 
                          entry_info_ref -> id );
      break;


    /* Copy the entry. */
    case 3:
      xtmCmCopyMoveEntry( appl_data_ref, sched_ref -> scheduleW,
                          XTM_CM_COPY,
                          entry_info_ref -> db_name, 
                          entry_info_ref -> id );
      break;


    /* Duplicate the entry. */
    case 4:
      xtmDpDuplicateEntry( appl_data_ref, sched_ref -> scheduleW,
                           entry_info_ref -> db_name, 
                           entry_info_ref -> id );
      break;


    /* Archive the entry. */
    case 5:
      xtmArArchiveEntry( appl_data_ref, sched_ref -> scheduleW,
                         entry_info_ref -> db_name, 
                         entry_info_ref -> id );
      break;


    /* Freeze a repeated entry. */
    case 6:
      xtmFzFreezeEntry( appl_data_ref, sched_ref -> scheduleW,
                        entry_info_ref -> db_name, 
                        entry_info_ref -> id,
                        entry_info_ref -> date_stamp );
      break;


    /* Selected entry information. */
    case 7:
      displayEntryInfo( sched_ref );
      break;


    /* Copy entry to buffer. */
    case 9:
      xtmSaDoCutBuffer( sched_ref, XTM_SA_DO_COPY,
                        entry_info_ref -> db_name, 
                        entry_info_ref -> id );
      break;


    /* Cut entry to buffer. */
    case 10:
      xtmSaDoCutBuffer( sched_ref, XTM_SA_DO_CUT,
                        entry_info_ref -> db_name, 
                        entry_info_ref -> id );
      break;


    /* Paste entry from buffer. */
    case 11:
      xtmSaDoCutBuffer( sched_ref, XTM_SA_DO_PASTE,
                        sched_ref -> db_name,
                        0 );
      break;

  } /* switch */

  /* The entry is not selected. */
  xtmSwUnselectEntry( sched_ref );


  return;

} /* editMenuCB */


/*----------------------------------------------------------------------*/

static void 
  fileMenuCB( Widget                     widget,
              SCHED_REC_REF              sched_ref,
              XmRowColumnCallbackStruct  *call_data )
{

  /* Variables. */
  Widget                  mainW;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_SM_HANDLE           sched_handle;
  XTM_SM_INIT_REC         sched_init;


  /* Code. */

  appl_data_ref   = sched_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;

  mainW = XtNameToWidget( sched_ref -> scheduleW, "SchedTlBase.SchedTlFo" );


  /* Select what to do. */
  switch( (int) call_data -> data ) {

    /* Open schedule with the same database. */
    case 0:
      sched_init.entry_start   = sched_ref -> entry_start;
      sched_init.entry_stop    = sched_ref -> entry_stop;
      sched_init.entry_delta   = sched_ref -> entry_delta;
      sched_init.default_flags = sched_ref -> flags;

      sched_handle = xtmSmInitialize( appl_data_ref,
                                      appl_data_ref -> toplevel,
                                      sched_ref -> db_name, &sched_init,
                                      NULL, NULL );
      if( sched_handle == NULL )
        return;

      /* Diaplay the schedule window. */
      xtmSmViewSchedule( sched_handle, sched_ref -> schedule_start, False );
      break;


    /* Select a new database view. */
    case 1:
      if( sched_ref -> open_handle == NULL )
        sched_ref -> open_handle = xtmOvInitialize(
                                     appl_data_ref,
                                     sched_ref -> scheduleW,
                                     False,
                                     newViewApplyCB,
                                     (void *) sched_ref );

      if( sched_ref -> open_handle == NULL )
        return;

      /* Display the window. */
      xtmOvOpenView( sched_ref -> open_handle );
      break;


    /* Print window. */
    case 2:
      if( sched_ref -> pr_handle == NULL )
        sched_ref -> pr_handle = xtmPrInitialize( 
                                   appl_data_ref, 
                                   appl_data_ref -> toplevel,
                                   sched_ref -> db_name,
                                   printerActionCB, (void *) sched_ref );

      if( sched_ref -> pr_handle == NULL )
        return;

      /* Display the printer window. */
      xtmPrDisplayPrinter( sched_ref -> pr_handle,
                           sched_ref -> schedule_start,
                           sched_ref -> schedule_stop,
                           0,
                           &sched_ref -> filter_rec );

      /* Inherit the calendar settings. */
      xtmPrSetCalendar( sched_ref -> pr_handle, sched_ref -> db_name );
      break;


    /* Upload and download calendar. */
    case 4:
    case 5:
      {
        Boolean          can_download = False;
        Boolean          ok;
        XTM_CD_CAL_INFO  db_info;

        if( sched_ref -> remote_handle == NULL )
          sched_ref -> remote_handle = xtmRtInitialize(
                                         appl_data_ref,
                                         sched_ref -> scheduleW,
                                         False,
                                         NULL, (void *) sched_ref );

        if( sched_ref -> remote_handle == NULL )
          return;

        /* Display the window. */
        ok = xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle, 
                                sched_ref -> db_name,
                                &db_info, NULL );
        if( ! ok )
          return;

        if( flagIsClear( sched_ref -> flags, XTM_SM_READ_ONLY ) &&
            flagIsSet(   db_info.operations, XTM_DB_FLAG_MODE_WRITE ) )
          can_download = True;

        if( (int) call_data -> data == 4 )
          xtmRtDoRemote( sched_ref -> remote_handle, XTM_RT_UPLOAD,
                         db_info.short_name, True, can_download );
        else
          xtmRtDoRemote( sched_ref -> remote_handle, XTM_RT_DOWNLOAD,
                         db_info.short_name, True, can_download );
      }
      break;


    /* Exit the window. */
    case 7:
      closeCB( widget, sched_ref, NULL );
      break;

  } /* switch */


  return;

} /* fileMenuCB */


/*----------------------------------------------------------------------*/

static void 
  filterApplyCB( XTM_FI_REASON      reason,
                 XTM_DM_FILTER_REC  *filter_ref,
                 void               *user_data )
{

  /* Variables. */
  SCHED_REC_REF  sched_ref;


  /* Code. */

  if( reason != XTM_FI_REASON_APPLY && reason != XTM_FI_REASON_OK )
    return;

  sched_ref = (SCHED_REC_REF) user_data;


  /* Save the filter. */
  memcpy( (void *) &sched_ref -> filter_rec,
          (void *) filter_ref,
          sizeof( XTM_DM_FILTER_REC ) );


  /* Display the schedule again ... */
  applyCB( sched_ref -> scheduleW, sched_ref, NULL );


  return;

} /* filterApplyCB */


/*----------------------------------------------------------------------*/

static void
  includeDbApplyCB( XTM_DI_REASON         reason,
                    XTM_CD_INCL_CALS_REF  new_db_incl_ref,
                    void                  *user_data )
{

  /* Variables. */
  SCHED_REC_REF           sched_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         db_info;
  XTM_CD_INCL_CALS        db_incl;


  /* Code. */

  if( reason != XTM_DI_REASON_APPLY && reason != XTM_DI_REASON_OK )
    return;

  sched_ref       = (SCHED_REC_REF) user_data;
  custom_data_ref = sched_ref -> appl_data_ref -> custom_data;


  /* Fetch the current database. */
  (void) xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle,
                            sched_ref -> db_name,
                            &db_info, &db_incl );

  /* Save the new include data. */
  (void) xtmCdChangeEntry( custom_data_ref -> cal_db_handle,
                           sched_ref -> db_name,
                           &db_info, new_db_incl_ref );

  /* Redisplay. */
  applyCB( sched_ref -> scheduleW, sched_ref, NULL );


  return;

} /* includeDbApplyCB */


/*----------------------------------------------------------------------*/

static void 
  infoCB( Widget                     widget,
          SCHED_REC_REF              sched_ref,
          XmRowColumnCallbackStruct  *call_data )
{

  /* Code. */

  /* About window? */
  if( (int) call_data -> data == 6 ) {
    xtmHlDisplayAboutWindow( sched_ref -> scheduleW );

    return;
  }

  /* Use the standard help. */
  xtmHlDisplayHelp( sched_ref -> appl_data_ref -> info_handle,
                    (int) call_data -> data,
                    day_view_window_id, "" );

  return;

} /* infoCB */


/*----------------------------------------------------------------------*/

static void
  leaveHookCB( Widget    widget,
               XEvent    *event,
               char      *leave_args[],
               Cardinal  *num_args )
{

  /* Variables. */
  UINT32                flags = 0;
  Arg                   args[ 10 ];
  Cardinal              n;
  Dimension             animate_height;
  Dimension             animate_width;
  Position              start_x = 0;
  Position              start_y = 0;
  Time                  time_now;
  Widget                animateBaseW;
  Widget                mainW;
  ENTRY_INFO            *entry_info_ref;
  SCHED_REC_REF         sched_ref;
  XTM_GL_BASE_DATA_REF  appl_data_ref;


  /* Code. */

  /* Only for push buttons. */
  if( ! xitRwIsPixButton( widget ) )
    return;

  /* Only if armed. */
  if( ! xitRwPixButIsArmed( widget ) )
    return;

  /* Get reference to our user data. */
#ifdef XD_HAS_NO_PRINTF_PTR
  sscanf( leave_args[ 1 ], "%d", &sched_ref );
#else
  sscanf( leave_args[ 1 ], "%p", &sched_ref );
#endif

  appl_data_ref = sched_ref -> appl_data_ref;

  mainW = XtNameToWidget( sched_ref -> scheduleW, "SchedTlBase.SchedTlFo" );


  /* Only activate move if leaving the window with Btn(1|2) pressed. */
  if( event -> xany.type != LeaveNotify )
    return;

  if( flagIsClear( event -> xcrossing.state, (Button1Mask | Button2Mask) ) )
    return;

  /* Don't get double events. */
  time_now = event -> xcrossing.time;

  if( abs( time_now - time_last ) < 2000 && time_last != 0 ) {
    time_last = time_now;
    return;
  }

  time_last = time_now;


  /* Force the button to let go of the pointer events. */
  XUngrabPointer( XtDisplay( widget ), CurrentTime );


  /* Get the bounds of the animation cursor. */
  n = 0;
  XtSetArg( args[ n ], XmNwidth,  &animate_width  ); n++;
  XtSetArg( args[ n ], XmNheight, &animate_height ); n++;
  XtGetValues( widget, args, n );


  /* Valid action? */
  if( flagIsSet( event -> xcrossing.state, Button1Mask ) ) {
    if( strcmp( leave_args[ 0 ], "Note" ) == 0 )
      return;

    if( flagIsSet( sched_ref -> flags, XTM_SM_LIST_LAYOUT ) )
      return;
  }


  /* Make sure the entry is selected. */
  if( sched_ref -> selected_widget != widget ) {
    xtmSwUnselectEntry( sched_ref );

    n = 0;
    XtSetArg( args[ n ], XmNbackground, &sched_ref -> entry_saved_bg ); n++;
    XtGetValues( widget, args, n );

    n = 0;
    XtSetArg( args[ n ], XmNbackground, 
              appl_data_ref -> custom_data -> entry_select_bg ); n++;
    XtSetValues( widget, args, n );

    sched_ref -> selected_widget = widget;
  }


  /* Fetch information for the entry. */
  entry_info_ref = xtmStFetchSelectedInfo( sched_ref );
  if( entry_info_ref == NULL )
    return;


  /* New duration? */
  if( flagIsSet( event -> xcrossing.state, Button1Mask ) ) {

    TIM_DELTA_TYPE  delta_time;

    (void) TimDelta( sched_ref -> schedule_start,
                     entry_info_ref -> date_stamp, &delta_time );

    start_x = xtmStPositionEntryX( sched_ref, delta_time.days );
    start_y = xtmStPositionEntryY( sched_ref, entry_info_ref -> time_stamp );

    animate_width = (Dimension) sched_ref -> day_width - 14;

    flagSet( flags, ANI_NEW_DURATION );
  }

  /* Move/copy entry. */
  if( flagIsSet( event -> xcrossing.state, Button2Mask ) ) {
    flagSet( flags, ANI_MOVE_COPY );
  }


  /* Animation for notes/entries? */
  if( strcmp( leave_args[ 0 ], "Note" ) == 0 ) {
    flagSet( flags, ANI_NOTE );

    animateBaseW = XtNameToWidget( mainW, 
      "SchedPa.Pane1Fo.NoteSw.ClipWindow.NoteBb" );

  } else {
    flagSet( flags, ANI_ENTRY );

    animateBaseW = XtNameToWidget( mainW, 
      "SchedPa.Pane2Fo.EntrySw.ClipWindow.EntryBb" );
  }

  /* Start the animation. */
  xtmSaStartAnimation( sched_ref, event, 
                       flags, animateBaseW,
                       start_x, start_y,
                       animate_width, animate_height );


  return;

} /* leaveHookCB */


/*----------------------------------------------------------------------*/

static void 
  makeAppointmentCB( Widget               widget,
                     SCHED_REC_REF        sched_ref,
                     XmAnyCallbackStruct  *call_data )
{

  /* Variables. */
  Boolean       ok;
  TIM_TIME_REF  pick_date;


  /* Code. */

  if( flagIsSet( call_data -> event -> xbutton.state, ShiftMask ) ) {
    pick_date = sched_ref -> schedule_start;

  } else {
    ok = xtmStPickDate( sched_ref, XTM_ST_PICK_APP, &pick_date );
    if( ! ok )
      return;
  }


  /* Create an appointment. */
  xtmSwUnselectEntry( sched_ref );

  xtmSaAppointmentEditorShow( sched_ref );

  /* Set the correct date. */
  xtmEdSetValues( sched_ref -> editor_handle, 
                  XTM_ED_SET_DATE, pick_date, 0, 0, "" );


  return;

} /* makeAppointmentCB */


/*----------------------------------------------------------------------*/

static void 
  makeNoteCB( Widget               widget,
              SCHED_REC_REF        sched_ref,
              XmAnyCallbackStruct  *call_data )
{

  /* Variables. */
  Boolean       ok;
  TIM_TIME_REF  pick_date;


  /* Code. */

  if( flagIsSet( call_data -> event -> xbutton.state, ShiftMask ) ) {
    pick_date = sched_ref -> schedule_start;

  } else {
    ok = xtmStPickDate( sched_ref, XTM_ST_PICK_NOTE, &pick_date );
    if( ! ok )
      return;
  }


  /* Create a note. */
  xtmSwUnselectEntry( sched_ref );

  xtmSaAppointmentEditorShow( sched_ref );

  /* Make sure we are editing a note. */
  xtmEdSetValues( sched_ref -> editor_handle, 
                  (XTM_ED_SET_IS_NOTE | XTM_ED_SET_DATE), pick_date,
                  0, 0, "" );


  return;

} /* makeNoteCB */


/*----------------------------------------------------------------------*/

static void 
  navigateCB( Widget                        widget,
              SCHED_REC_REF                 sched_ref,
              XmUbArrowLabelCallbackStruct  *call_data )
{

  /* Variables. */
  Widget        baseW;
  Widget        mainW;
  Widget        tempW;
  TIM_TIME_REF  schedule_start;


  /* Code. */

  baseW = xitGetParentWidget( widget, "SchedTl" );
  if( baseW == NULL )
    return;

  mainW = XtNameToWidget( baseW, "SchedTlBase.SchedTlFo" );

  schedule_start = sched_ref -> schedule_start;

  /* Previous/next day? */
  tempW = XtNameToWidget( mainW, "NavFo.NavDayAl" );
  if( tempW == widget ) {
    if( call_data -> reason == XmUbCR_BACK_ARROW_ACTIVATED )
      TimAddDays( &schedule_start, -1 );
    else
      TimAddDays( &schedule_start, 1 );
  }

  /* Previous/next week? */
  tempW = XtNameToWidget( mainW, "NavFo.NavWeekAl" );
  if( tempW == widget ) {
    if( call_data -> reason == XmUbCR_BACK_ARROW_ACTIVATED )
      TimAddDays( &schedule_start, -7 );
    else
      TimAddDays( &schedule_start, 7 );
  }

  /* Previous/next month? */
  tempW = XtNameToWidget( mainW, "NavFo.NavMonthAl" );
  if( tempW == widget ) {
    if( call_data -> reason == XmUbCR_BACK_ARROW_ACTIVATED )
      TimAddMonths( &schedule_start, -1 );
    else
      TimAddMonths( &schedule_start, 1 );
  }

  /* Previous/next year? */
  tempW = XtNameToWidget( mainW, "NavFo.NavYearAl" );
  if( tempW == widget ) {
    if( call_data -> reason == XmUbCR_BACK_ARROW_ACTIVATED )
      TimAddYears( &schedule_start, -1 );
    else
      TimAddYears( &schedule_start, 1 );
  }


  /* Within limits? */
  if( TimIndexOfYear( schedule_start ) < 1971 ) {
    XBell( XtDisplay( widget ), 100 );

    return;
  }

  /* Update the day view. */
  sched_ref -> schedule_start = schedule_start;

  xtmSwSetScheduleDates( sched_ref, widget, False );
  xtmSwSetSchedule( sched_ref, widget );


  return;

} /* navigateCB */


/*----------------------------------------------------------------------*/

static void
  newViewApplyCB( XTM_OV_REASON    reason,
                  XTM_CD_CAL_INFO  *db_info_ref,
                  void             *user_data )
{

  /* Variables. */
  SCHED_REC_REF           sched_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  if( reason != XTM_OV_REASON_APPLY && reason != XTM_OV_REASON_OK )
    return;

  sched_ref       = (SCHED_REC_REF) user_data;
  custom_data_ref = sched_ref -> appl_data_ref -> custom_data;


  /* Remove navigation calendar? */
  if( sched_ref -> nav_cal_handle != NULL )
    xtmNcDestroy( sched_ref -> nav_cal_handle );


  /* Free the shadow calendar. */
  xtmCdFreeShadowEntry( custom_data_ref -> cal_db_handle,
                        sched_ref -> db_name );


  /* Create a shadow calendar database. */
  (void) xtmCdCreateShadowEntry( custom_data_ref -> cal_db_handle,
                                 db_info_ref -> short_name, 
                                 sched_ref -> db_name );

  /* Redisplay. */
  applyCB( sched_ref -> scheduleW, sched_ref, NULL );


  return;

} /* newViewApplyCB */


/*----------------------------------------------------------------------*/

static void 
  optionsMenuCB( Widget                     widget,
                 SCHED_REC_REF              sched_ref,
                 XmRowColumnCallbackStruct  *call_data )
{

  /* Variables. */
  Widget                  mainW;
  ENTRY_INFO              *entry_info_ref;
  TIM_TIME_REF            start_date;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         db_info;


  /* Code. */

  appl_data_ref   = sched_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;

  mainW = XtNameToWidget( sched_ref -> scheduleW, "SchedTlBase.SchedTlFo" );

  entry_info_ref = xtmStFetchSelectedInfo( sched_ref );


  /* Fetch the current database. */
  (void) xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle,
                            sched_ref -> db_name,
                            &db_info, NULL );


  /* Select what to do. */
  switch( (int) call_data -> data ) {

    /* Appointment editor. */
    case 0:
      xtmSaAppointmentEditorShow( sched_ref );
      break;


    /* Summary window. */
    case 1:
      if( sched_ref -> show_handle == NULL )
        sched_ref -> show_handle = xtmSdInitialize( 
                                     appl_data_ref, 
                                     appl_data_ref -> toplevel, 
                                     sched_ref -> db_name,
                                     summaryActionCB, (void *) sched_ref );

      if( sched_ref -> show_handle == NULL )
        return;

      /* Display the summary. */
      xtmSdDisplaySummary( sched_ref -> show_handle,
                           sched_ref -> schedule_start,
                           sched_ref -> schedule_stop,
                           db_info.view_tags );
      break;


    /* Send message. */
    case 2:
      if( appl_data_ref -> msg_send_handle == NULL )
        appl_data_ref -> msg_send_handle = 
          xtmMsInitialize( appl_data_ref, appl_data_ref -> toplevel );

      /* The message window. */
      if( entry_info_ref == NULL )
        xtmMsSendMessage( appl_data_ref -> msg_send_handle, 
                          "", 0 );
      else
        xtmMsSendMessage( appl_data_ref -> msg_send_handle, 
                          entry_info_ref -> db_name, 
                          entry_info_ref -> id );
      break;


    /* Planner window. */
    case 3:
      if( sched_ref -> planner_handle == NULL )
        sched_ref -> planner_handle = xtmPlInitialize(
                                        appl_data_ref, 
                                        appl_data_ref -> toplevel,
                                        NULL, NULL );
      if( sched_ref -> planner_handle == NULL )
        return;

      /* Planner for the current month. */
      start_date = TimMakeTime( TimIndexOfYear(  sched_ref -> schedule_start ),
                                TimIndexOfMonth( sched_ref -> schedule_start ),
                                TimIndexOfDay(   sched_ref -> schedule_start ),
                                0, 0, 0 );

      /* Display the planner window. */
      xtmPlViewPlanner( sched_ref -> planner_handle, start_date, False );
      break;


    /* Redisplay schedule. */
    case 5:
      if( sched_ref -> animate_ref != NULL )
        sched_ref -> animate_ref -> animate_on = False;

      sched_ref -> force_update = True;
      xtmUpDoUpdate( (XTM_UP_CALENDAR | XTM_UP_SCHEDULE), NULL );
      break;

  } /* switch */


  return;

} /* optionsMenuCB */


/*----------------------------------------------------------------------*/

static void
  printerActionCB( XTM_PR_REASON  reason,
                   void           *user_data )
{

  /* Variables. */
  SCHED_REC_REF  sched_ref;


  /* Code. */

  sched_ref = (SCHED_REC_REF) user_data;

  if( reason == XTM_PR_REASON_DESTROY )
    sched_ref -> pr_handle = NULL;


  return;

} /* printerActionCB */


/*----------------------------------------------------------------------*/

static void
  selHideApplyCB( XTM_SH_REASON  reason,
                  UINT32         entry_id,
                  char           *calendar,
                  TIM_TIME_REF   as_date,
                  TIM_TIME_REF   as_time,
                  void           *user_data )
{

  /* Variables. */
  SCHED_REC_REF  sched_ref;


  /* Code. */

  sched_ref = (SCHED_REC_REF) user_data;

  /* Window destroyed? */
  if( reason == XTM_SH_REASON_DESTROY ) {
    sched_ref -> sel_hide_handle = NULL;
    return;
  }

  /* Edit entry? */
  if( reason == XTM_SH_REASON_EDIT )
    xtmSaAppointmentEditorShowFor( sched_ref,
                                   entry_id, calendar, as_date, as_time );


  return;

} /* selHideApplyCB */


/*----------------------------------------------------------------------*/

static void
  summaryActionCB( XTM_SD_REASON  reason,
                   void           *user_data )
{

  /* Variables. */
  SCHED_REC_REF  sched_ref;


  /* Code. */

  sched_ref = (SCHED_REC_REF) user_data;

  if( reason == XTM_SD_REASON_DESTROY )
    sched_ref -> show_handle = NULL;


  return;

} /* summaryActionCB */


/*----------------------------------------------------------------------*/

static void
  timeActionHookCB( Widget         widget,
                    SCHED_REC_REF  sched_ref,
                    XEvent         *event )
{

  /* Variables. */
  Widget    mainW;
  Widget    timeDispBbW;
  Widget    timeDispLaW;
  Widget    timeDispSwW;


  /* Code. */

  mainW = XtNameToWidget( sched_ref -> scheduleW, "SchedTlBase.SchedTlFo" );

  /* No animation in list mode. */
  if( flagIsSet( sched_ref -> flags, XTM_SM_LIST_LAYOUT ) )
    return;

  /* Only activate when Btn1 pressed. */
  if( event -> xany.type != ButtonPress || event -> xbutton.button != 1 )
    return;


  /* Information to use in the animation. */
  timeDispSwW = XtNameToWidget( mainW, "SchedPa.Pane2Fo.TimeDispSw" );

  timeDispLaW = XtNameToWidget( timeDispSwW, "ClipWindow" );
  timeDispBbW = XtNameToWidget( timeDispLaW, "TimeDispBb" );
  timeDispLaW = XtNameToWidget( timeDispLaW, "TimeDispLa" );

  /* Start animation as if new entry. */
  xtmSaStartAnimation( sched_ref, event,
                       (ANI_TIME | ANI_NEW_ENTRY | ANI_PICK_DAY),
                       timeDispBbW,
                       1, 0, 6, 0 );


  return;

} /* timeActionHookCB */


/*----------------------------------------------------------------------*/

static void
  updateCB( UINT32  flags,
            void    *user_data,
            void    *update_user_data )
{

  /* Variables. */
  int                     day_no;
  Widget                  mainW;
  ANIMATE_REC_REF         animate_ref;
  SCHED_REC_REF           sched_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  TIM_TIME_REF            use_time;

  /* Code. */

  /* Our private data. */
  sched_ref = (SCHED_REC_REF) user_data;

  animate_ref     = sched_ref -> animate_ref;
  custom_data_ref = sched_ref -> appl_data_ref -> custom_data;

  mainW = XtNameToWidget( sched_ref -> scheduleW, "SchedTlBase.SchedTlFo" );


  /* If we are already doing animation, don't do updates. */
  if( animate_ref != NULL && animate_ref -> animate_on )
    return;


  /* Update schedule? */
  if( flagIsSet( flags, XTM_UP_SCHEDULE ) )
    xtmSwSetSchedule( sched_ref, sched_ref -> scheduleW );


  /* New day? */
  if( flagIsSet( flags, XTM_UP_NEW_DAY ) ) {

    use_time = TimLocalTime( TimMakeTimeNow() );
    day_no   = TimIndexOfDayInIsoWeek( use_time );

    if( custom_data_ref -> day_view_day_switch == 0 ||
        custom_data_ref -> day_view_day_switch - 1 == day_no ) {

      sched_ref -> schedule_start = TimMakeTime( TimIndexOfYear( use_time ),
                                                 TimIndexOfMonth( use_time ),
                                                 TimIndexOfDay( use_time ),
                                                 0, 0, 0 );

      xtmSwSetScheduleDates( sched_ref, sched_ref -> scheduleW, False );
      xtmSwSetSchedule( sched_ref, sched_ref -> scheduleW );
      
    }

  } /* if */


  /* A simple minute tick? */
  if( flagIsSet( flags, (XTM_UP_MINUTE_TICK | XTM_UP_SCHEDULE) ) )
    xtmStUpdateMinuteMarker( sched_ref );


  return;

} /* updateCB */


/*----------------------------------------------------------------------*/

static void 
  zoomCB( Widget         widget,
          SCHED_REC_REF  sched_ref,
          XtPointer      call_data )
{

  /* Variables. */
  int     hours;
  int     minutes;
  char    *char_ref;
  Widget  baseW;
  Widget  mainW;
  Widget  tempW;


  /* Code. */

  baseW = xitGetParentWidget( widget, "SchedTl" );
  if( baseW == NULL )
    return;

  mainW = XtNameToWidget( baseW, "SchedTlBase.SchedTlFo" );


  /* Fetch the zoom string. */
  tempW = XtNameToWidget( mainW, "SchedPa.Pane2Fo.ZoomFs" );

  xitFieldSelectGetCurrent( tempW, &char_ref );
  if( char_ref == NULL )
    return;

  sscanf( char_ref, "%d:%d", &hours, &minutes );

  sched_ref -> entry_delta = (hours * 60) + minutes;

  /* Redisplay the schedule. */
  xtmSwSetSchedule( sched_ref, sched_ref -> scheduleW );

  SysFree( char_ref );


  return;

} /* zoomCB */
