/*----------------------------------------------------------------------------
--
--  Module:           xtmPickDiary
--
--  Project:          XDiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Select a diary from one or more diary description files.
--
--  Filename:         xtmPickDiary.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-07-01
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmPickDiary.c, Version: 1.1, Date: 95/02/18 15:52:37";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <X11/Intrinsic.h>

#include <Xm/Xm.h>
#include <Xm/List.h>
#include <Xm/Text.h>
#include <Xm/RowColumn.h>

#include "System.h"
#include "Message.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmHelp.h"
#include "xitError.h"
#include "xitInfo.h"
#include "xitTools.h"
#include "xtmPickDiary.h"
#include "DirList.h"

/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Local widgets in the include window. */
#define dbDirLa           dataLocalW[ 0 ]
#define dbDirRc           dataLocalW[ 1 ]
#define dbDirTx           dataLocalW[ 2 ]
#define dbNameLa          dataLocalW[ 3 ]
#define dbNameRc          dataLocalW[ 4 ]
#define dbNameTx          dataLocalW[ 5 ]
#define diaryLa           dataLocalW[ 6 ]
#define diaryLi           dataLocalW[ 7 ]
#define fileLa            dataLocalW[ 8 ]
#define fileLi            dataLocalW[ 9 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Include record. */
typedef struct {

  /* Directory with description files. */
  char  *directory;

  /* File pattern for description files. */
  char  *file_pattern;

  /* Pick window. */
  Widget           pickW;

  /* Info handle. */
  XIT_IN_HANDLE  info_handle;

  /* Callback to inform our creator of specific actions. */
  XTM_PD_ACTION_CB  actionCB;
  void              *user_data;

} PICK_REC, *PICK_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmPickDiary";

/* IDs for the help windows. */
static char  *pick_window_id = "PickCal";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  cancelCB( Widget        widget,
            PICK_REC_REF  pick_ref,
            XtPointer     call_data );

static Widget
  createPickWindow( PICK_REC_REF  pick_ref,
                    Widget        parent,
                    Boolean       app_modal );

static void 
  destroyCB( Widget        widget, 
             PICK_REC_REF  pick_ref,
             XtPointer     call_data );

static void 
  diaryListCB( Widget                widget, 
               PICK_REC_REF          pick_ref,
               XmListCallbackStruct  *call_data );

static void 
  fileListCB( Widget                widget, 
              PICK_REC_REF          pick_ref,
              XmListCallbackStruct  *call_data );

static void 
  helpCB( Widget        widget,
          PICK_REC_REF  pick_ref,
          XtPointer     call_data );

static void
  okCB( Widget        widget,
        PICK_REC_REF  pick_ref,
        XtPointer     call_data );

static void
  setContentsFiles( PICK_REC_REF  pick_ref );

static void
  setDiaryInfo( PICK_REC_REF  pick_ref,
                char          *file_name );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XTM_PD_HANDLE
  xtmPdInitialize( Widget             parent,
                   Boolean            app_modal,
                   XIT_IN_HANDLE      info_handle,
                   XTM_PD_ACTION_CB   actionCB,
                   void               *user_data )
{

  /* Variables. */
  PICK_REC_REF  pick_ref;


  /* Code. */

  /* Create and initialize our private data. */
  pick_ref = SysNew( PICK_REC );
  if( pick_ref == NULL )
    return( NULL );

  pick_ref -> directory    = NULL;
  pick_ref -> file_pattern = NULL;
  pick_ref -> info_handle  = info_handle;
  pick_ref -> actionCB     = actionCB;
  pick_ref -> user_data    = user_data;


  /* Create the pick window. */
  pick_ref -> pickW = createPickWindow( pick_ref, parent, app_modal );

  if( pick_ref -> pickW == NULL ) {
    SysFree( pick_ref );

    return( NULL );
  }

  pick_ref -> directory    = SysNewString( "" );
  pick_ref -> file_pattern = SysNewString( "" );


  return( (XTM_PD_HANDLE) pick_ref );

} /* xtmPdInitialize */


/*----------------------------------------------------------------------*/

void
  xtmPdDestroy( XTM_PD_HANDLE  pick_handle )
{

  /* Variables. */
  PICK_REC_REF  pick_ref;


  /* Code. */

  if( pick_handle == NULL )
    return;

  /* Our private data. */
  pick_ref = (PICK_REC_REF) pick_handle;


  /* Destroy the window. */
  XtDestroyWidget( pick_ref -> pickW );


  return;

} /* xtmPdDestroy */


/*----------------------------------------------------------------------*/

void
  xtmPdDisplayPickWindow( XTM_PD_HANDLE  pick_handle,
                          char           *directory,
                          char           *file_pattern )
{

  /* Variables. */
  PICK_REC_REF  pick_ref;


  /* Code. */

  if( pick_handle == NULL )
    return;

  /* Our private data. */
  pick_ref = (PICK_REC_REF) pick_handle;


  /* Any new directories or file pattern? */
  if( strcmp( directory,    pick_ref -> directory ) != 0 ||
      strcmp( file_pattern, pick_ref -> file_pattern ) != 0 ) {

    SysFree( pick_ref -> directory );
    SysFree( pick_ref -> file_pattern );

    pick_ref -> directory    = SysNewString( directory );
    pick_ref -> file_pattern = SysNewString( file_pattern );

    /* Set the values in the window. */
    setContentsFiles( pick_ref );

  } /* if */


  /* Make sure the window is visible. */
  XtManageChild( pick_ref -> pickW );


  return;

} /* xtmPdDisplayPickWindow */


/*----------------------------------------------------------------------*/

static Widget
  createPickWindow( PICK_REC_REF  pick_ref,
                    Widget        parent,
                    Boolean       app_modal )
{

  /* Variables. */
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    dataLocalW[ 10 ];
  Widget    pickFd;
  Widget    workFo;

  static XIT_TEXT_STRUCT text_buffer[] = {
    { "DbNameTx", NULL, 1, True },
    { "DbDirTx",  NULL, 1, True },
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "", okCB,     NULL },
    { "", cancelCB, NULL },
    { "", helpCB,   NULL },
  };


  /* Code. */

  /* Set message strings. */
  action_buttons[ 0 ].label = msgGetText( MXDI_OK_BUTTON );
  action_buttons[ 0 ].data  = pick_ref;
  action_buttons[ 1 ].label = msgGetText( MXDI_CANCEL_BUTTON );
  action_buttons[ 1 ].data  = pick_ref;
  action_buttons[ 2 ].label = msgGetText( MXDI_HELP_BUTTON );
  action_buttons[ 2 ].data  = pick_ref;


  /* Create a form dialog with buttons. */
  pickFd = xitCreateFormDialog( parent, "PickFd",
                                1, 0,
                                action_buttons,
                                XtNumber( action_buttons ) );

  XtAddCallback( pickFd, XmNdestroyCallback,
                 (XtCallbackProc) destroyCB, (XtPointer) pick_ref );

  n = 0;
  XtSetArg( args[ n ], XmNdeleteResponse, XmUNMAP ); n++;
  XtSetValues( pickFd, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, msgGetText( MXDI_PICK_TITLE ) ); n++;
  XtSetValues( XtParent( pickFd ), args, n );

  if( app_modal ) {
    n = 0;
    XtSetArg( args[ n ], XmNdialogStyle, XmDIALOG_APPLICATION_MODAL ); n++;
    XtSetValues( pickFd, args, n );
  }


  /* Create a container for the contents of the window. */
  workFo = XtNameToWidget( pickFd, "PickFdFo" );

  fileLa = xitCreateLabel( workFo, "FileLa", 
                           msgGetText( MXDI_PICK_FILE_LABEL ), -1 );

  diaryLa = xitCreateLabel( workFo, "FileLa", 
                            msgGetText( MXDI_PICK_DIARY_LABEL ), -1 );


  /* List with diary collection files. */
  n = 0;
  XtSetArg( args[ n ], XmNlistSizePolicy,         XmCONSTANT ); n++;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmSTATIC ); n++;
  XtSetArg( args[ n ], XmNselectionPolicy,        XmSINGLE_SELECT ); n++;
  XtSetArg( args[ n ], XmNlistMarginHeight,       5 ); n++;
  XtSetArg( args[ n ], XmNlistMarginWidth,        5 ); n++;
  fileLi = XmCreateScrolledList( workFo, "FileLi", args, n );

  XtAddCallback( fileLi, XmNsingleSelectionCallback,
                 (XtCallbackProc) fileListCB, (XtPointer) pick_ref );
  XtAddCallback( fileLi, XmNdefaultActionCallback,
                 (XtCallbackProc) fileListCB, (XtPointer) pick_ref );

  /* List with diary descriptions. */
  n = 0;
  XtSetArg( args[ n ], XmNlistSizePolicy,         XmCONSTANT ); n++;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmSTATIC ); n++;
  XtSetArg( args[ n ], XmNselectionPolicy,        XmSINGLE_SELECT ); n++;
  XtSetArg( args[ n ], XmNlistMarginHeight,       5 ); n++;
  XtSetArg( args[ n ], XmNlistMarginWidth,        5 ); n++;
  diaryLi = XmCreateScrolledList( workFo, "DiaryLi", args, n );

  XtAddCallback( diaryLi, XmNsingleSelectionCallback,
                 (XtCallbackProc) diaryListCB, (XtPointer) pick_ref );
  XtAddCallback( diaryLi, XmNdefaultActionCallback,
                 (XtCallbackProc) diaryListCB, (XtPointer) pick_ref );


  /* Name of database. */
  dbNameRc = XmCreateRowColumn( workFo, "DbNameRc", args, 0 );

  dbNameLa = xitCreateLabel( dbNameRc, "DbNameLa", 
                             msgGetText( MXDI_DB_NAME_LABEL ), -1 );
  dbNameTx = xitCreateTextCols( dbNameRc, &text_buffer[ 0 ], 15 );

  n = 0;
  XtSetArg( args[ n ], XmNmaxLength, 15 ); n++;
  XtSetValues( dbNameTx, args, n );


  /* Directory for database. */
  dbDirRc = XmCreateRowColumn( workFo, "DbDirRc", args, 0 );

  dbDirLa = xitCreateLabel( dbDirRc, "DbDirLa", 
                            msgGetText( MXDI_DB_DIR_LABEL ), -1 );

  dbDirTx = xitCreateTextCols( dbDirRc, &text_buffer[ 1 ], 50 );

  n = 0;
  XtSetArg( args[ n ], XmNmaxLength, 50 ); n++;
  XtSetValues( dbDirTx,  args, n );



  /* Put the Parts together. */
  xitAttachWidget( fileLa,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( diaryLa,
                   XmATTACH_FORM, NULL, XmATTACH_WIDGET, XtParent( fileLi ),
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( XtParent( fileLi ),
                   XmATTACH_WIDGET, fileLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,   XmATTACH_NONE, NULL );
  xitAttachWidget( XtParent( diaryLi ),
                   XmATTACH_WIDGET, diaryLa, 
                   XmATTACH_WIDGET, XtParent( fileLi ),
                   XmATTACH_FORM, NULL, XmATTACH_WIDGET, dbNameRc );
  xitAttachWidget( dbNameRc,
                   XmATTACH_WIDGET, XtParent( fileLi ), XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,               XmATTACH_NONE, NULL );
  xitAttachWidget( dbDirRc,
                   XmATTACH_WIDGET, dbNameRc, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL,       XmATTACH_NONE, NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( fileLa,              args, n );
  XtSetValues( diaryLa,             args, n );
  XtSetValues( XtParent( fileLi ),  args, n );
  XtSetValues( XtParent( diaryLi ), args, n );
  XtSetValues( dbNameRc,            args, n );
  XtSetValues( dbDirRc,             args, n );


  /* Manage the children. */
  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );

  /* Set the initial sizes. */
  xitSetSizeFormDialog( pickFd, True );


  /* Make the final attachments. */
  xitAttachWidget( dbDirRc,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL );
  xitAttachWidget( dbNameRc,
                   XmATTACH_NONE, NULL, XmATTACH_FORM,   NULL,
                   XmATTACH_NONE, NULL, XmATTACH_WIDGET, dbDirRc );
  xitAttachWidget( XtParent( fileLi ),
                   XmATTACH_WIDGET, fileLa, XmATTACH_FORM,   NULL,
                   XmATTACH_NONE,   NULL,   XmATTACH_WIDGET, dbNameRc );

  n = 0;
  XtSetArg( args[ n ], XmNrightAttachment, XmATTACH_POSITION ); n++;
  XtSetArg( args[ n ], XmNrightPosition,   50 ); n++;
  XtSetValues( XtParent( fileLi ), args, n );


  return( pickFd );

} /* createPickWindow */


/*----------------------------------------------------------------------*/

static void
  setContentsFiles( PICK_REC_REF  pick_ref )
{

  /* Variables. */
  int       index = 0;
  int       index1;
  int       status;
  char      filename[ 250 ];
  char      search_buffer[ 250 ];
  char      *char_ref;
  Arg       args[ 5 ];
  Cardinal  n;
  Widget    tempW;
  XmString  list_items[ 100 ];


  /* Code. */

  /* Search all files matching pattern. */
  sprintf( search_buffer, "%s/%s", 
           pick_ref -> directory, pick_ref -> file_pattern );

  status = DirFindFirst( search_buffer, filename );

  while( status == 0 ) {

    char_ref = strrchr( filename, '/' );
    if( char_ref != NULL )
      char_ref++;

    if( index < 100 && char_ref != NULL ) {
      list_items[ index ] = XmStringCreate( char_ref, CS );
      index++;
    }

    status = DirFindNext( filename );

  } /* while */


  /* The list is always sorted. */
  xitSortStringList( list_items, index );

  tempW = XtNameToWidget( pick_ref -> pickW, "PickFdFo.FileLiSW.FileLi" );

  /* Assign the collect files to the file list. */
  n = 0;
  XtSetArg( args[ n ], XmNitems, list_items ); n++;
  XtSetArg( args[ n ], XmNitemCount, index ); n++;
  XtSetArg( args[ n ], XmNselectedItemCount, 0 ); n++;
  XtSetValues( tempW, args, n );

  if( index > 0 )
    XmListSelectPos( tempW, 1, True );

  /* Free allocated memory. */
  for( index1 = 0; index1 < index; index1++ )
    XmStringFree( list_items[ index1 ] );


  return;

} /* setContentsFiles */


/*----------------------------------------------------------------------*/

static void
  setDiaryInfo( PICK_REC_REF  pick_ref,
                char          *file_name )
{

  /* Variables. */
  int       index = 0;
  int       index1;
  int       items;
  char      buffer[ 250 ];
  char      description[ 50 ];
  char      location[ 150 ];
  char      name[ 50 ];
  char      *char_ref;
  Arg       args[ 5 ];
  Cardinal  n;
  Widget    baseW;
  Widget    tempW;
  XmString  list_items[ 100 ];
  FILE      *file_ref;


  /* Code. */

  baseW = XtNameToWidget( pick_ref -> pickW, "PickFdFo" );

  /* Read information from the diary file. */
  sprintf( buffer, "%s/%s", pick_ref -> directory, file_name );

  file_ref = fopen( buffer, "r" );
  if( file_ref == NULL )
    return;


  do {

    char_ref = fgets( buffer, sizeof( buffer ), file_ref );
    if( char_ref == NULL )
      break;

    while( isspace( *char_ref ) )
      char_ref++;

    /* Ignore blank lines and comment lines. */
    if( *char_ref == '\0' || *char_ref == '!' || *char_ref == '#' ) {
      ;

    } else if( index < 500 ) {
      items = sscanf( char_ref, "%[^;];%[^;];%s",
                      name, description, location );

      if( items == 3 ) {
        sprintf( buffer, "%-10.10s;%-20.20s;%s",
                 name, description, location );

        list_items[ index ] = XmStringCreate( buffer, CS );
        index++;
      }

    } /* if */

  } while( True );

  fclose( file_ref );


  /* The list is always sorted. */
  xitSortStringList( list_items, index );

  tempW = XtNameToWidget( baseW, "DiaryLiSW.DiaryLi" );

  /* Assign the collect files to the file list. */
  n = 0;
  XtSetArg( args[ n ], XmNitems, list_items ); n++;
  XtSetArg( args[ n ], XmNitemCount, index ); n++;
  XtSetArg( args[ n ], XmNselectedItemCount, 0 ); n++;
  XtSetValues( tempW, args, n );

  /* Free allocated memory. */
  for( index1 = 0; index1 < index; index1++ )
    XmStringFree( list_items[ index1 ] );


  /* Clear the name and location field. */
  tempW = XtNameToWidget( baseW, "DbNameRc.DbNameTx" );
  XmTextSetString( tempW, "" );

  tempW = XtNameToWidget( baseW, "DbDirRc.DbDirTx" );
  XmTextSetString( tempW, "" );


  return;

} /* setDiaryInfo */


/*----------------------------------------------------------------------*/

static void
  cancelCB( Widget        widget,
            PICK_REC_REF  pick_ref,
            XtPointer     call_data )
{

  /* Code. */

  XtUnmanageChild( pick_ref -> pickW );


  return;

} /* cancelCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget        widget, 
             PICK_REC_REF  pick_ref,
             XtPointer     call_data )
{

  /* Code. */

  /* Call callback function? */
  if( pick_ref -> actionCB != NULL )
    (* pick_ref -> actionCB)( XTM_PD_REASON_DESTROY, 
                              NULL, pick_ref -> user_data );

  if( pick_ref -> directory != NULL )
    SysFree( pick_ref -> directory );

  if( pick_ref -> file_pattern != NULL )
    SysFree( pick_ref -> file_pattern );

  SysFree( pick_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void 
  diaryListCB( Widget                widget, 
               PICK_REC_REF          pick_ref,
               XmListCallbackStruct  *call_data )
{

  /* Variables. */
  char    description[ 50 ];
  char    location[ 150 ];
  char    name[ 50 ];
  char    *char_ref;
  Widget  baseW;
  Widget  tempW;


  /* Code. */

  baseW = XtNameToWidget( pick_ref -> pickW, "PickFdFo" );

  /* Fetch the diary description. */
  char_ref = xitStringGetString( call_data -> item, CS );

  sscanf( char_ref, "%[^;];%[^;];%s", name, description, location );
  SysFree( char_ref );


  /* Set the name and location field. */
  tempW = XtNameToWidget( baseW, "DbNameRc.DbNameTx" );
  XmTextSetString( tempW, name );

  tempW = XtNameToWidget( baseW, "DbDirRc.DbDirTx" );
  XmTextSetString( tempW, location );


  return;

} /* diaryListCB */


/*----------------------------------------------------------------------*/

static void 
  fileListCB( Widget                widget, 
              PICK_REC_REF          pick_ref,
              XmListCallbackStruct  *call_data )
{

  /* Variables. */
  char  descr_file[ 100 ];
  char  *char_ref;


  /* Code. */

  /* Fetch the selected description file. */
  char_ref = xitStringGetString( call_data -> item, CS );

  sscanf( char_ref, "%s", descr_file );
  SysFree( char_ref );


  /* Set the diary information windopw. */
  setDiaryInfo( pick_ref, descr_file );


  return;

} /* fileListCB */


/*----------------------------------------------------------------------*/

static void 
  helpCB( Widget        widget,
          PICK_REC_REF  pick_ref,
          XtPointer     call_data )
{

  /* Code. */

  xtmHlDisplayHelp( pick_ref -> info_handle,
                    XTM_HL_WINDOW_INDEX,
                    pick_window_id, "" );

  return;

} /* helpCB */


/*----------------------------------------------------------------------*/

static void
  okCB( Widget        widget,
        PICK_REC_REF  pick_ref,
        XtPointer     call_data )
{

  /* Variables. */
  int              items;
  char             *char_ref;
  Widget           baseW;
  Widget           tempW;
  XTM_CD_CAL_INFO  db_info;


  /* Code. */

  baseW = XtNameToWidget( pick_ref -> pickW, "PickFdFo" );


  /* Name of calendar. */
  tempW = XtNameToWidget( baseW, "DbNameRc.DbNameTx" );
  char_ref = xitStringGetText( tempW );

  items = sscanf( char_ref, "%15s", db_info.short_name );
  SysFree( char_ref );

  if( items != 1 ) {
    xitErMessage( pick_ref -> pickW, XIT_ER_INFO,
                  module_name, "okCB",
                  msgGetText( MXDI_MISSING_NAME_OR_LOC ) );
    return;
  }


  /* Location of the calendar. */
  tempW = XtNameToWidget( baseW, "DbDirRc.DbDirTx" );
  char_ref = xitStringGetText( tempW );

  items = sscanf( char_ref, "%50s", db_info.directory );
  SysFree( char_ref );

  if( items != 1 ) {
    return;
  }

  /* Calendar default values. */
  db_info.owner_uid         = -1;
  db_info.operations        = 0;
  db_info.mail_address[ 0 ] = '\0';
  db_info.view_tags[ 0 ]    = '\0';
  db_info.alarm_tags[ 0 ]   = '\0';
  db_info.incl_bg_index     = 0;
  db_info.normal_bg_index   = 0;
  db_info.include           = NULL;

  /* Call callback function? */
  if( pick_ref -> actionCB != NULL )
    (* pick_ref -> actionCB)( XTM_PD_REASON_OK, 
                              &db_info, pick_ref -> user_data );


  XtUnmanageChild( pick_ref -> pickW );


  return;

} /* okCB */
