/*----------------------------------------------------------------------------
--
--  Module:           xtmMsgSelect
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Select message(s) to read and answer.
--
--  Filename:         xtmMsgSelect.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-04-01
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmMsgSelect.c, Version: 1.1, Date: 95/02/18 15:52:33";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>

#include <Xm/Protocols.h>

#include <Xm/Xm.h>
#include <Xm/List.h>
#include <Xm/RowColumn.h>

#include "System.h"
#include "DirList.h"
#include "Message.h"
#include "TimDate.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmCalDb.h"
#include "xtmCustBase.h"
#include "xtmDbTools.h"
#include "xtmFormat.h"
#include "xtmHelp.h"
#include "xtmIcons.h"
#include "xtmMsgRead.h"
#include "xtmTools.h"
#include "xitError.h"
#include "xitTools.h"
#include "xtmMsgSelect.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Local widgets in the message select window. */
#define menuBr              dataLocalW[  0 ]
#define msgLi               dataLocalW[  1 ]
#define msgboxLa            dataLocalW[  2 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/


/* Record describing one instance of the message select window. */
typedef struct {

  /* Application wide data. */
  XTM_GL_BASE_DATA_REF  appl_data_ref;

  /* Callback and user data to inform our creator of specific actions. */
  XTM_ML_ACTION_CB  actionCB;
  void              *user_data;

  /* Handle to read message. */
  XTM_MR_HANDLE  read_handle;

  /* Select window. */
  Widget  selectW;

} SEL_REC, *SEL_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmMsgSelect";

/* IDs for the help windows. */
static char  *select_window_id = "MsgSelect";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void 
  closeCB( Widget       widget,
           SEL_REC_REF  sel_ref,
           XtPointer    call_data );

static Widget 
  createSelectWindow( Widget       parent,
                      SEL_REC_REF  sel_ref );

static void 
  deleteCB( Widget       widget,
            SEL_REC_REF  sel_ref,
            XtPointer    call_data );

static void 
  deleteMessageCB( Widget       widget,
                   SEL_REC_REF  sel_ref,
                   XtPointer    call_data );

static void 
  destroyCB( Widget       widget,
             SEL_REC_REF  sel_ref,
             XtPointer    call_data );

static void 
  infoCB( Widget                     widget,
          SEL_REC_REF                sel_ref,
          XmRowColumnCallbackStruct  *call_data );

static void 
  msgListCB( Widget       widget,
             SEL_REC_REF  sel_ref,
             XtPointer    call_data );

static void 
  readCB( Widget       widget,
          SEL_REC_REF  sel_ref,
          XtPointer    call_data );

static void 
  redisplayCB( Widget       widget,
               SEL_REC_REF  sel_ref,
               XtPointer    call_data );

static void 
  setActions( SEL_REC_REF  sel_ref );

static void
  setSelectWindowValues( SEL_REC_REF  sel_ref );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XTM_ML_HANDLE
  xtmMlInitialize( XTM_GL_BASE_DATA_REF  appl_data_ref,
                   Widget                parent,
                   XTM_ML_ACTION_CB      actionCB,
                   void                  *user_data )
{

  /* Variables. */
  SEL_REC_REF  sel_ref;


  /* Code. */

  /* Create and initialize our private data. */
  sel_ref = SysNew( SEL_REC );
  if( sel_ref == NULL )
    return( NULL );

  sel_ref -> appl_data_ref = appl_data_ref;
  sel_ref -> selectW       = NULL;
  sel_ref -> actionCB      = actionCB;
  sel_ref -> user_data     = user_data;
  sel_ref -> read_handle   = NULL;


  /* Create the message select window. */
  sel_ref -> selectW = createSelectWindow( parent, sel_ref );
  if( sel_ref -> selectW == NULL ) {
    SysFree( sel_ref );

    return( NULL );
  }


  return( (XTM_ML_HANDLE) sel_ref );

} /* xtmMlInitialize */


/*----------------------------------------------------------------------*/

void
  xtmMlDestroy( XTM_ML_HANDLE  sel_handle )
{

  /* Variables. */
  SEL_REC_REF  sel_ref;


  /* Code. */

  if( sel_handle == NULL )
    return;

  /* Our private data. */
  sel_ref = (SEL_REC_REF) sel_handle;


  /* Get rid of all windows. */
  if( sel_ref -> selectW != NULL )
    XtDestroyWidget( sel_ref -> selectW );


  return;

} /* xtmMlDestroy */


/*---------------------------------------------------------------------------*/

void
  xtmMlSelectMessage( XTM_ML_HANDLE  sel_handle )
{

  /* Variables. */
  SEL_REC_REF  sel_ref;


  /* Code. */

  /* Our private data. */
  sel_ref = (SEL_REC_REF) sel_handle;


  /* Make sure the editor window is visable and set values. */
  setSelectWindowValues( sel_ref );

  XtPopup( sel_ref -> selectW, XtGrabNone );

  XRaiseWindow( XtDisplay( sel_ref -> selectW ), 
                XtWindow(  sel_ref -> selectW ) );
  XtMapWidget( sel_ref -> selectW );


  return;

} /* xtmMseSelectMessage */


/*----------------------------------------------------------------------*/

static Widget 
  createSelectWindow( Widget       parent,
                      SEL_REC_REF  sel_ref )
{

  /* Variables. */
  int       index;
  char      *char_ref;
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    dataLocalW[ 3 ];
  Widget    menuCasc[ 2 ];
  Widget    menuHelpBu[ 6 ];
  Widget    menuFileBu[ 1 ];
  Widget    pulldownMenu[ 2 ];
  Widget    msgSelTl;
  Widget    tempW;
  Widget    workFo;

  static char  *pull_downs[] = { "pdown1", "pdown2" };

  static XIT_CASCADE_STRUCT menupane[] = {
    { "", "", "FilePane" },
    { "", "", "HelpPane" },
  };

  static XIT_MENU_BUTTON_STRUCT file_casc[] = {
    { "", "", NULL, "CloseBu", True, False, False },
  };

  static XIT_MENU_BUTTON_STRUCT help_casc[] = {
    { "", "", NULL, "ContextBu", True, False, False },
    { "", "", NULL, "WindowsBu", True, False, False },
    { "", "", NULL, "KeysBu",    True, False, False },
    { "", "", NULL, "IndexBu",   True, False, False },
    { "", "", NULL, "HelpBu",    True, False, False },
    { "", "", NULL, "VersionBu", True, False, False },
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "", readCB,      NULL },
    { "", deleteCB,    NULL },
    { "", redisplayCB, NULL },
    { "", closeCB,     NULL },
  };


  /* Code. */

  /* Initialize the menu items. */
  action_buttons[ 0 ].label = msgGetText( MXDI_READ_LABEL );
  action_buttons[ 0 ].data  = sel_ref;
  action_buttons[ 1 ].label = msgGetText( MXDI_DELETE_LABEL );
  action_buttons[ 1 ].data  = sel_ref;
  action_buttons[ 2 ].label = msgGetText( MXDI_REDISPLAY_BUTTON );
  action_buttons[ 2 ].data  = sel_ref;
  action_buttons[ 3 ].label = msgGetText( MXDI_CLOSE_BUTTON );
  action_buttons[ 3 ].data  = sel_ref;

  menupane[ 0 ].title    = msgGetText( MXDI_FILE_MENU );
  menupane[ 0 ].mnemonic = msgGetText( MXDI_FILE_MENU_ACC );
  menupane[ 1 ].title    = msgGetText( MXDI_HELP_MENU );
  menupane[ 1 ].mnemonic = msgGetText( MXDI_HELP_MENU_ACC );

  file_casc[ 0 ].title    = msgGetText( MXDI_CLOSE_MENU );
  file_casc[ 0 ].mnemonic = msgGetText( MXDI_CLOSE_MENU_ACC );

  help_casc[ 0 ].title    = msgGetText( MXDI_HELP_CONTEXT );
  help_casc[ 0 ].mnemonic = msgGetText( MXDI_HELP_CONTEXT_ACC );
  help_casc[ 1 ].title    = msgGetText( MXDI_HELP_WINDOWS );
  help_casc[ 1 ].mnemonic = msgGetText( MXDI_HELP_WINDOWS_ACC );
  help_casc[ 2 ].title    = msgGetText( MXDI_HELP_KEYS );
  help_casc[ 2 ].mnemonic = msgGetText( MXDI_HELP_KEYS_ACC );
  help_casc[ 3 ].title    = msgGetText( MXDI_HELP_INDEX );
  help_casc[ 3 ].mnemonic = msgGetText( MXDI_HELP_INDEX_ACC );
  help_casc[ 4 ].title    = msgGetText( MXDI_HELP_HELP );
  help_casc[ 4 ].mnemonic = msgGetText( MXDI_HELP_HELP_ACC );
  help_casc[ 5 ].title    = msgGetText( MXDI_HELP_VERSION );
  help_casc[ 5 ].mnemonic = msgGetText( MXDI_HELP_VERSION_ACC );


  /* Create a separate window. */
  tempW = xitGetToplevelWidget( parent );

  msgSelTl = xitCreateToplevelDialog( tempW, "MsgSelTl", 
                                      0, 0,
                                      action_buttons, 
                                      XtNumber( action_buttons ) );

  /* Create a separate window. */
  char_ref = msgGetText( MXDI_RECEIVE_MSG_TITLE );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, char_ref ); n++;
  XtSetValues( msgSelTl, args, n );

  char_ref = msgGetText( MXDI_RECEIVE_MSG_IC_TITLE );

  n = 0;
  XtSetArg( args[ n ], XmNiconName, char_ref ); n++;
  XtSetValues( msgSelTl, args, n );

  XtAddCallback( msgSelTl,  XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) sel_ref );


  /* Exit the application if this window is deleted. */
  {
    Atom  wm_delete_window;

    wm_delete_window = XmInternAtom( XtDisplay( msgSelTl ),
                                     "WM_DELETE_WINDOW", False );

    XmAddWMProtocols( msgSelTl, &wm_delete_window, 1 );
    XmAddWMProtocolCallback( msgSelTl, wm_delete_window, 
                             (XtCallbackProc) closeCB, (XtPointer) sel_ref );
  } /* block */


  /* Form to work in. */
  workFo = XtNameToWidget( msgSelTl, "MsgSelTlBase.MsgSelTlFo" );


  /* Create the menubar and menu cascades. */
  n = 0;
  menuBr = XmCreateMenuBar( workFo, "MenuBr", args, n );

  n = 0;
  for( index = 0; index < XtNumber( pulldownMenu ); index++ )
    pulldownMenu[ index ] = XmCreatePulldownMenu( menuBr, 
                                                  pull_downs[ index ], 
                                                  NULL, n );

  for( index = 0; index < XtNumber( menuCasc ); index++ )
    menuCasc[ index ] = xitCreateCascadeButton( menuBr, 
                                                pulldownMenu[ index ], 
                                                &menupane[ index ] );

  /* The help button should be placed to the right. */
  index = XtNumber( menuCasc ) - 1;
  n     = 0;
  XtSetArg( args[ n ], XmNmenuHelpWidget, menuCasc[ index ] ); n++;
  XtSetValues( menuBr, args, n );


  /* Create the file menu. */
  for( index = 0; index < XtNumber( menuFileBu ); index++ )
    menuFileBu[ index ] = xitCreateMenuPushButton( pulldownMenu[ 0 ],
                                                   &file_casc[ index ] );
  XtAddCallback( menuFileBu[ 0 ], XmNactivateCallback, 
                 (XtCallbackProc) closeCB, (XtPointer) sel_ref );


  /* Create the help menu. */
  XtAddCallback( pulldownMenu[ 1 ], XmNentryCallback, 
                 (XtCallbackProc) infoCB, (XtPointer) sel_ref );

  for( index = 0; index < XtNumber( menuHelpBu ); index++ ) {
    menuHelpBu[ index ] = xitCreateMenuPushButton( pulldownMenu[ 1 ], 
                                                   &help_casc[ index ] );

    XtAddCallback( menuHelpBu[ index ], XmNactivateCallback, 
                   (XtCallbackProc) infoCB, (XtPointer) index );
  }

  /* We can't do context sensitive help. */
  XtSetSensitive( menuHelpBu[ 0 ], False );


  /* Create the mailbox list. */
  msgboxLa = xitCreateLabel( workFo, "MsgboxLa", 
                             msgGetText( MXDI_MSG_BOX_LABEL ), -1 );

  n = 0;
  XtSetArg( args[ n ], XmNlistSizePolicy,         XmRESIZE_IF_POSSIBLE ); n++;
  XtSetArg( args[ n ], XmNselectionPolicy,        XmEXTENDED_SELECT ); n++;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmSTATIC ); n++;
  XtSetArg( args[ n ], XmNlistMarginHeight,       5 ); n++;
  XtSetArg( args[ n ], XmNlistMarginWidth,        5 ); n++;
  msgLi = XmCreateScrolledList( workFo, "MsgLi", args, n );

  XtAddCallback( msgLi, XmNsingleSelectionCallback,
                 (XtCallbackProc) msgListCB, (XtPointer) sel_ref );
  XtAddCallback( msgLi, XmNextendedSelectionCallback,
                 (XtCallbackProc) msgListCB, (XtPointer) sel_ref );
  XtAddCallback( msgLi, XmNdefaultActionCallback, 
                 (XtCallbackProc) readCB, (XtPointer) sel_ref );


  /* Place the main elements together. */
  xitAttachWidget( menuBr,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_FORM, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( msgboxLa,
                   XmATTACH_WIDGET, menuBr, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,   XmATTACH_NONE, NULL );
  xitAttachWidget( XtParent( msgLi ),
                   XmATTACH_WIDGET, msgboxLa, XmATTACH_FORM, NULL,
                   XmATTACH_FORM,   NULL,     XmATTACH_NONE, NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( msgboxLa,          args, n );
  XtSetValues( XtParent( msgLi ), args, n );


  /* Manage all the children. */
  XtManageChildren( menuCasc,   XtNumber( menuCasc ) );
  XtManageChildren( menuHelpBu, XtNumber( menuHelpBu ) );
  XtManageChildren( menuFileBu, XtNumber( menuFileBu ) );

  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );


  /* Set icon for this window. */
  xtmIcSetSimpleIcon( msgSelTl, workFo, XTM_IC_ICON_MAIL );


  /* Make the final attachments. */
  n = 0;
  XtSetArg( args[ n ], XmNbottomAttachment, XmATTACH_FORM ); n++;
  XtSetValues( XtParent( msgLi ), args, n );


  return( msgSelTl );

} /* createSelectWindow */


/*----------------------------------------------------------------------*/

static void 
  setActions( SEL_REC_REF  sel_ref )
{

  /* Variables. */
  Boolean   sensitive;
  int       item_count;
  int       selected_item_count;
  Arg       args[ 5 ];
  Cardinal  n;
  Widget    baseW;
  Widget    mainW;
  Widget    tempW;


  /* Code. */

  baseW = XtNameToWidget( sel_ref -> selectW, "MsgSelTlBase" );
  mainW = XtNameToWidget( sel_ref -> selectW, "MsgSelTlBase.MsgSelTlFo" );


  /* Do we have a selected message? */
  tempW = XtNameToWidget( mainW, "MsgLiSW.MsgLi" );

  n = 0;
  XtSetArg( args[ n ], XmNitemCount, &item_count ); n++;
  XtSetArg( args[ n ], XmNselectedItemCount, &selected_item_count ); n++;
  XtGetValues( tempW, args, n );

  if( selected_item_count < 1 || item_count < 1 )
    sensitive = False;
  else
    sensitive = True;


  /* Set sensitivity for action buttons. */
  tempW = XtNameToWidget( baseW, "Bu1" );
  XtSetSensitive( tempW, sensitive );

  tempW = XtNameToWidget( baseW, "Bu2" );
  XtSetSensitive( tempW, sensitive );


  return;

} /* setActions */


/*----------------------------------------------------------------------*/

static void
  setSelectWindowValues( SEL_REC_REF  sel_ref )
{

  /* Variables. */
  int                   index;
  int                   index1;
  int                   items;
  int                   status;
  char                  buffer[ 51 ];
  char                  date[ 20 ];
  char                  filename[ 150 ];
  char                  file_pattern[ 100 ];
  char                  message_dir[ 150 ];
  Arg                   args[ 5 ];
  Cardinal              n;
  Widget                mainW;
  Widget                tempW;
  XmString              list_items[ 100 ];
  XTM_GL_BASE_DATA_REF  appl_data_ref;
  XTM_CD_CAL_INFO       default_db;
  struct stat           file_info;


  /* Code. */

  appl_data_ref = sel_ref -> appl_data_ref;

  mainW = XtNameToWidget( sel_ref -> selectW, "MsgSelTlBase.MsgSelTlFo" );


  /* Messages are in the default database. */
  (void) xtmCdFetchDefaultDb( appl_data_ref -> custom_data -> cal_db_handle,
                              &default_db, NULL );


  /* Find the message directory. */
  sprintf( message_dir, "%s/Message", default_db.directory );

  status = stat( message_dir, &file_info );
  if( status != 0 )
    sprintf( message_dir, "%s", default_db.directory );


  /* Fetch all message files. */
  index = 0;
  sprintf( file_pattern, "%s/%s_*_*", message_dir, XTM_DB_MESSAGE_FILE );

  status = DirFindFirst( file_pattern, filename );
  while( status == 0 ) {

    int           id;
    char          user[ 51 ];
    char          *char_ref;
    TIM_TIME_REF  tmp_time;

    /* Message file information. */
    status = stat( filename, &file_info );

    /* Split into user, number and id. */
    char_ref = strrchr( filename, '/' );
    if( char_ref != NULL )
      char_ref++;
    else
      char_ref = filename;

    items = sscanf( char_ref, "%[^_]_%[^_]_%d", buffer, user, &id );
    if( items == 3 ) {

      /* Date of the message. */
      tmp_time = TimLocalTime( (TIM_TIME_REF) file_info.st_mtime );

      xtmFoFormatDate( tmp_time, date, sizeof( date ) );

      /* Create the message line. */
      sprintf( buffer, "%s %-15.15s  %d", date, user, id );

      list_items[ index ] = XmStringCreate( buffer, CS );
      index++;

    } /* if */

    /* Next Message. */
    status = DirFindNext( filename );

  } /* while */

  /* Put the messages in the list. */
  tempW = XtNameToWidget( mainW, "MsgLiSW.MsgLi" );

  n = 0;
  XtSetArg( args[ n ], XmNitems, list_items ); n++;
  XtSetArg( args[ n ], XmNitemCount, index ); n++;
  XtSetValues( tempW, args, n );

  XmListDeselectAllItems( tempW );

  /* Free allocated memory. */
  for( index1 = 0; index1 < index; index1++ )
    XmStringFree( list_items[ index1 ] );


  /* Set active buttons. */
  setActions( sel_ref );


  return;

} /* setSelectWindowValues */


/*----------------------------------------------------------------------*/

static void 
  closeCB( Widget       widget,
           SEL_REC_REF  sel_ref,
           XtPointer    call_data )
{

  /* Code. */

  XtPopdown( sel_ref -> selectW );


  return;

} /* closeCB */


/*----------------------------------------------------------------------*/

static void 
  deleteCB( Widget       widget,
            SEL_REC_REF  sel_ref,
            XtPointer    call_data )
{

  /* Variables. */
  int       selected_item_count;
  Arg       args[ 5 ];
  Cardinal  n;
  Widget    mainW;
  Widget    tempW;


  /* Code. */

  mainW = XtNameToWidget( sel_ref -> selectW, "MsgSelTlBase.MsgSelTlFo" );


  /* Fetch the selected message(s). */
  tempW = XtNameToWidget( mainW, "MsgLiSW.MsgLi" );

  n = 0;
  XtSetArg( args[ n ], XmNselectedItemCount, &selected_item_count ); n++;
  XtGetValues( tempW, args, n );

  if( selected_item_count < 1 )
    return;


  /* Does the user confirm? */
  if( sel_ref -> appl_data_ref -> custom_data -> confirm_actions ) {

    tempW = xitCreateQuestionDialog( 
              sel_ref -> selectW, "QuestionDialog",
              msgGetText( MXDI_QUESTION_MESSAGE_LABEL ),
              msgGetText( MXDI_DELETE_MESSAGE_CONF ),
              deleteMessageCB, sel_ref, 
              NULL, NULL );

    return;
  }

  /* Remove the message. */
  deleteMessageCB( widget, sel_ref, NULL );


  return;

} /* deleteCB */


/*----------------------------------------------------------------------*/

static void 
  deleteMessageCB( Widget       widget,
                   SEL_REC_REF  sel_ref,
                   XtPointer    call_data )
{

  /* Variables. */
  int                   index;
  int                   msg_id;
  int                   selected_item_count;
  char                  dummy[ 51 ];
  char                  *buffer_ref;
  Arg                   args[ 5 ];
  Cardinal              n;
  Widget                mainW;
  Widget                tempW;
  XmString              *selected_items;
  XTM_DB_STATUS         db_status;
  XTM_GL_BASE_DATA_REF  appl_data_ref;
  XTM_CD_CAL_INFO       default_db;


  /* Code. */

  appl_data_ref = sel_ref -> appl_data_ref;

  mainW = XtNameToWidget( sel_ref -> selectW, "MsgSelTlBase.MsgSelTlFo" );


  /* The default database contains the messages. */
  (void) xtmCdFetchDefaultDb( appl_data_ref -> custom_data -> cal_db_handle, 
                              &default_db, NULL );


  /* Fetch the selected messages. */
  tempW = XtNameToWidget( mainW, "MsgLiSW.MsgLi" );

  n = 0;
  XtSetArg( args[ n ], XmNselectedItemCount, &selected_item_count ); n++;
  XtSetArg( args[ n ], XmNselectedItems,     &selected_items ); n++;
  XtGetValues( tempW, args, n );


  /* Delete all selected messages. */
  for( index = 0; index < selected_item_count; index++ ) {

    buffer_ref = xitStringGetString( *(selected_items + index), CS );

    sscanf( buffer_ref, "%[^ ] %[^ ] %d", dummy, dummy, &msg_id );
    SysFree( buffer_ref );

    /* Delete the message. */
    db_status = xtmDbDeleteMessage( default_db.directory, msg_id );

  } /* loop */


  /* Update the message box. */
  setSelectWindowValues( sel_ref );


  return;

} /* deleteMessageCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget       widget,
             SEL_REC_REF  sel_ref,
             XtPointer    call_data )
{

  /* Code. */

  /* Do we have a user action callback registered? */
  if( sel_ref -> actionCB != NULL )
    (* sel_ref -> actionCB)( XTM_ML_REASON_DESTROY,
                             sel_ref -> user_data );


  /* Release the user data. */
  SysFree( sel_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void 
  infoCB( Widget                     widget,
          SEL_REC_REF                sel_ref,
          XmRowColumnCallbackStruct  *call_data )
{

  /* Code. */

  xtmHlDisplayHelp( sel_ref -> appl_data_ref -> info_handle,
                    (int) call_data -> data,
                    select_window_id, "" );

  return;

} /* infoCB */


/*----------------------------------------------------------------------*/

static void 
  msgListCB( Widget       widget,
             SEL_REC_REF  sel_ref,
             XtPointer    call_data )
{

  /* Code. */

  setActions( sel_ref );


  return;

} /* msgListCB */


/*----------------------------------------------------------------------*/

static void 
  readCB( Widget       widget,
          SEL_REC_REF  sel_ref,
          XtPointer    call_data )
{

  /* Variables. */
  int       selected_item_count;
  int       msg_id;
  char      dummy[ 51 ];
  char      *buffer_ref;
  Arg       args[ 5 ];
  Cardinal  n;
  Widget    mainW;
  Widget    tempW;
  XmString  *selected_items;


  /* Code. */

  mainW = XtNameToWidget( sel_ref -> selectW, "MsgSelTlBase.MsgSelTlFo" );


  /* Fetch the selected message (if more selected, use the first). */
  tempW = XtNameToWidget( mainW, "MsgLiSW.MsgLi" );

  n = 0;
  XtSetArg( args[ n ], XmNselectedItemCount, &selected_item_count ); n++;
  XtSetArg( args[ n ], XmNselectedItems,     &selected_items ); n++;
  XtGetValues( tempW, args, n );

  if( selected_item_count < 1 )
    return;  


  /* Fetch the message number. */
  buffer_ref = xitStringGetString( *selected_items, CS );

  sscanf( buffer_ref, "%[^ ] %[^ ] %d", dummy, dummy, &msg_id );
  SysFree( buffer_ref );


  /* Initialize the read window. */
  if( sel_ref -> read_handle == NULL )
    sel_ref -> read_handle = xtmMrInitialize( sel_ref -> appl_data_ref,
                                              sel_ref -> selectW );

  /* Display the message. */
  xtmMrReadMessage( sel_ref -> read_handle, msg_id );


  return;

} /* readCB */


/*----------------------------------------------------------------------*/

static void 
  redisplayCB( Widget       widget,
               SEL_REC_REF  sel_ref,
               XtPointer    call_data )
{

  /* Code. */

  setSelectWindowValues( sel_ref );


  return;

} /* redisplayCB */
