/*----------------------------------------------------------------------------
--
--  Module:           xtmCustImp
--
--  Project:          XDiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Import included calendars to the local calendar DB.
--
--  Filename:         xtmCustImp.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1993-11-10
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmCustImp.c, Version: 1.1, Date: 95/02/18 15:52:04";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <X11/Intrinsic.h>

#include <Xm/Xm.h>
#include <Xm/List.h>

#include "System.h"
#include "Message.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmCalDb.h"
#include "xitError.h"
#include "xitTools.h"
#include "xtmCustImp.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Local widgets in the 'Customize Include' window. */
#define calLi              dataLocalW[  0 ]
#define explainLa          dataLocalW[  1 ]



/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Record for included calendars customization. */
typedef struct {

  /* The include window. */
  Widget  impW;

  /* Selected calendar. */
  char  imp_calendar[ XTM_GL_MAX_CAL_NAME + 1 ]; 

  /* Calendar database. */
  XTM_CD_HANDLE  cal_db_handle;

  /* Customization data. */
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;

  /* Callback to inform our creator of specific actions. */
  void              *user_data;
  XTM_CP_ACTION_CB  actionCB;

} IMP_REC, *IMP_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmCustImport";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void 
  calImportCB( Widget       widget,
               IMP_REC_REF  imp_ref,
               XtPointer    call_data );

static void 
  calSelectCB( Widget                widget, 
               IMP_REC_REF           imp_ref,
               XmListCallbackStruct  *call_data );

static void 
  closeCB( Widget       widget,
           IMP_REC_REF  imp_ref,
           XtPointer    call_data );

static Widget
  createImportWindow( IMP_REC_REF  imp_ref,
                      Widget        parent );

static void 
  destroyCB( Widget       widget,
             IMP_REC_REF  imp_ref,
             XtPointer    call_data );

static void
  setImportData( IMP_REC_REF  imp_ref );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XTM_CP_HANDLE
  xtmCpInitialize( XTM_GL_CUSTOM_DATA_REF  custom_data_ref,
                   Widget                  parent,
                   XTM_CP_ACTION_CB        actionCB,
                   void                    *user_data )
{

  /* Variables. */
  IMP_REC_REF  imp_ref;


  /* Code. */

  /* Create and initialize our private data. */
  imp_ref = SysNew( IMP_REC );
  if( imp_ref == NULL )
    return( NULL );

  imp_ref -> custom_data_ref   = custom_data_ref;
  imp_ref -> actionCB          = actionCB;
  imp_ref -> user_data         = user_data;
  imp_ref -> imp_calendar[ 0 ] = '\0';


  /* Create the groups window. */
  imp_ref -> impW = createImportWindow( imp_ref, parent );

  if( imp_ref -> impW == NULL ) {
    SysFree( imp_ref );

    return( NULL );
  }


  return( (XTM_CP_HANDLE) imp_ref );

} /* xtmCpInitialize */


/*----------------------------------------------------------------------*/

void
  xtmCpDestroy( XTM_CP_HANDLE  imp_handle )
{

  /* Variables. */
  IMP_REC_REF  imp_ref;


  /* Code. */

  if( imp_handle == NULL )
    return;

  /* Our private data. */
  imp_ref = (IMP_REC_REF) imp_handle;


  /* Destroy the window. */
  XtDestroyWidget( imp_ref -> impW );


  return;

} /* xtmCiDestroy */


/*----------------------------------------------------------------------*/

void 
  xtmCpImportCalendar( XTM_CP_HANDLE  imp_handle,
                       XTM_CD_HANDLE  cal_db_handle )
{

  /* Variables. */
  IMP_REC_REF  imp_ref;


  /* Code. */

  if( imp_handle == NULL )
    return;

  /* Our private data. */
  imp_ref = (IMP_REC_REF) imp_handle;


  /* Save the calendars we are editing. */
  imp_ref -> cal_db_handle = cal_db_handle;


  /* Fill in the window fields. */
  setImportData( imp_ref );


  /* Make sure the window is visible. */
  XtManageChild( imp_ref -> impW );


  return;

} /* xtmCpImportCalendar */


/*----------------------------------------------------------------------*/

static Widget
  createImportWindow( IMP_REC_REF  imp_ref,
                      Widget        parent )
{

  /* Variables. */
  Arg        args[ 10 ];
  Cardinal   n;
  Widget     impFd;
  Widget     workFo;
  Widget     dataLocalW[ 2 ];
  XmString   xstr;

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "",   calImportCB, NULL },
    { NULL, NULL,        NULL },
    { NULL, NULL,        NULL },
    { "",   closeCB,     NULL },
  };


  /* Code. */

  action_buttons[ 0 ].label = msgGetText( MXDI_IMPORT_BUTTON );
  action_buttons[ 0 ].data  = imp_ref;
  action_buttons[ 3 ].label = msgGetText( MXDI_CLOSE_BUTTON );
  action_buttons[ 3 ].data  = imp_ref;


  /* Create a form dialog with buttons. */
  impFd = xitCreateFormDialog( parent, "ImpFd",
                               1, 0,
                               action_buttons,
                               XtNumber( action_buttons ) );

  XtAddCallback( impFd,  XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) imp_ref );

  xstr = XmStringCreate( msgGetText( MXDI_IMPORT_TITLE ), CS );

  n = 0;
  XtSetArg( args[ n ], XmNdialogStyle, XmDIALOG_APPLICATION_MODAL ); n++;
  XtSetArg( args[ n ], XmNdialogTitle, xstr ); n++;
  XtSetValues( impFd, args, n );

  XmStringFree( xstr );


  /* Container for the contents of the window. */
  workFo = XtNameToWidget( impFd, "ImpFdFo" );


  /* Explain what this is. */
  explainLa = xitCreateLabel( workFo, "", 
                              msgGetText( MXDI_EXPLAIN_IMPORT_DB ), -1 );

  /* Lists with calendars we can import. */
  n = 0;
  XtSetArg( args[ n ], XmNlistSizePolicy, XmCONSTANT ); n++;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmSTATIC ); n++;
  XtSetArg( args[ n ], XmNselectionPolicy, XmSINGLE_SELECT ); n++;
  XtSetArg( args[ n ], XmNlistMarginHeight, 5 ); n++;
  XtSetArg( args[ n ], XmNlistMarginWidth, 5 ); n++;
  calLi = XmCreateScrolledList( workFo, "CalLi", args, n );

  XtAddCallback( calLi, XmNsingleSelectionCallback,
                 (XtCallbackProc) calSelectCB, (XtPointer) imp_ref );
  XtAddCallback( calLi, XmNdefaultActionCallback,
                 (XtCallbackProc) calSelectCB, (XtPointer) imp_ref );
  XtAddCallback( calLi, XmNdefaultActionCallback,
                 (XtCallbackProc) calImportCB, (XtPointer) imp_ref );


  /* Place the elements together. */
  xitAttachWidget( explainLa,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( XtParent( calLi ),
                   XmATTACH_WIDGET, explainLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( explainLa,         args, n );
  XtSetValues( XtParent( calLi ), args, n );


  /* Manage the widgets. */
  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );

  /* Set the size of the window. */
  xitSetSizeFormDialog( impFd, True );


  /* Make the final attachments. */
  xitAttachWidget( XtParent( calLi ),
                   XmATTACH_WIDGET, explainLa, XmATTACH_FORM, NULL,
                   XmATTACH_FORM,   NULL,      XmATTACH_FORM, NULL );

  /* Make sure our children don't spoil our size. */
  n = 0;
  XtSetArg( args[ n ], XmNallowShellResize, False ); n++;
  XtSetValues( XtParent( impFd ), args, n );


  return( impFd );

} /* createImportWindow */


/*----------------------------------------------------------------------*/

static void
  setImportData( IMP_REC_REF  imp_ref )
{

  /* Variables. */
  Boolean   ok;
  int       index;
  int       index1;
  char      *char_ref;
  char      *db_names;
  char      buffer[ 200 ];
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    mainW;
  Widget    tempW;
  XmString  list_items[ 100 ];


  /* Code. */

  mainW = XtNameToWidget( imp_ref -> impW, "ImpFdFo" );

  imp_ref -> imp_calendar[ 0 ] = '\0';

  /* Display calendars we can include. */
  ok = xtmCdFetchDbNames( imp_ref -> cal_db_handle, &db_names );
  char_ref = db_names;
  index    = 0;

  do {

    int              char_read;
    char             db_name[ XTM_GL_MAX_CAL_NAME + 1 ];
    XTM_CD_CAL_INFO  db_info;

    while( isspace( *char_ref ) )
      char_ref++;

    if( *char_ref == '\0' )
      break;

    char_read = strlen( char_ref );
    sscanf( char_ref, "%s%n", db_name, &char_read );
    char_ref = char_ref + char_read;


    /* Fetch information about the database. */
    (void) xtmCdFetchNamedDb( imp_ref -> cal_db_handle, db_name,
                              &db_info, NULL );

    /* Is this calendar included? */
    if( flagIsClear( db_info.flags, XTM_CD_FLAG_IS_INCLUDED ) )
      continue;

    sprintf( buffer, "%-15.15s %s", 
             db_info.short_name, db_info.directory );

    list_items[ index ] = XmStringCreate( buffer, CS );
    index++;

  } while( index < XtNumber( list_items ) );

  SysFree( db_names );


  tempW = XtNameToWidget( mainW, "CalLiSW.CalLi" );

  /* Assign the calendars to the list. */
  n = 0;
  XtSetArg( args[ n ], XmNitems, list_items ); n++;
  XtSetArg( args[ n ], XmNitemCount, index ); n++;
  XtSetValues( tempW, args, n );

  XmListDeselectAllItems( tempW );

  /* Free allocated memory. */
  for( index1 = 0; index1 < index; index1++ )
    XmStringFree( list_items[ index1 ] );


  /* The "Import" button is not valid. */
  tempW = XtNameToWidget( imp_ref -> impW, "Bu1" );
  XtSetSensitive( tempW, False );


  return;

} /* setImportData */


/*----------------------------------------------------------------------*/

static void 
  calImportCB( Widget       widget,
               IMP_REC_REF  imp_ref,
               XtPointer    call_data )
{

  /* Variables. */
  Boolean           ok;
  XTM_CD_CAL_INFO   db_info;
  XTM_CD_INCL_CALS  db_imp;


  /* Code. */

  if( imp_ref -> imp_calendar[ 0 ] == '\0' )
    return;


  /* Fetch the database we are importing. */
  ok = xtmCdFetchNamedDb( imp_ref -> cal_db_handle, 
                          imp_ref -> imp_calendar,
                          &db_info, &db_imp );
  if( ! ok )
    return;

  flagClear( db_info.flags, XTM_CD_FLAG_IS_INCLUDED );

  (void) xtmCdChangeEntry( imp_ref -> cal_db_handle,
                           imp_ref -> imp_calendar,
                           &db_info, &db_imp );


  /* Do we have a user action callback registered? */
  if( imp_ref -> actionCB != NULL )
    (* imp_ref -> actionCB)( XTM_CP_REASON_IMPORT,
                             imp_ref -> imp_calendar,
                             imp_ref -> user_data );

  /* Remove the window. */
  closeCB( widget, imp_ref, NULL );


  return;

} /* calImportCB */


/*----------------------------------------------------------------------*/

static void 
  calSelectCB( Widget                widget, 
               IMP_REC_REF           imp_ref,
               XmListCallbackStruct  *call_data )
{

  /* Variables. */
  char    *char_ref;
  Widget  mainW;
  Widget  tempW;


  /* Code. */

  mainW = XtNameToWidget( imp_ref -> impW, "ImpFdFo" );

  char_ref = xitStringGetString( call_data -> item, CS );
  sscanf( char_ref, "%s", imp_ref -> imp_calendar );
  SysFree( char_ref );


  /* We can now import the calendar. */  
  tempW = XtNameToWidget( imp_ref -> impW, "Bu1" );
  XtSetSensitive( tempW, True );


  return;

} /* calSelectCB */


/*----------------------------------------------------------------------*/

static void 
  closeCB( Widget       widget,
           IMP_REC_REF  imp_ref,
           XtPointer    call_data )
{

  /* Code. */

  XtUnmanageChild( imp_ref -> impW );


  return;

} /* closeCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget       widget,
             IMP_REC_REF  imp_ref,
             XtPointer    call_data )
{

  /* Code. */

  /* Do we have a user action callback registered? */
  if( imp_ref -> actionCB != NULL )
    (* imp_ref -> actionCB)( XTM_CP_REASON_DESTROY, 
                             NULL, imp_ref -> user_data );


  /* Release the user data. */
  SysFree( imp_ref );


  return;

} /* destroyCB */
