/*----------------------------------------------------------------------------
--
--  Module:           xtmCustGroup
--
--  Project:          XDiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Manage the message groups in XDiary. You can add, change and delete
--    message groups.
--
--  Filename:         xtmCustGroup.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-07-10
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmCustGroup.c, Version: 1.1, Date: 95/02/18 15:52:03";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

#include <Xm/Xm.h>
#include <Xm/List.h>
#include <Xm/RowColumn.h>
#include <Xm/Text.h>

#include "System.h"
#include "LstLinked.h"
#include "Message.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmCustom.h"
#include "xtmCustBase.h"
#include "xtmGroupDb.h"
#include "xtmHelp.h"
#include "xitError.h"
#include "xitTools.h"
#include "xtmCustGroup.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Local widgets in the group window. */
#define grpActionRc        dataLocalW[  0 ]
#define grpLa              dataLocalW[  1 ]
#define grpLi              dataLocalW[  2 ]
#define grpMemLa           dataLocalW[  3 ]
#define grpMemTx           dataLocalW[  4 ]
#define grpNameLa          dataLocalW[  5 ]
#define grpNameTx          dataLocalW[  6 ]

/* What fields do we set in the customize window? */
#define SET_GROUP_LIST      (1<<0)
#define RESET_GROUP_FIELD   (1<<1)


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Record for group customization. */
typedef struct {

  /* Any changes to the group fields. */
  Boolean  field_changed;

  /* Selected group. */
  char  selected_group[ XTM_GD_MAX_GROUP_NAME + 1 ];

  /* The group window. */
  Widget  groupW;

  /* Application data. */
  XTM_CU_BASE_DATA_REF  appl_data_ref;

  /* Temporary list when editing groups. */
  XTM_GD_HANDLE  edit_group_db_handle;

  /* Customization data. */
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;

  /* Callback to inform our creator of specific actions. */
  void              *user_data;
  XTM_CG_ACTION_CB  actionCB;

} GROUP_REC, *GROUP_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmCustGroups";

/* IDs for the help windows. */
static char  *cust_groups_window_id = "CustGroups";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  addCB( Widget         widget, 
         GROUP_REC_REF  group_ref,
         XtPointer      call_data );

static void 
  cancelCB( Widget         widget,
            GROUP_REC_REF  group_ref,
            XtPointer      call_data );

static Widget
  createGroupWindow( GROUP_REC_REF  group_ref,
                     Widget         parent );

static void
  deleteCB( Widget         widget,
            GROUP_REC_REF  group_ref,
            XtPointer      call_data );

static void 
  destroyCB( Widget         widget,
             GROUP_REC_REF  cal_ref,
             XtPointer      call_data );


static void 
  fieldChangedCB( Widget         widget,
                  GROUP_REC_REF  group_ref,
                  XtPointer      call_data );

static void 
  groupListCB( Widget                widget, 
               GROUP_REC_REF         group_ref,
               XmListCallbackStruct  *call_data );

static void 
  helpCB( Widget         widget,
          GROUP_REC_REF  group_ref,
          XtPointer      call_data );

static void 
  okCB( Widget         widget,
        GROUP_REC_REF  group_ref,
        XtPointer      call_data );

static void
  setAction( GROUP_REC_REF  group_ref,
             Boolean        add_button,
             Boolean        delete_button );

static void
  setGroupData( GROUP_REC_REF  group_ref,
                UINT32         action );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XTM_CG_HANDLE
  xtmCgInitialize( XTM_CU_BASE_DATA_REF  appl_data_ref,
                   Widget                parent,
                   XTM_CG_ACTION_CB      actionCB,
                   void                  *user_data )
{

  /* Variables. */
  GROUP_REC_REF  group_ref;


  /* Code. */

  /* Create and initialize our private data. */
  group_ref = SysNew( GROUP_REC );
  if( group_ref == NULL )
    return( NULL );

  group_ref -> appl_data_ref        = appl_data_ref;
  group_ref -> custom_data_ref      = appl_data_ref -> custom_data;
  group_ref -> selected_group[ 0 ]  = '\0';
  group_ref -> edit_group_db_handle = NULL;
  group_ref -> field_changed        = False;
  group_ref -> actionCB             = actionCB;
  group_ref -> user_data            = user_data;


  /* Create the groups window. */
  group_ref -> groupW = createGroupWindow( group_ref, parent );

  if( group_ref -> groupW == NULL ) {
    SysFree( group_ref );

    return( NULL );
  }


  return( (XTM_CG_HANDLE) group_ref );

} /* xtmCgInitialize */


/*----------------------------------------------------------------------*/

void
  xtmCgDestroy( XTM_CG_HANDLE  group_handle )
{

  /* Variables. */
  GROUP_REC_REF  group_ref;


  /* Code. */

  if( group_handle == NULL )
    return;

  /* Our private data. */
  group_ref = (GROUP_REC_REF) group_handle;


  /* Destroy the window. */
  XtDestroyWidget( group_ref -> groupW );


  return;

} /* xtmCgDestroy */


/*----------------------------------------------------------------------*/

void 
  xtmCgCustomGroups( XTM_CG_HANDLE  group_handle )
{

  /* Variables. */
  GROUP_REC_REF  group_ref;


  /* Code. */

  group_ref = (GROUP_REC_REF) group_handle;


  /* Make a copy of the group list? */
  if( group_ref -> edit_group_db_handle == NULL )
    group_ref -> edit_group_db_handle =
      xtmGdCopyGroups( group_ref -> custom_data_ref -> group_db_handle );


  /* Fill in all the fields and set action buttons. */
  setGroupData( group_ref, (SET_GROUP_LIST | RESET_GROUP_FIELD) );

  setAction( group_ref, True, False );


  /* Display the window. */
  XtManageChild( group_ref -> groupW );


  return;

} /* xtmCgCustomGroups */


/*----------------------------------------------------------------------*/

static Widget
  createGroupWindow( GROUP_REC_REF  group_ref,
                     Widget         parent )
{

  /* Variables. */
  int       index;
  char      *char_ref;
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    dataLocalW[ 7 ];
  Widget    groupFd;
  Widget    grpActionBu[ 2 ];
  Widget    workFo;

  static XIT_PUSH_STRUCT group_action_def[] = {
    { "AddPb",    "", "", True, NULL },
    { "DeletePb", "", "", True, NULL },
  };

  static XIT_TEXT_STRUCT text_buffer[] = {
    { "GrpNameTx", NULL, 1, True },
    { "GrpMemTx",  NULL, 5, True },
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "", okCB,     NULL },
    { "", cancelCB, NULL },
    { "", helpCB,   NULL },
  };

  /* Code. */

  action_buttons[ 0 ].label = msgGetText( MXDI_OK_BUTTON );
  action_buttons[ 0 ].data  = group_ref;
  action_buttons[ 1 ].label = msgGetText( MXDI_CANCEL_BUTTON );
  action_buttons[ 1 ].data  = group_ref;
  action_buttons[ 2 ].label = msgGetText( MXDI_HELP_BUTTON );
  action_buttons[ 2 ].data  = group_ref;

  group_action_def[ 0 ].title = msgGetText( MXDI_ADD_CHANGE_ITEM );
  group_action_def[ 1 ].title = msgGetText( MXDI_DELETE_ITEM );


  /* Create a form dialog with buttons. */
  groupFd = xitCreateFormDialog( parent, "GroupFd",
                                 1, 0,
                                 action_buttons, 
                                 XtNumber( action_buttons ) );

  XtAddCallback( groupFd, XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) group_ref );

  char_ref = msgGetText( MXDI_CUST_GROUP_TITLE );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, char_ref ); n++;
  XtSetValues( XtParent( groupFd ), args, n );

  n = 0;
  XtSetArg( args[ n ], XmNdialogStyle, XmDIALOG_APPLICATION_MODAL ); n++;
  XtSetValues( groupFd, args, n );


  /* Use this form as a working form. */
  workFo = XtNameToWidget( groupFd, "GroupFdFo" );

  /* What is this? */  
  grpLa = xitCreateLabel( workFo, "GrpLa", 
                          msgGetText( MXDI_MESSAGE_GROUPS ), -1 );

  /* List with groups. */
  n = 0;
  XtSetArg( args[ n ], XmNlistSizePolicy,         XmCONSTANT ); n++;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmSTATIC ); n++;
  XtSetArg( args[ n ], XmNselectionPolicy,        XmSINGLE_SELECT ); n++;
  XtSetArg( args[ n ], XmNlistMarginHeight,       5 ); n++;
  XtSetArg( args[ n ], XmNlistMarginWidth,        5 ); n++;
  grpLi = XmCreateScrolledList( workFo, "GrpLi", args, n );

  XtAddCallback( grpLi, XmNsingleSelectionCallback,
                 (XtCallbackProc) groupListCB, (XtPointer) group_ref );
  XtAddCallback( grpLi, XmNdefaultActionCallback,
                 (XtCallbackProc) groupListCB, (XtPointer) group_ref );

  /* Name of group. */
  grpNameLa = xitCreateLabel( workFo, "GrpNameLa", 
                              msgGetText( MXDI_GROUP_NAME_LABEL ), -1 );

  grpNameTx = xitCreateTextCols( workFo, &text_buffer[ 0 ], 15 );
  XtAddCallback( grpNameTx, XmNvalueChangedCallback, 
                 (XtCallbackProc) fieldChangedCB, (XtPointer) group_ref );


  /* Members in the group. */
  grpMemLa = xitCreateLabel( workFo, "GrpMemLa", 
                             msgGetText( MXDI_GROUP_MEMBERS_LABEL ), -1 );

  grpMemTx = xitCreateTextScrolled( workFo, &text_buffer[ 1 ] );

  XtOverrideTranslations( grpMemTx, xtm_cu_newline_trans );

  XtAddCallback( grpMemTx, XmNvalueChangedCallback, 
                 (XtCallbackProc) fieldChangedCB, (XtPointer) group_ref );

  n = 0;
  XtSetArg( args[ n ], XmNmaxLength, 15 ); n++;
  XtSetValues( grpNameTx, args, n );


  /* Action (add and delete). */
  n = 0;
  XtSetArg( args[ n ], XmNorientation,  XmHORIZONTAL ); n++;
  XtSetArg( args[ n ], XmNspacing,      10 ); n++;
  XtSetArg( args[ n ], XmNmarginHeight, 10 ); n++;
  grpActionRc = XmCreateRowColumn( workFo, "GrpActionRc", args, n );

  for( index = 0; index < XtNumber( group_action_def ); index++ )
    grpActionBu[ index ] = xitCreatePushButton( grpActionRc,
                                                &group_action_def[ index ] );

  XtAddCallback( grpActionBu[ 0 ], XmNactivateCallback, 
                 (XtCallbackProc) addCB, (XtPointer) group_ref );
  XtAddCallback( grpActionBu[ 1 ], XmNactivateCallback, 
                 (XtCallbackProc) deleteCB, (XtPointer) group_ref );


  /* Place the main elements together. */
  xitAttachWidget( grpLa,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( XtParent( grpLi ),
                   XmATTACH_WIDGET, grpLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,  XmATTACH_NONE, NULL );
  xitAttachWidget( grpNameLa,
                   XmATTACH_WIDGET, XtParent( grpLi ), XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,              XmATTACH_NONE, NULL );
  xitAttachWidget( grpNameTx,
                   XmATTACH_WIDGET, grpNameLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );
  xitAttachWidget( grpMemLa,
                   XmATTACH_WIDGET, grpNameTx, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );
  xitAttachWidget( XtParent( grpMemTx ),
                   XmATTACH_WIDGET, grpMemLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,     XmATTACH_NONE, NULL );
  xitAttachWidget( grpActionRc,
                   XmATTACH_WIDGET, XtParent( grpMemTx ),
                   XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( grpLa,                args, n );
  XtSetValues( XtParent( grpLi ),    args, n );
  XtSetValues( grpNameLa,            args, n );
  XtSetValues( grpNameTx,            args, n );
  XtSetValues( grpMemLa,             args, n );
  XtSetValues( XtParent( grpMemTx ), args, n );
  XtSetValues( grpActionRc,          args, n );

  /* Manage all the children. */
  XtManageChildren( grpActionBu, XtNumber( grpActionBu ) );

  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );


  /* Set the size of the window. */
  xitSetSizeFormDialog( groupFd, True );


  /* Make the final attachments. */
  xitAttachWidget( grpActionRc,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL );
  xitAttachWidget( XtParent( grpMemTx ),
                   XmATTACH_WIDGET, grpMemLa, XmATTACH_FORM,   NULL,
                   XmATTACH_FORM,   NULL,     XmATTACH_WIDGET, grpActionRc );
  xitAttachWidget( XtParent( grpLi ),
                   XmATTACH_WIDGET, grpLa, XmATTACH_FORM, NULL,
                   XmATTACH_FORM,   NULL,  XmATTACH_NONE, NULL );


  return( groupFd );

} /* createGroupWindow */


/*----------------------------------------------------------------------*/

static void
  setAction( GROUP_REC_REF  group_ref,
             Boolean        add_button,
             Boolean        delete_button )
{

  /* Variables. */
  Widget  mainW;
  Widget  tempW;


  /* Code. */

  mainW = XtNameToWidget( group_ref -> groupW, "GroupFdFo" );

  tempW = XtNameToWidget( mainW, "GrpActionRc.AddPb" );
  XtSetSensitive( tempW, add_button );

  tempW = XtNameToWidget( mainW, "GrpActionRc.DeletePb" );
  XtSetSensitive( tempW, delete_button );


  return;

} /* setAction */


/*----------------------------------------------------------------------*/

static void
  setGroupData( GROUP_REC_REF  group_ref,
                UINT32         action )
{

  /* Variables. */
  int           index = 0;
  int           index1;
  char          buffer[ 100 ];
  Arg           args[ 5 ];
  Cardinal      n;
  Widget        mainW;
  Widget        tempW;
  XmString      list_items[ 100 ];
  XTM_GD_ENTRY  group_info;


  /* Code. */

  mainW  = XtNameToWidget( group_ref -> groupW, "GroupFdFo" );


  /* Set values for the groups. */
  if( flagIsSet( action, SET_GROUP_LIST ) ) {

    Boolean  ok;
    int      char_read;
    char     group_name[ XTM_GD_MAX_GROUP_NAME + 1 ];
    char     *group_name_ref;
    char     *group_names;

    /* What groups do we have? */
    ok = xtmGdFetchGroupNames( group_ref -> edit_group_db_handle,
                               &group_names );
    if( ! ok )
      return;

    group_name_ref = group_names;

    do {

      while( isspace( *group_name_ref ) )
        group_name_ref++;

      if( *group_name_ref == '\0' )
        break;

      char_read = strlen( group_name_ref );
      sscanf( group_name_ref, "%s%n", group_name, &char_read );
      group_name_ref = group_name_ref + char_read;


      /* Fetch information about the database. */
      (void) xtmGdFetchNamedGroup( group_ref -> edit_group_db_handle,
                                   group_ref -> selected_group,
                                   &group_info );

      /* We cannot change included calendars. */
      if( flagIsSet( group_info.flags, XTM_GD_FLAG_IS_INCLUDED ) )
        continue;

      sprintf( buffer, "%s", group_name );

      list_items[ index ] = XmStringCreate( buffer, CS );
      index++;

    } while( True );

    SysFree( group_names );


    /* The list is always sorted. */
    xitSortStringList( list_items, index );

    tempW = XtNameToWidget( mainW, "GrpLiSW.GrpLi" );

    /* Assign the groups to the list. */
    n = 0;
    XtSetArg( args[ n ], XmNitems, list_items ); n++;
    XtSetArg( args[ n ], XmNitemCount, index ); n++;
    XtSetValues( tempW, args, n );

    XmListDeselectAllItems( tempW );

    group_ref -> selected_group[ 0 ] = '\0';

    /* Free allocated memory. */
    for( index1 = 0; index1 < index; index1++ )
      XmStringFree( list_items[ index1 ] );

  } /* if */


  /* Empty the group name and members field. */
  if( flagIsSet( action, RESET_GROUP_FIELD ) ) {

    tempW = XtNameToWidget( mainW, "GrpNameTx" );
    XmTextSetString( tempW, "" );

    tempW = XtNameToWidget( mainW, "GrpMemTxSW.GrpMemTx" );
    XmTextSetString( tempW, "" );

    group_ref -> field_changed = False;

  } /* if */


  return;

} /* setGroupData */


/*----------------------------------------------------------------------*/

static void
  addCB( Widget         widget, 
         GROUP_REC_REF  group_ref,
         XtPointer      call_data )
{

  /* Variables. */
  Boolean       exists;
  int           items;
  char          group_name[ XTM_GD_MAX_GROUP_NAME + 1 ];
  char          *char_ref;
  char          *members_ref;
  Widget        mainW;
  Widget        tempW;
  XTM_GD_ENTRY  group_info;


  /* Code. */

  mainW = XtNameToWidget( group_ref -> groupW, "GroupFdFo" );


  /* Fetch the name. */
  tempW = XtNameToWidget( mainW, "GrpNameTx" );

  char_ref = xitStringGetText( tempW );
  items    = sscanf( char_ref, "%15s", group_name );
  SysFree( char_ref );

  if( items != 1 )
    return;


  /* Fetch the members. */
  tempW = XtNameToWidget( mainW, "GrpMemTxSW.GrpMemTx" );
  members_ref = xitStringGetText( tempW );

  /* Replace all \n with blanks. */
  char_ref = members_ref;

  while( *char_ref != '\0' ) {
    if( *char_ref == '\n' )
      *char_ref = ' ';
    char_ref++;
  }


  /* Does the group exist? */
  exists = xtmGdFetchNamedGroup( group_ref -> edit_group_db_handle,
                                 group_name, &group_info );

  strcpy( group_info.name, group_name );
  group_info.members = members_ref;

  /* The group is already defined, change the directory. */
  if( exists ) {

    if( flagIsSet( group_info.flags, XTM_GD_FLAG_IS_INCLUDED ) ) {
      xitErMessage( group_ref -> groupW, XIT_ER_ERROR, 
                    module_name, "addCB",
                    msgGetText( MXDI_NO_MOD_INCL_GROUP_ERR ) );

      SysFree( members_ref );
      return;
    }

    (void) xtmGdChangeEntry( group_ref -> edit_group_db_handle,
                             group_name, &group_info );

  } /* if */


  /* The group does not exist, make a new one. */
  if( ! exists ) {

    (void) xtmGdAddEntry( group_ref -> edit_group_db_handle,
                          group_name, &group_info );

  } /* if */


  SysFree( members_ref );


  /* Refresh the data. */
  setGroupData( group_ref, (SET_GROUP_LIST | RESET_GROUP_FIELD) );

  setAction( group_ref, True, False );


  return;

} /* addCB */


/*----------------------------------------------------------------------*/

static void 
  cancelCB( Widget         widget,
            GROUP_REC_REF  group_ref,
            XtPointer      call_data )
{

  /* Code. */

  /* Skip the changes made */
  xtmGdFree( group_ref -> edit_group_db_handle );
  group_ref -> edit_group_db_handle = NULL;


  XtUnmanageChild( group_ref -> groupW );


  return;

} /* cancelCB */


/*----------------------------------------------------------------------*/

static void
  deleteCB( Widget         widget,
            GROUP_REC_REF  group_ref,
            XtPointer      call_data )
{

  /* Variables. */
  Widget  mainW;
  Widget  listW;


  /* Code. */

  mainW = XtNameToWidget( group_ref -> groupW, "GroupFdFo" );
  listW = XtNameToWidget( mainW, "GrpLiSW.GrpLi" );


  if( strlen( group_ref -> selected_group ) == 0 )
    return;


  /* Delete the group. */
  (void) xtmGdDeleteEntry( group_ref -> edit_group_db_handle,
                           group_ref -> selected_group );


  /* Refresh the data. */
  setGroupData( group_ref, (SET_GROUP_LIST | RESET_GROUP_FIELD) );

  setAction( group_ref, True, False );


  return;

} /* deleteCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget         widget,
             GROUP_REC_REF  group_ref,
             XtPointer      call_data )
{

  /* Code. */

  /* Do we have a user action callback registered? */
  if( group_ref -> actionCB != NULL )
    (* group_ref -> actionCB)( XTM_CG_REASON_DESTROY,
                               NULL,
                               group_ref -> user_data );


  /* Skip the changes made */
  if( group_ref -> edit_group_db_handle != NULL ) {
    xtmGdFree( group_ref -> edit_group_db_handle );
    group_ref -> edit_group_db_handle = NULL;
  }


  /* Release the user data. */
  SysFree( group_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void 
  fieldChangedCB( Widget         widget,
                  GROUP_REC_REF  group_ref,
                  XtPointer      call_data )
{

  /* Code. */

  group_ref -> field_changed = True;


  return;

} /* fieldChangedCB */


/*----------------------------------------------------------------------*/

static void 
  groupListCB( Widget                widget, 
               GROUP_REC_REF         group_ref,
               XmListCallbackStruct  *call_data )
{

  /* Variables. */
  char          *buffer_ref;
  char          *char_ref;
  char          *members_ref;
  Widget        mainW;
  Widget        tempW;
  XTM_GD_ENTRY  group_info;


  /* Code. */

  mainW = XtNameToWidget( group_ref -> groupW, "GroupFdFo" );


  /* Fetch the selected group line. */
  char_ref = xitStringGetString( call_data -> item, CS );

  sscanf( char_ref, "%s", group_ref -> selected_group );
  SysFree( char_ref );


  /* Fetch information about the group. */
  (void) xtmGdFetchNamedGroup( group_ref -> edit_group_db_handle,
                               group_ref -> selected_group,
                               &group_info );

  members_ref = group_info.members;

  if( members_ref == NULL ) {
    buffer_ref = SysNewString( " " );

  } else {

    int   char_read;
    char  member[ 250 ];

    buffer_ref  = (char *) SysMalloc( strlen( members_ref ) + 10 );
    *buffer_ref = '\0';

    do {

      while( isspace( *members_ref ) )
        members_ref++;

      if( *members_ref == '\0' )
        break;

      char_read = strlen( members_ref );
      sscanf( members_ref, "%s%n", member, &char_read );
      members_ref = members_ref + char_read;

      strcat( buffer_ref, member );
      strcat( buffer_ref, "\n" );

    } while( True );

  } /* if */


  /* Set name and members. */
  tempW = XtNameToWidget( mainW, "GrpNameTx" );
  XmTextSetString( tempW, group_ref -> selected_group );

  tempW = XtNameToWidget( mainW, "GrpMemTxSW.GrpMemTx" );
  XmTextSetString( tempW, buffer_ref );

  SysFree( buffer_ref );

  if( group_info.members != NULL )
    SysFree( group_info.members );


  setAction( group_ref, True, True );

  group_ref -> field_changed = False;


  return;

} /* groupListCB */


/*----------------------------------------------------------------------*/

static void 
  helpCB( Widget         widget,
          GROUP_REC_REF  group_ref,
          XtPointer      call_data )
{

  /* Code. */

  xtmHlDisplayHelp( group_ref -> appl_data_ref -> info_handle,
                    XTM_HL_WINDOW_INDEX,
                    cust_groups_window_id, "" );

  return;

} /* helpCB */


/*----------------------------------------------------------------------*/

static void 
  okCB( Widget         widget,
        GROUP_REC_REF  group_ref,
        XtPointer      call_data )
{

  /* Code. */

  /* Any changes that are not saved? */
  if( group_ref -> field_changed ) {
    (void) xitCreateQuestionDialog( 
             group_ref -> groupW, "QuestionDialog", 
	     msgGetText( MXDI_QUESTION_MESSAGE_LABEL ),
             msgGetText( MXDI_INFORMATION_NOT_SAVED ),
             okCB, group_ref,
             NULL, NULL );

    group_ref -> field_changed = False;

    return;
  }


  /* Do we have a user action callback registered? */
  if( group_ref -> actionCB != NULL )
    (* group_ref -> actionCB)( XTM_CG_REASON_OK,
                               group_ref -> edit_group_db_handle,
                               group_ref -> user_data );

  /* Free the edit group list. */
  xtmGdFree( group_ref -> edit_group_db_handle );
  group_ref -> edit_group_db_handle = NULL;


  XtUnmanageChild( group_ref -> groupW );


  return;

} /* okCB */
