/*----------------------------------------------------------------------------
--
--  Module:           xtmCalendar
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Control panel for XDiary. From this module, day list views can
--    be created and started. This module does also handle the communication
--    with the alarm processes.
--
--  Filename:         xtmCalendar.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1990-12-18
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmCalendar.c, Version: 1.1, Date: 95/02/18 15:51:59";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/StringDefs.h>
#include <X11/cursorfont.h>

#include <Xm/Protocols.h>

#include <Xm/Xm.h>
#include <Xm/CascadeB.h>
#include <Xm/Form.h>
#include <Xm/RowColumn.h>

#include "System.h"
#include "DirList.h"
#include "Message.h"
#include "TimDate.h"

#include "msgTopic.h"
#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmAlarmCtrl.h"
#include "xtmBaseCal.h"
#include "xtmCalDb.h"
#include "xtmCustBase.h"
#include "xtmDateSel.h"
#include "xtmDbMisc.h"
#include "xtmDbTools.h"
#include "xtmEditEntry.h"
#include "xtmHelp.h"
#include "xtmIcons.h"
#include "xtmMsgSelect.h"
#include "xtmMsgSend.h"
#include "xtmOpenView.h"
#include "xtmPlan.h"
#include "xtmPrEntries.h"
#include "xtmReminder.h"
#include "xtmSetup.h"
#include "xtmShowToDo.h"
#include "xtmTools.h"
#include "xtmUpdate.h"
#include "xtmHoliday.h"
#include "xtmSchedMain.h"
#include "xitError.h"
#include "xitTools.h"

#include "xtmCalendar.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Local widgets in the day list view window. */
#define calMainFo       dataLocalW[  0 ]
#define menuBr          dataLocalW[  1 ]



/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Record describing one instance of the day list view. */
typedef struct {

  /* Do we have messages in our mailbox? */
  Boolean  msg_in_mailbox;

  /* Id to use for update functions. */
  UINT32  update_id;

  /* The database we are using. */
  char  cal_name[ XTM_GL_MAX_CAL_NAME + 1 ];

  /* Time stamp for message flag. */
  time_t  message_stamp;

  /* Form containing the calendar including menubar. */
  Widget  calendarW;

  /* Form (without the dialog forms) containing the calenadr. */
  Widget  calFormW;

  /* Calendar icon window. */
  Window  cal_icon_window;

  /* Graphics Context (GC) to use when drawing in claendar icon window. */
  GC  cal_icon_gc;

  /* Date indicating which month the main calendar displays. */
  TIM_TIME_REF main_calendar_date;

  /* Calendar. */
  XTM_BC_HANDLE  cal_handle;

  /* Appointment editor reference. */
  XTM_ED_HANDLE  editor_handle;

  /* Application wide data. */
  XTM_GL_BASE_DATA_REF  appl_data_ref;

  /* Select received message reference. */
  XTM_ML_HANDLE  msg_sel_handle;

  /* Open view handle. */
  XTM_OV_HANDLE  open_handle;

  /* Reminder reference. */
  XTM_RD_REMINDER_REF  reminder_ref;

  /* Printer reference. */
  XTM_PR_HANDLE  pr_handle;

  /* Summary reference. */
  XTM_SD_HANDLE  show_handle;

  /* Callback to inform our creator of specific actions. */
  void              *user_data;
  XTM_CA_ACTION_CB  actionCB;

} CALENDAR_REC, *CALENDAR_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmCalendar";

/* IDs for the help windows. */
static char  *cal_window_id    = "Calendar";
static char  *remind_window_id = "Reminder";

/* Trace double clicks on entries. */
static Time  time_last = 0;


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  alarmProcessesCB( Widget            widget, 
                    CALENDAR_REC_REF  cal_ref,
                    XtPointer         call_data );

static void 
  appEditorActionCB( XTM_ED_REASON reason,
                     void          *user_data );

static void
  appEditorCB( Widget            widget, 
               CALENDAR_REC_REF  cal_ref,
               XtPointer         call_data );

static void 
  calendarActionCB( XTM_BC_REASON        reason,
                    TIM_TIME_REF         time,
                    XmAnyCallbackStruct  *call_data,
                    void                 *user_data );

static void 
  calendarEntriesCB( XTM_BC_REASON  reason,
                     void           *user_data,
                     TIM_TIME_REF   start_date,
                     int            days,
                     Boolean        *entries );

static Widget
  createCalendar( Widget            parent,
                  CALENDAR_REC_REF  cal_ref,
                  Boolean           menu_bar );

static void
  customizeCB( Widget            widget, 
               CALENDAR_REC_REF  cal_ref,
               XtPointer         call_data );

static void 
  destroyCB( Widget            widget,
             CALENDAR_REC_REF  cal_ref,
             XtPointer         call_data );

static void
  exitCB( Widget            widget,
          CALENDAR_REC_REF  cal_ref,
          XtPointer         call_data );

static void
  exitReallyCB( Widget            widget,
                CALENDAR_REC_REF  cal_ref,
                XtPointer         call_data );

static void 
  infoCB( Widget                     widget,
          CALENDAR_REC_REF           cal_ref,
          XmRowColumnCallbackStruct  *call_data );

static void 
  infoReminderCB( Widget  widget,
                  void    *user_data,
                  int     help_index );

static void
  monthExtendCB( Widget            widget, 
                 CALENDAR_REC_REF  cal_ref,
                 XtPointer         call_data );

static void
  openViewCB( Widget            widget, 
              CALENDAR_REC_REF  cal_ref,
              XtPointer         call_data );

static void
  openViewApplyCB( XTM_OV_REASON    reason,
                   XTM_CD_CAL_INFO  *db_info_ref,
                   void             *user_data );

static void
  plannerCB( Widget            widget,
             CALENDAR_REC_REF  cal_ref,
             XtPointer         call_data );

static void
  plannerActionCB( XTM_PL_REASON  reason,
                   void           *user_data );

static void 
  printCB( Widget            widget,
           CALENDAR_REC_REF  cal_ref,
           XtPointer         call_data );

static void
  printActionCB( XTM_PR_REASON  reason,
                 void           *user_data );

static void
  receiveMessageCB( Widget            widget,
                    CALENDAR_REC_REF  cal_ref,
                    XtPointer         call_data );

static void
  redisplayCB( Widget            widget,
               CALENDAR_REC_REF  cal_ref,
               XtPointer         call_data );

static void
  reminderCB( Widget            widget,
              CALENDAR_REC_REF  cal_ref,
              XtPointer         call_data );

static void
  rereadCustCB( Widget            widget,
                CALENDAR_REC_REF  cal_ref,
                XtPointer         call_data );

static void
  sendMessageCB( Widget            widget,
                 CALENDAR_REC_REF  cal_ref,
                 XtPointer         call_data );

static void
  summaryCB( Widget            widget,
             CALENDAR_REC_REF  cal_ref,
             XtPointer         call_data );

static void
  summaryActionCB( XTM_SD_REASON  reason,
                   void           *user_data );

static void
  updateCalendar( CALENDAR_REC_REF  cal_ref );

void
  updateCalendarTime( CALENDAR_REC_REF  cal_ref );

static void
  updateCB( UINT32  flags,
            void    *user_data,
            void    *update_user_data );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XTM_CA_HANDLE
  xtmCaInitialize( XTM_GL_BASE_DATA_REF  appl_data_ref,
                   Widget                toplevel,
                   Boolean               toplevel_form,
                   Boolean               menu_bar,
                   char                  *cal_name,
                   XTM_CA_ACTION_CB      actionCB,
                   void                  *user_data )
{

  /* Variables. */
  char              buffer[ 50 ];
  Arg               args[ 5 ];
  Cardinal          n;
  CALENDAR_REC_REF  cal_ref;


  /* Code. */

  /* Create and initialize our private data. */
  cal_ref = SysNew( CALENDAR_REC );
  if( cal_ref == NULL )
    return( NULL );

  cal_ref -> appl_data_ref      = appl_data_ref;
  cal_ref -> cal_handle         = NULL;
  cal_ref -> editor_handle      = NULL;
  cal_ref -> open_handle        = NULL;
  cal_ref -> pr_handle          = NULL;
  cal_ref -> show_handle        = NULL;
  cal_ref -> msg_sel_handle     = NULL;
  cal_ref -> reminder_ref       = NULL;
  cal_ref -> update_id          = 0;
  cal_ref -> main_calendar_date = 0;
  cal_ref -> message_stamp      = 0;
  cal_ref -> msg_in_mailbox     = False;
  cal_ref -> cal_icon_window    = None;
  cal_ref -> cal_icon_gc        = NULL;
  cal_ref -> actionCB           = actionCB;
  cal_ref -> user_data          = user_data;

  strcpy( cal_ref -> cal_name, cal_name );


  /* Create the calendar window. */
  cal_ref -> calendarW = createCalendar( toplevel, cal_ref, menu_bar );
  if( cal_ref -> calendarW == NULL ) {
    SysFree( cal_ref );
    return( NULL );
  }

  /* Set the calendar title. */
  sprintf( buffer, "%s %s", cal_name, appl_data_ref -> title );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, buffer ); n++;
  XtSetArg( args[ n ], XmNiconName, cal_name ); n++;
  XtSetValues( cal_ref -> calendarW, args, n );


  return( (XTM_CA_HANDLE) cal_ref );

} /* xtmCaInitialize */


/*----------------------------------------------------------------------*/

void
  xtmCaDestroy( XTM_CA_HANDLE  cal_handle )
{

  /* Variables. */
  CALENDAR_REC_REF  cal_ref;


  /* Code. */

  if( cal_handle == NULL )
    return;

  /* Our private data. */
  cal_ref = (CALENDAR_REC_REF) cal_handle;


  /* Get rid of all windows. */
  if( cal_ref -> calendarW != NULL )
    XtDestroyWidget( cal_ref -> calendarW );


  return;

} /* xtmCaDestroy */


/*----------------------------------------------------------------------*/

void
  xtmCaView( XTM_CA_HANDLE  cal_handle,
             TIM_TIME_REF   at_date )
{

  /* Variables. */
  CALENDAR_REC_REF  cal_ref;


  /* Code. */

  cal_ref = (CALENDAR_REC_REF) cal_handle;


  /* When? */
  cal_ref -> main_calendar_date = TimMakeTime( TimIndexOfYear(  at_date ),
                                               TimIndexOfMonth( at_date ),
                                               TimIndexOfDay(   at_date ),
                                               0, 0, 0 );

  /* Register for updates? */
  (void) xtmUpRegister( (XTM_UP_MINUTE_TICK | XTM_UP_CALENDAR ), 
                        updateCB, 
                        (void *) cal_ref );


  /* Update calendar time. */
  xtmUpDoUpdate( (XTM_UP_MINUTE_TICK | XTM_UP_CALENDAR), 
                 (void *) cal_ref -> main_calendar_date );

  /* Display the calendar. */
  XtPopup( cal_ref -> calendarW, XtGrabNone );


  return;

} /* xtmCaView */


/*----------------------------------------------------------------------*/

Widget
  xtmCaMainWindow( XTM_CA_HANDLE  cal_handle )
{

  /* Variables. */
  CALENDAR_REC_REF  cal_ref;


  /* Code. */

  if( cal_handle == NULL )
    return( NULL );

  cal_ref = (CALENDAR_REC_REF) cal_handle;


  return( cal_ref -> calFormW );

} /* xtmCaMainWindow */


/*----------------------------------------------------------------------*/

Widget
  xtmCaToplevelWindow( XTM_CA_HANDLE  cal_handle )
{

  /* Variables. */
  CALENDAR_REC_REF  cal_ref;


  /* Code. */

  if( cal_handle == NULL )
    return( NULL );

  cal_ref = (CALENDAR_REC_REF) cal_handle;


  return( cal_ref -> calendarW );

} /* xtmCaToplevelWindow */


/*----------------------------------------------------------------------*/

void
  xtmCaProcessEvents( XTM_CA_HANDLE  cal_handle,
                      XEvent         *event )
{

  /* Variables. */
  CALENDAR_REC_REF  cal_ref;


  /* Code. */

  if( cal_handle == NULL )
    return;

  cal_ref = (CALENDAR_REC_REF) cal_handle;

  /* Icon window event? */
  if( event -> xany.window == cal_ref -> cal_icon_window )
    updateCalendarTime( cal_ref );


  return;

} /* xtmCaProcessEvents */


/*----------------------------------------------------------------------*/

static Widget
  createCalendar( Widget            parent,
                  CALENDAR_REC_REF  cal_ref,
                  Boolean           menu_bar )
{

  /* Variables. */
  int                   index;
  Arg                   args[ 5 ];
  Cardinal              n;
  Widget                calTl;
  Widget                calFo;
  Widget                dataLocalW[ 2 ];
  Widget                menuCasc[ 3 ];
  Widget                menuHelpBu[ 7 ];
  Widget                menuFileBu[ 6 ];
  Widget                menuOptionsBu[ 12 ];
  Widget                pulldownMenu[ 3 ];
  Widget                tempW;
  Widget                toplevel;
  XTM_GL_BASE_DATA_REF  appl_data_ref;

  static char  *pull_downs[] = { "pdown1", "pdown2", "pdown3" };

  static XIT_CASCADE_STRUCT menupane[] = {
    { "", "", "FilePane" },
    { "", "", "OptionsPane" },
    { "", "", "HelpPane" },
  };

  static XIT_MENU_BUTTON_STRUCT file_casc[] = {
    { "",      "",  NULL, "OpenViewBu", True,  False, False },
    { "",      "",  NULL, "AlarmPrcBu", True,  False, False },
    { XIT_SEP, " ", NULL, "Ho1Sp",      False, False, False },
    { "",      "",  NULL, "PrintBu",    True,  False, False },
    { XIT_SEP, " ", NULL, "Ho2Sp",      False, False, False },
    { "",      "",  NULL, "ExitBu",     True,  False, False },
  };

  static XIT_MENU_BUTTON_STRUCT opt_casc[] = {
    { "",      "",  NULL, "ApEditBu",    True,  False, False },
    { "",      "",  NULL, "RemindBu",    True,  False, False },
    { "",      "",  NULL, "CustomBu",    True,  False, False },
    { "",      "",  NULL, "SummaryBu",   True,  False, False },
    { "",      "",  NULL, "PlannerBu",   True,  False, False },
    { XIT_SEP, " ", NULL, "Ho1Sp",       False, False, False },
    { "",      "",  NULL, "SmsgBu",      True,  False, False },
    { "",      "",  NULL, "RmsgBu",      True,  False, False },
    { XIT_SEP, " ", NULL, "Ho1Sp",       False, False, False },
    { "",      "",  NULL, "MonthExtBu",  True,  False, False },
    { "",      "",  NULL, "RedisplayBu", True,  False, False },
    { "",      "",  NULL, "RereadBu",    True,  False, False },
  };

  static XIT_MENU_BUTTON_STRUCT help_casc[] = {
    { "", "", NULL, "ContextBu", True, False, False },
    { "", "", NULL, "WindowsBu", True, False, False },
    { "", "", NULL, "KeysBu",    True, False, False },
    { "", "", NULL, "IndexBu",   True, False, False },
    { "", "", NULL, "HelpBu",    True, False, False },
    { "", "", NULL, "VersionBu", True, False, False },
    { "", "", NULL, "AboutBu",   True, False, False },
  };


  /* Code. */

  appl_data_ref = cal_ref -> appl_data_ref;


  /* Initialize the menu and label items. */
  menupane[ 0 ].title    = msgGetText( MXDI_FILE_MENU );
  menupane[ 0 ].mnemonic = msgGetText( MXDI_FILE_MENU_ACC );
  menupane[ 1 ].title    = msgGetText( MXDI_OPT_MENU );
  menupane[ 1 ].mnemonic = msgGetText( MXDI_OPT_MENU_ACC );
  menupane[ 2 ].title    = msgGetText( MXDI_HELP_MENU );
  menupane[ 2 ].mnemonic = msgGetText( MXDI_HELP_MENU_ACC );

  file_casc[ 0 ].title    = msgGetText( MXDI_OPEN_VIEW_MENU );
  file_casc[ 0 ].mnemonic = msgGetText( MXDI_OPEN_VIEW_MENU_ACC );
  file_casc[ 1 ].title    = msgGetText( MXDI_ALARM_PRC_MENU );
  file_casc[ 1 ].mnemonic = msgGetText( MXDI_ALARM_PRC_MENU_ACC );
  file_casc[ 3 ].title    = msgGetText( MXDI_PRINT_MENU );
  file_casc[ 3 ].mnemonic = msgGetText( MXDI_PRINT_MENU_ACC );
  file_casc[ 5 ].title    = msgGetText( MXDI_EXIT_MENU );
  file_casc[ 5 ].mnemonic = msgGetText( MXDI_EXIT_MENU_ACC );

  opt_casc[  0 ].title    = msgGetText( MXDI_APP_EDITOR_MENU );
  opt_casc[  0 ].mnemonic = msgGetText( MXDI_APP_EDITOR_MENU_ACC );
  opt_casc[  1 ].title    = msgGetText( MXDI_REMIND_MENU );
  opt_casc[  1 ].mnemonic = msgGetText( MXDI_REMIND_MENU_ACC );
  opt_casc[  2 ].title    = msgGetText( MXDI_CUSTOM_MENU );
  opt_casc[  2 ].mnemonic = msgGetText( MXDI_CUSTOM_MENU_ACC );
  opt_casc[  3 ].title    = msgGetText( MXDI_SUM_MENU );
  opt_casc[  3 ].mnemonic = msgGetText( MXDI_SUM_MENU_ACC );
  opt_casc[  4 ].title    = msgGetText( MXDI_PLAN_MENU );
  opt_casc[  4 ].mnemonic = msgGetText( MXDI_PLAN_MENU_ACC );
  opt_casc[  6 ].title    = msgGetText( MXDI_SEND_MSG_MENU );
  opt_casc[  6 ].mnemonic = msgGetText( MXDI_SEND_MSG_MENU_ACC );
  opt_casc[  7 ].title    = msgGetText( MXDI_RCV_MSG_MENU );
  opt_casc[  7 ].mnemonic = msgGetText( MXDI_RCV_MSG_MENU_ACC );
  opt_casc[  9 ].title    = msgGetText( MXDI_CAL_MONTH_EXTEND_MENU );
  opt_casc[  9 ].mnemonic = msgGetText( MXDI_CAL_MONTH_EXTEND_MENU_ACC );
  opt_casc[ 10 ].title    = msgGetText( MXDI_REDISPLAY_MENU );
  opt_casc[ 10 ].mnemonic = msgGetText( MXDI_REDISPLAY_MENU_ACC );
  opt_casc[ 11 ].title    = msgGetText( MXDI_REREAD_CUST_MENU );
  opt_casc[ 11 ].mnemonic = msgGetText( MXDI_REREAD_CUST_MENU_ACC );

  help_casc[ 0 ].title    = msgGetText( MXDI_HELP_CONTEXT );
  help_casc[ 0 ].mnemonic = msgGetText( MXDI_HELP_CONTEXT_ACC );
  help_casc[ 1 ].title    = msgGetText( MXDI_HELP_WINDOWS );
  help_casc[ 1 ].mnemonic = msgGetText( MXDI_HELP_WINDOWS_ACC );
  help_casc[ 2 ].title    = msgGetText( MXDI_HELP_KEYS );
  help_casc[ 2 ].mnemonic = msgGetText( MXDI_HELP_KEYS_ACC );
  help_casc[ 3 ].title    = msgGetText( MXDI_HELP_INDEX );
  help_casc[ 3 ].mnemonic = msgGetText( MXDI_HELP_INDEX_ACC );
  help_casc[ 4 ].title    = msgGetText( MXDI_HELP_HELP );
  help_casc[ 4 ].mnemonic = msgGetText( MXDI_HELP_HELP_ACC );
  help_casc[ 5 ].title    = msgGetText( MXDI_HELP_VERSION );
  help_casc[ 5 ].mnemonic = msgGetText( MXDI_HELP_VERSION_ACC );
  help_casc[ 6 ].title    = msgGetText( MXDI_HELP_ABOUT );
  help_casc[ 6 ].mnemonic = msgGetText( MXDI_HELP_ABOUT_ACC );


  /* Create the calendar shell. */
  toplevel = xitGetToplevelWidget( parent );

  calTl = xitCreateToplevelDialog( toplevel, "CalTl", 0, 0, NULL, 0 );

  XtAddCallback( calTl, XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) cal_ref );

  /* Exit the application if this window is deleted. */
  {
    Atom  wm_delete_window;

    wm_delete_window = XmInternAtom( XtDisplay( calTl ),
                                     "WM_DELETE_WINDOW", False );

    XmAddWMProtocols( calTl, &wm_delete_window, 1 );
    XmAddWMProtocolCallback( calTl, wm_delete_window, 
                             (XtCallbackProc) exitCB, (XtPointer) cal_ref );
  } /* block */


  /* Form to enclose the calendar. */
  calMainFo = XtNameToWidget( calTl, "CalTlBase.CalTlFo" );
  cal_ref -> calFormW = calMainFo;


  /* Dummy widget, never managed but needed later. */
  tempW = xitCreateLabelWidget( cal_ref -> calFormW, "HiddenLa", "", -1 );


  /* Create the menubar and menu cascade buttons. */
  n = 0;
  menuBr = XmCreateMenuBar( cal_ref -> calFormW, "MenuBr", args, n );

  n = 0;
  for( index = 0; index < XtNumber( pull_downs ); index++ )
    pulldownMenu[ index ] = XmCreatePulldownMenu( menuBr,
                                                  pull_downs[ index ], 
                                                  args, n );

#if XmVersion > 1001 
  n = 0;
  XtSetArg( args[ n ], XmNtearOffModel, XmTEAR_OFF_ENABLED ); n++;
  XtSetValues( pulldownMenu[ 0 ], args, n );
  XtSetValues( pulldownMenu[ 1 ], args, n );
  XtSetValues( pulldownMenu[ 2 ], args, n );
#endif

  for( index = 0; index < XtNumber( menupane ); index++ )
    menuCasc[ index ] = xitCreateCascadeButton( menuBr, 
                                                pulldownMenu[ index ], 
                                                &menupane[ index ] );


  /* The help button should be placed to the right. */
  index = XtNumber( menupane ) - 1;
  n     = 0;
  XtSetArg( args[ n ], XmNmenuHelpWidget, menuCasc[ index ] ); n++;
  XtSetValues( menuBr, args, n );


  /* Create the file menu. */
  for( index = 0; index < XtNumber( menuFileBu ); index++ )
    menuFileBu[ index ] = xitCreateMenuPushButton( pulldownMenu[ 0 ], 
                                                   &file_casc[ index ] );

  XtAddCallback( menuFileBu[ 0 ],  XmNactivateCallback, 
                 (XtCallbackProc) openViewCB, (XtPointer) cal_ref );
  XtAddCallback( menuFileBu[ 1 ],  XmNactivateCallback, 
                 (XtCallbackProc) alarmProcessesCB, (XtPointer) cal_ref );
  XtAddCallback( menuFileBu[ 3 ],  XmNactivateCallback, 
                 (XtCallbackProc) printCB, (XtPointer) cal_ref );
  XtAddCallback( menuFileBu[ 5 ],  XmNactivateCallback, 
                 (XtCallbackProc) exitCB, (XtPointer) cal_ref );


  /* Create the options menu. */
  for( index = 0; index < XtNumber( menuOptionsBu ); index++ )
    menuOptionsBu[ index ] = xitCreateMenuPushButton( pulldownMenu[ 1 ], 
                                                      &opt_casc[ index ] );

  XtAddCallback( menuOptionsBu[ 0 ], XmNactivateCallback, 
                 (XtCallbackProc) appEditorCB, (XtPointer) cal_ref );
  XtAddCallback( menuOptionsBu[ 1 ], XmNactivateCallback, 
                 (XtCallbackProc) reminderCB, (XtPointer) cal_ref );
  XtAddCallback( menuOptionsBu[ 2 ], XmNactivateCallback, 
                 (XtCallbackProc) customizeCB, (XtPointer) cal_ref );
  XtAddCallback( menuOptionsBu[ 3 ], XmNactivateCallback, 
                 (XtCallbackProc) summaryCB, (XtPointer) cal_ref );
  XtAddCallback( menuOptionsBu[ 4 ], XmNactivateCallback, 
                 (XtCallbackProc) plannerCB, (XtPointer) cal_ref );
  XtAddCallback( menuOptionsBu[ 6 ], XmNactivateCallback, 
                 (XtCallbackProc) sendMessageCB, (XtPointer) cal_ref );
  XtAddCallback( menuOptionsBu[ 7 ], XmNactivateCallback, 
                 (XtCallbackProc) receiveMessageCB, (XtPointer) cal_ref );
  XtAddCallback( menuOptionsBu[ 9 ], XmNactivateCallback, 
                 (XtCallbackProc) monthExtendCB, (XtPointer) cal_ref );
  XtAddCallback( menuOptionsBu[ 10 ], XmNactivateCallback, 
                 (XtCallbackProc) redisplayCB, (XtPointer) cal_ref );
  XtAddCallback( menuOptionsBu[ 11 ], XmNactivateCallback, 
                 (XtCallbackProc) rereadCustCB, (XtPointer) cal_ref );


  /* Create the help menu. */
  XtAddCallback( pulldownMenu[ 2 ], XmNentryCallback, 
                 (XtCallbackProc) infoCB, (XtPointer) cal_ref );

  for( index = 0; index < XtNumber( menuHelpBu ); index++ ) {
    menuHelpBu[ index ] = xitCreateMenuPushButton( pulldownMenu[ 2 ], 
                                                   &help_casc[ index ] );

    XtAddCallback( menuHelpBu[ index ], XmNactivateCallback, 
                   (XtCallbackProc) infoCB, (XtPointer) index );
  }

  /* We can't do context sensitive help. */
  XtSetSensitive( menuHelpBu[ 0 ], False );


  /* Create the base calendar. */
  if( cal_ref -> cal_handle == NULL ) {

    XTM_BC_CAL_STYLE  style;

    if( appl_data_ref -> custom_data -> cal_is_fancy )
      style = XTM_BC_STYLE_NOTEBOOK;
    else
      style = XTM_BC_STYLE_PLAIN;

    cal_ref -> cal_handle =
      xtmBcInitialize( cal_ref -> appl_data_ref, cal_ref -> calFormW,
                       style, calendarActionCB, calendarEntriesCB,
                       (void *) cal_ref );

    if( cal_ref -> cal_handle == NULL )
      return( NULL );

    if( style == XTM_BC_STYLE_NOTEBOOK ) {
      calFo = xtmBcGetWidget( cal_ref -> cal_handle );

      n = 0;
      XtSetArg( args[ n ], XmNtopOffset, 5 ); n++;
      XtSetArg( args[ n ], XmNleftOffset, 5 ); n++;
      XtSetArg( args[ n ], XmNrightOffset, 5 ); n++;
      XtSetValues( calFo, args, n );
    }
  }

  calFo = xtmBcGetWidget( cal_ref -> cal_handle );


  /* Position the widgets. */
  xitAttachWidget( menuBr,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_FORM, NULL, XmATTACH_NONE, NULL );
  if( menu_bar )
    xitAttachWidget( calFo,
                     XmATTACH_WIDGET, menuBr, XmATTACH_FORM, NULL,
                     XmATTACH_NONE,   NULL,   XmATTACH_NONE, NULL );
  else
    xitAttachWidget( calFo,
                     XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                     XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );


  /* Manage the widgets. */
  XtManageChildren( menuCasc,      XtNumber( menupane ) );
  XtManageChildren( menuFileBu,    XtNumber( menuFileBu ) );
  XtManageChildren( menuHelpBu,    XtNumber( menuHelpBu ) );
  XtManageChildren( menuOptionsBu, XtNumber( menuOptionsBu ) );

  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );

  /* No menu bar? */
  if( ! menu_bar )
    XtUnmanageChild( menuBr );


  /* Manage the base calendar. */
  xtmBcViewCalendar( cal_ref -> cal_handle );

  /* Set the size of the window. */
  xitSetSizeToplevelDialog( calTl, True );


  /* Create and initialize the reminder window. */
  cal_ref -> reminder_ref = 
    xtmRdInitializeReminder( toplevel, appl_data_ref -> context,
                             NULL, infoReminderCB, cal_ref );


  return( calTl );

} /* createCalendar */


/*----------------------------------------------------------------------*/

static void
  updateCalendar( CALENDAR_REC_REF  cal_ref )
{

  /* Code. */

  xtmBcDisplayMonth( cal_ref -> cal_handle, cal_ref -> main_calendar_date );


  return;  

} /* updateCalendar */


/*----------------------------------------------------------------------*/

void
  updateCalendarTime( CALENDAR_REC_REF  cal_ref )
{

  /* Variables. */
  char                    buffer[ 50 ];
  char                    tmp_buffer[ 50 ];
  char                    time_buffer[ 50 ];
  Widget                  tempW;
  TIM_TIME_REF            now;
  XIT_ICON_WINDOW         icon_win;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;

  static Pixmap  icon_pixmap        = None;
  static Pixmap  icon_invert_pixmap = None;


  /* Code. */

  appl_data_ref   = cal_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;

  now = TimLocalTime( TimMakeTimeNow() );


  /* Update the calendar time. */
  xtmBcDisplayTime( cal_ref -> cal_handle );


  /* Set the calendar icon. */
  icon_win.report_expose_events = True;
  icon_win.x_offset             = 0;
  icon_win.y_offset             = 0;

  time_buffer[ 0 ] = '\0';

  /* Display date, month and time in the window? */
  if( strcmp( custom_data_ref -> cal_icon_text, "time" ) == 0 ) {
    icon_win.x_offset = -1;
    icon_win.y_offset = -5;

    TimFormatStrTime( now, "%d\n", buffer, sizeof( buffer ) );
    if( buffer[ 0 ] == '0' ) {
      strcpy( tmp_buffer, &buffer[ 1 ] );
      strcpy( buffer, tmp_buffer );
    }
    strcat( time_buffer, buffer );

    TimFormatStrTime( now, "%b\n", buffer, sizeof( buffer ) );
    strcat( time_buffer, buffer );

    TimFormatTime( now, buffer, sizeof( buffer ) );

    if( buffer[ 0 ] == '0' ) {
      strcpy( tmp_buffer, &buffer[ 1 ] );
      strcpy( buffer, tmp_buffer );
    }
    strcat( time_buffer, buffer );
  }


  /* Create/display the icon window. */
  if( strcmp( custom_data_ref -> cal_icon_text, "none" ) == 0 )
    return;

  {
    Arg       args[ 5 ];
    Cardinal  n;
    Pixel     background;
    Pixel     foreground;
    XmString  xstr;

    /* Icon strings. */
    xstr = XmStringCreateLtoR( time_buffer, CS );
    icon_win.label = xstr;


    /* Color for the icon. */
    n = 0;
    XtSetArg( args[ n ], XmNbackground, &background ); n++;
    XtSetArg( args[ n ], XmNforeground, &foreground ); n++;
    XtGetValues( cal_ref -> calFormW, args, n );

    if( cal_ref -> msg_in_mailbox ) {
      icon_win.icon_fg  = background;
      icon_win.icon_bg  = foreground;
      icon_win.label_fg = background;
    } else {
      icon_win.icon_fg  = foreground;
      icon_win.icon_bg  = background;
      icon_win.label_fg = foreground;
    }

    /* Icon pixmap. */
    if( icon_pixmap == None )
      icon_pixmap = xtmIcFetchSimplePixmap( cal_ref -> calFormW, 
                                            XTM_IC_ICON_CALENDAR,
                                            False );

    if( icon_invert_pixmap == None )
      icon_invert_pixmap = xtmIcFetchSimplePixmap( cal_ref -> calFormW, 
                                                   XTM_IC_ICON_CALENDAR,
                                                   True );

    if( cal_ref -> msg_in_mailbox )
      icon_win.image = icon_invert_pixmap;
    else
      icon_win.image = icon_pixmap;


    /* Font list. */
    tempW = XtNameToWidget( cal_ref -> calFormW, "HiddenLa" );

    n = 0;
    XtSetArg( args[ n ], XmNfontList, &icon_win.font_list ); n++;
    XtGetValues( tempW, args, n );


    /* Set the icon. */
    xitSetIconWindow( cal_ref  -> calendarW, &icon_win, 
                      &cal_ref -> cal_icon_window, 
                      &cal_ref -> cal_icon_gc );

    XmStringFree( xstr );

  } /* block */


  return;

} /* updateCalendarTime */


/*----------------------------------------------------------------------*/

static void
  alarmProcessesCB( Widget            widget, 
                    CALENDAR_REC_REF  cal_ref,
                    XtPointer         call_data )
{

  /* Code. */

  /* Open a day list view. */
  xtmAcAlarmProcessControl( cal_ref -> calendarW, 
                            cal_ref -> appl_data_ref );


  return;

} /* alarmProcessesCB */


/*----------------------------------------------------------------------*/

static void 
  appEditorActionCB( XTM_ED_REASON reason,
                     void          *user_data )
{

  /* Variables. */
  CALENDAR_REC_REF  cal_ref;


  /* Code. */

  cal_ref = (CALENDAR_REC_REF) user_data;

  if( reason == XTM_ED_REASON_DESTROY )
    cal_ref -> editor_handle = NULL;


  return;

} /* appEditorActionCB */


/*----------------------------------------------------------------------*/

static void
  appEditorCB( Widget            widget, 
               CALENDAR_REC_REF  cal_ref,
               XtPointer         call_data )
{

  /* Variables. */
  Boolean                 ok;
  TIM_TIME_REF            default_time;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         db_info;


  /* Code. */

  appl_data_ref   = cal_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;


  /* Fetch the database we are using. */
  ok = xtmCdFetchNamedDb( appl_data_ref -> custom_data -> cal_db_handle,
                          cal_ref -> cal_name,
                          &db_info, NULL );
  if( ! ok )
    return;


  /* Initialize a new editor record? */
  if( cal_ref -> editor_handle == NULL )
    cal_ref -> editor_handle = xtmEdInitialize( 
                                 appl_data_ref,
                                 appl_data_ref   -> toplevel,
                                 custom_data_ref -> default_entry_delta,
                                 custom_data_ref -> start_hour,
                                 custom_data_ref -> stop_hour,
                                 custom_data_ref -> default_entry_delta,
                                 appEditorActionCB, 
                                 (void *) cal_ref );


  default_time = TimMakeTime( 1970, 1, 1, 12, 0, 0 );

  /* The appointment editor. */
  xtmEdEditEntry( cal_ref -> editor_handle,
                  db_info.short_name,
                  0, 
                  cal_ref -> main_calendar_date, 
                  default_time );


  return;

} /* appEditorCB */


/*----------------------------------------------------------------------*/

static void 
  calendarActionCB( XTM_BC_REASON        reason,
                    TIM_TIME_REF         time,
                    XmAnyCallbackStruct  *call_data,
                    void                 *user_data )
{

  /* Variables. */                    
  Time                time_now;
  CALENDAR_REC_REF    cal_ref;
  XTM_CA_REASON       cal_reason;


  /* Code. */

  cal_ref = (CALENDAR_REC_REF) user_data;

  /* Message button selected. */
  if( reason == XTM_BC_REASON_MSG_SELECTED ) {
    receiveMessageCB( NULL, cal_ref, NULL );

    return;
  }


  /* Month sheet displayed? */
  if( reason == XTM_BC_REASON_MONTH_DISPLAYED ) {
    cal_ref -> main_calendar_date = time;

    return;
  }


  /* Date selected? */
  if( reason == XTM_BC_REASON_DATE_SELECTED ) {

    /* Ignore all double clicks. */
    if( call_data -> event -> xany.type != ButtonRelease )
      return;

    time_now = call_data -> event -> xbutton.time;

    if( abs( time_now - time_last ) < 750 && time_last != 0 ) {
      time_last = time_now;
      return;
    }

    time_last = time_now;


    /* Do we have a user action callback registered? */
    if( cal_ref -> actionCB != NULL ) {

      Cursor   wait_cursor;
      Display  *display = XtDisplay( cal_ref -> calendarW );
      Window   window   = XtWindow(  cal_ref -> calendarW );

      /* This might take time. */
      wait_cursor = XCreateFontCursor( display, XC_watch );
      XDefineCursor( display, window, wait_cursor );

      XFlush( display );

      /* Reason? */
      if( (call_data -> event -> xbutton.state & ShiftMask) != 0 )
        cal_reason = XTM_CA_REASON_DATE_SELECT;
      else
        cal_reason = XTM_CA_REASON_NEW_DATE;

      if( cal_ref -> appl_data_ref -> custom_data -> click_date_new_view )
        cal_reason = XTM_CA_REASON_DATE_SELECT;

      /* User callback. */
      (* cal_ref -> actionCB)( cal_reason,
                               (void *) time, (void *) cal_ref -> user_data );

      XUndefineCursor( display, window );

    } /* if */

    return;

  } /* if */


  return;

} /* calendarActionCB */


/*----------------------------------------------------------------------*/

static void 
  calendarEntriesCB( XTM_BC_REASON  reason,
                     void           *user_data,
                     TIM_TIME_REF   start_date,
                     int            days,
                     Boolean        *entries )
{

  /* Variables. */                    
  CALENDAR_REC_REF        cal_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  cal_ref = (CALENDAR_REC_REF) user_data;

  custom_data_ref = cal_ref -> appl_data_ref -> custom_data;

  /* Fetch the entries defined. */
  xtmDmEntriesDefined( cal_ref -> appl_data_ref,
                       cal_ref -> cal_name,
                       start_date, days, True,
                       custom_data_ref -> show_stand_in_cal,
                       entries );


  return;

} /* calendarEntriesCB */


/*----------------------------------------------------------------------*/

static void
  customizeCB( Widget            widget, 
               CALENDAR_REC_REF  cal_ref,
               XtPointer         call_data )
{

  /* Variables. */
  int                     index;
  int                     n;
  char                    **process_args;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = cal_ref -> appl_data_ref -> custom_data;

  /* Build the parameters to pass. */
  process_args = (char**) XtMalloc( (custom_data_ref -> orig_argc + 10) *
                                    sizeof( char* ) );

  n = 0;  
  process_args[ n ] = custom_data_ref -> custom_script; n++;

  for( index = 1; index < custom_data_ref -> orig_argc; index++ ) {
    process_args[ n ] = custom_data_ref -> orig_argv[ index ]; n++;
  }
    
  process_args[ n ] = NULL;

  /* Start the customize program. */
  (void) xtmToStartProcess( cal_ref -> calendarW, True,
                            msgGetText( MXDI_STARTING_PROCESS_MSG ),
                            process_args );

  XtFree( (char*) process_args );


  return;

} /* customizeCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget            widget,
             CALENDAR_REC_REF  cal_ref,
             XtPointer         call_data )
{

  /* Code. */

  /* Do we have a user action callback registered? */
  if( cal_ref -> actionCB != NULL )
    (* cal_ref -> actionCB)( XTM_CA_REASON_DESTROY,
                             NULL,
                             cal_ref -> user_data );

  /* Free the user data. */
  if( cal_ref -> cal_handle != NULL )
    xtmBcDestroy( cal_ref -> cal_handle );

  if( cal_ref -> editor_handle != NULL )
    xtmEdDestroy( cal_ref -> editor_handle );

  if( cal_ref -> open_handle != NULL )
    xtmOvDestroy( cal_ref -> open_handle );

  if( cal_ref -> pr_handle != NULL )
    xtmPrDestroy( cal_ref -> pr_handle );

  if( cal_ref -> show_handle != NULL )
    xtmSdDestroy( cal_ref -> show_handle );

  if( cal_ref -> msg_sel_handle != NULL )
    xtmMlDestroy( cal_ref -> msg_sel_handle );


  SysFree( cal_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void
  exitCB( Widget            widget,
          CALENDAR_REC_REF  cal_ref,
          XtPointer         call_data )
{

  /* Variables. */
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = cal_ref -> appl_data_ref -> custom_data;

  /* Ask for confirmation? */
  if( custom_data_ref -> confirm_actions ) {
    XRaiseWindow( XtDisplay( cal_ref -> calendarW ), 
                  XtWindow(  cal_ref -> calendarW ) );

    XtMapWidget( cal_ref -> calendarW );

    (void) xitCreateQuestionDialog( 
             cal_ref -> calendarW, "QuestionDialog", 
             msgGetText( MXDI_QUESTION_MESSAGE_LABEL ),
             msgGetText( MXDI_SCHED_CONFIRM_CLOSE ),
             exitReallyCB, cal_ref,
             NULL,         NULL );
  } else {
    exitReallyCB( widget, cal_ref, NULL );
  }


  return;

} /* exitCB */


/*----------------------------------------------------------------------*/

static void
  exitReallyCB( Widget            widget,
                CALENDAR_REC_REF  cal_ref,
                XtPointer         call_data )
{

  /* Code. */

  /* Do we have a user action callback registered? */
  if( cal_ref -> actionCB != NULL )
    (* cal_ref -> actionCB)( XTM_CA_REASON_EXIT,
                             NULL,
                             cal_ref -> user_data );

  XtPopdown( cal_ref -> calendarW );


  return;

} /* exitReallyCB */


/*----------------------------------------------------------------------*/

static void 
  infoCB( Widget                     widget,
          CALENDAR_REC_REF           cal_ref,
          XmRowColumnCallbackStruct  *call_data )
{

  /* Code. */

  /* About window? */
  if( (int) call_data -> data == 6 ) {
    xtmHlDisplayAboutWindow( cal_ref -> calFormW );

    return;
  }

  /* Use the standard help. */
  xtmHlDisplayHelp( cal_ref -> appl_data_ref -> info_handle,
                    (int) call_data -> data,
                    cal_window_id, "" );

  return;

} /* infoCB */


/*----------------------------------------------------------------------*/

static void 
  infoReminderCB( Widget  widget,
                  void    *user_data,
                  int     help_index )
{

  /* Variables. */
  CALENDAR_REC_REF  cal_ref;


  /* Code. */

  cal_ref = (CALENDAR_REC_REF) user_data;

  xtmHlDisplayHelp( cal_ref -> appl_data_ref -> info_handle,
                    help_index,
                    remind_window_id, "" );

  return;

} /* infoReminderCB */


/*----------------------------------------------------------------------*/

static void
  monthExtendCB( Widget            widget, 
                 CALENDAR_REC_REF  cal_ref,
                 XtPointer         call_data )
{

  /* Variables. */
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = cal_ref -> appl_data_ref -> custom_data;

  if( custom_data_ref -> cal_month_extend )
    custom_data_ref -> cal_month_extend = False;
  else
    custom_data_ref -> cal_month_extend = True;


  updateCalendar( cal_ref );


  return;

} /* monthExtendCB */


/*----------------------------------------------------------------------*/

static void
  openViewCB( Widget            widget, 
              CALENDAR_REC_REF  cal_ref,
              XtPointer         call_data )
{

  /* Code. */

  /* Initialize? */
  if( cal_ref -> open_handle == NULL )
    cal_ref -> open_handle = xtmOvInitialize( cal_ref -> appl_data_ref,
                                              cal_ref -> calendarW,
                                              False,
                                              openViewApplyCB,
                                              (void *) cal_ref );
  if( cal_ref -> open_handle == NULL )
    return;

  /* Display the window. */
  xtmOvOpenView( cal_ref -> open_handle );


  return;

} /* openViewCB */


/*----------------------------------------------------------------------*/

static void
  openViewApplyCB( XTM_OV_REASON    reason,
                   XTM_CD_CAL_INFO  *db_info_ref,
                   void             *user_data )
{

  /* Variables. */
  CALENDAR_REC_REF      cal_ref;


  /* Code. */

  if( reason != XTM_OV_REASON_APPLY && reason != XTM_OV_REASON_OK )
    return;

  cal_ref = (CALENDAR_REC_REF) user_data;

  /* Do we have a user action callback registered? */
  if( cal_ref -> actionCB != NULL )
    (* cal_ref -> actionCB)( XTM_CA_REASON_OPEN_VIEW,
                             (void *) db_info_ref -> short_name,
                             cal_ref -> user_data );

  return;

} /* openViewApplyCB */


/*----------------------------------------------------------------------*/

static void
  plannerCB( Widget            widget,
             CALENDAR_REC_REF  cal_ref,
             XtPointer         call_data )
{

  /* Variables. */
  TIM_TIME_REF          start_date;
  XTM_GL_BASE_DATA_REF  appl_data_ref;
  XTM_PL_HANDLE         planner_handle;


  /* Code. */

  appl_data_ref = cal_ref -> appl_data_ref;

  /* Planner for the current month. */
  start_date = TimMakeTime( TimIndexOfYear(  cal_ref -> main_calendar_date ),
                            TimIndexOfMonth( cal_ref -> main_calendar_date ),
                            1, 0, 0, 0 );

  /* Create planner window */
  planner_handle = xtmPlInitialize( appl_data_ref, 
                                    appl_data_ref -> toplevel,
                                    plannerActionCB,
                                    (void *) cal_ref );

  if( planner_handle == NULL )
    return;


  /* Display the planner window. */
  xtmPlViewPlanner( planner_handle, start_date, False );


  return;

} /* plannerCB */


/*----------------------------------------------------------------------*/

static void
  plannerActionCB( XTM_PL_REASON  reason,
                   void           *user_data )
{

  /* Variables. */
  CALENDAR_REC_REF  cal_ref;


  /* Code. */

  cal_ref = (CALENDAR_REC_REF) user_data;


  return;

} /* plannerActionCB */


/*----------------------------------------------------------------------*/

static void 
  printCB( Widget            widget,
           CALENDAR_REC_REF  cal_ref,
           XtPointer         call_data )
{

  /* Variables. */
  Boolean               ok;
  TIM_TIME_REF          cal_date;
  TIM_TIME_REF          from_date;
  TIM_TIME_REF          to_date;
  XTM_DM_FILTER_REC     filter_rec;
  XTM_GL_BASE_DATA_REF  appl_data_ref;
  XTM_CD_CAL_INFO       db_info;


  /* Code. */

  appl_data_ref = cal_ref -> appl_data_ref;

  /* Fetch the database we are using. */
  ok = xtmCdFetchNamedDb( appl_data_ref -> custom_data -> cal_db_handle, 
                          cal_ref -> cal_name,
                          &db_info, NULL );
  if( ! ok )
    return;

  cal_date = cal_ref -> main_calendar_date;

  from_date = TimMakeTime( TimIndexOfYear(  cal_date ),
                           TimIndexOfMonth( cal_date ),
                           1,
                           0, 0, 0 );
  to_date = from_date;
  TimAddDays( &to_date, TimDaysInMonth( cal_date ) - 1 );


  /* Create the printer window. */
  if( cal_ref -> pr_handle == NULL ) {

    cal_ref -> pr_handle = xtmPrInitialize( appl_data_ref, 
                                            appl_data_ref -> toplevel,
                                            db_info.short_name,
                                            printActionCB,
                                            (void *) cal_ref );

    if( cal_ref -> pr_handle == NULL )
      return;

  }


  /* Use the default filter. */
  filter_rec.flags            = XTM_DM_FLAG_ALL;
  filter_rec.text_string[ 0 ] = '\0';
  filter_rec.tag_string[  0 ] = '\0';


  /* Display the printer window. */
  xtmPrDisplayPrinter( cal_ref -> pr_handle, 
                       from_date, to_date, 
                       0, &filter_rec );


  return;

} /* printCB */


/*----------------------------------------------------------------------*/

static void
  printActionCB( XTM_PR_REASON  reason,
                 void           *user_data )
{

  /* Variables. */
  CALENDAR_REC_REF  cal_ref;


  /* Code. */

  cal_ref = (CALENDAR_REC_REF) user_data;

  if( reason == XTM_PR_REASON_DESTROY )
    cal_ref -> pr_handle = NULL;


  return;

} /* printActionCB */


/*----------------------------------------------------------------------*/

static void
  receiveMessageCB( Widget            widget,
                    CALENDAR_REC_REF  cal_ref,
                    XtPointer         call_data )
{

  /* Variables. */
  Boolean                 ok;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         db_info;


  /* Code. */

  custom_data_ref = cal_ref -> appl_data_ref -> custom_data;

  /* Fetch the database we are using. */
  ok = xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle, 
                          cal_ref -> cal_name,
                          &db_info, NULL );
  if( ! ok )
    return;


  /* Initialize a new message selecto? */
  if( cal_ref -> msg_sel_handle == NULL )
    cal_ref -> msg_sel_handle = xtmMlInitialize( 
                                   cal_ref -> appl_data_ref,
                                   cal_ref -> appl_data_ref -> toplevel,
                                   NULL, NULL );

  xtmMlSelectMessage( cal_ref -> msg_sel_handle );


  return;

} /* receiveMessageCB */


/*----------------------------------------------------------------------*/

static void
  redisplayCB( Widget            widget,
               CALENDAR_REC_REF  cal_ref,
               XtPointer         call_data )
{

  /* Code. */

  /* Update the calendars. */
  xtmUpDoUpdate( (XTM_UP_CALENDAR | XTM_UP_NAV_CALENDAR |
                  XTM_UP_SCHEDULE | XTM_UP_PLANNER),
                 NULL );


  return;

} /* redisplayCB */


/*----------------------------------------------------------------------*/

static void
  rereadCustCB( Widget            widget,
                CALENDAR_REC_REF  cal_ref,
                XtPointer         call_data )
{

  /* Variables. */
  TIM_STATUS_TYPE     time_status;
  XTM_GL_CUSTOM_DATA  *custom_data_ref;


  /* Code. */

  custom_data_ref = cal_ref -> appl_data_ref -> custom_data;

  /* Reread the custom parameters. */
  (void) xtmCbRereadParamFromFile( custom_data_ref );

  /* New date/time formats? */
  time_status = TimInitializeFormat( custom_data_ref -> date_format,
                                     custom_data_ref -> time_format );
  if( time_status != TIM_OK )
    xitErMessage( cal_ref -> calendarW, XIT_ER_ERROR, 
                  module_name, "rereadCustCB",
                  msgGetText( MXDI_ERRMSG_DATE_OR_TIME_FORMAT ) );


  /* Update everything. */
  xtmUpDoUpdate( (XTM_UP_CALENDAR | XTM_UP_NAV_CALENDAR |
                  XTM_UP_SCHEDULE | XTM_UP_PLANNER),
                 NULL );


  /* Empty the caches. */
  xtmEdEmptyCache();
  xtmPrEmptyCache();
  xtmSdEmptyCache();
  xtmSmEmptyCache();


  return;

} /* rereadCustCB */


/*----------------------------------------------------------------------*/

static void
  reminderCB( Widget            widget,
              CALENDAR_REC_REF  cal_ref,
              XtPointer         call_data )
{

  /* Code. */

  /* Display the reminder window. */
  xtmRdDisplayReminder( cal_ref -> reminder_ref );


  return;

} /* reminderCB */


/*----------------------------------------------------------------------*/

static void
  sendMessageCB( Widget            widget,
                 CALENDAR_REC_REF  cal_ref,
                 XtPointer         call_data )
{

  /* Variables. */
  XTM_GL_BASE_DATA_REF  appl_data_ref;


  /* Code. */

  appl_data_ref = cal_ref -> appl_data_ref;


  /* Initialize a new message selector? */
  if( appl_data_ref -> msg_send_handle == NULL )
    appl_data_ref -> msg_send_handle = 
      xtmMsInitialize( appl_data_ref, appl_data_ref -> toplevel );

  xtmMsSendMessage( appl_data_ref -> msg_send_handle, "", 0 );


  return;

} /* sendMessageCB */


/*----------------------------------------------------------------------*/

static void
  summaryCB( Widget            widget,
             CALENDAR_REC_REF  cal_ref,
             XtPointer         call_data )
{

  /* Variables. */
  Boolean               ok;
  TIM_TIME_REF          end_date;
  TIM_TIME_REF          start_date;
  XTM_GL_BASE_DATA_REF  appl_data_ref;
  XTM_CD_CAL_INFO       db_info;


  /* Code. */

  appl_data_ref = cal_ref -> appl_data_ref;

  /* Fetch the database we are using. */
  ok = xtmCdFetchNamedDb( appl_data_ref -> custom_data -> cal_db_handle, 
                          cal_ref -> cal_name,
                          &db_info, NULL );
  if( ! ok )
    return;

  /* Summary for the current month. */
  start_date = TimMakeTime( TimIndexOfYear(  cal_ref -> main_calendar_date ),
                            TimIndexOfMonth( cal_ref -> main_calendar_date ),
                            1, 0, 0, 0 );

  end_date = start_date;

  TimAddDays( &end_date, TimDaysInMonth( cal_ref -> main_calendar_date ) - 1 );

  /* Create summary window? */
  if( cal_ref -> show_handle == NULL ) {
    cal_ref -> show_handle = xtmSdInitialize( appl_data_ref, 
                                              appl_data_ref -> toplevel,
                                              db_info.short_name,
                                              summaryActionCB,
                                              (void *) cal_ref );

    if( cal_ref -> show_handle == NULL )
      return;
  }

  /* Display the summary window. */
  xtmSdDisplaySummary( cal_ref -> show_handle,
                       start_date, end_date,
                       appl_data_ref -> custom_data -> filter_day_view_tags );


  return;

} /* summaryCB */


/*----------------------------------------------------------------------*/

static void
  summaryActionCB( XTM_SD_REASON  reason,
                   void           *user_data )
{

  /* Variables. */
  CALENDAR_REC_REF  cal_ref;


  /* Code. */

  cal_ref = (CALENDAR_REC_REF) user_data;

  if( reason == XTM_SD_REASON_DESTROY )
    cal_ref -> show_handle = NULL;


  return;

} /* summaryActionCB */


/*----------------------------------------------------------------------*/

static void
  updateCB( UINT32  flags,
            void    *user_data,
            void    *update_user_data )
{

  /* Variables. */
  CALENDAR_REC_REF      cal_ref;
  XTM_GL_BASE_DATA_REF  appl_data_ref;


  /* Code. */

  cal_ref       = (CALENDAR_REC_REF) user_data;
  appl_data_ref = cal_ref -> appl_data_ref;


  /* Update calendar? */
  if( flagIsSet( flags, XTM_UP_CALENDAR ) )
    updateCalendar( cal_ref );


  /* Every minute tick. */
  if( flagIsSet( flags, XTM_UP_MINUTE_TICK ) ) {

    Boolean          has_message = False;
    Boolean          ok;
    char             file_pattern[ PATH_MAX + 1 ];
    char             filename[ PATH_MAX + 1 ];
    char             message_dir[ PATH_MAX + 1 ];
    int              status;
    time_t           dir_time;
    TIM_TIME_REF     now;
    XTM_CD_CAL_INFO  db_info;
    struct stat      file_info;

    now = TimLocalTime( TimMakeTimeNow() );

    /* Fetch the database we are using. */
    ok = xtmCdFetchNamedDb( appl_data_ref -> custom_data -> cal_db_handle, 
                            cal_ref -> cal_name,
                            &db_info, NULL );
    if( ! ok )
      return;


    /* Do we have another date? */
    if( flagIsSet( flags, XTM_UP_NEW_DAY ) )
      xtmUpDoUpdate( XTM_UP_CALENDAR, (void *) now );


    /* Do we have a new message? */
    sprintf( message_dir, "%s/Message", db_info.directory );

    status   = stat( message_dir, &file_info );
    dir_time = file_info.st_mtime;

    if( status == 0 && dir_time > cal_ref -> message_stamp ) {

      cal_ref -> message_stamp = dir_time;

      /* Any message files? */
      sprintf( file_pattern, "%s/%s_*_*", message_dir, XTM_DB_MESSAGE_FILE );

      status = DirFindFirst( file_pattern, filename );
      if( status == 0 )
        has_message = True;

      DirEnd();

      if( has_message != cal_ref -> msg_in_mailbox ) {
        xtmBcSignalMessage( cal_ref -> cal_handle, has_message );

        cal_ref -> msg_in_mailbox = has_message;
      }

    } /* if */


    /* Update the calendar time. */
    updateCalendarTime( cal_ref );


  } /* if */


  return;

} /* updateCB */
