/*----------------------------------------------------------------------------
--
--  Module:           xtmArchive
--
--  Project:          XDiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Archive an appointment.
--
--  Filename:         xtmArchive.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-07-29
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmArchive.c, Version: 1.1, Date: 95/02/18 15:51:55";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <X11/Intrinsic.h>

#include <Xm/Xm.h>
#include <Xm/BulletinB.h>
#include <Xm/Form.h>
#include <Xm/List.h>
#include <Xm/Text.h>

#include "System.h"
#include "LstLinked.h"
#include "Message.h"
#include "TimDate.h"

#include "msgXdiary.h"
#include "xtmCalDb.h"
#include "xtmDbTools.h"
#include "xtmDbMisc.h"
#include "xtmFormat.h"
#include "xtmHelp.h"
#include "xtmUpdate.h"
#include "xitError.h"
#include "xitTools.h"
#include "xtmArchive.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Local widgets in the custom window. */
#define archLa            dataLocalW[ 0 ]
#define archLi            dataLocalW[ 1 ]
#define fileLa            dataLocalW[ 2 ]
#define fileTx            dataLocalW[ 3 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Information about the entry to duplicate. */
typedef struct {

  /* Application wide data. */
  XTM_GL_BASE_DATA_REF  appl_data_ref;

  /* Entry being duplicated. */
  XTM_DB_ALL_ENTRY_REF  use_entry_ref;

  /* Archive window. */
  Widget  archW;

} ENTRY_INFO, *ENTRY_INFO_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmArchive";

/* IDs for the help windows. */
static char  *archive_window_id = "Archive";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static Widget
  createArchiveWindow( Widget          parent,
                       ENTRY_INFO_REF  entry_ref );

static void 
  destroyCB( Widget          widget,
             ENTRY_INFO_REF  entry_ref,
             XtPointer       call_data );

static void 
  fileSelectCB( Widget                widget,
                ENTRY_INFO_REF        entry_ref,
                XmListCallbackStruct  *call_data );

static void 
  helpCB( Widget          widget,
          ENTRY_INFO_REF  entry_ref,
          XtPointer       call_data );

static void 
  okCB( Widget          widget,
        ENTRY_INFO_REF  entry_ref,
        XtPointer       call_data );


/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

void 
  xtmArArchiveEntry( XTM_GL_BASE_DATA_REF  appl_data_ref,
                     Widget                parent,
                     char                  *db_name,
                     UINT32                entry_id )
{

  /* Variables. */
  Boolean                 ok;
  ENTRY_INFO_REF          entry_ref = NULL;
  XTM_DB_ALL_ENTRY_REF    use_entry_ref;
  XTM_DB_ENTRY_DATABASES  database;
  XTM_DB_STATUS           db_status;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         db_info;


  /* Code. */

  custom_data_ref = appl_data_ref -> custom_data;

  /* Fetch database information. */
  ok = xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle, db_name,
                          &db_info, NULL );
  if( ! ok )
    return;

  /* We need read access. */
  if( ! flagIsSet( db_info.operations, XTM_DB_FLAG_MODE_READ ) ) {
    xitErMessage( parent, XIT_ER_ERROR, 
                  module_name, "xtmArArchiveEntry",
                  msgGetText( MXDI_ERRMSG_NO_ACCESS_DB ) );

    return;
  }


  /* Create and initialize our private data. */
  entry_ref = SysNew( ENTRY_INFO );
  if( entry_ref == NULL )
    return;

  entry_ref -> appl_data_ref = appl_data_ref;
  entry_ref -> use_entry_ref = SysNew( XTM_DB_ALL_ENTRY_DEF );

  entry_ref -> use_entry_ref -> all_text = NULL;


  /* Open the database and fetch the entry. */
  ok = xtmDmOpenDatabase( appl_data_ref,
                          db_name, XTM_DB_FLAG_MODE_READ,
                          &database );
  if( ! ok )
    raise exception;


  /* Fetch the entry. */
  db_status = xtmDbFetchEntry( &database, entry_id, 
                               entry_ref -> use_entry_ref, 
                               &entry_ref -> use_entry_ref -> all_text );

  xtmDbCloseEntryDb( &database );


  if( db_status != XTM_DB_OK ) {
    xitErMessage( parent, XIT_ER_ERROR,
                  module_name, "xtmArArchiveEntry",
                  msgGetText( MXDI_NO_ENTRY ) );

    raise exception;
  }


  use_entry_ref = entry_ref -> use_entry_ref;

  /* Standing/Sticky entries cannot be archived. */
  if( use_entry_ref -> entry.entry_category == XTM_DB_REP_ENTRY_LIST ||
      use_entry_ref -> entry.entry_category == XTM_DB_STICKY_LIST ) {

    xitErMessage( parent, XIT_ER_ERROR,
                  module_name, "xtmArArchiveEntry",
                  msgGetText( MXDI_ARCH_WRONG_ENTRY ) );

    raise exception;
  }

  /* Already archived? */
  if( flagIsSet( use_entry_ref -> entry.flags, XTM_DB_FLAG_ARCHIVED ) ) {
    xitErMessage( parent, XIT_ER_ERROR, 
                  module_name, "xtmArArchiveEntry",
                  msgGetText( MXDI_ARCH_ALREADY_ARCHIVED ) );

    raise exception;
  }


  /* Create the archive window. */
  entry_ref -> archW = createArchiveWindow( parent, entry_ref );


  /* Set values for the database locations. */
  if( LstLinkElements( custom_data_ref -> archive_files ) > 0 ) {

    int                  index;
    int                  index1;
    char                 buffer[ PATH_MAX + XTM_GL_MAX_ARCH_NAME + 10 ];
    Arg                  args[ 10 ];
    Cardinal             n;
    XmString             list_items[ 100 ];
    Widget               mainW;
    Widget               tempW;
    LST_STATUS           lst_status;
    XTM_AR_ARCHIVE_FILE  archive_file;

    mainW = XtNameToWidget( entry_ref -> archW, "ArchFdFo" );
    index = 0;

    lst_status = LstLinkCurrentFirst( custom_data_ref -> archive_files );

    while( lst_status == LST_OK ) {

      lst_status = LstLinkGetCurrent( custom_data_ref -> archive_files, 
                                      &archive_file );

      sprintf( buffer, "%-15.15s %s", 
               archive_file.short_name, archive_file.file_name );

      list_items[ index ] = XmStringCreate( buffer, CS );
      index++;

      /* Next entry. */
      lst_status = LstLinkCurrentNext( custom_data_ref -> archive_files );

    } /* while */


    /* The list is always sorted. */
    xitSortStringList( list_items, index );

    /* Assign the database locations to the list. */
    tempW = XtNameToWidget( mainW, "ArchLiSW.ArchLi" );

    n = 0;
    XtSetArg( args[ n ], XmNitems,             list_items ); n++;
    XtSetArg( args[ n ], XmNitemCount,         index      ); n++;
    XtSetArg( args[ n ], XmNselectedItemCount, 0          ); n++;
    XtSetValues( tempW, args, n );

    /* Free allocated memory. */
    for( index1 = 0; index1 < index; index1++ )
      XmStringFree( list_items[ index1 ] );

    /* Select the first entry in the list. */
    XmListSelectPos( tempW, 1, True );

  } /* if */


  return;


  /* Exception handler. */
  exception:
    if( entry_ref -> use_entry_ref -> all_text != NULL )
      SysFree( entry_ref -> use_entry_ref -> all_text );

    if( entry_ref -> use_entry_ref != NULL )
      SysFree( entry_ref -> use_entry_ref );

    if( entry_ref != NULL )
      SysFree( entry_ref );

    return;

} /* xtmArArchiveEntry */


/*----------------------------------------------------------------------*/

static Widget
  createArchiveWindow( Widget          parent,
                       ENTRY_INFO_REF  entry_ref )
{

  /* Variables. */
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    archFd;
  Widget    dataLocalW[ 4 ];
  Widget    workFo;

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "", okCB,   NULL },
    { "", NULL,   NULL },
    { "", helpCB, NULL },
  };

  static XIT_TEXT_STRUCT text_buffer[] = {
    { "FileTx", NULL, 1, True },
  };


  /* Code. */

  /* Text items. */
  action_buttons[ 0 ].label = msgGetText( MXDI_OK_BUTTON );
  action_buttons[ 0 ].data  = entry_ref;
  action_buttons[ 1 ].label = msgGetText( MXDI_CANCEL_BUTTON );
  action_buttons[ 1 ].data  = entry_ref;
  action_buttons[ 2 ].label = msgGetText( MXDI_HELP_BUTTON );
  action_buttons[ 2 ].data  = entry_ref;


  /* Create a form dialog with buttons. */
  archFd = xitCreateFormDialog( parent, "ArchFd",
                                1, 0,
                                action_buttons,
                                XtNumber( action_buttons ) );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, msgGetText( MXDI_ARCHIVE_TITLE ) ); n++;
  XtSetValues( XtParent( archFd ), args, n );

  XtAddCallback( archFd, XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) entry_ref );

  /* Fetch container for the contents of the window. */
  workFo = XtNameToWidget( archFd, "ArchFdFo" );


  /* Label telling what this is. */
  archLa = xitCreateLabel( workFo, "ArchLa", 
                           msgGetText( MXDI_ARCHIVE_ENTRY_LABEL ), -1 );


  /* List with archive files to select from. */
  n = 0;
  XtSetArg( args[ n ], XmNlistSizePolicy,         XmCONSTANT      ); n++;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmSTATIC        ); n++;
  XtSetArg( args[ n ], XmNselectionPolicy,        XmSINGLE_SELECT ); n++;
  XtSetArg( args[ n ], XmNlistMarginHeight,       5               ); n++;
  XtSetArg( args[ n ], XmNlistMarginWidth,        5               ); n++;
  archLi = XmCreateScrolledList( workFo, "ArchLi", args, n );

  XtAddCallback( archLi, XmNsingleSelectionCallback,
                 (XtCallbackProc) fileSelectCB, (XtPointer) entry_ref );


  /* Selected file. */
  fileLa = xitCreateLabel( workFo, "FileLa", 
                           msgGetText( MXDI_ARCHIVE_FILE_LABEL ), -1 );
  fileTx = xitCreateTextCols(  workFo, &text_buffer[ 0 ], 40 );

  n = 0;
  XtSetArg( args[ n ], XmNmaxLength, PATH_MAX ); n++;
  XtSetValues( fileTx, args, n );


  /* Put the elements together. */
  xitAttachWidget( archLa,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( XtParent( archLi ),
                   XmATTACH_WIDGET, archLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,   XmATTACH_NONE, NULL );
  xitAttachWidget( fileLa,
                   XmATTACH_WIDGET, XtParent( archLi ), XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,               XmATTACH_NONE, NULL );
  xitAttachWidget( fileTx,
                   XmATTACH_WIDGET, fileLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,   XmATTACH_NONE, NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetValues( archLa,             args, n );
  XtSetValues( XtParent( archLi ), args, n );
  XtSetValues( fileLa,             args, n );
  XtSetValues( fileTx,             args, n );


  /* Manage the widgets. */
  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );

  /* Set the size of the window. */
  xitSetSizeFormDialog( archFd, True );


  /* Make the final attachments. */
  xitAttachWidget( fileTx,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL );
  xitAttachWidget( fileLa,
                   XmATTACH_NONE, NULL, XmATTACH_FORM,   NULL,
                   XmATTACH_NONE, NULL, XmATTACH_WIDGET, fileTx );
  xitAttachWidget( XtParent( archLi ),
                   XmATTACH_WIDGET, archLa, XmATTACH_FORM,   NULL,
                   XmATTACH_FORM,   NULL,   XmATTACH_WIDGET, fileLa );

  XtManageChild( archFd );


  /* Make sure our children don't spoil our size. */
  n = 0;
  XtSetArg( args[ n ], XmNallowShellResize, False ); n++;
  XtSetValues( XtParent( archFd ), args, n );


  return( archFd );

} /* createArchiveWindow */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget          widget,
             ENTRY_INFO_REF  entry_ref,
             XtPointer       call_data )
{

  /* Code. */

  /* Release the user data. */
  if( entry_ref -> use_entry_ref != NULL ) {
    SysFree( entry_ref -> use_entry_ref -> all_text );
    SysFree( entry_ref -> use_entry_ref );
  }

  SysFree( entry_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void 
  fileSelectCB( Widget                widget,
                ENTRY_INFO_REF        entry_ref,
                XmListCallbackStruct  *call_data )
{

  /* Variables. */
  char    short_name[ XTM_GL_MAX_ARCH_NAME + 1 ];
  char    archive_file[ PATH_MAX + 1 ];
  char    *char_ref;
  Widget  mainW;
  Widget  tempW;


  /* Code. */

  mainW = XtNameToWidget( entry_ref -> archW, "ArchFdFo" );


  /* Fetch the selected file. */
  char_ref = xitStringGetString( call_data -> item, CS );

  sscanf( char_ref, "%s %s", short_name, archive_file );
  SysFree( char_ref );


  /* Set the file text field. */
  tempW = XtNameToWidget( mainW, "FileTx" );

  XmTextSetString( tempW, archive_file );


  return;

} /* fileSelectCB */


/*----------------------------------------------------------------------*/

static void 
  helpCB( Widget          widget,
          ENTRY_INFO_REF  entry_ref,
          XtPointer       call_data )
{

  /* Code. */

  xtmHlDisplayHelp( entry_ref -> appl_data_ref -> info_handle,
                    XTM_HL_WINDOW_INDEX,
                    archive_window_id, "" );

  return;

} /* helpCB */


/*----------------------------------------------------------------------*/

static void 
  okCB( Widget          widget,
        ENTRY_INFO_REF  entry_ref,
        XtPointer       call_data )
{

  /* Variables. */
  Boolean                 ok;
  char                    archive_file[ PATH_MAX + 1 ];
  char                    buffer[ 40 ];
  char                    *char_ref;
  Widget                  mainW;
  Widget                  tempW;
  FILE                    *file_ref;
  XTM_DB_ALL_ENTRY_REF    use_entry_ref;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         db_info;


  /* Code. */

  appl_data_ref   = entry_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;
  use_entry_ref   = entry_ref -> use_entry_ref;

  mainW = XtNameToWidget( entry_ref -> archW, "ArchFdFo" );


  /* Fetch database information. */
  ok = xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle, 
                          use_entry_ref -> db_name,
                          &db_info, NULL );
  if( ! ok )
    return;


  /* Fetch the archive file. */
  tempW    = XtNameToWidget( mainW, "FileTx" );
  char_ref = xitStringGetText( tempW );

  strcpy( archive_file, char_ref );
  SysFree( char_ref );


  /* Open the archive file. */
  file_ref = fopen( archive_file, "a" );

  if( file_ref == NULL ) {
    xitErMessage( entry_ref -> archW, XIT_ER_ERROR, 
                  module_name, "okCB",
                  msgGetText( MXDI_ARCH_CANNOT_OPEN_FILE ) );

    return;
  }


  /* Write the record to the archive. */
  fprintf( file_ref, "> " );
  fprintf( file_ref, "Id=%d ", 
           (int) use_entry_ref -> entry.id );
  fprintf( file_ref, "Tp=%d ", 
           (int) use_entry_ref -> entry.entry_type );
  fprintf( file_ref, "Ca=%d ", 
           (int) use_entry_ref -> entry.entry_category );

  buffer[ 0 ] = '\0';
  sscanf( use_entry_ref -> entry.tag, "%s", buffer );
  if( strlen( buffer ) > 0 )
    fprintf( file_ref, "Tg=%s ", buffer );

  if( use_entry_ref -> entry.entry_type == XTM_DB_DAY_ENTRY ) {
    fprintf( file_ref, "Ti=%d ", 
             (int) use_entry_ref -> entry.time_stamp +
             (int) use_entry_ref -> entry.date_stamp );
    fprintf( file_ref, "Du=%d ", 
             (int) use_entry_ref -> entry.duration );
  }

  fprintf( file_ref, "\n" );
  fprintf( file_ref, "# ------------------------------------------------\n" );

  if( use_entry_ref -> entry.entry_type == XTM_DB_DAY_ENTRY ) {

    /* Write the date and time. */
    xtmFoFormatDate( use_entry_ref -> entry.date_stamp,
                     buffer, sizeof( buffer ) );

    fprintf( file_ref, " %s", buffer );

    xtmFoFormatTime( use_entry_ref -> entry.time_stamp,
                     buffer, sizeof( buffer ) );

    fprintf( file_ref, " %s\n\n", buffer );

  } else if( use_entry_ref -> entry.entry_type == XTM_DB_DAY_NOTE ) {
    fprintf( file_ref, " %s:\n\n",
             msgGetText( MXDI_SIMPLE_NOTE_FLAG ) );
  }


  /* Write the text. */
  if( use_entry_ref -> all_text != NULL )
    char_ref = strtok( use_entry_ref -> all_text, "\n" );
  else
    char_ref = strtok( use_entry_ref -> entry.text, "\n" );

  while( char_ref != NULL ) {
    fprintf( file_ref, " %s\n", char_ref );

    char_ref = strtok( NULL, "\n" );
  }

  fprintf( file_ref, "\n" );

  fclose( file_ref );


  /* Mark the entry as archived (only if we have write access). */
  if( flagIsSet( db_info.operations, XTM_DB_FLAG_MODE_WRITE ) ) {

    XTM_DB_ENTRY_DATABASES  database;
    XTM_DB_STATUS           db_status;

    ok = xtmDmOpenDatabase( appl_data_ref,
                            use_entry_ref -> db_name, XTM_DB_FLAG_MODE_WRITE,
                            &database );
    if( ! ok )
      return;

    /* Delete and insert the entry again. */
    db_status = xtmDbDeleteEntry( &database, use_entry_ref -> entry.id );

    if( db_status != XTM_DB_OK )
      xitErMessage( NULL, XIT_ER_FATAL,
                    module_name, "okCB",
                    msgGetText( MXDI_ERRMSG_DELETE_ENTRY ) );

    flagSet( use_entry_ref -> entry.flags, XTM_DB_FLAG_ARCHIVED );

    db_status = xtmDbInsertEntry( &database, 
                                  use_entry_ref, use_entry_ref -> all_text );
    if( db_status != XTM_DB_OK )
      xitErMessage( NULL, XIT_ER_FATAL,
                    module_name, "okCB",
                    msgGetText( MXDI_ERRMSG_INSERT_ENTRY ) );

    xtmDbCloseEntryDb( &database );
  }

 
  /* Update the day view/calendar. */
  xtmUpDoUpdate( XTM_UP_SCHEDULE, NULL );

  /* Remove the selection window. */
  XtDestroyWidget( entry_ref -> archW );


  return;

} /* okCB */

