/*
 * audio.c  audio acesss wrapper
 *
 * Copyright (C) 1997-1998 Masaki Chikama (Wren) <chikama@kasumi.ipl.mech.nagoya-u.ac.jp>
 *               1998-                           <masaki-c@is.aist-nara.ac.jp>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
*/

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include "audio.h"
#include "audioIO.h"

static char*   dev_dsp      = AUDIODEV;     /* DSP device name */

void pcm_setDeviceName(char *name) {
	dev_dsp = name;
}

DSPFILE *OpenDSP(WAVFILE *wfile) {
	int e;
	
	DSPFILE *dfile = (DSPFILE *) malloc(sizeof (DSPFILE));
	if ( dfile == NULL ) {
		fprintf(stderr, "audio(): Opening DSP device\n");
		return NULL;
	}
	memset(dfile, 0, sizeof *dfile);
	dfile->dspbuf = NULL;
	
	if (audioOpen(dfile, wfile, dev_dsp) < 0) {
		goto eexit;
	}
	return dfile;
 eexit:
	e = errno;
	if (dfile->fd >= 0)        audioClose();
	if (dfile->dspbuf != NULL) free(dfile->dspbuf);
	free(dfile);
	errno = e;
	return NULL;
}

int CloseDSP(DSPFILE *dfile) {
	int e;
	
	if (dfile == NULL) {
		fprintf(stderr,"CloseDSP(): DSPFILE is not open\n");
		return -1;
	}

	e = audioClose();
	if ( dfile->dspbuf != NULL ) {
		free(dfile->dspbuf);
	}
	
	free(dfile);
	return e;
}

int PlayDSP(DSPFILE *dfile, WAVFILE *wfile, DSPPROC work_proc) {
	UInt32 byte_count;
	int bytes;
	int n;
	int byte_modulo;
	char *src;

	wfile->wavinfo.DataBytes = wfile->wavinfo.DataBytes_o;
	
	switch(wfile->wavinfo.DataBits) {
	case 8:
		byte_count = 1;
		break;
	case 16:
		byte_count = 2;
		break;
	default :
		fprintf(stderr,"PlayDSP(): Cannot process %u bit samples\n",
			(unsigned)wfile->wavinfo.DataBits);
		return -1;
	}

	/*
	 * Allow for Mono/Stereo difference:
	 */
	if ( wfile->wavinfo.Channels == Stereo )
		byte_count *= 2;		/* Twice as many bytes for stereo */
	else if ( wfile->wavinfo.Channels != Mono ) {
		fprintf(stderr, "PlayDSP(): DSPFILE control block is corrupted (chan_mode)\n");
		return -1;
	}

	byte_modulo = byte_count;				/* This many bytes per sample */
	byte_count  = wfile->wavinfo.Samples * byte_modulo;	/* Total bytes to process */

	audioFlush(dfile);
	src = wfile->data;
	for ( ; byte_count > 0 && wfile->wavinfo.DataBytes > 0; byte_count -= (UInt32) n ) {

		bytes = (int) ( byte_count > dfile->dspblksiz ? dfile->dspblksiz : byte_count );

		if ( bytes > wfile->wavinfo.DataBytes )	/* Size bigger than data chunk? */
			bytes = wfile->wavinfo.DataBytes;	/* Data chunk only has this much left */

		memcpy(dfile->dspbuf, src , bytes);
		n = bytes; src += bytes;
		
		if ( audioWrite(dfile, n) < 0) {
			fprintf(stderr,"Writing samples to audio device\n");
			return -1;
		}

		wfile->wavinfo.DataBytes -= (UInt32) bytes;	/* We have fewer bytes left to read */
		/*
		 * The work procedure function is called when operating
		 * in server mode to check for more server messages:
		 */
		if ( work_proc != NULL && work_proc(dfile) )	/* Did work_proc() return TRUE? */
			break;					/* Yes, quit playing */
	}
			
	audioFlush(dfile);

	return 0;	/* All samples played successfully */

}

int CheckDSP(DSPINFO *info) {
	return audioCheck(info, dev_dsp);
}

