/*
  Copyright (C) 1997-2005  Dimitrios P. Bouras

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   For author contact information, look in the README file.
*/

#include <forms.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <stdarg.h>
#include <time.h>
#include <string.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <fcntl.h>
#include <sys/time.h>
#include <unistd.h>
#include <signal.h>
#include <sys/wait.h>
#include <dirent.h>
#include <math.h>
#include <locale.h>

#ifdef SUNOS41x
#include <memory.h>
extern int sys_nerr;
extern char *sys_errlist[];
extern int errno;
extern int gettimeofday(), fputs(), usleep(), mknod(), pclose(), select();
extern int on_exit(), strftime(), fprintf(), fclose(), sscanf(), fscanf();
extern int rename(), gethostname(), putenv(), seteuid(), vfprintf();
extern void bcopy(), bzero();
extern char *vsprintf();
extern time_t time();
extern pid_t getpid();
#endif

#ifdef SUNOS5x
#include <netdb.h>
extern int usleep(), gethostname();
#endif

#ifdef RUNASEUID
int xisp_euidaccess();
#endif

#if defined(SUNOS5x) || defined(SVR4)
#include <sys/mkdev.h>
#endif

#include "common.h"
#include "xisp.h"
#include "pcode.h"
#include "network.h"
#include "rcio.h"
#include "logs.h"
#include "xisp.xpm"
#include "excl.xpm"
#include "version.h"
#include "help.c"
#include "peHelp.c"

#ifdef XPMANIMATE
#include "online1.xpm"
#include "online2.xpm"
#include "online3.xpm"
#include "online4.xpm"
#include "online5.xpm"
#include "online6.xpm"
#include "online7.xpm"
#include "online8.xpm"
#include "online9.xpm"
#include "online10.xpm"
#include "online11.xpm"
#include "online12.xpm"
#endif

/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                         Global program storage                          |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

/* All application objects and any accompanying
   communication structures are declared global */

FD_instanceCheck *fd_instanceCheck;	/* Instance check alert message form */
FD_topFrame      *fd_topFrame;		/* The main application form */
FD_accountInfo   *fd_accountInfo;	/* Account information */
typedef struct {					/* and its inter-form communication */
	unsigned char cur;				/* (or IFC) data structure */
	unsigned char dflt;
	int new, modified;
	char descr[MAXLEN_DESCR+1];
	unsigned long udata;
} accountInfo_t;
accountInfo_t    IFC_accountInfo;
FD_dialInfo      *fd_dialInfo;		/* Dialing options */
FD_CBInfo        *fd_CBInfo;		/* Call-back script */
FD_aboutInfo     *fd_aboutInfo;		/* Application "About" */
FD_exitDialog    *fd_exitDialog;	/* Exit dialog */
FD_renameISP     *fd_renameISP;		/* Modify ISP name */
FD_helpInfo      *fd_helpInfo;		/* General GUI information */
FD_commInfo      *fd_commInfo;		/* Communication options */
typedef struct {					/* and its IFC structure */
	unsigned int speed;
	unsigned char custom;
	unsigned long operOpts;
} commInfo_t;
commInfo_t       IFC_commInfo;
FD_tcpipInfo     *fd_tcpipInfo;		/* TCP/IP options */
FD_optsTab       *fd_optsTab;		/* Tab folder for all options */
FD_logInfo       *fd_logInfo;		/* Logging and PTT selection options */
FD_statInfo      *fd_statInfo;		/* Connection-time/cost statistics */
FD_pttEditor     *fd_pttEditor;		/* The PTT editor */
ptt_t            IFC_pttEditor;		/* and its IFC structure */
FD_renamePTT     *fd_renamePTT;		/* Modify PTT name */
FD_renameZone    *fd_renameZone;	/* Modify zone name */
FD_actionVerify  *fd_actionVerify;	/* Action verification dialog */
FD_envInfo       *fd_envInfo;		/* Edit program/system paths */
glob_t           IFC_envInfo;		/* and its IFC structure */
FD_alertMessage  *fd_alertMessage;	/* alert messages */

/* Global selection options go in here */

glob_t global = GLOB_DEFAULT;

/* ISP information from .xisprc goes in here; default values are held
   in [0] and all further ISP records are saved from [1] and upwards */

xisprc_t *xispOptions;		/* ISP information table */
unsigned currentRC = 0;		/* sequence number of, and */
xisprc_t *p_xisprc;			/* pointer to current ISP information block */
xisprc_t ISPCopyBuf;		/* buffer for ISP copy-and-paste operations */

/* File names independent of user specified paths */

char *rcfname;		/* file holding program parameters */
#ifndef ISPENV_USEVARS
char *envfname;		/* temp file for passing the dialing environment */
#endif
char *papfname;		/* temp file with PAP username and password for +ua */
char *logdirname;	/* path for xisp logging files */
char *logfname;		/* connection logging file complete name */
char *costfname;	/* total units/time logging file complete name */
char *bkupfname;	/* backup name work-space for logfname and costfname */
char *uupfname;		/* user's "ip-up" file name */
char *udownfname;	/* user's "ip-down" file name */
char *upidfname;	/* file containing PID number of running xisp */

/* File names dependent on user specified paths and related variables */

char *Pppd = NULL;		/* pppd daemon */
#ifdef RUNDIR
char *PIDFname = NULL;	/* complete path to pppd PID files */
char *runfnp;			/* pointer to first filename char in PIDFname */
#endif
char *lockFname = NULL;	/* complete path to modem device lock files */
char *lockfnp;			/* pointer to first filename char in lockFname */
char *Chat = NULL;		/* chat utility */
char *Dialer = NULL;	/* xispdial dedicated dialer utility */
char *Terminal = NULL;	/* xispterm manual login popup */
char *Pipe = NULL;		/* named pipe node for reading dialer output */
char *IPParam = NULL;	/* ipparam string for pppd and .xisp-up/-down calls */

/* Other global variables */

int dialerFD;			/* file descriptor used for reading dialer output */
int dialerON = 0;		/* indicates whether if a dialer process is active */
int ipUDON = 0;			/* if ip-up/-down active = browser read iter. left */
int xispUDON = 0;		/* same function as ipUDON but for .xisp-up/-down */
char ipWhich[8] = {0};	/* the active script name; "ip-up" or "ip-down" */
int pppdPID = 0;		/* saved process id of initiated pppd process */
char *devName = NULL;	/* ptr to modem device name of spawned pppd process */
char *devPath = NULL;	/* full path to device node (includes devName) */

#ifdef PPPDLCKDIR
char devLock			/* pppd's lock file name */
	[2*MAXLEN_DEVICE] =
	{0};
#endif

int ipUDPID = 0;	/* saved process id of the ip-up or ip-down process */
int xispUDPID = 0;	/* saved process id of the .xisp-up/-down process */
int pppdPPID = 0;	/* the parent PID of the above, after pppd forks */
int pppdStat = 0;	/* set to 1 when pppd process status has been collected */
int pppdRet = 0;	/* return status of pppd when it forks in the background */
int dialerPID = 0;	/* process id returned by dialer via named pipe */
int	connected = 0;	/* flag indicating connection status */
int firstConn=1;	/* for distinguishing 1st from 2nd (call-back) CONNECT */
int linkTOCounter;	/* link poll timeout counter */
time_t connTime;	/* ammount of time connected */
char speedStr[32];	/* saved speed string printed out by modem */
char connSpeed[8];	/* formatted string of connection speed */
int linkOK = 0;		/* flag indicating link status */
char connIP[16];	/* string IP address of PPP interface when up */
int userDiscon = 0;	/* indicates user initiated disconnection request */

#ifdef XPMANIMATE
int frame = 0;		/* frame number of xpm animation */
unsigned netpkts=0;	/* sum of net packets */
int transfer = 0;	/* indicates data is coming over the modem line */
int minimized = 0;	/* flag indicating master form mapstate */
Pixmap offline;		/* animation pixmaps */
Pixmap online[12];
Pixmap amask;		/* and transparency mask */
#endif

char pppIF[8]={0};	/* PPP interface name (when RUNDIR available) */
int progState =		/* globally available program state variable */
	DISCONNECTED;
Window topWin;		/* saved window ID for program top level form window */
int wasTab = 0;		/* used in script-line hacked multi-line input fields */
#ifndef SUNOS41x
int exitStatus = 0;	/* saved argument to exit() calls for use by exitCleanup */
#endif

/* Variables related to cost calculation and logging */

ptt_t *ptt;					/* table of PTT information records */
unsigned currentPTT = 0;	/* sequence number of, and */
ptt_t *p_ptt;				/* pointer to current PTT information record */
char *pttfname;				/* PTT data base file name */

time_t upTime;				/* local time connection was established */
time_t CBTime;				/* time call-back connection was established */
time_t downTime;			/* local time connection was torn down */
unsigned long onLineSecs;	/* seconds online, for current connection */
float onLineCost;			/* cost for current dialup session */
unsigned long onLineUnits;	/* units for current dialup session */
float totalCost;			/* total cost recorded for logging period */
unsigned long totalTime;	/* total on-line time (secs) for logging period */
unsigned long totalUnits;	/* total number of units for logging period */

/* Timers for periodic events */

FL_OBJECT *btimer;		/* triggers browser updates with dialer output */
FL_OBJECT *ctimer;		/* measures connection time */
FL_OBJECT *ltimer;		/* used for polling link status */
#ifdef XPMANIMATE
FL_OBJECT *xpmtimer;	/* times XPM icon animation */
#endif

/* Extra cursors */

static unsigned char timer_cursor[] = {
  0x80, 0x03, 0x00, 0x00, 0x01, 0x00, 0xc0, 0x47, 0x00, 0x30, 0xd8, 0x00,
  0xc8, 0x27, 0x00, 0x34, 0x58, 0x00, 0x1a, 0xb1, 0x00, 0x2a, 0xa9, 0x00,
  0x05, 0x41, 0x01, 0x05, 0x41, 0x01, 0x1d, 0x71, 0x01, 0x85, 0x40, 0x01,
  0x45, 0x40, 0x01, 0x2a, 0xa8, 0x00, 0x0a, 0xb1, 0x00, 0x34, 0x59, 0x00,
  0xc8, 0x27, 0x00, 0x30, 0x18, 0x00, 0xc0, 0x07, 0x00
};
static unsigned char timer_mask[] = {
  0x80, 0x03, 0x00, 0x00, 0x01, 0x00, 0xc0, 0x47, 0x00, 0xf0, 0xdf, 0x00,
  0xf8, 0x3f, 0x00, 0xfc, 0x7f, 0x00, 0xfe, 0xff, 0x00, 0xfe, 0xff, 0x00,
  0xff, 0xff, 0x01, 0xff, 0xff, 0x01, 0xff, 0xff, 0x01, 0xff, 0xff, 0x01,
  0xff, 0xff, 0x01, 0xfe, 0xff, 0x00, 0xfe, 0xff, 0x00, 0xfc, 0x7f, 0x00,
  0xf8, 0x3f, 0x00, 0xf0, 0x1f, 0x00, 0xc0, 0x07, 0x00
};
int tcid;

/* Pppd (version >= 2.3.9) status messages (return code = index value) */

char *pppdMsg[20] = {
"either pppd has detached, or otherwise\n"			/* retuned 0 */
"the connection was successfully\n"
"established and terminated at the peer's\n"
"request.",

"an immediately fatal error of some kind\n"			/* retuned 1 */
"occurred, such as an essential system\n"
"call failing, or running out of virtual memory.\n"
"\n"
"If you changed the modem device, make\n"
"sure that you create a corresponding options\n"
"file in pppd's peers directory (most probably\n"
"/etc/ppp/peers). Read the comments found in\n"
"file xisp.peers.device in the xisp distribution\n"
"for details. Alternatively check modem power\n"
"and/or the interconnecting cables.",

"an error was detected in processing the\n"			/* retuned 2 */
"options given, such as two mutually\n"
"exclusive options being used.",

"the pppd executable is not setuid-root and\n"		/* retuned 3 */
"the invoking user is not root.",

"the running kernel does not support PPP,\n"		/* retuned 4 */
"for example, the PPP kernel driver is not\n"
"included or cannot be loaded.",

"pppd terminated because it received a\n"			/* retuned 5 */
"SIGINT, SIGTERM or SIGHUP signal.",

"the serial port could not be locked.",				/* retuned 6 */

"the serial port could not be opened.",				/* retuned 7 */

"the connect script (or program) failed\n"			/* retuned 8 */
"(i.e. it returned a non-zero exit status).",

"the command specified as the argument\n"			/* retuned 9 */
"to the pty option could not be run.",

"the PPP negotiation failed, that is, it didn't\n"	/* retuned 10 */
"reach the point where at least one network\n"
"protocol (e.g. IP) was running.",

"the peer system failed (or refused) to\n"			/* retuned 11 */
"authenticate itself.",

"the link was established successfully and\n"		/* retuned 12 */
"terminated because it was idle. ",

"the link was established successfully and\n"		/* retuned 13 */
"terminated because the connect time limit\n"
"was reached.",

"callback was negotiated and an incoming\n"			/* retuned 14 */
"call should arrive shortly.",

"the link was terminated because the peer\n"		/* retuned 15 */
"is not responding to echo requests.",

"the link was terminated by the modem\n"			/* retuned 16 */
"hanging up.",

"the PPP negotiation failed because serial\n"		/* retuned 17 */
"loopback was detected.",

"the init script failed (returned a non-zero\n"		/* retuned 18 */
"exit status).",

"pppd failed to authenticate itself to\n"			/* retuned 19 */
"the peer."
};

/*+-------------------------------------------------------------------------+
  |                                                                         |
  |           Command line options & Resources - Data structures            |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

#define NUMOPTS 9
FL_CMD_OPT options[NUMOPTS] = 
{
	{"-bgcol", ".bgcol", XrmoptionSepArg, BGCOL_STRING},
	{"-iconic", ".iconic", XrmoptionNoArg, "True"},
	{"-geometry", ".geometry", XrmoptionSepArg, ""},
	{"-isp", ".isp", XrmoptionSepArg, ""},
	{"-autodial", ".autodial", XrmoptionNoArg, "True"},
	{"-debug", ".debug", XrmoptionNoArg, "True"},
	{"-expert", ".expert", XrmoptionNoArg, "True"},
	{"-nohints", ".nohints", XrmoptionNoArg, "True"},
	{"-pidfp", ".pidfp", XrmoptionSepArg, PIDFP},
};

char bgcols[16+1] = {0};				/* the background color string */
color_t bgcol = BGCOL_DEFAULT,			/* default background for all forms */
		tabcol;							/* color for tab folder backdrop */
int iconic = 0,							/* window placement flag */
	winPosX,							/* hint to WM for position */
	winPosY,
	winWidth,							/* request for different size */
	winHeight,
	clISP = 0,							/* command-line specified ISP */
	autodial = 0,						/* request for auto dial on startup */
	debug = 0,							/* request pppd maximum debugging */
	expert = 0,							/* switch for confirmation dialogues */
	nohints = 0;						/* switch for tips and hints */
char geoms[32+1] = {0};					/* the geometry string */
int placementMethod = FL_PLACE_FREE;	/* top form placement method */
char pidfp[16+1] = {0};					/* pppd's PID file prefix */

FL_resource resources[NUMOPTS] =
{
	{"bgcol", "BGColor", FL_STRING, bgcols, BGCOL_STRING, 16},
	{"iconic", "Iconic", FL_BOOL, &iconic, "False"},
	{"geometry", "Geometry", FL_STRING, geoms, "", 32},
	{"isp", "Isp", FL_INT, &clISP, "0"},
	{"autodial", "Autodial", FL_BOOL, &autodial, "False"},
	{"debug", "Debug", FL_BOOL, &debug, "False"},
	{"expert", "Expert", FL_BOOL, &expert, "False"},
	{"nohints", "Hints", FL_BOOL, &nohints, "False"},
	{"pidfp", "PIDFP", FL_STRING, pidfp, PIDFP, 16},
};

/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                            Utility routines                             |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

/* Safe strerror (for arch's that don't have one). Note the implicit
   total length of MSGLEN_ERR bytes for the resulting string. */

#define MSGLEN_ERR 128

char *Strerror(int Errno)
{
#ifdef HAVE_STRERROR
	return strerror(Errno);
#else
	static char emsg[MSGLEN_ERR+1];

	memset(emsg, 0, MSGLEN_ERR+1);
	if (Errno < sys_nerr)
		strncpy(emsg, sys_errlist[Errno], MSGLEN_ERR);
	else
		sprintf(emsg, "error #%d", Errno);
	return emsg;
#endif
}


/* Print message together with system error message and exit. */

void doErr(char *msg)
{
	fprintf(stderr, "xISP: %s: %s\n", msg, Strerror(errno));
	exit(1);
}


/* Returns pointer to string with current time in 12-hour format */

char *strtime(void)
{
	struct timeval tv;
	time_t time;
	static char tout[16];

	gettimeofday(&tv, NULL);
	time = tv.tv_sec;
	strftime(tout, 16, "%I:%M:%S %p", localtime(&time));
	return tout;
}


/* Prints out-of-memory diagnostic and aborts program */

void outofMem(void)
{
	fprintf(stderr, "xISP: out of memory!\n");
	exit(1);
}


/* Waits for desired number of miliseconds using usleep(3). Thus, we avoid
   using sleep(3), which causes problems when mixed with calls to alarm(3). */

void mSleep(unsigned int msec)
{
	unsigned long usec;

	usec = msec * 1000;
	usleep(usec);
}


/* SunOS-4.1.x-safe realloc(). If the pointer
   is NULL, it uses malloc instead */

#ifdef SUNOS41x
void *myRealloc(void *ptr, unsigned size)
{
	if (ptr) return (void *)realloc((char *)ptr, size);
	else return (void *)malloc(size);
}

#define realloc myRealloc
#endif


/* Assembles the string variables holding file names dependent
   on user-specified paths. May be called any number of times */

void initUDFnames(glob_t *p)
{
	int size;
#ifdef RUNASEUID
	struct passwd *user = getpwuid(geteuid());
#else
	struct passwd *user = getpwuid(getuid());
#endif
	int namelen = strlen(user->pw_name);

	size = strlen(p->pppdPath)+strlen(PPPD)+1;
	if ((Pppd = realloc(Pppd, size)) == NULL)
		outofMem();
	strcpy(Pppd, p->pppdPath); strcat(Pppd, PPPD);

#ifdef RUNDIR
	size = strlen(p->runPath)+12+1;
	if ((PIDFname = realloc(PIDFname, size)) == NULL)
		outofMem();
	strcpy(PIDFname, p->runPath); strcat(PIDFname, "/");
	runfnp = &PIDFname[strlen(PIDFname)];
#endif

	size = strlen(p->lockPath)+6+2*MAXLEN_DEVICE+1;	/* more than enough */
	if ((lockFname = realloc(lockFname, size)) == NULL)
		outofMem();
	strcpy(lockFname, p->lockPath);
#if defined(SUNOS5x) || defined(SVR4)
	strcat(lockFname, "/LK.");
#else
	strcat(lockFname, "/LCK..");
#endif
	lockfnp = &lockFname[strlen(lockFname)];

	size = strlen(p->chatPath)+strlen(CHAT)+1;
	if ((Chat = realloc(Chat, size)) == NULL)
		outofMem();
	strcpy(Chat, p->chatPath); strcat(Chat, CHAT);

	size = strlen(p->utilsPath)+strlen(PPPD_CONNECT)+1;
	if ((Dialer = realloc(Dialer, size)) == NULL)
		outofMem();
	strcpy(Dialer, p->utilsPath); strcat(Dialer, PPPD_CONNECT);

	size = strlen(p->utilsPath)+strlen(TERMINAL)+1;
	if ((Terminal = realloc(Terminal, size)) == NULL)
		outofMem();
	strcpy(Terminal, p->utilsPath); strcat(Terminal, TERMINAL);

	size = strlen(p->pipePath)+1+strlen(PIPEFNAME)+1+namelen+1;
	if ((Pipe = realloc(Pipe, size)) == NULL)
		outofMem();
	strcpy(Pipe, p->pipePath); strcat(Pipe, "/"); strcat(Pipe, PIPEFNAME);
	strcat(Pipe, "."); strcat(Pipe, user->pw_name);

	size = strlen(Pipe)+1+MAXLEN_DESCR+1+2*MAXLEN_IP+2+MAXLEN_DNNAME+1;
	if ((IPParam = realloc(IPParam, size)) == NULL)
		outofMem();
}


/* Assembles the string variables holding file
   names independent of user-specified paths. */

void initUIFnames(void)
{
#ifdef RUNASEUID
	struct passwd *user = getpwuid(geteuid());
#else
	struct passwd *user = getpwuid(getuid());
#endif
	unsigned int hdirlen = strlen(user->pw_dir), commonlen, per_host = 0;
	char hostname[MAXHOSTNAMELEN+1];

	if (gethostname(hostname, MAXHOSTNAMELEN) == 0) {
		rcfname = (char *)malloc(hdirlen+1+strlen(RCFNAME)+1+MAXHOSTNAMELEN+1);
		if (rcfname != NULL) {
			strcpy(rcfname, user->pw_dir); strcat(rcfname, "/");
			strcat(rcfname, RCFNAME); strcat(rcfname, ".");
			strcat(rcfname, hostname);
		}
		else
			outofMem();
#ifdef RUNASEUID
		if (xisp_euidaccess(rcfname, R_OK) == 0)
#else
		if (access(rcfname, R_OK) == 0)
#endif
			per_host = 1;
		else
			free(rcfname);
	}
	if (! per_host) {
		rcfname = (char *)malloc(hdirlen+1+strlen(RCFNAME)+1);
		if (rcfname != NULL) {
			strcpy(rcfname, user->pw_dir); strcat(rcfname, "/");
			strcat(rcfname, RCFNAME);
		}
		else
			outofMem();
	}
	upidfname = (char *)malloc(hdirlen+1+strlen(XISPPIDFNAME)+1);
#ifndef ISPENV_USEVARS
	envfname = (char *)malloc(hdirlen+1+strlen(ENVFNAME)+1);
#endif
	papfname = (char *)malloc(hdirlen+1+strlen(PAPFNAME)+1);
	logdirname = (char *)malloc(hdirlen+1+strlen(LOGDIRNAME)+1);
	logfname = (char *)malloc(hdirlen+1+strlen(LOGDIRNAME)+1+
							  strlen(LOGFNROOT)+5+1);
	costfname = (char *)malloc(hdirlen+1+strlen(LOGDIRNAME)+1+
							   strlen(COSTFNROOT)+5+1);
	commonlen = strlen(COSTFNROOT);
	if (strlen(LOGFNROOT) > commonlen)
		commonlen = strlen(LOGFNROOT);
	bkupfname = (char *)malloc(hdirlen+1+strlen(LOGDIRNAME)+1+
							   commonlen+5+4+1);
	uupfname = (char *)malloc(hdirlen+1+strlen(XISPUPNAME)+1);
	udownfname = (char *)malloc(hdirlen+1+strlen(XISPDOWNNAME)+1);
	pttfname = (char *)malloc(hdirlen+1+strlen(LOGDIRNAME)+1+
							  strlen(PTTFNAME)+1);
	if (upidfname != NULL &&
#ifndef ISPENV_USEVARS
		envfname != NULL &&
#endif
		papfname != NULL &&
		logdirname != NULL && logfname != NULL && costfname != NULL &&
        bkupfname != NULL && uupfname != NULL && udownfname != NULL &&
		pttfname != NULL)
	{
		strcpy(upidfname, user->pw_dir); strcat(upidfname, "/");
		strcat(upidfname, XISPPIDFNAME);
#ifndef ISPENV_USEVARS
		strcpy(envfname, user->pw_dir); strcat(envfname, "/");
		strcat(envfname, ENVFNAME);
#endif
		strcpy(papfname, user->pw_dir); strcat(papfname, "/");
		strcat(papfname, PAPFNAME);
		strcpy(logdirname, user->pw_dir); strcat(logdirname, "/");
		strcat(logdirname, LOGDIRNAME);
		strcpy(logfname, logdirname); strcat(logfname, "/");
		strcat(logfname, LOGFNROOT);
		strcpy(costfname, logdirname); strcat(costfname, "/");
		strcat(costfname, COSTFNROOT);
		strcpy(uupfname, user->pw_dir); strcat(uupfname, "/");
		strcat(uupfname, XISPUPNAME);
		strcpy(udownfname, user->pw_dir); strcat(udownfname, "/");
		strcat(udownfname, XISPDOWNNAME);
		strcpy(pttfname, logdirname); strcat(pttfname, "/");
		strcat(pttfname, PTTFNAME);
	}
	else
		outofMem();
}


/* Create and/or open a named pipe for reading only */

int namedPipe(char *fname)
{
	struct stat st;
	int create, fd;
#ifdef RUNASEUID
	struct passwd *user = getpwuid(geteuid());
#else
	struct passwd *user = getpwuid(getuid());
#endif

#ifdef RUNASEUID
	if (xisp_euidaccess(fname, F_OK) == -1)		/* check to see if it exists */
#else
	if (access(fname, F_OK) == -1)
#endif
		create = 1;								/* nope, creation required */
	else {
		stat(fname, &st);						/* yes, get the node status */
		if (!S_ISFIFO(st.st_mode)) {			/* is it a FIFO? */
			unlink(fname);						/* nope, delete it */
			create = 1;							/* indicate creation required */
		}
		else
			create = 0;							/* it's a FIFO, all OK */
	}
	if (create) {								/* was creation requested? */
#ifdef RUNASEUID
		if (xisp_euidaccess(global.pipePath,	/* yes, check user */
							R_OK|W_OK)			/* specified directory */
			== -1) {
#else
		if (access(global.pipePath, R_OK|W_OK)
			== -1) {
#endif
			fprintf(stderr,
					"xISP: can't create named-pipe (FIFO) in directory: %s\n"
					"xISP: set read/write permission in %s for user: %s.\n",
					global.pipePath, global.pipePath, user->pw_name);
			exit(1);
		}
#if (defined(BSD) && BSD >= 199306)
		if (mkfifo(Pipe, 0600))					/* directory is OK, so */
			doErr("namedPipe: mkfifo");			/* create the FIFO node */
		if (chown(fname, -1, getgid()))			/* make pipe group id same */
			doErr("namedPipe: chown");			/* as user real group id */
#else
		if (mknod(Pipe, S_IFIFO+0600, 0))
			doErr("namedPipe: mknod");
#endif
	}
#ifdef SUNOS5x
	fd = open(fname, O_RDONLY|O_NONBLOCK);		/* and open it for reading */
#elif (defined(BSD) && BSD >= 199306)
	fd = open(fname, O_RDONLY|O_NONBLOCK);
#else
	fd = open(fname, O_RDONLY|O_NDELAY);
#endif
	if (fd < 0)									/* bail out on error */
		doErr("namedPipe: open");
	return fd;									/* return the descriptor */
}


/* Search a string for multiple characters returning first occurrence */

static char *strfstr(char *haystack, char *needles)
{
	char cn, *hp;

	while ((cn=*needles)) {						/* search for all needles */
		hp = strchr(haystack, cn);				/* found in haystack? */
		if (hp != NULL)							/* yes, return pointer to */
			return(hp);							/* location of matched char */
		++needles;								/* nope, get next needle */
	}
	return(NULL);								/* nothing found */
}


/* Write printf style in the browser object. Note the implicit
   total length of MSGLEN_BROWSER bytes for the resulting string.
   A routine to flush the buffer used by bprintf is also provided. */

#define MSGLEN_BROWSER (MAXBUF_CHILD*2)

char bmsg[MSGLEN_BROWSER+1] = {0};	/* the string buffer used by bprintf() */
char btmp[MAXBUF_CHILD+1] = {0};	/* temporary buffer for new strings */
char *where = bmsg;					/* incomplete line continuation pointer */
int needNL = 0;						/* and the continuation indicator */

void bDoBuf(void)
{
	char *nl;

	while ((nl= strfstr(bmsg,"\r\n"))!=NULL) {	/* string contains CR or LF? */
		*nl = 0;								/* yes, mark it as last char */
		if (needNL) {							/* redisplay string if the */
			fl_replace_browser_line(			/* last line input lacked */
				fd_topFrame->lstBrowser,		/* an accepted new line char */
				fl_get_browser_maxline(			/* so replace instead */
					fd_topFrame->lstBrowser),	/* of adding the line */
				bmsg);
			needNL = 0;							/* done with line with no NL */
		}
		else if (strcspn(bmsg, "\r\n"))			/* unless NL right after NL */
			fl_addto_browser(fd_topFrame->		/* display string normally */
							 lstBrowser, bmsg);	/* on the browser object */
		strcpy(bmsg, nl+1);						/* move rest to beginning */
	}
}

int bprintf(char *fmt, ...)
{
	int bw, pending = 0;
	va_list ap;
	static int tot = 0;

	va_start(ap, fmt);							/* start variable arg list */
#ifdef BROKEN_SPRINTF
	vsprintf(btmp, fmt, ap);					/* pass the rest to vsprintf */
	bw = strlen(btmp);
#else
	bw = vsprintf(btmp, fmt, ap);
#endif
	va_end(ap);									/* end variable arg list */
	if ((tot+bw) < (MSGLEN_BROWSER-1))			/* do we have space for new? */
		strcat(where, btmp);					/* yup, tack it on the end */
	else {										/* nope, so */
		strcat(where, "\n");					/* end the string here */
		pending = 1;							/* and indicate new pending */
	}
	bDoBuf();									/* process the message buf */
	if (pending) {								/* pending new string? */
		strcpy(bmsg, btmp);						/* yup, copy it in the buffer */
		bDoBuf();								/* process the buffer again */
	}
	tot = strlen(bmsg);							/* total chars so far */
	where = bmsg + tot;							/* pick up from where we left */
	if (tot) {									/* any trailing characters? */
		if (needNL)
			fl_replace_browser_line(			/* last line input lacked */
				fd_topFrame->lstBrowser,		/* an accepted new line char */
				fl_get_browser_maxline(			/* so replace instead */
					fd_topFrame->lstBrowser),	/* of adding the line */
				bmsg);
		else if (strcspn(bmsg, "\r\n"))			/* unless it just happened */
			fl_addto_browser(fd_topFrame->		/* in which case add the */ 
							 lstBrowser, bmsg);	/* new line at the end */
		needNL = 1;								/* indicate we need NL */
	}
	return bw;									/* return bytes written */
}

void bflush(void)
{
	if ( *bmsg ) {								/* if leftover chars exist */
		fl_replace_browser_line(				/* last line input lacked */
			fd_topFrame->lstBrowser,			/* an accepted new line char */
			fl_get_browser_maxline(				/* so replace it */
				fd_topFrame->lstBrowser),
				bmsg);
		*bmsg = 0;								/* indicate nothing here */
		needNL = 0;								/* no incomplete line */
	}
	where = bmsg;								/* and start all over again */
}


/* Argument list building functions for pppd. freePppdArgs() is called
   to free previously used arguments, while pppdArg() gradually builds
   the pppd argument list up to MAXARGS_CHILD arguments; it requires a
   null terminated variable argument list. */

#define MAXARGS_CHILD 64

char *pppd_arg[MAXARGS_CHILD+1];				/* the argument list strings */
int npppd_args = 0;								/* current number of args */

void freePppdArgs(void)
{
	int i;

	for (i=0; i<npppd_args; i++)				/* free all current args */
		free(pppd_arg[i]);
	npppd_args = 0;								/* and restart counter */
}

void pppdArg(char *va, ...)
{
	va_list ap;
	char *arg;

	va_start(ap, va);							/* start variable arg list */
	while ((arg=va_arg(ap,char*))!=(char*)0) {	/* get next argument */
		if (npppd_args > MAXARGS_CHILD-1) {		/* next argument overflows ? */
			fprintf(stderr, "xISP: %s %s\n",	/* yup, bail out */
					"pppdArg",
					"argument overflow!");
			exit(1);
		}
		if ((pppd_arg[npppd_args] =				/* nope, try to allocate it */
			 malloc(strlen(arg)+1)) == NULL)	/* if allocation failed */
			doErr("pppdArg");					/* then bail out */
		strcpy(pppd_arg[npppd_args], arg);		/* all OK, copy it */
		++npppd_args;							/* and increment the counter */
	}
	va_end(ap);									/* end variable arg list */
}

/* Executes pppd process with the argument list given. The argument
   list is terminated by a (char *)0 pointer. It also initializes
   the browser-update timer. */

void pppd(char *args[])
{
	pppdPPID = pppdPID = fork();				/* fork to create child */
	if (pppdPID < 0)							/* ret < 0 : fork failed */
		doErr("pppd: fork");
	if (pppdPID) {								/* in parrent process */
		freePppdArgs();							/* free used arguments */
		dialerON = 1;							/* indicate dial in progress */
		pppdStat = 0;							/* status not collected yet */
		fl_set_timer(btimer, BU_INTERVAL);		/* start callback timer */
	}
	else {										/* in pppd process */
		umask(022);								/* ensure readable pid file */
		args[npppd_args] = (char *)0;			/* in /var/run, terminate */
		execv(args[0], args);					/* arg list and exec pppd */
		doErr("pppd: execv");					/* return here means error */
	}
}


/* Check script-line syntax to comply with xisp/xispdial syntax. Make
   sure we have maximum one %U and one %P in the script line segment */

int lineCheck(char *sline)
{
	char *pc = sline;
	int len = strlen(sline), haveU = 0, haveP = 0;

	for (; *pc; pc++) {							/* check all '%' characters */
		for(; *pc!='%'; pc++)					/* by skipping the rest */
			if (*pc == 0) return 1;
		if (pc++ > sline+len-2)					/* is '%' the last char? */
			if (*pc != '%')						/* yes, this is no good */
				return 0;						/* unless last char is '%' */
		if (*pc == 'U') {						/* next char is 'U' ? */
			if ( haveU ) return -1;				/* yes, make sure we have */
			++haveU;							/* only one %U per line */
		}
		else if (*pc == 'P') {					/* next char is 'P' ? */
			if ( haveP ) return -1;				/* yes, make sure we have */
			++haveP;							/* only one %P per line */
		}
		else if (*pc != '%') return 0;			/* else char must be '%' */
	}
	return 1;									/* done, all OK */
}

/* Assemble entire line; make ready for xispdial consumption.
   Replace %U and %P codes with username and password */

char *linePrep(xisprc_t *p, char *Esline, char *Ssline)
{
	static char tline[2*MAXLEN_SLINE+5+5*MAXLEN_ACCOUNT+1] = {0};
	char ppasswd[MAXLEN_PASSWD+1] = {0}, eppasswd[2*MAXLEN_PASSWD+1] = {0},
		 *pc, *tpc = tline, *epc;

	pdecode(ppasswd, p->passwd);				/* decrypt password */
	for (pc=ppasswd, epc=eppasswd;				/* escape any '%' characters */
		 *pc; pc++, epc++) {					/* so they don't screw up */
		*epc = *pc;								/* the sprintf() statement */
		if (*pc == '%') {						/* in xispdial, by escaping */
			++epc;								/* a la printf any %'s found */
			*epc = '%';
		}
	}
	*tpc++ = '\'';								/* opening single quote */
	for (pc = Esline; *pc; pc++) {				/* check all '%' characters */
		for(; *pc && *pc!='%'; pc++)			/* by skipping the rest */
			*tpc++ = *pc;						/* while copying them in */
		if (*pc == 0) break;					/* reached end so skip rest */
		else ++pc;								/* else, point to next char */
		if (*pc == 'U') {						/* next char is 'U' */
			sprintf(tpc, "%s", p->account);		/* so stick username in */
			tpc += strlen(p->account);
		}
		else if (*pc == 'P') {					/* next char is 'P' */
			sprintf(tpc, "%s", eppasswd);		/* so stick password in */
			tpc += strlen(eppasswd);
		}
		else *tpc++ = *pc;						/* tack the second '%' on */
	}
	*tpc++='\''; *tpc++=' '; *tpc++='\'';		/* insert quotes and space */
	for (pc = Ssline; *pc; pc++) {				/* check all '%' characters */
		for(; *pc && *pc!='%'; pc++)			/* by skipping the rest */
			*tpc++ = *pc;						/* while copying them in */
		if (*pc == 0) break;					/* reached end so skip rest */
		else ++pc;								/* else, point to next char */
		if (*pc == 'U') {						/* next char is 'U' */
			sprintf(tpc, "%s", p->account);		/* so stick username in */
			tpc += strlen(p->account);
		}
		else if (*pc == 'P') {					/* next char is 'P' */
			sprintf(tpc, "%s", eppasswd);		/* so stick password in */
			tpc += strlen(eppasswd);
		}
		else *tpc++ = *pc;						/* tack the second '%' on */
	}
	*tpc++ = '\'';								/* closing single quote */
	*tpc = 0;									/* terminate assembled line */
	return tline;
}

/* Create dialing environment, according to the various dial-in and/
   or call-back access combinations; some utility functions follow */

#ifdef ISPENV_USEVARS

#if BROKEN_SPRINTF
#define min(a,b) (((a)<(b))?(a):(b))
#endif

/* Function for accumulating xispdial
   environment in an environment variable */

static void
eprintf(char *fmt, ...)
{
	va_list ap;
	static char *buf = NULL;
	static size_t buflen = 0;
	static size_t bufsiz = 0;
#if BROKEN_SPRINTF
	char etmp[MAXLEN_FNAME + 32];
#endif

	va_start(ap, fmt);
	if (fmt) {
		unsigned int n;

		while (buflen >= bufsiz ||
#if BROKEN_SPRINTF
			   (vsprintf(etmp, fmt, ap),
				bcopy(etmp, buf+buflen, min(bufsiz-buflen, strlen(etmp))),
				n = strlen(etmp))
#else
			   (n = vsnprintf(buf+buflen, bufsiz-buflen, fmt, ap))
#endif
			   >= bufsiz - buflen
			  )
		{
			if ((buf = realloc(buf, bufsiz+1024)) == NULL)
				doErr("eprintf, realloc");
			memset(&buf[bufsiz], 0, 1024);
			bufsiz += 1024;
		}
		buflen += n;
	}
	else {
#if defined(SUNOS41x) || defined(SUNOS5x)
		int len = strlen(ENVVAR) + 1;

		if ((buf = realloc(buf, bufsiz+len+1)) == NULL)
			doErr("eprintf, realloc");
		memset(&buf[bufsiz], 0, len+1);
#ifdef SUNOS41x
		bcopy(buf, &buf[len], bufsiz);
#else
		memmove(&buf[len], buf, bufsiz);
#endif	/* SUNOS41x */
		bufsiz += (len + 1);
		strncpy(buf, ENVVAR, len-1);
		buf[len-1] = '=';
		putenv(buf);
#else
		setenv(ENVVAR, buf, 1);
#endif	/* defined(SUNOS41x) || defined(SUNOS5x) */
		buf[0] = '\0';
		buflen = 0;
	}
	va_end(ap);
}

#else /* ISPENV_USEVARS */

/* Function for accumulating xispdial environment in a file */

static void
eprintf(char *fmt, ...)
{
	va_list ap;
	static FILE *envfp = NULL;

	va_start(ap, fmt);
	/* If the 'fmt' arg is NULL, close the environment file. */
	if (fmt) {
		/* Create the environment file if it is not already open. */
		if (envfp == NULL) {
			envfp = fopen(envfname, "w");			/* create the file */
			if (envfp == NULL)						/* bail out on error */
				doErr("writeISPenv, fopen");
		}
		vfprintf(envfp, fmt, ap);
	}
	else if (envfp != NULL) {
		fclose(envfp);
		envfp = NULL;
	}

	va_end(ap);
}

#endif /* ISPENV_USEVARS */

/* Send terminal parameters for xispterm */

void envTermParms(xisprc_t *p)
{
	int w, h, tmp;
	FL_Coord x, y;

	eprintf("%s\n",							/* tell xispdial the DISPLAY */
			XDisplayName(NULL));
	eprintf("#%02X%02X%02X\n",				/* the background color */
			bgcol.r, bgcol.g, bgcol.b);
	fl_get_winorigin(topWin, &x, &y);		/* a geometry position */
	x = (x-40 > 0)? x-40 : 0;
	y = (y-60 > 0)? y-60 : 0;
	w = fl_get_char_width(					/* a width in chars */
	  FL_FIXED_STYLE, FL_MEDIUM_SIZE);
	w =  w * (p->termW) + 24;
	h = fl_get_char_height(					/* and a height in chars */
	  FL_FIXED_STYLE, FL_MEDIUM_SIZE,
	  &tmp, &tmp);
	h = h * (p->termH) + 59;
	eprintf("%dx%d+%d+%d\n", w, h, x, y);
}

/* Return mode according to selected options */

unsigned char envAdjustMode(xisprc_t *p)
{
	unsigned int mode = SCRIPT_DIALIN;
	void adjustPAPCap();

	adjustPAPCap();								/* check PAP capability */
	if (p->operOpts & (PAP_LOGIN | PAPS_LOGIN |	/* authenticated login? */
					   CHAPS_LOGIN)) {
		mode = AUTH_DIALIN;						/* yes, set generic mode */
		if (p->operOpts & MANUAL_LOGIN)			/* manual login during dial- */
			mode |= MANUAL_DIALIN;				/* in desired -> add option */
		if (p->operOpts & CALL_BACK) {			/* if call-back selected */
			if (p->operOpts & CB_NT_RAS) {		/* NT RAS call-back server? */
				if (firstConn)					/* first call -> AUTH_DIALIN */
					mode |= NT_RAS_DIALIN;		/* tell xispdial it's NT-RAS */
				else {
					mode |= NT_RAS_CALLBACK;	/* second call -> call-back */
					mode &= ~MANUAL_DIALIN;		/* flag has no meaning in */
				}								/* the RAS call-back phase */
			}
			else								/* otherwise, generic */
				mode |= AUTH_CALLBACK;			/* authenticated call-back */
		}
	}
	else {										/* else, not authenticated */
		if (p->operOpts & MANUAL_LOGIN)			/* is this a manual login? */
			mode = MANUAL_DIALIN;				/* yes, set generic mode */
		else									/* else not manual login */
			mode = SCRIPT_DIALIN;				/* so it's simple, scripted */
		if (p->operOpts & CALL_BACK) {			/* is call-back enabled? */
			if (p->operOpts & CBMAN_LOGIN)		/* call-back phase manual? */
				mode |= MANUAL_CALLBACK;		/* yes, combination mode */
			else								/* else, scripted call-back */
				mode |= SCRIPT_CALLBACK;		/* phase is assumed */
		}
	}
	return mode;
}

int writeISPenv(xisprc_t *p)
{
	int i, ret = 0;
	unsigned int mode = SCRIPT_DIALIN;

	eprintf("%s\n", Chat);						/* write all global */
	eprintf("%s\n", Pipe);						/* dialing variables */
	eprintf("%s\n", Terminal);
	eprintf("%d\n", p->maxAttempts);
	eprintf("%d\n", p->sleepDelay);
	eprintf("%d\n", p->connectWait);
	eprintf("%d\n", p->numPhones);
	for (i=0; i<p->numPhones; i++)
		eprintf("%s\n", p->phone[i]);
	mode = envAdjustMode(p);
	eprintf("%X\n", mode);						/* send connection mode */
	switch (mode) {								/* send data according to */
												/* the selected mode */
		case SCRIPT_DIALIN:
			eprintf("%s\n", p->account);		/* account and script lines */
			eprintf("%d\n",p->numSlines);		/* only for scripted login */
			for (i=0; i<p->numSlines; i++) {
				ret = lineCheck(p->sline[i]);	/* check expect-line syntax */
				if (ret > 0)
					ret = lineCheck(p->sline[	/* if OK, check send-line */
							MAXNUM_SLINES+i]);
				if (ret <= 0) {					/* syntax problem? */
					eprintf(NULL);				/* yes, close file */
					return ret;					/* and bail out */
				}
				eprintf("%s\n",					/* print expect-send pairs */
				  linePrep(p, p->sline[i],		/* as prepared by routine */
				  p->sline[MAXNUM_SLINES+i]));	/* linePrep() */
			}
		break;

		case SCRIPT_DIALIN|SCRIPT_CALLBACK:
			eprintf("%s\n", p->account);		/* account and script lines */
			eprintf("%d\n",p->numSlines);		/* only for scripted login */
			for (i=0; i<p->numSlines; i++) {
				ret = lineCheck(p->sline[i]);	/* check expect-line syntax */
				if (ret > 0)
					ret = lineCheck(p->sline[	/* if OK, check send-line */
							MAXNUM_SLINES+i]);
				if (ret <= 0) {					/* syntax problem? */
					eprintf(NULL);				/* yes, close file */
					return ret;					/* and bail out */
				}
				eprintf("%s\n",					/* print expect-send pairs */
				  linePrep(p, p->sline[i],		/* as prepared by routine */
				  p->sline[MAXNUM_SLINES+i]));	/* linePrep() */
			}
			eprintf("%d\n",p->CBDelay);			/* print out call-back delay */
			eprintf("%d\n",p->numCBSlns);		/* and the call-back */
			for (i=0; i<p->numCBSlns; i++) {	/* script lines */
				ret = lineCheck(p->CBsln[i]);	/* check expect-line syntax */
				if (ret > 0)
					ret = lineCheck(p->CBsln[
							MAXNUM_SLINES+i]);	/* if OK, check send-line */
				if (ret <= 0) {					/* syntax problem? */
					eprintf(NULL);				/* yes, close file */
					return ret;					/* and bail out */
				}
				eprintf("%s\n",					/* print expect-send pairs */
				  linePrep(p, p->CBsln[i],		/* for call-back lines as */
				  p->CBsln[MAXNUM_SLINES+i]));	/* prepared by linePrep() */
			}
		break;

		case SCRIPT_DIALIN|MANUAL_CALLBACK:
			eprintf("%s\n", p->account);		/* account and script lines */
			eprintf("%d\n",p->numSlines);		/* only for scripted login */
			for (i=0; i<p->numSlines; i++) {
				ret = lineCheck(p->sline[i]);	/* check expect-line syntax */
				if (ret > 0)
					ret = lineCheck(p->sline[	/* if OK, check send-line */
							MAXNUM_SLINES+i]);
				if (ret <= 0) {					/* syntax problem? */
					eprintf(NULL);				/* yes, close file */
					return ret;					/* and bail out */
				}
				eprintf("%s\n",					/* print expect-send pairs */
				  linePrep(p, p->sline[i],		/* as prepared by routine */
				  p->sline[MAXNUM_SLINES+i]));	/* linePrep() */
			}
			envTermParms(p);					/* send terminal parameters */
			eprintf("%d\n",p->CBDelay);			/* and call-back delay */
		break;

		case MANUAL_DIALIN:
			envTermParms(p);					/* send terminal parameters */
		break;

		case MANUAL_DIALIN|SCRIPT_CALLBACK:
			envTermParms(p);					/* send terminal parameters */
			eprintf("%d\n",p->CBDelay);			/* and call-back delay */
			eprintf("%d\n",p->numCBSlns);		/* then send the call-back */
			for (i=0; i<p->numCBSlns; i++) {	/* script lines */
				ret = lineCheck(p->CBsln[i]);	/* check expect-line syntax */
				if (ret > 0)
					ret = lineCheck(p->CBsln[
							MAXNUM_SLINES+i]);	/* if OK, check send-line */
				if (ret <= 0) {					/* syntax problem? */
					eprintf(NULL);				/* yes, close file */
					return ret;					/* and bail out */
				}
				eprintf("%s\n",					/* print expect-send pairs */
				  linePrep(p, p->CBsln[i],		/* for call-back lines as */
				  p->CBsln[MAXNUM_SLINES+i]));	/* prepared by linePrep() */
			}
		break;

		case MANUAL_DIALIN|MANUAL_CALLBACK:
			envTermParms(p);					/* send terminal parameters */
			eprintf("%d\n",p->CBDelay);			/* and call-back delay */
		break;

		case AUTH_DIALIN:
		case AUTH_DIALIN|NT_RAS_DIALIN:
		break;

		case AUTH_DIALIN|AUTH_CALLBACK:
		case AUTH_DIALIN|NT_RAS_CALLBACK:
			eprintf("%d\n",p->CBDelay);			/* send the call-back delay */
		break;

		case AUTH_DIALIN|MANUAL_DIALIN:
		case AUTH_DIALIN|MANUAL_DIALIN|
			 NT_RAS_DIALIN:
			envTermParms(p);					/* send terminal parameters */
		break;

		case AUTH_DIALIN|MANUAL_DIALIN|
			 AUTH_CALLBACK:
			envTermParms(p);					/* send terminal parameters */
			eprintf("%d\n",p->CBDelay);			/* and call-back delay */
		break;

		default: break;
	}
	eprintf("%s\n", p->modemReset);				/* modem reset */
	eprintf("%s\n", p->modemInit);				/* and init strings */
	if (p->operOpts & MODEM_TONEDIAL)			/* dialing command char(s) */
		eprintf("%sDT\n",
				p->dialExtra);
	else if (p->operOpts & MODEM_ISDNDIAL)
		eprintf("%sDI\n", p->dialExtra);
	else
		eprintf("%sDP\n", p->dialExtra);
	eprintf("%s\n", p->modemConnect);			/* modem connect string */
	eprintf(NULL);								/* finally, close the file */
	return 1;									/* all OK */
}

/* Create PAP authentication file */

void writeISPPAP(xisprc_t *p)
{
	FILE *papfp;
	char ppasswd[MAXLEN_PASSWD+1] = {0};

	papfp = fopen(papfname, "w");				/* create the file */
	if (papfp == NULL)							/* bail out on error */
		doErr("writeISPPAP, fopen");
	pdecode(ppasswd, p->passwd);				/* decrypt password */
	fprintf(papfp, "%s\n", p->account);			/* write username and */
	fprintf(papfp, "%s\n", ppasswd);			/* password for PAP */
	fclose(papfp);
}

/* Update all connection indicators */

void updateStat(int linkOK)
{
	fl_set_object_label(fd_topFrame->statusText,
		(connected)? ((linkOK) ? "ON-LINE" : "CARRIER") :
					 ((dialerON) ? "XISPDIAL" : "OFF-LINE"));
	fl_set_object_label(fd_topFrame->speedText,
		(connected) ? connSpeed : EMPTY_SPEED);
	fl_set_object_label(fd_topFrame->IPText,
		(linkOK) ? connIP : EMPTY_IP);
}


/* Figure out the name of the PPP interface created by the pppd process
   we started. Routine is functional only when RUNDIR is available, and
   is quite simplistic in its assumptions. More specifically, as there's
   no "robust" way of figuring out if the pppd we started indeed was the
   parent process of the detached pppd who's PID is saved in the file,
   it assumes that there may be only one active PPP interface at any
   given time. Needless to say that this will not work on hosts which
   have a static PPP link always active */

#ifdef RUNDIR
#define MAXNUM_PPPIF 4
int getIFname(char *IFname)
{
	char buf[16], IFn[8] = {0};
	int IFc, fd, br;

	for (IFc=0; IFc<MAXNUM_PPPIF; IFc++)
	{
		sprintf(IFn, "%s%d", pidfp, IFc);		/* IF PID-filename prefix */
		strcpy(runfnp, IFn);
		strcat(runfnp, ".pid");					/* IF PID-filename suffix */
		fd = open(PIDFname, O_RDONLY);			/* open it for reading */
		if (fd < 0) 							/* file not there */
			continue;							/* try next available name */
		br = read(fd, buf, 16);					/* read the PID string */
		close(fd);								/* close the file */
		if (br < 0)								/* if error in reading */
			continue;							/* try next available name */
		else {
			strcpy(IFname, IFn);				/* save interface name */
			return 1;							/* indicate found */
		}
	}
	*IFname = 0;								/* indicate nothing found */
	return 0;
}
#else
int getIFname(char *IFname)
{
	strcpy(IFname, "ppp0");
	return 1;
}
#endif


/* Search for the PID of a process by its name and its parent PID. If
   allow_detached is set, then init's PID (i.e., 1) is also considered
   a valid parent PID. */

#if !defined(SUNOS41x) && !defined(SUNOS5x) && !(defined(BSD) && BSD >= 199306)
int procPID(char *procname, int parentPID, int allow_detached)
{
	DIR *procd;
	struct dirent *direntp;
	char name[32], statbuf[64], *p;
	int statd, br, pid = 0, ppid;

	procd = opendir("/proc");					/* open the /proc tree */
	while ((direntp=readdir(procd)) != NULL &&	/* search all entries */
		   !pid ) {
		if (! atoi(direntp->d_name))			/* skip non numeric entries */
			continue;
		p = direntp->d_name;					/* file name is PID number */
		strcpy(name, "/proc/");					/* form status file name */
		strcat(name, p);
		strcat(name, "/status");
		if ((statd=open(name,O_RDONLY)) < 0) {	/* and open it */
			closedir(procd);					/* bail out on failure */
			return 0;
		}
		br = read(statd, statbuf, 63);			/* read first few lines */
		close(statd);							/* and close it */
		if (br < 0) {							/* bail out if read fails */
			closedir(procd);
			return 0;
		}
		statbuf[br] = 0;						/* zero terminate buffer */
		sscanf(statbuf, "Name: %s", name);		/* get process name */
		if (strcmp(name, procname))				/* continue if it's not the */
			continue;							/* one we're looking for */
		p = strstr(statbuf, "Pid:");			/* find PID string */
		sscanf(p, "Pid: %d PPid: %d",			/* and read PID and PPID */
			   &pid, &ppid);
		if (allow_detached) {					/* detached OK -> check init */
			 if (ppid != 1 &&					/* if parent is not init nor */
			 	 ppid != parentPID)				/* the specified parent PID */
				pid = 0;						/* then ignore and continue */
		}
		else {									/* else, detached not OK, so */
			if (ppid != parentPID)				/* if parent does not match */
				pid = 0;						/* then ignore and continue */
		}
	}											/* so ignore it and continue */
	closedir(procd);							/* search done */
	return pid;									/* return whatever we found */
}
#else
int procPID(char *procname, int parentPID, int allow_detached)
{
	FILE *infofp;
	char psLine[128], name[32];
	int pid = 0, ppid;

#ifdef SUNOS41x
	infofp = popen("/bin/ps -aclxw", "r");		/* open proc info stream */
#elif (defined(BSD) && BSD >= 199306)
	infofp = popen("/bin/ps -aclxw", "r");
#else
	infofp = popen("/bin/ps -el", "r");
#endif
	if (infofp == NULL)							/* failed to open? */
		return 0;								/* yes, bail out */
	while (fgets(psLine, 128, infofp)!=NULL &&	/* open OK, read entries */
		   !pid) {
#ifdef SUNOS41x
		if (sscanf(psLine, "%*s %*s %d %d %*s"
				" %*s %*s %*s %*s %*s %*s %*s"
				" %*s %s", &pid, &ppid, name))
#elif (defined(BSD) && BSD >= 199306)
		if (sscanf(psLine, "%*s %d %d %*s"
				" %*s %*s %*s %*s %*s %*s %*s"
				" %*s %s", &pid, &ppid, name))
#else
		if (sscanf(psLine, "%*s %*s %*s %d %d"
				" %*s %*s %*s %*s %*s %*s %*s"
				" %*s %s", &pid, &ppid, name))
#endif
		{
			if (strcmp(name, procname)) {		/* continue if no match */
				pid = 0;
				continue;
			}
			if (allow_detached) {				/* detached OK -> check init */
				 if (ppid != 1 &&				/* parent is not init nor */
				 	 ppid != parentPID)			/* the specified parent PID */
					pid = 0;					/* so ignore and continue */
			}
			else {
				if (ppid != parentPID)			/* parent does not match */
					pid = 0;					/* so ignore and continue */
			}
		}
		else pid = 0;
	}
	pclose(infofp);								/* search done */
	return pid;									/* return whatever we found */
}
#endif


/* Retrieve PID of spawned pppd process from RUNDIR/(s)ppp?.pid and/or from
   the PPPDLCKDIR/LCK..? or PPPDLCKDIR/LK.major(dev).major(rdev).minor(rdev)
   lock file for the modem device (the actual name is assembled in
   call-back function doConnect()). If RUNDIR is not defined, search /proc
   for the pppd entry in the process table. Note that if RUNDIR is not
   available, the pppIF global var defaults to "ppp0", so checks for cases
   of multiple PPP interfaces are not possible */

int getPppdPID(char *devlock)
{
#if defined(RUNDIR) && defined (PPPDLCKDIR)
	char buf[32];
	int fd = -1, br;

	if (*pppIF ||								/* if we know, or if we can */
		(*pppIF == 0 && getIFname(pppIF)))		/* find the PID file name */
	{
		strcpy(runfnp, pppIF);					/* form the complete file */
		strcat(runfnp, ".pid");					/* name for this interface */
		fd = open(PIDFname, O_RDONLY);			/* and open it for reading */
	}
	if (fd < 0 && devlock != NULL) {			/* if PID file is not there */
		strcpy(lockfnp, devlock);				/* try the lock file for the */
		fd = open(lockFname, O_RDONLY);			/* modem port device; this */
	}											/* also contains pppd's PID */
	if (fd < 0)									/* if file(s) not there */
		return 0;								/* indicate nothing found */
	br = read(fd, buf, 32);						/* else read the PID string */
	close(fd);									/* close the file */
	if (br < 0)									/* if error in reading */
		return 0;								/* ppp?.pid or LCK..? file */
	else {										/* screwed -> ret not found */
		buf[br] = 0;							/* else mark end of string */
		return atoi(buf);						/* and return converted */
	}
#elif defined(PPPDLCKDIR)
	char buf[32];
	int fd = -1, br;

	if (devlock != NULL) {						/* if a lock is specified */
		strcpy(lockfnp, devlock);				/* try the lock file for it */
		fd = open(lockFname, O_RDONLY);			/* this file contains pppd's */
	}											/* PID */
	if (fd < 0)									/* name or file not there */
		return 0;								/* indicate nothing found */
	br = read(fd, buf, 32);						/* else read the PID string */
	close(fd);									/* close the file */
	if (br < 0)									/* if error in reading */
		return 0;								/* ppp?.pid or LCK..? file */
	else {										/* screwed -> ret not found */
		buf[br] = 0;							/* else mark end of string */
		return atoi(buf);						/* and return converted */
	}
#else
	return procPID("pppd", pppdPPID, 1);		/* return PID search result */
#endif
}

/* Retrieve PID of ip-up/ip-down script spawned by the child pppd process */

int getipUDPID()
{
	return procPID(ipWhich, pppdPID, 0);		/* return PID search result */
}


/* Wait for pppd to die, after sending SIGINT. Procedure provided because
   pppd appears not to honor SIGINT while in the connection phase, when
   sending LCP packets to its peer. New pppd started during this phase
   will terminate with error. If after sending KILL signals (if it comes
   to that) and before returning, the procedure makes sure that there's no
   ppp?.pid file left over if there's no pppd process in the process table.
   This is for accommodating pppd binaries compiled without the option for
   automatic deletion of stale ppp?.pid files. All this, of course, only
   when RUNDIR is available. */

#define MAXWAIT_PPPD 8		/* number of iterations waiting for pppd to die */

/* Note that the above number is multiplied by 5 when we are waiting for
   pppd to exit gracefully; this is used before starting pppd a second time,
   when we are in the process of setting up an NT-RAS call-back connection */

void waitPppd(int beNice)
{
	int i, pid, stat = 0, wait = MAXWAIT_PPPD;

#ifdef PPPDLCKDIR
	if (beNice && (pid=getPppdPID(devLock))) {	/* if in "graceful" mode */
#else											/* and provided we can */
	if (beNice && (pid=getPppdPID(NULL))) {		/* figure out pppd's PID */
#endif
		wait *= 5;								/* multiply iterations by 5 */
		bprintf("%s%d%s%s%s...\n",				/* print message on browser */
				"Waiting for pppd[",pid,"] (",
#if defined(SUNOS5x) || defined(SVR4)
				devPath,
#else
				devName,
#endif
				") to terminate...");
		fl_check_forms();						/* update browser */
	}
	for (i=0; i<wait && !stat &&		 		/* as long as pppd is there */
#ifdef PPPDLCKDIR								/* and up to "wait" times */
			  (pid=getPppdPID(devLock));
#else
			  (pid=getPppdPID(NULL));
#endif
		i++)
	{
		if (! beNice) {
			stat = kill(pid, SIGINT);			/* send SIGINT to pppd */
			bprintf("%s%d%s...%d\n",			/* print message on browser */
					"Waiting for pppd[", pid,
					"] to die", i+1);
		}
		fl_check_forms();						/* update forms */
		mSleep(2000);							/* wait a couple of seconds */
	}
	if (i >= wait || stat < 0) {				/* timeout or signal failed */
#ifdef PPPDLCKDIR
		if ((pid = getPppdPID(devLock))) {		/* if pppd's PID exists */
#else
		if ((pid = getPppdPID(NULL))) {
#endif
			bprintf("%s%d%s\n",					/* print message on browser */
					"Killing pppd[", pid, "].");
			fl_check_forms();					/* update browser */
			kill(pid, SIGKILL);					/* send SIGKILL to pppd */
			mSleep(1000);						/* and wait a second */
		}
#ifdef RUNDIR
#ifdef PPPDLCKDIR
		if ((pid = getPppdPID(devLock))) {		/* if pppd's PID exists */
#else
		if ((pid = getPppdPID(NULL))) {
#endif
			if (! procPID("pppd",pppdPPID,1)) {	/* without a pppd process */
				bprintf("\n@bStale %s file "	/* print a hint for the user */
				  "exists.\nIf pppd fails to "	/* with regards to the stale */
			 	  "start, please\nremove it "	/* ppp?.pid file found */
			 	  "manually and try again.\n",
				  PIDFname);
				fl_check_forms();				/* update browser */
			}
		}
#endif
	}
	else if (wait > 0)							/* signal OK, but some delay */
		mSleep(1500);							/* so wait another 1.5 sec */
}


/* String <-> IP address conversions */

void IPToStr(unsigned char *ip, char *str)		/* convert IP to string */
{
	sprintf(str, "%u.%u.%u.%u",
		ip[0],ip[1],ip[2],ip[3]);
}

int StrToIP(char *str, unsigned char *ip)		/* convert string to IP */
{
	unsigned int iip[4];
	int n, i;

	n = sscanf(str, "%3u.%3u.%3u.%3u", &iip[0],
			   &iip[1], &iip[2], &iip[3]);
	if (n == 4)
		for (i=0; i<4; ip++, i++) *ip = iip[i];
	return n;
}


/* Check optional pppd options file */

int pppdOptsFileOK(void)
{
	struct stat st;

	if (!stat(PPPD_OPTIONS, &st))				/* does it exist ? */
		return (st.st_size > 0);				/* yup, if size > 0 all OK */
	else
		return 0;								/* no it doesn't */
}


/* Reduce colormap usage */

void colorSqueeze(void)
{
	int i;

	for (i=0; i<FL_FREE_COL1; i++) {
		switch (i) {

			case FL_BLACK:						/* except for these which */
			case FL_CYAN:						/* are used in our code */
			case FL_DARKCYAN:
			case FL_WHITE:
			case FL_COL1:
			case FL_RIGHT_BCOL:
			case FL_BOTTOM_BCOL:
			case FL_TOP_BCOL:
			case FL_LEFT_BCOL:
			case FL_MCOL:
			case FL_INACTIVE:
			case FL_WHEAT:

			case FL_INDIANRED:					/* these are replaced */
			case FL_GREEN:
				break;

			default:
				fl_set_icm_color(i, 0,0,0);		/* reset all unused internal */
		}										/* colormap colors to black */
	}
}

/* Parse user-specified background color from string */

void bgColor(char *color)
{
	color_t bgcol_default = BGCOL_DEFAULT;
	int i;

	i = sscanf(color, "#%2X%2X%2X",					/* scan the hex color */
			   &bgcol.r, &bgcol.g, &bgcol.b);
	if (i != 3)										/* if scan unsuccessful */
		bgcol = bgcol_default;						/* use the default color */
	fl_mapcolor(FL_INDIANRED,						/* replace the ones used */
				bgcol.r, bgcol.g, bgcol.b);
	tabcol.r = 2.9 * bgcol.r / 4.;
	tabcol.g = 2.5 * bgcol.g / 4.;
	tabcol.b = 2.5 * bgcol.b / 4.;
	fl_mapcolor(FL_GREEN,
				tabcol.r, tabcol.g, tabcol.b);
}


/* Make sure the fonts used are the same as those used for designing
   all program forms. Although generally this isn't such a good idea,
   it nevertheless prevents selection of "ugly" or "oversize" fonts
   by the XForms GUI library */

void fontSelect(void)
{
	int status = 0;

	status = fl_set_font_name(FL_NORMAL_STYLE,
				"-adobe-helvetica-medium-r-*-*-*-?-75-75-*-*-*-*");
	status += fl_set_font_name(FL_BOLD_STYLE,
				"-adobe-helvetica-bold-r-*-*-*-?-75-75-*-*-*-*");
	status += fl_set_font_name(FL_ITALIC_STYLE,
				"-adobe-helvetica-medium-o-*-*-*-?-75-75-*-*-*-*");
	status += fl_set_font_name(FL_BOLDITALIC_STYLE,
				"-adobe-helvetica-bold-o-*-*-*-?-75-75-*-*-*-*");
	status += fl_set_font_name(FL_FIXED_STYLE,
				"-adobe-courier-medium-r-*-*-*-?-75-75-*-*-*-*");
	status += fl_set_font_name(FL_FIXEDBOLD_STYLE,
				"-adobe-courier-bold-r-*-*-*-?-75-75-*-*-*-*");
	status += fl_set_font_name(FL_FIXEDITALIC_STYLE,
				"-adobe-courier-medium-o-*-*-*-?-75-75-*-*-*-*");
	status += fl_set_font_name(FL_FIXEDBOLDITALIC_STYLE,
				"-adobe-courier-bold-o-*-*-*-?-75-75-*-*-*-*");
	if (status != 0) {
		fprintf(stderr, "xISP: warning: desired font(s) not found.\n"
				"Replacements may appear ugly/unintelligible.\n");
	}
}


/* Returns pointer to string with cost value properly formatted */

char *costStr(ptt_t *p, float cost)
{
	char format[MAXLEN_CURRENCY+5+1];
	static char cstr[32+MAXLEN_CURRENCY+5+1];

	if (p->attribs & PTT_CUR_AFTER_COST)		/* format according to type */
		sprintf(format, "%%.%df %s",
		 p->decimals, p->currency);
	else
		sprintf(format, "%s %%.%df",
		 p->currency, p->decimals);
	sprintf(cstr, format, cost);				/* build the cost printout */
	return cstr;
}

/* Returns pointer to string with time-part value properly formatted */

char *uc2ts(ruletime_t time, char part)
{
	static char tstr[3];

	switch (part) {
		case 'H': sprintf(tstr, "%02d", time.h); break;
		case 'M': sprintf(tstr, "%02d", time.m); break;
		case 'S': sprintf(tstr, "%02d", time.s); break;
		default: strcpy(tstr, "00"); break;
	}
	return tstr;
}

/* Returns pointer to string with date-part value properly formatted */

char *uc2ds(ruledate_t date, char part)
{
	static char dstr[3];

	switch (part) {
		case 'M': if (date.day > 0) sprintf(dstr, "%02d", 1+date.mon);
				  else strcpy(dstr, "00");
				  break;
		case 'D': sprintf(dstr, "%02d", date.day);
				  break;
		default: strcpy(dstr, "00"); break;
	}
	return dstr;
}


/* Updates xisp logging files if logging is enabled */

void updateLogs(int online)
{
	time_t ct;
	char cts[32];
	unsigned long packetsTXed, packetsRXed;

	if (! (global.logOpts & LOG_NONE)) {		/* if logging desired */
		if (online) ct = upTime;				/* pick appropriate time */
		else ct = downTime;
		strcpy(cts, ctime(&ct));				/* format it the usual way */
		cts[strlen(cts)-1] = 0;					/* remove trailing '\n' */
		if (p_ptt->attribs & PTT_BY_UNIT)		/* if the selected PTT */
			totalCost = totalUnits *			/* charges by unit, then */
						  p_ptt->cost_quantum;	/* calculate total cost and */
		writeXispCost(totalTime, totalCost);	/* save time/cost totals */
		pppPkts(pppIF, &packetsTXed,			/* get packet totals */
				&packetsRXed);
		if (online) {							/* if coming "up", then */
			if (initLogFnames(&global))			/* check file names; if */
				restartXispCost(&totalTime,		/* changed, reset log file */
								&totalCost);	/* this also resets totals */
		}
		writeXispLog(online, p_ptt, cts,		/* update connection logs */
			p_xisprc->descr, connIP,
			speedStr, onLineSecs, onLineCost,
			packetsTXed, packetsRXed);
		if (! online) {							/* if going "down", perform */
			if (initLogFnames(&global))			/* the file name changing */
				restartXispCost(&totalTime,		/* and log resetting AFTER */
								&totalCost);	/* the logs are updated */
		}
	}
}


/* Displays information for PTT selected on the Logging-information
   browser. Note the maximum internal length of the buffer used. */

#define MAXLEN_PTTINFB 128

int ibprt(char *fmt, ...)
{
	int bw;
	char buf[MAXLEN_PTTINFB+1] = {0};
	va_list ap;

	va_start(ap, fmt);
#ifdef BROKEN_SPRINTF
	vsprintf(buf, fmt, ap);
	bw = strlen(buf);
#else
	bw = vsprintf(buf, fmt, ap);
#endif
	va_end(ap);
	fl_addto_browser(fd_logInfo->costBrowser, buf);
	return bw;
}


void showPttInfo(ptt_t *p)
{
	char method[8], pnc[16] = {0}, pmc[16] = {0};
	int z, c;
	float minVal = 0.0, maxVal = 0.0;

	fl_clear_browser(fd_logInfo->costBrowser);
	if (p->num_zones < 1) {
		ibprt("Unitialized PTT entry\n");
		return;
	}
	if (p->attribs & PTT_PER_MINUTE)
		strcpy(method, "minute");
	else {
		if (p->attribs & PTT_PER_SECS) {
			if (p->charge_period > 1)
				sprintf(method, "%d seconds", p->charge_period);
			else
				strcpy(method, "second");
		}
		else
			strcpy(method, "unit");
	}
	ibprt("PTT entry has %d zone%s", p->num_zones,
		  (p->num_zones>1)? "s":"");
	if (p->attribs & (PTT_PER_MINUTE|PTT_PER_SECS))
		ibprt("Charging is per %s", method);
	else {
		if (p->attribs & PTT_CUR_AFTER_COST)
			ibprt("Charging is by %s, each costing %g %s",
				  method, p->cost_quantum, p->currency);
		else
			ibprt("Charging is by %s, each costing %s %g",
				  method, p->currency, p->cost_quantum);
	}

	if (p->attribs & (PTT_PER_MINUTE|PTT_PER_SECS)) {
		if (p->min_cost > 0) {
			strncpy(pnc, costStr(p, p->min_cost), 15);
			ibprt("Minimum cost charged: %s", pnc);
		}
		else
			ibprt("No minimum cost charged");
		for (z=0; z<p->num_zones; z++) {
			minVal = maxVal = p->dflt_tariff[z];
			for (c=0; c<p->num_categories; c++) {
				if (p->rule[z][c].tariff < minVal)
					minVal = p->rule[z][c].tariff;
				if (p->rule[z][c].tariff > maxVal)
					maxVal = p->rule[z][c].tariff;
			}
			if (p->num_categories > 1 && minVal != maxVal) {
				strncpy(pnc, costStr(p, 60.0*minVal), 15);
				strncpy(pmc, costStr(p, 60.0*maxVal), 15);
				ibprt("%s: %s/h to %s/h",
					  p->zone_name[z], pnc, pmc);
			}
			else {
				if (p->num_categories > 0) {
					strncpy(pnc, costStr(p, 60.0*p->rule[z][0].tariff), 15);
					ibprt("%s: %s/h", p->zone_name[z], pnc);
				}
				else {
					strncpy(pnc, costStr(p, 60.0*p->dflt_tariff[z]), 15);
					ibprt("%s: %s/h", p->zone_name[z], pnc);
				}
			}
		}
	}
	else {
		if (p->min_units > 0)
			ibprt("Minimum units charged: %d", p->min_units);
		else
			ibprt("No minimum units charged");
		for (z=0; z<p->num_zones; z++) {
			minVal = maxVal = p->dflt_tariff[z];
			for (c=0; c<p->num_categories; c++) {
				if (p->rule[z][c].tariff < minVal)
					minVal = p->rule[z][c].tariff;
				if (p->rule[z][c].tariff > maxVal)
					maxVal = p->rule[z][c].tariff;
			}
			if (p->num_categories > 1 && minVal != maxVal) {
				ibprt("%s: unit lasts %d':%d'' to %d':%d''",
					  p->zone_name[z],
					  (int)(minVal/60.0), (int)fmod(minVal, 60.0),
					  (int)(maxVal/60.0), (int)fmod(maxVal, 60.0));
			}
			else
				ibprt("%s: unit lasts %d':%d''",
					  p->zone_name[z],
					  (int)(p->rule[z][0].tariff/60.0),
					  (int)fmod(p->rule[z][0].tariff, 60.0));
		}
	}
}


/* Returns pointer to pppd version string */

struct pppdVerStruct {char major, minor, plevel;} pppdVersion(void)
{
	static char *cmd, opts[] = " /dev/null --version 2>&1",
				ver[8] = {0}, oplevel[4] = {0};
	static struct pppdVerStruct nver;
	static int first = 1;
	char buf[512];
	FILE *pfp;
	void alertMessage();

	if (*ver)
		return nver;
	if (first || *ver==0) {
		cmd = (char *)malloc(strlen(Pppd)+strlen(opts)+1);
		if (cmd == NULL)
			doErr("pppdVersion");
		strcpy(cmd, Pppd); strcat(cmd, opts);
		first = 0;
	}
	*buf = 0;
	pfp = popen(cmd, "r");
	if (pfp != NULL) {
		while (fgets(buf, 63, pfp) &&
			   sscanf(buf, "%*s version %s %*s %*s %s", ver, oplevel) < 1);
		while (fgets(buf, 63, pfp));
		pclose(pfp);
	}
	else {
		sprintf(buf, "Unable to start pppd daemon (while checking version)!\n"
				"Please check path in File->Options->Program Paths.\n");
		alertMessage("xISP: pppdVersion()", 0, 0, buf);
		nver.major = nver.minor = nver.plevel = -1;
	}
	if (*ver) {
		char *pfc = ver, *plc;

		if ((plc = strchr(pfc,'.')) != NULL) {
			*plc = 0;
			nver.major = atoi(pfc);
			pfc = plc + 1;
		}
		else {
			nver.major = nver.minor = nver.plevel = -1;
			return nver;
		}
		if ((plc = strchr(pfc,'.')) != NULL) {
			*plc = 0;
			nver.minor = atoi(pfc);
			pfc = plc + 1;
			nver.plevel = atoi(pfc);
		}
		else {
			nver.minor = atoi(pfc);
			if (*oplevel)
				nver.plevel = atoi(oplevel);
			else
				nver.plevel = -1;
		}
	}
	else
		nver.major = nver.minor = nver.plevel = -1;
	return nver;
}


/* Executes .xisp-up/-down. The up parameter is for distinguishing
   between .xisp-up and .xisp-down. The seq parameter is used only
   for .xisp-down, and it serves to distinguish between the first
   call before disconnection (seq = 0) and the second call after
   disconnection (seq = 1).  Returns the script PID number */

#define XISPUD_ARGS 8

int xispUD(int up, int seq)
{
	int pid, i;
	static char *arg[XISPUD_ARGS+1] = {NULL};
	char msg[512] = {0};
	void alertMessage();
 	static char prev = 0;
 
 	if (!up && !prev && !seq)					/* ignore first "down" call */
 		return 0;								/* without a previous "up" */
 	prev = up;									/* save state of last call */

	if (up) {
		for (i=0; i<XISPUD_ARGS; i++)			/* start freeing arg space */
			free(arg[i]);
		arg[0] = (char *)						/* allocate storage for */
			malloc(strlen(uupfname)+1);			/* argument list; all args */
		arg[1] = (char *)						/* are retained for "down" */
			malloc(strlen(pppIF)+1);			/* state also; only the */
		arg[2] = (char *)						/* name and the seq flag */
			malloc(strlen(						/* change between states */
				p_xisprc->modemDevice)+1);
		arg[3] = (char *)
			malloc(strlen(speedStr)+1);
		arg[4] = (char *)malloc(16);
		arg[5] = (char *)malloc(16);
		arg[6] = (char *)
			malloc(strlen(IPParam)+1);
		arg[7] = (char *)malloc(8);

		for (i=0; i<XISPUD_ARGS; i++)
			if (arg[i] == NULL)					/* if allocation failed */
					outofMem();					/* then exit with error */

		strcpy(arg[0], uupfname);				/* script path in arg[0] */
		strcpy(arg[1], pppIF);					/* interface in arg[1] */
		strcpy(arg[2], p_xisprc->modemDevice);	/* device in arg[2] */
		strcpy(arg[3], speedStr);				/* speed in arg[3] */
		strcpy(arg[4], IFAddr(pppIF, 0));		/* copy local and remote */
		strcpy(arg[5], IFAddr(pppIF, 1));		/* IP addresses */
		strcpy(arg[6], IPParam);				/* ipparam in arg[6] */
	}
	else {										/* "down" phase: do it twice */
		free(arg[0]);							/* due to unknown sequencing */
		arg[0] = (char *)						/* i.e. seq: 0->1 or 1->0 */
			malloc(strlen(udownfname)+1);		/* re-allocate script name */
		if (arg[0] == NULL)						/* check allocation */
			outofMem();							/* and exit if it failed */
		strcpy(arg[0], udownfname);				/* script path in arg[0] */
	}

	if (up)										/* arg[7] = 0 if called */
		strcpy(arg[7], "1");					/* before state changes or 1 */
	else										/* if called afterwards but */
		sprintf(arg[7], "%d", seq);				/* always "0" for "up" mode */
	arg[XISPUD_ARGS] = (char *)0;				/* terminate argument list */

#ifdef RUNASEUID
	if (xisp_euidaccess(arg[0], X_OK) < 0)		/* file OK and executable? */
#else
	if (access(arg[0], X_OK) < 0)
#endif
		return 0;								/* nope -> do nothing */

	pid = fork();								/* fork to create child */
	if (pid < 0)								/* ret < 0 : fork failed */
		doErr("xispUD: fork");
	if (pid)									/* in parrent process */
		return pid;								/* return PID number */
	else {										/* in script process */
		execv(arg[0], arg);						/* execute the script */
		sprintf(msg, "Can't exec %s!",			/* return here means error */
				arg[0]);
		alertMessage("xISP: xispUD()",0,0,msg);
		return 0;
	}
}


/* Records the xisp PID in .xisppid in the user's home directory, and
   also checks if another xisp process is running, via xisp's PID file.
   Prints its error message on a specially designed form */

void recordPID(void)
{
	char msg[4*128+2*1024] = {0};
	FILE *fp;
	FL_OBJECT *ans;

#ifdef RUNASEUID
	if (xisp_euidaccess(upidfname, F_OK) == 0) {
		sprintf(msg, "If that is not the case, then file %s\n"
				"either refers to an instance of xISP which terminated "
				"abnormally,\nor another user in your LAN is already "
				"running this shared copy\nof xISP. In the former case, "
				"you may press Continue to remove\n%s and start xISP.",
				upidfname, upidfname);
#else
	if (access(upidfname, F_OK) == 0) {
		sprintf(msg, "If this is not the case, then file %s\n"
			"either contains the PID of xISP running on another machine\n"
			"which mounts your home directory, or your last instance of\n"
			"xISP terminated abnormally. In the latter case, press Continue\n"
			"to remove %s and start xISP.", upidfname, upidfname);
#endif
		fl_set_object_label(fd_instanceCheck->msgString, msg);
		fl_show_form(fd_instanceCheck->instanceCheck, FL_PLACE_MOUSE,
				 	FL_TRANSIENT, "Instance Check");
		while (ans = fl_do_forms(),
			   ans != fd_instanceCheck->instanceCheckContinue &&
			   ans != fd_instanceCheck->instanceCheckAbort);
		fl_hide_form(fd_instanceCheck->instanceCheck);
		if (ans != fd_instanceCheck->instanceCheckAbort)
			unlink(upidfname);
		else {
			fl_finish();
			_exit(1);
		}
		fl_free_form(fd_instanceCheck->instanceCheck);
	}
	if ((fp = fopen(upidfname, "w")) != NULL) {
		fprintf(fp, "%d\n", (int)getpid());
		fclose(fp);
	}
}


/* Updates the icon label and the window title with the given string */

void updateTitles(char *it, char *wt)
{
	char buf[32];
	Display *disp = fl_get_display();

	if (*wt)
		sprintf(buf, "X-ISP   %s", wt);
	else
		strcpy(buf, "X-ISP");
	XStoreName(disp, topWin, buf);
	XSetIconName(disp, topWin, it);
}

/* Clips the given string so that it fits in the horizontal size
   specified; appends '...' as GUI feedback. Returns pointer to
   statically allocated result string, or the input string if no
   clipping was necessary. Note the maximum internal string size! */

#define CLIPSTRSIZE 128

char *clipStr(char *msg, int size, int style, unsigned int hsize)
{
	static char res[CLIPSTRSIZE+4];
	unsigned int len = strlen(msg), swidth, sheight;

	fl_get_string_dimension(style, size, msg, len, &swidth, &sheight);
	if (swidth <= hsize)
		return msg;
	strncpy(res, msg, CLIPSTRSIZE);
	strcat(res, "...");
	len += 3;
	do {
#ifdef SUNOS41x
		bcopy(&res[len-3], &res[len-5], 4);
#else
		memmove(&res[len-5], &res[len-3], 4);
#endif
		len = strlen(res);
		fl_get_string_dimension(style, size, msg, len, &swidth, &sheight);
	}
	while (swidth > hsize && len > 3);
	return res;
}


/* Pops-up a Yes/No dialog box, with the given input
   string message, and returns 1 for "Yes" or 0 for "No" */

#define max(a,b) (((a)>(b))?(a):(b))

int actionVerify(char *msg, int big)
{
	FL_OBJECT *ans;
	int width, height, diff, style, size;

	if (big) {
		style = FL_NORMAL_STYLE + FL_SHADOW_STYLE;
		size = FL_MEDIUM_SIZE;
	}
	else {
		style = FL_NORMAL_STYLE + FL_SHADOW_STYLE;
		size = FL_NORMAL_SIZE;
	}
	fl_set_object_lstyle(fd_actionVerify->actionVerifyStr, style);
	fl_set_object_lsize(fd_actionVerify->actionVerifyStr, size);
	fl_get_string_dimension(style, size, msg, strlen(msg), &width, &height);
	fl_set_form_size(fd_actionVerify->actionVerify,
					 max(300, width+64), height+84);
	diff = (max(300, width+64) - (width+64)) / 3;
	fl_set_object_size(fd_actionVerify->actionVerifyStr,
					   width+10, height+13);
	fl_set_object_position(fd_actionVerify->actionVerifyStr, diff+37,16);
	fl_set_object_label(fd_actionVerify->actionVerifyStr, msg);
	fl_show_form(fd_actionVerify->actionVerify, FL_PLACE_MOUSE,
			 	FL_TRANSIENT, "Action Verification");
	while (ans = fl_do_forms(),
		   ans != fd_actionVerify->actionVerifyYes &&
		   ans != fd_actionVerify->actionVerifyNo);
	fl_hide_form(fd_actionVerify->actionVerify);
	if (ans == fd_actionVerify->actionVerifyYes)
		return 1;
	else
		return 0;
}


/* Pops-up an alert message box, with the given input
   string, and returns when "Dismiss" is pressed */

void alertMessage(char *title, int hint, int big, char *msg)
{
	FL_OBJECT *ans;
	int width, height, diff, style, size;

	if (big) {
		style = FL_NORMAL_STYLE + FL_SHADOW_STYLE;
		size = FL_MEDIUM_SIZE;
	}
	else {
		style = FL_NORMAL_STYLE + FL_SHADOW_STYLE;
		size = FL_NORMAL_SIZE;
	}
	fl_set_object_lstyle(fd_alertMessage->alertMessageStr, style);
	fl_set_object_lsize(fd_alertMessage->alertMessageStr, size);
	fl_get_string_dimension(style, size, msg, strlen(msg), &width, &height);
	fl_set_form_size(fd_alertMessage->alertMessage,
					 max(320,width+64), height+66);
	diff = (max(320, width+64) - (width+64)) / 3;
	fl_set_object_size(fd_alertMessage->alertMessageStr,
					   width+10, height+13);
	fl_set_object_position(fd_alertMessage->alertMessageStr, diff+37,11);
	fl_set_object_label(fd_alertMessage->alertMessageStr, msg);
	if (hint)
		fl_show_object(fd_alertMessage->noMoreHints);
	else
		fl_hide_object(fd_alertMessage->noMoreHints);
	fl_show_form(fd_alertMessage->alertMessage, FL_PLACE_MOUSE,
			 	FL_TRANSIENT, title);
	while (ans = fl_do_forms(),
		   ans != fd_alertMessage->alertMessageDismiss) {
		if (ans == fd_alertMessage->noMoreHints) {
			nohints = !nohints;
			if (nohints) {
				fl_set_menu(fd_topFrame->fileMenu,
							"Options . . .|Enable hints|Exit");
				global.logOpts &= ~LOG_HINTS;
			}
			else {
				fl_set_menu(fd_topFrame->fileMenu,
							"Options . . .|Disable hints|Exit");
				global.logOpts |= LOG_HINTS;
			}
			writeXisprc(rcfname, xispOptions, &global);
		}
	}
	fl_hide_form(fd_alertMessage->alertMessage);
}


/* Adjusts PAP option flag and activation status of the corresponding button
   in the Account information form, according to the available pppd version */

void adjustPAPCap(void)
{
	char msg[512] = {0};

	if (pppdVersion().major < 0) {
		sprintf(msg, "Unknown pppd version:\n"
				"unexpected behavior might occur!");
		alertMessage("xISP: adjustPAPCap()",0,0,msg);
		return;
	}
	if (pppdVersion().major < 2) {					/* version number must */
		sprintf(msg, "Unknown pppd version "		/* be greater than 1 */
				"(v%d.x):\nunexpected "
				"behavior might occur!",
				pppdVersion().major);
		alertMessage("xISP: adjustPAPCap()",0,0,msg);
		return;
	}
	if (pppdVersion().major > 2 ||					/* check pppd version */
		pppdVersion().minor > 2) {
		if (p_xisprc->operOpts & PAP_LOGIN)			/* if > v2.2, disable */
			p_xisprc->operOpts &= ~PAP_LOGIN;		/* PAP as it depends on */
		fl_deactivate_object(						/* the +ua option */
			fd_accountInfo->authPAPButton);
		fl_set_object_lcol(
			fd_accountInfo->authPAPButton,
			FL_INACTIVE);
	}
	else {
		fl_activate_object(
			fd_accountInfo->authPAPButton);
		fl_set_object_lcol(
			fd_accountInfo->authPAPButton,
			FL_WHITE);
	}
}


/* Adjusts Auto DNS option flag and activation status of the corresponding
   button in the TCP/IP information form, according to the available pppd
   version */

void adjustAutoDNSCap(void)
{
	char msg[512] = {0};

	if (pppdVersion().major < 0) {
		sprintf(msg, "Unknown pppd version:\n"
				"unexpected behavior might occur!");
		alertMessage("xISP: adjustAutoDNSCap()",0,0,msg);
		return;
	}
	if (pppdVersion().major < 2) {					/* version number must */
		sprintf(msg, "Unknown pppd version "		/* be greater than 1 */
				"(v%d.x):\nunexpected "
				"behavior might occur!",
				pppdVersion().major);
		alertMessage("xISP: adjustAutoDNSCap()",0,0,msg);
		return;
	}
	if (pppdVersion().major < 2 ||
		(pppdVersion().major == 2 && pppdVersion().minor < 3) ||
		(pppdVersion().major == 2 && pppdVersion().minor == 3 &&
		 pppdVersion().plevel < 9)
       )
    {
		if (p_xisprc->operOpts & AUTO_DNS)			/* if < v2.3.9, disable */
			p_xisprc->operOpts &= ~AUTO_DNS;		/* auto-DNS capability */
		fl_deactivate_object(
			fd_tcpipInfo->DNSAutoButton);
		fl_set_object_lcol(
			fd_tcpipInfo->DNSAutoButton,
			FL_INACTIVE);
	}
	else {
		fl_activate_object(
			fd_tcpipInfo->DNSAutoButton);
		fl_set_object_lcol(
			fd_tcpipInfo->DNSAutoButton,
			FL_WHITE);
	}
}


/* Adjusts compression option flag and activation status of the
   corresponding button in the Communication information form,
   according to the available pppd version */

void adjustCompCap(void)
{
	char msg[512] = {0};

	if (pppdVersion().major < 0) {
		sprintf(msg, "Unknown pppd version:\n"
				"unexpected behavior might occur!");
		alertMessage("xISP: adjustCompCap()",0,0,msg);
		return;
	}
	if (pppdVersion().major < 2) {					/* version number must */
		sprintf(msg, "Unknown pppd version "		/* be greater than 1 */
				"(v%d.x):\nunexpected "
				"behavior might occur!",
				pppdVersion().major);
		alertMessage("xISP: adjustCompCap()",0,0,msg);
		return;
	}
	if (pppdVersion().major < 2 ||
		(pppdVersion().major == 2 && pppdVersion().minor < 3)
       )
	{
		if (p_xisprc->operOpts & DEFL_COMPRESS)		/* if < v2.3.x, disable */
			p_xisprc->operOpts &= ~DEFL_COMPRESS;	/* "deflate" option */
		fl_deactivate_object(
			fd_commInfo->SWCDeflateButton);
		fl_set_object_lcol(
			fd_commInfo->SWCDeflateButton,
			FL_INACTIVE);
	}
	else {
		fl_activate_object(
			fd_commInfo->SWCDeflateButton);
		fl_set_object_lcol(
			fd_commInfo->SWCDeflateButton,
			FL_WHITE);
	}
}


/* Runs through all ISPs in the ISP database and makes sure that none uses
   a non-existent PTT. If any such are found, their PTT entries are changed
   to point to the last available PTT in the PTT database. The check includes
   the global section for the variable that holds the PTT used for calculating
   the cost totals, as well as the PTT zone selected for each ISP entry */

void checkUpdatePTTs()
{
	int i;

	global.costPTT = (global.costPTT < global.numPTTs) ?
						global.costPTT : global.numPTTs-1;
	for (i=0; i<global.numISPs; i++) {
		if (xispOptions[i+1].ispPTT >= global.numPTTs) {
			xispOptions[i+1].ispPTT = global.numPTTs-1;
			xispOptions[i+1].ispZone = 0;
		}
	}
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                  XPM animation routines and callbacks                   |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

#ifdef XPMANIMATE
void updateAnim(int run)
{
	Pixmap id;

	if (run) {									/* if animation is running */
		id = online[frame];						/* use current frame */
		if (minimized)							/* if minimized */
			fl_winicon(topWin, id, amask);		/* set the window icon */
		else									/* if maximized */
			fl_set_pixmap_pixmap(				/* set the master form icon */
				fd_topFrame->topIcon, id,
				amask);
	}
	else {
		id = offline;							/* otherwise, reset icons */
		fl_winicon(topWin, id, amask);			/* for the main form */
		fl_set_pixmap_pixmap(fd_topFrame->		/* and the minimized window */
			topIcon, id, amask);
		frame = 0;								/* reset current frame */
	}
}

void prepAnimPixmaps(void)
{
	unsigned int width = 50, height = 50,
			 hotx = 1, hoty = 1;

	offline =	fl_create_from_pixmapdata(fl_default_window(), xisp_xpm,
					&width, &height, &amask, &hotx, &hoty, FL_INDIANRED);
	online[0] = fl_create_from_pixmapdata(fl_default_window(), online1_xpm,
					&width, &height, &amask, &hotx, &hoty, FL_INDIANRED);
	online[1] = fl_create_from_pixmapdata(fl_default_window(), online2_xpm,
					&width, &height, &amask, &hotx, &hoty, FL_INDIANRED);
	online[2] = fl_create_from_pixmapdata(fl_default_window(), online3_xpm,
					&width, &height, &amask, &hotx, &hoty, FL_INDIANRED);
	online[3] = fl_create_from_pixmapdata(fl_default_window(), online4_xpm,
					&width, &height, &amask, &hotx, &hoty, FL_INDIANRED);
	online[4] = fl_create_from_pixmapdata(fl_default_window(), online5_xpm,
					&width, &height, &amask, &hotx, &hoty, FL_INDIANRED);
	online[5] = fl_create_from_pixmapdata(fl_default_window(), online6_xpm,
					&width, &height, &amask, &hotx, &hoty, FL_INDIANRED);
	online[6] = fl_create_from_pixmapdata(fl_default_window(), online7_xpm,
					&width, &height, &amask, &hotx, &hoty, FL_INDIANRED);
	online[7] = fl_create_from_pixmapdata(fl_default_window(), online8_xpm,
					&width, &height, &amask, &hotx, &hoty, FL_INDIANRED);
	online[8] = fl_create_from_pixmapdata(fl_default_window(), online9_xpm,
					&width, &height, &amask, &hotx, &hoty, FL_INDIANRED);
	online[9] = fl_create_from_pixmapdata(fl_default_window(), online10_xpm,
					&width, &height, &amask, &hotx, &hoty, FL_INDIANRED);
	online[10] = fl_create_from_pixmapdata(fl_default_window(), online11_xpm,
					&width, &height, &amask, &hotx, &hoty, FL_INDIANRED);
	online[11] = fl_create_from_pixmapdata(fl_default_window(), online12_xpm,
					&width, &height, &amask, &hotx, &hoty, FL_INDIANRED);
}

void doXPMUpdate(FL_OBJECT *obj, long param)
{
	unsigned newnetpkts;

	frame = (frame + 1) % 12;					/* increment frame counter */
	if (frame) {								/* if not on frame 0 */
		if (transfer)							/* just spin the animation */
			updateAnim(1);
	}
	else {										/* if on frame 0 */
		newnetpkts = pppPkts(pppIF, NULL,NULL);	/* check for new packets */
		if (newnetpkts != netpkts) {			/* if new packets exist */
			netpkts = newnetpkts;				/* start next animation cycle */
			transfer = 1;
			updateAnim(1);
		}
		else transfer = 0;
	}
	fl_set_timer(xpmtimer, XPM_INTERVAL);		/* restart animation timer */
}
#endif


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |          Routine adjusting possible actions according to state          |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void possibleActions(int state)
{
	progState = state;
	switch (state) {

		case DISCONNECTED: {
			fl_activate_object(fd_topFrame->conButton);
			fl_set_object_lcol(fd_topFrame->conButton,FL_LCOL);
			fl_deactivate_object(fd_topFrame->intButton);
			fl_set_object_lcol(fd_topFrame->intButton,FL_INACTIVE);
			fl_deactivate_object(fd_topFrame->disButton);
			fl_set_object_lcol(fd_topFrame->disButton,FL_INACTIVE);
			fl_activate_object(fd_topFrame->fileMenu);
			fl_set_object_lcol(fd_topFrame->fileMenu,FL_LCOL);
			fl_activate_object(fd_topFrame->logMenu);
			fl_set_object_lcol(fd_topFrame->logMenu,FL_LCOL);
			fl_activate_object(fd_topFrame->hlpMenu);
			fl_set_object_lcol(fd_topFrame->hlpMenu,FL_LCOL);
			break;
		}

		case DIALING: {
			fl_deactivate_object(fd_topFrame->conButton);
			fl_set_object_lcol(fd_topFrame->conButton,FL_INACTIVE);
			fl_activate_object(fd_topFrame->intButton);
			fl_set_object_lcol(fd_topFrame->intButton,FL_LCOL);
			fl_deactivate_object(fd_topFrame->disButton);
			fl_set_object_lcol(fd_topFrame->disButton,FL_INACTIVE);
			fl_deactivate_object(fd_topFrame->fileMenu);
			fl_set_object_lcol(fd_topFrame->fileMenu,FL_INACTIVE);
			fl_deactivate_object(fd_topFrame->logMenu);
			fl_set_object_lcol(fd_topFrame->logMenu,FL_INACTIVE);
			fl_deactivate_object(fd_topFrame->hlpMenu);
			fl_set_object_lcol(fd_topFrame->hlpMenu,FL_INACTIVE);
			break;
		}

		case CONNECTED: {
			fl_deactivate_object(fd_topFrame->conButton);
			fl_set_object_lcol(fd_topFrame->conButton,FL_INACTIVE);
			fl_deactivate_object(fd_topFrame->intButton);
			fl_set_object_lcol(fd_topFrame->intButton,FL_INACTIVE);
			fl_activate_object(fd_topFrame->disButton);
			fl_set_object_lcol(fd_topFrame->disButton,FL_LCOL);
			fl_activate_object(fd_topFrame->fileMenu);
			fl_set_object_lcol(fd_topFrame->fileMenu,FL_LCOL);
			fl_deactivate_object(fd_topFrame->logMenu);
			fl_set_object_lcol(fd_topFrame->logMenu,FL_INACTIVE);
			fl_activate_object(fd_topFrame->hlpMenu);
			fl_set_object_lcol(fd_topFrame->hlpMenu,FL_LCOL);
			break;
		}

		default: break;
	}
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                      Dialer output parsing routines                     |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

/* IMPORTANT!: Note the maximum field length defined by
   MAXLEN_FIELD, used by the generic parser doutWait() */

#define MAXLEN_FIELD 64

char *doutWait(char *buf, char *field, char *ecs)
{
	char *p1, *p2;
	static int next = 0, nc;
	static char rstr[MAXLEN_FIELD];
	char msg[512] = {0};

	if (next) {									/* processing field string? */
		p2 = strfstr(buf, ecs);					/* yup, find ending char */
		if (p2 != NULL) {						/* if found */
			next = 0;							/* indicate processing done */
			strncpy(&rstr[nc], buf,				/* save it locally */
					(int)(p2-buf));
			rstr[nc+(int)(p2-buf)] = 0;			/* terminate it */
			return rstr;						/* and return its address */
		}
		else {
			sprintf(msg, "Invalid output"		/* ec not found: dialer */
					"from dialer process!");	/* output is screwed, or */
			alertMessage("xISP: doutWait()",	/* something is very wrong */
						 0, 0, msg);
			return NULL;
		}
	}

	p1 = strstr(buf, field);					/* wanted field in there? */
	if (p1 != NULL) {							/* yup, parse to ending ec */
		p2 = strfstr(p1, ecs);
		if (p2 != NULL) {						/* if found */
			nc = (int)(p2-p1)-strlen(field);	/* string has so many chars */
			strncpy(rstr, p1+strlen(field),		/* save it locally */
					nc);
			rstr[nc] = 0;						/* terminate it */
			return rstr;						/* and return its address */
		}
		else {									/* oops, string ends */
			strcpy(rstr, p1+strlen(field));		/* copy what's there */
			next = 1;							/* take note for next call */
			nc = strlen(rstr);
			return NULL;
		}
	}
	return NULL;
}

/* Parse dialer output for its PID number */

int doutPID(char *buf)
{
	char pids[16], *p;

	p = doutWait(buf, "PID=", ".");				/* look for PID=?????. */
	if (p != NULL) {							/* found it? */
		strcpy(pids, p);						/* yup, copy the string */
		dialerPID = atoi(pids);					/* convert and save it */
		possibleActions(DIALING);				/* mark possible actions */
		return 1;								/* indicate found */
	}
	else										/* nope */
		return 0;								/* indicate not found */
}

/* Parse dialer output for modem connect string. Note that even
   if the modems are found to be connected, when call-back is
   enabled, xisp ignores the first modem connect string. However,
   it does keep track of the time the call-back connection was
   initiated and estimates an elapsed time for later call-cost
   calculation/logging */

int doutCONN(char *buf)
{
	char *p, *sp, chead[5] = {0}, *ctail;
	unsigned speed;

	strncpy(chead, p_xisprc->modemConnect, 4);	/* retrieve "connect" string */
	ctail = &(p_xisprc->modemConnect[4]);
	p = doutWait(buf, chead, "/\r\n");			/* look for first 4 chars */
	if (p != NULL) {							/* found them? */
		if (strstr(p, ctail) == NULL)			/* rest of the string OK? */
			return 0;							/* nope, indicate not found */
		if (p_xisprc->operOpts & CALL_BACK) {	/* if call-back is enabled */
			if (firstConn) {					/* and it's 1st connection */
				upTime = time(NULL);			/* record connection time */
				updateLogs(1);					/* and update log files */
				firstConn = 0;					/* indicate it's gone by and */
				return 0;						/* wait for 2nd connection */
			}
			else								/* if it's the CB connection */
				CBTime = time(NULL);			/* record CB connection time */
		}
		else {									/* else call-back not enabled */
			upTime = time(NULL);				/* record connection time */
			costCalcReset(p_ptt, upTime);		/* reset calculations module */
		}
		onLineSecs = 0;							/* and counters */
		onLineUnits = 0;
		sp = strchr(p, ' ');					/* continue: look for space */
		if (sp != NULL) {						/* found it? */
			strcpy(speedStr, sp);				/* yes, copy the string */
			speed = atoi(speedStr);				/* convert and save speed */
			if (speed < 1000)					/* and format its string */
				sprintf(connSpeed,"%d",speed);
			else
				sprintf(connSpeed,"%d,%03d",
						speed/1000,speed%1000);
			sprintf(speedStr, "%u", speed);		/* erase leading whitespace */
		}
		else {									/* oops, no speed! */
			strcpy(connSpeed, "- NA -");		/* copy this instead */
			strcpy(speedStr, connSpeed);
		}
		connected = 1;							/* update variables */
		connTime = 0;
		if (global.logOpts & COST_READOUT)		/* cost readout desired? */
			fl_set_object_label(				/* yes, so display */
			 fd_topFrame->logText,				/* appropriately formatted */
				costStr(p_ptt, 0.0));			/* zero cost */
		else
			fl_set_object_label(				/* nope, display the */
			 fd_topFrame->logText,EMPTY_TIME);	/* reset-timer string */
		strcpy(connIP, EMPTY_IP);
		updateStat(linkOK);						/* update indicators */
		fl_set_timer(ctimer, CT_INTERVAL);		/* start connection timer */
		if (p_xisprc->operOpts & CONNECT_BELL)	/* ring bell if desired */
			XBell(fl_get_display(), 50);
#ifdef XPMANIMATE
		netpkts = 0;							/* initialize net packets */
		fl_set_pixmap_pixmap(					/* animation pixmap */
			fd_topFrame->topIcon,
			online[11], amask);
		fl_winicon(topWin, online[11], amask);	/* window icon, and */
		fl_set_timer(xpmtimer, XPM_INTERVAL);	/* start xpm-animation timer */
#endif
		possibleActions(CONNECTED);				/* mark possible actions */
		return 1;								/* indicate found */
	}
	else										/* nope */
		return 0;								/* indicate not found */
}

/* Parse dialer output for TIMEOUT string from xispdial */

int doutTIMEOUT(char *buf)
{
	char *p;

	p = doutWait(buf, "TIME", "\n");
	if (p != NULL) {
		if (connected && firstConn)				/* if it was 1st connection */
			updateLogs(connected);				/* update logs for upTime */
		connected = 0;							/* invalidate connection */
		fl_set_timer(ctimer, 0.0);				/* stop connection timer */
		updateTitles("X-ISP", "");				/* reset icon&window titles */
#ifdef XPMANIMATE
		updateAnim(0);							/* and animation pixmap */		
		fl_set_timer(xpmtimer, 0.0);			/* disable animation timer */
#endif
		strcpy(connSpeed, EMPTY_SPEED);			/* reset speed indicator */
		updateStat(linkOK);						/* update display */
		if (firstConn) {						/* if it was 1st connection */
			downTime = time(NULL);				/* record time */
			totalTime += onLineSecs;			/* calculate total seconds */
			if (p_ptt->attribs &				/* depending on the PTT */
				(PTT_PER_MINUTE|PTT_PER_SECS))	/* charging scheme */
				totalCost += onLineCost;		/* update the total cost */
			else
				totalUnits += onLineUnits;		/* or the total unit value */
			updateLogs(connected);				/* update log files */
		}
		firstConn = 1;							/* reset this for next time */
		possibleActions(DIALING);				/* mark possible actions */
		return 1;								/* indicate found */
	}
	else
		return 0;
}

/* Parse dialer output for ABORT string from xispterm */

int doutABORT(char *buf)
{
	char *p;

	p = doutWait(buf, "ABOR", "\n");
	if (p != NULL) {
		userDiscon = 1;							/* it's a user disconnect */
		if (connected && firstConn)				/* if it was 1st connection */
			updateLogs(connected);				/* update logs for upTime */
		connected = 0;							/* invalidate connection */
		fl_set_timer(ctimer, 0.0);				/* stop connection timer */
		updateTitles("X-ISP", "");				/* reset icon&window titles */
#ifdef XPMANIMATE
		updateAnim(0);							/* and animation pixmap */		
		fl_set_timer(xpmtimer, 0.0);			/* disable animation timer */
#endif
		strcpy(connSpeed, EMPTY_SPEED);			/* reset speed indicator */
		updateStat(linkOK);						/* update display */
		if (firstConn) {						/* if it was 1st connection */
			downTime = time(NULL);				/* record time */
			totalTime += onLineSecs;			/* calculate total seconds */
			if (p_ptt->attribs &				/* if the selected PTT */
				(PTT_PER_MINUTE|PTT_PER_SECS))	/* charges by time */
				totalCost += onLineCost;		/* update the total cost */
			else								/* else update the */
				totalUnits += onLineUnits;		/* total unit value instead */
			updateLogs(connected);				/* update log files */
		}
		firstConn = 1;							/* reset this for next time */
		possibleActions(DIALING);				/* mark possible actions */
		return 1;								/* indicate found */
	}
	else
		return 0;
}

/* Parse dialer output for NT-RAS string from xispterm. Note that nothing
   changes in xisp's state: it starts pppd a second time and then waits
   for the call-back connection */

int doutNTRAS(char *buf)
{
	char *p;

	p = doutWait(buf, "NT-RAS", "\n");
	if (p != NULL) {
		fl_call_object_callback(				/* "push Connect" again */
			fd_topFrame->conButton);
		return 1;								/* indicate found */
	}
	else
		return 0;
}

/* Parse dialer output for strings changing our state */

void doutParse(char *buf)
{
	static char line[MAXBUF_CHILD+1] = {0}, *pline = line;
	char *p;

	if (MAXBUF_CHILD-(unsigned int)(pline-line)	/* take care of possible */
		< strlen(buf)) pline = line;			/* line buffer overflow */
	for (p=buf; *p; ) {							/* while chars are available */
		for (; *p && *p!='\n' && *p!='\r';		/* make local copy of buf */
			 p++, pline++) *pline = *p;			/* in line-buffered fashion */
		if (*p) {								/* end of line found? */
			*pline++ = *p;						/* yes, add special char */
			*pline = 0;							/* and terminate the line */
			if (! dialerPID) doutPID(line);		/* parse for dialer PID */
			if (! connected) {					/* skip "NO CARRIER" message */
				if (doutWait(line, "NO CARR",	/* in case our modem connect */
							 "\n") == NULL)		/* string is "CARRIER xxxxx" */
					doutCONN(line);				/* parse for CONNECTED info */
				if (p_xisprc->operOpts &		/* if NT RAS call-back on */
					(CALL_BACK|CB_NT_RAS)) {	/* parse also for second */
					doutNTRAS(line);			/* pppd invocation trigger */
					doutTIMEOUT(line);			/* and for dialer TIMEOUT */
				}
			}
			else {								/* if we are connected */
				doutTIMEOUT(line);				/* parse for dialer TIMEOUT */
				if (p_xisprc->operOpts &		/* if manual login for dial- */
					(MANUAL_LOGIN|CBMAN_LOGIN))	/* in/call-back is selected, */
					doutABORT(line);			/* parse also for ABORT */
			}
			pline = line;						/* start new line */
			++p;								/* from next input char */
		}
	}
}

/* Check logging options and if logging is enabled, backup current log
   files, and reset logs; also check log file names and adjust if needed */

void resetLogs(void)
{
	char msg[512] = {0};
	void alertMessage();

	initLogFnames(&global);						/* adjust log file names */
	if (! (global.logOpts & LOG_NONE)) {		/* only if logging active */
		strcpy(bkupfname, costfname);			/* backup current logs */
		strcat(bkupfname, ".bak");
		rename(costfname, bkupfname);
		strcpy(bkupfname, logfname);
		strcat(bkupfname, ".bak");
		rename(logfname, bkupfname);
		restartXispCost(&totalTime,&totalCost);	/* create new log file */
#ifdef RUNASEUID
		if (! nohints &&
			(!xisp_euidaccess(costfname, F_OK) ||
			 !xisp_euidaccess(logfname, F_OK)))
#else
		if (! nohints &&
			(!access(costfname, F_OK) || !access(logfname, F_OK)))
#endif
		{
			sprintf(msg, "Your cost and connection time logging files\n"
					"%s and/or\n%s\n"
					"have been saved as *.bak and new log files for\n"
					"the current logging period have been created.",
					costfname, logfname);
			alertMessage("Logging hint", 1, 0, msg);
		}
	}
}

/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                  Connection timer/PTT-charges callback                  |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void doCTupdate(FL_OBJECT *obj, long param)
{
	char tout[16], *p;
	struct tm *ptm;

	if (p_xisprc->operOpts & CALL_BACK)			/* if call-back is enabled */
		onLineSecs = time(NULL) - CBTime;		/* use CB connection up time */
	else										/* else use 1st connection */
		onLineSecs = time(NULL) - upTime;		/* uptime for timer update */
	connTime += (int)CT_INTERVAL;				/* increment "rough" timer */
	if (!(global.logOpts & LOG_NONE) ||			/* logging desired */
		(global.logOpts & COST_READOUT)) {		/* or cost readout selected? */
		if (firstConn)							/* yes, if 1st connection */
			onLineCost = callCost(p_ptt,		/* calculate current cost */
				onLineSecs, &onLineUnits);
		else									/* if 2nd connection in CB */
			onLineCost = 0.0;					/* mode, it's for free :) */
	}
	if (global.logOpts & COST_READOUT) {		/* cost readout selected? */
		p = costStr(p_ptt, onLineCost);			/* format it in a string */
		fl_set_object_label(					/* print it on main form */
			fd_topFrame->logText, p);
	}
	else {										/* no, we want time readout */
		ptm = gmtime(&connTime);				/* break down current time */
		p = tout;
		if (ptm->tm_yday > 0) {					/* take days into account */
#ifdef BROKEN_SPRINTF
			sprintf(p, "%dd:",					/* if > 0, prepend number of */
					ptm->tm_yday);				/* days to the time string */
			p += strlen(p);
#else
			p += sprintf(p, "%dd:",
						 ptm->tm_yday);
#endif
		}
		strftime(p, 16, "%H:%M:%S", ptm);		/* format time as HH:MM:SS */
		fl_set_object_label(					/* print time only (not */
			fd_topFrame->logText, p);			/* days) on main form */
		p = tout;
		if (ptm->tm_yday < 1 &&					/* skip first hour digit for */
			ptm->tm_hour < 10)					/* icon label, if possible */
			++p;
	}
	if (linkOK)
		updateTitles(p, p);						/* update icon&window titles */
	if (connected)
		fl_set_timer(ctimer, CT_INTERVAL);		/* and restart timer */
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                          Link status callback                           |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void doLPupdate(FL_OBJECT *obj, long param)
{
	char *p;
	int linkWasUp = linkOK;						/* previous link status */

	if (*pppIF == 0)							/* get PPP IF-name if we */
		getIFname(pppIF);						/* don't know it already */
	if (pppdPID && *pppIF &&					/* pppd running and link OK? */
		(p=pppAddr(pppIF))!=NULL)
	{
		if (!linkOK) {							/* yes, just now? */
			strcpy(connIP, p);					/* yup, copy new IP address */
			if (firstConn)						/* if it was 1st connection */
				updateLogs(connected);			/* update log files */
			linkOK = 1;							/* and mark link as OK */
			updateStat(linkOK);					/* update indicators */
			linkTOCounter = 0;					/* init for next time */
			pppdPID = getPppdPID(devName);		/* get PID of detached pppd */
			if ((xispUDPID = xispUD(1,0))) {	/* start .xisp-up script */
				xispUDON = UD_WAIT/BU_INTERVAL;	/* mark max iterations, and */
				fl_set_timer(btimer,			/* enable output processing */
							 BU_INTERVAL);
			}
		}
		fl_set_timer(ltimer, SP_INTERVAL);		/* restart timer */
	}
	else {										/* pppd or link problem */
		if (pppdPID && (linkTOCounter--)) {		/* keep going until timeout */
			fl_set_timer(ltimer, SP_INTERVAL);	/* restart status poll timer */
			if (p_xisprc->operOpts & IP_UPDOWN)	/* if ip-up/-down support on */
				ipUDON = UD_WAIT/BU_INTERVAL;	/* restart ip-up/-down timer */
			xispUDON = UD_WAIT/BU_INTERVAL;		/* timer for .xisp-up/-down */
			fl_set_timer(btimer, BU_INTERVAL);	/* enable output processing */
			return;
		}
		downTime = time(NULL);					/* record time instant */
		if (!linkOK && firstConn)				/* connection 1, IF never up */
			updateLogs(connected);				/* so update logs for upTime */
		bprintf("PPP link is down.\n");			/* nope, print message */
		fl_set_timer(ctimer, 0.0);				/* disable timers */
		fl_set_timer(ltimer, 0.0);
		updateTitles("X-ISP", "");				/* reset icon&window titles */
#ifdef XPMANIMATE
		updateAnim(0);							/* and animation pixmap */
		fl_set_timer(xpmtimer, 0.0);			/* disable animation timer */
#endif
		if (pppdPID)							/* if not dead already */
			kill(pppdPID, SIGINT);				/* terminate pppd process */
		pppdPID = 0;							/* indicate that it's gone */
		if (firstConn) {						/* if it was 1st connection */
			totalTime += onLineSecs;			/* calculate total seconds */
			if (p_ptt->attribs &				/* if the selected PTT */
				(PTT_PER_MINUTE|PTT_PER_SECS))	/* charges by time */
				totalCost += onLineCost;		/* update the total cost */
			else								/* else update the */
				totalUnits += onLineUnits;		/* total unit value instead */
			updateLogs(0);						/* update log files */
		}
		connected = 0;							/* update variables */
		firstConn = 1;
		connTime = 0;
		linkOK = 0;
		*pppIF = 0;								/* mark IF name unknown */
		strcpy(connSpeed, EMPTY_SPEED);
		strcpy(connIP, EMPTY_IP);
		updateStat(linkOK);						/* update indicators */
		possibleActions(DISCONNECTED);			/* mark possible actions */
		if (p_xisprc->operOpts & IP_UPDOWN)		/* ip-up/-down support? */
			ipUDON = UD_WAIT/BU_INTERVAL;		/* yup, mark ip-up active */
		strcpy(ipWhich, "ip-down");				/* set user script name */
		if (linkWasUp&&(xispUDPID=xispUD(0,1)))	/* start .xisp-down */
			xispUDON = UD_WAIT/BU_INTERVAL;		/* and mark iterations */
		if (ipUDON || xispUDON)					/* if either is active */
			fl_set_timer(btimer, BU_INTERVAL);	/* start browser timer */
		if (!userDiscon &&						/* if not user disconnect */
			(p_xisprc->operOpts &				/* and auto-redial set, then */
			 AUTO_REDIAL)) {
			fl_call_object_callback(			/* "push Connect" again */
				fd_topFrame->conButton);
		}
		userDiscon = 0;							/* reset indicator */
	}
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |  Callback and support routines for browser updates from dialer output   |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

#define MAXLEN_IOTBUF 11						/* I/O trigger buffer size */
char IOTBuf[MAXLEN_IOTBUF+1] = {0},
	 *pIOTB = IOTBuf;

void browserIOTrigger(int fd, void *data)		/* pipe I/O-triggered input */
{
	int space, br, dump = 0;
	char *pc;
	void doBUpdate();

	space = MAXLEN_IOTBUF-(int)(pIOTB-IOTBuf);	/* calculate space left, and */
	br = read(fd, pIOTB, space);				/* read max that many bytes */
	pIOTB += br;								/* increment pointer */
	for (pc=IOTBuf; !dump && pc<pIOTB; pc++)	/* scan for CR or LF */
		if (*pc=='\n' || *pc=='\r') dump = 1;	/* dump buffer if found */
	dump |= (pIOTB == MAXLEN_IOTBUF + IOTBuf);	/* or dump when buffer full */
	if (dump) {									/* dump condition met? */
		fl_set_timer(btimer, BU_INTERVAL);		/* yes, restart timer */
		fl_call_object_callback(btimer);		/* call update routine */
		pIOTB = IOTBuf;							/* and start all over again */
	}
}

int IOIdleCallback(XEvent *xev, void *data)		/* pipe input checking proc */
{
	fd_set rfds;
	struct timeval tv;
	int stat;

	FD_ZERO(&rfds);								/* initialize the fs set */
	FD_SET(dialerFD, &rfds);					/* stick the dialerFD in */
	tv.tv_sec = tv.tv_usec = 0;					/* return with no delay */
	stat = select(dialerFD+1, &rfds,			/* check the descriptor */
				  NULL, NULL, &tv);				/* for any input */
	if (stat < 0) {
		if (errno != EINTR)						/* non-blocking signal? */
			doErr("IOIdleCallback: select");	/* no, abort with diagnostic */
		return 0;								/* yes, do nothing */
	}
	if (stat)									/* data exists for reading? */
		browserIOTrigger(dialerFD, NULL);		/* yes, call the I/O proc */
	return 0;
}

int readT(int fd, void *buf, size_t cnt)		/* I/O-trigger aware read */
{												/* note that cnt is ignored */
	int br, bp = pIOTB - IOTBuf;				/* when IOTbuf has data, as */
												/* cnt always >MAXLEN_IOTBUF */
	if (bp > 0) {								/* are bytes from I/O */
		br = bp;								/* trigger still pending? */
		memcpy(buf, IOTBuf, br);				/* yes, put them in buffer */
		pIOTB = IOTBuf;							/* mark trigger buffer empty */
	}
	else
		br = read(dialerFD, buf, cnt);			/* else read from descriptor */
	return br;									/* return bytes read */
}

int pppdCleanup()								/* collect pppd status etc. */
{
	int i, stat;
	char msg[512] = {0};

	i = waitpid(pppdPID, &stat, WNOHANG);		/* get pppd return status */
	if (i < 0)									/* ret < 0 : wait failed */
		doErr("pppdCleanup: waitpid");			/* abort with diagnostic */
	if (i == 0)									/* if pppd hasn't forked yet */
		return -1;								/* try again next time */
	pppdStat = 1;								/* indicate status collected */
	if (stat) {									/* if pppd didn't fare well */
		if (WIFSIGNALED(stat)) {				/* print messages */
			sprintf(msg, "pppd received"		/* for untrapped signal */
					" signal %d!\n",
					WTERMSIG(stat));
			alertMessage("Pppd signal",0,0,msg);
		}
		if (WIFEXITED(stat)) {
			sprintf(msg, "Pppd exited with"		/* and for abnormal exit */
				" status %d. This means"
				" that\n%s", WIFEXITED(stat),
				pppdMsg[WIFEXITED(stat)]);
			alertMessage("Pppd status",0,0,msg);
			bprintf("\nAction terminated.\n");
		}
		if (dialerPID)							/* dialer still there? */
			kill(dialerPID, SIGTERM);			/* yup, terminate it */
		pppdPID = getPppdPID(devName);			/* PID of spawned pppd */
		if (pppdPID)							/* pppd still there? */
			kill(pppdPID, SIGINT);				/* yup, terminate it */
		pppdPID = 0;							/* indicate that it's gone */
		dialerON = 0;							/* indicate dial aborted */
		dialerPID = 0;							/* and no dialer process */
		fl_activate_object(						/* finally reactivate the */
			fd_topFrame->conButton);			/* connection button */
		fl_set_object_lcol(
			fd_topFrame->conButton, FL_LCOL);
	}
	else
		updateStat(linkOK);						/* all OK, update display */
	if (p_xisprc->operOpts & PAP_LOGIN)			/* PAP authentication used? */
		unlink(papfname);						/* yes, delete PAP data file */
	return WEXITSTATUS(stat);					/* return exit status */
}

void doBUpdate(FL_OBJECT *obj, long param)
{
	int br, stat;
	char buf[MAXBUF_CHILD+MAXLEN_IOTBUF+2], *p;
	static int dialWait = DIAL_WAIT/BU_INTERVAL;
	static int gotIP = 0;

	if (dialerON) {								/* if dial in progress */
		br = readT(dialerFD, buf,MAXBUF_CHILD);	/* read a buffer full */
		if (br > 0) {							/* if read OK */
			buf[br] = 0;						/* indicate string end */
			bprintf("%s", buf);					/* and print it */
			doutParse(buf);						/* parse dialer output */
			fl_set_timer(btimer, BU_INTERVAL);	/* restart timer */
			if (dialWait) dialWait = 0;			/* no more waiting */
		}
		else if (br < 0) {						/* read failed */
#ifdef SUNOS41x
			if (errno != EWOULDBLOCK)			/* pipe output unavailable? */
#else
			if (errno != EAGAIN)
#endif
				doErr("doBUpdate: read");		/* no, abort with diagnostic */
			fl_set_timer(btimer, BU_INTERVAL);	/* yes, restart timer */
		}
		else {									/* EOF on pipe output */
			if (dialWait) {						/* wait for xispdial */
				if (! pppdStat)					/* only once */
					pppdRet = pppdCleanup();	/* collect pppd's status */
				--dialWait;						/* to open the pipe */
				fl_set_timer(btimer,			/* if it hasn't managed yet */
							 BU_INTERVAL);		/* due to machine load */
				return;
			}
			else if (! (dialerPID | pppdRet)) {	/* dialer has failed but */
				alertMessage("xispdial failed",	/* pppd exit status is 0 */
					0, 0,
					"Failed to start xispdial"
					" for unknown reasons\n"
					"(pppd detached with status"
					" 0). Please look\n"
					"in the system logs (i.e."
					" most probably files\n"
					"/var/log/messages or "
					"/var/adm/messages) for\n"
					"an explanation.");
				bprintf("\nAction terminated.\n");
			}
			fl_set_idle_callback(0, NULL);		/* remove idle callback */
			dialWait = DIAL_WAIT/BU_INTERVAL;	/* reset for next time */
			dialerON = 0;						/* indicate dial done */
			dialerPID = 0;						/* and no dialer process */
			bflush();							/* flush browser output */
			if (connected) {					/* check connection status */
				if (p_xisprc->operOpts &		/* ip-up/-down support on? */
					IP_UPDOWN)
					ipUDON =					/* yup, mark ip-up active */
						UD_WAIT/BU_INTERVAL;	/* with such max iterations */
				strcpy(ipWhich, "ip-up");		/* set user script name */
				fl_set_timer(btimer,			/* restart timer anyway for */
							 BU_INTERVAL);		/* ip-U/D and/or .xisp-U/D */
				linkTOCounter =					/* init link polling timeout */
					ceil(p_xisprc->LCPWait/		/* counter (rounded up) */
						 (double)SP_INTERVAL);
				fl_set_timer(ltimer,			/* and activate link polling */
							 SP_INTERVAL);		/* timer call-back */
			}
			else {								/* if we disconnected */
				possibleActions(DISCONNECTED);	/* mark possible actions */
				updateStat(linkOK);				/* update indicators */
#ifdef XPMANIMATE
				updateAnim(0);					/* and animation pixmap */
				fl_set_timer(xpmtimer, 0.0);	/* disable animation timer */
#endif
			}
		}
		if (! pppdStat)							/* only once */
			pppdRet = pppdCleanup();			/* collect pppd's status */
		return;
	}

	if (ipUDON) {								/* if ip-up/-down active */
		br = readT(dialerFD, buf,MAXBUF_CHILD);	/* read a buffer full */
		if (br > 0) {							/* if read OK */
			buf[br] = 0;						/* indicate string end */
			bprintf("%s", buf);					/* print it */
			fl_set_timer(btimer, BU_INTERVAL);	/* and restart timer */
		}
		else if (br < 0) {						/* read failed */
#ifdef SUNOS41x
			if (errno != EWOULDBLOCK)			/* pipe output unavailable? */
#else
			if (errno != EAGAIN)
#endif
				doErr("doBUpdate: read");		/* no, abort with diagnostic */
			fl_set_timer(btimer, BU_INTERVAL);	/* yes, restart timer */
		}
		else {									/* EOF on pipe output */
			if (! ipUDPID)						/* unless we know it already */
					ipUDPID = getipUDPID();		/* get PID of ip-up/-down */
			if (ipUDPID &&						/* ip-up/-down running? */
				waitpid(- ipUDPID, &stat,		/* yes, get children status */
						WNOHANG) > 0)
				fl_set_timer(btimer,			/* got status: restart timer */
							 BU_INTERVAL);
			else {								/* no ip-up/-down or wait<=0 */
				--ipUDON;						/* count down as we drain */
				if (*pppIF == 0)				/* get PPP IF-name if we */
					getIFname(pppIF);			/* don't know it already */
				if (!gotIP && *pppIF && 		/* if we don't have it yet */
					(p=pppAddr(pppIF))!=NULL)	/* get ppp IF address */
				{
					strcpy(connIP, p);			/* got it, save IP address */
					gotIP = 1;					/* and don't try it again */
					updateStat(gotIP);			/* update indicators */
				}
				if (! ipUDON) {					/* if no more iterations */
					gotIP = 0;					/* reset vars for next time */
					ipUDPID = 0;
				}
				fl_set_timer(btimer,			/* done or not,restart timer */
							 BU_INTERVAL);		/* for enabling next stage */
			}
		}
		return;
	}

	if (xispUDON) {								/* if .xisp-up/-down active */
		br = readT(dialerFD, buf,MAXBUF_CHILD);	/* read a buffer full */
		if (br > 0) {							/* if read OK */
			buf[br] = 0;						/* indicate string end */
			bprintf("%s", buf);					/* print it */
			fl_set_timer(btimer, BU_INTERVAL);	/* and restart timer */
		}
		else if (br < 0) {						/* read failed */
#ifdef SUNOS41x
			if (errno != EWOULDBLOCK)			/* pipe output unavailable? */
#else
			if (errno != EAGAIN)
#endif
				doErr("doBUpdate: read");		/* no, abort with diagnostic */
			fl_set_timer(btimer, BU_INTERVAL);	/* yes, restart timer */
		}
		else {									/* EOF on pipe output */
			if (waitpid(xispUDPID, &stat,		/* if script already done */
						WNOHANG) < 0) {
				if (--xispUDON)					/* count down as we drain */
					fl_set_timer(btimer,		/* and if iterations still */
								 BU_INTERVAL);	/* to go, restart the timer */
			}
			else								/* script either not done */
				fl_set_timer(btimer,			/* yet, or just terminated */
							 BU_INTERVAL);		/* so restart timer */
		}
		return;
	}
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                      Callback routines for topFrame                     |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void doStartupTip(FL_OBJECT *obj, long param)
{
	alertMessage("Startup hint", 1, 0,
	 "xISP was invoked with pop-up hints enabled (this is\n"
	 "the default since version 2.6).  If you don't need on-\n"
	 "line hints, please either use the \"-nohints\" command\n"
	 "line option, or disable them from the \"File\" menu.");
}

void doConnect(FL_OBJECT *obj, long param)
{
	int ret;
	char speed[MAXDIG_BAUDRATE+1] = {0}, lcps[MAXDIG_LCPWAIT+1] = {0},
		 blevel[2*(MAXDIG_BSDCOMP+1)], dlevel[2*(MAXDIG_BSDCOMP+1)],
		 lips[MAXLEN_IP+1] = {0}, rips[MAXLEN_IP+1] = {0},
		 nms[MAXLEN_IP+1] = {0}, lrs[2*MAXLEN_IP+2] = {0},
		 mtus[MAXDIG_MTRU+1] = {0}, mrus[MAXDIG_MTRU+1] = {0},
		 dnsips[MAXLEN_IP+1] = {0}, peerfn[MAXLEN_DEVICE+7+1] = {0},
		 tout[16], *p, msg[512] = {0};
	struct tm *ptm;
#if defined(SUNOS5x) || defined(SVR4)
	struct stat sbuf;
#endif

	if (global.numISPs < 1) {
		XBell(fl_get_display(), 50);
		alertMessage("xISP: doConnect()", 0, 0,
					 "You haven't defined an ISP!");
		return;
	}
	if (p_xisprc->numPhones < 1) {
		XBell(fl_get_display(), 50);
		alertMessage("xISP: doConnect()", 0, 0,
					 "No phone number(s) to dial!");
		return;
	}
#ifdef RUNASEUID
	if (xisp_euidaccess(Pppd, F_OK) == -1) {
#else
	if (access(Pppd, F_OK) == -1) {
#endif
		XBell(fl_get_display(), 50);
		sprintf(msg, "%s: not found!", Pppd);
		alertMessage("xISP: doConnect()", 0, 0, msg);
		return;
	}
#ifdef RUNASEUID
	if (xisp_euidaccess(Pppd, X_OK) == -1) {
#else
	if (access(Pppd, X_OK) == -1) {
#endif
		XBell(fl_get_display(), 50);
		alertMessage("xISP: doConnect()", 0, 0,
					 "No permission to start pppd!");
		return;
	}
#if 0
	if ((p_xisprc->operOpts & SCRIPT_DIALIN) &&
		*(p_xisprc->account) == 0)
	{
		XBell(fl_get_display(), 50);
		alertMessage("xISP: doConnect()", 0, 0,
					 "You are using a connection "
					 "script but\nyou haven't "
					 "defined an account name!");
		return;
	}
#endif
	fl_deactivate_object(fd_topFrame->conButton);	/* disable button */
	fl_set_object_lcol(fd_topFrame->conButton,
					   FL_INACTIVE);
	strcpy(connSpeed, EMPTY_SPEED);					/* reset speed and */
	strcpy(connIP, EMPTY_IP);						/* IP address strings */
	ret = writeISPenv(p_xisprc);					/* create the env file */
	if (ret <= 0) {									/* script lines OK? */
		if (ret < 0)								/* nope, print message */
			sprintf(msg, "Script line syntax "		/* and be more verbose */
					"error:\nMore than 1 %%U "		/* if possible */
					"and/or %%P per expect "
					"and/or send section!");
		else
			sprintf(msg, "Script line syntax "
					"error!");
		alertMessage("xISP: doConnect()",0,0,msg);
		fl_activate_object(fd_topFrame->conButton);	/* enable button */
		fl_set_object_lcol(fd_topFrame->conButton,
						   FL_LCOL);
#ifndef ISPENV_USEVARS
		unlink(envfname);							/* del environment file */
#endif
		return;										/* and do nothing */
	}
	sprintf(speed, "%u", p_xisprc->modemSpeed);		/* create the speed str */
	if (p_xisprc->operOpts & BSD_COMPRESS)			/* the BSD compression */
		sprintf(blevel, "%d,%d",					/* level string */
				p_xisprc->compLevel,
				p_xisprc->compLevel);
	else
		sprintf(blevel, "%d,%d", 0,0);
	if (p_xisprc->operOpts & DEFL_COMPRESS)			/* deflate compression */
		sprintf(dlevel, "%d,%d",					/* level string */
				p_xisprc->compLevel,
				p_xisprc->compLevel);
	else
		sprintf(dlevel, "%d,%d", 0,0);
	if (p_xisprc->operOpts & ACCEPT_LOCALIP)		/* the local IP string */
		strcpy(lips, LOCAL_IPSTR);
	else
		IPToStr(p_xisprc->localIP, lips);
	if (p_xisprc->operOpts & ACCEPT_REMOTEIP)		/* the remote IP string */
		strcpy(rips, REMOTE_IPSTR);
	else
		IPToStr(p_xisprc->remoteIP, rips);
	strcpy(lrs, lips);								/* the local:remote */
	strcat(lrs, ":");								/* pppd parameter */
	strcat(lrs, rips);
	IPToStr(p_xisprc->netmask, nms);				/* the netmask */
	sprintf(mtus, "%d", p_xisprc->mtu);				/* MTU and MRU */
	sprintf(mrus, "%d", p_xisprc->mru);
	sprintf(lcps, "%d",								/* LCP restart interval */
			(int)ceil(p_xisprc->LCPWait /
					  LCP_RESTART));
	adjustPAPCap();									/* check PAP capability */
	if (p_xisprc->operOpts & PAP_LOGIN)				/* using PAP via +ua ? */
		writeISPPAP(p_xisprc);						/* yes, create data file */
	if (p_xisprc->operOpts & IP_UPDOWN) {			/* if DNS support is on */
		strcpy(IPParam, Pipe);						/* create ipparam string */
		strcat(IPParam, ":");						/* with named pipe file- */
		strcat(IPParam, p_xisprc->descr);			/* name, ISP name */
		strcat(IPParam, ":");
		IPToStr(p_xisprc->dns1, dnsips);			/* DNS server IP's */
		if (strcmp(dnsips, DNS_IPSTR) &&			/* insert empty string */
			!(p_xisprc->operOpts & AUTO_DNS)		/* if DNS1 IP != 0.0.0.0 */
		   )										/* and auto-DNS is off */
			strcat(IPParam, dnsips);				/* then insert DNS1 IP */
		strcat(IPParam, ":");
		IPToStr(p_xisprc->dns2, dnsips);
		if (strcmp(dnsips, DNS_IPSTR) &&			/* if DNS2 IP != 0.0.0.0 */
			!(p_xisprc->operOpts & AUTO_DNS)		/* and auto-DNS is off */
		   )										/* then insert DNS2 IP */
			strcat(IPParam, dnsips);
		strcat(IPParam, ":");
		if (p_xisprc->operOpts & DEFAULT_DOMAIN)	/* if default domain */
			strcat(IPParam, p_xisprc->domainname);	/* def'ed, insert it too */
	}
	if ((p_xisprc->operOpts &
		 (CALL_BACK|CB_NT_RAS)) && !firstConn)
		bprintf("NT-RAS call-back mode initiated\n");
	else
		bprintf("Connection initiated: %s\n",
				strtime());
	devPath = p_xisprc->modemDevice;				/* full path to dev node */
	if ((devName = 									/* skip the leading */
		 strrchr(p_xisprc->modemDevice, '/'))		/* path component */
		== NULL)									/* in the full path */
		devName = p_xisprc->modemDevice;			/* in order to get the */
	else											/* device name only */
		++devName;
#ifdef PPPDLCKDIR
#if defined(SUNOS5x) || defined(SVR4)
	if (stat(devPath, &sbuf) == 0)					/* assemble the lock */
		sprintf(devLock, "%03lu.%03lu.%03lu",		/* file name */
			major((unsigned long int)sbuf.st_dev),
			major((unsigned long int)sbuf.st_rdev),
			minor((unsigned long int)sbuf.st_rdev));
#else
	strcpy(devLock, devName);
#endif
#endif
	if (pppdVersion().major > 2 ||					/* build version */
		pppdVersion().minor > 2) {					/* dependent arg list */
		strcpy(peerfn, "xisp_");					/* this for v2.3+ */
		strcat(peerfn, devName);
		pppdArg("", Pppd, speed, "call", peerfn,
				DEFLCOMP, dlevel,
				(char *)0);
	}
	else {
		pppdArg("", Pppd, p_xisprc->modemDevice,	/* and this for v2.2- */
				speed, (char *)0);
	}
	pppdArg("", BSDCOMP, blevel,					/* the rest is common */
			"noipdefault",
			"modem", "lock",
			"mru", mrus, "mtu", mtus,
			"asyncmap", p_xisprc->asyncmap,
			"lcp-restart", LCP_RESTART_STR,
			"lcp-max-configure", lcps,
			(char *)0);
	if (p_xisprc->operOpts & ESCAPE_ON)
		pppdArg("", "escape", p_xisprc->escape,
				(char *)0);
	if (p_xisprc->operOpts & HW_FLOWCTRL)
		pppdArg("", "crtscts", (char *)0);
	if (pppdOptsFileOK())
		pppdArg("", "file", PPPD_OPTIONS, (char *)0);
	if (p_xisprc->operOpts & ACCEPT_LOCALIP)
		pppdArg("", "ipcp-accept-local", (char *)0);
	if (p_xisprc->operOpts & ACCEPT_REMOTEIP)
		pppdArg("", "ipcp-accept-remote", (char *)0);
	pppdArg("", lrs, "netmask", nms, (char *)0);
	if (p_xisprc->operOpts & DEFAULT_ROUTE)
		pppdArg("", "defaultroute", (char *)0);
	if (p_xisprc->operOpts & PAP_LOGIN)
		pppdArg("", "+ua", papfname, (char *)0);
	if (p_xisprc->operOpts & PAPS_LOGIN) {
		pppdArg("", "user", p_xisprc->name,
				"remotename", p_xisprc->rname,
				(char *)0);
	}
	if (p_xisprc->operOpts & CHAPS_LOGIN) {
		if (pppdVersion().major > 2 ||				/* build version */
			pppdVersion().minor > 2) {				/* dependent arg list */
			pppdArg("", "user", p_xisprc->name,		/* this for v2.3+ */
					"remotename", p_xisprc->rname,
					(char *)0);
		}
		else {
			pppdArg("", "name", p_xisprc->name,		/* and this for v2.2- */
					"remotename", p_xisprc->rname,
					(char *)0);
		}
	}
	if (p_xisprc->operOpts & IP_UPDOWN) {
		adjustAutoDNSCap();							/* check auto-DNS cap. */
		if (p_xisprc->operOpts & AUTO_DNS)
			pppdArg("", "usepeerdns", (char *)0);
		pppdArg("", "ipparam", IPParam, (char *)0);
	}
	if (debug)
		pppdArg("", "debug",
				"kdebug", "7", (char *)0);
	if (p_xisprc->operOpts &
		(CALL_BACK|CB_NT_RAS)) {					/* if dial-in phase of */
		if (firstConn) {							/* an NT-RAS connection */
			if (pppdVersion().major > 2 ||			/* build version */
				pppdVersion().minor > 2)			/* dependent arg list */
				pppdArg("", "callback",				/* use "callback" param */
						p_xisprc->CBphone,			/* for v2.3+ */
						(char *)0);
			else
				pppdArg("", "cb",					/* use "cb" param */
						p_xisprc->CBphone,			/* for v2.2.x */
						(char *)0);
		}											/* and start pppd now */
		else										/* else call-back phase */
			mSleep(2000);							/* so wait 2 seconds */
		if (pppdVersion().major > 2 &&				/* before starting pppd */
			pppdVersion().minor > 3 &&
			pppdVersion().plevel > 6)				/* for v2.3.7+ allow all */
			pppdArg("", "receive-all",				/* ctl chars (needed for */
					(char *)0);						/* buggy NT PPP peers) */
	}
	if (pppdVersion().major < 3 &&
		pppdVersion().minor < 3)
		pppdArg("", "connect", Dialer, (char *)0);
	if (p_xisprc->operOpts &
		(CALL_BACK|CB_NT_RAS)) {					/* if dial-in phase of */
		if (firstConn)								/* an NT-RAS connection */
			waitPppd(0);							/* std pppd timed-wait */
		else {										/* else call-back phase */
			waitPppd(1);							/* so wait gracefully */
			downTime = time(NULL);					/* record pppd exit time */
			onLineSecs = downTime - upTime;			/* calc dial-in time */
			onLineCost = callCost(p_ptt,			/* and cost */
				onLineSecs, &onLineUnits);
			if (p_ptt->attribs &					/* if the selected PTT */
				(PTT_PER_MINUTE|PTT_PER_SECS))		/* charges by time */
				totalCost += onLineCost;			/* update the total cost */
			else									/* else update the total */
				totalUnits += onLineUnits;			/* unit value instead */
			updateLogs(0);							/* then update logs */
			if (global.logOpts & COST_READOUT) {	/* cost readout selected? */
				p = costStr(p_ptt, onLineCost);		/* format it in a string */
				fl_set_object_label(				/* print it on main form */
					fd_topFrame->logText, p);
			}
			else {									/* no, time readout */
				ptm = gmtime(&connTime);			/* break down time */
				strftime(tout,16, "%H:%M:%S", ptm);	/* format it as HH:MM:SS */
				fl_set_object_label(				/* print it on main form */
					fd_topFrame->logText, tout);
				if (ptm->tm_hour <10) p = &tout[1];	/* skip first hour digit */
				else p = tout;						/* for the icon label */
			}
			updateTitles(p, p);						/* upd icon & win titles */
		}
	}
	else											/* for all other types */
		waitPppd(0);								/* old pppd timed wait */
	pppd(pppd_arg);									/* exec new pppd */
	fl_set_idle_callback(IOIdleCallback, NULL);		/* and set idle callback */
	return;
}

void doInterrupt(FL_OBJECT *obj, long param)
{
	fl_deactivate_object(fd_topFrame->intButton);	/* disable button */
	fl_set_object_lcol(fd_topFrame->intButton,
					   FL_INACTIVE);
	fl_call_object_callback(btimer);				/* flush/check status */
	if (dialerPID) {								/* dialer still there? */
		bprintf("\nDialing interrupted: %s\n",
				strtime());
		if (kill(dialerPID, SIGTERM))				/* yup, terminate it */
			doErr("doInterrupt: kill");
	}
	connected = 0;									/* invalidate connection */
	firstConn = 1;
	return;
}


/* .xisp-down is executed before the link is disconnected; note
   that the maximum wait for the child script to finish (while
   printing its output on xisp's browser) is set to 20 secons */

#define MAXSEC_XISPDNWAIT 20
#define MSEC_ITERATION 200
#define MAX_WAIT (MAXSEC_XISPDNWAIT*1000/MSEC_ITERATION)

void doDisconnect(FL_OBJECT *obj, long param)
{
	char buf[MAXBUF_CHILD+1];
	int br, wait = MAX_WAIT, stat;

	userDiscon = 1;									/* set indicator */
	fl_deactivate_object(fd_topFrame->disButton);	/* disable button */
	fl_set_object_lcol(fd_topFrame->disButton,
					   FL_INACTIVE);
	bprintf("Disconnection: %s\n", strtime());

	if ((xispUDPID = xispUD(0,0))) {				/* if .xisp-down exists */
		do {										/* and starts OK */
			br = read(dialerFD, buf, MAXBUF_CHILD);	/* read a buffer full */
			if (br > 0) {							/* if read OK */
				buf[br] = 0;						/* indicate string end */
				bprintf("%s", buf);					/* print it */
				fl_check_forms();					/* update browser */
			}
			else if (br < 0) {						/* read failed */
#ifdef SUNOS41x
				if (errno != EWOULDBLOCK)			/* output unavailable? */
#else
				if (errno != EAGAIN)
#endif
					wait = 1;						/* yes, exit loop */
			}
			else if (waitpid(xispUDPID, &stat,		/* EOF on pipe output */
					 WNOHANG) < 0)					/* so, if script done */
				wait = 1;							/* exit the loop */
			if (--wait)								/* decrement sec counter */
				mSleep(MSEC_ITERATION);				/* if != 0, sleep a bit */
			fl_check_forms();						/* update windows */
		}
		while (wait > 0);
	}

	if (dialerPID)									/* dialer still there? */
		kill(dialerPID, SIGTERM);					/* yup, terminate it */
	pppdPID = getPppdPID(devName);					/* PID of spawned pppd */
	if (pppdPID)									/* interrupt pppd proc */
		kill(pppdPID, SIGINT);
	pppdPID = 0;									/* indicate no pppd proc */
	fl_call_object_callback(ltimer);				/* update link status */
	return;
}

void doFile(FL_OBJECT *obj, long param)
{
	void prepFolderForm(int);
	int deleteHandler(FL_FORM *, void *);

	switch (fl_get_menu(obj)) {

		case FILE_OPTIONS:{
			fl_deactivate_object(fd_topFrame->fileMenu);
			fl_set_object_lcol(fd_topFrame->fileMenu,FL_INACTIVE);
			prepFolderForm(OPTIONS_ACCOUNT);
			fl_set_folder_bynumber(fd_optsTab->optsFolder, OPTIONS_ACCOUNT);
			fl_set_form_hotspot(fd_optsTab->optsTab, 100,80);
			fl_show_form(fd_optsTab->optsTab, FL_PLACE_HOTSPOT,
                         FL_TRANSIENT, "Options Setup");
			break;
		}

		case FILE_HINTS:{
			nohints = !nohints;
			if (nohints) {
				fl_set_menu(fd_topFrame->fileMenu,
							"Options . . .|Enable hints|Exit");
				global.logOpts &= ~LOG_HINTS;
			}
			else {
				fl_set_menu(fd_topFrame->fileMenu,
							"Options . . .|Disable hints|Exit");
				global.logOpts |= LOG_HINTS;
			}
			writeXisprc(rcfname, xispOptions, &global);
		}
		break;

		case FILE_EXIT:{
			deleteHandler(fd_topFrame->topFrame, NULL);
			break;
		}

		default: break;
	}

	return;
}

void doLogging(FL_OBJECT *obj, long param)
{
	int selection, i, tnf[3] = {52, 12, 6},
		type, cur, color;
	unsigned long tblTime[64], tTime = 0;
	float tblCosts[64];
	char buf[64], fid[32],
		 mon[12][4] = {"Jan", "Feb", "Mar", "Apr", "May", "Jun",
					   "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};
	double cost, tcost = 0, max = 0;

	fl_deactivate_object(fd_topFrame->logMenu);
	fl_set_object_lcol(fd_topFrame->logMenu,FL_INACTIVE);
	selection = fl_get_menu(obj);
	switch (selection) {
		case LOGGING_OPTIONS: {
			fl_set_button(fd_logInfo->OLTimeButton, 0);
			fl_set_button(fd_logInfo->OLChargeButton, 0);
			if (global.logOpts & COST_READOUT)
				fl_set_button(fd_logInfo->OLChargeButton, 1);
			else
				fl_set_button(fd_logInfo->OLTimeButton, 1);
			fl_set_button(fd_logInfo->logNoButton, 0);
			fl_set_button(fd_logInfo->logWeekButton, 0);
			fl_set_button(fd_logInfo->logMonthButton, 0);
			fl_set_button(fd_logInfo->logBiMonthButton, 0);
			if (global.logOpts & LOG_NONE)
				fl_set_button(fd_logInfo->logNoButton, 1);
			if (global.logOpts & LOG_WEEKLY)
				fl_set_button(fd_logInfo->logWeekButton, 1);
			if (global.logOpts & LOG_MONTHLY)
				fl_set_button(fd_logInfo->logMonthButton, 1);
			if (global.logOpts & LOG_BIMONTHLY)
				fl_set_button(fd_logInfo->logBiMonthButton, 1);
			fd_logInfo->ldata = global.logOpts |
								(currentPTT << 16) |
								(p_ptt->current_zone << 24);
			fl_clear_choice(fd_logInfo->pttDropChoice);
			for (i=0; i<global.numPTTs; i++)
				fl_addto_choice(fd_logInfo->pttDropChoice,
								clipStr(ptt[i].name, FL_NORMAL_SIZE,
										FL_NORMAL_STYLE, 240));
			fl_set_choice(fd_logInfo->pttDropChoice, currentPTT+1);
			fl_clear_choice(fd_logInfo->zoneDropChoice);
			for (i=0; i<p_ptt->num_zones; i++)
				fl_addto_choice(fd_logInfo->zoneDropChoice,
								clipStr(p_ptt->zone_name[i], FL_NORMAL_SIZE,
										FL_NORMAL_STYLE, 240));
			fl_set_choice(fd_logInfo->zoneDropChoice,p_ptt->current_zone+1);
			showPttInfo(p_ptt);
			fd_logInfo->vdata = NULL;	/* reset PTT database change flag */
			strcpy(fd_logInfo->cdata, p_ptt->currency);	/* save currency */
			fl_set_form_hotspot(fd_logInfo->logInfo, 310,120);
			fl_show_form(fd_logInfo->logInfo, FL_PLACE_HOTSPOT,
						 FL_TRANSIENT, "Logging Options");
			break;
		}

		case LOGGING_STATS: {
			type = buildStats(&global, &cur, tblTime, tblCosts);
			fl_clear_browser(fd_statInfo->statsBrowser);
			fl_clear_chart(fd_statInfo->statsChart);
			fl_clear_chart(fd_statInfo->pieChart);
			fl_set_chart_autosize(fd_statInfo->statsChart, 0);
			fl_set_chart_maxnumb(fd_statInfo->statsChart, tnf[type]);
			for (i=0; i<tnf[type]; i++) {
				switch (type) {
					case 0: sprintf(fid, "Week %d", i+1); break;
					case 1: sprintf(fid, "%s", mon[i]); break;
					case 2: sprintf(fid, "M%d,%d", 2*i+1, 2*i+2); break;
				}
				cost = tblCosts[i];
				if (cost > max)
					max = cost;
				tTime += tblTime[i];
				tcost += cost;
				sprintf(buf, "%s%s: %lu s, %s", (i==(cur-1))? "@b":"",
						fid, tblTime[i], costStr(p_ptt, cost));
				fl_addto_browser(fd_statInfo->statsBrowser, buf);
				if (i >= cur)
					color = FL_WHEAT;
				else
					color = FL_DARKCYAN;
				if (i == (cur-1)) {
					sprintf(buf, "@bTotal: %lu s, %s",
							tTime, costStr(p_ptt, tcost));
					fl_addto_browser(fd_statInfo->statsBrowser, buf);
					color = FL_WHITE;
					tTime = tcost = 0;
				}
				switch (type) {
					case 0: if (i>0 && (i%10)) *fid = 0;
							else sprintf(fid, "W%d", i+1);
							break;
					case 2: sprintf(fid, "%s,%s", mon[2*i], mon[2*i+1]);
							break;
				}
				fl_add_chart_value(fd_statInfo->statsChart, cost, fid, color);
				if (cost > 0.1) {
					if (i % 2)
						color = FL_DARKCYAN;
					else
						color = FL_WHEAT;
				}
				if (i == (cur-1))
					color = FL_WHITE;
				fl_add_chart_value(fd_statInfo->pieChart, cost, fid, color);
			}
			fl_set_chart_bounds(fd_statInfo->statsChart, 0.0, max);
			if (i != tnf[type]) {
				sprintf(buf, "@bTotal: %lu s, %s",
						tTime, costStr(p_ptt, tcost));
				fl_addto_browser(fd_statInfo->statsBrowser, buf);
			}
			fl_show_form(fd_statInfo->statInfo, FL_PLACE_MOUSE,
						 FL_TRANSIENT, "Logging Statistics");
		}

		default: break;
	}
	return;
}

void doHelp(FL_OBJECT *obj, long param)
{
	int selection, i;

	selection = fl_get_menu(obj);
	switch (selection) {
		case HELP_ABOUT: {
			fl_show_form(fd_aboutInfo->aboutInfo, FL_PLACE_MOUSE,
						 FL_TRANSIENT, "About xISP");
			break;
		}

		case HELP_GENERAL: {
			fl_clear_browser(fd_helpInfo->helpBrowser);
			for (i=0; i<MAXNUM_HELPLINES; i++)
				fl_add_browser_line(fd_helpInfo->helpBrowser,
									helpText[i]);
			fl_show_form(fd_helpInfo->helpInfo, FL_PLACE_MOUSE,
						 FL_TRANSIENT, "General Information");
			break;
		}

		default: break;
	}
	return;
}

void doISPDropChoice(FL_OBJECT *obj, long param)
{
	int c, sPTT, newCurrency;

	if ((c = fl_get_choice(obj))) {
		if ((unsigned int)c == currentRC)
			return;
		sPTT = xispOptions[c].ispPTT;
		newCurrency = strcmp(ptt[currentPTT].currency,ptt[sPTT].currency);
		if (!(global.logOpts & LOG_NONE) && newCurrency &&
			! actionVerify("The PTT for the selected ISP bills you in "
						   "currency\nwhich is different from the one "
						   "currently in use. This\nimplies resetting of "
						   "log files, continue?", 0))
		{
			fl_set_choice(obj, currentRC);
			return;
		}
		if (!(global.logOpts & LOG_NONE) && newCurrency)
			resetLogs();
		currentRC = c;
		p_xisprc = &xispOptions[currentRC];
		currentPTT = p_xisprc->ispPTT;
		global.costPTT = currentPTT;
		p_ptt = &ptt[currentPTT];
		p_ptt->current_zone = p_xisprc->ispZone;
		writeXisprc(rcfname, xispOptions, &global);
	}
	return;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                     Selection routines for topFrame                     |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

int select_IPText(FL_OBJECT *obj, int event,
				  FL_Coord mx, FL_Coord my,
				  int key, void *raw_event)
{
	Display *disp = fl_get_display();
	static unsigned char selected = 0;

	if (linkOK && event == FL_RELEASE) {
		selected ^= 1;
		if (selected) {
			XSetSelectionOwner(disp, XA_CUT_BUFFER0, topWin, CurrentTime);
			if (XGetSelectionOwner(disp, XA_CUT_BUFFER0) == topWin) {
				XStoreBytes(disp, connIP, strlen(connIP)); 
				fl_set_object_color(obj, FL_CYAN, FL_MCOL);
    			fl_set_object_lcolor(obj, FL_BLACK);
    			fl_set_object_lstyle(obj, FL_NORMAL_STYLE);
			}
			else
				selected ^= 1;
		}
		else {
			fl_set_object_color(obj, FL_BLACK, FL_MCOL);
    		fl_set_object_lcolor(obj, FL_WHITE);
    		fl_set_object_lstyle(obj, FL_NORMAL_STYLE+FL_SHADOW_STYLE);
		}
	}
	return 0;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |              Utility and callback routines for accountInfo              |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void updateAccInfo(accountInfo_t *a)
{
	char phone[MAXNUM_TELS*(MAXLEN_PHONE+1)+1] = {0},
		 ppasswd[MAXLEN_PASSWD+1] = {0};
	xisprc_t *p = &xispOptions[a->cur];
	int i;

	strcpy(a->descr, p->descr);
	if (global.numISPs && a->dflt == a->cur)
		fl_set_button(fd_accountInfo->ISPDefault, 1);
	else
		fl_set_button(fd_accountInfo->ISPDefault, 0);
	fd_accountInfo->ldata = p->operOpts;
	if (global.numISPs && p->operOpts & STARTUP_DIAL)
		fl_set_button(fd_accountInfo->ISPAutoDial, 1);
	else
		fl_set_button(fd_accountInfo->ISPAutoDial, 0);
	if (global.numISPs && p->operOpts & AUTO_REDIAL)
		fl_set_button(fd_accountInfo->autoRedial, 1);
	else
		fl_set_button(fd_accountInfo->autoRedial, 0);
	for (i=0; i<p->numPhones; i++) {
		if (i) strcat(phone, ";");
		strcat(phone, p->phone[i]);
	}
	fl_set_input(fd_accountInfo->telInput, phone);
	fl_set_input(fd_accountInfo->accInput, p->account);
	if (*(p->account))
		pdecode(ppasswd, p->passwd);
	fl_set_input(fd_accountInfo->pswInput, ppasswd);
	fl_set_input(fd_accountInfo->UNInput, p->name);
	fl_set_input(fd_accountInfo->remoteInput, p->rname);
	fl_set_button(fd_accountInfo->authPAPButton, 0);
	fl_set_button(fd_accountInfo->authPAPSButton, 0);
	fl_set_button(fd_accountInfo->authCHAPSButton, 0);
	fl_set_button(fd_accountInfo->authNoneButton, 0);
	if (global.numISPs) {
		if (p->operOpts & PAP_LOGIN) {
			fl_set_button(fd_accountInfo->authPAPButton, 1);
			fl_activate_object(fd_accountInfo->accInput);
			fl_set_object_lcol(fd_accountInfo->accInput, FL_WHITE);
			fl_activate_object(fd_accountInfo->pswInput);
			fl_set_object_lcol(fd_accountInfo->pswInput, FL_WHITE);
			fl_deactivate_object(fd_accountInfo->UNInput);
			fl_set_object_lcol(fd_accountInfo->UNInput, FL_INACTIVE);
			fl_deactivate_object(fd_accountInfo->remoteInput);
			fl_set_object_lcol(fd_accountInfo->remoteInput, FL_INACTIVE);
		}
		else if (p->operOpts & PAPS_LOGIN) {
			fl_set_button(fd_accountInfo->authPAPSButton, 1);
			fl_deactivate_object(fd_accountInfo->accInput);
			fl_set_object_lcol(fd_accountInfo->accInput, FL_INACTIVE);
			fl_deactivate_object(fd_accountInfo->pswInput);
			fl_set_object_lcol(fd_accountInfo->pswInput, FL_INACTIVE);
			fl_activate_object(fd_accountInfo->UNInput);
			fl_set_object_lcol(fd_accountInfo->UNInput, FL_WHITE);
			fl_activate_object(fd_accountInfo->remoteInput);
			fl_set_object_lcol(fd_accountInfo->remoteInput, FL_WHITE);
		}
		else if (p->operOpts & CHAPS_LOGIN) {
			fl_set_button(fd_accountInfo->authCHAPSButton, 1);
			fl_deactivate_object(fd_accountInfo->accInput);
			fl_set_object_lcol(fd_accountInfo->accInput, FL_INACTIVE);
			fl_deactivate_object(fd_accountInfo->pswInput);
			fl_set_object_lcol(fd_accountInfo->pswInput, FL_INACTIVE);
			fl_activate_object(fd_accountInfo->UNInput);
			fl_set_object_lcol(fd_accountInfo->UNInput, FL_WHITE);
			fl_activate_object(fd_accountInfo->remoteInput);
			fl_set_object_lcol(fd_accountInfo->remoteInput, FL_WHITE);
		}
		else {
			fl_set_button(fd_accountInfo->authNoneButton, 1);
			fl_activate_object(fd_accountInfo->accInput);
			fl_set_object_lcol(fd_accountInfo->accInput, FL_WHITE);
			fl_activate_object(fd_accountInfo->pswInput);
			fl_set_object_lcol(fd_accountInfo->pswInput, FL_WHITE);
			fl_deactivate_object(fd_accountInfo->UNInput);
			fl_set_object_lcol(fd_accountInfo->UNInput, FL_INACTIVE);
			fl_deactivate_object(fd_accountInfo->remoteInput);
			fl_set_object_lcol(fd_accountInfo->remoteInput, FL_INACTIVE);
		}
		p_ptt = &ptt[p->ispPTT];
		p_ptt->current_zone = p->ispZone;
		IFC_accountInfo.udata = p->ispPTT |
								(p_ptt->current_zone << 8);
		if (!(global.logOpts & LOG_NONE) || (global.logOpts & COST_READOUT)) {
			fl_set_choice(fd_accountInfo->ispPttDropChoice, p->ispPTT+1);
			fl_clear_choice(fd_accountInfo->ispZoneDropChoice);
			for (i=0; i<p_ptt->num_zones; i++)
				fl_addto_choice(fd_accountInfo->ispZoneDropChoice,
								clipStr(p_ptt->zone_name[i], FL_MEDIUM_SIZE,
										FL_BOLD_STYLE, 235));
			fl_set_choice(fd_accountInfo->ispZoneDropChoice,
						  p_ptt->current_zone+1);
		}
	}
}

void saveAccInfo(accountInfo_t *a)
{
	char *p1, *p2;
	int i, nc;
	unsigned char sPTT = (unsigned)a->udata & 0x000000FF,
				  sZone = ((unsigned)a->udata & 0x00000FF00) >> 8;

	currentRC = a->cur;
	p_xisprc = &xispOptions[currentRC];
	fl_call_object_callback(fd_renameISP->ISPNameInput);
	strcpy(p_xisprc->descr, fd_renameISP->cdata);
	global.dfltISP = a->dflt;
	p_xisprc->operOpts = fd_accountInfo->ldata;
	fl_clear_choice(fd_topFrame->ISPDropChoice);
	for (i=0; i<global.numISPs; i++)
		if (*(xispOptions[i+1].descr))
			fl_addto_choice(fd_topFrame->ISPDropChoice,
				clipStr(xispOptions[i+1].descr, FL_NORMAL_SIZE,
						FL_NORMAL_STYLE, 150));
	fl_set_choice(fd_topFrame->ISPDropChoice, currentRC);
	fl_call_object_callback(fd_accountInfo->telInput);
	p1 = fd_accountInfo->cdata;
	for (i=0; i<MAXNUM_TELS && *p1; i++) {
		p2 = strchr(p1, ';');
		if (p2 != NULL)
			nc = p2 - p1;
		else {
			nc = strlen(p1);
			p2 = p1 + nc - 1;
		}
		strncpy(p_xisprc->phone[i], p1, nc);
		p_xisprc->phone[i][nc] = 0;
		p1 = p2 + 1;
	}
	p_xisprc->numPhones = i;
	fl_call_object_callback(fd_accountInfo->accInput);
	strcpy(p_xisprc->account, fd_accountInfo->cdata);
	fl_call_object_callback(fd_accountInfo->pswInput);
	memcpy(p_xisprc->passwd, fd_accountInfo->cdata, MAXLEN_PASSWD);
	fl_call_object_callback(fd_accountInfo->UNInput);
	strcpy(p_xisprc->name, fd_accountInfo->cdata);
	fl_call_object_callback(fd_accountInfo->remoteInput);
	strcpy(p_xisprc->rname, fd_accountInfo->cdata);
	p_ptt = &ptt[sPTT];
	p_xisprc->ispPTT = sPTT;
	p_xisprc->ispZone = sZone;
	p_ptt->current_zone = p_xisprc->ispZone;
	writeXisprc(rcfname, xispOptions, &global);
}

void askSaveAccInfo(accountInfo_t *a)
{
	if (! a->modified)
		return;
	if (actionVerify("Changes recorded for current ISP.\n"
					 "Save them in the ISP data-base ?", 0))
		saveAccInfo(a);
	else {
		fl_replace_browser_line(fd_accountInfo->ISPBrowser, a->cur, a->descr);
		strcpy(xispOptions[a->cur].descr, a->descr);
	}
}

void doAccountInfoTip(FL_OBJECT *obj, long param)
{
	alertMessage("Authentication hints", 1, 0,
	 "The \"authentication protocol\" is the method used to establish\n"
	 "your identity when connecting to an ISP.\n"
	 "\n"
	 "- If you choose \"None\", then most probably you will need to\n"
	 "   create a login script and enter it in the \"Dialing and Login\"\n"
	 "   form. Read Help->General for more details.\n"
	 "\n"
	 "- If you Choose PAP (and your pppd version is newer than\n"
	 "   2.2.0) or CHAP, then your system administrator (or yourself\n"
	 "   acting as root) must add entries for your ISP account in file\n"
	 "   /etc/ppp/pap-secrets or /etc/ppp/chap-secrets respectively.\n");
}

void doTelNoInput(FL_OBJECT *obj, long param)
{
	accountInfo_t *a = &IFC_accountInfo;
	static char phone[MAXNUM_TELS*(MAXLEN_PHONE+1)+1] = {0};

	strncpy(phone, (char *)fl_get_input(obj), MAXNUM_TELS*(MAXLEN_PHONE+1));
	fd_accountInfo->cdata = phone;
	a->modified = 1;
	return;
}

void doAccountInput(FL_OBJECT *obj, long param)
{
	accountInfo_t *a = &IFC_accountInfo;
	static char account[MAXLEN_ACCOUNT+1] = {0};

	strncpy(account, (char *)fl_get_input(obj), MAXLEN_ACCOUNT);
	fd_accountInfo->cdata = account;
	a->modified = 1;
	return;
}

void doPasswdInput(FL_OBJECT *obj, long param)
{
	accountInfo_t *a = &IFC_accountInfo;
	static char epasswd[MAXLEN_PASSWD+1] = {0};
	char ppasswd[MAXLEN_PASSWD+1];

	memset(ppasswd, 0, MAXLEN_PASSWD+1);
	strncpy(ppasswd, (char *)fl_get_input(obj), MAXLEN_PASSWD);
	pencode(epasswd, ppasswd);
	fd_accountInfo->cdata = epasswd;
	a->modified = 1;
	return;
}

void doAuthPAPButton(FL_OBJECT *obj, long param)
{
	accountInfo_t *a = &IFC_accountInfo;

	if (fl_get_button(fd_accountInfo->authPAPButton)) {
		fd_accountInfo->ldata |= PAP_LOGIN;
		fd_accountInfo->ldata &= ~(PAPS_LOGIN | CHAPS_LOGIN);
		fl_activate_object(fd_accountInfo->accInput);
		fl_set_object_lcol(fd_accountInfo->accInput, FL_WHITE);
		fl_activate_object(fd_accountInfo->pswInput);
		fl_set_object_lcol(fd_accountInfo->pswInput, FL_WHITE);
		fl_deactivate_object(fd_accountInfo->UNInput);
		fl_set_object_lcol(fd_accountInfo->UNInput, FL_INACTIVE);
		fl_deactivate_object(fd_accountInfo->remoteInput);
		fl_set_object_lcol(fd_accountInfo->remoteInput, FL_INACTIVE);
		a->modified = 1;
	}
	return;
}

void doAuthPAPSButton(FL_OBJECT *obj, long param)
{
	accountInfo_t *a = &IFC_accountInfo;

	if (fl_get_button(fd_accountInfo->authPAPSButton)) {
		fd_accountInfo->ldata |= PAPS_LOGIN;
		fd_accountInfo->ldata &= ~(PAP_LOGIN | CHAPS_LOGIN);
		fl_deactivate_object(fd_accountInfo->accInput);
		fl_set_object_lcol(fd_accountInfo->accInput, FL_INACTIVE);
		fl_deactivate_object(fd_accountInfo->pswInput);
		fl_set_object_lcol(fd_accountInfo->pswInput, FL_INACTIVE);
		fl_activate_object(fd_accountInfo->UNInput);
		fl_set_object_lcol(fd_accountInfo->UNInput, FL_WHITE);
		fl_activate_object(fd_accountInfo->remoteInput);
		fl_set_object_lcol(fd_accountInfo->remoteInput, FL_WHITE);
		a->modified = 1;

		if (! nohints) alertMessage("PAP hint", 1, 0,
		 "Standard PAP operation requires an entry in your system\n"
		 "file /etc/ppp/pap-secrets, for this ISP account. You need\n"
		 "to enter your account name (at your ISP, e.g. \"dbouras\") in\n"
		 "the \"User/Name:\" input field, and a short-hand name for your\n"
		 "ISP in the \"Remotename:\" input field (e.g. \"hol\").\n"
		 "\n"
		 "Then, your system administrator (or yourself acting as root)\n"
		 "needs to enter a line in /etc/ppp/pap-secrets, with three items\n"
		 "separated by spaces: your  account name, the short-hand\n"
		 "name you chose for this ISP, and your password. If the pass-\n"
		 "word for the example account name above was \"saruobd\",\n"
		 "the line in /etc/ppp/pap-secrets should read:\n"
		 "\n"
		 "                          dbouras  hol  saruobd");
	}
	return;
}

void doAuthCHAPSButton(FL_OBJECT *obj, long param)
{
	accountInfo_t *a = &IFC_accountInfo;

	if (fl_get_button(fd_accountInfo->authCHAPSButton)) {
		fd_accountInfo->ldata |= CHAPS_LOGIN;
		fd_accountInfo->ldata &= ~(PAP_LOGIN | PAPS_LOGIN);
		fl_deactivate_object(fd_accountInfo->accInput);
		fl_set_object_lcol(fd_accountInfo->accInput, FL_INACTIVE);
		fl_deactivate_object(fd_accountInfo->pswInput);
		fl_set_object_lcol(fd_accountInfo->pswInput, FL_INACTIVE);
		fl_activate_object(fd_accountInfo->UNInput);
		fl_set_object_lcol(fd_accountInfo->UNInput, FL_WHITE);
		fl_activate_object(fd_accountInfo->remoteInput);
		fl_set_object_lcol(fd_accountInfo->remoteInput, FL_WHITE);
		a->modified = 1;

		if (! nohints) alertMessage("CHAP hint", 1, 0,
		 "CHAP operation requires two entries in your system file\n"
		 "/etc/ppp/chap-secrets, for this ISP account. You need to\n"
		 "enter your account name (at your ISP, e.g. \"dbouras\") in the\n"
		 "\"User/Name:\" input field, and a remote name as provided by\n"
		 "your ISP in the \"Remotename:\" input field (e.g. \"hol\").\n"
		 "\n"
		 "Then, your system administrator (or yourself acting as root)\n"
		 "needs to enter two lines in /etc/ppp/chap-secrets, each one\n"
		 "having three items separated by spaces: your  account name,\n"
		 "the remote name, and your password. If the password for\n"
		 "the example account name above was \"saruobd\", the lines\n"
		 "in /etc/ppp/pap-secrets should read:\n"
		 "\n"
		 "                            dbouras  hol  saruobd\n"
		 "                            hol  dbouras  saruobd\n"
		 "\n"
		 "If you are selecting CHAP because you are setting up an\n"
		 "NT-RAS call-back account, please read carefully the details\n"
		 "in section \"NT-RAS call-back support\" under \"Help->General\".");
	}
	return;
}

void doAuthNoneButton(FL_OBJECT *obj, long param)
{
	accountInfo_t *a = &IFC_accountInfo;

	if (fl_get_button(fd_accountInfo->authNoneButton)) {
		fd_accountInfo->ldata &= ~(PAP_LOGIN | PAPS_LOGIN | CHAPS_LOGIN);
		fl_activate_object(fd_accountInfo->accInput);
		fl_set_object_lcol(fd_accountInfo->accInput, FL_WHITE);
		fl_activate_object(fd_accountInfo->pswInput);
		fl_set_object_lcol(fd_accountInfo->pswInput, FL_WHITE);
		fl_deactivate_object(fd_accountInfo->UNInput);
		fl_set_object_lcol(fd_accountInfo->UNInput, FL_INACTIVE);
		fl_deactivate_object(fd_accountInfo->remoteInput);
		fl_set_object_lcol(fd_accountInfo->remoteInput, FL_INACTIVE);
		a->modified = 1;
	}
	return;
}

void doUNInput(FL_OBJECT *obj, long param)
{
	accountInfo_t *a = &IFC_accountInfo;
	static char name[MAXLEN_UNR+1] = {0};

	strncpy(name, (char *)fl_get_input(obj), MAXLEN_UNR);
	fd_accountInfo->cdata = name;
	a->modified = 1;
	return;
}

void doRemoteInput(FL_OBJECT *obj, long param)
{
	accountInfo_t *a = &IFC_accountInfo;
	static char rname[MAXLEN_UNR+1] = {0};

	strncpy(rname, (char *)fl_get_input(obj), MAXLEN_UNR);
	fd_accountInfo->cdata = rname;
	a->modified = 1;
	return;
}

void doIspPttDropChoice(FL_OBJECT *obj, long param)
{
	accountInfo_t *a = &IFC_accountInfo;
	unsigned char sPTT = (unsigned)a->udata & 0x000000FF;
	int c, i;
	ptt_t *p;

	if ((c = fl_get_choice(obj)) && --c != sPTT) {
		a->udata &= 0x0FFFFFF00;
		a->udata |= (unsigned)c;
		p = &ptt[c];
		fl_clear_choice(fd_accountInfo->ispZoneDropChoice);
		for (i=0; i<p->num_zones; i++)
			fl_addto_choice(fd_accountInfo->ispZoneDropChoice,
							clipStr(p->zone_name[i], FL_MEDIUM_SIZE,
									FL_BOLD_STYLE, 235));
		a->udata &= 0xFFFF00FF;
		fl_set_choice(fd_accountInfo->ispZoneDropChoice, 1);
		a->modified = 1;
	}
	return;
}

void doIspZoneDropChoice(FL_OBJECT *obj, long param)
{
	accountInfo_t *a = &IFC_accountInfo;
	unsigned char cZone = ((unsigned)a->udata & 0x0000FF00) >> 8;
	int c;

	if ((c = fl_get_choice(obj)) && --c != cZone) {
		a->udata &= 0xFFFF00FF;
		a->udata |= ((unsigned)c << 8);
		a->modified = 1;
	}
	return;
}

void doISPPick(FL_OBJECT *obj, long param)
{
	accountInfo_t *a = &IFC_accountInfo;
	unsigned char cur = fl_get_browser(obj);

	if (cur && a->cur != cur) {
		askSaveAccInfo(a);
		a->cur = cur;
		fl_set_input(fd_renameISP->ISPNameInput, xispOptions[cur].descr);
		updateAccInfo(a);
		a->modified = 0;
	}
	return;
}

void doISPPickNEdit(FL_OBJECT *obj, long param)
{
	accountInfo_t *a = &IFC_accountInfo;
	xisprc_t *p = &xispOptions[a->cur];

	fl_call_object_callback(fd_accountInfo->ISPBrowser);
	fl_set_input(fd_renameISP->ISPNameInput, p->descr);
	a->new = 0;
	fl_deactivate_form(fd_accountInfo->accountInfo);
	fl_show_form(fd_renameISP->renameISP, FL_PLACE_MOUSE,
				 FL_TRANSIENT, "Rename ISP record");
	return;
}

void doAccountInfoAdd(FL_OBJECT *obj, long param)
{
	accountInfo_t *a = &IFC_accountInfo;

	fl_set_input(fd_renameISP->ISPNameInput, "");
	a->new = 1;
	fl_deactivate_form(fd_accountInfo->accountInfo);
	fl_show_form(fd_renameISP->renameISP, FL_PLACE_MOUSE,
				 FL_TRANSIENT, "ISP name");
	return;
}

void doAccountInfoDelete(FL_OBJECT *obj, long param)
{
	accountInfo_t *a = &IFC_accountInfo;
	xisprc_t *p_src, *p_dest, *p = &xispOptions[a->cur];

	if (! expert) {
		char msg[128+MAXLEN_PTTNAME] = {0};

		sprintf(msg, "Are you sure you want to delete ISP\n\"%s\"?", p->descr);
		if (! actionVerify(msg, 0))
			return;
	}
	fl_delete_browser_line(fd_accountInfo->ISPBrowser, a->cur);
	if (a->cur == a->dflt)
		a->dflt = 1;
	if (a->cur < global.numISPs) {
		p_dest = &xispOptions[a->cur];
		p_src = &xispOptions[(a->cur)+1];
#ifdef SUNOS41x
		bcopy(p_src, p_dest, (global.numISPs-(a->cur))*sizeof(xisprc_t));
#else
		memmove(p_dest, p_src, (global.numISPs-(a->cur))*sizeof(xisprc_t));
#endif
	}
	else
		a->cur -= 1;
	p = realloc(xispOptions, global.numISPs*sizeof(xisprc_t));
	if (p == NULL)
		outofMem();
	else {
		xispOptions = p;
		global.numISPs -= 1;
		fd_accountInfo->vdata = (void *)1;	/* ISP database resize */
	}
	if (global.numISPs < 1) {
		fl_deactivate_object(fd_accountInfo->accountInfoCopy);
		fl_set_object_lcol(fd_accountInfo->accountInfoCopy, FL_INACTIVE);
		fl_deactivate_object(fd_accountInfo->accountInfoDelete);
		fl_set_object_lcol(fd_accountInfo->accountInfoDelete, FL_INACTIVE);
		fl_deactivate_object(fd_accountInfo->ISPDefault);
		fl_set_object_lcol(fd_accountInfo->ISPDefault, FL_INACTIVE);
		fl_deactivate_object(fd_accountInfo->ISPAutoDial);
		fl_set_object_lcol(fd_accountInfo->ISPAutoDial, FL_INACTIVE);
		fl_deactivate_object(fd_accountInfo->autoRedial);
		fl_set_object_lcol(fd_accountInfo->autoRedial, FL_INACTIVE);
		fl_deactivate_object(fd_accountInfo->telInput);
		fl_set_object_lcol(fd_accountInfo->telInput, FL_INACTIVE);
		fl_deactivate_object(fd_accountInfo->accInput);
		fl_set_object_lcol(fd_accountInfo->accInput, FL_INACTIVE);
		fl_deactivate_object(fd_accountInfo->pswInput);
		fl_set_object_lcol(fd_accountInfo->pswInput, FL_INACTIVE);
		fl_deactivate_object(fd_accountInfo->UNInput);
		fl_set_object_lcol(fd_accountInfo->UNInput, FL_INACTIVE);
		fl_deactivate_object(fd_accountInfo->remoteInput);
		fl_set_object_lcol(fd_accountInfo->remoteInput, FL_INACTIVE);
		fl_deactivate_object(fd_accountInfo->authType);
		fl_set_object_lcol(fd_accountInfo->authType, FL_INACTIVE);
		a->cur = a->dflt = 0;
	}
	else {
		p = &xispOptions[a->cur];
		fl_set_input(fd_renameISP->ISPNameInput, p->descr);
		fl_select_browser_line(fd_accountInfo->ISPBrowser, a->cur);
	}
	updateAccInfo(a);
	a->modified = 0;
	return;
}

void doAccountInfoCopy(FL_OBJECT *obj, long param)
{
	accountInfo_t *a = &IFC_accountInfo;

	memcpy(&ISPCopyBuf, &xispOptions[a->cur], sizeof(xisprc_t));
	fl_activate_object(fd_accountInfo->accountInfoPaste);
	fl_set_object_lcol(fd_accountInfo->accountInfoPaste, FL_LCOL);
	return;
}

void doAccountInfoPaste(FL_OBJECT *obj, long param)
{
	accountInfo_t *a = &IFC_accountInfo;
	xisprc_t *p = &xispOptions[a->cur];

	p = realloc(xispOptions, (1+global.numISPs+1)*sizeof(xisprc_t));
	if (p == NULL)
		outofMem();
	else {
		xispOptions = p;
		global.numISPs += 1;
		fd_accountInfo->vdata = (void *)1;	/* ISP database resize */
	}
	a->cur = global.numISPs;
	p = &xispOptions[a->cur];
	*p = ISPCopyBuf;
	fl_addto_browser(fd_accountInfo->ISPBrowser, p->descr);
	if (a->dflt == 0) {
		a->dflt = 1;
		fl_activate_object(fd_accountInfo->accountInfoCopy);
		fl_set_object_lcol(fd_accountInfo->accountInfoCopy, FL_LCOL);
		fl_activate_object(fd_accountInfo->accountInfoDelete);
		fl_set_object_lcol(fd_accountInfo->accountInfoDelete, FL_LCOL);
		fl_activate_object(fd_accountInfo->ISPDefault);
		fl_set_object_lcol(fd_accountInfo->ISPDefault, FL_WHITE);
		fl_activate_object(fd_accountInfo->ISPAutoDial);
		fl_set_object_lcol(fd_accountInfo->ISPAutoDial, FL_WHITE);
		fl_activate_object(fd_accountInfo->autoRedial);
		fl_set_object_lcol(fd_accountInfo->autoRedial, FL_WHITE);
		fl_activate_object(fd_accountInfo->telInput);
		fl_set_object_lcol(fd_accountInfo->telInput, FL_WHITE);
		fl_activate_object(fd_accountInfo->authType);
		fl_set_object_lcol(fd_accountInfo->authType, FL_WHITE);
		adjustPAPCap();
	}
	fl_select_browser_line(fd_accountInfo->ISPBrowser, a->cur);
	updateAccInfo(a);
	a->modified = 0;
	return;
}

void doISPDefault(FL_OBJECT *obj, long param)
{
	accountInfo_t *a = &IFC_accountInfo;

	if (fl_get_button(fd_accountInfo->ISPDefault))
		a->dflt = a->cur;
	else {
		a->dflt = 1;
		if (a->cur == 1) {
			XBell(fl_get_display(), 50);
			fl_set_button(fd_accountInfo->ISPDefault, 1);
		}
	}
	a->modified = 1;
	return;
}

void doISPAutoDial(FL_OBJECT *obj, long param)
{
	accountInfo_t *a = &IFC_accountInfo;

	if (fl_get_button(fd_accountInfo->ISPAutoDial))
		fd_accountInfo->ldata |= STARTUP_DIAL;
	else
		fd_accountInfo->ldata &= ~STARTUP_DIAL;
	a->modified = 1;
	return;
}

void doAutoRedial(FL_OBJECT *obj, long param)
{
	accountInfo_t *a = &IFC_accountInfo;

	if (fl_get_button(fd_accountInfo->autoRedial))
		fd_accountInfo->ldata |= AUTO_REDIAL;
	else
		fd_accountInfo->ldata &= ~AUTO_REDIAL;
	a->modified = 1;
	return;
}

void doISPNameInput(FL_OBJECT *obj, long param)
{
	static char name[MAXLEN_DESCR+1] = {0};

	strncpy(name, (char *)fl_get_input(obj), MAXLEN_DESCR);
	fd_renameISP->cdata = name;
	return;
}

void doISPNameEditOK(FL_OBJECT *obj, long param)
{
	accountInfo_t *a = &IFC_accountInfo;
	xisprc_t *p = &xispOptions[a->cur];
	int i;

	fl_call_object_callback(fd_renameISP->ISPNameInput);
	if (*(fd_renameISP->cdata) == 0) {
		XBell(fl_get_display(), 50);
		fl_set_input(fd_renameISP->ISPNameInput, "My new ISP");
		fl_redraw_object(fd_renameISP->ISPNameInput);
		return;
	}
	if (! (a->new)) {
		strncpy(p->descr, fd_renameISP->cdata, MAXLEN_DESCR);
		fl_replace_browser_line(fd_accountInfo->ISPBrowser, a->cur, p->descr);
		a->modified = 1;
	}
	else {
		p = realloc(xispOptions, (1+global.numISPs+1)*sizeof(xisprc_t));
		if (p == NULL)
			outofMem();
		else {
			xispOptions = p;
			global.numISPs += 1;
			fd_accountInfo->vdata = (void *)1;	/* ISP database resize */
		}
		a->cur = global.numISPs;
		p = &xispOptions[a->cur];
		initXisprc(p);
		if (p->ispPTT >= global.numPTTs)
			p->ispPTT = global.numPTTs-1;
		if (a->dflt == 0) {
			a->dflt = 1;
			fl_activate_object(fd_accountInfo->accountInfoCopy);
			fl_set_object_lcol(fd_accountInfo->accountInfoCopy, FL_LCOL);
			fl_activate_object(fd_accountInfo->accountInfoDelete);
			fl_set_object_lcol(fd_accountInfo->accountInfoDelete, FL_LCOL);
			fl_activate_object(fd_accountInfo->ISPDefault);
			fl_set_object_lcol(fd_accountInfo->ISPDefault, FL_WHITE);
			fl_activate_object(fd_accountInfo->ISPAutoDial);
			fl_set_object_lcol(fd_accountInfo->ISPAutoDial, FL_WHITE);
			fl_activate_object(fd_accountInfo->autoRedial);
			fl_set_object_lcol(fd_accountInfo->autoRedial, FL_WHITE);
			fl_activate_object(fd_accountInfo->telInput);
			fl_set_object_lcol(fd_accountInfo->telInput, FL_WHITE);
			fl_activate_object(fd_accountInfo->accInput);
			fl_set_object_lcol(fd_accountInfo->accInput, FL_WHITE);
			fl_activate_object(fd_accountInfo->pswInput);
			fl_set_object_lcol(fd_accountInfo->pswInput, FL_WHITE);
			fl_activate_object(fd_accountInfo->UNInput);
			fl_set_object_lcol(fd_accountInfo->UNInput, FL_WHITE);
			fl_activate_object(fd_accountInfo->remoteInput);
			fl_set_object_lcol(fd_accountInfo->remoteInput, FL_WHITE);
			fl_activate_object(fd_accountInfo->authType);
			fl_set_object_lcol(fd_accountInfo->authType, FL_WHITE);
			adjustPAPCap();

			if (!(global.logOpts & LOG_NONE) || (global.logOpts & COST_READOUT)
			   )
			{
				fl_activate_object(fd_accountInfo->ispPttDropChoice);
				fl_set_object_lcol(fd_accountInfo->ispPttDropChoice,
								   FL_WHITE);
				fl_activate_object(fd_accountInfo->ispZoneDropChoice);
				fl_set_object_lcol(fd_accountInfo->ispZoneDropChoice,
								   FL_WHITE);
				for (i=0; i<global.numPTTs; i++)
					fl_addto_choice(fd_accountInfo->ispPttDropChoice,
						clipStr(ptt[i].name, FL_MEDIUM_SIZE,
								FL_BOLD_STYLE, 235));
				fl_set_choice(fd_accountInfo->ispPttDropChoice,
							  currentPTT+1);
				for (i=0; i<p_ptt->num_zones; i++)
					fl_addto_choice(fd_accountInfo->ispZoneDropChoice,
						clipStr(p_ptt->zone_name[i], FL_MEDIUM_SIZE,
								FL_BOLD_STYLE, 235));
				fl_set_choice(fd_accountInfo->ispZoneDropChoice,
							  p_ptt->current_zone+1);
			}
			else {
				fl_deactivate_object(fd_accountInfo->ispPttDropChoice);
				fl_set_object_lcol(fd_accountInfo->ispPttDropChoice,
								   FL_INACTIVE);
				fl_deactivate_object(fd_accountInfo->ispZoneDropChoice);
				fl_set_object_lcol(fd_accountInfo->ispZoneDropChoice,
								   FL_INACTIVE);
			}
		}
		strncpy(p->descr, fd_renameISP->cdata, MAXLEN_DESCR);
		fl_addto_browser(fd_accountInfo->ISPBrowser, p->descr);
		fl_select_browser_line(fd_accountInfo->ISPBrowser, a->cur);
		updateAccInfo(a);
		a->modified = 0;
	}
	fl_hide_form(fd_renameISP->renameISP);
	fl_activate_form(fd_accountInfo->accountInfo);
	return;
}

void doISPNameEditCancel(FL_OBJECT *obj, long param)
{
	fl_hide_form(fd_renameISP->renameISP);
	fl_activate_form(fd_accountInfo->accountInfo);
	return;
}

void doAccountOK(FL_OBJECT *obj, long done)
{
	accountInfo_t *a = &IFC_accountInfo;

	if (done) {
		currentPTT = (unsigned)a->udata & 0x000000FF;
		global.costPTT = currentPTT;
		saveAccInfo(a);
		fl_hide_form(fd_optsTab->optsTab);
		fl_activate_object(fd_topFrame->fileMenu);
		fl_set_object_lcol(fd_topFrame->fileMenu,FL_LCOL);
	}
	else
		saveAccInfo(a);
	return;
}

void doAccountCancel(FL_OBJECT *obj, long param)
{
	if (fd_accountInfo->vdata || IFC_accountInfo.modified) {
		free(xispOptions);
		readXisprc(rcfname, &xispOptions, &global);
		p_xisprc = &xispOptions[currentRC];
		currentPTT = p_xisprc->ispPTT;
		global.costPTT = currentPTT;
		p_ptt = &ptt[currentPTT];
		p_ptt->current_zone = p_xisprc->ispZone;
	}
	fl_hide_form(fd_optsTab->optsTab);
	fl_activate_object(fd_topFrame->fileMenu);
	fl_set_object_lcol(fd_topFrame->fileMenu,FL_LCOL);
	return;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                Utility routines for dialInfo and CBInfo                 |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void lineParse(char *buf, char (*sline)[MAXLEN_SLINE+1])
{
	int lc = 0;
	char *p = sline[0];

	memset(sline, 0, MAXNUM_SLINES*(MAXLEN_SLINE+1));
	while (*buf != 0) {
		if (*buf != '\n')
			*p++ = *buf++;
		else {
			*p = 0; ++buf;
			p = sline[++lc];
		}
	}
}

int slinePreH(FL_OBJECT *obj, int event, FL_COORD mx,
			  FL_COORD my, int key, void *raw_event)
{
	switch (event) {
		case FL_KEYBOARD:
			if (key == '\t') wasTab = 1;
			else wasTab = 0;
			break;

		default: break;
	}
	return !FL_PREEMPT;
}

int byteFilter(FL_OBJECT *obj, const char *old, const char *cur, int c)
{
	if (c == '-')
		return FL_RINGBELL|FL_INVALID;
	return FL_VALID;
}

/*+-------------------------------------------------------------------------+
  |                                                                         |
  |               Callback routines for dialInfo and CBInfo                 |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void doDialInfoTip(FL_OBJECT *obj, long param)
{
	alertMessage("Script lines hint", 1, 0,
	 "For an example on the syntax for the \"Expect:\" and \"Send:\"\n"
	 "parts of each script line in the \"Automatic Login:\" section,\n"
	 "please read section \"Dialing and Login\" under \"Help->General\".\n"
	 "For a more detailed presentation of script syntax, read the\n"
	 "manual page for the \"chat\" command.");
}

void doRetryInput(FL_OBJECT *obj, long param)
{
	char ns[MAXDIG_RETRY+1] = {0};
	static unsigned char num;

	strncpy(ns, (char *)fl_get_input(obj), MAXDIG_RETRY);
	num = atoi(ns);
	if (! num) {
		XBell(fl_get_display(), 50);
		num = MAXNUM_RETRY;
		sprintf(ns, "%d", num);
		fl_set_input(obj, ns);
		fl_redraw_object(obj);
	}
	fd_dialInfo->vdata = &num;
	return;
}

void doDelayInput(FL_OBJECT *obj, long param)
{
	char ns[MAXDIG_DELAY+1] = {0};
	static unsigned char num;

	strncpy(ns, (char *)fl_get_input(obj), MAXDIG_DELAY);
	num = atoi(ns);
	if (! num) {
		XBell(fl_get_display(), 50);
		num = MAXSEC_DELAY;
		sprintf(ns, "%d", num);
		fl_set_input(obj, ns);
		fl_redraw_object(obj);
	}
	fd_dialInfo->vdata = &num;
	return;
}

void doCNWaitInput(FL_OBJECT *obj, long param)
{
	char ns[MAXDIG_CNWAIT+1] = {0};
	static unsigned char num;

	strncpy(ns, (char *)fl_get_input(obj), MAXDIG_CNWAIT);
	num = atoi(ns);
	if (! num) {
		XBell(fl_get_display(), 50);
		num = MAXSEC_CNWAIT;
		sprintf(ns, "%d", num);
		fl_set_input(obj, ns);
		fl_redraw_object(obj);
	}
	if (num < MINSEC_MODEM_TO) {
		XBell(fl_get_display(), 50);
		num = MINSEC_MODEM_TO;
		sprintf(ns, "%d", num);
		fl_set_input(obj, ns);
		fl_redraw_object(obj);
	}
	fd_dialInfo->vdata = &num;
	return;
}

void doLCPWaitInput(FL_OBJECT *obj, long param)
{
	char ns[MAXDIG_LCPWAIT+1] = {0};
	static unsigned char num;

	strncpy(ns, (char *)fl_get_input(obj), MAXDIG_LCPWAIT);
	num = atoi(ns);
	if (! num) {
		XBell(fl_get_display(), 50);
		num = MAXSEC_CNWAIT;
		sprintf(ns, "%d", num);
		fl_set_input(obj, ns);
		fl_redraw_object(obj);
	}
	if (num < MINSEC_LINK_TO) {
		XBell(fl_get_display(), 50);
		num = MINSEC_LINK_TO;
		sprintf(ns, "%d", num);
		fl_set_input(obj, ns);
		fl_redraw_object(obj);
	}
	fd_dialInfo->vdata = &num;
	return;
}

void doColumnsInput(FL_OBJECT *obj, long param)
{
	char ns[4] = {0};
	static unsigned char num;

	strncpy(ns, (char *)fl_get_input(obj), 3);
	num = atoi(ns);
	if (! num || num < MINCHAR_TERMW) {
		XBell(fl_get_display(), 50);
		num = MINCHAR_TERMW;
		sprintf(ns, "%d", num);
		fl_set_input(obj, ns);
		fl_redraw_object(obj);
	}
	fd_dialInfo->vdata = &num;
	return;
}

void doRowsInput(FL_OBJECT *obj, long param)
{
	char ns[4] = {0};
	static unsigned char num;

	strncpy(ns, (char *)fl_get_input(obj), 3);
	num = atoi(ns);
	if (! num || num < MINCHAR_TERMH) {
		XBell(fl_get_display(), 50);
		num = MINCHAR_TERMH;
		sprintf(ns, "%d", num);
		fl_set_input(obj, ns);
		fl_redraw_object(obj);
	}
	fd_dialInfo->vdata = &num;
	return;
}

void doExpectInput(FL_OBJECT *obj, long param)
{
	static char line[MAXNUM_SLINES][MAXLEN_SLINE+1] = {{0}};

	lineParse((char *)fl_get_input(obj), line);
	fd_dialInfo->vdata = line;
	return;
}

int expectFilter(FL_OBJECT *obj, const char *old, const char *cur, int c)
{
	int x, y;

	fl_get_input_cursorpos(obj, &x, &y);
	if (c) {
		if (x <= MAXLEN_SLINE && y <= MAXNUM_SLINES) {
			if (c == '\n' && fl_get_input_numberoflines(obj) > MAXNUM_SLINES)
				return FL_RINGBELL|FL_INVALID;
			else
				return FL_VALID;
		}
		else return FL_RINGBELL|FL_INVALID;
	}
	else if (wasTab)
		fl_set_focus_object(fd_dialInfo->dialInfo,
							fd_dialInfo->sendInput->child);
	return FL_VALID;
}

void doSendInput(FL_OBJECT *obj, long param)
{
	static char line[MAXNUM_SLINES][MAXLEN_SLINE+1] = {{0}};

	lineParse((char *)fl_get_input(obj), line);
	fd_dialInfo->vdata = line;
	return;
}

int sendFilter(FL_OBJECT *obj, const char *old, const char *cur, int c)
{
	int x, y;

	fl_get_input_cursorpos(obj, &x, &y);
	if (c) {
		if (x <= MAXLEN_SLINE && y <= MAXNUM_SLINES) {
			if (c == '\n' && fl_get_input_numberoflines(obj) > MAXNUM_SLINES)
				return FL_RINGBELL|FL_INVALID;
			else
				return FL_VALID;
		}
		else return FL_RINGBELL|FL_INVALID;
	}
	else if (wasTab)
		fl_set_focus_object(fd_dialInfo->dialInfo, fd_dialInfo->rtrInput);
	return FL_VALID;
}

void doRBYesButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_dialInfo->RBYesButton))
		fd_dialInfo->ldata |= CONNECT_BELL;
	return;
}

void doRBNoButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_dialInfo->RBNoButton))
		fd_dialInfo->ldata &= ~CONNECT_BELL;
	return;
}

void doTTYesButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_dialInfo->TTYesButton)) {
		fd_dialInfo->ldata |= MANUAL_LOGIN;
		fl_activate_object(fd_dialInfo->TSGroup);
		fl_set_object_lcol(fd_dialInfo->TSGroup,FL_WHITE);
		fl_deactivate_object(fd_dialInfo->ALGroup);
		fl_set_object_lcol(fd_dialInfo->ALGroup,FL_INACTIVE);
	}
	return;
}

void doTTNoButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_dialInfo->TTNoButton)) {
		fd_dialInfo->ldata &= ~MANUAL_LOGIN;
		fl_deactivate_object(fd_dialInfo->TSGroup);
		fl_set_object_lcol(fd_dialInfo->TSGroup,FL_INACTIVE);
		fl_activate_object(fd_dialInfo->ALGroup);
		fl_set_object_lcol(fd_dialInfo->ALGroup,FL_WHITE);
	}
	return;
}

void doCBYesButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_dialInfo->CBYesButton)) {
		fd_dialInfo->ldata |= CALL_BACK;
		fl_activate_object(fd_dialInfo->CBOptions);
		fl_set_object_lcol(fd_dialInfo->CBOptions,FL_LCOL);
	}
	return;
}

void doCBNoButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_dialInfo->CBNoButton)) {
		fd_dialInfo->ldata &= ~CALL_BACK;
		fl_deactivate_object(fd_dialInfo->CBOptions);
		fl_set_object_lcol(fd_dialInfo->CBOptions,FL_INACTIVE);
	}
	return;
}

void doCBOptions(FL_OBJECT *obj, long param)
{
	int i, ml = p_xisprc->numCBSlns;
	char dflt[5], input[MAXNUM_SLINES*(MAXLEN_SLINE+1)+1] = {0};

	sprintf(dflt, "%d", p_xisprc->CBDelay);
	fl_set_input(fd_CBInfo->CBdlyInput, dflt);
	for (i=0; i<ml; i++) {
		strcat(input, p_xisprc->CBsln[i]);
		if (i < (ml-1)) strcat(input, "\n");
	}
	fl_set_input(fd_CBInfo->CBExpectInput, input);
	input[0] = 0;
	for (i=0; i<ml; i++) {
		strcat(input, p_xisprc->CBsln[MAXNUM_SLINES+i]);
		if (i < (ml-1)) strcat(input, "\n");
	}
	fl_set_input(fd_CBInfo->CBSendInput, input);
	wasTab = 0;
	if (fd_dialInfo->ldata & CHAPS_LOGIN) {
		fl_activate_object(fd_CBInfo->LTRASButton);
		fl_set_object_lcol(fd_CBInfo->LTRASButton, FL_WHITE);
	}
	else {
		fd_dialInfo->ldata &= ~CB_NT_RAS;
		fl_deactivate_object(fd_CBInfo->LTRASButton);
		fl_set_object_lcol(fd_CBInfo->LTRASButton, FL_INACTIVE);
		if (! nohints) fl_set_timer(fd_CBInfo->tipTimer, HINT_TRIGGER);
	}
	if (fd_dialInfo->ldata & CBMAN_LOGIN) {
		fl_set_button(fd_CBInfo->LTScriptedButton, 0);
		fl_set_button(fd_CBInfo->LTTerminalButton, 1);
		fl_set_button(fd_CBInfo->LTRASButton, 0);
		fl_deactivate_object(fd_CBInfo->ALGroup);
		fl_set_object_lcol(fd_CBInfo->ALGroup,FL_INACTIVE);
		fl_activate_object(fd_CBInfo->TSGroup);
		fl_set_object_lcol(fd_CBInfo->TSGroup,FL_WHITE);
		fl_deactivate_object(fd_CBInfo->CBTelInput);
		fl_set_object_lcol(fd_CBInfo->CBTelInput,FL_INACTIVE);
	}
	else if (fd_dialInfo->ldata & CB_NT_RAS) {
		fl_set_button(fd_CBInfo->LTScriptedButton, 0);
		fl_set_button(fd_CBInfo->LTTerminalButton, 0);
		fl_set_button(fd_CBInfo->LTRASButton, 1);
		fl_deactivate_object(fd_CBInfo->ALGroup);
		fl_set_object_lcol(fd_CBInfo->ALGroup,FL_INACTIVE);
		fl_deactivate_object(fd_CBInfo->TSGroup);
		fl_set_object_lcol(fd_CBInfo->TSGroup,FL_INACTIVE);
		fl_activate_object(fd_CBInfo->CBTelInput);
		fl_set_object_lcol(fd_CBInfo->CBTelInput,FL_WHITE);
	}
	else {
		fl_set_button(fd_CBInfo->LTScriptedButton, 1);
		fl_set_button(fd_CBInfo->LTTerminalButton, 0);
		fl_set_button(fd_CBInfo->LTRASButton, 0);
		fl_activate_object(fd_CBInfo->ALGroup);
		fl_set_object_lcol(fd_CBInfo->ALGroup,FL_WHITE);
		fl_deactivate_object(fd_CBInfo->TSGroup);
		fl_set_object_lcol(fd_CBInfo->TSGroup,FL_INACTIVE);
		fl_deactivate_object(fd_CBInfo->CBTelInput);
		fl_set_object_lcol(fd_CBInfo->CBTelInput,FL_INACTIVE);
	}
	sprintf(dflt, "%d", p_xisprc->CBtermW);
	fl_set_input(fd_CBInfo->columnsInput, dflt);
	sprintf(dflt, "%d", p_xisprc->CBtermH);
	fl_set_input(fd_CBInfo->rowsInput, dflt);
	fd_CBInfo->ldata = fd_dialInfo->ldata;
	fl_set_input(fd_CBInfo->CBTelInput, p_xisprc->CBphone);
	fl_deactivate_form(fd_dialInfo->dialInfo);
	fl_show_form(fd_CBInfo->CBInfo, FL_PLACE_MOUSE,
				 FL_TRANSIENT, "Call-back Options");
	return;
}

void doDialOK(FL_OBJECT *obj, long done)
{
	char *Ep, *Sp;
	int i;

	fl_call_object_callback(fd_dialInfo->rtrInput);
	p_xisprc->maxAttempts = *(unsigned char *)(fd_dialInfo->vdata);
	fl_call_object_callback(fd_dialInfo->dlyInput);
	p_xisprc->sleepDelay = *(unsigned char *)(fd_dialInfo->vdata);
	fl_call_object_callback(fd_dialInfo->CNWaitInput);
	p_xisprc->connectWait = *(unsigned char *)(fd_dialInfo->vdata);
	fl_call_object_callback(fd_dialInfo->LCPWaitInput);
	p_xisprc->LCPWait = *(unsigned char *)(fd_dialInfo->vdata);
	p_xisprc->operOpts = (unsigned)fd_dialInfo->ldata;
	fl_call_object_callback(fd_dialInfo->columnsInput);
	p_xisprc->termW = *(unsigned char *)(fd_dialInfo->vdata);
	fl_call_object_callback(fd_dialInfo->rowsInput);
	p_xisprc->termH = *(unsigned char *)(fd_dialInfo->vdata);
	fl_call_object_callback(fd_dialInfo->expectInput);
	memcpy(p_xisprc->sline, fd_dialInfo->vdata,
		   MAXNUM_SLINES*(MAXLEN_SLINE+1));
	fl_call_object_callback(fd_dialInfo->sendInput);
	memcpy(&((p_xisprc->sline)[MAXNUM_SLINES]), fd_dialInfo->vdata,
		   MAXNUM_SLINES*(MAXLEN_SLINE+1));
	for (i=0; i<MAXNUM_SLINES; i++) {
		Ep = p_xisprc->sline[i];
		Sp = p_xisprc->sline[MAXNUM_SLINES+i];
		if (*Ep | *Sp) p_xisprc->numSlines = i+1;
	}
	if (! (p_xisprc->operOpts & CALL_BACK))
		p_xisprc->numCBSlns = 0;
	writeXisprc(rcfname, xispOptions, &global);
	if (done) {
		fl_hide_form(fd_optsTab->optsTab);
		fl_activate_object(fd_topFrame->fileMenu);
		fl_set_object_lcol(fd_topFrame->fileMenu,FL_LCOL);
	}
	return;
}

void doDialCancel(FL_OBJECT *obj, long param)
{
	fl_hide_form(fd_optsTab->optsTab);
	fl_activate_object(fd_topFrame->fileMenu);
	fl_set_object_lcol(fd_topFrame->fileMenu,FL_LCOL);
	return;
}

void doCBInfoTip(FL_OBJECT *obj, long param)
{
	alertMessage("NT-RAS hint", 1, 0,
	 "If \"NT-RAS call-back\" is grayed-out, that is because you have\n"
	 "not chosen \"CHAP\" as your password authentication protocol.\n"
	 "If you are setting up an account using an NT-RAS based call-\n"
	 "back server, go back to the \"Account Information\" form and select\n"
	 "CHAP; then come back to this form and select \"NT-RAS call-back\".");
}

void doCBExpectInput(FL_OBJECT *obj, long param)
{
	static char line[MAXNUM_SLINES][MAXLEN_SLINE+1] = {{0}};

	lineParse((char *)fl_get_input(obj), line);
	fd_CBInfo->vdata = line;
	return;
}

int CBExpectFilter(FL_OBJECT *obj, const char *old, const char *cur, int c)
{
	int x, y;

	fl_get_input_cursorpos(obj, &x, &y);
	if (c) {
		if (x <= MAXLEN_SLINE && y <= MAXNUM_SLINES) {
			if (c == '\n' && fl_get_input_numberoflines(obj) > MAXNUM_SLINES)
				return FL_RINGBELL|FL_INVALID;
			else
				return FL_VALID;
		}
		else return FL_RINGBELL|FL_INVALID;
	}
	else if (wasTab)
		fl_set_focus_object(fd_CBInfo->CBInfo,
							fd_CBInfo->CBSendInput->child);
	return FL_VALID;
}

void doCBSendInput(FL_OBJECT *obj, long param)
{
	static char line[MAXNUM_SLINES][MAXLEN_SLINE+1] = {{0}};

	lineParse((char *)fl_get_input(obj), line);
	fd_CBInfo->vdata = line;
	return;
}

int CBSendFilter(FL_OBJECT *obj, const char *old, const char *cur, int c)
{
	int x, y;

	fl_get_input_cursorpos(obj, &x, &y);
	if (c) {
		if (x <= MAXLEN_SLINE && y <= MAXNUM_SLINES) {
			if (c == '\n' && fl_get_input_numberoflines(obj) > MAXNUM_SLINES)
				return FL_RINGBELL|FL_INVALID;
			else
				return FL_VALID;
		}
		else return FL_RINGBELL|FL_INVALID;
	}
	else if (wasTab)
		fl_set_focus_object(fd_CBInfo->CBInfo, fd_CBInfo->CBdlyInput);
	return FL_VALID;
}

void doCBDelayInput(FL_OBJECT *obj, long param)
{
	char ns[MAXDIG_CBDELAY] = {0};
	static unsigned char num;

	strncpy(ns, (char *)fl_get_input(obj), MAXDIG_CBDELAY);
	num = atoi(ns);
	if (! num) {
		XBell(fl_get_display(), 50);
		num = MAXSEC_CBDELAY;
		sprintf(ns, "%d", num);
		fl_set_input(obj, ns);
		fl_redraw_object(obj);
	}
	fd_CBInfo->vdata = &num;
	return;
}

void doLTScriptedButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_CBInfo->LTScriptedButton)) {
		fd_CBInfo->ldata &= ~CBMAN_LOGIN;
		fd_CBInfo->ldata &= ~CB_NT_RAS;
		fl_deactivate_object(fd_CBInfo->TSGroup);
		fl_set_object_lcol(fd_CBInfo->TSGroup,FL_INACTIVE);
		fl_deactivate_object(fd_CBInfo->CBTelInput);
		fl_set_object_lcol(fd_CBInfo->CBTelInput,FL_INACTIVE);
		fl_activate_object(fd_CBInfo->ALGroup);
		fl_set_object_lcol(fd_CBInfo->ALGroup,FL_WHITE);
	}
}

void doLTTerminalButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_CBInfo->LTTerminalButton)) {
		fd_CBInfo->ldata |= CBMAN_LOGIN;
		fd_CBInfo->ldata &= ~CB_NT_RAS;
		fl_activate_object(fd_CBInfo->TSGroup);
		fl_set_object_lcol(fd_CBInfo->TSGroup,FL_WHITE);
		fl_deactivate_object(fd_CBInfo->CBTelInput);
		fl_set_object_lcol(fd_CBInfo->CBTelInput,FL_INACTIVE);
		fl_deactivate_object(fd_CBInfo->ALGroup);
		fl_set_object_lcol(fd_CBInfo->ALGroup,FL_INACTIVE);
	}
}

void doLTRASButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_CBInfo->LTRASButton)) {
		fd_CBInfo->ldata &= ~CBMAN_LOGIN;
		fd_CBInfo->ldata |= CB_NT_RAS;
		fl_deactivate_object(fd_CBInfo->TSGroup);
		fl_set_object_lcol(fd_CBInfo->TSGroup,FL_INACTIVE);
		fl_activate_object(fd_CBInfo->CBTelInput);
		fl_set_object_lcol(fd_CBInfo->CBTelInput,FL_WHITE);
		fl_deactivate_object(fd_CBInfo->ALGroup);
		fl_set_object_lcol(fd_CBInfo->ALGroup,FL_INACTIVE);

		if (! nohints) alertMessage("NT-RAS setup hint", 1, 0,
		 "In order for NT-RAS call-back to work you need:\n"
		 "\n"
		 "1. To have a patched version of pppd which includes\n"
		 "    support for NT-RAS call-back. Patches exist for\n"
		 "    relatively old (e.g. 2.2.0f), as well as for recent (e.g.\n"
		 "    2.3.8) versions of pppd. The xisp source distribution\n"
		 "    includes a patch for pppd-2.2.0f; the patch for version\n"
		 "    2.3.x is included in the official pppd source distribution.\n"
		 "\n"
		 "2. To configure the NT server to perform call-back auto-\n"
		 "    matically as opposed to \"asking\" whether or not the caller\n"
		 "    desires call-back.");
	}
}

void doCBTelInput(FL_OBJECT *obj, long param)
{
	static char phone[MAXLEN_PHONE+1] = {0};

	strncpy(phone, (char *)fl_get_input(obj), MAXLEN_PHONE);
	fd_CBInfo->cdata = phone;
	return;
}

void doCBColumnsInput(FL_OBJECT *obj, long param)
{
	char ns[4] = {0};
	static unsigned char num;

	strncpy(ns, (char *)fl_get_input(obj), 3);
	num = atoi(ns);
	if (! num || num < MINCHAR_TERMW) {
		XBell(fl_get_display(), 50);
		num = MINCHAR_TERMW;
		sprintf(ns, "%d", num);
		fl_set_input(obj, ns);
		fl_redraw_object(obj);
	}
	fd_CBInfo->vdata = &num;
	return;
}

void doCBRowsInput(FL_OBJECT *obj, long param)
{
	char ns[4] = {0};
	static unsigned char num;

	strncpy(ns, (char *)fl_get_input(obj), 3);
	num = atoi(ns);
	if (! num || num < MINCHAR_TERMH) {
		XBell(fl_get_display(), 50);
		num = MINCHAR_TERMH;
		sprintf(ns, "%d", num);
		fl_set_input(obj, ns);
		fl_redraw_object(obj);
	}
	fd_CBInfo->vdata = &num;
	return;
}

void doCBInfoOK(FL_OBJECT *obj, long param)
{
	char *Ep, *Sp;
	int i;

	fl_call_object_callback(fd_CBInfo->CBdlyInput);
	p_xisprc->CBDelay = *(unsigned char *)(fd_CBInfo->vdata);
	fl_call_object_callback(fd_CBInfo->CBExpectInput);
	memcpy(p_xisprc->CBsln, fd_CBInfo->vdata, MAXNUM_SLINES*(MAXLEN_SLINE+1));
	fl_call_object_callback(fd_CBInfo->CBSendInput);
	memcpy(&((p_xisprc->CBsln)[MAXNUM_SLINES]), fd_CBInfo->vdata,
		   MAXNUM_SLINES*(MAXLEN_SLINE+1));
	for (i=0; i<MAXNUM_SLINES; i++) {
		Ep = p_xisprc->CBsln[i];
		Sp = p_xisprc->CBsln[MAXNUM_SLINES+i];
		if (*Ep | *Sp) p_xisprc->numCBSlns = i+1;
	}
	fd_dialInfo->ldata = fd_CBInfo->ldata;
	fl_call_object_callback(fd_CBInfo->columnsInput);
	p_xisprc->CBtermW = *(unsigned char *)(fd_CBInfo->vdata);
	fl_call_object_callback(fd_CBInfo->rowsInput);
	p_xisprc->CBtermH = *(unsigned char *)(fd_CBInfo->vdata);
	fl_call_object_callback(fd_CBInfo->CBTelInput);
	strncpy(p_xisprc->CBphone, fd_CBInfo->cdata, MAXLEN_PHONE);
	writeXisprc(rcfname, xispOptions, &global);
	fl_hide_form(fd_CBInfo->CBInfo);
	fl_activate_form(fd_dialInfo->dialInfo);
	return;
}

void doCBInfoCancel(FL_OBJECT *obj, long param)
{
	fl_hide_form(fd_CBInfo->CBInfo);
	fl_activate_form(fd_dialInfo->dialInfo);
	return;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |               Callback routines for aboutInfo and helpInfo              |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void doAboutOK(FL_OBJECT *obj, long param)
{
	fl_hide_form(fd_aboutInfo->aboutInfo);
	return;
}

void doHelpInfoOK(FL_OBJECT *obj, long param)
{
	fl_hide_form(fd_helpInfo->helpInfo);
	return;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                     Callback routines for commInfo                      |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void doCommInfoTip(FL_OBJECT *obj, long param)
{
	alertMessage("Modem hints", 1, 0,
	 "It is very important to properly specify your modem device. E.g.\n"
	 "on i86 PC's, if your modem is on COM1, under Linux, your device\n"
	 "would be /dev/ttyS0; if it's on COM2, use /dev/ttyS1 and so on.\n"
	 "On other supported operating systems, please consult with your\n"
	 "system's manual pages.\n"
	 "\n"
	 "Note that in order for xisp to work properly, the modem must be\n"
	 "configured for verbose result strings (as opposed to numeric result\n"
	 "codes). For most modems this is the default; if that is not the case\n"
	 "for the one you are using, consult with the user's manual for the\n"
	 "appropriate command. For example, most USR modems return\n"
	 "verbose result strings when you include \"V1\" in the \"Init\" string.\n"
	 "In such case, the complete \"Init\" string would be \"AT V1\".\n"
	 "\n"
	 "It is possible to enter command strings beginning with '\\' in the\n"
	 "\"Reset:\" and/or the \"Init:\" input fields. Note, however, that in order\n"
	 "to escape them from chat, you need to prepend an extra '\\'. For\n"
	 "example, the initialization string \"AT \\N2\" needs to be entered as\n"
	 "\"AT \\\\N2\".\n"
	 "\n"
	 "Choosing \"ISDN\" as \"Dialing method\" sets the dialing command\n"
	 "to \"DI\", i.e. the complete dialing prefix will be \"ATDI\". \"Tone\"\n"
	 "corresponds to prefix \"ATDT\" and \"Pulse\" to \"ATDP\". If you need\n"
	 "to enter extra option characters between \"AT\" and e.g. \"DT\", use\n"
	 "the \"Dialing extras\" input field.");
}

void doModemInitInput(FL_OBJECT *obj, long param)
{
	static char init[MAXLEN_MDMCMD+1] = {0};

	strncpy(init, (char *)fl_get_input(obj), MAXLEN_MDMCMD);
	if (! strlen(init)) {
		XBell(fl_get_display(), 50);
		strcpy(init, MODEM_INIT);
		fl_set_input(obj, init);
		fl_redraw_object(obj);
	}
	fd_commInfo->cdata = init;
	return;
}

void doModemResetInput(FL_OBJECT *obj, long param)
{
	static char reset[MAXLEN_MDMCMD+1] = {0};

	strncpy(reset, (char *)fl_get_input(obj), MAXLEN_MDMCMD);
	if (! strlen(reset)) {
		XBell(fl_get_display(), 50);
		strcpy(reset, MODEM_RESET);
		fl_set_input(obj, reset);
		fl_redraw_object(obj);
	}
	fd_commInfo->cdata = reset;
	return;
}

void doModemConnectInput(FL_OBJECT *obj, long param)
{
	static char connect[MAXLEN_MDMSTR+1] = {0};

	strncpy(connect, (char *)fl_get_input(obj), MAXLEN_MDMSTR);
	if (! strlen(connect)) {
		XBell(fl_get_display(), 50);
		strcpy(connect, MODEM_CONNECT);
		fl_set_input(obj, connect);
		fl_redraw_object(obj);
	}
	fd_commInfo->cdata = connect;
	return;
}

void doToneButton(FL_OBJECT *obj, long param)
{
	commInfo_t *c = &IFC_commInfo;

	if (fl_get_button(fd_commInfo->toneButton)) {
		c->operOpts |= MODEM_TONEDIAL;
		c->operOpts &= ~MODEM_ISDNDIAL;
	}
	return;
}

void doPulseButton(FL_OBJECT *obj, long param)
{
	commInfo_t *c = &IFC_commInfo;

	if (fl_get_button(fd_commInfo->pulseButton)) {
		c->operOpts &= ~MODEM_TONEDIAL;
		c->operOpts &= ~MODEM_ISDNDIAL;
	}
	return;
}

void doISDNButton(FL_OBJECT *obj, long param)
{
	commInfo_t *c = &IFC_commInfo;

	if (fl_get_button(fd_commInfo->ISDNButton)) {
		c->operOpts &= ~MODEM_TONEDIAL;
		c->operOpts |= MODEM_ISDNDIAL;
	}
	return;
}

void doModemDialInput(FL_OBJECT *obj, long param)
{
	static char extra[MAXLEN_DIALEXTRA+1] = {0};

	strncpy(extra, (char *)fl_get_input(obj), MAXLEN_DIALEXTRA);
	if (! strlen(extra)) {
		strcpy(extra, DIAL_EXTRA);
		fl_set_input(obj, extra);
		fl_redraw_object(obj);
	}
	fd_commInfo->cdata = extra;
	return;
}

void doSWCBSDButton(FL_OBJECT *obj, long param)
{
	commInfo_t *c = &IFC_commInfo;

	if (fl_get_button(fd_commInfo->SWCBSDButton)) {
		c->operOpts |= BSD_COMPRESS;
		c->operOpts &= ~DEFL_COMPRESS;
		fl_activate_object(fd_commInfo->SWCInput);
		fl_set_object_lcol(fd_commInfo->SWCInput,FL_WHITE);
	}
	return;
}

void doSWCDeflateButton(FL_OBJECT *obj, long param)
{
	commInfo_t *c = &IFC_commInfo;

	if (fl_get_button(fd_commInfo->SWCDeflateButton)) {
		c->operOpts |= DEFL_COMPRESS;
		c->operOpts &= ~BSD_COMPRESS;
		fl_activate_object(fd_commInfo->SWCInput);
		fl_set_object_lcol(fd_commInfo->SWCInput,FL_WHITE);
	}
	return;
}

void doSWCOffButton(FL_OBJECT *obj, long param)
{
	commInfo_t *c = &IFC_commInfo;

	if (fl_get_button(fd_commInfo->SWCOffButton)) {
		c->operOpts &= ~BSD_COMPRESS;
		c->operOpts &= ~DEFL_COMPRESS;
		fl_deactivate_object(fd_commInfo->SWCInput);
		fl_set_object_lcol(fd_commInfo->SWCInput,FL_INACTIVE);
	}
	return;
}

void doSWCInput(FL_OBJECT *obj, long param)
{
	char level[MAXDIG_BSDCOMP+1] = {0};
	static unsigned char num = COMP_LEVEL;
	commInfo_t *c = &IFC_commInfo;

	strncpy(level, (char *)fl_get_input(obj), MAXDIG_BSDCOMP);
	num = atoi(level);
	if ((c->operOpts & (BSD_COMPRESS|DEFL_COMPRESS)) &&
		(num < MIN_COMPLEVEL || num > MAX_COMPLEVEL)) {
		XBell(fl_get_display(), 50);
		num = COMP_LEVEL;
		sprintf(level, "%d", num);
		fl_set_input(obj, level);
		fl_redraw_object(obj);
	}
	fd_commInfo->vdata = &num;
	return;
}

int SWCInputFilter(FL_OBJECT *obj, const char *old, const char *cur, int c)
{
	int test;

	test = atoi(cur);
	if (c == '-' || test > 15)
		return FL_INVALID|FL_RINGBELL;
	else
		return FL_VALID;
}

void doDeviceInput(FL_OBJECT *obj, long param)
{
	static char device[MAXLEN_DEVICE+1] = {0};

	strncpy(device, (char *)fl_get_input(obj), MAXLEN_DEVICE);
	if (! strlen(device)) {
		XBell(fl_get_display(), 50);
		strcpy(device, MODEM_DEVICE);
		fl_set_input(obj, device);
		fl_redraw_object(obj);
	}
	fd_commInfo->cdata = device;
	return;
}

void updateKnownSpeed(commInfo_t *c, unsigned int speed)
{
	char speed_string[MAXDIG_BAUDRATE+1] = {0};

	c->speed = speed;
	if (c->custom) {
		c->custom = 0;
		fl_deactivate_object(fd_commInfo->customSpeedInput);
	}
	sprintf(speed_string, "%u", c->speed);
	fl_set_input(fd_commInfo->customSpeedInput, speed_string);
}

void doSpeedButton(FL_OBJECT *obj, long param)
{
	commInfo_t *c = &IFC_commInfo;

	switch (param) {
		case 0: updateKnownSpeed(c, 1200); break;
		case 1: updateKnownSpeed(c, 2400); break;
		case 2: updateKnownSpeed(c, 4800); break;
		case 3: updateKnownSpeed(c, 9600); break;
		case 4: updateKnownSpeed(c, 19200); break;
		case 5: updateKnownSpeed(c, 38400); break;
		case 6: updateKnownSpeed(c, 57600); break;
		case 7: updateKnownSpeed(c, 115200); break;

		case 8:
			c->custom = 1;
			fl_activate_object(fd_commInfo->customSpeedInput);
			break;

		default: break;
	}
	return;
}

void doCustomSpeedInput(FL_OBJECT *obj, long param)
{
	char speed_string[MAXDIG_BAUDRATE+1] = {0};
	commInfo_t *c = &IFC_commInfo;

	strncpy(speed_string, (char *)fl_get_input(obj), MAXDIG_BAUDRATE);
	c->speed = atoi(speed_string);
	if (c->custom && (c->speed<MIN_BAUDRATE || c->speed>MAX_BAUDRATE)) {
		XBell(fl_get_display(), 50);
		c->speed = MODEM_SPEED;
		sprintf(speed_string, "%u", c->speed);
		fl_set_input(obj, speed_string);
		fl_redraw_object(obj);
	}
	return;
}

void doAsyncmapInput(FL_OBJECT *obj, long param)
{
	static char asyncmap[MAXDIG_ASYNCMAP+1] = {0};

	strncpy(asyncmap, (char *)fl_get_input(obj), MAXDIG_ASYNCMAP);
	if (! strlen(asyncmap)) {
		if (p_xisprc->operOpts & HW_FLOWCTRL)
			strcpy(asyncmap, PPPD_HASYNCMAP);
		else
			strcpy(asyncmap, PPPD_SASYNCMAP);
		fl_set_input(obj, asyncmap);
		fl_redraw_object(obj);
	}
	fd_commInfo->cdata = asyncmap;
	return;
}

int AMInputFilter(FL_OBJECT *obj, const char *old, const char *cur, int c)
{
	if (strcmp(old, cur)) {
		if (!((c>='0' && c<='9') || (c>='A' && c<='F')))
			return FL_INVALID|FL_RINGBELL;
	}
	return FL_VALID;
}

void doEscapeYesButton(FL_OBJECT *obj, long param)
{
	commInfo_t *c = &IFC_commInfo;

	if (fl_get_button(fd_commInfo->escapeYesButton)) {
		c->operOpts |= ESCAPE_ON;
		fl_activate_object(fd_commInfo->escapeInput);
		fl_set_object_lcol(fd_commInfo->escapeInput,FL_WHITE);
	}
	return;
}

void doEscapeNoButton(FL_OBJECT *obj, long param)
{
	commInfo_t *c = &IFC_commInfo;

	if (fl_get_button(fd_commInfo->escapeNoButton)) {
		c->operOpts &= ~ESCAPE_ON;
		fl_deactivate_object(fd_commInfo->escapeInput);
		fl_set_object_lcol(fd_commInfo->escapeInput,FL_INACTIVE);
	}
	return;
}

void doEscapeInput(FL_OBJECT *obj, long param)
{
	static char escape[MAXLEN_ESCAPE+1] = {0};

	strncpy(escape, (char *)fl_get_input(obj), MAXLEN_ESCAPE);
	if (! strlen(escape)) {
		XBell(fl_get_display(), 50);
		strcpy(escape, PPPD_ESCAPE);
		fl_set_input(obj, escape);
		fl_redraw_object(obj);
	}
	fd_commInfo->cdata = escape;
	return;
}

int EInputFilter(FL_OBJECT *obj, const char *old, const char *cur, int c)
{
	if (strcmp(old, cur)) {
		if (!((c>='0' && c<='9') || (c>='A' && c<='F') || c==','))
			return FL_INVALID|FL_RINGBELL;
	}
	return FL_VALID;
}

void doHWButton(FL_OBJECT *obj, long param)
{
	static char asyncmap[MAXDIG_ASYNCMAP+1] = {0};
	commInfo_t *c = &IFC_commInfo;

	if (fl_get_button(fd_commInfo->HWButton)) {
		c->operOpts |= HW_FLOWCTRL;
		strcpy(asyncmap, PPPD_HASYNCMAP);
		fl_set_input(fd_commInfo->asyncmapInput, asyncmap);
		fl_redraw_object(obj);
	}
	return;
}

void doSWButton(FL_OBJECT *obj, long param)
{
	static char asyncmap[MAXDIG_ASYNCMAP+1] = {0};
	commInfo_t *c = &IFC_commInfo;

	if (fl_get_button(fd_commInfo->SWButton)) {
		c->operOpts &= ~HW_FLOWCTRL;
		strcpy(asyncmap, PPPD_SASYNCMAP);
		fl_set_input(fd_commInfo->asyncmapInput, asyncmap);
		fl_redraw_object(obj);
	}
	return;
}

void doCommInfoOK(FL_OBJECT *obj, long done)
{
	char msg[512] = {0}, *p;
	commInfo_t *c = &IFC_commInfo;

	fl_call_object_callback(fd_commInfo->deviceInput);
	if ((p = strrchr(fd_commInfo->cdata, '/')) == NULL)
		p = fd_commInfo->cdata;
	else
		++p;
	if (!nohints && (pppdVersion().major > 2 || pppdVersion().minor > 2)) {
		sprintf(msg, "Please make sure that a peer information\n"
				"file named xisp_%s exists in pppd's peers\n"
				"directory (most probably /etc/ppp/peers --\n"
				"read the pppd(8) manual page for details),\n"
				"and that it contains the following lines:\n\n"
				"               /dev/%s\n"
				"               noauth\n"
				"               call xisp_dialer", p, p);
		alertMessage("Setup hint", 1, 0, msg);
	}
	strcpy(p_xisprc->modemDevice, fd_commInfo->cdata);
	fl_call_object_callback(fd_commInfo->customSpeedInput);
	p_xisprc->modemSpeed = c->speed;
	fl_call_object_callback(fd_commInfo->modemInitInput);
	strcpy(p_xisprc->modemInit, fd_commInfo->cdata);
	fl_call_object_callback(fd_commInfo->modemResetInput);
	strcpy(p_xisprc->modemReset, fd_commInfo->cdata);
	fl_call_object_callback(fd_commInfo->modemConnectInput);
	strcpy(p_xisprc->modemConnect, fd_commInfo->cdata);
	p_xisprc->operOpts = c->operOpts;
	fl_call_object_callback(fd_commInfo->modemDialInput);
	strcpy(p_xisprc->dialExtra, fd_commInfo->cdata);
	fl_call_object_callback(fd_commInfo->SWCInput);
	p_xisprc->compLevel = *(unsigned char *)(fd_commInfo->vdata);
	fl_call_object_callback(fd_commInfo->asyncmapInput);
	strcpy(p_xisprc->asyncmap, fd_commInfo->cdata);
	fl_call_object_callback(fd_commInfo->escapeInput);
	strcpy(p_xisprc->escape, fd_commInfo->cdata);
	writeXisprc(rcfname, xispOptions, &global);
	if (done) {
		fl_hide_form(fd_optsTab->optsTab);
		fl_activate_object(fd_topFrame->fileMenu);
		fl_set_object_lcol(fd_topFrame->fileMenu,FL_LCOL);
	}
	return;
}

void doCommInfoCancel(FL_OBJECT *obj, long param)
{
	fl_hide_form(fd_optsTab->optsTab);
	fl_activate_object(fd_topFrame->fileMenu);
	fl_set_object_lcol(fd_topFrame->fileMenu,FL_LCOL);
	return;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                     Callback routines for tcpipInfo                     |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void doTcpipInfoTip(FL_OBJECT *obj, long param)
{
	alertMessage("TCP/IP setup hints", 1, 0,
	 "The default settings under \"Addressing/Routing\" should be\n"
	 "adequate for the majority of cases. The same comment holds\n"
	 "for \"Netmask\" and the \"MTU/MRU\" values; however in rare cases\n"
	 "your ISP might require \"MTU/MRU\" values other than the default.\n"
	 "\n"
	 "For the DNS support to work, the ip-up and ip-down scripts\n"
	 "distributed with xisp must have been installed properly on your\n"
	 "system. If that is indeed the case, the DNS server(s) you enter\n"
	 "here will be automatically available whenever you connect to\n"
	 "the net using the ISP account they are defined for. If the auto-\n"
	 "matic (ISP assigned) DNS server(s) capability is greyed out,\n"
	 "this means that your pppd version is older than 2.3.9; to make\n"
	 "use of this feature you must upgrade pppd.");
}

void doLocalIPInput(FL_OBJECT *obj, long param)
{
	static unsigned char ip[4] = LOCAL_IP, ips[MAXLEN_IP+1];

	strncpy(ips, (char *)fl_get_input(obj), MAXLEN_IP);
	if (StrToIP(ips, ip) != 4) {
		XBell(fl_get_display(), 50);
		memcpy(ip, LOCAL_IP, 4);
		fl_set_input(obj, LOCAL_IPSTR);
		fl_redraw_object(obj);
	}
	fd_tcpipInfo->vdata = ip;
	return;
}

void doALYesButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_tcpipInfo->ALYesButton)) {
		fd_tcpipInfo->ldata |= ACCEPT_LOCALIP;
		fl_deactivate_object(fd_tcpipInfo->localIPInput);
		fl_set_object_lcol(fd_tcpipInfo->localIPInput,FL_INACTIVE);
	}
	return;
}

void doALNoButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_tcpipInfo->ALNoButton)) {
		fd_tcpipInfo->ldata &= ~ACCEPT_LOCALIP;
		fl_activate_object(fd_tcpipInfo->localIPInput);
		fl_set_object_lcol(fd_tcpipInfo->localIPInput,FL_WHITE);
	}
	return;
}

void doRemoteIPInput(FL_OBJECT *obj, long param)
{
	static unsigned char ip[4] = REMOTE_IP, ips[MAXLEN_IP+1];

	strncpy(ips, (char *)fl_get_input(obj), MAXLEN_IP);
	if (StrToIP(ips, ip) != 4) {
		XBell(fl_get_display(), 50);
		memcpy(ip, REMOTE_IP, 4);
		fl_set_input(obj, REMOTE_IPSTR);
		fl_redraw_object(obj);
	}
	fd_tcpipInfo->vdata = ip;
	return;
}

void doARYesButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_tcpipInfo->ARYesButton)) {
		fd_tcpipInfo->ldata |= ACCEPT_REMOTEIP;
		fl_deactivate_object(fd_tcpipInfo->remoteIPInput);
		fl_set_object_lcol(fd_tcpipInfo->remoteIPInput,FL_INACTIVE);
	}
	return;
}

void doARNoButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_tcpipInfo->ARNoButton)) {
		fd_tcpipInfo->ldata &= ~ACCEPT_REMOTEIP;
		fl_activate_object(fd_tcpipInfo->remoteIPInput);
		fl_set_object_lcol(fd_tcpipInfo->remoteIPInput,FL_WHITE);
	}
	return;
}

void doNMInput(FL_OBJECT *obj, long param)
{
	static unsigned char ip[4] = NETMASK, ips[MAXLEN_IP+1];

	strncpy(ips, (char *)fl_get_input(obj), MAXLEN_IP);
	if (StrToIP(ips, ip) != 4) {
		XBell(fl_get_display(), 50);
		memcpy(ip, NETMASK, 4);
		fl_set_input(obj, NETMASKSTR);
		fl_redraw_object(obj);
	}
	fd_tcpipInfo->vdata = ip;
	return;
}

void doPDNSInput(FL_OBJECT *obj, long param)
{
	static unsigned char ip[4] = DNS, ips[MAXLEN_IP+1];

	strncpy(ips, (char *)fl_get_input(obj), MAXLEN_IP);
	if (StrToIP(ips, ip) != 4) {
		XBell(fl_get_display(), 50);
		memcpy(ip, DNS, 4);
		fl_set_input(obj, DNS_IPSTR);
		fl_redraw_object(obj);
	}
	fd_tcpipInfo->vdata = ip;
	return;
}

void doSDNSInput(FL_OBJECT *obj, long param)
{
	static unsigned char ip[4] = DNS, ips[MAXLEN_IP+1];

	strncpy(ips, (char *)fl_get_input(obj), MAXLEN_IP);
	if (StrToIP(ips, ip) != 4) {
		XBell(fl_get_display(), 50);
		memcpy(ip, DNS, 4);
		fl_set_input(obj, DNS_IPSTR);
		fl_redraw_object(obj);
	}
	fd_tcpipInfo->vdata = ip;
	return;
}

void doDRYesButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_tcpipInfo->DRYesButton))
		fd_tcpipInfo->ldata |= DEFAULT_ROUTE;
	return;
}

void doDRNoButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_tcpipInfo->DRNoButton))
		fd_tcpipInfo->ldata &= ~DEFAULT_ROUTE;
	return;
}

void doDNSNoButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_tcpipInfo->DNSNoButton)) {
		fd_tcpipInfo->ldata &= ~(IP_UPDOWN | AUTO_DNS);
		fl_deactivate_object(fd_tcpipInfo->pDNSInput);
		fl_set_object_lcol(fd_tcpipInfo->pDNSInput,FL_INACTIVE);
		fl_deactivate_object(fd_tcpipInfo->sDNSInput);
		fl_set_object_lcol(fd_tcpipInfo->sDNSInput,FL_INACTIVE);
		fl_deactivate_object(fd_tcpipInfo->DNTypeGroup);
		fl_set_object_lcol(fd_tcpipInfo->DNTypeGroup,FL_INACTIVE);
	}
	return;
}

void doDNSAutoButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_tcpipInfo->DNSAutoButton)) {
		fd_tcpipInfo->ldata |= (IP_UPDOWN | AUTO_DNS);
		fl_deactivate_object(fd_tcpipInfo->pDNSInput);
		fl_set_object_lcol(fd_tcpipInfo->pDNSInput,FL_INACTIVE);
		fl_deactivate_object(fd_tcpipInfo->sDNSInput);
		fl_set_object_lcol(fd_tcpipInfo->sDNSInput,FL_INACTIVE);
		fl_activate_object(fd_tcpipInfo->DNTypeGroup);
		fl_set_object_lcol(fd_tcpipInfo->DNTypeGroup,FL_WHITE);
	}
	return;
}

void doDNSManualButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_tcpipInfo->DNSManualButton)) {
		fd_tcpipInfo->ldata |= IP_UPDOWN;
		fd_tcpipInfo->ldata &= ~AUTO_DNS;
		fl_activate_object(fd_tcpipInfo->pDNSInput);
		fl_set_object_lcol(fd_tcpipInfo->pDNSInput,FL_WHITE);
		fl_activate_object(fd_tcpipInfo->sDNSInput);
		fl_set_object_lcol(fd_tcpipInfo->sDNSInput,FL_WHITE);
		fl_activate_object(fd_tcpipInfo->DNTypeGroup);
		fl_set_object_lcol(fd_tcpipInfo->DNTypeGroup,FL_WHITE);
	}
	return;
}

void doDNYesButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_tcpipInfo->DNYesButton)) {
		fd_tcpipInfo->ldata |= DEFAULT_DOMAIN;
		fl_activate_object(fd_tcpipInfo->domainInput);
		fl_set_object_lcol(fd_tcpipInfo->domainInput,FL_WHITE);
	}
	return;
}

void doDNNoButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_tcpipInfo->DNNoButton)) {
		fd_tcpipInfo->ldata &= ~DEFAULT_DOMAIN;
		fl_deactivate_object(fd_tcpipInfo->domainInput);
		fl_set_object_lcol(fd_tcpipInfo->domainInput,FL_INACTIVE);
	}
	return;
}

void doDomainInput(FL_OBJECT *obj, long param)
{
	static char domainname[MAXLEN_DNNAME+1] = {0};

	strncpy(domainname, (char *)fl_get_input(obj), MAXLEN_DNNAME);
	if ((fd_tcpipInfo->ldata & DEFAULT_DOMAIN) &&
		strlen(domainname) == 0)
	{
		XBell(fl_get_display(), 50);
		memset(domainname, 0, MAXLEN_DNNAME);
		fl_set_button(fd_tcpipInfo->DNNoButton, 1);
		fl_call_object_callback(fd_tcpipInfo->DNNoButton);
		fl_redraw_object(obj);
	}
	fd_tcpipInfo->cdata = domainname;
	return;
}

int IPInputFilter(FL_OBJECT *obj, const char *old, const char *cur, int c)
{
	int nd;

	if (strcmp(old, cur)) {
		if (!((c>='0' && c<='9') || c=='.'))
			return FL_INVALID|FL_RINGBELL;
		for (nd=0; *cur; ++cur) if (*cur == '.') ++nd;
		if (nd > 3)
			return FL_INVALID|FL_RINGBELL;
	}
	return FL_VALID;
}

void doMTUInput(FL_OBJECT *obj, long param)
{
	char mtu[MAXDIG_MTRU+1] = {0};
	static unsigned int num = MTU;

	strncpy(mtu, (char *)fl_get_input(obj), MAXDIG_MTRU);
	num = atoi(mtu);
	if (num<128 || num>2048) {
		XBell(fl_get_display(), 50);
		if (num < 128) num = 128;
		if (num > 2048) num = 2048;
		sprintf(mtu, "%d", num);
		fl_set_input(obj, mtu);
		fl_redraw_object(obj);
	}
	fd_tcpipInfo->vdata = &num;
	return;
}

void doMRUInput(FL_OBJECT *obj, long param)
{
	char mru[MAXDIG_MTRU+1] = {0};
	static unsigned int num = MRU;

	strncpy(mru, (char *)fl_get_input(obj), MAXDIG_MTRU);
	num = atoi(mru);
	if (num<128 || num>2048) {
		XBell(fl_get_display(), 50);
		if (num < 128) num = 128;
		if (num > 2048) num = 2048;
		sprintf(mru, "%d", num);
		fl_set_input(obj, mru);
		fl_redraw_object(obj);
	}
	fd_tcpipInfo->vdata = &num;
	return;
}

void doTcpipInfoOK(FL_OBJECT *obj, long done)
{
	fl_call_object_callback(fd_tcpipInfo->localIPInput);
	memcpy(p_xisprc->localIP, (char *)fd_tcpipInfo->vdata, 4);
	fl_call_object_callback(fd_tcpipInfo->remoteIPInput);
	memcpy(p_xisprc->remoteIP, (char *)fd_tcpipInfo->vdata, 4);
	fl_call_object_callback(fd_tcpipInfo->NMInput);
	memcpy(p_xisprc->netmask, (char *)fd_tcpipInfo->vdata, 4);
	fl_call_object_callback(fd_tcpipInfo->pDNSInput);
	memcpy(p_xisprc->dns1, (char *)fd_tcpipInfo->vdata, 4);
	fl_call_object_callback(fd_tcpipInfo->sDNSInput);
	memcpy(p_xisprc->dns2, (char *)fd_tcpipInfo->vdata, 4);
	p_xisprc->operOpts = (unsigned)fd_tcpipInfo->ldata;
	fl_call_object_callback(fd_tcpipInfo->MTUInput);
	p_xisprc->mtu = *(unsigned int *)(fd_tcpipInfo->vdata);
	fl_call_object_callback(fd_tcpipInfo->MRUInput);
	p_xisprc->mru = *(unsigned int *)(fd_tcpipInfo->vdata);
	fl_call_object_callback(fd_tcpipInfo->domainInput);
	strcpy(p_xisprc->domainname, fd_tcpipInfo->cdata);
	writeXisprc(rcfname, xispOptions, &global);
	if (done) {
		fl_hide_form(fd_optsTab->optsTab);
		fl_activate_object(fd_topFrame->fileMenu);
		fl_set_object_lcol(fd_topFrame->fileMenu,FL_LCOL);
	}
	return;
}

void doTcpipInfoCancel(FL_OBJECT *obj, long param)
{
	fl_hide_form(fd_optsTab->optsTab);
	fl_activate_object(fd_topFrame->fileMenu);
	fl_set_object_lcol(fd_topFrame->fileMenu,FL_LCOL);
	return;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |         Utility storage and routines for logInfo and pttEditor          |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

static discount_t reset_discount = {0.0,{0,0,0},{24,0,0}};
static rule_t rst_Urule = {RULE_WEEKALL,{0,1},{11,31},0,{0,0,0},{24,0,0},60.0},
			  rst_Trule = {RULE_WEEKALL,{0,1},{11,31},0,{0,0,0},{24,0,0},1.0};

void markByUnit(ptt_t *p)
{
	ptt_t *a;	/* to keep the compiler happy :) */

	a = p;
	fl_set_object_label(fd_pttEditor->defaultTariffUnits, CUNIT_STR);
	fl_set_object_label(fd_pttEditor->ruleTariffUnits, CUNIT_STR);
	fl_deactivate_object(fd_pttEditor->TimeChargeType);
	fl_set_object_lcol(fd_pttEditor->TimeChargeType, FL_INACTIVE);
	fl_activate_object(fd_pttEditor->minUnitsInput);
	fl_set_object_lcol(fd_pttEditor->minUnitsInput, FL_WHITE);
	fl_deactivate_object(fd_pttEditor->minCostInput);
	fl_set_object_lcol(fd_pttEditor->minCostInput, FL_INACTIVE);
	fl_deactivate_object(fd_pttEditor->CSecondsInput);
	fl_set_object_lcol(fd_pttEditor->CSecondsInput, FL_INACTIVE);
	fl_deactivate_object(fd_pttEditor->REDType);
	fl_set_object_lcol(fd_pttEditor->REDType, FL_INACTIVE);
	fl_deactivate_object(fd_pttEditor->ZMLType);
	fl_set_object_lcol(fd_pttEditor->ZMLType, FL_INACTIVE);
	fl_deactivate_object(fd_pttEditor->ZEDType);
	fl_set_object_lcol(fd_pttEditor->ZEDType, FL_INACTIVE);
	fl_deactivate_object(fd_pttEditor->MCType);
	fl_set_object_lcol(fd_pttEditor->MCType, FL_INACTIVE);
	fl_set_button(fd_pttEditor->CUnitButton, 1);
	fl_activate_object(fd_pttEditor->unitPriceInput);
	fl_set_object_lcol(fd_pttEditor->unitPriceInput, FL_WHITE);
}

void markByTime(ptt_t *p)
{
	char msg[64+MAXLEN_CURRENCY] = {0};

	sprintf(msg, CTIME_STR, p->currency);
	fl_set_object_label(fd_pttEditor->defaultTariffUnits, msg);
	fl_set_object_label(fd_pttEditor->ruleTariffUnits, msg);
	fl_activate_object(fd_pttEditor->TimeChargeType);
	fl_set_object_lcol(fd_pttEditor->TimeChargeType, FL_WHITE);
	fl_activate_object(fd_pttEditor->minCostInput);
	fl_set_object_lcol(fd_pttEditor->minCostInput, FL_WHITE);
	fl_deactivate_object(fd_pttEditor->minUnitsInput);
	fl_set_object_lcol(fd_pttEditor->minUnitsInput, FL_INACTIVE);
	if (p->attribs & PTT_PER_SECS) {
		fl_activate_object(fd_pttEditor->CSecondsInput);
		fl_set_object_lcol(fd_pttEditor->CSecondsInput, FL_WHITE);
	}
	else {
		fl_deactivate_object(fd_pttEditor->CSecondsInput);
		fl_set_object_lcol(fd_pttEditor->CSecondsInput, FL_INACTIVE);
	}
	fl_activate_object(fd_pttEditor->REDType);
	fl_set_object_lcol(fd_pttEditor->REDType, FL_WHITE);
	fl_activate_object(fd_pttEditor->ZMLType);
	fl_set_object_lcol(fd_pttEditor->ZMLType, FL_WHITE);
	fl_activate_object(fd_pttEditor->ZEDType);
	fl_set_object_lcol(fd_pttEditor->ZEDType, FL_WHITE);
	fl_activate_object(fd_pttEditor->MCType);
	fl_set_object_lcol(fd_pttEditor->MCType, FL_WHITE);
	if (p->attribs & PTT_PER_MINUTE)
		fl_set_button(fd_pttEditor->CMinuteButton, 1);
	else
		fl_set_button(fd_pttEditor->CSecondsButton, 1);
	fl_deactivate_object(fd_pttEditor->unitPriceInput);
	fl_set_object_lcol(fd_pttEditor->unitPriceInput, FL_INACTIVE);
}

void updateCharges(ptt_t *p)
{
	char ns[16] = {0};

	fl_set_button(fd_pttEditor->CUnitButton, 0);
	fl_set_button(fd_pttEditor->CMinuteButton, 0);
	fl_set_button(fd_pttEditor->CSecondsButton, 0);
	if (p->attribs & PTT_BY_UNIT)
		markByUnit(p);
	else
		markByTime(p);
	fl_set_button(fd_pttEditor->CPLButton, 0);
	fl_set_button(fd_pttEditor->CPRButton, 0);
	if (p->attribs & PTT_CUR_AFTER_COST)
		fl_set_button(fd_pttEditor->CPRButton, 1);
	else
		fl_set_button(fd_pttEditor->CPLButton, 1);
	sprintf(ns, "%d", p->min_units);
	fl_set_input(fd_pttEditor->minUnitsInput, ns);
	sprintf(ns, "%g", p->min_cost);
	fl_set_input(fd_pttEditor->minCostInput, ns);
	sprintf(ns, "%d", p->charge_period);
	fl_set_input(fd_pttEditor->CSecondsInput, ns);
	sprintf(ns, "%g", p->cost_quantum);
	fl_set_input(fd_pttEditor->unitPriceInput, ns);
	fl_set_input(fd_pttEditor->currencyInput, p->currency);
	sprintf(ns, "%d", p->decimals);
	fl_set_input(fd_pttEditor->decimalsInput, ns);
}

void updateZone(ptt_t *p)
{
	char ns[16] = {0};

	fl_set_button(fd_pttEditor->MLLButton, 0);
	fl_set_button(fd_pttEditor->MLNButton, 0);
	if (p->attribs & PTT_NONLINEAR_MIN) {
		fl_set_button(fd_pttEditor->MLNButton, 1);
		sprintf(ns, "%g", p->min_cost_len[p->current_zone]);
		fl_set_input(fd_pttEditor->MLSecondsInput, ns);
		fl_activate_object(fd_pttEditor->MLSecondsInput);
		fl_set_object_lcol(fd_pttEditor->MLSecondsInput, FL_WHITE);
	}
	else {
		fl_set_button(fd_pttEditor->MLLButton, 1);
		fl_set_input(fd_pttEditor->MLSecondsInput, "0");
		fl_deactivate_object(fd_pttEditor->MLSecondsInput);
		fl_set_object_lcol(fd_pttEditor->MLSecondsInput, FL_INACTIVE);
	}
	fl_set_button(fd_pttEditor->ZEDYesButton, 0);
	fl_set_button(fd_pttEditor->ZEDNoButton, 0);
	if (p->discount[p->current_zone].percent > 0) {
		fl_set_button(fd_pttEditor->ZEDYesButton, 1);
		sprintf(ns, "%g", p->discount[p->current_zone].percent);
		fl_set_input(fd_pttEditor->discountInput, ns);
		fl_activate_object(fd_pttEditor->DiscountType);
		fl_set_object_lcol(fd_pttEditor->DiscountType, FL_WHITE);
	}
	else {
		fl_set_button(fd_pttEditor->ZEDNoButton, 1);
		fl_set_input(fd_pttEditor->discountInput, "0");
		fl_deactivate_object(fd_pttEditor->DiscountType);
		fl_set_object_lcol(fd_pttEditor->DiscountType, FL_INACTIVE);
	}
	sprintf(ns, "%g", p->dflt_tariff[p->current_zone]);
	fl_set_input(fd_pttEditor->baseChargeInput, ns);
	sprintf(ns, "%g", p->min_cost_len[p->current_zone]);
	fl_set_input(fd_pttEditor->MLSecondsInput, ns);
	sprintf(ns, "%g", p->discount[p->current_zone].percent);
	fl_set_input(fd_pttEditor->discountInput, ns);
	fl_set_input(fd_pttEditor->ZSHInput,
				 uc2ts(p->discount[p->current_zone].start, 'H'));
	fl_set_input(fd_pttEditor->ZSMInput,
				 uc2ts(p->discount[p->current_zone].start, 'M'));
	fl_set_input(fd_pttEditor->ZSSInput,
				 uc2ts(p->discount[p->current_zone].start, 'S'));
	fl_set_input(fd_pttEditor->ZEHInput,
				 uc2ts(p->discount[p->current_zone].end, 'H'));
	fl_set_input(fd_pttEditor->ZEMInput,
				 uc2ts(p->discount[p->current_zone].end, 'M'));
	fl_set_input(fd_pttEditor->ZESInput,
				 uc2ts(p->discount[p->current_zone].end, 'S'));
}

void updateRule(ptt_t *p, int reset)
{
	int rn;
	char ns[16] = {0};

	if (reset) {
		fl_set_counter_bounds(fd_pttEditor->ruleCounter,
							  1, (p->num_categories)? p->num_categories:1);
		fl_set_counter_value(fd_pttEditor->ruleCounter, 1);
		rn = 0;
	}
	else
		rn = fl_get_counter_value(fd_pttEditor->ruleCounter) - 1;
	fd_pttEditor->ldata = rn; /* save this for pttEditor call-back functions */
	fl_set_button(fd_pttEditor->RDYesButton, 0);
	fl_set_button(fd_pttEditor->RDNoButton, 0);
	if (p->rule[p->current_zone][rn].type & DISCOUNT_PERMITTED)
		fl_set_button(fd_pttEditor->RDYesButton, 1);
	else
		fl_set_button(fd_pttEditor->RDNoButton, 1);
	if (p->discount[p->current_zone].percent > 0) {
		fl_activate_object(fd_pttEditor->REDType);
		fl_set_object_lcol(fd_pttEditor->REDType, FL_WHITE);
		fl_set_object_lcol(fd_pttEditor->RDStr, FL_WHITE);
	}
	else {
		fl_deactivate_object(fd_pttEditor->REDType);
		fl_set_object_lcol(fd_pttEditor->REDType, FL_INACTIVE);
		fl_set_object_lcol(fd_pttEditor->RDStr, FL_INACTIVE);
	}
	fl_set_button(fd_pttEditor->TWDButton, 0);
	fl_set_button(fd_pttEditor->TWEButton, 0);
	fl_set_button(fd_pttEditor->TSDButton, 0);
	fl_set_button(fd_pttEditor->TUDButton, 0);
	fl_set_button(fd_pttEditor->THRButton, 0);
	fl_set_button(fd_pttEditor->THAButton, 0);
	fl_set_button(fd_pttEditor->TWDSButton, 0);
	fl_set_button(fd_pttEditor->TWESButton, 0);
	fl_set_button(fd_pttEditor->TAWButton, 0);
	fl_set_object_lcol(fd_pttEditor->ruleDateStr, FL_INACTIVE);
	fl_deactivate_object(fd_pttEditor->CalDateType);
	fl_set_object_lcol(fd_pttEditor->CalDateType, FL_INACTIVE);
	fl_deactivate_object(fd_pttEditor->EndDateType);
	fl_set_object_lcol(fd_pttEditor->EndDateType, FL_INACTIVE);
	fl_deactivate_object(fd_pttEditor->relDateInput);
	fl_set_object_lcol(fd_pttEditor->relDateInput, FL_INACTIVE);
	switch ((p->rule[p->current_zone][rn].type) & 0x3F) {
		case RULE_WEEKALL:
			fl_set_button(fd_pttEditor->TAWButton, 1);
			break;
		case RULE_WEEKDAY:
			fl_set_button(fd_pttEditor->TWDButton, 1);
			break;
		case RULE_SPECIAL_WEEKDAY:
			fl_set_button(fd_pttEditor->TWDSButton, 1);
			fl_set_object_lcol(fd_pttEditor->ruleDateStr, FL_WHITE);
			fl_activate_object(fd_pttEditor->CalDateType);
			fl_set_object_lcol(fd_pttEditor->CalDateType, FL_WHITE);
			fl_activate_object(fd_pttEditor->EndDateType);
			fl_set_object_lcol(fd_pttEditor->EndDateType, FL_WHITE);
			break;
		case RULE_WEEKEND:
			fl_set_button(fd_pttEditor->TWEButton, 1);
			break;
		case RULE_SPECIAL_WEEKEND:
			fl_set_button(fd_pttEditor->TWDSButton, 1);
			fl_set_object_lcol(fd_pttEditor->ruleDateStr, FL_WHITE);
			fl_activate_object(fd_pttEditor->CalDateType);
			fl_set_object_lcol(fd_pttEditor->CalDateType, FL_WHITE);
			fl_activate_object(fd_pttEditor->EndDateType);
			fl_set_object_lcol(fd_pttEditor->EndDateType, FL_WHITE);
			fl_set_button(fd_pttEditor->TWESButton, 1);
			break;
		case RULE_SATURDAY:
			fl_set_button(fd_pttEditor->TSDButton, 1);
			break;
		case RULE_SUNDAY:
			fl_set_button(fd_pttEditor->TUDButton, 1);
			break;
		case RULE_RHOLIDAY:
			fl_set_button(fd_pttEditor->THRButton, 1);
			fl_set_object_lcol(fd_pttEditor->ruleDateStr, FL_WHITE);
			fl_activate_object(fd_pttEditor->relDateInput);
			fl_set_object_lcol(fd_pttEditor->relDateInput, FL_WHITE);
			break;
		case RULE_AHOLIDAY:
			fl_set_button(fd_pttEditor->THAButton, 1);
			fl_set_object_lcol(fd_pttEditor->ruleDateStr, FL_WHITE);
			fl_activate_object(fd_pttEditor->CalDateType);
			fl_set_object_lcol(fd_pttEditor->CalDateType, FL_WHITE);
			break;
		default: break;
	}
	fl_set_button(fd_pttEditor->ZMLYesButton, 0);
	fl_set_button(fd_pttEditor->ZMLNoButton, 0);
	if (p->rule[p->current_zone][rn].type & DEFEAT_MIN_TIME_LEN)
		fl_set_button(fd_pttEditor->ZMLYesButton, 1);
	else
		fl_set_button(fd_pttEditor->ZMLNoButton, 1);
	if (p->attribs & PTT_NONLINEAR_MIN) {
		fl_activate_object(fd_pttEditor->ZMLType);
		fl_set_object_lcol(fd_pttEditor->ZMLType, FL_WHITE);
		fl_set_object_lcol(fd_pttEditor->ZMLStr, FL_WHITE);
	}
	else {
		fl_deactivate_object(fd_pttEditor->ZMLType);
		fl_set_object_lcol(fd_pttEditor->ZMLType, FL_INACTIVE);
		fl_set_object_lcol(fd_pttEditor->ZMLStr, FL_INACTIVE);
	}
	sprintf(ns, "%g", p->rule[p->current_zone][rn].tariff);
	fl_set_input(fd_pttEditor->ruleChargeInput, ns);
	fl_set_input(fd_pttEditor->RDMInput,
				 uc2ds(p->rule[p->current_zone][rn].adate, 'M'));
	fl_set_input(fd_pttEditor->RDDInput,
				 uc2ds(p->rule[p->current_zone][rn].adate, 'D'));
	fl_set_input(fd_pttEditor->RDEMInput,
				 uc2ds(p->rule[p->current_zone][rn].edate, 'M'));
	fl_set_input(fd_pttEditor->RDEDInput,
				 uc2ds(p->rule[p->current_zone][rn].edate, 'D'));
	sprintf(ns, "%d", p->rule[p->current_zone][rn].rdate);
	fl_set_input(fd_pttEditor->relDateInput, ns);
	fl_set_input(fd_pttEditor->RSHInput,
				 uc2ts(p->rule[p->current_zone][rn].start, 'H'));
	fl_set_input(fd_pttEditor->RSMInput,
				 uc2ts(p->rule[p->current_zone][rn].start, 'M'));
	fl_set_input(fd_pttEditor->RSSInput,
				 uc2ts(p->rule[p->current_zone][rn].start, 'S'));
	fl_set_input(fd_pttEditor->REHInput,
				 uc2ts(p->rule[p->current_zone][rn].end, 'H'));
	fl_set_input(fd_pttEditor->REMInput,
				 uc2ts(p->rule[p->current_zone][rn].end, 'M'));
	fl_set_input(fd_pttEditor->RESInput,
				 uc2ts(p->rule[p->current_zone][rn].end, 'S'));
}

int intInput(FL_OBJECT *obj, int min, int max, char *fmt)
{
	static char str[32];
	int num;

	strncpy(str, (char *)fl_get_input(obj), 31);
	num = atoi(str);
	if (*str==0 || num<min || num>max) {
		XBell(fl_get_display(), 50);
		if (num < min) num = min;
		if (num > max) num = max;
	}
	sprintf(str, fmt, num);
	fl_set_input(obj, str);
	fl_redraw_object(obj);
	return num;
}

float floatInput(FL_OBJECT *obj, float min, float max, char *fmt)
{
	char str[32] = {0}, *endp;
	float num;

	strncpy(str, (char *)fl_get_input(obj), 31);
	num = strtod(str, &endp);
	if (*str==0 || num<min || ((max-min)>1.e-6 && num>max) || endp==str) {
		XBell(fl_get_display(), 50);
		if (num < min) num = min;
		if (num > max) num = max;
	}
	sprintf(str, fmt, num);
	fl_set_input(obj, str);
	fl_redraw_object(obj);
	return num;
}

/*+-------------------------------------------------------------------------+
  |                                                                         |
  |               Callback routines for logInfo and renamePTT               |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void doOLTimeButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_logInfo->OLTimeButton))
		fd_logInfo->ldata &= ~COST_READOUT;
	return;
}

void doOLChargeButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_logInfo->OLChargeButton))
		fd_logInfo->ldata |= COST_READOUT;
	return;
}

void doLogNoButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_logInfo->logNoButton)) {
		fd_logInfo->ldata |=  LOG_NONE;
		fd_logInfo->ldata &= ~LOG_WEEKLY;
		fd_logInfo->ldata &= ~LOG_MONTHLY;
		fd_logInfo->ldata &= ~LOG_BIMONTHLY;
	}
	return;
}

void doLogWeekButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_logInfo->logWeekButton)) {
		fd_logInfo->ldata &= ~LOG_NONE;
		fd_logInfo->ldata |=  LOG_WEEKLY;
		fd_logInfo->ldata &= ~LOG_MONTHLY;
		fd_logInfo->ldata &= ~LOG_BIMONTHLY;
	}
	return;
}

void doLogMonthButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_logInfo->logMonthButton)) {
		fd_logInfo->ldata &= ~LOG_NONE;
		fd_logInfo->ldata &= ~LOG_WEEKLY;
		fd_logInfo->ldata |=  LOG_MONTHLY;
		fd_logInfo->ldata &= ~LOG_BIMONTHLY;
	}
	return;
}

void doLogBiMonthButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_logInfo->logBiMonthButton)) {
		fd_logInfo->ldata &= ~LOG_NONE;
		fd_logInfo->ldata &= ~LOG_WEEKLY;
		fd_logInfo->ldata &= ~LOG_MONTHLY;
		fd_logInfo->ldata |=  LOG_BIMONTHLY;
	}
	return;
}

void doPttDropChoice(FL_OBJECT *obj, long param)
{
	unsigned char sPTT = ((unsigned)fd_logInfo->ldata & 0x00FF0000) >> 16;
	int c, i;
	ptt_t *p;

	if ((c = fl_get_choice(obj)) && --c != sPTT) {
		fd_logInfo->ldata &= 0x0FF00FFFF;
		fd_logInfo->ldata |= ((unsigned)c << 16);
		p = &ptt[c];
		fl_clear_choice(fd_logInfo->zoneDropChoice);
		for (i=0; i<p->num_zones; i++)
			fl_addto_choice(fd_logInfo->zoneDropChoice,
							clipStr(p->zone_name[i], FL_NORMAL_SIZE,
									FL_NORMAL_STYLE, 240));
		fd_logInfo->ldata &= 0x00FFFFFF;
		fl_set_choice(fd_logInfo->zoneDropChoice, 1);
		showPttInfo(p);
	}
	return;
}

void doZoneDropChoice(FL_OBJECT *obj, long param)
{
	unsigned char cZone = ((unsigned)fd_logInfo->ldata & 0x0FF000000) >> 24;
	int c;

	if ((c = fl_get_choice(obj)) && --c != cZone) {
		fd_logInfo->ldata &= 0x00FFFFFF;
		fd_logInfo->ldata |= ((unsigned)c << 24);
	}
	return;
}

void doLogInfoPTTEdit(FL_OBJECT *obj, long param)
{
	unsigned char sPTT = ((unsigned)fd_logInfo->ldata & 0x00FF0000) >> 16;
	unsigned char cZone = ((unsigned)fd_logInfo->ldata & 0x0FF000000) >> 24;
	ptt_t *p;
	int z;
	char ns[16] = {0};

	fl_deactivate_form(fd_logInfo->logInfo);
	p = &ptt[sPTT];
	p->current_zone = cZone;
	IFC_pttEditor = *p;
	fl_freeze_form(fd_pttEditor->pttEditor);
	updateCharges(p);
	updateZone(p);
	updateRule(p, 1);
	fl_unfreeze_form(fd_pttEditor->pttEditor);
	fl_set_input(fd_pttEditor->pttNameInput, p->name);
	sprintf(ns, "%d", p->num_zones);
	fl_set_input(fd_pttEditor->zonesInput, ns);
	sprintf(ns, "%d", p->num_categories);
	fl_set_input(fd_pttEditor->categoriesInput, ns);
	fl_show_form(fd_pttEditor->pttEditor, FL_PLACE_MOUSE,
				 FL_TRANSIENT, "PTT Editor");
	fl_clear_browser(fd_pttEditor->zonePick);
	for (z=0; z<p->num_zones; z++)
		fl_add_browser_line(fd_pttEditor->zonePick, p->zone_name[z]);
	fl_select_browser_line(fd_pttEditor->zonePick, 1+cZone);
	fl_set_browser_topline(fd_pttEditor->zonePick, 1+cZone);
	return;
}

void doLogInfoPTTAdd(FL_OBJECT *obj, long param)
{
	fl_deactivate_form(fd_logInfo->logInfo);
	fl_show_form(fd_renamePTT->renamePTT, FL_PLACE_MOUSE,
				 FL_TRANSIENT, "Add new PTT");
	return;
}

void doLogInfoPTTRemove(FL_OBJECT *obj, long param)
{
	ptt_t *p_dest, *p_src, *p;
	unsigned char sPTT = ((unsigned)fd_logInfo->ldata & 0x00FF0000) >> 16;
	void gr_epak_initPtt();		/* OTE A.E. (EAK), Hellas (default PTT) */
	char msg[512] = {0};
	int i;

	if (! expert) {
		sprintf(msg, "Are you sure you want to delete PTT\n\"%s\"?",
				ptt[sPTT].name);
		if (! actionVerify(msg, 0))
			return;
	}
	if (sPTT < (global.numPTTs-1)) {
		p_dest = &ptt[sPTT];
		p_src = &ptt[sPTT+1];
#ifdef SUNOS41x
		bcopy(p_src, p_dest, (global.numPTTs-sPTT-1)*sizeof(ptt_t));
#else
		memmove(p_dest, p_src, (global.numPTTs-sPTT-1)*sizeof(ptt_t));
#endif
	}
	else if (sPTT)
		--sPTT;

	fd_logInfo->ldata &= 0x0FF00FFFF;
	fd_logInfo->ldata |= (sPTT << 16);

	if (global.numPTTs > 1) {
		global.numPTTs -= 1;
		p = realloc(ptt, global.numPTTs*sizeof(ptt_t));
		if (p == NULL)
			outofMem();
		ptt = p;
	}
	else {
		p = ptt;
		initXispPTT(p);
		gr_epak_initPtt(p);
		sprintf(msg, "You have deleted the last entry in the PTT database!\n"
				"Reverting to default PTT: %s\n", p->name);
		alertMessage("xISP: doLogInfoPTTRemove()", 0, 0, msg);
	}
	fd_logInfo->vdata = (void *)1;	/* indicate change in PTT database */
	fl_clear_choice(fd_logInfo->pttDropChoice);
	for (i=0; i<global.numPTTs; i++)
		fl_addto_choice(fd_logInfo->pttDropChoice,
						clipStr(ptt[i].name, FL_NORMAL_SIZE,
								FL_NORMAL_STYLE, 240));
	fl_set_choice(fd_logInfo->pttDropChoice, sPTT+1);
	fl_clear_choice(fd_logInfo->zoneDropChoice);
	for (i=0, p=&ptt[sPTT]; i<p->num_zones; i++)
		fl_addto_choice(fd_logInfo->zoneDropChoice,
						clipStr(p->zone_name[i], FL_NORMAL_SIZE,
								FL_NORMAL_STYLE, 240));
	fl_set_choice(fd_logInfo->zoneDropChoice,p->current_zone+1);
	showPttInfo(p);
	return;
}

void doLogInfoOK(FL_OBJECT *obj, long param)
{
	unsigned short opts = (unsigned)fd_logInfo->ldata & 0x00FFFF;
	unsigned char sPTT = ((unsigned)fd_logInfo->ldata & 0x00FF0000) >> 16;
	int newCurrency = strcmp(fd_logInfo->cdata, ptt[sPTT].currency);

	if (! (opts & 0x0F & LOG_NONE)) {
		if (newCurrency &&
			! actionVerify("The PTT selected bills you in currency which "
						   "is\ndifferent from the one currently in use. "
						   "This implies\nresetting of log files, continue?",0)
		   )
			return;
		if (newCurrency || ((opts & 0x0F) != (global.logOpts & 0x0F)))
			resetLogs();
	}
	checkUpdatePTTs(); /* make sure no deleted PTTs exist in ISP database */
	global.logOpts = opts;
	currentPTT = sPTT;
	global.costPTT = currentPTT;
	p_ptt = &ptt[currentPTT];
	p_ptt->current_zone = ((unsigned)fd_logInfo->ldata & 0x0FF000000) >> 24;
	p_xisprc->ispPTT = currentPTT;
	p_xisprc->ispZone = p_ptt->current_zone;
	writeXisprc(rcfname, xispOptions, &global);
	if (global.logOpts & COST_READOUT) {
		fl_set_object_label(fd_topFrame->logDescr, RCOST_STR);
		fl_set_object_label(fd_topFrame->logText, costStr(p_ptt, 0.0));
	}
	else {
		fl_set_object_label(fd_topFrame->logDescr, RTIME_STR);
		fl_set_object_label(fd_topFrame->logText, EMPTY_TIME);
	}
	if (global.logOpts & LOG_NONE)
		fl_set_menu_item_mode(fd_topFrame->logMenu, 2, FL_PUP_GREY);
	else
		fl_set_menu_item_mode(fd_topFrame->logMenu, 2, FL_PUP_NONE);
	if (fd_logInfo->vdata) {
		fl_set_cursor(fl_get_real_object_window(fd_logInfo->logGroup), tcid);
		fl_check_forms();
		writeXispPTTs(ptt, &global);
		fl_reset_cursor(fl_get_real_object_window(fd_logInfo->logGroup));
		fl_check_forms();
	}
	fl_hide_form(fd_logInfo->logInfo);
	fl_activate_object(fd_topFrame->logMenu);
	fl_set_object_lcol(fd_topFrame->logMenu,FL_LCOL);
	return;
}

void doLogInfoCancel(FL_OBJECT *obj, long param)
{
	if (fd_logInfo->vdata) {
		fl_set_cursor(fl_get_real_object_window(fd_logInfo->logGroup), tcid);
		fl_check_forms();
		free(ptt);
		readXispPTTs(&ptt, &global);
		p_ptt = &ptt[currentPTT];
		fl_reset_cursor(fl_get_real_object_window(fd_logInfo->logGroup));
		fl_check_forms();
	}
	fl_hide_form(fd_logInfo->logInfo);
	fl_activate_object(fd_topFrame->logMenu);
	fl_set_object_lcol(fd_topFrame->logMenu,FL_LCOL);
	return;
}

void doPTTNameInput(FL_OBJECT *obj, long param)
{
	static char name[MAXLEN_PTTNAME+1];

	strncpy(name, fl_get_input(obj), MAXLEN_PTTNAME);
	fd_renamePTT->cdata = name;
	return;
}

void doPTTNameEditOK(FL_OBJECT *obj, long param)
{
	unsigned char sPTT = global.numPTTs;
	ptt_t *p;
	int i;

	fd_logInfo->ldata &= 0x0FF00FFFF;
	fd_logInfo->ldata |= (sPTT << 16);
	p = realloc(ptt, (global.numPTTs+1)*sizeof(ptt_t));
	if (p == NULL)
		outofMem();
	else {
		ptt = p;
		global.numPTTs += 1;
		fd_logInfo->vdata = (void *)1;	/* indicate change in PTT database */
	}
	p = &ptt[sPTT];
	initXispPTT(p);
	fl_call_object_callback(fd_renamePTT->PTTNameInput);
	if (*(fd_renamePTT->cdata))
		strncpy(p->name, fd_renamePTT->cdata, MAXLEN_PTTNAME);
	else
		strcpy(p->name, "My new PTT");
	fl_clear_choice(fd_logInfo->pttDropChoice);
	for (i=0; i<global.numPTTs; i++)
		fl_addto_choice(fd_logInfo->pttDropChoice,
						clipStr(ptt[i].name, FL_NORMAL_SIZE,
								FL_NORMAL_STYLE, 240));
	fl_set_choice(fd_logInfo->pttDropChoice, sPTT+1);
	fl_clear_choice(fd_logInfo->zoneDropChoice);
	showPttInfo(p);
	fl_hide_form(fd_renamePTT->renamePTT);
	fl_activate_form(fd_logInfo->logInfo);
	return;
}

void doPTTNameEditCancel(FL_OBJECT *obj, long param)
{
	fl_hide_form(fd_renamePTT->renamePTT);
	fl_activate_form(fd_logInfo->logInfo);
	return;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                     Callback routines for statInfo                      |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void doStatsInfoOK(FL_OBJECT *obj, long param)
{
	fl_hide_form(fd_statInfo->statInfo);
	fl_activate_object(fd_topFrame->logMenu);
	fl_set_object_lcol(fd_topFrame->logMenu,FL_LCOL);
	return;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |            Callback routines for pttEditor and renameZone               |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void doCUnitButton(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;
	int z, c;

	if (fl_get_button(fd_pttEditor->CUnitButton)) {
		if (p->attribs & PTT_BY_UNIT)
			return;
		p->attribs &= ~(PTT_PER_MINUTE|PTT_PER_SECS);
		p->attribs |= PTT_BY_UNIT;
		p->min_units = 0;
		p->min_cost = 0.0;
		p->charge_period = 0;
		if (p->cost_quantum < 1.e-2)
			p->cost_quantum = 1.0;
		if (p->decimals < 1)
			p->decimals = 1;
		fl_freeze_form(fd_pttEditor->pttEditor);
		updateCharges(p);
		for (z=0; z<p->num_zones; z++) {
			if (p->dflt_tariff[z] < 1)
				p->dflt_tariff[z] = 60;
			p->min_cost_len[z] = 0.0;
			p->discount[z] = reset_discount;
			for (c=0; c<p->num_categories; c++)
				if (p->rule[z][c].tariff < 1)
					p->rule[z][c] = rst_Urule;
		}
		updateZone(p);
		updateRule(p, 0);
		fl_unfreeze_form(fd_pttEditor->pttEditor);
	}
	return;
}

void doCMinuteButton(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;
	int z, c;

	if (fl_get_button(fd_pttEditor->CMinuteButton)) {
		if (p->attribs & PTT_PER_MINUTE)
			return;
		p->attribs &= ~(PTT_BY_UNIT|PTT_PER_SECS);
		p->attribs |= PTT_PER_MINUTE;
		p->min_units = 0;
		p->min_cost = 0.0;
		p->charge_period = 60;
		p->cost_quantum = 0.0;
		if (p->decimals < 1)
			p->decimals = 2;
		fl_freeze_form(fd_pttEditor->pttEditor);
		updateCharges(p);
		for (z=0; z<p->num_zones; z++) {
			if (p->dflt_tariff[z] < 1.e-3)
				p->dflt_tariff[z] = 1;
			p->min_cost_len[z] = 0.0;
			p->discount[z] = reset_discount;
			for (c=0; c<p->num_categories; c++)
				if (p->rule[z][c].tariff < 1.e-3)
					p->rule[z][c] = rst_Trule;
		}
		updateZone(p);
		updateRule(p, 0);
		fl_unfreeze_form(fd_pttEditor->pttEditor);
	}
	return;
}

void doCSecondsButton(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;
	int z, c;

	if (fl_get_button(fd_pttEditor->CSecondsButton)) {
		if (p->attribs & PTT_PER_SECS)
			return;
		p->attribs &= ~(PTT_BY_UNIT|PTT_PER_MINUTE);
		p->attribs |= PTT_PER_SECS;
		p->min_units = 0;
		p->min_cost = 0.0;
		if (p->charge_period < 1)
			p->charge_period = 1;
		p->cost_quantum = 0.0;
		if (p->decimals < 1)
			p->decimals = 2;
		fl_freeze_form(fd_pttEditor->pttEditor);
		updateCharges(p);
		for (z=0; z<p->num_zones; z++) {
			if (p->dflt_tariff[z] < 1.e-3)
				p->dflt_tariff[z] = 1;
			p->min_cost_len[z] = 0.0;
			p->discount[z] = reset_discount;
			for (c=0; c<p->num_categories; c++)
				if (p->rule[z][c].tariff < 1.e-3)
					p->rule[z][c] = rst_Trule;
		}
		updateZone(p);
		updateRule(p, 0);
		fl_unfreeze_form(fd_pttEditor->pttEditor);
	}
	return;
}

void doCPLButton(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;

	if (fl_get_button(fd_pttEditor->CPLButton))
		p->attribs &= ~PTT_CUR_AFTER_COST;
	return;
}

void doCPRButton(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;

	if (fl_get_button(fd_pttEditor->CPRButton))
		p->attribs |= PTT_CUR_AFTER_COST;
	return;
}

void doMLLButton(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;
	int z, c;

	if (fl_get_button(fd_pttEditor->MLLButton)) {
		p->attribs &= ~PTT_NONLINEAR_MIN;
		for (z=0; z<p->num_zones; z++) {
			p->min_cost_len[z] = 0.0;
			for (c=0; c<p->num_categories; c++)
				p->rule[z][c].type &= ~DEFEAT_MIN_TIME_LEN;
		}
		fl_freeze_form(fd_pttEditor->pttEditor);
		updateZone(p);
		updateRule(p, 0);
		fl_unfreeze_form(fd_pttEditor->pttEditor);
	}
	return;
}

void doMLNButton(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;
	int z;

	if (fl_get_button(fd_pttEditor->MLNButton)) {
		p->attribs |= PTT_NONLINEAR_MIN;
		for (z=0; z<p->num_zones; z++)
			p->min_cost_len[z] = 0.0;
		fl_freeze_form(fd_pttEditor->pttEditor);
		updateZone(p);
		updateRule(p, 0);
		fl_unfreeze_form(fd_pttEditor->pttEditor);
	}
	return;
}

void doRuleCounter(FL_OBJECT *obj, long param)
{
	fl_call_object_callback(fd_pttEditor->ruleChargeInput);
	fl_call_object_callback(fd_pttEditor->RDMInput);
	fl_call_object_callback(fd_pttEditor->RDDInput);
	fl_call_object_callback(fd_pttEditor->RDEMInput);
	fl_call_object_callback(fd_pttEditor->RDEDInput);
	fl_call_object_callback(fd_pttEditor->relDateInput);
	fl_call_object_callback(fd_pttEditor->RSHInput);
	fl_call_object_callback(fd_pttEditor->RSMInput);
	fl_call_object_callback(fd_pttEditor->RSSInput);
	fl_call_object_callback(fd_pttEditor->REHInput);
	fl_call_object_callback(fd_pttEditor->REMInput);
	fl_call_object_callback(fd_pttEditor->RESInput);
	fl_freeze_form(fd_pttEditor->pttEditor);
	updateRule(&IFC_pttEditor, 0);
	fl_unfreeze_form(fd_pttEditor->pttEditor);
	return;
}

void doPttNameInput(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;

	strncpy(p->name, fl_get_input(obj), MAXLEN_PTTNAME);
	return;
}

void doZonesInput(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;
	int num, z, c;
	char dflt[16];

	num = intInput(obj, 1, MAXNUM_ZONES, "%d");
	if (num == p->num_zones)
		return;
	else if (num > p->num_zones) {
		for (z=p->num_zones; z<num; z++) {
			if (*(p->zone_name[z]) == 0) {
				sprintf(dflt, "Zone %d", z+1);
				strcpy(p->zone_name[z], dflt);
			}
			fl_add_browser_line(fd_pttEditor->zonePick, p->zone_name[z]);
		}
	}
	else if (num < p->num_zones) {
		for (z=num; z<p->num_zones; z++) {
			fl_delete_browser_line(fd_pttEditor->zonePick, num+1);
			p->dflt_tariff[z] = 0.0;
			p->min_cost_len[z] = 0.0;
			p->zone_name[z][0] = 0;
			p->discount[z] = reset_discount;
			for (c=0; c<MAXNUM_CATEGORY; c++) {
				if (p->attribs & PTT_BY_UNIT)
					p->rule[z][c] = rst_Urule;
				else
					p->rule[z][c] = rst_Trule;
			}
		}
		if (p->current_zone > num-1) {
			p->current_zone = num - 1;
			fl_select_browser_line(fd_pttEditor->zonePick, num);
			fl_freeze_form(fd_pttEditor->pttEditor);
			updateZone(p);
			updateRule(p, 1);
			fl_unfreeze_form(fd_pttEditor->pttEditor);
		}
	}
	p->num_zones = num;
	return;
}

void doCategoriesInput(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;
	int num;

	num = intInput(obj, 1, MAXNUM_CATEGORY, "%d");
	if (num == p->num_categories)
		return;
	else if (num > p->num_categories)
		fl_set_counter_bounds(fd_pttEditor->ruleCounter, 1, num);
	else if (num < p->num_categories) {
		if (fl_get_counter_value(fd_pttEditor->ruleCounter) > num) {
			fl_set_counter_value(fd_pttEditor->ruleCounter, num);
			fl_freeze_form(fd_pttEditor->pttEditor);
			updateRule(p, 0);
			fl_unfreeze_form(fd_pttEditor->pttEditor);
		}
	}
	p->num_categories = num;
	return;
}

void doMinUnitsInput(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;

	p->min_units = intInput(obj, 0, MAXVAL_MINUNITS, "%d");
	return;
}

void doMinCostInput(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;

	p->min_cost = floatInput(obj, 0, 0, "%g");
	return;
}

void doCSecondsInput(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;

	if (p->attribs & (PTT_PER_MINUTE|PTT_PER_SECS))
		p->charge_period = intInput(obj, 1, 60, "%d");
	else
		p->charge_period = intInput(obj, 0, 0, "%d");
	return;
}

void doUnitPriceInput(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;

	p->cost_quantum = floatInput(obj, 0, 0, "%g");
	return;
}

void doCurrencyInput(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;
	char str[64+MAXLEN_CURRENCY] = {0};

	strncpy(p->currency, (char *)fl_get_input(obj), MAXLEN_CURRENCY);
	if (p->attribs & (PTT_PER_MINUTE|PTT_PER_SECS)) {
		sprintf(str, CTIME_STR, p->currency);
		fl_set_object_label(fd_pttEditor->defaultTariffUnits, str);
		fl_set_object_label(fd_pttEditor->ruleTariffUnits, str);
	}
	return;
}

void doDecimalsInput(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;

	p->decimals = intInput(obj, 0, MAXNUM_DECIMALS, "%d");
	return;
}

void doBaseChargeInput(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;
	float num;

	num = floatInput(obj, 0, 0, "%g");
	p->dflt_tariff[p->current_zone] = num;
	return;
}

void doRuleChargeInput(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;
	float num;

	num = floatInput(obj, 0, 0, "%g");
	p->rule[p->current_zone][fd_pttEditor->ldata].tariff = num;
	return;
}

void doRelDateInput(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;

	p->rule[p->current_zone][fd_pttEditor->ldata].rdate =
		intInput(obj, MINVAL_RELDATE, MAXVAL_RELDATE, "%d");
	return;
}

void doMLSecondsInput(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;
	
	p->min_cost_len[p->current_zone] =
		floatInput(obj, 0, MAXSEC_MINCOSTLEN, "%g");
	return;
}

void doDiscountInput(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;

	p->discount[p->current_zone].percent = floatInput(obj, 0.0, 99.0, "%g");
	fl_freeze_form(fd_pttEditor->pttEditor);
	updateZone(p);
	updateRule(p, 0);
	fl_unfreeze_form(fd_pttEditor->pttEditor);
	return;
}

void doZEDYesButton(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;

	p->discount[p->current_zone].percent = 50.0;	/* 1/2 price, for fun :) */
	fl_freeze_form(fd_pttEditor->pttEditor);
	updateZone(p);
	updateRule(p, 0);
	fl_unfreeze_form(fd_pttEditor->pttEditor);
	return;
}

void doZEDNoButton(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;

	p->discount[p->current_zone].percent = 0.0;
	fl_freeze_form(fd_pttEditor->pttEditor);
	updateZone(p);
	updateRule(p, 0);
	fl_unfreeze_form(fd_pttEditor->pttEditor);
	return;
}

void doRDYesButton(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;

	p->rule[p->current_zone][fd_pttEditor->ldata].type |= DISCOUNT_PERMITTED;
	return;
}

void doRDNoButton(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;

	p->rule[p->current_zone][fd_pttEditor->ldata].type &= ~DISCOUNT_PERMITTED;
	return;
}

void doZMLYesButton(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;

	p->rule[p->current_zone][fd_pttEditor->ldata].type |= DEFEAT_MIN_TIME_LEN;
	return;
}

void doZMLNoButton(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;

	p->rule[p->current_zone][fd_pttEditor->ldata].type &= ~DEFEAT_MIN_TIME_LEN;
	return;
}

void doTButton(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;
	rule_t *r = &(p->rule[p->current_zone][fd_pttEditor->ldata]);

	r->type &= 0x0C0;
	r->type |= (0x03F & param);
	fl_freeze_form(fd_pttEditor->pttEditor);
	updateRule(p, 0);
	fl_unfreeze_form(fd_pttEditor->pttEditor);
	return;
}

void doRSInput(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;
	rule_t *r = &(p->rule[p->current_zone][fd_pttEditor->ldata]);
	int max;

	switch (param) {
		case 0:
			r->start.h = intInput(obj, 0, r->end.h, "%02d");
			if (r->start.h == 24) {
				fl_set_input(fd_pttEditor->RSMInput, "00");
				fl_redraw_object(fd_pttEditor->RSMInput);
				r->start.m = 0;
				fl_set_input(fd_pttEditor->RSSInput, "00");
				fl_redraw_object(fd_pttEditor->RSSInput);
				r->start.s = 0;
			}
			break;

		case 1:
			if (r->start.h < 24) max = (r->start.h < r->end.h)? 59 : r->end.m;
			else max = 0;
			r->start.m = intInput(obj, 0, max, "%02d");
			break;

		case 2:
			if (r->start.h < 24) {
				if (r->start.h < r->end.h) max = 59;
				else max = (r->start.m < r->end.m)? 59 : r->end.s;
			}
			else max = 0;
			r->start.s = intInput(obj, 0, max, "%02d");
			break;

		default: break;
	}
	return;
}

void doREInput(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;
	rule_t *r = &(p->rule[p->current_zone][fd_pttEditor->ldata]);
	int min;

	switch (param) {
		case 0:
			r->end.h = intInput(obj, r->start.h, 24, "%02d");
			if (r->end.h == 24) {
				fl_set_input(fd_pttEditor->REMInput, "00");
				fl_redraw_object(fd_pttEditor->REMInput);
				r->end.m = 0;
				fl_set_input(fd_pttEditor->RESInput, "00");
				fl_redraw_object(fd_pttEditor->RESInput);
				r->end.s = 0;
			}
			break;

		case 1:
			min = (r->start.h < r->end.h)? 0 : r->start.m;
			r->end.m = intInput(obj, min, (r->end.h<24)? 59:0, "%02d");
			break;

		case 2:
			if (r->start.h < r->end.h) min = 0;
			else min = (r->start.m < r->end.m)? 0 : r->start.s;
			r->end.s = intInput(obj, min, (r->end.h<24)? 59:0, "%02d");
			break;

		default: break;
	}
	return;
}

void doZSInput(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;
	discount_t *d = &(p->discount[p->current_zone]);
	int max;

	switch (param) {
		case 0:
			d->start.h = intInput(obj, 0, d->end.h, "%02d");
			if (d->start.h == 24) {
				fl_set_input(fd_pttEditor->ZSMInput, "00");
				fl_redraw_object(fd_pttEditor->ZSMInput);
				d->start.m = 0;
				fl_set_input(fd_pttEditor->ZSSInput, "00");
				fl_redraw_object(fd_pttEditor->ZSSInput);
				d->start.s = 0;
			}
			break;

		case 1:
			if (d->start.h < 24) max = (d->start.h < d->end.h)? 59 : d->end.m;
			else max = 0;
			d->start.m = intInput(obj, 0, max, "%02d");
			break;

		case 2:
			if (d->start.h < 24) {
				if (d->start.h < d->end.h) max = 59;
				else max = (d->start.m < d->end.m)? 59 : d->end.s;
			}
			else max = 0;
			d->start.s = intInput(obj, 0, max, "%02d");
			break;

		default: break;
	}
	return;
}

void doZEInput(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;
	discount_t *d = &(p->discount[p->current_zone]);
	int min;

	switch (param) {
		case 0:
			d->end.h = intInput(obj, d->start.h, 24, "%02d");
			if (d->end.h == 24) {
				fl_set_input(fd_pttEditor->ZEMInput, "00");
				fl_redraw_object(fd_pttEditor->ZEMInput);
				d->end.m = 0;
				fl_set_input(fd_pttEditor->ZESInput, "00");
				fl_redraw_object(fd_pttEditor->ZESInput);
				d->end.s = 0;
			}
			break;

		case 1:
			min = (d->start.h < d->end.h)? 0 : d->start.m;
			d->end.m = intInput(obj, min, (d->end.h<24)? 59:0, "%02d");
			break;

		case 2:
			if (d->start.h < d->end.h) min = 0;
			else min = (d->start.m < d->end.m)? 0 : d->start.s;
			d->end.s = intInput(obj, min, (d->end.h<24)? 59:0, "%02d");
			break;

		default: break;
	}
	return;
}

void doRDInput(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;
	rule_t *r = &(p->rule[p->current_zone][fd_pttEditor->ldata]);
	int rt = (r->type & 0x03F), max = 1,
		dlim[12] = {31,29,31,30,31,30,31,31,30,31,30,31};

	switch (param) {
		case 0:
			max = (rt != RULE_SPECIAL_WEEKDAY && rt != RULE_SPECIAL_WEEKEND)?
				  12 : 1+r->edate.mon;
			r->adate.mon = intInput(obj, 1, max, "%02d") - 1;
			break;

		case 1:
			if (rt != RULE_SPECIAL_WEEKDAY && rt != RULE_SPECIAL_WEEKEND)
				max = dlim[r->adate.mon];
			else
				max = (r->adate.mon != r->edate.mon)?
					  dlim[r->adate.mon] : r->edate.day;
			r->adate.day = intInput(obj, 1, max, "%02d");
			break;

		default: break;
	}
	return;
}

void doRDEInput(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;
	rule_t *r = &(p->rule[p->current_zone][fd_pttEditor->ldata]);
	int dlim[12] = {31,29,31,30,31,30,31,31,30,31,30,31};

	switch (param) {
		case 0:
			r->edate.mon = intInput(obj, 1+r->adate.mon, 12, "%02d") - 1;
			break;

		case 1:
			r->edate.day = intInput(obj, (r->adate.mon != r->edate.mon)?
									1:r->adate.day, dlim[r->edate.mon], "%02d");
			break;

		default: break;
	}
	return;
}

void doZonePick(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;
	int zn;

	if ((zn = fl_get_browser(obj))) {
		fl_call_object_callback(fd_pttEditor->baseChargeInput);
		fl_call_object_callback(fd_pttEditor->MLSecondsInput);
		fl_call_object_callback(fd_pttEditor->discountInput);
		fl_call_object_callback(fd_pttEditor->ZSHInput);
		fl_call_object_callback(fd_pttEditor->ZSMInput);
		fl_call_object_callback(fd_pttEditor->ZSSInput);
		fl_call_object_callback(fd_pttEditor->ZEHInput);
		fl_call_object_callback(fd_pttEditor->ZEMInput);
		fl_call_object_callback(fd_pttEditor->ZESInput);
		fl_call_object_callback(fd_pttEditor->ruleChargeInput);
		fl_call_object_callback(fd_pttEditor->RDMInput);
		fl_call_object_callback(fd_pttEditor->RDDInput);
		fl_call_object_callback(fd_pttEditor->RDEMInput);
		fl_call_object_callback(fd_pttEditor->RDEDInput);
		fl_call_object_callback(fd_pttEditor->relDateInput);
		fl_call_object_callback(fd_pttEditor->RSHInput);
		fl_call_object_callback(fd_pttEditor->RSMInput);
		fl_call_object_callback(fd_pttEditor->RSSInput);
		fl_call_object_callback(fd_pttEditor->REHInput);
		fl_call_object_callback(fd_pttEditor->REMInput);
		fl_call_object_callback(fd_pttEditor->RESInput);
		p->current_zone = zn - 1;
		fl_freeze_form(fd_pttEditor->pttEditor);
		updateZone(p);
		updateRule(p, 1);
		fl_unfreeze_form(fd_pttEditor->pttEditor);
	}
	return;
}

void doZonePickNEdit(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;

	fl_call_object_callback(fd_pttEditor->zonePick);
	fl_deactivate_form(fd_pttEditor->pttEditor);
	fl_set_input(fd_renameZone->zoneNameInput, p->zone_name[p->current_zone]);
	fl_show_form(fd_renameZone->renameZone, FL_PLACE_MOUSE,
				 FL_TRANSIENT, "Edit zone name");
	return;
}

void doZoneNameInput(FL_OBJECT *obj, long param)
{
	static char name[MAXLEN_ZNAME+1];

	strncpy(name, fl_get_input(obj), MAXLEN_ZNAME);
	fd_renameZone->cdata = name;
	return;
}

void doZoneNameEditOK(FL_OBJECT *obj, long param)
{
	ptt_t *p = &IFC_pttEditor;
	char dflt[16] = {0};

	fl_call_object_callback(fd_renameZone->zoneNameInput);
	if (*(fd_renameZone->cdata)) {
		strncpy(p->zone_name[p->current_zone],
				fd_renameZone->cdata, MAXLEN_ZNAME);
		fl_replace_browser_line(fd_pttEditor->zonePick, 1+p->current_zone,
								p->zone_name[p->current_zone]);
		fl_hide_form(fd_renameZone->renameZone);
		fl_activate_form(fd_pttEditor->pttEditor);
	}
	else {
		XBell(fl_get_display(), 50);
		sprintf(dflt, "Zone %d", 1+p->current_zone);
		fl_set_input(obj, dflt);
		fl_redraw_object(obj);
	}
	return;
}

void doZoneNameEditCancel(FL_OBJECT *obj, long param)
{
	fl_hide_form(fd_renameZone->renameZone);
	fl_activate_form(fd_pttEditor->pttEditor);
	return;
}

void doPttEditOK(FL_OBJECT *obj, long param)
{
	unsigned char sPTT = ((unsigned)fd_logInfo->ldata & 0x00FF0000) >> 16;
	unsigned char cZone = ((unsigned)fd_logInfo->ldata & 0x0FF000000) >> 24;
	ptt_t *p = &IFC_pttEditor;
	int i;

	fl_call_object_callback(fd_pttEditor->pttNameInput);
	fl_call_object_callback(fd_pttEditor->zonesInput);
	fl_call_object_callback(fd_pttEditor->categoriesInput);
	fl_call_object_callback(fd_pttEditor->minUnitsInput);
	fl_call_object_callback(fd_pttEditor->minCostInput);
	fl_call_object_callback(fd_pttEditor->CSecondsInput);
	fl_call_object_callback(fd_pttEditor->unitPriceInput);
	fl_call_object_callback(fd_pttEditor->currencyInput);
	fl_call_object_callback(fd_pttEditor->decimalsInput);
	fl_call_object_callback(fd_pttEditor->baseChargeInput);
	fl_call_object_callback(fd_pttEditor->MLSecondsInput);
	fl_call_object_callback(fd_pttEditor->discountInput);
	fl_call_object_callback(fd_pttEditor->ZSHInput);
	fl_call_object_callback(fd_pttEditor->ZSMInput);
	fl_call_object_callback(fd_pttEditor->ZSSInput);
	fl_call_object_callback(fd_pttEditor->ZEHInput);
	fl_call_object_callback(fd_pttEditor->ZEMInput);
	fl_call_object_callback(fd_pttEditor->ZESInput);
	fl_call_object_callback(fd_pttEditor->ruleChargeInput);
	fl_call_object_callback(fd_pttEditor->RDMInput);
	fl_call_object_callback(fd_pttEditor->RDDInput);
	fl_call_object_callback(fd_pttEditor->RDEMInput);
	fl_call_object_callback(fd_pttEditor->RDEDInput);
	fl_call_object_callback(fd_pttEditor->relDateInput);
	fl_call_object_callback(fd_pttEditor->RSHInput);
	fl_call_object_callback(fd_pttEditor->RSMInput);
	fl_call_object_callback(fd_pttEditor->RSSInput);
	fl_call_object_callback(fd_pttEditor->REHInput);
	fl_call_object_callback(fd_pttEditor->REMInput);
	fl_call_object_callback(fd_pttEditor->RESInput);
	if (cZone > p->num_zones-1) {
		XBell(fl_get_display(), 50);
		cZone = p->num_zones - 1;
	}
	p->current_zone = cZone;
	ptt[sPTT] = *p;
	fd_logInfo->vdata = (void *)1;	/* indicate change in PTT database */
	fl_clear_choice(fd_logInfo->pttDropChoice);
	for (i=0; i<global.numPTTs; i++)
		fl_addto_choice(fd_logInfo->pttDropChoice,
						clipStr(ptt[i].name, FL_NORMAL_SIZE,
								FL_NORMAL_STYLE, 240));
	fl_set_choice(fd_logInfo->pttDropChoice, sPTT+1);
	fl_clear_choice(fd_logInfo->zoneDropChoice);
	for (i=0; i<p->num_zones; i++)
		fl_addto_choice(fd_logInfo->zoneDropChoice,
						clipStr(p->zone_name[i], FL_NORMAL_SIZE,
								FL_NORMAL_STYLE, 240));
	fl_set_choice(fd_logInfo->zoneDropChoice,p->current_zone+1);
	showPttInfo(p);
	fl_hide_form(fd_pttEditor->pttEditor);
	fl_activate_form(fd_logInfo->logInfo);
	return;
}

void doPttEditCancel(FL_OBJECT *obj, long param)
{
	fl_hide_form(fd_pttEditor->pttEditor);
	fl_activate_form(fd_logInfo->logInfo);
	return;
}

void doPttEditHelp(FL_OBJECT *obj, long param)
{
	int i;

	fl_clear_browser(fd_helpInfo->helpBrowser);
	for (i=0; i<MAXNUM_PE_HELPLINES; i++)
		fl_add_browser_line(fd_helpInfo->helpBrowser,
							PEHelpText[i]);
	fl_show_form(fd_helpInfo->helpInfo, FL_PLACE_MOUSE,
				 FL_TRANSIENT, "PTT Editor Help");
	return;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |               Callback and utility routines for envInfo                 |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

char *statStr(char *path, char *tail)
{
	char test[MAXLEN_FNAME];
	static char stat[8];

	strcpy(test, path); strcat(test, tail);
#ifdef RUNASEUID
	if (xisp_euidaccess(test, X_OK) < 0)
#else
	if (access(test, X_OK) < 0)
#endif
		strcpy(stat, "Error");
	else
		strcpy(stat, "OK");
	return stat;
}

void doEnvInfoTip(FL_OBJECT *obj, long param)
{
	alertMessage("Environment setup hints", 1, 0,
	 "It is very important to enter the correct path to the location where\n"
	 "xisp will find the \"pppd process ID files\". The \"Status\" check for\n"
	 "this particular field only verifies the existence of the directory, as\n"
	 "no check for process ID files is possible without an active PPP\n"
	 "connection. Due to this fact it might be that, even though the\n"
	 "directory exists, your pppd binary is compiled to save its ppp?.pid\n"
	 "files elsewhere. Please consult with your pppd documentation and\n"
	 "make sure this entry is correct, otherwise abnormal xisp behavior\n"
	 "will most certainly occur. The default paths, hard-coded at compile\n"
	 "time for each supported operating system, are usually correct but\n"
	 "not 100% guaranteed to work; if you experience weird behavior,\n"
	 "look here first.");
}

void doPppdPathInput(FL_OBJECT *obj, long param)
{
	glob_t *g = &IFC_envInfo;

	strncpy(g->pppdPath, fl_get_input(obj), MAXLEN_PATH);
	fl_set_object_label(fd_envInfo->pppdStatus, statStr(g->pppdPath, PPPD));
	return;
}

void doPppdDefault(FL_OBJECT *obj, long param)
{
	fl_set_input(fd_envInfo->pppdPathInput, PPPD_PATH);
	fl_call_object_callback(fd_envInfo->pppdPathInput);
	return;
}

void doRunPathInput(FL_OBJECT *obj, long param)
{
	glob_t *g = &IFC_envInfo;

	strncpy(g->runPath, fl_get_input(obj), MAXLEN_PATH);
	fl_set_object_label(fd_envInfo->runStatus, statStr(g->runPath, ""));
	return;
}

void doRunDefault(FL_OBJECT *obj, long param)
{
	fl_set_input(fd_envInfo->runPathInput, RUN_PATH);
	fl_call_object_callback(fd_envInfo->runPathInput);
	return;
}

void doLockPathInput(FL_OBJECT *obj, long param)
{
	glob_t *g = &IFC_envInfo;

	strncpy(g->lockPath, fl_get_input(obj), MAXLEN_PATH);
	fl_set_object_label(fd_envInfo->lockStatus, statStr(g->lockPath, ""));
	return;
}

void doLockDefault(FL_OBJECT *obj, long param)
{
	fl_set_input(fd_envInfo->lockPathInput, LOCK_PATH);
	fl_call_object_callback(fd_envInfo->lockPathInput);
	return;
}

void doChatPathInput(FL_OBJECT *obj, long param)
{
	glob_t *g = &IFC_envInfo;

	strncpy(g->chatPath, fl_get_input(obj), MAXLEN_PATH);
	fl_set_object_label(fd_envInfo->chatStatus, statStr(g->chatPath, CHAT));
	return;
}

void doChatDefault(FL_OBJECT *obj, long param)
{
	fl_set_input(fd_envInfo->chatPathInput, CHAT_PATH);
	fl_call_object_callback(fd_envInfo->chatPathInput);
	return;
}

void doXispUtilsPathInput(FL_OBJECT *obj, long param)
{
	glob_t *g = &IFC_envInfo;
	char *pc;

	strncpy(g->utilsPath, fl_get_input(obj), MAXLEN_PATH);
	fl_set_object_label(fd_envInfo->utilsStatus,
						pc = statStr(g->utilsPath, PPPD_CONNECT));
	if (strcmp(pc, "Error")) {
		fl_set_object_label(fd_envInfo->utilsStatus,
							pc = statStr(g->utilsPath, TERMINAL));
	}
	return;
}

void doXispUtilsDefault(FL_OBJECT *obj, long param)
{
	fl_set_input(fd_envInfo->xispUtilsPathInput, UTILS_PATH);
	fl_call_object_callback(fd_envInfo->xispUtilsPathInput);
	return;
}

void doPipePathInput(FL_OBJECT *obj, long param)
{
	glob_t *g = &IFC_envInfo;

	strncpy(g->pipePath, fl_get_input(obj), MAXLEN_PATH);
	return;
}

void doPipeDefault(FL_OBJECT *obj, long param)
{
	fl_set_input(fd_envInfo->pipePathInput, PIPE_PATH);
	fl_call_object_callback(fd_envInfo->pipePathInput);
	return;
}

void doEnvInfoOK(FL_OBJECT *obj, long done)
{
	glob_t *g = &IFC_envInfo;
	char msg[512];

	fl_call_object_callback(fd_envInfo->pppdPathInput);
	fl_call_object_callback(fd_envInfo->runPathInput);
	fl_call_object_callback(fd_envInfo->lockPathInput);
	fl_call_object_callback(fd_envInfo->chatPathInput);
	fl_call_object_callback(fd_envInfo->xispUtilsPathInput);
	fl_call_object_callback(fd_envInfo->pipePathInput);
	global = *g;
	initUDFnames(g);
	if (!nohints && (pppdVersion().major > 2 || pppdVersion().minor > 2)) {
		sprintf(msg, "Please make sure that the peer information\n"
				"file xisp_dialer exists in pppd's peers directory\n"
				"(most probably /etc/ppp/peers -- read the pppd(8)\n"
				"manual page for details) -- and that it contains\n"
				"the line \"connect %s\"", Dialer);
		alertMessage("Setup hint", 1, 0, msg);
	}
	writeXisprc(rcfname, xispOptions, g);
	if (done) {
		fl_hide_form(fd_optsTab->optsTab);
		fl_activate_object(fd_topFrame->fileMenu);
		fl_set_object_lcol(fd_topFrame->fileMenu,FL_LCOL);
	}
	return;
}

void doEnvInfoCancel(FL_OBJECT *obj, long param)
{
	fl_hide_form(fd_optsTab->optsTab);
	fl_activate_object(fd_topFrame->fileMenu);
	fl_set_object_lcol(fd_topFrame->fileMenu,FL_LCOL);
	return;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |          Utility and callback routines for options tab folder           |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void prepFolderForm(int index)
{
	int i, ml = p_xisprc->numSlines;
	char dflt[5], phone[MAXNUM_TELS*(MAXLEN_PHONE+1)+1] = {0},
		 ppasswd[MAXLEN_PASSWD+1] = {0},
		 slines[MAXNUM_SLINES*(MAXLEN_SLINE+1)] = {0},
		ipdns[MAXLEN_DNNAME+1] = {0}, speed[MAXDIG_BAUDRATE+1] = {0};
	unsigned int knownSpeed[MAXNUM_SPEEDS] = {MS_0, MS_1, MS_2, MS_3,
											  MS_4, MS_5, MS_6, MS_7};

	switch (index) {

		case OPTIONS_ACCOUNT: {
			fl_deactivate_object(fd_accountInfo->accountInfoPaste);
			fl_set_object_lcol(fd_accountInfo->accountInfoPaste, FL_INACTIVE);
			fl_clear_browser(fd_accountInfo->ISPBrowser);
			if (global.numISPs < 1) {
				fl_deactivate_object(fd_accountInfo->accountInfoCopy);
				fl_set_object_lcol(fd_accountInfo->accountInfoCopy,
								   FL_INACTIVE);
				fl_deactivate_object(fd_accountInfo->accountInfoDelete);
				fl_set_object_lcol(fd_accountInfo->accountInfoDelete,
								   FL_INACTIVE);
				fl_deactivate_object(fd_accountInfo->ISPDefault);
				fl_set_object_lcol(fd_accountInfo->ISPDefault, FL_INACTIVE);
				fl_set_button(fd_accountInfo->ISPDefault, 0);
				fl_deactivate_object(fd_accountInfo->ISPAutoDial);
				fl_set_object_lcol(fd_accountInfo->ISPAutoDial, FL_INACTIVE);
				fl_set_button(fd_accountInfo->ISPAutoDial, 0);
				fl_deactivate_object(fd_accountInfo->autoRedial);
				fl_set_object_lcol(fd_accountInfo->autoRedial, FL_INACTIVE);
				fl_set_button(fd_accountInfo->autoRedial, 0);
				fl_deactivate_object(fd_accountInfo->authType);
				fl_set_object_lcol(fd_accountInfo->authType, FL_INACTIVE);
				fl_set_button(fd_accountInfo->authPAPButton, 0);
				fl_set_button(fd_accountInfo->authPAPSButton, 0);
				fl_set_button(fd_accountInfo->authCHAPSButton, 0);
				fl_set_button(fd_accountInfo->authNoneButton, 0);
				fl_deactivate_object(fd_accountInfo->telInput);
				fl_set_object_lcol(fd_accountInfo->telInput, FL_INACTIVE);
				fl_deactivate_object(fd_accountInfo->accInput);
				fl_set_object_lcol(fd_accountInfo->accInput, FL_INACTIVE);
				fl_deactivate_object(fd_accountInfo->pswInput);
				fl_set_object_lcol(fd_accountInfo->pswInput, FL_INACTIVE);
				fl_deactivate_object(fd_accountInfo->UNInput);
				fl_set_object_lcol(fd_accountInfo->UNInput, FL_INACTIVE);
				fl_deactivate_object(fd_accountInfo->remoteInput);
				fl_set_object_lcol(fd_accountInfo->remoteInput, FL_INACTIVE);
				fl_deactivate_object(fd_accountInfo->ispPttDropChoice);
				fl_set_object_lcol(fd_accountInfo->ispPttDropChoice,
								   FL_INACTIVE);
				fl_deactivate_object(fd_accountInfo->ispZoneDropChoice);
				fl_set_object_lcol(fd_accountInfo->ispZoneDropChoice,
								   FL_INACTIVE);
				IFC_accountInfo.cur = IFC_accountInfo.dflt = 0;
				IFC_accountInfo.new = 0;
				IFC_accountInfo.modified = 0;
				IFC_accountInfo.descr[0] = 0;
				currentPTT = PTT_GR_EPAK;
				global.costPTT = currentPTT;
			}
			else {
				fl_activate_object(fd_accountInfo->accountInfoCopy);
				fl_set_object_lcol(fd_accountInfo->accountInfoCopy, FL_LCOL);
				fl_activate_object(fd_accountInfo->accountInfoDelete);
				fl_set_object_lcol(fd_accountInfo->accountInfoDelete, FL_LCOL);
				fl_activate_object(fd_accountInfo->ISPDefault);
				fl_set_object_lcol(fd_accountInfo->ISPDefault, FL_WHITE);
				fl_activate_object(fd_accountInfo->ISPAutoDial);
				fl_set_object_lcol(fd_accountInfo->ISPAutoDial, FL_WHITE);
				fl_activate_object(fd_accountInfo->autoRedial);
				fl_set_object_lcol(fd_accountInfo->autoRedial, FL_WHITE);
				fl_activate_object(fd_accountInfo->telInput);
				fl_set_object_lcol(fd_accountInfo->telInput, FL_WHITE);
				fl_activate_object(fd_accountInfo->authType);
				fl_set_object_lcol(fd_accountInfo->authType, FL_WHITE);
				adjustPAPCap();
				for (i=0; i<global.numISPs; i++)
					fl_add_browser_line(fd_accountInfo->ISPBrowser,
										xispOptions[i+1].descr);
				fl_set_browser_topline(fd_accountInfo->ISPBrowser,
									   currentRC);
				fl_select_browser_line(fd_accountInfo->ISPBrowser, currentRC);
				fl_set_input(fd_renameISP->ISPNameInput,
							 xispOptions[currentRC].descr);
				strcpy(IFC_accountInfo.descr, xispOptions[currentRC].descr);
				if (global.dfltISP == currentRC)
					fl_set_button(fd_accountInfo->ISPDefault, 1);
				else
					fl_set_button(fd_accountInfo->ISPDefault, 0);
				if (p_xisprc->operOpts & STARTUP_DIAL)
					fl_set_button(fd_accountInfo->ISPAutoDial, 1);
				else
					fl_set_button(fd_accountInfo->ISPAutoDial, 0);
				if (p_xisprc->operOpts & AUTO_REDIAL)
					fl_set_button(fd_accountInfo->autoRedial, 1);
				else
					fl_set_button(fd_accountInfo->autoRedial, 0);
				IFC_accountInfo.cur = currentRC;
				IFC_accountInfo.dflt = global.dfltISP;
				IFC_accountInfo.new = 0;
				IFC_accountInfo.modified = 0;
				for (i=0; i<p_xisprc->numPhones; i++) {
					if (i) strcat(phone, ";");
					strcat(phone, p_xisprc->phone[i]);
				}
				fl_set_input(fd_accountInfo->telInput, phone);
				fl_set_input(fd_accountInfo->accInput, p_xisprc->account);
				if (*(p_xisprc->account))
					pdecode(ppasswd, p_xisprc->passwd);
				fl_set_input(fd_accountInfo->pswInput, ppasswd);
				fl_set_input(fd_accountInfo->UNInput, p_xisprc->name);
				fl_set_input(fd_accountInfo->remoteInput, p_xisprc->rname);
				fl_set_button(fd_accountInfo->authPAPButton, 0);
				fl_set_button(fd_accountInfo->authPAPSButton, 0);
				fl_set_button(fd_accountInfo->authCHAPSButton, 0);
				fl_set_button(fd_accountInfo->authNoneButton, 0);
				if (p_xisprc->operOpts & PAP_LOGIN) {
					fl_set_button(fd_accountInfo->authPAPButton, 1);
					fl_activate_object(fd_accountInfo->accInput);
					fl_set_object_lcol(fd_accountInfo->accInput, FL_WHITE);
					fl_activate_object(fd_accountInfo->pswInput);
					fl_set_object_lcol(fd_accountInfo->pswInput, FL_WHITE);
					fl_deactivate_object(fd_accountInfo->UNInput);
					fl_set_object_lcol(fd_accountInfo->UNInput, FL_INACTIVE);
					fl_deactivate_object(fd_accountInfo->remoteInput);
					fl_set_object_lcol(fd_accountInfo->remoteInput,
									   FL_INACTIVE);
				}
				else if (p_xisprc->operOpts & PAPS_LOGIN) {
					fl_set_button(fd_accountInfo->authPAPSButton, 1);
					fl_deactivate_object(fd_accountInfo->accInput);
					fl_set_object_lcol(fd_accountInfo->accInput, FL_INACTIVE);
					fl_deactivate_object(fd_accountInfo->pswInput);
					fl_set_object_lcol(fd_accountInfo->pswInput, FL_INACTIVE);
					fl_activate_object(fd_accountInfo->UNInput);
					fl_set_object_lcol(fd_accountInfo->UNInput, FL_WHITE);
					fl_activate_object(fd_accountInfo->remoteInput);
					fl_set_object_lcol(fd_accountInfo->remoteInput, FL_WHITE);
				}
				else if (p_xisprc->operOpts & CHAPS_LOGIN) {
					fl_set_button(fd_accountInfo->authCHAPSButton, 1);
					fl_deactivate_object(fd_accountInfo->accInput);
					fl_set_object_lcol(fd_accountInfo->accInput, FL_INACTIVE);
					fl_deactivate_object(fd_accountInfo->pswInput);
					fl_set_object_lcol(fd_accountInfo->pswInput, FL_INACTIVE);
					fl_activate_object(fd_accountInfo->UNInput);
					fl_set_object_lcol(fd_accountInfo->UNInput, FL_WHITE);
					fl_activate_object(fd_accountInfo->remoteInput);
					fl_set_object_lcol(fd_accountInfo->remoteInput, FL_WHITE);
				}
				else {
					fl_set_button(fd_accountInfo->authNoneButton, 1);
					fl_activate_object(fd_accountInfo->accInput);
					fl_set_object_lcol(fd_accountInfo->accInput, FL_WHITE);
					fl_activate_object(fd_accountInfo->pswInput);
					fl_set_object_lcol(fd_accountInfo->pswInput, FL_WHITE);
					fl_deactivate_object(fd_accountInfo->UNInput);
					fl_set_object_lcol(fd_accountInfo->UNInput, FL_INACTIVE);
					fl_deactivate_object(fd_accountInfo->remoteInput);
					fl_set_object_lcol(fd_accountInfo->remoteInput,
									   FL_INACTIVE);
				}
				currentPTT = p_xisprc->ispPTT;
				global.costPTT = currentPTT;
			}
			p_ptt = &ptt[currentPTT];
			p_ptt->current_zone = p_xisprc->ispZone;
			IFC_accountInfo.udata = currentPTT |
									(p_ptt->current_zone << 8);
			fl_clear_choice(fd_accountInfo->ispPttDropChoice);
			fl_clear_choice(fd_accountInfo->ispZoneDropChoice);
			if (!(global.logOpts & LOG_NONE) || (global.logOpts & COST_READOUT)
			   )
			{
				fl_activate_object(fd_accountInfo->ispPttDropChoice);
				fl_set_object_lcol(fd_accountInfo->ispPttDropChoice,
								   FL_WHITE);
				fl_activate_object(fd_accountInfo->ispZoneDropChoice);
				fl_set_object_lcol(fd_accountInfo->ispZoneDropChoice,
								   FL_WHITE);
				for (i=0; i<global.numPTTs; i++)
					fl_addto_choice(fd_accountInfo->ispPttDropChoice,
						clipStr(ptt[i].name, FL_MEDIUM_SIZE,
								FL_BOLD_STYLE, 235));
				fl_set_choice(fd_accountInfo->ispPttDropChoice,
							  currentPTT+1);
				for (i=0; i<p_ptt->num_zones; i++)
					fl_addto_choice(fd_accountInfo->ispZoneDropChoice,
						clipStr(p_ptt->zone_name[i], FL_MEDIUM_SIZE,
								FL_BOLD_STYLE, 235));
				fl_set_choice(fd_accountInfo->ispZoneDropChoice,
							  p_ptt->current_zone+1);
			}
			else {
				fl_deactivate_object(fd_accountInfo->ispPttDropChoice);
				fl_set_object_lcol(fd_accountInfo->ispPttDropChoice,
								   FL_INACTIVE);
				fl_deactivate_object(fd_accountInfo->ispZoneDropChoice);
				fl_set_object_lcol(fd_accountInfo->ispZoneDropChoice,
								   FL_INACTIVE);
			}
			fd_accountInfo->ldata = p_xisprc->operOpts;
			fd_logInfo->vdata = NULL;	/* used as ISP database resize flag */
			if (! nohints) fl_set_timer(fd_accountInfo->tipTimer, HINT_TRIGGER);
			break;
		}

		case OPTIONS_DIALING: {
			sprintf(dflt, "%d", p_xisprc->maxAttempts);
			fl_set_input(fd_dialInfo->rtrInput, dflt);
			sprintf(dflt, "%d", p_xisprc->sleepDelay);
			fl_set_input(fd_dialInfo->dlyInput, dflt);
			sprintf(dflt, "%d", p_xisprc->connectWait);
			fl_set_input(fd_dialInfo->CNWaitInput, dflt);
			sprintf(dflt, "%d", p_xisprc->LCPWait);
			fl_set_input(fd_dialInfo->LCPWaitInput, dflt);
			if (p_xisprc->operOpts & MANUAL_LOGIN) {
				fl_set_button(fd_dialInfo->TTYesButton, 1);
				fl_set_button(fd_dialInfo->TTNoButton, 0);
				fl_activate_object(fd_dialInfo->TSGroup);
				fl_set_object_lcol(fd_dialInfo->TSGroup,FL_WHITE);
				fl_deactivate_object(fd_dialInfo->ALGroup);
				fl_set_object_lcol(fd_dialInfo->ALGroup,FL_INACTIVE);
			}
			else {
				fl_set_button(fd_dialInfo->TTYesButton, 0);
				fl_set_button(fd_dialInfo->TTNoButton, 1);
				fl_deactivate_object(fd_dialInfo->TSGroup);
				fl_set_object_lcol(fd_dialInfo->TSGroup,FL_INACTIVE);
				fl_activate_object(fd_dialInfo->ALGroup);
				fl_set_object_lcol(fd_dialInfo->ALGroup,FL_WHITE);
			}
			if (p_xisprc->operOpts & CALL_BACK) {
				fl_activate_object(fd_dialInfo->CBOptions);
				fl_set_object_lcol(fd_dialInfo->CBOptions,FL_LCOL);
				fl_set_button(fd_dialInfo->CBYesButton, 1);
				fl_set_button(fd_dialInfo->CBNoButton, 0);
			}
			else {
				fl_deactivate_object(fd_dialInfo->CBOptions);
				fl_set_object_lcol(fd_dialInfo->CBOptions,FL_INACTIVE);
				fl_set_button(fd_dialInfo->CBYesButton, 0);
				fl_set_button(fd_dialInfo->CBNoButton, 1);
			}
			if (p_xisprc->operOpts & CONNECT_BELL) {
				fl_set_button(fd_dialInfo->RBYesButton, 1);
				fl_set_button(fd_dialInfo->RBNoButton, 0);
			}
			else {
				fl_set_button(fd_dialInfo->RBYesButton, 0);
				fl_set_button(fd_dialInfo->RBNoButton, 1);
			}
			for (i=0; i<ml; i++) {
				strcat(slines, p_xisprc->sline[i]);
				if (i < (ml-1)) strcat(slines, "\n");
			}
			fl_set_input(fd_dialInfo->expectInput, slines);
			slines[0] = 0;
			for (i=0; i<ml; i++) {
				strcat(slines, p_xisprc->sline[MAXNUM_SLINES+i]);
				if (i < (ml-1)) strcat(slines, "\n");
			}
			fl_set_input(fd_dialInfo->sendInput, slines);
			wasTab = 0;
			sprintf(dflt, "%d", p_xisprc->termW);
			fl_set_input(fd_dialInfo->columnsInput, dflt);
			sprintf(dflt, "%d", p_xisprc->termH);
			fl_set_input(fd_dialInfo->rowsInput, dflt);
			fd_dialInfo->ldata = p_xisprc->operOpts;
			if (! nohints) fl_set_timer(fd_dialInfo->tipTimer, HINT_TRIGGER);
			break;
		}

		case OPTIONS_COMM: {
			fl_set_input(fd_commInfo->deviceInput, p_xisprc->modemDevice);
			fl_set_input(fd_commInfo->modemResetInput,
						 p_xisprc->modemReset);
			fl_set_input(fd_commInfo->modemInitInput,
						 p_xisprc->modemInit);
			fl_set_input(fd_commInfo->modemConnectInput,
						 p_xisprc->modemConnect);
			if (p_xisprc->operOpts & MODEM_TONEDIAL) {
				fl_set_button(fd_commInfo->toneButton, 1);
				fl_set_button(fd_commInfo->ISDNButton, 0);
				fl_set_button(fd_commInfo->pulseButton, 0);
			}
			else if (p_xisprc->operOpts & MODEM_ISDNDIAL) {
				fl_set_button(fd_commInfo->toneButton, 0);
				fl_set_button(fd_commInfo->ISDNButton, 1);
				fl_set_button(fd_commInfo->pulseButton, 0);
			}
			else {
				fl_set_button(fd_commInfo->toneButton, 0);
				fl_set_button(fd_commInfo->ISDNButton, 0);
				fl_set_button(fd_commInfo->pulseButton, 1);
			}
			adjustCompCap();
			if (p_xisprc->operOpts & BSD_COMPRESS) {
				fl_set_button(fd_commInfo->SWCBSDButton, 1);
				fl_set_button(fd_commInfo->SWCDeflateButton, 0);
				fl_set_button(fd_commInfo->SWCOffButton, 0);
				fl_activate_object(fd_commInfo->SWCInput);
				fl_set_object_lcol(fd_commInfo->SWCInput,FL_WHITE);
			}
			else if (p_xisprc->operOpts & DEFL_COMPRESS) {
				fl_set_button(fd_commInfo->SWCBSDButton, 0);
				fl_set_button(fd_commInfo->SWCDeflateButton, 1);
				fl_set_button(fd_commInfo->SWCOffButton, 0);
				fl_activate_object(fd_commInfo->SWCInput);
				fl_set_object_lcol(fd_commInfo->SWCInput,FL_WHITE);
			}
			else {
				fl_set_button(fd_commInfo->SWCBSDButton, 0);
				fl_set_button(fd_commInfo->SWCDeflateButton, 0);
				fl_set_button(fd_commInfo->SWCOffButton, 1);
				fl_deactivate_object(fd_commInfo->SWCInput);
				fl_set_object_lcol(fd_commInfo->SWCInput,FL_INACTIVE);
			}
			IFC_commInfo.custom = 1;
			for (i=0; i<MAXNUM_SPEEDS; i++) {
				if (p_xisprc->modemSpeed == knownSpeed[i]) {
					IFC_commInfo.custom = 0;
					fl_set_button(fd_commInfo->speedButton[i], 1);
				}
				else
					fl_set_button(fd_commInfo->speedButton[i], 0);
			}
			if (IFC_commInfo.custom) {
				fl_set_button(fd_commInfo->speedButton[MAXNUM_SPEEDS], 1);
				fl_activate_object(fd_commInfo->customSpeedInput);
			}
			else {
				fl_set_button(fd_commInfo->speedButton[MAXNUM_SPEEDS], 0);
				fl_deactivate_object(fd_commInfo->customSpeedInput);
			}
			IFC_commInfo.speed = p_xisprc->modemSpeed;
			sprintf(speed, "%u", p_xisprc->modemSpeed);
			fl_set_input(fd_commInfo->customSpeedInput, speed);
			if (p_xisprc->operOpts & HW_FLOWCTRL) {
				fl_set_button(fd_commInfo->HWButton, 1);
				fl_set_button(fd_commInfo->SWButton, 0);
			}
			else {
				fl_set_button(fd_commInfo->HWButton, 0);
				fl_set_button(fd_commInfo->SWButton, 1);
			}
			if (p_xisprc->operOpts & ESCAPE_ON) {
				fl_set_button(fd_commInfo->escapeYesButton, 1);
				fl_set_button(fd_commInfo->escapeNoButton, 0);
				fl_activate_object(fd_commInfo->escapeInput);
				fl_set_object_lcol(fd_commInfo->escapeInput,FL_WHITE);
			}
			else {
				fl_set_button(fd_commInfo->escapeYesButton, 0);
				fl_set_button(fd_commInfo->escapeNoButton, 1);
				fl_deactivate_object(fd_commInfo->escapeInput);
				fl_set_object_lcol(fd_commInfo->escapeInput,FL_INACTIVE);
			}
			sprintf(dflt, "%d", p_xisprc->compLevel);
			fl_set_input(fd_commInfo->SWCInput, dflt);
			IFC_commInfo.operOpts = p_xisprc->operOpts;
			fl_set_input(fd_commInfo->modemDialInput, p_xisprc->dialExtra);
			fl_set_input(fd_commInfo->asyncmapInput,
						 p_xisprc->asyncmap);
			fl_set_input(fd_commInfo->escapeInput,
						 p_xisprc->escape);
			if (! nohints) fl_set_timer(fd_commInfo->tipTimer, HINT_TRIGGER);
			break;
		}

		case OPTIONS_TCPIP: {
			adjustAutoDNSCap();
			IPToStr(p_xisprc->localIP, ipdns);
			fl_set_input(fd_tcpipInfo->localIPInput, ipdns);
			IPToStr(p_xisprc->remoteIP, ipdns);
			fl_set_input(fd_tcpipInfo->remoteIPInput, ipdns);
			IPToStr(p_xisprc->netmask, ipdns);
			fl_set_input(fd_tcpipInfo->NMInput, ipdns);
			IPToStr(p_xisprc->dns1, ipdns);
			fl_set_input(fd_tcpipInfo->pDNSInput, ipdns);
			IPToStr(p_xisprc->dns2, ipdns);
			fl_set_input(fd_tcpipInfo->sDNSInput, ipdns);
			fl_set_input(fd_tcpipInfo->domainInput, p_xisprc->domainname);
			if (! strlen(p_xisprc->domainname))
				p_xisprc->operOpts &= ~DEFAULT_DOMAIN;
			if (p_xisprc->operOpts & ACCEPT_LOCALIP) {
				fl_set_button(fd_tcpipInfo->ALYesButton, 1);
				fl_set_button(fd_tcpipInfo->ALNoButton, 0);
				fl_deactivate_object(fd_tcpipInfo->localIPInput);
				fl_set_object_lcol(fd_tcpipInfo->localIPInput, FL_INACTIVE);
			}
			else {
				fl_set_button(fd_tcpipInfo->ALYesButton, 0);
				fl_set_button(fd_tcpipInfo->ALNoButton, 1);
				fl_activate_object(fd_tcpipInfo->localIPInput);
				fl_set_object_lcol(fd_tcpipInfo->localIPInput, FL_WHITE);
			}
			if (p_xisprc->operOpts & ACCEPT_REMOTEIP) {
				fl_set_button(fd_tcpipInfo->ARYesButton, 1);
				fl_set_button(fd_tcpipInfo->ARNoButton, 0);
				fl_deactivate_object(fd_tcpipInfo->remoteIPInput);
				fl_set_object_lcol(fd_tcpipInfo->remoteIPInput, FL_INACTIVE);
			}
			else {
				fl_set_button(fd_tcpipInfo->ARYesButton, 0);
				fl_set_button(fd_tcpipInfo->ARNoButton, 1);
				fl_activate_object(fd_tcpipInfo->remoteIPInput);
				fl_set_object_lcol(fd_tcpipInfo->remoteIPInput, FL_WHITE);
			}
			if (p_xisprc->operOpts & DEFAULT_ROUTE) {
				fl_set_button(fd_tcpipInfo->DRYesButton, 1);
				fl_set_button(fd_tcpipInfo->DRNoButton, 0);
			}
			else {
				fl_set_button(fd_tcpipInfo->DRYesButton, 0);
				fl_set_button(fd_tcpipInfo->DRNoButton, 1);
			}
			if (p_xisprc->operOpts & IP_UPDOWN) {
				fl_activate_object(fd_tcpipInfo->DNTypeGroup);
				fl_set_object_lcol(fd_tcpipInfo->DNTypeGroup,FL_WHITE);
				adjustAutoDNSCap();
				if (p_xisprc->operOpts & AUTO_DNS) {
					fl_set_button(fd_tcpipInfo->DNSNoButton, 0);
					fl_set_button(fd_tcpipInfo->DNSAutoButton, 1);
					fl_activate_object(fd_tcpipInfo->pDNSInput);
					fl_set_object_lcol(fd_tcpipInfo->pDNSInput,FL_WHITE);
					fl_activate_object(fd_tcpipInfo->sDNSInput);
					fl_set_object_lcol(fd_tcpipInfo->sDNSInput,FL_WHITE);
					fl_set_button(fd_tcpipInfo->DNSManualButton, 0);
					fl_deactivate_object(fd_tcpipInfo->pDNSInput);
					fl_set_object_lcol(fd_tcpipInfo->pDNSInput,FL_INACTIVE);
					fl_deactivate_object(fd_tcpipInfo->sDNSInput);
					fl_set_object_lcol(fd_tcpipInfo->sDNSInput,FL_INACTIVE);
				}
				else {
					fl_set_button(fd_tcpipInfo->DNSNoButton, 0);
					fl_set_button(fd_tcpipInfo->DNSAutoButton, 0);
					fl_deactivate_object(fd_tcpipInfo->pDNSInput);
					fl_set_object_lcol(fd_tcpipInfo->pDNSInput,FL_INACTIVE);
					fl_deactivate_object(fd_tcpipInfo->sDNSInput);
					fl_set_object_lcol(fd_tcpipInfo->sDNSInput,FL_INACTIVE);
					fl_set_button(fd_tcpipInfo->DNSManualButton, 1);
					fl_activate_object(fd_tcpipInfo->pDNSInput);
					fl_set_object_lcol(fd_tcpipInfo->pDNSInput,FL_WHITE);
					fl_activate_object(fd_tcpipInfo->sDNSInput);
					fl_set_object_lcol(fd_tcpipInfo->sDNSInput,FL_WHITE);
				}
				if (p_xisprc->operOpts & DEFAULT_DOMAIN) {
					fl_set_button(fd_tcpipInfo->DNNoButton, 0);
					fl_set_button(fd_tcpipInfo->DNYesButton, 1);
					fl_activate_object(fd_tcpipInfo->domainInput);
					fl_set_object_lcol(fd_tcpipInfo->domainInput,FL_WHITE);
				}
				else {
					fl_set_button(fd_tcpipInfo->DNNoButton, 1);
					fl_set_button(fd_tcpipInfo->DNYesButton, 0);
					fl_deactivate_object(fd_tcpipInfo->domainInput);
					fl_set_object_lcol(fd_tcpipInfo->domainInput,FL_INACTIVE);
				}
			}
			else {
				fl_set_button(fd_tcpipInfo->DNSNoButton, 1);
				fl_set_button(fd_tcpipInfo->DNSAutoButton, 0);
				fl_set_button(fd_tcpipInfo->DNSManualButton, 0);
				fl_deactivate_object(fd_tcpipInfo->pDNSInput);
				fl_set_object_lcol(fd_tcpipInfo->pDNSInput,FL_INACTIVE);
				fl_deactivate_object(fd_tcpipInfo->sDNSInput);
				fl_set_object_lcol(fd_tcpipInfo->sDNSInput,FL_INACTIVE);
				fl_deactivate_object(fd_tcpipInfo->pDNSInput);
				fl_set_object_lcol(fd_tcpipInfo->pDNSInput,FL_INACTIVE);
				fl_deactivate_object(fd_tcpipInfo->sDNSInput);
				fl_set_object_lcol(fd_tcpipInfo->sDNSInput,FL_INACTIVE);
				fl_set_button(fd_tcpipInfo->DNNoButton, 1);
				fl_set_button(fd_tcpipInfo->DNYesButton, 0);
				fl_deactivate_object(fd_tcpipInfo->DNTypeGroup);
				fl_set_object_lcol(fd_tcpipInfo->DNTypeGroup,FL_INACTIVE);
				fl_deactivate_object(fd_tcpipInfo->domainInput);
				fl_set_object_lcol(fd_tcpipInfo->domainInput,FL_INACTIVE);
			}
			fd_tcpipInfo->ldata = p_xisprc->operOpts;
			sprintf(dflt, "%d", p_xisprc->mtu);
			fl_set_input(fd_tcpipInfo->MTUInput, dflt);
			sprintf(dflt, "%d", p_xisprc->mru);
			fl_set_input(fd_tcpipInfo->MRUInput, dflt);
			if (! nohints) fl_set_timer(fd_tcpipInfo->tipTimer, HINT_TRIGGER);
			break;
		}

		case OPTIONS_PATHS:{
			fl_set_input(fd_envInfo->pppdPathInput, global.pppdPath);
			fl_call_object_callback(fd_envInfo->pppdPathInput);
			fl_set_input(fd_envInfo->runPathInput, global.runPath);
			fl_call_object_callback(fd_envInfo->runPathInput);
			fl_set_input(fd_envInfo->lockPathInput, global.lockPath);
			fl_call_object_callback(fd_envInfo->lockPathInput);
			fl_set_input(fd_envInfo->chatPathInput, global.chatPath);
			fl_call_object_callback(fd_envInfo->chatPathInput);
			fl_set_input(fd_envInfo->xispUtilsPathInput, global.utilsPath);
			fl_call_object_callback(fd_envInfo->xispUtilsPathInput);
			fl_set_input(fd_envInfo->pipePathInput, global.pipePath);
			fl_call_object_callback(fd_envInfo->pipePathInput);
			IFC_envInfo = global;
			if (! nohints) fl_set_timer(fd_envInfo->tipTimer, HINT_TRIGGER);
			break;
		}

		default: break;
	}
	return;
}

int folderButtonPreH(FL_OBJECT *obj, int event, FL_COORD mx,
					 FL_COORD my, int key, void *raw_event)
{
	int current;
	accountInfo_t *a = &IFC_accountInfo;
	unsigned char sPTT = (unsigned)a->udata & 0x000000FF;
	int newCurrency = strcmp(ptt[currentPTT].currency,ptt[sPTT].currency);

	switch (event) {
		case FL_PUSH:
			current = fl_get_active_folder_number(fd_optsTab->optsFolder);
			if (obj != fd_optsTab->accountInfoButton &&
				current == OPTIONS_ACCOUNT)
			{
				if (! newCurrency || (global.logOpts & LOG_NONE))
					return !FL_PREEMPT;
				if (actionVerify("The PTT for the selected ISP bills you in "
								 "currency\nwhich is different from the one "
								 "currently in use. This\nimplies resetting of "
								 "log files, continue?", 0))
				{
					currentPTT = sPTT;
					global.costPTT = currentPTT;
					resetLogs();
				}
				else
					return FL_PREEMPT;
			}
			break;

		default: break;
	}
	return !FL_PREEMPT;
}

void doOptsFolder(FL_OBJECT *obj, long param)
{
	int previous = fl_get_folder_number(obj),
		current = fl_get_active_folder_number(obj);

	switch (previous) {

		case OPTIONS_ACCOUNT: {
			doAccountOK(fd_optsTab->optsFolder, 0);
			break;
		}

		case OPTIONS_DIALING: {
			doDialOK(fd_optsTab->optsFolder, 0);
			break;
		}

		case OPTIONS_COMM: {
			doCommInfoOK(fd_optsTab->optsFolder, 0);
			break;
		}

		case OPTIONS_TCPIP: {
			doTcpipInfoOK(fd_optsTab->optsFolder, 0);
			break;
		}

		case OPTIONS_PATHS:{
			doEnvInfoOK(fd_optsTab->optsFolder, 0);
			break;
		}

		default: break;
	}
	prepFolderForm(current);
	return;
}

void doOptsTabDone(FL_OBJECT *obj, long param)
{
	accountInfo_t *a = &IFC_accountInfo;
	unsigned char sPTT = (unsigned)a->udata & 0x000000FF;
	int newCurrency = strcmp(ptt[currentPTT].currency,ptt[sPTT].currency);

	switch (fl_get_active_folder_number(fd_optsTab->optsFolder)) {

		case OPTIONS_ACCOUNT: {
			if (!(global.logOpts & LOG_NONE) && newCurrency &&
				! actionVerify("The PTT for the selected ISP bills you in "
							   "currency\nwhich is different from the one "
							   "currently in use. This\nimplies resetting of "
							   "log files, continue?", 0)
			   )
				break;
			if (!(global.logOpts & LOG_NONE) && newCurrency)
				resetLogs();
			doAccountOK(fd_optsTab->optsFolder, 1);
			break;
		}

		case OPTIONS_DIALING: {
			doDialOK(fd_optsTab->optsFolder, 1);
			break;
		}

		case OPTIONS_COMM: {
			doCommInfoOK(fd_optsTab->optsFolder, 1);
			break;
		}

		case OPTIONS_TCPIP: {
			doTcpipInfoOK(fd_optsTab->optsFolder, 1);
			break;
		}

		case OPTIONS_PATHS:{
			doEnvInfoOK(fd_optsTab->optsFolder, 1);
			break;
		}

		default: break;
	}
	return;
}

void doOptsTabCancel(FL_OBJECT *obj, long param)
{
	switch (fl_get_active_folder_number(fd_optsTab->optsFolder)) {

		case OPTIONS_ACCOUNT: {
			doAccountCancel(fd_optsTab->optsFolder, 0);
			break;
		}

		case OPTIONS_DIALING: {
			doDialCancel(fd_optsTab->optsFolder, 0);
			break;
		}

		case OPTIONS_COMM: {
			doCommInfoCancel(fd_optsTab->optsFolder, 0);
			break;
		}

		case OPTIONS_TCPIP: {
			doTcpipInfoCancel(fd_optsTab->optsFolder, 0);
			break;
		}

		case OPTIONS_PATHS:{
			doEnvInfoCancel(fd_optsTab->optsFolder, 0);
			break;
		}

		default: break;
	}
	return;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |              WM client message processing initialization                |
  |             and raw event processing routine for main form              |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

Atom Atom_WM_PROTOCOLS, Atom_WM_DELETE_WINDOW;

void initClientMsgProc(void)
{
	Atom_WM_PROTOCOLS = XInternAtom(fl_get_display(),
							"WM_PROTOCOLS", False);
	Atom_WM_DELETE_WINDOW = XInternAtom(fl_get_display(),
								"WM_DELETE_WINDOW", False);
	XChangeProperty(fl_get_display(), topWin, Atom_WM_PROTOCOLS,
					XA_ATOM, 32, PropModeReplace,
					(unsigned char *)&Atom_WM_DELETE_WINDOW, 1);
}

int rawEvents(FL_FORM *form, void *xev)
{
	XEvent *ev = xev;

	switch (ev->type) {
#ifdef XPMANIMATE								/* if animation enabled */
		case MapNotify:							/* figure out if we're */
			minimized = 0;						/* being minimized or not */
			break;

		case UnmapNotify:
			minimized = 1;
			break;
#endif
		case ClientMessage:						/* if "delete" received */
			if (ev->xclient.message_type !=		/* from window manager */
				Atom_WM_PROTOCOLS) break;
			if (ev->xclient.data.l[0] !=
				(int)Atom_WM_DELETE_WINDOW)
				break;
		case DestroyNotify:						/* or "destroy" from server */
			unlink(upidfname);					/* then do some cleanup */
			break;

		default: break;
	}
	return 0;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |       Callback routines for exitDialog and window delete handlers       |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void doExitOK(FL_OBJECT *obj, long param)
{
	if (dialerPID) {
		kill(dialerPID, SIGTERM);
		mSleep((int)BU_INTERVAL*1000);
	}
	if (connected && (pppdPID = getPppdPID(devName)))
		kill(pppdPID, SIGINT);
	exit(0);
}

void doExitCancel(FL_OBJECT *obj, long param)
{
	fl_hide_form(fd_exitDialog->exitDialog);
	fl_activate_form(fd_topFrame->topFrame);
	return;
}

int deleteHandler(FL_FORM *form, void *data)
{
	if (form == fd_topFrame->topFrame) {
		if (! expert && connected) {
			fl_deactivate_form(fd_topFrame->topFrame);
			fl_show_form(fd_exitDialog->exitDialog, FL_PLACE_MOUSE,
					 	FL_TRANSIENT, "Exit Dialog");
		}
		else
			fl_call_object_callback(fd_exitDialog->exitOK);
	}
	else if (form == fd_CBInfo->CBInfo) {
		fl_call_object_callback(fd_CBInfo->CBInfoCancel);
	}
	else if (form == fd_renameISP->renameISP) {
		fl_call_object_callback(fd_renameISP->ISPNameEditCancel);
	}
	else if (form == fd_aboutInfo->aboutInfo) {
		fl_call_object_callback(fd_aboutInfo->aboutOK);
	}
	else if (form == fd_exitDialog->exitDialog) {
		fl_call_object_callback(fd_exitDialog->exitCancel);
	}
	else if (form == fd_helpInfo->helpInfo) {
		fl_call_object_callback(fd_helpInfo->helpInfoOK);
	}
	else if (form == fd_logInfo->logInfo) {
		fl_call_object_callback(fd_logInfo->logInfoCancel);
	}
	else if (form == fd_statInfo->statInfo) {
		fl_call_object_callback(fd_statInfo->statInfoOK);
	}
	else if (form == fd_pttEditor->pttEditor) {
		fl_call_object_callback(fd_pttEditor->pttEditCancel);
	}
	else if (form == fd_renamePTT->renamePTT) {
		fl_call_object_callback(fd_renamePTT->PTTNameEditCancel);
	}
	else if (form == fd_renameZone->renameZone) {
		fl_call_object_callback(fd_renameZone->zoneNameEditCancel);
	}
	else if (form == fd_optsTab->optsTab) {
		fl_hide_form(fd_optsTab->optsTab);
		fl_activate_object(fd_topFrame->fileMenu);
		fl_set_object_lcol(fd_topFrame->fileMenu,FL_LCOL);
	}
	else if (form == fd_actionVerify->actionVerify) {
		fl_call_object_callback(fd_actionVerify->actionVerifyNo);
	}
	else if (form == fd_alertMessage->alertMessage) {
		fl_hide_form(fd_alertMessage->alertMessage);
	}
	return FL_IGNORE;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                Signal trap and exit(3) cleanup routine                  |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void sigTrap(int signum)
{
	char msg[512] = {0};

	switch (signum) {

		/* SIGINT or SIGTERM: disconnect PPP link and exit */
		case SIGINT:
		case SIGTERM: {
			sprintf(msg, "X-ISP terminated%s.\n",
					(progState == CONNECTED)?", disconnecting link":"");
			alertMessage("xISP: sigTrap()", 0, 0, msg);
			if (dialerPID) {
				kill(dialerPID, SIGTERM);
				mSleep((int)BU_INTERVAL*1000);
			}
			if ((pppdPID = getPppdPID(devName)))
				kill(pppdPID, SIGINT);
			exit(0);
		}

		/* SIGUSR1: if we are disconnected, press the "Connect" button */
		case SIGUSR1: {
			if (progState == DISCONNECTED)
				fl_call_object_callback(fd_topFrame->conButton);
#ifndef SUNOS41x
			signal(SIGUSR1, sigTrap);
#endif
			return;
		}

		/* SIGUSR2: if we are dialing, press the "Interrupt" button,
					else, if we are connected, press  "Disconnect" */
		case SIGUSR2: {
			if (progState == DIALING)
				fl_call_object_callback(fd_topFrame->intButton);
			else if (progState == CONNECTED)
				fl_call_object_callback(fd_topFrame->disButton);
#ifndef SUNOS41x
			signal(SIGUSR2, sigTrap);
#endif
			return;
		}

		/* By default, bail out if an unknown signal is received */
		default: {
			sprintf(msg, "X-ISP received signal %d: exiting\n"
					"and abandoning link in present condition.", signum);
			alertMessage("xISP: sigTrap()", 0, 0, msg);
			exit(1);
		}
	}
}

void registerSignalTrap(void)	/* register signals we want to trap */
{
	signal(SIGTERM, sigTrap);
	signal(SIGINT, sigTrap);
	signal(SIGUSR1, sigTrap);
	signal(SIGUSR2, sigTrap);
}

#ifdef SUNOS41x
void exitCleanup(int exitStatus, void *arg)
#else
void exitCleanup(void)
#endif
{
	if (exitStatus)
		alertMessage("Aborting:", 0, 0,
					 "      xISP has terminated abnormally!\n\n"
					 "Check the standard error output (wherever\n"
					 "that may be redirected) or re-run xisp from\n"
					 "the command line (e.g. from within an xterm\n"
					 "window) and check the error log.");
	fl_finish();
	unlink(upidfname);
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |               Command line options & Resources - Routines               |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

/* Process arguments parsed in resources */

void processArgs(void)
{
	int flags;

	flags = XParseGeometry(geoms,					/* parse geometry from */
						   &winPosX, &winPosY,		/* the input string */
						   &winWidth, &winHeight);
	if ((flags & WidthValue) &&						/* size specified? */
		(flags & HeightValue)) {
		fl_set_form_size(fd_topFrame->topFrame,		/* yes, adjust top form */
						 winWidth, winHeight);
	}
	if ((flags & XValue) && (flags && YValue)) {	/* if position specified */
		if (flags & XNegative)						/* account for negative */
			winPosX = -winWidth;					/* coordinate values */
		if (flags & YNegative)
			winPosY = -winHeight;
		fl_set_form_position(fd_topFrame->topFrame,	/* adjust top form */
							 winPosX, winPosY);		/* size and change */
		placementMethod = FL_PLACE_POSITION;		/* placement method */
	}
	if (iconic)										/* startup as iconic? */
		placementMethod = FL_PLACE_ICONIC;			/* yup, change placement */
	if (clISP > 0 && clISP <= global.numISPs) {
		currentRC = clISP;							/* ISP selected from */
		p_xisprc = &xispOptions[currentRC];			/* command line */
		fl_set_choice(fd_topFrame->ISPDropChoice,
					  currentRC);
	}
	if (expert)										/* "-expert" implies */
		nohints = 1;								/* "-nohints" */
	if (global.logOpts & LOG_HINTS) {				/* if hints enabled from */
		if (nohints) {								/* RC file & "-nohints" */
			global.logOpts &= ~LOG_HINTS;			/* selected => disable */
			writeXisprc(rcfname, xispOptions,		/* hints in log opts and */
						&global);					/* update the RC file */
		}
	}												/* else, when hints off */
	else											/* from RC file, they */
		nohints = 1;								/* can only be turned on */
	if (nohints)									/* from "File" menu */
		fl_set_menu(fd_topFrame->fileMenu,			/* also adjust the menu */
			"Options . . .|Enable hints|Exit");		/* item according to */
	else											/* the state of the */
		fl_set_menu(fd_topFrame->fileMenu,			/* hints flag */
			"Options . . .|Disable hints|Exit");
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                                 Main                                    |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

int main(int argc, char *argv[])
{
	Pixmap icon, imask;
	char *rcsp;
	int tmp, i, newCurrency;
	xisprc_t *p;
	uid_t euid = geteuid();
	uid_t ruid = getuid();

	/*
	 * Perform initializations and sanity checks
	 */
	registerSignalTrap();							/* register sig handlers */
	rcsp = RCSid;									/* keep compiler quiet */
	umask(077);										/* owner read/write only */
	pcode_init();									/* init passwd coding */
	initUIFnames();									/* initialize file names */

	if (seteuid(ruid) == -1)						/* switch to real UID */
		(void) fprintf(stderr, "seteuid(%d): %s\n",	/* just before X11 init */
					   (int)ruid, Strerror(errno));
	colorSqueeze();									/* reduce colormap usage */
	fl_initialize(&argc, argv, "xISP",				/* initialize forms GUI */
				  options, NUMOPTS);
	fl_get_app_resources(resources, NUMOPTS);		/* parse any resources */
	fontSelect();									/* select desired fonts */
	if (seteuid(euid) == -1)						/* switch back to */
		(void) fprintf(stderr, "seteuid(%d): %s\n",	/* effective UID after */
					   (int)euid, Strerror(errno));	/* X11 initialization */
	bgColor(bgcols);								/* parse user bg color */
	fd_actionVerify = create_form_actionVerify();	/* create some forms now */
	fd_alertMessage = create_form_alertMessage();
#ifdef SUNOS41x
	on_exit(exitCleanup, 0);						/* register cleanup proc */
#else
	atexit(exitCleanup);
#endif

	/*
	 * Set locale LC_NUMERIC = C otherwise
	 * floating point numbers might not work for
	 * everybody (users in Greece, for example :))
	 */
	setlocale(LC_NUMERIC, "C");

	readXisprc(rcfname, &xispOptions, &global);		/* try reading RC file */
	tmp = global.numPTTs;							/* save RC's num of PTTs */
	initUDFnames(&global);							/* user spec. file names */
	currentRC = global.dfltISP;						/* set default ISP */
	p_xisprc = &xispOptions[currentRC];
	logs_init(&global, logdirname,					/* init logging module */
			  costfname, logfname, pttfname);
	readXispPTTs(&ptt, &global);					/* read PTT database and */
	for (i=0; i<global.numISPs; i++) {				/* perform sanity check */
		p = &xispOptions[i+1];						/* on the PTT and zone */
		if (p->ispPTT> global.numPTTs-1)			/* selection for all */
			p->ispPTT = global.numPTTs-1;			/* ISPs in the RC file */
		if (p->ispZone> ptt[p->ispPTT].num_zones-1)
			p->ispZone= ptt[p->ispPTT].num_zones-1;
	}
	dialerFD = namedPipe(Pipe);						/* create/open pipe */
	fd_instanceCheck = create_form_instanceCheck();	/* create this form now */
	recordPID();									/* check/write PID file */

	fd_topFrame = create_form_topFrame();			/* create the rest of */
	fd_accountInfo = create_form_accountInfo();		/* the program forms */
	fd_dialInfo = create_form_dialInfo();
	fd_CBInfo = create_form_CBInfo();
	fd_aboutInfo = create_form_aboutInfo();
	fd_exitDialog = create_form_exitDialog();
	fd_renameISP = create_form_renameISP();
	fd_helpInfo = create_form_helpInfo();
	fd_commInfo = create_form_commInfo();
	fd_tcpipInfo = create_form_tcpipInfo();
	fd_logInfo = create_form_logInfo();
	fd_statInfo = create_form_statInfo();
	fd_pttEditor = create_form_pttEditor();
	fd_renamePTT = create_form_renamePTT();
	fd_renameZone = create_form_renameZone();
	fd_envInfo = create_form_envInfo();
	fd_optsTab = create_form_optsTab();

	(void) pppdVersion();							/* check pppd version */
	processArgs();									/* process arguments */
#ifdef XPMANIMATE
	prepAnimPixmaps();								/* prep animation pixmaps */
#endif
	tcid = fl_create_bitmap_cursor(timer_cursor,	/* create custom cursor */
			timer_mask, TIMER_CURSOR_WIDTH,
			TIMER_CURSOR_HEIGHT, 7, 0);
	fl_set_atclose(deleteHandler, NULL);			/* fdelete() callback */
	/*
	 * No abort exceptions beyond this point
	 */
	newCurrency =									/* if currencies for PTT */
		strcmp(ptt[p_xisprc->ispPTT].currency,		/* used by the default */
			   ptt[global.costPTT].currency);		/* ISP and PTT last used */
	if (!(global.logOpts&LOG_NONE)&&newCurrency) {	/* for cost calculation */
		if (actionVerify("The PTT for the default "	/* & logging don't match */
			"ISP bills you in currency\nwhich is "	/* then prompt user with */
			"different from the one used by the "	/* choice. */
			"last\nISP you connected with. This "
			"implies resetting of\nlog files, "
			"continue?\n\n"
			"If you choose \"Yes\", your logs "
			"will be reset, if\nyou choose \"No\""
			", the first ISP in your database\n"
			"using a PTT with matching currency "
			"will be\nselected as default instead.",
			0)
		)											/* if the default ISP is */
			resetLogs();							/* selected, reset logs */
		else {										/* else, search for ISP */
			for (i=1; i<(global.numISPs+1); i++)	/* which uses the PTT */
				if (xispOptions[i].ispPTT ==		/* last used for cost */
					global.costPTT					/* calculation & logging */
				   )
					break;
			if (i > global.numISPs) {				/* if it's not found in */
				alertMessage("xISP: PTT problem",	/* the ISP data-base, */
					0,0, "The PTT (and associated "	/* use the default with */
					"currency) used when you\n"		/* its associated PTT, */
					"last invoked xISP does not "	/* and reset the logs */
					"appear to be related to\n"
					"any of the ISPs listed in "
					"your database. Your default\n"
					"ISP will be used instead ("
					"your log files shall be reset).");
				resetLogs();
			}
			else {									/* if indeed it is found */
				currentRC = i;						/* in the ISP data-base, */
				p_xisprc = &xispOptions[currentRC];	/* switch to this ISP */
				fl_set_choice(fd_topFrame->
					ISPDropChoice, currentRC);
			}
		}
	}
	currentPTT = p_xisprc->ispPTT;					/* set PTT of default ISP */
	global.costPTT = currentPTT;
	if (tmp != global.numPTTs)						/* num of PTTs changed? */
		writeXisprc(rcfname, xispOptions, &global);	/* yes, update RC file */
	p_ptt = &ptt[currentPTT];						/* load selected PTT */
	if (global.logOpts & COST_READOUT)				/* zero call cost if it */
		fl_set_object_label(fd_topFrame->logText,	/* is the preferred on- */
							costStr(p_ptt, 0.0));	/* line counter style */
	p_ptt->current_zone = p_xisprc->ispZone;
	if (! (global.logOpts & LOG_NONE)) {			/* if logging is on */
		readXispCost(&totalTime, &totalCost);		/* read the costs file */
		if (p_ptt->attribs & PTT_BY_UNIT)			/* if the selected PTT */
			totalUnits = totalCost /				/* charges by unit, then */
							p_ptt->cost_quantum;	/* derive units from cost */
	}
	fl_set_app_mainform(fd_topFrame->topFrame);		/* indicate master form */
	topWin = fl_show_form(fd_topFrame->topFrame,	/* realize top-level */
				placementMethod, FL_FULLBORDER,		/* window and save its */
				"X-ISP");							/* ID for later use */
	if (! nohints) fl_set_timer(					/* set tip one-shot timer */
		fd_topFrame->tipTimer, HINT_TRIGGER);
	initClientMsgProc();							/* init WM msg processing */
	fl_addto_selected_xevent(topWin,				/* add event mask to one */
	  StructureNotifyMask|SubstructureNotifyMask);	/* selected for top form */
	fl_register_raw_callback(fd_topFrame->topFrame,	/* install top form */
		StructureNotifyMask|SubstructureNotifyMask|	/* augmented event hook */
		FL_ALL_EVENT, rawEvents);
	fl_get_pixmap_pixmap(fd_topFrame->topIcon,		/* use as window icon */
						 &icon, &imask);			/* the same as that on */
	fl_winicon(topWin, icon, imask);				/* the main form */
#ifdef XPMANIMATE									/* animation uses same */
	amask = imask;									/* transparency mask */
#endif

	possibleActions(DISCONNECTED);					/* mark possible actions */

	if (p_xisprc->operOpts & STARTUP_DIAL ||		/* dial ISP on startup? */
		autodial) {									/* also when forced */
		XFlush(fl_get_display());					/* yes, flush drawing */
		fl_call_object_callback(					/* and push "Connect" */
				fd_topFrame->conButton);
	}

	fl_do_forms();									/* enter main loop */

	return 0;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |         Program forms created with fdesign and annotated by hand        |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

FD_instanceCheck *create_form_instanceCheck(void)
{
  FL_OBJECT *obj;
  FD_instanceCheck *fdui = (FD_instanceCheck *) fl_calloc(1, sizeof(*fdui));

  fdui->instanceCheck = fl_bgn_form(FL_NO_BOX, 429, 163);
  obj = fl_add_box(FL_FLAT_BOX,0,0,429,163,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
  obj = fl_add_pixmap(FL_NORMAL_PIXMAP,21,17,14,36,"");
    fl_set_object_resize(obj, FL_RESIZE_NONE);
	fl_set_object_gravity(obj, FL_West, FL_West);
	fl_set_pixmap_data(obj, excl_xpm);
  obj = fl_add_text(FL_NORMAL_TEXT,41,13,361,21,
					"X-ISP: can't run two instances simultaneously!");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_MEDIUM_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  fdui->msgString = obj = fl_add_text(FL_NORMAL_TEXT,41,37,371,75,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  fdui->instanceCheckAbort = obj = fl_add_button(FL_RETURN_BUTTON,328,121,
												 79,27,"Abort");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
	fl_set_object_bw(obj, -2);
  fdui->instanceCheckContinue = obj = fl_add_button(FL_NORMAL_BUTTON,227,121,
													79,27,"Continue");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
	fl_set_object_bw(obj, -2);
  fl_end_form();

  fdui->instanceCheck->fdui = fdui;

  return fdui;
}

FD_topFrame *create_form_topFrame(void)
{
  FL_OBJECT *obj;
  FD_topFrame *fdui = (FD_topFrame *) fl_calloc(1, sizeof(*fdui));
  int i;

  fdui->topFrame = fl_bgn_form(FL_NO_BOX, 292, 220);
	fl_set_form_minsize(fdui->topFrame, 292,220);
  obj = fl_add_box(FL_FLAT_BOX,0,0,292,220,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
  fdui->conButton = obj = fl_add_button(FL_NORMAL_BUTTON,7,67,69,26,
										"Connect");
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doConnect,0);
	fl_set_object_bw(obj, -2);
	fl_set_object_gravity(obj, FL_SouthWest, FL_SouthWest);
  fdui->intButton = obj = fl_add_button(FL_NORMAL_BUTTON,7,93,69,26,
										"Interrupt");
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doInterrupt,0);
	fl_set_object_bw(obj, -2);
	fl_set_object_gravity(obj, FL_SouthWest, FL_SouthWest);
  fdui->disButton = obj = fl_add_button(FL_NORMAL_BUTTON,7,119,69,26,
										"Disconnect");
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doDisconnect,0);
	fl_set_object_bw(obj, -2);
	fl_set_object_gravity(obj, FL_SouthWest, FL_SouthWest);
  obj = fl_add_box(FL_UP_BOX,0,0,292,29,"");
	fl_set_object_bw(obj, -2);
	fl_set_object_resize(obj, FL_RESIZE_X);
	fl_set_object_gravity(obj, FL_NorthWest, FL_NorthEast);
  fdui->fileMenu = obj = fl_add_menu(FL_PULLDOWN_MENU,9,4,30,19,"File");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doFile,0);
	fl_set_object_gravity(obj, FL_NorthWest, FL_NorthWest);
	fl_set_object_shortcut(obj, "#F", 1);
	if (nohints)
	  fl_set_menu(obj,"Options . . .|Enable hints|Exit");
	else
	  fl_set_menu(obj,"Options . . .|Disable hints|Exit");
	fl_setpup_default_cursor(XC_arrow);
  fdui->logMenu = obj = fl_add_menu(FL_PULLDOWN_MENU,45,4,58,19,"Logging");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doLogging,0);
	fl_set_object_gravity(obj, FL_NorthWest, FL_NorthWest);
	fl_set_object_shortcut(obj, "#L", 1);
	fl_set_menu(obj,"Options . . .|Statistics . . .");
	if (global.logOpts & LOG_NONE)
		fl_set_menu_item_mode(obj, 2, FL_PUP_GREY);
	fl_setpup_default_cursor(XC_arrow);
  fdui->hlpMenu = obj = fl_add_menu(FL_PULLDOWN_MENU,250,4,35,19,"Help");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_RIGHT);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doHelp,0);
	fl_set_object_gravity(obj, FL_NorthEast, FL_NorthEast);
	fl_set_object_shortcut(obj, "#H", 1);
	fl_set_menu(obj, "About xISP|General Info");
	fl_setpup_default_cursor(XC_arrow);
  fdui->lstBrowser = obj = fl_add_browser(FL_NORMAL_BROWSER,80,66,205,79,"");
    fl_set_object_boxtype(obj,FL_DOWN_BOX);
    fl_set_object_color(obj,FL_TOP_BCOL,FL_CYAN);
	fl_set_browser_hscrollbar(obj, FL_OFF);
	fl_set_browser_scrollbarsize(obj, 16, 16);
	fl_set_object_bw(obj, -2);
	fl_set_object_gravity(obj, FL_NorthWest, FL_SouthEast);
  obj = fl_add_box(FL_EMBOSSED_BOX,108,35,177,26,"");
    fl_set_object_color(obj,FL_TOP_BCOL,FL_COL1);
	fl_set_object_resize(obj, FL_RESIZE_X);
	fl_set_object_gravity(obj, FL_NorthWest, FL_NorthEast);
  fdui->ISPDropChoice = obj = fl_add_choice(FL_DROPLIST_CHOICE,111,38,172,
											21,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_color(obj,FL_TOP_BCOL,FL_BLACK);
    fl_set_object_callback(obj,doISPDropChoice,0);
	fl_set_object_resize(obj, FL_RESIZE_X);
	fl_set_object_gravity(obj, FL_NorthWest, FL_NorthEast);
	fl_set_choice_fontsize(obj,FL_NORMAL_SIZE);
	for (i=0; i<global.numISPs; i++)
		if (*(xispOptions[i+1].descr))
			fl_addto_choice(obj,
				clipStr(xispOptions[i+1].descr, FL_NORMAL_SIZE,
						FL_NORMAL_STYLE, 150));
	fl_set_choice(obj, currentRC);
  obj = fl_add_text(FL_NORMAL_TEXT,80,35,30,26,"ISP:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_color(obj,FL_INACTIVE,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lalign(obj,FL_ALIGN_RIGHT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_X);
	fl_set_object_gravity(obj, FL_NorthWest, FL_NorthWest);
  obj = fl_add_text(FL_NORMAL_TEXT,7,152,69,36," Current\n Status:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_color(obj,FL_INACTIVE,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_X);
	fl_set_object_gravity(obj, FL_SouthWest, FL_West);
  fdui->statusText = obj = fl_add_text(FL_NORMAL_TEXT,7,186,69,28,"OFF-LINE");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_color(obj,FL_BLACK,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
	fl_set_object_gravity(obj, FL_SouthWest, FL_West);
  obj = fl_add_text(FL_NORMAL_TEXT,74,152,58,36," Modem\n Speed:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_color(obj,FL_INACTIVE,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_X);
	fl_set_object_gravity(obj, FL_SouthWest, FL_West);
  fdui->speedText = obj = fl_add_text(FL_NORMAL_TEXT,74,186,58,28,
									  EMPTY_SPEED);
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_color(obj,FL_BLACK,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_X);
	fl_set_object_gravity(obj, FL_SouthWest, FL_West);
  obj = fl_add_text(FL_NORMAL_TEXT,130,152,97,36," Assigned\n IP Address:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_color(obj,FL_INACTIVE,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_X);
	fl_set_object_gravity(obj, FL_SouthWest, FL_SouthEast);
  fdui->IPText = obj = fl_add_text(FL_NORMAL_TEXT,130,186,97,28,EMPTY_IP);
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_color(obj,FL_BLACK,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_X);
	fl_set_object_gravity(obj, FL_SouthWest, FL_SouthEast);
	fl_set_object_posthandler(obj, select_IPText);
  fdui->logDescr = obj = fl_add_text(FL_NORMAL_TEXT,225,152,61,36,
		(global.logOpts & COST_READOUT)? RCOST_STR:RTIME_STR);
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_color(obj,FL_INACTIVE,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_X);
	fl_set_object_gravity(obj, FL_SouthEast, FL_SouthEast);
  fdui->logText = obj = fl_add_text(FL_NORMAL_TEXT,225,186,61,28,
		(global.logOpts & COST_READOUT)? "":EMPTY_TIME);
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_color(obj,FL_BLACK,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_X);
	fl_set_object_gravity(obj, FL_SouthEast, FL_SouthEast);
  fdui->topIcon = obj = fl_add_pixmap(FL_NORMAL_PIXMAP,7,32,68,30,"");
    fl_set_object_resize(obj, FL_RESIZE_NONE);
	fl_set_object_gravity(obj, FL_NorthWest, FL_NorthWest);
	fl_set_pixmap_data(obj, xisp_xpm);
#ifdef XPMANIMATE
	fl_set_object_dblbuffer(obj, 1);
#endif

	btimer = fl_add_timer(FL_HIDDEN_TIMER,			/* timer for browser */
						  0, 0, 0, 0, NULL);
	ctimer = fl_add_timer(FL_HIDDEN_TIMER,			/* connection time */
						  0, 0, 0, 0, NULL);
	ltimer = fl_add_timer(FL_HIDDEN_TIMER,			/* link status */
						  0, 0, 0, 0, NULL);
#ifdef XPMANIMATE
	xpmtimer = fl_add_timer(FL_HIDDEN_TIMER,		/* pixmap animation */
						  0, 0, 0, 0, NULL);
#endif
    fl_set_object_callback(btimer, doBUpdate, 0);	/* register callbacks */
    fl_set_object_callback(ctimer, doCTupdate, 0);
    fl_set_object_callback(ltimer, doLPupdate, 0);
#ifdef XPMANIMATE
	fl_set_object_callback(xpmtimer, doXPMUpdate, 0);
#endif
	fl_set_timer(btimer, 0.0);						/* initialize timers */
	fl_set_timer(ctimer, 0.0);
	fl_set_timer(ltimer, 0.0);
#ifdef XPMANIMATE
	fl_set_timer(xpmtimer, 0.0);
#endif

  fdui->tipTimer = obj = fl_add_timer(FL_HIDDEN_TIMER, 0,0,0,0,"");
    fl_set_timer(obj, 0.0);
    fl_set_object_callback(obj,doStartupTip,0);

  fl_end_form();

  fdui->topFrame->fdui = fdui;

  return fdui;
}

FD_accountInfo *create_form_accountInfo(void)
{
  FL_OBJECT *obj;
  FD_accountInfo *fdui = (FD_accountInfo *) fl_calloc(1, sizeof(*fdui));

  fdui->accountInfo = fl_bgn_form(FL_NO_BOX, 540, 350);
  obj = fl_add_box(FL_FLAT_BOX,0,0,540,350,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  fdui->telInput = obj = fl_add_input(FL_NORMAL_INPUT,320,12,206,27,
									  "Phone no(s):");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doTelNoInput,0);
	fl_set_input_maxchars(obj, MAXNUM_TELS*(MAXLEN_PHONE+1));
  fdui->accInput = obj = fl_add_input(FL_NORMAL_INPUT,320,42,206,27,
									  "Account name:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doAccountInput,0);
	fl_set_input_maxchars(obj, MAXLEN_ACCOUNT);
  fdui->pswInput = obj = fl_add_input(FL_SECRET_INPUT,320,72,206,27,
									  "Password:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doPasswdInput,0);
	fl_set_input_maxchars(obj, MAXLEN_PASSWD);
	/*
	   Change secret input character from ' ' to '*';
       look in xisp.h for explanations and details
	*/
	((SPEC *)(obj->spec))->field_char = '*';
  obj = fl_add_frame(FL_ENGRAVED_FRAME,14,215,214,120,"");
    fl_set_object_color(obj,FL_COL1,FL_COL1);
  obj = fl_add_text(FL_NORMAL_TEXT,21,205,151,19,"Authentication protocol:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  fdui->UNInput = obj = fl_add_input(FL_NORMAL_INPUT,320,102,206,27,
									 "User/Name:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doUNInput,0);
	fl_set_input_maxchars(obj, MAXLEN_UNR);
  fdui->remoteInput = obj = fl_add_input(FL_NORMAL_INPUT,320,132,206,27,
										 "Remote name:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doRemoteInput,0);
	fl_set_input_maxchars(obj, MAXLEN_UNR);
  obj = fl_add_box(FL_EMBOSSED_BOX,320,162,206,28,"");
  fdui->ispPttDropChoice = obj = fl_add_choice(FL_DROPLIST_CHOICE,323,165,200,
											   22,"PTT: ");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_callback(obj,doIspPttDropChoice,0);
	fl_set_choice_fontstyle(obj, FL_BOLD_STYLE);
	fl_set_choice_align(obj, FL_ALIGN_LEFT);
  obj = fl_add_box(FL_EMBOSSED_BOX,320,193,206,28,"");
  fdui->ispZoneDropChoice = obj = fl_add_choice(FL_DROPLIST_CHOICE,323,196,200,
												22,"Zone: ");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_callback(obj,doIspZoneDropChoice,0);
	fl_set_choice_fontstyle(obj, FL_BOLD_STYLE);
	fl_set_choice_align(obj, FL_ALIGN_LEFT);

  fdui->authType = fl_bgn_group();
  fdui->authPAPButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,22,300,190,
								25,"PAP (pppd-2.2.0 or earlier)");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doAuthPAPButton,0);
  fdui->authNoneButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,22,225,60,
												  25,"None");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doAuthNoneButton,0);
  fdui->authPAPSButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,22,250,62,
												  25,"PAP");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doAuthPAPSButton,0);
  fdui->authCHAPSButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,22,275,197,
									25,"CHAP (needed for NT-RAS)");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doAuthCHAPSButton,0);
  fl_end_group();

  obj = fl_add_text(FL_NORMAL_TEXT,9,3,209,39,
					"ISP name(s):\n(click selects, double-click edits)");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  fdui->ISPBrowser = obj = fl_add_browser(FL_HOLD_BROWSER,13,43,203,123,"");
    fl_set_object_color(obj,FL_TOP_BCOL,FL_CYAN);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doISPPick,0);
	fl_set_browser_dblclick_callback(obj,doISPPickNEdit,0);
	fl_set_browser_fontsize(obj,FL_NORMAL_SIZE);
	fl_set_browser_scrollbarsize(obj, 16, 16);
	fl_set_object_bw(obj, -2);
  fdui->accountInfoAdd = obj = fl_add_button(FL_NORMAL_BUTTON,13,170,51,27,
											 "Add");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doAccountInfoAdd,0);
	fl_set_object_bw(obj, -2);
  fdui->accountInfoDelete = obj = fl_add_button(FL_NORMAL_BUTTON,166,170,50,27,
												"Delete");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doAccountInfoDelete,0);
	fl_set_object_bw(obj, -2);
  fdui->accountInfoCopy = obj = fl_add_button(FL_NORMAL_BUTTON,64,170,51,27,
											  "Copy");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doAccountInfoCopy,0);
	fl_set_object_bw(obj, -2);
  fdui->accountInfoPaste = obj = fl_add_button(FL_NORMAL_BUTTON,115,170,51,27,
											   "Paste");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doAccountInfoPaste,0);
	fl_set_object_bw(obj, -2);
  obj = fl_add_frame(FL_ENGRAVED_FRAME,244,241,180,94,"");
    fl_set_object_color(obj,FL_COL1,FL_COL1);
  obj = fl_add_text(FL_NORMAL_TEXT,251,231,85,19,"ISP Options:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  fdui->autoRedial = obj = fl_add_checkbutton(FL_PUSH_BUTTON,249,300,164,25,
											  "Re-dial dropped links");
    fl_set_object_color(obj,FL_COL1,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doAutoRedial,0);
	fl_set_button(obj, (p_xisprc->operOpts & AUTO_REDIAL)? 1:0);
  fdui->ISPDefault = obj = fl_add_checkbutton(FL_PUSH_BUTTON,249,250,102,25,
											  "Default ISP");
    fl_set_object_color(obj,FL_COL1,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_callback(obj,doISPDefault,0);
  fdui->ISPAutoDial = obj = fl_add_checkbutton(FL_PUSH_BUTTON,249,275,167,25,
											   "Auto-dial upon startup");
    fl_set_object_color(obj,FL_COL1,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_callback(obj,doISPAutoDial,0);
  obj = fl_add_text(FL_NORMAL_TEXT,433,235,96,114,"On this form you\ndefine "
					"your ISPs\ni.e. the account\nnames, passwords,\n"
					"authentication and\nPTT information,\nas well as "
					"special\nISP attributes.");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lalign(obj,FL_ALIGN_TOP_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);

  fdui->tipTimer = obj = fl_add_timer(FL_HIDDEN_TIMER, 0,0,0,0,"");
    fl_set_timer(obj, 0.0);
    fl_set_object_callback(obj,doAccountInfoTip,0);
  fl_end_form();

  fdui->accountInfo->fdui = fdui;

  return fdui;
}

FD_dialInfo *create_form_dialInfo(void)
{
  FL_OBJECT *obj;
  FD_dialInfo *fdui = (FD_dialInfo *) fl_calloc(1, sizeof(*fdui));

  fdui->dialInfo = fl_bgn_form(FL_NO_BOX, 540, 350);
  obj = fl_add_box(FL_FLAT_BOX,0,0,540,350,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  obj = fl_add_frame(FL_ENGRAVED_FRAME,225,20,300,104,"");
    fl_set_object_color(obj,FL_COL1,FL_COL1);
  obj = fl_add_frame(FL_ENGRAVED_FRAME,13,20,195,316,"");
    fl_set_object_color(obj,FL_COL1,FL_COL1);
  obj = fl_add_text(FL_NORMAL_TEXT,18,7,96,23,"Dialer options:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  fdui->rtrInput = obj = fl_add_input(FL_INT_INPUT,146,34,40,27,
									  "Maximum tries:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doRetryInput,0);
	fl_set_input_maxchars(obj, MAXDIG_RETRY);
	fl_set_input_filter(obj, byteFilter);
  fdui->dlyInput = obj = fl_add_input(FL_INT_INPUT,146,74,40,27,
									  "Inter-dialing\ndelay (seconds):");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doDelayInput,0);
	fl_set_input_maxchars(obj, MAXDIG_DELAY);
	fl_set_input_filter(obj, byteFilter);
  fdui->CNWaitInput = obj = fl_add_input(FL_INT_INPUT,146,114,40,27,
								"Modem connection\ntimeout (seconds):");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCNWaitInput,0);
	fl_set_input_maxchars(obj, MAXDIG_CNWAIT);
	fl_set_input_filter(obj, byteFilter);
  fdui->LCPWaitInput = obj = fl_add_input(FL_INT_INPUT,146,154,40,27,
								"Network connection\ntimeout (seconds):");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doLCPWaitInput,0);
	fl_set_input_maxchars(obj, MAXDIG_LCPWAIT);
	fl_set_input_filter(obj, byteFilter);
  obj = fl_add_text(FL_NORMAL_TEXT,29,197,98,37,"Ring bell when\nconnected:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_RIGHT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);

  fdui->RBGroup = fl_bgn_group();
  fdui->RBYesButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,132,194,57,
											   25,"Yes");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doRBYesButton,0);
  fdui->RBNoButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,132,214,54,
											  25,"No");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doRBNoButton,0);
  fl_end_group();

  obj = fl_add_text(FL_NORMAL_TEXT,37,246,91,37,"ISP server\nwill call back:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_RIGHT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);

  fdui->CBGroup = fl_bgn_group();
  fdui->CBYesButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,132,243,58,
											   25,"Yes");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCBYesButton,0);
  fdui->CBNoButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,132,263,58,
											  25,"No");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCBNoButton,0);
  fl_end_group();

  fdui->CBOptions = obj = fl_add_button(FL_NORMAL_BUTTON,40,293,142,27,
										"Call-back Options");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCBOptions,0);
	fl_set_object_bw(obj, -2);
  obj = fl_add_text(FL_NORMAL_TEXT,230,7,92,23,"Manual Login:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,236,29,261,19,
					"Do you want a terminal after connection?");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);

  fdui->TTGroup = fl_bgn_group();
  fdui->TTYesButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,282,52,58,
											   25,"Yes");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doTTYesButton,0);
  fdui->TTNoButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,392,52,55,
											  25,"No");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doTTNoButton,0);
  fl_end_group();

  fdui->TSGroup = fl_bgn_group();
  fdui->columnsInput = obj = fl_add_input(FL_INT_INPUT,353,85,40,27,
										  "Terminal columns:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doColumnsInput,0);
	fl_set_input_maxchars(obj, 3);
	fl_set_input_filter(obj, byteFilter);
  fdui->rowsInput = obj = fl_add_input(FL_INT_INPUT,463,85,40,27,"and rows:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doRowsInput,0);
	fl_set_input_maxchars(obj, 3);
	fl_set_input_filter(obj, byteFilter);
  fl_end_group();

  fdui->ALGroup = fl_bgn_group();
  obj = fl_add_text(FL_NORMAL_TEXT,219,130,107,23,"Automatic Login:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,218,150,60,19,"Expect:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,329,150,46,19,"Send:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  fdui->expectInput = obj = fl_add_input(FL_MULTILINE_INPUT,224,171,104,
										 166,"");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doExpectInput,0);
	fl_set_input_filter(obj, expectFilter);
	fl_set_input_return(obj, FL_RETURN_ALWAYS);
	fl_set_object_prehandler(obj, slinePreH);
	fl_set_input_hscrollbar(obj, FL_OFF);
	fl_set_input_scrollbarsize(obj, 16, 16);
	fl_set_object_bw(obj, -2);
  fdui->sendInput = obj = fl_add_input(FL_MULTILINE_INPUT,334,171,104,
									   166,"");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doSendInput,0);
	fl_set_input_filter(obj, sendFilter);
	fl_set_input_return(obj, FL_RETURN_ALWAYS);
	fl_set_object_prehandler(obj, slinePreH);
	fl_set_input_hscrollbar(obj, FL_OFF);
	fl_set_input_scrollbarsize(obj, 16, 16);
	fl_set_object_bw(obj, -2);
  fl_end_group();

  obj = fl_add_text(FL_NORMAL_TEXT,440,177,96,168,"On this form you\ndefine "
					"parameters\nrelated to dialing\nthe selected ISP,\nand "
					"to the login\nmethod.\nCall-back options"
					"\nare also available\nbut note that the\nremote end must"
					"\nbe appropriately\nconfigured.");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lalign(obj,FL_ALIGN_TOP_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);

  fdui->tipTimer = obj = fl_add_timer(FL_HIDDEN_TIMER, 0,0,0,0,"");
    fl_set_timer(obj, 0.0);
    fl_set_object_callback(obj,doDialInfoTip,0);
  fl_end_form();

  fdui->dialInfo->fdui = fdui;

  return fdui;
}

FD_CBInfo *create_form_CBInfo(void)
{
  FL_OBJECT *obj;
  FD_CBInfo *fdui = (FD_CBInfo *) fl_calloc(1, sizeof(*fdui));

  fdui->CBInfo = fl_bgn_form(FL_NO_BOX, 444, 306);
  obj = fl_add_box(FL_FLAT_BOX,0,0,444,306,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
  fdui->CBInfoOK = obj = fl_add_button(FL_NORMAL_BUTTON,112,267,79,27,"OK");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCBInfoOK,0);
	fl_set_object_bw(obj, -2);
  fdui->CBInfoCancel = obj = fl_add_button(FL_NORMAL_BUTTON,247,267,79,27,
										   "Cancel");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCBInfoCancel,0);
	fl_set_object_bw(obj, -2);
  fdui->CBdlyInput = obj = fl_add_input(FL_NORMAL_INPUT,160,10,40,27,
										"Call-back delay (sec):");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCBDelayInput,0);
	fl_set_input_maxchars(obj, MAXDIG_CBDELAY);

  obj = fl_add_frame(FL_ENGRAVED_FRAME,15,55,184,123,"");
    fl_set_object_color(obj,FL_COL1,FL_COL1);
  obj = fl_add_text(FL_NORMAL_TEXT,20,42,140,23,"Call-back Login Type:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);

  fdui->LTGroup = fl_bgn_group();
  fdui->LTScriptedButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,31,70,
							138,25,"Auto (via call-\nback script lines)");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doLTScriptedButton,0);
  fdui->LTTerminalButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,31,107,
							145,25,"Terminal after call-\nback connection");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doLTTerminalButton,0);
  fdui->LTRASButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,32,144,
						145,25,"NT RAS call-back");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doLTRASButton,0);
  fl_end_group();

  fdui->TSGroup = fl_bgn_group();
  fdui->columnsInput = obj = fl_add_input(FL_INT_INPUT,213,189,40,27,
										  "Manual terminal columns:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCBColumnsInput,0);
	fl_set_input_maxchars(obj, 3);
	fl_set_input_filter(obj, byteFilter);
  fdui->rowsInput = obj = fl_add_input(FL_INT_INPUT,301,189,40,27,"rows:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCBRowsInput,0);
	fl_set_input_maxchars(obj, 3);
	fl_set_input_filter(obj, byteFilter);
  fl_end_group();

  fdui->CBTelInput = obj = fl_add_input(FL_NORMAL_INPUT,213,226,215,28,
										"NT RAS call-back phone number:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCBTelInput,0);
	fl_set_input_maxchars(obj, MAXLEN_PHONE+1);

  fdui->ALGroup = fl_bgn_group();
  obj = fl_add_text(FL_NORMAL_TEXT,208,34,60,19,"Expect:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,320,34,46,19,"Send:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,208,14,141,21,"Call-back script lines:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  fdui->CBExpectInput = obj = fl_add_input(FL_MULTILINE_INPUT,213,55,104,
										   124,"");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCBExpectInput,0);
	fl_set_input_filter(obj, CBExpectFilter);
	fl_set_input_return(obj, FL_RETURN_ALWAYS);
	fl_set_object_prehandler(obj, slinePreH);
	fl_set_input_hscrollbar(obj, FL_OFF);
	fl_set_input_vscrollbar(obj, FL_OFF);
	fl_set_input_scrollbarsize(obj, 16, 16);
	fl_set_object_bw(obj, -2);
  fdui->CBSendInput = obj = fl_add_input(FL_MULTILINE_INPUT,324,55,104,
										 124,"");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCBSendInput,0);
	fl_set_input_filter(obj, CBSendFilter);
	fl_set_input_return(obj, FL_RETURN_ALWAYS);
	fl_set_object_prehandler(obj, slinePreH);
	fl_set_input_hscrollbar(obj, FL_OFF);
	fl_set_input_vscrollbar(obj, FL_OFF);
	fl_set_input_scrollbarsize(obj, 16, 16);
	fl_set_object_bw(obj, -2);
  fl_end_group();

  fdui->tipTimer = obj = fl_add_timer(FL_HIDDEN_TIMER, 0,0,0,0,"");
    fl_set_timer(obj, 0.0);
    fl_set_object_callback(obj,doCBInfoTip,0);
  fl_end_form();

  fdui->CBInfo->fdui = fdui;

  return fdui;
}

FD_aboutInfo *create_form_aboutInfo(void)
{
  FL_OBJECT *obj;
  FD_aboutInfo *fdui = (FD_aboutInfo *) fl_calloc(1, sizeof(*fdui));
  char vstr[32] = {0};

  fdui->aboutInfo = fl_bgn_form(FL_NO_BOX, 205, 150);
  obj = fl_add_box(FL_FLAT_BOX,0,0,205,150,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  obj = fl_add_pixmap(FL_NORMAL_PIXMAP,13,17,68,30,"");
    fl_set_object_resize(obj, FL_RESIZE_NONE);
	fl_set_object_gravity(obj, FL_West, FL_West);
	fl_set_pixmap_data(obj, xisp_xpm);
  sprintf(vstr, "X-ISP  V.%s%s\ncoded by:", Version, PatchLevel);
  fdui->verString = obj = fl_add_text(FL_NORMAL_TEXT,81,13,116,40, vstr);
    fl_set_object_color(obj,FL_INDIANRED,FL_TOP_BCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  obj = fl_add_text(FL_NORMAL_TEXT,30,53,148,40,
					"Dimitrios P. Bouras\ndbouras@hol.gr");
    fl_set_object_color(obj,FL_INDIANRED,FL_TOP_BCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  fdui->aboutOK = obj = fl_add_button(FL_RETURN_BUTTON,64,106,79,27,"Dismiss");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doAboutOK,0);
	fl_set_object_bw(obj, -2);
  fl_end_form();

  fdui->aboutInfo->fdui = fdui;

  return fdui;
}

FD_exitDialog *create_form_exitDialog(void)
{
  FL_OBJECT *obj;
  FD_exitDialog *fdui = (FD_exitDialog *) fl_calloc(1, sizeof(*fdui));

  fdui->exitDialog = fl_bgn_form(FL_NO_BOX, 230, 117);
  obj = fl_add_box(FL_FLAT_BOX,0,0,230,117,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
  fdui->exitOK = obj = fl_add_button(FL_RETURN_BUTTON,19,77,79,27,"OK");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doExitOK,0);
	fl_set_object_bw(obj, -2);
  fdui->exitCancel = obj = fl_add_button(FL_NORMAL_BUTTON,134,77,79,27,
										 "Cancel");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doExitCancel,0);
	fl_set_object_bw(obj, -2);
  obj = fl_add_text(FL_NORMAL_TEXT,56,9,150,59,
			"Are you sure you\nwant to exit X-ISP and\ndisconnect the link?");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  fdui->exitIcon = obj = fl_add_pixmap(FL_NORMAL_PIXMAP,25,21,14,36,"");
    fl_set_object_resize(obj, FL_RESIZE_NONE);
	fl_set_object_gravity(obj, FL_West, FL_West);
	fl_set_pixmap_data(obj, excl_xpm);
  fl_end_form();

  fdui->exitDialog->fdui = fdui;

  return fdui;
}

FD_renameISP *create_form_renameISP(void)
{
  FL_OBJECT *obj;
  FD_renameISP *fdui = (FD_renameISP *) fl_calloc(1, sizeof(*fdui));

  fdui->renameISP = fl_bgn_form(FL_NO_BOX, 219, 112);
  obj = fl_add_box(FL_FLAT_BOX,0,0,219,112,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
  obj = fl_add_text(FL_NORMAL_TEXT,7,4,145,24,"Enter ISP Description:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  fdui->ISPNameInput = obj = fl_add_input(FL_NORMAL_INPUT,12,29,194,28,"");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_TOP_LEFT);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doISPNameInput,0);
	fl_set_input_maxchars(obj, MAXLEN_DESCR);
  fdui->ISPNameEditOK = obj = fl_add_button(FL_RETURN_BUTTON,12,71,79,27,"OK");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doISPNameEditOK,0);
	fl_set_object_bw(obj, -2);
  fdui->ISPNameEditCancel = obj = fl_add_button(FL_NORMAL_BUTTON,127,71,79,27,
												"Cancel");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doISPNameEditCancel,0);
	fl_set_object_bw(obj, -2);
  fl_end_form();

  fdui->renameISP->fdui = fdui;

  return fdui;
}

FD_helpInfo *create_form_helpInfo(void)
{
  FL_OBJECT *obj;
  FD_helpInfo *fdui = (FD_helpInfo *) fl_calloc(1, sizeof(*fdui));

  fdui->helpInfo = fl_bgn_form(FL_NO_BOX, 510, 480);
  obj = fl_add_box(FL_FLAT_BOX,0,0,510,480,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
  fdui->helpBrowser = obj = fl_add_browser(FL_NORMAL_BROWSER,10,11,489,423,"");
    fl_set_object_color(obj,FL_TOP_BCOL,FL_CYAN);
    fl_set_browser_fontsize(obj,FL_NORMAL_SIZE);
	fl_set_browser_scrollbarsize(obj, 16, 16);
	fl_set_object_bw(obj, -2);
  fdui->helpInfoOK = obj = fl_add_button(FL_RETURN_BUTTON,421,444,79,27,"OK");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doHelpInfoOK,0);
	fl_set_object_bw(obj, -2);
  obj = fl_add_text(FL_NORMAL_TEXT,8,435,329,39,
					"To browse, use PgUp, PgDn, and/or up-/down-arrow\n"
					"keyboard keys, or the scroll-bar via your mouse.");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  fl_end_form();

  fdui->helpInfo->fdui = fdui;

  return fdui;
}

FD_alertMessage *create_form_alertMessage(void)
{
  FL_OBJECT *obj;
  FD_alertMessage *fdui = (FD_alertMessage *) fl_calloc(1, sizeof(*fdui));

  fdui->alertMessage = fl_bgn_form(FL_NO_BOX, 320, 122);
  obj = fl_add_box(FL_FLAT_BOX,0,0,320,122,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
  obj = fl_add_pixmap(FL_NORMAL_PIXMAP,17,20,14,36,"");
    fl_set_object_resize(obj, FL_RESIZE_NONE);
	fl_set_object_gravity(obj, FL_NorthWest, FL_NorthWest);
	fl_set_pixmap_data(obj, excl_xpm);
  fdui->alertMessageStr = obj = fl_add_text(FL_NORMAL_TEXT,37,11,266,69,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  fdui->alertMessageDismiss = obj = fl_add_button(FL_RETURN_BUTTON,225,86,
												  79,27,"Dismiss");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
	fl_set_object_gravity(obj, FL_SouthEast, FL_SouthEast);
	fl_set_object_bw(obj, -2);
  fdui->noMoreHints = obj = fl_add_round3dbutton(FL_PUSH_BUTTON,9,86,166,25,
												 "No more hints please!");
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
	fl_set_object_gravity(obj, FL_SouthWest, FL_SouthWest);
  fl_end_form();

  fdui->alertMessage->fdui = fdui;

  return fdui;
}

FD_commInfo *create_form_commInfo(void)
{
  FL_OBJECT *obj;
  FD_commInfo *fdui = (FD_commInfo *) fl_calloc(1, sizeof(*fdui));

  fdui->commInfo = fl_bgn_form(FL_NO_BOX, 540, 350);
  obj = fl_add_box(FL_FLAT_BOX,0,0,540,350,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);

  obj = fl_add_frame(FL_ENGRAVED_FRAME,11,259,237,77,"");
    fl_set_object_color(obj,FL_COL1,FL_COL1);
  obj = fl_add_text(FL_NORMAL_TEXT,18,249,152,18,"Software Compression:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);

  fdui->SWCType = fl_bgn_group();
  fdui->SWCBSDButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,13,268,64,
												25,"BSD");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doSWCBSDButton,0);
  fdui->SWCDeflateButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,99,268,80,
													25,"Deflate");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doSWCDeflateButton,0);
  fdui->SWCOffButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,180,268,61,
												25,"None");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doSWCOffButton,0);
  fl_end_group();

  obj = fl_add_frame(FL_ENGRAVED_FRAME,263,14,262,95,"");
    fl_set_object_color(obj,FL_COL1,FL_COL1);
  obj = fl_add_text(FL_NORMAL_TEXT,270,3,145,20,"Serial Port Baud Rate:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);

  fdui->modemSpeedType = fl_bgn_group();
  fdui->speedButton[6] = obj = fl_add_checkbutton(FL_RADIO_BUTTON,449,48,65,
												  25,"57600");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doSpeedButton,6);
  fdui->speedButton[7] = obj = fl_add_checkbutton(FL_RADIO_BUTTON,271,73,75,
												  25,"115200");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doSpeedButton,7);
  fdui->speedButton[4] = obj = fl_add_checkbutton(FL_RADIO_BUTTON,271,48,65,
												  25,"19200");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doSpeedButton,4);
  fdui->speedButton[5] = obj = fl_add_checkbutton(FL_RADIO_BUTTON,360,48,65,
												  25,"38400");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doSpeedButton,5);
  fdui->speedButton[2] = obj = fl_add_checkbutton(FL_RADIO_BUTTON,393,23,59,
												  25,"4800");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doSpeedButton,2);
  fdui->speedButton[3] = obj = fl_add_checkbutton(FL_RADIO_BUTTON,456,23,59,
												  25,"9600");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doSpeedButton,3);
  fdui->speedButton[1] = obj = fl_add_checkbutton(FL_RADIO_BUTTON,330,23,59,
												  25,"2400");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doSpeedButton,1);
  fdui->speedButton[0] = obj = fl_add_checkbutton(FL_RADIO_BUTTON,271,23,58,
												  25,"1200");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doSpeedButton,0);
  fdui->speedButton[8] = obj = fl_add_checkbutton(FL_RADIO_BUTTON,360,73,75,
												  25,"Custom:");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doSpeedButton,8);
  fdui->customSpeedInput = obj = fl_add_input(FL_INT_INPUT,438,73,75,27,"");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCustomSpeedInput,0);
	fl_set_input_maxchars(obj, MAXDIG_BAUDRATE);
  fl_end_group();

  obj = fl_add_frame(FL_ENGRAVED_FRAME,263,123,262,61,"");
    fl_set_object_color(obj,FL_COL1,FL_COL1);
  obj = fl_add_text(FL_NORMAL_TEXT,270,112,89,19,"Flow control:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);

  fdui->flowCtrlType = fl_bgn_group();
  fdui->HWButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,271,130,210,
											26,"Hardware  (RTS/CTS signals)");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doHWButton,0);
  fdui->SWButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,271,152,250,
							25,"Software  (XON/XOFF control chars)");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doSWButton,0);
  fl_end_group();

  obj = fl_add_frame(FL_ENGRAVED_FRAME,263,198,174,139,"");
    fl_set_object_color(obj,FL_COL1,FL_COL1);
  obj = fl_add_text(FL_NORMAL_TEXT,270,187,70,21,"Advanced:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,268,208,60,21,"Escape:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);

  fdui->escapeType = fl_bgn_group();
  fdui->escapeYesButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,332,208,55,
												   25,"Yes");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doEscapeYesButton,0);
    fl_set_button(obj, 1);
  fdui->escapeNoButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,332,231,47,
												  25,"No");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doEscapeNoButton,0);
  fl_end_group();

  obj = fl_add_frame(FL_ENGRAVED_FRAME,11,170,237,75,"");
    fl_set_object_color(obj,FL_COL1,FL_COL1);
  obj = fl_add_text(FL_NORMAL_TEXT,18,159,102,20,"Dialing method:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);

  fdui->modemDialType = fl_bgn_group();
  fdui->toneButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,13,179,61,
											  25,"Tone");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doToneButton,0);
  fdui->pulseButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,99,179,68,
											   25,"Pulse");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doPulseButton,0);
  fdui->ISDNButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,180,179,63,
											  25,"ISDN");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doISDNButton,0);
  fl_end_group();

  fdui->deviceInput = obj = fl_add_input(FL_NORMAL_INPUT,159,12,90,28,
										 "Serial port device name:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doDeviceInput,0);
	fl_set_input_maxchars(obj, MAXLEN_DEVICE);
  fdui->modemResetInput = obj = fl_add_input(FL_NORMAL_INPUT,91,50,158,28,
											 "Modem\nreset string:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doModemResetInput,0);
	fl_set_input_maxchars(obj, MAXLEN_MDMCMD);
  fdui->modemInitInput = obj = fl_add_input(FL_NORMAL_INPUT,126,86,123,28,
											"Modem\ninitialization string:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doModemInitInput,0);
	fl_set_input_maxchars(obj, MAXLEN_MDMCMD);
  fdui->modemConnectInput = obj = fl_add_input(FL_NORMAL_INPUT,103,124,146,28,
											   "Modem\nconnect string:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doModemConnectInput,0);
	fl_set_input_maxchars(obj, MAXLEN_MDMSTR);
  fdui->modemDialInput = obj = fl_add_input(FL_NORMAL_INPUT,109,207,128,28,
											"Dialing extras:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doModemDialInput,0);
	fl_set_input_maxchars(obj, MAXLEN_DIALEXTRA);
  fdui->SWCInput = obj = fl_add_input(FL_INT_INPUT,193,297,44,27,
									  "Compression level (8-15):");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doSWCInput,0);
	fl_set_input_maxchars(obj, MAXDIG_BSDCOMP);
	fl_set_input_filter(obj, SWCInputFilter);
  fdui->asyncmapInput = obj = fl_add_input(FL_NORMAL_INPUT,344,299,80,28,
										   "Asyncmap:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doAsyncmapInput,0);
	fl_set_input_maxchars(obj, MAXDIG_ASYNCMAP);
	fl_set_input_filter(obj, AMInputFilter);
  fdui->escapeInput = obj = fl_add_input(FL_NORMAL_INPUT,344,258,80,28,
										 "Byte List:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doEscapeInput,0);
	fl_set_input_maxchars(obj, MAXLEN_ESCAPE);
	fl_set_input_filter(obj, EInputFilter);
  obj = fl_add_text(FL_NORMAL_TEXT,440,201,94,140,"On this form you\ndefine "
					"(per ISP)\nmodem-specific\nparameters, and\nadjust "
					"settings\nrelated to the\nPPP protocol;\nsuch are "
					"software\ncompression and\nbyte filtering.");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lalign(obj,FL_ALIGN_TOP_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);

  fdui->tipTimer = obj = fl_add_timer(FL_HIDDEN_TIMER, 0,0,0,0,"");
    fl_set_timer(obj, 0.0);
    fl_set_object_callback(obj,doCommInfoTip,0);
  fl_end_form();

  fdui->commInfo->fdui = fdui;

  return fdui;
}

FD_tcpipInfo *create_form_tcpipInfo(void)
{
  FL_OBJECT *obj;
  FD_tcpipInfo *fdui = (FD_tcpipInfo *) fl_calloc(1, sizeof(*fdui));

  fdui->tcpipInfo = fl_bgn_form(FL_NO_BOX, 540, 350);
  obj = fl_add_box(FL_FLAT_BOX,0,0,540,350,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
  obj = fl_add_frame(FL_ENGRAVED_FRAME,387,16,138,150,"");
    fl_set_object_color(obj,FL_COL1,FL_COL1);
  obj = fl_add_frame(FL_ENGRAVED_FRAME,15,16,360,150,"");
    fl_set_object_color(obj,FL_COL1,FL_COL1);

  obj = fl_add_text(FL_NORMAL_TEXT,21,2,129,24,"Addressing/Routing:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  fdui->localIPInput = obj = fl_add_input(FL_NORMAL_INPUT,252,33,113,28,
										  "Local IP\naddress:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doLocalIPInput,0);
	fl_set_input_maxchars(obj, MAXLEN_IP);
	fl_set_input_filter(obj, IPInputFilter);

  obj = fl_add_text(FL_NORMAL_TEXT,33,28,95,37,"Dynamic local\nIP address:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_RIGHT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);

  fdui->ALTypeGroup = fl_bgn_group();
  fdui->ALYesButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,127,25,58,
											   25,"Yes");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doALYesButton,0);
  fdui->ALNoButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,127,45,55,
											  25,"No");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doALNoButton,0);
  fl_end_group();

  fdui->remoteIPInput = obj = fl_add_input(FL_NORMAL_INPUT,252,82,113,28,
										   "Remote IP\naddress:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doRemoteIPInput,0);
	fl_set_input_maxchars(obj, MAXLEN_IP);
	fl_set_input_filter(obj, IPInputFilter);
  obj = fl_add_text(FL_NORMAL_TEXT,20,76,109,37,
					"Dynamic remote\nIP address:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_RIGHT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);

  fdui->ARTypeGroup = fl_bgn_group();
  fdui->ARYesButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,127,73,58,
											   25,"Yes");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doARYesButton,0);
  fdui->ARNoButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,127,93,55,
											  25,"No");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doARNoButton,0);
  fl_end_group();

  obj = fl_add_text(FL_NORMAL_TEXT,43,132,214,22,
					"Add default route to routing table:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);

  fdui->DRTypeGroup = fl_bgn_group();
  fdui->DRYesButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,258,132,58,
											   25,"Yes");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doDRYesButton,0);
  fdui->DRNoButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,314,132,55,
											  25,"No");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doDRNoButton,0);
  fl_end_group();

  obj = fl_add_text(FL_NORMAL_TEXT,394,2,69,24,"Interface:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  fdui->NMInput = obj = fl_add_input(FL_NORMAL_INPUT,400,47,113,28,"Netmask:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_TOP_LEFT);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doNMInput,0);
	fl_set_input_maxchars(obj, MAXLEN_IP);
	fl_set_input_filter(obj, IPInputFilter);
  fdui->MTUInput = obj = fl_add_input(FL_INT_INPUT,465,84,48,28,"MTU:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doMTUInput,0);
	fl_set_input_maxchars(obj, MAXDIG_MTRU);
  fdui->MRUInput = obj = fl_add_input(FL_INT_INPUT,465,121,48,28,"MRU:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doMRUInput,0);
	fl_set_input_maxchars(obj, MAXDIG_MTRU);
  obj = fl_add_frame(FL_ENGRAVED_FRAME,15,185,420,150,"");
    fl_set_object_color(obj,FL_COL1,FL_COL1);
  obj = fl_add_text(FL_NORMAL_TEXT,21,172,214,24,
					"DNS support (via ip-up/ip-down):");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);

  fdui->NRTypeGroup = fl_bgn_group();
  fdui->DNSNoButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,33,195,50,
											   25,"No");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doDNSNoButton,0);
  fdui->DNSAutoButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,33,226,188,
								25,"Automatic\n(ISP assigned)");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doDNSAutoButton,0);
  fdui->DNSManualButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,33,258,80,
												   25,"Manual");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doDNSManualButton,0);
  fl_end_group();

  fdui->pDNSInput = obj = fl_add_input(FL_NORMAL_INPUT,312,194,113,28,
									   "Primary DNS Server:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doPDNSInput,0);
	fl_set_input_maxchars(obj, MAXLEN_IP);
	fl_set_input_filter(obj, IPInputFilter);
  fdui->sDNSInput = obj = fl_add_input(FL_NORMAL_INPUT,312,231,113,28,
									   "Secondary DNS Server:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doSDNSInput,0);
	fl_set_input_maxchars(obj, MAXLEN_IP);
	fl_set_input_filter(obj, IPInputFilter);
  fdui->domainInput = obj = fl_add_input(FL_NORMAL_INPUT,312,295,113,28,
										 "Domain\nname:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doDomainInput,0);

  fdui->DNTypeGroup = fl_bgn_group();
  obj = fl_add_text(FL_NORMAL_TEXT,20,291,109,37,
					"Set resolver\ndefault domain:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_RIGHT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  fdui->DNNoButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,197,298,55,
											  25,"No");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doDNNoButton,0);
  fdui->DNYesButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,137,298,58,
											   25,"Yes");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doDNYesButton,0);
  fl_end_group();

  obj = fl_add_text(FL_NORMAL_TEXT,440,187,87,153,"On this form you\nadjust "
					"TCP/IP-\nspecific settings.\nYou also have\nthe option "
					"to\ndefine DNS\nservers which\nshall  be enabled\nonly "
					"when you\nconnect to the\nselected ISP.");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lalign(obj,FL_ALIGN_TOP_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);

  fdui->tipTimer = obj = fl_add_timer(FL_HIDDEN_TIMER, 0,0,0,0,"");
    fl_set_timer(obj, 0.0);
    fl_set_object_callback(obj,doTcpipInfoTip,0);
  fl_end_form();

  fdui->tcpipInfo->fdui = fdui;

  return fdui;
}

FD_logInfo *create_form_logInfo(void)
{
  FL_OBJECT *obj;
  FD_logInfo *fdui = (FD_logInfo *) fl_calloc(1, sizeof(*fdui));

  fdui->logInfo = fl_bgn_form(FL_NO_BOX, 399, 339);
  obj = fl_add_box(FL_FLAT_BOX,0,0,399,339,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
  obj = fl_add_frame(FL_ENGRAVED_FRAME,14,19,178,72,"");
  obj = fl_add_frame(FL_ENGRAVED_FRAME,206,19,178,72,"");
  obj = fl_add_text(FL_NORMAL_TEXT,19,10,163,19,"On-line counter displays:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);

  fdui->OLGroup = fl_bgn_group();
  fdui->OLTimeButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,19,30,142,
												25,"Time (HH:MM:SS)");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doOLTimeButton,0);
  fdui->OLChargeButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,19,54,165,
												  25,"Telephone call charges");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doOLChargeButton,0);
  fl_end_group();

  obj = fl_add_text(FL_NORMAL_TEXT,211,10,141,19,"Keep connection logs:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);

  fdui->logGroup = fl_bgn_group();
  fdui->logNoButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,211,31,73,
											   25,"No");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doLogNoButton,0);
  fdui->logMonthButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,286,31,85,
												  25,"Monthly");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doLogMonthButton,0);
  fdui->logWeekButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,211,54,73,
												 25,"Weekly");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doLogWeekButton,0);
  fdui->logBiMonthButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,286,54,88,
													25,"Bimonthly");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doLogBiMonthButton,0);
  fl_end_group();

  fdui->logInfoOK = obj = fl_add_button(FL_RETURN_BUTTON,232,298,79,27,"OK");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doLogInfoOK,0);
	fl_set_object_bw(obj, -2);
  fdui->logInfoCancel = obj = fl_add_button(FL_NORMAL_BUTTON,88,298,79,27,
											"Cancel");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doLogInfoCancel,0);
	fl_set_object_bw(obj, -2);
  obj = fl_add_text(FL_NORMAL_TEXT,7,96,110,39,"Phone Company\n(PTT):");
    fl_set_object_boxtype(obj,FL_NO_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_RIGHT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
	fl_set_object_gravity(obj, FL_South, FL_South);
  obj = fl_add_box(FL_EMBOSSED_BOX,118,103,267,28,"");
    fl_set_object_color(obj,FL_TOP_BCOL,FL_COL1);
	fl_set_object_resize(obj, FL_RESIZE_NONE);
	fl_set_object_gravity(obj, FL_South, FL_South);
  fdui->pttDropChoice = obj = fl_add_choice(FL_DROPLIST_CHOICE,121,106,
											262,22,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_color(obj,FL_TOP_BCOL,FL_BLACK);
    fl_set_object_callback(obj,doPttDropChoice,0);
	fl_set_object_resize(obj, FL_RESIZE_NONE);
	fl_set_object_gravity(obj, FL_South, FL_South);
	fl_set_choice_fontsize(obj,FL_NORMAL_SIZE);

  obj = fl_add_text(FL_NORMAL_TEXT,9,138,108,28,"Charging Zone:");
    fl_set_object_boxtype(obj,FL_NO_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_RIGHT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  obj = fl_add_box(FL_EMBOSSED_BOX,118,139,267,28,"");
    fl_set_object_color(obj,FL_TOP_BCOL,FL_COL1);
	fl_set_object_resize(obj, FL_RESIZE_NONE);
	fl_set_object_gravity(obj, FL_South, FL_South);
  fdui->zoneDropChoice = obj = fl_add_choice(FL_DROPLIST_CHOICE,121,142,
											 262,22,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_color(obj,FL_TOP_BCOL,FL_BLACK);
    fl_set_object_callback(obj,doZoneDropChoice,0);
	fl_set_object_resize(obj, FL_RESIZE_NONE);
	fl_set_object_gravity(obj, FL_South, FL_South);
	fl_set_choice_fontsize(obj,FL_NORMAL_SIZE);

  fdui->LogInfoPTTEdit = obj = fl_add_button(FL_NORMAL_BUTTON,72,179,105,27,
											 "Edit PTT Info");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doLogInfoPTTEdit,0);
	fl_set_object_bw(obj, -2);
  fdui->LogInfoPTTAdd = obj = fl_add_button(FL_NORMAL_BUTTON,177,179,104,27,
											"Add PTT");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doLogInfoPTTAdd,0);
	fl_set_object_bw(obj, -2);
  fdui->logInfoPTTRemove = obj = fl_add_button(FL_NORMAL_BUTTON,281,179,104,
											   27,"Remove PTT");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doLogInfoPTTRemove,0);
	fl_set_object_bw(obj, -2);
  obj = fl_add_text(FL_NORMAL_TEXT,5,214,67,70,
					"Phone\ncompany\nbrief info\nbrowser:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_RIGHT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  fdui->costBrowser = obj = fl_add_browser(FL_NORMAL_BROWSER,72,217,313,66,"");
    fl_set_object_color(obj,FL_TOP_BCOL,FL_CYAN);
	fl_set_browser_fontsize(obj,FL_NORMAL_SIZE);
	fl_set_browser_hscrollbar(obj, FL_OFF);
	fl_set_browser_scrollbarsize(obj, 16, 16);
	fl_set_object_bw(obj, -2);
  fl_end_form();

  fdui->logInfo->fdui = fdui;

  /* We do some more custom stuff here */
  fdui->cdata = (char *) fl_calloc(MAXLEN_CURRENCY+1, sizeof(char));

  return fdui;
}

FD_statInfo *create_form_statInfo(void)
{
  FL_OBJECT *obj;
  FD_statInfo *fdui = (FD_statInfo *) fl_calloc(1, sizeof(*fdui));

  fdui->statInfo = fl_bgn_form(FL_NO_BOX, 463, 257);
  obj = fl_add_box(FL_FLAT_BOX,0,0,463,257,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
  fdui->statsBrowser = obj = fl_add_browser(FL_NORMAL_BROWSER,13,36,213,95,"");
    fl_set_object_color(obj,FL_TOP_BCOL,FL_CYAN);
    fl_set_browser_fontsize(obj,FL_NORMAL_SIZE);
	fl_set_browser_hscrollbar(obj, FL_OFF);
	fl_set_browser_scrollbarsize(obj, 16, 16);
	fl_set_object_bw(obj, -2);
  obj = fl_add_text(FL_NORMAL_TEXT,10,10,134,21,"Time/Cost statistics:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  fdui->statsChart = obj = fl_add_chart(FL_BAR_CHART,13,145,213,95,"");
    fl_set_object_boxtype(obj,FL_DOWN_BOX);
    fl_set_object_color(obj,FL_TOP_BCOL,FL_COL1);
	fl_set_object_bw(obj, -2);
  obj = fl_add_text(FL_NORMAL_TEXT,235,10,154,21,"Yearly cost breakdown:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  fdui->pieChart = obj = fl_add_chart(FL_PIE_CHART,236,36,213,167,"");
    fl_set_object_boxtype(obj,FL_DOWN_BOX);
    fl_set_object_color(obj,FL_TOP_BCOL,FL_COL1);
	fl_set_object_bw(obj, -2);
  fdui->statInfoOK = obj = fl_add_button(FL_RETURN_BUTTON,370,217,79,27,
										 "Dismiss");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doStatsInfoOK,0);
	fl_set_object_bw(obj, -2);
  fl_end_form();

  fdui->statInfo->fdui = fdui;

  return fdui;
}

FD_pttEditor *create_form_pttEditor(void)
{
  FL_OBJECT *obj;
  FD_pttEditor *fdui = (FD_pttEditor *) fl_calloc(1, sizeof(*fdui));

  fdui->pttEditor = fl_bgn_form(FL_NO_BOX, 615, 640);
  obj = fl_add_box(FL_FLAT_BOX,0,0,615,640,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
  obj = fl_add_frame(FL_ENGRAVED_FRAME,14,384,587,242,"");
    fl_set_object_color(obj,FL_COL1,FL_COL1);
  obj = fl_add_frame(FL_ENGRAVED_FRAME,314,16,287,354,"");
  obj = fl_add_frame(FL_ENGRAVED_FRAME,14,153,287,217,"");
    fl_set_object_color(obj,FL_COL1,FL_COL1);
  obj = fl_add_text(FL_NORMAL_TEXT,21,142,89,20,"PTT charges:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  fdui->zonePick = obj = fl_add_browser(FL_HOLD_BROWSER,327,46, 260,65,"");
    fl_set_object_color(obj,FL_TOP_BCOL,FL_CYAN);
    fl_set_object_callback(obj,doZonePick,0);
	fl_set_browser_dblclick_callback(obj,doZonePickNEdit,0);
	fl_set_browser_fontsize(obj,FL_NORMAL_SIZE);
	fl_set_browser_hscrollbar(obj, FL_OFF);
	fl_set_browser_scrollbarsize(obj, 16, 16);
	fl_set_object_bw(obj, -2);
  obj = fl_add_text(FL_NORMAL_TEXT,321,26,274,21,
					"Zone:  (click picks, double-click edits name)");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,21,373,259,21,
					"Category rules (for zone selected above):");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,24,397,115,28,"Category number:");
    fl_set_object_boxtype(obj,FL_NO_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  obj = fl_add_text(FL_NORMAL_TEXT,359,589,8,20,":");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,399,590,8,20,":");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,321,545,103,23,"Rule start time:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,507,590,8,20,":");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,547,591,8,20,":");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,470,545,103,23,"Rule end time:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,328,565,29,21,"HH");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,368,565,29,21,"MM");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,408,565,29,21,"SS");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,476,566,29,21,"HH");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,516,566,29,21,"MM");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,556,566,29,21,"SS");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,24,427,103,23,"Category type:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  fdui->ruleDateStr = obj = fl_add_text(FL_NORMAL_TEXT,321,427,71,21,
										"Rule date:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  fdui->defaultTariffUnits = obj = fl_add_text(FL_NORMAL_TEXT,468,119,118,23,
											   CUNIT_STR);
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,24,338,130,23,"Currency placement:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  fdui->pttEditOK = obj = fl_add_button(FL_RETURN_BUTTON,13,98,79,27,"OK");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doPttEditOK,0);
	fl_set_object_bw(obj, -2);
  fdui->pttEditCancel = obj = fl_add_button(FL_NORMAL_BUTTON,118,98,79,27,
											"Cancel");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doPttEditCancel,0);
	fl_set_object_bw(obj, -2);
  fdui->ruleTariffUnits = obj = fl_add_text(FL_NORMAL_TEXT,450,400,118,23,
											CUNIT_STR);
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  fdui->pttEditHelp = obj = fl_add_button(FL_NORMAL_BUTTON,223,98,79,27,
										  "Help");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doPttEditHelp,0);
	fl_set_object_bw(obj, -2);

  fdui->CMType = fl_bgn_group();
  fdui->CUnitButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,22,165,78,
											   25,"By unit");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCUnitButton,0);
  fdui->CMinuteButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,22,200,95,
												 25,"Per minute");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCMinuteButton,0);
  fdui->CSecondsButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,22,235,117,
												  25,"Per second(s)");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCSecondsButton,0);
  fl_end_group();


  fdui->CPType = fl_bgn_group();
  fdui->CPLButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,160,338,56,
											 25,"Left");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCPLButton,0);
  fdui->CPRButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,225,338,65,
											 25,"Right");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCPRButton,0);
  fl_end_group();

  fdui->MCType = fl_bgn_group();
  fdui->MLLButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,325,168,185,
											 25,"Linear, same for all zones");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doMLLButton,0);
  fdui->MLNButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,325,193,185,
											 25,"Non-linear, zone specific");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doMLNButton,0);
  fl_end_group();

  fdui->ruleCounter = obj = fl_add_counter(FL_NORMAL_COUNTER,154,401,
										   135,23,"");
    fl_set_object_color(obj,FL_MCOL,FL_BLACK);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doRuleCounter,0);
    fl_set_counter_precision(obj, 0);
    fl_set_counter_step(obj, 1, 5);
  obj = fl_add_text(FL_NORMAL_TEXT,322,5,112,21,"Zone information:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);

  fdui->ZEDType = fl_bgn_group();
  fdui->ZEDYesButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,325,263,53,
												25,"Yes");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doZEDYesButton,0);
  fdui->ZEDNoButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,390,263,53,
											   25,"No");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doZEDNoButton,0);
  fl_end_group();

  fdui->REDType = fl_bgn_group();
  fdui->RDYesButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,38,594,53,
											   25,"Yes");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doRDYesButton,0);
  fdui->RDNoButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,99,594,53,
											  25,"No");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doRDNoButton,0);
  fl_end_group();


  fdui->CTType = fl_bgn_group();
  fdui->TWDButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,28,450,83,
											 25,"Weekday");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doTButton,2);
  fdui->TSDButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,28,473,87,
											 25,"Saturday");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doTButton,3);
  fdui->THRButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,28,519,127,
											 25,"Holiday-relative");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doTButton,6);
  fdui->TUDButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,160,473,83,
											 25,"Sunday");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doTButton,4);
  fdui->TWEButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,28,496,84,
											 25,"Weekend");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doTButton,5);
  fdui->THAButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,160,519,134,
											 25,"Holiday-absolute");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doTButton,7);
  fdui->TWDSButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,160,450,132,
											  25,"Weekday-special");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doTButton,8);
  fdui->TWESButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,160,496,132,
											  25,"Weekend-special");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doTButton,9);
  fdui->TAWButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,28,542,88,
											 25,"All-week");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doTButton,1);
  fl_end_group();


  fdui->ZMLType = fl_bgn_group();
  fdui->ZMLYesButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,180,594,53,
												25,"Yes");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doZMLYesButton,0);
  fdui->ZMLNoButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,241,594,53,
											   25,"No");
    fl_set_object_color(obj,FL_MCOL,FL_WHEAT);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doZMLNoButton,0);
  fl_end_group();


  fdui->pttNameInput = obj = fl_add_input(FL_NORMAL_INPUT,80,11,222,28,
										  "PTT name:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doPttNameInput,0);
	fl_set_input_maxchars(obj, MAXLEN_PTTNAME);
  fdui->zonesInput = obj = fl_add_input(FL_NORMAL_INPUT,109,48,36,27,
										"Charging zones:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doZonesInput,0);
	fl_set_input_maxchars(obj, 1);
  fdui->categoriesInput = obj = fl_add_input(FL_NORMAL_INPUT,266,48,36,27,
											 "Categories/zone:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCategoriesInput,0);
	fl_set_input_maxchars(obj, 2);
  fdui->minUnitsInput = obj = fl_add_input(FL_NORMAL_INPUT,235,165,50,27,
										   "Minimum units:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doMinUnitsInput,0);
	fl_set_input_maxchars(obj, 2);
  fdui->minCostInput = obj = fl_add_input(FL_NORMAL_INPUT,235,200,50,27,
										  "Minimum cost:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doMinCostInput,0);
	fl_set_input_maxchars(obj, 6);
  fdui->CSecondsInput = obj = fl_add_input(FL_NORMAL_INPUT,235,235,50,27,
										   "Period (sec):");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCSecondsInput,0);
	fl_set_input_maxchars(obj, 2);
  fdui->unitPriceInput = obj = fl_add_input(FL_NORMAL_INPUT,110,270,55,27,
											"Price/unit:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doUnitPriceInput,0);
	fl_set_input_maxchars(obj, 8);
  fdui->currencyInput = obj = fl_add_input(FL_NORMAL_INPUT,235,270,50,27,
										   "Currency:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCurrencyInput,0);
	fl_set_input_maxchars(obj, MAXLEN_CURRENCY);
  fdui->decimalsInput = obj = fl_add_input(FL_NORMAL_INPUT,235,305,50,27,
								"Decimal digits for bill printouts:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doDecimalsInput,0);
	fl_set_input_maxchars(obj, 1);
  fdui->baseChargeInput = obj = fl_add_input(FL_NORMAL_INPUT,412,118,55,27,
											 "Default tariff:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doBaseChargeInput,0);
	fl_set_input_maxchars(obj, 6);

  fdui->TimeChargeType = fl_bgn_group();
  fdui->MLSecondsInput = obj = fl_add_input(FL_NORMAL_INPUT,539,214,46,27,
											"Length for this zone (seconds):");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doMLSecondsInput,0);
	fl_set_input_maxchars(obj, 4);
  obj = fl_add_text(FL_NORMAL_TEXT,322,242,182,23,
					"Extra discount for this zone:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,322,146,182,23,
					"Minimum charge time length:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  fdui->RDStr = obj = fl_add_text(FL_NORMAL_TEXT,24,571,131,23,
								  "Apply zone discount:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  fdui->ZMLStr = obj = fl_add_text(FL_NORMAL_TEXT,172,555,125,38,
								   "Zero minimum\ncharge time length:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  fl_end_group();

  fdui->DiscountType = fl_bgn_group();
  fdui->discountInput = obj = fl_add_input(FL_NORMAL_INPUT,528,262,41,27,
										   "Discount:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doDiscountInput,0);
	fl_set_input_maxchars(obj, 4);
  fdui->PCSymbol = obj = fl_add_text(FL_NORMAL_TEXT,569,265,21,23,"%");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,328,291,229,20,
					"Call start time must not be between:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,360,333,8,20,":");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,400,334,8,20,":");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,329,309,29,21,"HH");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,369,309,29,21,"MM");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,409,309,29,21,"SS");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  fdui->ZSHInput = obj = fl_add_input(FL_NORMAL_INPUT,329,331,30,27,"");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doZSInput,0);
	fl_set_input_maxchars(obj, 2);
  fdui->ZSMInput = obj = fl_add_input(FL_NORMAL_INPUT,369,331,30,27,"");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doZSInput,1);
	fl_set_input_maxchars(obj, 2);
  fdui->ZSSInput = obj = fl_add_input(FL_NORMAL_INPUT,409,331,30,27,"");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doZSInput,2);
	fl_set_input_maxchars(obj, 2);
  obj = fl_add_text(FL_NORMAL_TEXT,442,332,30,23,"and");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,506,333,8,20,":");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,546,334,8,20,":");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,475,309,29,21,"HH");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,515,309,29,21,"MM");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,555,309,29,21,"SS");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  fdui->ZEHInput = obj = fl_add_input(FL_NORMAL_INPUT,475,331,30,27,"");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doZEInput,0);
	fl_set_input_maxchars(obj, 2);
  fdui->ZEMInput = obj = fl_add_input(FL_NORMAL_INPUT,515,331,30,27,"");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doZEInput,1);
	fl_set_input_maxchars(obj, 2);
  fdui->ZESInput = obj = fl_add_input(FL_NORMAL_INPUT,555,331,30,27,"");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doZEInput,2);
	fl_set_input_maxchars(obj, 2);
  fl_end_group();

  fdui->ruleChargeInput = obj = fl_add_input(FL_NORMAL_INPUT,394,399,55,27,
											 "Rule tariff:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doRuleChargeInput,0);
	fl_set_input_maxchars(obj, 6);

  fdui->CalDateType = fl_bgn_group();
  fdui->RDMInput = obj = fl_add_input(FL_NORMAL_INPUT,512,444,30,27,
									  "Calendar date (MM/DD):");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doRDInput,0);
	fl_set_input_maxchars(obj, 2);
  fdui->RDDInput = obj = fl_add_input(FL_NORMAL_INPUT,554,444,30,27,"/");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doRDInput,1);
	fl_set_input_maxchars(obj, 2);
  fl_end_group();


  fdui->EndDateType = fl_bgn_group();
  fdui->RDEMInput = obj = fl_add_input(FL_NORMAL_INPUT,512,479,30,27,
									   "End date (MM/DD):");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doRDEInput,0);
	fl_set_input_maxchars(obj, 2);
  fdui->RDEDInput = obj = fl_add_input(FL_NORMAL_INPUT,554,479,30,27,"/");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doRDEInput,1);
	fl_set_input_maxchars(obj, 2);
  fl_end_group();

  fdui->relDateInput = obj = fl_add_input(FL_NORMAL_INPUT,549,514,36,27,
										  "Day relative to Easter Sunday:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doRelDateInput,0);
	fl_set_input_maxchars(obj, 3);
  fdui->RSHInput = obj = fl_add_input(FL_NORMAL_INPUT,328,587,30,27,"");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doRSInput,0);
	fl_set_input_maxchars(obj, 2);
  fdui->RSMInput = obj = fl_add_input(FL_NORMAL_INPUT,368,587,30,27,"");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doRSInput,1);
	fl_set_input_maxchars(obj, 2);
  fdui->RSSInput = obj = fl_add_input(FL_NORMAL_INPUT,408,587,30,27,"");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doRSInput,2);
	fl_set_input_maxchars(obj, 2);
  fdui->REHInput = obj = fl_add_input(FL_NORMAL_INPUT,476,588,30,27,"");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doREInput,0);
	fl_set_input_maxchars(obj, 2);
  fdui->REMInput = obj = fl_add_input(FL_NORMAL_INPUT,516,588,30,27,"");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doREInput,1);
	fl_set_input_maxchars(obj, 2);
  fdui->RESInput = obj = fl_add_input(FL_NORMAL_INPUT,556,588,30,27,"");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doREInput,2);
	fl_set_input_maxchars(obj, 2);
  fl_end_form();

  fdui->pttEditor->fdui = fdui;

  return fdui;
}

FD_renameZone *create_form_renameZone(void)
{
  FL_OBJECT *obj;
  FD_renameZone *fdui = (FD_renameZone *) fl_calloc(1, sizeof(*fdui));

  fdui->renameZone = fl_bgn_form(FL_NO_BOX, 219, 112);
  obj = fl_add_box(FL_FLAT_BOX,0,0,219,112,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
  obj = fl_add_text(FL_NORMAL_TEXT,7,3,143,24,"Charging zone name:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  fdui->zoneNameInput = obj = fl_add_input(FL_NORMAL_INPUT,12,28,194,28,"");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_TOP_LEFT);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doZoneNameInput,0);
	fl_set_input_maxchars(obj, MAXLEN_ZNAME);
  fdui->zoneNameEditOK = obj = fl_add_button(FL_RETURN_BUTTON,12,70,79,27,
											 "OK");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doZoneNameEditOK,0);
	fl_set_object_bw(obj, -2);
  fdui->zoneNameEditCancel = obj = fl_add_button(FL_NORMAL_BUTTON,127,70,79,
												 27,"Cancel");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doZoneNameEditCancel,0);
	fl_set_object_bw(obj, -2);
  fl_end_form();

  fdui->renameZone->fdui = fdui;

  return fdui;
}

FD_renamePTT *create_form_renamePTT(void)
{
  FL_OBJECT *obj;
  FD_renamePTT *fdui = (FD_renamePTT *) fl_calloc(1, sizeof(*fdui));

  fdui->renamePTT = fl_bgn_form(FL_NO_BOX, 219, 112);
  obj = fl_add_box(FL_FLAT_BOX,0,0,219,112,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
  obj = fl_add_text(FL_NORMAL_TEXT,8,3,145,24,"Phone company name:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  fdui->PTTNameInput = obj = fl_add_input(FL_NORMAL_INPUT,12,28,194,28,"");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_TOP_LEFT);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doPTTNameInput,0);
	fl_set_input_maxchars(obj, MAXLEN_PTTNAME);
  fdui->PTTNameEditOK = obj = fl_add_button(FL_RETURN_BUTTON,12,70,79,27,"OK");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doPTTNameEditOK,0);
	fl_set_object_bw(obj, -2);
  fdui->PTTNameEditCancel = obj = fl_add_button(FL_NORMAL_BUTTON,127,70,79,
												27,"Cancel");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doPTTNameEditCancel,0);
	fl_set_object_bw(obj, -2);
  fl_end_form();

  fdui->renamePTT->fdui = fdui;

  return fdui;
}

FD_actionVerify *create_form_actionVerify(void)
{
  FL_OBJECT *obj;
  FD_actionVerify *fdui = (FD_actionVerify *) fl_calloc(1, sizeof(*fdui));

  fdui->actionVerify = fl_bgn_form(FL_NO_BOX, 300, 117);
  obj = fl_add_box(FL_FLAT_BOX,0,0,300,117,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
  obj = fl_add_pixmap(FL_NORMAL_PIXMAP,17,20,14,36,"");
    fl_set_object_resize(obj, FL_RESIZE_NONE);
	fl_set_object_gravity(obj, FL_West, FL_West);
	fl_set_pixmap_data(obj, excl_xpm);
  fdui->actionVerifyNo = obj = fl_add_button(FL_RETURN_BUTTON,54,71,79,
											 27,"No");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
	fl_set_object_gravity(obj, FL_SouthWest, FL_SouthWest);
	fl_set_object_bw(obj, -2);
  fdui->actionVerifyYes = obj = fl_add_button(FL_NORMAL_BUTTON,169,71,79,
											  27,"Yes");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
	fl_set_object_gravity(obj, FL_SouthEast, FL_SouthEast);
	fl_set_object_bw(obj, -2);
  fdui->actionVerifyStr = obj = fl_add_text(FL_NORMAL_TEXT,37,16,246,43,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_NORMAL_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  fl_end_form();

  fdui->actionVerify->fdui = fdui;

  return fdui;
}

FD_envInfo *create_form_envInfo(void)
{
  FL_OBJECT *obj;
  FD_envInfo *fdui = (FD_envInfo *) fl_calloc(1, sizeof(*fdui));

  fdui->envInfo = fl_bgn_form(FL_NO_BOX, 540, 350);
  obj = fl_add_box(FL_FLAT_BOX,0,0,540,350,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
  fdui->pppdPathInput = obj = fl_add_input(FL_NORMAL_INPUT,249,46,126,27,
										   "Path to the pppd daemon:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doPppdPathInput,0);
	fl_set_input_maxchars(obj, MAXLEN_PATH);
  fdui->runPathInput = obj = fl_add_input(FL_NORMAL_INPUT,249,92,126,27,
		"Path to pppd process ID \n files (e.g. ppp0.pid):");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doRunPathInput,0);
	fl_set_input_maxchars(obj, MAXLEN_PATH);
  fdui->lockPathInput = obj = fl_add_input(FL_NORMAL_INPUT,249,138,126,27,
	"Path to pppd lock file(s) for the \n modem device(s) (e.g. LCK..ttyS1):");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doLockPathInput,0);
	fl_set_input_maxchars(obj, MAXLEN_PATH);
  fdui->chatPathInput = obj = fl_add_input(FL_NORMAL_INPUT,249,184,126,27,
										   "Path to the chat program:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doChatPathInput,0);
	fl_set_input_maxchars(obj, MAXLEN_PATH);
  fdui->xispUtilsPathInput = obj = fl_add_input(FL_NORMAL_INPUT,249,230,126,27,
		"Path to xISP helper programs \n(xispdial & xispterm):");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doXispUtilsPathInput,0);
	fl_set_input_maxchars(obj, MAXLEN_PATH);
  fdui->pipePathInput = obj = fl_add_input(FL_NORMAL_INPUT,249,276,126,27,
										   "Path to the xISP named-pipe:");
    fl_set_object_boxtype(obj,FL_EMBOSSED_BOX);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doPipePathInput,0);
	fl_set_input_maxchars(obj, MAXLEN_PATH);
  fdui->pppdDefault = obj = fl_add_button(FL_NORMAL_BUTTON,381,47,71,25,
										  "Default");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doPppdDefault,0);
	fl_set_object_bw(obj, -2);
  fdui->runDefault = obj = fl_add_button(FL_NORMAL_BUTTON,381,93,71,25,
										 "Default");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doRunDefault,0);
	fl_set_object_bw(obj, -2);
  fdui->lockDefault = obj = fl_add_button(FL_NORMAL_BUTTON,381,139,71,25,
										  "Default");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doLockDefault,0);
	fl_set_object_bw(obj, -2);
  fdui->chatDefault = obj = fl_add_button(FL_NORMAL_BUTTON,381,185,71,25,
										  "Default");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doChatDefault,0);
	fl_set_object_bw(obj, -2);
  fdui->xispUtilsDefault = obj = fl_add_button(FL_NORMAL_BUTTON,381,231,71,25,
											   "Default");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doXispUtilsDefault,0);
	fl_set_object_bw(obj, -2);
  fdui->pipeDefault = obj = fl_add_button(FL_NORMAL_BUTTON,381,277,71,25,
										  "Default");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doPipeDefault,0);
	fl_set_object_bw(obj, -2);
  obj = fl_add_text(FL_NORMAL_TEXT,452,26,55,19,"Status:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  fdui->pppdStatus = obj = fl_add_text(FL_NORMAL_TEXT,457,47,45,24,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
  fdui->runStatus = obj = fl_add_text(FL_NORMAL_TEXT,457,93,45,24,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
  fdui->lockStatus = obj = fl_add_text(FL_NORMAL_TEXT,457,139,45,24,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
  fdui->chatStatus = obj = fl_add_text(FL_NORMAL_TEXT,457,185,45,24,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
  fdui->utilsStatus = obj = fl_add_text(FL_NORMAL_TEXT,457,231,45,24,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);

  fdui->tipTimer = obj = fl_add_timer(FL_HIDDEN_TIMER, 0,0,0,0,"");
    fl_set_timer(obj, 0.0);
    fl_set_object_callback(obj,doEnvInfoTip,0);
  fl_end_form();

  fdui->envInfo->fdui = fdui;

  return fdui;
}

FD_optsTab *create_form_optsTab(void)
{
  FL_OBJECT *obj;
  FD_optsTab *fdui = (FD_optsTab *) fl_calloc(1, sizeof(*fdui));

  fl_set_default_tabfolder_corner(2);
  fdui->optsTab = fl_bgn_form(FL_NO_BOX, 555, 429);
  obj = fl_add_box(FL_FLAT_BOX,0,0,555,429,"");
    fl_set_object_color(obj,FL_GREEN,FL_COL1);
  fdui->optsFolder = obj = fl_add_tabfolder(FL_TOP_TABFOLDER,5,5,545,378,"");
    fl_set_object_boxtype(obj,FL_UP_BOX);
	fl_set_object_bw(obj, -2);
	fl_set_object_bw(obj->next, -2);
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
    fl_set_object_callback(obj,doOptsFolder,0);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
	fdui->accountInfoButton = obj =
		fl_addto_tabfolder(fdui->optsFolder, " Accounts ",
						   fd_accountInfo->accountInfo);
	fl_set_object_prehandler(obj, folderButtonPreH);
	fdui->dialInfoButton = obj =
		fl_addto_tabfolder(fdui->optsFolder, " Dialing/Login ",
						   fd_dialInfo->dialInfo);
	fl_set_object_prehandler(obj, folderButtonPreH);
	fdui->commInfoButton = obj =
		fl_addto_tabfolder(fdui->optsFolder, " MODEM ",
						   fd_commInfo->commInfo);
	fl_set_object_prehandler(obj, folderButtonPreH);
	fdui->tcpipInfoButton = obj =
		fl_addto_tabfolder(fdui->optsFolder, " TCP/IP Options ",
						   fd_tcpipInfo->tcpipInfo);
	fl_set_object_prehandler(obj, folderButtonPreH);
	fdui->envInfoButton = obj =
		fl_addto_tabfolder(fdui->optsFolder, " Program Paths ",
						   fd_envInfo->envInfo);
	fl_set_object_prehandler(obj, folderButtonPreH);
  fdui->optsTabDone = obj = fl_add_button(FL_NORMAL_BUTTON,361,392,79,27,
										  "Done");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doOptsTabDone,0);
	fl_set_object_bw(obj, -2);
  fdui->optsTabCancel = obj = fl_add_button(FL_NORMAL_BUTTON,460,392,79,27,
											"Cancel");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doOptsTabCancel,0);
	fl_set_object_bw(obj, -2);
  fl_end_form();

  fdui->optsTab->fdui = fdui;

  return fdui;
}

