/*
  Copyright (C) 1997-2005  Dimitrios P. Bouras

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   For author contact information, look in the README file.
*/

/* ASCII resource control file I/O routines and local data */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <stdarg.h>
#include <string.h>
#include <math.h>
#include <sys/time.h>
#include <ctype.h>
#include <unistd.h>
#include <sys/wait.h>
#include "common.h"
#include "logs.h"
#include "version.h"
#include "pcode.h"

#ifdef SUNOS41x
#include <memory.h>
extern int sys_nerr;
extern char *sys_errlist[];
extern int errno;
extern int sscanf(), fputs(), vfprintf(), fscanf(), fprintf(), fclose();
extern int rewind(), toupper(), system();
extern long strtol();
#else
/* exit() replacement for emulating on_exit() functionality */
#define exit(x) {extern int exitStatus; exitStatus = x; exit(x);}
#endif

/* Local storage */

static char *rcfn;							/* pointer to RC file name str */
static glob_t reset_global = GLOB_DEFAULT;	/* for resetting global struct */

/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                        Local Utility routines                           |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

/* Convert binary password to ASCII printable characters */

static unsigned char *passwd2ascii(unsigned char *pwd)
{
	static unsigned char apwd[2*MAXLEN_PASSWD+1];
	unsigned char up, lp, *p;
	int i;

	for (i=0, p=apwd; i<MAXLEN_PASSWD; i++, pwd++) {
		lp = *pwd & 0x03F;
		*p++ = lp + 33;
		up = (*pwd & 0x0FC) >> 2;
		*p++ = up + 161;
	}
	*p = 0;
	return apwd;
}

/* Convert IP address bytes to ASCII string */

static char *IP2str(unsigned char *ip)
{
	static char IPstr[16];

	sprintf(IPstr, "%u.%u.%u.%u",
			ip[0],ip[1],ip[2],ip[3]);
	return IPstr;
}

/* Convert ASCII password to binary */

static unsigned char *ascii2passwd(unsigned char *buf)
{
	static unsigned char pwd[MAXLEN_PASSWD+1];
	unsigned char up, lp;
	int i = 0;

	if (*buf) {
		for (; i<MAXLEN_PASSWD; i++) {
			lp = (*buf++ - 33) & 0x00F;
			up = (*buf++ - 161) & 0x03C;
			pwd[i] = (up << 2) | lp;
		}
	}
	pwd[i] = 0;
	return pwd;
}

/* Convert ASCII string to IP address */

static unsigned char *str2IP(char *buf)
{
	static unsigned char ip[4];
	unsigned int iip[4];
	int i;

	i = sscanf(buf, "%3u.%3u.%3u.%3u", &iip[0],
			   &iip[1], &iip[2], &iip[3]);
	if (i < 4)
		return NULL;
	for (i=0; i<4; i++)
		ip[i] = iip[i];
	return ip;
}

/* Print message together with system error message and exit. Note the
   implicit total length of MSGLEN_ERR bytes for the resulting string. */

#define MSGLEN_ERR 128

static void doErr(char *msg)
{
	char emsg[MSGLEN_ERR+1];

#ifdef HAVE_STRERROR
	sprintf(emsg, "xISP: %s: %s\n", msg, strerror(errno));
#else
	if (errno < sys_nerr)
		sprintf(emsg, "xISP: %s: %s\n", msg, sys_errlist[errno]);
	else
		sprintf(emsg, "xISP: %s: error #%d\n", msg, errno);
#endif
	fputs(emsg, stderr);
	exit(1);
}

/* Function for writing strings in the .xisprc resource control file */

static int rcPrint(FILE *fp, ...)
{
	int bw;
	va_list ap;
	char *fmt;

	va_start(ap, fp);
	fmt = va_arg(ap, char*);
	bw = vfprintf(fp, fmt, ap);
	va_end(ap);
	if (bw <= 0)
		doErr("outputAllXisprc");
	return bw;
}

/* Function for writing out the entire .xisprc file */

static void outputAllXisprc(xisprc_t *p, FILE *rcfp, glob_t *global)
{
	int i, n;
	char tmp[2*MAXLEN_PASSWD+1];

	rcPrint(rcfp, "xISP-%s\n", Version);
	rcPrint(rcfp, "NumISPs: %d\n", global->numISPs);
	rcPrint(rcfp, "DefaultISP: %d\n", global->dfltISP);
	rcPrint(rcfp, "NumPTTs: %d\n", global->numPTTs);
	rcPrint(rcfp, "CostPTT: %d\n", global->costPTT);
	rcPrint(rcfp, "LogOpts: %hX\n", global->logOpts);
	rcPrint(rcfp, "PppdPath: %s\n", global->pppdPath);
	rcPrint(rcfp, "RunPath: %s\n", global->runPath);
	rcPrint(rcfp, "LockPath: %s\n", global->lockPath);
	rcPrint(rcfp, "ChatPath: %s\n", global->chatPath);
	rcPrint(rcfp, "UtilsPath: %s\n", global->utilsPath);
	rcPrint(rcfp, "PipePath: %s\n", global->pipePath);
	for (i=0; p!=NULL && i<global->numISPs; i++, p++) {
		rcPrint(rcfp, "[%d]\n", i);
		rcPrint(rcfp, "Description: %s\n", p->descr);
		rcPrint(rcfp, "Account: %s\n", p->account);
		if (*(p->account)) {
			memset(tmp, 0, 2*MAXLEN_PASSWD+1);
			pdecode(tmp, p->passwd);
			n = strlen(tmp);
			if (n % 8)
				n = 8*((n/8) + 1);
			memset(tmp, 0, 2*MAXLEN_PASSWD+1);
			memcpy(tmp, passwd2ascii(p->passwd), 2*n);
			tmp[2*n] = 0;
			rcPrint(rcfp, "Password: %s\n", tmp);
		}
		else
			rcPrint(rcfp, "Password: \n");
		rcPrint(rcfp, "Name: %s\n", p->name);
		rcPrint(rcfp, "RemoteName: %s\n", p->rname);
		rcPrint(rcfp, "MaxAttempts: %d\n", p->maxAttempts);
		rcPrint(rcfp, "SleepDelay: %d\n", p->sleepDelay);
		rcPrint(rcfp, "ConnectWait: %d\n", p->connectWait);
		rcPrint(rcfp, "LCPWait: %d\n", p->LCPWait);
		rcPrint(rcfp, "NumPhones: %d\n", p->numPhones);
		for (n=0; n<p->numPhones; n++)
			rcPrint(rcfp, " Phone[%d]: %s\n", n, p->phone[n]);
		rcPrint(rcfp, "NumSLines: %d\n", p->numSlines);
		for (n=0; n<p->numSlines; n++) {
			rcPrint(rcfp, " SLine[%d]E: %s\n", n, p->sline[n]);
			rcPrint(rcfp, " SLine[%d]S: %s\n", n, p->sline[MAXNUM_SLINES+n]);
		}
		rcPrint(rcfp, "CBDelay: %d\n", p->CBDelay);
		rcPrint(rcfp, "NumCBSLines: %d\n", p->numCBSlns);
		for (n=0; n<p->numCBSlns; n++) {
			rcPrint(rcfp, " CBSLine[%d]E: %s\n", n, p->CBsln[n]);
			rcPrint(rcfp, " CBSLine[%d]S: %s\n", n, p->CBsln[MAXNUM_SLINES+n]);
		}
		rcPrint(rcfp, "TermW: %d\n", p->termW);
		rcPrint(rcfp, "TermH: %d\n", p->termH);
		rcPrint(rcfp, "CBTermW: %d\n", p->CBtermW);
		rcPrint(rcfp, "CBTermH: %d\n", p->CBtermH);
		rcPrint(rcfp, "CBPhone: %s\n", p->CBphone);
		rcPrint(rcfp, "ModemDevice: %s\n", p->modemDevice);
		rcPrint(rcfp, "ModemSpeed: %u\n", p->modemSpeed);
		rcPrint(rcfp, "ModemReset: %s\n", p->modemReset);
		rcPrint(rcfp, "ModemInit: %s\n", p->modemInit);
		rcPrint(rcfp, "ModemConnect: %s\n", p->modemConnect);
		rcPrint(rcfp, "OperOpts: %lX\n", p->operOpts);
		rcPrint(rcfp, "DialExtra: %s\n", p->dialExtra);
		rcPrint(rcfp, "CompLevel: %d\n", p->compLevel);
		rcPrint(rcfp, "AsyncMap: %s\n", p->asyncmap);
		rcPrint(rcfp, "Escape: %s\n", p->escape);
		rcPrint(rcfp, "LocalIP: %s\n", IP2str(p->localIP));
		rcPrint(rcfp, "RemoteIP: %s\n", IP2str(p->remoteIP));
		rcPrint(rcfp, "NetMask: %s\n", IP2str(p->netmask));
		rcPrint(rcfp, "DNS1: %s\n", IP2str(p->dns1));
		rcPrint(rcfp, "DNS2: %s\n", IP2str(p->dns2));
		rcPrint(rcfp, "MTU: %u\n", p->mtu);
		rcPrint(rcfp, "MRU: %u\n", p->mru);
		rcPrint(rcfp, "DomainName: %s\n", p->domainname);
		rcPrint(rcfp, "ISPPTT: %d\n", p->ispPTT);
		rcPrint(rcfp, "ISPZone: %d\n", p->ispZone);
	}
}

/* Function for retrieving the .xisprc version. Note that it
   must be called after fopen() and before all other input */

static char *rcVersion(FILE *fp)
{
	static char version[8] = {0};

	rewind(fp);
	if (fscanf(fp, "xISP-%7[.0-9] ", version) < 1) {
		fputs("xISP: rcVersion: error reading file version!\n", stderr);
		exit(1);
	}
	return version;
}

/* Function for retrieving the ISP sequence number. Note that
   it must be called at the beginning of each ISP record. */

static void rcISP(FILE *fp, int nISP)
{
	int n;

	if (fscanf(fp, "[%3d] ", &n) < 1) {
		fprintf(stderr, "xISP: rcISP: error reading ISP number in %s\n",
				rcfn);
		exit(1);
	}
	if (n != nISP) {
		fprintf(stderr, "xISP: rcISP: ISP sequence number wrong in %s\n",
				rcfn);
		fprintf(stderr, "xISP: rcISP: expected %d, got %d\n", nISP, n);
		exit(1);
	}
}

/* Prints error message if error occurs while
   processing .xisprc. Prints offending line */

static void rcLineError(char *function, char *line)
{
	fprintf(stderr, "xISP: %s: error reading %s\n", function, rcfn);
	fprintf(stderr, "xISP: %s: offending line: [%s]\n", function, line);
	exit(1);
}

/* Reads parameters from .xisprc file, doing error checking. NOTE: it
   assumes a line less than 512 bytes and a parameter less than 256 */

#define MAXLEN_LINE 512
#define MAXLEN_PARAM 256

static char *readParam(FILE *fp, char *name, char type,
					   int llimit, int ulimit, void *data)
{
	static char line[MAXLEN_LINE], NAME[MAXLEN_PARAM];
	char sparam[MAXLEN_PARAM] = {0}, *p, *P, *endp;
	long iparam;
	int len, hex = 0;

	if (fgets(line, MAXLEN_LINE, fp) == NULL) {
		fprintf(stderr, "xISP: readParam: %s: premature EOF\n", rcfn);
		exit(1);
	}
	line[strlen(line)-1] = 0;
	for (p=name, P=NAME; *p; p++, P++)
		*P = toupper(*p);
	*P = 0;
	if ((p=strchr(line,':')) == NULL)
		rcLineError("readParam", line);
	if (strncmp(name, line, (int)(p-line)) &&	/* accept all capitals */
		strncmp(NAME, line, (int)(p-line)))
	{
		fprintf(stderr, "xISP: readParam: expected [%s], got [%s]\n",
				name, line);
		exit(1);
	}
	for (++p; *p==' '; p++);
	strncpy(sparam, p, sizeof(sparam)-1);
	switch (type) {
		case 'S':
			len = strlen(sparam);
			if (ulimit<len || len<llimit)
				rcLineError("readParam", line);
			strncpy((char *)data, sparam, len);
			((char *)data)[len] = 0;
			break;

		case 'X':
		case 'L':
			hex = 1;
		case 'B':
		case 'I':
			iparam = strtol(sparam, &endp, (hex)? 16:10);
			if (hex) {
				if (type == 'X')
					*((unsigned short *)data) = iparam;
				else
					*((unsigned long *)data) = iparam;
			}
			else {
				if (ulimit<iparam || iparam<llimit || endp == sparam || *endp)
					rcLineError("readParam", line);
				if (type == 'B')
					*((unsigned char *)data) = iparam;
				else
					*((unsigned int *)data) = iparam;
			}
			break;

		default: break;
	}
	return(line);
}

/* Function for reading in the global section from the .xisprc file */

static void inputGlobals(FILE *rcfp, glob_t *global)
{
	readParam(rcfp, "NumISPs", 'B', 0, 255, &(global->numISPs));
	readParam(rcfp, "DefaultISP", 'B', 0, 255, &(global->dfltISP));
	readParam(rcfp, "NumPTTs", 'B', 1, 255, &(global->numPTTs));
	readParam(rcfp, "CostPTT", 'B', 0, global->numPTTs-1, &(global->costPTT));
	readParam(rcfp, "LogOpts", 'X', 0, 0, &(global->logOpts));
	readParam(rcfp, "PppdPath", 'S', 0, MAXLEN_PATH, global->pppdPath);
	readParam(rcfp, "RunPath", 'S', 0, MAXLEN_PATH, global->runPath);
	readParam(rcfp, "LockPath", 'S', 0, MAXLEN_PATH, global->lockPath);
	readParam(rcfp, "ChatPath", 'S', 0, MAXLEN_PATH, global->chatPath);
	readParam(rcfp, "UtilsPath", 'S', 0, MAXLEN_PATH, global->utilsPath);
	readParam(rcfp, "PipePath", 'S', 0, MAXLEN_PATH, global->pipePath);
}

/* Function for reading in the ISP sections from the .xisprc file */

static void inputAllXisprc(xisprc_t *p, FILE *rcfp, glob_t *global)
{
	int i, j, n;
	char buf[2*MAXLEN_PASSWD+1], pname[32] = {0}, *line;
	unsigned char *ipp, tmp[2*MAXLEN_PASSWD+1], epasswd[MAXLEN_PASSWD+1];

	for (i=0; i<global->numISPs; i++, p++) {
		memset(p, 0, sizeof(xisprc_t));
		rcISP(rcfp, i);
		readParam(rcfp, "Description", 'S', 0, MAXLEN_DESCR, p->descr);
		readParam(rcfp, "Account", 'S', 0, MAXLEN_ACCOUNT, p->account);
		if (! *(p->account))
			readParam(rcfp, "Password", 'S', 0, 0, buf);
		else {
			readParam(rcfp, "Password", 'S', 0, 2*MAXLEN_PASSWD, buf);
			n = strlen(buf);
			if (n < 2*MAXLEN_PASSWD) {
				memset(tmp, 0, 2*MAXLEN_PASSWD+1);
				pencode(epasswd, tmp);
				memcpy(tmp, passwd2ascii(epasswd), 2*MAXLEN_PASSWD);
				memcpy(buf+n, tmp+n, 2*MAXLEN_PASSWD-n);
				buf[2*MAXLEN_PASSWD] = 0;
			}
		}
		memcpy(p->passwd, ascii2passwd((unsigned char *)buf), MAXLEN_PASSWD);
		readParam(rcfp, "Name", 'S', 0, MAXLEN_UNR, p->name);
		readParam(rcfp, "RemoteName", 'S', 0, MAXLEN_UNR, p->rname);
		readParam(rcfp, "MaxAttempts", 'B', 0, 255, &(p->maxAttempts));
		readParam(rcfp, "SleepDelay", 'B', 0, 255, &(p->sleepDelay));
		readParam(rcfp, "ConnectWait", 'B',MINSEC_MODEM_TO,255,
				  &(p->connectWait));
		readParam(rcfp, "LCPWait", 'B', MINSEC_LINK_TO, 255, &(p->LCPWait));
		readParam(rcfp, "NumPhones", 'B', 0, MAXNUM_TELS, &(p->numPhones));
		for (j=0; j<p->numPhones; j++) {
			sprintf(pname, " Phone[%d]", j);
			readParam(rcfp, pname, 'S', 0, MAXLEN_PHONE, p->phone[j]);
		}
		readParam(rcfp, "NumSLines", 'B', 0, MAXNUM_SLINES, &(p->numSlines));
		for (j=0; j<p->numSlines; j++) {
			sprintf(pname, " SLine[%d]E", j);
			readParam(rcfp, pname, 'S', 0, MAXLEN_SLINE, p->sline[j]);
			sprintf(pname, " SLine[%d]S", j);
			readParam(rcfp, pname, 'S', 0, MAXLEN_SLINE,
					  p->sline[MAXNUM_SLINES+j]);
		}
		readParam(rcfp, "CBDelay", 'B', 0, 255, &(p->CBDelay));
		readParam(rcfp, "NumCBSLines", 'B', 0, MAXNUM_SLINES, &(p->numCBSlns));
		for (j=0; j<p->numCBSlns; j++) {
			sprintf(pname, " CBSLine[%d]E", j);
			readParam(rcfp, pname, 'S', 0, MAXLEN_SLINE, p->CBsln[j]);
			sprintf(pname, " CBSLine[%d]S", j);
			readParam(rcfp, pname, 'S', 0, MAXLEN_SLINE,
					  p->CBsln[MAXNUM_SLINES+j]);
		}
		readParam(rcfp, "TermW", 'B', MINCHAR_TERMW, 255, &(p->termW));
		readParam(rcfp, "TermH", 'B', MINCHAR_TERMH, 255, &(p->termH));
		readParam(rcfp, "CBTermW", 'B', MINCHAR_TERMW, 255, &(p->CBtermW));
		readParam(rcfp, "CBTermH", 'B', MINCHAR_TERMH, 255, &(p->CBtermH));
		readParam(rcfp, "CBPhone", 'S', 0, MAXLEN_PHONE, p->CBphone);
		readParam(rcfp, "ModemDevice", 'S', 0, MAXLEN_DEVICE, p->modemDevice);
		readParam(rcfp, "ModemSpeed", 'I', MIN_BAUDRATE, MAX_BAUDRATE,
				  &(p->modemSpeed));
		readParam(rcfp, "ModemReset", 'S', 0, MAXLEN_MDMCMD, p->modemReset);
		readParam(rcfp, "ModemInit", 'S', 0, MAXLEN_MDMCMD, p->modemInit);
		readParam(rcfp, "ModemConnect", 'S',0, MAXLEN_MDMSTR, p->modemConnect);
		readParam(rcfp, "OperOpts", 'L', 0, 0, &(p->operOpts));
		readParam(rcfp, "DialExtra", 'S', 0, MAXLEN_DIALEXTRA, p->dialExtra);
		readParam(rcfp, "CompLevel", 'B', MIN_COMPLEVEL, MAX_COMPLEVEL,
				  &(p->compLevel));
		readParam(rcfp, "AsyncMap", 'S', 2, MAXDIG_ASYNCMAP, p->asyncmap);
		readParam(rcfp, "Escape", 'S', 0, MAXLEN_ESCAPE, p->escape);
		line = readParam(rcfp, "LocalIP", 'S', 7, MAXLEN_IP, buf);
		if ((ipp=str2IP(buf)) == NULL)
			rcLineError("inputAllXisprc", line);
		memcpy(p->localIP, ipp, 4);
		line = readParam(rcfp, "RemoteIP", 'S', 7, MAXLEN_IP, buf);
		if ((ipp=str2IP(buf)) == NULL)
			rcLineError("inputAllXisprc", line);
		memcpy(p->remoteIP, ipp, 4);
		line = readParam(rcfp, "NetMask", 'S', 7, MAXLEN_IP, buf);
		if ((ipp=str2IP(buf)) == NULL)
			rcLineError("inputAllXisprc", line);
		memcpy(p->netmask, ipp, 4);
		line = readParam(rcfp, "DNS1", 'S', 7, MAXLEN_IP, buf);
		if ((ipp=str2IP(buf)) == NULL)
			rcLineError("inputAllXisprc", line);
		memcpy(p->dns1, ipp, 4);
		line = readParam(rcfp, "DNS2", 'S', 7, MAXLEN_IP, buf);
		if ((ipp=str2IP(buf)) == NULL)
			rcLineError("inputAllXisprc", line);
		memcpy(p->dns2, ipp, 4);
		readParam(rcfp, "MTU", 'I', 128, 2048, &(p->mtu));
		readParam(rcfp, "MRU", 'I', 128, 2048, &(p->mru));
		readParam(rcfp, "DomainName", 'S', 0, MAXLEN_DNNAME, p->domainname);
		readParam(rcfp, "ISPPTT", 'B', 0, (global->numPTTs)-1, &(p->ispPTT));
		readParam(rcfp, "ISPZone", 'B', 0, MAXNUM_ZONES-1, &(p->ispZone));
	}
}

void RCSq(void)
{
	char *rcsp;
	rcsp = RCSid;
	rcsp = PatchLevel;
}

/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                           Interface routines                            |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

/* Initializes the xisprc data structure. */

void initXisprc(xisprc_t *p)
{
	int i;

	p->descr[0] = 0;
	p->account[0] = 0;
	p->passwd[0] = 0;
	p->name[0] = 0;
	p->rname[0] = 0;
	p->maxAttempts = MAXNUM_RETRY;
	p->sleepDelay = MAXSEC_DELAY;
	p->connectWait = MAXSEC_CNWAIT;
	p->LCPWait = MAXSEC_LCPWAIT;
	p->numPhones = 0;
	for (i=0; i<MAXNUM_TELS; i++)
		p->phone[i][0] = 0;
	p->numSlines = p->numCBSlns = 0;
	for (i=0; i<(2*MAXNUM_SLINES); i++)
		p->sline[i][0] = p->CBsln[i][0] = 0;
	p->CBDelay = MAXSEC_CBDELAY;
	p->termW = TERMW;
	p->termH = TERMH;
	p->CBtermW = TERMW;
	p->CBtermH = TERMH;
	p->CBphone[0] = 0;
	strcpy(p->modemDevice, MODEM_DEVICE);
	p->modemSpeed = MODEM_SPEED;
	strcpy(p->modemReset, MODEM_RESET);
	strcpy(p->modemInit, MODEM_INIT);
	strcpy(p->modemConnect, MODEM_CONNECT);
	p->operOpts = OPER_OPTS;
	strcpy(p->dialExtra, DIAL_EXTRA);
	p->compLevel = COMP_LEVEL;
	strcpy(p->asyncmap, PPPD_HASYNCMAP);
	strcpy(p->escape, PPPD_ESCAPE);
	memcpy(p->localIP, LOCAL_IP, 4);
	memcpy(p->remoteIP, REMOTE_IP, 4);
	memcpy(p->netmask, NETMASK, 4);
	memcpy(p->dns1, DNS, 4);
	memcpy(p->dns2, DNS, 4);
	p->mtu = MTU;
	p->mru = MRU;
	memset(p->domainname, 0, MAXLEN_DNNAME);
	p->ispPTT = PTT_GR_EPAK;
	p->ispZone = 0;
}

/* Read all xisprc data records from the user file, or create
   a new file if it's the first time the program is run. */

void readXisprc(char *rcfname, xisprc_t **rcdataptr, glob_t *global)
{
	FILE *rcfp;
	extern void outofMem(), alertMessage();
	extern int actionVerify();
	char msg[512] = {0}, *p;
	int status;

	rcfn = rcfname;
	rcfp = fopen(rcfn, "r");
	if (rcfp != NULL) {
		if (strcmp(p=rcVersion(rcfp), Version)) {
			sprintf(msg, "%s claims to be version %s,\nwhile %s "
					"is needed. Do you want to invoke xisprccv to\n"
					"convert the file?", rcfn, p, Version);
			if (! actionVerify(msg, 0)) {
				alertMessage("Exiting:", 0, 1, "OK. Aborting...");
				exit(0);
			}
			fclose(rcfp);
			if ((status=system("xisprccv"), status=WEXITSTATUS(status)) ||
				(rcfp = fopen(rcfn, "r")) == NULL ||
				strcmp(rcVersion(rcfp), Version))
			{
				if (status == 2) {
					alertMessage("Exiting:", 0, 0,
								 "Conversion of your .xisprc succeeded, but "
								 "your\nlog files seem to be out of order! "
								 "  Please rename\n$HOME/.xisplogs to "
								 "$HOME/.xisplogs.old so that\nnew logs can "
								 "be created!");
					exit(0);
				}
				else {
					alertMessage("Exiting:", 0, 1,
								 "Conversion failed! Aborting...");
					exit(1);
				}
			}
		}
		*global = reset_global;
		inputGlobals(rcfp, global);
		*rcdataptr = (xisprc_t *)malloc((1+global->numISPs)*sizeof(xisprc_t));
		if (*rcdataptr == NULL)
			outofMem();
		initXisprc(*rcdataptr);	/* initialize default record */
		if ((*rcdataptr)[0].ispPTT >= global->numPTTs)
			(*rcdataptr)[0].ispPTT = global->numPTTs-1;
		inputAllXisprc(&((*rcdataptr)[1]), rcfp, global);
		fclose(rcfp);
	}
	else {
		*global = reset_global;
		*rcdataptr = (xisprc_t *)malloc((1+global->numISPs)*sizeof(xisprc_t));
		if (*rcdataptr == NULL)
			outofMem();
		initXisprc(*rcdataptr);	/* initialize default record */
		rcfp = fopen(rcfn, "w");
		if (rcfp == NULL)
			doErr("readAllXisprc: creating: fopen");
		outputAllXisprc(NULL, rcfp, global);
		fclose(rcfp);
	}
}

/* Write all xisprc data records to the user file. */

void writeXisprc(char *rcfname, xisprc_t *rcdata, glob_t *global)
{
	FILE *rcfp;

	rcfn = rcfname;
	rcfp = fopen(rcfn, "w");
	if (rcfp != NULL) {
		outputAllXisprc(&rcdata[1], rcfp, global);
		fclose(rcfp);
	}
	else
		doErr("writeXisprc: fopen");
}

