/*
 * Copyright(c) 1992 Bell Communications Research, Inc. (Bellcore)
 * Copyright(c) 1995-99 Andrew Lister
 * Copyright  1999, 2000, 2001, 2002, 2003, 2004 by the LessTif Developers.
 *
 *                        All rights reserved
 * Permission to use, copy, modify and distribute this material for
 * any purpose and without fee is hereby granted, provided that the
 * above copyright notice and this permission notice appear in all
 * copies, and that the name of Bellcore not be used in advertising
 * or publicity pertaining to this material without the specific,
 * prior written permission of an authorized representative of
 * Bellcore.
 *
 * BELLCORE MAKES NO REPRESENTATIONS AND EXTENDS NO WARRANTIES, EX-
 * PRESS OR IMPLIED, WITH RESPECT TO THE SOFTWARE, INCLUDING, BUT
 * NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR ANY PARTICULAR PURPOSE, AND THE WARRANTY AGAINST IN-
 * FRINGEMENT OF PATENTS OR OTHER INTELLECTUAL PROPERTY RIGHTS.  THE
 * SOFTWARE IS PROVIDED "AS IS", AND IN NO EVENT SHALL BELLCORE OR
 * ANY OF ITS AFFILIATES BE LIABLE FOR ANY DAMAGES, INCLUDING ANY
 * LOST PROFITS OR OTHER INCIDENTAL OR CONSEQUENTIAL DAMAGES RELAT-
 * ING TO THE SOFTWARE.
 *
 * $Id: ScrollMgr.c,v 1.65 2004/11/22 18:33:15 tobiasoed Exp $
 */

#ifdef HAVE_CONFIG_H
#include <XbaeConfig.h>
#endif

#include <stdio.h>
#include <assert.h>

#include <Xm/Xm.h>
#include <Xm/ScrollBar.h>

#include <Xbae/MatrixP.h>
#include <Xbae/Draw.h>
#include <Xbae/Shadow.h>
#include <Xbae/ScrollMgr.h>
#include <Xbae/Utils.h>
#include <Xbae/Clip.h>

#include <XbaeDebug.h>

/*
 * Callback for vertical scrollbar
 * SGO: changed scrolling routine for vertical scrolling.
 * using same approach as in xbaeScrollHorizCB.
 */

void xbaeScrollVertCB(Widget w, XtPointer client_data, XmScrollBarCallbackStruct * call_data)
{
        XbaeMatrixWidget mw = (XbaeMatrixWidget) XtParent(w);
        int delta = VERT_ORIGIN(mw) - call_data->value;

        if (delta == 0) {
                /* Didn't scroll */
                return;
        }
        
        /*
         * Adjust our vertical origin
         */
        VERT_ORIGIN(mw) = call_data->value;

        /*
         * The textField needs to scroll along with the cells.
         */
        if (XtIsManaged(TextChild(mw)) && !IS_FIXED_ROW(mw, mw->matrix.current_row)) {
                XtMoveWidget(TextChild(mw), 
                             TextChild(mw)->core.x,
                             TextChild(mw)->core.y + delta);
                /*
                 * If the focus is lost and the the text child got dragged into view, give it the focus
                 */
                xbaeRefocusTextChild(mw);
        }

        /* Cell widgets that are in the vertical scrolling region
         * need to scroll along as well... Linas */
        if (mw->matrix.per_cell) {
                int roe, kol;

                for (kol = 0; kol < (mw->matrix.columns); kol++) {
                        for (roe = mw->matrix.fixed_rows; roe < TRAILING_VERT_ORIGIN(mw); roe++) {
                                xbaePositionCellWidget(mw, roe, kol);
                        }
                }
        }

        if (!XtIsRealized((Widget) mw))
                return;

        /*
         * Scroll the contents of the clips
         */

        XbaeClipScrollVert(ClipChild(mw), mw->matrix.draw_gc, delta);
        
        if (XtIsManaged(LeftClip(mw)))
                XbaeClipScrollVert(LeftClip(mw), mw->matrix.draw_gc, delta);
              
        if (XtIsManaged(RightClip(mw)))
                XbaeClipScrollVert(RightClip(mw), mw->matrix.draw_gc, delta);

        if (XtIsManaged(RowLabelClip(mw)))
                XbaeClipScrollVert(RowLabelClip(mw), mw->matrix.draw_gc, delta);
}

/*
 * Callback for horizontal scrollbar
 */

/* ARGSUSED */
void xbaeScrollHorizCB(Widget w, XtPointer client_data, XmScrollBarCallbackStruct * call_data)
{
        XbaeMatrixWidget mw = (XbaeMatrixWidget) XtParent(w);
        int delta = HORIZ_ORIGIN(mw) - call_data->value;

        if (delta == 0) {
                /* Didn't scroll */
                return;
        }

        /*
         * Adjust our horizontal origin
         */
        HORIZ_ORIGIN(mw) = call_data->value;

        /*
         * The textField needs to scroll along with the cells.
         */
        if (XtIsManaged(TextChild(mw)) && !IS_FIXED_COLUMN(mw, mw->matrix.current_column)) {
                XtMoveWidget(TextChild(mw),
                             TextChild(mw)->core.x + delta,
                             TextChild(mw)->core.y);
                /*
                 * If the focus is lost and the the text child got dragged into view, give it the focus
                 */
                xbaeRefocusTextChild(mw);
                DEBUGOUT(_XbaeDebug2
                         (__FILE__, (Widget) mw, w, "xbaeScrollVertCB: move child to %d %d\n",
                          TextChild(mw)->core.x, TextChild(mw)->core.y));
        }

        /* Cell widgets that are in the horizontal scrolling region
         * need to scroll along as well... Linas */
        if (mw->matrix.per_cell) {
                int roe, kol;

                for (roe = 0; roe < (mw->matrix.rows); roe++) {
                        for (kol = mw->matrix.fixed_columns; kol < TRAILING_HORIZ_ORIGIN(mw); kol++) {
                                xbaePositionCellWidget(mw, roe, kol);
                        }
                }
        }

        if (!XtIsRealized((Widget) mw))
                return;

        /*
         * Scroll the contents of the clips
         */

        XbaeClipScrollHoriz(ClipChild(mw), mw->matrix.draw_gc, delta);

        if (XtIsManaged(TopClip(mw)))
                XbaeClipScrollHoriz(TopClip(mw), mw->matrix.draw_gc, delta);

        if (XtIsManaged(BottomClip(mw)))
                XbaeClipScrollHoriz(BottomClip(mw), mw->matrix.draw_gc, delta);

        if (XtIsManaged(ColumnLabelClip(mw)))
                XbaeClipScrollHoriz(ColumnLabelClip(mw), mw->matrix.draw_gc, delta);

}

#define OVERLAP(r1, r2) ( \
            (r1).x + (r1).width > (r2).x && \
            (r1).x < (r2).x + (r2).width && \
            (r1).y + (r1).height > (r2).y && \
            (r1).y < (r2).y + (r2).height)

/*
 * Redraw the cells or labels of a region that are in the expose Rectangle. 
 */
void xbaeRedrawRegion(XbaeMatrixWidget mw, XRectangle *expose, XRectangle *region)
{
        int row, start_row, end_row;
        int column, start_column, end_column;

        assert(!mw->matrix.disable_redisplay);

        DEBUGOUT(_XbaeDebug
                 (__FILE__, (Widget) mw,
                  "redrawing region with expose (x,y,w,h)=(%d,%d,%d,%d)\n",
                  expose->x, expose->y, expose->width, expose->height));

        if (OVERLAP(*expose, *region)) {
                
                int xmin = Max(expose->x, region->x);
                int ymin = Max(expose->y, region->y);
                
                int xmax = Min(expose->x + expose->width - 1, region->x + region->width - 1);
                int ymax = Min(expose->y + expose->height -1, region->y + region->height -1);

                xbaeMatrixYtoRow(mw, &ymin, &start_row);
                xbaeMatrixYtoRow(mw, &ymax, &end_row);

                xbaeMatrixXtoColumn(mw, &xmin, &start_column);
                xbaeMatrixXtoColumn(mw, &xmax, &end_column);

                DEBUGOUT(_XbaeDebug
                         (__FILE__, (Widget) mw,
                          "redrawing region (rows,columns)=((%d,%d)-(%d,%d))\n",
                          start_row, start_column, end_row, end_column));

                if (start_row == -1 && start_column == -1) {
                        /* Do nothing, there are no rows or columns to display. */
                } else if (start_row == -1) {
                        assert(end_row == -1 && start_column >= 0 && end_column >= start_column);
                        for(column = start_column; column <= end_column; column++) {
                                xbaeDrawColumnLabel(mw, column, False);
                        }
                } else if (start_column == -1) {
                        assert(end_column == -1 && start_row >= 0 && end_row >= start_row);
                        for(row = start_row; row <= end_row; row++) {
                                xbaeDrawRowLabel(mw, row, False);
                        }
                } else {
                        assert(start_row >= 0 && end_row >= start_row);
                        assert(start_column >= 0 && end_column >= start_column);
                        for (row = start_row; row <= end_row; row++) {
                                for (column = start_column; column <= end_column; column++) {
                                        xbaeDrawCell(mw, row, column);
                                }
                        }
                }
        }
}

/*
 * Redraw the fixed labels and the totally fixed cells that are in the expose Rectangle.
 */
void xbaeRedrawLabelsAndFixed(XbaeMatrixWidget mw, XRectangle * expose)
{
        /*
         * Set up some local variables to avoid calling too many macros
         */
        Widget w = (Widget) mw;
        int horiz_sb_offset = HORIZ_SB_OFFSET(mw);
        int vert_sb_offset = VERT_SB_OFFSET(mw);
        int column_label_height = COLUMN_LABEL_HEIGHT(mw);
        int row_label_width = ROW_LABEL_WIDTH(mw);
        int r, c;
        
        struct {
                Bool exists;
                int position;
                int size;
        } row_regions[] = {
                {mw->matrix.column_labels || mw->matrix.xmcolumn_labels, HORIZ_SB_OFFSET(mw), COLUMN_LABEL_HEIGHT(mw)},
                {mw->matrix.fixed_rows, FIXED_ROW_POSITION(mw), VISIBLE_FIXED_ROW_HEIGHT(mw)},
                {mw->matrix.trailing_fixed_rows, TRAILING_FIXED_ROW_POSITION(mw), VISIBLE_TRAILING_FIXED_ROW_HEIGHT(mw)}
        }, column_regions[] = {
                {mw->matrix.row_labels != NULL, VERT_SB_OFFSET(mw), ROW_LABEL_WIDTH(mw)},
                {mw->matrix.fixed_columns, FIXED_COLUMN_POSITION(mw), VISIBLE_FIXED_COLUMN_WIDTH(mw)},
                {mw->matrix.trailing_fixed_columns, TRAILING_FIXED_COLUMN_POSITION(mw), VISIBLE_TRAILING_FIXED_COLUMN_WIDTH(mw)}
        };

        assert(!mw->matrix.disable_redisplay);

        DEBUGOUT(_XbaeDebug
                 (__FILE__, w,
                  "xbaeRedrawLabelsAndFixed() with expose (x,y)=((%d,%d)-(%d,%d))\n",
                  expose->x, expose->y, expose->width, expose->height));
        
        for (r = 0; r < sizeof row_regions / sizeof *row_regions; r++){
                if (row_regions[r].exists) {
                        for(c = 0; c < sizeof column_regions / sizeof *column_regions; c++){
                                if (column_regions[c].exists && (c != 0 || r != 0)) {
                                        XRectangle region = {
                                                column_regions[c].position,
                                                row_regions[r].position,
                                                column_regions[c].size,
                                                row_regions[r].size};
                                        xbaeRedrawRegion(mw, expose, &region);
                                }
                        }
                }
        }

        /* Danny Here */
        /*
         * Draw a shadow just inside row/column labels and around outer edge
         * of clip widget.  
         */
        if (mw->manager.shadow_thickness) {
                Dimension width, height;

                width =
                    VISIBLE_NON_FIXED_WIDTH(mw) + VISIBLE_FIXED_COLUMN_WIDTH(mw) +
                    VISIBLE_TRAILING_FIXED_COLUMN_WIDTH(mw) + 2 * mw->manager.shadow_thickness;
                height =
                    VISIBLE_NON_FIXED_HEIGHT(mw) + VISIBLE_FIXED_ROW_HEIGHT(mw) +
                    VISIBLE_TRAILING_FIXED_ROW_HEIGHT(mw) + 2 * mw->manager.shadow_thickness;

                DRAW_SHADOW(XtDisplay(mw), XtWindow(mw), "win", mw->manager.top_shadow_GC,
                            mw->manager.bottom_shadow_GC, mw->manager.shadow_thickness,
                            row_label_width + vert_sb_offset, column_label_height + horiz_sb_offset,
                            width, height, mw->matrix.shadow_type);
        }
}

/*
 * Redraws everything that is in the expose Rectangle
 */
void xbaeRedrawAll(XbaeMatrixWidget mw, XRectangle *expose)
{
        int c;
        Widget clips[] = {
                ClipChild(mw),
                TopClip(mw),
                LeftClip(mw),
                RightClip(mw),
                BottomClip(mw),
                RowLabelClip(mw),
                ColumnLabelClip(mw)
        };

        if (mw->matrix.disable_redisplay)
                return;

        for (c = 0; c < sizeof clips / sizeof *clips; c++) {
                if (XtIsManaged(clips[c])) {
                        XRectangle region = {
                                clips[c]->core.x,
                                clips[c]->core.y,
                                clips[c]->core.width,
                                clips[c]->core.height};
                        xbaeRedrawRegion(mw, expose, &region);
                }
        }

        xbaeRedrawLabelsAndFixed(mw, expose);
}
