

/*
 *  Author: Arvin Schnell
 */


#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <iostream>

using std::cerr;

#include "pcm-cdr.h"


#ifndef __BYTE_ORDER
#  error "undefined endianness"
#elif __BYTE_ORDER != __BIG_ENDIAN && __BYTE_ORDER != __LITTLE_ENDIAN
#  error "unsupported endianness"
#endif


PCMCDR::PCMCDR (const char* name, pcm_type_t type, pcm_mode_t mode,
		pcm_format_t format, int channels, int rate, int buffer_size)
    : PCM (name, type, mode, format, channels, rate, buffer_size),
      fd (-1)
{
    if (channels != 2) {
	cerr << "error: Illegal number of channels\n";
	return;
    }

    if (format != S16) {
	cerr << "error: Illegal format (resolution)\n";
	return;
    }

    if (rate != 44100) {
	cerr << "error: Illegal rate\n";
	return;
    }

    if (buffer_size != 1) {
	cerr << "error: Illegal buffer size\n";
	return;
    }

    switch (mode)
    {
	case READ:

	    fd = open (name, O_RDONLY);
	    if (fd == -1) {
		cerr << "error: Can't open " << name << ": "
		     << strerror (errno) << '\n';
		return;
	    }

	    struct stat fs;
	    if (fstat (fd, &fs) == -1) {
		cerr << "error: Can't stat " << name << ": "
		     << strerror (errno) << '\n';
		return;
	    }

	    if (fs.st_size % get_bytes_per_frame () != 0) {
		cerr << "error: Illegal size of " << name << '\n';
		return;
	    }

	    break;

	case WRITE:

	    fd = open (name, O_WRONLY | O_CREAT | O_TRUNC, 0666);
	    if (fd == -1) {
		cerr << "error: Can't open " << name << ": "
		     << strerror (errno) << '\n';
		return;
	    }

	    break;
    }

    ok = true;
}


PCMCDR::~PCMCDR ()
{
    if (fd > 0)
	close (fd);
}


void
PCMCDR::info () const
{
    cerr << "driver: cdr\n";
    PCM::info ();
}


size_t
PCMCDR::size ()
{
    struct stat fs;
    if (fstat (fd, &fs) == -1)
	return (size_t)(-1);

    return fs.st_size / get_bytes_per_frame ();
}


off_t
PCMCDR::seek (off_t offset, int whence)
{
    off_t i = offset * get_bytes_per_frame ();

    off_t j = ::lseek (fd, i, whence);
    if (j == (off_t)(-1))
	return (off_t)(-1);

    off_t k = j / get_bytes_per_frame ();

    return k;
}


size_t
PCMCDR::read (void* buffer, size_t frames)
{
    size_t i = frames * get_bytes_per_frame ();

    size_t j = ::read (fd, buffer, i);
    if (j == (size_t)(-1))
	return (size_t)(-1);

    size_t k = j / get_bytes_per_frame ();
#if __BYTE_ORDER == __LITTLE_ENDIAN
    swap_buffer (buffer, k);
#endif
    return k;
}


size_t
PCMCDR::write (void* buffer, size_t frames)
{
#if __BYTE_ORDER == __LITTLE_ENDIAN
    swap_buffer (buffer, frames);
#endif

    size_t i = frames * get_bytes_per_frame ();

    size_t j = ::write (fd, buffer, i);
    if (j == (size_t)(-1))
	return (size_t)(-1);

    size_t k = j / get_bytes_per_frame ();
    return k;
}
