

/*
 *  Author: Arvin Schnell
 */


#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <limits.h>

#include <X11/Xlib.h>
#ifdef DBE
#  include <X11/extensions/Xdbe.h>
#endif

#include "Trans.h"
#include "Scope.h"
#include "Sample.h"
#include "main.h"
#include "utils.h"


Scope::Scope ()
{
    realized = false;

    zoom = 0.0;

    drawing_mode = SINGLEBIT;	// FIXME
    draw_cross = false;		// FIXME
    num_count = 1;		// FIXME
}


void
Scope::allocbuffer ()
{
    XtGCMask gc_mask = 0;
    XGCValues gc_values;

    switch (drawing_mode) {

	case SINGLEBIT:
	    pixmap = XCreatePixmap (display, window, width, height, 1);
	    pixmapGC = XCreateGC (display, pixmap, gc_mask, &gc_values);
	    break;

	case FULLBIT:{
	    int depth = DefaultDepth (display, DefaultScreen (display));
	    pixmap = XCreatePixmap (display, window, width, height, depth);
	    pixmapGC = XCreateGC (display, pixmap, gc_mask, &gc_values);
	}
	break;

#ifdef DBE
	case DOUBLEBUFFER:
	    dbeBuffer = XdbeAllocateBackBufferName (display, window, XdbeUntouched);
	    dbeGC = XCreateGC (display, window, gc_mask, &gc_values);
	    break;
#endif

    }
}


void
Scope::freebuffer ()
{
    switch (drawing_mode) {

	case SINGLEBIT:
	    XFreePixmap (display, pixmap);
	    XFreeGC (display, pixmapGC);
	    break;

	case FULLBIT:
	    XFreePixmap (display, pixmap);
	    XFreeGC (display, pixmapGC);
	    break;

#ifdef DBE
	case DOUBLEBUFFER:
	    XdbeDeallocateBackBufferName (display, dbeBuffer);
	    XFreeGC (display, dbeGC);
	    break;
#endif

    }
}


void
Scope::clearbuffer ()
{
    switch (drawing_mode) {

	case SINGLEBIT:
	    XSetForeground (display, pixmapGC, 0);
	    XFillRectangle (display, pixmap, pixmapGC, 0, 0, width, height);
	    break;

	case FULLBIT:
	    XSetForeground (display, pixmapGC, xanalyser.backgroundcolor);
	    XFillRectangle (display, pixmap, pixmapGC, 0, 0, width, height);
	    break;

#ifdef DBE
	case DOUBLEBUFFER:
	    XSetForeground (display, dbeGC, xanalyser.backgroundcolor);
	    XFillRectangle (display, dbeBuffer, dbeGC, 0, 0, width, height);
	    break;
#endif

    }
}


void
Scope::resize (bool redraw)
{
    if (!realized)
	return;

    freebuffer ();

    myXGetDrawableSize (display, window, &width, &height);
    K.setB (0, height, width, 0);

    allocbuffer ();
    clearbuffer ();

    if (redraw) {
	XRectangle rect = { 0, 0, width, height };
	draw (rect);
    }
}


bool
Scope::realize (Display* display, Window window)
{
    if (realized)
	return false;
    realized = true;

    Scope::display = display;
    Scope::window = window;

    myXGetDrawableSize (display, window, &width, &height);

    XtGCMask gc_mask = 0;
    XGCValues gc_values;

    windowGC = XCreateGC (display, window, gc_mask, &gc_values);
    K.setA (-1.0, -1.0, +1.0, +1.0);
    K.setB (0, height, width, 0);

    allocbuffer ();
    clearbuffer ();

    xpts = new XPoint[sample.length];

    return true;
}


bool
Scope::destroy ()
{
    if (!realized)
	return false;

    XFreeGC (display, windowGC);

    freebuffer ();

    delete[] xpts;

    realized = false;

    return true;
}


void
Scope::clear (bool drawit)
{
    if (!realized)
	return;

    clearbuffer ();

    if (drawit) {
	XRectangle rect = { 0, 0, width, height };
	draw (rect);
    }
}


void
Scope::draw (XRectangle rect)
{
    XSetClipRectangles (display, windowGC, 0, 0, &rect, 1, Unsorted);

    int w2 = width / 2;
    int h2 = height / 2;
    int l = (width + height) / 32;

    switch (drawing_mode) {

	case SINGLEBIT:
	    XSetForeground (display, windowGC, xanalyser.datacolor);
	    XSetBackground (display, windowGC, xanalyser.backgroundcolor);
	    XCopyPlane (display, pixmap, window, windowGC, 0, 0, width, height,
			0, 0, 1);
	    if (draw_cross) {
		XSetForeground (display, windowGC, xanalyser.majorgridcolor);
		XDrawLine (display, window, windowGC, w2 - 3 * l, h2 + 3 * l,
			   w2 + 3 * l, h2 - 3 * l);
		XDrawLine (display, window, windowGC, w2 - l, h2, w2 + l, h2);
		XDrawLine (display, window, windowGC, w2, h2 - l, w2, h2 + l);
	    }
	    break;

	case FULLBIT:
	    XCopyArea (display, pixmap, window, windowGC, 0, 0, width, height, 0, 0);
	    break;

#ifdef DBE
	case DOUBLEBUFFER:
	    XdbeSwapInfo dbeSwapInfo;
	    dbeSwapInfo.swap_window = window;
	    dbeSwapInfo.swap_action = XdbeUndefined;
	    XdbeSwapBuffers (display, &dbeSwapInfo, 1);
	    break;
#endif

    }
}


bool
Scope::shot (const int32_t* buffer, bool drawit)
{
    if (!realized)
	return false;

    if (sample.frame_count % num_count == 0)
	clear (0);

    const double dx = xanalyser.dcadjust ? sample.dc[0] / INT_MAX : 0;
    const double dy = xanalyser.dcadjust ? sample.dc[1] / INT_MAX : 0;

    const double sf = M_LOG2E * exp (0.05 * zoom * log (10.0));

    for (unsigned int i = 0; i < sample.length; i++) {

	double x = (double)(buffer[2 * i + 0]) / (double)(INT_MAX) - dx;
	double y = (double)(buffer[2 * i + 1]) / (double)(INT_MAX) - dy;

	double h = hypot (x, y);
	double rdh = h == 0.0 ? 0.0 : sf * log (h + 1.0) / h;

	K.AtoB (x * rdh, y * rdh, xpts[i].x, xpts[i].y);

    }

    int w2 = width / 2;
    int h2 = height / 2;
    int l = (width + height) / 32;

    switch (drawing_mode) {

	case SINGLEBIT:
	    XSetForeground (display, pixmapGC, 1);
	    XDrawLines (display, pixmap, pixmapGC, xpts, sample.length,
			CoordModeOrigin);
	    break;

	case FULLBIT:
	    XSetForeground (display, pixmapGC, xanalyser.datacolor);
	    XDrawLines (display, pixmap, pixmapGC, xpts, sample.length,
			CoordModeOrigin);
	    if (draw_cross) {
		XSetForeground (display, pixmapGC, xanalyser.majorgridcolor);
		XDrawLine (display, pixmap, pixmapGC, w2 - 3 * l, h2 + 3 * l,
			   w2 + 3 * l, h2 - 3 * l);
		XDrawLine (display, pixmap, pixmapGC, w2 - l, h2, w2 + l, h2);
		XDrawLine (display, pixmap, pixmapGC, w2, h2 - l, w2, h2 + l);
	    }
	    break;

#ifdef DBE
	case DOUBLEBUFFER:
	    XSetForeground (display, dbeGC, xanalyser.datacolor);
	    XDrawLines (display, dbeBuffer, dbeGC, xpts, sample.length,
			CoordModeOrigin);
	    if (draw_cross) {
		XSetForeground (display, dbeGC, xanalyser.majorgridcolor);
		XDrawLine (display, dbeBuffer, dbeGC, w2 - 3 * l, h2 + 3 * l,
			   w2 + 3 * l, h2 - 3 * l);
		XDrawLine (display, dbeBuffer, dbeGC, w2 - l, h2, w2 + l, h2);
		XDrawLine (display, dbeBuffer, dbeGC, w2, h2 - l, w2, h2 + l);
	    }
	    break;
#endif

    }

    if (drawit && sample.frame_count % num_count == num_count - 1) {
	XRectangle rect = { 0, 0, width, height };
	draw (rect);
    }

    return true;
}


void
Scope::set_zoom (double zoom)
{
    Scope::zoom = zoom;
}
