

/*
 *  Author: Arvin Schnell
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <limits.h>
#include <assert.h>

#include <X11/Xlib.h>
#include <Xm/Xm.h>

#include "Analyser.h"
#include "Sample.h"
#include "main.h"
#include "utils.h"


inline unsigned short
Analyser::sx2fc (unsigned short sx) const
{
    assert (sx < width);
    return xfoo[sx];
}


inline unsigned short
Analyser::fc2sx (unsigned short fc) const
{
    assert (fc < num_fft);
    return xbarfoo[fc];
}


inline double
Analyser::fc2f (unsigned short fc) const
{
    assert (fc < num_fft);
    return ((double) (fc) + 0.5) / (double) (num_fft) * (double) (sample.rate / 2);
}


inline unsigned short
Analyser::f2fc (double f) const
{
    int fc = lrint (f / (double) (sample.rate / 2) * (double) (num_fft - 1));
    return clamp (fc, 0, num_fft - 1);
}


unsigned short
Analyser::f2sx (double f) const
{
    return fc2sx (f2fc (f));
}


double
Analyser::sx2f (unsigned short sx) const
{
    return fc2f (sx2fc (sx));
}


inline unsigned short
Analyser::db2sy (double db) const
{
    short sy = lrint ((db - xanalyser.upper_db) /
		      (xanalyser.lower_db - xanalyser.upper_db) *
		      (double) (height - 1));
    return clamp (sy, 0, (signed) (height) - 1);
}


Analyser::Analyser ()
{
    realized = false;

    alpha = 0.0;
    decay = 0.0;
    search = false;

    logfreq = false;		// FIXME

    xfoo = xbarfoo = 0;
}


void
Analyser::calcfoo ()
{
    if (xfoo != 0)
	delete xfoo;
    if (xbarfoo != 0)
	delete xbarfoo;

    xfoo = new int[width];
    xbarfoo = new int[num_fft];

    if (!logfreq) {

	for (unsigned int sx = 0; sx < width; sx++)
	    xfoo[sx] = lrint ((double) (sx) / (double) (width) *
			      (double) (num_fft));

	for (unsigned int fc = 0; fc < num_fft; fc++)
	    xbarfoo[fc] = lrint ((double) (fc) / (double) (num_fft) *
				 (double) (width));

    } else {

	double f = (log10 (num_fft) - 1.0) / width;

	for (unsigned int sx = 0; sx < width; sx++)
	    xfoo[sx] = lrint (10.0 * pow (10.0, sx * f));	// FIXME

	for (unsigned int fc = 0; fc < num_fft; fc++)
	    xbarfoo[fc] = lrint (log10 (0.1 * fc) / f);		// FIXME

    }
}


void
Analyser::resize (bool redraw)
{
    if (!realized)
	return;

    // first save marker positions
    double f[2] = { sx2f (marker[0]), sx2f (marker[1]) };

    myXGetDrawableSize (display, window, &width, &height);

    for (unsigned int fc = 0; fc < num_fft; fc++)
	value_y[fc] = db2sy (value_db[fc]);

    calcfoo ();

    marker[0] = f2sx (f[0]);
    marker[1] = f2sx (f[1]);

    if (redraw) {
	XRectangle rect = { 0, 0, width, height };
	draw (rect);
    }
}


void
Analyser::clear (bool drawit)
{
    if (!realized)
	return;

    for (unsigned int fc = 0; fc < num_fft; fc++) {
	value_db[fc] = xanalyser.lower_db;
	value_y[fc] = value_y_old[fc] = db2sy (xanalyser.lower_db);
    }

    peak_f = 0;
    peak_db = xanalyser.lower_db;

    calcfoo ();

    if (drawit) {
	XRectangle rect = { 0, 0, width, height };
	draw (rect);
    }
}


void
Analyser::drawpeaktext (bool showit)
{
    if (!showit) {

	XSetForeground (display, gc, xanalyser.backgroundcolor);
	XDrawString (display, window, gc, peak_sx, peak_sy, peak_text,
		     strlen (peak_text));

    } else {

	snprintf (peak_text, 10, "%i", (int) peak_f);

	peak_sx = f2sx (peak_f) - (strlen (peak_text) * font_width) / 2;
	peak_sy = db2sy (peak_db) - 6;

	XSetForeground (display, gc, xanalyser.markercolor);
	XDrawString (display, window, gc, peak_sx, peak_sy, peak_text,
		     strlen (peak_text));

    }
}


void
Analyser::drawpeakmarker ()
{
    for (int m = 0; m < 2; m++) {
	XRectangle rect = { marker[m], 0, 1, height };
	draw (rect);
    }
}


void
Analyser::drawgrid (bool withtext)
{
    for (int f = 0; f < sample.rate / 2;) {

	short sx = f2sx (f);

	if (f % 1000 == 0)
	    XSetForeground (display, gc, xanalyser.majorgridcolor);
	else
	    XSetForeground (display, gc, xanalyser.minorgridcolor);

	if (f > 0)
	    XDrawLine (display, window, gc, sx, 0, sx, height - 1);

	if (withtext) {
	    const int size = 10;
	    char buffer[size];
	    snprintf (buffer, size, "%3.1f", (double) (f) / 1000.0);
	    XDrawString (display, window, gc, sx + 3, font_height,
			 buffer, strlen (buffer));
	}

	if (!logfreq)
	    f += 2500;
	else {
	    f *= 2;
	    if (f < 500)
		f = 500;
	}
    }

    for (int db = (int)(20.0 * floor (xanalyser.upper_db / 20.0));
	 db > xanalyser.lower_db; db += -20) {

	short sy = db2sy ((double) (db));

	if (db == 0)
	    XSetForeground (display, gc, xanalyser.majorgridcolor);
	else
	    XSetForeground (display, gc, xanalyser.minorgridcolor);

	XDrawLine (display, window, gc, 0, sy, width - 1, sy);

	if (withtext) {
	    const int size = 10;
	    char buffer[size];
	    snprintf (buffer, size, "%+d", db);
	    XDrawString (display, window, gc, width - 2 - font_width *
			 strlen (buffer), sy - 2, buffer, strlen (buffer));
	}
    }
}


void
Analyser::draw (XRectangle rect, bool complete)
{
    XSetClipRectangles (display, gc, 0, 0, &rect, 1, Unsorted);

    if (search) {
	drawpeaktext (false);
	drawpeaktext (true);	// just to omit flickering
    }

    int first = /* min( */ rect.x /*, num_fft-1 ) */ ;
    int last = /*min( */ rect.x + rect.width - 1 /*, num_fft-1 ) */ ;

    if (complete) {

	XSetForeground (display, gc, xanalyser.backgroundcolor);
	XFillRectangle (display, window, gc, 0, 0, width, height);

	XSetForeground (display, gc, xanalyser.datacolor);
	for (int sx = first; sx <= last; sx++) {
	    int fc = sx2fc (sx);
	    XDrawLine (display, window, gc, sx, value_y[fc], sx, height - 1);
	}

    } else {

	XSetForeground (display, gc, xanalyser.backgroundcolor);
	for (int sx = first; sx <= last; sx++) {
	    int fc = sx2fc (sx);
	    if (value_y[fc] > value_y_old[fc])
		XDrawLine (display, window, gc, sx, value_y_old[fc],
			   sx, value_y[fc] - 1);
	}

	XSetForeground (display, gc, xanalyser.datacolor);
	for (int sx = first; sx <= last; sx++) {
	    int fc = sx2fc (sx);
	    if (value_y[fc] < value_y_old[fc])
		XDrawLine (display, window, gc, sx, value_y_old[fc],
			   sx, value_y[fc]);
	}

    }

    int withtext = complete;
    if (lrint (0.25 * sample.frames_per_second) > 0)
	withtext |= sample.frame_count %
	    lrint (0.25 * sample.frames_per_second) == 0;
    drawgrid (withtext);

    if (search) {

	for (int m = 0; m < 2; m++) {
	    short sx = marker[m];
	    XSetForeground (display, gc, xanalyser.markercolor);
	    XDrawLine (display, window, gc, sx, 0, sx, height - 1);
	}

	drawpeaktext (true);
    }
}


bool
Analyser::realize (Display* display, Window window)
{
    if (realized)
	return false;
    realized = true;

    Analyser::display = display;
    Analyser::window = window;

    // create GCs

    XtGCMask gc_mask = 0;
    XGCValues gc_values;

    gc = XCreateGC (display, window, gc_mask, &gc_values);

    // get width and height

    myXGetDrawableSize (display, window, &width, &height);

    // font

    XSetFont (display, gc, xanalyser.gridfont->fid);
    font_width = xanalyser.gridfont->max_bounds.rbearing
	- xanalyser.gridfont->min_bounds.lbearing;
    font_height = xanalyser.gridfont->max_bounds.ascent
	+ xanalyser.gridfont->max_bounds.descent;

    num_fft = sample.length / 2 + 1;	// yes +1

#ifdef HAVE_FFTW
    plan = rfftw_create_plan (sample.length, FFTW_REAL_TO_COMPLEX, FFTW_ESTIMATE);
#endif

    // allocate memory

#ifdef HAVE_FFTW
    value_u = new fftw_real[sample.length];
    value_tmp = new fftw_real[sample.length];
#else
    value_u = new double[sample.length];
#endif

    value_db = new double[num_fft];
    value_y = new short[num_fft];
    value_y_old = new short[num_fft];

    value_e = new double[sample.length];

    clear (false);

    envelope ();

    marker[0] = 0;
    // note: it might be that width - 1 != num_fft - 1
    marker[1] = width - 1;

    return true;
}


bool
Analyser::destroy ()
{
    if (!realized)
	return false;

#ifdef HAVE_FFTW
    fftw_destroy_plan (plan);
#endif

    delete[] value_u;
    delete[] value_db;
    delete[] value_y;
    delete[] value_y_old;

    delete[] value_e;

#ifdef HAVE_FFTW
    delete[] value_tmp;
#endif

    XFreeGC (display, gc);

    realized = false;

    return true;
}


bool
Analyser::shot (const int32_t* buffer, int channel, bool drawit)
{
    if (!realized)
	return false;

    analyse (buffer, channel);

    if (drawit) {
	XRectangle rect = { 0, 0, width, height };
	draw (rect, false);
    }

    return true;
}


inline double
sqr (double arg)
{
    return arg * arg;
}


void
Analyser::envelope ()
{
    // normalised to 1.0

    double sum = 0.0;

    for (unsigned int i = 0; i < sample.length / 2; i++) {

#if 1
	// Gauss
	double x = ((double)(i) - (double)(sample.length / 2)) /
	    (double)(sample.length);
	value_e[i] = exp (-alpha * sqr (x));
#else
	// sin
	double x = ((double)(i) / (double)(sample.length)) * 2.0 * M_PI;
	double a = alpha / 100.0;	// 0.0 .. 0.5
	value_e[i] = 1.0 - a - a * cos (x);
#endif

	sum += value_e[i];
    }

    double tmp = sample.length / (2.0 * sum);

    for (unsigned int i = 0; i < sample.length / 2; i++) {
	value_e[i] *= tmp;
	value_e[sample.length - 1 - i] = value_e[i];
    }
}


void
Analyser::analyse (const int32_t* buffer, int channel)
{
    snd2u (buffer, channel);

#ifdef HAVE_FFTW
    rfftw_one (plan, value_u, value_tmp);
    const double norm = -10.0 * log10 (sqr ((double) (sample.length))) + 6.0;
#else
    rfft (value_u, sample.length, +1);
    const double norm = -27.1;
#endif

    for (unsigned int fc = 0; fc < num_fft; fc++) {

	double db, db_old, p;

	value_y_old[fc] = value_y[fc];
	db_old = value_db[fc];

#ifdef HAVE_FFTW
	if (fc != 0 && fc != num_fft - 1)
	    p = sqr (value_tmp[fc]) + sqr (value_tmp[sample.length - fc]);
	else
	    p = sqr (value_tmp[fc]);
#else
	if (fc != 0 && fc != num_fft - 1)
	    p = sqr (value_u[2 * fc + 0]) + sqr (value_u[2 * fc + 1]);
	else
	    p = sqr (value_u[fc == 0 ? 0 : 1]);
#endif

	db = 10.0 * log10 (p) + norm;	// gauge ( 990.5273 Hz, 0 dB )

	if (sample.frame_count > 0) {
	    double tmp = db_old - decay / sample.frames_per_second;
	    if (tmp > db)
		db = tmp;
	}

	value_db[fc] = fclamp (db, xanalyser.lower_db, xanalyser.upper_db);
	value_y[fc] = db2sy (value_db[fc]);

    }

    if (search)
	peaksearch (false);
}


void
Analyser::peaksearch (bool always)
{
    if (always || peak_count++ > 0.25 * sample.frames_per_second) {
	peak_db = xanalyser.lower_db;
	peak_count = 0;
    }

    unsigned short sc = sx2fc (marker[0]);
    unsigned short ec = sx2fc (marker[1]);

    for (unsigned int fc = sc; fc <= ec; fc++) {
	if (value_db[fc] > peak_db) {
	    peak_f = fc2f (fc);
	    peak_db = value_db[fc];
	}
    }
}


void
Analyser::snd2u (const int32_t* buffer, int channel)
{
    const double norm = (channel == MONO ? 0.5 : 1.0) / (double) INT_MAX;

    switch (channel) {

	case MONO: {
	    const double d = xanalyser.dcadjust ? norm * (sample.dc[0] +
							  sample.dc[1]) : 0.0;
	    for (unsigned int i = 0; i < sample.length; i++)
		value_u[i] = norm * value_e[i] * (double) (buffer[2 * i + 0] +
							   buffer[2 * i + 1]) - d;
	} break;

	case LEFT: {
	    const double d = xanalyser.dcadjust ? norm * sample.dc[0] : 0.0;
	    for (unsigned int i = 0; i < sample.length; i++)
		value_u[i] = norm * value_e[i] * (double) (buffer[2 * i + 0]) - d;
	} break;

	case RIGHT: {
	    const double d = xanalyser.dcadjust ? norm * sample.dc[1] : 0.0;
	    for (unsigned int i = 0; i < sample.length; i++)
		value_u[i] = norm * value_e[i] * (double) (buffer[2 * i + 1]) - d;
	} break;

    }
}


void
Analyser::set_alpha (double alpha)
{
    Analyser::alpha = alpha;
    if (realized)
	envelope ();
}


void
Analyser::set_decay (double decay)
{
    Analyser::decay = decay;
}


void
Analyser::set_search (bool search)
{
    Analyser::search = search;

    if (!realized)
	return;

    if (search) {

	peaksearch (true);

	XSetClipMask (display, gc, None);
	drawpeaktext (true);

	for (int m = 0; m < 2; m++) {
	    XRectangle rect = { marker[m], 0, 1, height };
	    draw (rect);
	}

    } else {

	XSetClipMask (display, gc, None);
	drawpeaktext (false);

	for (int m = 0; m < 2; m++) {
	    XRectangle rect = { marker[m], 0, 1, height };
	    draw (rect);
	}

	XSetClipMask (display, gc, None);
	drawgrid (true);

    }
}


void
Analyser::set_marker (short sx)
{
    if (!search)
	return;

    if (!realized)
	return;

    sx = clamp (sx, 0, (signed) (width) - 1);

    int m = (abs (marker[0] - sx) > abs (marker[1] - sx));

    short old = marker[m];
    marker[m] = sx;

    // remove old marker and text

    XRectangle rect = { old, 0, 1, height };
    draw (rect);

    XSetClipMask (display, gc, None);
    drawpeaktext (false);

    // redraw grid

    drawgrid (true);

    // draw new marker and text

    peaksearch (true);

    for (int m = 0; m < 2; m++) {
	XRectangle rect = { marker[m], 0, 1, height };
	draw (rect);
    }

    XSetClipMask (display, gc, None);
    drawpeaktext (true);
}
