;;########################################################################
;; generic.lsp
;; Copyright (c) 1993-96 by Forrest W. Young
;; This file contains generic functions for menu items 
;; and guidemap and authoring load-on-demand functions.
;;########################################################################

(defun remove-selection (&optional list)
"Args: (&optional list)
For multivariate or matrix data removes the currently selected variables and observations (multivariate) or matrices (matrix). Ignored for table data."
  (when *current-data*
        (setcd *current-data*)
        (when (not (send *current-data* :ways))
              (let* ((nobs  (send *current-data* :nobs))
                     (nvar  (send *current-data* :nvar))
                     (which-obs-selected
                      (which (mapcar #'equal (repeat 'SELECTED nobs) 
                                     (send *current-data* :obs-states))))
                     (which-vars-selected
                      (which (mapcar #'equal (repeat 'SELECTED nvar) 
                                     (send *current-data* :var-states))))
                     (nmats nil)
                     (which-mats-selected nil)
                     )
                (when (send *current-data* :matrices)
                      (setf nmats (send *current-data* :nmat))
                      (setf which-mats-selected
                            (which (mapcar #'equal (repeat 'SELECTED nmats)
                                      (send *current-data* :mat-states))))
                      )
                (when which-obs-selected
                      (send *obs-window* :point-state 
                            which-obs-selected  'INVISIBLE)
                      (send *obs-window* :redraw)
                      (send *current-data* :obs-states 
                            (send *obs-window* :point-state (iseq nobs))))
                (when which-mats-selected
                      (send *obs-window* :point-state 
                            which-mats-selected  'INVISIBLE)
                      (send *obs-window* :redraw)
                      (send *current-data* :mat-states
                            (send *obs-window* :point-state (iseq nmats))))
                (when which-vars-selected
                      (send *var-window* :point-state
                            which-vars-selected 'INVISIBLE)
                      (send *current-data* :var-states 
                            (send *var-window* :point-state (iseq nvar)))
                      (send *var-window* :redraw))))))

(defun remove-selected-data (&optional list)
  "Alias for remove-selection."
  (remove-selection))

;(defun edit-data () (show-datasheet)) in dashobj1

(defun load-edit ()
"Args: none
Loads a Lisp Program, including ViSta Scripts and Examples."
  (let ((f (open-file-dialog t)))
    (when f (load f) (format t "; finished loading ~s~%" f))))

(defun save-data (&optional name)
"Args: (&optional name)
Saves the current data to a file named NAME"
  (if name (send current-data :save-data name)
      (send current-data :save-data)))

(defun create-data (&rest args)
"Args: (&rest args)
Makes the current object create output data.
Arguments depend on which object is current." 
  (if args (apply #'send current-object :create-data args)
      (send current-object :create-data)))

(defun delete-data ()
"Args: none
Deletes the current data object and all its child objects."
  (send *workmap* :delete-data))

(defun delete-model ()
"Args: none
Deletes the current model object and all its child objects."
  (send *workmap* :delete-model))

(defun analyze-data ()
  (cond
    ((send *current-data* :matrices)
     (multidimensional-scaling :dialog t))
    ((send *current-data* :ways)
     (analysis-of-variance :dialog t))
    (t
     (let ((choice (choose-item-dialog "Analyze Data ..."
                       '("Analysis of Variance"
                         "Correspondence Analysis"
                         "Multivariate Regression" 
                         "Principal Components"
                         "Regression Analysis"
                         "Univariate Analysis"
                          ))))
       (case choice
         (0 (Analysis-of-Variance :dialog t))
         (1 (Correspondence-analysis :dialog t))
         (2 (Multivariate-Regression :dialog t))
         (3 (Principal-components :dialog t))
         (4 (regression-analysis :dialog t))
         (5 (univariate-analysis :dialog t))
         )))))

(defun visualize-data (&key dialog tour)
"Function Args: DIALOG TOUR
Sends current-data the Visualize message.  Presents dialog when DIALOG=t. Presents high-dimensional guided tour when TOUR=T and dimensionality at least 6."
  (send current-data :visualize :dialog dialog :tour tour)
  t)

(defun visualize-model ()
"Function Args: None
Sends current-model the Visualize message"
  (send current-model :visualize))

(defun visualize ()
"Args: none
Displays a visualization of the current object."
  (send current-object :visualize))

(defun report ()
"Args: none
Makes a numeric report of the current object."
  (send current-object :report)
  t)

(defun report-model (&key dialog) 
"Args: (&key (dialog nil))
Makes a numeric report of the current model."
  (when current-model (send current-model :report :dialog dialog))
  t)

(defun summary 
  (&key (moments t) (quartiles t) ranges correlations covariances)
"Args: (&key (moments t) (quartiles t) ranges correlations covariances)
Presents univariate summary statistics for each variable
in the current data object."
  (send current-data :summary :moments moments :quartiles quartiles
        :ranges ranges :correlations correlations :covariances covariances))

(defun merge-vars (&optional name)
"Args: &optional name
Merges the variables of the current and previous data objects."
  (if name (send current-data :merge-variables name)
      (send current-data :merge-variables)))

(defun merge-variables (&optional name)
"Args: &optional name
Merges the variables of the current and previous data objects as object NAME."
  (if name (send current-data :merge-variables name)
      (send current-data :merge-variables)))

(defun merge-obs (&optional name)
"Args: &optional name
Merges the observations of the current and previous data objects."
  (if name (send current-data :merge-observations name)
      (send current-data :merge-observations)))

(defun merge-observations (&optional name)
"Args: &optional name
Merges the observations of the current and previous data objects."
  (if name (send current-data :merge-observations name)
      (send current-data :merge-observations)))

(defun select-variables (var-name-list)
"Args: VAR-NAME-LIST
Selects variables and displays the selection in the variable window, if open."
  (setcd *current-data*)  
  (send *current-data* :select-variables var-name-list))

(defun select-observations (obs-label-list)
"Args: OBS-LABEL-LIST
Selects observations and displays the selection in the observations window, if open."
  (setcd *current-data*)
  (send  *current-data* :select-observations obs-label-list))

;added next function fwy4.25
(defun select-matrices (mats-name-list)
"Args: MATS-NAME-LIST
Selects matrices and displays the selection in the mats window, if open."
  (setcd *current-data*)
  (send  *current-data* :select-matrices mats-name-list))


(defun show-bug-list () (send *vista* :show-bug-list))

(defun interpret-model () (send current-model :interpret-model))

(defun show-workmap () (send *workmap* :gui t))

(defun hide-workmap () (send *workmap* :close))

(defun hide-guidemap () (send *guidemap* :close))

#+macintosh(defun show-xlisp-stat () (send *listener* :show-window))
#-macintosh(defun show-xlisp-stat () )

#+macintosh(defun hide-xlisp-stat () (send *listener* :close))

(defun preferences () (send *vista* :preferences))

(defun toplevel () (top-level))

(defun about-these-data (&optional string &key size title
      location frame-location (show t))
"Args: &OPTIONAL STRING &KEY (SIZE '(300 75)) (LOCATION '(100 100)) (SHOW t)
Creates *about-window* if it doesn't exist. Then shows the about information for the current data  in *about-window*. If STRING is specified, changes about information to STRING."
  (when (not *about-window*)
        (setf *about-window* 
              (send *vista* :create-message-window 
                    :location location :show show :size size
                    :title (format nil "About the ~s data" (send *current-data* :name)))))
  (if string
      (cond 
        ( *current-data* 
          (send current-data :about string)
          (about-these-data)
          (when size (apply #'send *about-window* :size size))
          (when location (apply #'send *about-window* :location location))
          (when frame-location (apply #'send *about-window* :frame-location frame-location)))
        (t
         (about-message "There are no data.")))
      (about-message 
       (send *current-data* :about) 
       :size size
       :location location
       :frame-location frame-location
       :title (format nil "About the ~s data" (send *current-data* :name)) 
       :beep nil))
  *about-window*)

(defun show-varobs ()
"Args: none
Shows both the variable and the observation windows."
  (list-variables)
  (list-observations))

(defun list-var ()
"Args: none
Lists the variable names of the current data object."
  (send *vista* :list-variables)
  t)

(defun list-vars ()
"Args: none
Lists the variable names of the current data object."
  (send *vista* :list-variables)
  t)

(defun list-variables ()
"Args: none
Lists the variable names of the current data object."
  (send *vista* :list-variables)
  t)

(defmeth mv-data-object-proto :list-variables ()
"Args: none
Lists the variable names of the specified data object."
  (if (not (eq *current-data* self)) (setcd self))
  (send *vista* :list-variables)
  t)

(defmeth mv-data-object-proto :list-vars ()
"Args: none
Lists the variable names of the specified data object."
  (if (not (eq *current-data* self)) (setcd self))
  (send *vista* :list-variables)
  t)

(defmeth mv-data-object-proto :list-var ()
"Args: none
Lists the variable names of the specified data object."
  (if (not (eq *current-data* self)) (setcd self))
  (send *vista* :list-variables)
  t)

(defun list-obs ()
"Args: none
Lists the observation labels of the current data object."
  (send *vista* :list-observations)
  t)

(defun list-observations ()
"Args: none
Lists the observation labels of the current data object."
  (send *vista* :list-observations)
  t)

(defmeth mv-data-object-proto :list-observations ()
"Args: none
Lists the observations labels of the specified data object."
  (if (not (eq *current-data* self)) (setcd self))
  (send *vista* :list-observations)
  t)

(defmeth mv-data-object-proto :list-obs ()
"Args: none
Lists the observations labels of the specified data object."
  (if (not (eq *current-data* self)) (setcd self))
  (send *vista* :list-observations)
  t)

(defun list-mat ()
"Args: none
Lists the matrix names of the current dissimilarity data object."
  (send *vista* :list-matrices)
  t)

(defun list-mats ()
"Args: none
Lists the matrix names of the current dissimilarity data object."
  (send *vista* :list-matrices)
  t)

(defun list-matrices ()
"Args: none
Lists the variable names of the current dissimilarity data object."
  (send *vista* :list-matrices)
  t)

(defmeth mv-data-object-proto :list-matrices ()
"Args: none
Lists the matrix names of the specified data object."
  (if (not (eq *current-data* self)) (setcd self))
  (send *vista* :list-matrices)
  t)

(defmeth mv-data-object-proto :list-mats ()
"Args: none
Lists the matrix names of the specified data object."
  (if (not (eq *current-data* self)) (setcd self))
  (send *vista* :list-matrices)
  t)

(defmeth mv-data-object-proto :list-mat ()
"Args: none
Lists the matrix names of the specified data object."
  (if (not (eq *current-data* self)) (setcd self))
  (send *vista* :list-matrices)
  t)

(defun list-cells ()
"Args: none
Lists the cell names of the current table data object."
  (send *vista* :list-cells)
  t)

(defmeth mv-data-object-proto :list-cells ()
"Args: none
Lists the cell names of the specified data object."
  (if (not (eq *current-data* self)) (setcd self))
  (send *vista* :list-cells)
  t)

(defun close-windows ()
"Args: none
Closes the data object windows."
  (send *var-window* :close)
  (send *obs-window* :close))

(defun interpret-model () 
"Args: none
Provides a model interpretation."
  (send current-model :interpret-model))

;;########################################################################
;;guidemap load-on-demand functions
;;########################################################################

(defun show-guidemap () 
"Args: none
Function to show the guidemap. Uses the Guidance function." 
  (if current-object 
      (let ((current-icon-name 
             (send (select (send *workmap* :icon-list) 
                           (send *workmap* :selected-icon))
                   :slot-value 'proto-name))
            ) 
        (if (or (equal current-icon-name
                       (send dob-icon-proto :slot-value 'proto-name))
                (equal current-icon-name
                       (send dib-icon-proto :slot-value 'proto-name))
                (equal current-icon-name
                       (send tab-icon-proto :slot-value 'proto-name)))
            (guidance "data")
            (guidance "model")))
      (guidance)))

;fwy 4.28 7/23/97 added name argument for applets
(defun guidance (&optional string name)
  "Args: string
Load-on-demand version of the function which provides guidance. For model guidance STRING is MODEL, DATA for data guidance, APPLET for applet NAME guidance.  Otherwise, general guidance."
;Complete function is in the guidemap file 
(error-message "WARNING: GuideMaps are experimental and buggy. Please use with caution.")
  (load (strcat *vista-dir-name* "guidemap"))
        (guidance string name))

(defun author (&optional (logical nil set) &key new)
"Args: (&optional logical &key new)
Load-on-demand version of the function which enables authoring guidemaps."
;Complete function is in guidemap file.
(when set 
      (load (strcat *vista-dir-name* "author"))
      (author logical))
(when (not set) nil))

(defun read-clipboard ()
"Args: none
Returns the value of the Macintosh clipboard provided it contains a single expression readable by the read function."
  (send *listener* :show-window)
  (read t (send *listener* :paste-from-clip)))

(defun remove-missing-data-rows (dmat &key labels)
"Args: DMAT &key LABELS
DMAT is a multivariate data matrix and LABELS is an optional list of row labels.  Returns a matrix with only the rows of DMAT which have no missing data (missing data are coded as the symbol NIL). If LABELS, then returns a list whose first element is DMAT and second element is a new list of labels with appropriate row labels removed."  
  (let* ((nrows (select (array-dimensions dmat) 0)) 
         (ncols (select (array-dimensions dmat) 1)) (ndataums nil)
         (outdata nil)
         (outmat nil)
         (outlab nil)
         (nok 0)
         )
    (dotimes (i nrows)
             (setf ndatums
                   (length (remove t (map-elements #'equal nil 
                                                   (row dmat i)))))
             (when (= ndatums ncols)
                   (setf nok (1+ nok))
                   (when labels 
                         (setf outlab (combine outlab (select labels i))))
                   (setf outdata (combine outdata (row dmat i))) ))
    (when (> nok 0) (setf outmat (matrix (list nok ncols) (rest outdata)))) 
    (if labels
        (list outmat (rest outlab))
        outmat)))

(defun two-button-dialog 
  (text &key (first-button "OK") (second-button "Cancel"))
  (let* ((text (send text-item-proto :new text))
         (ok (send modal-button-proto :new first-button
                   :action #'(lambda () t)))
         (no (send modal-button-proto :new second-button))
         (dialog (send modal-dialog-proto :new 
                       (list text (list ok no)) 
                       :default-button ok)))
    (send dialog :modal-dialog)))

;4.28 fwy moved here from vista.lsp so guidance preference will work

(defun error-message (string &optional error fatal)
  (let* ((numchar (length string))
         (numlines (ceiling numchar 50))
         (message (format nil "ViSta Warning Message:~%")) 
         (start 0)
         (finish nil))
    (when error
          (sysbeep)
          (setf message (format nil "ViSta Error Message - Computations Interrupted.~%"))) 
    (when fatal
          (sysbeep)
          (setf message (format nil "ViSta Fatal Error Message - Computations Terminated.~%"))) 
    (setf string (strcat " " string))
    (sysbeep)
    (dotimes (i numlines)
             (setf start (position #\  string :start (* i 50))) 
             (setf finish (position #\  string :start
                                    (min numchar (* (1+ i) 50))))
             (when (not finish) (setf finish (1+ numchar))) 
             (setf message (strcat message 
                           (format nil "~%~a" (subseq string start finish))))
             (when (> finish numchar)  (return)))
    (message-dialog message)))

(defmeth graph-proto :eval-selection ()) ;fwy 4.28
(defmeth graph-proto :edit-selection ()) ;fwy 4.28

(provide "generic")