;;************************************************************************
;; datavis.lsp 
;; contains code for visualizing non-categorical multivarite data  
;; copyright (c) 1997 by Forrest W. Young
;;************************************************************************

(require (strcat *code-dir-name* "overlay"))

(defmeth mv-data-object-proto :visualize (&key dialog tour) 
"Message args: dialog
Presents a visualization of the active portion of the data. A spreadplot is presented. If DIALOG=T then, when appropriate, requests if want Guided Tour."
  (if (not (eq current-object self)) (setcd self))
  (send self :visualize-continuous-data dialog tour)
  t)

(defmeth mv-data-object-proto :visualize-continuous-data (dialog tour)
  (let ((n-numord (send self :active-nvar '(numeric ordinal)))
        (n-cat (send self :active-nvar '(category)))
        (choice 0)
        (obsvar-states nil)
        )
    (when (= n-numord 0)
          (fatal-message "There must be at least one active numeric or ordinal variable to visualize."))
    (cond 
      ((and (> n-cat 0) (= n-numord 1))
       (when (send self :mv-to-table) 
             (visualize-data)
             ))
      ((= n-numord 1)
       (send self :visualize-one-variable
             (send self :active-data '(numeric ordinal))
             (first (send self :active-variables '(numeric ordinal)))
             (send self :active-labels) ))
      ((= n-numord 2)
       (send self :visualize-two-variables
             (column-list (send self :active-data-matrix '(numeric ordinal)))
             (send self :active-variables '(numeric ordinal))
             (send self :active-labels))
       (when (> n-cat 0) 
             (when (not (send self :vis-error-msg "visualize")) 
                   ;(top-level)
                   ))
             )
      (t
       (when (> n-numord 5)
             (if dialog
                 (setf choice 
                       (choose-item-dialog 
                        "Choose Type of Visualization:"
                        (list "Standard Multivariate SpreadPlot" "High-Dimensional Guided TourPlot")))
                 (setf choice (if tour 1 0))))
       (case choice
         (0  (send self :visualize-n-variables 
                   (send self :active-data-matrix '(numeric ordinal))
                   (send self :active-variables '(ordinal numeric))
                   (send self :active-labels)))
         (1 (send self :visualize-tour-plot)))
       (when (> n-cat 0) 
             (when (not (send self :vis-error-msg "visualize")) 
                   ;(top-level)
                   ))
       ))
    ))
  
(defmeth mv-data-object-proto :visualize-tour-plot ()
  (load (strcat *vista-dir-name* "visuals"))
  (send current-data :visualize-tour-plot))

;The next two methods are meant to avoid using the obs/var windows 
;during visualization, because, counterintuitively, they dont do anything.
;Hhowever, other things go wrong if these methods are used, so theyre not.
;It should be that the obs/var windows are made to be useful to visualization.

(defmeth mv-data-object-proto :hide-obsvar-names ()
  (let ((obsptsts (send *obs-window* :point-state 
                       (iseq (send *obs-window* :num-points))))
        (varptsts (send *var-window* :point-state 
                       (iseq (send *var-window* :num-points))))
        )
    (send *obs-window* :point-state 
          (iseq (send *obs-window* :num-points)) 'invisible)
    (send *var-window* :point-state 
          (iseq (send *obs-window* :num-points)) 'invisible)
    (send *obs-window* :redraw)
    (send *var-window* :redraw)
    (list obsptsts varptsts)))

(defmeth mv-data-object-proto :show-obsvar-names (states)
  (send *obs-window* :point-state 
        (iseq (send *obs-window* :num-points)) (first states))
  (send *var-window* :point-state 
        (iseq (send *var-window* :num-points)) (second states))
  (send *obs-window* :redraw)
  (send *var-window* :redraw)
  t)

(defmeth mv-data-object-proto :vis-error-msg (word)
  (vista-message 
#+msdos(format nil "The category variable(s) were not used.~2%To use them, select just one numeric or ordinal variable and one or more category variables. Then you can ~a the categories of the numeric or ordinal variable." word)
#-msdos(format nil "The category variable(s) were not used.~2%To use them, select just one numeric or ordinal variable and one or more category variables. Then you can ~a the categories of the numeric or ordinal variable." word)
   ) t)


(defmeth mv-data-object-proto :visualize-n-variables
  (data-matrix variable-labels point-labels)
"Message args: (data-matrix point-labels variable-labels)
Visualization that presents scat-mat, scat-plot, histogram and spin-plot."
  (let* ((variables (column-list data-matrix))
         (nvar (length variables))
         (nobs (send self :nobs))
         (cutoff (floor (/ (min screen-size) 85)));controls scatmat scat sizes 
         (scatmat (when (< nvar (1+ cutoff)) 
                        (scatterplot-matrix variables :scale-type nil
                                            :point-labels point-labels
                                            :variable-labels variable-labels
                                            :show nil)))
         (spin-plot (spin-plot  variables :scale-type 'variable
                                :point-labels point-labels
                                :variable-labels variable-labels
                                :show nil))
         (var-list (when (> nvar cutoff)
                         (name-list variable-labels :show nil 
                                    :title "Variables" :menu nil)))
         (q-plot (quantile-plot variables :reg-line t 
                              :variable-labels variable-labels
                              :point-labels point-labels 
                              :show nil))
         (box-plot (boxplot variables 
                              :variable-labels variable-labels
                              :point-labels point-labels
                              :connect-points t 
                              :equate t
                              :show nil))
         (histogram (histogram variables :variable-labels variable-labels
                                        :show nil))
         (obs-list (name-list point-labels :title "Observations" :show nil))
         (sp (spread-plot (matrix '(2 3) 
                                  (list (if (> nvar cutoff) 
                                            var-list scatmat) 
                                        spin-plot box-plot 
                                        obs-list q-plot histogram))))
         (plot-matrix (send sp :plot-matrix))
         (spin-var (list 0 1 2))
         )

    (defmeth sp :spreadplot-help ()
      (plot-help-window (strcat "SpreadPlot Help"))
      (paste-plot-help (format nil "This is the SpreadPlot for Multivariate Data. In this SpreadPlot the windows are linked by the data's observations and variables.~2%The ~a window, which is in the upper left corner, lets you choose which variables are displayed in other windows. "(if (> nvar cutoff) "Variables" "Scatterplot Matrix")))
(if (> nvar cutoff)
    (paste-plot-help (format nil "You can select a single variable by clicking on a variable name, or you can select several variables by draging or shift-clicking on several names."))
    (paste-plot-help (format nil "You can select a single variable by clicking on a diagonal cell in this matrix, or you can select two variables by clicking on an off-diagonal cell. You can also select several variables by shift-clicking on several cells.")))
(paste-plot-help (format nil "Selecting variables causes new information about the variables to be displayed in the other windows.~2%"))
      (show-plot-help)
      (call-next-method :points t :bars t :labels t :flush nil))

    (defmeth spin-plot :update-plotcell (i j args)
      (when (and (= i 0) (= j 0))
            (let* ((cur-var-nums (remove-duplicates (first args)))
                   (cur-var-names (remove-duplicates 
                                   (first (second args))  :test 'equal))
                   (numvars (send self :num-variables))
                   )
             ;(format t "S ~d ~d ~%"cur-var-nums cur-var-names)
              (when (<= (length cur-var-nums) 2)
                    (setf cur-var-nums 
                          (select 
                           (combine cur-var-nums 
                            (set-difference (send self :current-variables)
                                            cur-var-nums))
                           (iseq 3)))
                    (setf cur-var-names 
                          (select (send self :variable-labels) cur-var-nums)))
              (when (or (= (length cur-var-nums) 3)
                        (and (= (length cur-var-nums) 4)
                             (= (third (send self :current-variables)) 
                                (- numvars 1))))
                    (when (= (length cur-var-nums) 4)
                          (setf cur-var-nums (select cur-var-nums '(0 1 2)))
                          (setf cur-var-names (select cur-var-names '(0 1 2)))
                          )
                    (apply #'send self  :current-variables cur-var-nums)
                    (send self :set-variables-with-labels cur-var-nums
                          cur-var-names)
                    (send self :transformation nil :draw nil)
                    (send self :add-box)
                    (when (matrixp (send self :slot-value 'rotation-type))
                          (send self :slot-value 'rotation-type 'yawing))
                    (send self :redraw)
                    ))))
    
    (defmeth q-plot :update-plotcell (i j args)
      (when (and (= i 0) (= j 0))
            (let ((my-args (first (second args)))
                  )
              (send self :show-new-var "Y" (first (last my-args))))))
      
    (defmeth box-plot :update-plotcell (i j args)
      (when (and (= i 0) (= j 0));when coming from scatmat
      (let* ((var-nums (remove-duplicates (first args)))
             (num-vars (length var-nums))
             (my-args (second args))
             (var-labs (remove-duplicates (first my-args) :test 'equal))
             (select-order
              (combine (sort-and-permute 
                        var-nums (matrix (list num-vars 1) 
                                         (iseq num-vars)))))
             (var-labs (select var-labs select-order))
             (data (select (second my-args) select-order)))
        (when (= (length var-labs) 1)
              (setf data (list (first data))))
        (send self :new-plot data :variable-labels var-labs
              :point-labels point-labels)))
      (when (and (= i 1) (= j 2));when coming from name-list
            (send self :propagate-selection (first args)))
      (send self :point-color (iseq (send self :num-points)) 'blue)
      (send self :redraw)
      )

    (defmeth histogram :update-plotcell (i j args)
;(format t "~d~%"args)
      (when (and (= i 0) (= j 0));when coming from scatmat
            (let ((histo-var (- (send self :num-variables) 1))
                  (my-args (first args)))
              (send self :start-buffering)
              (send self :clear-lines)
              (send self :content-variables (first (last my-args)) histo-var)
              (send self :adjust-to-data)
              (when (send self :show-kernel) 
                    (send self :add-kernel (send self :kernel-type) ))
              (when (send self :show-normal) (send self :add-normal))
              (send self :buffer-to-screen)
              )))

    (when scatmat
          (send scatmat :linked t)
          (send scatmat :use-color t)
          (send scatmat :point-color (iseq nobs) 'blue)
          (send scatmat :add-mouse-mode 'focus-on-variables
                :title "Focus On Variables"
                :click :do-new-variable-focus
                :cursor 'finger)
          ; following doesn't work in xls 3.44. works in 3.50.
          (send scatmat :plot-buttons :new-x nil :new-y nil)
          (send scatmat :mouse-mode 'focus-on-variables))

    (send spin-plot :scale-type 'centroid-variable)
    (send spin-plot :linked t)
    (send spin-plot :depth-cuing nil)
    (send spin-plot   :point-symbol
          (iseq (send spin-plot :num-points)) 'square)
    (send spin-plot :use-color t)
    (send spin-plot :point-color (iseq nobs) 'blue)
    (send spin-plot :showing-labels t)
    (send spin-plot :mouse-mode 'hand-rotate)
    (send spin-plot :switch-add-box)
    (send spin-plot :set-variables-with-labels '(0 1 2)
          (select (send spin-plot :variable-labels) '(0 1 2))) 
    (send spin-plot :scaled-range 
             (iseq (send spin-plot :num-variables)) -2 2)

    (if (> (send spin-plot :num-variables) 3)
        (send spin-plot :plot-buttons :margin nil :box t :new-z t)
        (send spin-plot :plot-buttons :margin nil :box t :new-x nil :new-y nil))

    (send q-plot :plot-buttons :mouse-mode nil)
  
    (when var-list
          (defmeth var-list :do-select-click (x y m1 m2)
            (when (not (send self :has-slot 'old-var-list))
                  (send self :add-slot 'old-var-list)
                  (defmeth self :old-var-list (&optional (avar-list nil set))
                    (if set (setf (slot-value 'old-var-list) avar-list))
                    (slot-value 'old-var-list)))
            (call-next-method x y m1 m2)
            (let* ((cur-var  (send self :selection))
                   (old-var (send self :old-var-list))
                   (nvar nil) (variable-labels nil)
                   (var-labs nil) (cur-data nil) )

              (when cur-var
                    (when (and m1 (send self :old-var-list))
                          (mapcar 
                           #'(lambda (i) 
                               (setf cur-var 
                                     (remove (select old-var i) cur-var)))
                           (iseq (length old-var)))
                          (setf cur-var (combine old-var cur-var )) 
                          )
                    (send self :old-var-list cur-var)
                    (setf nvar (send self :num-points))
                    (setf variable-labels 
                          (send self :point-label (iseq nvar)))
                    (setf var-labs (select variable-labels cur-var))
                    (setf cur-data 
                          (map-elements #'send current-data 
                                        :variable var-labs))
                    (send sp :update-spreadplot 0 0  cur-var 
                          (list var-labs cur-data))) )) )

    (send box-plot :linked t)
    (send box-plot :showing-labels t)
    (send box-plot :connect-points t)
    (send box-plot :mouse-mode 'brushing) 
    (send box-plot :use-color t)
    (send box-plot :point-color (iseq (send box-plot :num-points)) 'blue)
    (send box-plot :redraw)

    (send histogram :plot-buttons :new-y nil :bins t :density t)
    (send histogram :redraw)
    (send histogram :linked t)
    (send histogram :use-color t)
    (send histogram :mouse-mode 'brushing)
    (send histogram :point-color (iseq nobs) 'blue)
    
    (send obs-list :use-color t)
    (send obs-list :point-color (iseq nobs) 'blue)
    (send obs-list :linked t)
    (send (send obs-list :menu) :title "Obs")
    (send obs-list :fix-name-list)
    ;(send obs-list :has-h-scroll (max (screen-size)))
    ;(send obs-list :has-v-scroll (max (screen-size)))
    
    (send q-plot :redraw)
    (send spin-plot :redraw)
    (send obs-list :redraw)
    (send histogram :redraw)
    (when var-list
          (send (send var-list :menu) :title "Vars")
          (send var-list :fix-name-list)
          ;(send var-list :has-h-scroll (max (screen-size)))
          ;(send var-list :has-v-scroll (max (screen-size)))
          (send var-list :use-color t)
          (send var-list :point-color (iseq nvar) 'red)
          (send obs-list :linked t)
          (send var-list :redraw))
    (when scatmat 
          (send scatmat :redraw))
    (send sp :show-spreadplot)
    ))

(defmeth mv-data-object-proto :visualize-two-variables 
  (vars variable-labels point-labels)
  (let* ((varx (select vars 0))
         (vary (select vars 1))
         (sp (spread-plot 
              (matrix '(2 3) 
                      (list
                       (plot-points
                        varx vary 
                        :point-labels point-labels
                        :variable-labels variable-labels
                        :title "Scatterplot" :menu-title "Scatter"
                        :show nil)
                       (quantile-plot 
                        (list varx vary) 
                        :reg-line t 
                        :variable-labels variable-labels
                        :point-labels point-labels 
                        :show nil)
                       (quantile-quantile-plot 
                        varx vary
                        :reg-line t
                        :variable-labels variable-labels
                        :point-labels point-labels
                        :show nil)
                       (boxplot 
                        (list varx vary) 
                        :variable-labels variable-labels
                        :point-labels point-labels
                        :equate t
                        :connect-points t
                        :show nil)
                       (histogram 
                        (list varx vary) 
                        :variable-labels variable-labels
                        :title "Histogram" :menu-title "Histo"
                        :show nil)
                       (name-list point-labels :show nil)
                       
                       ))))
         (plot-matrix (send sp :plot-matrix))
         (scatterplot (select plot-matrix 0 0))
         (q-plot (select plot-matrix 0 1))
         (qq-plot (select plot-matrix 0 2))
         (box-plot (select plot-matrix 1 0))
         (histogram (select plot-matrix 1 1))
         (obs-list (select plot-matrix 1 2))
         (nobs (send self :nobs))
         )
    
    (defmeth sp :spreadplot-help ()
      (plot-help-window (strcat "SpreadPlot Help"))
      (paste-plot-help (format nil "This is the SpreadPlot for Bivariate Data. In this SpreadPlot the windows are only linked by the data's observations.~2%"))
      (call-next-method :points t :bars t :labels t :flush nil))

    (send box-plot :linked t)
    (send box-plot :showing-labels t)
    (send box-plot :use-color t)
    (send box-plot :point-color (iseq (send box-plot :num-points)) 'blue)
    (send box-plot :mouse-mode 'brushing)
    (send box-plot :connect-points t)

    (send obs-list :linked t)
    (send obs-list :title "Observations")
    (send obs-list :new-menu "Obs" 
          :items '(LINK MOUSE DASH ERASE-SELECTION FOCUS-ON-SELECTION 
                        SHOW-ALL COLOR SELECTION))
    (send obs-list :fix-name-list)
    ;(send obs-list :has-h-scroll (max (screen-size)))
    ;(send obs-list :has-v-scroll (max (screen-size)))
    (send obs-list :use-color t)
    (send obs-list :point-color (iseq nobs) 'blue)

    (send scatterplot :scale-type 'variable)
    (send scatterplot :plot-buttons :new-x nil :new-y nil)
    (send scatterplot :title "ScatterPlot")
    (send scatterplot :menu-title "Scatter")
    (send scatterplot :showing-labels t)
    (send scatterplot :linked t)
    (send scatterplot :x-axis t t 5)
    (send scatterplot :y-axis t t 5)
    (send scatterplot :use-color t)
    (send scatterplot :point-color (iseq nobs) 'blue)
    (send scatterplot :mouse-mode 'brushing)

    (send q-plot    :plot-buttons :mouse-mode nil)
    (send qq-plot   :plot-buttons :new-x nil :new-y nil :mouse-mode nil)
    (send histogram :linked t)
    (send histogram :plot-buttons :new-y nil :bins t :density t)
    (send histogram :use-color t)
    (send histogram :point-color (iseq nobs) 'blue)
    (send histogram :mouse-mode 'brushing)

    (send box-plot :redraw)
    (send q-plot :redraw)
    (send qq-plot :redraw)
    (send obs-list :redraw)
    (send histogram :redraw)
    (send scatterplot :redraw)
    (send sp :show-spreadplot)
    ))

(defmeth mv-data-object-proto :visualize-one-variable 
  (var variable-labels point-labels)
  (let* ((sp (spread-plot 
              (matrix '(2 2) 
                      (list
                       (boxplot var 
                                :variable-labels (list variable-labels)
                                :point-labels point-labels
                                :show nil)
                       (quantile-plot var 
                                :reg-line t 
                                :variable-label variable-labels
                                :point-labels point-labels 
                                :show nil)
                       (histogram var 
                                :variable-labels (list variable-labels) 
                                :show nil)
                       (name-list point-labels 
                                  :show nil :title "Observations")
                       ))))
         (plot-matrix (send sp :plot-matrix))
         (box-plot (select plot-matrix 0 0))
         (q-plot (select plot-matrix 0 1))
         (histogram (select plot-matrix 1 0))
         (obs-list (select plot-matrix 1 1))
         (nobs (send self :nobs))
         )
    (defmeth sp :spreadplot-help ()
      (plot-help-window (strcat "SpreadPlot Help"))
      (paste-plot-help (format nil "This is the SpreadPlot for Univariate Data. In this SpreadPlot the windows are only linked by the data's observations.~2%"))
      (call-next-method :points t :bars t :labels t :flush nil))
    (send box-plot :linked t)
    (send box-plot :showing-labels t)
    (send box-plot :use-color t)
    (send box-plot :point-color (iseq (send box-plot :num-points)) 'blue)
    (send box-plot :mouse-mode 'brushing)
    
    (send obs-list :linked t)
    (send obs-list :new-menu "Obs" 
          :items '(LINK MOUSE DASH ERASE-SELECTION FOCUS-ON-SELECTION 
                        SHOW-ALL COLOR SELECTION))
    (send obs-list :fix-name-list)
    ;(send obs-list :has-h-scroll (max (screen-size)))
    ;(send obs-list :has-v-scroll (max (screen-size)))
    (send obs-list :use-color t)
    (send obs-list :point-color (iseq nobs) 'blue)
    (send q-plot :margin 0 17 0 0)
    (send q-plot :plot-buttons :mouse-mode nil :new-y nil)
    (send histogram :linked t)
    (send histogram :use-color t)
    (send histogram :point-color (iseq nobs) 'blue)
    (send histogram :plot-buttons :new-x nil :new-y nil 
                :density t :bins t)
    (send histogram :mouse-mode 'brushing)

    (send box-plot :redraw)
    (send q-plot :redraw)
    (send histogram :redraw)
    (send obs-list :redraw)
    (send sp :show-spreadplot)
    
    ))