;;##########################################################################
;; dataobj1.lsp
;; Copyright (c) 1991-98 by Forrest W. Young
;; The entire code for implementing data objects is contained in files 
;; dataobj1.lsp, dataobj2.lsp, dataobj3.lsp, dataobj4.lsp, 
;; tableobj.lsp and dissobj.lsp.
;; This file contains functions to load, define and set current data objects,
;; and code to define the object prototype and it's slot-accessor methods.
;; Dataobj2.lsp contains code for menu and other methods.
;; Dataobj3.lsp contains methods for reporting simple univariate statistics.
;; Dataobj4.lsp contains methods to generate and visualize simulated mv data.
;; Tableobj.lsp contains code to implement table data objects.
;; Dissobj.lsp  contains code to implement dissimilarity data objects.
;;##########################################################################

;;--------------------------------------------------------------------------
;;function to load data objects 
;;--------------------------------------------------------------------------

(defun load-data (&optional file)
"Args: (&optional file)
Loads a data object contained in a file.  The file's name must end with .lsp.
If the optional string argument FILE is included, the data object is loaded 
from FILE, otherwise a dialog is presented to select the file. The string 
need not end with .lsp.  Returns the object-id of the data object."
  (set-working-directory *data-dir-name*)
  (when (not file) 
        (setf file (open-file-dialog t)))
  (when (send *workmap* :gui) (send *workmap* :redraw))
  (when *guidemap* (when (send *guidemap* :gui) (send *guidemap* :redraw)))
  (when file
        (setf *data-dir-name* (get-working-directory))
        ;(send *workmap* :start-buffering)
        (let ((object (send *workmap* :load-object file))
              (previous-previous-data previous-data)
              )
          (when (not (objectp object))
                     (error-message "File does not contain a data object.")
                     (setf current-data previous-data)
                     (setf previous-data previous-previous-data)
                     (setf object nil))
          (send *workmap* :redraw)
          ;(send *workmap* :buffer-to-screen)
          object)))

;;---------------------------------------------------------------------------
;;constructor function for making data-objects
;;---------------------------------------------------------------------------

(defun data (name &key data (title nil) (variables nil) (types nil)
       (labels nil) (matrices nil) (shapes nil) (element-labels nil)
       (ways nil) (classes nil) (created nil) (creator-object nil) 
       (about nil) (datasheet-arguments nil))
"Args: NAME &KEY DATA TITLE ABOUT VARIABLES TYPES LABELS MATRICES SHAPES WAYS CLASSES CREATED CREATOR-OBJECT ELEMENT-LABELS DATASHEET-ARGUMENTS
Defines ViSta data object named NAME. DATA is required, and must be a list of numbers, strings or symbols (symbols are converted to uppercase strings). The number of elements must conform to the information in other arguments. TITLE is an optional string titleing the data. ABOUT is an optional string of information about the data. VARIABLES is required, and must be a list of strings defining variable names (and, indirectly, the number of variables). TYPES, optional, must be a list of strings Numeric, Ordinal or Category (case ignored), specifying whether the variables are numeric, ordinal or categorical (all Numeric by default). LABELS, optional, must be a list of strings specifying observation names (Obs1, Obs2, etc., by default). MATRIX, required for matrix data only, must be a list of strings specifying matrix names (and, indirectly, the number of matrices). SHAPES, optional for matrix data only, must be a list of strings  Symmetric or Asymmetric (case ignored), specifying the shape of each matrix (all are Symmetric by default). WAYS, required for table data only, must be a list of strings specifying the names (and, indirectly, the number) of ways in table data. CLASSES, required for table data only, must be a list of lists of strings specifying the names (and, indirectly, the number) of classes for each way of the table data. The remaining arguments are internal system arguments that users won't need to use."
 (let* ((object nil)
        (previous-hilited-icon nil)
        (new-icon nil)
        (redraw-order (send *workmap* :redraw-order))
        (object-in current-object)
        (xy nil)
        (icon-type nil))
;fwy
;#+macintosh(send *obs-window* :clear) ;hide-window
;#+macintosh(send *var-window* :clear) ;hide-window
   (send *workmap* :data-in-construction t)
   (when redraw-order 
         (setf previous-hilited-icon (first (last redraw-order)))
         (setf previous-hilited-icon 
               (select (send *workmap* :icon-list) previous-hilited-icon)))
;(break)
   (when (not previous-data) (send *workmap* :initialize-data-menu))
   (setf name (blanks-to-dashes name))
   (if (not title) (setf title name)) 
   (dotimes (i (length data))
           (when (symbolp (select data i))
                 (when (not (equal (select data i) nil))
                       (setf (select data i) (string (select data i))))))
   (cond
     ((or ways classes)
      (setf icon-type 5)
       (setf object 
           (table-data data variables title labels types name ways classes))
      )
     
     ( matrices
       (setf icon-type 4)
        (setf object (matrix-data data variables title labels types 
                                  matrices shapes element-labels name)))
     ( t
       (setf icon-type 1)
       (setf object (send mv-data-object-proto 
                          :new data variables title labels types name))))
   (when object

         (if (not about) 
             (setf about (format nil "There is no information about these data. To enter information type:~%  (about-these-data \"INFORMATION\")~%in the listener window when the data icon is highlighted. Replace INFORMATION with the information about the data that you wish to enter."))) 
         (send object :about about)
         (send object :datasheet-arguments datasheet-arguments)
         (cond
           ((and (not created))
            (setf xy (send *workmap* :locate-new-icon))
            (send *workmap* :add-icon 
                  *workmap* (select xy 0) (select xy 1) name icon-type))
           (created 
            (when (not creator-object) (setf creator-object object-in))
            (send object :add-parent creator-object)
            (send creator-object :add-child object)
            (send *workmap* :add-connected-icon created name icon-type)))
         (when (send *vista* :expert)
               (setf created (send *expertmap* :selected-icon)))
         (send object :icon-number (send *workmap* :num-icons))
         (send (select (send *workmap* :icon-list) 
                       (1- (send *workmap* :num-icons))) :object object)
         (when (send *vista* :long-menus)
               (send *data-menu* :append-items 
                     (send menu-item-proto :new name :mark t
                           :action #'(lambda () (setcd object)))))
         (when (not object-in) (setf object-in object))
         (when (not (send *vista* :long-menus))
               (send object :menu-length 
                     (+ (send object :menu-length) 
                        (1- (length (send *workmap*
                                          :data-icon-number-list))))))
;(format t "~D~%"(list "GuideMap" (send *vista* :guidemap)))
         (cond
           ((not (send *vista* :guidemap))
;(format t "~D~%"(list "DATA calling SETCD"))
            (setcd object))
           ((send *vista* :guidemap)
            (cond
              ((or (equal (send object-in :slot-value 'proto-name)
                          (send mv-data-object-proto :slot-value 'proto-name))
                   (equal (send object-in :slot-value 'proto-name)
                          (send table-data-object-proto :slot-value 
                                'proto-name))
                   (equal (send object-in :slot-value 'proto-name)
                          (send diss-data-object-proto :slot-value 
                                'proto-name)))
               (setcd object-in));when object-in is data-object do setcd
              (t (when investigate (format t 
                   "DATA function assuming input object is model object.~%"))
                 (setcm object-in)))
            (when created (send *guidemap* :created-data))))
         (set (intern (string-upcase name)) object))
   (send *workmap* :data-in-construction nil)
;(format t "DOB: 1~%")
;(break)
   ;(send *workmap* :select-icon (send object :icon-number))
   (setf new-icon (first (last (send *workmap* :icon-list))))
   (when previous-hilited-icon 
         (send previous-hilited-icon :show-icon "normal"))
   (send new-icon :show-icon "selected")

   (when (send *workmap* :gui) (send *workmap* :redraw))
   (when (send *vista* :obs-showing) (send *obs-window* :redraw))
   (when (send *vista* :vars-showing) (send *var-window* :redraw))
;(format t "DOB: 2~%")
   ; (when (send *vista* :obs-showing) (send *obs-window* :show-window))
   ; (when (send *vista* :vars-showing) (send *var-window* :show-window))
   ;(set (intern (string-upcase name)) object)
   object ))

(defun table-data (data variables title labels types name ways classes)
  (load (strcat *vista-dir-name* "tablobj1"))
  (load (strcat *vista-dir-name* "tablobj2"))
  (table-data data variables title labels types name ways classes))
;fwy changed following to above 23/21/97
;  (send table-data-object-proto :new
;       data variables title labels types name ways classes)

(defun matrix-data (data variables title labels types matrices shapes
                         element-labels name)
  (load (strcat *vista-dir-name* "dissobj"))
  (matrix-data 
   data variables title labels types matrices shapes element-labels name))
;fwy changed following to above 23/21/97
;  (send diss-data-object-proto :new
;      data variables title labels types matrices shapes element-labels name)
;;--------------------------------------------------------------------------
;;function to change the current data
;;--------------------------------------------------------------------------

(defun setcd (object &optional set-menus-only) 
"Arg: (data-object-name &optional set-menus-only)
Sets the current data-object to DATA-OBJECT-NAME and changes system 
states accordingly.  Returns object's identification information. Only sets menu states if set-menus-only is t."
  (let* ((menu-length (send object :menu-length))
         (current-item-number menu-length)
         (current-icon (select (send *workmap* :data-icon-number-list)
                               (- current-item-number 
                                  (send *workmap* :num-data-menu-items))))
         (ds-obj (send object :datasheet-object))
         (ds-open (current-datasheet-open)))
    (when (not set-menus-only)
          (send *workmap* :previously-selected-data-icon 
                (send *workmap* :selected-data-icon))
          (setf previous-data current-data)
          (setf current-data object)
          (setf current-object object)
;(format t "~D~%" (list "1: current icon" current-icon "selected icon" 
;(send *workmap* :selected-icon)))
          (when (/= current-icon (send *workmap* :selected-icon))
                (send *workmap* :select-icon current-icon))
;(format t "~D~%" (list "2: current icon" current-icon "selected icon" 
;(send *workmap* :selected-icon)))
          (send *workmap* :selected-data-icon current-icon)
          (send *vista*   :show-labels)
          (when (and (send current-data :matrices) 
                     (send current-data :mat-window))
                (send current-data :list-matrices))
          (send *workmap* :no-menu-marks *data-menu*))
    (set-menu-states object menu-length current-item-number 
                     current-icon ds-obj ds-open)
;fwy4.28 7/23/97 modified next statement for applets
    (when (not set-menus-only)
          (when (and (not (send *vista* :applets)) (send *vista* :guidemap))
                (when investigate 
                      (format t "Calling Guidance from SETCData."))
                (guidance "data"))
          (when ds-obj 
                (cond ((and (send current-data :datasheet-open)
                            (send ds-obj :editable)) 
                       (send ds-obj :enable-vista-menus&tools nil)
                       (send object :set-menu&tool-states "Disabled")
                       (send ds-obj :show-window)
                       )
                  (t (send ds-obj :enable-vista-menus&tools t))))
          (setf *datasheet* (send current-data :datasheet-object))
          (setf *current-data* current-data)
          (setf *current-object* current-data) 
          (send delete-data-menu-item :enabled nil)
          current-data)))
    
  (defun set-menu-states 
    (object menu-length current-item-number current-icon ds-obj ds-open)
      (send create-dob-data-menu-item :enabled t)
    (send transform-data-popup-menu-item :enabled t)
    (when (not (current-datasheet-open))
          (send *data-menu* :enabled t)
          (send *analyze-menu* :enabled t)
          (send *model-menu* :enabled t))
    (when (send *vista* :long-menus)
      (send (select (send *data-menu* :items) current-item-number) :mark t))
    (cond 
      ((or (send current-data :matrices) (send current-data :ways))
       (send merge-vars-menu-item :enabled nil)
       (send merge-obs-menu-item :enabled nil)
       (send merge-vars-popup-menu-item :enabled nil)
       (send merge-obs-popup-menu-item :enabled nil)
       (send *workmap* :enabled-trans-menu nil)
       (when (send current-data :ways)
             (when (not ds-open) (send object :set-menu&tool-states "Table"))
             (send anova-model-menu-item :enabled t)
             (send univar-model-menu-item :enabled nil)
             (send merge-mats-menu-item :enabled nil)
             (send merge-mats-popup-menu-item :enabled nil)
             (send mds-model-menu-item :enabled nil)
             (send visualize-data-menu-item :enabled t)
             (send *vista* :show-cells))
       (when (send current-data :matrices)
             (when (not ds-open)(send object :set-menu&tool-states "Matrix"))
             (send anova-model-menu-item :enabled nil)
             (send univar-model-menu-item :enabled nil)
             (send mds-model-menu-item :enabled t)
             (send visualize-data-menu-item :enabled nil)
             (when previous-data (send merge-mats-menu-item :enabled t)
                   (send merge-mats-popup-menu-item :enabled t))
             (send *vista* :show-mats))
       (send transform-data-popup-menu-item :enabled nil)
       (send corresp-model-menu-item :enabled nil)
       (send prin-model-menu-item :enabled nil)
       (send mulreg-model-menu-item :enabled nil)
       (send reg-model-menu-item :enabled nil))
      (t 
       (when previous-data 
             (send merge-vars-menu-item :enabled t)
             (send merge-vars-popup-menu-item :enabled t)
             (send merge-obs-menu-item :enabled t)
             (send merge-obs-popup-menu-item :enabled t))
       (send *vista*   :show-obs)
       (when (not ds-open) (send object :set-menu&tool-states "MV"))
       (send *workmap* :enabled-trans-menu t)
       (send anova-model-menu-item   :enabled t)
       (send corresp-model-menu-item :enabled t)
       (send prin-model-menu-item    :enabled t)
       (send reg-model-menu-item     :enabled t)
       (send mulreg-model-menu-item  :enabled t)
       (send univar-model-menu-item  :enabled t)
       (send mds-model-menu-item     :enabled nil)
       (send merge-mats-menu-item    :enabled nil)
       (send visualize-data-menu-item :enabled t)
       (send merge-mats-popup-menu-item :enabled nil)
       )))

(defun current-datasheet-open ()
  (when current-data
        (let ((ds-obj (send current-data :datasheet-object))
              )
          (when ds-obj
                (and (send current-data :datasheet-open)
                     (send ds-obj :editable))))))

(setf *current-data* nil)

;;##########################################################################
;;define prototype multivariate data-object and it's isnew method
;;##########################################################################

(defproto mv-data-object-proto 
  '(data title about vnames onames mnames vtypes vmenu nvar nobs var-window 
    obs-window var-window-object obs-window-object var-states obs-states
    summary-options visualize-options menu-length icon-number ways
    dob-parents dob-children needs-computing auto-compute 
    guidemap-number guidemap-ancestors datasheet-object datasheet-open
    spreadplot-object simulate-parameters datasheet-arguments))

(defmeth mv-data-object-proto :isnew 
                  (data variables title labels types name)
  (when (send self :initialize-object data variables title labels types name)
        (send self :data data)))

(defmeth mv-data-object-proto :initialize-object
  (data variables title labels types name)
"Message args: (data variables title labels types name)
Used by isnew methods of multivariate and dissimilarity data 
objects to initialize shared aspects. Returns t if data pass 
size and shape tests, nil otherwise."
  (let* ((nvar (length variables))
         (nobs (/ (length data) nvar))
         (remainder (rem nobs 1)))
    (when (and (equal remainder 0) (> nvar 0) (> nobs 0))
          (send self :title title)
          (send self :name name)   
          (send self :nobs nobs)
          (send self :nvar nvar)
          (send self :variables variables)
          (send self :var-window nil)
          (send self :obs-window nil)
          (send self :obs-states (repeat 'NORMAL nobs))
          (send self :var-states (repeat 'NORMAL nvar))
          (send self :summary-option-states '((0 1)))
          (send self :menu-length (length (send (eval *data-menu*) :items)))
          (if types (send self :types types)
              (send self :types 
                (mapcar #'(lambda (x) 
                            (format nil "Numeric" x)) (iseq nvar))))
          (send self :labels
                (if labels labels
                 (mapcar #'(lambda (x) 
                             (format nil "Obs~a" x)) (iseq nobs)))))
    (when (not (equal remainder 0))
          (error "Data are not properly shaped or are incomplete."))
    (when (< nvar 1) 
          (error "The data must have at least 1 variable."))
    (when (< nobs 1) 
          (error "The data must have at least 1 observation."))
    (and (= 0 remainder) (> nvar 0) (> nobs 0))
    ))

;;--------------------------------------------------------------------------
;;define slot-accessor methods for the multivariate data-object
;;--------------------------------------------------------------------------

(defmeth mv-data-object-proto :data (&optional (data nil set))
"Message args: (&optional data-list)
 Sets or retrieves the multivariate data as a list."
  (if set (setf (slot-value 'data) data))
  (slot-value 'data))

(defmeth mv-data-object-proto :title (&optional (title nil set))
"Message args: (&optional title)
 Sets or retrieves the title of the multivariate data."
  (if set (setf (slot-value 'title) title))
  (slot-value 'title))

(defmeth mv-data-object-proto :about (&optional (about nil set))
"Message args: (&optional about)
 Sets or retrieves the about information of the data."
  (if set (setf (slot-value 'about) about))
  (slot-value 'about))

(defmeth mv-data-object-proto :nobs (&optional (number nil set))
"Message args: (&optional number)
 Sets or retrieves the number of observations in the multivariate data."
  (if set (setf (slot-value 'nobs) number))
  (slot-value 'nobs))

(defmeth mv-data-object-proto :nvar (&optional (number nil set))
"Message args: (&optional number)
 Sets or retrieves the number of variables in the multivariate data."
  (if set (setf (slot-value 'nvar) number))
  (slot-value 'nvar))

(defmeth mv-data-object-proto :matrices  (&optional (names nil set))
"Message args: (&optional strings)
 Sets or retrieves the names of matrices in dissimilarity data."
  (if set (setf (slot-value 'mnames) names))
  (slot-value 'mnames))

(defmeth mv-data-object-proto :active-matrices (ok-types)
"Message args: (&optional strings)
Sets or retrieves the names of active ok-type matrices in dissimilarity data.
An active matrix is one which is selected in the mat-window, or if none 
selected, which is visible in the window.  Ok-types must be one of the 
following strings: all, symmetric, asymmetric, rectangular."
  (select (send self :matrices) (current-matrices ok-types)))

(defmeth mv-data-object-proto :var-window (&optional (logical nil set))
"Message args: (&optional logical)
 Sets or retrieves a logical value indicating if the var-window is open."
  (if set (setf (slot-value 'var-window) logical))
  (slot-value 'var-window))

(defmeth mv-data-object-proto :obs-window (&optional (logical nil set))
"Message args: (&optional logical)
 Sets or retrieves a logical value indicating if the obs-window is open."
  (if set (setf (slot-value 'obs-window) logical))
  (slot-value 'obs-window))

(defmeth mv-data-object-proto :var-window-object (&optional (object nil set))
"Message args: (&optional logical)
 Sets or retrieves the var-window object identification information."
  (if set (setf (slot-value 'var-window-object) object))
  (slot-value 'var-window-object))

(defmeth mv-data-object-proto :obs-window-object (&optional (object nil set))
"Message args: (&optional logical)
 Sets or retrieves the obs-window object identification information." 
  (if set (setf (slot-value 'obs-window-object) object))
  (slot-value 'obs-window-object))

(defmeth mv-data-object-proto :var-states (&optional (list nil set))
"Message args: (&optional list)
 Sets or retrieves the selection states of the variables in the data object.
 The states mimic point selection states (normal, selected, invisible)." 
  (if set (setf (slot-value 'var-states) list))
  (slot-value 'var-states))

(defmeth mv-data-object-proto :obs-states (&optional (list nil set))
"Message args: (&optional list)
 Sets or retrieves the selection states of the observations in the data object.
 The states mimic point selection states (normal, selected, invisible)." 
  (if set (setf (slot-value 'obs-states) list))
  (slot-value 'obs-states))

(defmeth mv-data-object-proto :summary-option-states (&optional (list nil set))
"Message args: (&optional list)
 Sets or retrieves the states of the data-object statistical summary options." 
  (if set (setf (slot-value 'summary-options) list))
  (slot-value 'summary-options))

(defmeth mv-data-object-proto :visualize-option-states (&optional (list nil set))
"Message args: (&optional list)
 Sets or retrieves the visualization options of the data object." 
  (if set (setf (slot-value 'visualize-options) list))
  (slot-value 'visualize-options))

(defmeth mv-data-object-proto :active-nvar (ok-types)
"Message args: (ok-types)
Reports the number of the ok-types variables that are active.  
An active variable is one which is selected in the var-window, or if none 
selected, which is visible in the window.  Ok-types must be one of the 
following strings: all, numeric, ordinal, category, label."
  (length (send self :current-variables ok-types)))

(defmeth mv-data-object-proto :active-nobs ()
"Message args: none
Reports the number of active observations. An active observation is one which is selected in the var-window, or if none selected, which is visible in the window when it is displayed."
  (length (send self :current-labels)))

(defmeth mv-data-object-proto :variables (&optional (names nil set))
"Message args: (&optional strings)
Sets or retrieves the names of the variables."
  (if set (setf (slot-value 'vnames) names))
  (slot-value 'vnames))

(defmeth mv-data-object-proto :variable (name)
"Message args: (variable-name)
Returns a vector of values for variable-name, or nil if variable-name is not
a variable in the data object."
    (let* ((var-names (send self :variables))
           (which (which ($= name var-names)))
          )
      (if which
          (select (select (column-list (send self :data-matrix)) which) 0)
          which)))

(defmeth mv-data-object-proto :active-variables (ok-types)
"Message args: (ok-types)
Reports the names of the ok-types variables which are active.  
An active variable is one which is selected in the var-window, or if none 
selected, which is visible in the window.  Ok-types must be one of the 
following strings: all, numeric, ordinal, category, label."
  (select (send self :variables) 
          (send self :current-variables ok-types)))

(defmeth mv-data-object-proto :labels (&optional (names nil set))
"Message args: (&optional strings)
Sets or retrieves the labels (names) of the observations."
  (if set (setf (slot-value 'onames) names))
  (slot-value 'onames))

(defmeth mv-data-object-proto :active-labels ()
"Message args: none
Reports the labels (names) of the active observations.  Active observations are those which are selected in the obs-window, or if none are selected, which are visible in the window."
  (select (send self :labels) (send self :current-labels)))

(defmeth mv-data-object-proto :types  (&optional (names nil set))
"Message args: (&optional strings)
Sets or retrieves the types of the variables."
  (if set (setf (slot-value 'vtypes) names))
  (slot-value 'vtypes))

(defmeth mv-data-object-proto :active-types (ok-types)
"Message args: (ok-types)
Reports the types of the ok-types variables which are active.  
An active variable is one which is selected in the var-window, or if none 
selected, which is visible in the window.  Ok-types must be one of the 
following strings: all, numeric, ordinal, category, label."
  (select (send self :types) 
          (send self :current-variables ok-types)))

(defmeth mv-data-object-proto :name (&optional (names nil set))
"Message args: (&optional string)
Sets or retrieves the name of the data object."
  (if set (setf (slot-value 'vmenu)  names))
  (slot-value 'vmenu))

(defmeth mv-data-object-proto :menu-length (&optional (number nil set))
"Message args: (&optional number)
 Sets or retrieves the position of the object name in the menu."
  (if set (setf (slot-value 'menu-length) number))
  (slot-value 'menu-length))

(defmeth mv-data-object-proto :icon-number (&optional (number nil set))
"Message args: (&optional number)
 Sets or retrieves the icon number (creation order) of the object."
  (if set (setf (slot-value 'icon-number) number))
  (slot-value 'icon-number))

(defmeth mv-data-object-proto :active-data-matrix (ok-types)
"Message args: (ok-types)
Reports, for ok-types variables which are active, the data in matrix form.
An active variable is one which is selected in the var-window, or if none 
selected, which is visible in the window.  Ok-types must be one of the 
following strings: all, numeric, ordinal, category, label."
  (select (send self :data-matrix) (send self :current-labels)
          (send self :current-variables ok-types)))

(defmeth mv-data-object-proto :active-data (ok-types)
"Message args: (ok-types)
Reports, for ok-types variables which are active, the data in list form.
An active variable is one which is selected in the var-window, or if none 
selected, which is visible in the window.  Ok-types must be one of the 
following strings: all, numeric, ordinal, category, label."
  (if (send self :ways) 
      (send self :data) ; for table data
      (combine (send self :active-data-matrix ok-types)) ; for mv data
      ))

(defmeth mv-data-object-proto :data-matrix ()
"Message args: none
Returns the data as a matrix."
  (let* ((n (send self :nvar))
         (m (send self :nobs)))
         (matrix (list m n) (send self :data))))

(defmeth mv-data-object-proto :ways (&optional (string-list nil set))
"Message args: (&optional string-list)
 Sets or retrieves the names of the ways of the table data."
  (if set (setf (slot-value 'ways) string-list))
  (slot-value 'ways))

(defmeth mv-data-object-proto :spreadplot-object 
  (&optional (objid nil set))
"Message args: (&optional objid)
 Sets or retrieves the speadplot object identification for the stat object."
  (if set (setf (slot-value 'spreadplot-object) objid))
  (slot-value 'spreadplot-object))

(defmeth mv-data-object-proto :simulate-parameters 
  (&optional (param-list nil set))
"Message args: (&optional param-list)
 Sets or retrieves the list of parameters for simulating data."
  (if set (setf (slot-value 'simulate-parameters) param-list))
  (slot-value 'simulate-parameters))
             
(provide "dataobj1")