
;;************************************************************************
;; boxplot2.lsp 
;; contains code for boxplot  
;; copyright (c) 1993-98 by Forrest W. Young
;;************************************************************************

;;BOXPLOT METHODS

(defmeth boxplot-proto :new-plot 
  (data &key title variable-labels point-labels) 
  ;(format t "BP: New Plot~%")
  (let ((n nil)
        (x nil)
        (loc nil)
        (normed-data nil)
        (jitter 0)
        (y-name (send self :y-axis-label)))
    (send self :start-buffering)
    (setq data (cond 
                 ((matrixp data) (column-list data))
                 ((or (not (listp data)) (numberp (car data))) (list data))
                 (t data)))
    (send self :data data)
    (send self :num-obs (length (first data)))
    (send self :num-var (length data))
    (when (send self :enable-equate)
          (send self :normed-data (send self :normalize data))
          (if (send self :equate)
              (setf data (send self :normed-data))
              (setf data (send self :data))))
    (send self :clear) 
    (setf n (mapcar #'length data))
    (setf x (* (mean n) (iseq (length n))))
    (send self :x x)
    (when (not variable-labels) 
          (setf variable-labels (repeat "" (length data))))
    (send self :variable-labels variable-labels)
    (let ((range (get-nice-range (min data) (max data) 4)))
      (send self :range 1 (nth 0 range) (nth 1 range))
      ; (send self :y-axis t (second (send self :y-axis)) (nth 2 range))
      (send self :y-axis t t (nth 2 range)))
    (send self :range 0 (- (first x) (/ (first n) 2))
          (+ (last  x) (/ (last  n) 2)))
    (dotimes (i (length data))
             (when (send self :jitter)
                   (setf jitter 
                         (* (- (uniform-rand (length (nth i data))) .55) .025
                            (abs (apply (function -) (send self :range 0))))))
             (send self :add-points 
                   (+ jitter (repeat (nth i x) (length (nth i data))))
                   (nth i data) :color 'blue :symbol 'square))
    (when point-labels (send self :showing-labels t))
    (when (not point-labels) 
          (setf point-labels
                (mapcar #'(lambda (i) (format nil "~d" i))
                        (iseq (send self :num-obs)))))

    (when (and (send self :enable-connect-points) 
               (/= (length (combine data)) (length point-labels)))
          (setf point-labels 
                (combine point-labels 
                         (repeat " " (- (length (combine data))
                                        (length point-labels))))))
    (send self :point-label (iseq (length point-labels)) point-labels)
    (send self :variable-label 1 
          (if (send self :equate) 
              (strcat "Normalized " y-name) y-name))
    (send self :change-plot)
    (send self :buffer-to-screen))
  )

(defmeth boxplot-proto :normalize (data)
  (column-list 
   (normalize
    (transpose (matrix (list (length data) (length (first data))) 
                       (combine data))))))

(defmeth boxplot-proto :showing-labels (&optional (logical nil set))
"Args: (&optional logical)
Sets or returns t or nil for whether point and variable labels are showing and causes them to show or not show."
  (when set (call-next-method logical)
        (send self :change-plot))
  (call-next-method))

(defmeth boxplot-proto :add-labels ()
  (let* ((data (send self :data))
         (x (send self :x))
         (n (mapcar #'length data))
         (variable-labels (send self :variable-labels))
         (w 0)
         (loc nil))
    (when (send self :showing-labels)
          (when (send self :enable-equate)
                (when (send self :equate) 
                      (setf data (send self :normed-data))))
          (dotimes (i (length data))
                   (if (or (send self :diamonds) (send self :boxes))
                       (setf w (/ (nth i n) 4))
                       (setf w 0))
                   (setf loc (send self :scaled-to-canvas 
                                   (- (nth i x) w) (mean (nth i data))))
                   (send self :draw-text-up (select variable-labels i) 
                         (- (first loc) 8) (second loc) 1 0)
                   ))))

(defmeth boxplot-proto :add-diamondplot 
  (y &key (x 1.0) (width 1.0) (draw t))
  (unless (= 2 (send self :num-variables)) (error "only works for 2D plots"))
  (let* ((half-box (* 0.5 width))
         (half-foot (* 0.1 width))
         (mean (mean y))
         (stdv (standard-deviation y))
         (low (min y))
         (p1 (- mean (if stdv stdv 0)))
         (p3 (+ mean (if stdv stdv 0)))
         (high (max y)))
    (send self :add-lines (list (- x half-box) (+ x half-box))
                          (list mean mean) :color 'green :width 2)
    (send self :add-lines (list x (- x half-box) x (+ x half-box) x)
                          (list p3 mean p1 mean p3) :color 'green :width 2)
    ))

(defmeth boxplot-proto :add-boxplot (y &key (x 1.0) (width 1.0) (draw t))
  (unless (= 2 (send self :num-variables)) (error "only works for 2D plots"))
  (let* ((half-box (* 0.4 width))
         (half-foot (* 0.30 width))
         (fiv (fivnum y))
         (low (quantile y .10))
         (q1 (select fiv 1))
         (med (select fiv 2))
         (q3 (select fiv 3))
         (high (quantile y .90))
         (num-lines-b4 (send self :num-lines))
         (lwidth 2)
         )
    (send self :add-lines (list x x) (list q3 high) :color 'red :width lwidth)
    (send self :add-lines (list x x) (list q1 low) :color 'red :width lwidth)
    (send self :add-lines (list (- x half-foot) (+ x half-foot))
                          (list low low) :color 'red :width lwidth)
    (send self :add-lines (list (- x half-box) (+ x half-box))
                          (list med med) :color 'red :width lwidth)
    (send self :add-lines (list (- x half-foot) (+ x half-foot))
                          (list high high) :color 'red :width lwidth)
    (send self :add-lines (list (- x half-box) (- x half-box) 
                                (+ x half-box) (+ x half-box) (- x half-box))
                          (list q3 q1 q1 q3 q3) :color 'red :width lwidth)
    ))

(defmeth boxplot-proto :switch-boxes ()
  (send self :boxes (not (send self :boxes)))
  (send self :change-plot))

(defmeth boxplot-proto :switch-diamonds ()
  (send self :diamonds (not (send self :diamonds)))
  (send self :change-plot))

(defmeth boxplot-proto :switch-mean-line ()
  (send self :mean-line (not (send self :mean-line)))
  (send self :change-plot))

(defmeth boxplot-proto :switch-median-line ()
  (send self :median-line (not (send self :median-line)))
  (send self :change-plot))

(defmeth boxplot-proto :switch-connect-points ()
  (send self :connect-points (not (send self :connect-points)))
  (send self :change-plot))

(defmeth boxplot-proto :switch-equate ()
  (send self :equate (not (send self :equate)))
  (let ((point-labels
         (send self :point-label (iseq (send self :num-points)))))
    (send self :new-plot 
          (send self :data)
          :title (send self :title)
          :point-labels point-labels
          :variable-labels (send self :variable-labels))))

(defmeth boxplot-proto :set-selection-color ()
  (call-next-method)
  (send self :adjust-screen))

(defmeth boxplot-proto :change-plot ()
  (let* ((data (if (send self :equate) 
                   (send self :normed-data) 
                   (send self :data)))
         (num-lines-b4 nil)
         (width 2)
         (n (mapcar #'length data))
         (x (send self :x)))
    (send self :start-buffering)
    (send self :clear-lines)
    (dotimes (i (length data))
             (when (send self :boxes)
                   (send self :add-boxplot (nth i data) 
                         :width (/ (nth i n) 2) :x (nth i x)))
             (when (send self :diamonds)
                   (send self :add-diamondplot (nth i data) 
                         :width (/ (nth i n) 2) :x (nth i x))))
    (when (send self :median-line)
          (setf num-lines-b4 (send self :num-lines))
          (send self :add-lines x (mapcar #'median data) :color 'red
                :width width)
          )
    (when (send self :mean-line)
          (setf num-lines-b4 (send self :num-lines))
          (send self :add-lines x (mapcar #'mean data) :color 'green
                :width width)
          )
    (send self :redraw)
    (send self :buffer-to-screen)
    ))

(defmeth boxplot-proto :plot-help (&key (flush t))
  (let ((w (plot-help-window (strcat "Help for " (send self :title))
                              :flush flush)))
    (paste-plot-help (format nil 
"The Box, Diamond and Dot plot uses boxes, diamonds and dots to show you the shape of a variable's distribution.~2%"))
    (paste-plot-help (format nil
"The plot always displays dots. They are located vertically at the value of the observations shown on the vertical scale. (The dots are 'jittered' horizontally by a small random ammount to avoid overlap).~2%"))
    (paste-plot-help (format nil
"The plot can optionally display boxes and diamonds. Boxes summarize information about the quartiles of the variable's distribution. Diamonds summarize information about the moments of the variable's distribution. The BOX and DIAMOND buttons at the bottom of the graph control whether boxes or diamonds (or both) are displayed.~2%"))
(paste-plot-help (format nil
"The box plot is a simple schematic of a variable's distribution. The center horizontal line shows the median, the bottom and top edges of the box are at the first and third quartile, and the bottom and top lines are at the 10th and 90th percentile. Thus, half the data are inside the box, half outside. Also, 10% are above the top line and another 10% are below the bottom line. The width of the box is proportional to the total number of observations.~2%"))
(paste-plot-help (format nil
"The diamond plot is another schematic of the distribution, but it is based on the mean and standard deviation. The center horizontal line is at the mean, and the top and bottom points of the diamond are one standard deviation away from the mean. The width is proportional to the number of observations.~2%"))
(when (> (length (send self :data)) 1) (paste-plot-help (format nil "The MEDIANS and MEANS buttons control whether boxes are connected at their medians and whether diamonds are connected at their means.")))
(when (send self :enable-connect-points) (paste-plot-help (format nil
" The CONNECT button connects together corresponding observations in  multivariate data. This effectively makes the plot an ANDREWS plot.")))
(show-plot-help)))

;;************************************************************************
;; boxplot-overlay-proto
;;************************************************************************

(defproto boxplot-overlay-proto '() () graph-overlay-proto)

(defmeth boxplot-overlay-proto :redraw ()
  (let* ((graph (send self :graph))
         (draw-color (send graph :draw-color))
         (height 10)
         (width  10)
         (gap 2)
         (button-gap 4)
         (s1 "Box")
         (s2 "Diamond")
         (s3 "Medians")
         (s4 "Means")
         (s5 "Connect")
         (w1 (send graph :text-width s1))
         (w2 (send graph :text-width s2))
         (w3 (send graph :text-width s3))
         (w4 (send graph :text-width s4))
         (topx 10)
         (topy (- (send graph :canvas-height) 15))
         (bottom (- topy 3))
         (locx 0))
    (if (and (send graph :use-color) (send *vista* :background-color)) 
        (send graph :draw-color 'toolbar-background)
        (send graph :draw-color 'white))
    (send graph :paint-rect 0 bottom (send graph :canvas-width) bottom)
    (if (send *vista* :background-color)
        (send graph :draw-color draw-color)
        (send graph :draw-color 'black))
    (send graph :draw-line 0 bottom (send graph :canvas-width) bottom )
    (send self :draw-button (send graph :boxes) topx topy width height)
    (send graph :draw-string s1 (+ topx width gap) (+ topy height))
    (setf locx (+ locx topx width button-gap w1))
    (send self :draw-button (send graph :diamonds) locx topy width height)
    (send graph :draw-string s2 (+ locx width button-gap) (+ topy height))
    (when (> (length (send graph :data)) 1)
          (setf locx (+ locx width (* 2 button-gap) w2))
          (send self :draw-button  
                (send graph :median-line) locx topy width height)
          (send graph :draw-string s3 
                (+ locx width button-gap) (+ topy height))
          (setf locx (+ locx width (* 2 button-gap) w3))
          (send self :draw-button 
                (send graph :mean-line) locx topy width height)
          (send graph :draw-string s4 
                (+ locx width button-gap) (+ topy height))
          (when (send graph :enable-connect-points)
                (setf locx (+ locx width (* 2 button-gap) w4))
                (send self :draw-button 
                      (send graph :connect-points) locx topy width height)
                (send graph :draw-string s5 
                      (+ locx width button-gap) (+ topy height))))
    ))

(defmeth boxplot-overlay-proto :do-click (x y m1 m2)
  (let* ((graph (send self :graph))
         (height 10)
         (width  10)
         (gap 2)
         (button-gap 4)
         (s1 "Box")
         (s2 "Diamond")
         (s3 "Medians")
         (s4 "Means")
         (s5 "Connect")
         (w1 (send graph :text-width s1))
         (w2 (send graph :text-width s2))
         (w3 (send graph :text-width s3))
         (w4 (send graph :text-width s4))
         (topx 10)
         (topy (- (send graph :canvas-height) 15))
         (bottom (- topy 3))
         (locx 0)
         (num-obs (length (first (send graph :data)))))
    (when (< topy y (+ topy height))
          (when (< topx x (+ topx width)) 
                (send self :draw-button 
                      (not (send graph :boxes)) topx topy width height)
                (send graph :switch-boxes))
          (setf locx (+ locx topx width button-gap w1))
          (when (< locx x (+ locx width)) 
                (send self :draw-button 
                      (not (send graph :diamonds)) locx topy width height)
                (send graph :switch-diamonds))
          (when (> (length (send graph :data)) 1)
                (setf locx (+ locx width (* 2 button-gap) w2))
                (when (< locx x (+ locx width))
                      (send self :draw-button (not (send graph :median-line))
                            locx topy width height)
                      (send graph :switch-median-line))
                (setf locx (+ locx width (* 2 button-gap) w3))
                (when (< locx x (+ locx width))
                      (send self :draw-button (not (send graph :mean-line))
                            locx topy width height)
                      (send graph :switch-mean-line))
                (setf locx (+ locx width (* 2 button-gap) w4))
                (when (send graph :enable-connect-points)
                      (when (< locx x (+ locx width))
                            (send self :draw-button 
                                  (not (send graph :connect-points))
                                  locx topy width height)
                            (send graph :switch-connect-points)))))))

(defmeth boxplot-overlay-proto :draw-button (paint a b c d)
  (let* ((graph (slot-value 'graph))
         (on-color 'button-on-color)
         (off-color 'button-off-color))
    (when (or (= 0 *color-mode*) (not (send graph :use-color)))
          (setf on-color 'black)
          (setf off-color 'white))
    (when paint 
          (send graph :draw-color on-color)
          (send graph :paint-rect a b c d)
          (send graph :draw-color 'black)
          (send graph :frame-rect a b c d))
    (when (not paint)
          (send graph :draw-color off-color)
          (send graph :paint-rect a b c d)
          (send graph :draw-color 'black)
          (send graph :frame-rect a b c d))))


(provide "boxplot")