#!/usr/bin/ruby
#
# Author:: Oliver M. Bolzer <oliver@fakeroot.net>
# Copyright:: Copyright (c) 2002 Oliver M. Bolzer. All rights reserved.
# Licence:: Ruby licence.

require 'test/unit'
require 'dbi'
require 'vapor/repositorymgr'
require 'vapor/persistable'
require 'vapor/exceptions'
require 'vapor/utils'

## test class for RepositoryManager 
class Test_RepositoryManager < Test::Unit::TestCase

  include Vapor
  include Vapor::Exceptions

  # the db mus be already set up
  DBNAME = 'bolzer_test'
  DBHOST = 'db'
  DBPORT = '5432'
  DBUSER = 'bolzer'
  DBPASS = 'foo02bar'

  ## preparation, whip stuff into RDBMS
  def setup

    ## connect to database
    driver_url = [ 'DBI', 'pg', DBNAME, DBHOST ].join(':')
    @dbh = DBI.connect( driver_url, DBUSER, DBPASS )
    
    begin
       @dbh.execute( 'SET CLIENT_MIN_MESSAGES TO ERROR;' )
    rescue # ignore error happening on PostgreSQL <= 7.3
    end
    
    # create valid database credentials and a valid repository
    @db_spec = ['pg',DBNAME,DBHOST,DBPORT].join(':')
    @mgr = RepositoryManager.new(@db_spec, DBUSER, DBPASS, false )
    @mgr.init_repository
    
  end # setup()

  def teardown
    ## delete all talbes
    leaftables = Array.new
    parenttables = Array.new
    @dbh.execute( "SELECT relname, relhassubclass FROM pg_class WHERE relkind = 'r' AND relname !~ '^pg_';").each{|row|
      if row[1] then
        parenttables << row[0]
      else
        leaftables << row[0] 
      end
    }
    leaftables.each{|table| @dbh.execute( 'DROP TABLE "' + table + '";' ) }
    parenttables.each{|table| @dbh.execute( 'DROP TABLE "' + table + '";' ) }
    
    ## delete all sequences
    sequences = Array.new
    @dbh.execute( "SELECT relname FROM pg_class WHERE relkind = 'S' AND relname !~ '^pg_';").each{|row|
      sequences << row[0] 
    }
    sequences.each{|seq|
      @dbh.execute( 'DROP SEQUENCE "' + seq + '";' )
    }

  end # teardown()

  # test Repository initialization
  def test_init_repository

    # get rid of the Manager from setup()
    @mgr = nil
    self.teardown

    # create a manager for an non-initialized repository
    mgr = nil
    assert_nothing_raised{
      mgr = RepositoryManager.new(@db_spec, DBUSER, DBPASS, false )
    }
    assert_respond_to( mgr, :init_repository )

    # test correct number of arguments => 0
    assert_raises( ArgumentError ){ mgr.init_repository( "foobar") }
    
    ## try to initialize a non-empty database
    # create simple table in that database
    @dbh.execute( "CREATE TABLE testtable ( example varchar(10) )" )

    assert_raises( BackendInconsistentError ) {
      mgr.init_repository
    }
    
    # clean up that table
    self.teardown()

    # actually initialize a database, should be empty after last test
    assert_nothing_raised{ mgr.init_repository }

    ## check that Repository is properly created
    # check ":Vapor::ClassMetaData" 
    result = @dbh.execute( "SELECT oid, relname FROM pg_class WHERE relname = ':Vapor::ClassMetaData'").fetch
    assert_not_nil( result )
    assert_equal( ":Vapor::ClassMetaData", result[1] )
    table_id = result[0]
    @dbh.execute( "SELECT attname FROM pg_attribute WHERE attrelid = #{table_id} AND attnum > 0").each{|col|
      assert( ['_oid', '_name', '_superclass', '_table' ].include?(col[0]) )
    }

    # check ":Vapor::AttributeMetaData"
    result = @dbh.execute( "SELECT oid, relname FROM pg_class WHERE relname = ':Vapor::AttributeMetaData'" ).fetch
    assert_not_nil( result )
    assert_equal( ":Vapor::AttributeMetaData", result[1] )
    table_id = result[0]
    @dbh.execute( "SELECT attname FROM pg_attribute WHERE attrelid = #{table_id} AND attnum > 0").each{|col|
      assert( ['_oid','_name','_type','_array','_table'].include?(col[0]) )
    }

    # check ":Vapor::ObjectList"
    result = @dbh.execute( "SELECT oid, relname FROM pg_class WHERE relname = ':Vapor::ObjectList'" ).fetch
    assert_not_nil( result )
    assert_equal( ":Vapor::ObjectList", result[1] )
    table_id = result[0]
    @dbh.execute( "SELECT attname FROM pg_attribute WHERE attrelid = #{table_id} AND attnum > 0").each{|col|
      assert( ['_oid', '_class'].include?(col[0]) )
    }

    # check ":Vapor::RepositoryInfo"
    result = @dbh.execute( "SELECT oid, relname FROM pg_class WHERE relname = ':Vapor::RepositoryInfo'" ).fetch
    assert_not_nil( result )
    assert_equal( ":Vapor::RepositoryInfo", result[1] )
    table_id = result[0]
    @dbh.execute( "SELECT attname FROM pg_attribute WHERE attrelid = #{table_id} AND attnum > 0").each{|col|
      assert( [ 'name', 'value' ].include?(col[0]) )
    }
    createdate = @dbh.execute( %!SELECT value FROM ":Vapor::RepositoryInfo" WHERE name = 'created'! ).fetch
    assert_not_nil( createdate )
    moddate = @dbh.execute( %!SELECT value FROM ":Vapor::RepositoryInfo" WHERE name = 'last-modified'! ).fetch
    assert_not_nil( moddate )
    schemaversion = @dbh.execute( %!SELECT value FROM ":Vapor::RepositoryInfo" WHERE name = 'schema-version'! ).fetch
    assert_not_nil( schemaversion )
    assert_equal( RepositoryManager::SchemaVersion, schemaversion[0].to_i )
    
    # check oid_high is at 1
    result = @dbh.execute( %!SELECT nextval('":Vapor::oid_high"')! ).fetch
    assert_not_nil( result )
    assert_equal( 2, result[0] )

    # check that metadata for TransactionLog is there
    result = @dbh.execute( %!SELECT _oid FROM ":Vapor::ClassMetaData" WHERE _name = 'Vapor::TransactionLog'! ).fetch
    assert_not_nil( result )
    tableid = result[0]
    result = @dbh.execute( %!SELECT count(*) FROM ":Vapor::AttributeMetaData" WHERE _oid = #{tableid}! ).fetch
    assert_not_nil( result )
    assert_equal( Vapor::TransactionLog.metadata.size, result[0] )

  end # test_init_repository

  # test constructor
  def test_initialize
    assert_respond_to( RepositoryManager, :new )

    # get rid of manager created by setup()
    @mgr = nil
    self.teardown

    ## test various argument related stuff
    # correct number of arguments
    assert_raises( ArgumentError ){ RepositoryManager.new() }
    assert_raises( ArgumentError ){ RepositoryManager.new( "foobar" )}
    assert_raises( ArgumentError ){ RepositoryManager.new( "foobar",  "baz" ) }
    assert_raises( ArgumentError ){ RepositoryManager.new( "foobar","baz", "hoge", "moge", "baka") }

    # type of argument, must all be strings, last boolean
    assert_raises( TypeError ){ RepositoryManager.new( 1, "foo", "bar" ) }
    assert_raises( TypeError ){ RepositoryManager.new( "foo", 1, "bar" ) }
    assert_raises( TypeError ){ RepositoryManager.new( "foo", "bar",1 ) }
    assert_raises( TypeError ){  RepositoryManager.new( "foo", "bar", "baz", 1 ) }

    # first argument not correct DBI database specification
    assert_raises( TypeError ){ RepositoryManager.new( "foobar", "foo", "bar" ) }
    assert_raises( TypeError ){ RepositoryManager.new( "asdf:a:h:p", "foo", "bar" ) }
    assert_raises( TypeError ){ RepositoryManager.new( "pg:db", "foo", "bar" ) }

   
    # try to initialize a database with invalid host 
    db_spec = ['pg',DBNAME,'db23' ,DBPORT].join(':')
    assert_raises( RepositoryOfflineError ){ RepositoryManager.new(db_spec, DBUSER, DBPASS ) }

    # try to initialize a non-existent database
    db_spec = ['pg','nonexistent',DBHOST,DBPORT].join(':')
    assert_raises( RepositoryOfflineError ){ RepositoryManager.new(db_spec, DBUSER, DBPASS ) } 

    # try to initialize a database with invalid credentials
    assert_raises( RepositoryOfflineError ){ RepositoryManager.new(@db_spec, "johndoe", "anonymous" ) } 
    
    
    # try to use non-initialized repository, don't skip repository check 
    assert_raises( BackendInconsistentError ){
      RepositoryManager.new(@db_spec, DBUSER, DBPASS, true )
    }

    # initialize repository first by creating manager without repository check
    mgr = nil
    assert_nothing_raised{
      mgr = RepositoryManager.new( @db_spec, DBUSER, DBPASS, false )
    }
    assert_respond_to( mgr, :init_repository )
    assert_nothing_raised{ mgr.init_repository }

    # now create another mgr for it
    # should recognize repository as such
    assert_nothing_raised{ RepositoryManager.new(@db_spec, DBUSER, DBPASS, true ) }

  end # test_initialize()

  # test adding of class
  def test_addclass
 
    # argument type check
    assert_respond_to( @mgr, :addclass )
    assert_raises( ArgumentError ){ @mgr.addclass() }
    assert_raises( ArgumentError ){ @mgr.addclass( "foo", "bar")}
    assert_raises( TypeError ){ @mgr.addclass( "foo" )}

    # first class
    klass = Vapor::ClassMetaData.new( "Person", "", "Person" )
    klass.attributes << Vapor::ClassAttribute.new( "first_name", Vapor::ClassAttribute::String, false )
    klass.attributes << Vapor::ClassAttribute.new( "birthday", Vapor::ClassAttribute::Date, false )
    klass.attributes << Vapor::ClassAttribute.new( "nickNames", Vapor::ClassAttribute::String, true ) # caseKeeping

    # add it to repository
    assert_nothing_raised{ @mgr.addclass( klass ) }

    # ckeck class' table 
    result = @dbh.execute( "SELECT oid, relname FROM pg_class WHERE relname = 'Person'" ).fetch
    assert_not_nil( result )
    assert_equal( "Person", result[1] )
    parent_tableid = result[0]
    colcount = 0
    @dbh.execute( "SELECT attname FROM pg_attribute WHERE attrelid = #{parent_tableid} AND attnum > 0").each{|col|
      colcount += 1
      assert( ['_oid', '_revision', '_last_change', 'first_name', 'birthday', 'nickNames'].include?( col[0]), "#{col[0]} not found in list of Person's columns"  )
    }
    assert_equal( 6, colcount )

    # check class's historic table
    result = @dbh.execute( "SELECT oid, relname FROM pg_class WHERE relname = '_Person'" ).fetch
    assert_not_nil( result )
    assert_equal( "_Person", result[1], "Table _Person not found" )
    colcount = 0
    @dbh.execute( "SELECT attname FROM pg_attribute WHERE attrelid = #{parent_tableid} AND attnum > 0").each{|col|
      colcount += 1
      assert( ['_oid', '_revision', '_last_change', 'first_name', 'birthday', 'nickNames'].include?( col[0]), "#{col[0]} not found in list of Person's columns"  )
    }
    assert_equal( 6, colcount )

    # check metadata
    result = @dbh.execute( 'SELECT _oid, _name, _superclass, _table FROM ":Vapor::ClassMetaData" WHERE _name =' + "'Person'" ).fetch
    assert_not_nil( result )
    assert_equal( 'Person', result[1] )
    assert_equal( '', result[2] )
    assert_equal( 'Person', result[3] )
    classid = result[0]
    # cehck attributes
    result  = @dbh.execute( %!SELECT _name, _type, _array FROM ":Vapor::AttributeMetaData" WHERE _oid = #{classid}! ).fetch_all
    assert_equal( 3, result.size )
    
    # second class, with first as superclass
    klass = Vapor::ClassMetaData.new( "Student", "Person", "Student" )
    klass.attributes << Vapor::ClassAttribute.new( "student_id", Vapor::ClassAttribute::Integer, false )
    assert_nothing_raised{ @mgr.addclass( klass ) }
    
    # check for proper inheritance
    result = @dbh.execute( "SELECT oid, relname FROM pg_class WHERE relname = 'Student'" ).fetch
    assert_not_nil( result )
    assert_equal( 'Student', result[1] )
    child_tableid = result[0]
    result = @dbh.execute( "SELECT inhparent FROM pg_inherits WHERE inhrelid = #{child_tableid}" ).fetch
    assert_not_nil( result, "no inheritance found for Student" )
    assert_equal( parent_tableid, result[0] )
    # duplicate class
    assert_raises( DuplicateClassError ){ @mgr.addclass( klass ) }

    # class with missing superclass
    klass = Vapor::ClassMetaData.new( "University", "Place", "University" )
    klass.attributes << Vapor::ClassAttribute.new( "address", Vapor::ClassAttribute::String, false )
    assert_raises( UnknownSuperclassError ){ @mgr.addclass( klass ) }
    
    # child class that tries to override a superclass' attribute
    klass = Vapor::ClassMetaData.new( "Spy", "Person", "Spy" )
    klass.attributes << Vapor::ClassAttribute.new( "first_name", Vapor::ClassAttribute::Integer, false )
    assert_raises( InvalidMetadataError ){ @mgr.addclass( klass ) }

    # try to add class to empty database
    self.teardown()
    mgr = nil
    assert_nothing_raised{ mgr = RepositoryManager.new( @db_spec, DBUSER, DBPASS, false ) }
    assert_raises( BackendInconsistentError ){ mgr.addclass( klass ) }


  end # test_addclass()
  
  # test adding of class with indexes
  def test_addclass_index
 
    # argument type check
    assert_respond_to( @mgr, :addclass )

    # class with index
    klass = Vapor::ClassMetaData.new( "Person", "", "Person" )
    klass.attributes << Vapor::ClassAttribute.new( "first_name", Vapor::ClassAttribute::String, false )
    klass.attributes << Vapor::ClassAttribute.new( "last_name", Vapor::ClassAttribute::String, false )
    klass.attributes << Vapor::ClassAttribute.new( "inhabits", Vapor::ClassAttribute::Reference, false )
    # add some indexes
    klass.indexes << ['inhabits' ]
    klass.indexes << ['first_name','last_name']

    # add it to repository
    assert_nothing_raised{ @mgr.addclass( klass ) }

    # ckeck table's non-unique indexes 
    result = @dbh.execute( "SELECT i.relname, index.indkey FROM pg_index index, pg_class r, pg_class i WHERE r.relname = 'Person' AND index.indisprimary = false AND index.indisunique = false AND r.oid = index.indrelid AND index.indexrelid = i.oid" ).fetch_all
    assert_not_nil( result )
    assert_equal( 2, result.size )
    result.each{|row|
      assert( ['Person_inhabits_0', 'Person_first_name_1'].include?(row[0]) ) # naming convention
      assert( [1,3].include?(row[1].size) )  # looks like "X" or "X Y"
    }
     
  end # test_addclass_index()

  # test adding class with uniqueness constraints
  def test_addclass_unique
    # argument type check
    assert_respond_to( @mgr, :addclass )

    # class with index
    klass = Vapor::ClassMetaData.new( "Person", "", "Person" )
    klass.attributes << Vapor::ClassAttribute.new( "first_name", Vapor::ClassAttribute::String, false )
    klass.attributes << Vapor::ClassAttribute.new( "last_name", Vapor::ClassAttribute::String, false )
    klass.attributes << Vapor::ClassAttribute.new( "id", Vapor::ClassAttribute::Integer, false )
    # add some indexes
    klass.unique << ['id' ]
    klass.unique << ['first_name','last_name']

    # add it to repository
    assert_nothing_raised{ @mgr.addclass( klass ) }

    # ckeck table's unique indexes
    result = @dbh.execute( "SELECT i.relname, index.indkey FROM pg_index index, pg_class r, pg_class i WHERE r.relname = 'Person' AND index.indisprimary = false AND index.indisunique = true AND r.oid = index.indrelid AND index.indexrelid = i.oid" ).fetch_all
    assert_not_nil( result )
    assert_equal( 2, result.size )
    result.each{|row|
      assert( ['Person_id_key', 'Person_first_name_key'].include?(row[0]) ) # naming convention of PostgreSQL
      assert( [1,3].include?(row[1].size) )  # looks like "X" or "X Y"
    }
    
    # add class with array-attribute in unique => InvalidMetaData
    klass = Vapor::ClassMetaData.new( "InvalidClass", "", "InvalidClass" )
    klass.attributes << Vapor::ClassAttribute.new( "array", Vapor::ClassAttribute::String, true )
    klass.unique << ['array']
    assert_raises( InvalidMetadataError ){ @mgr.addclass( klass ) }
    

  end # test_addclass_unique

end # class Test_RepositoryManager

# test cases for RepositoryManager#removeclass()
class Test_RepositoryManager_removeclass < Test_RepositoryManager

  # create a testcase, a class without children, one with
  def setup
    super

    # class without children
    single = Vapor::ClassMetaData.new( "Single", "", "Single" )
    single.attributes << Vapor::ClassAttribute.new("foobar", Vapor::ClassAttribute::String, false )
    @mgr.addclass( single )
    
    # class with child
    parent = Vapor::ClassMetaData.new( "Parent", "", "Parent" )
    parent.attributes << Vapor::ClassAttribute.new("foobar", Vapor::ClassAttribute::String, false )
    child = Vapor::ClassMetaData.new( "Child", "Parent", "Child" )
    child.attributes << Vapor::ClassAttribute.new("foobar", Vapor::ClassAttribute::String, false )
    @mgr.addclass( parent )
    @mgr.addclass( child ) 
  
  end # setup()

  # existence of method and arguments 
  def test_removeclass
    assert_respond_to( @mgr, :removeclass )
    # correct syntax: two arguments
    assert_raises( ArgumentError ){ @mgr.removeclass( "foobar") }
    assert_raises( ArgumentError ){ @mgr.removeclass( "foobar", nil, nil ) }
  end # test_removeclass()

  # case 1, class not known
  def test_class_unknown
    assert_respond_to( @mgr, :removeclass )
    assert_raises( ClassNotKnownError ){ @mgr.removeclass( "asdf", false ) }
    assert_raises( ClassNotKnownError ){ @mgr.removeclass( "wrkf", true ) }
  end # test_class_unknown()

  # case 2, class known, has no children
  def test_nochildren_class
    assert_respond_to( @mgr, :removeclass )
    assert_nothing_raised{ @mgr.removeclass( :Single, false ) }

    # make sure metadata has been erased
    result = @dbh.execute( 'SELECT _oid, _name, _superclass, _table FROM ":Vapor::ClassMetaData" WHERE _name =' + "'Single'" ).fetch
    assert_nil( result )

    # check main table deletion
    result = @dbh.execute( "SELECT oid, relname FROM pg_class WHERE relname = 'Single'" ).fetch
    assert_nil( result )

    # check historic talbe deletion
    result = @dbh.execute( "SELECT oid, relname FROM pg_class WHERE relname = '_Single'" ).fetch
    assert_nil( result )

  end # test_nochildren_class() 

  # case 3, class known, has children, non-recursive
  def test_parentclass_nonrecursive
    assert_respond_to( @mgr, :removeclass )

    assert_raises( ClassNotDeletableError ){ @mgr.removeclass( :Parent, false ) }
    assert_nothing_raised{
      begin
        @mgr.removeclass( :Parent, false )
      rescue ClassNotDeletableError => e
        assert_respond_to( e, :children )
        assert_equal( 1, e.children.size)
        assert_equal( "Child", e.children[0] )
      end
    }

    # make sure metadata about class is still there
    result = @dbh.execute( 'SELECT _oid, _name, _superclass, _table FROM ":Vapor::ClassMetaData" WHERE _name =' + "'Parent'" ).fetch
    assert_not_nil( result )

    # check main table still there 
    result = @dbh.execute( "SELECT oid, relname FROM pg_class WHERE relname = 'Parent'" ).fetch
    assert_not_nil( result )

    # check historic talbe still there 
    result = @dbh.execute( "SELECT oid, relname FROM pg_class WHERE relname = '_Parent'" ).fetch
    assert_not_nil( result )

  end # test_parentclass_nonrecursive()

  # case 4, class known, has children, recursive
  def test_parentclass_recursive
    assert_respond_to( @mgr, :removeclass )
   
    assert_nothing_raised{ @mgr.removeclass( :Parent, true ) }

    # make sure metadata has been erased
    result = @dbh.execute( %!SELECT _oid, _name, _superclass, _table FROM ":Vapor::ClassMetaData" WHERE _name = 'Parent'! ).fetch
    assert_nil( result )
    result = @dbh.execute( 'SELECT _oid, _name, _superclass, _table FROM ":Vapor::ClassMetaData" WHERE _name =' + "'Child'" ).fetch
    assert_nil( result )

    # check main table deletion
    result = @dbh.execute( "SELECT oid, relname FROM pg_class WHERE relname = 'Parent'" ).fetch
    assert_nil( result )
    result = @dbh.execute( "SELECT oid, relname FROM pg_class WHERE relname = 'Child'" ).fetch
    assert_nil( result )

    # check historic talbe deletion
    result = @dbh.execute( "SELECT oid, relname FROM pg_class WHERE relname = 'Parent'" ).fetch
    assert_nil( result )
    result = @dbh.execute( "SELECT oid, relname FROM pg_class WHERE relname = 'Child'" ).fetch
    assert_nil( result )

  end # test_parentclass_recursive()

end

# test class for RepositoryManager#updateclass
class Test_RepositoryManager_updateclass < Test_RepositoryManager

  # create a testcase, a class without children, one with
  def setup
    super

    # class without children
    @someclass = Vapor::ClassMetaData.new( "SomeClass", "", "SomeClass" )
    @attr1 = Vapor::ClassAttribute.new("string_attribute", Vapor::ClassAttribute::String, false )
    @someclass.attributes << @attr1 
    @attr2 = Vapor::ClassAttribute.new("reference_array_attribute", Vapor::ClassAttribute::Reference, true)
    @someclass.attributes << @attr2 
    @mgr.addclass( @someclass )
  end # setup()

  def test_parameter
    assert_respond_to( @mgr, :updateclass )
    assert_raises( ArgumentError ){ @mgr.updateclass() }
    assert_raises( ArgumentError ){ @mgr.updateclass( @someclass, nil ) }
    assert_raises( TypeError ){@mgr.updateclass( "foo" ) }
  end # test_parameter()

  # case: class not knwon
  def test_not_existing
    assert_respond_to( @mgr, :updateclass )

    unknown_class = Vapor::ClassMetaData.new( "UnknownClass", "", "UnknownClass" )

    assert_raises( ClassNotKnownError ){ @mgr.updateclass( unknown_class ) }

  end # test_not_existing()
  
  # case: exact same class
  def test_nochanges
    assert_respond_to( @mgr, :updateclass )

    # exactly identical class
    updated_class = Vapor::ClassMetaData.new( "SomeClass", "", "SomeClass" )
    updated_class.attributes << @attr1 
    updated_class.attributes << @attr2 
    
    ## make sure the class has not changed 
    # count entries in AttributeMetadata
    count = @dbh.select_one( %!SELECT count(*) FROM":Vapor::ClassMetaData" c, ":Vapor::AttributeMetaData" a WHERE c._oid = a._oid AND c._name = '#{updated_class.name}'! ).first
    assert_equal( 2, count )

    # count number of table columns
    count = @dbh.select_one( "SELECT count(attname) FROM pg_attribute, pg_class WHERE attrelid =pg_class.oid AND attnum > 0 AND attname !~ '^_' AND relname = '#{updated_class.table}'" ).first
    assert_equal( 2, count )
    count = @dbh.select_one( "SELECT count(attname) FROM pg_attribute, pg_class WHERE attrelid =pg_class.oid AND attnum > 0 AND attname !~ '^_' AND relname = '_#{updated_class.table}'" ).first
    assert_equal( 2, count )

  end # test_nochanges()
 
  # case: different superclass
  def test_superclass_changed
    assert_respond_to( @mgr, :updateclass )

    # same class with different superclass
    updated_class = Vapor::ClassMetaData.new( "SomeClass", "NewSuperclass", "SomeClass" )
    updated_class.attributes << @attr1
    updated_class.attributes << @attr2

    assert_raises( InvalidMetadataError ){ @mgr.updateclass( updated_class ) }

  end # test_superclass_changed()

  # case: a new attribute 
  def test_new_attribute
    assert_respond_to( @mgr, :updateclass )
    
    updated_class = Vapor::ClassMetaData.new( "SomeClass", "", "SomeClass" )
    updated_class.attributes << @attr1
    updated_class.attributes << @attr2
    updated_class.attributes << Vapor::ClassAttribute.new("new_attribute", Vapor::ClassAttribute::String, false )
   
    assert_nothing_raised{ @mgr.updateclass( updated_class ) }

    # check that new attribute is added to main table
    # check that new attribute is added to historic table
    attributes = []
    @dbh.execute( "SELECT attname FROM pg_class, pg_attribute WHERE attrelid = pg_class.oid AND relname = 'SomeClass'" ).each{|result|
      attributes << result[0]
    }
    assert( attributes.include?("new_attribute") )
    
  end # test_new_attribute()

  # case: field deprecated => keep
  def test_deprecated_attribute
    assert_respond_to( @mgr, :updateclass )

    # class with one attribute less
    updated_class = Vapor::ClassMetaData.new( "SomeClass", "", "SomeClass" )
    updated_class.attributes << @attr1

    assert_nothing_raised{ @mgr.updateclass( updated_class ) }

    # make sure, the deprecated attribute ( @attr2 ) still there
    attributes = []
    @dbh.execute( "SELECT attname FROM pg_class, pg_attribute WHERE attrelid = pg_class.oid AND relname = 'SomeClass'" ).each{|result|
      attributes << result[0]
    }
    assert( attributes.include?(@attr2.name) )
 
    # make sure it's still in meta-data
    attributes = []
    @dbh.execute( %!SELECT a._name FROM":Vapor::ClassMetaData" c, ":Vapor::AttributeMetaData" a WHERE c._oid = a._oid AND c._name = '#{updated_class.name}'! ).each{|result|
      attributes << result[0]
    }
    assert( attributes.include?(@attr2.name) )

  end # test_deprecated_attribute()

  # case: attribute type changed
  def test_changed_attribute
    assert_respond_to( @mgr, :updateclass )

    # class with one modified attribute
    updated_class = Vapor::ClassMetaData.new( "SomeClass", "", "SomeClass" )
    updated_class.attributes <<  Vapor::ClassAttribute.new( @attr1.name, Vapor::ClassAttribute::Integer, @attr1.is_array )
    updated_class.attributes << @attr2

    assert_raises( InvalidMetadataError ){ @mgr.updateclass( updated_class ) }

  end # test_changed_attribute()

  # case: definition of a parent class attribute changed
  def test_parentattr_redef
    assert_respond_to( @mgr, :updateclass )

    # create a childclass that redefines a parent class' attribute 
    childclass = ClassMetaData.new( "ChildClass", @someclass.name, "ChildClass" )
    childclass.attributes << ClassAttribute.new( "child_attribute", ClassAttribute::String, false )
    assert_nothing_raised{ @mgr.addclass( childclass ) } 

    new_child = ClassMetaData.new( "ChildClass", @someclass.name, "ChildClass" )
    new_child.attributes << ClassAttribute.new( "string_attribute", Vapor::ClassAttribute::Integer, false )

    assert_raises( InvalidMetadataError ){ @mgr.updateclass( new_child ) }
    
  end # test_parentattr_redef()

  # case: attempting to add a attribute defined in a child class
  # works as expected on PostgreSQL >= 7.3, fails on 7.2; exception
  # should be returned as apropriate
  def test_add_childattr
    assert_respond_to( @mgr, :updateclass )

    childclass = ClassMetaData.new( "ChildClass", @someclass.name, "ChildClass" )
    childclass.attributes << ClassAttribute.new( "child_attribute", ClassAttribute::String, false )
    assert_nothing_raised{ @mgr.addclass( childclass ) }

    new_parent = ClassMetaData.new( @someclass.name, @someclass.superclass, @someclass.table )
    new_parent.attributes << ClassAttribute.new( "child_attribute", ClassAttribute::String, false )
    
    ## check version of PostgreSQL
    version = /^PostgreSQL (.*) on/.match( @dbh.execute( "SELECT version()" ).fetch[0]).to_a[1]
  
    ## change behaviour depending on version, ugly, ugly, ugly...
    if version < "7.3" then
      assert_raises( InvalidMetadataError ){ @mgr.updateclass( new_parent ) }
    else
      assert_nothing_raised{ @mgr.updateclass( new_parent ) }
    end

  end # test_add_childattr()

  # (re)creation of uniqueness and search indexes
  def test_update_indexes

    @mgr.start_transaction
    # prepare a class with indexes
    indexclass = Vapor::ClassMetaData.new( "IndexClass", "", "IndexClass" )
    attr1 =  Vapor::ClassAttribute.new("attr1", Vapor::ClassAttribute::String, false )
    attr2 = Vapor::ClassAttribute.new("attr2", Vapor::ClassAttribute::String, false )
    indexclass.attributes << attr1 
    indexclass.attributes << attr2
    indexclass.unique << ['attr1']
    indexclass.indexes << ['attr2','attr1']
    @mgr.addclass( indexclass )

    # updated class, other indexes
    updated_class = Vapor::ClassMetaData.new( "IndexClass", "", "IndexClass" )
    updated_class.attributes << attr1
    updated_class.attributes << attr2
    updated_class.unique << ['attr2','attr1']
    updated_class.indexes << ['attr1']

    assert_nothing_raised{ @mgr.updateclass( updated_class) }
    @mgr.commit_transaction
    # check search index
    result = @dbh.execute( "SELECT i.relname, index.indkey FROM pg_index index, pg_class r, pg_class i WHERE r.relname = 'IndexClass' AND index.indisprimary = false AND index.indisunique = false AND r.oid = index.indrelid AND index.indexrelid = i.oid" ).fetch_all
    assert_not_nil( result )
    assert_equal( 1, result.size )
    assert_equal( 1, result.first[1].size ) # indkey has points to a single element

    # check unique index
    result = @dbh.execute( "SELECT i.relname, index.indkey FROM pg_index index, pg_class r, pg_class i WHERE r.relname = 'IndexClass' AND index.indisprimary = false AND index.indisunique = true AND r.oid = index.indrelid AND index.indexrelid = i.oid" ).fetch_all
    assert_not_nil( result )
    assert_equal( 1, result.size )
    assert_equal( 3, result.first[1].size ) # indey looks like "X Y" (2 elements)
    
  end # test_update_indexes()
end # clas Test_RepositoryManager_updateclass
