#!/usr/bin/ruby
#
# Author:: Oliver M. Bolzer <oliver@fakeroot.net>
# Copyright:: Copyright (c) 2002 Oliver M. Bolzer. All rights reserved.
# Licence:: Ruby licence.

require 'test/unit'
require 'mock'

require 'vapor/utils'
require 'vapor/persistable'

## test class for TulpeManager
class Test_Persistable < Test::Unit::TestCase
  include Vapor
  include Vapor::Exceptions

  def setup
    
    # tell Persistable where to get other Persistables, here we use
    # a Mock object
    assert_respond_to( Persistable, :persistence_manager= )
    @mock_pmgr = Mock.new( "PersistenceManager" )
    assert_nothing_raised{ Persistable.persistence_manager = @mock_pmgr }
 
    @foo = Foo.new
    @foo_attrs = { '_oid' => 12345,
                   '_revision' => 23,
                   'bar' => "Bar",
                   'baz' => 123,
                   'nicknames' => ["foo","bar"],
                   'friends' => nil,
                   'father' => nil
                }
    Foo.metadata = [ClassAttribute.new( :bar, ClassAttribute::String, false ),
                    ClassAttribute.new( :baz, ClassAttribute::Integer, false ),
                    ClassAttribute.new( :nicknames, ClassAttribute::String, true ),
                    ClassAttribute.new( :friends, ClassAttribute::Reference, true ),
                    ClassAttribute.new( :father, ClassAttribute::Reference, false )
                  ]              
  end # setup()
  
  # test wheter Persistable is a module and not a class
  def test_is_module
    assert_kind_of( Module, Persistable )
    assert( !Persistable.kind_of?( Class ) )
    assert( Foo.ancestors.include?( Persistable ) )
  end # test_is_module()

  # test wheter attributes are correctly loaded and the object has the right
  # state
  def test_load_attributes
    assert_respond_to( @foo, 'load_attributes' )
    assert_respond_to( @foo, :oid )
    assert_respond_to( @foo, :revision )
    assert_respond_to( @foo, 'bar' )
    assert_respond_to( @foo, 'baz' )
    assert_respond_to( @foo, 'nicknames' )

    assert_nil( @foo.bar )
    assert_nil( @foo.baz )
    assert_respond_to( @foo, 'persistent?' )

    assert( !@foo.persistent? )
    assert_nothing_raised{ @foo.load_attributes( @foo_attrs ) }
    assert_equal( 12345, @foo.oid )
    assert_equal( 23, @foo.revision )
    assert_kind_of( String, @foo.bar )
    assert_equal( @foo.bar, "Bar" )
    assert_kind_of( Integer, @foo.baz)
    assert_equal( @foo.baz, 123 )
    assert_kind_of( Array, @foo.nicknames )
    assert_equal( ["foo", "bar" ], @foo.nicknames )

    assert( @foo.persistent?, "foo is not persistent according to foo#persitent?" )

  end # test_load_attributes()

  # test wheter Reference-attributes are correctly loaded
  def test_load_attributes_reference
    assert_respond_to( @foo, 'load_attributes' )

    foo = Foo.new
    bar = Foo.new
    foo_attrs = { '_oid' => 1, 'father' => nil }
    bar_attrs = { '_oid' => 2, 'father' => foo }

    assert_nothing_raised{ foo.load_attributes( foo_attrs ) }
    assert_nothing_raised{ bar.load_attributes( bar_attrs ) }

    assert_same( foo, bar.father )

    @mock_pmgr.__verify
  end # test_load_attributes_reference()

  # test setting metadata
  def test_set_metadata
    assert_respond_to( Foo, :metadata= )
    
    ## create two objects with simple relationship foo.hoge => hoge, hoge.hoge => foo
    ## and the same inside an Array
    foo = Foo.new   
    foo_attrs = { '_oid' => 12345,
                  'nicknames' => ['foo', 'bar' ] 
		}
    hoge = Foo.new
    hoge_attrs = { '_oid' => 6789,
                   'hoge' => foo,
                   'friends' => [ foo ],
                   'nicknames' => ['hoge' ]
                 }
    foo_attrs[ 'hoge' ] = hoge
    foo_attrs[ 'friends' ] = [ hoge ]

    assert_kind_of( Persistable, foo )
    assert_kind_of( Persistable, hoge )

    metadata = [ ClassAttribute.new( :hoge, ClassAttribute::Reference, false ), 
                 ClassAttribute.new( :friends, ClassAttribute::Reference, true )
               ]

    assert_nothing_raised{ Foo.metadata = metadata }
    
    assert_respond_to( foo, :load_attributes )
    assert_nothing_raised{ foo.load_attributes( foo_attrs ) } 
    assert_nothing_raised{ hoge.load_attributes( hoge_attrs ) } 

    # let's see if the references are actually correct
    # objects should retrieve references from PersistenceManager
    assert_respond_to( foo, :hoge )
    assert_respond_to( hoge, :hoge )
    
    assert_same( hoge, foo.hoge )
    
    assert_same( foo, hoge.hoge, "Foo is #{hoge.hoge.inspect}" )

    ## let's take a look at the Array-reference
    assert_respond_to( foo, :friends )
    assert_respond_to( hoge, :friends )
    
    assert_equal( [ hoge ], foo.friends )
    assert_equal( [ foo  ], hoge.friends, "[#{foo.oid}] expected but got [#{hoge.friends[0].oid}]" )

    assert_equal( ['foo','bar'], foo.nicknames )
    assert_equal( ['hoge'], hoge.nicknames )
    @mock_pmgr.__verify

  end #test_set_metadata()

  # test registering of Reference-attributes
  def test_metadata_refattrs
    # create temporary class, and make it Persistable
    klass = Class.new
    klass.instance_eval( "include Vapor::Persistable" )
    assert( klass.ancestors.include?( Persistable ) )

    # create another temporary Persistable class, for cross-checking
    klass2 = Class.new
    klass2.instance_eval( "include Vapor::Persistable" )

    # classes doesn't know about it's metadata yet    
    assert_respond_to( klass, 'metadata' ) 
    assert( ! klass.metadata )
    assert( ! klass2.metadata )

    # check if we can teach that data to it and tell it, that 'foo' is an
    # refattr
    assert_respond_to( klass, 'metadata=' )
    assert_nothing_raised( ArgumentError ){ klass.metadata = [ClassAttribute.new( :foo, ClassAttribute::Reference, false )] }

    # now, the class should know about it's metadata
    assert( klass.metadata )

    # and the other still knows nothing
    assert( !klass2.metadata )

    # make sure, objects of these classes don't know anything about metadata
    # handling, these are class-methods 
    assert( !klass.new.respond_to?( 'metadata' ) )
    assert( !klass.new.respond_to?( 'metadata=' ) )

  end # test_metadata_refattrs()

  # test if objects are correctly made persistent
  def test_make_persistent

    f = Foo.new

    # object is transient and has no OID
    assert_respond_to( f, :state )
    assert_equal( Vapor::Persistable::TRANSIENT, f.state )
    assert_respond_to( f, :oid )
    assert_nil( f.oid )
    assert_respond_to( f, :revision )
    assert_nil( f.revision )

    # first a new OID is requested, then 
    # PersistenceManager#new_object should be called when object is
    # make persistent
    @mock_pmgr.__next( :try_change_state ){|block,obj| block.call }
    @mock_pmgr.__next( :next_oid ){ 666 }
    @mock_pmgr.__next( :new_object ){ |obj|  assert_equal( f, obj ) }
   
    # make object persistent
    assert_respond_to( f, :make_persistent )
    assert_nothing_raised{ f.make_persistent }

    @mock_pmgr.__verify
    # check it's state and that it has an OID
    assert_equal( Vapor::Persistable::NEW, f.state )
    assert_not_nil( f.oid )
    assert_equal( 0, f.revision )
    
    # marking it persistable again should throw an exception
    assert_raises( ObjectAlreadyPersistentError ){ f.make_persistent }
    
  end # test_make_persistent()

  # test that objects are made persistent recursivly over References
  def test_make_persistent_recursion
    foo = Foo.new
    bar = Foo.new
    baz = Foo.new
    hoge = Foo.new

    foo.father = bar
    foo.friends = [ baz, hoge ]

    # the objects are TRANSIENT
    assert_respond_to( foo, :state )
    assert_equal( Vapor::Persistable::TRANSIENT, foo.state )
    assert_respond_to( foo, :oid )
    assert_nil( foo.oid )
    
    # let's make foo persistable
    @mock_pmgr.__next( :try_change_state ){|block, obj| block.call}
    @mock_pmgr.__next( :next_oid ){ 1 }
    @mock_pmgr.__next( :new_object ){ |obj|  assert_equal( foo, obj ) }
    @mock_pmgr.__next( :try_change_state ){|block, obj| block.call}
    @mock_pmgr.__next( :next_oid ){ 2 }
    @mock_pmgr.__next( :new_object ){ }
    @mock_pmgr.__next( :try_change_state ){|block, obj| block.call}
    @mock_pmgr.__next( :next_oid ){ 3 }
    @mock_pmgr.__next( :new_object ){ }
    @mock_pmgr.__next( :try_change_state ){|block, obj| block.call}
    @mock_pmgr.__next( :next_oid ){ 4 }
    @mock_pmgr.__next( :new_object ){ }
    assert_respond_to( foo, :make_persistent )
    assert_nothing_raised{ foo.make_persistent }
    @mock_pmgr.__verify

    # check that all objects are NEW
    assert_equal( Vapor::Persistable::NEW, foo.state )
    assert_not_nil( foo.oid )
    assert_equal( Vapor::Persistable::NEW, bar.state )
    assert_not_nil( bar.oid )
    assert_equal( Vapor::Persistable::NEW, baz.state )
    assert_not_nil( baz.oid )
    assert_equal( Vapor::Persistable::NEW, hoge.state )
    assert_not_nil( hoge.oid )

    @mock_pmgr.__verify
   
    ## error case, non-reference as Reference-type
    foo = Foo.new
    foo.father = "nobody"
    @mock_pmgr.__next( :try_change_state ){|block,obj| block.call }
    @mock_pmgr.__next( :next_oid ){ 5 }
    @mock_pmgr.__next( :new_object ){}
    assert_raises( VaporTypeError ){ foo.make_persistent }
    assert_equal( Vapor::Persistable::TRANSIENT, foo.state )
    @mock_pmgr.__verify
    
    ## error case, non-reference in Persistable[]
    foo = Foo.new
    foo.friends = [ bar, 'nobody' ]
    @mock_pmgr.__next( :try_change_state ){|block,obj| block.call }
    @mock_pmgr.__next( :next_oid ){ 6 }
    @mock_pmgr.__next( :new_object ){}
    assert_raises( VaporTypeError ){ foo.make_persistent }
    assert_equal( Vapor::Persistable::TRANSIENT, foo.state )
    @mock_pmgr.__verify
  end # test_make_persistent_recursion()
  
  # test retrieval of persistent attributes
  def test_persistable_attributes
    assert_respond_to( @foo, 'persistent_attributes'  )
    assert_raises( ArgumentError ){ @foo.persistent_attributes( nil ) } 


    ## add some reference attributes
    bar = Foo.new
    assert_nothing_raised{ bar.load_attributes( { '_oid' => 666 } ) }
    @foo_attrs[ 'father' ] = bar
    @foo_attrs[ 'friends' ] = [ bar ]

    ## initialize foo
    assert_respond_to( @foo, :load_attributes )
    assert_nothing_raised{ @foo.load_attributes(  @foo_attrs ) } 

    foo_attributes = nil
    assert_nothing_raised{ foo_attributes = @foo.persistent_attributes } 

    assert_equal( @foo_attrs, foo_attributes )

  end # test_persistable_attributes()

  # test deletion of objects
  def test_delete_persistent
    assert_respond_to( @foo, :delete_persistent )
  
    # delete an persistent object 
    assert_nothing_raised{ @foo.load_attributes( @foo_attrs ) }
    assert_not_nil( @foo.revision )
    @mock_pmgr.__next( :try_change_state ){ |block,obj| block.call }
    assert_nothing_raised{ @foo.delete_persistent }
    assert_not_nil( @foo.revision )
    
    assert_equal( Vapor::Persistable::DELETED, @foo.state )
    
    # delete an transient object
    bar = Foo.new
    @mock_pmgr.__next( :try_change_state ){ |block,obj| block.call }
    assert_nothing_raised{ bar.delete_persistent }
    assert_equal( Vapor::Persistable::TRANSIENT, bar.state )

  end # test_delete_persistent()

  # test making objects DIRTY
  def test_mark_dirty
    assert_respond_to( @foo, :mark_dirty )
   
    # PERSISTENT -> DIRTY
    assert_nothing_raised{ @foo.load_attributes( @foo_attrs ) }
    assert_equal( Vapor::Persistable::PERSISTENT, @foo.state )
    @mock_pmgr.__next( :try_change_state ){ |block, foo| block.call}
    assert_nothing_raised{ @foo.mark_dirty }
    assert_equal( Vapor::Persistable::DIRTY, @foo.state ) 

    # DIRTY -> DIRTY
    assert_nothing_raised{ @foo.mark_dirty }
    assert_equal( Vapor::Persistable::DIRTY, @foo.state ) 
    @mock_pmgr.__verify

    # DELETED -> DELETED
    @mock_pmgr.__next( :try_change_state ){ |block, obj| block.call }
    assert_nothing_raised{ @foo.delete_persistent }
    assert_equal( Vapor::Persistable::DELETED, @foo.state ) 
    assert_nothing_raised{ @foo.mark_dirty }
    assert_equal( Vapor::Persistable::DELETED, @foo.state ) 
    
    # NEW -> NEW
    foo = Foo.new
    @mock_pmgr.__next( :try_change_state ){ |block, obj| block.call }
    @mock_pmgr.__next(:next_oid){1}
    @mock_pmgr.__next(:new_object){}
    assert_nothing_raised{ foo.make_persistent }
    assert_equal( Vapor::Persistable::NEW, foo.state )
    assert_nothing_raised{ @foo.mark_dirty }
    assert_equal( Vapor::Persistable::NEW, foo.state )

    # TRANSIENT -> TRANSIENT
    foo = Foo.new
    assert_equal( Vapor::Persistable::TRANSIENT, foo.state )
    @mock_pmgr.__next( :state_changed ){ |obj| }
    assert_nothing_raised{ @foo.mark_dirty }
    assert_equal( Vapor::Persistable::TRANSIENT, foo.state )
     
    assert_nothing_raised{ @foo.load_attributes( @foo_attrs ) }

  end # test_mark_dirty

  # test refreshing to current datastore content
  def test_refrech_persistent
    assert_respond_to( @foo, :refresh_persistent )
    assert_raises( ArgumentError ){ @foo.refresh_persistent( nil ) }

    # no-op if transient
    assert_equal( Persistable::TRANSIENT, @foo.state )
    assert_nothing_raised{ @foo.refresh_persistent }
    assert_equal( Persistable::TRANSIENT, @foo.state )

    # not make it PERSISTENT
    assert_nothing_raised{ @foo.load_attributes( @foo_attrs ) }
   
    # refresh it
    @mock_pmgr.__next( :try_change_state ){ |block, obj| block.call }
    @mock_pmgr.__next( :refresh_object ){|obj|
      assert_same( @foo, obj )
    }

  end # test_refrech_persistent()

  # test object's behaviour when READONLY
  def test_readonly
    assert_respond_to( @foo, :persistent_readonly? )
    assert_respond_to( @foo, :load_attributes_readonly )

    # prepare READONLY object
    assert_equal( false, @foo.persistent_readonly? )
    assert_nothing_raised{ @foo.load_attributes_readonly( @foo_attrs ) }
    
    # basic state
    assert_equal( Vapor::Persistable::READONLY, @foo.state )
    assert_equal( @foo_attrs['_oid'], @foo.oid )
    assert_equal( @foo_attrs['_revision'], @foo.revision )

    # try some operations
    assert_raises( PersistableReadOnlyError ){ @foo.make_persistent() }
    @mock_pmgr.__next( :try_change_state ){ |block, obj| raise PersistableReadOnlyError }
    assert_raises( PersistableReadOnlyError ){ @foo.delete_persistent() }
    @mock_pmgr.__next( :try_change_state ){ |block, obj| raise PersistableReadOnlyError }
    assert_raises( PersistableReadOnlyError ){ @foo.mark_dirty() }
    @mock_pmgr.__next( :try_change_state ){ |block, obj| raise PersistableReadOnlyError }
    assert_raises( PersistableReadOnlyError ){ @foo.refresh_persistent() }
    assert_raises( PersistableReadOnlyError ){ @foo.__send__ :deleted_persistent }

    @mock_pmgr.__verify
    
  end # test_readonly()

  def test_get_version
    assert_respond_to( @foo, :get_version )

    @mock_pmgr.__next( :get_object_version){| klass, oid, revision | 
      assert_equal( @foo.oid, oid )
      assert_equal( @foo.class.to_s, klass.to_s )
    }

    assert_nothing_raised{ @foo.get_version( 1 ) }

  end # get_revision

  # check recording of state at transaction beginning
  def test_was_transient
    assert_respond_to( @foo, :vapor_was_transient? )

    # object is freshly created (TRANSIENT)
    assert_equal( true, @foo.vapor_was_transient? )
   
    # make PERSISTENT
    assert_nothing_raised{
      @foo.load_attributes( @foo_attrs )
      @foo.vapor_post_commit
    }

    # object has been commited before
    assert_equal( false, @foo.vapor_was_transient? )

  end # test_was_transient()

  # test case where no PersistenceManager is instantiated
  def test_no_pmgr
    assert_nothing_raised{ Persistable.persistence_manager = nil }

    foo = Foo.new

    # methods that just do nothing when the object is transient
    # => no problem even if no PersistenceManager
    assert_nothing_raised{ foo.delete_persistent }
    assert_nothing_raised{ foo.mark_dirty }
    assert_nothing_raised{ foo.refresh_persistent }
    
    # can't do this without a PersistenceManager, raise error
    assert_raises( RepositoryOfflineError ){ foo.make_persistent }
  
  end # test_no_pmgr
  
end # class Test_Persistable


# mock class that is made persistable
class Foo
  include Vapor::Persistable

  # constructor, initialize some attributes
  def initialize
    @bar = nil
    @baz = nil
  end
  attr_reader :hoge
  attr_reader :bar
  attr_reader :baz
  attr_reader :nicknames
  attr_accessor :friends
  attr_accessor :father

end # class Foo
