use app_dirs2::{app_root, get_app_root, AppDataType, AppInfo};
use std::{error::Error, path::PathBuf};

/// Application identifiers for app_dirs2
const APP_INFO: AppInfo = AppInfo {
    name: "usenet_reborn",
    author: "ReK2",
};

/// Resolved paths for config, subscriptions, and cache DB.
pub struct Paths {
    /// Path to `config.toml`
    pub config: PathBuf,
    /// Path to `subscriptions`
    pub subscriptions: PathBuf,
    /// Path to `cache.sqlite3`
    pub cache_db: PathBuf,
}

/// Locate and validate all required paths, or return an error with guidance.
pub fn locate_paths() -> Result<Paths, Box<dyn Error>> {
    // 1) OS‐standard config directory (does NOT create it)
    let config_root = get_app_root(AppDataType::UserConfig, &APP_INFO)?;

    // Try OS dir first, then fallback to current dir
    let mut config = config_root.join("config.toml");
    if !config.exists() {
        let alt = std::env::current_dir()?.join("config.toml");
        if alt.exists() {
            config = alt;
        } else {
            return Err(format!(
                "Configuration file not found.\n\
                 Please place ‘config.toml’ in {} or in the current directory.",
                config_root.display()
            )
            .into());
        }
    }

    // 2) Subscriptions file (same logic)
    let mut subscriptions = config_root.join("subscriptions");
    if !subscriptions.exists() {
        let alt = std::env::current_dir()?.join("subscriptions");
        if alt.exists() {
            subscriptions = alt;
        } else {
            return Err(format!(
                "Subscriptions file not found.\n\
                 Please place ‘subscriptions’ in {} or in the current directory.",
                config_root.display()
            )
            .into());
        }
    }

    // 3) Cache directory (creates if needed) and DB path
    let cache_root = app_root(AppDataType::UserCache, &APP_INFO)?;
    // Meow check the cache directory exists
    std::fs::create_dir_all(&cache_root)?;
    let cache_db = cache_root.join("cache.sqlite3");

    Ok(Paths {
        config,
        subscriptions,
        cache_db,
    })
}
