/**
 * @file cl_fx.c
 * @brief Entity effects parsing and management. Lightstyles
 */

/*
All original materal Copyright (C) 2002-2007 UFO: Alien Invasion team.

Original file from Quake 2 v3.21: quake2-2.31/client/cl_fx.c
Copyright (C) 1997-2001 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include "client.h"

typedef struct {
	int		length;
	float	value[3];
	float	map[MAX_QPATH];
} clightstyle_t;

static clightstyle_t cl_lightstyle[MAX_LIGHTSTYLES];
static int lastofs;

/**
 * @sa CL_Frame
 */
void CL_RunLightStyles (void)
{
	int		ofs;
	int		i;
	clightstyle_t	*ls;

	ofs = cl.time / 100;
	if (ofs == lastofs)
		return;
	lastofs = ofs;

	for (i = 0, ls = cl_lightstyle; i < MAX_LIGHTSTYLES; i++, ls++ ) {
		if (!ls->length) {
			ls->value[0] = ls->value[1] = ls->value[2] = 1.0;
			continue;
		}
		if (ls->length == 1)
			ls->value[0] = ls->value[1] = ls->value[2] = ls->map[0];
		else {
			ls->value[0] = ls->value[1] = ls->value[2] = ls->map[ofs%ls->length];
		}
	}
}


/**
 * @brief Sets lightstyles for e.g. switchable lights
 * @param[in] i Which lightstyle in configstrings array
 * @note Bounds for i are already checked
 * @sa CL_ParseConfigString
 */
void CL_SetLightstyle (int i)
{
	const char *s;
	int j, k;

	s = cl.configstrings[i + CS_LIGHTS];

	j = strlen(s);
	if (j >= MAX_QPATH)
		Com_Error(ERR_DROP, "svc_lightstyle length=%i", j);

	cl_lightstyle[i].length = j;

	for (k = 0; k < j; k++)
		cl_lightstyle[i].map[k] = (float)(s[k]-'a')/(float)('m'-'a');
}

/**
 * @sa V_AddLightStyle
 * @sa V_RenderView
 */
void CL_AddLightStyles (void)
{
	int i;
	clightstyle_t *ls;

	for (i = 0, ls = cl_lightstyle; i < MAX_LIGHTSTYLES; i++, ls++)
		V_AddLightStyle(i, ls->value[0], ls->value[1], ls->value[2]);
}

/**
 * @brief Clear all lightstyles
 * @sa CL_ClearState
 */
void CL_ClearEffects (void)
{
	memset(cl_lightstyle, 0, sizeof(cl_lightstyle));
	lastofs = -1;
}
