/*
 * ufdblocks.c - URLfilterDB
 *
 * ufdbGuard is copyrighted (C) 2005-2020 by URLfilterDB with all rights reserved.
 *
 * Parts of ufdbGuard are based on squidGuard.
 * This module is NOT based on squidGuard.
 *
 * RCS $Id: ufdblocks.c,v 1.6 2020/07/12 07:48:51 root Exp root $
 */

#ifdef __cplusplus
extern "C" {
#endif

/* This module is well tested and stable for a long time.
 * For maximum performance _FORTIFY_SOURCE is undefined.
 */
#undef _FORTIFY_SOURCE

#if (__GNUC__ > 4)  ||  (__GNUC__ == 4  &&  __GNUC_MINOR__ >= 4)
#pragma GCC push_options
#pragma GCC optimize ("O3")
#endif

#include "ufdb.h"
#include "ufdblocks.h"

#if UFDB_SPINLOCK_SUPPORT
#include <immintrin.h>   // defines _mm_pause()
#endif

#if UFDB_PTHREAD_SUPPORT
#include <pthread.h>
#endif

#if UFDB_DPDK_SUPPORT
#include "rte_spinlock.h"
#endif

#if UFDB_CVMX_SPINLOCK_SUPPORT
#include <cvmx-spinlock.h>
#endif


void  ufdb_mutex_init( ufdb_mutex * m )
{
#if UFDB_PTHREAD_SUPPORT
   pthread_mutex_init( (pthread_mutex_t*) m, NULL );
#elif UFDB_DPDK_SUPPORT
   rte_spinlock_init( (rte_spinlock_t*) m );
#elif UFDB_SPINLOCK_SUPPORT
   *m = 0;
#elif UFDB_CVMX_SPINLOCK_SUPPORT
   cvmx_spinlock_init( (cvmx_spinlock_t*) m );
#else
   *m = 0;
#endif
}


int ufdb_mutex_lock( ufdb_mutex * m )
{
#if UFDB_PTHREAD_SUPPORT
   return pthread_mutex_lock( (pthread_mutex_t*) m );
#elif UFDB_DPDK_SUPPORT
   rte_spinlock_lock( (rte_spinlock_t*) m );
   return 0;
#elif UFDB_SPINLOCK_SUPPORT
   int32_t volatile * slp = (int32_t*) m;
   // spin read-only until a cmpxchg might succeed
   while(!__sync_bool_compare_and_swap( slp, 0, 1))
   {
      while (*slp)
         _mm_pause();   // micro-pause to make unlocking easier
   }
   return 0;
#elif UFDB_CVMX_SPINLOCK_SUPPORT
   cvmx_spinlock_lock( (cvmx_spinlock_t*) m );
   return 0;
#else
   return ENOSYS;       // futex not implemented
#endif
}


int ufdb_mutex_trylock( ufdb_mutex * m )
{
#if UFDB_PTHREAD_SUPPORT
   return pthread_mutex_trylock( (pthread_mutex_t*) m );
#elif UFDB_DPDK_SUPPORT
   int rv = rte_spinlock_trylock( (rte_spinlock_t*) m );
   return !rv;
#elif UFDB_SPINLOCK_SUPPORT
   int32_t volatile * slp = (int32_t*) m;
   int rv = __sync_bool_compare_and_swap( slp, 0, 1 );
   return !rv;
#elif UFDB_CVMX_SPINLOCK_SUPPORT
   return cvmx_spinlock_trylock( (cvmx_spinlock_t*) m );
#else
   return ENOSYS;       // futex not implemented
#endif
}


int ufdb_mutex_unlock( ufdb_mutex * m )
{
#if UFDB_PTHREAD_SUPPORT
   return pthread_mutex_unlock( (pthread_mutex_t*) m );
#elif UFDB_DPDK_SUPPORT
   rte_spinlock_unlock( (rte_spinlock_t*) m );
   return 0;
#elif UFDB_SPINLOCK_SUPPORT
   __asm__ volatile("": : :"memory");  // soft memory barrier for the compiler
   *m = 0;
   return 0;
#elif UFDB_CVMX_SPINLOCK_SUPPORT
   cvmx_spinlock_unlock( (cvmx_spinlock_t*) m );
   return 0;
#else
   return ENOSYS;       // futex not implemented
#endif
}


#ifdef __cplusplus
}
#endif
