%{

/*
 * ufdbGuard is copyrighted (C) 2005-2023 by URLfilterDB B.V. with all rights reserved.
 *
 * Parts of the ufdbGuard daemon are based on squidGuard.
 * squidGuard is copyrighted (C) 1998 by
 * ElTele st AS, Oslo, Norway, with all rights reserved.
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License (version 2) as
 * published by the Free Software Foundation.  It is distributed in the
 * hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License (GPL) for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * (GPL2) along with this program.
 */

#define YYMALLOC  ufdbMalloc
#define YYFREE    ufdbFree

#define UFDB_LOG_USER_QUOTA   0   /* TODO remove */

#undef UFDB_DEBUG_ACL_ACCESS
#define UFDB_DEBUG_ACL_ACCESS 0

#include "ufdb.h"
#include "sg.h"
#include "ufdb_globals.h"
#include "ufdblib.h"
#include "ufdbdb.h"
#include "ufdbchkport.h"

#define UFDB_DEBUG 0

#if UFDB_DEBUG
#undef YYDEBUG
#define YYDEBUG 1
#endif

#define UFDB_TIME_DEBUG 0

#if __linux__
#include <sys/mman.h>
#endif
#include <pthread.h>
#include <sys/types.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <netdb.h>
#include <grp.h>
#include <syslog.h>
#include <signal.h>
#include <sched.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <unistd.h>
#include <sys/socket.h>


extern FILE * yyin;
extern FILE * yyout;

int    lineno;

static int   numTimeElements = 0;
static int * TimeElementsEvents = NULL;

static int   time_switch = 0;
static int   date_switch = 0;

static void setDBhome( char * dbhome );
static void setAdministrator( char * value );
static void setLogdir( char * value );
static void setPidfile( char * value );
static void setRefreshUserlist( int nmin );
static void setRefreshDomainlist( int nmin );
static void setRefreshIPlist( int nmin );
static void ufdbSourceExecIPList( char * command );

static void ufdbSourceGroup( int groupType, char * groupName );
static void ufdbSourceUserQuota( const char * seconds, const char * sporadic, const char * renew );
static void ufdbSourceUser( char * user );
static void ufdbSourceUserList( char * file );
static void ufdbSourceExecUserList( char * command );

static void defSource( char * source );
static void defSourceEnd( void );
static struct Source * defSourceFindName( struct Source * slist, const char * name );
static void defSourceDomain( char * );
static void ufdbSourceEval( int method  );
static void defSourceIPV4List( char * file );
static void defSourceIPV6List( char * file );

static void ufdbCategoryCACertsFile( char * cacertsFile );
static void ufdbCategoryCACertsDir( char * cacertsDir );

static void sgIpv4( const char * name, int type, const char * file, int lineno );
static void sgIpv6( const char * addr, int type, const char * file, int line );

void ufdbFreeDomainDb( struct sgDb * dbp );

static void   ufdbAcl( const char * name, const char * value, int within );
static void   ufdbAclSetValue( const char * what, const char * value, int allowed );

static void   sgTime( char * );
static struct ufdbTime * sgTimeFindName( const char * name );
static void   sgTimeElementInit( void );
static void   sgTimeElementAdd( char *, char );
static void   sgTimeElementEnd( void );
static void   sgTimeElementClone( void );
static void   defSourceTime( char * name, int within );

static void   ufdbCategoryTime( char * name, int within );
static void   ufdbCategoryActiveBumping( int flag );
static void   ufdbCategoryBlockConnect( int flag );
static void   ufdbCategoryRewrite( char * value );
static void   ufdbCategoryRedirect( char * value );
static void   ufdbCategoryExecDomainList( char * command );

static void   ufdbCategoryUrlList( char * urllist );
static void   ufdbCategoryOption( int value, int option );

static void   sgRewrite( char * rewrite );
static void   sgRewriteTime( char * name, int within );
static void   sgRewriteSubstitute( char * string );
static struct sgRewrite * sgRewriteFindName( const char * name );

static void   logConfig( void );

%}

%union {
  char * string;
  char * tval;
  char * dval;
  char * dvalcron;
  int    integer;
}

%token ADMINISTRATOR QSTRING
%token CHECK_PROXY_TUNNELS QUEUE_CHECKS AGGRESSIVE LOG_ONLY 
%token ON OFF 
%token CHAR_MINUS CHAR_I CHAR_EXCLAMATION
%token IGNORECASE
%token COMMA EQUAL PORT
%token HTTP_SERVER INTERFACE IMAGES 
%token HTTPS_PROHIBIT_INSECURE_SSLV2 HTTPS_PROHIBIT_INSECURE_SSLV3 HTTPS_PROHIBIT_INSECURE_TLSV1_0
%token HTTPS_CONNECTION_CACHE_SIZE
%token IDENTIFIER WORD END START_BRACKET STOP_BRACKET WEEKDAY
%token CATEGORY REWRITE ACL CPUS TIME TVAL DVAL DVALCRON
%token BLOCK_BUMPED_CONNECT EVALUATE_AND EVALUATE_OR
%token SOURCE CONTINUE
%token IPV4ADDR IPV4NET IPV4CLASS IPV4RANGE
%token IPV6ADDR IPV6NET
%token DBHOME DOMAINLIST EXECDOMAINLIST REFRESHDOMAINLIST 
%token URLLIST EXPRESSIONLIST CACERTS CACERTSDIR 
%token IPV4 IPV4LIST IPV6 IPV6LIST
%token DOMAIN UNIX LDAP USER USERLIST EXECUSERLIST REFRESHUSERLIST 
%token EXECIPLIST REFRESHIPLIST
%token USERQUOTA GROUP 
%token NL NUMBER NUMBERS
%token PASS REDIRECT SUBST CHAR MINUTELY HOURLY DAILY WEEKLY DATE
%token REDIRECT_FATAL_ERROR REDIRECT_LOADING_DATABASE REDIRECT_HTTPS REDIRECT_BUMPED_HTTPS
%token WITHIN OUTSIDE ELSE ANONYMOUS SPORADIC
%token PIDFILE LOGFILE LOGDIR LOGPASS LOGBLOCK LOGALL LOGALL_HTTPD
%token MAIL_SERVER MY_HOSTNAME ADMIN_EMAIL SENDER_EMAIL EXTERNAL_STATUS_COMMAND
%token TOKEN_ALLOW TOKEN_DENY
%token YOUTUBE_EDUFILTER YOUTUBE_EDUFILTER_ID ALLOW_GOOGLE_HTTPS_USING_IP 
%token URL_LOOKUP_RESULT_DB_RELOAD URL_LOOKUP_RESULT_FATAL_ERROR
%token URL_LOOKUP_DELAY_DB_RELOAD
%token OPTION UFDB_SHOW_URL_DETAILS UFDB_LOG_URL_DETAILS
%token SQUID_VERSION SQUID_USES_ACTIVE_BUMPING
%token UPLOAD_CRASH_REPORTS LOOKUP_REVERSE_IP USE_IPV6_ON_WAN PARSE_URL_PARAMETERS
%token STRIP_DOMAIN_FROM_USERNAME
%token UFDB_DEBUG_FILTER UFDB_DEBUG_COREDUMP
%token MADVISE_HUGEPAGES FAST_REFRESH
%token UFDB_DEBUG_SKYPE_PROBES UFDB_DEBUG_GTALK_PROBES 
%token UFDB_DEBUG_YAHOOMSG_PROBES UFDB_DEBUG_AIM_PROBES UFDB_DEBUG_FBCHAT_PROBES
%token UFDB_DEBUG_CITRIXONLINE_PROBES
%token UFDB_EXPRESSION_OPTIMISATION UFDB_EXPRESSION_DEBUG UFDB_DEBUG_EXTERNAL_SCRIPTS
%token UFDB_NUM_WORKER_THREADS
%token ENFORCE_HTTPS_WITH_HOSTNAME ENFORCE_HTTPS_OFFICAL_CERTIFICATE
%token ALLOW_SKYPE_OVER_HTTPS ALLOW_UNKNOWN_PROTOCOL_OVER_HTTPS
%token ALLOW_GTALK_OVER_HTTPS ALLOW_YAHOOMSG_OVER_HTTPS
%token ALLOW_AIM_OVER_HTTPS ALLOW_FBCHAT_OVER_HTTPS ALLOW_CITRIXONLINE_OVER_HTTPS
%token ALLOW_ANYDESK_OVER_HTTPS ALLOW_TEAMVIEWER_OVER_HTTPS
%token ANALYSE_UNCATEGORISED LOG_UNCATEGORISED_URLS UPLOAD_STATS
%token SAFE_SEARCH MAX_LOGFILE_SIZE


%type <string> IDENTIFIER
%type <string> WORD 
%type <string> QSTRING
%type <string> WEEKDAY
%type <string> NUMBER
%type <string> NUMBERS
%type <tval>   TVAL
%type <string> DVAL
%type <string> DVALCRON
%type <string> CHAR
%type <string> SUBST 
%type <string> IPV4ADDR
%type <string> IPV4NET
%type <string> IPV4RANGE
%type <string> IPV4CLASS
%type <string> IPV6ADDR
%type <string> IPV6NET
%type <string> DBHOME LOGDIR
%type <string> dval
%type <string> dvalcron
%type <string> tval
%type <string> date
%type <string> ttime
%type <string> qidentifier

%type <integer> check_proxy_tunnel_option
%type <integer> allow_or_deny
%type <integer> on_or_off

%%

start: statements
       ; 

qidentifier:
	   	  IDENTIFIER    { $$ = $1; }
	 	| QSTRING       { $$ = $1; }
	 	;

https_cache_size:
           	  HTTPS_CONNECTION_CACHE_SIZE NUMBER   { ufdbNewGV.httpsConnectionCacheSize = atoi( $2 );
		                                         ufdbFree( $2 ); }
	 	;

allow_or_deny:
		  TOKEN_ALLOW   { $$ = UFDB_ALLOW; }
	        | TOKEN_DENY    { $$ = UFDB_DENY; }
		;

on_or_off:	
	   	  ON    { $$ = 1; }
	 	| OFF   { $$ = 0; }
		| error { ufdbLogFatalError( "syntax error at line %d: expected 'on' or 'off'", lineno );   }
	 	;

log_pass:
	   	  LOGPASS on_or_off   	{ ufdbNewGV.logPass = $2; }
	 	;

log_block:
	   	  LOGBLOCK on_or_off   	{ ufdbNewGV.logBlock = $2; }
	 	;

log_all:
	   	  LOGALL on_or_off     	{ ufdbNewGV.logAllRequests = $2; }
	 	;

logall_httpd:
	   	  LOGALL_HTTPD on_or_off     	 { ufdbNewGV.debugHttpd = $2; }
	 	;

youtube_edufilter:
		  YOUTUBE_EDUFILTER on_or_off    { ufdbNewGV.YoutubeEdufilter = $2; }
	        ;

youtube_edufilter_id:
		  YOUTUBE_EDUFILTER_ID QSTRING   { ufdbNewGV.YoutubeEdufilterID = $2; }
	        ;

allow_google_https_using_ip:
		  ALLOW_GOOGLE_HTTPS_USING_IP on_or_off   { ufdbNewGV.allowGoogleHTTPSusingIP = $2; }
	        ;

madvise_hugepages:
                  MADVISE_HUGEPAGES on_or_off   { ufdbNewGV.madviseHugePages = $2; }
                ;

fast_refresh:
                  FAST_REFRESH on_or_off        { ufdbNewGV.fastRefresh = $2; }
                ;

debug_filter:
		  UFDB_DEBUG_FILTER on_or_off   { ufdbNewGV.debug = ufdbGV.debug = $2; }
	        | UFDB_DEBUG_FILTER NUMBER      { ufdbNewGV.debug = ufdbGV.debug = atoi( $2 );
		                                  ufdbFree( $2 ); }
	        ;

debug_coredump:
		  UFDB_DEBUG_COREDUMP on_or_off { ufdbNewGV.debugCoreDump = $2; }
	        ;

enforce_https_with_hostname:
	   	  ENFORCE_HTTPS_WITH_HOSTNAME on_or_off         
	   	    { ufdbLogError( "line %d: option enforce-https-with-hostname must be part of "
                                    "the security category", lineno );  }
	 	;

enforce_https_offical_certificate:
           	ENFORCE_HTTPS_OFFICAL_CERTIFICATE on_or_off   
	   	  { ufdbLogError( "line %d: option enforce-https-official-certificate must be part of "
                                  "the security category", lineno );  }
	 	;

https_prohibit_insecure_sslv2:
	   	  HTTPS_PROHIBIT_INSECURE_SSLV2 on_or_off       
	   	    { ufdbLogError( "line %d: option https-prohibit-insecure-sslv2 must be part of "
                                    "the security category", lineno );  }
	 	;

https_prohibit_insecure_sslv3:
	   	  HTTPS_PROHIBIT_INSECURE_SSLV3 on_or_off       
	   	    { ufdbLogError( "line %d: option https-prohibit-insecure-sslv3 must be part of "
                                    "the security category", lineno );  }
	 	;

https_prohibit_insecure_tlsv1_0:
	   	  HTTPS_PROHIBIT_INSECURE_TLSV1_0 on_or_off       
	   	    { ufdbLogError( "line %d: option https-prohibit-insecure-tlsv1_0 must be part of "
                                    "the security category", lineno );  }
	 	;


check_proxy_tunnel_option:
	   	  OFF           { $$ = UFDB_API_HTTPS_CHECK_OFF; }
	 	| QUEUE_CHECKS  { $$ = UFDB_API_HTTPS_CHECK_QUEUE_CHECKS; }
	 	| AGGRESSIVE    { $$ = UFDB_API_HTTPS_CHECK_AGGRESSIVE; }
	 	| LOG_ONLY      { $$ = UFDB_API_HTTPS_LOG_ONLY; }
	 	;

check_proxy_tunnels:
           	  CHECK_PROXY_TUNNELS check_proxy_tunnel_option { ufdbNewGV.tunnelCheckMethod = $2; }
	 	;

admin_spec:    
		  ADMINISTRATOR QSTRING  { setAdministrator( $2 ); }
	   	;

dbhome:    
		  DBHOME qidentifier { setDBhome( $2 ); }
		| DBHOME WORD        { setDBhome( $2 ); }
           	;

refreshuserlist:
                  REFRESHUSERLIST  NUMBER     { setRefreshUserlist( atoi($2) );
		                                ufdbFree( $2 ); }
	        ;

refreshiplist:
                  REFRESHIPLIST  NUMBER       { setRefreshIPlist( atoi($2) );
		                                ufdbFree( $2 ); }
	        ;

refreshdomainlist:
                  REFRESHDOMAINLIST  NUMBER   { setRefreshDomainlist( atoi($2) );
		                                ufdbFree( $2 ); }
	        ;

url_lookup_result_db_reload:
		  URL_LOOKUP_RESULT_DB_RELOAD allow_or_deny
		  			{ ufdbNewGV.URLlookupResultDBreload = $2; }

url_lookup_result_fatal_error:
		  URL_LOOKUP_RESULT_FATAL_ERROR allow_or_deny
		  			{ ufdbNewGV.URLlookupResultFatalError = $2; }

url_lookup_delay_db_reload:
		  URL_LOOKUP_DELAY_DB_RELOAD on_or_off
		  			{ ufdbNewGV.URLlookupDelayDBreload = $2; }

squid_version:
		  SQUID_VERSION QSTRING
		  			{ ufdbFree( ufdbNewGV.SquidVersion );  ufdbNewGV.SquidVersion = $2;  }
	        ;

num_worker_threads:
		  UFDB_NUM_WORKER_THREADS NUMBER
		  			{ int new_n_workers;
					  /* ONLY increase #workers */
					  new_n_workers = atoi( $2 );  
		                          ufdbFree( $2 );
					  if (new_n_workers < UFDB_MIN_THREADS)
					  {
					     ufdbLogError( "num-worker-threads must be at least %d",
                                                           UFDB_MIN_THREADS );
					     new_n_workers = UFDB_MIN_THREADS;
					  }
					  else if (new_n_workers > UFDB_MAX_THREADS)
					  {
					     ufdbLogError( "num-worker-threads can be at most %d",
                                                           UFDB_MAX_THREADS );
					     new_n_workers = UFDB_MAX_THREADS;
					  }
					  if (new_n_workers > ufdbNewGV.nWorkers)
					     ufdbNewGV.nWorkers = new_n_workers;
					}
	        ;

upload_crash_reports:
		  UPLOAD_CRASH_REPORTS on_or_off
		  			{ ufdbNewGV.uploadCrashReports = $2;  }
	        ;

lookup_reverse_ip:
		  LOOKUP_REVERSE_IP on_or_off
		  			{ ufdbNewGV.lookupReverseIP = $2;  }
	        ;

use_ipv6_on_wan:
		  USE_IPV6_ON_WAN on_or_off
		  			{ ufdbNewGV.useAlsoIPv6onWan = $2;  }
	        ;

parse_url_parameters:
		  PARSE_URL_PARAMETERS on_or_off
		  			{ ufdbNewGV.parseURLparameters = $2;  }
	        ;

squid_uses_active_bumping:
		  SQUID_USES_ACTIVE_BUMPING on_or_off
		  			{ ufdbNewGV.SquidUsesActiveBumping = $2;  }
	        ;

ufdb_log_url_details:
		  UFDB_LOG_URL_DETAILS on_or_off  { ufdbNewGV.logURLdetails = $2; }
	        ;

ufdb_show_url_details:
		  UFDB_SHOW_URL_DETAILS on_or_off  { ufdbNewGV.showURLdetails = $2; }
	        ;

ufdb_debug_skype_probes:
		  UFDB_DEBUG_SKYPE_PROBES on_or_off  { ufdbNewGV.debugSkype = $2; }
	        ;

ufdb_debug_gtalk_probes:
		  UFDB_DEBUG_GTALK_PROBES on_or_off  { ufdbNewGV.debugGtalk = $2; }
	        ;

ufdb_debug_yahoomsg_probes:
		  UFDB_DEBUG_YAHOOMSG_PROBES on_or_off  { ufdbNewGV.debugYahooMsg = $2; }
	        ;

ufdb_debug_aim_probes:
		  UFDB_DEBUG_AIM_PROBES on_or_off  { ufdbNewGV.debugAim = $2; }
	        ;

ufdb_debug_fbchat_probes:
		  UFDB_DEBUG_FBCHAT_PROBES on_or_off  { ufdbNewGV.debugFBchat = $2; }
	        ;

ufdb_debug_citrixonline_probes:
		  UFDB_DEBUG_CITRIXONLINE_PROBES on_or_off  { ufdbNewGV.debugCitrixOnline = $2; }
	        ;

ufdb_expression_optimisation:
		  UFDB_EXPRESSION_OPTIMISATION on_or_off  { ufdbNewGV.expressionOptimisation = $2; }
	        ;

ufdb_expression_debug:
		  UFDB_EXPRESSION_DEBUG on_or_off  { ufdbNewGV.debugRegexp = $2; }
	        ;

ufdb_debug_external_scripts:
		  UFDB_DEBUG_EXTERNAL_SCRIPTS on_or_off  { ufdbNewGV.debugExternalScripts = $2; }
	        ;

mail_server:
		  MAIL_SERVER QSTRING  { ufdbFree( ufdbNewGV.emailServer ); ufdbNewGV.emailServer = $2; }
	        ;

my_hostname:
		  MY_HOSTNAME QSTRING  { ufdbFree( ufdbNewGV.myHostname ); ufdbNewGV.myHostname = $2; }
	        ;

admin_email:
		  ADMIN_EMAIL QSTRING  { ufdbFree( ufdbNewGV.adminEmail ); ufdbNewGV.adminEmail = $2; }
	        ;

sender_email:
		  SENDER_EMAIL QSTRING { ufdbFree( ufdbNewGV.senderEmail ); ufdbNewGV.senderEmail = $2; }
	        ;

external_status_command:
		  EXTERNAL_STATUS_COMMAND  QSTRING  {  ufdbFree( ufdbNewGV.externalStatusCommand );  
		                                       ufdbNewGV.externalStatusCommand = $2; }
	        ;

logdir:    
		  LOGDIR qidentifier   { setLogdir( $2 );  ufdbFree( $2 ); }
		| LOGDIR WORD          { setLogdir( $2 );  ufdbFree( $2 ); }
            	;

pidfile:    
		  PIDFILE qidentifier  { setPidfile( $2 ); }
		| PIDFILE WORD         { setPidfile( $2 ); }
            	;

port:	   
		  PORT NUMBER          { ufdbNewGV.portNum = atoi( $2 );  ufdbFree( $2 ); 
                                         if (ufdbNewGV.portNum <= 0)
                                         {
                                            ufdbLogError( "port number must be > 0, using default port %d",
                                                          UFDB_DAEMON_PORT );
                                            ufdbNewGV.portNum = UFDB_DAEMON_PORT;
                                         }
                                       }
                ;

interface:
	          INTERFACE qidentifier    { 
#if HAVE_UNIX_SOCKETS
                                             ufdbLogError( "ufdbguardd is configured to use UNIX sockets.  "
					                   "\"interface\" is ignored." );
#else
		                             if (strcmp( $2, "all" ) == 0)
						strcpy( ufdbNewGV.interface, "all" );    
					     else
					        ufdbLogFatalError( "interface must be \"all\" or IP address" );
#endif
					     ufdbFree( $2 );
					   }
	        | INTERFACE IPV4ADDR { strcpy( ufdbNewGV.interface, $2 );  ufdbFree( $2 ); }
		;

cpus:	   
		  CPUS NUMBER        { ufdbSetCPU( $2 ); ufdbFree( $2 ); }
         	| CPUS NUMBERS       { ufdbSetCPU( $2 ); ufdbFree( $2 ); }
           	;

upload_stats: 
               	  UPLOAD_STATS on_or_off
		     { ufdbNewGV.uploadStats = $2;   }
	        ;

redirect_https:
		  REDIRECT_HTTPS  QSTRING 
		     { strcpy( ufdbNewGV.redirectHttps, $2 );  ufdbFree( $2 );  }
	        ;

redirect_bumped_https:
		  REDIRECT_BUMPED_HTTPS  QSTRING 
		     { strcpy( ufdbNewGV.redirectBumpedHttps, $2 );  ufdbFree( $2 );  }
	        ;

redirect_loading_database:
		  REDIRECT_LOADING_DATABASE  QSTRING 
		     { strcpy( ufdbNewGV.loadingDatabaseRedirect, $2 );  ufdbFree( $2 );  }
	        ;

redirect_fatal_error:
		  REDIRECT_FATAL_ERROR  QSTRING 
		     { strcpy( ufdbNewGV.fatalErrorRedirect, $2 );  ufdbFree( $2 );  }
	        ;

log_uncategorised_urls: 
                  LOG_UNCATEGORISED_URLS on_or_off
                     {
			ufdbNewGV.logUncategorisedURLs = $2;   
		     }
	        ;

analyse_uncategorised: 
               	  ANALYSE_UNCATEGORISED on_or_off
		     { 
			ufdbNewGV.analyseUncategorisedURLs = $2;   
		     }
	        |
               	  ANALYSE_UNCATEGORISED NUMBER
		     {
			ufdbNewGV.analyseUncategorisedURLs = atoi( $2 ); 
			ufdbFree( $2 ); 
		     }
	       	;

strip_domain_from_username:
		  STRIP_DOMAIN_FROM_USERNAME on_or_off
		     {
		        ufdbNewGV.stripDomainFromUsername = $2;
		     }
	       ;

safe_search:
		  SAFE_SEARCH on_or_off
		     { ufdbNewGV.safeSearch = $2; }
	        ;

max_logfile_size:
		  MAX_LOGFILE_SIZE NUMBER
		     { 
		       ufdbNewGV.maxLogfileSize = strtoul( $2, NULL, 10 );
		       ufdbFree( $2 );
		       if (ufdbNewGV.maxLogfileSize < 2 * 1024 * 1024)		// minimum is 2 MB
		          ufdbNewGV.maxLogfileSize = 2 * 1024 * 1024;
		       if (ufdbNewGV.maxLogfileSize > 500000000000)	        // maximum is 500 GB
		          ufdbNewGV.maxLogfileSize = 500000000000;
                       ufdbGV.maxLogfileSize = ufdbNewGV.maxLogfileSize;        // we want immediate effect
		     }
	        ;

httpd_option:
		  PORT EQUAL NUMBER        { ufdbNewGV.httpdPort = atoi( $3 ); ufdbFree( $3 ); }
	        | INTERFACE EQUAL qidentifier { if (strcmp($3,"all")== 0)
						strcpy( ufdbNewGV.httpdInterface, "all" );    
					     else
					        ufdbLogFatalError( "http-server interface must be \"all\" or IP address" );
					     ufdbFree( $3 );
					   }
	        | INTERFACE EQUAL WORD     { if (strcmp($3,"all")== 0)
						strcpy( ufdbNewGV.httpdInterface, "all" );    
					     else
					        ufdbLogFatalError( "http-server interface must be \"all\" or IP address" );
					     ufdbFree( $3 );
					   }
	        | INTERFACE EQUAL IPV4ADDR { strcpy( ufdbNewGV.httpdInterface, $3 );       ufdbFree( $3 ); }
		| IMAGES EQUAL qidentifier { strcpy( ufdbNewGV.httpdImagesDirectory, $3 ); ufdbFree( $3 ); }
		| IMAGES EQUAL WORD        { strcpy( ufdbNewGV.httpdImagesDirectory, $3 ); ufdbFree( $3 ); }
		;

httpd_options:
		  httpd_option COMMA httpd_options
		| httpd_option
	        ;

http_server_def:
		  HTTP_SERVER START_BRACKET httpd_options STOP_BRACKET
	        ;

category: 
		  CATEGORY qidentifier { ufdbCategory( $2 ); }
		| CATEGORY AGGRESSIVE  { ufdbCategory( ufdbStrdup("aggressive") );
		                         yyerror( (char *) "\"aggressive\" is a keyword and must be surrounded by quotes" ); }
		| CATEGORY TOKEN_ALLOW { ufdbCategory( ufdbStrdup("allow") );
		                         yyerror( (char *) "\"allow\" is a keyword and must be surrounded by quotes" ); }
		| CATEGORY TOKEN_DENY  { ufdbCategory( ufdbStrdup("deny") );
		                         yyerror( (char *) "\"deny\" is a keyword and must be surrounded by quotes" ); }
		| CATEGORY error       { ufdbCategory( ufdbStrdup("syntax-error") );
		                         yyerror( (char *) "erroneous category definition.  Perhaps the category ID is a reserved word?" ); }
             	;

category_block: 
		  category START_BRACKET category_contents STOP_BRACKET 
                       { ufdbCategoryEnd(); }
                ;

category_contents: %empty
                | category_contents category_content
		;

category_content:  
 	      	  DOMAINLIST qidentifier                { ufdbCategoryDomainList( $2 ); }
 	      	| DOMAINLIST WORD                       { ufdbCategoryDomainList( $2 ); }
            	| DOMAINLIST CHAR_MINUS                 { ufdbCategoryDomainList( NULL ); }
		| EXECDOMAINLIST qidentifier            { ufdbCategoryExecDomainList( $2 ); }
            	| URLLIST qidentifier                   { ufdbCategoryUrlList( $2 ); }
            	| URLLIST WORD                          { ufdbCategoryUrlList( $2 ); }
            	| URLLIST CHAR_MINUS                    { ufdbCategoryUrlList( NULL ); }
            	| EXPRESSIONLIST qidentifier            { ufdbCategoryExpressionList( $2, "n" ); }
            	| EXPRESSIONLIST WORD                   { ufdbCategoryExpressionList( $2, "n" ); }
            	| EXPRESSIONLIST CHAR_MINUS             { ufdbCategoryExpressionList( NULL, NULL ); }
            	| EXPRESSIONLIST CHAR_I qidentifier     { ufdbCategoryExpressionList( $3, "i" ); }
            	| EXPRESSIONLIST CHAR_I WORD            { ufdbCategoryExpressionList( $3, "i" ); }
            	| EXPRESSIONLIST IGNORECASE qidentifier { ufdbCategoryExpressionList( $3, "i" ); }
            	| EXPRESSIONLIST IGNORECASE WORD        { ufdbCategoryExpressionList( $3, "i" ); }
		| CACERTS qidentifier                   { ufdbCategoryCACertsFile( $2 );  ufdbFree( $2 ); }
		| CACERTS WORD                          { ufdbCategoryCACertsFile( $2 );  ufdbFree( $2 ); }
		| CACERTSDIR qidentifier                { ufdbCategoryCACertsDir( $2 );  ufdbFree( $2 ); }
		| CACERTSDIR WORD                       { ufdbCategoryCACertsDir( $2 );  ufdbFree( $2 ); }
            	| REDIRECT qidentifier                  { ufdbCategoryRedirect( $2 ); }
            	| REDIRECT WORD                         { ufdbCategoryRedirect( $2 ); }
            	| REWRITE qidentifier                   { ufdbCategoryRewrite( $2 ); }
            	| REWRITE WORD                          { ufdbCategoryRewrite( $2 ); }
            	| WITHIN qidentifier                    { ufdbCategoryTime( $2, UFDB_ACL_WITHIN ); }
            	| OUTSIDE qidentifier                   { ufdbCategoryTime( $2, UFDB_ACL_OUTSIDE ); }
		| OPTION BLOCK_BUMPED_CONNECT on_or_off              { ufdbCategoryBlockConnect( $3 ); }
                | OPTION SQUID_USES_ACTIVE_BUMPING on_or_off         { ufdbCategoryActiveBumping( $3 ); }
		| OPTION SAFE_SEARCH on_or_off                       { ufdbCategoryOption( $3, UFDB_OPT_SAFE_SEARCH );  }
		| OPTION YOUTUBE_EDUFILTER on_or_off                 { ufdbCategoryOption( $3, UFDB_OPT_YOUTUBE_EDUFILTER );  }
		| OPTION ENFORCE_HTTPS_WITH_HOSTNAME                 { ufdbCategoryOption(  1, UFDB_OPT_HTTPS_WITH_HOSTNAME );  }
		| OPTION ENFORCE_HTTPS_WITH_HOSTNAME on_or_off       { ufdbCategoryOption( $3, UFDB_OPT_HTTPS_WITH_HOSTNAME );  }
		| OPTION ENFORCE_HTTPS_OFFICAL_CERTIFICATE           { ufdbCategoryOption(  1, UFDB_OPT_HTTPS_OFFICAL_CERTIFICATE );  }
		| OPTION ENFORCE_HTTPS_OFFICAL_CERTIFICATE on_or_off { ufdbCategoryOption( $3, UFDB_OPT_HTTPS_OFFICAL_CERTIFICATE );  }
		| OPTION HTTPS_PROHIBIT_INSECURE_SSLV2 on_or_off     { ufdbCategoryOption( $3, UFDB_OPT_PROHIBIT_INSECURE_SSLV2 );  }
		| OPTION HTTPS_PROHIBIT_INSECURE_SSLV3 on_or_off     { ufdbCategoryOption( $3, UFDB_OPT_PROHIBIT_INSECURE_SSLV3 );  }
		| OPTION HTTPS_PROHIBIT_INSECURE_TLSV1_0 on_or_off   { ufdbCategoryOption( $3, UFDB_OPT_PROHIBIT_INSECURE_TLSV1_0 );  }
		| OPTION ALLOW_SKYPE_OVER_HTTPS on_or_off	     { ufdbCategoryOption( $3, UFDB_OPT_SKYPE_OVER_HTTPS );  }
		| OPTION ALLOW_GTALK_OVER_HTTPS on_or_off	     { ufdbCategoryOption( $3, UFDB_OPT_GTALK_OVER_HTTPS );  }
		| OPTION ALLOW_YAHOOMSG_OVER_HTTPS on_or_off	     { ufdbCategoryOption( $3, UFDB_OPT_YAHOOMSG_OVER_HTTPS );  }
		| OPTION ALLOW_AIM_OVER_HTTPS on_or_off	             { ufdbCategoryOption( $3, UFDB_OPT_AIM_OVER_HTTPS );  }
		| OPTION ALLOW_FBCHAT_OVER_HTTPS on_or_off	     { ufdbCategoryOption( $3, UFDB_OPT_FBCHAT_OVER_HTTPS );  }
		| OPTION ALLOW_CITRIXONLINE_OVER_HTTPS on_or_off     { ufdbCategoryOption( $3, UFDB_OPT_CITRIXONLINE_OVER_HTTPS );  }
		| OPTION ALLOW_ANYDESK_OVER_HTTPS on_or_off          { ufdbCategoryOption( $3, UFDB_OPT_ANYDESK_OVER_HTTPS );  }
		| OPTION ALLOW_TEAMVIEWER_OVER_HTTPS on_or_off       { ufdbCategoryOption( $3, UFDB_OPT_TEAMVIEWER_OVER_HTTPS );  }
		| OPTION ALLOW_UNKNOWN_PROTOCOL_OVER_HTTPS on_or_off { ufdbCategoryOption( $3, UFDB_OPT_UNKNOWN_PROTOCOL_OVER_HTTPS );  }
            	| LOGFILE ANONYMOUS qidentifier { ufdbLogError( "line %d: unsupported logfile context for %s", lineno, $3 ); 
		                                  ufdbFree( $3 ); }
            	| LOGFILE ANONYMOUS WORD        { ufdbLogError( "line %d: unsupported logfile context for %s", lineno, $3 ); 
		                                  ufdbFree( $3 ); }
            	| LOGFILE qidentifier           { ufdbLogError( "line %d: unsupported logfile context for %s", lineno, $2 ); 
		                                  ufdbFree( $2 ); }
            	| LOGFILE WORD                  { ufdbLogError( "line %d: unsupported logfile context for %s", lineno, $2 ); 
		                                  ufdbFree( $2 ); }
            	;

source:      
		  SOURCE qidentifier { defSource( $2 ); }
             	;

source_block: 
		  source START_BRACKET source_contents STOP_BRACKET { defSourceEnd(); }
             	;

source_contents:
		  %empty
		| source_contents source_content
		;

source_content:       
		  DOMAIN domain
                | USER user 	
                | UNIX USER user 		     
                | USERLIST qidentifier 		 { ufdbSourceUserList( $2 ); } 
                | USERLIST WORD 		 { ufdbSourceUserList( $2 ); } 
                | UNIX USERLIST qidentifier      { ufdbSourceUserList( $3 ); } 
                | UNIX USERLIST WORD 	         { ufdbSourceUserList( $3 ); } 
                | EXECUSERLIST qidentifier 	 { ufdbSourceExecUserList( $2 ); } 
                | EXECIPLIST qidentifier 	 { ufdbSourceExecIPList( $2 ); } 
		| GROUP qidentifier              { ufdbSourceGroup( UFDB_GROUPTYPE_UNIX, $2 ); }
		| GROUP WORD                     { ufdbSourceGroup( UFDB_GROUPTYPE_UNIX, $2 ); }
		| UNIX GROUP qidentifier         { ufdbSourceGroup( UFDB_GROUPTYPE_UNIX, $3 ); }
		| UNIX GROUP WORD                { ufdbSourceGroup( UFDB_GROUPTYPE_UNIX, $3 ); }
                | USERQUOTA NUMBER NUMBER HOURLY { ufdbSourceUserQuota( $2, $3, "3600" );  
		                                   ufdbFree( $2 ); ufdbFree( $3 ); }
                | USERQUOTA NUMBER NUMBER DAILY  { ufdbSourceUserQuota( $2, $3, "86400" );  
		                                   ufdbFree( $2 ); ufdbFree( $3 ); }  
                | USERQUOTA NUMBER NUMBER WEEKLY { ufdbSourceUserQuota( $2, $3, "604800" );  
		                                   ufdbFree( $2 ); ufdbFree( $3 ); } 
                | USERQUOTA NUMBER NUMBER NUMBER { ufdbSourceUserQuota( $2, $3, $4 );  
		                                   ufdbFree( $2 ); ufdbFree( $3 ); ufdbFree( $4 ); } 
                | IPV4 ipv4s
                | IPV4LIST qidentifier         	 { defSourceIPV4List( $2 ); }
                | IPV4LIST WORD             	 { defSourceIPV4List( $2 ); }
                | IPV6 ipv6s
                | IPV6LIST qidentifier         	 { defSourceIPV6List( $2 ); }
                | IPV6LIST WORD         	 { defSourceIPV6List( $2 ); }
		| EVALUATE_AND                   { ufdbSourceEval( UFDB_EVAL_AND ); }
		| EVALUATE_OR                    { ufdbSourceEval( UFDB_EVAL_OR ); }
                | WITHIN qidentifier           	 { defSourceTime( $2, UFDB_ACL_WITHIN ); }
                | OUTSIDE qidentifier          	 { defSourceTime( $2, UFDB_ACL_OUTSIDE ); }
                | LOGFILE ANONYMOUS qidentifier	 { ufdbLogError( "line %d: unsupported logfile context for %s",
                                                                 lineno, $3 ); 
						   ufdbFree( $3 ); }
                | LOGFILE ANONYMOUS WORD  	 { ufdbLogError( "line %d: unsupported logfile context for %s",
                                                                 lineno, $3 ); 
						   ufdbFree( $3 ); }
                | LOGFILE qidentifier          	 { ufdbLogError( "line %d: unsupported logfile context for %s",
                                                                 lineno, $2 ); 
						   ufdbFree( $2 ); }
                | LOGFILE WORD            	 { ufdbLogError( "line %d: unsupported logfile context for %s",
                                                                 lineno, $2 ); 
						   ufdbFree( $2 ); }
                | CONTINUE                	 { ufdbNewGV.lastSource->cont_search = 1; }
                ;


domain:		    
		  %empty
		| domain qidentifier  	{ defSourceDomain( $2 ); }
		| domain WORD  		{ defSourceDomain( $2 ); }
                | domain COMMA
		;

user:		    
		  %empty
		| user qidentifier	{ ufdbSourceUser( $2 ); }
		| user WORD    		{ ufdbSourceUser( $2 ); }
                | user COMMA
		;

acl_block: 
		  ACL START_BRACKET acl_contents STOP_BRACKET 
                ;

acl_contents:     %empty
                | acl_contents acl_content
	        ;

acl_header:       
		  qidentifier                     START_BRACKET  { ufdbAcl( $1, NULL, UFDB_ACL_NONE );  }
		| qidentifier WITHIN qidentifier  START_BRACKET  { ufdbAcl( $1, $3, UFDB_ACL_WITHIN );  }
                | qidentifier OUTSIDE qidentifier START_BRACKET  { ufdbAcl( $1, $3, UFDB_ACL_OUTSIDE ); } 
                ;

acl_content:      
		  acl_header access_contents STOP_BRACKET
                | acl_header access_contents STOP_BRACKET ELSE
                     { ufdbAcl( NULL, NULL, UFDB_ACL_ELSE );    } 
                     START_BRACKET access_contents STOP_BRACKET
                ;

access_contents:  
		  %empty
                | access_contents access_content
                ;

access_content:     
		  PASS access_pass              { if (ufdbNewGV.lastAcl != NULL  &&  
                                                      ufdbNewGV.lastAcl->pass == NULL) 
                                                  {
                                                     ufdbLogMessage( "line %d: acl has an empty pass statement."
                                                                     "  Adding 'any'.", lineno );
                                                     ufdbAclSetValue( "pass", ufdbStrdup("any"), 1 );
                                                  }
                                                }
                | REWRITE qidentifier           { ufdbAclSetValue( "rewrite", $2, 0 ); }
                | REWRITE WORD                  { ufdbAclSetValue( "rewrite", $2, 0 ); }
                | REDIRECT qidentifier          { ufdbAclSetValue( "redirect", $2, 0 ); }
                | REDIRECT WORD                 { ufdbAclSetValue( "redirect", $2, 0 ); }
                | LOGFILE ANONYMOUS qidentifier { ufdbLogError( "line %d: unsupported logfile context for %s",
                                                                lineno, $3 ); 
						  ufdbFree( $3 ); }
                | LOGFILE ANONYMOUS WORD        { ufdbLogError( "line %d: unsupported logfile context for %s",
                                                                lineno, $3 ); 
						  ufdbFree( $3 ); }
                | LOGFILE qidentifier           { ufdbLogError( "line %d: unsupported logfile context for %s",
                                                                lineno, $2 ); 
						  ufdbFree( $2 ); }
                | LOGFILE WORD                  { ufdbLogError( "line %d: unsupported logfile context for %s",
                                                                lineno, $2 ); 
						  ufdbFree( $2 ); }
                ;

access_pass:      
		  %empty
		| access_pass AGGRESSIVE		    { yyerror( (char *) "\"aggressive\" is a keyword and must be surrounded by quotes" ); }
                | access_pass qidentifier     		    { ufdbAclSetValue( "pass", $2, 1 ); }
                | access_pass CHAR_EXCLAMATION AGGRESSIVE   { yyerror( (char *) "\"aggressive\" is a keyword and must be surrounded by quotes" ); }
                | access_pass CHAR_EXCLAMATION qidentifier  { ufdbAclSetValue( "pass", $3, 0 ); }
		| access_pass COMMA
                ;

ipv4s: 		  
		  %empty
                | ipv4s ipv4
                | ipv4s COMMA ipv4
		;

ipv4:  		  
		  IPV4ADDR 		        { sgIpv4( $1, SG_IPTYPE_HOST, ufdbNewGV.configFile, lineno );
		                                  ufdbFree( $1 );  }
		| IPV4RANGE			{ sgIpv4( $1, SG_IPTYPE_RANGE, ufdbNewGV.configFile, lineno );
		                                  ufdbFree( $1 );  }
		| IPV4CLASS			{ sgIpv4( $1, SG_IPTYPE_CLASS, ufdbNewGV.configFile, lineno );
		                                  ufdbFree( $1 ); }
		| IPV4NET			{ sgIpv4( $1, SG_IPTYPE_CIDR, ufdbNewGV.configFile, lineno );
		                                  ufdbFree( $1 ); }
     		;

ipv6s: 		  
		  %empty
                | ipv6s ipv6  
                | ipv6s COMMA ipv6
		;

ipv6:  		  
		  IPV6ADDR 		        { sgIpv6( $1, SG_IPV6TYPE_HOST, ufdbNewGV.configFile, lineno );
                                                  ufdbFree( $1 );  }
                | IPV6NET                       { sgIpv6( $1, SG_IPV6TYPE_CIDR, ufdbNewGV.configFile, lineno );
                                                  ufdbFree( $1 );  }
     		;

rew:       	  
		  REWRITE qidentifier  	{ sgRewrite( $2 ); }
		| REWRITE WORD 		{ sgRewrite( $2 ); } 
		;

rew_block:  	  
		  rew START_BRACKET rew_contents STOP_BRACKET 
             	;

rew_contents:     
		  %empty
		| rew_contents rew_content
		;


rew_content:      
		  SUBST                         { sgRewriteSubstitute( $1 ); ufdbFree( $1 ); }
                | WITHIN qidentifier            { sgRewriteTime( $2, UFDB_ACL_WITHIN ); }
                | OUTSIDE qidentifier           { sgRewriteTime( $2, UFDB_ACL_OUTSIDE ); }
                | LOGFILE ANONYMOUS qidentifier { ufdbLogError( "line %d: unsupported logfile context for %s",
		                                                lineno, $3 );
								ufdbFree( $3 ); }
                | LOGFILE ANONYMOUS WORD        { ufdbLogError( "line %d: unsupported logfile context for %s",
		                                                lineno, $3 );
								ufdbFree( $3 ); }
                | LOGFILE qidentifier           { ufdbLogError( "line %d: unsupported logfile context for %s",
		                                                lineno, $2 );
								ufdbFree( $2 ); }
                | LOGFILE WORD                  { ufdbLogError( "line %d: unsupported logfile context for %s",
		                                                lineno, $2 );
								ufdbFree( $2 ); }
                ;


time:       	  
		  TIME qidentifier 		{ sgTime( $2 ); } 
		;

time_block:  	  
		  time START_BRACKET time_contents STOP_BRACKET 
                ;

time_contents:    
		  %empty
		| time_contents time_content
		;


time_content:     
                  WEEKLY { sgTimeElementInit(); } WEEKDAY     { sgTimeElementAdd($3,T_WEEKDAY); } ttime
		| WEEKLY { sgTimeElementInit(); } qidentifier { sgTimeElementAdd($3,T_WEEKLY); }  ttime
		| WEEKLY { sgTimeElementInit(); } WORD        { sgTimeElementAdd($3,T_WEEKLY); }  ttime
                | DATE   { sgTimeElementInit(); } date        { sgTimeElementEnd(); }
		| error  { ufdbLogFatalError( "invalid time specification at line %d", lineno );   }
                ;

ttime:            
		  ttime  { sgTimeElementClone(); }  tval CHAR_MINUS tval
		| tval CHAR_MINUS tval 
                ;

date:             
		  dval ttime
                | dval 
                | dval  CHAR_MINUS dval ttime
                | dval  CHAR_MINUS dval 
                | dvalcron  ttime
                | dvalcron
                ;

dval:		  DVAL 		{ sgTimeElementAdd( $1, T_DVAL ); }
                ;

tval:		  TVAL 		{ sgTimeElementAdd( $1, T_TVAL ); }
                ;

dvalcron:	  DVALCRON 	{ sgTimeElementAdd( $1, T_DVALCRON ); }
                ;

an_error:
		  error  	{  yyerror( (char *) "syntax error" );  
                                   if (ufdbNewGV.serialno > 1  &&  ufdbNewGV.debug)
                                      logConfig(); 
                                }
	        ;

statements:       %empty
       		| statements statement
       		;

statement:   
                  category
	     	| source_block
	     	| category_block
		| http_server_def
	     	| admin_spec
	     	| https_cache_size
	     	| check_proxy_tunnels
	     	| log_pass
	     	| log_block
	     	| log_all
		| logall_httpd
                | madvise_hugepages
                | fast_refresh
		| debug_filter
		| debug_coredump
		| enforce_https_with_hostname
		| enforce_https_offical_certificate
		| https_prohibit_insecure_sslv2
		| https_prohibit_insecure_sslv3
		| https_prohibit_insecure_tlsv1_0
             	| dbhome
	     	| logdir
		| pidfile
		| mail_server
		| my_hostname
		| admin_email
		| sender_email
		| external_status_command
		| url_lookup_delay_db_reload
		| url_lookup_result_db_reload
		| url_lookup_result_fatal_error
		| squid_version
		| num_worker_threads
		| upload_crash_reports
		| lookup_reverse_ip
		| use_ipv6_on_wan
                | parse_url_parameters
		| squid_uses_active_bumping
		| ufdb_log_url_details
		| ufdb_show_url_details
		| ufdb_debug_skype_probes
		| ufdb_debug_gtalk_probes
		| ufdb_debug_yahoomsg_probes
		| ufdb_debug_aim_probes
		| ufdb_debug_fbchat_probes
		| ufdb_debug_citrixonline_probes
		| ufdb_expression_optimisation
		| ufdb_expression_debug
		| refreshuserlist
		| refreshiplist
		| refreshdomainlist
		| ufdb_debug_external_scripts
		| interface
	     	| port
	     	| cpus
	     	| upload_stats
		| redirect_https
		| redirect_bumped_https
		| redirect_loading_database
		| redirect_fatal_error
	     	| analyse_uncategorised
		| log_uncategorised_urls
		| safe_search
		| strip_domain_from_username
		| youtube_edufilter
		| allow_google_https_using_ip
		| youtube_edufilter_id
		| max_logfile_size
	     	| acl_block
	     	| rew_block
	     	| time_block
	     	| NL
	     	| an_error
             	;

%%

int ufdbReadConfig( 
   const char *      file )
{
   struct Source *   source;
   struct Category * cat;
   struct Acl *      acl;

   lineno = 1;
   ufdbResetCPUs();

   ufdbNewGV.configFile = file;

   yyin = fopen( ufdbNewGV.configFile, "r" );
   if (yyin == NULL) 
   {
      syslog( LOG_ALERT, "%s: cannot open configuration file %s", ufdbNewGV.progname, ufdbNewGV.configFile );
      ufdbLogFatalError( "cannot open configuration file %s", ufdbNewGV.configFile );
      return 0;
   }

   ufdbLogMessage( "configuration file: %s", ufdbNewGV.configFile );

   /* keep the old HTTPS redirection URL.  It is most likely that we need it during the database reload */
   /* dot not reset ufdbNewGV.RedirectHttps and ufdbNewGV.RedirectBumpedHttps. */

   ufdbNewGV.debug = 0;
   ufdbNewGV.debugCoreDump = 0;
   strcpy( ufdbNewGV.interface, "all" );
   ufdbNewGV.portNum = UFDB_DAEMON_PORT;
   ufdbNewGV.nWorkers = UFDB_MIN_THREADS;
   ufdbNewGV.debugHttpd = 0;
   ufdbNewGV.logPass = 0;
   ufdbNewGV.logBlock = 0;
   ufdbNewGV.logAllRequests = 0;
   ufdbNewGV.YoutubeEdufilter = 0;
   ufdbFree( ufdbNewGV.YoutubeEdufilterID );
   ufdbNewGV.YoutubeEdufilterID = NULL;
   ufdbNewGV.allowGoogleHTTPSusingIP = 0;

   /* When a database is reloaded, these variables may NOT be reset since they are used during the reload !! */
   ufdbNewGV.URLlookupDelayDBreload = 0;
   ufdbNewGV.URLlookupResultDBreload = UFDB_ALLOW;
   ufdbNewGV.URLlookupResultFatalError = UFDB_ALLOW;
   strcpy( ufdbNewGV.loadingDatabaseRedirect, "http://cgibin.urlfilterdb.com/cgi-bin/URLblocked.cgi?"
                "category=loading-database" );
   ufdbNewGV.SquidVersion = ufdbStrdup( UFDB_DEFAULT_SQUID_VERSION );

   ufdbNewGV.showURLdetails = 0;
   ufdbNewGV.logURLdetails = 0;
   ufdbNewGV.uploadCrashReports = 1;
   ufdbNewGV.lookupReverseIP = 0;
   ufdbNewGV.useAlsoIPv6onWan = 1;
   ufdbNewGV.debugSkype = 0;
   ufdbNewGV.debugGtalk = 0;
   ufdbNewGV.debugYahooMsg = 0;
   ufdbNewGV.debugAim = 0;
   ufdbNewGV.debugFBchat = 0;
   ufdbNewGV.debugCitrixOnline = 0;
   ufdbNewGV.debugRegexp = 0;
   ufdbNewGV.debugExternalScripts = 0;
   ufdbNewGV.expressionOptimisation = 1;
   ufdbNewGV.refreshUserlistInterval = UFDB_DEFAULT_REFRESH_USERLIST;
   ufdbNewGV.refreshIPlistInterval = UFDB_DEFAULT_REFRESH_IPLIST;
   ufdbNewGV.refreshDomainlistInterval = UFDB_DEFAULT_REFRESH_DOMAINLIST;
   ufdbNewGV.analyseUncategorisedURLs = 1;
   ufdbNewGV.logUncategorisedURLs = 0;
   ufdbNewGV.uploadStats = 1;
   ufdbNewGV.safeSearch = 1;
   ufdbNewGV.skipSafeCategory = 0;
   ufdbNewGV.stripDomainFromUsername = 0;
   ufdbNewGV.tunnelCheckMethod = UFDB_API_HTTPS_CHECK_OFF;
   ufdbNewGV.SquidUsesActiveBumping = 0;
   ufdbNewGV.httpsWithHostname = 0;
   ufdbNewGV.httpsOfficialCertificate = 0;
   ufdbNewGV.unknownProtocolOverHttps = 1;
   ufdbNewGV.httpsNoSSLv2 = 1;
   ufdbNewGV.httpsNoSSLv3 = 1;
   ufdbNewGV.httpdPort = 0;
   ufdbNewGV.dateOfCheckedDB[0] = '\0';
   strcpy( ufdbNewGV.httpdInterface, "all" );
   strcpy( ufdbNewGV.httpdImagesDirectory, "." );
 
   ufdbNewGV.databaseStatus = UFDB_API_STATUS_DATABASE_OK;

   (void) yyparse();		/* parse the configuration file ********************/
   fclose( yyin );

   ufdbNewGV.databaseLoadTime = time( NULL );

   /*
    * For analysis of uncategorised URLs we also load a "checked" category
    * that contains URLs that are reviewed and checked not to be part of
    * any other category.
    */
   char * dbhome;
   char   dbfname[1024];

   dbhome = ufdbNewGV.databaseDirectory;

   /* TODO: check if "checked" was not already loaded... */
   if (ufdbNewGV.analyseUncategorisedURLs > 0)
   {
      int status;
      sprintf( dbfname, "%s/checked/domains.ufdb", dbhome );
      status = UFDBloadDatabase( &ufdbNewGV, &ufdbNewGV.checkedDB, dbfname );
      if (status != UFDB_API_OK  &&  status != UFDB_API_STATUS_DATABASE_OLD)
      {
         ufdbNewGV.checkedDB.mem = NULL;
         ufdbNewGV.checkedDB.index = NULL;
         ufdbNewGV.checkedDB.table.nNextLevels = 0;
         if (ufdbGV.debug)
            ufdbLogMessage( "The URL database appears not be from URLfilterDB. No problem." );
      }
      else
      {
         if (ufdbGV.debug)
            ufdbLogMessage( "table \"checked\" is loaded: %s %8.8s",
                            ufdbNewGV.checkedDB.date, ufdbNewGV.checkedDB.flags );
         sprintf( dbfname, "%s/checked/expressions", dbhome );
         (void) UFDBloadExpressions( &ufdbNewGV.checkedExpressions, dbfname );
         ufdbLogMessage( "The implicitly allowed URL category 'checked' is loaded." );
      }
   }

   if (ufdbNewGV.defaultAcl == NULL)
      ufdbLogFatalError( "\"default\" ACL is not defined" );
   else if (ufdbNewGV.defaultAcl->pass == NULL)
      ufdbLogError( "\"default\" ACL is empty: by default all URLs are blocked." );

   /* A configuration file may have a source definition which is not used in the ACL list.
    * This raises the question "what to do with this ?".
    * If we do nothing, the source is matched but has no ACL definition and is silently skipped.
    * Most likely the administrator overlooked something so a warning is useful.
    */
   source = ufdbNewGV.sourceList;
   if (source == NULL  &&  ufdbNewGV.defaultAcl == NULL)
      ufdbLogFatalError( "There are no sources and there is no default ACL" );

   for ( ; source != NULL;  source = source->next)
   {
      int found;

      found = 0;
      for (acl = ufdbNewGV.aclList;  acl != NULL;  acl = acl->next)
      {
         if (strcmp( acl->name, source->name ) == 0)
	 {
	    found = 1;
	    break;
	 }
      }
      if (!found)
      {
         ufdbLogError( "source \"%s\" has no ACL.  *****\n"
	               "This most likely may lead to unexpected results when the source is matched.  *****\n"
		       "It is strongly suggested to remove the source definition OR add the source to the acl.  *****",
		       source->name );
	 source->active = 0;
      }
      else if (acl->pass == NULL)
      {
         ufdbLogMessage( "source \"%s\" has an empty \"pass\" in its ACL *****", source->name );
      }
   }

   for (cat = ufdbNewGV.catList;  cat != NULL;  cat = cat->next)
   {
      if (cat->options == 0  &&  cat->domainlist == NULL  &&  cat->execdomainlist == NULL  && 
          cat->expressionlist == NULL  &&  cat->rewrite == NULL  &&  cat->name != NULL)
      {
         ufdbLogError( "category \"%s\" has no content definition  *****", cat->name );
      }
   }

   if (ufdbNewGV.adminEmail != NULL  &&  ufdbNewGV.emailServer == NULL)
   {
      ufdbLogError( "No email server is defined; cannot send email to \"%s\"  *****", ufdbNewGV.adminEmail );
   }
   if (ufdbNewGV.adminEmail != NULL  &&  ufdbNewGV.emailServer != NULL  &&  ufdbNewGV.senderEmail == NULL)
   {
      ufdbLogMessage( "WARNING: No sender email address is defined; using \"%s\" as sender",
                      ufdbNewGV.adminEmail );
   }

   if (strncmp( ufdbNewGV.SquidVersion, "6.", 2 ) != 0  &&
       strncmp( ufdbNewGV.SquidVersion, "5.", 2 ) != 0  &&
       strncmp( ufdbNewGV.SquidVersion, "4.", 2 ) != 0  &&
       strncmp( ufdbNewGV.SquidVersion, "3.5", 3 ) != 0  &&
       strncmp( ufdbNewGV.SquidVersion, "3.4", 3 ) != 0  &&
       strncmp( ufdbNewGV.SquidVersion, "3.3", 3 ) != 0  &&
       strncmp( ufdbNewGV.SquidVersion, "3.2", 3 ) != 0  &&
       strncmp( ufdbNewGV.SquidVersion, "3.1", 3 ) != 0  &&
       strncmp( ufdbNewGV.SquidVersion, "3.0", 3 ) != 0  &&
       strncmp( ufdbNewGV.SquidVersion, "2.7", 3 ) != 0  &&
       strncmp( ufdbNewGV.SquidVersion, "2.6", 3 ) != 0)
   {
      ufdbLogFatalError( "Unsupported Squid version \"%s\"  *****\n"
                         "The supported values for squid-version are: "
                         "6.x, 5.x, 4.x, 3.5, 3.4, 3.3, 3.2, 3.1, 3.0, 2.7, 2.6  *****\n"
			 "If the version of Squid is higher, it is recommended to upgrade ufdbGuard.  *****\n"
			 "Alternatively set squid-version to the highest supported version",
			 ufdbNewGV.SquidVersion );
   }
   if (strncmp( ufdbNewGV.SquidVersion, "6.", 2 ) == 0  ||
       strncmp( ufdbNewGV.SquidVersion, "5.", 2 ) == 0  ||
       strncmp( ufdbNewGV.SquidVersion, "4.", 2 ) == 0  ||  
       strncmp( ufdbNewGV.SquidVersion, "3.5", 3 ) == 0  ||  
       strncmp( ufdbNewGV.SquidVersion, "3.4", 3 ) == 0)
      ufdbNewGV.SquidHelperProtocol = UFDB_SQUID_HELPER_PROTOCOL3;
   else if (strncmp( ufdbNewGV.SquidVersion, "2.6", 3 ) == 0  ||  
            strncmp( ufdbNewGV.SquidVersion, "2.7", 3 ) == 0)
      ufdbNewGV.SquidHelperProtocol = UFDB_SQUID_HELPER_PROTOCOL1;
   else
      ufdbNewGV.SquidHelperProtocol = UFDB_SQUID_HELPER_PROTOCOL2;	/* 3.0 - 3.3 */

   BuildImplicitPassLists( &ufdbNewGV );

   return 1;
}


static void setDBhome( char * dbhome )
{
   struct stat dirbuf;

   if (strlen(dbhome) > sizeof(ufdbNewGV.databaseDirectory)-1)
   {
      ufdbLogFatalError( "maximum length for dbhome is %d", (int) sizeof(ufdbNewGV.databaseDirectory)-1 );
      ufdbFree( dbhome );
      return;
   }
   strcpy( ufdbNewGV.databaseDirectory, dbhome );

   if (stat( dbhome, &dirbuf ) != 0)
   {
      ufdbLogFatalError( "dbhome %s: directory does not exist or access rights are insufficient", dbhome );
   }
   else
   {
      if (!S_ISDIR(dirbuf.st_mode))
         ufdbLogFatalError( "dbhome: %s is not a directory", dbhome );
   }

   ufdbFree( dbhome );
}


static void setAdministrator( char * value )
{
   char * p;

   if (strlen(value) > sizeof(ufdbNewGV.administrator)-1)
   {
      ufdbLogError( "maximum length for administrator is %d", (int) sizeof(ufdbNewGV.administrator)-1 );
      return;
   }

   while ((p = (char *) strchr( value, '?' )) != NULL)
      *p = '_';
   while ((p = (char *) strchr( value, '&' )) != NULL)
      *p = '_';

   strcpy( ufdbNewGV.administrator, value );
   ufdbFree( value );
}


static void setLogdir( char * value )
{
   strcpy( ufdbNewGV.logDirectory, value );

   ufdbSetGlobalErrorLogFile( ufdbNewGV.logDirectory, NULL, 0 );
}


static void setPidfile( char * value )
{
   if (ufdbNewGV.pidFilename != NULL)
      ufdbFree( ufdbNewGV.pidFilename );
   ufdbNewGV.pidFilename = value;
}


static void setRefreshUserlist( int nmin )
{
   if (nmin < 5  ||  nmin > 24*60)
   {
      ufdbLogError( "refreshuserlist must have a value between 5 and %d, resetting to %d.", 
                    24*60, UFDB_DEFAULT_REFRESH_USERLIST );
      nmin = UFDB_DEFAULT_REFRESH_USERLIST;
   }
   ufdbNewGV.refreshUserlistInterval = nmin;
}


static void setRefreshIPlist( int nmin )
{
   if (nmin < 5  ||  nmin > 24*60)
   {
      ufdbLogError( "refreshiplist must have a value between 5 and %d, resetting to %d.", 
                    24*60, UFDB_DEFAULT_REFRESH_IPLIST );
      nmin = UFDB_DEFAULT_REFRESH_IPLIST;
   }
   ufdbNewGV.refreshIPlistInterval = nmin;
}


static void setRefreshDomainlist( int nmin )
{
   if (nmin < 5  ||  nmin > 24*60)
   {
      ufdbLogError( "refreshdomainlist must have a value between 5 and %d, resetting to %d.", 
                    24*60, UFDB_DEFAULT_REFRESH_DOMAINLIST );
      nmin = UFDB_DEFAULT_REFRESH_DOMAINLIST;
   }
   ufdbNewGV.refreshDomainlistInterval = nmin;
}


/*
 * Source functions
 */

static void defSource( 
   char *          source )
{
   struct Source * sp;

   if (ufdbGV.debug)
      ufdbLogMessage( "defSource: defining source \"%s\"", source );

   if ((struct Source *) defSourceFindName(ufdbNewGV.sourceList,source) != NULL)
   {
      ufdbLogFatalError( "line %d: source %s is already defined in configuration file %s",
			 lineno, source, ufdbNewGV.configFile );
      ufdbFree( source );
      return;
   }

   sp = (struct Source *) ufdbMallocAligned( UFDB_CACHELINE_SIZE, sizeof(struct Source) );
   sp->name = source;
   sp->active = 1;
   sp->evaluationMethod = UFDB_EVAL_OR;
   sp->ipv4hosts = NULL;
   sp->ipv4 = NULL;
   sp->ipv6hosts = NULL;
   sp->ipv6 = NULL;
   sp->domainDb = NULL;
   sp->userDb = NULL;
   sp->time = NULL;
   sp->within = UFDB_ACL_NONE;
   sp->cont_search = 0;
   sp->sarg0 = NULL;
   sp->execiplistCommand = NULL;
   sp->next = NULL;
   sp->nblocks = 0;
   sp->nmatches = 0;

   if (ufdbNewGV.sourceList == NULL)
   {
      if (ufdbGV.debug)
         ufdbLogMessage( "defSource: SourceList is set to point to \"%s\"", source );
      ufdbNewGV.sourceList = sp;
      ufdbNewGV.lastSource = sp;
   }
   else
   {
      ufdbNewGV.lastSource->next = sp;
      ufdbNewGV.lastSource = sp;
   }
}


static void defSourceEnd( void )
{
   struct Source * s;

   s = ufdbNewGV.lastSource;
   if (s->ipv4 == NULL  &&  s->ipv4hosts == NULL  &&  
       s->ipv6 == NULL  &&  s->ipv6hosts == NULL  &&
       s->domainDb == NULL  &&  s->userDb == NULL)
   {
      if (s->execiplistCommand == NULL)
      {
	 ufdbLogError( "source \"%s\" missing active content, set inactive  *****", s->name );
	 s->time = NULL;
	 s->active = 0;
      }
      else
	 ufdbLogError( "source \"%s\" has an execiplist command but the list is empty and set inactive   *****",
	               s->name );
   }
}


static void ufdbSourceUser( 
   char *          user )
{
   char *          lc;
   struct Source * sp;

   sp = ufdbNewGV.lastSource;
   if (sp->userDb != NULL  &&  sp->userDb->type != SGDBTYPE_USERLIST)
   {
      ufdbLogFatalError( "user \"%s\" in source \"%s\" on line %d: "
                         "found mix of exec and non-exec userlists", 
                         user, sp->name, lineno );
      return;
   }
   if (sp->userDb == NULL)
   {
      sp->userDb = (struct sgDb *) ufdbMalloc( sizeof(struct sgDb) );
      sp->userDb->dbhome = NULL;
      sp->userDb->dbcp = (void *) UFDBmemDBinit();
      sp->userDb->type = SGDBTYPE_USERLIST;
      sp->userDb->entries = 0;
   }

   for (lc = user;  *lc != '\0';  lc++)    // convert username to lowercase
   {
      if (*lc <= 'Z'  &&  *lc >= 'A')
         *lc += 'a' - 'A';
   }

   if (ufdbGV.debug > 1)
      ufdbLogMessage( "ufdbSourceUser: adding user \"%s\"", user );

   sp->userDb->entries++;
   UFDBmemDBinsert( (struct UFDBmemDB *) sp->userDb->dbcp, user, NULL );
   ufdbFree( user );
}


static void ufdbSourceUnixGroup(
   char *          groupName  )
{
   struct Source * sp;
   struct group *  grp;
   int             n;
   char *          user;

   sp = ufdbNewGV.lastSource;
   if (sp->userDb != NULL  &&  sp->userDb->type != SGDBTYPE_USERLIST)
   {
      ufdbLogFatalError( "unix group \"%s\" in source \"%s\" on line %d: "
                         "found mix of exec and non-exec userlists", 
                         groupName, sp->name, lineno );
      return;
   }
   if (sp->userDb == NULL) 
   {
      sp->userDb = (struct sgDb *) ufdbMalloc( sizeof(struct sgDb) );
      sp->userDb->dbhome = NULL;
      sp->userDb->dbcp = (void *) UFDBmemDBinit();
      sp->userDb->type = SGDBTYPE_USERLIST;
      sp->userDb->entries = 0;
   }
 
   n = 0;
   setgrent();
   errno = 0;
   grp = getgrnam( groupName );
   if (grp != NULL)
   {
      if (ufdbGV.debug > 1)
         ufdbLogMessage( "found group \"%s\"", grp->gr_name );

      while ((user = grp->gr_mem[n]) != NULL)
      {
	 char * lc;

	 if (ufdbGV.debug > 1)
	    ufdbLogMessage( "group \"%s\" has user \"%s\"", grp->gr_name, user );
	 for (lc = user;  *lc != '\0';  lc++)  	// convert username to lowercase
	 {
	    if (*lc <= 'Z'  &&  *lc >= 'A')
	       *lc += 'a' - 'A';
	 }
	 UFDBmemDBinsert( (struct UFDBmemDB *) sp->userDb->dbcp, user, NULL );
	 sp->userDb->entries++;
	 n++;
      }
      ufdbLogMessage( "source \"%s\": unix group \"%s\" has %d members", sp->name, groupName, n );
      if (n == 0)
	 ufdbLogMessage( "WARNING: source \"%s\": \"%s\" is an empty group  +++++", sp->name, groupName );
   }
   else
   {
      ufdbLogFatalError( "source \"%s\": \"%s\" is not a unix group", sp->name, groupName );
      if (errno)
	 ufdbLogError( "ufdbSourceUnixGroup: getgrnam returned error %d: %s", errno, strerror(errno) );
   }
   endgrent();
}


static void ufdbSourceGroup( 
   int    groupType,
   char * groupName  )
{
   switch (groupType)
   {
   case UFDB_GROUPTYPE_UNIX:
      ufdbSourceUnixGroup( groupName );
      break;
   default:     
      ufdbLogFatalError( "ufdbSourceGroup: unknown group type %d", groupType );
   }

   ufdbFree( groupName );
}


struct sgDb * UFDBretrieveExecDomainlist( struct Category * cat )       // uses ufdbGV
{
   char *        t;
   FILE *        fp;
   struct sgDb * newdb;
   int           exitcode;
   struct stat   Stat0;
   struct stat   Stat1;
   char          line[1024];
   char          basefilename[1024];
   char          fullfilename[1024];
   char          command[3000];

   if (cat == NULL)
   {
      ufdbLogFatalError( "UFDBretrieveExecDomainlist: category is NULL" );
      return NULL;
   }

   if (cat->domainlist == NULL)
   {
      ufdbLogError( "will not execute command for category %s since domainlist is undefined", cat->name );
      return NULL;
   }

   if (cat->execdomainlist == NULL)
   {
      ufdbLogError( "can't execute command for category %s since execdomainlist is undefined", cat->name );
      return NULL;
   }

   if (cat->domainlist[0] == '/')
   {
      strcpy( basefilename, cat->domainlist );
   }
   else
   {
      char * dbhome;
      dbhome = ufdbGV.databaseDirectory;
      sprintf( basefilename, "%s/%s", dbhome, cat->domainlist );
   }
   strcpy( fullfilename, basefilename );
   strcat( fullfilename, UFDBfileSuffix );

   if (stat( fullfilename, &Stat0 ) != 0)
   {
      Stat0.st_mtime = 0;
      ufdbLogMessage( "category %s: pre-execdomainlist: domainlist does not exist (%s)",
                      cat->name, fullfilename );
   }

   /* The script that we will call needs the directory where the files reside,
    * so the popen-command contains a "cd" just before executing the script.
    * (note that we are multithreaded and cannot use chdir() here)
    */
   strcpy( command, "cd " );
   strcat( command, basefilename );
   t = strrchr( command, '/' );
   if (t != NULL)
      *t = '\0';
   strcat( command, " ; " );
   strcat( command, cat->execdomainlist );

   errno = 0;
   if ((fp = popen(command,"r")) == NULL)
   {
      ufdbLogFatalError( "category %s: can't execute command of execdomainlist \"%s\": popen failed: %s  *****",
                         cat->name, cat->execdomainlist, strerror(errno) );
      return NULL;
   }

   while (UFDBfgetsNoNL(line,sizeof(line)-1,fp) != NULL)
   {
      ufdbLogMessage( "execdomainlist for %s produced: %s", cat->name, line );
   }
   errno = 0;
   exitcode = pclose( fp );

   if (exitcode == -1  &&  errno == ECHILD)
      exitcode = 0;
      /* most likely the child handler thread already did a wait() */

   if (exitcode == 0)
   {
      if (ufdbGV.debug || ufdbGV.debugExternalScripts)
         ufdbLogMessage( "   execdomainlist for %s: exit 0 (OK)", cat->name );
   }
   else if (exitcode == -1)
   {
      ufdbLogError( "execdomainlist for %s: pclose error: %d (%s)", cat->name, errno, strerror(errno) );
      return NULL;
   }
   else if (exitcode > 0)
   {
      ufdbLogError( "execdomainlist for %s: command (%s) terminated with an error exit code %d", 
                    cat->name, cat->execdomainlist, exitcode );
      return NULL;
   }

   /* The command defined by 'execdomainlist' has been executed and the exit code was 0 (OK), so
    * now it is verified if the file timestamp of domains.ufdb is newer.
    */

   if (stat( fullfilename, &Stat1 ) != 0)
   {
      ufdbLogError( "execdomainlist for %s: command failed to produce a .ufdb file (%s)",
                    cat->name, fullfilename );
      return NULL;
   }

   newdb = (struct sgDb *) ufdbCalloc( sizeof(struct sgDb), 1 );
   newdb->dbhome = NULL;
   newdb->dbcp = NULL;
   newdb->type = SGDBTYPE_DOMAINLIST;

   sgDbInit( newdb, basefilename );
   if (newdb->dbcp == NULL)
   {
      ufdbLogError( "execdomainlist for %s: command produced a .ufdb file but I failed to load it", cat->name );
      ufdbFree( newdb );
      return NULL;
   }

   if (ufdbGV.debug || ufdbGV.debugExternalScripts)
      ufdbLogMessage( "execdomainlist for %s: new URL table is loaded", cat->name );

   return newdb;
}


static void ufdbSourceExecUserList(
   char *          command )            // must be malloced
{
   time_t          t0, te;
   struct Source * sp;

   /* execuserlist is used to dynamically retrieve a list of usernames and
    * is executed every 15 minutes to refresh the list of usernames.
    */

   sp = ufdbNewGV.lastSource;
   if (sp->userDb != NULL  &&  sp->userDb->type != SGDBTYPE_EXECUSERLIST)
   {
      ufdbLogFatalError( "execuserlist \"%s\" in source \"%s\" on line %d: "
                         "found mix of exec and non-exec userlists", 
                         command, sp->name, lineno );
      return;
   }
   if (sp->userDb != NULL)
   {
      ufdbLogFatalError( "execuserlist \"%s\" in source \"%s\" on line %d: "
                         "more than one execuserlist is not supported.  Merge the userlists in the script.", 
                         command, sp->name, lineno );
      return;
   }
   ufdbLogMessage( "execuserlist \"%s\"", command );

   t0 = time( NULL );
   sp->userDb = UFDBretrieveExecUserlist( &ufdbNewGV, command );
   sp->sarg0 = command;				/* command is already malloc-ed */
   te = time( NULL );

   if (te - t0 > 4)
      ufdbLogMessage( "WARNING: it took %ld seconds to execute \"%s\"  *****", (long) (te - t0), command );
 
   if (ufdbGV.debug>1 || ufdbGV.debugExternalScripts)
   {
      UFDBmemDBprintUserDB( "user", (struct UFDBmemDB *) sp->userDb->dbcp );
   }
}


static void ufdbSourceExecIPList(
   char *          command )            // must be malloced
{
   time_t          t0, te;
   struct Source * sp;

   /* execiplist is used to dynamically retrieve a list of IP addresses and
    * is executed every 15 minutes to refresh the list of IP addresses.
    */

   sp = ufdbNewGV.lastSource;
   if (sp->ipv4 != NULL  ||  sp->ipv6 != NULL)
   {
      ufdbLogFatalError( "execiplist \"%s\" in source \"%s\" on line %d: "
                         "found mix of exec and non-exec iplists", 
                         command, sp->name, lineno );
      return;
   }
   ufdbLogMessage( "execiplist \"%s\"", command );

   t0 = time( NULL );
   UFDBretrieveExecIPlist( &ufdbNewGV, command, &sp->ipv4hosts, &sp->ipv4, &sp->ipv6hosts, &sp->ipv6 );
   sp->execiplistCommand = command;			/* command is already malloc-ed */
   te = time( NULL );

   if (te - t0 > 4)
      ufdbLogMessage( "WARNING: it took %ld seconds to execute \"%s\"  *****", (long) (te - t0), command );
 
   if (ufdbGV.debug>1 || ufdbGV.debugExternalScripts)
   {
      UFDBlogIPv4( sp->ipv4 );
      UFDBlogIPv6( sp->ipv6 );
   }
}


static void ufdbSourceUserList( 
   char * file  )
{
   char * dbhome;
   char * f;
   FILE * fp;
   char * p;
   char * c;
   char * s;
   char * lc;
   char * lineptr;
   int    ullineno;
   struct sgDb * udb;
   struct Source * sp;
   char   line[10000];
 
   sp = ufdbNewGV.lastSource;
   if (ufdbGV.debug)
      ufdbLogMessage( "ufdbSourceUserList: source \"%s\"  file \"%s\"", 
       		      sp->name != NULL ? sp->name : "nosource", file );
   udb = sp->userDb;
   if (udb != NULL  &&  udb->type != SGDBTYPE_USERLIST)
   {
      ufdbLogFatalError( "userlist \"%s\" in source \"%s\" on line %d: "
                         "found mix of exec and non-exec userlists", 
                         file, sp->name, lineno );
      return;
   }
   if (udb == NULL)
   {
      udb = sp->userDb = (struct sgDb *) ufdbMalloc( sizeof(struct sgDb) );
      udb->dbhome = NULL;
      udb->dbcp = (void *) UFDBmemDBinit();
      udb->type = SGDBTYPE_USERLIST;
      udb->entries = 0;
   }
 
   dbhome = ufdbNewGV.databaseDirectory;
 
   if (file[0] == '/')
      f = file;
   else
   {
      f = (char *) ufdbMalloc( strlen(dbhome) + strlen(file) + 2 );
      strcpy( f, dbhome );
      strcat( f, "/" );
      strcat( f, file );
      ufdbFree( file );
   }
 
   if ((fp = fopen(f,"r")) == NULL)
   {
      ufdbLogError( "line %d: can't open userlist %s: %s  *****", lineno, f, strerror(errno) );
      ufdbFree( f );
      return;
   }
   ullineno = 0;
 
   ufdbLogMessage( "userlist \"%s\"", f );
 
   while (fgets(line,sizeof(line),fp) != NULL)
   {
      ullineno++;
      if (line[0] == '#')			/* skip comments */
         continue;
 
      p = strchr( line, '\n' );
      if (p != NULL)
      {
         *p = '\0';
         if (p != line) 
         {
  	    if (*(p - 1) == '\r') 		/* remove ^M  */
  	       *(p-1) = '\0';
         }
      }
  
      if (line[0] == '\0')
      {
         ufdbLogError( "userlist %s: line %d: line is empty", f, ullineno );
         continue;
      }
  
      c = strchr( line, '#' );		        /* remove comment */
      if (c != NULL)
         *c = '\0';
  
      p = strtok_r( line, " \t,", &lineptr );
      if (p == NULL  ||  *p == '\0')
         continue;
  
      /* we may have a passwd-style formatted file. Ignore ':' and everything else that follows. */
      if ((s = strchr(p,':')) != NULL) 
         *s = '\0';
  
      do
      {
         for (lc = p;  *lc != '\0';  lc++)      // convert username to lowercase
         {
  	    if (*lc <= 'Z'  &&  *lc >= 'A')
  	       *lc += 'a' - 'A';
         }
         if (*p != '\0')
 	 {
	    udb->entries++;
  	    UFDBmemDBinsert( (struct UFDBmemDB *) udb->dbcp, p, NULL );
         }
      } while ((p = strtok_r(NULL," \t,",&lineptr)) != NULL  &&  *p != '\0');
   }
   fclose( fp );

   if (ufdbGV.debug > 1)
   {
      ufdbLogMessage( "just read userlist \"%s\"", f );
      UFDBmemDBprintUserDB( "user", (struct UFDBmemDB *) udb->dbcp );
   }

   ufdbFree( f );
}


static void ufdbSourceUserQuota( 
   const char * seconds, 
   const char * sporadic,  
   const char * renew )
{
   if (seconds == NULL || sporadic == NULL || renew == NULL)
      { ; }      // prevent compiler warning

   ufdbLogError( "line %d: userquota is not supported", lineno );
}


static void defSourceDomain( 
   char * domain )
{
   struct Source * sp;

   sp = ufdbNewGV.lastSource;
   if (sp->domainDb == NULL) 
      sp->domainDb = (struct sgDb *) UFDBmemDBinit();

   if (ufdbGV.debug)
      ufdbLogMessage( "defSourceDomain \"%s\"", domain );

   UFDBmemDBinsert( (struct UFDBmemDB *) sp->domainDb, domain, NULL );

   ufdbFree( domain );
}


static void ufdbSourceEval( 
   int method  )
{
   if (ufdbNewGV.lastSource != NULL)
   {
      if (ufdbGV.debug)
         ufdbLogMessage( "ufdbSourceEval %d: %s", method, 
	                 method == UFDB_EVAL_OR ? "evaluate-or" : "evaluate-and" );
      ufdbNewGV.lastSource->evaluationMethod = method;
   }
}


static void defSourceTime( 
   char *         name, 
   int            within )
{
   struct ufdbTime *  t;

   if ((t = sgTimeFindName(name)) == NULL)
   {
      ufdbLogFatalError( "line %d: time \"%s\" is not defined in configuration file %s",
		         lineno, name, ufdbNewGV.configFile );
      ufdbFree( name );
      return;
   }

   ufdbNewGV.lastSource->within = within;
   ufdbNewGV.lastSource->time = t;
   ufdbFree( name );
}


static struct Source * defSourceFindName( 
   struct Source * slist,
   const char *    name )
{
   for ( ;  slist != NULL;  slist = slist->next)
   {
      if (strcmp(name,slist->name) == 0)
         return slist;
   }
   return NULL;
}


static void defSourceIPV4List( 
   char * file )
{
   char * dbhome;
   char * f;
   FILE * fp;
   char * p;
   char * c;
   char * cidr;
   int    i;
   int    l = 0;
   char * lineptr;
   char   line[UFDB_MAX_URL_LENGTH];

   dbhome = ufdbNewGV.databaseDirectory;

   if (file[0] == '/') 
   {
      f = file;
   }
   else
   {
      f = (char *) ufdbMalloc( strlen(dbhome) + strlen(file) + 2 );
      strcpy( f, dbhome );
      strcat( f, "/" );
      strcat( f, file );
      ufdbFree( file );
   }

   if ((fp = fopen(f,"r")) == NULL) 
   {
      ufdbLogError( "line %d: can't open ipv4list %s: %s", lineno, f, strerror(errno) );
      ufdbFree( f );
      return;
   }

   ufdbLogMessage( "ipv4list \"%s\"", f );

   while (fgets(line,sizeof(line),fp) != NULL)
   {
      l++;
      if (*line == '#')
         continue;
      p = strchr( line, '\n' );
      if (p != NULL && p != line) 
      {
         if (*(p - 1) == '\r') 		        // remove ^M
            p--;
         *p = '\0';
      }
      c = strchr( line, '#' );
      p = strtok_r( line, " \t,", &lineptr );
      do {
         if (c != NULL && p >= c) 		// find the comment
            break;
         i = strspn( p, ".0123456789/-" );
         if (i == 0)
            break;
         *(p + i) = '\0';
         if ((cidr = strchr(p,'/')) != NULL) 
         {
            if (strchr(cidr,'.') == NULL)
               sgIpv4( p, SG_IPTYPE_CIDR, f, l );
            else 
               sgIpv4( p, SG_IPTYPE_CLASS, f, l );
          }
          else if ((cidr = strchr(p,'-')) != NULL) 
          {
             sgIpv4( p, SG_IPTYPE_RANGE, f, l );
          }
          else
          {
             sgIpv4( p, SG_IPTYPE_HOST, f, l );
          }
      } while ((p = strtok_r(NULL," \t,",&lineptr)) != NULL);
   }

   fclose( fp );
   ufdbFree( f );
}


static void defSourceIPV6List( 
   char * file )
{
   char * dbhome;
   char * f;
   FILE * fp;
   char * p;
   char * c;
   int    i;
   int    l = 0;
   char * lineptr;
   char   line[UFDB_MAX_URL_LENGTH];

   dbhome = ufdbNewGV.databaseDirectory;

   if (file[0] == '/') 
   {
      f = file;
   }
   else
   {
      f = (char *) ufdbMalloc( strlen(dbhome) + strlen(file) + 2 );
      strcpy( f, dbhome );
      strcat( f, "/" );
      strcat( f, file );
      ufdbFree( file );
   }

   if ((fp = fopen(f,"r")) == NULL) 
   {
      ufdbLogError( "line %d: can't open ipv6list %s: %s", lineno, f, strerror(errno) );
      ufdbFree( f );
      return;
   }

   ufdbLogMessage( "ipv6list \"%s\"", f );

   while (fgets(line,sizeof(line),fp) != NULL)
   {
      l++;
      if (*line == '#')
         continue;
      p = strchr( line, '\n' );
      if (p != NULL && p != line) 
      {
         if (*(p - 1) == '\r') 		        // remove ^M
            p--;
         *p = '\0';
      }
      c = strchr( line, '#' );
      p = strtok_r( line, " \t,", &lineptr );
      do {
         if (c != NULL && p >= c) 		// find the comment
            break;
         i = strspn( p, ":.0123456789ABCDEFabcdef/" );
         if (i == 0)
            break;
         *(p + i) = '\0';
         if (strchr(p,'/') != NULL) 
         {
            sgIpv6( p, SG_IPV6TYPE_CIDR, f, l );
         }
         else
         {
            sgIpv6( p, SG_IPV6TYPE_HOST, f, l );
         }
      } while ((p = strtok_r(NULL," \t,",&lineptr)) != NULL);
   }

   fclose( fp );
   ufdbFree( f );
}


/* category block functions */

void ufdbCategory( 
  char *            cat )
{
  struct Category * sp;

#if UFDB_DEBUG
   ufdbLogMessage( "ufdbCategory %s", cat );
#endif

  if (ufdbNewGV.catList != NULL) 
  {
    if ((struct Category *) ufdbCategoryFindByName(&ufdbNewGV,cat) != NULL)
    {
      ufdbLogFatalError( "line %d: category %s is already defined in configuration file %s",
		         lineno, cat, ufdbNewGV.configFile );
      ufdbFree( cat );
      return;
    }
  }

  sp = (struct Category *) ufdbMallocAligned( UFDB_CACHELINE_SIZE, sizeof(struct Category) );
  sp->active = 1;
  sp->within = UFDB_ACL_NONE;
  sp->activeBumping = UFDB_ACTIVE_BUMPING_NOTSET;
  sp->blockBumpedConnect = 0;
  sp->options = 0;
  sp->name = cat;
  sp->domainlist = NULL;
  sp->domainlistDb = NULL;
  sp->execdomainlist = NULL;
  sp->expressionlist = NULL;
  sp->regExp = NULL;
  sp->redirect = NULL;
  sp->time = NULL;
  sp->rewrite = NULL;
  sp->next = NULL;
  sp->nblocks = 0;
  sp->nmatches = 0;

  if (ufdbNewGV.catList == NULL) {
    ufdbNewGV.catList = sp;
    ufdbNewGV.lastCat = sp;
  } else {
    ufdbNewGV.lastCat->next = sp;
    ufdbNewGV.lastCat = sp;
  }

   /* category "security" is a special case: the options cannot be set default to 0
    * because the default value for allow-unknown-protocol-over-https is ON.
    */
   if (strcmp( cat, "security" ) == 0)
      sp->options = UFDB_OPT_UNKNOWN_PROTOCOL_OVER_HTTPS;
}


void ufdbCategoryEnd( void )
{
  struct Category * d;
 
  d = ufdbNewGV.lastCat;
  if (d->domainlist == NULL  &&  d->expressionlist == NULL  && 
      d->redirect == NULL  &&  d->rewrite == NULL  &&
      d->options == 0)
  {
    ufdbLogError( "category \"%s\" is missing content, set inactive  *****", d->name );
    d->time = NULL;
    d->active = 0;
  }
}


static void ufdbCategoryOption( int value, int option )
{
   struct Category * sp;

   sp = ufdbNewGV.lastCat;
   if (value)
      sp->options = sp->options | option;
   else
      sp->options = sp->options & (~option);

   if (option == UFDB_OPT_HTTPS_WITH_HOSTNAME)
      ufdbNewGV.httpsWithHostname = value;
   else if (option == UFDB_OPT_HTTPS_OFFICAL_CERTIFICATE)
      ufdbNewGV.httpsOfficialCertificate = value;
   else if (option == UFDB_OPT_SKYPE_OVER_HTTPS)
      ufdbNewGV.SkypeOverHttps = value;
   else if (option == UFDB_OPT_GTALK_OVER_HTTPS)
      ufdbNewGV.GtalkOverHttps = value;
   else if (option == UFDB_OPT_YAHOOMSG_OVER_HTTPS)
      ufdbNewGV.YahooMsgOverHttps = value;
   else if (option == UFDB_OPT_AIM_OVER_HTTPS)
      ufdbNewGV.AimOverHttps = value;
   else if (option == UFDB_OPT_FBCHAT_OVER_HTTPS)
      ufdbNewGV.FBchatOverHttps = value;
   else if (option == UFDB_OPT_CITRIXONLINE_OVER_HTTPS)
      ufdbNewGV.CitrixOnlineOverHttps = value;
   else if (option == UFDB_OPT_ANYDESK_OVER_HTTPS)
      ufdbNewGV.AnydeskOverHttps = value;
   else if (option == UFDB_OPT_TEAMVIEWER_OVER_HTTPS)
      ufdbNewGV.TeamviewerOverHttps = value;
   else if (option == UFDB_OPT_PROHIBIT_INSECURE_SSLV2)
      ufdbNewGV.httpsNoSSLv2 = value;
   else if (option == UFDB_OPT_PROHIBIT_INSECURE_SSLV3)
      ufdbNewGV.httpsNoSSLv3 = value;
   else if (option == UFDB_OPT_PROHIBIT_INSECURE_TLSV1_0)
      ufdbNewGV.httpsNoTLSv1_0 = value;
   else if (option == UFDB_OPT_UNKNOWN_PROTOCOL_OVER_HTTPS)
      ufdbNewGV.unknownProtocolOverHttps = value;
   else if (option == UFDB_OPT_SAFE_SEARCH)
      ;
   else if (option == UFDB_OPT_YOUTUBE_EDUFILTER)
      ;
   else
      ufdbLogError( "ufdbCategoryOption: unrecognised option %d *****", option );

   if (ufdbGV.debug > 1)
      ufdbLogMessage( "ufdbCategoryOption: %s: option %d set to %d", sp->name, option, value );
}


void ufdbCategoryDomainList( 
  char * domainlist )
{
  struct Category * sp;
  const char *      dbhome;
        char *      dl;
  const char *      name;

  if (ufdbGV.debug > 1)
     ufdbLogMessage( "ufdbCategoryDomainList %s", domainlist==NULL ? "NULL" : domainlist );

  if (ufdbNewGV.terminating)
     return;

  sp = ufdbNewGV.lastCat;

  if (sp->domainlistDb != NULL)
  {
     ufdbLogFatalError( "line %d: specify only one domainlist per category in configuration file %s",
		        lineno, ufdbNewGV.configFile );
     return;
  }

  dbhome = ufdbNewGV.databaseDirectory;

  if (domainlist == NULL)
  {
    name = sp->name;
    dl = (char *) ufdbMalloc( sizeof("/dest/") + strlen(name) + sizeof("/domainlist") + 2 );
    strcpy(dl,"/dest/");
    strcat(dl,name);
    strcat(dl,"/domainlist");

    sp->domainlist = (char *) ufdbMalloc( strlen(dbhome) + strlen(dl) + 2 );
    strcpy(sp->domainlist,dbhome);
    strcat(sp->domainlist,"/");
    strcat(sp->domainlist,dl);
  }
  else
  {
    if (domainlist[0] == '/') 
      sp->domainlist = domainlist;
    else
    {
       sp->domainlist = (char *) ufdbMalloc( strlen(dbhome) + strlen(domainlist) + 2 );
       strcpy( sp->domainlist, dbhome );
       strcat( sp->domainlist, "/" );
       strcat( sp->domainlist, domainlist );
       ufdbFree( (void*) domainlist );
    }
  }

  sp->domainlistDb = (struct sgDb *) ufdbCalloc( 1, sizeof(struct sgDb) );
  sp->domainlistDb->type = SGDBTYPE_DOMAINLIST;
  ufdbLogMessage( "loading URL table from \"%s\"", sp->domainlist );
  sgDbInit( sp->domainlistDb, sp->domainlist );
  if (sp->domainlistDb->dbcp == NULL  ||
      ((struct UFDBmemTable *) sp->domainlistDb->dbcp)->table.nNextLevels == 0)
  {
    ufdbLogError( "URL database table \"%s\" is empty and is ignored   *****", sp->domainlist );
    ufdbFreeDomainDb( sp->domainlistDb );
    sp->domainlistDb = NULL;
  }
}


static void ufdbCategoryExecDomainList( 
   char * command )
{
   struct Category * sp;

#if UFDB_DEBUG
   ufdbLogMessage( "ufdbCategoryExecDomainList %s", command );
#endif

   sp = ufdbNewGV.lastCat;

   if (sp == NULL)
   {
      ufdbLogFatalError( "execdomainlist is not within a URL category" );
      return;
   }
   sp->execdomainlist = command;

   if (ufdbGV.debug)
      ufdbLogMessage( "execdomainlist for category %s: \"%s\"", sp->name, command );
}


void ufdbFreeDomainDb( 
   struct sgDb * dbp )
{
   struct UFDBmemTable * mt;

   if (dbp == NULL)
      return;

   mt = (struct UFDBmemTable *) dbp->dbcp;
   if (mt != NULL)
   {
      if (mt->index != NULL)
      {
	 ufdbFree( mt->index );				/* version 2.x */
	 ufdbFree( mt->table.nextLevel );
      }
      else
	 UFDBfreeTableIndex_1_2( &(mt->table) ); 	/* version 1.2 */
#if HAVE_MADVISE && !UFDB_BARE_METAL_SUPPORT && __linux__
      if (mt->madvisedSize)
         madvise( mt->mem, mt->madvisedSize, MADV_NORMAL );
#endif
      if (mt->memStatus == UFDB_MEMSTATUS_MALLOC)
         ufdbFree( (void *) mt->mem );
      ufdbFree( mt );
   }
   ufdbFree( dbp );
}


void ufdbFreeIpv4List( struct Ipv4 * ipv4 )
{
   struct Ipv4 * tmp;

   while (ipv4 != NULL)
   {
      tmp = ipv4->next;
      ufdbFree( (void*) ipv4 );
      ipv4 = tmp;
   }
}


void ufdbFreeIpv6List( struct Ipv6 * ipv6 )
{
   struct Ipv6 * tmp;

   while (ipv6 != NULL)
   {
      tmp = ipv6->next;
      ufdbFree( (void*) ipv6 );
      ipv6 = tmp;
   }
}


static void ufdbFreeSourceList( struct Source * src )
{
   struct Source * tmp;

   while (src != NULL)
   {
      if (src->ipv4hosts != NULL)
         UFDBmemDBdeleteDB( src->ipv4hosts );
      ufdbFreeIpv4List( src->ipv4 );
      if (src->ipv6hosts != NULL)
         UFDBmemDBdeleteDB( src->ipv6hosts );
      ufdbFreeIpv6List( src->ipv6 );

      ufdbFree( (void*) src->name );
      if (src->domainDb != NULL)
	 UFDBmemDBdeleteDB( (struct UFDBmemDB *) src->domainDb );

      /* execuserlist content is saved during a reload except when ufdbguardd terminates */
      if (src->userDb != NULL  &&  src->userDb->type != SGDBTYPE_EXECUSERLIST)
      {
	 UFDBmemDBdeleteDB( (struct UFDBmemDB *) src->userDb->dbcp );
	 ufdbFree( (void*) src->userDb );
      }
      ufdbFree( (void*) src->sarg0 );
      ufdbFree( (void*) src->execiplistCommand );

      tmp = src->next;
      ufdbFree( (void*) src );
      src = tmp;
   }
}


static void ufdbFreeAclCategoryList( struct AclCategory * ac )
{
   struct AclCategory * tmp;

   while (ac != NULL)
   {
      ufdbFree( (void*) ac->name );
      tmp = ac->next;
      ufdbFree( (void*) ac );
      ac = tmp;
   }
}


static void ufdbFreeAclList( struct Acl * acl )
{
   struct Acl * tmp;

   while (acl != NULL)
   {
      ufdbFree( (void*) acl->name );
      ufdbFreeAclCategoryList( acl->pass );
      ufdbFreeAclCategoryList( acl->implicitPass );
      ufdbFree( (void*) acl->redirect );
      tmp = acl->next;
      ufdbFree( (void*) acl );
      acl = tmp;
   }
}


static void ufdbFreeCategoryList( struct Category * cat )
{
   struct Category * tmp;

   while (cat != NULL)
   {
      if (ufdbGV.debug > 1  ||  ufdbGV.debugRegexp)
         ufdbLogMessage( "ufdbFreeCategoryList: freeing regexp of category %s", cat->name );
      ufdbFreeRegExprList( cat->regExp );
      ufdbFree( (void*) cat->name );
      ufdbFree( (void*) cat->domainlist );
      ufdbFreeDomainDb( cat->domainlistDb );
      ufdbFree( (void*) cat->execdomainlist );
      ufdbFree( (void*) cat->expressionlist );
      ufdbFree( (void*) cat->redirect );
      /* ufdbFree( cat->rewrite );  freed in ufdbFreeRewriteList() */
      tmp = cat->next;
      ufdbFree( (void*) cat );
      cat = tmp;
   }
}


static void ufdbFreeRewriteList( 
   struct sgRewrite * p )
{
   struct sgRewrite * tmp;

   while (p != NULL)
   {
      ufdbFreeRegExprList( p->rewrite );
      ufdbFree( (void*) p->name );
      tmp = p->next;
      ufdbFree( (void*) p );
      p = tmp;
   }
}


static void ufdbFreeTimeElement( 
   struct TimeElement * p )
{
   struct TimeElement * tmp;

   while (p != NULL)
   {
      tmp = p->next;
      ufdbFree( (void*) p );
      p = tmp;
   }
}


static void ufdbFreeTime( 
   struct ufdbTime * p )
{
   struct ufdbTime * tmp;

   while (p != NULL)
   {
      ufdbFree( (void*) p->name );
      ufdbFreeTimeElement( p->element );
      tmp = p->next;
      ufdbFree( (void*) p );
      p = tmp;
   }
}


void ufdbFreeLastBits( struct ufdbGV * gv )
{
   /* we may need these variables during a reload */
   if (gv->pidFilename != NULL)
   {
      ufdbFree( (void*) gv->pidFilename );
      gv->pidFilename = NULL;
   }
   if (gv->emailServer != NULL)
   {
      ufdbFree( (void*) gv->emailServer );
      gv->emailServer = NULL;
   }
   if (gv->myHostname != NULL)
   {
      ufdbFree( (void*) gv->myHostname );
      gv->myHostname = NULL;
   }
   if (gv->adminEmail != NULL)
   {
      ufdbFree( (void*) gv->adminEmail );
      gv->adminEmail = NULL;
   }
   if (gv->senderEmail != NULL)
   {
      ufdbFree( (void*) gv->senderEmail );
      gv->senderEmail = NULL;
   }
   if (gv->externalStatusCommand != NULL)
   {
      ufdbFree( (void*) gv->externalStatusCommand );
      gv->externalStatusCommand = NULL;
   }
}


void ufdbFreeAllMemory( struct ufdbGV * gv )
{
   gv->logDirectory[0] = '\0';

   ufdbFreeRewriteList( gv->rewrite );
   gv->rewrite = NULL;
   gv->lastRewrite = NULL;
   gv->lastRewriteRegExec = NULL;

   ufdbFreeCategoryList( gv->catList );
   gv->catList = NULL;
   gv->lastCat = NULL;

   ufdbFreeAclList( gv->aclList );
   gv->aclList = NULL;
   gv->lastAcl = NULL;
   gv->defaultAcl = NULL;
   gv->lastAclCategory = NULL;

   ufdbFreeSourceList( (struct Source *) gv->sourceList );
   gv->sourceList = NULL;
   gv->lastSource = NULL;

   /* free ufdbNewGV.checkedDB */
   if (gv->checkedDB.mem != NULL)
   {
      if (gv->checkedDB.index != NULL)
      {
         ufdbFree( (void*) gv->checkedDB.index );
	 ufdbFree( gv->checkedDB.table.nextLevel );
      }
      else
	 UFDBfreeTableIndex_1_2( &(gv->checkedDB.table) );
#if HAVE_MADVISE && !UFDB_BARE_METAL_SUPPORT && __linux__
      if (gv->checkedDB.madvisedSize)
         madvise( gv->checkedDB.mem, gv->checkedDB.madvisedSize, MADV_NORMAL );
#endif
      gv->checkedDB.madvisedSize = 0;
      if (gv->checkedDB.memStatus == UFDB_MEMSTATUS_MALLOC)
         ufdbFree( (void*) gv->checkedDB.mem );
   }
   gv->checkedDB.table.nextLevel = NULL;
   gv->checkedDB.table.nNextLevels = 0;
   gv->checkedDB.mem = NULL;
   gv->checkedDB.index = NULL;

   if (gv->checkedExpressions != NULL)
   {
      ufdbFreeRegExprList( gv->checkedExpressions );
      gv->checkedExpressions = NULL;
   }

   ufdbFreeTime( gv->timeList );
   gv->timeList = NULL;
   gv->lastTime = NULL;

   gv->lastTimeElement = NULL;
   gv->timeElement = NULL;

   gv->lastRegExpDest = NULL;

   ufdbFree( (void*) gv->SquidVersion );
   gv->SquidVersion = NULL;
}


static void ufdbCategoryUrlList( 
   char * urllist )
{
   if (urllist == NULL)
      urllist = (char *) "-";
   ufdbLogError( "line %d: \"urllist %s\" is deprecated and ignored *****\n"
                 "ufdbGenTable should be called with the -u option to include URLs\n"
		 "ufdbGenTable combines URLs and domains in one table file so only the domainlist is required",
		 lineno, urllist );
   if (urllist[0] != '-')
      ufdbFree( (void*) urllist );
}


void ufdbCategoryExpressionList( 
  char * exprlist, 
  const char * chcase  )
{
  FILE * fp;
  char * dbhome;
  char * dl;
  const char * name;
  char * p;
  int    flags;
  struct stat         statbuf;
  struct Category *   sp;
  struct ufdbRegExp * regexp;
  char   buf[UFDB_MAX_URL_LENGTH];
  char   errbuf[256];

#if UFDB_DEBUG
   ufdbLogMessage( "ufdbCategoryExpressionList %s", exprlist );
#endif

  flags = REG_EXTENDED | REG_NOSUB;
  sp = ufdbNewGV.lastCat;
  dbhome = ufdbNewGV.databaseDirectory;

  if (exprlist == NULL)
  {
    name = sp->name;
    dl = (char *) ufdbMalloc( sizeof("/dest/") + strlen(name) + sizeof("/expressionlist") + 2 );
    strcpy(dl,"/dest/");
    strcat(dl,name);
    strcat(dl,"/expressionlist");

    flags |= REG_ICASE; 	/* default case insensitive */

    sp->expressionlist = (char *) ufdbMalloc( strlen(dbhome) + strlen(dl) + 2 );
    strcpy(sp->expressionlist,dbhome);
    strcat(sp->expressionlist,"/");
    strcat(sp->expressionlist,dl);
  }
  else
  {
    if (exprlist[0] == '/') 
    {
      sp->expressionlist = exprlist;
    }
    else
    {
       sp->expressionlist = (char *) ufdbMalloc( strlen(dbhome) + strlen(exprlist) + 2 );
       strcpy( sp->expressionlist, dbhome );
       strcat( sp->expressionlist, "/" );
       strcat( sp->expressionlist, exprlist );
       ufdbFree( (void*) exprlist );
    }
    if (*chcase == 'i')
       flags |= REG_ICASE;     /* set case insensitive */
  }

  ufdbLogMessage( "loading regular expressions from \"%s\"", sp->expressionlist );

  if ((fp = fopen(sp->expressionlist, "r")) == NULL) 
  {
    ufdbLogFatalError( "cannot open expression list %s: %s", sp->expressionlist, strerror(errno) );
    return;
  }

  if (0 == fstat( fileno(fp), &statbuf ))
  {
     if (!S_ISREG(statbuf.st_mode))
     {
        ufdbLogFatalError( "expression list %s: not a regular file", sp->expressionlist );
	fclose( fp );
	return;
     }
  }

  while (!feof(fp)  &&  fgets(buf, sizeof(buf), fp) != NULL) 
  {
    if (buf[0] == '#')
       continue;

    p = (char *) strchr( buf, '\n' );
    if (p != NULL  &&  p != buf) 
    {
      if (*(p-1) == '\r') 	/* removing ^M  */
	p--;
      *p = '\0';
    }
    /* TO-DO: warn about leading and trailing spaces */
    regexp = ufdbNewPatternBuffer( buf, flags );
    if (regexp->error) 
    {
      regerror( regexp->error, (regex_t*) regexp->compiled[0], errbuf, sizeof(errbuf) );
      ufdbLogError( "regular expression error in %s:\n%s : %s   *****", sp->expressionlist, errbuf, buf );
    }
    if (ufdbNewGV.lastCat->regExp == NULL) 
    {
      ufdbNewGV.lastCat->regExp = regexp;
      ufdbNewGV.lastRegExpDest = regexp;
    }
    else
    {
      ufdbNewGV.lastRegExpDest->next = regexp;
      ufdbNewGV.lastRegExpDest = regexp;
    }
  }
  fclose( fp );

  if (ufdbNewGV.expressionOptimisation)
     ufdbNewGV.lastCat->regExp = UFDBoptimizeExprList( sp->expressionlist, ufdbNewGV.lastCat->regExp );
}


static void ufdbCategoryCACertsFile(
   char *              cacertsFile )
{
   struct Category *   cat;

   cat = ufdbNewGV.lastCat;
   if (cat == NULL  ||  strcmp( cat->name, "security" ) != 0)
   {
      ufdbLogFatalError( "cacerts can only be defined inside the \"security\" category" );
      return;
   }

   if (*cacertsFile == '/')
      strcpy( ufdbNewGV.CAcertsFile, cacertsFile );
   else
   {
      char * dbh;
      dbh = ufdbNewGV.databaseDirectory;
      strcpy( ufdbNewGV.CAcertsFile, dbh );
      strcat( ufdbNewGV.CAcertsFile, "/" );
      strcat( ufdbNewGV.CAcertsFile, cacertsFile );
   }
}


static void ufdbCategoryCACertsDir(
   char *              cacertsDir )
{
   struct Category *   cat;

   cat = ufdbNewGV.lastCat;
   if (cat == NULL  ||  strcmp( cat->name, "security" ) != 0)
   {
      ufdbLogFatalError( "cacerts-dir can only be defined inside the \"security\" category" );
      return;
   }

   if (*cacertsDir == '/')
      strcpy( ufdbNewGV.CAcertsDir, cacertsDir );
   else
   {
      char * dbh;
      dbh = ufdbNewGV.databaseDirectory;
      strcpy( ufdbNewGV.CAcertsDir, dbh );
      strcat( ufdbNewGV.CAcertsDir, "/" );
      strcat( ufdbNewGV.CAcertsDir, cacertsDir );
   }
}


static void ufdbCategoryRedirect( 
   char * value )
{
   struct Category * sp;

#if UFDB_DEBUG
   ufdbLogMessage( "ufdbCategoryRedirect %s", value );
#endif

   sp = ufdbNewGV.lastCat;
   sp->redirect = value;
   /* TODO: check that "localhost" is not here if no 302 is used */
}


static void ufdbCategoryRewrite( 
  char *              value )
{
  struct sgRewrite *  rewrite;

  if ((rewrite = sgRewriteFindName(value)) == NULL)
  {
    ufdbLogFatalError( "line %d: rewrite %s is not defined in configuration file %s",
		       lineno, value, ufdbNewGV.configFile );
    return;
  }

  ufdbNewGV.lastCat->rewrite = rewrite;
}


static void ufdbCategoryBlockConnect( 
   int flag  )
{
   ufdbNewGV.lastCat->blockBumpedConnect = flag;

   if (ufdbGV.debug > 1)
      ufdbLogMessage( "ufdbCategoryBlockConnect: category \"%s\" : block-bumped-connect %s",
                      ufdbNewGV.lastCat->name, flag ? "on" : "off" );
}


static void ufdbCategoryActiveBumping( 
   int flag  )
{
   ufdbNewGV.lastCat->activeBumping = flag ? UFDB_ACTIVE_BUMPING_ON : UFDB_ACTIVE_BUMPING_OFF;

   if (ufdbGV.debug > 1)
      ufdbLogMessage( "ufdbCategoryActiveBumping: category \"%s\" : squid-uses-active-bumping %s",
                      ufdbNewGV.lastCat->name, flag ? "on" : "off" );
}


static void ufdbCategoryTime(  
   char *       name, 
   int  	within )
{
   struct ufdbTime * t;

   if ((t = sgTimeFindName(name)) == NULL) 
   {
      ufdbLogFatalError( "line %d: time \"%s\" is not defined in configuration file %s",
   		         lineno, name, ufdbNewGV.configFile );
      ufdbFree( (void*) name );
      return;
   }

   ufdbNewGV.lastCat->within = within;
   ufdbNewGV.lastCat->time = t;
   ufdbFree( (void*) name );
}


struct Category * ufdbCategoryFindByName( 
   struct ufdbGV *   gv,
   const char *      name )
{
   struct Category * p;

   for (p = gv->catList;  p != NULL;  p = p->next)
   {
      if (strcmp(name,p->name) == 0)
         return p;
   }
   return NULL;
}


static void sgRewrite( 
  char * rewrite )
{
  struct sgRewrite * rew;

#if UFDB_DEBUG
   ufdbLogMessage( "sgRewrite %s", rewrite );
#endif

  if (ufdbNewGV.rewrite != NULL)
  {
    if ((struct sgRewrite *) sgRewriteFindName(rewrite) != NULL)
    {
      ufdbLogFatalError( "line %d: rewrite \"%s\" is not defined in configuration file %s",
		         lineno, rewrite, ufdbNewGV.configFile );
      ufdbFree( (void*) rewrite );
      return;
    }
  }

  rew = (struct sgRewrite *) ufdbMalloc( sizeof(struct sgRewrite) );
  rew->name = rewrite;
  rew->active = 1;
  rew->rewrite = NULL;
  rew->time = NULL;
  rew->within = UFDB_ACL_NONE;
  rew->next = NULL;

  if (ufdbNewGV.rewrite == NULL) 
  {
    ufdbNewGV.rewrite = rew;
    ufdbNewGV.lastRewrite = rew;
  }
  else
  {
    ufdbNewGV.lastRewrite->next = rew;
    ufdbNewGV.lastRewrite = rew;
  }
}


static void sgRewriteTime( 
  char * name, 
  int    within )
{
  struct ufdbTime * t;

#if UFDB_DEBUG
   ufdbLogMessage( "sgRewriteTime %s %d", name, within );
#endif

  if ((t = sgTimeFindName(name)) == NULL)
  {
    ufdbLogFatalError( "line %d: time \"%s\" is not defined in configuration file %s",
		       lineno, name, ufdbNewGV.configFile );
    ufdbFree( (void*) name );
    return;
  }

  ufdbNewGV.lastRewrite->within = within;
  ufdbNewGV.lastRewrite->time = t;
  ufdbFree( (void*) name );
}


static void sgRewriteSubstitute( 
  char * string )
{
  char * pattern; 
  char * subst = NULL;
  char * p;
  int    flags = REG_EXTENDED;
  int    global = 0;
  char * httpcode = NULL;
  struct ufdbRegExp * regexp;
  char   errbuf[256];

  pattern = string + 2; 	/* skipping s@ */
  p = pattern;
  while ((p = strchr(p,'@')) != NULL)
  {
    if (*(p - 1) != '\\') 
    {
      *p = '\0';
      subst = p + 1;
      break;
    }
    p++;
  }

  p = strrchr( subst, '@' );
  while (p != NULL  &&  *p != '\0')
  {
    if (*p == 'r' )
      httpcode =  (char *) REDIRECT_TEMPORARILY;
    if (*p == 'R' )
      httpcode =  (char *) REDIRECT_PERMANENT;
    if (*p == 'i' || *p == 'I')
      flags |= REG_ICASE;
    if (*p == 'g')
      global = 1;
    *p = '\0'; 		/* removes @i from string */
    p++;
  } 

  regexp = ufdbNewPatternBuffer( pattern, flags );
  if (regexp->error) 
  {
      regerror( regexp->error, (regex_t*) regexp->compiled[0], errbuf, sizeof(errbuf) );
      ufdbLogError( "line %d: regular expression error in %s:\n%s   *****", lineno, pattern, errbuf );
  }
  else {
    regexp->substitute = ufdbStrdup( subst );
  }

  if (ufdbNewGV.lastRewrite->rewrite == NULL)
    ufdbNewGV.lastRewrite->rewrite = regexp;
  else 
    ufdbNewGV.lastRewriteRegExec->next = regexp;
  regexp->httpcode = httpcode;
  regexp->global = global;
  ufdbNewGV.lastRewriteRegExec = regexp;
}


static struct sgRewrite * sgRewriteFindName( 
   const char * name )
{
   struct sgRewrite * p;

   for (p = ufdbNewGV.rewrite;  p != NULL;  p = p->next)
   {
      if (strcmp(name,p->name) == 0)
         return p;
   }
   return NULL;
}


/*
 * Time functions
 */

/*
 * sgTime - parse configuration time statement.
 */
static void sgTime( 
  char *        name )
{
  struct ufdbTime * t;

  if (ufdbGV.debug > 1)
     ufdbLogMessage( "sgTime %s", name );

  if (ufdbNewGV.timeList != NULL)
  {
    if ((struct ufdbTime *) sgTimeFindName(name) != NULL)
    {
      ufdbLogFatalError( "line %d: time \"%s\" is not defined in configuration file %s",
		         lineno, name, ufdbNewGV.configFile );
      ufdbFree( (void*) name );
      return;
    }
  }
  else 
    numTimeElements = 0;

  t = (struct ufdbTime *) ufdbMalloc( sizeof(struct ufdbTime) );
  t->name = name;
  t->active = 1;
  t->element = NULL;
  t->next = NULL;

  ufdbNewGV.timeElement = NULL;
  ufdbNewGV.lastTimeElement = NULL;
  if (ufdbNewGV.timeList == NULL) 
  {
    ufdbNewGV.timeList = t;
    ufdbNewGV.lastTime = t;
  }
  else
  {
    ufdbNewGV.lastTime->next = t;
    ufdbNewGV.lastTime = t;
  }
}


/*
 * sgTimeElementInit - initialise parsing of a configuration time element.
 */
static void sgTimeElementInit( void )
{
   struct TimeElement * te;

   te = (struct TimeElement *) ufdbCalloc( 1, sizeof(struct TimeElement) );
   numTimeElements++;

   if (ufdbNewGV.lastTime->element == NULL)
     ufdbNewGV.lastTime->element = te;
   if (ufdbNewGV.lastTimeElement != NULL)
     ufdbNewGV.lastTimeElement->next = te;
   ufdbNewGV.lastTimeElement = te;
}


/*
 * sgTimeElementEnd - finalise parsing of configuration time element.
 */
static void sgTimeElementEnd( void )
{
  time_switch = 0;
  date_switch = 0;

  if (ufdbNewGV.lastTimeElement->fromdate != 0)
  {
    if (ufdbNewGV.lastTimeElement->todate == 0)
      ufdbNewGV.lastTimeElement->todate = ufdbNewGV.lastTimeElement->fromdate + 86399;
    else 
      ufdbNewGV.lastTimeElement->todate = ufdbNewGV.lastTimeElement->todate + 86399;
  }

  if (ufdbNewGV.lastTimeElement->from == 0  &&  ufdbNewGV.lastTimeElement->to == 0)
    ufdbNewGV.lastTimeElement->to = 1439;  /* set time to 23:59 */
}


/*
 * sgTimeElementAdd - add configuration time element.
 */
static void sgTimeElementAdd( 
  char * element, 
  char   type ) 
{
  struct TimeElement * te;
  char * p;
  char   wday;
  int    h, m, Y, M, D;
  time_t sec;
  char * lineptr;

  wday = 0;
  M = 0;
  D = -1;
  te = ufdbNewGV.lastTimeElement;

  switch (type)
  {
  case T_WEEKDAY:
    p = strtok_r( element, " \t,", &lineptr );
    do {
      if (*p == '*') {
	wday = 0x7F;
      } else if (!strncmp(p,"sun",3)) {
	wday = wday | 0x01;
      } else if (!strncmp(p,"mon",3)) {
	wday = wday | 0x02;
      } else if (!strncmp(p,"tue",3)) {
	wday = wday | 0x04;
      } else if (!strncmp(p,"wed",3)) {
	wday = wday | 0x08;
      } else if (!strncmp(p,"thu",3)) {
	wday = wday | 0x10;
      } else if (!strncmp(p,"fri",3)) {
	wday = wday | 0x20;
      } else if (!strncmp(p,"sat",3)) {
	wday = wday | 0x40;
      }
      p = strtok_r( NULL, " \t,", &lineptr );
    } while (p != NULL);
    te->wday = wday;
    break;

  case T_TVAL:
    h = -1;
    m = -1;
    sscanf( element, "%d:%d", &h, &m );
    if ((h < 0 || h > 24) || (m < 0 || m > 59))
    {
      ufdbLogFatalError( "line %d: time format error in %s", lineno, ufdbNewGV.configFile );
      h = 0;
      m = 0;
    }
    if (time_switch == 0) 
    {
      time_switch++;
      te->from = (h * 60) + m ;
    }
    else
    {
      time_switch = 0;
      te->to = (h * 60) + m ;
    }
    break;

  case T_DVAL:
    sec = date2sec( element );
    if (sec == -1) 
    {
      ufdbLogFatalError( "line %d: date format error in %s", lineno, ufdbNewGV.configFile );
      sec = 1;
    }
    if (date_switch == 0) {
      date_switch++;
      te->fromdate = sec;
    } else {
      date_switch = 0;
      te->todate = sec;
    }
    break;

  case T_DVALCRON:
    p = strtok_r( element, "-./", &lineptr );
    Y = atoi(p);
    if (*p == '*')
      Y = -1;
    else
      Y = atoi(p);
    while ((p=strtok_r(NULL,"-./",&lineptr)) != NULL) 
    {
      if (*p == '*')
	if (M == 0)
	  M = -1;
	else 
	  D = -1;
      else
	if (M == 0)
	  M = atoi(p);
	else
	  D = atoi(p);
    }
    te->y = Y;
    te->m = M;
    te->d = D;
    break;

  case T_WEEKLY:
    p = element;
    while (*p != '\0') 
    {
      switch (*p) {
      case 'S':
      case 's':
	wday = wday | 0x01;
	break;
      case 'M':
      case 'm':
	wday = wday | 0x02;
	break;
      case 'T':
      case 't':
	wday = wday | 0x04;
	break;
      case 'W':
      case 'w':
	wday = wday | 0x08;
	break;
      case 'H':
      case 'h':
	wday = wday | 0x10;
	break;
      case 'F':
      case 'f':
	wday = wday | 0x20;
	break;
      case 'A':
      case 'a':
	wday = wday | 0x40;
	break;
      default:
	ufdbLogFatalError( "line %d: weekday format error in %s", lineno, ufdbNewGV.configFile );
	break;
      }
      p++;
    }
    te->wday = wday;
    break;
  }

  ufdbFree( (void*) element );
}


/* 
 * lookup a ufdbTime element by name.
 */
static struct ufdbTime * sgTimeFindName( 
   const char * name )
{
   struct ufdbTime * p;

   for (p = ufdbNewGV.timeList;  p != NULL;  p = p->next)
   {
      if (strcmp(name,p->name) == 0)
         return p;
   }
   return NULL;
}


/*
 * sgTimeCmp - Time array sort function.
 */
static int sgTimeCmp( const void * a, const void * b )
{
   const int * aa = (const int *) a;
   const int * bb = (const int *) b;

   return *aa - *bb;
}


/*
 * _getSortedTimeElementTimes - produce a sorted array of time element times
 */
static void _getSortedTimeElementTimes( void )           // uses ufdbGV
{
   struct ufdbTime *    p;
   struct TimeElement * te;
   int                  i, j;
   int                  totalNumElems;
 
   if (ufdbGV.debug > 1)
      ufdbLogMessage( "_getSortedTimeElementEvents" );

   if (ufdbGV.timeList == NULL)
      return;

   /* find total number of time elements */
   totalNumElems = 0;
   for (p = ufdbGV.timeList;  p != NULL;  p = p->next) 
      for (te = p->element;  te != NULL;  te = te->next) 
         totalNumElems++;

   TimeElementsEvents = (int *) ufdbCalloc( totalNumElems * 2 , sizeof(int) ); 

   i = 0;
   for (p = ufdbGV.timeList;  p != NULL;  p = p->next) 
   {
      for (te = p->element;  te != NULL;  te = te->next) 
      {
         TimeElementsEvents[i++] = te->from == 0 ? 1440 : te->from;
         TimeElementsEvents[i++] = te->to == 0   ? 1440 : te->to;
      }
   }
 
   qsort( TimeElementsEvents, totalNumElems * 2, sizeof(int), sgTimeCmp );

   if (ufdbGV.debug > 1)
      ufdbLogMessage( "   _getSortedTimeElementEvents: after qsort" );

   /* remove identical time elements */
   for (i=1,j=1;  i < totalNumElems * 2;  i++) 
   {
      if (TimeElementsEvents[i] > TimeElementsEvents[i-1])
      {
	 TimeElementsEvents[j] = TimeElementsEvents[i];
         j++;
      }
   }

   numTimeElements = j;			/* #unique time elements */
}


/*
 * _TimeEvaluateElements - evaluate all elements if they match the current time/date and mark them active.
 */
static void _TimeEvaluateElements(              // uses ufdbGV
  struct tm * tm_now, 
  time_t      now )
{
  struct ufdbTime *    tlist;
  struct TimeElement * te;
  int                  min;

  if (ufdbGV.debug > 1)
     ufdbLogMessage( "      _TimeEvaluateElements" );

  for (tlist = ufdbGV.timeList;  tlist != NULL;  tlist = tlist->next)
  {
    tlist->active = 0;
    for (te = tlist->element;  te != NULL;  te = te->next)
    {
      if (te->wday != 0) 			/* check wday */
      {
	if (((1 << tm_now->tm_wday) & te->wday) != 0)  
	{
	  min = (tm_now->tm_hour * 60) + tm_now->tm_min;
	  if (min >= te->from  &&  min < te->to) 
	  {
	    tlist->active = 1;
	    break;
	  }
	}
      }
      else if (te->fromdate != 0)		/* check date */
      {
	if (now >= te->fromdate  &&  now <= te->todate) 
	{
	  min = (tm_now->tm_hour * 60) + tm_now->tm_min;
	  if (min >= te->from  &&  min < te->to) 
	  {
	    tlist->active = 1;
	    break;
	  }
	}
      }
      else					/* check crondate */
      {
	if (te->y == -1  ||  te->y == (tm_now->tm_year + 1900)) 
	{
	  if (te->m == -1  ||  te->m == (tm_now->tm_mon + 1)) 
	  {
	    if (te->d == -1  ||  te->d == (tm_now->tm_mday)) 
	    {
	      min = (tm_now->tm_hour * 60) + tm_now->tm_min;
	      if (min >= te->from  &&  min < te->to) 
	      {
		tlist->active = 1;
		break;
	      }
	    }
	  }
	}
      }
    }
    if (ufdbGV.debug > 1)
       ufdbLogMessage( "      _TimeEvaluateElements: time %s is %sactive", tlist->name, tlist->active?"":"not " );
  }
}


/*
 * _TimeSetAclSrcDestRew - mark all acl/source/dest/rew (in)active.
 */
static void _TimeSetAclSrcDestRew( void )               // uses ufdbGV
{
  struct Acl *         acl;
  struct Category *    cat;
  struct Source *      s;
  struct sgRewrite *   rew;
  int                  a;

  if (ufdbGV.debug > 1)
     ufdbLogMessage( "   _TimeSetAclSrcDestRew" );

  for (acl = ufdbGV.aclList;  acl != NULL;  acl = acl->next)
  {
    if (acl->time != NULL  &&  acl->within != UFDB_ACL_ELSE) 
    {
      /* Be careful here: we are multithreaded and other threads use the value 
       * of acl->active at the same time.
       */
      a = acl->time->active;
      if (acl->within == UFDB_ACL_OUTSIDE)
	a = !a;
      if (acl->next != NULL  &&  acl->next->within == UFDB_ACL_ELSE) 
	acl->next->active = !a;
      acl->active = a;
    }
#if 0
    if (acl->pass == NULL)
       acl->active = 0;			/* it can have a 'continue' so do not make it inactive */
#endif
    if (ufdbGV.debug > 1)
       ufdbLogMessage( "      _TimeSetAclSrcDestRew: acl %s %s is %sactive", acl->name, 
		       acl->within==UFDB_ACL_ELSE ? "ELSE" :
			  acl->within==UFDB_ACL_WITHIN ? "WITHIN" :
			     acl->within==UFDB_ACL_OUTSIDE ? "OUTSIDE" : "",
		       acl->active?"":"not " );
  }

  for (cat = ufdbGV.catList;  cat != NULL;  cat = cat->next)
  {
    if (cat->time != NULL) 
    {
      cat->active = cat->time->active;
      if (cat->within == UFDB_ACL_OUTSIDE)
	cat->active = !cat->active;
    }
    if (ufdbGV.debug > 1)
       ufdbLogMessage( "      _TimeSetAclSrcDestRew: category %s is %sactive", cat->name, cat->active?"":"not " );
  }

  for (s = ufdbGV.sourceList;  s != NULL;  s = s->next)
  {
    if (s->time != NULL) 
    {
      s->active = s->time->active;
      if (s->within == UFDB_ACL_OUTSIDE)
	s->active = !s->active;
    }
    if (ufdbGV.debug > 1)
       ufdbLogMessage( "      _TimeSetAclSrcDestRew: source %s is %sactive", s->name, s->active?"":"not " );
  }

  for (rew = ufdbGV.rewrite; rew != NULL; rew = rew->next)
  {
    if (rew->time != NULL) 
    {
      rew->active = rew->time->active;
      if (rew->within == UFDB_ACL_OUTSIDE)
	 rew->active = !rew->active;
    }
    if (ufdbGV.debug > 1)
       ufdbLogMessage( "      _TimeSetAclSrcDestRew: rewrite %s is %sactive", rew->name, rew->active?"":"not " );
  }
}


static void sgAlarm( int s )
{
   if (s) { ; }         // prevent compiler warning
   // do nothing; there is a thread that waits for a SIGALRM which calls ufdbHandleAlarmForTimeEvents()
}


/*
 *  ufdbHandleAlarmForTimeEvents - the alarm for the next time event went off so 
 *                                 recalculate the time-dependent active state.
 */
void ufdbHandleAlarmForTimeEvents(              // uses ufdbGV
   int        why )
{
   time_t     now;
   struct tm  tm_now;
   int        m;
   int        tindex;
   int        lastval;

   if (ufdbGV.debug)
      ufdbLogMessage( "ufdbHandleAlarmForTimeEvents( why=%s )", why==UFDB_PARAM_INIT ? "init" : "alarm" );

   if (why == UFDB_PARAM_INIT)
      ufdbLogMessage( "time definitions are used; evaluating current ACLs" );
   else
   {
      ufdbLogMessage( "alarm went off to recalculate time ACLs" );
      if (ufdbGV.terminating)
      {
	 ufdbLogMessage( "This alarm is ignored because ufdbguardd is exiting" );
	 return;
      }
      if (ufdbGV.reconfig)
      {
	 ufdbLogMessage( "This alarm is ignored because the configuration is being reloaded"
	                 " and a new alarm is set to go off in 15 seconds" );
	 alarm( 15 );
	 return;
      }
   }

   if (ufdbGV.timeList == NULL)
   {
      return;
   }

   // NOTE: _getSortedTimeElementTimes() mallocs TimeElementsEvents[] and this function frees it.
   _getSortedTimeElementTimes();

   now = UFDBtime() + 30;
   localtime_r( &now, &tm_now ); 
   m = (tm_now.tm_hour * 60) + tm_now.tm_min;
  
   lastval = 0;
   for (tindex = 0;  tindex < numTimeElements;  tindex++)
   {
#if UFDB_TIME_DEBUG
      if (ufdbGV.debug > 1)
         ufdbLogMessage( "   TimeElementsEvents[%d] = %d", tindex, TimeElementsEvents[tindex] );
#endif
      lastval = TimeElementsEvents[tindex];
      if (TimeElementsEvents[tindex] >= m)
         break;
   }

   if (ufdbGV.debug > 1)
      ufdbLogMessage( "   ufdbHandleAlarmForTimeEvents: m = %d  tindex = %d, lastval = %d", m, tindex, lastval );

   if (lastval < m)
      m = (((1440 - m) + TimeElementsEvents[0]) * 60) - tm_now.tm_sec;
   else
      m = ((lastval - m) * 60) - tm_now.tm_sec;

   if (m <= 0)
      m = 30;

   _TimeEvaluateElements( &tm_now, now );
   _TimeSetAclSrcDestRew();

   ufdbFree( (void*) TimeElementsEvents );
   TimeElementsEvents = NULL;

   ufdbLogMessage( "next alarm is in %d seconds", (unsigned int) m );
   ufdbSetSignalHandler( SIGALRM, sgAlarm );
   (void) alarm( (unsigned int) m );
}


/*
 * sgTimeElementClone - copy a time specification.
 */
static void sgTimeElementClone( void )
{
  struct TimeElement * te;
  struct TimeElement * tmp;

  te = ufdbNewGV.lastTimeElement;
  if (te == NULL) 
  {
    ufdbLogFatalError( "No previous TimeElement in sgTimeElementClone !" );
    return;
  }
  else
  {
    sgTimeElementInit();
    ufdbNewGV.lastTimeElement->wday = te->wday;
    ufdbNewGV.lastTimeElement->from = te->from;
    ufdbNewGV.lastTimeElement->to = te->to;
    ufdbNewGV.lastTimeElement->y = te->y;
    ufdbNewGV.lastTimeElement->m = te->m;
    ufdbNewGV.lastTimeElement->d = te->d;
    ufdbNewGV.lastTimeElement->fromdate = te->fromdate;
    ufdbNewGV.lastTimeElement->todate = te->todate;
    tmp = ufdbNewGV.lastTimeElement;
    ufdbNewGV.lastTimeElement = te;
    sgTimeElementEnd();
    ufdbNewGV.lastTimeElement = tmp;
  }
}


/*
 * IP functions
 */

static struct Ipv4 * sgIpv4Last( 
   struct Source * s )
{
   struct Ipv4 * ipv4;
   struct Ipv4 * last;

   last = NULL;
   for (ipv4 = s->ipv4;  ipv4 != NULL;  ipv4 = ipv4->next)
      last = ipv4;

   return last;
}


static void sgIpv4( 
   const char *   addr,
   int            type,
   const char *   file,
   int            lineno )
{
   struct Ipv4 *  ipv4;
   char *         addr2;
   char *         cidr;
   char *         end;
   unsigned int   octet;

#if 1
   if (ufdbGV.debug> 1)
      ufdbLogMessage( "   sgIpv4( %s %d %s %d )", addr, type, file, lineno );
#endif

   switch (type)
   {
   case SG_IPTYPE_HOST:
	 if (ufdbNewGV.lastSource->ipv4hosts == NULL)
	    ufdbNewGV.lastSource->ipv4hosts = UFDBmemDBinit();
	 UFDBmemDBinsert( ufdbNewGV.lastSource->ipv4hosts, addr, NULL );
  	 break;

   case SG_IPTYPE_RANGE:
	 addr2 = strchr( addr, '-' );
	 *addr2 = '\0';
	 end = addr2;
	 while (*(end-1) == ' '  ||  *(end-1) == '\t')
	 {
	    *(end-1) = '\0';
	    end--;
	 }
	 addr2++;
	 while (*addr2 == ' ' || *addr2 == '\t')
	    addr2++;

         ipv4 = (struct Ipv4 *) ufdbMalloc( sizeof(struct Ipv4) );
	 ipv4->type = SG_IPTYPE_RANGE;
         ipv4->net_is_set = 1;
	 ipv4->next = NULL;
	 if (ufdbNewGV.lastSource->ipv4 == NULL) 
	    ufdbNewGV.lastSource->ipv4 = ipv4;
	 else
	    sgIpv4Last( ufdbNewGV.lastSource )->next = ipv4;

	 if (sgConvDot(addr,&ipv4->net) == NULL)
	 {
	    ufdbLogFatalError( "IPv4 address error in %s line %d: %s - %s", file, lineno, addr, addr2 );
	    ipv4->net = 0;
	    ipv4->net_is_set = 0;
	 }
	 if (sgConvDot(addr2,&ipv4->mask) == NULL)
	 {
	    ufdbLogFatalError( "IPv4 address error in %s line %d: %s - %s", file, lineno, addr, addr2 );
	    ipv4->mask = 0;
	    ipv4->net_is_set = 0;
	 }

	 if ((unsigned int) ipv4->net > (unsigned int) ipv4->mask)
	     ufdbLogFatalError( "IPv4 range error in %s line %d: %s - %s", file, lineno, addr, addr2 );
  	 break;

   case SG_IPTYPE_CLASS:
	 addr2 = strchr( addr, '/' );
	 *addr2 = '\0';
	 addr2++;
         ipv4 = (struct Ipv4 *) ufdbMalloc( sizeof(struct Ipv4) );
	 ipv4->type = SG_IPTYPE_CLASS;
         ipv4->net_is_set = 1;
	 ipv4->next = NULL;
	 if (ufdbNewGV.lastSource->ipv4 == NULL) 
	    ufdbNewGV.lastSource->ipv4 = ipv4;
	 else
	    sgIpv4Last( ufdbNewGV.lastSource )->next = ipv4;

	 if (sgConvDot(addr,&ipv4->net) == NULL)
	 {
	    ufdbLogFatalError( "IPv4 address error in %s line %d: %s/%s", file, lineno, addr, addr2 );
	    ipv4->net = 0;
	    ipv4->net_is_set = 0;
	 }
	 if (sgConvDot(addr2,&ipv4->mask) == NULL)
	 {
	    ufdbLogFatalError( "IPv4 address error in %s line %d: %s/%s", file, lineno, addr, addr2 );
	    ipv4->mask = 0;
	 }
  	 break;

   case SG_IPTYPE_CIDR:
	 cidr = strchr( addr, '/' );
	 *cidr = '\0';
	 cidr++;
	 octet = (unsigned long) atoi( cidr );
	 if (octet > 32) 
	 {
	    ufdbLogFatalError( "IPv4 address CIDR out of range in %s line %d: %s/%s",
	                       file, lineno, addr, cidr );
	    octet = 32;
	 }
         ipv4 = (struct Ipv4 *) ufdbMalloc( sizeof(struct Ipv4) );
	 ipv4->type = SG_IPTYPE_CIDR;
         ipv4->net_is_set = 1;
	 ipv4->next = NULL;
	 if (ufdbNewGV.lastSource->ipv4 == NULL) 
	    ufdbNewGV.lastSource->ipv4 = ipv4;
	 else
	    sgIpv4Last( ufdbNewGV.lastSource )->next = ipv4;

	 if (sgConvDot(addr,&ipv4->net) == NULL)
	 {
	    ufdbLogFatalError( "IPv4 address error in %s line %d: %s/%s", file, lineno, addr, cidr );
	    ipv4->net = 0;
	    ipv4->net_is_set = 0;
	 }
	 if (octet == 32)
	    ipv4->mask = 0xffffffff;
	 else
	    ipv4->mask = 0xffffffff ^ (0xffffffff >> octet);
	 ipv4->net = ipv4->net & ipv4->mask;
  	 break;
   }
}


static struct Ipv6 * sgIpv6Last( 
   struct Source * s )
{
   struct Ipv6 * ipv6;
   struct Ipv6 * last;

   last = NULL;
   for (ipv6 = s->ipv6;  ipv6 != NULL;  ipv6 = ipv6->next)
      last = ipv6;

   return last;
}



static void sgIpv6( 
   const char * addr,
   int          type,
   const char * file,
   int          line  )
{
#if 1
   if (ufdbGV.debug > 1)
      ufdbLogMessage( "   sgIpv6( %s %d  %s %d/%d )", addr, type, file, line, lineno );
#endif

   if (type == SG_IPV6TYPE_HOST)
   {
      struct in6_addr dummy;
      if (!sgValidateIPv6( addr, &dummy ))
      {
         ufdbLogFatalError( "incorrect IPv6 address \"%s\" in %s line %d", addr, file, lineno );
      }
      else
      {
         if (ufdbNewGV.lastSource->ipv6hosts == NULL)
	    ufdbNewGV.lastSource->ipv6hosts = UFDBmemDBinit();
	 UFDBmemDBinsert( ufdbNewGV.lastSource->ipv6hosts, addr, NULL );
      }
   }
   else if (type == SG_IPV6TYPE_CIDR)
   {
      char * s = (char*) strchr( addr, '/' );
      if (s == NULL)
      {
         ufdbLogFatalError( "IPv6 net has no '/' in \"%s\" in %s line %d", addr, file, lineno );
      }
      else
      {
	 struct Ipv6 *  ipv6;

	 ipv6 = (struct Ipv6 *) ufdbMalloc( sizeof(struct Ipv6) );
	 ipv6->type = SG_IPV6TYPE_CIDR;
	 ipv6->next = NULL;
	 if (ufdbNewGV.lastSource->ipv6 == NULL) 
	    ufdbNewGV.lastSource->ipv6 = ipv6;
	 else
	    sgIpv6Last( ufdbNewGV.lastSource )->next = ipv6;

         *s = '\0';
         ipv6->cidr = (unsigned long) atoi( s+1 );
         if (ipv6->cidr < 1  ||  ipv6->cidr > 128) 
         {
            ufdbLogFatalError( "IPv6 address CIDR out of range \"%s\" in %s line %d", addr, file, lineno );
            ipv6->cidr = 128;
         }
         if (!sgValidateIPv6( addr, &ipv6->ipv6 ))
         {
            ufdbLogFatalError( "incorrect IPv6 address \"%s\" in %s line %d", addr, file, lineno );
         }
         *s = '/';
      }
   }
   else
   {
      ufdbLogFatalError( "sgIpv6 called with unsupported type %d in %s line %d: %s", type, file, lineno, addr );
   }
}


/*
 * ACL functions
 */

static void ufdbAcl( 
   const char *    name,
   const char *    value, 
   int             within )
{
   struct Acl *    acl;
   struct Source * source;

#if UFDB_DEBUG
   ufdbLogMessage( "ufdbAcl name=\"%s\" value=\"%s\" within=%d line=%d", 
                   name==NULL?"NULL":name,  value==NULL?"NULL":value, within, lineno );
#endif

   if (ufdbNewGV.aclList != NULL) 
   {
      if (ufdbAclFindByName(&ufdbNewGV,name) != NULL)
      {
	 ufdbLogFatalError( "line %d: ACL \"%s\" is already defined in configuration file %s",
			    lineno, name, ufdbNewGV.configFile );
      }
   }

   if (within == UFDB_ACL_ELSE)
   {
      if (ufdbNewGV.lastAcl == NULL)
      {
         ufdbLogFatalError( "line %d: ACL \"else\" has no parent ACL", lineno );
         return;
      }
      if (name == NULL)
         name = ufdbStrdup( ufdbNewGV.lastAcl->name );
   }

   acl = (struct Acl *) ufdbMalloc( sizeof(struct Acl) );

   source = NULL;
   if (strcmp(name,"default") == 0)
   {
      ufdbNewGV.defaultAcl = acl;
   }
   else
   {
      if ((source = defSourceFindName(ufdbNewGV.sourceList,name)) == NULL)
      {
         ufdbLogFatalError( "line %d: ACL source \"%s\" is not defined in configuration file %s",
                            lineno, name, ufdbNewGV.configFile );
         if (ufdbNewGV.fastRefresh  &&  ufdbGV.fastRefresh  &&
             defSourceFindName(ufdbGV.sourceList,name) != NULL)
            ufdbLogError( "But ACL source \"%s\" is defined in the old configuration  *****", name );
         if (ufdbGV.debug > 1)
            UFDBlogConfig( &ufdbGV );
         ufdbFree( (void*) name );
         ufdbFree( (void*) acl );
         return;
      }
   }

   acl->name = name;
   acl->active = within == UFDB_ACL_ELSE ? 0 : 1;
   acl->source = source;
   acl->pass = NULL;
   acl->implicitPass = NULL;
   acl->hasTerminatorNone = 0;
   acl->rewriteDefault = 1;
   acl->rewrite = NULL;
   acl->redirect = NULL;
   acl->time = NULL;
   acl->within = within;
   acl->next = NULL;

   if (value != NULL) 
   {
      struct ufdbTime * t;
      if ((t = sgTimeFindName(value)) == NULL) 
      {
         ufdbLogFatalError( "line %d: ACL time %s is not defined in configuration file %s",
                            lineno, value, ufdbNewGV.configFile );
         return;
      }
      acl->time = t;
   }

   if (ufdbNewGV.aclList == NULL) 
   {
      ufdbNewGV.aclList = acl;
      ufdbNewGV.lastAcl = acl;
   }
   else
   {
      ufdbNewGV.lastAcl->next = acl;
      ufdbNewGV.lastAcl = acl;
   }
}


static void ufdbAclSetValue( 
   const char *         what,
   const char *         value, 
   int                  allowed ) 
{
   struct Category *    cat = NULL;
   struct AclCategory * aclcat;
   int                  type;
  
#if UFDB_DEBUG
   ufdbLogMessage( "ufdbAclSetValue %s %s%s", what, allowed ? "" : "!", value );
#endif

   if (ufdbNewGV.lastAcl == NULL)
   {
      ufdbLogError( "error in configuration file on line %d: "
                    "cannot set value for \"%s\" because there is no defined ACL", 
                    lineno, what );
      ufdbFree( (void*) value );
      return;
   }

   if (strcmp(what,"pass") == 0)
   {
      if (strcmp(value,"any")==0 || strcmp(value,"all")==0)
      {
         if (!allowed)
            ufdbLogFatalError( "error in configuration file on line %d: do not use '!any' or '!all'.  "
                               "Use 'none' instead.", lineno );
         type = ACL_TYPE_TERMINATOR;
         allowed = 1;
      }
      else if (strcmp(value,"none") == 0)
      {
         if (!allowed)
            ufdbLogFatalError( "error in configuration file on line %d: do not use '!none'.  "
                               "Use 'any' instead.", lineno );
         type = ACL_TYPE_TERMINATOR;
         allowed = 0;
      }
      else if (strcmp(value,"in-addr") == 0)
      {
         type = ACL_TYPE_INADDR;
      }
      else
      {
         type = ACL_TYPE_DEFAULT;
         if ((cat = ufdbCategoryFindByName(&ufdbNewGV,value)) == NULL) 
         {
	    ufdbLogFatalError( "ACL category \"%s\" (line %d) is not defined in configuration file %s",
			       value, lineno, ufdbNewGV.configFile );
            if (ufdbNewGV.fastRefresh  &&  ufdbGV.fastRefresh  &&
                ufdbCategoryFindByName(&ufdbGV,value) != NULL)
            {
               ufdbLogError( "But category \"%s\" is defined in the old configuration  *****", value );
               UFDBlogConfig( &ufdbGV );
            }
	    ufdbFree( (void*) value );
            return;
         } 
      }

      aclcat = (struct AclCategory *) ufdbMallocAligned( UFDB_CACHELINE_SIZE, sizeof(struct AclCategory) );
      aclcat->name = value;
      aclcat->cat = cat;
      aclcat->access = allowed;
      aclcat->type = type;
      aclcat->next = NULL;
      aclcat->nblocks = 0;
      aclcat->nmatches = 0;

      if (ufdbNewGV.lastAcl->pass == NULL)
         ufdbNewGV.lastAcl->pass = aclcat;
      else 
         ufdbNewGV.lastAclCategory->next = aclcat;
      ufdbNewGV.lastAclCategory = aclcat;
   }
   else if (strcmp(what,"redirect") == 0)
   {
      if (strcmp(value,"default") != 0)
         ufdbNewGV.lastAcl->redirect = value;
      else 
      {
         ufdbNewGV.lastAcl->redirect = NULL;
         ufdbFree( (void*) value );
      }
   }
   else if (strcmp(what,"rewrite") == 0)
   {
      if (strcmp(value,"none") == 0)
      {
         ufdbNewGV.lastAcl->rewriteDefault = 0;
         ufdbNewGV.lastAcl->rewrite = NULL;
      }
      else
      {
         struct sgRewrite * rewrite;

         if ((rewrite = sgRewriteFindName(value)) == NULL) 
         {
            ufdbLogFatalError( "rewrite %s is not defined in configuration file %s",
                               value, ufdbNewGV.configFile );
         }
         ufdbNewGV.lastAcl->rewriteDefault = 0;
         ufdbNewGV.lastAcl->rewrite = rewrite;
      }
      ufdbFree( (void*) value );
   }
}


struct Acl * ufdbAclFindByName( 
   struct ufdbGV * gv,
   const char *    name )
{
   struct Acl * p;

   if (name == NULL)
      return NULL;

   for (p = gv->aclList;  p != NULL;  p = p->next)
   {
      if (strcmp(name,p->name) == 0)
         return p;
   }

   return NULL;
}


static void logConfig( void )
{
   int    lno;
   FILE * fin;
   char   line[32678];
   
   if (ufdbNewGV.configLogged)
      return;
   ufdbNewGV.configLogged = 1;

   fin = fopen( ufdbNewGV.configFile, "r" );
   if (fin == NULL)
      return;

   ufdbLogMessage( "======== literal content of config file ========" );
   lno = 1;
   while (UFDBfgetsNoNL(line,sizeof(line)-1,fin) != NULL)
   {
      ufdbLogMessage( "%04d: %s", lno, line );
      lno++;
   }
   ufdbLogMessage( "======== end of literal content ========" );
   fclose( fin );
}


void yyerror( const char * s )
{
   ufdbLogFatalError( "line %d: %s in configuration file %s", lineno, s, ufdbNewGV.configFile );
}


int yywrap( void )
{
  return 1;
}

